#include <stdio.h>
#include "ptrie.h"
#include "stack.h"

static char *program;
static int conv_ok, unknown, ambig, total;

static void
#ifdef __STDC__
stack_free_func(char *data)
#else /* !__STDC__ */
stack_free_func(data)
char *data;
#endif /* !__STDC__ */
{
    if (data != NULL)
      free(data);
}

static void
#ifdef __STDC__
ptrie_free_func(stack data)
#else /* !__STDC__ */
ptrie_free_func(data)
stack data;
#endif /* !__STDC__ */
{
    if (data != NULL)
      stack_free(data, stack_free_func);
}

static int
#ifdef __STDC__
add_line(char *line, ptrie tree)
#else /* !__STDC__ */
add_line(line, tree)
char *line;
ptrie tree;
#endif /* !__STDC__ */
{
    char *pinyin, *ptr;
    unsigned char danci[3];
    stack nstack;
    int dcount;

    ptr = line;
    /*
     * Go to the end of the Pinyin.
     */
    while(*ptr && *ptr != ' ')
      ptr++;

    /*
     * Make a copy of the Pinyin.
     */
    pinyin = (char *)malloc(sizeof(char) * (int)(ptr - line + 1));
    (void)strncpy(pinyin, line, (int)(ptr - line));

    /*
     * Move to the first danci.
     */
    while(*ptr && !(*ptr & 0x80))
      ptr++;

    /*
     * Do this while more danci remain in the line.
     */
    dcount = 0;
    while(*ptr != '\0') {
        danci[0] = *ptr++;
        danci[1] = *ptr++;
        danci[2] = '\0';
        dcount++;
        nstack = (stack)ptrie_find(tree, (unsigned char *)danci);
        if (nstack == NULL) {
            nstack = stack_create(20);
            /*
             * A stack overflow on a newly created stack should
             * not be possible if the stack size is greater than 0.
             * But, may as well be careful.
             */
            if (stack_push(nstack, (void *)pinyin) < 0) {
                fprintf(stderr, "%s: 0 stack overflow (%s %d)\n", program,
                        pinyin, dcount);
                stack_free(nstack, stack_free_func);
                return(-1);
            }
            /*
             * Insert the danci/pinyin stack pair into the search
             * trie.
             */
            ptrie_insert(tree, (unsigned char *)danci, (void *)nstack);
        } else {
            /*
             * Make sure we don't overflow the stack when pushing
             * a new Pinyin on it.
             */
            if (stack_push(nstack, (void *)pinyin) < 0) {
                fprintf(stderr, "%s: 1 stack overflow (%s %d)\n", program,
                        pinyin, dcount);
                return(-1);
            }
        }

        /*
         * Move to next danci.
         */
        while(*ptr && !(*ptr & 0x80))
          ptr++;
    }
    return(0);
}

static ptrie
#ifdef __STDC__
load_danci(FILE *dfile)
#else /* !__STDC__ */
load_danci(dfile)
FILE *dfile;
#endif /* !__STDC__ */
{
    char buf[BUFSIZ];
    ptrie tree;

    /*
     * Create the search trie for the danci.
     */
    tree = ptrie_create();

    fgets(buf, BUFSIZ, dfile);
    while(!feof(dfile)) {
        if (add_line((char *)buf, tree) < 0) {
            ptrie_free(tree, ptrie_free_func);
            return(NULL);
        }
        fgets(buf, BUFSIZ, dfile);
    }
    return(tree);
}

static void
#ifdef __STDC__
topinyin(FILE *in, FILE *msg, ptrie tree)
#else /* !__STDC__ */
topinyin(in, msg, tree)
FILE *in, *msg;
ptrie tree;
#endif /* !__STDC__ */
{
    int c, line_no = 0;
    unsigned char danci[3];
    stack st;
    char *pinyin;

    c = getc(in);
    while(!feof(in)) {
        if (c & 0x80) {
            danci[0] = c;
            danci[1] = getc(in);
            danci[2] = '\0';
            st = (stack)ptrie_find(tree, (unsigned char *)danci);
            if (st == NULL) {
                printf("<%s unknown>", (char *)danci);
                fprintf(msg, "<%s unknown> line (%d)\n",
                        (char *)danci, line_no);
                unknown++;
            } else {
                if (stack_size(st) > 1) {
                    printf("<%s>", (char *)danci);
                    fprintf(msg, "<%s ambig> line (%d)\n",
                            (char *)danci, line_no);
                    ambig++;
                } else {
                    pinyin = (char *)stack_top(st);
                    printf("%s", pinyin);
                    conv_ok++;
                }
                total++;
            }
        } else if (c == '\n') {
            line_no++;
            putchar(c);
        } else
          putchar(c);
        c = getc(in);
    }
}

static void
#ifdef __STDC__
report_stats(char *fname)
#else /* !__STDC__ */
report_stats(fname)
char *fname;
#endif /* !__STDC__ */
{
    FILE *statfile;
    double res;

    statfile = fopen("stats", "a");
    if (statfile != NULL) {
        fprintf(statfile, "%s: for file \"%s\" out of %d total characters:\n",
                program, fname, total);
        res = (((double)conv_ok)/((double)total)) * 100.0;
        fprintf(statfile, "\t%g%% (%d) converted ok\n", res, conv_ok);
        res = (((double)ambig)/((double)total)) * 100.0;
        fprintf(statfile, "\t%g%% (%d) ambiguous\n\n", res, ambig);
        fclose(statfile);
    }
}

static void
#ifdef __STDC__
usage(void)
#else /* !__STDC__ */
usage()
#endif /* !__STDC__ */
{
    fprintf(stderr, "usage  %s -d danci-file [input-file]\n",
            program);
    exit(-1);
}

int
#ifdef __STDC__
main(int argc, char **argv)
#else /* !__STDC__ */
main(argc, argv)
int argc;
char **argv;
#endif /* !__STDC__ */
{
    FILE *dfile = NULL, *msgfile = stderr, *infile = stdin;
    char *dfile_name, *msgfile_name, *infile_name;
    ptrie tree = NULL;
    char msgbuf[BUFSIZ];

    program = argv[0];

    if (argc < 3)
      usage();

    argc--;
    *argv++;
    while(argc != 0) {
        if (argv[0][0] == '-') {
            switch(argv[0][1]) {
              case 'd': case 'D':
                argc--;
                *argv++;
                dfile_name = argv[0];

                /*
                 * A new danci file needs to be loaded.
                 */
                dfile = fopen(dfile_name, "r");
                if (dfile == NULL) {
                    fprintf(stderr, "%s: problem opening danci file \"%s\"\n",
                            program, dfile_name);
                    if (msgfile != NULL && msgfile != stderr)
                      fclose(msgfile);
                    if (tree != NULL)
                      ptrie_free(tree, ptrie_free_func);
                    exit(-1);
                }
                if (tree != NULL)
                  ptrie_free(tree, ptrie_free_func);
                tree = load_danci(dfile);
                fclose(dfile);
                if (tree == NULL) {
                    fprintf(stderr, "%s: problem loading danci file \"%s\"\n",
                            program, dfile_name);
                    if (msgfile != NULL && msgfile != stderr)
                      fclose(msgfile);
                    exit(-1);
                }
                break;
              default:
                if (tree != NULL)
                  ptrie_free(tree, ptrie_free_func);
                fprintf(stderr, "%s: unknown command line switch\n", program);
                usage();
                break;
            }
        } else {
            if (argc != 0) {
                infile_name = argv[0];
                infile = fopen(infile_name, "r");
                if (infile == NULL) {
                    fprintf(stderr, "%s: problem opening input file \"%s\"\n",
                            program, infile_name);
                    if (tree != NULL)
                      ptrie_free(tree, ptrie_free_func);
                }
                sprintf(msgbuf, "%s.msg", infile_name);
            } else
              sprintf(msgbuf, "stdin.msg");
            msgfile = fopen(msgbuf, "w");
            if (msgfile == NULL) {
                if (tree != NULL)
                  ptrie_free(tree, ptrie_free_func);
                fclose(infile);
                fprintf(stderr, "%s: couldn't open the message file \"%s\"\n",
                        program, msgbuf);
                exit(0);
            }
            conv_ok = unknown = ambig = total = 0;
            topinyin(infile, msgfile, tree);
            if (infile == stdin)
              report_stats("stdin");
            else
              report_stats(infile_name);
            if (infile != stdin) {
                fclose(infile);
                infile = stdin;
            }
        }
        argc--;
        *argv++;
    }
}
