/*
 * The new sysinstall program.
 *
 * This is probably the last attempt in the `sysinstall' line, the next
 * generation being slated for what's essentially a complete rewrite.
 *
 * $FreeBSD: src/release/sysinstall/main.c,v 1.50.2.3 1999/08/29 15:54:06 peter Exp $
 *
 * Copyright (c) 1995
 *	Jordan Hubbard.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JORDAN HUBBARD ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "sysinstall.h"
#include <sys/signal.h>
#include <sys/fcntl.h>

static void
screech(int sig)
{
    msgDebug("\007Signal %d caught!  That's bad!\n", sig);
    longjmp(BailOut, sig);
}

int
main(int argc, char **argv)
{
    int choice, scroll, curr, max, status;
#ifdef AT_KANJI
#ifdef  USE_KON
    int use_kon = 0;
#endif  /* USE_KON */
#if defined(JAPANESE) || defined(CHINESE) || defined(KOREAN)
    char *lang_var = 0;
#endif	/* JAPANESE || CHINESE || KOREAN */
#endif	/* AT_KANJI */

    /* Catch fatal signals and complain about them if running as init */
    if (getpid() == 1) {
	signal(SIGBUS, screech);
	signal(SIGSEGV, screech);
    }

    /* We don't work too well when running as non-root anymore */
    if (geteuid() != 0) {
	fprintf(stderr, "Error: This utility should only be run as root.\n");
	return 1;
    }
#ifdef PC98
    {	char *p;

    	p = getenv("TERM");
    	if (p && strcmp(p, "cons25") == 0)
    		putenv("TERM=cons25w");
    }
#endif

    /* Set up whatever things need setting up */
    systemInitialize(argc, argv);

#ifdef AT_KANJI
#if defined(JAPANESE) || defined(CHINESE) || defined(KOREAN)
    dmenuOpenSimple(&MenuI18N, FALSE);
    lang_var = variable_get("language");
    if (lang_var) {
	if (strcmp(lang_var, "japanese") == 0) {
	    language = LANG_JAPANESE;
	}
	else if (strcmp(lang_var, "chinese") == 0) {
	    language = LANG_CHINESE;
	}
	else if (strcmp(lang_var, "korean") == 0) {
	    language = LANG_KOREAN;
	}
    }
    if (USE_KON_LANG(language)) {
	msgInfo(NULL);
        use_kon = 1;
    }
#else
    language = LANG_ENGLISH;
#endif
    init_i18n();
    override_termcap();

#ifdef  USE_KON
    /* this if () condition is not complete, but I think it's */
    /* acceptable for almost all cases */
    if (use_kon && 
	(RunningAsInit || strncmp(getenv("TERM") + 1, "ons2", 4) == 0)) {
	kon2_main();
    }
#endif  /* USE_KON */
#endif	/* AT_KANJI */

    /* Set default flag and variable values */
    installVarDefaults(NULL);
    /* only when multi-user is it reasonable to do this here */
    if (!RunningAsInit)
	installEnvironment();

    if (argc > 1 && !strcmp(argv[1], "-fake")) {
	variable_set2(VAR_DEBUG, "YES", 0);
	Fake = TRUE;
	/* AT å: ե⡼ɤޤ */
	KANJI_MESSAGE("ʸϤΤ褦˸ޤǤ?");
	msgConfirm(KM("I'll be just faking it from here on out, OK?"));
    }

    /* Try to preserve our scroll-back buffer */
    if (OnVTY) {
	for (curr = 0; curr < 25; curr++)
	    putchar('\n');
    }
    /* Move stderr aside */
    if (DebugFD)
	dup2(DebugFD, 2);

#if 1	/* ư */
#ifdef KANJI_ONLY
    kanji = 1;
#elif defined(KANJI)
    kanji = !dialog_yesno("Japanese Message Check", "Do you use Japanese message", -1, -1);
    dialog_clear();
#endif	/* KANJI_ONLY, KANJI */
#endif

#ifdef PCCARD
    /* Initialize PC-card */
    pccardInitialize();
#endif

    /* Probe for all relevant devices on the system */
    deviceGetAll();

    /* First, see if we have any arguments to process (and argv[0] counts if it's not "sysinstall") */
    if (!RunningAsInit) {
	int i, start_arg;

	if (!strstr(argv[0], "sysinstall"))
	    start_arg = 0;
	else if (Fake)
	    start_arg = 2;
	else
	    start_arg = 1;
	for (i = start_arg; i < argc; i++) {
	    if (DITEM_STATUS(dispatchCommand(argv[i])) != DITEM_SUCCESS)
		systemShutdown(1);
	}
	if (argc > start_arg)
	    systemShutdown(0);
    }
    else
	dispatch_load_file_int(TRUE);

    status = setjmp(BailOut);
    if (status) {
	KANJI_MESSAGE(
	"ʥ %d ȯޤ. Ǥ¤ΤΤ¸ƥåȥ\n"
	"ޤ. Ƹ, ץ˥塼 Debug ͭˤ, \n"
	"ΥǥХåͭǤ;ʬʥǥХå󶡤Ƥ.");
	msgConfirm(KM("A signal %d was caught - I'm saving what I can and shutting\n"
		   "If you can reproduce the problem, please turn Debug on in\n"
		   "the Options menu for the extra information it provides in\n"
		   "debugging problems like this."), status);
	systemShutdown(status);
    }

#if 0	/* ˰ư */
#ifdef KANJI_ONLY
    kanji = 1;
#elif defined(KANJI)
    kanji = !dialog_yesno("Japanese Message Check", "Do you use Japanese message", -1, -1);
    dialog_clear();
#endif	/* KANJI_ONLY, KANJI */
#endif

#ifdef KANJI
    /* tzsetup ܸɽΤɬפǤ */
    if (kanji)
	putenv("LANG=ja_JP.EUC");
#endif	/* KANJI */
#ifdef PC98
    /* IDE ͥ */
    if (getpid() == 1) {
	static void Ide24Check(void);

	Ide24Check();
#if 0
	if (file_readable("/IDE2")) {		/* boot.flp  /IDE2  */
	    DistsIdeFlg |= DIST_98IDE2;
	    msgDebug("IDE2 look!\n");
	}
	else if(file_readable("/IDE4"))	{	/* boot.flp  /IDE4  */
	    DistsIdeFlg |= DIST_98IDE4;
	    msgDebug("IDE4 look!\n");
	}
	if (file_readable("/stand/help.JP/miss")) {		/* 2HC  help ե뤬­ʤ */
		msgDebug("/stand/help.JP/miss ǧޤ\n");	/* debug */
		getHelpFloppy();
		msgDebug("end getHelpFloppy\n");	/* debug */
	}
#endif
    }
#endif	/* PC98 */
    /* Begin user dialog at outer menu */
    dialog_clear();
    while (1) {
	choice = scroll = curr = max = 0;
	dmenuOpen(&MenuInitial, &choice, &scroll, &curr, &max, TRUE);
	/* : ...Ȥޤ. \n"  . θΥڡϲ̤줵ʤ˽פǤ */
	KANJI_MESSAGE(
	"˽λƹޤ? ƥϥ֡Ȥޤ. \n"
	"(եåԡɥ饤֤ȴƲ)");
	if (getpid() != 1
#ifdef __alpha__
	    || !msgYesNo("Are you sure you wish to exit?  The system will halt.")
#else
	    || !msgYesNo(KM("Are you sure you wish to exit?  The system will reboot\n"
		         "(be sure to remove any floppies from the drives)."))
#endif
	    )
	    break;
    }

    /* Say goodnight, Gracie */
    systemShutdown(0);

    return 0; /* We should never get here */
}

#ifdef PC98
#include <sys/sysctl.h>
static void Ide24Check(void)
{
	char buff[300];
	int mib[2], len;

	mib[0] = CTL_KERN;
	mib[1] = KERN_VERSION;
	len = sizeof(buff);
	sysctl(mib, 2, &buff, &len, NULL, 0);

	msgDebug("%s\n", buff);	/* debug */
	if (strstr(buff, "IDE2")) {		/* version  IDE2  */
	    DistsIdeFlg |= DIST_98IDE2;
	    msgDebug("IDE2 look!\n");
	}
	else if(strstr(buff, "IDE4"))	{	/* version  IDE4  */
	    DistsIdeFlg |= DIST_98IDE4;
	    msgDebug("IDE4 look!\n");
	}
	else if(strstr(buff, "UIDE"))	{	/* version  UIDE  */
	    DistsIdeFlg |= DIST_98UIDE;
	    msgDebug("UIDE look!\n");
	}
#if 0
	if (file_readable("/stand/help.JP/miss")) {		/* 2HC  help ե뤬­ʤ */
		msgDebug("/stand/help.JP/miss ǧޤ\n");	/* debug */
		getHelpFloppy();
		msgDebug("end getHelpFloppy\n");	/* debug */
	}	
#endif
}
#endif
