/*
 * The new sysinstall program.
 *
 * This is probably the last attempt in the `sysinstall' line, the next
 * generation being slated to essentially a complete rewrite.
 *
 * $FreeBSD: src/release/sysinstall/network.c,v 1.33.2.10 1999/12/17 02:46:34 jkh Exp $
 *
 * Copyright (c) 1995
 *	Jordan Hubbard.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JORDAN HUBBARD ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/* These routines deal with getting things off of network media */

#include "sysinstall.h"
#include <signal.h>
#include <termios.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

static Boolean	networkInitialized;
static pid_t	startPPP(Device *devp);
#ifdef DHCP_SUPPORT
static pid_t	startDHCP(Device *devp);
#endif

static pid_t	pppPID;
#ifdef DHCP_SUPPORT
static pid_t	pppDHCP;
#endif

Boolean
mediaInitNetwork(Device *dev)
{
    int i;
    char *rp;
    char *cp, ifconfig[255];
    WINDOW *w;
#ifdef DHCP_SUPPORT
    DevInfo * di = (DevInfo *)dev->private;
#endif

    if (!RunningAsInit || networkInitialized)
	return TRUE;

    if (isDebug())
	msgDebug("Init routine called for network device %s.\n", dev->name);

    if (!file_readable("/etc/resolv.conf")) {
	if (DITEM_STATUS(configResolv(NULL)) == DITEM_FAILURE) {
	    KANJI_MESSAGE("/etc/resolv.conf ˽񤭹ʤ褦Ǥ. ͥåȤϻȤޤ.");
	    msgConfirm(KM("Can't seem to write out /etc/resolv.conf.  Net cannot be used."));
	    return FALSE;
	}
    }

    w = savescr();
    dialog_clear_norefresh();

    /* Old PPP process lying around? */
    if (pppPID) {
	KANJI_MESSAGE(" PPP ץ %d ߤƤޤ");
	msgConfirm(KM("Killing previous PPP process %d."), pppPID);
	kill(pppPID, SIGTERM);
	pppPID = 0;
    }
#ifdef DHCP_SUPPORT
    /* Old DHCP client lying around? */
    else if (pppDHCP) {
	KANJI_MESSAGE("˵ưƤ DHCP 饤 %d ߤޤ");
        msgNotify(KM("Killing previous DHCP client %d."), pppDHCP);
        kill(pppDHCP, SIGTERM);
        unlink("/etc/services");
        pppDHCP = 0;
    }
#endif

    if (!strncmp("ppp", dev->name, 3)) {	/* PPP? */
	if (!(pppPID = startPPP(dev))) {
	    KANJI_MESSAGE("PPP 򳫻ϤǤޤ! Υ󥹥ȡˡϻѤǤޤ.");
	    msgConfirm(KM("Unable to start PPP!  This installation method cannot be used."));
	    return FALSE;
	}
	networkInitialized = TRUE;
	return TRUE;
    }
    else if (!strncmp("sl", dev->name, 2)) {	/* SLIP? */
	char *val;
	char attach[256];

	/* Cheesy slip attach */
	snprintf(attach, 256, "slattach -a -h -l -s 9600 %s", dev->devname);
#ifdef PC98
	KANJI_MESSAGE(
	"ٹ: ǤΥ󥹥ȡ SLIP ؤΥݡȤ, ʬǽ㤤\n"
	"ȤʤäƤޤ. ϥ󥰤ԤʤΥ桼ƥƥ˷礱\n"
	"Ƥ뤳ȤǤ. ⤷ PPP ѤǤΤǤ, \n"
	"Ѥޤ礦. SLIP ľܥꥢ³줿˻ѤƲ\n"
	". ޤ, ʲ˼ slattach ޥɤԽƲ (ǤΥ\n"
	"եȤ, VJ , ϡɥե, ꥢ̵, 9600 ܡ\n"
	"ž졼ȤǤ). Ǥ, [RETURN] 򲡤Ƽ¹ԤƲ.");
	val = msgGetInput(attach, KM(
			  "Warning:  SLIP is rather poorly supported in this revision\n"
			  "of the installation due to the lack of a dialing utility.\n"
			  "If you can use PPP for this instead then you're much better\n"
			  "off doing so, otherwise SLIP works fairly well for *hardwired*\n"
			  "links.  Please edit the following slattach command for\n"
			  "correctness (default here is: VJ compression, Hardware flow-\n"
			  "control, ignore carrier and 9600 baud data rate).  When you're\n"
			  "ready, press [RETURN] to execute it."));
#else
	KANJI_MESSAGE(
	"ٹ: ǤΥ󥹥ȡ SLIP ؤΥݡȤ, ʬǽ㤤\n"
	"ȤʤäƤޤ. ϥ󥰤ԤʤΥ桼ƥƥ˷礱\n"
	"Ƥ뤳ȤǤ. ⤷ PPP ѤǤΤǤ, \n"
	"Ѥޤ礦. SLIP ľܥꥢ³줿˻ѤƲ\n"
	". ޤ, ʲ˼ slattach ޥɤԽƲ (ǤΥ\n"
	"եȤ, VJ , ϡɥե, ꥢ̵, 9600 ܡ\n"
	"ž졼ȤǤ). Ǥ, [ENTER] 򲡤Ƽ¹ԤƲ.");
	val = msgGetInput(attach, KM(
			  "Warning:  SLIP is rather poorly supported in this revision\n"
			  "of the installation due to the lack of a dialing utility.\n"
			  "If you can use PPP for this instead then you're much better\n"
			  "off doing so, otherwise SLIP works fairly well for *hardwired*\n"
			  "links.  Please edit the following slattach command for\n"
			  "correctness (default here is: VJ compression, Hardware flow-\n"
			  "control, ignore carrier and 9600 baud data rate).  When you're\n"
			  "ready, press [ENTER] to execute it."));
#endif	/* PC98 */
	if (!val) {
	    KANJI_MESSAGE("slattach ޥɤǤ. ʤ!");
	    msgConfirm(KM("slattach command was empty.  Try again!"));
	    restorescr(w);
	    return FALSE;
	}
	else
	    SAFE_STRCPY(attach, val);
	/*
	 * Doing this with vsystem() is actually bogus since we should be storing the pid of slattach
	 * for later killing.  It's just too convenient to call vsystem(), however, rather than
	 * constructing a proper argument for exec() so we punt on doing slip right for now.
	 */
	if (vsystem(attach)) {
	    KANJI_MESSAGE(
	    "slattach ơ֤ޤ. ޥɤ⤦ٳǧƺ\n"
	    "٤ƤߤƲ.");
	    msgConfirm(KM("slattach returned a bad status!  Please verify that\n"
		       "the command is correct and try this operation again."));
	    restorescr(w);
	    return FALSE;
	}
    }

#ifdef DHCP_SUPPORT
    if (di && !strcmp(di->ipaddr, "DHCP")) {
        if (!(pppDHCP = startDHCP(dev))) {
            KANJI_MESSAGE(
            "DHCP 饤Ȥ󥿡ե %s ưǤޤ!\n"
            "Υ󥹥ȡˡϻѤǤޤ");
            msgConfirm(KM("Unable to start DHCP in %s interface!\n"
            		"This installation method cannot be used."), dev->name);
	    return FALSE;
	}
	networkInitialized = TRUE;
	return TRUE;
    }
#endif

    snprintf(ifconfig, 255, "%s%s", VAR_IFCONFIG, dev->name);
    cp = variable_get(ifconfig);
    if (!cp) {
	KANJI_MESSAGE(
	"ǥХ %s ѲǽȤʤäƤޤ. κȤԤʤ˥ͥå\n"
	"˥塼ǥǥХνԤʤɬפޤ.");
	msgConfirm(KM("The %s device is not configured.  You will need to do so\n"
		   "in the Networking configuration menu before proceeding."), dev->name);
	return FALSE;
    }
#ifdef DHCP_SUPPORT
    else if (!strcmp(cp, "DHCP"))
	goto bail;
#endif
    msgDebug("ifconfig %s %s", dev->name, cp);
    i = vsystem("ifconfig %s %s", dev->name, cp);
    if (i) {
	KANJI_MESSAGE(
	"󥿡ե %s Ǥޤ!\n"
	"Υ󥹥ȡˡϻѤǤޤ.");
	msgConfirm(KM("Unable to configure the %s interface!\n"
		   "This installation method cannot be used."), dev->name);
	return FALSE;
    }

    rp = variable_get(VAR_GATEWAY);
    if (!rp || *rp == '0') {
	KANJI_MESSAGE(
	"ȥꤵƤޤ. ͥåȥʳ˥\n"
	"뤳ȤϤǤޤ.");
	msgConfirm(KM("No gateway has been set. You may be unable to access hosts\n"
		   "not on your local network"));
    }
    else {
	/* KANJI_MESSAGE("%s ˥ǥեȷϩꤷƤޤ."); */
	msgDebug("Adding default route to %s.", rp);
	vsystem("route -n add default %s", rp);
    }
bail:
    if (isDebug())
	msgDebug("Network initialized successfully.\n");
    networkInitialized = TRUE;
    return TRUE;
}

void
mediaShutdownNetwork(Device *dev)
{
    char *cp;

    if (!RunningAsInit || !networkInitialized)
	return;

    msgDebug("Shutdown called for network device %s\n", dev->name);
    if (pppPID) {
        KANJI_MESSAGE("˵ưƤ PPP ץ %d ߤޤ");
        msgNotify(KM("Killing previous PPP process %d."), pppPID);
        kill(pppPID, SIGTERM);
        pppPID = 0;
    }
#ifdef DHCP_SUPPORT
    else if (pppDHCP) {
	KANJI_MESSAGE("˵ưƤ DHCP 饤 %d ߤޤ");
        msgNotify(KM("Killing previous DHCP client %d."), pppDHCP);
        kill(pppDHCP, SIGTERM);
        unlink("/etc/services");
        pppDHCP = 0;
    }
#endif
    /* Not a serial device? */
    else if (strncmp("sl", dev->name, 2) && strncmp("ppp", dev->name, 3)) {
	int i;
	char ifconfig[255];

	snprintf(ifconfig, 255, "%s%s", VAR_IFCONFIG, dev->name);
	cp = variable_get(ifconfig);
	if (!cp)
	    return;
	msgDebug("ifconfig %s down", dev->name);
	i = vsystem("ifconfig %s down", dev->name);
	if (i) {
	    KANJI_MESSAGE("ٹ: 󥿡ե %s ˽λǤޤ.");
	    msgConfirm(KM("Warning: Unable to down the %s interface properly"), dev->name);
	}
	cp = variable_get(VAR_GATEWAY);
	if (cp) {
	    /* KANJI_MESSAGE("ǥեȷϩƤޤ."); */
	    msgDebug("Deleting default route.");
	    vsystem("route -n delete default");
	}
    }
    networkInitialized = FALSE;
}

/* Start PPP on the 3rd screen */
static pid_t
startPPP(Device *devp)
{
    int fd2, pulse;
    FILE *fp;
    char *val;
    pid_t pid = 0;
    char myaddr[16], provider[16], speed[16], authname[32], authkey[16];
    char phone[16];
    WINDOW *w = savescr();

    /* These are needed to make ppp work */
    Mkdir("/var/log");
    Mkdir("/var/run");
    Mkdir("/var/spool/lock");
    Mkdir("/etc/ppp");

    dialog_clear_norefresh();
    if (!variable_get(VAR_SERIAL_SPEED))
	variable_set2(VAR_SERIAL_SPEED, "115200", 0);
    /* Get any important user values */
    KANJI_MESSAGE(
    "ǥؤΥܡ졼ȤꤷƲ. ۤȤɤΥǥ, Ф\n"
    "®٤̤®٤ǥԥ塼ȥǡžǽȤʤäƤΤ,\n"
    "ͤϥǥब󶡤ºݤκǹ®٤⤯ꤹ뤳ȤǤ\n"
    ". ⤷ꤷɤΤ狼ʤ, ǥեȤ򤷤Ʋ\n"
    ".");
    val = variable_get_value(VAR_SERIAL_SPEED, KM(
		      "Enter the baud rate for your modem - this can be higher than the actual\n"
		      "maximum data rate since most modems can talk at one speed to the\n"
		      "computer and at another speed to the remote end.\n\n"
		      "If you're not sure what to put here, just select the default."), 0);
    SAFE_STRCPY(speed, (val && *val) ? val : "115200");

    val = variable_get(VAR_GATEWAY);
    SAFE_STRCPY(provider, (val && *val) ? val : "0");

    dialog_clear_norefresh();
    KANJI_MESSAGE(
    "ʤΥӥץХ IP ɥ쥹ϤƲ. ⤷\n"
    "狼ʤ, 뤤ưŪʥͥԤʤ\n"
    ", 0 ϤƲ.");
    val = msgGetInput(provider, KM("Enter the IP address of your service provider or 0 if you\n"
		      "don't know it and would prefer to negotiate it dynamically."));
    SAFE_STRCPY(provider, (val && *val) ? val : "0");

    if (devp->private && ((DevInfo *)devp->private)->ipaddr[0])
	SAFE_STRCPY(myaddr, ((DevInfo *)devp->private)->ipaddr);
    else
	strcpy(myaddr, "0");

    if (!Fake)
	fp = fopen("/etc/ppp/ppp.linkup", "w");
    else
	fp = fopen("/dev/stderr", "w");
    if (fp != NULL) {
	fprintf(fp, "MYADDR:\n");
	fprintf(fp, " delete ALL\n");
        fprintf(fp, " add 0 0 HISADDR\n");
	fchmod(fileno(fp), 0755);
	fclose(fp);
    }
    if (!Fake)
	fd2 = open("/etc/ppp/ppp.secret", O_CREAT);
    else
	fd2 = -1;
    if (fd2 != -1) {
	fchmod(fd2, 0700);
	close(fd2);
    }
    if (!Fake)
	fp = fopen("/etc/ppp/ppp.conf", "a");
    else
	fp = fopen("/dev/stderr", "w");
    if (!fp) {
	KANJI_MESSAGE("/etc/ppp/ppp.conf 򥪡ץǤޤ!  εǽϻǽǤ");
	msgConfirm(KM("Couldn't open /etc/ppp/ppp.conf file!  This isn't going to work"));
	restorescr(w);
	return 0;
    }
    authname[0] = '\0';
    pulse = 0;
    dialog_clear_norefresh();
    KANJI_MESSAGE("ISP  PAP  CHAP ppp 򥵥ݡȤƤޤ?");
    if (!dialog_yesno("", KM("Does your ISP support PAP or CHAP ppp logins?"), -1, -1)) {
	KANJI_MESSAGE("ץХΥ˻Ѥ̾ϤƤ.");
	val = msgGetInput(NULL, KM("Enter the name you use to login to your provider."));
	SAFE_STRCPY(authname, val);
	dialog_clear_norefresh();
	KANJI_MESSAGE("ץХΥ˻ѤѥɤϤƤ.");
	val = msgGetInput(NULL, KM("Enter the password you use to login to your provider."));
	SAFE_STRCPY(authkey, val);
	dialog_clear_norefresh();
	KANJI_MESSAGE("ץХΥѤֹϤƤ.");
	val = msgGetInput(NULL, KM("Enter the your provider's login phone number."));
	SAFE_STRCPY(phone, val);
	dialog_clear_norefresh();
	KANJI_MESSAGE("òϥȡ򥵥ݡȤƤޤ?");
	pulse = dialog_yesno("", KM("Does your telephone line support tone dialing?"), -1, -1);
    }
    fprintf(fp, "\ninstall:\n");
    fprintf(fp, " set speed %s\n", speed);
    fprintf(fp, " set device %s\n", devp->devname);
    fprintf(fp, " set ifaddr %s %s\n", myaddr, provider);
    fprintf(fp, " set timeout 0\n");
    fprintf(fp, " enable dns\n");
    fprintf(fp, " set log local phase\n");
    if(authname[0] != '\0'){
	fprintf(fp, " set dial \"ABORT BUSY ABORT NO\\\\sCARRIER TIMEOUT 5 \\\"\\\" AT OK-AT-OK ATE1Q0 OK \\\\dATD%c\\\\T TIMEOUT 40 CONNECT\"\n", pulse ? 'P' : 'T');
	fprintf(fp, " set login\n");
	fprintf(fp, " set authname %s\n", authname);
	fprintf(fp, " set authkey %s\n", authkey);
	fprintf(fp, " set phone %s\n", phone);
    }
    if (fchmod(fileno(fp), 0600) != 0) {
	KANJI_MESSAGE("ٹ: /etc/ppp/ppp.conf Υѡߥ˼Ԥޤ!");
	msgConfirm(KM("Warning: Failed to fix permissions on /etc/ppp/ppp.conf !"));
    }
    fclose(fp);

    /* Make the ppp config persistent */
    variable_set2(VAR_PPP_ENABLE, "YES", 0);
    variable_set2(VAR_PPP_PROFILE, "install", 0);

    if (!Fake && !file_readable("/dev/tun0") && mknod("/dev/tun0", 0600 | S_IFCHR, makedev(52, 0))) {
	KANJI_MESSAGE("ٹ:  ǥХ /dev/tun0 ޤ. PPP ưޤ!");
	msgConfirm(KM("Warning:  No /dev/tun0 device.  PPP will not work!"));
	restorescr(w);
	return 0;
    }

    if (isDebug())
	msgDebug("About to start PPP on device %s @ %s baud.  Provider = %s\n", devp->devname, speed, provider);

    if (!Fake && !(pid = fork())) {
	int i, fd;
	struct termios foo;
	extern int login_tty(int);

	for (i = getdtablesize(); i >= 0; i--)
	    close(i);

	/* We're going over to VTY2 */
	fd = open("/dev/ttyv2", O_RDWR);
	ioctl(0, TIOCSCTTY, &fd);
	dup2(0, 1);
	dup2(0, 2);
	DebugFD = 2;
	if (login_tty(fd) == -1)
	    msgDebug("ppp: Can't set the controlling terminal.\n");
	signal(SIGTTOU, SIG_IGN);
	if (tcgetattr(fd, &foo) != -1) {
	    foo.c_cc[VERASE] = '\010';
	    if (tcsetattr(fd, TCSANOW, &foo) == -1)
		msgDebug("ppp: Unable to set the erase character.\n");
	}
	else
	    msgDebug("ppp: Unable to get the terminal attributes!\n");
	execlp("ppp", "ppp", "install", (char *)NULL);
	msgDebug("PPP process failed to exec!\n");
	exit(1);
    }
    else {
	dialog_clear_norefresh();
#ifdef PC98
	KANJI_MESSAGE(
	": VTY3  PPP ޥɤưޤ (GRPH-F3 ǤĤȤ\n"
	"ޥɤԤʤȤǤޤ. GRPH-F1 Ǹޤ).\n"
	"PAP  CHAP Ȥʤñ \"dial\", Ϥޤ, ¾ξ\n"
        " \"term\" ȥפ, üߥ졼ư, \n"
	"ǥƥӥץХ˥ǤϤǤ. ö\n"
	"³Ω, Υ꡼äƥ꥿ǤäƲ.\n\n"
	"³ΩޤǤФ [RETRUN] 򲡤ʤ!");
	msgConfirm(KM("NOTICE: The PPP command is now started on VTY3 (type GRPH-F3 to\n"
           "interact with it, GRPH-F1 to switch back here). If you are using\n"
           "a PAP or CHAP login simply enter \"dial\", otherwise you'll need\n"
           "to use the \"term\" command which starts a terminal emulator\n"
           "which you can use to talk to your modem and dial the service\n"
           "provider.  Once you're connected, come back to this screen and\n"
           "press return.\n\n"
           "DO NOT PRESS [RETURN] HERE UNTIL THE CONNECTION IS FULLY\n"
           "ESTABLISHED!"));
#else
	KANJI_MESSAGE(
	": VTY3  PPP ޥɤưޤ (ALT-F3 ǤĤȤΥ\n"
	"ޥɤԤʤȤǤޤ. ALT-F1 Ǹޤ).\n"
	"PAP  CHAP Ȥʤñ \"dial\", Ϥޤ, ¾ξ\n"
        " \"term\" ȥפ, üߥ졼ư, \n"
	"ǥƥӥץХ˥ǤϤǤ. ö\n"
	"³Ω, Υ꡼äƥ꥿ǤäƲ.\n\n"
	"³ΩޤǤФ [ENTER] 򲡤ʤ!");
	msgConfirm(KM("NOTICE: The PPP command is now started on VTY3 (type ALT-F3 to\n"
           "interact with it, ALT-F1 to switch back here). If you are using\n"
           "a PAP or CHAP login simply enter \"dial\", otherwise you'll need\n"
           "to use the \"term\" command which starts a terminal emulator\n"
           "which you can use to talk to your modem and dial the service\n"
           "provider.  Once you're connected, come back to this screen and\n"
           "press return.\n\n"
           "DO NOT PRESS [ENTER] HERE UNTIL THE CONNECTION IS FULLY\n"
           "ESTABLISHED!"));
#endif
    }
    restorescr(w);
    return pid;
}

#ifdef DHCP_SUPPORT
/* Start WIDE-DHCP client */
static pid_t
startDHCP(Device *devp)
{
    DevInfo * di = (DevInfo *)devp->private;
    FILE *fp;
    char dhcp[255];
    int i;

    /* These are needed to make DHCP client work */
    Mkdir("/var/db");
    Mkdir("/var/run");
    symlink("/stand/etc/services", "/etc/services");

    msgNotify("dhcpc %s %s", di->extras, devp->name);
    i = vsystem("dhcpc %s %s", di->extras, devp->name);
    if (i) {
        msgDebug("dhcp: DHCP client failed to exec!\n");
        return 0;
    }
    sleep(2);
    snprintf(dhcp, sizeof(dhcp), "/var/run/dhcpc.%s.pid", devp->name);
    fp = fopen(dhcp, "r");
    if (!fp) {
        KANJI_MESSAGE("%s 򥪡ץǤޤ!  εǽϻǽǤ");
        msgConfirm(KM("Couldn't open %s file!  This isn't going to work"), dhcp);
        return 0;
    }

    if (!fgets(dhcp, sizeof(dhcp), fp)) {
        msgDebug("dhcp: Unable to get the process id\n");
        return 0;
    }
    fclose(fp);
    return (pid_t)atoi(dhcp);
}
#endif
