/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_get-products-image.c
 * @brief implement GET /products/$HASH/image
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include "taler-merchant-httpd_get-products-image.h"
#include "taler-merchant-httpd_helper.h"
#include <taler/taler_error_codes.h>


MHD_RESULT
TMH_get_products_image (const struct TMH_RequestHandler *rh,
                        struct MHD_Connection *connection,
                        struct TMH_HandlerContext *hc)
{
  struct TMH_MerchantInstance *mi = hc->instance;
  const char *image_hash = hc->infix;
  char *image = NULL;
  enum GNUNET_DB_QueryStatus qs;
  (void) rh;

  {
    /* Just simple check if the string is what we really expect */
    size_t ih_len = strlen (image_hash);

    if ( (sizeof (struct GNUNET_ShortHashCode) * 2) != ih_len)
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "image_hash");

    for (size_t i = 0; i < ih_len; i++)
      if (! isxdigit ((unsigned char) image_hash[i]))
        return TALER_MHD_reply_with_error (connection,
                                           MHD_HTTP_BAD_REQUEST,
                                           TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                           "image_hash");
  }

  qs = TMH_db->lookup_product_image_by_hash (TMH_db->cls,
                                             mi->settings.id,
                                             image_hash,
                                             &image);
  if (0 > qs)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       "lookup_product_image_by_hash");
  }
  if ( (0 == qs) ||
       (NULL == image) )
  {
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_NOT_FOUND,
                                       TALER_EC_MERCHANT_GENERIC_PRODUCT_UNKNOWN,
                                       image_hash);
  }

  {
    MHD_RESULT ret;

    ret = TALER_MHD_REPLY_JSON_PACK (connection,
                                     MHD_HTTP_OK,
                                     GNUNET_JSON_pack_string ("image",
                                                              image));
    GNUNET_free (image);
    return ret;
  }
}
