/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-patch-pot.c
 * @brief implementation of PATCH /private/pots/$POT_ID
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-patch-pot-ID.h"
#include <taler/taler_json_lib.h>


MHD_RESULT
TMH_private_patch_pot (const struct TMH_RequestHandler *rh,
                       struct MHD_Connection *connection,
                       struct TMH_HandlerContext *hc)
{
  const char *pot_id_str = hc->infix;
  unsigned long long pot_id;
  const char *pot_name;
  const char *description;
  size_t expected_pot_total_len;
  struct TALER_Amount *expected_pot_totals;
  bool no_expected_total;
  size_t new_pot_total_len;
  struct TALER_Amount *new_pot_totals;
  bool no_new_total;
  enum GNUNET_DB_QueryStatus qs;
  bool conflict_total;
  bool conflict_name;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("pot_name",
                             &pot_name),
    GNUNET_JSON_spec_string ("description",
                             &description),
    GNUNET_JSON_spec_mark_optional (
      TALER_JSON_spec_amount_any_array ("expected_pot_total",
                                        &expected_pot_total_len,
                                        &expected_pot_totals),
      &no_expected_total),
    GNUNET_JSON_spec_mark_optional (
      TALER_JSON_spec_amount_any_array ("new_pot_totals",
                                        &new_pot_total_len,
                                        &new_pot_totals),
      &no_new_total),
    GNUNET_JSON_spec_end ()
  };

  (void) rh;
  {
    char dummy;

    if (1 != sscanf (pot_id_str,
                     "%llu%c",
                     &pot_id,
                     &dummy))
    {
      GNUNET_break_op (0);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "pot_id");
    }
  }

  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
    {
      GNUNET_break_op (0);
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    }
  }

  qs = TMH_db->update_money_pot (TMH_db->cls,
                                 hc->instance->settings.id,
                                 pot_id,
                                 pot_name,
                                 description,
                                 expected_pot_total_len,
                                 no_expected_total
                                 ? NULL
                                 : expected_pot_totals,
                                 new_pot_total_len,
                                 no_new_total
                                 ? NULL
                                 : new_pot_totals,
                                 &conflict_total,
                                 &conflict_name);
  GNUNET_JSON_parse_free (spec);
  if (qs < 0)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "update_money_pot");
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    return TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_MERCHANT_GENERIC_MONEY_POT_UNKNOWN,
      pot_id_str);
  }
  if (conflict_total)
  {
    /* Pot total mismatch - expected_pot_total didn't match current value */
    return TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_MONEY_POT_CONFLICTING_TOTAL,
      NULL);
  }
  if (conflict_name)
  {
    /* Pot name conflict - name exists */
    return TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_MONEY_POT_CONFLICTING_NAME,
      pot_name);
  }

  return TALER_MHD_reply_static (connection,
                                 MHD_HTTP_NO_CONTENT,
                                 NULL,
                                 NULL,
                                 0);
}
