// GoToSocial
// Copyright (C) GoToSocial Authors admin@gotosocial.org
// SPDX-License-Identifier: AGPL-3.0-or-later
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package admin

import (
	"errors"
	"fmt"
	"net/http"

	apimodel "code.superseriousbusiness.org/gotosocial/internal/api/model"
	apiutil "code.superseriousbusiness.org/gotosocial/internal/api/util"
	"code.superseriousbusiness.org/gotosocial/internal/gtserror"
	"github.com/gin-gonic/gin"
)

// DomainLimitUpdatePUTHandler swagger:operation PUT /api/v1/admin/domain_limits/{id} domainLimitUpdate
//
// Update a domain limit.
//
//	---
//	tags:
//	- admin
//
//	consumes:
//	- multipart/form-data
//	- application/json
//
//	produces:
//	- application/json
//
//	parameters:
//	-
//		name: id
//		type: string
//		description: The id of the domain limit.
//		in: path
//		required: true
//	-
//		name: media_policy
//		in: formData
//		description: |-
//			Policy to apply to media files originating from the limited domain.
//			No action = default (not limited).
//			Mark sensitive = mark all media from the limited domain as sensitive.
//			Reject = do not download media from the limited domain. Serve a link to the media instead.
//			Omit to keep current value.
//		type: string
//		enum:
//			- no_action
//			- mark_sensitive
//			- reject
//	-
//		name: follows_policy
//		in: formData
//		description: |-
//			Policy to apply to follow (requests) originating from the limited domain.
//			No action = default (not limited).
//			Manual approval = require manual approval for all follows from limited domain.
//			Reject non mutual = automatically reject follows from the limited domain when they're not follow-backs.
//			Reject all = automatically reject all follows from the limited domain.
//			Omit to keep current value.
//		type: string
//		enum:
//			- no_action
//			- manual_approval
//			- reject_non_mutual
//			- reject_all
//	-
//		name: statuses_policy
//		in: formData
//		description: |-
//			Policy to apply to statuses of non-followed accounts on the limited domain.
//			No action = default (not limited).
//			Filter warn = trigger a warn filter pointing to this domain limit.
//			Filter hide = trigger a hide filter pointing to this domain limit.
//			Omit to keep current value.
//		type: string
//		enum:
//			- no_action
//			- filter_warn
//			- filter_hide
//	-
//		name: accounts_policy
//		in: formData
//		description: |-
//			Policy to apply to non-followed accounts on the limited domain.
//			No action = default (not limited).
//			Mute = mute all non-followed accounts on the limited domain.
//			Omit to keep current value.
//		type: string
//		enum:
//			- no_action
//			- mute
//		default: no_action
//	-
//		name: content_warning
//		in: formData
//		description: Content warning to prepend to posts from accounts on this instance. Omit to keep current value.
//		type: string
//	-
//		name: public_comment
//		in: formData
//		description: >-
//			Public comment about this domain limit.
//			This will be displayed alongside the domain limit if you choose to share limits.
//			Omit to keep current value.
//		type: string
//	-
//		name: private_comment
//		in: formData
//		description: >-
//			Private comment about this domain limit. Will only be shown to other admins, so this
//			is a useful way of internally keeping track of why a certain domain ended up limited.
//			Omit to keep current value.
//		type: string
//
//	security:
//	- OAuth2 Bearer:
//		- admin:write:domain_limits
//
//	responses:
//		'200':
//			description: The updated domain limit.
//			schema:
//				"$ref": "#/definitions/domainLimit"
//		'400':
//			description: bad request
//		'401':
//			description: unauthorized
//		'403':
//			description: forbidden
//		'404':
//			description: not found
//		'406':
//			description: not acceptable
//		'500':
//			description: internal server error
func (m *Module) DomainLimitPUTHandler(c *gin.Context) {
	authed, errWithCode := apiutil.TokenAuth(c,
		true, true, true, true,
		apiutil.ScopeAdminWriteDomainLimits,
	)
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	if !*authed.User.Admin {
		err := fmt.Errorf("user %s not an admin", authed.User.ID)
		apiutil.ErrorHandler(c, gtserror.NewErrorForbidden(err, err.Error()), m.processor.InstanceGetV1)
		return
	}

	if authed.Account.IsMoving() {
		apiutil.ForbiddenAfterMove(c)
		return
	}

	if _, errWithCode := apiutil.NegotiateAccept(c, apiutil.JSONAcceptHeaders...); errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	id, errWithCode := apiutil.ParseID(c.Param(apiutil.IDKey))
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	form := new(apimodel.DomainLimitRequest)
	if err := c.ShouldBind(form); err != nil {
		apiutil.ErrorHandler(c, gtserror.NewErrorBadRequest(err, err.Error()), m.processor.InstanceGetV1)
		return
	}

	// Ensure something is set to update.
	if form.MediaPolicy == nil &&
		form.FollowsPolicy == nil &&
		form.StatusesPolicy == nil &&
		form.AccountsPolicy == nil &&
		form.ContentWarning == nil &&
		form.PublicComment == nil &&
		form.PrivateComment == nil {
		const text = "nothing to update; at least one of media_policy, follows_policy, statuses_policy, accounts_policy, content_warning, public_comment, or private_comment must be set"
		errWithCode := gtserror.NewErrorBadRequest(errors.New(text), text)
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	domainLimit, errWithCode := m.processor.Admin().DomainLimitUpdate(
		c.Request.Context(),
		id,
		form.MediaPolicy,
		form.FollowsPolicy,
		form.StatusesPolicy,
		form.AccountsPolicy,
		form.ContentWarning,
		form.PublicComment,
		form.PrivateComment,
	)
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	apiutil.JSON(c, http.StatusOK, domainLimit)
}
