// GoToSocial
// Copyright (C) GoToSocial Authors admin@gotosocial.org
// SPDX-License-Identifier: AGPL-3.0-or-later
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package admin

import (
	"errors"
	"fmt"
	"net/http"
	"net/url"

	apimodel "code.superseriousbusiness.org/gotosocial/internal/api/model"
	apiutil "code.superseriousbusiness.org/gotosocial/internal/api/util"
	"code.superseriousbusiness.org/gotosocial/internal/gtserror"
	"code.superseriousbusiness.org/gotosocial/internal/gtsmodel"
	"code.superseriousbusiness.org/gotosocial/internal/util"
	"github.com/gin-gonic/gin"
)

// DomainPermissionSubscriptionPATCHHandler swagger:operation PATCH /api/v1/admin/domain_permission_subscriptions/${id} domainPermissionSubscriptionUpdate
//
// Update a domain permission subscription with the given parameters.
//
//	---
//	tags:
//	- admin
//
//	consumes:
//	- multipart/form-data
//	- application/json
//
//	produces:
//	- application/json
//
//	parameters:
//	-
//		name: id
//		required: true
//		in: path
//		description: ID of the domain permission subscription.
//		type: string
//	-
//		name: priority
//		in: formData
//		description: >-
//			Priority of this subscription compared to others of the same permission type.
//			0-255 (higher = higher priority). Higher priority subscriptions will overwrite
//			permissions generated by lower priority subscriptions. When two subscriptions
//			have the same `priority` value, priority is indeterminate, so it's recommended
//			to always set this value manually.
//		type: number
//		minimum: 0
//		maximum: 255
//	-
//		name: title
//		in: formData
//		description: Optional title for this subscription.
//		type: string
//	-
//		name: uri
//		in: formData
//		description: URI to call in order to fetch the permissions list.
//		type: string
//	-
//		name: as_draft
//		in: formData
//		description: >-
//			If true, domain permissions arising from this subscription will be
//			created as drafts that must be approved by a moderator to take effect.
//			If false, domain permissions from this subscription will come into force immediately.
//			Defaults to "true".
//		type: boolean
//		default: true
//	-
//		name: adopt_orphans
//		in: formData
//		description: >-
//			If true, this domain permission subscription will "adopt" domain permissions
//			which already exist on the instance, and which meet the following conditions:
//			1) they have no subscription ID (ie., they're "orphaned") and 2) they are present
//			in the subscribed list. Such orphaned domain permissions will be given this
//			subscription's subscription ID value and be managed by this subscription.
//		type: boolean
//		default: false
//	-
//		name: remove_retracted
//		in: formData
//		description: >-
//			If true, then when a list is processed, if the list does *not* contain entries that
//			it *did* contain previously, ie., retracted entries, then domain permissions
//			corresponding to those entries will be removed. If false, they will just be orphaned instead.
//		type: boolean
//		default: true
//	-
//		name: content_type
//		in: formData
//		description: >-
//			MIME content type to use when parsing the permissions list.
//			One of "text/plain", "text/csv", and "application/json".
//		type: string
//	-
//		name: fetch_username
//		in: formData
//		description: >-
//			Optional basic auth username to provide when fetching given uri.
//			If set, will be transmitted along with `fetch_password` when doing the fetch.
//		type: string
//	-
//		name: fetch_password
//		in: formData
//		description: >-
//			Optional basic auth password to provide when fetching given uri.
//			If set, will be transmitted along with `fetch_username` when doing the fetch.
//		type: string
//
//	security:
//	- OAuth2 Bearer:
//		- admin:write
//
//	responses:
//		'200':
//			description: The updated domain permission subscription.
//			schema:
//				"$ref": "#/definitions/domainPermissionSubscription"
//		'400':
//			schema:
//				"$ref": "#/definitions/error"
//			description: bad request
//		'401':
//			schema:
//				"$ref": "#/definitions/error"
//			description: unauthorized
//		'403':
//			schema:
//				"$ref": "#/definitions/error"
//			description: forbidden
//		'406':
//			schema:
//				"$ref": "#/definitions/error"
//			description: not acceptable
//		'409':
//			schema:
//				"$ref": "#/definitions/error"
//			description: conflict
//		'500':
//			schema:
//				"$ref": "#/definitions/error"
//			description: internal server error
func (m *Module) DomainPermissionSubscriptionPATCHHandler(c *gin.Context) {
	authed, errWithCode := apiutil.TokenAuth(c,
		true, true, true, true,
		apiutil.ScopeAdminWrite,
	)
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	if !*authed.User.Admin {
		err := fmt.Errorf("user %s not an admin", authed.User.ID)
		apiutil.ErrorHandler(c, gtserror.NewErrorForbidden(err, err.Error()), m.processor.InstanceGetV1)
		return
	}

	if authed.Account.IsMoving() {
		apiutil.ForbiddenAfterMove(c)
		return
	}

	if _, errWithCode := apiutil.NegotiateAccept(c, apiutil.JSONAcceptHeaders...); errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	id, errWithCode := apiutil.ParseID(c.Param(apiutil.IDKey))
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	// Parse + validate form.
	form := new(apimodel.DomainPermissionSubscriptionRequest)
	if err := c.ShouldBind(form); err != nil {
		apiutil.ErrorHandler(c, gtserror.NewErrorBadRequest(err, err.Error()), m.processor.InstanceGetV1)
		return
	}

	// Normalize priority if set.
	var priority *uint8
	if form.Priority != nil {
		prioInt := *form.Priority
		if prioInt < 0 || prioInt > 255 {
			const errText = "priority must be a number in the range 0 to 255"
			errWithCode := gtserror.NewErrorBadRequest(errors.New(errText), errText)
			apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
			return
		}

		priority = util.Ptr(uint8(prioInt)) // #nosec G115 -- Just validated.
	}

	// Validate URI if set.
	var uriStr *string
	if form.URI != nil {
		uri, err := url.Parse(*form.URI)
		if err != nil {
			err := fmt.Errorf("invalid uri provided: %w", err)
			errWithCode := gtserror.NewErrorBadRequest(err, err.Error())
			apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
			return
		}

		// Normalize URI by converting back to string.
		uriStr = util.Ptr(uri.String())
	}

	// Validate content type if set.
	var contentType *gtsmodel.DomainPermSubContentType
	if form.ContentType != nil {
		ct, errWithCode := parseDomainPermSubContentType(*form.ContentType)
		if errWithCode != nil {
			apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
			return
		}

		contentType = &ct
	}

	// Make sure at least one field is set,
	// otherwise we're trying to update nothing.
	if priority == nil &&
		form.Title == nil &&
		uriStr == nil &&
		contentType == nil &&
		form.AsDraft == nil &&
		form.AdoptOrphans == nil &&
		form.RemoveRetracted == nil &&
		form.FetchUsername == nil &&
		form.FetchPassword == nil {
		const errText = "no updateable fields set on request"
		errWithCode := gtserror.NewErrorBadRequest(errors.New(errText), errText)
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	permSub, errWithCode := m.processor.Admin().DomainPermissionSubscriptionUpdate(
		c.Request.Context(),
		id,
		priority,
		form.Title,
		uriStr,
		contentType,
		form.AsDraft,
		form.AdoptOrphans,
		form.RemoveRetracted,
		form.FetchUsername,
		form.FetchPassword,
	)
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	apiutil.JSON(c, http.StatusOK, permSub)
}
