{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The ATK interface provided by UI components
-- which occupy a physical area on the screen.
-- which the user can activate\/interact with.
-- 
-- t'GI.Atk.Interfaces.Component.Component' should be implemented by most if not all UI elements
-- with an actual on-screen presence, i.e. components which can be
-- said to have a screen-coordinate bounding box.  Virtually all
-- widgets will need to have t'GI.Atk.Interfaces.Component.Component' implementations provided
-- for their corresponding t'GI.Atk.Objects.Object.Object' class.  In short, only UI
-- elements which are *not* GUI elements will omit this ATK interface.
-- 
-- A possible exception might be textual information with a
-- transparent background, in which case text glyph bounding box
-- information is provided by t'GI.Atk.Interfaces.Text.Text'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Atk.Interfaces.Component
    ( 

-- * Exported types
    Component(..)                           ,
    IsComponent                             ,
    toComponent                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [contains]("GI.Atk.Interfaces.Component#g:method:contains"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Atk.Interfaces.Component#g:method:grabFocus"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refAccessibleAtPoint]("GI.Atk.Interfaces.Component#g:method:refAccessibleAtPoint"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeFocusHandler]("GI.Atk.Interfaces.Component#g:method:removeFocusHandler"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [scrollTo]("GI.Atk.Interfaces.Component#g:method:scrollTo"), [scrollToPoint]("GI.Atk.Interfaces.Component#g:method:scrollToPoint"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAlpha]("GI.Atk.Interfaces.Component#g:method:getAlpha"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getExtents]("GI.Atk.Interfaces.Component#g:method:getExtents"), [getLayer]("GI.Atk.Interfaces.Component#g:method:getLayer"), [getMdiZorder]("GI.Atk.Interfaces.Component#g:method:getMdiZorder"), [getPosition]("GI.Atk.Interfaces.Component#g:method:getPosition"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getSize]("GI.Atk.Interfaces.Component#g:method:getSize").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setExtents]("GI.Atk.Interfaces.Component#g:method:setExtents"), [setPosition]("GI.Atk.Interfaces.Component#g:method:setPosition"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setSize]("GI.Atk.Interfaces.Component#g:method:setSize").

#if defined(ENABLE_OVERLOADING)
    ResolveComponentMethod                  ,
#endif

-- ** contains #method:contains#

#if defined(ENABLE_OVERLOADING)
    ComponentContainsMethodInfo             ,
#endif
    componentContains                       ,


-- ** getAlpha #method:getAlpha#

#if defined(ENABLE_OVERLOADING)
    ComponentGetAlphaMethodInfo             ,
#endif
    componentGetAlpha                       ,


-- ** getExtents #method:getExtents#

#if defined(ENABLE_OVERLOADING)
    ComponentGetExtentsMethodInfo           ,
#endif
    componentGetExtents                     ,


-- ** getLayer #method:getLayer#

#if defined(ENABLE_OVERLOADING)
    ComponentGetLayerMethodInfo             ,
#endif
    componentGetLayer                       ,


-- ** getMdiZorder #method:getMdiZorder#

#if defined(ENABLE_OVERLOADING)
    ComponentGetMdiZorderMethodInfo         ,
#endif
    componentGetMdiZorder                   ,


-- ** getPosition #method:getPosition#

#if defined(ENABLE_OVERLOADING)
    ComponentGetPositionMethodInfo          ,
#endif
    componentGetPosition                    ,


-- ** getSize #method:getSize#

#if defined(ENABLE_OVERLOADING)
    ComponentGetSizeMethodInfo              ,
#endif
    componentGetSize                        ,


-- ** grabFocus #method:grabFocus#

#if defined(ENABLE_OVERLOADING)
    ComponentGrabFocusMethodInfo            ,
#endif
    componentGrabFocus                      ,


-- ** refAccessibleAtPoint #method:refAccessibleAtPoint#

#if defined(ENABLE_OVERLOADING)
    ComponentRefAccessibleAtPointMethodInfo ,
#endif
    componentRefAccessibleAtPoint           ,


-- ** removeFocusHandler #method:removeFocusHandler#

#if defined(ENABLE_OVERLOADING)
    ComponentRemoveFocusHandlerMethodInfo   ,
#endif
    componentRemoveFocusHandler             ,


-- ** scrollTo #method:scrollTo#

#if defined(ENABLE_OVERLOADING)
    ComponentScrollToMethodInfo             ,
#endif
    componentScrollTo                       ,


-- ** scrollToPoint #method:scrollToPoint#

#if defined(ENABLE_OVERLOADING)
    ComponentScrollToPointMethodInfo        ,
#endif
    componentScrollToPoint                  ,


-- ** setExtents #method:setExtents#

#if defined(ENABLE_OVERLOADING)
    ComponentSetExtentsMethodInfo           ,
#endif
    componentSetExtents                     ,


-- ** setPosition #method:setPosition#

#if defined(ENABLE_OVERLOADING)
    ComponentSetPositionMethodInfo          ,
#endif
    componentSetPosition                    ,


-- ** setSize #method:setSize#

#if defined(ENABLE_OVERLOADING)
    ComponentSetSizeMethodInfo              ,
#endif
    componentSetSize                        ,




 -- * Signals


-- ** boundsChanged #signal:boundsChanged#

    ComponentBoundsChangedCallback          ,
#if defined(ENABLE_OVERLOADING)
    ComponentBoundsChangedSignalInfo        ,
#endif
    afterComponentBoundsChanged             ,
    onComponentBoundsChanged                ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.Atk.Enums as Atk.Enums
import {-# SOURCE #-} qualified GI.Atk.Objects.Object as Atk.Object
import {-# SOURCE #-} qualified GI.Atk.Objects.Relation as Atk.Relation
import {-# SOURCE #-} qualified GI.Atk.Objects.RelationSet as Atk.RelationSet
import {-# SOURCE #-} qualified GI.Atk.Objects.StateSet as Atk.StateSet
import {-# SOURCE #-} qualified GI.Atk.Structs.PropertyValues as Atk.PropertyValues
import {-# SOURCE #-} qualified GI.Atk.Structs.Rectangle as Atk.Rectangle
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.ValueArray as GObject.ValueArray

#else
import {-# SOURCE #-} qualified GI.Atk.Enums as Atk.Enums
import {-# SOURCE #-} qualified GI.Atk.Objects.Object as Atk.Object
import {-# SOURCE #-} qualified GI.Atk.Structs.Rectangle as Atk.Rectangle
import qualified GI.GObject.Objects.Object as GObject.Object

#endif

-- interface Component 
-- | Memory-managed wrapper type.
newtype Component = Component (SP.ManagedPtr Component)
    deriving (Eq)

instance SP.ManagedPtrNewtype Component where
    toManagedPtr (Component p) = p

foreign import ccall "atk_component_get_type"
    c_atk_component_get_type :: IO B.Types.GType

instance B.Types.TypedObject Component where
    glibType = c_atk_component_get_type

instance B.Types.GObject Component

-- | Type class for types which can be safely cast to t'Component', for instance with `toComponent`.
class (SP.GObject o, O.IsDescendantOf Component o) => IsComponent o
instance (SP.GObject o, O.IsDescendantOf Component o) => IsComponent o

instance O.HasParentTypes Component
type instance O.ParentTypes Component = '[GObject.Object.Object]

-- | Cast to t'Component', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toComponent :: (MIO.MonadIO m, IsComponent o) => o -> m Component
toComponent = MIO.liftIO . B.ManagedPtr.unsafeCastTo Component

-- | Convert t'Component' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Component) where
    gvalueGType_ = c_atk_component_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Component)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Component)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Component ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Component
type instance O.AttributeList Component = ComponentAttributeList
type ComponentAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveComponentMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveComponentMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveComponentMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveComponentMethod "contains" o = ComponentContainsMethodInfo
    ResolveComponentMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveComponentMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveComponentMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveComponentMethod "grabFocus" o = ComponentGrabFocusMethodInfo
    ResolveComponentMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveComponentMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveComponentMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveComponentMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveComponentMethod "refAccessibleAtPoint" o = ComponentRefAccessibleAtPointMethodInfo
    ResolveComponentMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveComponentMethod "removeFocusHandler" o = ComponentRemoveFocusHandlerMethodInfo
    ResolveComponentMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveComponentMethod "scrollTo" o = ComponentScrollToMethodInfo
    ResolveComponentMethod "scrollToPoint" o = ComponentScrollToPointMethodInfo
    ResolveComponentMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveComponentMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveComponentMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveComponentMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveComponentMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveComponentMethod "getAlpha" o = ComponentGetAlphaMethodInfo
    ResolveComponentMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveComponentMethod "getExtents" o = ComponentGetExtentsMethodInfo
    ResolveComponentMethod "getLayer" o = ComponentGetLayerMethodInfo
    ResolveComponentMethod "getMdiZorder" o = ComponentGetMdiZorderMethodInfo
    ResolveComponentMethod "getPosition" o = ComponentGetPositionMethodInfo
    ResolveComponentMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveComponentMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveComponentMethod "getSize" o = ComponentGetSizeMethodInfo
    ResolveComponentMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveComponentMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveComponentMethod "setExtents" o = ComponentSetExtentsMethodInfo
    ResolveComponentMethod "setPosition" o = ComponentSetPositionMethodInfo
    ResolveComponentMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveComponentMethod "setSize" o = ComponentSetSizeMethodInfo
    ResolveComponentMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveComponentMethod t Component, O.OverloadedMethod info Component p) => OL.IsLabel t (Component -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveComponentMethod t Component, O.OverloadedMethod info Component p, R.HasField t Component p) => R.HasField t Component p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveComponentMethod t Component, O.OverloadedMethodInfo info Component) => OL.IsLabel t (O.MethodProxy info Component) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Component::contains
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the components top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_contains" atk_component_contains :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO CInt

-- | Checks whether the specified point is within the extent of the /@component@/.
-- 
-- Toolkit implementor note: ATK provides a default implementation for
-- this virtual method. In general there are little reason to
-- re-implement it.
componentContains ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: the t'GI.Atk.Interfaces.Component.Component'
    -> Int32
    -- ^ /@x@/: x coordinate
    -> Int32
    -- ^ /@y@/: y coordinate
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the components top level window
    -> m Bool
    -- ^ __Returns:__ 'P.True' or 'P.False' indicating whether the specified point is within
    -- the extent of the /@component@/ or not
componentContains component x y coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let coordType' = (fromIntegral . fromEnum) coordType
    result <- atk_component_contains component' x y coordType'
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentContainsMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentContainsMethodInfo a signature where
    overloadedMethod = componentContains

instance O.OverloadedMethodInfo ComponentContainsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentContains",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentContains"
        })


#endif

-- method Component::get_alpha
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_alpha" atk_component_get_alpha :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    IO CDouble

-- | Returns the alpha value (i.e. the opacity) for this
-- /@component@/, on a scale from 0 (fully transparent) to 1.0
-- (fully opaque).
-- 
-- /Since: 1.12/
componentGetAlpha ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> m Double
    -- ^ __Returns:__ An alpha value from 0 to 1.0, inclusive.
componentGetAlpha component = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    result <- atk_component_get_alpha component'
    let result' = realToFrac result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentGetAlphaMethodInfo
instance (signature ~ (m Double), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetAlphaMethodInfo a signature where
    overloadedMethod = componentGetAlpha

instance O.OverloadedMethodInfo ComponentGetAlphaMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetAlpha",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetAlpha"
        })


#endif

-- method Component::get_extents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put x coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put y coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put width"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put height"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the components top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_extents" atk_component_get_extents :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO ()

-- | Gets the rectangle which gives the extent of the /@component@/.
-- 
-- If the extent can not be obtained (e.g. a non-embedded plug or missing
-- support), all of x, y, width, height are set to -1.
componentGetExtents ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the components top level window
    -> m ((Int32, Int32, Int32, Int32))
componentGetExtents component coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    let coordType' = (fromIntegral . fromEnum) coordType
    atk_component_get_extents component' x y width height coordType'
    x' <- peek x
    y' <- peek y
    width' <- peek width
    height' <- peek height
    touchManagedPtr component
    freeMem x
    freeMem y
    freeMem width
    freeMem height
    return (x', y', width', height')

#if defined(ENABLE_OVERLOADING)
data ComponentGetExtentsMethodInfo
instance (signature ~ (Atk.Enums.CoordType -> m ((Int32, Int32, Int32, Int32))), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetExtentsMethodInfo a signature where
    overloadedMethod = componentGetExtents

instance O.OverloadedMethodInfo ComponentGetExtentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetExtents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetExtents"
        })


#endif

-- method Component::get_layer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Atk" , name = "Layer" })
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_layer" atk_component_get_layer :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    IO CUInt

-- | Gets the layer of the component.
componentGetLayer ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> m Atk.Enums.Layer
    -- ^ __Returns:__ an t'GI.Atk.Enums.Layer' which is the layer of the component
componentGetLayer component = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    result <- atk_component_get_layer component'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentGetLayerMethodInfo
instance (signature ~ (m Atk.Enums.Layer), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetLayerMethodInfo a signature where
    overloadedMethod = componentGetLayer

instance O.OverloadedMethodInfo ComponentGetLayerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetLayer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetLayer"
        })


#endif

-- method Component::get_mdi_zorder
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_mdi_zorder" atk_component_get_mdi_zorder :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    IO Int32

-- | Gets the zorder of the component. The value G_MININT will be returned
-- if the layer of the component is not ATK_LAYER_MDI or ATK_LAYER_WINDOW.
componentGetMdiZorder ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> m Int32
    -- ^ __Returns:__ a gint which is the zorder of the component, i.e. the depth at
    -- which the component is shown in relation to other components in the same
    -- container.
componentGetMdiZorder component = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    result <- atk_component_get_mdi_zorder component'
    touchManagedPtr component
    return result

#if defined(ENABLE_OVERLOADING)
data ComponentGetMdiZorderMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetMdiZorderMethodInfo a signature where
    overloadedMethod = componentGetMdiZorder

instance O.OverloadedMethodInfo ComponentGetMdiZorderMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetMdiZorder",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetMdiZorder"
        })


#endif

-- method Component::get_position
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put x coordinate position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put y coordinate position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the components top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_position" atk_component_get_position :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO ()

{-# DEPRECATED componentGetPosition ["Since 2.12. Use 'GI.Atk.Interfaces.Component.componentGetExtents' instead."] #-}
-- | Gets the position of /@component@/ in the form of
-- a point specifying /@component@/\'s top-left corner.
-- 
-- If the position can not be obtained (e.g. a non-embedded plug or missing
-- support), x and y are set to -1.
componentGetPosition ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the components top level window
    -> m ((Int32, Int32))
componentGetPosition component coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    let coordType' = (fromIntegral . fromEnum) coordType
    atk_component_get_position component' x y coordType'
    x' <- peek x
    y' <- peek y
    touchManagedPtr component
    freeMem x
    freeMem y
    return (x', y')

#if defined(ENABLE_OVERLOADING)
data ComponentGetPositionMethodInfo
instance (signature ~ (Atk.Enums.CoordType -> m ((Int32, Int32))), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetPositionMethodInfo a signature where
    overloadedMethod = componentGetPosition

instance O.OverloadedMethodInfo ComponentGetPositionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetPosition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetPosition"
        })


#endif

-- method Component::get_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put width of @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of #gint to put height of @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_get_size" atk_component_get_size :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    IO ()

{-# DEPRECATED componentGetSize ["Since 2.12. Use 'GI.Atk.Interfaces.Component.componentGetExtents' instead."] #-}
-- | Gets the size of the /@component@/ in terms of width and height.
-- 
-- If the size can not be obtained (e.g. a non-embedded plug or missing
-- support), width and height are set to -1.
componentGetSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> m ((Int32, Int32))
componentGetSize component = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    atk_component_get_size component' width height
    width' <- peek width
    height' <- peek height
    touchManagedPtr component
    freeMem width
    freeMem height
    return (width', height')

#if defined(ENABLE_OVERLOADING)
data ComponentGetSizeMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGetSizeMethodInfo a signature where
    overloadedMethod = componentGetSize

instance O.OverloadedMethodInfo ComponentGetSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGetSize"
        })


#endif

-- method Component::grab_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_grab_focus" atk_component_grab_focus :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    IO CInt

-- | Grabs focus for this /@component@/.
componentGrabFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, 'P.False' otherwise.
componentGrabFocus component = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    result <- atk_component_grab_focus component'
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentGrabFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentGrabFocusMethodInfo a signature where
    overloadedMethod = componentGrabFocus

instance O.OverloadedMethodInfo ComponentGrabFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentGrabFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentGrabFocus"
        })


#endif

-- method Component::ref_accessible_at_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the components top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Atk" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_ref_accessible_at_point" atk_component_ref_accessible_at_point :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO (Ptr Atk.Object.Object)

-- | Gets a reference to the accessible child, if one exists, at the
-- coordinate point specified by /@x@/ and /@y@/.
componentRefAccessibleAtPoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: the t'GI.Atk.Interfaces.Component.Component'
    -> Int32
    -- ^ /@x@/: x coordinate
    -> Int32
    -- ^ /@y@/: y coordinate
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the components top level window
    -> m (Maybe Atk.Object.Object)
    -- ^ __Returns:__ a reference to the accessible
    -- child, if one exists
componentRefAccessibleAtPoint component x y coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let coordType' = (fromIntegral . fromEnum) coordType
    result <- atk_component_ref_accessible_at_point component' x y coordType'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Atk.Object.Object) result'
        return result''
    touchManagedPtr component
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ComponentRefAccessibleAtPointMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> m (Maybe Atk.Object.Object)), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentRefAccessibleAtPointMethodInfo a signature where
    overloadedMethod = componentRefAccessibleAtPoint

instance O.OverloadedMethodInfo ComponentRefAccessibleAtPointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentRefAccessibleAtPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentRefAccessibleAtPoint"
        })


#endif

-- method Component::remove_focus_handler
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the #AtkComponent to remove the focus handler from"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "handler_id"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the handler id of the focus handler to be removed\nfrom @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_remove_focus_handler" atk_component_remove_focus_handler :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Word32 ->                               -- handler_id : TBasicType TUInt
    IO ()

{-# DEPRECATED componentRemoveFocusHandler ["(Since version 2.9.4)","If you need to track when an object gains or","lose the focus, use the [Object::stateChange](\"GI.Atk.Objects.Object#g:signal:stateChange\") \\\"focused\\\" notification instead."] #-}
-- | Remove the handler specified by /@handlerId@/ from the list of
-- functions to be executed when this object receives focus events
-- (in or out).
componentRemoveFocusHandler ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: the t'GI.Atk.Interfaces.Component.Component' to remove the focus handler from
    -> Word32
    -- ^ /@handlerId@/: the handler id of the focus handler to be removed
    -- from /@component@/
    -> m ()
componentRemoveFocusHandler component handlerId = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    atk_component_remove_focus_handler component' handlerId
    touchManagedPtr component
    return ()

#if defined(ENABLE_OVERLOADING)
data ComponentRemoveFocusHandlerMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentRemoveFocusHandlerMethodInfo a signature where
    overloadedMethod = componentRemoveFocusHandler

instance O.OverloadedMethodInfo ComponentRemoveFocusHandlerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentRemoveFocusHandler",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentRemoveFocusHandler"
        })


#endif

-- method Component::scroll_to
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "ScrollType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "specify where the object should be made visible."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_scroll_to" atk_component_scroll_to :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    CUInt ->                                -- type : TInterface (Name {namespace = "Atk", name = "ScrollType"})
    IO CInt

-- | Makes /@component@/ visible on the screen by scrolling all necessary parents.
-- 
-- Contrary to atk_component_set_position, this does not actually move
-- /@component@/ in its parent, this only makes the parents scroll so that the
-- object shows up on the screen, given its current position within the parents.
-- 
-- /Since: 2.30/
componentScrollTo ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Atk.Enums.ScrollType
    -- ^ /@type@/: specify where the object should be made visible.
    -> m Bool
    -- ^ __Returns:__ whether scrolling was successful.
componentScrollTo component type_ = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let type_' = (fromIntegral . fromEnum) type_
    result <- atk_component_scroll_to component' type_'
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentScrollToMethodInfo
instance (signature ~ (Atk.Enums.ScrollType -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentScrollToMethodInfo a signature where
    overloadedMethod = componentScrollTo

instance O.OverloadedMethodInfo ComponentScrollToMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentScrollTo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentScrollTo"
        })


#endif

-- method Component::scroll_to_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coords"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specify whether coordinates are relative to the screen or to the\nparent object."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x-position where to scroll to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y-position where to scroll to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_scroll_to_point" atk_component_scroll_to_point :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    CUInt ->                                -- coords : TInterface (Name {namespace = "Atk", name = "CoordType"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    IO CInt

-- | Move the top-left of /@component@/ to a given position of the screen by
-- scrolling all necessary parents.
-- 
-- /Since: 2.30/
componentScrollToPoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Atk.Enums.CoordType
    -- ^ /@coords@/: specify whether coordinates are relative to the screen or to the
    -- parent object.
    -> Int32
    -- ^ /@x@/: x-position where to scroll to
    -> Int32
    -- ^ /@y@/: y-position where to scroll to
    -> m Bool
    -- ^ __Returns:__ whether scrolling was successful.
componentScrollToPoint component coords x y = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let coords' = (fromIntegral . fromEnum) coords
    result <- atk_component_scroll_to_point component' coords' x y
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentScrollToPointMethodInfo
instance (signature ~ (Atk.Enums.CoordType -> Int32 -> Int32 -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentScrollToPointMethodInfo a signature where
    overloadedMethod = componentScrollToPoint

instance O.OverloadedMethodInfo ComponentScrollToPointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentScrollToPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentScrollToPoint"
        })


#endif

-- method Component::set_extents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "width to set for @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "height to set for @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the components top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_set_extents" atk_component_set_extents :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    Int32 ->                                -- width : TBasicType TInt
    Int32 ->                                -- height : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO CInt

-- | Sets the extents of /@component@/.
componentSetExtents ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Int32
    -- ^ /@x@/: x coordinate
    -> Int32
    -- ^ /@y@/: y coordinate
    -> Int32
    -- ^ /@width@/: width to set for /@component@/
    -> Int32
    -- ^ /@height@/: height to set for /@component@/
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the components top level window
    -> m Bool
    -- ^ __Returns:__ 'P.True' or 'P.False' whether the extents were set or not
componentSetExtents component x y width height coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let coordType' = (fromIntegral . fromEnum) coordType
    result <- atk_component_set_extents component' x y width height coordType'
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentSetExtentsMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> Int32 -> Atk.Enums.CoordType -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentSetExtentsMethodInfo a signature where
    overloadedMethod = componentSetExtents

instance O.OverloadedMethodInfo ComponentSetExtentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentSetExtents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentSetExtents"
        })


#endif

-- method Component::set_position
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y coordinate" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specifies whether the coordinates are relative to the screen\nor to the component's top level window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_set_position" atk_component_set_position :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO CInt

-- | Sets the position of /@component@/.
-- 
-- Contrary to atk_component_scroll_to, this does not trigger any scrolling,
-- this just moves /@component@/ in its parent.
componentSetPosition ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Int32
    -- ^ /@x@/: x coordinate
    -> Int32
    -- ^ /@y@/: y coordinate
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: specifies whether the coordinates are relative to the screen
    -- or to the component\'s top level window
    -> m Bool
    -- ^ __Returns:__ 'P.True' or 'P.False' whether or not the position was set or not
componentSetPosition component x y coordType = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    let coordType' = (fromIntegral . fromEnum) coordType
    result <- atk_component_set_position component' x y coordType'
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentSetPositionMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentSetPositionMethodInfo a signature where
    overloadedMethod = componentSetPosition

instance O.OverloadedMethodInfo ComponentSetPositionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentSetPosition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentSetPosition"
        })


#endif

-- method Component::set_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "component"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "Component" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkComponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "width to set for @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "height to set for @component"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_component_set_size" atk_component_set_size :: 
    Ptr Component ->                        -- component : TInterface (Name {namespace = "Atk", name = "Component"})
    Int32 ->                                -- width : TBasicType TInt
    Int32 ->                                -- height : TBasicType TInt
    IO CInt

-- | Set the size of the /@component@/ in terms of width and height.
componentSetSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsComponent a) =>
    a
    -- ^ /@component@/: an t'GI.Atk.Interfaces.Component.Component'
    -> Int32
    -- ^ /@width@/: width to set for /@component@/
    -> Int32
    -- ^ /@height@/: height to set for /@component@/
    -> m Bool
    -- ^ __Returns:__ 'P.True' or 'P.False' whether the size was set or not
componentSetSize component width height = liftIO $ do
    component' <- unsafeManagedPtrCastPtr component
    result <- atk_component_set_size component' width height
    let result' = (/= 0) result
    touchManagedPtr component
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentSetSizeMethodInfo
instance (signature ~ (Int32 -> Int32 -> m Bool), MonadIO m, IsComponent a) => O.OverloadedMethod ComponentSetSizeMethodInfo a signature where
    overloadedMethod = componentSetSize

instance O.OverloadedMethodInfo ComponentSetSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component.componentSetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#v:componentSetSize"
        })


#endif

-- signal Component::bounds-changed
-- | The \'bounds-changed\" signal is emitted when the position or
-- size of the component changes.
type ComponentBoundsChangedCallback =
    Atk.Rectangle.Rectangle
    -- ^ /@arg1@/: The AtkRectangle giving the new position and size.
    -> IO ()

type C_ComponentBoundsChangedCallback =
    Ptr Component ->                        -- object
    Ptr Atk.Rectangle.Rectangle ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ComponentBoundsChangedCallback`.
foreign import ccall "wrapper"
    mk_ComponentBoundsChangedCallback :: C_ComponentBoundsChangedCallback -> IO (FunPtr C_ComponentBoundsChangedCallback)

wrap_ComponentBoundsChangedCallback :: 
    GObject a => (a -> ComponentBoundsChangedCallback) ->
    C_ComponentBoundsChangedCallback
wrap_ComponentBoundsChangedCallback gi'cb gi'selfPtr arg1 _ = do
    B.ManagedPtr.withTransient  arg1 $ \arg1' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  arg1'


-- | Connect a signal handler for the [boundsChanged](#signal:boundsChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' component #boundsChanged callback
-- @
-- 
-- 
onComponentBoundsChanged :: (IsComponent a, MonadIO m) => a -> ((?self :: a) => ComponentBoundsChangedCallback) -> m SignalHandlerId
onComponentBoundsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ComponentBoundsChangedCallback wrapped
    wrapped'' <- mk_ComponentBoundsChangedCallback wrapped'
    connectSignalFunPtr obj "bounds-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [boundsChanged](#signal:boundsChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' component #boundsChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterComponentBoundsChanged :: (IsComponent a, MonadIO m) => a -> ((?self :: a) => ComponentBoundsChangedCallback) -> m SignalHandlerId
afterComponentBoundsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ComponentBoundsChangedCallback wrapped
    wrapped'' <- mk_ComponentBoundsChangedCallback wrapped'
    connectSignalFunPtr obj "bounds-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ComponentBoundsChangedSignalInfo
instance SignalInfo ComponentBoundsChangedSignalInfo where
    type HaskellCallbackType ComponentBoundsChangedSignalInfo = ComponentBoundsChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ComponentBoundsChangedCallback cb
        cb'' <- mk_ComponentBoundsChangedCallback cb'
        connectSignalFunPtr obj "bounds-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Component::bounds-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Component.html#g:signal:boundsChanged"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Component = ComponentSignalList
type ComponentSignalList = ('[ '("boundsChanged", ComponentBoundsChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


