// Copyright 2012 James Cooper. All rights reserved.
// Use of this source code is governed by a MIT-style
// license that can be found in the LICENSE file.

package gorp

import (
	"database/sql/driver"
	"log"
	"time"
)

// A nullable Time value
type NullTime struct {
	Time  time.Time
	Valid bool // Valid is true if Time is not NULL
}

// Scan implements the Scanner interface.
func (nt *NullTime) Scan(value interface{}) error {
	log.Printf("Time scan value is: %#v", value)
	switch t := value.(type) {
	case time.Time:
		nt.Time, nt.Valid = t, true
	case []byte:
		v := strToTime(string(t))
		if v != nil {
			nt.Valid = true
			nt.Time = *v
		}
	case string:
		v := strToTime(t)
		if v != nil {
			nt.Valid = true
			nt.Time = *v
		}
	}
	return nil
}

func strToTime(v string) *time.Time {
	for _, dtfmt := range []string{
		"2006-01-02 15:04:05.999999999",
		"2006-01-02T15:04:05.999999999",
		"2006-01-02 15:04:05",
		"2006-01-02T15:04:05",
		"2006-01-02 15:04",
		"2006-01-02T15:04",
		"2006-01-02",
		"2006-01-02 15:04:05-07:00",
	} {
		if t, err := time.Parse(dtfmt, v); err == nil {
			return &t
		}
	}
	return nil
}

// Value implements the driver Valuer interface.
func (nt NullTime) Value() (driver.Value, error) {
	if !nt.Valid {
		return nil, nil
	}
	return nt.Time, nil
}
