use crate::content_page::ContentPage;
use crate::i18n::i18n;
use crate::image_widget::ImageWidget;
use base64::Engine;
use base64::engine::general_purpose::STANDARD as base64_std;
use gdk4::Texture;
use glib::{Object, Properties, prelude::*, subclass::*};
use gtk4::{Accessible, Buildable, CompositeTemplate, ConstraintTarget, Widget, subclass::prelude::*};
use libadwaita::{Dialog, prelude::*, subclass::prelude::*};
use news_flash::models::ArticleID;
use std::cell::RefCell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ImageDialog)]
    #[template(file = "data/resources/ui_templates/media/image_dialog.blp")]
    pub struct ImageDialog {
        #[template_child]
        pub image: TemplateChild<ImageWidget>,

        #[property(get = Self::get_texture, set = Self::set_texture, nullable)]
        pub texture: RefCell<Option<Texture>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ImageDialog {
        const NAME: &'static str = "ImageDialog";
        type ParentType = Dialog;
        type Type = super::ImageDialog;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ImageDialog {}

    impl WidgetImpl for ImageDialog {}

    impl AdwDialogImpl for ImageDialog {}

    #[gtk4::template_callbacks]
    impl ImageDialog {
        #[template_callback]
        fn on_error(&self, _error: String) {
            self.obj().close();
        }

        fn set_texture(&self, texture: Option<Texture>) {
            self.image.set_texture(texture);
        }

        fn get_texture(&self) -> Option<Texture> {
            self.image.texture()
        }
    }
}

glib::wrapper! {
    pub struct ImageDialog(ObjectSubclass<imp::ImageDialog>)
        @extends Widget, Dialog,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ImageDialog {
    fn default() -> Self {
        Object::new()
    }
}

impl ImageDialog {
    pub fn new_url(article_id: &ArticleID, url: &str) -> Self {
        let dialog = glib::Object::new::<Self>();
        dialog.imp().image.load(article_id, url);
        dialog
    }

    pub fn new_base64_data(base64_data: &str) -> Option<Self> {
        let decoded_data = base64_data
            .find(',')
            .and_then(|start| base64_std.decode(&base64_data[start + 1..]).ok());
        let Some(decoded_data) = decoded_data else {
            ContentPage::instance().simple_message(&i18n("Could not decode image"));
            return None;
        };

        let bytes = glib::Bytes::from_owned(decoded_data);
        let texture = Texture::from_bytes(&bytes);

        let dialog = Self::default();
        dialog.set_texture(texture.ok());
        Some(dialog)
    }

    pub fn new_texture(texture: &Texture) -> Self {
        let dialog = Self::default();
        dialog.set_texture(Some(texture));
        dialog
    }
}
