/*
 * Copyright (c) 2019 Valve Corporation
 *
 * SPDX-License-Identifier: MIT
 *
 * This file was generated by aco_builder_h.py
 */

#ifndef _ACO_BUILDER_
#define _ACO_BUILDER_

#include "aco_ir.h"

namespace aco {
enum dpp_ctrl {
    _dpp_quad_perm = 0x000,
    _dpp_row_sl = 0x100,
    _dpp_row_sr = 0x110,
    _dpp_row_rr = 0x120,
    dpp_wf_sl1 = 0x130,
    dpp_wf_rl1 = 0x134,
    dpp_wf_sr1 = 0x138,
    dpp_wf_rr1 = 0x13C,
    dpp_row_mirror = 0x140,
    dpp_row_half_mirror = 0x141,
    dpp_row_bcast15 = 0x142,
    dpp_row_bcast31 = 0x143,
    _dpp_row_share = 0x150,
    _dpp_row_xmask = 0x160,
};

inline dpp_ctrl
dpp_quad_perm(unsigned lane0, unsigned lane1, unsigned lane2, unsigned lane3)
{
    assert(lane0 < 4 && lane1 < 4 && lane2 < 4 && lane3 < 4);
    return (dpp_ctrl)(lane0 | (lane1 << 2) | (lane2 << 4) | (lane3 << 6));
}

inline dpp_ctrl
dpp_row_sl(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_sl) | amount);
}

inline dpp_ctrl
dpp_row_sr(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_sr) | amount);
}

inline dpp_ctrl
dpp_row_rr(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_rr) | amount);
}

inline dpp_ctrl
dpp_row_share(unsigned lane)
{
    assert(lane < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_share) | lane);
}

inline dpp_ctrl
dpp_row_xmask(unsigned mask)
{
    assert(mask < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_xmask) | mask);
}

inline unsigned
ds_pattern_bitmode(unsigned and_mask, unsigned or_mask, unsigned xor_mask)
{
    assert(and_mask < 32 && or_mask < 32 && xor_mask < 32);
    return and_mask | (or_mask << 5) | (xor_mask << 10);
}

inline unsigned
ds_pattern_rotate(unsigned delta, unsigned mask)
{
    assert(delta < 32 && mask < 32);
    return mask | (delta << 5) | 0xc000;
}

aco_ptr<Instruction> create_s_mov(Definition dst, Operand src);

enum sendmsg {
   sendmsg_none = 0,
   sendmsg_gs = 2, /* gfx6 to gfx10.3 */
   sendmsg_gs_done = 3, /* gfx6 to gfx10.3 */
   sendmsg_hs_tessfactor = 2, /* gfx11+ */
   sendmsg_dealloc_vgprs = 3, /* gfx11+ */
   sendmsg_save_wave = 4, /* gfx8 to gfx10.3 */
   sendmsg_stall_wave_gen = 5, /* gfx9+ */
   sendmsg_halt_waves = 6, /* gfx9+ */
   sendmsg_ordered_ps_done = 7, /* gfx9+ */
   sendmsg_early_prim_dealloc = 8, /* gfx9 to gfx10 */
   sendmsg_gs_alloc_req = 9, /* gfx9+ */
   sendmsg_get_doorbell = 10, /* gfx9 to gfx10.3 */
   sendmsg_get_ddid = 11, /* gfx10 to gfx10.3 */
   sendmsg_id_mask = 0xf,
};

/* gfx11+ */
enum sendmsg_rtn {
   sendmsg_rtn_get_doorbell = 0,
   sendmsg_rtn_get_ddid = 1,
   sendmsg_rtn_get_tma = 2,
   sendmsg_rtn_get_realtime = 3,
   sendmsg_rtn_save_wave = 4,
   sendmsg_rtn_get_tba = 5,
   sendmsg_rtn_mask = 0xff,
};

enum bperm_swiz {
   bperm_b1_sign = 8,
   bperm_b3_sign = 9,
   bperm_b5_sign = 10,
   bperm_b7_sign = 11,
   bperm_0 = 12,
   bperm_255 = 13,
};

enum class alu_delay_wait {
   NO_DEP = 0,
   VALU_DEP_1 = 1,
   VALU_DEP_2 = 2,
   VALU_DEP_3 = 3,
   VALU_DEP_4 = 4,
   TRANS32_DEP_1 = 5,
   TRANS32_DEP_2 = 6,
   TRANS32_DEP_3 = 7,
   FMA_ACCUM_CYCLE_1 = 8,
   SALU_CYCLE_1 = 9,
   SALU_CYCLE_2 = 10,
   SALU_CYCLE_3 = 11,
};

class Builder {
public:
   struct Result {
      Instruction *instr;

      Result(Instruction *instr_) : instr(instr_) {}

      operator Instruction *() const {
         return instr;
      }

      operator Temp() const {
         return instr->definitions[0].getTemp();
      }

      operator Operand() const {
         return Operand((Temp)*this);
      }

      Definition& def(unsigned index) const {
         return instr->definitions[index];
      }

      aco_ptr<Instruction> get_ptr() const {
        return aco_ptr<Instruction>(instr);
      }

      Instruction * operator * () const {
         return instr;
      }

      Instruction * operator -> () const {
         return instr;
      }
   };

   struct Op {
      Operand op;
      Op(Temp tmp) : op(tmp) {}
      Op(Operand op_) : op(op_) {}
      Op(Result res) : op((Temp)res) {}
   };

   enum WaveSpecificOpcode {
      s_cselect = (unsigned) aco_opcode::s_cselect_b64,
      s_cmp_lg = (unsigned) aco_opcode::s_cmp_lg_u64,
      s_and = (unsigned) aco_opcode::s_and_b64,
      s_andn2 = (unsigned) aco_opcode::s_andn2_b64,
      s_or = (unsigned) aco_opcode::s_or_b64,
      s_orn2 = (unsigned) aco_opcode::s_orn2_b64,
      s_not = (unsigned) aco_opcode::s_not_b64,
      s_mov = (unsigned) aco_opcode::s_mov_b64,
      s_wqm = (unsigned) aco_opcode::s_wqm_b64,
      s_and_saveexec = (unsigned) aco_opcode::s_and_saveexec_b64,
      s_or_saveexec = (unsigned) aco_opcode::s_or_saveexec_b64,
      s_andn2_wrexec = (unsigned) aco_opcode::s_andn2_wrexec_b64,
      s_xnor = (unsigned) aco_opcode::s_xnor_b64,
      s_xor = (unsigned) aco_opcode::s_xor_b64,
      s_bcnt1_i32 = (unsigned) aco_opcode::s_bcnt1_i32_b64,
      s_bitcmp1 = (unsigned) aco_opcode::s_bitcmp1_b64,
      s_ff1_i32 = (unsigned) aco_opcode::s_ff1_i32_b64,
      s_flbit_i32 = (unsigned) aco_opcode::s_flbit_i32_b64,
      s_lshl = (unsigned) aco_opcode::s_lshl_b64,
   };

   Program *program;
   bool use_iterator;
   bool start; // only when use_iterator == false
   RegClass lm;

   std::vector<aco_ptr<Instruction>> *instructions;
   std::vector<aco_ptr<Instruction>>::iterator it;
   bool is_precise = false;
   bool is_sz_preserve = false;
   bool is_inf_preserve = false;
   bool is_nan_preserve = false;
   bool is_nuw = false;

   Builder(Program *pgm) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(NULL) {}
   Builder(Program *pgm, Block *block) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(&block->instructions) {}
   Builder(Program *pgm, std::vector<aco_ptr<Instruction>> *instrs) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(instrs) {}

   Builder precise() const {
      Builder res = *this;
      res.is_precise = true;
      return res;
   };

   Builder nuw() const {
      Builder res = *this;
      res.is_nuw = true;
      return res;
   }

   void moveEnd(Block *block) {
      instructions = &block->instructions;
   }

   void reset() {
      use_iterator = false;
      start = false;
      instructions = NULL;
   }

   void reset(Block *block) {
      use_iterator = false;
      start = false;
      instructions = &block->instructions;
   }

   void reset(std::vector<aco_ptr<Instruction>> *instrs) {
      use_iterator = false;
      start = false;
      instructions = instrs;
   }

   void reset(std::vector<aco_ptr<Instruction>> *instrs, std::vector<aco_ptr<Instruction>>::iterator instr_it) {
      use_iterator = true;
      start = false;
      instructions = instrs;
      it = instr_it;
   }

   Result insert(aco_ptr<Instruction> instr) {
      Instruction *instr_ptr = instr.get();
      if (instructions) {
         if (use_iterator) {
            it = instructions->emplace(it, std::move(instr));
            it = std::next(it);
         } else if (!start) {
            instructions->emplace_back(std::move(instr));
         } else {
            instructions->emplace(instructions->begin(), std::move(instr));
         }
      }
      return Result(instr_ptr);
   }

   Result insert(Instruction* instr) {
      if (instructions) {
         if (use_iterator) {
            it = instructions->emplace(it, aco_ptr<Instruction>(instr));
            it = std::next(it);
         } else if (!start) {
            instructions->emplace_back(aco_ptr<Instruction>(instr));
         } else {
            instructions->emplace(instructions->begin(), aco_ptr<Instruction>(instr));
         }
      }
      return Result(instr);
   }

   Temp tmp(RegClass rc) {
      return program->allocateTmp(rc);
   }

   Temp tmp(RegType type, unsigned size) {
      return tmp(RegClass(type, size));
   }

   Definition def(RegClass rc) {
      return Definition(program->allocateTmp(rc));
   }

   Definition def(RegType type, unsigned size) {
      return def(RegClass(type, size));
   }

   Definition def(RegClass rc, PhysReg reg) {
      return Definition(tmp(rc), reg);
   }

   inline aco_opcode w64(WaveSpecificOpcode opcode) const {
      return (aco_opcode) opcode;
   }

   inline aco_opcode w32(WaveSpecificOpcode opcode) const {
      switch (opcode) {
      case s_cselect:
         return aco_opcode::s_cselect_b32;
      case s_cmp_lg:
         return aco_opcode::s_cmp_lg_u32;
      case s_and:
         return aco_opcode::s_and_b32;
      case s_andn2:
         return aco_opcode::s_andn2_b32;
      case s_or:
         return aco_opcode::s_or_b32;
      case s_orn2:
         return aco_opcode::s_orn2_b32;
      case s_not:
         return aco_opcode::s_not_b32;
      case s_mov:
         return aco_opcode::s_mov_b32;
      case s_wqm:
         return aco_opcode::s_wqm_b32;
      case s_and_saveexec:
         return aco_opcode::s_and_saveexec_b32;
      case s_or_saveexec:
         return aco_opcode::s_or_saveexec_b32;
      case s_andn2_wrexec:
         return aco_opcode::s_andn2_wrexec_b32;
      case s_xnor:
         return aco_opcode::s_xnor_b32;
      case s_xor:
         return aco_opcode::s_xor_b32;
      case s_bcnt1_i32:
         return aco_opcode::s_bcnt1_i32_b32;
      case s_bitcmp1:
         return aco_opcode::s_bitcmp1_b32;
      case s_ff1_i32:
         return aco_opcode::s_ff1_i32_b32;
      case s_flbit_i32:
         return aco_opcode::s_flbit_i32_b32;
      case s_lshl:
         return aco_opcode::s_lshl_b32;
      default:
         UNREACHABLE("Unsupported wave specific opcode.");
      }
   }

   inline aco_opcode w64or32(WaveSpecificOpcode opcode) const {
      if (program->wave_size == 64)
         return w64(opcode);
      else
         return w32(opcode);
   }

   Operand m0(Temp tmp) {
       Operand op(tmp);
       op.setPrecolored(aco::m0);
       return op;
   }

   Definition m0(Definition def) {
       def.setPrecolored(aco::m0);
       return def;
   }

   Operand vcc(Temp tmp) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(tmp.type() == RegType::sgpr && tmp.bytes() <= 8);
       Operand op(tmp);
       op.setPrecolored(aco::vcc);
       return op;
   }

   Definition vcc(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setPrecolored(aco::vcc);
       return def;
   }

   Operand exec(Temp tmp) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(tmp.type() == RegType::sgpr && tmp.bytes() <= 8);
       Operand op(tmp);
       op.setPrecolored(aco::exec);
       return op;
   }

   Definition exec(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setPrecolored(aco::exec);
       return def;
   }

   Operand scc(Temp tmp) {
       Operand op(tmp);
       op.setPrecolored(aco::scc);
       return op;
   }

   Definition scc(Definition def) {
       def.setPrecolored(aco::scc);
       return def;
   }


   Operand set16bit(Operand op) {
       op.set16bit(true);
       return op;
   }

   Operand set24bit(Operand op) {
       op.set24bit(true);
       return op;
   }

   /* hand-written helpers */
   Temp as_uniform(Op op)
   {
      assert(op.op.isTemp());
      if (op.op.getTemp().type() == RegType::vgpr)
         return pseudo(aco_opcode::p_as_uniform, def(RegType::sgpr, op.op.size()), op);
      else
         return op.op.getTemp();
   }

   Result v_mul_imm(Definition dst, Temp tmp, uint32_t imm, bool tmpu24=false, bool tmpi24=false)
   {
      assert(tmp.type() == RegType::vgpr);
      /* Assume 24bit if high 8 bits of tmp don't impact the result. */
      if ((imm & 0xff) == 0) {
         tmpu24 = true;
         tmpi24 = true;
      }
      tmpu24 &= imm <= 0xffffffu;
      tmpi24 &= imm <= 0x7fffffu || imm >= 0xff800000u;
      bool has_lshl_add = program->gfx_level >= GFX9;
      /* v_mul_lo_u32 has 1.6x the latency of most VALU on GFX10 (8 vs 5 cycles),
       * compared to 4x the latency on <GFX10. */
      unsigned mul_cost = program->gfx_level >= GFX10 ? 1 : (4 + Operand::c32(imm).isLiteral());
      if (imm == 0) {
         return copy(dst, Operand::zero());
      } else if (imm == 1) {
         return copy(dst, Operand(tmp));
      } else if (imm == 0xffffffff) {
         return vsub32(dst, Operand::zero(), tmp);
      } else if (util_is_power_of_two_or_zero(imm)) {
         return vop2(aco_opcode::v_lshlrev_b32, dst, Operand::c32(ffs(imm) - 1u), tmp);
      } else if (tmpu24) {
        return vop2(aco_opcode::v_mul_u32_u24, dst, Operand::c32(imm), tmp);
      } else if (tmpi24) {
        return vop2(aco_opcode::v_mul_i32_i24, dst, Operand::c32(imm), tmp);
      } else if (util_is_power_of_two_nonzero(imm - 1u)) {
         return vadd32(dst, vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand::c32(ffs(imm - 1u) - 1u), tmp), tmp);
      } else if (mul_cost > 2 && util_is_power_of_two_nonzero(imm + 1u)) {
         return vsub32(dst, vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand::c32(ffs(imm + 1u) - 1u), tmp), tmp);
      }

      unsigned instrs_required = util_bitcount(imm);
      if (!has_lshl_add) {
         instrs_required = util_bitcount(imm) - (imm & 0x1); /* shifts */
         instrs_required += util_bitcount(imm) - 1; /* additions */
      }
      if (instrs_required < mul_cost) {
         Result res(NULL);
         Temp cur;
         while (imm) {
            unsigned shift = u_bit_scan(&imm);
            Definition tmp_dst = imm ? def(v1) : dst;

            if (shift && cur.id())
               res = vadd32(Definition(tmp_dst), vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand::c32(shift), tmp), cur);
            else if (shift)
               res = vop2(aco_opcode::v_lshlrev_b32, Definition(tmp_dst), Operand::c32(shift), tmp);
            else if (cur.id())
               res = vadd32(Definition(tmp_dst), tmp, cur);
            else
               tmp_dst = Definition(tmp);

            cur = tmp_dst.getTemp();
         }
         return res;
      }

      Temp imm_tmp = copy(def(s1), Operand::c32(imm));
      return vop3(aco_opcode::v_mul_lo_u32, dst, imm_tmp, tmp);
   }

   Result v_mul24_imm(Definition dst, Temp tmp, uint32_t imm)
   {
      return v_mul_imm(dst, tmp, imm & 0xffffffu, true);
   }

   Result copy(Definition dst, Op op) {
      return pseudo(aco_opcode::p_parallelcopy, dst, op);
   }

   Result vadd32(Definition dst, Op a, Op b, bool carry_out=false, Op carry_in=Op(Operand(s2)), bool post_ra=false) {
      if (b.op.isConstant() || b.op.regClass().type() != RegType::vgpr)
         std::swap(a, b);
      if (!post_ra && (!b.op.hasRegClass() || b.op.regClass().type() == RegType::sgpr))
         b = copy(def(v1), b);

      if (!carry_in.op.isUndefined())
         return vop2(aco_opcode::v_addc_co_u32, Definition(dst), def(lm), a, b, carry_in);
      else if (program->gfx_level >= GFX10 && carry_out)
         return vop3(aco_opcode::v_add_co_u32_e64, Definition(dst), def(lm), a, b);
      else if (program->gfx_level < GFX9 || carry_out)
         return vop2(aco_opcode::v_add_co_u32, Definition(dst), def(lm), a, b);
      else
         return vop2(aco_opcode::v_add_u32, Definition(dst), a, b);
   }

   Result vsub32(Definition dst, Op a, Op b, bool carry_out=false, Op borrow=Op(Operand(s2)))
   {
      if (!borrow.op.isUndefined() || program->gfx_level < GFX9)
         carry_out = true;

      bool reverse = !b.op.isTemp() || b.op.regClass().type() != RegType::vgpr;
      if (reverse)
         std::swap(a, b);
      if (!b.op.hasRegClass() || b.op.regClass().type() == RegType::sgpr)
         b = copy(def(v1), b);

      aco_opcode op;
      Temp carry;
      if (carry_out) {
         carry = tmp(lm);
         if (borrow.op.isUndefined())
            op = reverse ? aco_opcode::v_subrev_co_u32 : aco_opcode::v_sub_co_u32;
         else
            op = reverse ? aco_opcode::v_subbrev_co_u32 : aco_opcode::v_subb_co_u32;
      } else {
         op = reverse ? aco_opcode::v_subrev_u32 : aco_opcode::v_sub_u32;
      }
      bool vop3 = false;
      if (program->gfx_level >= GFX10 && op == aco_opcode::v_subrev_co_u32) {
        vop3 = true;
        op = aco_opcode::v_subrev_co_u32_e64;
      } else if (program->gfx_level >= GFX10 && op == aco_opcode::v_sub_co_u32) {
        vop3 = true;
        op = aco_opcode::v_sub_co_u32_e64;
      }

      int num_ops = borrow.op.isUndefined() ? 2 : 3;
      int num_defs = carry_out ? 2 : 1;
      aco_ptr<Instruction> sub;
      if (vop3)
        sub.reset(create_instruction(op, Format::VOP3, num_ops, num_defs));
      else
        sub.reset(create_instruction(op, Format::VOP2, num_ops, num_defs));
      sub->operands[0] = a.op;
      sub->operands[1] = b.op;
      if (!borrow.op.isUndefined())
         sub->operands[2] = borrow.op;
      sub->definitions[0] = dst;
      if (carry_out)
         sub->definitions[1] = Definition(carry);

      return insert(std::move(sub));
   }

   Result readlane(Definition dst, Op vsrc, Op lane)
   {
      if (program->gfx_level >= GFX8)
         return vop3(aco_opcode::v_readlane_b32_e64, dst, vsrc, lane);
      else
         return vop2(aco_opcode::v_readlane_b32, dst, vsrc, lane);
   }
   Result writelane(Definition dst, Op val, Op lane, Op vsrc) {
      if (program->gfx_level >= GFX8)
         return vop3(aco_opcode::v_writelane_b32_e64, dst, val, lane, vsrc);
      else
         return vop2(aco_opcode::v_writelane_b32, dst, val, lane, vsrc);
   }
        
   Result pseudo(aco_opcode opcode)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2, Op op3)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Definition def4, Definition def5, Definition def6, Definition def7, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 8);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setSZPreserve(is_sz_preserve);
            instr->definitions[3].setInfPreserve(is_inf_preserve);
            instr->definitions[3].setNaNPreserve(is_nan_preserve);
            instr->definitions[3].setNUW(is_nuw);
            instr->definitions[4] = def4;
            instr->definitions[4].setPrecise(is_precise);
            instr->definitions[4].setSZPreserve(is_sz_preserve);
            instr->definitions[4].setInfPreserve(is_inf_preserve);
            instr->definitions[4].setNaNPreserve(is_nan_preserve);
            instr->definitions[4].setNUW(is_nuw);
            instr->definitions[5] = def5;
            instr->definitions[5].setPrecise(is_precise);
            instr->definitions[5].setSZPreserve(is_sz_preserve);
            instr->definitions[5].setInfPreserve(is_inf_preserve);
            instr->definitions[5].setNaNPreserve(is_nan_preserve);
            instr->definitions[5].setNUW(is_nuw);
            instr->definitions[6] = def6;
            instr->definitions[6].setPrecise(is_precise);
            instr->definitions[6].setSZPreserve(is_sz_preserve);
            instr->definitions[6].setInfPreserve(is_inf_preserve);
            instr->definitions[6].setNaNPreserve(is_nan_preserve);
            instr->definitions[6].setNUW(is_nuw);
            instr->definitions[7] = def7;
            instr->definitions[7].setPrecise(is_precise);
            instr->definitions[7].setSZPreserve(is_sz_preserve);
            instr->definitions[7].setInfPreserve(is_inf_preserve);
            instr->definitions[7].setNaNPreserve(is_nan_preserve);
            instr->definitions[7].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, Op op7)
   {
      unsigned num_ops = 8;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
            instr->operands[7] = op7.op;


            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6)
   {
      unsigned num_ops = 7;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;


            
       
      return insert(instr);
   }

        
   Result sop1(aco_opcode opcode, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP1), num_ops, 0);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Op op0)
   {
       return sop1(w64or32(opcode), op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP1), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);


            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0)
   {
       return sop1(w64or32(opcode), def0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP1), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Op op0)
   {
       return sop1(w64or32(opcode), def0, op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Definition def1, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP1), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0)
   {
       return sop1(w64or32(opcode), def0, def1, op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP1), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
       return sop1(w64or32(opcode), def0, def1, def2, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP2), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1)
   {
       return sop2(w64or32(opcode), def0, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP2), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
       return sop2(w64or32(opcode), def0, op0, op1, op2);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP2), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
       return sop2(w64or32(opcode), def0, def1, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOP2), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
       return sop2(w64or32(opcode), def0, def1, op0, op1, op2);
   }

        
   Result sopk(aco_opcode opcode, uint32_t imm=0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 0);


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Op op0, uint32_t imm=0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 0);
            instr->operands[0] = op0.op;


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, uint32_t imm=0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Op op0, uint32_t imm=0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Definition def1, uint32_t imm=0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Definition def1, Op op0, uint32_t imm=0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPK), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, uint32_t imm=0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPP), num_ops, 0);


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, Op op0, uint32_t imm=0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPP), num_ops, 0);
            instr->operands[0] = op0.op;


      instr->salu().imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopc(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SOPC), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   inline Result sopc(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1)
   {
       return sopc(w64or32(opcode), def0, op0, op1);
   }

        
   Result smem(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Op op0, Op op1, Op op2, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, Op op0, Op op1, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, Op op0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}})
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SMEM), num_ops, 0);


      instr->smem().sync = sync;
      instr->smem().cache = cache;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Op op0, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Op op0, Op op1, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Op op0, Op op1, Op op2, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint16_t offset0=0, uint8_t offset1=0, bool gds=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::DS), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->ds().offset0 = offset0;
      instr->ds().offset1 = offset1;
      instr->ds().gds = gds;
            
       
      return insert(instr);
   }

        
   Result ldsdir(aco_opcode opcode, Definition def0, Op op0, uint8_t attr=0, uint8_t attr_chan=0, memory_sync_info sync=memory_sync_info(), uint8_t wait_vdst=15, uint8_t wait_vsrc=1)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::LDSDIR), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


      instr->ldsdir().attr = attr;
      instr->ldsdir().attr_chan = attr_chan;
      instr->ldsdir().sync = sync;
      instr->ldsdir().wait_vdst = wait_vdst;
      instr->ldsdir().wait_vsrc = wait_vsrc;
            
       
      return insert(instr);
   }

        
   Result mubuf(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned offset, bool offen, bool idxen=false, bool addr64=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lds=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MUBUF), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mubuf().offset = offset;
      instr->mubuf().offen = offen;
      instr->mubuf().idxen = idxen;
      instr->mubuf().addr64 = addr64;
      instr->mubuf().disable_wqm = disable_wqm;
      instr->mubuf().cache = cache;
      instr->mubuf().tfe = tfe;
      instr->mubuf().lds = lds;
            
       
      return insert(instr);
   }

        
   Result mubuf(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned offset, bool offen, bool idxen=false, bool addr64=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lds=false)
   {
      unsigned num_ops = 3;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MUBUF), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mubuf().offset = offset;
      instr->mubuf().offen = offen;
      instr->mubuf().idxen = idxen;
      instr->mubuf().addr64 = addr64;
      instr->mubuf().disable_wqm = disable_wqm;
      instr->mubuf().cache = cache;
      instr->mubuf().tfe = tfe;
      instr->mubuf().lds = lds;
            
       
      return insert(instr);
   }

        
   Result mubuf(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, unsigned offset, bool offen, bool idxen=false, bool addr64=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lds=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MUBUF), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mubuf().offset = offset;
      instr->mubuf().offen = offen;
      instr->mubuf().idxen = idxen;
      instr->mubuf().addr64 = addr64;
      instr->mubuf().disable_wqm = disable_wqm;
      instr->mubuf().cache = cache;
      instr->mubuf().tfe = tfe;
      instr->mubuf().lds = lds;
            
       
      return insert(instr);
   }

        
   Result mtbuf(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned dfmt, unsigned nfmt, unsigned offset, bool offen, bool idxen=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MTBUF), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mtbuf().dfmt = dfmt;
      instr->mtbuf().nfmt = nfmt;
      instr->mtbuf().offset = offset;
      instr->mtbuf().offen = offen;
      instr->mtbuf().idxen = idxen;
      instr->mtbuf().disable_wqm = disable_wqm;
      instr->mtbuf().cache = cache;
      instr->mtbuf().tfe = tfe;
            
       
      return insert(instr);
   }

        
   Result mtbuf(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned dfmt, unsigned nfmt, unsigned offset, bool offen, bool idxen=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false)
   {
      unsigned num_ops = 3;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MTBUF), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mtbuf().dfmt = dfmt;
      instr->mtbuf().nfmt = nfmt;
      instr->mtbuf().offset = offset;
      instr->mtbuf().offen = offen;
      instr->mtbuf().idxen = idxen;
      instr->mtbuf().disable_wqm = disable_wqm;
      instr->mtbuf().cache = cache;
      instr->mtbuf().tfe = tfe;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 3;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 5;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 6;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 7;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 3;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 5;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 6;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 7;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, Op op7, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 8;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
            instr->operands[7] = op7.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, Op op7, Op op8, Op op9, Op op10, Op op11, Op op12, Op op13, unsigned dmask=0xF, bool da=false, bool unrm=false, bool disable_wqm=false, ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool tfe=false, bool lwe=false, bool r128=false, bool a16=false, bool d16=false)
   {
      unsigned num_ops = 14;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::MIMG), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
            instr->operands[7] = op7.op;
            instr->operands[8] = op8.op;
            instr->operands[9] = op9.op;
            instr->operands[10] = op10.op;
            instr->operands[11] = op11.op;
            instr->operands[12] = op12.op;
            instr->operands[13] = op13.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->mimg().dmask = dmask;
      instr->mimg().da = da;
      instr->mimg().unrm = unrm;
      instr->mimg().disable_wqm = disable_wqm;
      instr->mimg().cache = cache;
      instr->mimg().tfe = tfe;
      instr->mimg().lwe = lwe;
      instr->mimg().r128 = r128;
      instr->mimg().a16 = a16;
      instr->mimg().d16 = d16;
            
       
      return insert(instr);
   }

        
   Result exp(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned enabled_mask, unsigned dest, bool compr=false, bool done=false, bool vm=false, bool disable_wqm=false)
   {
      unsigned num_ops = 4;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::EXP), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->exp().enabled_mask = enabled_mask;
      instr->exp().dest = dest;
      instr->exp().compressed = compr;
      instr->exp().done = done;
      instr->exp().valid_mask = vm;
      instr->exp().disable_wqm = disable_wqm;
            
       
      return insert(instr);
   }

        
   Result exp(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, unsigned enabled_mask, unsigned dest, bool compr=false, bool done=false, bool vm=false, bool disable_wqm=false)
   {
      unsigned num_ops = 5;
      num_ops += disable_wqm * 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::EXP), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;

        if (disable_wqm) {
           instr_exact_mask(instr) = Operand();
           instr_wqm_mask(instr) = Operand();
        }

      instr->exp().enabled_mask = enabled_mask;
      instr->exp().dest = dest;
      instr->exp().compressed = compr;
      instr->exp().done = done;
      instr->exp().valid_mask = vm;
      instr->exp().disable_wqm = disable_wqm;
            
       
      return insert(instr);
   }

        
   Result branch(aco_opcode opcode, uint32_t target0=0, uint32_t target1=0)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO_BRANCH), num_ops, 0);


      instr->branch().target[0] = target0;
      instr->branch().target[1] = target1;
            
       
      return insert(instr);
   }

        
   Result branch(aco_opcode opcode, Op op0, uint32_t target0=0, uint32_t target1=0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO_BRANCH), num_ops, 0);
            instr->operands[0] = op0.op;


      instr->branch().target[0] = target0;
      instr->branch().target[1] = target1;
            
       
      return insert(instr);
   }

        
   Result barrier(aco_opcode opcode, memory_sync_info sync, sync_scope exec_scope=scope_invocation)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO_BARRIER), num_ops, 0);


      instr->barrier().sync = sync;
      instr->barrier().exec_scope = exec_scope;
            
       
      return insert(instr);
   }

        
   Result reduction(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, ReduceOp op, unsigned cluster_size=0)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO_REDUCTION), num_ops, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setSZPreserve(is_sz_preserve);
            instr->definitions[2].setInfPreserve(is_inf_preserve);
            instr->definitions[2].setNaNPreserve(is_nan_preserve);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->reduction().reduce_op = op;
      instr->reduction().cluster_size = cluster_size;
            
       
      return insert(instr);
   }

        
   Result call(aco_opcode opcode, ABI abi)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::PSEUDO_CALL), num_ops, 0);


      instr->call().abi = abi;
            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode)
   {
      unsigned num_ops = 0;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1), num_ops, 0);


            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop1_sdwa(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::SDWA), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vopc(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vopc(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vopc_sdwa(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::SDWA), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vopc_sdwa(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::SDWA), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            instr->sdwa().sel[0] = SubdwordSel(op0.op.bytes(), 0, false);instr->sdwa().sel[1] = SubdwordSel(op1.op.bytes(), 0, false);instr->sdwa().dst_sel = SubdwordSel(def0.bytes(), 0, false);

       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
       
      return insert(instr);
   }

        
   Result vop3p(aco_opcode opcode, Definition def0, Op op0, Op op1, uint8_t opsel_lo, uint8_t opsel_hi)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
       
      return insert(instr);
   }

        
   Result vop3p(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint8_t opsel_lo, uint8_t opsel_hi)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
       
      return insert(instr);
   }

        
   Result vopd(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, aco_opcode opy)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPD), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->vopd().opy = opy;
            
       
      return insert(instr);
   }

        
   Result vopd(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, aco_opcode opy)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPD), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->vopd().opy = opy;
            
       
      return insert(instr);
   }

        
   Result vopd(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, aco_opcode opy)
   {
      unsigned num_ops = 4;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPD), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;


      instr->vopd().opy = opy;
            
       
      return insert(instr);
   }

        
   Result vopd(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, Op op4, aco_opcode opy)
   {
      unsigned num_ops = 5;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPD), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;


      instr->vopd().opy = opy;
            
       
      return insert(instr);
   }

        
   Result vopd(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, aco_opcode opy)
   {
      unsigned num_ops = 6;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPD), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;


      instr->vopd().opy = opy;
            
       
      return insert(instr);
   }

        
   Result vinterp_inreg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint8_t opsel=0, unsigned wait_exp=7)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VINTERP_INREG), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->vinterp_inreg().opsel = opsel;
      instr->vinterp_inreg().wait_exp = wait_exp;
            
       
      return insert(instr);
   }

        
   Result vintrp(aco_opcode opcode, Definition def0, Op op0, Op op1, unsigned attribute, unsigned component, bool high_16bits=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VINTRP), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->vintrp().attribute = attribute;
      instr->vintrp().component = component;
      instr->vintrp().high_16bits = high_16bits;
            
       
      return insert(instr);
   }

        
   Result vintrp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned attribute, unsigned component, bool high_16bits=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VINTRP), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->vintrp().attribute = attribute;
      instr->vintrp().component = component;
      instr->vintrp().high_16bits = high_16bits;
            
       
      return insert(instr);
   }

        
   Result vop1_dpp(aco_opcode opcode, Definition def0, Op op0, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp(aco_opcode opcode, Definition def0, Op op0, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3p_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint8_t opsel_lo, uint8_t opsel_hi, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3p_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint8_t opsel_lo, uint8_t opsel_hi, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop1_dpp8(aco_opcode opcode, Definition def0, Op op0, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp8(aco_opcode opcode, Definition def0, Op op0, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3p_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint8_t opsel_lo, uint8_t opsel_hi, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop3p_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint8_t opsel_lo, uint8_t opsel_hi, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP3P|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->valu().opsel_lo = opsel_lo;
      instr->valu().opsel_hi = opsel_hi;
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop1_e64(aco_opcode opcode, Definition def0, Op op0)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
       
      return insert(instr);
   }

        
   Result vopc_e64(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
       
      return insert(instr);
   }

        
   Result vopc_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
       
      return insert(instr);
   }

        
   Result vop1_e64_dpp(aco_opcode opcode, Definition def0, Op op0, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_e64_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3|(int)Format::DPP16), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_e64_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3|(int)Format::DPP16), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp16().dpp_ctrl = dpp_ctrl;
      instr->dpp16().row_mask = row_mask;
      instr->dpp16().bank_mask = bank_mask;
      instr->dpp16().bound_ctrl = bound_ctrl;
      instr->dpp16().fetch_inactive = fetch_inactive;
            instr->dpp16().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop1_e64_dpp8(aco_opcode opcode, Definition def0, Op op0, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 1;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP1|(int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vop2_e64_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_e64_dpp8(aco_opcode opcode, Definition def0, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3|(int)Format::DPP8), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result vopc_e64_dpp8(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint32_t lane_sel=0, bool fetch_inactive=true)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3|(int)Format::DPP8), num_ops, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setSZPreserve(is_sz_preserve);
            instr->definitions[1].setInfPreserve(is_inf_preserve);
            instr->definitions[1].setNaNPreserve(is_nan_preserve);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


            
            
      instr->dpp8().lane_sel = lane_sel;
      instr->dpp8().fetch_inactive = fetch_inactive;
            instr->dpp8().fetch_inactive &= program->gfx_level >= GFX10;

       
      return insert(instr);
   }

        
   Result flat(aco_opcode opcode, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::FLAT), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result flat(aco_opcode opcode, Definition def0, Op op0, Op op1, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::FLAT), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result flat(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::FLAT), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result global(aco_opcode opcode, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::GLOBAL), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result global(aco_opcode opcode, Definition def0, Op op0, Op op1, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::GLOBAL), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result global(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::GLOBAL), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result scratch(aco_opcode opcode, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SCRATCH), num_ops, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result scratch(aco_opcode opcode, Definition def0, Op op0, Op op1, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 2;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SCRATCH), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

        
   Result scratch(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, int32_t offset=0, memory_sync_info sync=memory_sync_info(), ac_hw_cache_flags cache={{0, 0, 0, 0, 0}}, bool lds=false, bool nv=false)
   {
      unsigned num_ops = 3;
      Instruction* instr = create_instruction(opcode, (Format)((int)Format::SCRATCH), num_ops, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setSZPreserve(is_sz_preserve);
            instr->definitions[0].setInfPreserve(is_inf_preserve);
            instr->definitions[0].setNaNPreserve(is_nan_preserve);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;


      instr->flatlike().offset = offset;
      instr->flatlike().sync = sync;
      instr->flatlike().cache = cache;
      instr->flatlike().lds = lds;
      instr->flatlike().nv = nv;
            
       
      return insert(instr);
   }

};

void hw_init_scratch(Builder& bld, Definition def, Operand scratch_addr, Operand scratch_offset);

} // namespace aco

#endif /* _ACO_BUILDER_ */
