dnl ##########################################################################
dnl ###  WSG_PKG - define a package
dnl ##########################################################################

AC_DEFUN(WSG_PKG, [
  PACKAGE="$1";
  VERSION="$2";
  AC_DEFINE_UNQUOTED(VERSION, "${VERSION}", [Version of this package.])
  WSG_PKG_DEFS="PACKAGE = ${PACKAGE}\\
VERSION = ${VERSION}";
  AC_SUBST(WSG_PKG_DEFS)
])


dnl ##########################################################################
dnl ###  WSG_ENCAP - check for Encap tools
dnl ##########################################################################

AC_DEFUN(WSG_ENCAP, [
  MKENCAP_OPTS=$1;

  dnl ### must call WSG_PKG first
  if test -z "${PACKAGE}" || test -z "${VERSION}"; then
    AC_MSG_ERROR([[WSG_PKG] must be used before [WSG_ENCAP]])
  fi

  dnl ### allow user to disable Encap support
  AC_ARG_ENABLE(encap,
    [  --disable-encap         Do not configure as an Encap package],
    ,
    enable_encap=default)

  if test "$enable_encap" != "no"; then
    dnl ### look for epkg and mkencap
    AC_PATH_PROG(EPKG, epkg)
    AC_PATH_PROG(MKENCAP, mkencap)

    dnl ### enable by default if epkg or mkencap are found
    if test "${EPKG:+set}" = "set" || test "${MKENCAP:+set}" = "set" && test "$enable_encap" = "default"; then
      enable_encap=yes;
    fi
  fi

  if test "$enable_encap" = "yes"; then
    dnl ###################################################################
    dnl ### generate fallback values for ${ENCAP_SOURCE} and ${ENCAP_TARGET}
    dnl ### from the environment or the default prefix
    dnl ###################################################################
    if test -z "${ENCAP_SOURCE}" && test -z "${ENCAP_TARGET}"; then
      ENCAP_SOURCE="${ac_default_prefix}/encap";
      ENCAP_TARGET="${ac_default_prefix}";
    elif test -z "${ENCAP_TARGET}"; then
      ENCAP_TARGET="`dirname ${ENCAP_SOURCE}`";
    elif test -z "${ENCAP_SOURCE}"; then
      ENCAP_SOURCE="${ENCAP_TARGET}/encap";
    fi

    dnl ###################################################################
    dnl ### if --prefix is specified:
    dnl ###   1) if its next-to-last componant is "encap", assume that it
    dnl ###      points to the package directory
    dnl ###   2) otherwise, assume it points to the target directory
    dnl ###################################################################
    if test "${prefix}" != "NONE"; then
      prefixdir="`dirname ${prefix}`";
      prefixbase="`basename ${prefix}`";
      if test "`basename ${prefixdir}`" = "encap"; then
        ENCAP_SOURCE="${prefixdir}";
        ENCAP_TARGET="`dirname ${ENCAP_SOURCE}`";
      elif test "${prefixdir}" != "${ENCAP_SOURCE}"; then
        ENCAP_SOURCE="${prefix}/encap";
        ENCAP_TARGET="${prefix}";
      fi
      if ( test "`basename ${prefixdir}`" = "encap" || \
           test "${prefixdir}" = "${ENCAP_SOURCE}" ) && \
         test "${prefixbase}" != "${PACKAGE}-${VERSION}"; then
        ENCAP_PKGSPEC="${prefixbase}";
      fi
    fi

    dnl ### display results
    AC_MSG_CHECKING([for Encap source directory])
    AC_MSG_RESULT(${ENCAP_SOURCE})
    AC_MSG_CHECKING([for Encap target directory])
    AC_MSG_RESULT(${ENCAP_TARGET})
    AC_MSG_CHECKING([for Encap package directory])
    if test "${ENCAP_PKGSPEC:-unset}" = "unset"; then
      ENCAP_PKGSPEC='${PACKAGE}-${VERSION}';
      AC_MSG_RESULT(${ENCAP_SOURCE}/${PACKAGE}-${VERSION})
    else
      AC_MSG_RESULT(${ENCAP_SOURCE}/${ENCAP_PKGSPEC})
    fi
    prefix='${ENCAP_SOURCE}/${ENCAP_PKGSPEC}';

    dnl ### override default sysconfdir and localstatedir
    if test "$sysconfdir" = '${prefix}/etc'; then
      sysconfdir='${ENCAP_TARGET}/etc';
    fi
    if test "$localstatedir" = '${prefix}/var'; then
      localstatedir='/var/lib/${PACKAGE}';
    fi

    dnl ### check for --disable-epkg-install
    AC_ARG_ENABLE(epkg-install,
      [  --disable-epkg-install  Do not run epkg during make install],
      ,
      enable_epkg_install=yes)
    if test "$enable_epkg_install" = "no"; then
      EPKG="";
    fi

    dnl ### generate Makefile variables
    WSG_ENCAP_DEFS="ENCAP_SOURCE = ${ENCAP_SOURCE}\\
ENCAP_TARGET = ${ENCAP_TARGET}\\
ENCAP_PKGSPEC = ${ENCAP_PKGSPEC}\\
EPKG = ${EPKG:-:}\\
MKENCAP = ${MKENCAP:-:}\\
MKENCAP_OPTS = ${MKENCAP_OPTS}";
    AC_SUBST(WSG_ENCAP_DEFS)

    dnl ### generate rules for make install target
    WSG_INSTALL_RULES='${MKENCAP} ${MKENCAP_OPTS} -s ${DESTDIR}${ENCAP_SOURCE} -e ${ENCAP_PKGSPEC};\
	if test -z \"${DESTDIR}\"; then \\\
		${EPKG} -s ${ENCAP_SOURCE} -t ${ENCAP_TARGET} ${ENCAP_PKGSPEC}; \\\
	fi';
    AC_SUBST(WSG_INSTALL_RULES)
  fi
])


dnl ##########################################################################
dnl ###  WSG_FUNC_BASENAME - check for working basename()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_BASENAME, [
  AC_DEFINE(NEED_BASENAME, 1, [Define if you want to use the basename function])
  AC_CHECK_HEADERS(libgen.h)
  AC_CACHE_CHECK([for working basename],
    wsg_cv_func_basename_works,
    [AC_TRY_RUN(
changequote(<<, >>)<<
#include <stdio.h>
#ifdef HAVE_LIBGEN_H
# include <libgen.h>
#endif

typedef struct {
  char *test;
  char *result;
} test_t;

const test_t tests[] = {
  { "/usr/local/foo", "foo" },
  { "/usr/local/foo/", "foo" },
  { NULL, NULL }
};

int main() {
  char test1[1024];
  int i;

  for (i = 0; tests[i].test; i++) {
    strcpy(test1, tests[i].test);
    if (strcmp(basename(test1), tests[i].result) ||
        strcmp(test1, tests[i].test))
      exit(1);
  }

  exit(0);
}
>>changequote([, ]),
      wsg_cv_func_basename_works=yes,
      wsg_cv_func_basename_works=no,
      wsg_cv_func_basename_works=no
    )]
  )
  if test "$wsg_cv_func_basename_works" = "yes"; then
    AC_DEFINE(HAVE_BASENAME, 1, [Define if your system has a working basename])
  else
    LIBOBJS="$LIBOBJS basename.o"
  fi
])


dnl ##########################################################################
dnl ###  WSG_FUNC_DIRNAME - check for working dirname()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_DIRNAME, [
  AC_DEFINE(NEED_DIRNAME, 1, [Define if you want to use the dirname function])
  AC_CHECK_HEADERS(libgen.h)
  AC_CACHE_CHECK([for working dirname],
    wsg_cv_func_dirname_works,
    [AC_TRY_RUN(
changequote(<<, >>)<<
#include <stdio.h>
#ifdef HAVE_LIBGEN_H
# include <libgen.h>
#endif

typedef struct {
  char *test;
  char *result;
} test_t;

const test_t tests[] = {
  { "/usr/local/foo", "/usr/local" },
  { "/usr/local/foo/", "/usr/local" },
  { "/", "/" },
  { "", "." },
  { NULL, NULL }
};

int main() {
  char test1[1024];
  int i;

  for (i = 0; tests[i].test; i++) {
    strcpy(test1, tests[i].test);
    if (strcmp(dirname(test1), tests[i].result) ||
        strcmp(test1, tests[i].test))
      exit(1);
  }

  exit(0);
}
>>changequote([, ]),
      wsg_cv_func_dirname_works=yes,
      wsg_cv_func_dirname_works=no,
      wsg_cv_func_dirname_works=no
    )]
  )
  if test "$wsg_cv_func_dirname_works" = "yes"; then
    AC_DEFINE(HAVE_DIRNAME, 1, [Define if your system has a working dirname])
  else
    LIBOBJS="$LIBOBJS dirname.o"
  fi
])


dnl ##########################################################################
dnl ###  WSG_FUNC_FNMATCH - check for working fnmatch()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_FNMATCH, [
  AC_DEFINE(NEED_FNMATCH, 1, [Define if you want to use the fnmatch function])
  AC_CHECK_HEADERS(fnmatch.h)
  if test "$ac_cv_header_fnmatch_h" = "yes"; then
    AC_FUNC_FNMATCH
  fi
  if test "$ac_cv_func_fnmatch_works" != "yes"; then
    AC_CHECK_HEADERS(ctype.h)
    LIBOBJS="$LIBOBJS fnmatch.o"
  fi
])


dnl ##########################################################################
dnl ###  WSG_FUNC_GLOB - check for working glob()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_GLOB, [
  AC_DEFINE(NEED_GLOB, 1, [Define if you want to use the glob function])
  AC_CHECK_HEADERS(glob.h)
  AC_CACHE_CHECK([for working glob],
    wsg_cv_func_glob_works,
    [AC_TRY_RUN(
changequote(<<, >>)<<
#include <stdio.h>
#ifdef HAVE_GLOB_H
# include <glob.h>
#endif

#ifndef GLOB_ABORTED
# define GLOB_ABORTED GLOB_ABEND
#endif

int main() {
  glob_t g;
  int status;

  status = glob("conf*", 0, NULL, &g);
  switch (status) {
    case 0:
    case GLOB_NOSPACE:
    case GLOB_ABORTED:
    case GLOB_NOMATCH:
      exit(0);
      break;
    default:
      exit(1);
      break;
  }
}
>>changequote([, ]),
      wsg_cv_func_glob_works=yes,
      wsg_cv_func_glob_works=no,
      wsg_cv_func_glob_works=no
    )]
  )
  if test "$wsg_cv_func_glob_works" = "yes"; then
    AC_DEFINE(HAVE_GLOB, 1, [Define if your system has a working glob])
  else
    LIBOBJS="$LIBOBJS glob.o"
    AC_CHECK_FUNCS(issetugid)
  fi
])


dnl ##########################################################################
dnl ###  WSG_FUNC_SNPRINTF - check for working snprintf()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_SNPRINTF, [
  AC_DEFINE(NEED_SNPRINTF, 1, [Define if you want to use the snprintf function])
  AC_CACHE_CHECK([for working snprintf],
    wsg_cv_func_snprintf_works,
    [AC_TRY_RUN(
changequote(<<, >>)<<
#include <stdio.h>

typedef struct {
  int length;
  char *test;
  int retval;
  char *result;
} test_t;

const test_t tests[] = {
  { 10, "12345678901234567890", 20, "123456789" },
#if 0
  { 0, "12345678901234567890", 20, NULL },
  { -1, "12345678901234567890", -1, NULL },
#endif
  { 0, NULL, 0, NULL }
};

int main() {
  char test1[1024];
  int i;

  for (i = 0; tests[i].test; i++) {
    memset(test1, 'X', sizeof(test1));
    if ((snprintf(test1, tests[i].length, "%s", tests[i].test)
         != tests[i].retval) ||
        (tests[i].result && strcmp(tests[i].result, test1)))
      exit(1);
  }

  exit(0);
}
>>changequote([, ]),
      wsg_cv_func_snprintf_works=yes,
      wsg_cv_func_snprintf_works=no,
      wsg_cv_func_snprintf_works=no
    )]
  )
  if test "$wsg_cv_func_snprintf_works" = "yes"; then
    AC_DEFINE(HAVE_SNPRINTF, 1, [Define if your system has a working snprintf])
  else
    LIBOBJS="$LIBOBJS snprintf.o"
  fi
])


dnl ##########################################################################
dnl ###  WSG_REPLACE_FUNC - wrapper for AC_REPLACE_FUNCS()
dnl ##########################################################################

AC_DEFUN(WSG_REPLACE_FUNC, [
  AC_DEFINE(NEED_[]translit($1,[a-z],[A-Z]), 1,
	    Define if you want to use the $1 function)
  AC_REPLACE_FUNCS($1)
])


dnl ##########################################################################
dnl ###  WSG_FUNC_STRTOK_R - check for working strtok_r()
dnl ##########################################################################

AC_DEFUN(WSG_FUNC_STRTOK_R, [
  AC_DEFINE(NEED_STRTOK_R, 1, [Define if you want to use the strtok_r function])
  AC_REPLACE_FUNCS(strtok_r)
  if test "$ac_cv_func_strtok_r" = "yes"; then
    AC_CACHE_CHECK([what to define for strtok_r prototype],
      wsg_cv_func_strtok_r_define,
      [AC_EGREP_HEADER(strtok_r, string.h,
        wsg_cv_func_strtok_r_define=none,
        [AC_EGREP_CPP(strtok_r, [
            #define _REENTRANT
            #include <string.h>
          ],
          wsg_cv_func_strtok_r_define=-D_REENTRANT,
          [AC_EGREP_CPP(strtok_r, [
              #define _THREAD_SAFE
              #include <string.h>
            ],
            wsg_cv_func_strtok_r_define=-D_THREAD_SAFE,
            wsg_cv_func_strtok_r_define=none
          )]
        )]
      )]
    )
    if test $wsg_cv_func_strtok_r_define != none; then
      CFLAGS="$CFLAGS $wsg_cv_func_strtok_r_define"
    fi
  fi
])


dnl ##########################################################################
dnl ###  WSG_LIB_READLINE - check for GNU readline library
dnl ##########################################################################

AC_DEFUN(WSG_LIB_READLINE, [
  AC_CHECK_HEADERS(readline/readline.h)
  AC_CHECK_HEADERS(readline/history.h)
  if test "$ac_cv_header_readline_readline_h" = "yes"; then
    AC_SEARCH_LIBS(tputs, termcap curses)
    AC_CHECK_LIB(readline, rl_callback_handler_install)
  fi
])


dnl ##########################################################################
dnl ###  WSG_LIB_TAR - check for libtar
dnl ##########################################################################

AC_DEFUN(WSG_LIB_TAR, [
  wsg_old_libs="$LIBS"
  LIBS="$LIBS -ltar"
  AC_CACHE_CHECK([for usable version of libtar],
    wsg_cv_lib_tar_usable,
    [AC_TRY_RUN(
changequote(<<,>>)<<
#include <stdio.h>
#include <string.h>
#include <libtar.h>

int main(int argc, char *argv[]) {
  return (strcmp(libtar_version, "1.2") >= 0 ? 0 : 1);
}
>>changequote([,]),
      wsg_cv_lib_tar_usable=yes,
      wsg_cv_lib_tar_usable=no,
      wsg_cv_lib_tar_usable=no
    )]
  )
  if test "$wsg_cv_lib_tar_usable" = "yes"; then
    AC_DEFINE(HAVE_LIBTAR, 1,
              [Define if your system has a current version of libtar])
  else
    LIBS="$wsg_old_libs"
  fi
])


dnl ##########################################################################
dnl ###  WSG_LIB_FGET - check for libfget
dnl ##########################################################################

AC_DEFUN(WSG_LIB_FGET, [
  wsg_old_libs="$LIBS"
  AC_CHECK_LIB(socket, socket)
  AC_CHECK_LIB(nsl, gethostbyname)
  LIBS="$LIBS -lfget"
  AC_CACHE_CHECK([for usable version of libfget],
    wsg_cv_lib_fget_usable,
    [AC_TRY_COMPILE([
        #include <libfget.h>
      ], changequote(<<,>>)<<
	FTP *ftp;
	char buf[10240];

        ftp_connect(&ftp, "host.com", buf, sizeof(buf), 0, 0,
		    -1, -1, NULL, NULL);
      >>changequote([,]),
      wsg_cv_lib_fget_usable=yes,
      wsg_cv_lib_fget_usable=no
    )]
  )
  if test "$wsg_cv_lib_fget_usable" = "yes"; then
    AC_DEFINE(HAVE_LIBFGET, 1,
	      [Define if your system has a current version of libfget])
  else
    LIBS="$wsg_old_libs";
  fi
])


dnl ##########################################################################
dnl ###  WSG_LIB_WRAP - check for TCP Wrappers library
dnl ##########################################################################

AC_DEFUN(WSG_LIB_WRAP, [
  AC_CHECK_HEADERS(tcpd.h)
  if test "$ac_cv_header_tcpd_h" = "yes"; then
    wsg_old_libs="$LIBS"
    LIBS="$LIBS -lwrap"
    AC_CACHE_CHECK([for libwrap library], wsg_cv_lib_wrap_hosts_ctl,
      AC_TRY_LINK(
        [
          #include <stdio.h>
          #include <tcpd.h>
          int allow_severity;
          int deny_severity;
        ], [
          hosts_ctl("test", STRING_UNKNOWN, "10.0.0.1", STRING_UNKNOWN);
        ],
        wsg_cv_lib_wrap_hosts_ctl=yes,
        wsg_cv_lib_wrap_hosts_ctl=no
      )
    )
    if test "$wsg_cv_lib_wrap_hosts_ctl" = "yes"; then
      AC_DEFINE(HAVE_LIBWRAP, 1, [Define if you have libwrap])
    else
      LIBS="$wsg_old_libs"
    fi
  fi
])


dnl ##########################################################################
dnl ###  WSG_TYPE - check for arbitrary type in arbitrary header file(s)
dnl ##########################################################################

dnl ### Usage: WSG_TYPE(type_t, default, [ #include <hdr1.h> ... ])
AC_DEFUN(WSG_TYPE, [
  AC_CACHE_CHECK(if $1 is defined, wsg_cv_type_$1,
    AC_TRY_LINK($3,
      $1 x;,
      wsg_cv_type_$1=yes,
      wsg_cv_type_$1=no
    )
  )
  if test "$wsg_cv_type_$1" != "yes"; then
    AC_DEFINE($1, $2, Define if your system does not have $1)
  fi
])


dnl ##########################################################################
dnl ###  WSG_SHLIB - check how to build shared libraries (very rudimentary)
dnl ##########################################################################

dnl ### Usage: WSG_SHLIB([ #include <hdr1.h> ... ], [ func1(); ... ])
AC_DEFUN(WSG_SHLIB, [
  AC_CACHE_CHECK([how to compile shared libraries], wsg_cv_cflags_shlib,
    for flags in "-fpic" "+Z"; do
      wsg_old_cflags="$CFLAGS";
      CFLAGS="$CFLAGS $flags";
      AC_TRY_COMPILE($1, $2,
        wsg_cv_cflags_shlib="$flags",
        wsg_cv_cflags_shlib=no
      )
      CFLAGS="$wsg_old_cflags";
      if test "$wsg_cv_cflags_shlib" != "no"; then
	break;
      fi
    done
  )
  if test "$wsg_cv_cflags_shlib" != "no"; then
    SHLIB_CFLAGS="$wsg_cv_cflags_shlib";
  else
    SHLIB_CFLAGS="";
  fi
  AC_SUBST(SHLIB_CFLAGS)

  AC_CACHE_CHECK([how to link shared libraries], wsg_cv_ldflags_shlib,
    for flags in "-shared" "-G"; do
      wsg_old_cflags="$CFLAGS";
      wsg_old_ldflags="$LDFLAGS";
      CFLAGS="$CFLAGS $SHLIB_CFLAGS";
      LDFLAGS="$LDFLAGS $flags";
      AC_TRY_LINK($1, $2,
        wsg_cv_ldflags_shlib="$flags",
        wsg_cv_ldflags_shlib=no
      )
      LDFLAGS="$wsg_old_ldflags";
      CFLAGS="$wsg_old_cflags";
      if test "$wsg_cv_ldflags_shlib" != "no"; then
        break;
      fi
    done
  )
  if test "$wsg_cv_ldflags_shlib" != "no"; then
    SHLIB_LDFLAGS="$wsg_cv_ldflags_shlib";
  else
    SHLIB_LDFLAGS="";
  fi
  AC_SUBST(SHLIB_LDFLAGS)
])


