/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QString>

class QByteArray; /* Forward declaration. */

/*!
 * @brief Creates complete directory path.
 *
 * @param[in] dirPath Directory path to be created.
 * @return True if directory structure already exists or was successfully
 *     created, false on error.
 */
bool createDirRecursive(const QString &dirPath);

/*!
 * @brief Copy directory and its content.
 *
 * Example: copyDirRecursively("/foo/baz", "/bar") copies the directory
 *     '/foo/baz' and its content into '/bar'.
 *     The result is a directory '/bar/baz'.
 *
 * @param[in] fromDirPath Source directory.
 * @param[in] toPath Target path.
 * @param[in] deleteOnError Delete created files and directories when cannot
 *                          be entirely written.
 */
bool copyDirRecursively(const QString &fromDirPath, const QString &toPath,
    bool deleteOnError = false);

/*!
 * @brief Check if file with given path exists.
 *
 * @param[in] filePath Full file path.
 * @return Full path with non-conflicting file if file already exists,
 *     null string on error.
 */
QString nonconflictingFileName(QString filePath);

/*!
 * @brief Identifies the cause why a file could not be written.
 */
enum WriteFileState {
	WF_SUCCESS = 0, /*!< File was successfully created and written. */
	WF_CANNOT_CREATE, /*!< File could not be created. */
	WF_CANNOT_READ, /*!< File could not be read. */
	WF_CANNOT_WRITE_WHOLE, /*!< File could not be entirely written. */
	WF_ERROR /*!< Different error. */
};

/*!
 * @brief Create file and write data to it.
 *
 * @param[in] filePath      File path.
 * @param[in] data          Data to be written into file.
 * @param[in] deleteOnError Delete created file when cannot be entirely
 *                          written.
 * @return Status identifier.
 */
enum WriteFileState writeFile(const QString &filePath, const QByteArray &data,
    bool deleteOnError = false);

/*!
 * @brief Generates an auxiliary file name.
 *
 * @note repeated calls with same \a filePath may result in different returned
 *     paths - depending on the time change.
 *
 * @param[in] filePath Original file name.
 * @rerturn New file name consisting of concatenated original file name and a
 *     time-based suffix.
 */
QString auxConfFilePath(const QString &filePath);
