/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QDialog>
#include <QMap>

#include "src/datovka_shared/identifiers/account_id.h"
#include "src/datovka_shared/utility/network_speed.h"
#include "src/worker/task_authenticate_message.h"

class TaskAuthenticateMessage; /* Forward declaration. */

namespace Ui {
	class DlgMessageCheck;
}

/*!
 * @brief Dialogue for the back-up progress visualisation.
 */
class DlgMessageCheck : public QDialog {
	Q_OBJECT

private:
	/*!
	 * @brief Constructor.
	 *
	 * @param[in] parent Parent object.
	 */
	DlgMessageCheck(QWidget *parent = Q_NULLPTR);

public:
	/*!
	 * @brief Destructor.
	 */
	virtual
	~DlgMessageCheck(void);

	/*!
	 * @brief Authenticate file.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[in] fileName Name of file containing message data.
	 * @param[in] sizeType Determines how the message is processed, normal or VoDZ.
	 */
	static
	void authenticateMessage(const AcntId &acntId, const QString &fileName,
	    enum TaskAuthenticateMessage::MsgSizeType sizeType,
	    QWidget *parent = Q_NULLPTR);

	/*!
	 * @brief Authenticate data.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[in] data Message data.
	 * @param[in] sizeType Determines how the message is processed, normal or VoDZ.
	 */
	static
	void authenticateMessage(const AcntId &acntId, const QByteArray &data,
	    enum TaskAuthenticateMessage::MsgSizeType sizeType,
	    QWidget *parent = Q_NULLPTR);

public Q_SLOTS:
	/*!
	 * @brief Check if a worker is working on the background and show
	 *     a dialogue whether the user wants to close this dialogue.
	 *
	 * @note Used instead of closeEvent().
	 *     https://stackoverflow.com/questions/13129380/show-a-dialog-before-closing-the-program-in-qt
	 *
	 * @param[in] resultCode Dialog result code.
	 */
	virtual
	void done(int resultCode) Q_DECL_OVERRIDE;

private Q_SLOTS:
/* Handle worker signals. */
	/*!
	 * @brief Set upload progress.
	 */
	void watchUploadProgress(const AcntId &acntId,
	    const QString &transactId, qint64 uploadTotal,
	    qint64 uploadCurrent);

	/*!
	 * @brief Hide upload progress.
	 */
	void watchUploadProgressFinished(const AcntId &acntId,
	    const QString &transactId, int result, const QString &resultDesc,
	    const QVariant &resultVal);

	/*!
	 * @brief Set download progress.
	 */
	void watchDownloadProgress(const AcntId &acntId,
	    const QString &transactId, const MsgId &msgId,
	    qint64 downloadTotal, qint64 downloadCurrent);

	/*!
	 * @brief Set result.
	 */
	void watchDownloadProgressFinished(const AcntId &acntId,
	    const QString &transactId, const MsgId &msgId,
	    int result, const QString &resultDesc);

private:
	/*!
	 * @brief Initialise the dialogue.
	 */
	void initDialogue(void);

	Ui::DlgMessageCheck *m_ui; /*!< UI generated from UI file. */

	QMap<QString, TaskAuthenticateMessage *> m_tasks; /*!< Created tasks. */

	QMap<QString, NetworkSpeed> m_speeds; /* Network speeds. */
};
