#include "dnsdist-rust-bridge.hh"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) & noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
template <typename T, typename = void *>
struct operator_new {
  void *operator()(::std::size_t sz) { return ::operator new(sz); }
};

template <typename T>
struct operator_new<T, decltype(T::operator new(sizeof(T)))> {
  void *operator()(::std::size_t sz) { return T::operator new(sz); }
};

class Fail final {
  ::rust::repr::PtrLen &throw$;
public:
  Fail(::rust::repr::PtrLen &throw$) noexcept : throw$(throw$) {}
  void operator()(char const *) noexcept;
  void operator()(std::string const &) noexcept;
};
} // namespace detail

template <typename T>
union MaybeUninit {
  T value;
  void *operator new(::std::size_t sz) { return detail::operator_new<T>{}(sz); }
  MaybeUninit() {}
  ~MaybeUninit() {}
};

namespace {
template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};
} // namespace
} // namespace cxxbridge1

namespace behavior {
class missing {};
missing trycatch(...);

template <typename Try, typename Fail>
static typename ::std::enable_if<
    ::std::is_same<decltype(trycatch(::std::declval<Try>(), ::std::declval<Fail>())),
                 missing>::value>::type
trycatch(Try &&func, Fail &&fail) noexcept try {
  func();
} catch (::std::exception const &e) {
  fail(e.what());
}
} // namespace behavior
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace dnsdist {
  namespace rust {
    namespace settings {
      struct ResponseConfig;
      struct SOAParams;
      struct SVCRecordAdditionalParams;
      struct SVCRecordParameters;
      struct SharedDNSSelector;
      struct SharedDNSAction;
      struct SharedDNSResponseAction;
      struct AllowActionConfiguration;
      struct ContinueActionConfiguration;
      struct DelayActionConfiguration;
      struct DnstapLogActionConfiguration;
      struct DropActionConfiguration;
      struct SetEDNSOptionActionConfiguration;
      struct ERCodeActionConfiguration;
      struct HTTPStatusActionConfiguration;
      struct KeyValueStoreLookupActionConfiguration;
      struct KeyValueStoreRangeLookupActionConfiguration;
      struct LogActionConfiguration;
      struct LuaActionConfiguration;
      struct LuaFFIActionConfiguration;
      struct LuaFFIPerThreadActionConfiguration;
      struct NegativeAndSOAActionConfiguration;
      struct NoneActionConfiguration;
      struct PoolActionConfiguration;
      struct QPSActionConfiguration;
      struct QPSPoolActionConfiguration;
      struct RCodeActionConfiguration;
      struct RemoteLogActionConfiguration;
      struct SetAdditionalProxyProtocolValueActionConfiguration;
      struct SetDisableECSActionConfiguration;
      struct SetDisableValidationActionConfiguration;
      struct SetECSActionConfiguration;
      struct SetECSOverrideActionConfiguration;
      struct SetECSPrefixLengthActionConfiguration;
      struct SetExtendedDNSErrorActionConfiguration;
      struct SetMacAddrActionConfiguration;
      struct SetMaxReturnedTTLActionConfiguration;
      struct SetNoRecurseActionConfiguration;
      struct SetProxyProtocolValuesActionConfiguration;
      struct SetSkipCacheActionConfiguration;
      struct SetTagActionConfiguration;
      struct SetTempFailureCacheTTLActionConfiguration;
      struct SNMPTrapActionConfiguration;
      struct SpoofActionConfiguration;
      struct SpoofCNAMEActionConfiguration;
      struct SpoofPacketActionConfiguration;
      struct SpoofRawActionConfiguration;
      struct SpoofSVCActionConfiguration;
      struct TCActionConfiguration;
      struct TeeActionConfiguration;
      struct AllowResponseActionConfiguration;
      struct ClearRecordTypesResponseActionConfiguration;
      struct DelayResponseActionConfiguration;
      struct DnstapLogResponseActionConfiguration;
      struct DropResponseActionConfiguration;
      struct LimitTTLResponseActionConfiguration;
      struct LogResponseActionConfiguration;
      struct LuaResponseActionConfiguration;
      struct LuaFFIResponseActionConfiguration;
      struct LuaFFIPerThreadResponseActionConfiguration;
      struct RemoteLogResponseActionConfiguration;
      struct SetEDNSOptionResponseActionConfiguration;
      struct SetExtendedDNSErrorResponseActionConfiguration;
      struct SetMaxReturnedTTLResponseActionConfiguration;
      struct SetMaxTTLResponseActionConfiguration;
      struct SetMinTTLResponseActionConfiguration;
      struct SetReducedTTLResponseActionConfiguration;
      struct SetSkipCacheResponseActionConfiguration;
      struct SetTagResponseActionConfiguration;
      struct SNMPTrapResponseActionConfiguration;
      struct TCResponseActionConfiguration;
      struct AllSelectorConfiguration;
      struct AndSelectorConfiguration;
      struct ByNameSelectorConfiguration;
      struct DNSSECSelectorConfiguration;
      struct DSTPortSelectorConfiguration;
      struct EDNSOptionSelectorConfiguration;
      struct EDNSVersionSelectorConfiguration;
      struct ERCodeSelectorConfiguration;
      struct HTTPHeaderSelectorConfiguration;
      struct HTTPPathSelectorConfiguration;
      struct HTTPPathRegexSelectorConfiguration;
      struct KeyValueStoreLookupSelectorConfiguration;
      struct KeyValueStoreRangeLookupSelectorConfiguration;
      struct LuaSelectorConfiguration;
      struct LuaFFISelectorConfiguration;
      struct LuaFFIPerThreadSelectorConfiguration;
      struct MaxQPSSelectorConfiguration;
      struct MaxQPSIPSelectorConfiguration;
      struct NetmaskGroupSelectorConfiguration;
      struct NotSelectorConfiguration;
      struct OpcodeSelectorConfiguration;
      struct OrSelectorConfiguration;
      struct PayloadSizeSelectorConfiguration;
      struct PoolAvailableSelectorConfiguration;
      struct PoolOutstandingSelectorConfiguration;
      struct ProbaSelectorConfiguration;
      struct ProxyProtocolValueSelectorConfiguration;
      struct QClassSelectorConfiguration;
      struct QNameSelectorConfiguration;
      struct QNameLabelsCountSelectorConfiguration;
      struct QNameSetSelectorConfiguration;
      struct QNameSuffixSelectorConfiguration;
      struct QNameWireLengthSelectorConfiguration;
      struct QTypeSelectorConfiguration;
      struct RCodeSelectorConfiguration;
      struct RDSelectorConfiguration;
      struct RE2SelectorConfiguration;
      struct RecordsCountSelectorConfiguration;
      struct RecordsTypeCountSelectorConfiguration;
      struct RegexSelectorConfiguration;
      struct SNISelectorConfiguration;
      struct TagSelectorConfiguration;
      struct TCPSelectorConfiguration;
      struct TrailingDataSelectorConfiguration;
      struct GlobalConfiguration;
      struct MetricsConfiguration;
      struct CarbonConfiguration;
      struct RemoteLoggingConfiguration;
      struct ProtobufLoggerConfiguration;
      struct DnstapLoggerConfiguration;
      struct ProtoBufMetaConfiguration;
      struct LmdbKvStoreConfiguration;
      struct CdbKvStoreConfiguration;
      struct KvsLookupKeySourceIpConfiguration;
      struct KvsLookupKeyQnameConfiguration;
      struct KvsLookupKeySuffixConfiguration;
      struct KvsLookupKeyTagConfiguration;
      struct KvsLookupKeysConfiguration;
      struct KeyValueStoresConfiguration;
      struct WebserverConfiguration;
      struct ConsoleConfiguration;
      struct EbpfMapConfiguration;
      struct EbpfConfiguration;
      struct EdnsClientSubnetConfiguration;
      struct DynamicRulesSettingsConfiguration;
      struct DynamicRuleConfiguration;
      struct DynamicRulesConfiguration;
      struct RingBuffersConfiguration;
      struct IncomingTlsCertificateKeyPairConfiguration;
      struct IncomingTlsConfiguration;
      struct OutgoingTlsConfiguration;
      struct HttpCustomResponseHeaderConfiguration;
      struct HttpResponsesMapConfiguration;
      struct IncomingDohConfiguration;
      struct IncomingDoqConfiguration;
      struct IncomingQuicConfiguration;
      struct IncomingDnscryptCertificateKeyPairConfiguration;
      struct IncomingDnscryptConfiguration;
      struct OutgoingDohConfiguration;
      struct IncomingTcpConfiguration;
      struct BindConfiguration;
      struct OutgoingTcpConfiguration;
      struct ProxyProtocolValueConfiguration;
      struct LazyHealthCheckConfiguration;
      struct HealthCheckConfiguration;
      struct OutgoingAutoUpgradeConfiguration;
      struct BackendConfiguration;
      struct TuningConfiguration;
      struct TcpTuningConfiguration;
      struct UdpTuningConfiguration;
      struct TlsEngineConfiguration;
      struct TlsTuningConfiguration;
      struct DohTuningConfiguration;
      struct CacheSettingsConfiguration;
      struct SecurityPollingConfiguration;
      struct StructuredLoggingConfiguration;
      struct LoggingConfiguration;
      struct GeneralConfiguration;
      struct NetmaskGroupConfiguration;
      struct PacketCacheConfiguration;
      struct ProxyProtocolConfiguration;
      struct SnmpConfiguration;
      struct QueryCountConfiguration;
      struct PoolConfiguration;
      struct CustomLoadBalancingPolicyConfiguration;
      struct LoadBalancingPoliciesConfiguration;
      struct QueryRuleConfiguration;
      struct ResponseRuleConfiguration;
      struct XskConfiguration;
      using DNSSelector = ::dnsdist::rust::settings::DNSSelector;
      using DNSActionWrapper = ::dnsdist::rust::settings::DNSActionWrapper;
      using DNSResponseActionWrapper = ::dnsdist::rust::settings::DNSResponseActionWrapper;
    }
  }
}

namespace dnsdist {
namespace rust {
namespace settings {
#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig
struct ResponseConfig final {
  bool set_aa CXX_DEFAULT_VALUE(false);
  bool set_ad CXX_DEFAULT_VALUE(false);
  bool set_ra CXX_DEFAULT_VALUE(false);
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);

  bool operator==(ResponseConfig const &) const noexcept;
  bool operator!=(ResponseConfig const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams
struct SOAParams final {
  ::std::uint32_t serial CXX_DEFAULT_VALUE(0);
  ::std::uint32_t refresh CXX_DEFAULT_VALUE(0);
  ::std::uint32_t retry CXX_DEFAULT_VALUE(0);
  ::std::uint32_t expire CXX_DEFAULT_VALUE(0);
  ::std::uint32_t minimum CXX_DEFAULT_VALUE(0);

  bool operator==(SOAParams const &) const noexcept;
  bool operator!=(SOAParams const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams
struct SVCRecordAdditionalParams final {
  ::std::uint16_t key CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(SVCRecordAdditionalParams const &) const noexcept;
  bool operator!=(SVCRecordAdditionalParams const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters
struct SVCRecordParameters final {
  ::rust::Vec<::std::uint16_t> mandatory_params;
  ::rust::Vec<::rust::String> alpns;
  ::rust::Vec<::rust::String> ipv4_hints;
  ::rust::Vec<::rust::String> ipv6_hints;
  ::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> additional_params;
  ::rust::String target;
  ::std::uint16_t port CXX_DEFAULT_VALUE(0);
  ::std::uint16_t priority CXX_DEFAULT_VALUE(0);
  bool no_default_alpn CXX_DEFAULT_VALUE(false);

  bool operator==(SVCRecordParameters const &) const noexcept;
  bool operator!=(SVCRecordParameters const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector
struct SharedDNSSelector final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> selector;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction
struct SharedDNSAction final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction
struct SharedDNSResponseAction final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration
struct AllowActionConfiguration final {
  ::rust::String name;

  bool operator==(AllowActionConfiguration const &) const noexcept;
  bool operator!=(AllowActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration
struct ContinueActionConfiguration final {
  ::rust::String name;
  ::dnsdist::rust::settings::SharedDNSAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration
struct DelayActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t msec CXX_DEFAULT_VALUE(0);

  bool operator==(DelayActionConfiguration const &) const noexcept;
  bool operator!=(DelayActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration
struct DnstapLogActionConfiguration final {
  ::rust::String name;
  ::rust::String identity;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;

  bool operator==(DnstapLogActionConfiguration const &) const noexcept;
  bool operator!=(DnstapLogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration
struct DropActionConfiguration final {
  ::rust::String name;

  bool operator==(DropActionConfiguration const &) const noexcept;
  bool operator!=(DropActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration
struct SetEDNSOptionActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);
  ::rust::String data;

  bool operator==(SetEDNSOptionActionConfiguration const &) const noexcept;
  bool operator!=(SetEDNSOptionActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration
struct ERCodeActionConfiguration final {
  ::rust::String name;
  ::rust::String rcode;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(ERCodeActionConfiguration const &) const noexcept;
  bool operator!=(ERCodeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration
struct HTTPStatusActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t status CXX_DEFAULT_VALUE(0);
  ::rust::String body;
  ::rust::String content_type;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(HTTPStatusActionConfiguration const &) const noexcept;
  bool operator!=(HTTPStatusActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration
struct KeyValueStoreLookupActionConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;
  ::rust::String destination_tag;

  bool operator==(KeyValueStoreLookupActionConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreLookupActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration
struct KeyValueStoreRangeLookupActionConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;
  ::rust::String destination_tag;

  bool operator==(KeyValueStoreRangeLookupActionConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreRangeLookupActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration
struct LogActionConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  bool binary CXX_DEFAULT_VALUE(false);
  bool append CXX_DEFAULT_VALUE(false);
  bool buffered CXX_DEFAULT_VALUE(false);
  bool verbose_only CXX_DEFAULT_VALUE(false);
  bool include_timestamp CXX_DEFAULT_VALUE(false);

  bool operator==(LogActionConfiguration const &) const noexcept;
  bool operator!=(LogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration
struct LuaActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaActionConfiguration const &) const noexcept;
  bool operator!=(LuaActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration
struct LuaFFIActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFIActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration
struct LuaFFIPerThreadActionConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration
struct NegativeAndSOAActionConfiguration final {
  ::rust::String name;
  bool nxd CXX_DEFAULT_VALUE(false);
  ::rust::String zone;
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);
  ::rust::String mname;
  ::rust::String rname;
  ::dnsdist::rust::settings::SOAParams soa_parameters;
  bool soa_in_authority CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(NegativeAndSOAActionConfiguration const &) const noexcept;
  bool operator!=(NegativeAndSOAActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration
struct NoneActionConfiguration final {
  ::rust::String name;

  bool operator==(NoneActionConfiguration const &) const noexcept;
  bool operator!=(NoneActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration
struct PoolActionConfiguration final {
  ::rust::String name;
  ::rust::String pool_name;
  bool stop_processing CXX_DEFAULT_VALUE(false);

  bool operator==(PoolActionConfiguration const &) const noexcept;
  bool operator!=(PoolActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration
struct QPSActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t limit CXX_DEFAULT_VALUE(0);

  bool operator==(QPSActionConfiguration const &) const noexcept;
  bool operator!=(QPSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration
struct QPSPoolActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t limit CXX_DEFAULT_VALUE(0);
  ::rust::String pool_name;
  bool stop_processing CXX_DEFAULT_VALUE(false);

  bool operator==(QPSPoolActionConfiguration const &) const noexcept;
  bool operator!=(QPSPoolActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration
struct RCodeActionConfiguration final {
  ::rust::String name;
  ::rust::String rcode;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(RCodeActionConfiguration const &) const noexcept;
  bool operator!=(RCodeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration
struct RemoteLogActionConfiguration final {
  ::rust::String name;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;
  ::rust::String server_id;
  ::rust::String ip_encrypt_key;
  ::rust::Vec<::rust::String> export_tags;
  ::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> metas;

  bool operator==(RemoteLogActionConfiguration const &) const noexcept;
  bool operator!=(RemoteLogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration
struct SetAdditionalProxyProtocolValueActionConfiguration final {
  ::rust::String name;
  ::std::uint8_t proxy_type CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(SetAdditionalProxyProtocolValueActionConfiguration const &) const noexcept;
  bool operator!=(SetAdditionalProxyProtocolValueActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration
struct SetDisableECSActionConfiguration final {
  ::rust::String name;

  bool operator==(SetDisableECSActionConfiguration const &) const noexcept;
  bool operator!=(SetDisableECSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration
struct SetDisableValidationActionConfiguration final {
  ::rust::String name;

  bool operator==(SetDisableValidationActionConfiguration const &) const noexcept;
  bool operator!=(SetDisableValidationActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration
struct SetECSActionConfiguration final {
  ::rust::String name;
  ::rust::String ipv4;
  ::rust::String ipv6;

  bool operator==(SetECSActionConfiguration const &) const noexcept;
  bool operator!=(SetECSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration
struct SetECSOverrideActionConfiguration final {
  ::rust::String name;
  bool override_existing CXX_DEFAULT_VALUE(false);

  bool operator==(SetECSOverrideActionConfiguration const &) const noexcept;
  bool operator!=(SetECSOverrideActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration
struct SetECSPrefixLengthActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t ipv4 CXX_DEFAULT_VALUE(0);
  ::std::uint16_t ipv6 CXX_DEFAULT_VALUE(0);

  bool operator==(SetECSPrefixLengthActionConfiguration const &) const noexcept;
  bool operator!=(SetECSPrefixLengthActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration
struct SetExtendedDNSErrorActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t info_code CXX_DEFAULT_VALUE(0);
  ::rust::String extra_text;

  bool operator==(SetExtendedDNSErrorActionConfiguration const &) const noexcept;
  bool operator!=(SetExtendedDNSErrorActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration
struct SetMacAddrActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);

  bool operator==(SetMacAddrActionConfiguration const &) const noexcept;
  bool operator!=(SetMacAddrActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration
struct SetMaxReturnedTTLActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxReturnedTTLActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxReturnedTTLActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration
struct SetNoRecurseActionConfiguration final {
  ::rust::String name;

  bool operator==(SetNoRecurseActionConfiguration const &) const noexcept;
  bool operator!=(SetNoRecurseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration
struct SetProxyProtocolValuesActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> values;

  bool operator==(SetProxyProtocolValuesActionConfiguration const &) const noexcept;
  bool operator!=(SetProxyProtocolValuesActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration
struct SetSkipCacheActionConfiguration final {
  ::rust::String name;

  bool operator==(SetSkipCacheActionConfiguration const &) const noexcept;
  bool operator!=(SetSkipCacheActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration
struct SetTagActionConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;

  bool operator==(SetTagActionConfiguration const &) const noexcept;
  bool operator!=(SetTagActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration
struct SetTempFailureCacheTTLActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);

  bool operator==(SetTempFailureCacheTTLActionConfiguration const &) const noexcept;
  bool operator!=(SetTempFailureCacheTTLActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration
struct SNMPTrapActionConfiguration final {
  ::rust::String name;
  ::rust::String reason;

  bool operator==(SNMPTrapActionConfiguration const &) const noexcept;
  bool operator!=(SNMPTrapActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration
struct SpoofActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> ips;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofActionConfiguration const &) const noexcept;
  bool operator!=(SpoofActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration
struct SpoofCNAMEActionConfiguration final {
  ::rust::String name;
  ::rust::String cname;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofCNAMEActionConfiguration const &) const noexcept;
  bool operator!=(SpoofCNAMEActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration
struct SpoofPacketActionConfiguration final {
  ::rust::String name;
  ::rust::String response;
  ::std::uint32_t len CXX_DEFAULT_VALUE(0);

  bool operator==(SpoofPacketActionConfiguration const &) const noexcept;
  bool operator!=(SpoofPacketActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration
struct SpoofRawActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> answers;
  ::rust::String qtype_for_any;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofRawActionConfiguration const &) const noexcept;
  bool operator!=(SpoofRawActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration
struct SpoofSVCActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> parameters;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofSVCActionConfiguration const &) const noexcept;
  bool operator!=(SpoofSVCActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration
struct TCActionConfiguration final {
  ::rust::String name;

  bool operator==(TCActionConfiguration const &) const noexcept;
  bool operator!=(TCActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration
struct TeeActionConfiguration final {
  ::rust::String name;
  ::rust::String rca;
  ::rust::String lca;
  bool add_ecs CXX_DEFAULT_VALUE(false);
  bool add_proxy_protocol CXX_DEFAULT_VALUE(false);

  bool operator==(TeeActionConfiguration const &) const noexcept;
  bool operator!=(TeeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration
struct AllowResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(AllowResponseActionConfiguration const &) const noexcept;
  bool operator!=(AllowResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration
struct ClearRecordTypesResponseActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::std::uint16_t> types;

  bool operator==(ClearRecordTypesResponseActionConfiguration const &) const noexcept;
  bool operator!=(ClearRecordTypesResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration
struct DelayResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t msec CXX_DEFAULT_VALUE(0);

  bool operator==(DelayResponseActionConfiguration const &) const noexcept;
  bool operator!=(DelayResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration
struct DnstapLogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String identity;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;

  bool operator==(DnstapLogResponseActionConfiguration const &) const noexcept;
  bool operator!=(DnstapLogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration
struct DropResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(DropResponseActionConfiguration const &) const noexcept;
  bool operator!=(DropResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration
struct LimitTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t min CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::std::uint16_t> types;

  bool operator==(LimitTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(LimitTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration
struct LogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  bool append CXX_DEFAULT_VALUE(false);
  bool buffered CXX_DEFAULT_VALUE(false);
  bool verbose_only CXX_DEFAULT_VALUE(false);
  bool include_timestamp CXX_DEFAULT_VALUE(false);

  bool operator==(LogResponseActionConfiguration const &) const noexcept;
  bool operator!=(LogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration
struct LuaResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration
struct LuaFFIResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFIResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration
struct LuaFFIPerThreadResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration
struct RemoteLogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;
  ::rust::String server_id;
  ::rust::String ip_encrypt_key;
  bool include_cname CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> export_tags;
  ::rust::String export_extended_errors_to_meta;
  ::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> metas;

  bool operator==(RemoteLogResponseActionConfiguration const &) const noexcept;
  bool operator!=(RemoteLogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration
struct SetEDNSOptionResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);
  ::rust::String data;

  bool operator==(SetEDNSOptionResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetEDNSOptionResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration
struct SetExtendedDNSErrorResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t info_code CXX_DEFAULT_VALUE(0);
  ::rust::String extra_text;

  bool operator==(SetExtendedDNSErrorResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetExtendedDNSErrorResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration
struct SetMaxReturnedTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxReturnedTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxReturnedTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration
struct SetMaxTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration
struct SetMinTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t min CXX_DEFAULT_VALUE(0);

  bool operator==(SetMinTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMinTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration
struct SetReducedTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint8_t percentage CXX_DEFAULT_VALUE(0);

  bool operator==(SetReducedTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetReducedTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration
struct SetSkipCacheResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(SetSkipCacheResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetSkipCacheResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration
struct SetTagResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;

  bool operator==(SetTagResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetTagResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration
struct SNMPTrapResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String reason;

  bool operator==(SNMPTrapResponseActionConfiguration const &) const noexcept;
  bool operator!=(SNMPTrapResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration
struct TCResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(TCResponseActionConfiguration const &) const noexcept;
  bool operator!=(TCResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration
struct AllSelectorConfiguration final {
  ::rust::String name;

  bool operator==(AllSelectorConfiguration const &) const noexcept;
  bool operator!=(AllSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration
struct AndSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration
struct ByNameSelectorConfiguration final {
  ::rust::String name;
  ::rust::String selector_name;

  bool operator==(ByNameSelectorConfiguration const &) const noexcept;
  bool operator!=(ByNameSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration
struct DNSSECSelectorConfiguration final {
  ::rust::String name;

  bool operator==(DNSSECSelectorConfiguration const &) const noexcept;
  bool operator!=(DNSSECSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration
struct DSTPortSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t port CXX_DEFAULT_VALUE(0);

  bool operator==(DSTPortSelectorConfiguration const &) const noexcept;
  bool operator!=(DSTPortSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration
struct EDNSOptionSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t option_code CXX_DEFAULT_VALUE(0);

  bool operator==(EDNSOptionSelectorConfiguration const &) const noexcept;
  bool operator!=(EDNSOptionSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration
struct EDNSVersionSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t version CXX_DEFAULT_VALUE(0);

  bool operator==(EDNSVersionSelectorConfiguration const &) const noexcept;
  bool operator!=(EDNSVersionSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration
struct ERCodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String rcode;

  bool operator==(ERCodeSelectorConfiguration const &) const noexcept;
  bool operator!=(ERCodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration
struct HTTPHeaderSelectorConfiguration final {
  ::rust::String name;
  ::rust::String header;
  ::rust::String expression;

  bool operator==(HTTPHeaderSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPHeaderSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration
struct HTTPPathSelectorConfiguration final {
  ::rust::String name;
  ::rust::String path;

  bool operator==(HTTPPathSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPPathSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration
struct HTTPPathRegexSelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(HTTPPathRegexSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPPathRegexSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration
struct KeyValueStoreLookupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;

  bool operator==(KeyValueStoreLookupSelectorConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreLookupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration
struct KeyValueStoreRangeLookupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;

  bool operator==(KeyValueStoreRangeLookupSelectorConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreRangeLookupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration
struct LuaSelectorConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaSelectorConfiguration const &) const noexcept;
  bool operator!=(LuaSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration
struct LuaFFISelectorConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFISelectorConfiguration const &) const noexcept;
  bool operator!=(LuaFFISelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration
struct LuaFFIPerThreadSelectorConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadSelectorConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration
struct MaxQPSSelectorConfiguration final {
  ::rust::String name;
  ::std::uint32_t qps CXX_DEFAULT_VALUE(0);
  ::std::uint32_t burst CXX_DEFAULT_VALUE(0);

  bool operator==(MaxQPSSelectorConfiguration const &) const noexcept;
  bool operator!=(MaxQPSSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration
struct MaxQPSIPSelectorConfiguration final {
  ::rust::String name;
  ::std::uint32_t qps CXX_DEFAULT_VALUE(0);
  ::std::uint8_t ipv4_mask CXX_DEFAULT_VALUE(0);
  ::std::uint8_t ipv6_mask CXX_DEFAULT_VALUE(0);
  ::std::uint32_t burst CXX_DEFAULT_VALUE(0);
  ::std::uint32_t expiration CXX_DEFAULT_VALUE(0);
  ::std::uint32_t cleanup_delay CXX_DEFAULT_VALUE(0);
  ::std::uint32_t scan_fraction CXX_DEFAULT_VALUE(0);
  ::std::uint32_t shards CXX_DEFAULT_VALUE(0);

  bool operator==(MaxQPSIPSelectorConfiguration const &) const noexcept;
  bool operator!=(MaxQPSIPSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration
struct NetmaskGroupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String netmask_group_name;
  ::rust::Vec<::rust::String> netmasks;
  bool source CXX_DEFAULT_VALUE(false);
  bool quiet CXX_DEFAULT_VALUE(false);

  bool operator==(NetmaskGroupSelectorConfiguration const &) const noexcept;
  bool operator!=(NetmaskGroupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration
struct NotSelectorConfiguration final {
  ::rust::String name;
  ::dnsdist::rust::settings::SharedDNSSelector selector;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration
struct OpcodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(OpcodeSelectorConfiguration const &) const noexcept;
  bool operator!=(OpcodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration
struct OrSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration
struct PayloadSizeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String comparison;
  ::std::uint16_t size CXX_DEFAULT_VALUE(0);

  bool operator==(PayloadSizeSelectorConfiguration const &) const noexcept;
  bool operator!=(PayloadSizeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration
struct PoolAvailableSelectorConfiguration final {
  ::rust::String name;
  ::rust::String pool;

  bool operator==(PoolAvailableSelectorConfiguration const &) const noexcept;
  bool operator!=(PoolAvailableSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration
struct PoolOutstandingSelectorConfiguration final {
  ::rust::String name;
  ::rust::String pool;
  ::std::uint64_t max_outstanding CXX_DEFAULT_VALUE(0);

  bool operator==(PoolOutstandingSelectorConfiguration const &) const noexcept;
  bool operator!=(PoolOutstandingSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration
struct ProbaSelectorConfiguration final {
  ::rust::String name;
  double probability CXX_DEFAULT_VALUE(0);

  bool operator==(ProbaSelectorConfiguration const &) const noexcept;
  bool operator!=(ProbaSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration
struct ProxyProtocolValueSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t option_type CXX_DEFAULT_VALUE(0);
  ::rust::String option_value;

  bool operator==(ProxyProtocolValueSelectorConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolValueSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration
struct QClassSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qclass;
  ::std::uint16_t numeric_value CXX_DEFAULT_VALUE(0);

  bool operator==(QClassSelectorConfiguration const &) const noexcept;
  bool operator!=(QClassSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration
struct QNameSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qname;

  bool operator==(QNameSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration
struct QNameLabelsCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t min_labels_count CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max_labels_count CXX_DEFAULT_VALUE(0);

  bool operator==(QNameLabelsCountSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameLabelsCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration
struct QNameSetSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> qnames;

  bool operator==(QNameSetSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSetSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration
struct QNameSuffixSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> suffixes;
  bool quiet CXX_DEFAULT_VALUE(false);

  bool operator==(QNameSuffixSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSuffixSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration
struct QNameWireLengthSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t min CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max CXX_DEFAULT_VALUE(0);

  bool operator==(QNameWireLengthSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameWireLengthSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration
struct QTypeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qtype;
  ::std::uint16_t numeric_value CXX_DEFAULT_VALUE(0);

  bool operator==(QTypeSelectorConfiguration const &) const noexcept;
  bool operator!=(QTypeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration
struct RCodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String rcode;

  bool operator==(RCodeSelectorConfiguration const &) const noexcept;
  bool operator!=(RCodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration
struct RDSelectorConfiguration final {
  ::rust::String name;

  bool operator==(RDSelectorConfiguration const &) const noexcept;
  bool operator!=(RDSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration
struct RE2SelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(RE2SelectorConfiguration const &) const noexcept;
  bool operator!=(RE2SelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration
struct RecordsCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t section CXX_DEFAULT_VALUE(0);
  ::std::uint16_t minimum CXX_DEFAULT_VALUE(0);
  ::std::uint16_t maximum CXX_DEFAULT_VALUE(0);

  bool operator==(RecordsCountSelectorConfiguration const &) const noexcept;
  bool operator!=(RecordsCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration
struct RecordsTypeCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t section CXX_DEFAULT_VALUE(0);
  ::std::uint16_t record_type CXX_DEFAULT_VALUE(0);
  ::std::uint16_t minimum CXX_DEFAULT_VALUE(0);
  ::std::uint16_t maximum CXX_DEFAULT_VALUE(0);

  bool operator==(RecordsTypeCountSelectorConfiguration const &) const noexcept;
  bool operator!=(RecordsTypeCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration
struct RegexSelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(RegexSelectorConfiguration const &) const noexcept;
  bool operator!=(RegexSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration
struct SNISelectorConfiguration final {
  ::rust::String name;
  ::rust::String server_name;

  bool operator==(SNISelectorConfiguration const &) const noexcept;
  bool operator!=(SNISelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration
struct TagSelectorConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;
  bool empty_as_wildcard CXX_DEFAULT_VALUE(false);

  bool operator==(TagSelectorConfiguration const &) const noexcept;
  bool operator!=(TagSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration
struct TCPSelectorConfiguration final {
  ::rust::String name;
  bool tcp CXX_DEFAULT_VALUE(false);

  bool operator==(TCPSelectorConfiguration const &) const noexcept;
  bool operator!=(TCPSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration
struct TrailingDataSelectorConfiguration final {
  ::rust::String name;

  bool operator==(TrailingDataSelectorConfiguration const &) const noexcept;
  bool operator!=(TrailingDataSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration
struct CacheSettingsConfiguration final {
  ::std::uint32_t stale_entries_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint16_t cleaning_delay CXX_DEFAULT_VALUE(0);
  ::std::uint16_t cleaning_percentage CXX_DEFAULT_VALUE(0);

  bool operator==(CacheSettingsConfiguration const &) const noexcept;
  bool operator!=(CacheSettingsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration
struct ConsoleConfiguration final {
  ::rust::String listen_address;
  ::rust::String key;
  ::rust::Vec<::rust::String> acl;
  ::std::uint32_t maximum_output_size CXX_DEFAULT_VALUE(0);
  bool log_connections CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_concurrent_connections CXX_DEFAULT_VALUE(0);

  bool operator==(ConsoleConfiguration const &) const noexcept;
  bool operator!=(ConsoleConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration
struct DynamicRulesSettingsConfiguration final {
  ::std::uint64_t purge_interval CXX_DEFAULT_VALUE(0);
  ::rust::String default_action;

  bool operator==(DynamicRulesSettingsConfiguration const &) const noexcept;
  bool operator!=(DynamicRulesSettingsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration
struct EbpfMapConfiguration final {
  ::std::uint32_t max_entries CXX_DEFAULT_VALUE(0);
  ::rust::String pinned_path;

  bool operator==(EbpfMapConfiguration const &) const noexcept;
  bool operator!=(EbpfMapConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration
struct EbpfConfiguration final {
  ::dnsdist::rust::settings::EbpfMapConfiguration ipv4;
  ::dnsdist::rust::settings::EbpfMapConfiguration ipv6;
  ::dnsdist::rust::settings::EbpfMapConfiguration cidr_ipv4;
  ::dnsdist::rust::settings::EbpfMapConfiguration cidr_ipv6;
  ::dnsdist::rust::settings::EbpfMapConfiguration qnames;
  bool external CXX_DEFAULT_VALUE(false);

  bool operator==(EbpfConfiguration const &) const noexcept;
  bool operator!=(EbpfConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration
struct EdnsClientSubnetConfiguration final {
  bool override_existing CXX_DEFAULT_VALUE(false);
  ::std::uint8_t source_prefix_v4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t source_prefix_v6 CXX_DEFAULT_VALUE(0);

  bool operator==(EdnsClientSubnetConfiguration const &) const noexcept;
  bool operator!=(EdnsClientSubnetConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration
struct GeneralConfiguration final {
  ::std::uint16_t edns_udp_payload_size_self_generated_answers CXX_DEFAULT_VALUE(0);
  bool add_edns_to_self_generated_answers CXX_DEFAULT_VALUE(false);
  bool truncate_tc_answers CXX_DEFAULT_VALUE(false);
  bool fixup_case CXX_DEFAULT_VALUE(false);
  bool allow_empty_responses CXX_DEFAULT_VALUE(false);
  bool drop_empty_queries CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> capabilities_to_retain;

  bool operator==(GeneralConfiguration const &) const noexcept;
  bool operator!=(GeneralConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration
struct KvsLookupKeysConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> source_ip_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> qname_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> suffix_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> tag_keys;

  bool operator==(KvsLookupKeysConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeysConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration
struct KeyValueStoresConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> lmdb;
  ::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> cdb;
  ::dnsdist::rust::settings::KvsLookupKeysConfiguration lookup_keys;

  bool operator==(KeyValueStoresConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoresConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration
struct LoadBalancingPoliciesConfiguration final {
  ::rust::String default_policy;
  bool servfail_on_no_server CXX_DEFAULT_VALUE(false);
  bool round_robin_servfail_on_no_server CXX_DEFAULT_VALUE(false);
  double weighted_balancing_factor CXX_DEFAULT_VALUE(0);
  double consistent_hashing_balancing_factor CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> custom_policies;
  ::std::uint32_t hash_perturbation CXX_DEFAULT_VALUE(0);

  bool operator==(LoadBalancingPoliciesConfiguration const &) const noexcept;
  bool operator!=(LoadBalancingPoliciesConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration
struct StructuredLoggingConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::rust::String level_prefix;
  ::rust::String time_format;

  bool operator==(StructuredLoggingConfiguration const &) const noexcept;
  bool operator!=(StructuredLoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration
struct LoggingConfiguration final {
  bool verbose CXX_DEFAULT_VALUE(false);
  bool verbose_health_checks CXX_DEFAULT_VALUE(false);
  ::rust::String verbose_log_destination;
  ::rust::String syslog_facility;
  ::dnsdist::rust::settings::StructuredLoggingConfiguration structured;

  bool operator==(LoggingConfiguration const &) const noexcept;
  bool operator!=(LoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration
struct MetricsConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> carbon;

  bool operator==(MetricsConfiguration const &) const noexcept;
  bool operator!=(MetricsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration
struct ProxyProtocolConfiguration final {
  ::rust::Vec<::rust::String> acl;
  ::std::uint32_t maximum_payload_size CXX_DEFAULT_VALUE(0);
  bool apply_acl_to_proxied_clients CXX_DEFAULT_VALUE(false);

  bool operator==(ProxyProtocolConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration
struct QueryCountConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::rust::String filter_function_name;
  ::rust::String filter_function_code;
  ::rust::String filter_function_file;

  bool operator==(QueryCountConfiguration const &) const noexcept;
  bool operator!=(QueryCountConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration
struct RemoteLoggingConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> protobuf_loggers;
  ::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> dnstap_loggers;

  bool operator==(RemoteLoggingConfiguration const &) const noexcept;
  bool operator!=(RemoteLoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration
struct RingBuffersConfiguration final {
  ::std::uint64_t size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t shards CXX_DEFAULT_VALUE(0);
  ::std::uint64_t lock_retries CXX_DEFAULT_VALUE(0);
  bool record_queries CXX_DEFAULT_VALUE(false);
  bool record_responses CXX_DEFAULT_VALUE(false);

  bool operator==(RingBuffersConfiguration const &) const noexcept;
  bool operator!=(RingBuffersConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration
struct SecurityPollingConfiguration final {
  ::std::uint32_t polling_interval CXX_DEFAULT_VALUE(0);
  ::rust::String suffix;

  bool operator==(SecurityPollingConfiguration const &) const noexcept;
  bool operator!=(SecurityPollingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration
struct SnmpConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  bool traps_enabled CXX_DEFAULT_VALUE(false);
  ::rust::String daemon_socket;

  bool operator==(SnmpConfiguration const &) const noexcept;
  bool operator!=(SnmpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration
struct DohTuningConfiguration final {
  ::std::uint32_t outgoing_worker_threads CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_cleanup_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_connection_per_backend CXX_DEFAULT_VALUE(0);

  bool operator==(DohTuningConfiguration const &) const noexcept;
  bool operator!=(DohTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration
struct TcpTuningConfiguration final {
  ::std::uint32_t worker_threads CXX_DEFAULT_VALUE(0);
  ::std::uint32_t receive_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint32_t send_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queries_per_connection CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_connection_duration CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queued_connections CXX_DEFAULT_VALUE(0);
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_cleanup_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_connection_per_backend CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_connections_per_client CXX_DEFAULT_VALUE(0);
  ::rust::String fast_open_key;
  ::std::uint8_t connections_overload_threshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_connection_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_rate_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tls_new_session_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tls_resumed_session_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_read_ios_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint32_t ban_duration_for_exceeding_max_read_ios_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint32_t ban_duration_for_exceeding_tcp_tls_rate CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v6 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v4_port CXX_DEFAULT_VALUE(0);

  bool operator==(TcpTuningConfiguration const &) const noexcept;
  bool operator!=(TcpTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration
struct TlsTuningConfiguration final {
  ::std::uint16_t outgoing_tickets_cache_cleanup_delay CXX_DEFAULT_VALUE(0);
  ::std::uint16_t outgoing_tickets_cache_validity CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max_outgoing_tickets_per_backend CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> providers;
  ::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> engines;

  bool operator==(TlsTuningConfiguration const &) const noexcept;
  bool operator!=(TlsTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration
struct UdpTuningConfiguration final {
  ::std::uint32_t messages_per_round CXX_DEFAULT_VALUE(0);
  ::std::uint32_t send_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t receive_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_outstanding_per_backend CXX_DEFAULT_VALUE(0);
  ::std::uint8_t timeout CXX_DEFAULT_VALUE(0);
  bool randomize_outgoing_sockets_to_backend CXX_DEFAULT_VALUE(false);
  bool randomize_ids_to_backend CXX_DEFAULT_VALUE(false);

  bool operator==(UdpTuningConfiguration const &) const noexcept;
  bool operator!=(UdpTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration
struct TuningConfiguration final {
  ::dnsdist::rust::settings::DohTuningConfiguration doh;
  ::dnsdist::rust::settings::TcpTuningConfiguration tcp;
  ::dnsdist::rust::settings::TlsTuningConfiguration tls;
  ::dnsdist::rust::settings::UdpTuningConfiguration udp;

  bool operator==(TuningConfiguration const &) const noexcept;
  bool operator!=(TuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration
struct WebserverConfiguration final {
  ::rust::Vec<::rust::String> listen_addresses;
  ::rust::String password;
  ::rust::String api_key;
  ::rust::Vec<::rust::String> acl;
  bool api_requires_authentication CXX_DEFAULT_VALUE(false);
  bool stats_require_authentication CXX_DEFAULT_VALUE(false);
  bool dashboard_requires_authentication CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_concurrent_connections CXX_DEFAULT_VALUE(0);
  bool hash_plaintext_credentials CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> custom_headers;
  ::rust::String api_configuration_directory;
  bool api_read_write CXX_DEFAULT_VALUE(false);

  bool operator==(WebserverConfiguration const &) const noexcept;
  bool operator!=(WebserverConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration
struct GlobalConfiguration final {
  ::rust::Vec<::rust::String> acl;
  ::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> backends;
  ::rust::Vec<::dnsdist::rust::settings::BindConfiguration> binds;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> cache_hit_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> cache_inserted_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> cache_miss_rules;
  ::dnsdist::rust::settings::CacheSettingsConfiguration cache_settings;
  ::dnsdist::rust::settings::ConsoleConfiguration console;
  ::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> dynamic_rules;
  ::dnsdist::rust::settings::DynamicRulesSettingsConfiguration dynamic_rules_settings;
  ::dnsdist::rust::settings::EbpfConfiguration ebpf;
  ::dnsdist::rust::settings::EdnsClientSubnetConfiguration edns_client_subnet;
  ::dnsdist::rust::settings::GeneralConfiguration general;
  ::dnsdist::rust::settings::KeyValueStoresConfiguration key_value_stores;
  ::dnsdist::rust::settings::LoadBalancingPoliciesConfiguration load_balancing_policies;
  ::dnsdist::rust::settings::LoggingConfiguration logging;
  ::dnsdist::rust::settings::MetricsConfiguration metrics;
  ::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> netmask_groups;
  ::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> packet_caches;
  ::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> pools;
  ::dnsdist::rust::settings::ProxyProtocolConfiguration proxy_protocol;
  ::dnsdist::rust::settings::QueryCountConfiguration query_count;
  ::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> query_rules;
  ::dnsdist::rust::settings::RemoteLoggingConfiguration remote_logging;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> response_rules;
  ::dnsdist::rust::settings::RingBuffersConfiguration ring_buffers;
  ::dnsdist::rust::settings::SecurityPollingConfiguration security_polling;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> self_answered_response_rules;
  ::dnsdist::rust::settings::SnmpConfiguration snmp;
  ::dnsdist::rust::settings::TuningConfiguration tuning;
  ::dnsdist::rust::settings::WebserverConfiguration webserver;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> xfr_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::XskConfiguration> xsk;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> timeout_response_rules;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration
struct CarbonConfiguration final {
  ::rust::String address;
  ::rust::String name;
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  ::rust::String name_space;
  ::rust::String instance;

  bool operator==(CarbonConfiguration const &) const noexcept;
  bool operator!=(CarbonConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration
struct ProtobufLoggerConfiguration final {
  ::rust::String name;
  ::rust::String address;
  ::std::uint16_t timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queued_entries CXX_DEFAULT_VALUE(0);
  ::std::uint8_t reconnect_wait_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_count CXX_DEFAULT_VALUE(0);

  bool operator==(ProtobufLoggerConfiguration const &) const noexcept;
  bool operator!=(ProtobufLoggerConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration
struct DnstapLoggerConfiguration final {
  ::rust::String name;
  ::rust::String transport;
  ::rust::String address;
  ::std::uint64_t buffer_hint CXX_DEFAULT_VALUE(0);
  ::std::uint64_t flush_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t input_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t output_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t queue_notify_threshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t reopen_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_count CXX_DEFAULT_VALUE(0);

  bool operator==(DnstapLoggerConfiguration const &) const noexcept;
  bool operator!=(DnstapLoggerConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration
struct ProtoBufMetaConfiguration final {
  ::rust::String key;
  ::rust::String value;

  bool operator==(ProtoBufMetaConfiguration const &) const noexcept;
  bool operator!=(ProtoBufMetaConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration
struct LmdbKvStoreConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  ::rust::String database_name;
  bool no_lock CXX_DEFAULT_VALUE(false);

  bool operator==(LmdbKvStoreConfiguration const &) const noexcept;
  bool operator!=(LmdbKvStoreConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration
struct CdbKvStoreConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  ::std::uint32_t refresh_delay CXX_DEFAULT_VALUE(0);

  bool operator==(CdbKvStoreConfiguration const &) const noexcept;
  bool operator!=(CdbKvStoreConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration
struct KvsLookupKeySourceIpConfiguration final {
  ::rust::String name;
  ::std::uint8_t v4_mask CXX_DEFAULT_VALUE(0);
  ::std::uint8_t v6_mask CXX_DEFAULT_VALUE(0);
  bool include_port CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeySourceIpConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeySourceIpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration
struct KvsLookupKeyQnameConfiguration final {
  ::rust::String name;
  bool wire_format CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeyQnameConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeyQnameConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration
struct KvsLookupKeySuffixConfiguration final {
  ::rust::String name;
  ::std::uint16_t minimum_labels CXX_DEFAULT_VALUE(0);
  bool wire_format CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeySuffixConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeySuffixConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration
struct KvsLookupKeyTagConfiguration final {
  ::rust::String name;
  ::rust::String tag;

  bool operator==(KvsLookupKeyTagConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeyTagConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration
struct DynamicRuleConfiguration final {
  ::rust::String rule_type;
  ::std::uint32_t seconds CXX_DEFAULT_VALUE(0);
  ::std::uint32_t action_duration CXX_DEFAULT_VALUE(0);
  ::rust::String comment;
  ::std::uint32_t rate CXX_DEFAULT_VALUE(0);
  double ratio CXX_DEFAULT_VALUE(0);
  ::rust::String action;
  ::std::uint32_t warning_rate CXX_DEFAULT_VALUE(0);
  double warning_ratio CXX_DEFAULT_VALUE(0);
  ::rust::String tag_name;
  ::rust::String tag_value;
  ::rust::String visitor_function_name;
  ::rust::String visitor_function_code;
  ::rust::String visitor_function_file;
  ::rust::String rcode;
  ::rust::String qtype;
  ::std::uint32_t minimum_number_of_responses CXX_DEFAULT_VALUE(0);
  double minimum_global_cache_hit_ratio CXX_DEFAULT_VALUE(0);

  bool operator==(DynamicRuleConfiguration const &) const noexcept;
  bool operator!=(DynamicRuleConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration
struct DynamicRulesConfiguration final {
  ::rust::String name;
  ::std::uint8_t mask_ipv4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t mask_ipv6 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t mask_port CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> exclude_ranges;
  ::rust::Vec<::rust::String> include_ranges;
  ::rust::Vec<::rust::String> exclude_domains;
  ::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> rules;

  bool operator==(DynamicRulesConfiguration const &) const noexcept;
  bool operator!=(DynamicRulesConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration
struct IncomingTlsCertificateKeyPairConfiguration final {
  ::rust::String certificate;
  ::rust::String key;
  ::rust::String password;

  bool operator==(IncomingTlsCertificateKeyPairConfiguration const &) const noexcept;
  bool operator!=(IncomingTlsCertificateKeyPairConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration
struct IncomingTlsConfiguration final {
  ::rust::String provider;
  ::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> certificates;
  ::rust::String ciphers;
  ::rust::String ciphers_tls_13;
  ::rust::String minimum_version;
  ::rust::String ticket_key_file;
  ::std::uint32_t tickets_keys_rotation_delay CXX_DEFAULT_VALUE(0);
  ::std::uint32_t number_of_tickets_keys CXX_DEFAULT_VALUE(0);
  bool prefer_server_ciphers CXX_DEFAULT_VALUE(false);
  ::std::uint32_t session_timeout CXX_DEFAULT_VALUE(0);
  bool session_tickets CXX_DEFAULT_VALUE(false);
  ::std::uint32_t number_of_stored_sessions CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> ocsp_response_files;
  ::rust::String key_log_file;
  bool release_buffers CXX_DEFAULT_VALUE(false);
  bool enable_renegotiation CXX_DEFAULT_VALUE(false);
  bool async_mode CXX_DEFAULT_VALUE(false);
  bool ktls CXX_DEFAULT_VALUE(false);
  bool read_ahead CXX_DEFAULT_VALUE(false);
  bool proxy_protocol_outside_tls CXX_DEFAULT_VALUE(false);
  bool ignore_configuration_errors CXX_DEFAULT_VALUE(false);

  bool operator==(IncomingTlsConfiguration const &) const noexcept;
  bool operator!=(IncomingTlsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration
struct OutgoingTlsConfiguration final {
  ::rust::String provider;
  ::rust::String subject_name;
  ::rust::String subject_address;
  bool validate_certificate CXX_DEFAULT_VALUE(false);
  ::rust::String ca_store;
  ::rust::String ciphers;
  ::rust::String ciphers_tls_13;
  ::rust::String key_log_file;
  bool release_buffers CXX_DEFAULT_VALUE(false);
  bool enable_renegotiation CXX_DEFAULT_VALUE(false);
  bool ktls CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingTlsConfiguration const &) const noexcept;
  bool operator!=(OutgoingTlsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration
struct HttpCustomResponseHeaderConfiguration final {
  ::rust::String key;
  ::rust::String value;

  bool operator==(HttpCustomResponseHeaderConfiguration const &) const noexcept;
  bool operator!=(HttpCustomResponseHeaderConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration
struct HttpResponsesMapConfiguration final {
  ::rust::String expression;
  ::std::uint16_t status CXX_DEFAULT_VALUE(0);
  ::rust::String content;
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> headers;

  bool operator==(HttpResponsesMapConfiguration const &) const noexcept;
  bool operator!=(HttpResponsesMapConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration
struct IncomingDohConfiguration final {
  ::rust::String provider;
  ::rust::Vec<::rust::String> paths;
  ::std::uint64_t idle_timeout CXX_DEFAULT_VALUE(0);
  ::rust::String server_tokens;
  bool send_cache_control_headers CXX_DEFAULT_VALUE(false);
  bool keep_incoming_headers CXX_DEFAULT_VALUE(false);
  bool trust_forwarded_for_header CXX_DEFAULT_VALUE(false);
  bool early_acl_drop CXX_DEFAULT_VALUE(false);
  bool exact_path_matching CXX_DEFAULT_VALUE(false);
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> custom_response_headers;
  ::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> responses_map;

  bool operator==(IncomingDohConfiguration const &) const noexcept;
  bool operator!=(IncomingDohConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration
struct IncomingDoqConfiguration final {
  ::std::uint64_t max_concurrent_queries_per_connection CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingDoqConfiguration const &) const noexcept;
  bool operator!=(IncomingDoqConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration
struct IncomingQuicConfiguration final {
  ::std::uint64_t idle_timeout CXX_DEFAULT_VALUE(0);
  ::rust::String congestion_control_algorithm;
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingQuicConfiguration const &) const noexcept;
  bool operator!=(IncomingQuicConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration
struct IncomingDnscryptCertificateKeyPairConfiguration final {
  ::rust::String certificate;
  ::rust::String key;

  bool operator==(IncomingDnscryptCertificateKeyPairConfiguration const &) const noexcept;
  bool operator!=(IncomingDnscryptCertificateKeyPairConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration
struct IncomingDnscryptConfiguration final {
  ::rust::String provider_name;
  ::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> certificates;

  bool operator==(IncomingDnscryptConfiguration const &) const noexcept;
  bool operator!=(IncomingDnscryptConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration
struct OutgoingDohConfiguration final {
  ::rust::String path;
  bool add_x_forwarded_headers CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingDohConfiguration const &) const noexcept;
  bool operator!=(OutgoingDohConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration
struct IncomingTcpConfiguration final {
  ::std::uint32_t max_in_flight_queries CXX_DEFAULT_VALUE(0);
  ::std::uint32_t listen_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t fast_open_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_concurrent_connections CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingTcpConfiguration const &) const noexcept;
  bool operator!=(IncomingTcpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration
struct BindConfiguration final {
  ::rust::String listen_address;
  bool reuseport CXX_DEFAULT_VALUE(false);
  ::rust::String protocol;
  ::std::uint32_t threads CXX_DEFAULT_VALUE(0);
  ::rust::String interface;
  ::rust::String cpus;
  bool enable_proxy_protocol CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::IncomingTcpConfiguration tcp;
  ::dnsdist::rust::settings::IncomingTlsConfiguration tls;
  ::dnsdist::rust::settings::IncomingDohConfiguration doh;
  ::dnsdist::rust::settings::IncomingDoqConfiguration doq;
  ::dnsdist::rust::settings::IncomingQuicConfiguration quic;
  ::dnsdist::rust::settings::IncomingDnscryptConfiguration dnscrypt;
  ::rust::Vec<::rust::String> additional_addresses;
  ::rust::String xsk;

  bool operator==(BindConfiguration const &) const noexcept;
  bool operator!=(BindConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration
struct OutgoingTcpConfiguration final {
  ::std::uint16_t retries CXX_DEFAULT_VALUE(0);
  ::std::uint16_t connect_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint16_t send_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint16_t receive_timeout CXX_DEFAULT_VALUE(0);
  bool fast_open CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingTcpConfiguration const &) const noexcept;
  bool operator!=(OutgoingTcpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration
struct ProxyProtocolValueConfiguration final {
  ::std::uint8_t key CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(ProxyProtocolValueConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolValueConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration
struct LazyHealthCheckConfiguration final {
  ::std::uint16_t interval CXX_DEFAULT_VALUE(0);
  ::std::uint16_t min_sample_count CXX_DEFAULT_VALUE(0);
  ::rust::String mode;
  ::std::uint16_t sample_size CXX_DEFAULT_VALUE(0);
  ::std::uint16_t threshold CXX_DEFAULT_VALUE(0);
  bool use_exponential_back_off CXX_DEFAULT_VALUE(false);
  ::std::uint16_t max_back_off CXX_DEFAULT_VALUE(0);

  bool operator==(LazyHealthCheckConfiguration const &) const noexcept;
  bool operator!=(LazyHealthCheckConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration
struct HealthCheckConfiguration final {
  ::rust::String mode;
  ::rust::String qname;
  ::rust::String qclass;
  ::rust::String qtype;
  ::rust::String function;
  ::rust::String lua;
  ::rust::String lua_file;
  ::std::uint16_t timeout CXX_DEFAULT_VALUE(0);
  bool set_cd CXX_DEFAULT_VALUE(false);
  ::std::uint8_t max_failures CXX_DEFAULT_VALUE(0);
  ::std::uint8_t rise CXX_DEFAULT_VALUE(0);
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  bool must_resolve CXX_DEFAULT_VALUE(false);
  bool use_tcp CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::LazyHealthCheckConfiguration lazy;

  bool operator==(HealthCheckConfiguration const &) const noexcept;
  bool operator!=(HealthCheckConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration
struct OutgoingAutoUpgradeConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  bool keep CXX_DEFAULT_VALUE(false);
  ::rust::String pool;
  ::std::uint8_t doh_key CXX_DEFAULT_VALUE(0);
  bool use_lazy_health_check CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingAutoUpgradeConfiguration const &) const noexcept;
  bool operator!=(OutgoingAutoUpgradeConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration
struct BackendConfiguration final {
  ::rust::String address;
  ::rust::String id;
  ::rust::String name;
  ::rust::String protocol;
  ::dnsdist::rust::settings::OutgoingTlsConfiguration tls;
  ::dnsdist::rust::settings::OutgoingDohConfiguration doh;
  bool use_client_subnet CXX_DEFAULT_VALUE(false);
  bool use_proxy_protocol CXX_DEFAULT_VALUE(false);
  ::std::uint32_t queries_per_second CXX_DEFAULT_VALUE(0);
  ::std::uint32_t order CXX_DEFAULT_VALUE(0);
  ::std::uint32_t weight CXX_DEFAULT_VALUE(0);
  ::std::uint8_t udp_timeout CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> pools;
  ::dnsdist::rust::settings::OutgoingTcpConfiguration tcp;
  bool ip_bind_addr_no_port CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::HealthCheckConfiguration health_checks;
  ::rust::String source;
  ::std::uint32_t sockets CXX_DEFAULT_VALUE(0);
  bool disable_zero_scope CXX_DEFAULT_VALUE(false);
  bool reconnect_on_up CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_in_flight CXX_DEFAULT_VALUE(0);
  bool tcp_only CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::OutgoingAutoUpgradeConfiguration auto_upgrade;
  ::std::uint32_t max_concurrent_tcp_connections CXX_DEFAULT_VALUE(0);
  bool proxy_protocol_advertise_tls CXX_DEFAULT_VALUE(false);
  ::rust::String mac_address;
  ::rust::String cpus;
  ::rust::String xsk;
  ::std::uint8_t dscp CXX_DEFAULT_VALUE(0);

  bool operator==(BackendConfiguration const &) const noexcept;
  bool operator!=(BackendConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration
struct TlsEngineConfiguration final {
  ::rust::String name;
  ::rust::String default_string;

  bool operator==(TlsEngineConfiguration const &) const noexcept;
  bool operator!=(TlsEngineConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration
struct NetmaskGroupConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> netmasks;

  bool operator==(NetmaskGroupConfiguration const &) const noexcept;
  bool operator!=(NetmaskGroupConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration
struct PacketCacheConfiguration final {
  ::rust::String name;
  ::std::uint64_t size CXX_DEFAULT_VALUE(0);
  bool deferrable_insert_lock CXX_DEFAULT_VALUE(false);
  bool dont_age CXX_DEFAULT_VALUE(false);
  bool keep_stale_data CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_negative_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t min_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t shards CXX_DEFAULT_VALUE(0);
  bool parse_ecs CXX_DEFAULT_VALUE(false);
  ::std::uint32_t stale_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t temporary_failure_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t truncated_ttl CXX_DEFAULT_VALUE(0);
  bool cookie_hashing CXX_DEFAULT_VALUE(false);
  ::std::uint32_t maximum_entry_size CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> options_to_skip;
  ::rust::Vec<::std::uint16_t> payload_ranks;

  bool operator==(PacketCacheConfiguration const &) const noexcept;
  bool operator!=(PacketCacheConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration
struct PoolConfiguration final {
  ::rust::String name;
  ::rust::String packet_cache;
  ::rust::String policy;

  bool operator==(PoolConfiguration const &) const noexcept;
  bool operator!=(PoolConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration
struct CustomLoadBalancingPolicyConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;
  bool ffi CXX_DEFAULT_VALUE(false);
  bool per_thread CXX_DEFAULT_VALUE(false);

  bool operator==(CustomLoadBalancingPolicyConfiguration const &) const noexcept;
  bool operator!=(CustomLoadBalancingPolicyConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration
struct QueryRuleConfiguration final {
  ::rust::String name;
  ::rust::String uuid;
  ::dnsdist::rust::settings::SharedDNSSelector selector;
  ::dnsdist::rust::settings::SharedDNSAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration
struct ResponseRuleConfiguration final {
  ::rust::String name;
  ::rust::String uuid;
  ::dnsdist::rust::settings::SharedDNSSelector selector;
  ::dnsdist::rust::settings::SharedDNSResponseAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration
struct XskConfiguration final {
  ::rust::String name;
  ::rust::String interface;
  ::std::uint16_t queues CXX_DEFAULT_VALUE(0);
  ::std::uint32_t frames CXX_DEFAULT_VALUE(0);
  ::rust::String map_path;

  bool operator==(XskConfiguration const &) const noexcept;
  bool operator!=(XskConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration

extern "C" {
bool dnsdist$rust$settings$cxxbridge1$ResponseConfig$operator$eq(ResponseConfig const &, ResponseConfig const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ResponseConfig$operator$ne(ResponseConfig const &, ResponseConfig const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SOAParams$operator$eq(SOAParams const &, SOAParams const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SOAParams$operator$ne(SOAParams const &, SOAParams const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SVCRecordAdditionalParams$operator$eq(SVCRecordAdditionalParams const &, SVCRecordAdditionalParams const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SVCRecordAdditionalParams$operator$ne(SVCRecordAdditionalParams const &, SVCRecordAdditionalParams const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SVCRecordParameters$operator$eq(SVCRecordParameters const &, SVCRecordParameters const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SVCRecordParameters$operator$ne(SVCRecordParameters const &, SVCRecordParameters const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllowActionConfiguration$operator$eq(AllowActionConfiguration const &, AllowActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllowActionConfiguration$operator$ne(AllowActionConfiguration const &, AllowActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DelayActionConfiguration$operator$eq(DelayActionConfiguration const &, DelayActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DelayActionConfiguration$operator$ne(DelayActionConfiguration const &, DelayActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLogActionConfiguration$operator$eq(DnstapLogActionConfiguration const &, DnstapLogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLogActionConfiguration$operator$ne(DnstapLogActionConfiguration const &, DnstapLogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DropActionConfiguration$operator$eq(DropActionConfiguration const &, DropActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DropActionConfiguration$operator$ne(DropActionConfiguration const &, DropActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetEDNSOptionActionConfiguration$operator$eq(SetEDNSOptionActionConfiguration const &, SetEDNSOptionActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetEDNSOptionActionConfiguration$operator$ne(SetEDNSOptionActionConfiguration const &, SetEDNSOptionActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ERCodeActionConfiguration$operator$eq(ERCodeActionConfiguration const &, ERCodeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ERCodeActionConfiguration$operator$ne(ERCodeActionConfiguration const &, ERCodeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPStatusActionConfiguration$operator$eq(HTTPStatusActionConfiguration const &, HTTPStatusActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPStatusActionConfiguration$operator$ne(HTTPStatusActionConfiguration const &, HTTPStatusActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupActionConfiguration$operator$eq(KeyValueStoreLookupActionConfiguration const &, KeyValueStoreLookupActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupActionConfiguration$operator$ne(KeyValueStoreLookupActionConfiguration const &, KeyValueStoreLookupActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupActionConfiguration$operator$eq(KeyValueStoreRangeLookupActionConfiguration const &, KeyValueStoreRangeLookupActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupActionConfiguration$operator$ne(KeyValueStoreRangeLookupActionConfiguration const &, KeyValueStoreRangeLookupActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LogActionConfiguration$operator$eq(LogActionConfiguration const &, LogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LogActionConfiguration$operator$ne(LogActionConfiguration const &, LogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaActionConfiguration$operator$eq(LuaActionConfiguration const &, LuaActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaActionConfiguration$operator$ne(LuaActionConfiguration const &, LuaActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIActionConfiguration$operator$eq(LuaFFIActionConfiguration const &, LuaFFIActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIActionConfiguration$operator$ne(LuaFFIActionConfiguration const &, LuaFFIActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadActionConfiguration$operator$eq(LuaFFIPerThreadActionConfiguration const &, LuaFFIPerThreadActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadActionConfiguration$operator$ne(LuaFFIPerThreadActionConfiguration const &, LuaFFIPerThreadActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NegativeAndSOAActionConfiguration$operator$eq(NegativeAndSOAActionConfiguration const &, NegativeAndSOAActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NegativeAndSOAActionConfiguration$operator$ne(NegativeAndSOAActionConfiguration const &, NegativeAndSOAActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NoneActionConfiguration$operator$eq(NoneActionConfiguration const &, NoneActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NoneActionConfiguration$operator$ne(NoneActionConfiguration const &, NoneActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolActionConfiguration$operator$eq(PoolActionConfiguration const &, PoolActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolActionConfiguration$operator$ne(PoolActionConfiguration const &, PoolActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QPSActionConfiguration$operator$eq(QPSActionConfiguration const &, QPSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QPSActionConfiguration$operator$ne(QPSActionConfiguration const &, QPSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QPSPoolActionConfiguration$operator$eq(QPSPoolActionConfiguration const &, QPSPoolActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QPSPoolActionConfiguration$operator$ne(QPSPoolActionConfiguration const &, QPSPoolActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RCodeActionConfiguration$operator$eq(RCodeActionConfiguration const &, RCodeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RCodeActionConfiguration$operator$ne(RCodeActionConfiguration const &, RCodeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLogActionConfiguration$operator$eq(RemoteLogActionConfiguration const &, RemoteLogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLogActionConfiguration$operator$ne(RemoteLogActionConfiguration const &, RemoteLogActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetAdditionalProxyProtocolValueActionConfiguration$operator$eq(SetAdditionalProxyProtocolValueActionConfiguration const &, SetAdditionalProxyProtocolValueActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetAdditionalProxyProtocolValueActionConfiguration$operator$ne(SetAdditionalProxyProtocolValueActionConfiguration const &, SetAdditionalProxyProtocolValueActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetDisableECSActionConfiguration$operator$eq(SetDisableECSActionConfiguration const &, SetDisableECSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetDisableECSActionConfiguration$operator$ne(SetDisableECSActionConfiguration const &, SetDisableECSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetDisableValidationActionConfiguration$operator$eq(SetDisableValidationActionConfiguration const &, SetDisableValidationActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetDisableValidationActionConfiguration$operator$ne(SetDisableValidationActionConfiguration const &, SetDisableValidationActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSActionConfiguration$operator$eq(SetECSActionConfiguration const &, SetECSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSActionConfiguration$operator$ne(SetECSActionConfiguration const &, SetECSActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSOverrideActionConfiguration$operator$eq(SetECSOverrideActionConfiguration const &, SetECSOverrideActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSOverrideActionConfiguration$operator$ne(SetECSOverrideActionConfiguration const &, SetECSOverrideActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSPrefixLengthActionConfiguration$operator$eq(SetECSPrefixLengthActionConfiguration const &, SetECSPrefixLengthActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetECSPrefixLengthActionConfiguration$operator$ne(SetECSPrefixLengthActionConfiguration const &, SetECSPrefixLengthActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorActionConfiguration$operator$eq(SetExtendedDNSErrorActionConfiguration const &, SetExtendedDNSErrorActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorActionConfiguration$operator$ne(SetExtendedDNSErrorActionConfiguration const &, SetExtendedDNSErrorActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMacAddrActionConfiguration$operator$eq(SetMacAddrActionConfiguration const &, SetMacAddrActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMacAddrActionConfiguration$operator$ne(SetMacAddrActionConfiguration const &, SetMacAddrActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLActionConfiguration$operator$eq(SetMaxReturnedTTLActionConfiguration const &, SetMaxReturnedTTLActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLActionConfiguration$operator$ne(SetMaxReturnedTTLActionConfiguration const &, SetMaxReturnedTTLActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetNoRecurseActionConfiguration$operator$eq(SetNoRecurseActionConfiguration const &, SetNoRecurseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetNoRecurseActionConfiguration$operator$ne(SetNoRecurseActionConfiguration const &, SetNoRecurseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetProxyProtocolValuesActionConfiguration$operator$eq(SetProxyProtocolValuesActionConfiguration const &, SetProxyProtocolValuesActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetProxyProtocolValuesActionConfiguration$operator$ne(SetProxyProtocolValuesActionConfiguration const &, SetProxyProtocolValuesActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetSkipCacheActionConfiguration$operator$eq(SetSkipCacheActionConfiguration const &, SetSkipCacheActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetSkipCacheActionConfiguration$operator$ne(SetSkipCacheActionConfiguration const &, SetSkipCacheActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTagActionConfiguration$operator$eq(SetTagActionConfiguration const &, SetTagActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTagActionConfiguration$operator$ne(SetTagActionConfiguration const &, SetTagActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTempFailureCacheTTLActionConfiguration$operator$eq(SetTempFailureCacheTTLActionConfiguration const &, SetTempFailureCacheTTLActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTempFailureCacheTTLActionConfiguration$operator$ne(SetTempFailureCacheTTLActionConfiguration const &, SetTempFailureCacheTTLActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNMPTrapActionConfiguration$operator$eq(SNMPTrapActionConfiguration const &, SNMPTrapActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNMPTrapActionConfiguration$operator$ne(SNMPTrapActionConfiguration const &, SNMPTrapActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofActionConfiguration$operator$eq(SpoofActionConfiguration const &, SpoofActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofActionConfiguration$operator$ne(SpoofActionConfiguration const &, SpoofActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofCNAMEActionConfiguration$operator$eq(SpoofCNAMEActionConfiguration const &, SpoofCNAMEActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofCNAMEActionConfiguration$operator$ne(SpoofCNAMEActionConfiguration const &, SpoofCNAMEActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofPacketActionConfiguration$operator$eq(SpoofPacketActionConfiguration const &, SpoofPacketActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofPacketActionConfiguration$operator$ne(SpoofPacketActionConfiguration const &, SpoofPacketActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofRawActionConfiguration$operator$eq(SpoofRawActionConfiguration const &, SpoofRawActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofRawActionConfiguration$operator$ne(SpoofRawActionConfiguration const &, SpoofRawActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofSVCActionConfiguration$operator$eq(SpoofSVCActionConfiguration const &, SpoofSVCActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SpoofSVCActionConfiguration$operator$ne(SpoofSVCActionConfiguration const &, SpoofSVCActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCActionConfiguration$operator$eq(TCActionConfiguration const &, TCActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCActionConfiguration$operator$ne(TCActionConfiguration const &, TCActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TeeActionConfiguration$operator$eq(TeeActionConfiguration const &, TeeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TeeActionConfiguration$operator$ne(TeeActionConfiguration const &, TeeActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllowResponseActionConfiguration$operator$eq(AllowResponseActionConfiguration const &, AllowResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllowResponseActionConfiguration$operator$ne(AllowResponseActionConfiguration const &, AllowResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ClearRecordTypesResponseActionConfiguration$operator$eq(ClearRecordTypesResponseActionConfiguration const &, ClearRecordTypesResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ClearRecordTypesResponseActionConfiguration$operator$ne(ClearRecordTypesResponseActionConfiguration const &, ClearRecordTypesResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DelayResponseActionConfiguration$operator$eq(DelayResponseActionConfiguration const &, DelayResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DelayResponseActionConfiguration$operator$ne(DelayResponseActionConfiguration const &, DelayResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLogResponseActionConfiguration$operator$eq(DnstapLogResponseActionConfiguration const &, DnstapLogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLogResponseActionConfiguration$operator$ne(DnstapLogResponseActionConfiguration const &, DnstapLogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DropResponseActionConfiguration$operator$eq(DropResponseActionConfiguration const &, DropResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DropResponseActionConfiguration$operator$ne(DropResponseActionConfiguration const &, DropResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LimitTTLResponseActionConfiguration$operator$eq(LimitTTLResponseActionConfiguration const &, LimitTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LimitTTLResponseActionConfiguration$operator$ne(LimitTTLResponseActionConfiguration const &, LimitTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LogResponseActionConfiguration$operator$eq(LogResponseActionConfiguration const &, LogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LogResponseActionConfiguration$operator$ne(LogResponseActionConfiguration const &, LogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaResponseActionConfiguration$operator$eq(LuaResponseActionConfiguration const &, LuaResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaResponseActionConfiguration$operator$ne(LuaResponseActionConfiguration const &, LuaResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIResponseActionConfiguration$operator$eq(LuaFFIResponseActionConfiguration const &, LuaFFIResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIResponseActionConfiguration$operator$ne(LuaFFIResponseActionConfiguration const &, LuaFFIResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadResponseActionConfiguration$operator$eq(LuaFFIPerThreadResponseActionConfiguration const &, LuaFFIPerThreadResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadResponseActionConfiguration$operator$ne(LuaFFIPerThreadResponseActionConfiguration const &, LuaFFIPerThreadResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLogResponseActionConfiguration$operator$eq(RemoteLogResponseActionConfiguration const &, RemoteLogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLogResponseActionConfiguration$operator$ne(RemoteLogResponseActionConfiguration const &, RemoteLogResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetEDNSOptionResponseActionConfiguration$operator$eq(SetEDNSOptionResponseActionConfiguration const &, SetEDNSOptionResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetEDNSOptionResponseActionConfiguration$operator$ne(SetEDNSOptionResponseActionConfiguration const &, SetEDNSOptionResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorResponseActionConfiguration$operator$eq(SetExtendedDNSErrorResponseActionConfiguration const &, SetExtendedDNSErrorResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorResponseActionConfiguration$operator$ne(SetExtendedDNSErrorResponseActionConfiguration const &, SetExtendedDNSErrorResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLResponseActionConfiguration$operator$eq(SetMaxReturnedTTLResponseActionConfiguration const &, SetMaxReturnedTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLResponseActionConfiguration$operator$ne(SetMaxReturnedTTLResponseActionConfiguration const &, SetMaxReturnedTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxTTLResponseActionConfiguration$operator$eq(SetMaxTTLResponseActionConfiguration const &, SetMaxTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMaxTTLResponseActionConfiguration$operator$ne(SetMaxTTLResponseActionConfiguration const &, SetMaxTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMinTTLResponseActionConfiguration$operator$eq(SetMinTTLResponseActionConfiguration const &, SetMinTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetMinTTLResponseActionConfiguration$operator$ne(SetMinTTLResponseActionConfiguration const &, SetMinTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetReducedTTLResponseActionConfiguration$operator$eq(SetReducedTTLResponseActionConfiguration const &, SetReducedTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetReducedTTLResponseActionConfiguration$operator$ne(SetReducedTTLResponseActionConfiguration const &, SetReducedTTLResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetSkipCacheResponseActionConfiguration$operator$eq(SetSkipCacheResponseActionConfiguration const &, SetSkipCacheResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetSkipCacheResponseActionConfiguration$operator$ne(SetSkipCacheResponseActionConfiguration const &, SetSkipCacheResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTagResponseActionConfiguration$operator$eq(SetTagResponseActionConfiguration const &, SetTagResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SetTagResponseActionConfiguration$operator$ne(SetTagResponseActionConfiguration const &, SetTagResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNMPTrapResponseActionConfiguration$operator$eq(SNMPTrapResponseActionConfiguration const &, SNMPTrapResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNMPTrapResponseActionConfiguration$operator$ne(SNMPTrapResponseActionConfiguration const &, SNMPTrapResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCResponseActionConfiguration$operator$eq(TCResponseActionConfiguration const &, TCResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCResponseActionConfiguration$operator$ne(TCResponseActionConfiguration const &, TCResponseActionConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllSelectorConfiguration$operator$eq(AllSelectorConfiguration const &, AllSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$AllSelectorConfiguration$operator$ne(AllSelectorConfiguration const &, AllSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ByNameSelectorConfiguration$operator$eq(ByNameSelectorConfiguration const &, ByNameSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ByNameSelectorConfiguration$operator$ne(ByNameSelectorConfiguration const &, ByNameSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DNSSECSelectorConfiguration$operator$eq(DNSSECSelectorConfiguration const &, DNSSECSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DNSSECSelectorConfiguration$operator$ne(DNSSECSelectorConfiguration const &, DNSSECSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DSTPortSelectorConfiguration$operator$eq(DSTPortSelectorConfiguration const &, DSTPortSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DSTPortSelectorConfiguration$operator$ne(DSTPortSelectorConfiguration const &, DSTPortSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EDNSOptionSelectorConfiguration$operator$eq(EDNSOptionSelectorConfiguration const &, EDNSOptionSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EDNSOptionSelectorConfiguration$operator$ne(EDNSOptionSelectorConfiguration const &, EDNSOptionSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EDNSVersionSelectorConfiguration$operator$eq(EDNSVersionSelectorConfiguration const &, EDNSVersionSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EDNSVersionSelectorConfiguration$operator$ne(EDNSVersionSelectorConfiguration const &, EDNSVersionSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ERCodeSelectorConfiguration$operator$eq(ERCodeSelectorConfiguration const &, ERCodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ERCodeSelectorConfiguration$operator$ne(ERCodeSelectorConfiguration const &, ERCodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPHeaderSelectorConfiguration$operator$eq(HTTPHeaderSelectorConfiguration const &, HTTPHeaderSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPHeaderSelectorConfiguration$operator$ne(HTTPHeaderSelectorConfiguration const &, HTTPHeaderSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPPathSelectorConfiguration$operator$eq(HTTPPathSelectorConfiguration const &, HTTPPathSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPPathSelectorConfiguration$operator$ne(HTTPPathSelectorConfiguration const &, HTTPPathSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPPathRegexSelectorConfiguration$operator$eq(HTTPPathRegexSelectorConfiguration const &, HTTPPathRegexSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HTTPPathRegexSelectorConfiguration$operator$ne(HTTPPathRegexSelectorConfiguration const &, HTTPPathRegexSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupSelectorConfiguration$operator$eq(KeyValueStoreLookupSelectorConfiguration const &, KeyValueStoreLookupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupSelectorConfiguration$operator$ne(KeyValueStoreLookupSelectorConfiguration const &, KeyValueStoreLookupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupSelectorConfiguration$operator$eq(KeyValueStoreRangeLookupSelectorConfiguration const &, KeyValueStoreRangeLookupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupSelectorConfiguration$operator$ne(KeyValueStoreRangeLookupSelectorConfiguration const &, KeyValueStoreRangeLookupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaSelectorConfiguration$operator$eq(LuaSelectorConfiguration const &, LuaSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaSelectorConfiguration$operator$ne(LuaSelectorConfiguration const &, LuaSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFISelectorConfiguration$operator$eq(LuaFFISelectorConfiguration const &, LuaFFISelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFISelectorConfiguration$operator$ne(LuaFFISelectorConfiguration const &, LuaFFISelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadSelectorConfiguration$operator$eq(LuaFFIPerThreadSelectorConfiguration const &, LuaFFIPerThreadSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadSelectorConfiguration$operator$ne(LuaFFIPerThreadSelectorConfiguration const &, LuaFFIPerThreadSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MaxQPSSelectorConfiguration$operator$eq(MaxQPSSelectorConfiguration const &, MaxQPSSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MaxQPSSelectorConfiguration$operator$ne(MaxQPSSelectorConfiguration const &, MaxQPSSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MaxQPSIPSelectorConfiguration$operator$eq(MaxQPSIPSelectorConfiguration const &, MaxQPSIPSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MaxQPSIPSelectorConfiguration$operator$ne(MaxQPSIPSelectorConfiguration const &, MaxQPSIPSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NetmaskGroupSelectorConfiguration$operator$eq(NetmaskGroupSelectorConfiguration const &, NetmaskGroupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NetmaskGroupSelectorConfiguration$operator$ne(NetmaskGroupSelectorConfiguration const &, NetmaskGroupSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OpcodeSelectorConfiguration$operator$eq(OpcodeSelectorConfiguration const &, OpcodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OpcodeSelectorConfiguration$operator$ne(OpcodeSelectorConfiguration const &, OpcodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PayloadSizeSelectorConfiguration$operator$eq(PayloadSizeSelectorConfiguration const &, PayloadSizeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PayloadSizeSelectorConfiguration$operator$ne(PayloadSizeSelectorConfiguration const &, PayloadSizeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolAvailableSelectorConfiguration$operator$eq(PoolAvailableSelectorConfiguration const &, PoolAvailableSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolAvailableSelectorConfiguration$operator$ne(PoolAvailableSelectorConfiguration const &, PoolAvailableSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolOutstandingSelectorConfiguration$operator$eq(PoolOutstandingSelectorConfiguration const &, PoolOutstandingSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolOutstandingSelectorConfiguration$operator$ne(PoolOutstandingSelectorConfiguration const &, PoolOutstandingSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProbaSelectorConfiguration$operator$eq(ProbaSelectorConfiguration const &, ProbaSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProbaSelectorConfiguration$operator$ne(ProbaSelectorConfiguration const &, ProbaSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueSelectorConfiguration$operator$eq(ProxyProtocolValueSelectorConfiguration const &, ProxyProtocolValueSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueSelectorConfiguration$operator$ne(ProxyProtocolValueSelectorConfiguration const &, ProxyProtocolValueSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QClassSelectorConfiguration$operator$eq(QClassSelectorConfiguration const &, QClassSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QClassSelectorConfiguration$operator$ne(QClassSelectorConfiguration const &, QClassSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSelectorConfiguration$operator$eq(QNameSelectorConfiguration const &, QNameSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSelectorConfiguration$operator$ne(QNameSelectorConfiguration const &, QNameSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameLabelsCountSelectorConfiguration$operator$eq(QNameLabelsCountSelectorConfiguration const &, QNameLabelsCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameLabelsCountSelectorConfiguration$operator$ne(QNameLabelsCountSelectorConfiguration const &, QNameLabelsCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSetSelectorConfiguration$operator$eq(QNameSetSelectorConfiguration const &, QNameSetSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSetSelectorConfiguration$operator$ne(QNameSetSelectorConfiguration const &, QNameSetSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSuffixSelectorConfiguration$operator$eq(QNameSuffixSelectorConfiguration const &, QNameSuffixSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameSuffixSelectorConfiguration$operator$ne(QNameSuffixSelectorConfiguration const &, QNameSuffixSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameWireLengthSelectorConfiguration$operator$eq(QNameWireLengthSelectorConfiguration const &, QNameWireLengthSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QNameWireLengthSelectorConfiguration$operator$ne(QNameWireLengthSelectorConfiguration const &, QNameWireLengthSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QTypeSelectorConfiguration$operator$eq(QTypeSelectorConfiguration const &, QTypeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QTypeSelectorConfiguration$operator$ne(QTypeSelectorConfiguration const &, QTypeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RCodeSelectorConfiguration$operator$eq(RCodeSelectorConfiguration const &, RCodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RCodeSelectorConfiguration$operator$ne(RCodeSelectorConfiguration const &, RCodeSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RDSelectorConfiguration$operator$eq(RDSelectorConfiguration const &, RDSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RDSelectorConfiguration$operator$ne(RDSelectorConfiguration const &, RDSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RE2SelectorConfiguration$operator$eq(RE2SelectorConfiguration const &, RE2SelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RE2SelectorConfiguration$operator$ne(RE2SelectorConfiguration const &, RE2SelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RecordsCountSelectorConfiguration$operator$eq(RecordsCountSelectorConfiguration const &, RecordsCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RecordsCountSelectorConfiguration$operator$ne(RecordsCountSelectorConfiguration const &, RecordsCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RecordsTypeCountSelectorConfiguration$operator$eq(RecordsTypeCountSelectorConfiguration const &, RecordsTypeCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RecordsTypeCountSelectorConfiguration$operator$ne(RecordsTypeCountSelectorConfiguration const &, RecordsTypeCountSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RegexSelectorConfiguration$operator$eq(RegexSelectorConfiguration const &, RegexSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RegexSelectorConfiguration$operator$ne(RegexSelectorConfiguration const &, RegexSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNISelectorConfiguration$operator$eq(SNISelectorConfiguration const &, SNISelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SNISelectorConfiguration$operator$ne(SNISelectorConfiguration const &, SNISelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TagSelectorConfiguration$operator$eq(TagSelectorConfiguration const &, TagSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TagSelectorConfiguration$operator$ne(TagSelectorConfiguration const &, TagSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCPSelectorConfiguration$operator$eq(TCPSelectorConfiguration const &, TCPSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TCPSelectorConfiguration$operator$ne(TCPSelectorConfiguration const &, TCPSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TrailingDataSelectorConfiguration$operator$eq(TrailingDataSelectorConfiguration const &, TrailingDataSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TrailingDataSelectorConfiguration$operator$ne(TrailingDataSelectorConfiguration const &, TrailingDataSelectorConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MetricsConfiguration$operator$eq(MetricsConfiguration const &, MetricsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$MetricsConfiguration$operator$ne(MetricsConfiguration const &, MetricsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CarbonConfiguration$operator$eq(CarbonConfiguration const &, CarbonConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CarbonConfiguration$operator$ne(CarbonConfiguration const &, CarbonConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLoggingConfiguration$operator$eq(RemoteLoggingConfiguration const &, RemoteLoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RemoteLoggingConfiguration$operator$ne(RemoteLoggingConfiguration const &, RemoteLoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProtobufLoggerConfiguration$operator$eq(ProtobufLoggerConfiguration const &, ProtobufLoggerConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProtobufLoggerConfiguration$operator$ne(ProtobufLoggerConfiguration const &, ProtobufLoggerConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLoggerConfiguration$operator$eq(DnstapLoggerConfiguration const &, DnstapLoggerConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DnstapLoggerConfiguration$operator$ne(DnstapLoggerConfiguration const &, DnstapLoggerConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProtoBufMetaConfiguration$operator$eq(ProtoBufMetaConfiguration const &, ProtoBufMetaConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProtoBufMetaConfiguration$operator$ne(ProtoBufMetaConfiguration const &, ProtoBufMetaConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LmdbKvStoreConfiguration$operator$eq(LmdbKvStoreConfiguration const &, LmdbKvStoreConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LmdbKvStoreConfiguration$operator$ne(LmdbKvStoreConfiguration const &, LmdbKvStoreConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CdbKvStoreConfiguration$operator$eq(CdbKvStoreConfiguration const &, CdbKvStoreConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CdbKvStoreConfiguration$operator$ne(CdbKvStoreConfiguration const &, CdbKvStoreConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeySourceIpConfiguration$operator$eq(KvsLookupKeySourceIpConfiguration const &, KvsLookupKeySourceIpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeySourceIpConfiguration$operator$ne(KvsLookupKeySourceIpConfiguration const &, KvsLookupKeySourceIpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeyQnameConfiguration$operator$eq(KvsLookupKeyQnameConfiguration const &, KvsLookupKeyQnameConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeyQnameConfiguration$operator$ne(KvsLookupKeyQnameConfiguration const &, KvsLookupKeyQnameConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeySuffixConfiguration$operator$eq(KvsLookupKeySuffixConfiguration const &, KvsLookupKeySuffixConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeySuffixConfiguration$operator$ne(KvsLookupKeySuffixConfiguration const &, KvsLookupKeySuffixConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeyTagConfiguration$operator$eq(KvsLookupKeyTagConfiguration const &, KvsLookupKeyTagConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeyTagConfiguration$operator$ne(KvsLookupKeyTagConfiguration const &, KvsLookupKeyTagConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeysConfiguration$operator$eq(KvsLookupKeysConfiguration const &, KvsLookupKeysConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KvsLookupKeysConfiguration$operator$ne(KvsLookupKeysConfiguration const &, KvsLookupKeysConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoresConfiguration$operator$eq(KeyValueStoresConfiguration const &, KeyValueStoresConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$KeyValueStoresConfiguration$operator$ne(KeyValueStoresConfiguration const &, KeyValueStoresConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$WebserverConfiguration$operator$eq(WebserverConfiguration const &, WebserverConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$WebserverConfiguration$operator$ne(WebserverConfiguration const &, WebserverConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ConsoleConfiguration$operator$eq(ConsoleConfiguration const &, ConsoleConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ConsoleConfiguration$operator$ne(ConsoleConfiguration const &, ConsoleConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EbpfMapConfiguration$operator$eq(EbpfMapConfiguration const &, EbpfMapConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EbpfMapConfiguration$operator$ne(EbpfMapConfiguration const &, EbpfMapConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EbpfConfiguration$operator$eq(EbpfConfiguration const &, EbpfConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EbpfConfiguration$operator$ne(EbpfConfiguration const &, EbpfConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EdnsClientSubnetConfiguration$operator$eq(EdnsClientSubnetConfiguration const &, EdnsClientSubnetConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$EdnsClientSubnetConfiguration$operator$ne(EdnsClientSubnetConfiguration const &, EdnsClientSubnetConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRulesSettingsConfiguration$operator$eq(DynamicRulesSettingsConfiguration const &, DynamicRulesSettingsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRulesSettingsConfiguration$operator$ne(DynamicRulesSettingsConfiguration const &, DynamicRulesSettingsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRuleConfiguration$operator$eq(DynamicRuleConfiguration const &, DynamicRuleConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRuleConfiguration$operator$ne(DynamicRuleConfiguration const &, DynamicRuleConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRulesConfiguration$operator$eq(DynamicRulesConfiguration const &, DynamicRulesConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DynamicRulesConfiguration$operator$ne(DynamicRulesConfiguration const &, DynamicRulesConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RingBuffersConfiguration$operator$eq(RingBuffersConfiguration const &, RingBuffersConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$RingBuffersConfiguration$operator$ne(RingBuffersConfiguration const &, RingBuffersConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTlsCertificateKeyPairConfiguration$operator$eq(IncomingTlsCertificateKeyPairConfiguration const &, IncomingTlsCertificateKeyPairConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTlsCertificateKeyPairConfiguration$operator$ne(IncomingTlsCertificateKeyPairConfiguration const &, IncomingTlsCertificateKeyPairConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTlsConfiguration$operator$eq(IncomingTlsConfiguration const &, IncomingTlsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTlsConfiguration$operator$ne(IncomingTlsConfiguration const &, IncomingTlsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingTlsConfiguration$operator$eq(OutgoingTlsConfiguration const &, OutgoingTlsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingTlsConfiguration$operator$ne(OutgoingTlsConfiguration const &, OutgoingTlsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HttpCustomResponseHeaderConfiguration$operator$eq(HttpCustomResponseHeaderConfiguration const &, HttpCustomResponseHeaderConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HttpCustomResponseHeaderConfiguration$operator$ne(HttpCustomResponseHeaderConfiguration const &, HttpCustomResponseHeaderConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HttpResponsesMapConfiguration$operator$eq(HttpResponsesMapConfiguration const &, HttpResponsesMapConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HttpResponsesMapConfiguration$operator$ne(HttpResponsesMapConfiguration const &, HttpResponsesMapConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDohConfiguration$operator$eq(IncomingDohConfiguration const &, IncomingDohConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDohConfiguration$operator$ne(IncomingDohConfiguration const &, IncomingDohConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDoqConfiguration$operator$eq(IncomingDoqConfiguration const &, IncomingDoqConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDoqConfiguration$operator$ne(IncomingDoqConfiguration const &, IncomingDoqConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingQuicConfiguration$operator$eq(IncomingQuicConfiguration const &, IncomingQuicConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingQuicConfiguration$operator$ne(IncomingQuicConfiguration const &, IncomingQuicConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDnscryptCertificateKeyPairConfiguration$operator$eq(IncomingDnscryptCertificateKeyPairConfiguration const &, IncomingDnscryptCertificateKeyPairConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDnscryptCertificateKeyPairConfiguration$operator$ne(IncomingDnscryptCertificateKeyPairConfiguration const &, IncomingDnscryptCertificateKeyPairConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDnscryptConfiguration$operator$eq(IncomingDnscryptConfiguration const &, IncomingDnscryptConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingDnscryptConfiguration$operator$ne(IncomingDnscryptConfiguration const &, IncomingDnscryptConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingDohConfiguration$operator$eq(OutgoingDohConfiguration const &, OutgoingDohConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingDohConfiguration$operator$ne(OutgoingDohConfiguration const &, OutgoingDohConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTcpConfiguration$operator$eq(IncomingTcpConfiguration const &, IncomingTcpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$IncomingTcpConfiguration$operator$ne(IncomingTcpConfiguration const &, IncomingTcpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$BindConfiguration$operator$eq(BindConfiguration const &, BindConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$BindConfiguration$operator$ne(BindConfiguration const &, BindConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingTcpConfiguration$operator$eq(OutgoingTcpConfiguration const &, OutgoingTcpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingTcpConfiguration$operator$ne(OutgoingTcpConfiguration const &, OutgoingTcpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueConfiguration$operator$eq(ProxyProtocolValueConfiguration const &, ProxyProtocolValueConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueConfiguration$operator$ne(ProxyProtocolValueConfiguration const &, ProxyProtocolValueConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LazyHealthCheckConfiguration$operator$eq(LazyHealthCheckConfiguration const &, LazyHealthCheckConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LazyHealthCheckConfiguration$operator$ne(LazyHealthCheckConfiguration const &, LazyHealthCheckConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HealthCheckConfiguration$operator$eq(HealthCheckConfiguration const &, HealthCheckConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$HealthCheckConfiguration$operator$ne(HealthCheckConfiguration const &, HealthCheckConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingAutoUpgradeConfiguration$operator$eq(OutgoingAutoUpgradeConfiguration const &, OutgoingAutoUpgradeConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$OutgoingAutoUpgradeConfiguration$operator$ne(OutgoingAutoUpgradeConfiguration const &, OutgoingAutoUpgradeConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$BackendConfiguration$operator$eq(BackendConfiguration const &, BackendConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$BackendConfiguration$operator$ne(BackendConfiguration const &, BackendConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TuningConfiguration$operator$eq(TuningConfiguration const &, TuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TuningConfiguration$operator$ne(TuningConfiguration const &, TuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TcpTuningConfiguration$operator$eq(TcpTuningConfiguration const &, TcpTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TcpTuningConfiguration$operator$ne(TcpTuningConfiguration const &, TcpTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$UdpTuningConfiguration$operator$eq(UdpTuningConfiguration const &, UdpTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$UdpTuningConfiguration$operator$ne(UdpTuningConfiguration const &, UdpTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TlsEngineConfiguration$operator$eq(TlsEngineConfiguration const &, TlsEngineConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TlsEngineConfiguration$operator$ne(TlsEngineConfiguration const &, TlsEngineConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TlsTuningConfiguration$operator$eq(TlsTuningConfiguration const &, TlsTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$TlsTuningConfiguration$operator$ne(TlsTuningConfiguration const &, TlsTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DohTuningConfiguration$operator$eq(DohTuningConfiguration const &, DohTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$DohTuningConfiguration$operator$ne(DohTuningConfiguration const &, DohTuningConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CacheSettingsConfiguration$operator$eq(CacheSettingsConfiguration const &, CacheSettingsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CacheSettingsConfiguration$operator$ne(CacheSettingsConfiguration const &, CacheSettingsConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SecurityPollingConfiguration$operator$eq(SecurityPollingConfiguration const &, SecurityPollingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SecurityPollingConfiguration$operator$ne(SecurityPollingConfiguration const &, SecurityPollingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$StructuredLoggingConfiguration$operator$eq(StructuredLoggingConfiguration const &, StructuredLoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$StructuredLoggingConfiguration$operator$ne(StructuredLoggingConfiguration const &, StructuredLoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LoggingConfiguration$operator$eq(LoggingConfiguration const &, LoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LoggingConfiguration$operator$ne(LoggingConfiguration const &, LoggingConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$GeneralConfiguration$operator$eq(GeneralConfiguration const &, GeneralConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$GeneralConfiguration$operator$ne(GeneralConfiguration const &, GeneralConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NetmaskGroupConfiguration$operator$eq(NetmaskGroupConfiguration const &, NetmaskGroupConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$NetmaskGroupConfiguration$operator$ne(NetmaskGroupConfiguration const &, NetmaskGroupConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PacketCacheConfiguration$operator$eq(PacketCacheConfiguration const &, PacketCacheConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PacketCacheConfiguration$operator$ne(PacketCacheConfiguration const &, PacketCacheConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolConfiguration$operator$eq(ProxyProtocolConfiguration const &, ProxyProtocolConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$ProxyProtocolConfiguration$operator$ne(ProxyProtocolConfiguration const &, ProxyProtocolConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SnmpConfiguration$operator$eq(SnmpConfiguration const &, SnmpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$SnmpConfiguration$operator$ne(SnmpConfiguration const &, SnmpConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QueryCountConfiguration$operator$eq(QueryCountConfiguration const &, QueryCountConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$QueryCountConfiguration$operator$ne(QueryCountConfiguration const &, QueryCountConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolConfiguration$operator$eq(PoolConfiguration const &, PoolConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$PoolConfiguration$operator$ne(PoolConfiguration const &, PoolConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CustomLoadBalancingPolicyConfiguration$operator$eq(CustomLoadBalancingPolicyConfiguration const &, CustomLoadBalancingPolicyConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$CustomLoadBalancingPolicyConfiguration$operator$ne(CustomLoadBalancingPolicyConfiguration const &, CustomLoadBalancingPolicyConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LoadBalancingPoliciesConfiguration$operator$eq(LoadBalancingPoliciesConfiguration const &, LoadBalancingPoliciesConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$LoadBalancingPoliciesConfiguration$operator$ne(LoadBalancingPoliciesConfiguration const &, LoadBalancingPoliciesConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$XskConfiguration$operator$eq(XskConfiguration const &, XskConfiguration const &) noexcept;
bool dnsdist$rust$settings$cxxbridge1$XskConfiguration$operator$ne(XskConfiguration const &, XskConfiguration const &) noexcept;

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getAllowAction(::dnsdist::rust::settings::AllowActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getAllowAction$)(::dnsdist::rust::settings::AllowActionConfiguration const &) = ::dnsdist::rust::settings::getAllowAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getAllowAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getContinueAction(::dnsdist::rust::settings::ContinueActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getContinueAction$)(::dnsdist::rust::settings::ContinueActionConfiguration const &) = ::dnsdist::rust::settings::getContinueAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getContinueAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDelayAction(::dnsdist::rust::settings::DelayActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getDelayAction$)(::dnsdist::rust::settings::DelayActionConfiguration const &) = ::dnsdist::rust::settings::getDelayAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getDelayAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDnstapLogAction(::dnsdist::rust::settings::DnstapLogActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getDnstapLogAction$)(::dnsdist::rust::settings::DnstapLogActionConfiguration const &) = ::dnsdist::rust::settings::getDnstapLogAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getDnstapLogAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDropAction(::dnsdist::rust::settings::DropActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getDropAction$)(::dnsdist::rust::settings::DropActionConfiguration const &) = ::dnsdist::rust::settings::getDropAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getDropAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetEDNSOptionAction(::dnsdist::rust::settings::SetEDNSOptionActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetEDNSOptionAction$)(::dnsdist::rust::settings::SetEDNSOptionActionConfiguration const &) = ::dnsdist::rust::settings::getSetEDNSOptionAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetEDNSOptionAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getERCodeAction(::dnsdist::rust::settings::ERCodeActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getERCodeAction$)(::dnsdist::rust::settings::ERCodeActionConfiguration const &) = ::dnsdist::rust::settings::getERCodeAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getERCodeAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getHTTPStatusAction(::dnsdist::rust::settings::HTTPStatusActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getHTTPStatusAction$)(::dnsdist::rust::settings::HTTPStatusActionConfiguration const &) = ::dnsdist::rust::settings::getHTTPStatusAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getHTTPStatusAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getKeyValueStoreLookupAction(::dnsdist::rust::settings::KeyValueStoreLookupActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getKeyValueStoreLookupAction$)(::dnsdist::rust::settings::KeyValueStoreLookupActionConfiguration const &) = ::dnsdist::rust::settings::getKeyValueStoreLookupAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getKeyValueStoreLookupAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getKeyValueStoreRangeLookupAction(::dnsdist::rust::settings::KeyValueStoreRangeLookupActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getKeyValueStoreRangeLookupAction$)(::dnsdist::rust::settings::KeyValueStoreRangeLookupActionConfiguration const &) = ::dnsdist::rust::settings::getKeyValueStoreRangeLookupAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getKeyValueStoreRangeLookupAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLogAction(::dnsdist::rust::settings::LogActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getLogAction$)(::dnsdist::rust::settings::LogActionConfiguration const &) = ::dnsdist::rust::settings::getLogAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getLogAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaAction(::dnsdist::rust::settings::LuaActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getLuaAction$)(::dnsdist::rust::settings::LuaActionConfiguration const &) = ::dnsdist::rust::settings::getLuaAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getLuaAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFIAction(::dnsdist::rust::settings::LuaFFIActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getLuaFFIAction$)(::dnsdist::rust::settings::LuaFFIActionConfiguration const &) = ::dnsdist::rust::settings::getLuaFFIAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getLuaFFIAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFIPerThreadAction(::dnsdist::rust::settings::LuaFFIPerThreadActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getLuaFFIPerThreadAction$)(::dnsdist::rust::settings::LuaFFIPerThreadActionConfiguration const &) = ::dnsdist::rust::settings::getLuaFFIPerThreadAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getLuaFFIPerThreadAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getNegativeAndSOAAction(::dnsdist::rust::settings::NegativeAndSOAActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getNegativeAndSOAAction$)(::dnsdist::rust::settings::NegativeAndSOAActionConfiguration const &) = ::dnsdist::rust::settings::getNegativeAndSOAAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getNegativeAndSOAAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getNoneAction(::dnsdist::rust::settings::NoneActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getNoneAction$)(::dnsdist::rust::settings::NoneActionConfiguration const &) = ::dnsdist::rust::settings::getNoneAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getNoneAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getPoolAction(::dnsdist::rust::settings::PoolActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getPoolAction$)(::dnsdist::rust::settings::PoolActionConfiguration const &) = ::dnsdist::rust::settings::getPoolAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getPoolAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQPSAction(::dnsdist::rust::settings::QPSActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getQPSAction$)(::dnsdist::rust::settings::QPSActionConfiguration const &) = ::dnsdist::rust::settings::getQPSAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getQPSAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQPSPoolAction(::dnsdist::rust::settings::QPSPoolActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getQPSPoolAction$)(::dnsdist::rust::settings::QPSPoolActionConfiguration const &) = ::dnsdist::rust::settings::getQPSPoolAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getQPSPoolAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRCodeAction(::dnsdist::rust::settings::RCodeActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getRCodeAction$)(::dnsdist::rust::settings::RCodeActionConfiguration const &) = ::dnsdist::rust::settings::getRCodeAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getRCodeAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRemoteLogAction(::dnsdist::rust::settings::RemoteLogActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getRemoteLogAction$)(::dnsdist::rust::settings::RemoteLogActionConfiguration const &) = ::dnsdist::rust::settings::getRemoteLogAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getRemoteLogAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetAdditionalProxyProtocolValueAction(::dnsdist::rust::settings::SetAdditionalProxyProtocolValueActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetAdditionalProxyProtocolValueAction$)(::dnsdist::rust::settings::SetAdditionalProxyProtocolValueActionConfiguration const &) = ::dnsdist::rust::settings::getSetAdditionalProxyProtocolValueAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetAdditionalProxyProtocolValueAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetDisableECSAction(::dnsdist::rust::settings::SetDisableECSActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetDisableECSAction$)(::dnsdist::rust::settings::SetDisableECSActionConfiguration const &) = ::dnsdist::rust::settings::getSetDisableECSAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetDisableECSAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetDisableValidationAction(::dnsdist::rust::settings::SetDisableValidationActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetDisableValidationAction$)(::dnsdist::rust::settings::SetDisableValidationActionConfiguration const &) = ::dnsdist::rust::settings::getSetDisableValidationAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetDisableValidationAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetECSAction(::dnsdist::rust::settings::SetECSActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetECSAction$)(::dnsdist::rust::settings::SetECSActionConfiguration const &) = ::dnsdist::rust::settings::getSetECSAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetECSAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetECSOverrideAction(::dnsdist::rust::settings::SetECSOverrideActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetECSOverrideAction$)(::dnsdist::rust::settings::SetECSOverrideActionConfiguration const &) = ::dnsdist::rust::settings::getSetECSOverrideAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetECSOverrideAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetECSPrefixLengthAction(::dnsdist::rust::settings::SetECSPrefixLengthActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetECSPrefixLengthAction$)(::dnsdist::rust::settings::SetECSPrefixLengthActionConfiguration const &) = ::dnsdist::rust::settings::getSetECSPrefixLengthAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetECSPrefixLengthAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetExtendedDNSErrorAction(::dnsdist::rust::settings::SetExtendedDNSErrorActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetExtendedDNSErrorAction$)(::dnsdist::rust::settings::SetExtendedDNSErrorActionConfiguration const &) = ::dnsdist::rust::settings::getSetExtendedDNSErrorAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetExtendedDNSErrorAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetMacAddrAction(::dnsdist::rust::settings::SetMacAddrActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetMacAddrAction$)(::dnsdist::rust::settings::SetMacAddrActionConfiguration const &) = ::dnsdist::rust::settings::getSetMacAddrAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetMacAddrAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetMaxReturnedTTLAction(::dnsdist::rust::settings::SetMaxReturnedTTLActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetMaxReturnedTTLAction$)(::dnsdist::rust::settings::SetMaxReturnedTTLActionConfiguration const &) = ::dnsdist::rust::settings::getSetMaxReturnedTTLAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetMaxReturnedTTLAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetNoRecurseAction(::dnsdist::rust::settings::SetNoRecurseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetNoRecurseAction$)(::dnsdist::rust::settings::SetNoRecurseActionConfiguration const &) = ::dnsdist::rust::settings::getSetNoRecurseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetNoRecurseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetProxyProtocolValuesAction(::dnsdist::rust::settings::SetProxyProtocolValuesActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetProxyProtocolValuesAction$)(::dnsdist::rust::settings::SetProxyProtocolValuesActionConfiguration const &) = ::dnsdist::rust::settings::getSetProxyProtocolValuesAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetProxyProtocolValuesAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetSkipCacheAction(::dnsdist::rust::settings::SetSkipCacheActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetSkipCacheAction$)(::dnsdist::rust::settings::SetSkipCacheActionConfiguration const &) = ::dnsdist::rust::settings::getSetSkipCacheAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetSkipCacheAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetTagAction(::dnsdist::rust::settings::SetTagActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetTagAction$)(::dnsdist::rust::settings::SetTagActionConfiguration const &) = ::dnsdist::rust::settings::getSetTagAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetTagAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetTempFailureCacheTTLAction(::dnsdist::rust::settings::SetTempFailureCacheTTLActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSetTempFailureCacheTTLAction$)(::dnsdist::rust::settings::SetTempFailureCacheTTLActionConfiguration const &) = ::dnsdist::rust::settings::getSetTempFailureCacheTTLAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSetTempFailureCacheTTLAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSNMPTrapAction(::dnsdist::rust::settings::SNMPTrapActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSNMPTrapAction$)(::dnsdist::rust::settings::SNMPTrapActionConfiguration const &) = ::dnsdist::rust::settings::getSNMPTrapAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSNMPTrapAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSpoofAction(::dnsdist::rust::settings::SpoofActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSpoofAction$)(::dnsdist::rust::settings::SpoofActionConfiguration const &) = ::dnsdist::rust::settings::getSpoofAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSpoofAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSpoofCNAMEAction(::dnsdist::rust::settings::SpoofCNAMEActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSpoofCNAMEAction$)(::dnsdist::rust::settings::SpoofCNAMEActionConfiguration const &) = ::dnsdist::rust::settings::getSpoofCNAMEAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSpoofCNAMEAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSpoofPacketAction(::dnsdist::rust::settings::SpoofPacketActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSpoofPacketAction$)(::dnsdist::rust::settings::SpoofPacketActionConfiguration const &) = ::dnsdist::rust::settings::getSpoofPacketAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSpoofPacketAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSpoofRawAction(::dnsdist::rust::settings::SpoofRawActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSpoofRawAction$)(::dnsdist::rust::settings::SpoofRawActionConfiguration const &) = ::dnsdist::rust::settings::getSpoofRawAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSpoofRawAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSpoofSVCAction(::dnsdist::rust::settings::SpoofSVCActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getSpoofSVCAction$)(::dnsdist::rust::settings::SpoofSVCActionConfiguration const &) = ::dnsdist::rust::settings::getSpoofSVCAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getSpoofSVCAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTCAction(::dnsdist::rust::settings::TCActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getTCAction$)(::dnsdist::rust::settings::TCActionConfiguration const &) = ::dnsdist::rust::settings::getTCAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getTCAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTeeAction(::dnsdist::rust::settings::TeeActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> (*getTeeAction$)(::dnsdist::rust::settings::TeeActionConfiguration const &) = ::dnsdist::rust::settings::getTeeAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(getTeeAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getAllowResponseAction(::dnsdist::rust::settings::AllowResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getAllowResponseAction$)(::dnsdist::rust::settings::AllowResponseActionConfiguration const &) = ::dnsdist::rust::settings::getAllowResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getAllowResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getClearRecordTypesResponseAction(::dnsdist::rust::settings::ClearRecordTypesResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getClearRecordTypesResponseAction$)(::dnsdist::rust::settings::ClearRecordTypesResponseActionConfiguration const &) = ::dnsdist::rust::settings::getClearRecordTypesResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getClearRecordTypesResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDelayResponseAction(::dnsdist::rust::settings::DelayResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getDelayResponseAction$)(::dnsdist::rust::settings::DelayResponseActionConfiguration const &) = ::dnsdist::rust::settings::getDelayResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getDelayResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDnstapLogResponseAction(::dnsdist::rust::settings::DnstapLogResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getDnstapLogResponseAction$)(::dnsdist::rust::settings::DnstapLogResponseActionConfiguration const &) = ::dnsdist::rust::settings::getDnstapLogResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getDnstapLogResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDropResponseAction(::dnsdist::rust::settings::DropResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getDropResponseAction$)(::dnsdist::rust::settings::DropResponseActionConfiguration const &) = ::dnsdist::rust::settings::getDropResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getDropResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLimitTTLResponseAction(::dnsdist::rust::settings::LimitTTLResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getLimitTTLResponseAction$)(::dnsdist::rust::settings::LimitTTLResponseActionConfiguration const &) = ::dnsdist::rust::settings::getLimitTTLResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getLimitTTLResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLogResponseAction(::dnsdist::rust::settings::LogResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getLogResponseAction$)(::dnsdist::rust::settings::LogResponseActionConfiguration const &) = ::dnsdist::rust::settings::getLogResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getLogResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaResponseAction(::dnsdist::rust::settings::LuaResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getLuaResponseAction$)(::dnsdist::rust::settings::LuaResponseActionConfiguration const &) = ::dnsdist::rust::settings::getLuaResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getLuaResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFIResponseAction(::dnsdist::rust::settings::LuaFFIResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getLuaFFIResponseAction$)(::dnsdist::rust::settings::LuaFFIResponseActionConfiguration const &) = ::dnsdist::rust::settings::getLuaFFIResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getLuaFFIResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFIPerThreadResponseAction(::dnsdist::rust::settings::LuaFFIPerThreadResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getLuaFFIPerThreadResponseAction$)(::dnsdist::rust::settings::LuaFFIPerThreadResponseActionConfiguration const &) = ::dnsdist::rust::settings::getLuaFFIPerThreadResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getLuaFFIPerThreadResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRemoteLogResponseAction(::dnsdist::rust::settings::RemoteLogResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getRemoteLogResponseAction$)(::dnsdist::rust::settings::RemoteLogResponseActionConfiguration const &) = ::dnsdist::rust::settings::getRemoteLogResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getRemoteLogResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetEDNSOptionResponseAction(::dnsdist::rust::settings::SetEDNSOptionResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetEDNSOptionResponseAction$)(::dnsdist::rust::settings::SetEDNSOptionResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetEDNSOptionResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetEDNSOptionResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetExtendedDNSErrorResponseAction(::dnsdist::rust::settings::SetExtendedDNSErrorResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetExtendedDNSErrorResponseAction$)(::dnsdist::rust::settings::SetExtendedDNSErrorResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetExtendedDNSErrorResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetExtendedDNSErrorResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetMaxReturnedTTLResponseAction(::dnsdist::rust::settings::SetMaxReturnedTTLResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetMaxReturnedTTLResponseAction$)(::dnsdist::rust::settings::SetMaxReturnedTTLResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetMaxReturnedTTLResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetMaxReturnedTTLResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetMaxTTLResponseAction(::dnsdist::rust::settings::SetMaxTTLResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetMaxTTLResponseAction$)(::dnsdist::rust::settings::SetMaxTTLResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetMaxTTLResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetMaxTTLResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetMinTTLResponseAction(::dnsdist::rust::settings::SetMinTTLResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetMinTTLResponseAction$)(::dnsdist::rust::settings::SetMinTTLResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetMinTTLResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetMinTTLResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetReducedTTLResponseAction(::dnsdist::rust::settings::SetReducedTTLResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetReducedTTLResponseAction$)(::dnsdist::rust::settings::SetReducedTTLResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetReducedTTLResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetReducedTTLResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetSkipCacheResponseAction(::dnsdist::rust::settings::SetSkipCacheResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetSkipCacheResponseAction$)(::dnsdist::rust::settings::SetSkipCacheResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetSkipCacheResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetSkipCacheResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSetTagResponseAction(::dnsdist::rust::settings::SetTagResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSetTagResponseAction$)(::dnsdist::rust::settings::SetTagResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSetTagResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSetTagResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSNMPTrapResponseAction(::dnsdist::rust::settings::SNMPTrapResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getSNMPTrapResponseAction$)(::dnsdist::rust::settings::SNMPTrapResponseActionConfiguration const &) = ::dnsdist::rust::settings::getSNMPTrapResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getSNMPTrapResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTCResponseAction(::dnsdist::rust::settings::TCResponseActionConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> (*getTCResponseAction$)(::dnsdist::rust::settings::TCResponseActionConfiguration const &) = ::dnsdist::rust::settings::getTCResponseAction;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(getTCResponseAction$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getAllSelector(::dnsdist::rust::settings::AllSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getAllSelector$)(::dnsdist::rust::settings::AllSelectorConfiguration const &) = ::dnsdist::rust::settings::getAllSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getAllSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getAndSelector(::dnsdist::rust::settings::AndSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getAndSelector$)(::dnsdist::rust::settings::AndSelectorConfiguration const &) = ::dnsdist::rust::settings::getAndSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getAndSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getByNameSelector(::dnsdist::rust::settings::ByNameSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getByNameSelector$)(::dnsdist::rust::settings::ByNameSelectorConfiguration const &) = ::dnsdist::rust::settings::getByNameSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getByNameSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDNSSECSelector(::dnsdist::rust::settings::DNSSECSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getDNSSECSelector$)(::dnsdist::rust::settings::DNSSECSelectorConfiguration const &) = ::dnsdist::rust::settings::getDNSSECSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getDNSSECSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getDSTPortSelector(::dnsdist::rust::settings::DSTPortSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getDSTPortSelector$)(::dnsdist::rust::settings::DSTPortSelectorConfiguration const &) = ::dnsdist::rust::settings::getDSTPortSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getDSTPortSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getEDNSOptionSelector(::dnsdist::rust::settings::EDNSOptionSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getEDNSOptionSelector$)(::dnsdist::rust::settings::EDNSOptionSelectorConfiguration const &) = ::dnsdist::rust::settings::getEDNSOptionSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getEDNSOptionSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getEDNSVersionSelector(::dnsdist::rust::settings::EDNSVersionSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getEDNSVersionSelector$)(::dnsdist::rust::settings::EDNSVersionSelectorConfiguration const &) = ::dnsdist::rust::settings::getEDNSVersionSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getEDNSVersionSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getERCodeSelector(::dnsdist::rust::settings::ERCodeSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getERCodeSelector$)(::dnsdist::rust::settings::ERCodeSelectorConfiguration const &) = ::dnsdist::rust::settings::getERCodeSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getERCodeSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getHTTPHeaderSelector(::dnsdist::rust::settings::HTTPHeaderSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getHTTPHeaderSelector$)(::dnsdist::rust::settings::HTTPHeaderSelectorConfiguration const &) = ::dnsdist::rust::settings::getHTTPHeaderSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getHTTPHeaderSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getHTTPPathSelector(::dnsdist::rust::settings::HTTPPathSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getHTTPPathSelector$)(::dnsdist::rust::settings::HTTPPathSelectorConfiguration const &) = ::dnsdist::rust::settings::getHTTPPathSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getHTTPPathSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getHTTPPathRegexSelector(::dnsdist::rust::settings::HTTPPathRegexSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getHTTPPathRegexSelector$)(::dnsdist::rust::settings::HTTPPathRegexSelectorConfiguration const &) = ::dnsdist::rust::settings::getHTTPPathRegexSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getHTTPPathRegexSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getKeyValueStoreLookupSelector(::dnsdist::rust::settings::KeyValueStoreLookupSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getKeyValueStoreLookupSelector$)(::dnsdist::rust::settings::KeyValueStoreLookupSelectorConfiguration const &) = ::dnsdist::rust::settings::getKeyValueStoreLookupSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getKeyValueStoreLookupSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getKeyValueStoreRangeLookupSelector(::dnsdist::rust::settings::KeyValueStoreRangeLookupSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getKeyValueStoreRangeLookupSelector$)(::dnsdist::rust::settings::KeyValueStoreRangeLookupSelectorConfiguration const &) = ::dnsdist::rust::settings::getKeyValueStoreRangeLookupSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getKeyValueStoreRangeLookupSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaSelector(::dnsdist::rust::settings::LuaSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getLuaSelector$)(::dnsdist::rust::settings::LuaSelectorConfiguration const &) = ::dnsdist::rust::settings::getLuaSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getLuaSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFISelector(::dnsdist::rust::settings::LuaFFISelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getLuaFFISelector$)(::dnsdist::rust::settings::LuaFFISelectorConfiguration const &) = ::dnsdist::rust::settings::getLuaFFISelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getLuaFFISelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getLuaFFIPerThreadSelector(::dnsdist::rust::settings::LuaFFIPerThreadSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getLuaFFIPerThreadSelector$)(::dnsdist::rust::settings::LuaFFIPerThreadSelectorConfiguration const &) = ::dnsdist::rust::settings::getLuaFFIPerThreadSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getLuaFFIPerThreadSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getMaxQPSSelector(::dnsdist::rust::settings::MaxQPSSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getMaxQPSSelector$)(::dnsdist::rust::settings::MaxQPSSelectorConfiguration const &) = ::dnsdist::rust::settings::getMaxQPSSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getMaxQPSSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getMaxQPSIPSelector(::dnsdist::rust::settings::MaxQPSIPSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getMaxQPSIPSelector$)(::dnsdist::rust::settings::MaxQPSIPSelectorConfiguration const &) = ::dnsdist::rust::settings::getMaxQPSIPSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getMaxQPSIPSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getNetmaskGroupSelector(::dnsdist::rust::settings::NetmaskGroupSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getNetmaskGroupSelector$)(::dnsdist::rust::settings::NetmaskGroupSelectorConfiguration const &) = ::dnsdist::rust::settings::getNetmaskGroupSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getNetmaskGroupSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getNotSelector(::dnsdist::rust::settings::NotSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getNotSelector$)(::dnsdist::rust::settings::NotSelectorConfiguration const &) = ::dnsdist::rust::settings::getNotSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getNotSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getOpcodeSelector(::dnsdist::rust::settings::OpcodeSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getOpcodeSelector$)(::dnsdist::rust::settings::OpcodeSelectorConfiguration const &) = ::dnsdist::rust::settings::getOpcodeSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getOpcodeSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getOrSelector(::dnsdist::rust::settings::OrSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getOrSelector$)(::dnsdist::rust::settings::OrSelectorConfiguration const &) = ::dnsdist::rust::settings::getOrSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getOrSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getPayloadSizeSelector(::dnsdist::rust::settings::PayloadSizeSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getPayloadSizeSelector$)(::dnsdist::rust::settings::PayloadSizeSelectorConfiguration const &) = ::dnsdist::rust::settings::getPayloadSizeSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getPayloadSizeSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getPoolAvailableSelector(::dnsdist::rust::settings::PoolAvailableSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getPoolAvailableSelector$)(::dnsdist::rust::settings::PoolAvailableSelectorConfiguration const &) = ::dnsdist::rust::settings::getPoolAvailableSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getPoolAvailableSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getPoolOutstandingSelector(::dnsdist::rust::settings::PoolOutstandingSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getPoolOutstandingSelector$)(::dnsdist::rust::settings::PoolOutstandingSelectorConfiguration const &) = ::dnsdist::rust::settings::getPoolOutstandingSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getPoolOutstandingSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getProbaSelector(::dnsdist::rust::settings::ProbaSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getProbaSelector$)(::dnsdist::rust::settings::ProbaSelectorConfiguration const &) = ::dnsdist::rust::settings::getProbaSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getProbaSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getProxyProtocolValueSelector(::dnsdist::rust::settings::ProxyProtocolValueSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getProxyProtocolValueSelector$)(::dnsdist::rust::settings::ProxyProtocolValueSelectorConfiguration const &) = ::dnsdist::rust::settings::getProxyProtocolValueSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getProxyProtocolValueSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQClassSelector(::dnsdist::rust::settings::QClassSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQClassSelector$)(::dnsdist::rust::settings::QClassSelectorConfiguration const &) = ::dnsdist::rust::settings::getQClassSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQClassSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQNameSelector(::dnsdist::rust::settings::QNameSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQNameSelector$)(::dnsdist::rust::settings::QNameSelectorConfiguration const &) = ::dnsdist::rust::settings::getQNameSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQNameSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQNameLabelsCountSelector(::dnsdist::rust::settings::QNameLabelsCountSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQNameLabelsCountSelector$)(::dnsdist::rust::settings::QNameLabelsCountSelectorConfiguration const &) = ::dnsdist::rust::settings::getQNameLabelsCountSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQNameLabelsCountSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQNameSetSelector(::dnsdist::rust::settings::QNameSetSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQNameSetSelector$)(::dnsdist::rust::settings::QNameSetSelectorConfiguration const &) = ::dnsdist::rust::settings::getQNameSetSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQNameSetSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQNameSuffixSelector(::dnsdist::rust::settings::QNameSuffixSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQNameSuffixSelector$)(::dnsdist::rust::settings::QNameSuffixSelectorConfiguration const &) = ::dnsdist::rust::settings::getQNameSuffixSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQNameSuffixSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQNameWireLengthSelector(::dnsdist::rust::settings::QNameWireLengthSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQNameWireLengthSelector$)(::dnsdist::rust::settings::QNameWireLengthSelectorConfiguration const &) = ::dnsdist::rust::settings::getQNameWireLengthSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQNameWireLengthSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getQTypeSelector(::dnsdist::rust::settings::QTypeSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getQTypeSelector$)(::dnsdist::rust::settings::QTypeSelectorConfiguration const &) = ::dnsdist::rust::settings::getQTypeSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getQTypeSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRCodeSelector(::dnsdist::rust::settings::RCodeSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRCodeSelector$)(::dnsdist::rust::settings::RCodeSelectorConfiguration const &) = ::dnsdist::rust::settings::getRCodeSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRCodeSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRDSelector(::dnsdist::rust::settings::RDSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRDSelector$)(::dnsdist::rust::settings::RDSelectorConfiguration const &) = ::dnsdist::rust::settings::getRDSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRDSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRE2Selector(::dnsdist::rust::settings::RE2SelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRE2Selector$)(::dnsdist::rust::settings::RE2SelectorConfiguration const &) = ::dnsdist::rust::settings::getRE2Selector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRE2Selector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRecordsCountSelector(::dnsdist::rust::settings::RecordsCountSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRecordsCountSelector$)(::dnsdist::rust::settings::RecordsCountSelectorConfiguration const &) = ::dnsdist::rust::settings::getRecordsCountSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRecordsCountSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRecordsTypeCountSelector(::dnsdist::rust::settings::RecordsTypeCountSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRecordsTypeCountSelector$)(::dnsdist::rust::settings::RecordsTypeCountSelectorConfiguration const &) = ::dnsdist::rust::settings::getRecordsTypeCountSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRecordsTypeCountSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getRegexSelector(::dnsdist::rust::settings::RegexSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getRegexSelector$)(::dnsdist::rust::settings::RegexSelectorConfiguration const &) = ::dnsdist::rust::settings::getRegexSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getRegexSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getSNISelector(::dnsdist::rust::settings::SNISelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getSNISelector$)(::dnsdist::rust::settings::SNISelectorConfiguration const &) = ::dnsdist::rust::settings::getSNISelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getSNISelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTagSelector(::dnsdist::rust::settings::TagSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getTagSelector$)(::dnsdist::rust::settings::TagSelectorConfiguration const &) = ::dnsdist::rust::settings::getTagSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getTagSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTCPSelector(::dnsdist::rust::settings::TCPSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getTCPSelector$)(::dnsdist::rust::settings::TCPSelectorConfiguration const &) = ::dnsdist::rust::settings::getTCPSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getTCPSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$getTrailingDataSelector(::dnsdist::rust::settings::TrailingDataSelectorConfiguration const &config, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *return$) noexcept {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> (*getTrailingDataSelector$)(::dnsdist::rust::settings::TrailingDataSelectorConfiguration const &) = ::dnsdist::rust::settings::getTrailingDataSelector;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(getTrailingDataSelector$(config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen dnsdist$rust$settings$cxxbridge1$from_yaml_string(::rust::Str str, ::dnsdist::rust::settings::GlobalConfiguration *return$) noexcept;

void dnsdist$rust$settings$cxxbridge1$registerProtobufLogger(::dnsdist::rust::settings::ProtobufLoggerConfiguration const &config) noexcept {
  void (*registerProtobufLogger$)(::dnsdist::rust::settings::ProtobufLoggerConfiguration const &) = ::dnsdist::rust::settings::registerProtobufLogger;
  registerProtobufLogger$(config);
}

void dnsdist$rust$settings$cxxbridge1$registerDnstapLogger(::dnsdist::rust::settings::DnstapLoggerConfiguration const &config) noexcept {
  void (*registerDnstapLogger$)(::dnsdist::rust::settings::DnstapLoggerConfiguration const &) = ::dnsdist::rust::settings::registerDnstapLogger;
  registerDnstapLogger$(config);
}

void dnsdist$rust$settings$cxxbridge1$registerKVSObjects(::dnsdist::rust::settings::KeyValueStoresConfiguration const &config) noexcept {
  void (*registerKVSObjects$)(::dnsdist::rust::settings::KeyValueStoresConfiguration const &) = ::dnsdist::rust::settings::registerKVSObjects;
  registerKVSObjects$(config);
}

void dnsdist$rust$settings$cxxbridge1$registerNMGObjects(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const &nmgs) noexcept {
  void (*registerNMGObjects$)(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const &) = ::dnsdist::rust::settings::registerNMGObjects;
  registerNMGObjects$(nmgs);
}
} // extern "C"

bool ResponseConfig::operator==(ResponseConfig const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ResponseConfig$operator$eq(*this, rhs);
}

bool ResponseConfig::operator!=(ResponseConfig const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ResponseConfig$operator$ne(*this, rhs);
}

bool SOAParams::operator==(SOAParams const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SOAParams$operator$eq(*this, rhs);
}

bool SOAParams::operator!=(SOAParams const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SOAParams$operator$ne(*this, rhs);
}

bool SVCRecordAdditionalParams::operator==(SVCRecordAdditionalParams const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SVCRecordAdditionalParams$operator$eq(*this, rhs);
}

bool SVCRecordAdditionalParams::operator!=(SVCRecordAdditionalParams const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SVCRecordAdditionalParams$operator$ne(*this, rhs);
}

bool SVCRecordParameters::operator==(SVCRecordParameters const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SVCRecordParameters$operator$eq(*this, rhs);
}

bool SVCRecordParameters::operator!=(SVCRecordParameters const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SVCRecordParameters$operator$ne(*this, rhs);
}

bool AllowActionConfiguration::operator==(AllowActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllowActionConfiguration$operator$eq(*this, rhs);
}

bool AllowActionConfiguration::operator!=(AllowActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllowActionConfiguration$operator$ne(*this, rhs);
}

bool DelayActionConfiguration::operator==(DelayActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DelayActionConfiguration$operator$eq(*this, rhs);
}

bool DelayActionConfiguration::operator!=(DelayActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DelayActionConfiguration$operator$ne(*this, rhs);
}

bool DnstapLogActionConfiguration::operator==(DnstapLogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLogActionConfiguration$operator$eq(*this, rhs);
}

bool DnstapLogActionConfiguration::operator!=(DnstapLogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLogActionConfiguration$operator$ne(*this, rhs);
}

bool DropActionConfiguration::operator==(DropActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DropActionConfiguration$operator$eq(*this, rhs);
}

bool DropActionConfiguration::operator!=(DropActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DropActionConfiguration$operator$ne(*this, rhs);
}

bool SetEDNSOptionActionConfiguration::operator==(SetEDNSOptionActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetEDNSOptionActionConfiguration$operator$eq(*this, rhs);
}

bool SetEDNSOptionActionConfiguration::operator!=(SetEDNSOptionActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetEDNSOptionActionConfiguration$operator$ne(*this, rhs);
}

bool ERCodeActionConfiguration::operator==(ERCodeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ERCodeActionConfiguration$operator$eq(*this, rhs);
}

bool ERCodeActionConfiguration::operator!=(ERCodeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ERCodeActionConfiguration$operator$ne(*this, rhs);
}

bool HTTPStatusActionConfiguration::operator==(HTTPStatusActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPStatusActionConfiguration$operator$eq(*this, rhs);
}

bool HTTPStatusActionConfiguration::operator!=(HTTPStatusActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPStatusActionConfiguration$operator$ne(*this, rhs);
}

bool KeyValueStoreLookupActionConfiguration::operator==(KeyValueStoreLookupActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupActionConfiguration$operator$eq(*this, rhs);
}

bool KeyValueStoreLookupActionConfiguration::operator!=(KeyValueStoreLookupActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupActionConfiguration$operator$ne(*this, rhs);
}

bool KeyValueStoreRangeLookupActionConfiguration::operator==(KeyValueStoreRangeLookupActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupActionConfiguration$operator$eq(*this, rhs);
}

bool KeyValueStoreRangeLookupActionConfiguration::operator!=(KeyValueStoreRangeLookupActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupActionConfiguration$operator$ne(*this, rhs);
}

bool LogActionConfiguration::operator==(LogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LogActionConfiguration$operator$eq(*this, rhs);
}

bool LogActionConfiguration::operator!=(LogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LogActionConfiguration$operator$ne(*this, rhs);
}

bool LuaActionConfiguration::operator==(LuaActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaActionConfiguration$operator$eq(*this, rhs);
}

bool LuaActionConfiguration::operator!=(LuaActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaActionConfiguration$operator$ne(*this, rhs);
}

bool LuaFFIActionConfiguration::operator==(LuaFFIActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIActionConfiguration$operator$eq(*this, rhs);
}

bool LuaFFIActionConfiguration::operator!=(LuaFFIActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIActionConfiguration$operator$ne(*this, rhs);
}

bool LuaFFIPerThreadActionConfiguration::operator==(LuaFFIPerThreadActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadActionConfiguration$operator$eq(*this, rhs);
}

bool LuaFFIPerThreadActionConfiguration::operator!=(LuaFFIPerThreadActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadActionConfiguration$operator$ne(*this, rhs);
}

bool NegativeAndSOAActionConfiguration::operator==(NegativeAndSOAActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NegativeAndSOAActionConfiguration$operator$eq(*this, rhs);
}

bool NegativeAndSOAActionConfiguration::operator!=(NegativeAndSOAActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NegativeAndSOAActionConfiguration$operator$ne(*this, rhs);
}

bool NoneActionConfiguration::operator==(NoneActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NoneActionConfiguration$operator$eq(*this, rhs);
}

bool NoneActionConfiguration::operator!=(NoneActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NoneActionConfiguration$operator$ne(*this, rhs);
}

bool PoolActionConfiguration::operator==(PoolActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolActionConfiguration$operator$eq(*this, rhs);
}

bool PoolActionConfiguration::operator!=(PoolActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolActionConfiguration$operator$ne(*this, rhs);
}

bool QPSActionConfiguration::operator==(QPSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QPSActionConfiguration$operator$eq(*this, rhs);
}

bool QPSActionConfiguration::operator!=(QPSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QPSActionConfiguration$operator$ne(*this, rhs);
}

bool QPSPoolActionConfiguration::operator==(QPSPoolActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QPSPoolActionConfiguration$operator$eq(*this, rhs);
}

bool QPSPoolActionConfiguration::operator!=(QPSPoolActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QPSPoolActionConfiguration$operator$ne(*this, rhs);
}

bool RCodeActionConfiguration::operator==(RCodeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RCodeActionConfiguration$operator$eq(*this, rhs);
}

bool RCodeActionConfiguration::operator!=(RCodeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RCodeActionConfiguration$operator$ne(*this, rhs);
}

bool RemoteLogActionConfiguration::operator==(RemoteLogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLogActionConfiguration$operator$eq(*this, rhs);
}

bool RemoteLogActionConfiguration::operator!=(RemoteLogActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLogActionConfiguration$operator$ne(*this, rhs);
}

bool SetAdditionalProxyProtocolValueActionConfiguration::operator==(SetAdditionalProxyProtocolValueActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetAdditionalProxyProtocolValueActionConfiguration$operator$eq(*this, rhs);
}

bool SetAdditionalProxyProtocolValueActionConfiguration::operator!=(SetAdditionalProxyProtocolValueActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetAdditionalProxyProtocolValueActionConfiguration$operator$ne(*this, rhs);
}

bool SetDisableECSActionConfiguration::operator==(SetDisableECSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetDisableECSActionConfiguration$operator$eq(*this, rhs);
}

bool SetDisableECSActionConfiguration::operator!=(SetDisableECSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetDisableECSActionConfiguration$operator$ne(*this, rhs);
}

bool SetDisableValidationActionConfiguration::operator==(SetDisableValidationActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetDisableValidationActionConfiguration$operator$eq(*this, rhs);
}

bool SetDisableValidationActionConfiguration::operator!=(SetDisableValidationActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetDisableValidationActionConfiguration$operator$ne(*this, rhs);
}

bool SetECSActionConfiguration::operator==(SetECSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSActionConfiguration$operator$eq(*this, rhs);
}

bool SetECSActionConfiguration::operator!=(SetECSActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSActionConfiguration$operator$ne(*this, rhs);
}

bool SetECSOverrideActionConfiguration::operator==(SetECSOverrideActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSOverrideActionConfiguration$operator$eq(*this, rhs);
}

bool SetECSOverrideActionConfiguration::operator!=(SetECSOverrideActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSOverrideActionConfiguration$operator$ne(*this, rhs);
}

bool SetECSPrefixLengthActionConfiguration::operator==(SetECSPrefixLengthActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSPrefixLengthActionConfiguration$operator$eq(*this, rhs);
}

bool SetECSPrefixLengthActionConfiguration::operator!=(SetECSPrefixLengthActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetECSPrefixLengthActionConfiguration$operator$ne(*this, rhs);
}

bool SetExtendedDNSErrorActionConfiguration::operator==(SetExtendedDNSErrorActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorActionConfiguration$operator$eq(*this, rhs);
}

bool SetExtendedDNSErrorActionConfiguration::operator!=(SetExtendedDNSErrorActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorActionConfiguration$operator$ne(*this, rhs);
}

bool SetMacAddrActionConfiguration::operator==(SetMacAddrActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMacAddrActionConfiguration$operator$eq(*this, rhs);
}

bool SetMacAddrActionConfiguration::operator!=(SetMacAddrActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMacAddrActionConfiguration$operator$ne(*this, rhs);
}

bool SetMaxReturnedTTLActionConfiguration::operator==(SetMaxReturnedTTLActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLActionConfiguration$operator$eq(*this, rhs);
}

bool SetMaxReturnedTTLActionConfiguration::operator!=(SetMaxReturnedTTLActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLActionConfiguration$operator$ne(*this, rhs);
}

bool SetNoRecurseActionConfiguration::operator==(SetNoRecurseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetNoRecurseActionConfiguration$operator$eq(*this, rhs);
}

bool SetNoRecurseActionConfiguration::operator!=(SetNoRecurseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetNoRecurseActionConfiguration$operator$ne(*this, rhs);
}

bool SetProxyProtocolValuesActionConfiguration::operator==(SetProxyProtocolValuesActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetProxyProtocolValuesActionConfiguration$operator$eq(*this, rhs);
}

bool SetProxyProtocolValuesActionConfiguration::operator!=(SetProxyProtocolValuesActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetProxyProtocolValuesActionConfiguration$operator$ne(*this, rhs);
}

bool SetSkipCacheActionConfiguration::operator==(SetSkipCacheActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetSkipCacheActionConfiguration$operator$eq(*this, rhs);
}

bool SetSkipCacheActionConfiguration::operator!=(SetSkipCacheActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetSkipCacheActionConfiguration$operator$ne(*this, rhs);
}

bool SetTagActionConfiguration::operator==(SetTagActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTagActionConfiguration$operator$eq(*this, rhs);
}

bool SetTagActionConfiguration::operator!=(SetTagActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTagActionConfiguration$operator$ne(*this, rhs);
}

bool SetTempFailureCacheTTLActionConfiguration::operator==(SetTempFailureCacheTTLActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTempFailureCacheTTLActionConfiguration$operator$eq(*this, rhs);
}

bool SetTempFailureCacheTTLActionConfiguration::operator!=(SetTempFailureCacheTTLActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTempFailureCacheTTLActionConfiguration$operator$ne(*this, rhs);
}

bool SNMPTrapActionConfiguration::operator==(SNMPTrapActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNMPTrapActionConfiguration$operator$eq(*this, rhs);
}

bool SNMPTrapActionConfiguration::operator!=(SNMPTrapActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNMPTrapActionConfiguration$operator$ne(*this, rhs);
}

bool SpoofActionConfiguration::operator==(SpoofActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofActionConfiguration$operator$eq(*this, rhs);
}

bool SpoofActionConfiguration::operator!=(SpoofActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofActionConfiguration$operator$ne(*this, rhs);
}

bool SpoofCNAMEActionConfiguration::operator==(SpoofCNAMEActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofCNAMEActionConfiguration$operator$eq(*this, rhs);
}

bool SpoofCNAMEActionConfiguration::operator!=(SpoofCNAMEActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofCNAMEActionConfiguration$operator$ne(*this, rhs);
}

bool SpoofPacketActionConfiguration::operator==(SpoofPacketActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofPacketActionConfiguration$operator$eq(*this, rhs);
}

bool SpoofPacketActionConfiguration::operator!=(SpoofPacketActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofPacketActionConfiguration$operator$ne(*this, rhs);
}

bool SpoofRawActionConfiguration::operator==(SpoofRawActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofRawActionConfiguration$operator$eq(*this, rhs);
}

bool SpoofRawActionConfiguration::operator!=(SpoofRawActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofRawActionConfiguration$operator$ne(*this, rhs);
}

bool SpoofSVCActionConfiguration::operator==(SpoofSVCActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofSVCActionConfiguration$operator$eq(*this, rhs);
}

bool SpoofSVCActionConfiguration::operator!=(SpoofSVCActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SpoofSVCActionConfiguration$operator$ne(*this, rhs);
}

bool TCActionConfiguration::operator==(TCActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCActionConfiguration$operator$eq(*this, rhs);
}

bool TCActionConfiguration::operator!=(TCActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCActionConfiguration$operator$ne(*this, rhs);
}

bool TeeActionConfiguration::operator==(TeeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TeeActionConfiguration$operator$eq(*this, rhs);
}

bool TeeActionConfiguration::operator!=(TeeActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TeeActionConfiguration$operator$ne(*this, rhs);
}

bool AllowResponseActionConfiguration::operator==(AllowResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllowResponseActionConfiguration$operator$eq(*this, rhs);
}

bool AllowResponseActionConfiguration::operator!=(AllowResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllowResponseActionConfiguration$operator$ne(*this, rhs);
}

bool ClearRecordTypesResponseActionConfiguration::operator==(ClearRecordTypesResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ClearRecordTypesResponseActionConfiguration$operator$eq(*this, rhs);
}

bool ClearRecordTypesResponseActionConfiguration::operator!=(ClearRecordTypesResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ClearRecordTypesResponseActionConfiguration$operator$ne(*this, rhs);
}

bool DelayResponseActionConfiguration::operator==(DelayResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DelayResponseActionConfiguration$operator$eq(*this, rhs);
}

bool DelayResponseActionConfiguration::operator!=(DelayResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DelayResponseActionConfiguration$operator$ne(*this, rhs);
}

bool DnstapLogResponseActionConfiguration::operator==(DnstapLogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLogResponseActionConfiguration$operator$eq(*this, rhs);
}

bool DnstapLogResponseActionConfiguration::operator!=(DnstapLogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLogResponseActionConfiguration$operator$ne(*this, rhs);
}

bool DropResponseActionConfiguration::operator==(DropResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DropResponseActionConfiguration$operator$eq(*this, rhs);
}

bool DropResponseActionConfiguration::operator!=(DropResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DropResponseActionConfiguration$operator$ne(*this, rhs);
}

bool LimitTTLResponseActionConfiguration::operator==(LimitTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LimitTTLResponseActionConfiguration$operator$eq(*this, rhs);
}

bool LimitTTLResponseActionConfiguration::operator!=(LimitTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LimitTTLResponseActionConfiguration$operator$ne(*this, rhs);
}

bool LogResponseActionConfiguration::operator==(LogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LogResponseActionConfiguration$operator$eq(*this, rhs);
}

bool LogResponseActionConfiguration::operator!=(LogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LogResponseActionConfiguration$operator$ne(*this, rhs);
}

bool LuaResponseActionConfiguration::operator==(LuaResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaResponseActionConfiguration$operator$eq(*this, rhs);
}

bool LuaResponseActionConfiguration::operator!=(LuaResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaResponseActionConfiguration$operator$ne(*this, rhs);
}

bool LuaFFIResponseActionConfiguration::operator==(LuaFFIResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIResponseActionConfiguration$operator$eq(*this, rhs);
}

bool LuaFFIResponseActionConfiguration::operator!=(LuaFFIResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIResponseActionConfiguration$operator$ne(*this, rhs);
}

bool LuaFFIPerThreadResponseActionConfiguration::operator==(LuaFFIPerThreadResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadResponseActionConfiguration$operator$eq(*this, rhs);
}

bool LuaFFIPerThreadResponseActionConfiguration::operator!=(LuaFFIPerThreadResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadResponseActionConfiguration$operator$ne(*this, rhs);
}

bool RemoteLogResponseActionConfiguration::operator==(RemoteLogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLogResponseActionConfiguration$operator$eq(*this, rhs);
}

bool RemoteLogResponseActionConfiguration::operator!=(RemoteLogResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLogResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetEDNSOptionResponseActionConfiguration::operator==(SetEDNSOptionResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetEDNSOptionResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetEDNSOptionResponseActionConfiguration::operator!=(SetEDNSOptionResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetEDNSOptionResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetExtendedDNSErrorResponseActionConfiguration::operator==(SetExtendedDNSErrorResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetExtendedDNSErrorResponseActionConfiguration::operator!=(SetExtendedDNSErrorResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetExtendedDNSErrorResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetMaxReturnedTTLResponseActionConfiguration::operator==(SetMaxReturnedTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetMaxReturnedTTLResponseActionConfiguration::operator!=(SetMaxReturnedTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxReturnedTTLResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetMaxTTLResponseActionConfiguration::operator==(SetMaxTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxTTLResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetMaxTTLResponseActionConfiguration::operator!=(SetMaxTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMaxTTLResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetMinTTLResponseActionConfiguration::operator==(SetMinTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMinTTLResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetMinTTLResponseActionConfiguration::operator!=(SetMinTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetMinTTLResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetReducedTTLResponseActionConfiguration::operator==(SetReducedTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetReducedTTLResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetReducedTTLResponseActionConfiguration::operator!=(SetReducedTTLResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetReducedTTLResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetSkipCacheResponseActionConfiguration::operator==(SetSkipCacheResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetSkipCacheResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetSkipCacheResponseActionConfiguration::operator!=(SetSkipCacheResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetSkipCacheResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SetTagResponseActionConfiguration::operator==(SetTagResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTagResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SetTagResponseActionConfiguration::operator!=(SetTagResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SetTagResponseActionConfiguration$operator$ne(*this, rhs);
}

bool SNMPTrapResponseActionConfiguration::operator==(SNMPTrapResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNMPTrapResponseActionConfiguration$operator$eq(*this, rhs);
}

bool SNMPTrapResponseActionConfiguration::operator!=(SNMPTrapResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNMPTrapResponseActionConfiguration$operator$ne(*this, rhs);
}

bool TCResponseActionConfiguration::operator==(TCResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCResponseActionConfiguration$operator$eq(*this, rhs);
}

bool TCResponseActionConfiguration::operator!=(TCResponseActionConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCResponseActionConfiguration$operator$ne(*this, rhs);
}

bool AllSelectorConfiguration::operator==(AllSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllSelectorConfiguration$operator$eq(*this, rhs);
}

bool AllSelectorConfiguration::operator!=(AllSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$AllSelectorConfiguration$operator$ne(*this, rhs);
}

bool ByNameSelectorConfiguration::operator==(ByNameSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ByNameSelectorConfiguration$operator$eq(*this, rhs);
}

bool ByNameSelectorConfiguration::operator!=(ByNameSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ByNameSelectorConfiguration$operator$ne(*this, rhs);
}

bool DNSSECSelectorConfiguration::operator==(DNSSECSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DNSSECSelectorConfiguration$operator$eq(*this, rhs);
}

bool DNSSECSelectorConfiguration::operator!=(DNSSECSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DNSSECSelectorConfiguration$operator$ne(*this, rhs);
}

bool DSTPortSelectorConfiguration::operator==(DSTPortSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DSTPortSelectorConfiguration$operator$eq(*this, rhs);
}

bool DSTPortSelectorConfiguration::operator!=(DSTPortSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DSTPortSelectorConfiguration$operator$ne(*this, rhs);
}

bool EDNSOptionSelectorConfiguration::operator==(EDNSOptionSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EDNSOptionSelectorConfiguration$operator$eq(*this, rhs);
}

bool EDNSOptionSelectorConfiguration::operator!=(EDNSOptionSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EDNSOptionSelectorConfiguration$operator$ne(*this, rhs);
}

bool EDNSVersionSelectorConfiguration::operator==(EDNSVersionSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EDNSVersionSelectorConfiguration$operator$eq(*this, rhs);
}

bool EDNSVersionSelectorConfiguration::operator!=(EDNSVersionSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EDNSVersionSelectorConfiguration$operator$ne(*this, rhs);
}

bool ERCodeSelectorConfiguration::operator==(ERCodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ERCodeSelectorConfiguration$operator$eq(*this, rhs);
}

bool ERCodeSelectorConfiguration::operator!=(ERCodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ERCodeSelectorConfiguration$operator$ne(*this, rhs);
}

bool HTTPHeaderSelectorConfiguration::operator==(HTTPHeaderSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPHeaderSelectorConfiguration$operator$eq(*this, rhs);
}

bool HTTPHeaderSelectorConfiguration::operator!=(HTTPHeaderSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPHeaderSelectorConfiguration$operator$ne(*this, rhs);
}

bool HTTPPathSelectorConfiguration::operator==(HTTPPathSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPPathSelectorConfiguration$operator$eq(*this, rhs);
}

bool HTTPPathSelectorConfiguration::operator!=(HTTPPathSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPPathSelectorConfiguration$operator$ne(*this, rhs);
}

bool HTTPPathRegexSelectorConfiguration::operator==(HTTPPathRegexSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPPathRegexSelectorConfiguration$operator$eq(*this, rhs);
}

bool HTTPPathRegexSelectorConfiguration::operator!=(HTTPPathRegexSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HTTPPathRegexSelectorConfiguration$operator$ne(*this, rhs);
}

bool KeyValueStoreLookupSelectorConfiguration::operator==(KeyValueStoreLookupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupSelectorConfiguration$operator$eq(*this, rhs);
}

bool KeyValueStoreLookupSelectorConfiguration::operator!=(KeyValueStoreLookupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreLookupSelectorConfiguration$operator$ne(*this, rhs);
}

bool KeyValueStoreRangeLookupSelectorConfiguration::operator==(KeyValueStoreRangeLookupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupSelectorConfiguration$operator$eq(*this, rhs);
}

bool KeyValueStoreRangeLookupSelectorConfiguration::operator!=(KeyValueStoreRangeLookupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoreRangeLookupSelectorConfiguration$operator$ne(*this, rhs);
}

bool LuaSelectorConfiguration::operator==(LuaSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaSelectorConfiguration$operator$eq(*this, rhs);
}

bool LuaSelectorConfiguration::operator!=(LuaSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaSelectorConfiguration$operator$ne(*this, rhs);
}

bool LuaFFISelectorConfiguration::operator==(LuaFFISelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFISelectorConfiguration$operator$eq(*this, rhs);
}

bool LuaFFISelectorConfiguration::operator!=(LuaFFISelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFISelectorConfiguration$operator$ne(*this, rhs);
}

bool LuaFFIPerThreadSelectorConfiguration::operator==(LuaFFIPerThreadSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadSelectorConfiguration$operator$eq(*this, rhs);
}

bool LuaFFIPerThreadSelectorConfiguration::operator!=(LuaFFIPerThreadSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LuaFFIPerThreadSelectorConfiguration$operator$ne(*this, rhs);
}

bool MaxQPSSelectorConfiguration::operator==(MaxQPSSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MaxQPSSelectorConfiguration$operator$eq(*this, rhs);
}

bool MaxQPSSelectorConfiguration::operator!=(MaxQPSSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MaxQPSSelectorConfiguration$operator$ne(*this, rhs);
}

bool MaxQPSIPSelectorConfiguration::operator==(MaxQPSIPSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MaxQPSIPSelectorConfiguration$operator$eq(*this, rhs);
}

bool MaxQPSIPSelectorConfiguration::operator!=(MaxQPSIPSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MaxQPSIPSelectorConfiguration$operator$ne(*this, rhs);
}

bool NetmaskGroupSelectorConfiguration::operator==(NetmaskGroupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NetmaskGroupSelectorConfiguration$operator$eq(*this, rhs);
}

bool NetmaskGroupSelectorConfiguration::operator!=(NetmaskGroupSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NetmaskGroupSelectorConfiguration$operator$ne(*this, rhs);
}

bool OpcodeSelectorConfiguration::operator==(OpcodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OpcodeSelectorConfiguration$operator$eq(*this, rhs);
}

bool OpcodeSelectorConfiguration::operator!=(OpcodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OpcodeSelectorConfiguration$operator$ne(*this, rhs);
}

bool PayloadSizeSelectorConfiguration::operator==(PayloadSizeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PayloadSizeSelectorConfiguration$operator$eq(*this, rhs);
}

bool PayloadSizeSelectorConfiguration::operator!=(PayloadSizeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PayloadSizeSelectorConfiguration$operator$ne(*this, rhs);
}

bool PoolAvailableSelectorConfiguration::operator==(PoolAvailableSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolAvailableSelectorConfiguration$operator$eq(*this, rhs);
}

bool PoolAvailableSelectorConfiguration::operator!=(PoolAvailableSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolAvailableSelectorConfiguration$operator$ne(*this, rhs);
}

bool PoolOutstandingSelectorConfiguration::operator==(PoolOutstandingSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolOutstandingSelectorConfiguration$operator$eq(*this, rhs);
}

bool PoolOutstandingSelectorConfiguration::operator!=(PoolOutstandingSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolOutstandingSelectorConfiguration$operator$ne(*this, rhs);
}

bool ProbaSelectorConfiguration::operator==(ProbaSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProbaSelectorConfiguration$operator$eq(*this, rhs);
}

bool ProbaSelectorConfiguration::operator!=(ProbaSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProbaSelectorConfiguration$operator$ne(*this, rhs);
}

bool ProxyProtocolValueSelectorConfiguration::operator==(ProxyProtocolValueSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueSelectorConfiguration$operator$eq(*this, rhs);
}

bool ProxyProtocolValueSelectorConfiguration::operator!=(ProxyProtocolValueSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueSelectorConfiguration$operator$ne(*this, rhs);
}

bool QClassSelectorConfiguration::operator==(QClassSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QClassSelectorConfiguration$operator$eq(*this, rhs);
}

bool QClassSelectorConfiguration::operator!=(QClassSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QClassSelectorConfiguration$operator$ne(*this, rhs);
}

bool QNameSelectorConfiguration::operator==(QNameSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSelectorConfiguration$operator$eq(*this, rhs);
}

bool QNameSelectorConfiguration::operator!=(QNameSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSelectorConfiguration$operator$ne(*this, rhs);
}

bool QNameLabelsCountSelectorConfiguration::operator==(QNameLabelsCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameLabelsCountSelectorConfiguration$operator$eq(*this, rhs);
}

bool QNameLabelsCountSelectorConfiguration::operator!=(QNameLabelsCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameLabelsCountSelectorConfiguration$operator$ne(*this, rhs);
}

bool QNameSetSelectorConfiguration::operator==(QNameSetSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSetSelectorConfiguration$operator$eq(*this, rhs);
}

bool QNameSetSelectorConfiguration::operator!=(QNameSetSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSetSelectorConfiguration$operator$ne(*this, rhs);
}

bool QNameSuffixSelectorConfiguration::operator==(QNameSuffixSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSuffixSelectorConfiguration$operator$eq(*this, rhs);
}

bool QNameSuffixSelectorConfiguration::operator!=(QNameSuffixSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameSuffixSelectorConfiguration$operator$ne(*this, rhs);
}

bool QNameWireLengthSelectorConfiguration::operator==(QNameWireLengthSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameWireLengthSelectorConfiguration$operator$eq(*this, rhs);
}

bool QNameWireLengthSelectorConfiguration::operator!=(QNameWireLengthSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QNameWireLengthSelectorConfiguration$operator$ne(*this, rhs);
}

bool QTypeSelectorConfiguration::operator==(QTypeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QTypeSelectorConfiguration$operator$eq(*this, rhs);
}

bool QTypeSelectorConfiguration::operator!=(QTypeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QTypeSelectorConfiguration$operator$ne(*this, rhs);
}

bool RCodeSelectorConfiguration::operator==(RCodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RCodeSelectorConfiguration$operator$eq(*this, rhs);
}

bool RCodeSelectorConfiguration::operator!=(RCodeSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RCodeSelectorConfiguration$operator$ne(*this, rhs);
}

bool RDSelectorConfiguration::operator==(RDSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RDSelectorConfiguration$operator$eq(*this, rhs);
}

bool RDSelectorConfiguration::operator!=(RDSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RDSelectorConfiguration$operator$ne(*this, rhs);
}

bool RE2SelectorConfiguration::operator==(RE2SelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RE2SelectorConfiguration$operator$eq(*this, rhs);
}

bool RE2SelectorConfiguration::operator!=(RE2SelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RE2SelectorConfiguration$operator$ne(*this, rhs);
}

bool RecordsCountSelectorConfiguration::operator==(RecordsCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RecordsCountSelectorConfiguration$operator$eq(*this, rhs);
}

bool RecordsCountSelectorConfiguration::operator!=(RecordsCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RecordsCountSelectorConfiguration$operator$ne(*this, rhs);
}

bool RecordsTypeCountSelectorConfiguration::operator==(RecordsTypeCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RecordsTypeCountSelectorConfiguration$operator$eq(*this, rhs);
}

bool RecordsTypeCountSelectorConfiguration::operator!=(RecordsTypeCountSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RecordsTypeCountSelectorConfiguration$operator$ne(*this, rhs);
}

bool RegexSelectorConfiguration::operator==(RegexSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RegexSelectorConfiguration$operator$eq(*this, rhs);
}

bool RegexSelectorConfiguration::operator!=(RegexSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RegexSelectorConfiguration$operator$ne(*this, rhs);
}

bool SNISelectorConfiguration::operator==(SNISelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNISelectorConfiguration$operator$eq(*this, rhs);
}

bool SNISelectorConfiguration::operator!=(SNISelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SNISelectorConfiguration$operator$ne(*this, rhs);
}

bool TagSelectorConfiguration::operator==(TagSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TagSelectorConfiguration$operator$eq(*this, rhs);
}

bool TagSelectorConfiguration::operator!=(TagSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TagSelectorConfiguration$operator$ne(*this, rhs);
}

bool TCPSelectorConfiguration::operator==(TCPSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCPSelectorConfiguration$operator$eq(*this, rhs);
}

bool TCPSelectorConfiguration::operator!=(TCPSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TCPSelectorConfiguration$operator$ne(*this, rhs);
}

bool TrailingDataSelectorConfiguration::operator==(TrailingDataSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TrailingDataSelectorConfiguration$operator$eq(*this, rhs);
}

bool TrailingDataSelectorConfiguration::operator!=(TrailingDataSelectorConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TrailingDataSelectorConfiguration$operator$ne(*this, rhs);
}

bool MetricsConfiguration::operator==(MetricsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MetricsConfiguration$operator$eq(*this, rhs);
}

bool MetricsConfiguration::operator!=(MetricsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$MetricsConfiguration$operator$ne(*this, rhs);
}

bool CarbonConfiguration::operator==(CarbonConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CarbonConfiguration$operator$eq(*this, rhs);
}

bool CarbonConfiguration::operator!=(CarbonConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CarbonConfiguration$operator$ne(*this, rhs);
}

bool RemoteLoggingConfiguration::operator==(RemoteLoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLoggingConfiguration$operator$eq(*this, rhs);
}

bool RemoteLoggingConfiguration::operator!=(RemoteLoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RemoteLoggingConfiguration$operator$ne(*this, rhs);
}

bool ProtobufLoggerConfiguration::operator==(ProtobufLoggerConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProtobufLoggerConfiguration$operator$eq(*this, rhs);
}

bool ProtobufLoggerConfiguration::operator!=(ProtobufLoggerConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProtobufLoggerConfiguration$operator$ne(*this, rhs);
}

bool DnstapLoggerConfiguration::operator==(DnstapLoggerConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLoggerConfiguration$operator$eq(*this, rhs);
}

bool DnstapLoggerConfiguration::operator!=(DnstapLoggerConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DnstapLoggerConfiguration$operator$ne(*this, rhs);
}

bool ProtoBufMetaConfiguration::operator==(ProtoBufMetaConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProtoBufMetaConfiguration$operator$eq(*this, rhs);
}

bool ProtoBufMetaConfiguration::operator!=(ProtoBufMetaConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProtoBufMetaConfiguration$operator$ne(*this, rhs);
}

bool LmdbKvStoreConfiguration::operator==(LmdbKvStoreConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LmdbKvStoreConfiguration$operator$eq(*this, rhs);
}

bool LmdbKvStoreConfiguration::operator!=(LmdbKvStoreConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LmdbKvStoreConfiguration$operator$ne(*this, rhs);
}

bool CdbKvStoreConfiguration::operator==(CdbKvStoreConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CdbKvStoreConfiguration$operator$eq(*this, rhs);
}

bool CdbKvStoreConfiguration::operator!=(CdbKvStoreConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CdbKvStoreConfiguration$operator$ne(*this, rhs);
}

bool KvsLookupKeySourceIpConfiguration::operator==(KvsLookupKeySourceIpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeySourceIpConfiguration$operator$eq(*this, rhs);
}

bool KvsLookupKeySourceIpConfiguration::operator!=(KvsLookupKeySourceIpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeySourceIpConfiguration$operator$ne(*this, rhs);
}

bool KvsLookupKeyQnameConfiguration::operator==(KvsLookupKeyQnameConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeyQnameConfiguration$operator$eq(*this, rhs);
}

bool KvsLookupKeyQnameConfiguration::operator!=(KvsLookupKeyQnameConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeyQnameConfiguration$operator$ne(*this, rhs);
}

bool KvsLookupKeySuffixConfiguration::operator==(KvsLookupKeySuffixConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeySuffixConfiguration$operator$eq(*this, rhs);
}

bool KvsLookupKeySuffixConfiguration::operator!=(KvsLookupKeySuffixConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeySuffixConfiguration$operator$ne(*this, rhs);
}

bool KvsLookupKeyTagConfiguration::operator==(KvsLookupKeyTagConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeyTagConfiguration$operator$eq(*this, rhs);
}

bool KvsLookupKeyTagConfiguration::operator!=(KvsLookupKeyTagConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeyTagConfiguration$operator$ne(*this, rhs);
}

bool KvsLookupKeysConfiguration::operator==(KvsLookupKeysConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeysConfiguration$operator$eq(*this, rhs);
}

bool KvsLookupKeysConfiguration::operator!=(KvsLookupKeysConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KvsLookupKeysConfiguration$operator$ne(*this, rhs);
}

bool KeyValueStoresConfiguration::operator==(KeyValueStoresConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoresConfiguration$operator$eq(*this, rhs);
}

bool KeyValueStoresConfiguration::operator!=(KeyValueStoresConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$KeyValueStoresConfiguration$operator$ne(*this, rhs);
}

bool WebserverConfiguration::operator==(WebserverConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$WebserverConfiguration$operator$eq(*this, rhs);
}

bool WebserverConfiguration::operator!=(WebserverConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$WebserverConfiguration$operator$ne(*this, rhs);
}

bool ConsoleConfiguration::operator==(ConsoleConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ConsoleConfiguration$operator$eq(*this, rhs);
}

bool ConsoleConfiguration::operator!=(ConsoleConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ConsoleConfiguration$operator$ne(*this, rhs);
}

bool EbpfMapConfiguration::operator==(EbpfMapConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EbpfMapConfiguration$operator$eq(*this, rhs);
}

bool EbpfMapConfiguration::operator!=(EbpfMapConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EbpfMapConfiguration$operator$ne(*this, rhs);
}

bool EbpfConfiguration::operator==(EbpfConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EbpfConfiguration$operator$eq(*this, rhs);
}

bool EbpfConfiguration::operator!=(EbpfConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EbpfConfiguration$operator$ne(*this, rhs);
}

bool EdnsClientSubnetConfiguration::operator==(EdnsClientSubnetConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EdnsClientSubnetConfiguration$operator$eq(*this, rhs);
}

bool EdnsClientSubnetConfiguration::operator!=(EdnsClientSubnetConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$EdnsClientSubnetConfiguration$operator$ne(*this, rhs);
}

bool DynamicRulesSettingsConfiguration::operator==(DynamicRulesSettingsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRulesSettingsConfiguration$operator$eq(*this, rhs);
}

bool DynamicRulesSettingsConfiguration::operator!=(DynamicRulesSettingsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRulesSettingsConfiguration$operator$ne(*this, rhs);
}

bool DynamicRuleConfiguration::operator==(DynamicRuleConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRuleConfiguration$operator$eq(*this, rhs);
}

bool DynamicRuleConfiguration::operator!=(DynamicRuleConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRuleConfiguration$operator$ne(*this, rhs);
}

bool DynamicRulesConfiguration::operator==(DynamicRulesConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRulesConfiguration$operator$eq(*this, rhs);
}

bool DynamicRulesConfiguration::operator!=(DynamicRulesConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DynamicRulesConfiguration$operator$ne(*this, rhs);
}

bool RingBuffersConfiguration::operator==(RingBuffersConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RingBuffersConfiguration$operator$eq(*this, rhs);
}

bool RingBuffersConfiguration::operator!=(RingBuffersConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$RingBuffersConfiguration$operator$ne(*this, rhs);
}

bool IncomingTlsCertificateKeyPairConfiguration::operator==(IncomingTlsCertificateKeyPairConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTlsCertificateKeyPairConfiguration$operator$eq(*this, rhs);
}

bool IncomingTlsCertificateKeyPairConfiguration::operator!=(IncomingTlsCertificateKeyPairConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTlsCertificateKeyPairConfiguration$operator$ne(*this, rhs);
}

bool IncomingTlsConfiguration::operator==(IncomingTlsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTlsConfiguration$operator$eq(*this, rhs);
}

bool IncomingTlsConfiguration::operator!=(IncomingTlsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTlsConfiguration$operator$ne(*this, rhs);
}

bool OutgoingTlsConfiguration::operator==(OutgoingTlsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingTlsConfiguration$operator$eq(*this, rhs);
}

bool OutgoingTlsConfiguration::operator!=(OutgoingTlsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingTlsConfiguration$operator$ne(*this, rhs);
}

bool HttpCustomResponseHeaderConfiguration::operator==(HttpCustomResponseHeaderConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HttpCustomResponseHeaderConfiguration$operator$eq(*this, rhs);
}

bool HttpCustomResponseHeaderConfiguration::operator!=(HttpCustomResponseHeaderConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HttpCustomResponseHeaderConfiguration$operator$ne(*this, rhs);
}

bool HttpResponsesMapConfiguration::operator==(HttpResponsesMapConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HttpResponsesMapConfiguration$operator$eq(*this, rhs);
}

bool HttpResponsesMapConfiguration::operator!=(HttpResponsesMapConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HttpResponsesMapConfiguration$operator$ne(*this, rhs);
}

bool IncomingDohConfiguration::operator==(IncomingDohConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDohConfiguration$operator$eq(*this, rhs);
}

bool IncomingDohConfiguration::operator!=(IncomingDohConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDohConfiguration$operator$ne(*this, rhs);
}

bool IncomingDoqConfiguration::operator==(IncomingDoqConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDoqConfiguration$operator$eq(*this, rhs);
}

bool IncomingDoqConfiguration::operator!=(IncomingDoqConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDoqConfiguration$operator$ne(*this, rhs);
}

bool IncomingQuicConfiguration::operator==(IncomingQuicConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingQuicConfiguration$operator$eq(*this, rhs);
}

bool IncomingQuicConfiguration::operator!=(IncomingQuicConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingQuicConfiguration$operator$ne(*this, rhs);
}

bool IncomingDnscryptCertificateKeyPairConfiguration::operator==(IncomingDnscryptCertificateKeyPairConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDnscryptCertificateKeyPairConfiguration$operator$eq(*this, rhs);
}

bool IncomingDnscryptCertificateKeyPairConfiguration::operator!=(IncomingDnscryptCertificateKeyPairConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDnscryptCertificateKeyPairConfiguration$operator$ne(*this, rhs);
}

bool IncomingDnscryptConfiguration::operator==(IncomingDnscryptConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDnscryptConfiguration$operator$eq(*this, rhs);
}

bool IncomingDnscryptConfiguration::operator!=(IncomingDnscryptConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingDnscryptConfiguration$operator$ne(*this, rhs);
}

bool OutgoingDohConfiguration::operator==(OutgoingDohConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingDohConfiguration$operator$eq(*this, rhs);
}

bool OutgoingDohConfiguration::operator!=(OutgoingDohConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingDohConfiguration$operator$ne(*this, rhs);
}

bool IncomingTcpConfiguration::operator==(IncomingTcpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTcpConfiguration$operator$eq(*this, rhs);
}

bool IncomingTcpConfiguration::operator!=(IncomingTcpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$IncomingTcpConfiguration$operator$ne(*this, rhs);
}

bool BindConfiguration::operator==(BindConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$BindConfiguration$operator$eq(*this, rhs);
}

bool BindConfiguration::operator!=(BindConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$BindConfiguration$operator$ne(*this, rhs);
}

bool OutgoingTcpConfiguration::operator==(OutgoingTcpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingTcpConfiguration$operator$eq(*this, rhs);
}

bool OutgoingTcpConfiguration::operator!=(OutgoingTcpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingTcpConfiguration$operator$ne(*this, rhs);
}

bool ProxyProtocolValueConfiguration::operator==(ProxyProtocolValueConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueConfiguration$operator$eq(*this, rhs);
}

bool ProxyProtocolValueConfiguration::operator!=(ProxyProtocolValueConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolValueConfiguration$operator$ne(*this, rhs);
}

bool LazyHealthCheckConfiguration::operator==(LazyHealthCheckConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LazyHealthCheckConfiguration$operator$eq(*this, rhs);
}

bool LazyHealthCheckConfiguration::operator!=(LazyHealthCheckConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LazyHealthCheckConfiguration$operator$ne(*this, rhs);
}

bool HealthCheckConfiguration::operator==(HealthCheckConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HealthCheckConfiguration$operator$eq(*this, rhs);
}

bool HealthCheckConfiguration::operator!=(HealthCheckConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$HealthCheckConfiguration$operator$ne(*this, rhs);
}

bool OutgoingAutoUpgradeConfiguration::operator==(OutgoingAutoUpgradeConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingAutoUpgradeConfiguration$operator$eq(*this, rhs);
}

bool OutgoingAutoUpgradeConfiguration::operator!=(OutgoingAutoUpgradeConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$OutgoingAutoUpgradeConfiguration$operator$ne(*this, rhs);
}

bool BackendConfiguration::operator==(BackendConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$BackendConfiguration$operator$eq(*this, rhs);
}

bool BackendConfiguration::operator!=(BackendConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$BackendConfiguration$operator$ne(*this, rhs);
}

bool TuningConfiguration::operator==(TuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TuningConfiguration$operator$eq(*this, rhs);
}

bool TuningConfiguration::operator!=(TuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TuningConfiguration$operator$ne(*this, rhs);
}

bool TcpTuningConfiguration::operator==(TcpTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TcpTuningConfiguration$operator$eq(*this, rhs);
}

bool TcpTuningConfiguration::operator!=(TcpTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TcpTuningConfiguration$operator$ne(*this, rhs);
}

bool UdpTuningConfiguration::operator==(UdpTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$UdpTuningConfiguration$operator$eq(*this, rhs);
}

bool UdpTuningConfiguration::operator!=(UdpTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$UdpTuningConfiguration$operator$ne(*this, rhs);
}

bool TlsEngineConfiguration::operator==(TlsEngineConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TlsEngineConfiguration$operator$eq(*this, rhs);
}

bool TlsEngineConfiguration::operator!=(TlsEngineConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TlsEngineConfiguration$operator$ne(*this, rhs);
}

bool TlsTuningConfiguration::operator==(TlsTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TlsTuningConfiguration$operator$eq(*this, rhs);
}

bool TlsTuningConfiguration::operator!=(TlsTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$TlsTuningConfiguration$operator$ne(*this, rhs);
}

bool DohTuningConfiguration::operator==(DohTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DohTuningConfiguration$operator$eq(*this, rhs);
}

bool DohTuningConfiguration::operator!=(DohTuningConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$DohTuningConfiguration$operator$ne(*this, rhs);
}

bool CacheSettingsConfiguration::operator==(CacheSettingsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CacheSettingsConfiguration$operator$eq(*this, rhs);
}

bool CacheSettingsConfiguration::operator!=(CacheSettingsConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CacheSettingsConfiguration$operator$ne(*this, rhs);
}

bool SecurityPollingConfiguration::operator==(SecurityPollingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SecurityPollingConfiguration$operator$eq(*this, rhs);
}

bool SecurityPollingConfiguration::operator!=(SecurityPollingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SecurityPollingConfiguration$operator$ne(*this, rhs);
}

bool StructuredLoggingConfiguration::operator==(StructuredLoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$StructuredLoggingConfiguration$operator$eq(*this, rhs);
}

bool StructuredLoggingConfiguration::operator!=(StructuredLoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$StructuredLoggingConfiguration$operator$ne(*this, rhs);
}

bool LoggingConfiguration::operator==(LoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LoggingConfiguration$operator$eq(*this, rhs);
}

bool LoggingConfiguration::operator!=(LoggingConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LoggingConfiguration$operator$ne(*this, rhs);
}

bool GeneralConfiguration::operator==(GeneralConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$GeneralConfiguration$operator$eq(*this, rhs);
}

bool GeneralConfiguration::operator!=(GeneralConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$GeneralConfiguration$operator$ne(*this, rhs);
}

bool NetmaskGroupConfiguration::operator==(NetmaskGroupConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NetmaskGroupConfiguration$operator$eq(*this, rhs);
}

bool NetmaskGroupConfiguration::operator!=(NetmaskGroupConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$NetmaskGroupConfiguration$operator$ne(*this, rhs);
}

bool PacketCacheConfiguration::operator==(PacketCacheConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PacketCacheConfiguration$operator$eq(*this, rhs);
}

bool PacketCacheConfiguration::operator!=(PacketCacheConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PacketCacheConfiguration$operator$ne(*this, rhs);
}

bool ProxyProtocolConfiguration::operator==(ProxyProtocolConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolConfiguration$operator$eq(*this, rhs);
}

bool ProxyProtocolConfiguration::operator!=(ProxyProtocolConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$ProxyProtocolConfiguration$operator$ne(*this, rhs);
}

bool SnmpConfiguration::operator==(SnmpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SnmpConfiguration$operator$eq(*this, rhs);
}

bool SnmpConfiguration::operator!=(SnmpConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$SnmpConfiguration$operator$ne(*this, rhs);
}

bool QueryCountConfiguration::operator==(QueryCountConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QueryCountConfiguration$operator$eq(*this, rhs);
}

bool QueryCountConfiguration::operator!=(QueryCountConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$QueryCountConfiguration$operator$ne(*this, rhs);
}

bool PoolConfiguration::operator==(PoolConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolConfiguration$operator$eq(*this, rhs);
}

bool PoolConfiguration::operator!=(PoolConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$PoolConfiguration$operator$ne(*this, rhs);
}

bool CustomLoadBalancingPolicyConfiguration::operator==(CustomLoadBalancingPolicyConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CustomLoadBalancingPolicyConfiguration$operator$eq(*this, rhs);
}

bool CustomLoadBalancingPolicyConfiguration::operator!=(CustomLoadBalancingPolicyConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$CustomLoadBalancingPolicyConfiguration$operator$ne(*this, rhs);
}

bool LoadBalancingPoliciesConfiguration::operator==(LoadBalancingPoliciesConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LoadBalancingPoliciesConfiguration$operator$eq(*this, rhs);
}

bool LoadBalancingPoliciesConfiguration::operator!=(LoadBalancingPoliciesConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$LoadBalancingPoliciesConfiguration$operator$ne(*this, rhs);
}

bool XskConfiguration::operator==(XskConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$XskConfiguration$operator$eq(*this, rhs);
}

bool XskConfiguration::operator!=(XskConfiguration const &rhs) const noexcept {
  return dnsdist$rust$settings$cxxbridge1$XskConfiguration$operator$ne(*this, rhs);
}

::dnsdist::rust::settings::GlobalConfiguration from_yaml_string(::rust::Str str) {
  ::rust::MaybeUninit<::dnsdist::rust::settings::GlobalConfiguration> return$;
  ::rust::repr::PtrLen error$ = dnsdist$rust$settings$cxxbridge1$from_yaml_string(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}
} // namespace settings
} // namespace rust
} // namespace dnsdist

extern "C" {
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$new(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$drop(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$len(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$capacity(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> const *ptr) noexcept;
::dnsdist::rust::settings::SVCRecordAdditionalParams const *cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$data(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$reserve_total(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$set_len(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$truncate(::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> *ptr, ::std::size_t len) noexcept;

static_assert(sizeof(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>) == 2 * sizeof(void *), "");
static_assert(alignof(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>) == alignof(void *), "");
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSSelector$null(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSSelector$clone(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> const &self, ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector>(self);
}
::dnsdist::rust::settings::DNSSelector const *cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSSelector$get(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> const &self) noexcept {
  return self.get();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSSelector$drop(::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> *self) noexcept {
  self->~shared_ptr();
}

static_assert(sizeof(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>) == 2 * sizeof(void *), "");
static_assert(alignof(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>) == alignof(void *), "");
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSActionWrapper$null(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSActionWrapper$clone(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> const &self, ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper>(self);
}
::dnsdist::rust::settings::DNSActionWrapper const *cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSActionWrapper$get(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> const &self) noexcept {
  return self.get();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSActionWrapper$drop(::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> *self) noexcept {
  self->~shared_ptr();
}

static_assert(sizeof(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>) == 2 * sizeof(void *), "");
static_assert(alignof(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>) == alignof(void *), "");
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSResponseActionWrapper$null(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSResponseActionWrapper$clone(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> const &self, ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper>(self);
}
::dnsdist::rust::settings::DNSResponseActionWrapper const *cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSResponseActionWrapper$get(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> const &self) noexcept {
  return self.get();
}
void cxxbridge1$shared_ptr$dnsdist$rust$settings$DNSResponseActionWrapper$drop(::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> *self) noexcept {
  self->~shared_ptr();
}

void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$new(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$len(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::ProtoBufMetaConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$data(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$new(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$len(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::ProxyProtocolValueConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$data(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$new(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$drop(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$len(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$capacity(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> const *ptr) noexcept;
::dnsdist::rust::settings::SVCRecordParameters const *cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$data(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$reserve_total(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$set_len(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$truncate(::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$new(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$drop(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$len(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$capacity(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> const *ptr) noexcept;
::dnsdist::rust::settings::SharedDNSSelector const *cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$data(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$reserve_total(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$set_len(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$truncate(::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$new(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$len(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::BackendConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$data(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$new(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$len(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::BindConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$data(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::BindConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$new(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$len(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::ResponseRuleConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$data(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$new(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$len(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::QueryRuleConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$data(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$new(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$len(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::DynamicRulesConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$data(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$new(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$len(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::NetmaskGroupConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$data(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$new(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$len(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::PacketCacheConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$data(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$new(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$len(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::PoolConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$data(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$new(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$len(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::XskConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$data(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::XskConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$new(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$len(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::CarbonConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$data(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$new(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$len(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::ProtobufLoggerConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$data(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$new(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$len(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::DnstapLoggerConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$data(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$new(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$data(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$new(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$data(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$new(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$data(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$new(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::KvsLookupKeyTagConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$data(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$new(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$len(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::LmdbKvStoreConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$data(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$new(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$len(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::CdbKvStoreConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$data(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$new(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$len(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$data(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$new(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$len(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::DynamicRuleConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$data(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$new(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$len(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$data(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$new(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$len(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::HttpResponsesMapConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$data(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$new(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$len(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$data(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$new(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$len(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::TlsEngineConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$data(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$new(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$drop(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$len(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$capacity(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> const *ptr) noexcept;
::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration const *cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$data(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> const *ptr) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$reserve_total(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$set_len(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$truncate(::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> *ptr, ::std::size_t len) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$capacity(this);
}
template <>
::dnsdist::rust::settings::SVCRecordAdditionalParams const *Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordAdditionalParams$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::ProtoBufMetaConfiguration const *Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtoBufMetaConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::ProxyProtocolValueConfiguration const *Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProxyProtocolValueConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::SVCRecordParameters>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordParameters>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SVCRecordParameters>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SVCRecordParameters>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$capacity(this);
}
template <>
::dnsdist::rust::settings::SVCRecordParameters const *Vec<::dnsdist::rust::settings::SVCRecordParameters>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordParameters>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordParameters>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::SVCRecordParameters>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SVCRecordParameters$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::SharedDNSSelector>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::SharedDNSSelector>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SharedDNSSelector>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::SharedDNSSelector>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$capacity(this);
}
template <>
::dnsdist::rust::settings::SharedDNSSelector const *Vec<::dnsdist::rust::settings::SharedDNSSelector>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::SharedDNSSelector>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::SharedDNSSelector>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::SharedDNSSelector>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$SharedDNSSelector$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::BackendConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::BackendConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::BackendConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::BackendConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::BackendConfiguration const *Vec<::dnsdist::rust::settings::BackendConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::BackendConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::BackendConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::BackendConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BackendConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::BindConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::BindConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::BindConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::BindConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::BindConfiguration const *Vec<::dnsdist::rust::settings::BindConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::BindConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::BindConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::BindConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$BindConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::ResponseRuleConfiguration const *Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::ResponseRuleConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ResponseRuleConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::QueryRuleConfiguration const *Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::QueryRuleConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$QueryRuleConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::DynamicRulesConfiguration const *Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRulesConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRulesConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::NetmaskGroupConfiguration const *Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$NetmaskGroupConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::PacketCacheConfiguration const *Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::PacketCacheConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PacketCacheConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::PoolConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::PoolConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::PoolConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::PoolConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::PoolConfiguration const *Vec<::dnsdist::rust::settings::PoolConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::PoolConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::PoolConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::PoolConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$PoolConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::XskConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::XskConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::XskConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::XskConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::XskConfiguration const *Vec<::dnsdist::rust::settings::XskConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::XskConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::XskConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::XskConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$XskConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::CarbonConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::CarbonConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CarbonConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CarbonConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::CarbonConfiguration const *Vec<::dnsdist::rust::settings::CarbonConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::CarbonConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::CarbonConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::CarbonConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CarbonConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::ProtobufLoggerConfiguration const *Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$ProtobufLoggerConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::DnstapLoggerConfiguration const *Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DnstapLoggerConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration const *Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration const *Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyQnameConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration const *Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeySuffixConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::KvsLookupKeyTagConfiguration const *Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$KvsLookupKeyTagConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::LmdbKvStoreConfiguration const *Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$LmdbKvStoreConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::CdbKvStoreConfiguration const *Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CdbKvStoreConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration const *Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::DynamicRuleConfiguration const *Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::DynamicRuleConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$DynamicRuleConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration const *Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::HttpResponsesMapConfiguration const *Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$HttpResponsesMapConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration const *Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::TlsEngineConfiguration const *Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::TlsEngineConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$TlsEngineConfiguration$truncate(this, len);
}
template <>
Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::Vec() noexcept {
  cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$new(this);
}
template <>
void Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::drop() noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$drop(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::size() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$len(this);
}
template <>
::std::size_t Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::capacity() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$capacity(this);
}
template <>
::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration const *Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::data() const noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$data(this);
}
template <>
void Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$reserve_total(this, new_cap);
}
template <>
void Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$set_len(this, len);
}
template <>
void Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration$truncate(this, len);
}
} // namespace cxxbridge1
} // namespace rust
