
#line 1 "source/parser/TextFileParser.rl"
/*
 * Copyright (c) 2009-2022, Albertas Vyšniauskas
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *     * Neither the name of the software author nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "parser/TextFile.h"
#include "Color.h"
#include "math/Algorithms.h"
#include <cstring>
#include <functional>
#include <vector>
#include <string>
#include <cmath>
#include <algorithm>
#include <cstdint>
#include <memory>
#include <utility>
namespace text_file_parser {
enum struct Unit {
	unitless,
	percentage,
	degree,
	gradian,
	turn,
	radian,
};
struct FSM {
	int cs;
	int act;
	char ws;
	char *ts, *te;
	char buffer[8 * 1024];
	int line, column, lineStart, bufferOffset;
	int64_t numberI64;
	std::vector<std::pair<int64_t, Unit>> numbersI64;
	char *numberStart;
	std::vector<std::pair<double, Unit>> numbersDouble;
	std::function<void(const Color&)> addColor;
	void handleNewline() {
		line++;
		column = 0;
		lineStart = te - buffer;
	}
	int hexToInt(char hex) {
		if (hex >= '0' && hex <= '9') return hex - '0';
		if (hex >= 'a' && hex <= 'f') return hex - 'a' + 10;
		if (hex >= 'A' && hex <= 'F') return hex - 'A' + 10;
		return 0;
	}
	int hexPairToInt(const char *hexPair) {
		return hexToInt(hexPair[0]) << 4 | hexToInt(hexPair[1]);
	}
	void colorHexFull(bool withHashSymbol) {
		Color color;
		int startIndex = withHashSymbol ? 1 : 0;
		color.red = hexPairToInt(ts + startIndex) / 255.0f;
		color.green = hexPairToInt(ts + startIndex + 2) / 255.0f;
		color.blue = hexPairToInt(ts + startIndex + 4) / 255.0f;
		color.alpha = 1;
		addColor(color);
	}
	void colorHexShort(bool withHashSymbol) {
		Color color;
		int startIndex = withHashSymbol ? 1 : 0;
		color.red = hexToInt(ts[startIndex + 0]) / 15.0f;
		color.green = hexToInt(ts[startIndex + 1]) / 15.0f;
		color.blue = hexToInt(ts[startIndex + 2]) / 15.0f;
		color.alpha = 1;
		addColor(color);
	}
	void colorHexWithAlphaFull(bool withHashSymbol) {
		Color color;
		int startIndex = withHashSymbol ? 1 : 0;
		color.red = hexPairToInt(ts + startIndex) / 255.0f;
		color.green = hexPairToInt(ts + startIndex + 2) / 255.0f;
		color.blue = hexPairToInt(ts + startIndex + 4) / 255.0f;
		color.alpha = hexPairToInt(ts + startIndex + 6) / 255.0f;
		addColor(color);
	}
	void colorHexWithAlphaShort(bool withHashSymbol) {
		Color color;
		int startIndex = withHashSymbol ? 1 : 0;
		color.red = hexToInt(ts[startIndex + 0]) / 15.0f;
		color.green = hexToInt(ts[startIndex + 1]) / 15.0f;
		color.blue = hexToInt(ts[startIndex + 2]) / 15.0f;
		color.alpha = hexToInt(ts[startIndex + 3]) / 15.0f;
		addColor(color);
	}
	float getPercentage(size_t index, double unitlessMultiplier = 1.0, double percentageMultiplier = 1.0, double percentageOffset = 0) const {
		switch (numbersDouble[index].second) {
		case Unit::unitless:
			return static_cast<float>(numbersDouble[index].first * unitlessMultiplier);
		case Unit::percentage:
			return static_cast<float>(numbersDouble[index].first * (1 / 100.0) * percentageMultiplier + percentageOffset);
		case Unit::degree:
		case Unit::turn:
		case Unit::radian:
		case Unit::gradian:
			break;
		}
		return 0;
	}
	static double normalizeDegrees(double value) {
		double tmp;
		double result = std::modf(value, &tmp);
		if (result < 0)
			result += 1.0;
		return result;
	}
	float getDegrees(size_t index) const {
		switch (numbersDouble[index].second) {
		case Unit::unitless:
		case Unit::degree:
			return static_cast<float>(normalizeDegrees(numbersDouble[index].first * (1 / 360.0)));
		case Unit::turn:
			return static_cast<float>(normalizeDegrees(numbersDouble[index].first));
		case Unit::radian:
			return static_cast<float>(normalizeDegrees(numbersDouble[index].first * (1 / math::PI / 2)));
		case Unit::gradian:
			return static_cast<float>(normalizeDegrees(numbersDouble[index].first * (1 / 400.0)));
		case Unit::percentage:
			break;
		}
		return 0;
	}
	void colorRgb() {
		Color color;
		color.red = getPercentage(0, 1.0 / 255);
		color.green = getPercentage(1, 1.0 / 255);
		color.blue = getPercentage(2, 1.0 / 255);
		color.alpha = numbersDouble.size() == 4 ? getPercentage(3) : 1;
		numbersDouble.clear();
		addColor(color);
	}
	void colorHsl() {
		Color color;
		color.hsl.hue = getDegrees(0);
		color.hsl.saturation = getPercentage(1);
		color.hsl.lightness = getPercentage(2);
		color.alpha = numbersDouble.size() == 4 ? getPercentage(3) : 1;
		numbersDouble.clear();
		addColor(color.normalizeRgb().hslToRgb());
	}
	void colorOklch() {
		Color color;
		color.oklch.L = getPercentage(0);
		color.oklch.C = getPercentage(1, 1.0, 0.4);
		color.oklch.h = getDegrees(2) * 360.0f;
		color.alpha = numbersDouble.size() == 4 ? getPercentage(3) : 1;
		numbersDouble.clear();
		addColor(color.oklchToRgb().normalizeRgb());
	}
	void colorOklab() {
		Color color;
		color.oklab.L = getPercentage(0);
		color.oklab.a = getPercentage(1, 1.0, 0.8, -0.4);
		color.oklab.b = getPercentage(2, 1.0, 0.8, -0.4);
		color.alpha = numbersDouble.size() == 4 ? getPercentage(3) : 1;
		numbersDouble.clear();
		addColor(color.oklabToRgb().normalizeRgb());
	}
	void colorValues() {
		Color color;
		color.red = static_cast<float>(numbersDouble[0].first);
		color.green = static_cast<float>(numbersDouble[1].first);
		color.blue = static_cast<float>(numbersDouble[2].first);
		if (numbersDouble.size() > 3)
			color.alpha = static_cast<float>(numbersDouble[3].first);
		else
			color.alpha = 1;
		numbersDouble.clear();
		addColor(color);
	}
	void colorValueIntegers() {
		Color color;
		color.red = numbersI64[0].first / 255.0f;
		color.green = numbersI64[1].first / 255.0f;
		color.blue = numbersI64[2].first / 255.0f;
		if (numbersI64.size() > 3)
			color.alpha = numbersI64[3].first / 255.0f;
		else
			color.alpha = 1;
		numbersI64.clear();
		addColor(color);
	}
	double parseDouble(const char *start, const char *end) {
		std::string v(start, end);
		try {
			return std::stod(v.c_str());
		} catch(...) {
			return 0;
		}
	}
	void clearNumberStacks() {
		numbersI64.clear();
		numbersDouble.clear();
	}
};


#line 288 "source/parser/TextFileParser.rl"



#line 216 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
static const char _text_file_actions[] = {
	0, 1, 1, 1, 2, 1, 3, 1, 
	4, 1, 5, 1, 6, 1, 7, 1, 
	8, 1, 9, 1, 10, 1, 11, 1, 
	12, 1, 13, 1, 14, 1, 15, 1, 
	17, 1, 18, 1, 19, 1, 20, 1, 
	21, 1, 23, 1, 25, 1, 26, 1, 
	40, 1, 41, 1, 42, 1, 43, 1, 
	44, 1, 45, 1, 46, 1, 47, 1, 
	48, 1, 49, 1, 50, 1, 51, 1, 
	52, 1, 53, 1, 54, 1, 55, 1, 
	56, 1, 57, 1, 58, 1, 59, 1, 
	60, 1, 61, 1, 62, 1, 63, 1, 
	64, 1, 65, 1, 67, 1, 68, 1, 
	69, 1, 70, 1, 71, 1, 72, 1, 
	73, 1, 74, 1, 77, 1, 78, 1, 
	79, 1, 80, 1, 81, 2, 0, 1, 
	2, 1, 4, 2, 2, 4, 2, 2, 
	73, 2, 2, 74, 2, 4, 73, 2, 
	4, 74, 2, 4, 75, 2, 4, 76, 
	2, 6, 58, 2, 7, 58, 2, 8, 
	58, 2, 9, 58, 2, 10, 58, 2, 
	14, 48, 2, 14, 49, 2, 14, 50, 
	2, 14, 51, 2, 14, 52, 2, 14, 
	54, 2, 14, 56, 2, 14, 57, 2, 
	14, 59, 2, 14, 60, 2, 14, 61, 
	2, 15, 48, 2, 15, 49, 2, 15, 
	50, 2, 15, 51, 2, 15, 52, 2, 
	15, 54, 2, 15, 56, 2, 15, 57, 
	2, 15, 59, 2, 15, 60, 2, 15, 
	61, 2, 16, 22, 2, 16, 24, 2, 
	16, 66, 2, 19, 1, 2, 19, 27, 
	2, 19, 28, 2, 19, 29, 2, 19, 
	30, 2, 19, 31, 2, 19, 32, 2, 
	19, 33, 2, 19, 36, 2, 19, 37, 
	2, 19, 38, 2, 19, 39, 3, 0, 
	1, 3, 3, 2, 4, 73, 3, 2, 
	4, 74, 3, 19, 0, 1, 3, 19, 
	1, 4, 3, 19, 1, 34, 3, 19, 
	1, 35, 3, 19, 2, 36, 3, 19, 
	2, 37, 3, 19, 3, 36, 3, 19, 
	3, 37, 3, 19, 3, 39, 3, 19, 
	30, 1, 3, 19, 31, 1, 3, 19, 
	32, 1, 3, 19, 33, 1, 4, 19, 
	0, 1, 39, 5, 19, 0, 1, 3, 
	39, 5, 19, 0, 1, 34, 3, 5, 
	19, 0, 1, 35, 3
};

static const short _text_file_cond_offsets[] = {
	0, 0, 1, 9, 18, 20, 21, 26, 
	32, 34, 35, 41, 43, 44, 47, 48, 
	51, 52, 55, 56, 59, 65, 71, 73, 
	74, 82, 87, 93, 95, 96, 101, 107, 
	109, 110, 113, 114, 117, 118, 121, 122, 
	128, 137, 140, 141, 147, 156, 159, 162, 
	165, 168, 171, 174, 177, 180, 183, 186, 
	189, 192, 195, 198, 201, 204, 207, 210, 
	213, 216, 219, 222, 225, 228, 231, 234, 
	237, 240, 243, 246, 248, 249, 255, 257, 
	258, 265, 272, 274, 275, 281, 287, 289, 
	290, 298, 302, 308, 310, 311, 318, 321, 
	324, 327, 328, 333, 341, 345, 348, 349, 
	355, 364, 366, 369, 370, 374, 381, 387, 
	389, 390, 397, 400, 406, 408, 409, 416, 
	419, 422, 425, 426, 431, 439, 442, 445, 
	446, 451, 459, 462, 465, 466, 471, 479, 
	481, 483, 484, 490, 492, 493, 500, 507, 
	509, 510, 516, 522, 524, 525, 532, 535, 
	541, 543, 544, 551, 554, 557, 560, 561, 
	566, 574, 577, 580, 581, 586, 594, 596, 
	599, 600, 604, 611, 617, 619, 620, 627, 
	630, 636, 638, 639, 646, 649, 655, 657, 
	658, 665, 668, 671, 674, 675, 680, 688, 
	691, 694, 695, 700, 708, 711, 714, 715, 
	720, 728, 731, 734, 735, 740, 748, 750, 
	753, 755, 756, 762, 764, 765, 779, 786, 
	788, 789, 793, 795, 801, 803, 804, 808, 
	812, 818, 820, 821, 828, 831, 834, 837, 
	838, 843, 851, 854, 855, 857, 862, 865, 
	866, 868, 873, 879, 881, 882, 886, 889, 
	895, 897, 898, 902, 905, 908, 909, 911, 
	916, 919, 920, 922, 927, 929, 931, 934, 
	937, 938, 950, 952, 954, 956, 959, 961, 
	963, 966, 968, 970, 972, 975, 990, 992, 
	994, 995, 1001, 1003, 1004, 1018, 1025, 1027, 
	1028, 1032, 1034, 1040, 1042, 1043, 1047, 1050, 
	1056, 1058, 1059, 1066, 1069, 1072, 1075, 1076, 
	1081, 1089, 1092, 1093, 1095, 1100, 1103, 1104, 
	1106, 1111, 1117, 1119, 1120, 1124, 1127, 1133, 
	1135, 1136, 1140, 1143, 1149, 1151, 1152, 1159, 
	1162, 1165, 1168, 1169, 1174, 1182, 1185, 1186, 
	1188, 1193, 1196, 1197, 1199, 1204, 1206, 1208, 
	1211, 1214, 1215, 1227, 1229, 1231, 1233, 1236, 
	1238, 1240, 1243, 1245, 1247, 1249, 1252, 1267, 
	1269, 1272, 1274, 1276, 1278, 1279, 1285, 1287, 
	1288, 1294, 1300, 1302, 1303, 1309, 1315, 1317, 
	1318, 1332, 1335, 1337, 1339, 1342, 1345, 1346, 
	1358, 1360, 1362, 1364, 1367, 1369, 1371, 1374, 
	1376, 1378, 1380, 1383, 1398, 1400, 1403, 1404, 
	1408, 1415, 1417, 1420, 1421, 1425, 1432, 1434, 
	1436, 1438, 1439, 1445, 1447, 1448, 1454, 1460, 
	1462, 1463, 1469, 1475, 1477, 1478, 1492, 1495, 
	1501, 1503, 1504, 1511, 1514, 1517, 1520, 1521, 
	1526, 1534, 1536, 1538, 1541, 1544, 1545, 1557, 
	1559, 1561, 1563, 1566, 1568, 1570, 1573, 1575, 
	1577, 1579, 1582, 1597, 1599, 1602, 1603, 1607, 
	1614, 1616, 1619, 1620, 1624, 1631, 1633, 1635, 
	1637, 1638, 1644, 1646, 1647, 1653, 1659, 1661, 
	1662, 1668, 1674, 1676, 1677, 1692, 1696, 1698, 
	1700, 1704, 1707, 1708, 1721, 1723, 1725, 1727, 
	1731, 1733, 1735, 1739, 1741, 1743, 1745, 1749, 
	1765, 1767, 1770, 1771, 1775, 1782, 1784, 1787, 
	1788, 1792, 1799, 1801, 1803, 1805, 1806, 1812, 
	1814, 1815, 1821, 1827, 1829, 1830, 1836, 1842, 
	1844, 1845, 1852, 1855, 1858, 1861, 1862, 1867, 
	1875, 1877, 1880, 1881, 1885, 1892, 1894, 1897, 
	1898, 1902, 1909, 1911, 1915, 1917, 1921, 1923, 
	1927, 1929, 1931, 1933, 1934, 1940, 1942, 1943, 
	1949, 1955, 1957, 1958, 1964, 1970, 1972, 1973, 
	1980, 1983, 1989, 1991, 1992, 1999, 2002, 2005, 
	2008, 2009, 2014, 2022, 2025, 2028, 2029, 2034, 
	2042, 2044, 2047, 2048, 2052, 2059, 2061, 2064, 
	2065, 2069, 2076, 2078, 2082, 2084, 2088, 2090, 
	2094, 2096, 2098, 2100, 2101, 2107, 2109, 2110, 
	2116, 2122, 2124, 2125, 2131, 2137, 2139, 2140, 
	2148, 2152, 2156, 2159, 2160, 2166, 2175, 2177, 
	2180, 2181, 2185, 2192, 2194, 2197, 2198, 2202, 
	2209, 2211, 2215, 2217, 2221, 2223, 2227, 2233, 
	2236, 2242, 2247, 2250, 2255, 2258, 2261, 2264, 
	2267, 2273, 2281, 2289, 2297, 2305, 2313, 2321, 
	2329, 2337, 2345, 2353, 2361, 2369, 2377, 2385, 
	2393, 2401, 2409, 2417, 2425, 2433, 2441, 2449, 
	2457, 2465, 2473, 2481, 2489, 2497, 2505, 2513, 
	2526, 2539, 2552, 2565, 2578, 2591, 2596, 2601, 
	2609, 2614, 2622, 2627, 2635, 2640, 2648, 2653, 
	2661, 2674, 2687, 2700, 2713, 2718, 2723, 2731, 
	2736, 2744, 2749, 2757, 2770, 2783, 2796, 2809, 
	2814, 2819, 2827, 2832, 2840, 2845, 2853, 2866, 
	2879, 2884, 2889, 2897, 2910, 2923, 2928, 2933, 
	2941, 2954, 2967, 2972, 2977, 2985, 2998, 3011, 
	3016, 3021, 3029, 3042, 3047, 3060, 3065, 3078, 
	3083, 3096, 3101, 3114, 3119, 3132, 3137, 3150, 
	3155, 3168, 3173, 3182, 3188, 3197, 3202, 3208, 
	3213, 3219, 3225, 3231, 3237, 3246, 3259, 3272, 
	3285, 3298, 3311, 3324, 3337, 3350, 3363, 3376, 
	3389, 3402, 3415, 3428, 3441, 3454, 3467, 3480, 
	3493, 3506, 3519, 3532, 3545, 3558, 3571, 3584, 
	3597, 3610, 3623, 3636, 3639, 3642, 3645, 3648, 
	3651, 3654, 3657, 3660, 3663, 3666, 3669, 3672, 
	3675, 3678, 3681, 3684, 3687, 3690, 3693, 3696, 
	3699, 3702, 3705, 3708, 3711, 3714, 3717, 3720, 
	3723, 3726, 3729, 3732, 3735, 3738, 3741, 3744, 
	3747, 3750, 3753, 3756, 3759, 3762, 3765, 3768, 
	3771, 3785, 3785, 3787, 3792, 3795, 3796, 3800, 
	3803, 3809, 3817, 3820, 3821, 3825, 3831, 3840, 
	3841, 3844, 3847, 3850, 3853, 3856, 3859, 3862, 
	3865, 3868, 3871, 3874, 3877, 3880, 3883, 3886, 
	3889, 3892, 3895, 3898, 3901, 3904, 3907, 3910, 
	3913, 3916, 3919, 3921, 3923, 3925, 3927, 3929, 
	3931, 3933, 3935, 3937, 3939, 3941, 3943, 3945, 
	3947, 3949, 3951, 3953, 3955, 3957, 3959, 3961, 
	3967, 3973, 3974, 3977, 3978, 3986, 3994, 4002, 
	4010, 4018, 4026, 4034, 4042, 4050, 4058, 4066, 
	4074, 4082, 4090, 4098, 4106, 4114, 4122, 4130, 
	4138, 4146, 4154, 4162, 4170, 4178, 4186, 4195, 
	4208, 4211, 4217, 4230, 4243, 4256, 4261, 4269, 
	4282, 4295, 4308, 4313, 4321, 4334, 4347, 4352, 
	4360, 4373, 4386, 4391, 4399, 4412, 4425, 4438, 
	4443, 4451, 4464, 4477, 4482, 4490, 4503, 4516, 
	4521, 4529, 4542, 4555, 4560, 4568, 4581, 4594, 
	4599, 4607, 4620, 4633, 4638, 4646, 4659, 4672, 
	4677, 4685, 4694, 4703, 4707, 4713, 4717, 4730, 
	4743, 4756, 4769, 4782, 4795, 4808, 4821, 4834, 
	4847, 4860, 4873, 4886, 4899, 4912, 4925, 4938, 
	4951, 4964, 4977, 4990, 5003, 5016, 5029, 5042, 
	5055, 5056, 5057, 5059, 5062, 5065, 5068, 5071, 
	5074, 5077, 5080, 5083, 5086, 5089, 5092, 5095, 
	5098, 5101, 5104, 5107, 5110, 5113, 5116, 5119, 
	5122, 5125, 5128, 5131, 5134, 5137, 5137, 5137, 
	5137, 5137
};

static const char _text_file_cond_lengths[] = {
	0, 1, 8, 9, 2, 1, 5, 6, 
	2, 1, 6, 2, 1, 3, 1, 3, 
	1, 3, 1, 3, 6, 6, 2, 1, 
	8, 5, 6, 2, 1, 5, 6, 2, 
	1, 3, 1, 3, 1, 3, 1, 6, 
	9, 3, 1, 6, 9, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 2, 1, 6, 2, 1, 
	7, 7, 2, 1, 6, 6, 2, 1, 
	8, 4, 6, 2, 1, 7, 3, 3, 
	3, 1, 5, 8, 4, 3, 1, 6, 
	9, 2, 3, 1, 4, 7, 6, 2, 
	1, 7, 3, 6, 2, 1, 7, 3, 
	3, 3, 1, 5, 8, 3, 3, 1, 
	5, 8, 3, 3, 1, 5, 8, 2, 
	2, 1, 6, 2, 1, 7, 7, 2, 
	1, 6, 6, 2, 1, 7, 3, 6, 
	2, 1, 7, 3, 3, 3, 1, 5, 
	8, 3, 3, 1, 5, 8, 2, 3, 
	1, 4, 7, 6, 2, 1, 7, 3, 
	6, 2, 1, 7, 3, 6, 2, 1, 
	7, 3, 3, 3, 1, 5, 8, 3, 
	3, 1, 5, 8, 3, 3, 1, 5, 
	8, 3, 3, 1, 5, 8, 2, 3, 
	2, 1, 6, 2, 1, 14, 7, 2, 
	1, 4, 2, 6, 2, 1, 4, 4, 
	6, 2, 1, 7, 3, 3, 3, 1, 
	5, 8, 3, 1, 2, 5, 3, 1, 
	2, 5, 6, 2, 1, 4, 3, 6, 
	2, 1, 4, 3, 3, 1, 2, 5, 
	3, 1, 2, 5, 2, 2, 3, 3, 
	1, 12, 2, 2, 2, 3, 2, 2, 
	3, 2, 2, 2, 3, 15, 2, 2, 
	1, 6, 2, 1, 14, 7, 2, 1, 
	4, 2, 6, 2, 1, 4, 3, 6, 
	2, 1, 7, 3, 3, 3, 1, 5, 
	8, 3, 1, 2, 5, 3, 1, 2, 
	5, 6, 2, 1, 4, 3, 6, 2, 
	1, 4, 3, 6, 2, 1, 7, 3, 
	3, 3, 1, 5, 8, 3, 1, 2, 
	5, 3, 1, 2, 5, 2, 2, 3, 
	3, 1, 12, 2, 2, 2, 3, 2, 
	2, 3, 2, 2, 2, 3, 15, 2, 
	3, 2, 2, 2, 1, 6, 2, 1, 
	6, 6, 2, 1, 6, 6, 2, 1, 
	14, 3, 2, 2, 3, 3, 1, 12, 
	2, 2, 2, 3, 2, 2, 3, 2, 
	2, 2, 3, 15, 2, 3, 1, 4, 
	7, 2, 3, 1, 4, 7, 2, 2, 
	2, 1, 6, 2, 1, 6, 6, 2, 
	1, 6, 6, 2, 1, 14, 3, 6, 
	2, 1, 7, 3, 3, 3, 1, 5, 
	8, 2, 2, 3, 3, 1, 12, 2, 
	2, 2, 3, 2, 2, 3, 2, 2, 
	2, 3, 15, 2, 3, 1, 4, 7, 
	2, 3, 1, 4, 7, 2, 2, 2, 
	1, 6, 2, 1, 6, 6, 2, 1, 
	6, 6, 2, 1, 15, 4, 2, 2, 
	4, 3, 1, 13, 2, 2, 2, 4, 
	2, 2, 4, 2, 2, 2, 4, 16, 
	2, 3, 1, 4, 7, 2, 3, 1, 
	4, 7, 2, 2, 2, 1, 6, 2, 
	1, 6, 6, 2, 1, 6, 6, 2, 
	1, 7, 3, 3, 3, 1, 5, 8, 
	2, 3, 1, 4, 7, 2, 3, 1, 
	4, 7, 2, 4, 2, 4, 2, 4, 
	2, 2, 2, 1, 6, 2, 1, 6, 
	6, 2, 1, 6, 6, 2, 1, 7, 
	3, 6, 2, 1, 7, 3, 3, 3, 
	1, 5, 8, 3, 3, 1, 5, 8, 
	2, 3, 1, 4, 7, 2, 3, 1, 
	4, 7, 2, 4, 2, 4, 2, 4, 
	2, 2, 2, 1, 6, 2, 1, 6, 
	6, 2, 1, 6, 6, 2, 1, 8, 
	4, 4, 3, 1, 6, 9, 2, 3, 
	1, 4, 7, 2, 3, 1, 4, 7, 
	2, 4, 2, 4, 2, 4, 6, 3, 
	6, 5, 3, 5, 3, 3, 3, 3, 
	6, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 13, 
	13, 13, 13, 13, 13, 5, 5, 8, 
	5, 8, 5, 8, 5, 8, 5, 8, 
	13, 13, 13, 13, 5, 5, 8, 5, 
	8, 5, 8, 13, 13, 13, 13, 5, 
	5, 8, 5, 8, 5, 8, 13, 13, 
	5, 5, 8, 13, 13, 5, 5, 8, 
	13, 13, 5, 5, 8, 13, 13, 5, 
	5, 8, 13, 5, 13, 5, 13, 5, 
	13, 5, 13, 5, 13, 5, 13, 5, 
	13, 5, 9, 6, 9, 5, 6, 5, 
	6, 6, 6, 6, 9, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	14, 0, 2, 5, 3, 1, 4, 3, 
	6, 8, 3, 1, 4, 6, 9, 1, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 6, 
	6, 1, 3, 1, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 9, 13, 
	3, 6, 13, 13, 13, 5, 8, 13, 
	13, 13, 5, 8, 13, 13, 5, 8, 
	13, 13, 5, 8, 13, 13, 13, 5, 
	8, 13, 13, 5, 8, 13, 13, 5, 
	8, 13, 13, 5, 8, 13, 13, 5, 
	8, 13, 13, 5, 8, 13, 13, 5, 
	8, 9, 9, 4, 6, 4, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	1, 1, 2, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 0, 0, 0, 
	0, 0
};

static const int _text_file_cond_keys[] = {
	48, 57, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	44, 44, 45, 45, 46, 46, 48, 57, 
	58, 58, 59, 59, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 48, 57, 9, 9, 
	32, 32, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	44, 44, 58, 58, 59, 59, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 44, 44, 58, 58, 
	59, 59, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 43, 43, 45, 45, 
	48, 57, 48, 57, 43, 43, 45, 45, 
	48, 57, 48, 57, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 69, 69, 
	101, 101, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 69, 69, 101, 101, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 66, 66, 98, 98, 
	40, 40, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 44, 44, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	44, 44, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 47, 47, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	41, 41, 47, 47, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	41, 41, 9, 9, 32, 32, 41, 41, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 41, 41, 
	47, 47, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 47, 47, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 46, 46, 
	47, 47, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 9, 9, 32, 32, 
	37, 37, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 44, 44, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 44, 44, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 41, 41, 9, 9, 
	32, 32, 41, 41, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 44, 44, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 44, 44, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 44, 44, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 44, 44, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 46, 46, 
	48, 57, 69, 69, 101, 101, 66, 66, 
	98, 98, 65, 65, 97, 97, 40, 40, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	44, 44, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 44, 44, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	47, 47, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 47, 47, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 41, 41, 9, 9, 32, 32, 
	41, 41, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 46, 46, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	47, 47, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	47, 47, 48, 57, 9, 9, 32, 32, 
	37, 37, 46, 46, 47, 47, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	9, 9, 32, 32, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 44, 44, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 44, 44, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	44, 44, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 41, 41, 
	9, 9, 32, 32, 41, 41, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 44, 44, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 48, 57, 
	9, 9, 32, 32, 37, 37, 44, 44, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 44, 44, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 48, 57, 
	9, 9, 32, 32, 37, 37, 44, 44, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 44, 44, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 44, 44, 48, 57, 
	9, 9, 32, 32, 37, 37, 44, 44, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	66, 66, 98, 98, 40, 40, 65, 65, 
	97, 97, 76, 76, 108, 108, 40, 40, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 44, 44, 
	48, 57, 68, 68, 69, 69, 71, 71, 
	82, 82, 84, 84, 100, 100, 101, 101, 
	103, 103, 114, 114, 116, 116, 9, 9, 
	32, 32, 43, 43, 44, 44, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 41, 41, 9, 9, 
	32, 32, 41, 41, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 46, 46, 
	48, 57, 69, 69, 101, 101, 43, 43, 
	45, 45, 48, 57, 48, 57, 37, 37, 
	48, 57, 37, 37, 46, 46, 48, 57, 
	69, 69, 101, 101, 43, 43, 45, 45, 
	48, 57, 48, 57, 37, 37, 48, 57, 
	37, 37, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 37, 37, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	44, 44, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 37, 37, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	41, 41, 43, 43, 45, 45, 48, 57, 
	48, 57, 37, 37, 48, 57, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	37, 37, 48, 57, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 69, 69, 
	101, 101, 71, 71, 103, 103, 9, 9, 
	32, 32, 44, 44, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	44, 44, 48, 57, 68, 68, 71, 71, 
	82, 82, 84, 84, 100, 100, 103, 103, 
	114, 114, 116, 116, 82, 82, 114, 114, 
	65, 65, 97, 97, 68, 68, 100, 100, 
	9, 9, 32, 32, 44, 44, 65, 65, 
	97, 97, 68, 68, 100, 100, 9, 9, 
	32, 32, 44, 44, 85, 85, 117, 117, 
	82, 82, 114, 114, 78, 78, 110, 110, 
	9, 9, 32, 32, 44, 44, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	68, 68, 69, 69, 71, 71, 82, 82, 
	84, 84, 100, 100, 101, 101, 103, 103, 
	114, 114, 116, 116, 76, 76, 108, 108, 
	65, 65, 97, 97, 40, 40, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	68, 68, 69, 69, 71, 71, 82, 82, 
	84, 84, 100, 100, 101, 101, 103, 103, 
	114, 114, 116, 116, 9, 9, 32, 32, 
	43, 43, 44, 44, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 37, 37, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 47, 47, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	41, 41, 9, 9, 32, 32, 41, 41, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 46, 46, 48, 57, 69, 69, 
	101, 101, 43, 43, 45, 45, 48, 57, 
	48, 57, 37, 37, 48, 57, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	37, 37, 48, 57, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 44, 44, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 44, 44, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 41, 41, 9, 9, 32, 32, 
	41, 41, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 46, 46, 48, 57, 
	69, 69, 101, 101, 43, 43, 45, 45, 
	48, 57, 48, 57, 37, 37, 48, 57, 
	37, 37, 46, 46, 48, 57, 69, 69, 
	101, 101, 43, 43, 45, 45, 48, 57, 
	48, 57, 37, 37, 48, 57, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	69, 69, 101, 101, 71, 71, 103, 103, 
	9, 9, 32, 32, 44, 44, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 44, 44, 48, 57, 68, 68, 
	71, 71, 82, 82, 84, 84, 100, 100, 
	103, 103, 114, 114, 116, 116, 82, 82, 
	114, 114, 65, 65, 97, 97, 68, 68, 
	100, 100, 9, 9, 32, 32, 44, 44, 
	65, 65, 97, 97, 68, 68, 100, 100, 
	9, 9, 32, 32, 44, 44, 85, 85, 
	117, 117, 82, 82, 114, 114, 78, 78, 
	110, 110, 9, 9, 32, 32, 44, 44, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 68, 68, 69, 69, 71, 71, 
	82, 82, 84, 84, 100, 100, 101, 101, 
	103, 103, 114, 114, 116, 116, 76, 76, 
	108, 108, 40, 40, 65, 65, 97, 97, 
	76, 76, 108, 108, 67, 67, 99, 99, 
	72, 72, 104, 104, 40, 40, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	41, 41, 48, 57, 68, 68, 69, 69, 
	71, 71, 82, 82, 84, 84, 100, 100, 
	101, 101, 103, 103, 114, 114, 116, 116, 
	9, 9, 32, 32, 41, 41, 69, 69, 
	101, 101, 71, 71, 103, 103, 9, 9, 
	32, 32, 41, 41, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	41, 41, 48, 57, 68, 68, 71, 71, 
	82, 82, 84, 84, 100, 100, 103, 103, 
	114, 114, 116, 116, 82, 82, 114, 114, 
	65, 65, 97, 97, 68, 68, 100, 100, 
	9, 9, 32, 32, 41, 41, 65, 65, 
	97, 97, 68, 68, 100, 100, 9, 9, 
	32, 32, 41, 41, 85, 85, 117, 117, 
	82, 82, 114, 114, 78, 78, 110, 110, 
	9, 9, 32, 32, 41, 41, 9, 9, 
	32, 32, 41, 41, 46, 46, 48, 57, 
	68, 68, 69, 69, 71, 71, 82, 82, 
	84, 84, 100, 100, 101, 101, 103, 103, 
	114, 114, 116, 116, 9, 9, 32, 32, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	9, 9, 32, 32, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	76, 76, 108, 108, 67, 67, 99, 99, 
	72, 72, 104, 104, 40, 40, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	47, 47, 48, 57, 68, 68, 69, 69, 
	71, 71, 82, 82, 84, 84, 100, 100, 
	101, 101, 103, 103, 114, 114, 116, 116, 
	9, 9, 32, 32, 47, 47, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 41, 41, 9, 9, 32, 32, 
	41, 41, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 46, 46, 48, 57, 
	69, 69, 101, 101, 69, 69, 101, 101, 
	71, 71, 103, 103, 9, 9, 32, 32, 
	47, 47, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 47, 47, 
	48, 57, 68, 68, 71, 71, 82, 82, 
	84, 84, 100, 100, 103, 103, 114, 114, 
	116, 116, 82, 82, 114, 114, 65, 65, 
	97, 97, 68, 68, 100, 100, 9, 9, 
	32, 32, 47, 47, 65, 65, 97, 97, 
	68, 68, 100, 100, 9, 9, 32, 32, 
	47, 47, 85, 85, 117, 117, 82, 82, 
	114, 114, 78, 78, 110, 110, 9, 9, 
	32, 32, 47, 47, 9, 9, 32, 32, 
	46, 46, 47, 47, 48, 57, 68, 68, 
	69, 69, 71, 71, 82, 82, 84, 84, 
	100, 100, 101, 101, 103, 103, 114, 114, 
	116, 116, 9, 9, 32, 32, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 9, 9, 
	32, 32, 37, 37, 46, 46, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	9, 9, 32, 32, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 76, 76, 
	108, 108, 67, 67, 99, 99, 72, 72, 
	104, 104, 40, 40, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 41, 41, 
	47, 47, 48, 57, 68, 68, 69, 69, 
	71, 71, 82, 82, 84, 84, 100, 100, 
	101, 101, 103, 103, 114, 114, 116, 116, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	69, 69, 101, 101, 71, 71, 103, 103, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	48, 57, 68, 68, 71, 71, 82, 82, 
	84, 84, 100, 100, 103, 103, 114, 114, 
	116, 116, 82, 82, 114, 114, 65, 65, 
	97, 97, 68, 68, 100, 100, 9, 9, 
	32, 32, 41, 41, 47, 47, 65, 65, 
	97, 97, 68, 68, 100, 100, 9, 9, 
	32, 32, 41, 41, 47, 47, 85, 85, 
	117, 117, 82, 82, 114, 114, 78, 78, 
	110, 110, 9, 9, 32, 32, 41, 41, 
	47, 47, 9, 9, 32, 32, 41, 41, 
	46, 46, 47, 47, 48, 57, 68, 68, 
	69, 69, 71, 71, 82, 82, 84, 84, 
	100, 100, 101, 101, 103, 103, 114, 114, 
	116, 116, 9, 9, 32, 32, 43, 43, 
	45, 45, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 9, 9, 
	32, 32, 37, 37, 46, 46, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	9, 9, 32, 32, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 76, 76, 
	108, 108, 65, 65, 97, 97, 66, 66, 
	98, 98, 40, 40, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 41, 41, 9, 9, 
	32, 32, 41, 41, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 48, 57, 9, 9, 
	32, 32, 37, 37, 41, 41, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 9, 9, 32, 32, 
	37, 37, 46, 46, 48, 57, 69, 69, 
	101, 101, 76, 76, 108, 108, 65, 65, 
	67, 67, 97, 97, 99, 99, 76, 76, 
	108, 108, 65, 65, 67, 67, 97, 97, 
	99, 99, 76, 76, 108, 108, 65, 65, 
	67, 67, 97, 97, 99, 99, 76, 76, 
	108, 108, 65, 65, 97, 97, 66, 66, 
	98, 98, 40, 40, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	46, 46, 48, 57, 48, 57, 9, 9, 
	32, 32, 37, 37, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	47, 47, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 47, 47, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 41, 41, 9, 9, 32, 32, 
	41, 41, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	41, 41, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 46, 46, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	47, 47, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	47, 47, 48, 57, 9, 9, 32, 32, 
	37, 37, 46, 46, 47, 47, 48, 57, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 48, 57, 
	9, 9, 32, 32, 37, 37, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	76, 76, 108, 108, 65, 65, 67, 67, 
	97, 97, 99, 99, 76, 76, 108, 108, 
	65, 65, 67, 67, 97, 97, 99, 99, 
	76, 76, 108, 108, 65, 65, 67, 67, 
	97, 97, 99, 99, 76, 76, 108, 108, 
	65, 65, 97, 97, 66, 66, 98, 98, 
	40, 40, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 46, 46, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	46, 46, 48, 57, 46, 46, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 46, 46, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	47, 47, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	9, 9, 32, 32, 41, 41, 47, 47, 
	43, 43, 45, 45, 48, 57, 48, 57, 
	9, 9, 32, 32, 37, 37, 41, 41, 
	47, 47, 48, 57, 9, 9, 32, 32, 
	37, 37, 41, 41, 46, 46, 47, 47, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 43, 43, 45, 45, 48, 57, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	48, 57, 9, 9, 32, 32, 37, 37, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 43, 43, 45, 45, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	37, 37, 48, 57, 9, 9, 32, 32, 
	37, 37, 46, 46, 48, 57, 69, 69, 
	101, 101, 76, 76, 108, 108, 65, 65, 
	67, 67, 97, 97, 99, 99, 76, 76, 
	108, 108, 65, 65, 67, 67, 97, 97, 
	99, 99, 76, 76, 108, 108, 65, 65, 
	67, 67, 97, 97, 99, 99, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 9, 9, 32, 32, 48, 57, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 9, 9, 32, 32, 
	44, 44, 58, 58, 59, 59, 9, 9, 
	32, 32, 48, 57, 9, 9, 32, 32, 
	44, 44, 58, 58, 59, 59, 9, 9, 
	32, 32, 48, 57, 9, 9, 32, 32, 
	48, 57, 9, 9, 32, 32, 48, 57, 
	9, 9, 32, 32, 48, 57, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 43, 43, 45, 45, 48, 57, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 43, 43, 45, 45, 48, 57, 
	65, 70, 97, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 43, 43, 45, 45, 48, 57, 
	65, 70, 97, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 43, 43, 
	44, 44, 45, 45, 46, 46, 48, 57, 
	58, 58, 59, 59, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 69, 69, 
	101, 101, 9, 9, 32, 32, 44, 44, 
	58, 58, 59, 59, 9, 9, 32, 32, 
	43, 43, 45, 45, 46, 46, 48, 57, 
	9, 9, 32, 32, 44, 44, 58, 58, 
	59, 59, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 9, 9, 
	32, 32, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 43, 43, 
	45, 45, 46, 46, 48, 57, 9, 9, 
	32, 32, 43, 43, 45, 45, 46, 46, 
	48, 57, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 35, 35, 
	43, 43, 45, 45, 46, 46, 47, 47, 
	48, 57, 65, 70, 72, 72, 79, 79, 
	82, 82, 97, 102, 104, 104, 111, 111, 
	114, 114, 46, 46, 48, 57, 9, 9, 
	32, 32, 48, 57, 69, 69, 101, 101, 
	48, 57, 69, 69, 101, 101, 48, 57, 
	46, 46, 48, 57, 69, 69, 101, 101, 
	9, 9, 32, 32, 48, 57, 9, 9, 
	32, 32, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 69, 69, 
	101, 101, 48, 57, 69, 69, 101, 101, 
	48, 57, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 69, 69, 101, 101, 
	48, 57, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 71, 71, 
	103, 103, 71, 71, 103, 103, 71, 71, 
	103, 103, 83, 83, 115, 115, 83, 83, 
	115, 115, 83, 83, 115, 115, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 75, 75, 107, 107, 75, 75, 
	107, 107, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 48, 57, 9, 9, 32, 32, 
	48, 57, 48, 57, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 69, 69, 101, 101, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 43, 43, 45, 45, 48, 57, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 43, 43, 
	45, 45, 48, 57, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 48, 57, 
	58, 58, 59, 59, 65, 70, 97, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 43, 43, 45, 45, 
	48, 57, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 48, 57, 58, 58, 
	59, 59, 65, 70, 97, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 43, 43, 45, 45, 48, 57, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 48, 57, 58, 58, 59, 59, 
	65, 70, 97, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	43, 43, 45, 45, 48, 57, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	48, 57, 58, 58, 59, 59, 65, 70, 
	97, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	69, 69, 101, 101, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 69, 69, 101, 101, 46, 46, 
	48, 57, 69, 69, 101, 101, 9, 9, 
	32, 32, 46, 46, 48, 57, 69, 69, 
	101, 101, 46, 46, 48, 57, 69, 69, 
	101, 101, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 9, 9, 
	32, 32, 44, 44, 46, 46, 48, 57, 
	58, 58, 59, 59, 65, 68, 69, 69, 
	70, 70, 97, 100, 101, 101, 102, 102, 
	9, 9, 32, 32, 44, 44, 46, 46, 
	48, 57, 58, 58, 59, 59, 65, 68, 
	69, 69, 70, 70, 97, 100, 101, 101, 
	102, 102, 9, 9, 32, 32, 44, 44, 
	46, 46, 48, 57, 58, 58, 59, 59, 
	65, 68, 69, 69, 70, 70, 97, 100, 
	101, 101, 102, 102, 9, 9, 32, 32, 
	44, 44, 46, 46, 48, 57, 58, 58, 
	59, 59, 65, 68, 69, 69, 70, 70, 
	97, 100, 101, 101, 102, 102, 47, 47, 
	42, 42, 42, 42, 47, 47, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 0
};

static const char _text_file_cond_spaces[] = {
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 17, 17, 17, 17, 17, 17, 
	17, 17, 17, 17, 17, 17, 2, 2, 
	2, 2, 2, 2, 18, 18, 18, 18, 
	18, 18, 22, 22, 22, 22, 22, 22, 
	41, 41, 41, 41, 41, 41, 3, 3, 
	3, 19, 19, 19, 23, 23, 23, 42, 
	42, 42, 26, 26, 26, 45, 45, 45, 
	51, 51, 51, 65, 65, 65, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	4, 4, 4, 4, 4, 4, 4, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 31, 31, 4, 5, 
	5, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 7, 7, 7, 7, 7, 
	7, 7, 7, 32, 32, 6, 7, 7, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	8, 8, 8, 8, 8, 8, 8, 8, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 9, 
	9, 9, 9, 9, 9, 9, 9, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 8, 
	9, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 8, 9, 
	33, 33, 33, 33, 33, 33, 8, 9, 
	33, 33, 33, 33, 33, 33, 8, 9, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 8, 9, 33, 33, 33, 33, 33, 
	33, 8, 9, 33, 33, 33, 33, 33, 
	33, 33, 33, 8, 9, 33, 33, 8, 
	33, 9, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 33, 
	33, 33, 33, 33, 33, 33, 33, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 34, 34, 10, 
	8, 10, 8, 36, 36, 10, 9, 10, 
	9, 61, 61, 10, 33, 10, 33, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 11, 11, 11, 11, 
	11, 11, 11, 11, 35, 35, 11, 8, 
	11, 8, 37, 37, 11, 9, 11, 9, 
	62, 62, 11, 33, 11, 33, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 10, 
	11, 38, 38, 38, 38, 38, 10, 11, 
	38, 38, 10, 11, 38, 38, 38, 38, 
	38, 38, 38, 10, 11, 38, 38, 38, 
	38, 10, 38, 11, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 63, 63, 38, 8, 38, 8, 64, 
	64, 38, 9, 38, 9, 80, 80, 38, 
	33, 38, 33, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 20, 12, 12, 0, 
	0, 12, 12, 12, 24, 12, 12, 1, 
	1, 12, 12, 12, 24, 12, 12, 1, 
	1, 12, 12, 12, 24, 12, 12, 1, 
	1, 12, 12, 12, 24, 12, 12, 1, 
	1, 12, 12, 12, 43, 12, 12, 17, 
	17, 12, 12, 12, 43, 12, 12, 17, 
	17, 12, 12, 12, 43, 12, 12, 17, 
	17, 12, 12, 12, 43, 12, 12, 17, 
	17, 12, 12, 12, 27, 12, 12, 2, 
	2, 12, 12, 12, 27, 12, 12, 2, 
	2, 12, 12, 12, 46, 12, 12, 18, 
	18, 12, 12, 12, 46, 12, 12, 18, 
	18, 12, 12, 12, 52, 12, 12, 22, 
	22, 12, 12, 12, 52, 12, 12, 22, 
	22, 12, 12, 12, 66, 12, 12, 41, 
	41, 12, 12, 12, 66, 12, 12, 41, 
	41, 12, 12, 12, 29, 12, 12, 3, 
	3, 12, 12, 12, 48, 12, 12, 19, 
	19, 12, 12, 12, 54, 12, 12, 23, 
	23, 12, 12, 12, 68, 12, 12, 42, 
	42, 12, 12, 12, 57, 12, 12, 26, 
	26, 12, 12, 12, 71, 12, 12, 45, 
	45, 12, 12, 12, 75, 12, 12, 51, 
	51, 12, 12, 12, 81, 12, 12, 65, 
	65, 13, 13, 13, 13, 21, 13, 13, 
	0, 21, 0, 0, 21, 0, 13, 13, 
	13, 13, 21, 13, 13, 0, 21, 0, 
	0, 21, 0, 13, 13, 13, 13, 21, 
	13, 13, 0, 21, 0, 0, 21, 0, 
	13, 13, 13, 13, 21, 13, 13, 0, 
	21, 0, 0, 21, 0, 13, 13, 13, 
	13, 21, 13, 13, 0, 21, 0, 0, 
	21, 0, 13, 13, 13, 13, 21, 13, 
	13, 0, 21, 0, 0, 21, 0, 13, 
	13, 21, 0, 0, 13, 13, 21, 0, 
	0, 13, 13, 13, 21, 13, 13, 0, 
	0, 13, 13, 21, 0, 0, 13, 13, 
	13, 21, 13, 13, 0, 0, 13, 13, 
	21, 0, 0, 13, 13, 13, 21, 13, 
	13, 0, 0, 13, 13, 21, 0, 0, 
	13, 13, 13, 21, 13, 13, 0, 0, 
	13, 13, 21, 0, 0, 13, 13, 13, 
	21, 13, 13, 0, 0, 13, 13, 13, 
	13, 25, 13, 13, 1, 25, 1, 1, 
	25, 1, 13, 13, 13, 13, 25, 13, 
	13, 1, 25, 1, 1, 25, 1, 13, 
	13, 13, 13, 25, 13, 13, 1, 25, 
	1, 1, 25, 1, 13, 13, 13, 13, 
	25, 13, 13, 1, 25, 1, 1, 25, 
	1, 13, 13, 25, 1, 1, 13, 13, 
	25, 1, 1, 13, 13, 13, 25, 13, 
	13, 1, 1, 13, 13, 25, 1, 1, 
	13, 13, 13, 25, 13, 13, 1, 1, 
	13, 13, 25, 1, 1, 13, 13, 13, 
	25, 13, 13, 1, 1, 13, 13, 13, 
	13, 44, 13, 13, 17, 44, 17, 17, 
	44, 17, 13, 13, 13, 13, 44, 13, 
	13, 17, 44, 17, 17, 44, 17, 13, 
	13, 13, 13, 44, 13, 13, 17, 44, 
	17, 17, 44, 17, 13, 13, 13, 13, 
	44, 13, 13, 17, 44, 17, 17, 44, 
	17, 13, 13, 44, 17, 17, 13, 13, 
	44, 17, 17, 13, 13, 13, 44, 13, 
	13, 17, 17, 13, 13, 44, 17, 17, 
	13, 13, 13, 44, 13, 13, 17, 17, 
	13, 13, 44, 17, 17, 13, 13, 13, 
	44, 13, 13, 17, 17, 13, 13, 13, 
	13, 28, 13, 13, 2, 28, 2, 2, 
	28, 2, 13, 13, 13, 13, 28, 13, 
	13, 2, 28, 2, 2, 28, 2, 13, 
	13, 28, 2, 2, 13, 13, 28, 2, 
	2, 13, 13, 13, 28, 13, 13, 2, 
	2, 13, 13, 13, 13, 47, 13, 13, 
	18, 47, 18, 18, 47, 18, 13, 13, 
	13, 13, 47, 13, 13, 18, 47, 18, 
	18, 47, 18, 13, 13, 47, 18, 18, 
	13, 13, 47, 18, 18, 13, 13, 13, 
	47, 13, 13, 18, 18, 13, 13, 13, 
	13, 53, 13, 13, 22, 53, 22, 22, 
	53, 22, 13, 13, 13, 13, 53, 13, 
	13, 22, 53, 22, 22, 53, 22, 13, 
	13, 53, 22, 22, 13, 13, 53, 22, 
	22, 13, 13, 13, 53, 13, 13, 22, 
	22, 13, 13, 13, 13, 67, 13, 13, 
	41, 67, 41, 41, 67, 41, 13, 13, 
	13, 13, 67, 13, 13, 41, 67, 41, 
	41, 67, 41, 13, 13, 67, 41, 41, 
	13, 13, 67, 41, 41, 13, 13, 13, 
	67, 13, 13, 41, 41, 13, 13, 13, 
	13, 30, 13, 13, 3, 30, 3, 3, 
	30, 3, 13, 13, 30, 3, 3, 13, 
	13, 13, 13, 49, 13, 13, 19, 49, 
	19, 19, 49, 19, 13, 13, 49, 19, 
	19, 13, 13, 13, 13, 55, 13, 13, 
	23, 55, 23, 23, 55, 23, 13, 13, 
	55, 23, 23, 13, 13, 13, 13, 69, 
	13, 13, 42, 69, 42, 42, 69, 42, 
	13, 13, 69, 42, 42, 13, 13, 13, 
	13, 58, 13, 13, 26, 58, 26, 26, 
	58, 26, 13, 13, 58, 26, 26, 13, 
	13, 13, 13, 72, 13, 13, 45, 72, 
	45, 45, 72, 45, 13, 13, 72, 45, 
	45, 13, 13, 13, 13, 76, 13, 13, 
	51, 76, 51, 51, 76, 51, 13, 13, 
	76, 51, 51, 13, 13, 13, 13, 82, 
	13, 13, 65, 82, 65, 65, 82, 65, 
	13, 13, 82, 65, 65, 39, 39, 13, 
	39, 13, 13, 39, 39, 39, 39, 39, 
	13, 13, 13, 39, 39, 39, 39, 13, 
	39, 39, 39, 13, 13, 39, 39, 39, 
	39, 39, 39, 39, 13, 13, 13, 39, 
	39, 39, 39, 39, 39, 39, 39, 13, 
	13, 13, 39, 39, 39, 13, 39, 13, 
	13, 39, 39, 13, 13, 13, 39, 39, 
	39, 13, 13, 13, 39, 39, 39, 39, 
	13, 39, 39, 39, 13, 13, 39, 39, 
	39, 13, 50, 39, 39, 0, 21, 0, 
	0, 21, 0, 39, 39, 39, 13, 50, 
	39, 39, 0, 21, 0, 0, 21, 0, 
	39, 39, 39, 13, 50, 39, 39, 0, 
	21, 0, 0, 21, 0, 39, 39, 39, 
	13, 50, 39, 39, 0, 21, 0, 0, 
	21, 0, 39, 39, 39, 13, 50, 39, 
	39, 0, 21, 0, 0, 21, 0, 39, 
	39, 39, 13, 50, 39, 39, 0, 21, 
	0, 0, 21, 0, 39, 39, 39, 13, 
	56, 39, 39, 1, 25, 1, 1, 25, 
	1, 39, 39, 39, 13, 56, 39, 39, 
	1, 25, 1, 1, 25, 1, 39, 39, 
	39, 13, 56, 39, 39, 1, 25, 1, 
	1, 25, 1, 39, 39, 39, 13, 56, 
	39, 39, 1, 25, 1, 1, 25, 1, 
	39, 39, 39, 13, 70, 39, 39, 17, 
	44, 17, 17, 44, 17, 39, 39, 39, 
	13, 70, 39, 39, 17, 44, 17, 17, 
	44, 17, 39, 39, 39, 13, 70, 39, 
	39, 17, 44, 17, 17, 44, 17, 39, 
	39, 39, 13, 70, 39, 39, 17, 44, 
	17, 17, 44, 17, 39, 39, 39, 13, 
	59, 39, 39, 2, 28, 2, 2, 28, 
	2, 39, 39, 39, 13, 59, 39, 39, 
	2, 28, 2, 2, 28, 2, 39, 39, 
	39, 13, 73, 39, 39, 18, 47, 18, 
	18, 47, 18, 39, 39, 39, 13, 73, 
	39, 39, 18, 47, 18, 18, 47, 18, 
	39, 39, 39, 13, 77, 39, 39, 22, 
	53, 22, 22, 53, 22, 39, 39, 39, 
	13, 77, 39, 39, 22, 53, 22, 22, 
	53, 22, 39, 39, 39, 13, 84, 39, 
	39, 41, 67, 41, 41, 67, 41, 39, 
	39, 39, 13, 84, 39, 39, 41, 67, 
	41, 41, 67, 41, 39, 39, 39, 13, 
	60, 39, 39, 3, 30, 3, 3, 30, 
	3, 39, 39, 39, 13, 74, 39, 39, 
	19, 49, 19, 19, 49, 19, 39, 39, 
	39, 13, 78, 39, 39, 23, 55, 23, 
	23, 55, 23, 39, 39, 39, 13, 85, 
	39, 39, 42, 69, 42, 42, 69, 42, 
	39, 39, 39, 13, 79, 39, 39, 26, 
	58, 26, 26, 58, 26, 39, 39, 39, 
	13, 86, 39, 39, 45, 72, 45, 45, 
	72, 45, 39, 39, 39, 13, 87, 39, 
	39, 51, 76, 51, 51, 76, 51, 39, 
	39, 39, 13, 88, 39, 39, 65, 82, 
	65, 65, 82, 65, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	17, 17, 17, 17, 17, 17, 17, 17, 
	17, 17, 17, 17, 17, 17, 17, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	18, 18, 18, 18, 18, 18, 18, 18, 
	18, 22, 22, 22, 22, 22, 22, 22, 
	22, 22, 41, 41, 41, 41, 41, 41, 
	41, 41, 41, 3, 3, 3, 3, 3, 
	3, 19, 19, 19, 19, 19, 19, 23, 
	23, 23, 23, 23, 23, 42, 42, 42, 
	42, 42, 42, 26, 26, 26, 26, 26, 
	26, 45, 45, 45, 45, 45, 45, 51, 
	51, 51, 51, 51, 51, 65, 65, 65, 
	65, 65, 65, 83, 13, 13, 13, 40, 
	88, 65, 32, 80, 31, 65, 32, 80, 
	31, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 0, 0, 0, 1, 1, 1, 17, 
	17, 17, 0, 0, 0, 2, 2, 2, 
	18, 18, 18, 0, 0, 0, 22, 22, 
	22, 1, 1, 1, 41, 41, 41, 17, 
	17, 17, 3, 3, 3, 19, 19, 19, 
	0, 0, 0, 23, 23, 23, 1, 1, 
	1, 42, 42, 42, 17, 17, 17, 26, 
	26, 26, 2, 2, 2, 45, 45, 45, 
	18, 18, 18, 51, 51, 51, 22, 22, 
	22, 65, 65, 65, 41, 41, 41, 4, 
	4, 5, 5, 31, 31, 6, 6, 7, 
	7, 32, 32, 8, 8, 9, 9, 33, 
	33, 10, 10, 34, 34, 36, 36, 61, 
	61, 11, 11, 35, 35, 37, 37, 62, 
	62, 38, 38, 63, 63, 64, 64, 80, 
	80, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 12, 
	12, 12, 12, 12, 12, 20, 12, 12, 
	0, 0, 12, 12, 12, 24, 12, 12, 
	1, 1, 12, 12, 12, 43, 12, 12, 
	17, 17, 12, 12, 12, 20, 12, 12, 
	0, 0, 12, 12, 12, 27, 12, 12, 
	2, 2, 12, 12, 12, 46, 12, 12, 
	18, 18, 12, 12, 12, 20, 12, 12, 
	0, 0, 12, 12, 12, 52, 12, 12, 
	22, 22, 12, 12, 12, 24, 12, 12, 
	1, 1, 12, 12, 12, 66, 12, 12, 
	41, 41, 12, 12, 12, 43, 12, 12, 
	17, 17, 12, 12, 12, 29, 12, 12, 
	3, 3, 12, 12, 12, 48, 12, 12, 
	19, 19, 12, 12, 12, 20, 12, 12, 
	0, 0, 12, 12, 12, 54, 12, 12, 
	23, 23, 12, 12, 12, 24, 12, 12, 
	1, 1, 12, 12, 12, 68, 12, 12, 
	42, 42, 12, 12, 12, 43, 12, 12, 
	17, 17, 12, 12, 12, 57, 12, 12, 
	26, 26, 12, 12, 12, 27, 12, 12, 
	2, 2, 12, 12, 12, 71, 12, 12, 
	45, 45, 12, 12, 12, 46, 12, 12, 
	18, 18, 12, 12, 12, 75, 12, 12, 
	51, 51, 12, 12, 12, 52, 12, 12, 
	22, 22, 12, 12, 12, 81, 12, 12, 
	65, 65, 12, 12, 12, 66, 12, 12, 
	41, 41, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 21, 
	13, 13, 0, 21, 0, 0, 21, 0, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 25, 13, 13, 
	1, 25, 1, 1, 25, 1, 13, 13, 
	13, 13, 44, 13, 13, 17, 44, 17, 
	17, 44, 17, 13, 13, 13, 13, 21, 
	13, 13, 0, 21, 0, 0, 21, 0, 
	13, 13, 21, 0, 0, 13, 13, 13, 
	21, 13, 13, 0, 0, 13, 13, 13, 
	13, 28, 13, 13, 2, 28, 2, 2, 
	28, 2, 13, 13, 13, 13, 47, 13, 
	13, 18, 47, 18, 18, 47, 18, 13, 
	13, 13, 13, 21, 13, 13, 0, 21, 
	0, 0, 21, 0, 13, 13, 21, 0, 
	0, 13, 13, 13, 21, 13, 13, 0, 
	0, 13, 13, 13, 13, 53, 13, 13, 
	22, 53, 22, 22, 53, 22, 13, 13, 
	13, 13, 25, 13, 13, 1, 25, 1, 
	1, 25, 1, 13, 13, 25, 1, 1, 
	13, 13, 13, 25, 13, 13, 1, 1, 
	13, 13, 13, 13, 67, 13, 13, 41, 
	67, 41, 41, 67, 41, 13, 13, 13, 
	13, 44, 13, 13, 17, 44, 17, 17, 
	44, 17, 13, 13, 44, 17, 17, 13, 
	13, 13, 44, 13, 13, 17, 17, 13, 
	13, 13, 13, 30, 13, 13, 3, 30, 
	3, 3, 30, 3, 13, 13, 13, 13, 
	49, 13, 13, 19, 49, 19, 19, 49, 
	19, 13, 13, 13, 13, 21, 13, 13, 
	0, 21, 0, 0, 21, 0, 13, 13, 
	21, 0, 0, 13, 13, 13, 21, 13, 
	13, 0, 0, 13, 13, 13, 13, 55, 
	13, 13, 23, 55, 23, 23, 55, 23, 
	13, 13, 13, 13, 25, 13, 13, 1, 
	25, 1, 1, 25, 1, 13, 13, 25, 
	1, 1, 13, 13, 13, 25, 13, 13, 
	1, 1, 13, 13, 13, 13, 69, 13, 
	13, 42, 69, 42, 42, 69, 42, 13, 
	13, 13, 13, 44, 13, 13, 17, 44, 
	17, 17, 44, 17, 13, 13, 44, 17, 
	17, 13, 13, 13, 44, 13, 13, 17, 
	17, 13, 13, 13, 13, 58, 13, 13, 
	26, 58, 26, 26, 58, 26, 13, 13, 
	13, 13, 28, 13, 13, 2, 28, 2, 
	2, 28, 2, 13, 13, 28, 2, 2, 
	13, 13, 13, 28, 13, 13, 2, 2, 
	13, 13, 13, 13, 72, 13, 13, 45, 
	72, 45, 45, 72, 45, 13, 13, 13, 
	13, 47, 13, 13, 18, 47, 18, 18, 
	47, 18, 13, 13, 47, 18, 18, 13, 
	13, 13, 47, 13, 13, 18, 18, 13, 
	13, 13, 13, 76, 13, 13, 51, 76, 
	51, 51, 76, 51, 13, 13, 13, 13, 
	53, 13, 13, 22, 53, 22, 22, 53, 
	22, 13, 13, 53, 22, 22, 13, 13, 
	13, 53, 13, 13, 22, 22, 13, 13, 
	13, 13, 82, 13, 13, 65, 82, 65, 
	65, 82, 65, 13, 13, 13, 13, 67, 
	13, 13, 41, 67, 41, 41, 67, 41, 
	13, 13, 67, 41, 41, 13, 13, 13, 
	67, 13, 13, 41, 41, 39, 39, 39, 
	13, 39, 39, 39, 13, 13, 39, 39, 
	39, 13, 39, 39, 39, 13, 13, 13, 
	39, 13, 13, 39, 39, 13, 39, 13, 
	13, 13, 39, 13, 13, 39, 39, 39, 
	13, 50, 39, 39, 0, 21, 0, 0, 
	21, 0, 39, 39, 39, 13, 56, 39, 
	39, 1, 25, 1, 1, 25, 1, 39, 
	39, 39, 13, 70, 39, 39, 17, 44, 
	17, 17, 44, 17, 39, 39, 39, 13, 
	50, 39, 39, 0, 21, 0, 0, 21, 
	0, 39, 39, 39, 13, 59, 39, 39, 
	2, 28, 2, 2, 28, 2, 39, 39, 
	39, 13, 73, 39, 39, 18, 47, 18, 
	18, 47, 18, 39, 39, 39, 13, 50, 
	39, 39, 0, 21, 0, 0, 21, 0, 
	39, 39, 39, 13, 77, 39, 39, 22, 
	53, 22, 22, 53, 22, 39, 39, 39, 
	13, 56, 39, 39, 1, 25, 1, 1, 
	25, 1, 39, 39, 39, 13, 84, 39, 
	39, 41, 67, 41, 41, 67, 41, 39, 
	39, 39, 13, 70, 39, 39, 17, 44, 
	17, 17, 44, 17, 39, 39, 39, 13, 
	60, 39, 39, 3, 30, 3, 3, 30, 
	3, 39, 39, 39, 13, 74, 39, 39, 
	19, 49, 19, 19, 49, 19, 39, 39, 
	39, 13, 50, 39, 39, 0, 21, 0, 
	0, 21, 0, 39, 39, 39, 13, 78, 
	39, 39, 23, 55, 23, 23, 55, 23, 
	39, 39, 39, 13, 56, 39, 39, 1, 
	25, 1, 1, 25, 1, 39, 39, 39, 
	13, 85, 39, 39, 42, 69, 42, 42, 
	69, 42, 39, 39, 39, 13, 70, 39, 
	39, 17, 44, 17, 17, 44, 17, 39, 
	39, 39, 13, 79, 39, 39, 26, 58, 
	26, 26, 58, 26, 39, 39, 39, 13, 
	59, 39, 39, 2, 28, 2, 2, 28, 
	2, 39, 39, 39, 13, 86, 39, 39, 
	45, 72, 45, 45, 72, 45, 39, 39, 
	39, 13, 73, 39, 39, 18, 47, 18, 
	18, 47, 18, 39, 39, 39, 13, 87, 
	39, 39, 51, 76, 51, 51, 76, 51, 
	39, 39, 39, 13, 77, 39, 39, 22, 
	53, 22, 22, 53, 22, 39, 39, 39, 
	13, 88, 39, 39, 65, 82, 65, 65, 
	82, 65, 39, 39, 39, 13, 84, 39, 
	39, 41, 67, 41, 41, 67, 41, 14, 
	15, 15, 14, 0, 0, 0, 1, 1, 
	1, 17, 17, 17, 0, 0, 0, 2, 
	2, 2, 18, 18, 18, 0, 0, 0, 
	22, 22, 22, 1, 1, 1, 41, 41, 
	41, 17, 17, 17, 3, 3, 3, 19, 
	19, 19, 0, 0, 0, 23, 23, 23, 
	1, 1, 1, 42, 42, 42, 17, 17, 
	17, 26, 26, 26, 2, 2, 2, 45, 
	45, 45, 18, 18, 18, 51, 51, 51, 
	22, 22, 22, 65, 65, 65, 41, 41, 
	41, 0
};

static const short _text_file_key_offsets[] = {
	0, 0, 2, 11, 21, 24, 26, 32, 
	39, 42, 44, 51, 54, 56, 60, 62, 
	66, 68, 72, 74, 78, 85, 92, 95, 
	97, 106, 111, 118, 121, 123, 128, 135, 
	138, 140, 144, 146, 150, 152, 156, 158, 
	165, 175, 179, 181, 188, 198, 204, 210, 
	216, 222, 228, 234, 240, 246, 252, 258, 
	276, 294, 312, 330, 336, 342, 360, 378, 
	396, 414, 456, 498, 504, 522, 540, 582, 
	600, 642, 684, 774, 776, 777, 784, 787, 
	789, 797, 805, 808, 810, 817, 824, 827, 
	829, 838, 842, 849, 852, 854, 862, 865, 
	868, 872, 874, 880, 889, 893, 897, 899, 
	906, 916, 918, 922, 924, 929, 937, 944, 
	947, 949, 957, 960, 967, 970, 972, 980, 
	983, 986, 990, 992, 998, 1007, 1010, 1014, 
	1016, 1022, 1031, 1034, 1038, 1040, 1046, 1055, 
	1057, 1059, 1060, 1067, 1070, 1072, 1080, 1088, 
	1091, 1093, 1100, 1107, 1110, 1112, 1120, 1123, 
	1130, 1133, 1135, 1143, 1146, 1149, 1153, 1155, 
	1161, 1170, 1173, 1177, 1179, 1185, 1194, 1196, 
	1200, 1202, 1207, 1215, 1222, 1225, 1227, 1235, 
	1238, 1245, 1248, 1250, 1258, 1261, 1268, 1271, 
	1273, 1281, 1284, 1287, 1291, 1293, 1299, 1308, 
	1311, 1315, 1317, 1323, 1332, 1335, 1339, 1341, 
	1347, 1356, 1359, 1363, 1365, 1371, 1380, 1386, 
	1389, 1391, 1392, 1399, 1402, 1404, 1419, 1427, 
	1430, 1432, 1437, 1439, 1446, 1449, 1451, 1456, 
	1460, 1467, 1470, 1472, 1480, 1483, 1486, 1490, 
	1492, 1498, 1507, 1511, 1513, 1516, 1522, 1526, 
	1528, 1531, 1537, 1544, 1547, 1549, 1554, 1557, 
	1564, 1567, 1569, 1574, 1577, 1581, 1583, 1586, 
	1592, 1596, 1598, 1601, 1607, 1609, 1611, 1614, 
	1618, 1620, 1633, 1635, 1637, 1639, 1642, 1644, 
	1646, 1649, 1651, 1653, 1655, 1658, 1674, 1676, 
	1678, 1679, 1686, 1689, 1691, 1706, 1714, 1717, 
	1719, 1724, 1726, 1733, 1736, 1738, 1743, 1746, 
	1753, 1756, 1758, 1766, 1769, 1772, 1776, 1778, 
	1784, 1793, 1797, 1799, 1802, 1808, 1812, 1814, 
	1817, 1823, 1830, 1833, 1835, 1840, 1843, 1850, 
	1853, 1855, 1860, 1863, 1870, 1873, 1875, 1883, 
	1886, 1889, 1893, 1895, 1901, 1910, 1914, 1916, 
	1919, 1925, 1929, 1931, 1934, 1940, 1942, 1944, 
	1947, 1951, 1953, 1966, 1968, 1970, 1972, 1975, 
	1977, 1979, 1982, 1984, 1986, 1988, 1991, 2007, 
	2013, 2016, 2018, 2020, 2022, 2023, 2030, 2033, 
	2035, 2042, 2049, 2052, 2054, 2061, 2068, 2071, 
	2073, 2088, 2091, 2093, 2095, 2098, 2102, 2104, 
	2117, 2119, 2121, 2123, 2126, 2128, 2130, 2133, 
	2135, 2137, 2139, 2142, 2158, 2160, 2164, 2166, 
	2171, 2179, 2181, 2185, 2187, 2192, 2200, 2202, 
	2204, 2206, 2207, 2214, 2217, 2219, 2226, 2233, 
	2236, 2238, 2245, 2252, 2255, 2257, 2272, 2275, 
	2282, 2285, 2287, 2295, 2298, 2301, 2305, 2307, 
	2313, 2322, 2324, 2326, 2329, 2333, 2335, 2348, 
	2350, 2352, 2354, 2357, 2359, 2361, 2364, 2366, 
	2368, 2370, 2373, 2389, 2391, 2395, 2397, 2402, 
	2410, 2412, 2416, 2418, 2423, 2431, 2437, 2443, 
	2449, 2452, 2473, 2482, 2488, 2509, 2530, 2539, 
	2545, 2566, 2587, 2596, 2602, 2646, 2654, 2660, 
	2666, 2674, 2686, 2692, 2730, 2736, 2742, 2748, 
	2756, 2762, 2768, 2776, 2782, 2788, 2794, 2802, 
	2849, 2855, 2867, 2873, 2888, 2912, 2918, 2930, 
	2936, 2951, 2975, 2977, 2979, 2981, 2982, 2989, 
	2992, 2994, 3001, 3008, 3011, 3013, 3020, 3027, 
	3030, 3032, 3040, 3043, 3046, 3050, 3052, 3058, 
	3067, 3069, 3073, 3075, 3080, 3088, 3090, 3094, 
	3096, 3101, 3109, 3115, 3119, 3125, 3129, 3143, 
	3151, 3153, 3155, 3157, 3158, 3165, 3168, 3170, 
	3177, 3184, 3187, 3189, 3196, 3203, 3206, 3208, 
	3216, 3219, 3226, 3229, 3231, 3239, 3242, 3245, 
	3249, 3251, 3257, 3266, 3269, 3273, 3275, 3281, 
	3290, 3292, 3296, 3298, 3303, 3311, 3313, 3317, 
	3319, 3324, 3332, 3338, 3342, 3348, 3352, 3366, 
	3374, 3380, 3386, 3392, 3395, 3416, 3425, 3431, 
	3452, 3473, 3482, 3488, 3509, 3530, 3539, 3545, 
	3568, 3576, 3584, 3596, 3602, 3619, 3645, 3651, 
	3663, 3669, 3684, 3708, 3714, 3726, 3732, 3747, 
	3771, 3785, 3793, 3807, 3815, 3845, 3857, 3864, 
	3868, 3875, 3880, 3884, 3889, 3893, 3897, 3901, 
	3905, 3912, 3927, 3942, 3957, 3972, 3987, 4002, 
	4017, 4032, 4047, 4062, 4093, 4124, 4155, 4186, 
	4201, 4216, 4247, 4278, 4309, 4340, 4403, 4466, 
	4481, 4512, 4543, 4606, 4637, 4700, 4763, 4890, 
	4914, 4938, 4962, 4986, 5010, 5034, 5046, 5058, 
	5073, 5085, 5100, 5112, 5127, 5139, 5154, 5166, 
	5181, 5205, 5229, 5253, 5277, 5289, 5301, 5316, 
	5328, 5343, 5355, 5370, 5422, 5474, 5526, 5578, 
	5606, 5634, 5665, 5693, 5724, 5752, 5783, 5807, 
	5831, 5843, 5855, 5870, 5922, 5974, 6002, 6030, 
	6061, 6113, 6165, 6193, 6221, 6252, 6360, 6468, 
	6528, 6588, 6651, 6675, 6687, 6739, 6767, 6819, 
	6847, 6955, 7015, 7067, 7095, 7203, 7263, 7371, 
	7431, 7651, 7775, 7799, 7814, 7838, 7853, 7868, 
	7883, 7898, 7913, 7928, 7943, 7967, 8009, 8051, 
	8093, 8135, 8177, 8219, 8261, 8303, 8345, 8387, 
	8465, 8543, 8621, 8699, 8741, 8783, 8861, 8939, 
	9017, 9095, 9245, 9395, 9437, 9515, 9593, 9743, 
	9821, 9971, 10121, 10415, 10421, 10427, 10433, 10439, 
	10445, 10451, 10457, 10463, 10469, 10475, 10481, 10487, 
	10505, 10523, 10541, 10559, 10577, 10583, 10589, 10595, 
	10613, 10631, 10649, 10667, 10685, 10703, 10745, 10787, 
	10829, 10835, 10841, 10859, 10877, 10895, 10913, 10955, 
	10997, 11015, 11033, 11075, 11117, 11159, 11201, 11291, 
	11381, 11686, 11687, 11690, 11696, 11700, 11702, 11707, 
	11711, 11718, 11727, 11731, 11733, 11738, 11745, 11755, 
	11757, 11763, 11769, 11787, 11793, 11799, 11817, 11823, 
	11841, 11847, 11889, 11907, 11913, 11931, 11937, 11955, 
	11961, 12003, 12021, 12039, 12045, 12087, 12105, 12147, 
	12165, 12255, 12297, 12299, 12301, 12307, 12309, 12311, 
	12317, 12319, 12321, 12327, 12329, 12335, 12341, 12355, 
	12357, 12363, 12369, 12383, 12389, 12403, 12417, 12447, 
	12454, 12461, 12463, 12467, 12469, 12484, 12499, 12530, 
	12545, 12560, 12591, 12606, 12637, 12652, 12715, 12746, 
	12761, 12792, 12807, 12838, 12853, 12916, 12947, 12978, 
	12993, 13056, 13087, 13150, 13181, 13308, 13371, 13381, 
	13405, 13409, 13416, 13440, 13492, 13516, 13528, 13543, 
	13567, 13619, 13643, 13655, 13670, 13722, 13746, 13758, 
	13773, 13881, 13933, 13961, 13992, 14016, 14068, 14092, 
	14104, 14119, 14171, 14195, 14207, 14222, 14330, 14382, 
	14410, 14441, 14493, 14517, 14529, 14544, 14652, 14704, 
	14732, 14763, 14871, 14923, 14951, 14982, 15202, 15310, 
	15370, 15433, 15457, 15535, 15566, 15625, 15656, 15698, 
	15740, 15818, 15860, 15902, 15980, 16022, 16100, 16142, 
	16292, 16370, 16412, 16490, 16532, 16610, 16652, 16802, 
	16880, 16958, 17000, 17150, 17228, 17378, 17456, 17750, 
	17900, 17901, 17902, 17904, 17910, 17916, 17934, 17940, 
	17946, 17964, 17970, 17988, 17994, 18036, 18054, 18060, 
	18078, 18084, 18102, 18108, 18150, 18168, 18186, 18192, 
	18234, 18252, 18294, 18312, 18402, 18444, 18447, 18448, 
	18449, 18451
};

static const int _text_file_trans_keys[] = {
	7216, 7225, 7177, 7200, 7212, 7237, 7269, 7216, 
	7225, 7226, 7227, 7177, 7200, 7212, 7214, 7211, 
	7213, 7216, 7225, 7226, 7227, 7214, 7216, 7225, 
	7216, 7225, 7177, 7200, 7237, 7269, 7216, 7225, 
	7177, 7200, 7211, 7213, 7214, 7216, 7225, 7214, 
	7216, 7225, 7216, 7225, 7177, 7200, 7211, 7213, 
	7214, 7216, 7225, 7214, 7216, 7225, 7216, 7225, 
	7211, 7213, 7216, 7225, 7216, 7225, 7211, 7213, 
	7216, 7225, 7216, 7225, 7211, 7213, 7216, 7225, 
	7216, 7225, 7177, 7200, 7216, 7225, 7177, 7200, 
	7214, 7237, 7269, 7216, 7225, 7177, 7200, 7211, 
	7213, 7214, 7216, 7225, 7214, 7216, 7225, 7216, 
	7225, 7177, 7200, 7212, 7237, 7269, 7216, 7225, 
	7226, 7227, 7177, 7200, 7212, 7226, 7227, 7177, 
	7200, 7211, 7213, 7214, 7216, 7225, 7214, 7216, 
	7225, 7216, 7225, 7177, 7200, 7212, 7226, 7227, 
	7177, 7200, 7211, 7213, 7214, 7216, 7225, 7214, 
	7216, 7225, 7216, 7225, 7211, 7213, 7216, 7225, 
	7216, 7225, 7211, 7213, 7216, 7225, 7216, 7225, 
	7211, 7213, 7216, 7225, 7216, 7225, 7177, 7200, 
	7212, 7216, 7225, 7226, 7227, 7177, 7200, 7212, 
	7214, 7237, 7269, 7216, 7225, 7226, 7227, 7211, 
	7213, 7216, 7225, 7216, 7225, 7177, 7200, 7212, 
	7216, 7225, 7226, 7227, 7177, 7200, 7212, 7214, 
	7237, 7269, 7216, 7225, 7226, 7227, 560, 569, 
	577, 582, 609, 614, 560, 569, 577, 582, 
	609, 614, 560, 569, 577, 582, 609, 614, 
	560, 569, 577, 582, 609, 614, 560, 569, 
	577, 582, 609, 614, 560, 569, 577, 582, 
	609, 614, 1072, 1081, 1089, 1094, 1121, 1126, 
	1072, 1081, 1089, 1094, 1121, 1126, 1072, 1081, 
	1089, 1094, 1121, 1126, 1072, 1081, 1089, 1094, 
	1121, 1126, 9264, 9273, 9281, 9286, 9313, 9318, 
	9520, 9529, 9537, 9542, 9569, 9574, 9776, 9785, 
	9793, 9798, 9825, 9830, 9264, 9273, 9281, 9286, 
	9313, 9318, 9520, 9529, 9537, 9542, 9569, 9574, 
	9776, 9785, 9793, 9798, 9825, 9830, 9264, 9273, 
	9281, 9286, 9313, 9318, 9520, 9529, 9537, 9542, 
	9569, 9574, 9776, 9785, 9793, 9798, 9825, 9830, 
	9264, 9273, 9281, 9286, 9313, 9318, 9520, 9529, 
	9537, 9542, 9569, 9574, 9776, 9785, 9793, 9798, 
	9825, 9830, 1584, 1593, 1601, 1606, 1633, 1638, 
	1584, 1593, 1601, 1606, 1633, 1638, 12336, 12345, 
	12353, 12358, 12385, 12390, 12592, 12601, 12609, 12614, 
	12641, 12646, 12848, 12857, 12865, 12870, 12897, 12902, 
	12336, 12345, 12353, 12358, 12385, 12390, 12592, 12601, 
	12609, 12614, 12641, 12646, 12848, 12857, 12865, 12870, 
	12897, 12902, 13360, 13369, 13377, 13382, 13409, 13414, 
	13616, 13625, 13633, 13638, 13665, 13670, 13872, 13881, 
	13889, 13894, 13921, 13926, 13360, 13369, 13377, 13382, 
	13409, 13414, 13616, 13625, 13633, 13638, 13665, 13670, 
	13872, 13881, 13889, 13894, 13921, 13926, 10288, 10297, 
	10305, 10310, 10337, 10342, 10544, 10553, 10561, 10566, 
	10593, 10598, 10800, 10809, 10817, 10822, 10849, 10854, 
	11056, 11065, 11073, 11078, 11105, 11110, 11312, 11321, 
	11329, 11334, 11361, 11366, 11568, 11577, 11585, 11590, 
	11617, 11622, 11824, 11833, 11841, 11846, 11873, 11878, 
	10288, 10297, 10305, 10310, 10337, 10342, 10544, 10553, 
	10561, 10566, 10593, 10598, 10800, 10809, 10817, 10822, 
	10849, 10854, 11056, 11065, 11073, 11078, 11105, 11110, 
	11312, 11321, 11329, 11334, 11361, 11366, 11568, 11577, 
	11585, 11590, 11617, 11622, 11824, 11833, 11841, 11846, 
	11873, 11878, 2096, 2105, 2113, 2118, 2145, 2150, 
	18480, 18489, 18497, 18502, 18529, 18534, 18736, 18745, 
	18753, 18758, 18785, 18790, 18992, 19001, 19009, 19014, 
	19041, 19046, 19504, 19513, 19521, 19526, 19553, 19558, 
	19760, 19769, 19777, 19782, 19809, 19814, 20016, 20025, 
	20033, 20038, 20065, 20070, 20528, 20537, 20545, 20550, 
	20577, 20582, 20784, 20793, 20801, 20806, 20833, 20838, 
	21040, 21049, 21057, 21062, 21089, 21094, 21296, 21305, 
	21313, 21318, 21345, 21350, 21552, 21561, 21569, 21574, 
	21601, 21606, 21808, 21817, 21825, 21830, 21857, 21862, 
	22064, 22073, 22081, 22086, 22113, 22118, 22576, 22585, 
	22593, 22598, 22625, 22630, 22832, 22841, 22849, 22854, 
	22881, 22886, 23088, 23097, 23105, 23110, 23137, 23142, 
	23600, 23609, 23617, 23622, 23649, 23654, 23856, 23865, 
	23873, 23878, 23905, 23910, 24112, 24121, 24129, 24134, 
	24161, 24166, 24368, 24377, 24385, 24390, 24417, 24422, 
	24624, 24633, 24641, 24646, 24673, 24678, 24880, 24889, 
	24897, 24902, 24929, 24934, 25136, 25145, 25153, 25158, 
	25185, 25190, 25648, 25657, 25665, 25670, 25697, 25702, 
	25904, 25913, 25921, 25926, 25953, 25958, 26160, 26169, 
	26177, 26182, 26209, 26214, 26416, 26425, 26433, 26438, 
	26465, 26470, 26672, 26681, 26689, 26694, 26721, 26726, 
	26928, 26937, 26945, 26950, 26977, 26982, 27184, 27193, 
	27201, 27206, 27233, 27238, 14384, 14393, 14401, 14406, 
	14433, 14438, 14640, 14649, 14657, 14662, 14689, 14694, 
	14896, 14905, 14913, 14918, 14945, 14950, 15152, 15161, 
	15169, 15174, 15201, 15206, 15408, 15417, 15425, 15430, 
	15457, 15462, 15664, 15673, 15681, 15686, 15713, 15718, 
	15920, 15929, 15937, 15942, 15969, 15974, 16176, 16185, 
	16193, 16198, 16225, 16230, 16432, 16441, 16449, 16454, 
	16481, 16486, 16688, 16697, 16705, 16710, 16737, 16742, 
	16944, 16953, 16961, 16966, 16993, 16998, 17200, 17209, 
	17217, 17222, 17249, 17254, 17456, 17465, 17473, 17478, 
	17505, 17510, 17712, 17721, 17729, 17734, 17761, 17766, 
	17968, 17977, 17985, 17990, 18017, 18022, 2626, 2658, 
	2600, 2569, 2592, 2603, 2605, 2606, 2608, 2617, 
	2606, 2608, 2617, 2608, 2617, 2569, 2592, 2597, 
	2604, 2629, 2661, 2608, 2617, 2569, 2592, 2604, 
	2606, 2603, 2605, 2608, 2617, 2606, 2608, 2617, 
	2608, 2617, 2569, 2592, 2597, 2629, 2661, 2608, 
	2617, 2569, 2592, 2603, 2605, 2606, 2608, 2617, 
	2606, 2608, 2617, 2608, 2617, 2569, 2592, 2597, 
	2601, 2607, 2629, 2661, 2608, 2617, 2569, 2592, 
	2601, 2607, 2569, 2592, 2603, 2605, 2606, 2608, 
	2617, 2606, 2608, 2617, 2608, 2617, 2569, 2592, 
	2597, 2601, 2629, 2661, 2608, 2617, 2569, 2592, 
	2601, 2569, 2592, 2601, 2603, 2605, 2608, 2617, 
	2608, 2617, 2569, 2592, 2597, 2601, 2608, 2617, 
	2569, 2592, 2597, 2601, 2606, 2629, 2661, 2608, 
	2617, 2569, 2592, 2601, 2607, 2603, 2605, 2608, 
	2617, 2608, 2617, 2569, 2592, 2597, 2601, 2607, 
	2608, 2617, 2569, 2592, 2597, 2601, 2606, 2607, 
	2629, 2661, 2608, 2617, 2569, 2592, 2603, 2605, 
	2608, 2617, 2608, 2617, 2569, 2592, 2597, 2608, 
	2617, 2569, 2592, 2597, 2606, 2629, 2661, 2608, 
	2617, 2569, 2592, 2603, 2605, 2606, 2608, 2617, 
	2606, 2608, 2617, 2608, 2617, 2569, 2592, 2597, 
	2604, 2629, 2661, 2608, 2617, 2569, 2592, 2604, 
	2569, 2592, 2603, 2605, 2606, 2608, 2617, 2606, 
	2608, 2617, 2608, 2617, 2569, 2592, 2597, 2601, 
	2629, 2661, 2608, 2617, 2569, 2592, 2601, 2569, 
	2592, 2601, 2603, 2605, 2608, 2617, 2608, 2617, 
	2569, 2592, 2597, 2601, 2608, 2617, 2569, 2592, 
	2597, 2601, 2606, 2629, 2661, 2608, 2617, 2569, 
	2592, 2604, 2603, 2605, 2608, 2617, 2608, 2617, 
	2569, 2592, 2597, 2604, 2608, 2617, 2569, 2592, 
	2597, 2604, 2606, 2629, 2661, 2608, 2617, 2569, 
	2592, 2604, 2603, 2605, 2608, 2617, 2608, 2617, 
	2569, 2592, 2597, 2604, 2608, 2617, 2569, 2592, 
	2597, 2604, 2606, 2629, 2661, 2608, 2617, 3138, 
	3170, 3137, 3169, 3112, 3081, 3104, 3115, 3117, 
	3118, 3120, 3129, 3118, 3120, 3129, 3120, 3129, 
	3081, 3104, 3109, 3116, 3141, 3173, 3120, 3129, 
	3081, 3104, 3116, 3118, 3115, 3117, 3120, 3129, 
	3118, 3120, 3129, 3120, 3129, 3081, 3104, 3109, 
	3141, 3173, 3120, 3129, 3081, 3104, 3115, 3117, 
	3118, 3120, 3129, 3118, 3120, 3129, 3120, 3129, 
	3081, 3104, 3109, 3119, 3141, 3173, 3120, 3129, 
	3081, 3104, 3119, 3081, 3104, 3115, 3117, 3118, 
	3120, 3129, 3118, 3120, 3129, 3120, 3129, 3081, 
	3104, 3109, 3113, 3141, 3173, 3120, 3129, 3081, 
	3104, 3113, 3081, 3104, 3113, 3115, 3117, 3120, 
	3129, 3120, 3129, 3081, 3104, 3109, 3113, 3120, 
	3129, 3081, 3104, 3109, 3113, 3118, 3141, 3173, 
	3120, 3129, 3081, 3104, 3119, 3115, 3117, 3120, 
	3129, 3120, 3129, 3081, 3104, 3109, 3119, 3120, 
	3129, 3081, 3104, 3109, 3118, 3119, 3141, 3173, 
	3120, 3129, 3081, 3104, 3115, 3117, 3120, 3129, 
	3120, 3129, 3081, 3104, 3109, 3120, 3129, 3081, 
	3104, 3109, 3118, 3141, 3173, 3120, 3129, 3081, 
	3104, 3115, 3117, 3118, 3120, 3129, 3118, 3120, 
	3129, 3120, 3129, 3081, 3104, 3109, 3116, 3141, 
	3173, 3120, 3129, 3081, 3104, 3116, 3081, 3104, 
	3115, 3117, 3118, 3120, 3129, 3118, 3120, 3129, 
	3120, 3129, 3081, 3104, 3109, 3116, 3141, 3173, 
	3120, 3129, 3081, 3104, 3116, 3081, 3104, 3115, 
	3117, 3118, 3120, 3129, 3118, 3120, 3129, 3120, 
	3129, 3081, 3104, 3109, 3113, 3141, 3173, 3120, 
	3129, 3081, 3104, 3113, 3081, 3104, 3113, 3115, 
	3117, 3120, 3129, 3120, 3129, 3081, 3104, 3109, 
	3113, 3120, 3129, 3081, 3104, 3109, 3113, 3118, 
	3141, 3173, 3120, 3129, 3081, 3104, 3116, 3115, 
	3117, 3120, 3129, 3120, 3129, 3081, 3104, 3109, 
	3116, 3120, 3129, 3081, 3104, 3109, 3116, 3118, 
	3141, 3173, 3120, 3129, 3081, 3104, 3116, 3115, 
	3117, 3120, 3129, 3120, 3129, 3081, 3104, 3109, 
	3116, 3120, 3129, 3081, 3104, 3109, 3116, 3118, 
	3141, 3173, 3120, 3129, 3081, 3104, 3116, 3115, 
	3117, 3120, 3129, 3120, 3129, 3081, 3104, 3109, 
	3116, 3120, 3129, 3081, 3104, 3109, 3116, 3118, 
	3141, 3173, 3120, 3129, 27714, 27746, 27970, 28002, 
	28226, 28258, 2600, 3137, 3169, 3660, 3692, 3624, 
	3593, 3616, 3627, 3629, 3630, 3632, 3641, 3630, 
	3632, 3641, 3632, 3641, 3593, 3616, 3628, 3652, 
	3653, 3655, 3666, 3668, 3684, 3685, 3687, 3698, 
	3700, 3632, 3641, 3593, 3616, 3628, 3630, 3627, 
	3629, 3632, 3641, 3630, 3632, 3641, 3632, 3641, 
	3621, 3653, 3685, 3632, 3641, 3593, 3616, 3593, 
	3616, 3627, 3629, 3630, 3632, 3641, 3630, 3632, 
	3641, 3632, 3641, 3621, 3653, 3685, 3632, 3641, 
	3593, 3616, 3625, 3631, 3593, 3616, 3627, 3629, 
	3630, 3632, 3641, 3630, 3632, 3641, 3632, 3641, 
	3593, 3616, 3621, 3625, 3653, 3685, 3632, 3641, 
	3593, 3616, 3625, 3593, 3616, 3625, 3627, 3629, 
	3632, 3641, 3632, 3641, 3593, 3616, 3621, 3625, 
	3632, 3641, 3593, 3616, 3621, 3625, 3630, 3653, 
	3685, 3632, 3641, 3627, 3629, 3632, 3641, 3632, 
	3641, 3621, 3632, 3641, 3621, 3630, 3653, 3685, 
	3632, 3641, 3627, 3629, 3632, 3641, 3632, 3641, 
	3621, 3632, 3641, 3621, 3630, 3653, 3685, 3632, 
	3641, 3593, 3616, 3627, 3629, 3630, 3632, 3641, 
	3630, 3632, 3641, 3632, 3641, 3621, 3653, 3685, 
	3632, 3641, 3593, 3616, 3628, 3593, 3616, 3627, 
	3629, 3630, 3632, 3641, 3630, 3632, 3641, 3632, 
	3641, 3621, 3653, 3685, 3632, 3641, 3593, 3616, 
	3625, 3627, 3629, 3632, 3641, 3632, 3641, 3621, 
	3632, 3641, 3621, 3630, 3653, 3685, 3632, 3641, 
	3627, 3629, 3632, 3641, 3632, 3641, 3621, 3632, 
	3641, 3621, 3630, 3653, 3685, 3632, 3641, 3653, 
	3685, 3655, 3687, 3593, 3616, 3628, 3627, 3629, 
	3632, 3641, 3632, 3641, 3593, 3616, 3628, 3652, 
	3655, 3666, 3668, 3684, 3687, 3698, 3700, 3632, 
	3641, 3666, 3698, 3649, 3681, 3652, 3684, 3593, 
	3616, 3628, 3649, 3681, 3652, 3684, 3593, 3616, 
	3628, 3669, 3701, 3666, 3698, 3662, 3694, 3593, 
	3616, 3628, 3593, 3616, 3628, 3630, 3652, 3653, 
	3655, 3666, 3668, 3684, 3685, 3687, 3698, 3700, 
	3632, 3641, 4172, 4204, 4161, 4193, 4136, 4105, 
	4128, 4139, 4141, 4142, 4144, 4153, 4142, 4144, 
	4153, 4144, 4153, 4105, 4128, 4140, 4164, 4165, 
	4167, 4178, 4180, 4196, 4197, 4199, 4210, 4212, 
	4144, 4153, 4105, 4128, 4140, 4142, 4139, 4141, 
	4144, 4153, 4142, 4144, 4153, 4144, 4153, 4133, 
	4165, 4197, 4144, 4153, 4105, 4128, 4105, 4128, 
	4139, 4141, 4142, 4144, 4153, 4142, 4144, 4153, 
	4144, 4153, 4133, 4165, 4197, 4144, 4153, 4105, 
	4128, 4143, 4105, 4128, 4139, 4141, 4142, 4144, 
	4153, 4142, 4144, 4153, 4144, 4153, 4105, 4128, 
	4133, 4137, 4165, 4197, 4144, 4153, 4105, 4128, 
	4137, 4105, 4128, 4137, 4139, 4141, 4144, 4153, 
	4144, 4153, 4105, 4128, 4133, 4137, 4144, 4153, 
	4105, 4128, 4133, 4137, 4142, 4165, 4197, 4144, 
	4153, 4139, 4141, 4144, 4153, 4144, 4153, 4133, 
	4144, 4153, 4133, 4142, 4165, 4197, 4144, 4153, 
	4139, 4141, 4144, 4153, 4144, 4153, 4133, 4144, 
	4153, 4133, 4142, 4165, 4197, 4144, 4153, 4105, 
	4128, 4139, 4141, 4142, 4144, 4153, 4142, 4144, 
	4153, 4144, 4153, 4133, 4165, 4197, 4144, 4153, 
	4105, 4128, 4140, 4105, 4128, 4139, 4141, 4142, 
	4144, 4153, 4142, 4144, 4153, 4144, 4153, 4133, 
	4165, 4197, 4144, 4153, 4105, 4128, 4140, 4105, 
	4128, 4139, 4141, 4142, 4144, 4153, 4142, 4144, 
	4153, 4144, 4153, 4105, 4128, 4133, 4137, 4165, 
	4197, 4144, 4153, 4105, 4128, 4137, 4105, 4128, 
	4137, 4139, 4141, 4144, 4153, 4144, 4153, 4105, 
	4128, 4133, 4137, 4144, 4153, 4105, 4128, 4133, 
	4137, 4142, 4165, 4197, 4144, 4153, 4139, 4141, 
	4144, 4153, 4144, 4153, 4133, 4144, 4153, 4133, 
	4142, 4165, 4197, 4144, 4153, 4139, 4141, 4144, 
	4153, 4144, 4153, 4133, 4144, 4153, 4133, 4142, 
	4165, 4197, 4144, 4153, 4165, 4197, 4167, 4199, 
	4105, 4128, 4140, 4139, 4141, 4144, 4153, 4144, 
	4153, 4105, 4128, 4140, 4164, 4167, 4178, 4180, 
	4196, 4199, 4210, 4212, 4144, 4153, 4178, 4210, 
	4161, 4193, 4164, 4196, 4105, 4128, 4140, 4161, 
	4193, 4164, 4196, 4105, 4128, 4140, 4181, 4213, 
	4178, 4210, 4174, 4206, 4105, 4128, 4140, 4105, 
	4128, 4140, 4142, 4164, 4165, 4167, 4178, 4180, 
	4196, 4197, 4199, 4210, 4212, 4144, 4153, 28748, 
	28780, 29004, 29036, 29260, 29292, 3624, 4161, 4193, 
	4684, 4716, 4675, 4707, 4680, 4712, 4648, 4617, 
	4640, 4651, 4653, 4654, 4656, 4665, 4654, 4656, 
	4665, 4656, 4665, 4617, 4640, 4645, 4677, 4709, 
	4656, 4665, 4617, 4640, 4651, 4653, 4654, 4656, 
	4665, 4654, 4656, 4665, 4656, 4665, 4617, 4640, 
	4645, 4677, 4709, 4656, 4665, 4617, 4640, 4651, 
	4653, 4654, 4656, 4665, 4654, 4656, 4665, 4656, 
	4665, 4617, 4640, 4649, 4676, 4677, 4679, 4690, 
	4692, 4708, 4709, 4711, 4722, 4724, 4656, 4665, 
	4617, 4640, 4649, 4677, 4709, 4679, 4711, 4617, 
	4640, 4649, 4651, 4653, 4656, 4665, 4656, 4665, 
	4617, 4640, 4649, 4676, 4679, 4690, 4692, 4708, 
	4711, 4722, 4724, 4656, 4665, 4690, 4722, 4673, 
	4705, 4676, 4708, 4617, 4640, 4649, 4673, 4705, 
	4676, 4708, 4617, 4640, 4649, 4693, 4725, 4690, 
	4722, 4686, 4718, 4617, 4640, 4649, 4617, 4640, 
	4649, 4654, 4676, 4677, 4679, 4690, 4692, 4708, 
	4709, 4711, 4722, 4724, 4656, 4665, 4617, 4640, 
	4651, 4653, 4656, 4665, 4656, 4665, 4617, 4640, 
	4645, 4656, 4665, 4617, 4640, 4645, 4654, 4677, 
	4709, 4656, 4665, 4617, 4640, 4651, 4653, 4656, 
	4665, 4656, 4665, 4617, 4640, 4645, 4656, 4665, 
	4617, 4640, 4645, 4654, 4677, 4709, 4656, 4665, 
	5196, 5228, 5187, 5219, 5192, 5224, 5160, 5129, 
	5152, 5163, 5165, 5166, 5168, 5177, 5166, 5168, 
	5177, 5168, 5177, 5129, 5152, 5157, 5189, 5221, 
	5168, 5177, 5129, 5152, 5163, 5165, 5166, 5168, 
	5177, 5166, 5168, 5177, 5168, 5177, 5129, 5152, 
	5157, 5189, 5221, 5168, 5177, 5129, 5152, 5163, 
	5165, 5166, 5168, 5177, 5166, 5168, 5177, 5168, 
	5177, 5129, 5152, 5167, 5188, 5189, 5191, 5202, 
	5204, 5220, 5221, 5223, 5234, 5236, 5168, 5177, 
	5129, 5152, 5167, 5129, 5152, 5163, 5165, 5166, 
	5168, 5177, 5166, 5168, 5177, 5168, 5177, 5129, 
	5152, 5157, 5161, 5189, 5221, 5168, 5177, 5129, 
	5152, 5161, 5129, 5152, 5161, 5163, 5165, 5168, 
	5177, 5168, 5177, 5129, 5152, 5157, 5161, 5168, 
	5177, 5129, 5152, 5157, 5161, 5166, 5189, 5221, 
	5168, 5177, 5189, 5221, 5191, 5223, 5129, 5152, 
	5167, 5163, 5165, 5168, 5177, 5168, 5177, 5129, 
	5152, 5167, 5188, 5191, 5202, 5204, 5220, 5223, 
	5234, 5236, 5168, 5177, 5202, 5234, 5185, 5217, 
	5188, 5220, 5129, 5152, 5167, 5185, 5217, 5188, 
	5220, 5129, 5152, 5167, 5205, 5237, 5202, 5234, 
	5198, 5230, 5129, 5152, 5167, 5129, 5152, 5166, 
	5167, 5188, 5189, 5191, 5202, 5204, 5220, 5221, 
	5223, 5234, 5236, 5168, 5177, 5129, 5152, 5163, 
	5165, 5168, 5177, 5168, 5177, 5129, 5152, 5157, 
	5168, 5177, 5129, 5152, 5157, 5166, 5189, 5221, 
	5168, 5177, 5129, 5152, 5163, 5165, 5168, 5177, 
	5168, 5177, 5129, 5152, 5157, 5168, 5177, 5129, 
	5152, 5157, 5166, 5189, 5221, 5168, 5177, 29772, 
	29804, 30028, 30060, 30284, 30316, 29763, 29795, 30019, 
	30051, 30275, 30307, 29768, 29800, 30024, 30056, 30280, 
	30312, 29736, 29992, 30248, 29705, 29728, 29739, 29741, 
	29742, 29961, 29984, 29995, 29997, 29998, 30217, 30240, 
	30251, 30253, 30254, 29744, 29753, 30000, 30009, 30256, 
	30265, 29742, 29998, 30254, 29744, 29753, 30000, 30009, 
	30256, 30265, 29744, 29753, 30000, 30009, 30256, 30265, 
	29705, 29728, 29733, 29765, 29797, 29961, 29984, 29989, 
	30021, 30053, 30217, 30240, 30245, 30277, 30309, 29744, 
	29753, 30000, 30009, 30256, 30265, 29705, 29728, 29739, 
	29741, 29742, 29961, 29984, 29995, 29997, 29998, 30217, 
	30240, 30251, 30253, 30254, 29744, 29753, 30000, 30009, 
	30256, 30265, 29742, 29998, 30254, 29744, 29753, 30000, 
	30009, 30256, 30265, 29744, 29753, 30000, 30009, 30256, 
	30265, 29705, 29728, 29733, 29765, 29797, 29961, 29984, 
	29989, 30021, 30053, 30217, 30240, 30245, 30277, 30309, 
	29744, 29753, 30000, 30009, 30256, 30265, 29705, 29728, 
	29739, 29741, 29742, 29961, 29984, 29995, 29997, 29998, 
	30217, 30240, 30251, 30253, 30254, 29744, 29753, 30000, 
	30009, 30256, 30265, 29742, 29998, 30254, 29744, 29753, 
	30000, 30009, 30256, 30265, 29744, 29753, 30000, 30009, 
	30256, 30265, 4649, 5167, 29705, 29728, 29764, 29765, 
	29767, 29778, 29780, 29796, 29797, 29799, 29810, 29812, 
	29961, 29984, 30020, 30021, 30023, 30034, 30036, 30052, 
	30053, 30055, 30066, 30068, 30217, 30240, 30276, 30277, 
	30279, 30290, 30292, 30308, 30309, 30311, 30322, 30324, 
	29744, 29753, 30000, 30009, 30256, 30265, 4649, 5167, 
	29705, 29728, 29961, 29984, 30217, 30240, 29765, 29797, 
	30021, 30053, 30277, 30309, 29767, 29799, 30023, 30055, 
	30279, 30311, 4649, 5167, 29705, 29728, 29961, 29984, 
	30217, 30240, 29739, 29741, 29995, 29997, 30251, 30253, 
	29744, 29753, 30000, 30009, 30256, 30265, 29744, 29753, 
	30000, 30009, 30256, 30265, 4649, 5167, 29705, 29728, 
	29764, 29767, 29778, 29780, 29796, 29799, 29810, 29812, 
	29961, 29984, 30020, 30023, 30034, 30036, 30052, 30055, 
	30066, 30068, 30217, 30240, 30276, 30279, 30290, 30292, 
	30308, 30311, 30322, 30324, 29744, 29753, 30000, 30009, 
	30256, 30265, 29778, 29810, 30034, 30066, 30290, 30322, 
	29761, 29793, 30017, 30049, 30273, 30305, 29764, 29796, 
	30020, 30052, 30276, 30308, 4649, 5167, 29705, 29728, 
	29961, 29984, 30217, 30240, 29761, 29793, 30017, 30049, 
	30273, 30305, 29764, 29796, 30020, 30052, 30276, 30308, 
	4649, 5167, 29705, 29728, 29961, 29984, 30217, 30240, 
	29781, 29813, 30037, 30069, 30293, 30325, 29778, 29810, 
	30034, 30066, 30290, 30322, 29774, 29806, 30030, 30062, 
	30286, 30318, 4649, 5167, 29705, 29728, 29961, 29984, 
	30217, 30240, 4649, 5167, 29705, 29728, 29742, 29764, 
	29765, 29767, 29778, 29780, 29796, 29797, 29799, 29810, 
	29812, 29961, 29984, 29998, 30020, 30021, 30023, 30034, 
	30036, 30052, 30053, 30055, 30066, 30068, 30217, 30240, 
	30254, 30276, 30277, 30279, 30290, 30292, 30308, 30309, 
	30311, 30322, 30324, 29744, 29753, 30000, 30009, 30256, 
	30265, 29705, 29728, 29961, 29984, 30217, 30240, 29739, 
	29741, 29995, 29997, 30251, 30253, 29744, 29753, 30000, 
	30009, 30256, 30265, 29744, 29753, 30000, 30009, 30256, 
	30265, 29705, 29728, 29733, 29961, 29984, 29989, 30217, 
	30240, 30245, 29744, 29753, 30000, 30009, 30256, 30265, 
	29705, 29728, 29733, 29742, 29765, 29797, 29961, 29984, 
	29989, 29998, 30021, 30053, 30217, 30240, 30245, 30254, 
	30277, 30309, 29744, 29753, 30000, 30009, 30256, 30265, 
	29705, 29728, 29961, 29984, 30217, 30240, 29739, 29741, 
	29995, 29997, 30251, 30253, 29744, 29753, 30000, 30009, 
	30256, 30265, 29744, 29753, 30000, 30009, 30256, 30265, 
	29705, 29728, 29733, 29961, 29984, 29989, 30217, 30240, 
	30245, 29744, 29753, 30000, 30009, 30256, 30265, 29705, 
	29728, 29733, 29742, 29765, 29797, 29961, 29984, 29989, 
	29998, 30021, 30053, 30217, 30240, 30245, 30254, 30277, 
	30309, 29744, 29753, 30000, 30009, 30256, 30265, 5708, 
	5740, 5697, 5729, 5698, 5730, 5672, 5641, 5664, 
	5675, 5677, 5678, 5680, 5689, 5678, 5680, 5689, 
	5680, 5689, 5641, 5664, 5669, 5701, 5733, 5680, 
	5689, 5641, 5664, 5675, 5677, 5678, 5680, 5689, 
	5678, 5680, 5689, 5680, 5689, 5641, 5664, 5669, 
	5701, 5733, 5680, 5689, 5641, 5664, 5675, 5677, 
	5678, 5680, 5689, 5678, 5680, 5689, 5680, 5689, 
	5641, 5664, 5669, 5673, 5701, 5733, 5680, 5689, 
	5641, 5664, 5673, 5641, 5664, 5673, 5675, 5677, 
	5680, 5689, 5680, 5689, 5641, 5664, 5669, 5673, 
	5680, 5689, 5641, 5664, 5669, 5673, 5678, 5701, 
	5733, 5680, 5689, 5641, 5664, 5675, 5677, 5680, 
	5689, 5680, 5689, 5641, 5664, 5669, 5680, 5689, 
	5641, 5664, 5669, 5678, 5701, 5733, 5680, 5689, 
	5641, 5664, 5675, 5677, 5680, 5689, 5680, 5689, 
	5641, 5664, 5669, 5680, 5689, 5641, 5664, 5669, 
	5678, 5701, 5733, 5680, 5689, 32844, 32876, 33100, 
	33132, 33356, 33388, 4675, 4707, 5697, 5729, 33868, 
	33900, 34124, 34156, 34380, 34412, 5187, 5219, 5697, 
	5729, 30796, 30828, 31052, 31084, 31308, 31340, 31564, 
	31596, 31820, 31852, 32076, 32108, 32332, 32364, 5697, 
	5729, 29763, 29795, 30019, 30051, 30275, 30307, 6220, 
	6252, 6209, 6241, 6210, 6242, 6184, 6153, 6176, 
	6187, 6189, 6190, 6192, 6201, 6190, 6192, 6201, 
	6192, 6201, 6153, 6176, 6181, 6213, 6245, 6192, 
	6201, 6153, 6176, 6187, 6189, 6190, 6192, 6201, 
	6190, 6192, 6201, 6192, 6201, 6153, 6176, 6181, 
	6213, 6245, 6192, 6201, 6153, 6176, 6187, 6189, 
	6190, 6192, 6201, 6190, 6192, 6201, 6192, 6201, 
	6153, 6176, 6181, 6191, 6213, 6245, 6192, 6201, 
	6153, 6176, 6191, 6153, 6176, 6187, 6189, 6190, 
	6192, 6201, 6190, 6192, 6201, 6192, 6201, 6153, 
	6176, 6181, 6185, 6213, 6245, 6192, 6201, 6153, 
	6176, 6185, 6153, 6176, 6185, 6187, 6189, 6192, 
	6201, 6192, 6201, 6153, 6176, 6181, 6185, 6192, 
	6201, 6153, 6176, 6181, 6185, 6190, 6213, 6245, 
	6192, 6201, 6153, 6176, 6191, 6187, 6189, 6192, 
	6201, 6192, 6201, 6153, 6176, 6181, 6191, 6192, 
	6201, 6153, 6176, 6181, 6190, 6191, 6213, 6245, 
	6192, 6201, 6153, 6176, 6187, 6189, 6192, 6201, 
	6192, 6201, 6153, 6176, 6181, 6192, 6201, 6153, 
	6176, 6181, 6190, 6213, 6245, 6192, 6201, 6153, 
	6176, 6187, 6189, 6192, 6201, 6192, 6201, 6153, 
	6176, 6181, 6192, 6201, 6153, 6176, 6181, 6190, 
	6213, 6245, 6192, 6201, 38988, 39020, 39244, 39276, 
	39500, 39532, 4675, 4707, 6209, 6241, 40012, 40044, 
	40268, 40300, 40524, 40556, 5187, 5219, 6209, 6241, 
	41036, 41068, 41292, 41324, 41548, 41580, 41804, 41836, 
	42060, 42092, 42316, 42348, 42572, 42604, 6209, 6241, 
	29763, 29795, 30019, 30051, 30275, 30307, 43084, 43116, 
	43340, 43372, 43596, 43628, 43073, 43105, 43329, 43361, 
	43585, 43617, 43074, 43106, 43330, 43362, 43586, 43618, 
	43048, 43304, 43560, 43017, 43040, 43051, 43053, 43054, 
	43273, 43296, 43307, 43309, 43310, 43529, 43552, 43563, 
	43565, 43566, 43056, 43065, 43312, 43321, 43568, 43577, 
	43054, 43310, 43566, 43056, 43065, 43312, 43321, 43568, 
	43577, 43056, 43065, 43312, 43321, 43568, 43577, 43017, 
	43040, 43045, 43077, 43109, 43273, 43296, 43301, 43333, 
	43365, 43529, 43552, 43557, 43589, 43621, 43056, 43065, 
	43312, 43321, 43568, 43577, 43017, 43040, 43051, 43053, 
	43054, 43273, 43296, 43307, 43309, 43310, 43529, 43552, 
	43563, 43565, 43566, 43056, 43065, 43312, 43321, 43568, 
	43577, 43054, 43310, 43566, 43056, 43065, 43312, 43321, 
	43568, 43577, 43056, 43065, 43312, 43321, 43568, 43577, 
	43017, 43040, 43045, 43077, 43109, 43273, 43296, 43301, 
	43333, 43365, 43529, 43552, 43557, 43589, 43621, 43056, 
	43065, 43312, 43321, 43568, 43577, 43017, 43040, 43051, 
	43053, 43054, 43273, 43296, 43307, 43309, 43310, 43529, 
	43552, 43563, 43565, 43566, 43056, 43065, 43312, 43321, 
	43568, 43577, 43054, 43310, 43566, 43056, 43065, 43312, 
	43321, 43568, 43577, 43056, 43065, 43312, 43321, 43568, 
	43577, 5673, 6191, 43017, 43040, 43045, 43077, 43109, 
	43273, 43296, 43301, 43333, 43365, 43529, 43552, 43557, 
	43589, 43621, 43056, 43065, 43312, 43321, 43568, 43577, 
	5673, 6191, 43017, 43040, 43273, 43296, 43529, 43552, 
	5673, 6191, 43017, 43040, 43273, 43296, 43529, 43552, 
	43051, 43053, 43307, 43309, 43563, 43565, 43056, 43065, 
	43312, 43321, 43568, 43577, 43056, 43065, 43312, 43321, 
	43568, 43577, 5673, 6191, 43017, 43040, 43045, 43273, 
	43296, 43301, 43529, 43552, 43557, 43056, 43065, 43312, 
	43321, 43568, 43577, 5673, 6191, 43017, 43040, 43045, 
	43054, 43077, 43109, 43273, 43296, 43301, 43310, 43333, 
	43365, 43529, 43552, 43557, 43566, 43589, 43621, 43056, 
	43065, 43312, 43321, 43568, 43577, 43017, 43040, 43273, 
	43296, 43529, 43552, 43051, 43053, 43307, 43309, 43563, 
	43565, 43056, 43065, 43312, 43321, 43568, 43577, 43056, 
	43065, 43312, 43321, 43568, 43577, 43017, 43040, 43045, 
	43273, 43296, 43301, 43529, 43552, 43557, 43056, 43065, 
	43312, 43321, 43568, 43577, 43017, 43040, 43045, 43054, 
	43077, 43109, 43273, 43296, 43301, 43310, 43333, 43365, 
	43529, 43552, 43557, 43566, 43589, 43621, 43056, 43065, 
	43312, 43321, 43568, 43577, 43017, 43040, 43273, 43296, 
	43529, 43552, 43051, 43053, 43307, 43309, 43563, 43565, 
	43056, 43065, 43312, 43321, 43568, 43577, 43056, 43065, 
	43312, 43321, 43568, 43577, 43017, 43040, 43045, 43273, 
	43296, 43301, 43529, 43552, 43557, 43056, 43065, 43312, 
	43321, 43568, 43577, 43017, 43040, 43045, 43054, 43077, 
	43109, 43273, 43296, 43301, 43310, 43333, 43365, 43529, 
	43552, 43557, 43566, 43589, 43621, 43056, 43065, 43312, 
	43321, 43568, 43577, 44108, 44140, 44364, 44396, 44620, 
	44652, 44876, 44908, 45132, 45164, 45388, 45420, 45644, 
	45676, 4675, 4707, 43073, 43105, 43329, 43361, 43585, 
	43617, 46156, 46188, 46412, 46444, 46668, 46700, 46924, 
	46956, 47180, 47212, 47436, 47468, 47692, 47724, 5187, 
	5219, 43073, 43105, 43329, 43361, 43585, 43617, 34892, 
	34924, 35148, 35180, 35404, 35436, 35660, 35692, 35916, 
	35948, 36172, 36204, 36428, 36460, 36684, 36716, 36940, 
	36972, 37196, 37228, 37452, 37484, 37708, 37740, 37964, 
	37996, 38220, 38252, 38476, 38508, 29763, 29795, 30019, 
	30051, 30275, 30307, 43073, 43105, 43329, 43361, 43585, 
	43617, 6665, 6688, 6700, 6704, 6713, 6714, 6715, 
	6665, 6688, 6704, 6713, 6665, 6688, 6700, 6704, 
	6713, 6714, 6715, 6665, 6688, 6700, 6714, 6715, 
	6665, 6688, 6704, 6713, 6665, 6688, 6700, 6714, 
	6715, 6665, 6688, 6704, 6713, 6665, 6688, 6704, 
	6713, 6665, 6688, 6704, 6713, 6665, 6688, 6704, 
	6713, 6665, 6688, 6700, 6704, 6713, 6714, 6715, 
	6665, 6688, 6700, 577, 582, 609, 614, 6714, 
	6715, 56368, 56377, 56624, 56633, 56880, 56889, 6665, 
	6688, 6700, 577, 582, 609, 614, 6714, 6715, 
	56368, 56377, 56624, 56633, 56880, 56889, 6665, 6688, 
	6700, 577, 582, 609, 614, 6714, 6715, 56368, 
	56377, 56624, 56633, 56880, 56889, 6665, 6688, 6700, 
	577, 582, 609, 614, 6714, 6715, 56368, 56377, 
	56624, 56633, 56880, 56889, 6665, 6688, 6700, 577, 
	582, 609, 614, 6714, 6715, 56368, 56377, 56624, 
	56633, 56880, 56889, 6665, 6688, 6700, 577, 582, 
	609, 614, 6714, 6715, 56368, 56377, 56624, 56633, 
	56880, 56889, 6665, 6688, 6700, 1089, 1094, 1121, 
	1126, 6714, 6715, 57392, 57401, 57648, 57657, 57904, 
	57913, 6665, 6688, 6700, 1089, 1094, 1121, 1126, 
	6714, 6715, 57392, 57401, 57648, 57657, 57904, 57913, 
	6665, 6688, 6700, 1089, 1094, 1121, 1126, 6714, 
	6715, 57392, 57401, 57648, 57657, 57904, 57913, 6665, 
	6688, 6700, 1089, 1094, 1121, 1126, 6714, 6715, 
	57392, 57401, 57648, 57657, 57904, 57913, 6665, 6688, 
	6700, 6714, 6715, 9281, 9286, 9313, 9318, 9537, 
	9542, 9569, 9574, 9793, 9798, 9825, 9830, 58416, 
	58425, 58672, 58681, 58928, 58937, 59184, 59193, 59440, 
	59449, 59696, 59705, 59952, 59961, 6665, 6688, 6700, 
	6714, 6715, 9281, 9286, 9313, 9318, 9537, 9542, 
	9569, 9574, 9793, 9798, 9825, 9830, 58416, 58425, 
	58672, 58681, 58928, 58937, 59184, 59193, 59440, 59449, 
	59696, 59705, 59952, 59961, 6665, 6688, 6700, 6714, 
	6715, 9281, 9286, 9313, 9318, 9537, 9542, 9569, 
	9574, 9793, 9798, 9825, 9830, 58416, 58425, 58672, 
	58681, 58928, 58937, 59184, 59193, 59440, 59449, 59696, 
	59705, 59952, 59961, 6665, 6688, 6700, 6714, 6715, 
	9281, 9286, 9313, 9318, 9537, 9542, 9569, 9574, 
	9793, 9798, 9825, 9830, 58416, 58425, 58672, 58681, 
	58928, 58937, 59184, 59193, 59440, 59449, 59696, 59705, 
	59952, 59961, 6665, 6688, 6700, 1601, 1606, 1633, 
	1638, 6714, 6715, 60464, 60473, 60720, 60729, 60976, 
	60985, 6665, 6688, 6700, 1601, 1606, 1633, 1638, 
	6714, 6715, 60464, 60473, 60720, 60729, 60976, 60985, 
	6665, 6688, 6700, 6714, 6715, 12353, 12358, 12385, 
	12390, 12609, 12614, 12641, 12646, 12865, 12870, 12897, 
	12902, 61488, 61497, 61744, 61753, 62000, 62009, 62256, 
	62265, 62512, 62521, 62768, 62777, 63024, 63033, 6665, 
	6688, 6700, 6714, 6715, 12353, 12358, 12385, 12390, 
	12609, 12614, 12641, 12646, 12865, 12870, 12897, 12902, 
	61488, 61497, 61744, 61753, 62000, 62009, 62256, 62265, 
	62512, 62521, 62768, 62777, 63024, 63033, 6665, 6688, 
	6700, 6714, 6715, 13377, 13382, 13409, 13414, 13633, 
	13638, 13665, 13670, 13889, 13894, 13921, 13926, 63536, 
	63545, 63792, 63801, 64048, 64057, 64304, 64313, 64560, 
	64569, 64816, 64825, 65072, 65081, 6665, 6688, 6700, 
	6714, 6715, 13377, 13382, 13409, 13414, 13633, 13638, 
	13665, 13670, 13889, 13894, 13921, 13926, 63536, 63545, 
	63792, 63801, 64048, 64057, 64304, 64313, 64560, 64569, 
	64816, 64825, 65072, 65081, 6665, 6688, 6700, 6714, 
	6715, 10305, 10310, 10337, 10342, 10561, 10566, 10593, 
	10598, 10817, 10822, 10849, 10854, 11073, 11078, 11105, 
	11110, 11329, 11334, 11361, 11366, 11585, 11590, 11617, 
	11622, 11841, 11846, 11873, 11878, 65584, 65593, 65840, 
	65849, 66096, 66105, 66352, 66361, 66608, 66617, 66864, 
	66873, 67120, 67129, 67376, 67385, 67632, 67641, 67888, 
	67897, 68144, 68153, 68400, 68409, 68656, 68665, 68912, 
	68921, 69168, 69177, 6665, 6688, 6700, 6714, 6715, 
	10305, 10310, 10337, 10342, 10561, 10566, 10593, 10598, 
	10817, 10822, 10849, 10854, 11073, 11078, 11105, 11110, 
	11329, 11334, 11361, 11366, 11585, 11590, 11617, 11622, 
	11841, 11846, 11873, 11878, 65584, 65593, 65840, 65849, 
	66096, 66105, 66352, 66361, 66608, 66617, 66864, 66873, 
	67120, 67129, 67376, 67385, 67632, 67641, 67888, 67897, 
	68144, 68153, 68400, 68409, 68656, 68665, 68912, 68921, 
	69168, 69177, 6665, 6688, 6700, 2113, 2118, 2145, 
	2150, 6714, 6715, 69680, 69689, 69936, 69945, 70192, 
	70201, 6665, 6688, 6700, 6714, 6715, 18497, 18502, 
	18529, 18534, 18753, 18758, 18785, 18790, 19009, 19014, 
	19041, 19046, 70704, 70713, 70960, 70969, 71216, 71225, 
	71472, 71481, 71728, 71737, 71984, 71993, 72240, 72249, 
	6665, 6688, 6700, 6714, 6715, 19521, 19526, 19553, 
	19558, 19777, 19782, 19809, 19814, 20033, 20038, 20065, 
	20070, 72752, 72761, 73008, 73017, 73264, 73273, 73520, 
	73529, 73776, 73785, 74032, 74041, 74288, 74297, 6665, 
	6688, 6700, 6714, 6715, 20545, 20550, 20577, 20582, 
	20801, 20806, 20833, 20838, 21057, 21062, 21089, 21094, 
	21313, 21318, 21345, 21350, 21569, 21574, 21601, 21606, 
	21825, 21830, 21857, 21862, 22081, 22086, 22113, 22118, 
	74800, 74809, 75056, 75065, 75312, 75321, 75568, 75577, 
	75824, 75833, 76080, 76089, 76336, 76345, 76592, 76601, 
	76848, 76857, 77104, 77113, 77360, 77369, 77616, 77625, 
	77872, 77881, 78128, 78137, 78384, 78393, 6665, 6688, 
	6700, 6714, 6715, 22593, 22598, 22625, 22630, 22849, 
	22854, 22881, 22886, 23105, 23110, 23137, 23142, 78896, 
	78905, 79152, 79161, 79408, 79417, 79664, 79673, 79920, 
	79929, 80176, 80185, 80432, 80441, 6665, 6688, 6700, 
	6714, 6715, 23617, 23622, 23649, 23654, 23873, 23878, 
	23905, 23910, 24129, 24134, 24161, 24166, 24385, 24390, 
	24417, 24422, 24641, 24646, 24673, 24678, 24897, 24902, 
	24929, 24934, 25153, 25158, 25185, 25190, 80944, 80953, 
	81200, 81209, 81456, 81465, 81712, 81721, 81968, 81977, 
	82224, 82233, 82480, 82489, 82736, 82745, 82992, 83001, 
	83248, 83257, 83504, 83513, 83760, 83769, 84016, 84025, 
	84272, 84281, 84528, 84537, 6665, 6688, 6700, 6714, 
	6715, 25665, 25670, 25697, 25702, 25921, 25926, 25953, 
	25958, 26177, 26182, 26209, 26214, 26433, 26438, 26465, 
	26470, 26689, 26694, 26721, 26726, 26945, 26950, 26977, 
	26982, 27201, 27206, 27233, 27238, 85040, 85049, 85296, 
	85305, 85552, 85561, 85808, 85817, 86064, 86073, 86320, 
	86329, 86576, 86585, 86832, 86841, 87088, 87097, 87344, 
	87353, 87600, 87609, 87856, 87865, 88112, 88121, 88368, 
	88377, 88624, 88633, 6665, 6688, 6700, 6714, 6715, 
	14401, 14406, 14433, 14438, 14657, 14662, 14689, 14694, 
	14913, 14918, 14945, 14950, 15169, 15174, 15201, 15206, 
	15425, 15430, 15457, 15462, 15681, 15686, 15713, 15718, 
	15937, 15942, 15969, 15974, 16193, 16198, 16225, 16230, 
	16449, 16454, 16481, 16486, 16705, 16710, 16737, 16742, 
	16961, 16966, 16993, 16998, 17217, 17222, 17249, 17254, 
	17473, 17478, 17505, 17510, 17729, 17734, 17761, 17766, 
	17985, 17990, 18017, 18022, 48176, 48185, 48432, 48441, 
	48688, 48697, 48944, 48953, 49200, 49209, 49456, 49465, 
	49712, 49721, 49968, 49977, 50224, 50233, 50480, 50489, 
	50736, 50745, 50992, 51001, 51248, 51257, 51504, 51513, 
	51760, 51769, 52016, 52025, 52272, 52281, 52528, 52537, 
	52784, 52793, 53040, 53049, 53296, 53305, 53552, 53561, 
	53808, 53817, 54064, 54073, 54320, 54329, 54576, 54585, 
	54832, 54841, 55088, 55097, 55344, 55353, 55600, 55609, 
	55856, 55865, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 582, 614, 7177, 7200, 7212, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 577, 580, 
	609, 612, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 7211, 7213, 577, 582, 609, 614, 
	105520, 105529, 105776, 105785, 106032, 106041, 7211, 7213, 
	577, 582, 609, 614, 105520, 105529, 105776, 105785, 
	106032, 106041, 7177, 7200, 7212, 577, 582, 609, 
	614, 7226, 7227, 105520, 105529, 105776, 105785, 106032, 
	106041, 7211, 7213, 577, 582, 609, 614, 105520, 
	105529, 105776, 105785, 106032, 106041, 7177, 7200, 7212, 
	577, 582, 609, 614, 7226, 7227, 105520, 105529, 
	105776, 105785, 106032, 106041, 7211, 7213, 577, 582, 
	609, 614, 105520, 105529, 105776, 105785, 106032, 106041, 
	7177, 7200, 7212, 577, 582, 609, 614, 7226, 
	7227, 105520, 105529, 105776, 105785, 106032, 106041, 7211, 
	7213, 577, 582, 609, 614, 105520, 105529, 105776, 
	105785, 106032, 106041, 7177, 7200, 7212, 577, 582, 
	609, 614, 7226, 7227, 105520, 105529, 105776, 105785, 
	106032, 106041, 7211, 7213, 577, 582, 609, 614, 
	105520, 105529, 105776, 105785, 106032, 106041, 7177, 7200, 
	7212, 577, 582, 609, 614, 7226, 7227, 105520, 
	105529, 105776, 105785, 106032, 106041, 1094, 1126, 7177, 
	7200, 7212, 7214, 106565, 106597, 106821, 106853, 107077, 
	107109, 1089, 1092, 1121, 1124, 7226, 7227, 106544, 
	106553, 106800, 106809, 107056, 107065, 1094, 1126, 7177, 
	7200, 7212, 7214, 106565, 106597, 106821, 106853, 107077, 
	107109, 1089, 1092, 1121, 1124, 7226, 7227, 106544, 
	106553, 106800, 106809, 107056, 107065, 1094, 1126, 7177, 
	7200, 7212, 7214, 106565, 106597, 106821, 106853, 107077, 
	107109, 1089, 1092, 1121, 1124, 7226, 7227, 106544, 
	106553, 106800, 106809, 107056, 107065, 1094, 1126, 7177, 
	7200, 7212, 7214, 106565, 106597, 106821, 106853, 107077, 
	107109, 1089, 1092, 1121, 1124, 7226, 7227, 106544, 
	106553, 106800, 106809, 107056, 107065, 7211, 7213, 1089, 
	1094, 1121, 1126, 106544, 106553, 106800, 106809, 107056, 
	107065, 7211, 7213, 1089, 1094, 1121, 1126, 106544, 
	106553, 106800, 106809, 107056, 107065, 7177, 7200, 7212, 
	1089, 1094, 1121, 1126, 7226, 7227, 106544, 106553, 
	106800, 106809, 107056, 107065, 7211, 7213, 1089, 1094, 
	1121, 1126, 106544, 106553, 106800, 106809, 107056, 107065, 
	7177, 7200, 7212, 1089, 1094, 1121, 1126, 7226, 
	7227, 106544, 106553, 106800, 106809, 107056, 107065, 7211, 
	7213, 1089, 1094, 1121, 1126, 106544, 106553, 106800, 
	106809, 107056, 107065, 7177, 7200, 7212, 1089, 1094, 
	1121, 1126, 7226, 7227, 106544, 106553, 106800, 106809, 
	107056, 107065, 7177, 7200, 7212, 7214, 9286, 9318, 
	9542, 9574, 9798, 9830, 107589, 107621, 107845, 107877, 
	108101, 108133, 108357, 108389, 108613, 108645, 108869, 108901, 
	109125, 109157, 7226, 7227, 9281, 9284, 9313, 9316, 
	9537, 9540, 9569, 9572, 9793, 9796, 9825, 9828, 
	107568, 107577, 107824, 107833, 108080, 108089, 108336, 108345, 
	108592, 108601, 108848, 108857, 109104, 109113, 7177, 7200, 
	7212, 7214, 9286, 9318, 9542, 9574, 9798, 9830, 
	107589, 107621, 107845, 107877, 108101, 108133, 108357, 108389, 
	108613, 108645, 108869, 108901, 109125, 109157, 7226, 7227, 
	9281, 9284, 9313, 9316, 9537, 9540, 9569, 9572, 
	9793, 9796, 9825, 9828, 107568, 107577, 107824, 107833, 
	108080, 108089, 108336, 108345, 108592, 108601, 108848, 108857, 
	109104, 109113, 7177, 7200, 7212, 7214, 9286, 9318, 
	9542, 9574, 9798, 9830, 107589, 107621, 107845, 107877, 
	108101, 108133, 108357, 108389, 108613, 108645, 108869, 108901, 
	109125, 109157, 7226, 7227, 9281, 9284, 9313, 9316, 
	9537, 9540, 9569, 9572, 9793, 9796, 9825, 9828, 
	107568, 107577, 107824, 107833, 108080, 108089, 108336, 108345, 
	108592, 108601, 108848, 108857, 109104, 109113, 7177, 7200, 
	7212, 7214, 9286, 9318, 9542, 9574, 9798, 9830, 
	107589, 107621, 107845, 107877, 108101, 108133, 108357, 108389, 
	108613, 108645, 108869, 108901, 109125, 109157, 7226, 7227, 
	9281, 9284, 9313, 9316, 9537, 9540, 9569, 9572, 
	9793, 9796, 9825, 9828, 107568, 107577, 107824, 107833, 
	108080, 108089, 108336, 108345, 108592, 108601, 108848, 108857, 
	109104, 109113, 7211, 7213, 9281, 9286, 9313, 9318, 
	9537, 9542, 9569, 9574, 9793, 9798, 9825, 9830, 
	107568, 107577, 107824, 107833, 108080, 108089, 108336, 108345, 
	108592, 108601, 108848, 108857, 109104, 109113, 7211, 7213, 
	9281, 9286, 9313, 9318, 9537, 9542, 9569, 9574, 
	9793, 9798, 9825, 9830, 107568, 107577, 107824, 107833, 
	108080, 108089, 108336, 108345, 108592, 108601, 108848, 108857, 
	109104, 109113, 7177, 7200, 7212, 7226, 7227, 9281, 
	9286, 9313, 9318, 9537, 9542, 9569, 9574, 9793, 
	9798, 9825, 9830, 107568, 107577, 107824, 107833, 108080, 
	108089, 108336, 108345, 108592, 108601, 108848, 108857, 109104, 
	109113, 7211, 7213, 9281, 9286, 9313, 9318, 9537, 
	9542, 9569, 9574, 9793, 9798, 9825, 9830, 107568, 
	107577, 107824, 107833, 108080, 108089, 108336, 108345, 108592, 
	108601, 108848, 108857, 109104, 109113, 7177, 7200, 7212, 
	7226, 7227, 9281, 9286, 9313, 9318, 9537, 9542, 
	9569, 9574, 9793, 9798, 9825, 9830, 107568, 107577, 
	107824, 107833, 108080, 108089, 108336, 108345, 108592, 108601, 
	108848, 108857, 109104, 109113, 7211, 7213, 9281, 9286, 
	9313, 9318, 9537, 9542, 9569, 9574, 9793, 9798, 
	9825, 9830, 107568, 107577, 107824, 107833, 108080, 108089, 
	108336, 108345, 108592, 108601, 108848, 108857, 109104, 109113, 
	7177, 7200, 7212, 7226, 7227, 9281, 9286, 9313, 
	9318, 9537, 9542, 9569, 9574, 9793, 9798, 9825, 
	9830, 107568, 107577, 107824, 107833, 108080, 108089, 108336, 
	108345, 108592, 108601, 108848, 108857, 109104, 109113, 1606, 
	1638, 7177, 7200, 7212, 7214, 109637, 109669, 109893, 
	109925, 110149, 110181, 1601, 1604, 1633, 1636, 7226, 
	7227, 109616, 109625, 109872, 109881, 110128, 110137, 1606, 
	1638, 7177, 7200, 7212, 7214, 109637, 109669, 109893, 
	109925, 110149, 110181, 1601, 1604, 1633, 1636, 7226, 
	7227, 109616, 109625, 109872, 109881, 110128, 110137, 7211, 
	7213, 1601, 1606, 1633, 1638, 109616, 109625, 109872, 
	109881, 110128, 110137, 7211, 7213, 1601, 1606, 1633, 
	1638, 109616, 109625, 109872, 109881, 110128, 110137, 7177, 
	7200, 7212, 1601, 1606, 1633, 1638, 7226, 7227, 
	109616, 109625, 109872, 109881, 110128, 110137, 7177, 7200, 
	7212, 7214, 12358, 12390, 12614, 12646, 12870, 12902, 
	110661, 110693, 110917, 110949, 111173, 111205, 111429, 111461, 
	111685, 111717, 111941, 111973, 112197, 112229, 7226, 7227, 
	12353, 12356, 12385, 12388, 12609, 12612, 12641, 12644, 
	12865, 12868, 12897, 12900, 110640, 110649, 110896, 110905, 
	111152, 111161, 111408, 111417, 111664, 111673, 111920, 111929, 
	112176, 112185, 7177, 7200, 7212, 7214, 12358, 12390, 
	12614, 12646, 12870, 12902, 110661, 110693, 110917, 110949, 
	111173, 111205, 111429, 111461, 111685, 111717, 111941, 111973, 
	112197, 112229, 7226, 7227, 12353, 12356, 12385, 12388, 
	12609, 12612, 12641, 12644, 12865, 12868, 12897, 12900, 
	110640, 110649, 110896, 110905, 111152, 111161, 111408, 111417, 
	111664, 111673, 111920, 111929, 112176, 112185, 7211, 7213, 
	12353, 12358, 12385, 12390, 12609, 12614, 12641, 12646, 
	12865, 12870, 12897, 12902, 110640, 110649, 110896, 110905, 
	111152, 111161, 111408, 111417, 111664, 111673, 111920, 111929, 
	112176, 112185, 7211, 7213, 12353, 12358, 12385, 12390, 
	12609, 12614, 12641, 12646, 12865, 12870, 12897, 12902, 
	110640, 110649, 110896, 110905, 111152, 111161, 111408, 111417, 
	111664, 111673, 111920, 111929, 112176, 112185, 7177, 7200, 
	7212, 7226, 7227, 12353, 12358, 12385, 12390, 12609, 
	12614, 12641, 12646, 12865, 12870, 12897, 12902, 110640, 
	110649, 110896, 110905, 111152, 111161, 111408, 111417, 111664, 
	111673, 111920, 111929, 112176, 112185, 7177, 7200, 7212, 
	7214, 13382, 13414, 13638, 13670, 13894, 13926, 112709, 
	112741, 112965, 112997, 113221, 113253, 113477, 113509, 113733, 
	113765, 113989, 114021, 114245, 114277, 7226, 7227, 13377, 
	13380, 13409, 13412, 13633, 13636, 13665, 13668, 13889, 
	13892, 13921, 13924, 112688, 112697, 112944, 112953, 113200, 
	113209, 113456, 113465, 113712, 113721, 113968, 113977, 114224, 
	114233, 7177, 7200, 7212, 7214, 13382, 13414, 13638, 
	13670, 13894, 13926, 112709, 112741, 112965, 112997, 113221, 
	113253, 113477, 113509, 113733, 113765, 113989, 114021, 114245, 
	114277, 7226, 7227, 13377, 13380, 13409, 13412, 13633, 
	13636, 13665, 13668, 13889, 13892, 13921, 13924, 112688, 
	112697, 112944, 112953, 113200, 113209, 113456, 113465, 113712, 
	113721, 113968, 113977, 114224, 114233, 7211, 7213, 13377, 
	13382, 13409, 13414, 13633, 13638, 13665, 13670, 13889, 
	13894, 13921, 13926, 112688, 112697, 112944, 112953, 113200, 
	113209, 113456, 113465, 113712, 113721, 113968, 113977, 114224, 
	114233, 7211, 7213, 13377, 13382, 13409, 13414, 13633, 
	13638, 13665, 13670, 13889, 13894, 13921, 13926, 112688, 
	112697, 112944, 112953, 113200, 113209, 113456, 113465, 113712, 
	113721, 113968, 113977, 114224, 114233, 7177, 7200, 7212, 
	7226, 7227, 13377, 13382, 13409, 13414, 13633, 13638, 
	13665, 13670, 13889, 13894, 13921, 13926, 112688, 112697, 
	112944, 112953, 113200, 113209, 113456, 113465, 113712, 113721, 
	113968, 113977, 114224, 114233, 7177, 7200, 7212, 7214, 
	10310, 10342, 10566, 10598, 10822, 10854, 11078, 11110, 
	11334, 11366, 11590, 11622, 11846, 11878, 114757, 114789, 
	115013, 115045, 115269, 115301, 115525, 115557, 115781, 115813, 
	116037, 116069, 116293, 116325, 116549, 116581, 116805, 116837, 
	117061, 117093, 117317, 117349, 117573, 117605, 117829, 117861, 
	118085, 118117, 118341, 118373, 7226, 7227, 10305, 10308, 
	10337, 10340, 10561, 10564, 10593, 10596, 10817, 10820, 
	10849, 10852, 11073, 11076, 11105, 11108, 11329, 11332, 
	11361, 11364, 11585, 11588, 11617, 11620, 11841, 11844, 
	11873, 11876, 114736, 114745, 114992, 115001, 115248, 115257, 
	115504, 115513, 115760, 115769, 116016, 116025, 116272, 116281, 
	116528, 116537, 116784, 116793, 117040, 117049, 117296, 117305, 
	117552, 117561, 117808, 117817, 118064, 118073, 118320, 118329, 
	7177, 7200, 7212, 7214, 10310, 10342, 10566, 10598, 
	10822, 10854, 11078, 11110, 11334, 11366, 11590, 11622, 
	11846, 11878, 114757, 114789, 115013, 115045, 115269, 115301, 
	115525, 115557, 115781, 115813, 116037, 116069, 116293, 116325, 
	116549, 116581, 116805, 116837, 117061, 117093, 117317, 117349, 
	117573, 117605, 117829, 117861, 118085, 118117, 118341, 118373, 
	7226, 7227, 10305, 10308, 10337, 10340, 10561, 10564, 
	10593, 10596, 10817, 10820, 10849, 10852, 11073, 11076, 
	11105, 11108, 11329, 11332, 11361, 11364, 11585, 11588, 
	11617, 11620, 11841, 11844, 11873, 11876, 114736, 114745, 
	114992, 115001, 115248, 115257, 115504, 115513, 115760, 115769, 
	116016, 116025, 116272, 116281, 116528, 116537, 116784, 116793, 
	117040, 117049, 117296, 117305, 117552, 117561, 117808, 117817, 
	118064, 118073, 118320, 118329, 7211, 7213, 10305, 10310, 
	10337, 10342, 10561, 10566, 10593, 10598, 10817, 10822, 
	10849, 10854, 11073, 11078, 11105, 11110, 11329, 11334, 
	11361, 11366, 11585, 11590, 11617, 11622, 11841, 11846, 
	11873, 11878, 114736, 114745, 114992, 115001, 115248, 115257, 
	115504, 115513, 115760, 115769, 116016, 116025, 116272, 116281, 
	116528, 116537, 116784, 116793, 117040, 117049, 117296, 117305, 
	117552, 117561, 117808, 117817, 118064, 118073, 118320, 118329, 
	7211, 7213, 10305, 10310, 10337, 10342, 10561, 10566, 
	10593, 10598, 10817, 10822, 10849, 10854, 11073, 11078, 
	11105, 11110, 11329, 11334, 11361, 11366, 11585, 11590, 
	11617, 11622, 11841, 11846, 11873, 11878, 114736, 114745, 
	114992, 115001, 115248, 115257, 115504, 115513, 115760, 115769, 
	116016, 116025, 116272, 116281, 116528, 116537, 116784, 116793, 
	117040, 117049, 117296, 117305, 117552, 117561, 117808, 117817, 
	118064, 118073, 118320, 118329, 7177, 7200, 7212, 7226, 
	7227, 10305, 10310, 10337, 10342, 10561, 10566, 10593, 
	10598, 10817, 10822, 10849, 10854, 11073, 11078, 11105, 
	11110, 11329, 11334, 11361, 11366, 11585, 11590, 11617, 
	11622, 11841, 11846, 11873, 11878, 114736, 114745, 114992, 
	115001, 115248, 115257, 115504, 115513, 115760, 115769, 116016, 
	116025, 116272, 116281, 116528, 116537, 116784, 116793, 117040, 
	117049, 117296, 117305, 117552, 117561, 117808, 117817, 118064, 
	118073, 118320, 118329, 2118, 2150, 7177, 7200, 7212, 
	7214, 118853, 118885, 119109, 119141, 119365, 119397, 2113, 
	2116, 2145, 2148, 7226, 7227, 118832, 118841, 119088, 
	119097, 119344, 119353, 7211, 7213, 2113, 2118, 2145, 
	2150, 118832, 118841, 119088, 119097, 119344, 119353, 7177, 
	7200, 7212, 7214, 18502, 18534, 18758, 18790, 19014, 
	19046, 119877, 119909, 120133, 120165, 120389, 120421, 120645, 
	120677, 120901, 120933, 121157, 121189, 121413, 121445, 7226, 
	7227, 18497, 18500, 18529, 18532, 18753, 18756, 18785, 
	18788, 19009, 19012, 19041, 19044, 119856, 119865, 120112, 
	120121, 120368, 120377, 120624, 120633, 120880, 120889, 121136, 
	121145, 121392, 121401, 7211, 7213, 18497, 18502, 18529, 
	18534, 18753, 18758, 18785, 18790, 19009, 19014, 19041, 
	19046, 119856, 119865, 120112, 120121, 120368, 120377, 120624, 
	120633, 120880, 120889, 121136, 121145, 121392, 121401, 7177, 
	7200, 7212, 7214, 19526, 19558, 19782, 19814, 20038, 
	20070, 121925, 121957, 122181, 122213, 122437, 122469, 122693, 
	122725, 122949, 122981, 123205, 123237, 123461, 123493, 7226, 
	7227, 19521, 19524, 19553, 19556, 19777, 19780, 19809, 
	19812, 20033, 20036, 20065, 20068, 121904, 121913, 122160, 
	122169, 122416, 122425, 122672, 122681, 122928, 122937, 123184, 
	123193, 123440, 123449, 7211, 7213, 19521, 19526, 19553, 
	19558, 19777, 19782, 19809, 19814, 20033, 20038, 20065, 
	20070, 121904, 121913, 122160, 122169, 122416, 122425, 122672, 
	122681, 122928, 122937, 123184, 123193, 123440, 123449, 7177, 
	7200, 7212, 7214, 20550, 20582, 20806, 20838, 21062, 
	21094, 21318, 21350, 21574, 21606, 21830, 21862, 22086, 
	22118, 123973, 124005, 124229, 124261, 124485, 124517, 124741, 
	124773, 124997, 125029, 125253, 125285, 125509, 125541, 125765, 
	125797, 126021, 126053, 126277, 126309, 126533, 126565, 126789, 
	126821, 127045, 127077, 127301, 127333, 127557, 127589, 7226, 
	7227, 20545, 20548, 20577, 20580, 20801, 20804, 20833, 
	20836, 21057, 21060, 21089, 21092, 21313, 21316, 21345, 
	21348, 21569, 21572, 21601, 21604, 21825, 21828, 21857, 
	21860, 22081, 22084, 22113, 22116, 123952, 123961, 124208, 
	124217, 124464, 124473, 124720, 124729, 124976, 124985, 125232, 
	125241, 125488, 125497, 125744, 125753, 126000, 126009, 126256, 
	126265, 126512, 126521, 126768, 126777, 127024, 127033, 127280, 
	127289, 127536, 127545, 7211, 7213, 20545, 20550, 20577, 
	20582, 20801, 20806, 20833, 20838, 21057, 21062, 21089, 
	21094, 21313, 21318, 21345, 21350, 21569, 21574, 21601, 
	21606, 21825, 21830, 21857, 21862, 22081, 22086, 22113, 
	22118, 123952, 123961, 124208, 124217, 124464, 124473, 124720, 
	124729, 124976, 124985, 125232, 125241, 125488, 125497, 125744, 
	125753, 126000, 126009, 126256, 126265, 126512, 126521, 126768, 
	126777, 127024, 127033, 127280, 127289, 127536, 127545, 7177, 
	7200, 7212, 7214, 22598, 22630, 22854, 22886, 23110, 
	23142, 128069, 128101, 128325, 128357, 128581, 128613, 128837, 
	128869, 129093, 129125, 129349, 129381, 129605, 129637, 7226, 
	7227, 22593, 22596, 22625, 22628, 22849, 22852, 22881, 
	22884, 23105, 23108, 23137, 23140, 128048, 128057, 128304, 
	128313, 128560, 128569, 128816, 128825, 129072, 129081, 129328, 
	129337, 129584, 129593, 7211, 7213, 22593, 22598, 22625, 
	22630, 22849, 22854, 22881, 22886, 23105, 23110, 23137, 
	23142, 128048, 128057, 128304, 128313, 128560, 128569, 128816, 
	128825, 129072, 129081, 129328, 129337, 129584, 129593, 7177, 
	7200, 7212, 7214, 23622, 23654, 23878, 23910, 24134, 
	24166, 24390, 24422, 24646, 24678, 24902, 24934, 25158, 
	25190, 130117, 130149, 130373, 130405, 130629, 130661, 130885, 
	130917, 131141, 131173, 131397, 131429, 131653, 131685, 131909, 
	131941, 132165, 132197, 132421, 132453, 132677, 132709, 132933, 
	132965, 133189, 133221, 133445, 133477, 133701, 133733, 7226, 
	7227, 23617, 23620, 23649, 23652, 23873, 23876, 23905, 
	23908, 24129, 24132, 24161, 24164, 24385, 24388, 24417, 
	24420, 24641, 24644, 24673, 24676, 24897, 24900, 24929, 
	24932, 25153, 25156, 25185, 25188, 130096, 130105, 130352, 
	130361, 130608, 130617, 130864, 130873, 131120, 131129, 131376, 
	131385, 131632, 131641, 131888, 131897, 132144, 132153, 132400, 
	132409, 132656, 132665, 132912, 132921, 133168, 133177, 133424, 
	133433, 133680, 133689, 7211, 7213, 23617, 23622, 23649, 
	23654, 23873, 23878, 23905, 23910, 24129, 24134, 24161, 
	24166, 24385, 24390, 24417, 24422, 24641, 24646, 24673, 
	24678, 24897, 24902, 24929, 24934, 25153, 25158, 25185, 
	25190, 130096, 130105, 130352, 130361, 130608, 130617, 130864, 
	130873, 131120, 131129, 131376, 131385, 131632, 131641, 131888, 
	131897, 132144, 132153, 132400, 132409, 132656, 132665, 132912, 
	132921, 133168, 133177, 133424, 133433, 133680, 133689, 7177, 
	7200, 7212, 7214, 25670, 25702, 25926, 25958, 26182, 
	26214, 26438, 26470, 26694, 26726, 26950, 26982, 27206, 
	27238, 134213, 134245, 134469, 134501, 134725, 134757, 134981, 
	135013, 135237, 135269, 135493, 135525, 135749, 135781, 136005, 
	136037, 136261, 136293, 136517, 136549, 136773, 136805, 137029, 
	137061, 137285, 137317, 137541, 137573, 137797, 137829, 7226, 
	7227, 25665, 25668, 25697, 25700, 25921, 25924, 25953, 
	25956, 26177, 26180, 26209, 26212, 26433, 26436, 26465, 
	26468, 26689, 26692, 26721, 26724, 26945, 26948, 26977, 
	26980, 27201, 27204, 27233, 27236, 134192, 134201, 134448, 
	134457, 134704, 134713, 134960, 134969, 135216, 135225, 135472, 
	135481, 135728, 135737, 135984, 135993, 136240, 136249, 136496, 
	136505, 136752, 136761, 137008, 137017, 137264, 137273, 137520, 
	137529, 137776, 137785, 7211, 7213, 25665, 25670, 25697, 
	25702, 25921, 25926, 25953, 25958, 26177, 26182, 26209, 
	26214, 26433, 26438, 26465, 26470, 26689, 26694, 26721, 
	26726, 26945, 26950, 26977, 26982, 27201, 27206, 27233, 
	27238, 134192, 134201, 134448, 134457, 134704, 134713, 134960, 
	134969, 135216, 135225, 135472, 135481, 135728, 135737, 135984, 
	135993, 136240, 136249, 136496, 136505, 136752, 136761, 137008, 
	137017, 137264, 137273, 137520, 137529, 137776, 137785, 7177, 
	7200, 7212, 7214, 14406, 14438, 14662, 14694, 14918, 
	14950, 15174, 15206, 15430, 15462, 15686, 15718, 15942, 
	15974, 16198, 16230, 16454, 16486, 16710, 16742, 16966, 
	16998, 17222, 17254, 17478, 17510, 17734, 17766, 17990, 
	18022, 138309, 138341, 138565, 138597, 138821, 138853, 139077, 
	139109, 139333, 139365, 139589, 139621, 139845, 139877, 140101, 
	140133, 140357, 140389, 140613, 140645, 140869, 140901, 141125, 
	141157, 141381, 141413, 141637, 141669, 141893, 141925, 142149, 
	142181, 142405, 142437, 142661, 142693, 142917, 142949, 143173, 
	143205, 143429, 143461, 143685, 143717, 143941, 143973, 144197, 
	144229, 144453, 144485, 144709, 144741, 144965, 144997, 145221, 
	145253, 145477, 145509, 145733, 145765, 145989, 146021, 7226, 
	7227, 14401, 14404, 14433, 14436, 14657, 14660, 14689, 
	14692, 14913, 14916, 14945, 14948, 15169, 15172, 15201, 
	15204, 15425, 15428, 15457, 15460, 15681, 15684, 15713, 
	15716, 15937, 15940, 15969, 15972, 16193, 16196, 16225, 
	16228, 16449, 16452, 16481, 16484, 16705, 16708, 16737, 
	16740, 16961, 16964, 16993, 16996, 17217, 17220, 17249, 
	17252, 17473, 17476, 17505, 17508, 17729, 17732, 17761, 
	17764, 17985, 17988, 18017, 18020, 138288, 138297, 138544, 
	138553, 138800, 138809, 139056, 139065, 139312, 139321, 139568, 
	139577, 139824, 139833, 140080, 140089, 140336, 140345, 140592, 
	140601, 140848, 140857, 141104, 141113, 141360, 141369, 141616, 
	141625, 141872, 141881, 142128, 142137, 142384, 142393, 142640, 
	142649, 142896, 142905, 143152, 143161, 143408, 143417, 143664, 
	143673, 143920, 143929, 144176, 144185, 144432, 144441, 144688, 
	144697, 144944, 144953, 145200, 145209, 145456, 145465, 145712, 
	145721, 145968, 145977, 7211, 7213, 14401, 14406, 14433, 
	14438, 14657, 14662, 14689, 14694, 14913, 14918, 14945, 
	14950, 15169, 15174, 15201, 15206, 15425, 15430, 15457, 
	15462, 15681, 15686, 15713, 15718, 15937, 15942, 15969, 
	15974, 16193, 16198, 16225, 16230, 16449, 16454, 16481, 
	16486, 16705, 16710, 16737, 16742, 16961, 16966, 16993, 
	16998, 17217, 17222, 17249, 17254, 17473, 17478, 17505, 
	17510, 17729, 17734, 17761, 17766, 17985, 17990, 18017, 
	18022, 138288, 138297, 138544, 138553, 138800, 138809, 139056, 
	139065, 139312, 139321, 139568, 139577, 139824, 139833, 140080, 
	140089, 140336, 140345, 140592, 140601, 140848, 140857, 141104, 
	141113, 141360, 141369, 141616, 141625, 141872, 141881, 142128, 
	142137, 142384, 142393, 142640, 142649, 142896, 142905, 143152, 
	143161, 143408, 143417, 143664, 143673, 143920, 143929, 144176, 
	144185, 144432, 144441, 144688, 144697, 144944, 144953, 145200, 
	145209, 145456, 145465, 145712, 145721, 145968, 145977, 7211, 
	7213, 7214, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 146480, 146489, 146490, 146491, 146736, 
	146745, 146746, 146747, 146992, 147001, 147002, 147003, 7211, 
	7213, 7214, 146441, 146464, 146697, 146720, 146953, 146976, 
	146480, 146489, 146736, 146745, 146992, 147001, 7214, 7237, 
	7269, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 146480, 146489, 146490, 146491, 146736, 146745, 
	146746, 146747, 146992, 147001, 147002, 147003, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 146490, 
	146491, 146746, 146747, 147002, 147003, 7211, 7213, 7214, 
	146441, 146464, 146697, 146720, 146953, 146976, 146480, 146489, 
	146736, 146745, 146992, 147001, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 146490, 146491, 146746, 
	146747, 147002, 147003, 7211, 7213, 7214, 146441, 146464, 
	146697, 146720, 146953, 146976, 146480, 146489, 146736, 146745, 
	146992, 147001, 7214, 7237, 7269, 146441, 146464, 146697, 
	146720, 146953, 146976, 146480, 146489, 146736, 146745, 146992, 
	147001, 7211, 7213, 7214, 146441, 146464, 146697, 146720, 
	146953, 146976, 146480, 146489, 146736, 146745, 146992, 147001, 
	7211, 7213, 7214, 146441, 146464, 146697, 146720, 146953, 
	146976, 146480, 146489, 146736, 146745, 146992, 147001, 7214, 
	7237, 7269, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 146480, 146489, 146490, 146491, 146736, 
	146745, 146746, 146747, 146992, 147001, 147002, 147003, 582, 
	614, 7214, 105541, 105573, 105797, 105829, 106053, 106085, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 577, 580, 609, 612, 146490, 146491, 146746, 
	146747, 147002, 147003, 147504, 147513, 147760, 147769, 148016, 
	148025, 148272, 148281, 148528, 148537, 148784, 148793, 149040, 
	149049, 582, 614, 7214, 105541, 105573, 105797, 105829, 
	106053, 106085, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 577, 580, 609, 612, 146490, 
	146491, 146746, 146747, 147002, 147003, 147504, 147513, 147760, 
	147769, 148016, 148025, 148272, 148281, 148528, 148537, 148784, 
	148793, 149040, 149049, 582, 614, 7214, 105541, 105573, 
	105797, 105829, 106053, 106085, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 577, 580, 609, 
	612, 146490, 146491, 146746, 146747, 147002, 147003, 147504, 
	147513, 147760, 147769, 148016, 148025, 148272, 148281, 148528, 
	148537, 148784, 148793, 149040, 149049, 582, 614, 7214, 
	105541, 105573, 105797, 105829, 106053, 106085, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 577, 
	580, 609, 612, 146490, 146491, 146746, 146747, 147002, 
	147003, 147504, 147513, 147760, 147769, 148016, 148025, 148272, 
	148281, 148528, 148537, 148784, 148793, 149040, 149049, 582, 
	614, 7214, 105541, 105573, 105797, 105829, 106053, 106085, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 577, 580, 609, 612, 146490, 146491, 146746, 
	146747, 147002, 147003, 147504, 147513, 147760, 147769, 148016, 
	148025, 148272, 148281, 148528, 148537, 148784, 148793, 149040, 
	149049, 582, 614, 7214, 105541, 105573, 105797, 105829, 
	106053, 106085, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 577, 580, 609, 612, 146490, 
	146491, 146746, 146747, 147002, 147003, 147504, 147513, 147760, 
	147769, 148016, 148025, 148272, 148281, 148528, 148537, 148784, 
	148793, 149040, 149049, 1094, 1126, 7214, 106565, 106597, 
	106821, 106853, 107077, 107109, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 1089, 1092, 1121, 
	1124, 146490, 146491, 146746, 146747, 147002, 147003, 149552, 
	149561, 149808, 149817, 150064, 150073, 150320, 150329, 150576, 
	150585, 150832, 150841, 151088, 151097, 1094, 1126, 7214, 
	106565, 106597, 106821, 106853, 107077, 107109, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 1089, 
	1092, 1121, 1124, 146490, 146491, 146746, 146747, 147002, 
	147003, 149552, 149561, 149808, 149817, 150064, 150073, 150320, 
	150329, 150576, 150585, 150832, 150841, 151088, 151097, 1094, 
	1126, 7214, 106565, 106597, 106821, 106853, 107077, 107109, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 1089, 1092, 1121, 1124, 146490, 146491, 146746, 
	146747, 147002, 147003, 149552, 149561, 149808, 149817, 150064, 
	150073, 150320, 150329, 150576, 150585, 150832, 150841, 151088, 
	151097, 1094, 1126, 7214, 106565, 106597, 106821, 106853, 
	107077, 107109, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 1089, 1092, 1121, 1124, 146490, 
	146491, 146746, 146747, 147002, 147003, 149552, 149561, 149808, 
	149817, 150064, 150073, 150320, 150329, 150576, 150585, 150832, 
	150841, 151088, 151097, 7214, 9286, 9318, 9542, 9574, 
	9798, 9830, 107589, 107621, 107845, 107877, 108101, 108133, 
	108357, 108389, 108613, 108645, 108869, 108901, 109125, 109157, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 9281, 9284, 9313, 9316, 9537, 9540, 9569, 
	9572, 9793, 9796, 9825, 9828, 146490, 146491, 146746, 
	146747, 147002, 147003, 151600, 151609, 151856, 151865, 152112, 
	152121, 152368, 152377, 152624, 152633, 152880, 152889, 153136, 
	153145, 153392, 153401, 153648, 153657, 153904, 153913, 154160, 
	154169, 154416, 154425, 154672, 154681, 154928, 154937, 155184, 
	155193, 7214, 9286, 9318, 9542, 9574, 9798, 9830, 
	107589, 107621, 107845, 107877, 108101, 108133, 108357, 108389, 
	108613, 108645, 108869, 108901, 109125, 109157, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 9281, 
	9284, 9313, 9316, 9537, 9540, 9569, 9572, 9793, 
	9796, 9825, 9828, 146490, 146491, 146746, 146747, 147002, 
	147003, 151600, 151609, 151856, 151865, 152112, 152121, 152368, 
	152377, 152624, 152633, 152880, 152889, 153136, 153145, 153392, 
	153401, 153648, 153657, 153904, 153913, 154160, 154169, 154416, 
	154425, 154672, 154681, 154928, 154937, 155184, 155193, 7214, 
	9286, 9318, 9542, 9574, 9798, 9830, 107589, 107621, 
	107845, 107877, 108101, 108133, 108357, 108389, 108613, 108645, 
	108869, 108901, 109125, 109157, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 9281, 9284, 9313, 
	9316, 9537, 9540, 9569, 9572, 9793, 9796, 9825, 
	9828, 146490, 146491, 146746, 146747, 147002, 147003, 151600, 
	151609, 151856, 151865, 152112, 152121, 152368, 152377, 152624, 
	152633, 152880, 152889, 153136, 153145, 153392, 153401, 153648, 
	153657, 153904, 153913, 154160, 154169, 154416, 154425, 154672, 
	154681, 154928, 154937, 155184, 155193, 7214, 9286, 9318, 
	9542, 9574, 9798, 9830, 107589, 107621, 107845, 107877, 
	108101, 108133, 108357, 108389, 108613, 108645, 108869, 108901, 
	109125, 109157, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 9281, 9284, 9313, 9316, 9537, 
	9540, 9569, 9572, 9793, 9796, 9825, 9828, 146490, 
	146491, 146746, 146747, 147002, 147003, 151600, 151609, 151856, 
	151865, 152112, 152121, 152368, 152377, 152624, 152633, 152880, 
	152889, 153136, 153145, 153392, 153401, 153648, 153657, 153904, 
	153913, 154160, 154169, 154416, 154425, 154672, 154681, 154928, 
	154937, 155184, 155193, 1606, 1638, 7214, 109637, 109669, 
	109893, 109925, 110149, 110181, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 1601, 1604, 1633, 
	1636, 146490, 146491, 146746, 146747, 147002, 147003, 155696, 
	155705, 155952, 155961, 156208, 156217, 156464, 156473, 156720, 
	156729, 156976, 156985, 157232, 157241, 1606, 1638, 7214, 
	109637, 109669, 109893, 109925, 110149, 110181, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 1601, 
	1604, 1633, 1636, 146490, 146491, 146746, 146747, 147002, 
	147003, 155696, 155705, 155952, 155961, 156208, 156217, 156464, 
	156473, 156720, 156729, 156976, 156985, 157232, 157241, 7214, 
	12358, 12390, 12614, 12646, 12870, 12902, 110661, 110693, 
	110917, 110949, 111173, 111205, 111429, 111461, 111685, 111717, 
	111941, 111973, 112197, 112229, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 12353, 12356, 12385, 
	12388, 12609, 12612, 12641, 12644, 12865, 12868, 12897, 
	12900, 146490, 146491, 146746, 146747, 147002, 147003, 157744, 
	157753, 158000, 158009, 158256, 158265, 158512, 158521, 158768, 
	158777, 159024, 159033, 159280, 159289, 159536, 159545, 159792, 
	159801, 160048, 160057, 160304, 160313, 160560, 160569, 160816, 
	160825, 161072, 161081, 161328, 161337, 7214, 12358, 12390, 
	12614, 12646, 12870, 12902, 110661, 110693, 110917, 110949, 
	111173, 111205, 111429, 111461, 111685, 111717, 111941, 111973, 
	112197, 112229, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 12353, 12356, 12385, 12388, 12609, 
	12612, 12641, 12644, 12865, 12868, 12897, 12900, 146490, 
	146491, 146746, 146747, 147002, 147003, 157744, 157753, 158000, 
	158009, 158256, 158265, 158512, 158521, 158768, 158777, 159024, 
	159033, 159280, 159289, 159536, 159545, 159792, 159801, 160048, 
	160057, 160304, 160313, 160560, 160569, 160816, 160825, 161072, 
	161081, 161328, 161337, 7214, 13382, 13414, 13638, 13670, 
	13894, 13926, 112709, 112741, 112965, 112997, 113221, 113253, 
	113477, 113509, 113733, 113765, 113989, 114021, 114245, 114277, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 13377, 13380, 13409, 13412, 13633, 13636, 13665, 
	13668, 13889, 13892, 13921, 13924, 146490, 146491, 146746, 
	146747, 147002, 147003, 161840, 161849, 162096, 162105, 162352, 
	162361, 162608, 162617, 162864, 162873, 163120, 163129, 163376, 
	163385, 163632, 163641, 163888, 163897, 164144, 164153, 164400, 
	164409, 164656, 164665, 164912, 164921, 165168, 165177, 165424, 
	165433, 7214, 13382, 13414, 13638, 13670, 13894, 13926, 
	112709, 112741, 112965, 112997, 113221, 113253, 113477, 113509, 
	113733, 113765, 113989, 114021, 114245, 114277, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 13377, 
	13380, 13409, 13412, 13633, 13636, 13665, 13668, 13889, 
	13892, 13921, 13924, 146490, 146491, 146746, 146747, 147002, 
	147003, 161840, 161849, 162096, 162105, 162352, 162361, 162608, 
	162617, 162864, 162873, 163120, 163129, 163376, 163385, 163632, 
	163641, 163888, 163897, 164144, 164153, 164400, 164409, 164656, 
	164665, 164912, 164921, 165168, 165177, 165424, 165433, 7214, 
	10310, 10342, 10566, 10598, 10822, 10854, 11078, 11110, 
	11334, 11366, 11590, 11622, 11846, 11878, 114757, 114789, 
	115013, 115045, 115269, 115301, 115525, 115557, 115781, 115813, 
	116037, 116069, 116293, 116325, 116549, 116581, 116805, 116837, 
	117061, 117093, 117317, 117349, 117573, 117605, 117829, 117861, 
	118085, 118117, 118341, 118373, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 10305, 10308, 10337, 
	10340, 10561, 10564, 10593, 10596, 10817, 10820, 10849, 
	10852, 11073, 11076, 11105, 11108, 11329, 11332, 11361, 
	11364, 11585, 11588, 11617, 11620, 11841, 11844, 11873, 
	11876, 146490, 146491, 146746, 146747, 147002, 147003, 165936, 
	165945, 166192, 166201, 166448, 166457, 166704, 166713, 166960, 
	166969, 167216, 167225, 167472, 167481, 167728, 167737, 167984, 
	167993, 168240, 168249, 168496, 168505, 168752, 168761, 169008, 
	169017, 169264, 169273, 169520, 169529, 169776, 169785, 170032, 
	170041, 170288, 170297, 170544, 170553, 170800, 170809, 171056, 
	171065, 171312, 171321, 171568, 171577, 171824, 171833, 172080, 
	172089, 172336, 172345, 172592, 172601, 172848, 172857, 173104, 
	173113, 173360, 173369, 173616, 173625, 7214, 10310, 10342, 
	10566, 10598, 10822, 10854, 11078, 11110, 11334, 11366, 
	11590, 11622, 11846, 11878, 114757, 114789, 115013, 115045, 
	115269, 115301, 115525, 115557, 115781, 115813, 116037, 116069, 
	116293, 116325, 116549, 116581, 116805, 116837, 117061, 117093, 
	117317, 117349, 117573, 117605, 117829, 117861, 118085, 118117, 
	118341, 118373, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 10305, 10308, 10337, 10340, 10561, 
	10564, 10593, 10596, 10817, 10820, 10849, 10852, 11073, 
	11076, 11105, 11108, 11329, 11332, 11361, 11364, 11585, 
	11588, 11617, 11620, 11841, 11844, 11873, 11876, 146490, 
	146491, 146746, 146747, 147002, 147003, 165936, 165945, 166192, 
	166201, 166448, 166457, 166704, 166713, 166960, 166969, 167216, 
	167225, 167472, 167481, 167728, 167737, 167984, 167993, 168240, 
	168249, 168496, 168505, 168752, 168761, 169008, 169017, 169264, 
	169273, 169520, 169529, 169776, 169785, 170032, 170041, 170288, 
	170297, 170544, 170553, 170800, 170809, 171056, 171065, 171312, 
	171321, 171568, 171577, 171824, 171833, 172080, 172089, 172336, 
	172345, 172592, 172601, 172848, 172857, 173104, 173113, 173360, 
	173369, 173616, 173625, 2118, 2150, 7214, 118853, 118885, 
	119109, 119141, 119365, 119397, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 2113, 2116, 2145, 
	2148, 146490, 146491, 146746, 146747, 147002, 147003, 174128, 
	174137, 174384, 174393, 174640, 174649, 174896, 174905, 175152, 
	175161, 175408, 175417, 175664, 175673, 7214, 18502, 18534, 
	18758, 18790, 19014, 19046, 119877, 119909, 120133, 120165, 
	120389, 120421, 120645, 120677, 120901, 120933, 121157, 121189, 
	121413, 121445, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 18497, 18500, 18529, 18532, 18753, 
	18756, 18785, 18788, 19009, 19012, 19041, 19044, 146490, 
	146491, 146746, 146747, 147002, 147003, 176176, 176185, 176432, 
	176441, 176688, 176697, 176944, 176953, 177200, 177209, 177456, 
	177465, 177712, 177721, 177968, 177977, 178224, 178233, 178480, 
	178489, 178736, 178745, 178992, 179001, 179248, 179257, 179504, 
	179513, 179760, 179769, 7214, 19526, 19558, 19782, 19814, 
	20038, 20070, 121925, 121957, 122181, 122213, 122437, 122469, 
	122693, 122725, 122949, 122981, 123205, 123237, 123461, 123493, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 19521, 19524, 19553, 19556, 19777, 19780, 19809, 
	19812, 20033, 20036, 20065, 20068, 146490, 146491, 146746, 
	146747, 147002, 147003, 180272, 180281, 180528, 180537, 180784, 
	180793, 181040, 181049, 181296, 181305, 181552, 181561, 181808, 
	181817, 182064, 182073, 182320, 182329, 182576, 182585, 182832, 
	182841, 183088, 183097, 183344, 183353, 183600, 183609, 183856, 
	183865, 7214, 20550, 20582, 20806, 20838, 21062, 21094, 
	21318, 21350, 21574, 21606, 21830, 21862, 22086, 22118, 
	123973, 124005, 124229, 124261, 124485, 124517, 124741, 124773, 
	124997, 125029, 125253, 125285, 125509, 125541, 125765, 125797, 
	126021, 126053, 126277, 126309, 126533, 126565, 126789, 126821, 
	127045, 127077, 127301, 127333, 127557, 127589, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 20545, 
	20548, 20577, 20580, 20801, 20804, 20833, 20836, 21057, 
	21060, 21089, 21092, 21313, 21316, 21345, 21348, 21569, 
	21572, 21601, 21604, 21825, 21828, 21857, 21860, 22081, 
	22084, 22113, 22116, 146490, 146491, 146746, 146747, 147002, 
	147003, 184368, 184377, 184624, 184633, 184880, 184889, 185136, 
	185145, 185392, 185401, 185648, 185657, 185904, 185913, 186160, 
	186169, 186416, 186425, 186672, 186681, 186928, 186937, 187184, 
	187193, 187440, 187449, 187696, 187705, 187952, 187961, 188208, 
	188217, 188464, 188473, 188720, 188729, 188976, 188985, 189232, 
	189241, 189488, 189497, 189744, 189753, 190000, 190009, 190256, 
	190265, 190512, 190521, 190768, 190777, 191024, 191033, 191280, 
	191289, 191536, 191545, 191792, 191801, 192048, 192057, 7214, 
	22598, 22630, 22854, 22886, 23110, 23142, 128069, 128101, 
	128325, 128357, 128581, 128613, 128837, 128869, 129093, 129125, 
	129349, 129381, 129605, 129637, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 22593, 22596, 22625, 
	22628, 22849, 22852, 22881, 22884, 23105, 23108, 23137, 
	23140, 146490, 146491, 146746, 146747, 147002, 147003, 192560, 
	192569, 192816, 192825, 193072, 193081, 193328, 193337, 193584, 
	193593, 193840, 193849, 194096, 194105, 194352, 194361, 194608, 
	194617, 194864, 194873, 195120, 195129, 195376, 195385, 195632, 
	195641, 195888, 195897, 196144, 196153, 7214, 23622, 23654, 
	23878, 23910, 24134, 24166, 24390, 24422, 24646, 24678, 
	24902, 24934, 25158, 25190, 130117, 130149, 130373, 130405, 
	130629, 130661, 130885, 130917, 131141, 131173, 131397, 131429, 
	131653, 131685, 131909, 131941, 132165, 132197, 132421, 132453, 
	132677, 132709, 132933, 132965, 133189, 133221, 133445, 133477, 
	133701, 133733, 146441, 146464, 146476, 146697, 146720, 146732, 
	146953, 146976, 146988, 23617, 23620, 23649, 23652, 23873, 
	23876, 23905, 23908, 24129, 24132, 24161, 24164, 24385, 
	24388, 24417, 24420, 24641, 24644, 24673, 24676, 24897, 
	24900, 24929, 24932, 25153, 25156, 25185, 25188, 146490, 
	146491, 146746, 146747, 147002, 147003, 196656, 196665, 196912, 
	196921, 197168, 197177, 197424, 197433, 197680, 197689, 197936, 
	197945, 198192, 198201, 198448, 198457, 198704, 198713, 198960, 
	198969, 199216, 199225, 199472, 199481, 199728, 199737, 199984, 
	199993, 200240, 200249, 200496, 200505, 200752, 200761, 201008, 
	201017, 201264, 201273, 201520, 201529, 201776, 201785, 202032, 
	202041, 202288, 202297, 202544, 202553, 202800, 202809, 203056, 
	203065, 203312, 203321, 203568, 203577, 203824, 203833, 204080, 
	204089, 204336, 204345, 7214, 25670, 25702, 25926, 25958, 
	26182, 26214, 26438, 26470, 26694, 26726, 26950, 26982, 
	27206, 27238, 134213, 134245, 134469, 134501, 134725, 134757, 
	134981, 135013, 135237, 135269, 135493, 135525, 135749, 135781, 
	136005, 136037, 136261, 136293, 136517, 136549, 136773, 136805, 
	137029, 137061, 137285, 137317, 137541, 137573, 137797, 137829, 
	146441, 146464, 146476, 146697, 146720, 146732, 146953, 146976, 
	146988, 25665, 25668, 25697, 25700, 25921, 25924, 25953, 
	25956, 26177, 26180, 26209, 26212, 26433, 26436, 26465, 
	26468, 26689, 26692, 26721, 26724, 26945, 26948, 26977, 
	26980, 27201, 27204, 27233, 27236, 146490, 146491, 146746, 
	146747, 147002, 147003, 204848, 204857, 205104, 205113, 205360, 
	205369, 205616, 205625, 205872, 205881, 206128, 206137, 206384, 
	206393, 206640, 206649, 206896, 206905, 207152, 207161, 207408, 
	207417, 207664, 207673, 207920, 207929, 208176, 208185, 208432, 
	208441, 208688, 208697, 208944, 208953, 209200, 209209, 209456, 
	209465, 209712, 209721, 209968, 209977, 210224, 210233, 210480, 
	210489, 210736, 210745, 210992, 211001, 211248, 211257, 211504, 
	211513, 211760, 211769, 212016, 212025, 212272, 212281, 212528, 
	212537, 7214, 14406, 14438, 14662, 14694, 14918, 14950, 
	15174, 15206, 15430, 15462, 15686, 15718, 15942, 15974, 
	16198, 16230, 16454, 16486, 16710, 16742, 16966, 16998, 
	17222, 17254, 17478, 17510, 17734, 17766, 17990, 18022, 
	138309, 138341, 138565, 138597, 138821, 138853, 139077, 139109, 
	139333, 139365, 139589, 139621, 139845, 139877, 140101, 140133, 
	140357, 140389, 140613, 140645, 140869, 140901, 141125, 141157, 
	141381, 141413, 141637, 141669, 141893, 141925, 142149, 142181, 
	142405, 142437, 142661, 142693, 142917, 142949, 143173, 143205, 
	143429, 143461, 143685, 143717, 143941, 143973, 144197, 144229, 
	144453, 144485, 144709, 144741, 144965, 144997, 145221, 145253, 
	145477, 145509, 145733, 145765, 145989, 146021, 146441, 146464, 
	146476, 146697, 146720, 146732, 146953, 146976, 146988, 14401, 
	14404, 14433, 14436, 14657, 14660, 14689, 14692, 14913, 
	14916, 14945, 14948, 15169, 15172, 15201, 15204, 15425, 
	15428, 15457, 15460, 15681, 15684, 15713, 15716, 15937, 
	15940, 15969, 15972, 16193, 16196, 16225, 16228, 16449, 
	16452, 16481, 16484, 16705, 16708, 16737, 16740, 16961, 
	16964, 16993, 16996, 17217, 17220, 17249, 17252, 17473, 
	17476, 17505, 17508, 17729, 17732, 17761, 17764, 17985, 
	17988, 18017, 18020, 89136, 89145, 89392, 89401, 89648, 
	89657, 89904, 89913, 90160, 90169, 90416, 90425, 90672, 
	90681, 90928, 90937, 91184, 91193, 91440, 91449, 91696, 
	91705, 91952, 91961, 92208, 92217, 92464, 92473, 92720, 
	92729, 92976, 92985, 93232, 93241, 93488, 93497, 93744, 
	93753, 94000, 94009, 94256, 94265, 94512, 94521, 94768, 
	94777, 95024, 95033, 95280, 95289, 95536, 95545, 95792, 
	95801, 96048, 96057, 96304, 96313, 96560, 96569, 96816, 
	96825, 97072, 97081, 97328, 97337, 97584, 97593, 97840, 
	97849, 98096, 98105, 98352, 98361, 98608, 98617, 98864, 
	98873, 99120, 99129, 99376, 99385, 99632, 99641, 99888, 
	99897, 100144, 100153, 100400, 100409, 100656, 100665, 100912, 
	100921, 101168, 101177, 101424, 101433, 101680, 101689, 101936, 
	101945, 102192, 102201, 102448, 102457, 102704, 102713, 102960, 
	102969, 103216, 103225, 103472, 103481, 103728, 103737, 103984, 
	103993, 104240, 104249, 104496, 104505, 104752, 104761, 105008, 
	105017, 146490, 146491, 146746, 146747, 147002, 147003, 560, 
	569, 577, 582, 609, 614, 560, 569, 577, 
	582, 609, 614, 560, 569, 577, 582, 609, 
	614, 560, 569, 577, 582, 609, 614, 560, 
	569, 577, 582, 609, 614, 560, 569, 577, 
	582, 609, 614, 560, 569, 577, 582, 609, 
	614, 1072, 1081, 1089, 1094, 1121, 1126, 1072, 
	1081, 1089, 1094, 1121, 1126, 1072, 1081, 1089, 
	1094, 1121, 1126, 1072, 1081, 1089, 1094, 1121, 
	1126, 1072, 1081, 1089, 1094, 1121, 1126, 9264, 
	9273, 9281, 9286, 9313, 9318, 9520, 9529, 9537, 
	9542, 9569, 9574, 9776, 9785, 9793, 9798, 9825, 
	9830, 9264, 9273, 9281, 9286, 9313, 9318, 9520, 
	9529, 9537, 9542, 9569, 9574, 9776, 9785, 9793, 
	9798, 9825, 9830, 9264, 9273, 9281, 9286, 9313, 
	9318, 9520, 9529, 9537, 9542, 9569, 9574, 9776, 
	9785, 9793, 9798, 9825, 9830, 9264, 9273, 9281, 
	9286, 9313, 9318, 9520, 9529, 9537, 9542, 9569, 
	9574, 9776, 9785, 9793, 9798, 9825, 9830, 9264, 
	9273, 9281, 9286, 9313, 9318, 9520, 9529, 9537, 
	9542, 9569, 9574, 9776, 9785, 9793, 9798, 9825, 
	9830, 1584, 1593, 1601, 1606, 1633, 1638, 1584, 
	1593, 1601, 1606, 1633, 1638, 1584, 1593, 1601, 
	1606, 1633, 1638, 12336, 12345, 12353, 12358, 12385, 
	12390, 12592, 12601, 12609, 12614, 12641, 12646, 12848, 
	12857, 12865, 12870, 12897, 12902, 12336, 12345, 12353, 
	12358, 12385, 12390, 12592, 12601, 12609, 12614, 12641, 
	12646, 12848, 12857, 12865, 12870, 12897, 12902, 12336, 
	12345, 12353, 12358, 12385, 12390, 12592, 12601, 12609, 
	12614, 12641, 12646, 12848, 12857, 12865, 12870, 12897, 
	12902, 13360, 13369, 13377, 13382, 13409, 13414, 13616, 
	13625, 13633, 13638, 13665, 13670, 13872, 13881, 13889, 
	13894, 13921, 13926, 13360, 13369, 13377, 13382, 13409, 
	13414, 13616, 13625, 13633, 13638, 13665, 13670, 13872, 
	13881, 13889, 13894, 13921, 13926, 13360, 13369, 13377, 
	13382, 13409, 13414, 13616, 13625, 13633, 13638, 13665, 
	13670, 13872, 13881, 13889, 13894, 13921, 13926, 10288, 
	10297, 10305, 10310, 10337, 10342, 10544, 10553, 10561, 
	10566, 10593, 10598, 10800, 10809, 10817, 10822, 10849, 
	10854, 11056, 11065, 11073, 11078, 11105, 11110, 11312, 
	11321, 11329, 11334, 11361, 11366, 11568, 11577, 11585, 
	11590, 11617, 11622, 11824, 11833, 11841, 11846, 11873, 
	11878, 10288, 10297, 10305, 10310, 10337, 10342, 10544, 
	10553, 10561, 10566, 10593, 10598, 10800, 10809, 10817, 
	10822, 10849, 10854, 11056, 11065, 11073, 11078, 11105, 
	11110, 11312, 11321, 11329, 11334, 11361, 11366, 11568, 
	11577, 11585, 11590, 11617, 11622, 11824, 11833, 11841, 
	11846, 11873, 11878, 10288, 10297, 10305, 10310, 10337, 
	10342, 10544, 10553, 10561, 10566, 10593, 10598, 10800, 
	10809, 10817, 10822, 10849, 10854, 11056, 11065, 11073, 
	11078, 11105, 11110, 11312, 11321, 11329, 11334, 11361, 
	11366, 11568, 11577, 11585, 11590, 11617, 11622, 11824, 
	11833, 11841, 11846, 11873, 11878, 2096, 2105, 2113, 
	2118, 2145, 2150, 2096, 2105, 2113, 2118, 2145, 
	2150, 18480, 18489, 18497, 18502, 18529, 18534, 18736, 
	18745, 18753, 18758, 18785, 18790, 18992, 19001, 19009, 
	19014, 19041, 19046, 18480, 18489, 18497, 18502, 18529, 
	18534, 18736, 18745, 18753, 18758, 18785, 18790, 18992, 
	19001, 19009, 19014, 19041, 19046, 19504, 19513, 19521, 
	19526, 19553, 19558, 19760, 19769, 19777, 19782, 19809, 
	19814, 20016, 20025, 20033, 20038, 20065, 20070, 19504, 
	19513, 19521, 19526, 19553, 19558, 19760, 19769, 19777, 
	19782, 19809, 19814, 20016, 20025, 20033, 20038, 20065, 
	20070, 20528, 20537, 20545, 20550, 20577, 20582, 20784, 
	20793, 20801, 20806, 20833, 20838, 21040, 21049, 21057, 
	21062, 21089, 21094, 21296, 21305, 21313, 21318, 21345, 
	21350, 21552, 21561, 21569, 21574, 21601, 21606, 21808, 
	21817, 21825, 21830, 21857, 21862, 22064, 22073, 22081, 
	22086, 22113, 22118, 20528, 20537, 20545, 20550, 20577, 
	20582, 20784, 20793, 20801, 20806, 20833, 20838, 21040, 
	21049, 21057, 21062, 21089, 21094, 21296, 21305, 21313, 
	21318, 21345, 21350, 21552, 21561, 21569, 21574, 21601, 
	21606, 21808, 21817, 21825, 21830, 21857, 21862, 22064, 
	22073, 22081, 22086, 22113, 22118, 22576, 22585, 22593, 
	22598, 22625, 22630, 22832, 22841, 22849, 22854, 22881, 
	22886, 23088, 23097, 23105, 23110, 23137, 23142, 22576, 
	22585, 22593, 22598, 22625, 22630, 22832, 22841, 22849, 
	22854, 22881, 22886, 23088, 23097, 23105, 23110, 23137, 
	23142, 23600, 23609, 23617, 23622, 23649, 23654, 23856, 
	23865, 23873, 23878, 23905, 23910, 24112, 24121, 24129, 
	24134, 24161, 24166, 24368, 24377, 24385, 24390, 24417, 
	24422, 24624, 24633, 24641, 24646, 24673, 24678, 24880, 
	24889, 24897, 24902, 24929, 24934, 25136, 25145, 25153, 
	25158, 25185, 25190, 23600, 23609, 23617, 23622, 23649, 
	23654, 23856, 23865, 23873, 23878, 23905, 23910, 24112, 
	24121, 24129, 24134, 24161, 24166, 24368, 24377, 24385, 
	24390, 24417, 24422, 24624, 24633, 24641, 24646, 24673, 
	24678, 24880, 24889, 24897, 24902, 24929, 24934, 25136, 
	25145, 25153, 25158, 25185, 25190, 25648, 25657, 25665, 
	25670, 25697, 25702, 25904, 25913, 25921, 25926, 25953, 
	25958, 26160, 26169, 26177, 26182, 26209, 26214, 26416, 
	26425, 26433, 26438, 26465, 26470, 26672, 26681, 26689, 
	26694, 26721, 26726, 26928, 26937, 26945, 26950, 26977, 
	26982, 27184, 27193, 27201, 27206, 27233, 27238, 25648, 
	25657, 25665, 25670, 25697, 25702, 25904, 25913, 25921, 
	25926, 25953, 25958, 26160, 26169, 26177, 26182, 26209, 
	26214, 26416, 26425, 26433, 26438, 26465, 26470, 26672, 
	26681, 26689, 26694, 26721, 26726, 26928, 26937, 26945, 
	26950, 26977, 26982, 27184, 27193, 27201, 27206, 27233, 
	27238, 14384, 14393, 14401, 14406, 14433, 14438, 14640, 
	14649, 14657, 14662, 14689, 14694, 14896, 14905, 14913, 
	14918, 14945, 14950, 15152, 15161, 15169, 15174, 15201, 
	15206, 15408, 15417, 15425, 15430, 15457, 15462, 15664, 
	15673, 15681, 15686, 15713, 15718, 15920, 15929, 15937, 
	15942, 15969, 15974, 16176, 16185, 16193, 16198, 16225, 
	16230, 16432, 16441, 16449, 16454, 16481, 16486, 16688, 
	16697, 16705, 16710, 16737, 16742, 16944, 16953, 16961, 
	16966, 16993, 16998, 17200, 17209, 17217, 17222, 17249, 
	17254, 17456, 17465, 17473, 17478, 17505, 17510, 17712, 
	17721, 17729, 17734, 17761, 17766, 17968, 17977, 17985, 
	17990, 18017, 18022, 14384, 14393, 14401, 14406, 14433, 
	14438, 14640, 14649, 14657, 14662, 14689, 14694, 14896, 
	14905, 14913, 14918, 14945, 14950, 15152, 15161, 15169, 
	15174, 15201, 15206, 15408, 15417, 15425, 15430, 15457, 
	15462, 15664, 15673, 15681, 15686, 15713, 15718, 15920, 
	15929, 15937, 15942, 15969, 15974, 16176, 16185, 16193, 
	16198, 16225, 16230, 16432, 16441, 16449, 16454, 16481, 
	16486, 16688, 16697, 16705, 16710, 16737, 16742, 16944, 
	16953, 16961, 16966, 16993, 16998, 17200, 17209, 17217, 
	17222, 17249, 17254, 17456, 17465, 17473, 17478, 17505, 
	17510, 17712, 17721, 17729, 17734, 17761, 17766, 17968, 
	17977, 17985, 17990, 18017, 18022, 10, 13, 44, 
	71, 103, 6955, 7211, 7213, 7214, 27474, 27506, 
	27730, 27762, 27986, 28018, 28242, 28274, 28488, 28520, 
	28744, 28776, 29000, 29032, 29256, 29288, 34639, 34671, 
	34895, 34927, 35151, 35183, 35407, 35439, 35663, 35695, 
	35919, 35951, 36175, 36207, 36431, 36463, 36687, 36719, 
	36943, 36975, 37199, 37231, 37455, 37487, 37711, 37743, 
	37967, 37999, 38223, 38255, 38479, 38511, 212783, 213039, 
	213295, 213551, 213795, 214051, 214307, 214563, 214819, 215075, 
	215331, 215587, 215843, 216099, 216355, 216611, 216867, 217123, 
	217379, 217635, 217891, 218147, 218403, 218659, 218915, 219171, 
	219427, 219683, 219939, 220195, 220451, 220707, 220963, 221219, 
	221475, 221731, -128, 34, 36, 42, 58, 64, 
	73, 78, 80, 81, 83, 96, 105, 110, 
	112, 113, 115, 127, 6957, 6958, 14145, 14150, 
	14177, 14182, 14401, 14406, 14433, 14438, 14657, 14662, 
	14689, 14694, 14913, 14918, 14945, 14950, 15169, 15174, 
	15201, 15206, 15425, 15430, 15457, 15462, 15681, 15686, 
	15713, 15718, 15937, 15942, 15969, 15974, 16193, 16198, 
	16225, 16230, 16449, 16454, 16481, 16486, 16705, 16710, 
	16737, 16742, 16961, 16966, 16993, 16998, 17217, 17222, 
	17249, 17254, 17473, 17478, 17505, 17510, 17729, 17734, 
	17761, 17766, 17985, 17990, 18017, 18022, 88880, 88889, 
	89136, 89145, 89392, 89401, 89648, 89657, 89904, 89913, 
	90160, 90169, 90416, 90425, 90672, 90681, 90928, 90937, 
	91184, 91193, 91440, 91449, 91696, 91705, 91952, 91961, 
	92208, 92217, 92464, 92473, 92720, 92729, 92976, 92985, 
	93232, 93241, 93488, 93497, 93744, 93753, 94000, 94009, 
	94256, 94265, 94512, 94521, 94768, 94777, 95024, 95033, 
	95280, 95289, 95536, 95545, 95792, 95801, 96048, 96057, 
	96304, 96313, 96560, 96569, 96816, 96825, 97072, 97081, 
	97328, 97337, 97584, 97593, 97840, 97849, 98096, 98105, 
	98352, 98361, 98608, 98617, 98864, 98873, 99120, 99129, 
	99376, 99385, 99632, 99641, 99888, 99897, 100144, 100153, 
	100400, 100409, 100656, 100665, 100912, 100921, 101168, 101177, 
	101424, 101433, 101680, 101689, 101936, 101945, 102192, 102201, 
	102448, 102457, 102704, 102713, 102960, 102969, 103216, 103225, 
	103472, 103481, 103728, 103737, 103984, 103993, 104240, 104249, 
	104496, 104505, 104752, 104761, 105008, 105017, 10, 7214, 
	7216, 7225, 7177, 7200, 7237, 7269, 7216, 7225, 
	7237, 7269, 7216, 7225, 7216, 7225, 7214, 7237, 
	7269, 7216, 7225, 7177, 7200, 7216, 7225, 7177, 
	7200, 7214, 7237, 7269, 7216, 7225, 7177, 7200, 
	7212, 7237, 7269, 7216, 7225, 7226, 7227, 7237, 
	7269, 7216, 7225, 7216, 7225, 7214, 7237, 7269, 
	7216, 7225, 7177, 7200, 7212, 7216, 7225, 7226, 
	7227, 7177, 7200, 7212, 7214, 7237, 7269, 7216, 
	7225, 7226, 7227, 7216, 7225, 560, 569, 577, 
	582, 609, 614, 1072, 1081, 1089, 1094, 1121, 
	1126, 9264, 9273, 9281, 9286, 9313, 9318, 9520, 
	9529, 9537, 9542, 9569, 9574, 9776, 9785, 9793, 
	9798, 9825, 9830, 560, 569, 577, 582, 609, 
	614, 1584, 1593, 1601, 1606, 1633, 1638, 12336, 
	12345, 12353, 12358, 12385, 12390, 12592, 12601, 12609, 
	12614, 12641, 12646, 12848, 12857, 12865, 12870, 12897, 
	12902, 560, 569, 577, 582, 609, 614, 13360, 
	13369, 13377, 13382, 13409, 13414, 13616, 13625, 13633, 
	13638, 13665, 13670, 13872, 13881, 13889, 13894, 13921, 
	13926, 1072, 1081, 1089, 1094, 1121, 1126, 10288, 
	10297, 10305, 10310, 10337, 10342, 10544, 10553, 10561, 
	10566, 10593, 10598, 10800, 10809, 10817, 10822, 10849, 
	10854, 11056, 11065, 11073, 11078, 11105, 11110, 11312, 
	11321, 11329, 11334, 11361, 11366, 11568, 11577, 11585, 
	11590, 11617, 11622, 11824, 11833, 11841, 11846, 11873, 
	11878, 9264, 9273, 9281, 9286, 9313, 9318, 9520, 
	9529, 9537, 9542, 9569, 9574, 9776, 9785, 9793, 
	9798, 9825, 9830, 2096, 2105, 2113, 2118, 2145, 
	2150, 18480, 18489, 18497, 18502, 18529, 18534, 18736, 
	18745, 18753, 18758, 18785, 18790, 18992, 19001, 19009, 
	19014, 19041, 19046, 560, 569, 577, 582, 609, 
	614, 19504, 19513, 19521, 19526, 19553, 19558, 19760, 
	19769, 19777, 19782, 19809, 19814, 20016, 20025, 20033, 
	20038, 20065, 20070, 1072, 1081, 1089, 1094, 1121, 
	1126, 20528, 20537, 20545, 20550, 20577, 20582, 20784, 
	20793, 20801, 20806, 20833, 20838, 21040, 21049, 21057, 
	21062, 21089, 21094, 21296, 21305, 21313, 21318, 21345, 
	21350, 21552, 21561, 21569, 21574, 21601, 21606, 21808, 
	21817, 21825, 21830, 21857, 21862, 22064, 22073, 22081, 
	22086, 22113, 22118, 9264, 9273, 9281, 9286, 9313, 
	9318, 9520, 9529, 9537, 9542, 9569, 9574, 9776, 
	9785, 9793, 9798, 9825, 9830, 22576, 22585, 22593, 
	22598, 22625, 22630, 22832, 22841, 22849, 22854, 22881, 
	22886, 23088, 23097, 23105, 23110, 23137, 23142, 1584, 
	1593, 1601, 1606, 1633, 1638, 23600, 23609, 23617, 
	23622, 23649, 23654, 23856, 23865, 23873, 23878, 23905, 
	23910, 24112, 24121, 24129, 24134, 24161, 24166, 24368, 
	24377, 24385, 24390, 24417, 24422, 24624, 24633, 24641, 
	24646, 24673, 24678, 24880, 24889, 24897, 24902, 24929, 
	24934, 25136, 25145, 25153, 25158, 25185, 25190, 12336, 
	12345, 12353, 12358, 12385, 12390, 12592, 12601, 12609, 
	12614, 12641, 12646, 12848, 12857, 12865, 12870, 12897, 
	12902, 25648, 25657, 25665, 25670, 25697, 25702, 25904, 
	25913, 25921, 25926, 25953, 25958, 26160, 26169, 26177, 
	26182, 26209, 26214, 26416, 26425, 26433, 26438, 26465, 
	26470, 26672, 26681, 26689, 26694, 26721, 26726, 26928, 
	26937, 26945, 26950, 26977, 26982, 27184, 27193, 27201, 
	27206, 27233, 27238, 13360, 13369, 13377, 13382, 13409, 
	13414, 13616, 13625, 13633, 13638, 13665, 13670, 13872, 
	13881, 13889, 13894, 13921, 13926, 14384, 14393, 14401, 
	14406, 14433, 14438, 14640, 14649, 14657, 14662, 14689, 
	14694, 14896, 14905, 14913, 14918, 14945, 14950, 15152, 
	15161, 15169, 15174, 15201, 15206, 15408, 15417, 15425, 
	15430, 15457, 15462, 15664, 15673, 15681, 15686, 15713, 
	15718, 15920, 15929, 15937, 15942, 15969, 15974, 16176, 
	16185, 16193, 16198, 16225, 16230, 16432, 16441, 16449, 
	16454, 16481, 16486, 16688, 16697, 16705, 16710, 16737, 
	16742, 16944, 16953, 16961, 16966, 16993, 16998, 17200, 
	17209, 17217, 17222, 17249, 17254, 17456, 17465, 17473, 
	17478, 17505, 17510, 17712, 17721, 17729, 17734, 17761, 
	17766, 17968, 17977, 17985, 17990, 18017, 18022, 10288, 
	10297, 10305, 10310, 10337, 10342, 10544, 10553, 10561, 
	10566, 10593, 10598, 10800, 10809, 10817, 10822, 10849, 
	10854, 11056, 11065, 11073, 11078, 11105, 11110, 11312, 
	11321, 11329, 11334, 11361, 11366, 11568, 11577, 11585, 
	11590, 11617, 11622, 11824, 11833, 11841, 11846, 11873, 
	11878, 2631, 2663, 3143, 3175, 27719, 27751, 27975, 
	28007, 28231, 28263, 3667, 3699, 4179, 4211, 28755, 
	28787, 29011, 29043, 29267, 29299, 4683, 4715, 5195, 
	5227, 29771, 29803, 30027, 30059, 30283, 30315, 5707, 
	5739, 32843, 32875, 33099, 33131, 33355, 33387, 33867, 
	33899, 34123, 34155, 34379, 34411, 30795, 30827, 31051, 
	31083, 31307, 31339, 31563, 31595, 31819, 31851, 32075, 
	32107, 32331, 32363, 6219, 6251, 38987, 39019, 39243, 
	39275, 39499, 39531, 40011, 40043, 40267, 40299, 40523, 
	40555, 41035, 41067, 41291, 41323, 41547, 41579, 41803, 
	41835, 42059, 42091, 42315, 42347, 42571, 42603, 43083, 
	43115, 43339, 43371, 43595, 43627, 44107, 44139, 44363, 
	44395, 44619, 44651, 44875, 44907, 45131, 45163, 45387, 
	45419, 45643, 45675, 46155, 46187, 46411, 46443, 46667, 
	46699, 46923, 46955, 47179, 47211, 47435, 47467, 47691, 
	47723, 34891, 34923, 35147, 35179, 35403, 35435, 35659, 
	35691, 35915, 35947, 36171, 36203, 36427, 36459, 36683, 
	36715, 36939, 36971, 37195, 37227, 37451, 37483, 37707, 
	37739, 37963, 37995, 38219, 38251, 38475, 38507, 6665, 
	6688, 6700, 6704, 6713, 6714, 6715, 6665, 6688, 
	6700, 6704, 6713, 6714, 6715, 6704, 6713, 6665, 
	6688, 6704, 6713, 6704, 6713, 6665, 6688, 6700, 
	577, 582, 609, 614, 6714, 6715, 56368, 56377, 
	56624, 56633, 56880, 56889, 6665, 6688, 6700, 1089, 
	1094, 1121, 1126, 6714, 6715, 57392, 57401, 57648, 
	57657, 57904, 57913, 6665, 6688, 6700, 6714, 6715, 
	9281, 9286, 9313, 9318, 9537, 9542, 9569, 9574, 
	9793, 9798, 9825, 9830, 58416, 58425, 58672, 58681, 
	58928, 58937, 59184, 59193, 59440, 59449, 59696, 59705, 
	59952, 59961, 6665, 6688, 6700, 577, 582, 609, 
	614, 6714, 6715, 56368, 56377, 56624, 56633, 56880, 
	56889, 6665, 6688, 6700, 1601, 1606, 1633, 1638, 
	6714, 6715, 60464, 60473, 60720, 60729, 60976, 60985, 
	6665, 6688, 6700, 6714, 6715, 12353, 12358, 12385, 
	12390, 12609, 12614, 12641, 12646, 12865, 12870, 12897, 
	12902, 61488, 61497, 61744, 61753, 62000, 62009, 62256, 
	62265, 62512, 62521, 62768, 62777, 63024, 63033, 6665, 
	6688, 6700, 577, 582, 609, 614, 6714, 6715, 
	56368, 56377, 56624, 56633, 56880, 56889, 6665, 6688, 
	6700, 6714, 6715, 13377, 13382, 13409, 13414, 13633, 
	13638, 13665, 13670, 13889, 13894, 13921, 13926, 63536, 
	63545, 63792, 63801, 64048, 64057, 64304, 64313, 64560, 
	64569, 64816, 64825, 65072, 65081, 6665, 6688, 6700, 
	1089, 1094, 1121, 1126, 6714, 6715, 57392, 57401, 
	57648, 57657, 57904, 57913, 6665, 6688, 6700, 6714, 
	6715, 10305, 10310, 10337, 10342, 10561, 10566, 10593, 
	10598, 10817, 10822, 10849, 10854, 11073, 11078, 11105, 
	11110, 11329, 11334, 11361, 11366, 11585, 11590, 11617, 
	11622, 11841, 11846, 11873, 11878, 65584, 65593, 65840, 
	65849, 66096, 66105, 66352, 66361, 66608, 66617, 66864, 
	66873, 67120, 67129, 67376, 67385, 67632, 67641, 67888, 
	67897, 68144, 68153, 68400, 68409, 68656, 68665, 68912, 
	68921, 69168, 69177, 6665, 6688, 6700, 6714, 6715, 
	9281, 9286, 9313, 9318, 9537, 9542, 9569, 9574, 
	9793, 9798, 9825, 9830, 58416, 58425, 58672, 58681, 
	58928, 58937, 59184, 59193, 59440, 59449, 59696, 59705, 
	59952, 59961, 6665, 6688, 6700, 2113, 2118, 2145, 
	2150, 6714, 6715, 69680, 69689, 69936, 69945, 70192, 
	70201, 6665, 6688, 6700, 6714, 6715, 18497, 18502, 
	18529, 18534, 18753, 18758, 18785, 18790, 19009, 19014, 
	19041, 19046, 70704, 70713, 70960, 70969, 71216, 71225, 
	71472, 71481, 71728, 71737, 71984, 71993, 72240, 72249, 
	6665, 6688, 6700, 577, 582, 609, 614, 6714, 
	6715, 56368, 56377, 56624, 56633, 56880, 56889, 6665, 
	6688, 6700, 6714, 6715, 19521, 19526, 19553, 19558, 
	19777, 19782, 19809, 19814, 20033, 20038, 20065, 20070, 
	72752, 72761, 73008, 73017, 73264, 73273, 73520, 73529, 
	73776, 73785, 74032, 74041, 74288, 74297, 6665, 6688, 
	6700, 1089, 1094, 1121, 1126, 6714, 6715, 57392, 
	57401, 57648, 57657, 57904, 57913, 6665, 6688, 6700, 
	6714, 6715, 20545, 20550, 20577, 20582, 20801, 20806, 
	20833, 20838, 21057, 21062, 21089, 21094, 21313, 21318, 
	21345, 21350, 21569, 21574, 21601, 21606, 21825, 21830, 
	21857, 21862, 22081, 22086, 22113, 22118, 74800, 74809, 
	75056, 75065, 75312, 75321, 75568, 75577, 75824, 75833, 
	76080, 76089, 76336, 76345, 76592, 76601, 76848, 76857, 
	77104, 77113, 77360, 77369, 77616, 77625, 77872, 77881, 
	78128, 78137, 78384, 78393, 6665, 6688, 6700, 6714, 
	6715, 9281, 9286, 9313, 9318, 9537, 9542, 9569, 
	9574, 9793, 9798, 9825, 9830, 58416, 58425, 58672, 
	58681, 58928, 58937, 59184, 59193, 59440, 59449, 59696, 
	59705, 59952, 59961, 6665, 6688, 6700, 6714, 6715, 
	22593, 22598, 22625, 22630, 22849, 22854, 22881, 22886, 
	23105, 23110, 23137, 23142, 78896, 78905, 79152, 79161, 
	79408, 79417, 79664, 79673, 79920, 79929, 80176, 80185, 
	80432, 80441, 6665, 6688, 6700, 1601, 1606, 1633, 
	1638, 6714, 6715, 60464, 60473, 60720, 60729, 60976, 
	60985, 6665, 6688, 6700, 6714, 6715, 23617, 23622, 
	23649, 23654, 23873, 23878, 23905, 23910, 24129, 24134, 
	24161, 24166, 24385, 24390, 24417, 24422, 24641, 24646, 
	24673, 24678, 24897, 24902, 24929, 24934, 25153, 25158, 
	25185, 25190, 80944, 80953, 81200, 81209, 81456, 81465, 
	81712, 81721, 81968, 81977, 82224, 82233, 82480, 82489, 
	82736, 82745, 82992, 83001, 83248, 83257, 83504, 83513, 
	83760, 83769, 84016, 84025, 84272, 84281, 84528, 84537, 
	6665, 6688, 6700, 6714, 6715, 12353, 12358, 12385, 
	12390, 12609, 12614, 12641, 12646, 12865, 12870, 12897, 
	12902, 61488, 61497, 61744, 61753, 62000, 62009, 62256, 
	62265, 62512, 62521, 62768, 62777, 63024, 63033, 6665, 
	6688, 6700, 6714, 6715, 25665, 25670, 25697, 25702, 
	25921, 25926, 25953, 25958, 26177, 26182, 26209, 26214, 
	26433, 26438, 26465, 26470, 26689, 26694, 26721, 26726, 
	26945, 26950, 26977, 26982, 27201, 27206, 27233, 27238, 
	85040, 85049, 85296, 85305, 85552, 85561, 85808, 85817, 
	86064, 86073, 86320, 86329, 86576, 86585, 86832, 86841, 
	87088, 87097, 87344, 87353, 87600, 87609, 87856, 87865, 
	88112, 88121, 88368, 88377, 88624, 88633, 6665, 6688, 
	6700, 6714, 6715, 13377, 13382, 13409, 13414, 13633, 
	13638, 13665, 13670, 13889, 13894, 13921, 13926, 63536, 
	63545, 63792, 63801, 64048, 64057, 64304, 64313, 64560, 
	64569, 64816, 64825, 65072, 65081, 6665, 6688, 6700, 
	6714, 6715, 14401, 14406, 14433, 14438, 14657, 14662, 
	14689, 14694, 14913, 14918, 14945, 14950, 15169, 15174, 
	15201, 15206, 15425, 15430, 15457, 15462, 15681, 15686, 
	15713, 15718, 15937, 15942, 15969, 15974, 16193, 16198, 
	16225, 16230, 16449, 16454, 16481, 16486, 16705, 16710, 
	16737, 16742, 16961, 16966, 16993, 16998, 17217, 17222, 
	17249, 17254, 17473, 17478, 17505, 17510, 17729, 17734, 
	17761, 17766, 17985, 17990, 18017, 18022, 48176, 48185, 
	48432, 48441, 48688, 48697, 48944, 48953, 49200, 49209, 
	49456, 49465, 49712, 49721, 49968, 49977, 50224, 50233, 
	50480, 50489, 50736, 50745, 50992, 51001, 51248, 51257, 
	51504, 51513, 51760, 51769, 52016, 52025, 52272, 52281, 
	52528, 52537, 52784, 52793, 53040, 53049, 53296, 53305, 
	53552, 53561, 53808, 53817, 54064, 54073, 54320, 54329, 
	54576, 54585, 54832, 54841, 55088, 55097, 55344, 55353, 
	55600, 55609, 55856, 55865, 6665, 6688, 6700, 6714, 
	6715, 10305, 10310, 10337, 10342, 10561, 10566, 10593, 
	10598, 10817, 10822, 10849, 10854, 11073, 11078, 11105, 
	11110, 11329, 11334, 11361, 11366, 11585, 11590, 11617, 
	11622, 11841, 11846, 11873, 11878, 65584, 65593, 65840, 
	65849, 66096, 66105, 66352, 66361, 66608, 66617, 66864, 
	66873, 67120, 67129, 67376, 67385, 67632, 67641, 67888, 
	67897, 68144, 68153, 68400, 68409, 68656, 68665, 68912, 
	68921, 69168, 69177, 7177, 7200, 7212, 7214, 7237, 
	7269, 7216, 7225, 7226, 7227, 582, 614, 7177, 
	7200, 7212, 7214, 105541, 105573, 105797, 105829, 106053, 
	106085, 577, 580, 609, 612, 7226, 7227, 105520, 
	105529, 105776, 105785, 106032, 106041, 7211, 7213, 7216, 
	7225, 7177, 7200, 7212, 7216, 7225, 7226, 7227, 
	1094, 1126, 7177, 7200, 7212, 7214, 106565, 106597, 
	106821, 106853, 107077, 107109, 1089, 1092, 1121, 1124, 
	7226, 7227, 106544, 106553, 106800, 106809, 107056, 107065, 
	7177, 7200, 7212, 7214, 9286, 9318, 9542, 9574, 
	9798, 9830, 107589, 107621, 107845, 107877, 108101, 108133, 
	108357, 108389, 108613, 108645, 108869, 108901, 109125, 109157, 
	7226, 7227, 9281, 9284, 9313, 9316, 9537, 9540, 
	9569, 9572, 9793, 9796, 9825, 9828, 107568, 107577, 
	107824, 107833, 108080, 108089, 108336, 108345, 108592, 108601, 
	108848, 108857, 109104, 109113, 582, 614, 7177, 7200, 
	7212, 7214, 105541, 105573, 105797, 105829, 106053, 106085, 
	577, 580, 609, 612, 7226, 7227, 105520, 105529, 
	105776, 105785, 106032, 106041, 7211, 7213, 577, 582, 
	609, 614, 105520, 105529, 105776, 105785, 106032, 106041, 
	7177, 7200, 7212, 577, 582, 609, 614, 7226, 
	7227, 105520, 105529, 105776, 105785, 106032, 106041, 1606, 
	1638, 7177, 7200, 7212, 7214, 109637, 109669, 109893, 
	109925, 110149, 110181, 1601, 1604, 1633, 1636, 7226, 
	7227, 109616, 109625, 109872, 109881, 110128, 110137, 7177, 
	7200, 7212, 7214, 12358, 12390, 12614, 12646, 12870, 
	12902, 110661, 110693, 110917, 110949, 111173, 111205, 111429, 
	111461, 111685, 111717, 111941, 111973, 112197, 112229, 7226, 
	7227, 12353, 12356, 12385, 12388, 12609, 12612, 12641, 
	12644, 12865, 12868, 12897, 12900, 110640, 110649, 110896, 
	110905, 111152, 111161, 111408, 111417, 111664, 111673, 111920, 
	111929, 112176, 112185, 582, 614, 7177, 7200, 7212, 
	7214, 105541, 105573, 105797, 105829, 106053, 106085, 577, 
	580, 609, 612, 7226, 7227, 105520, 105529, 105776, 
	105785, 106032, 106041, 7211, 7213, 577, 582, 609, 
	614, 105520, 105529, 105776, 105785, 106032, 106041, 7177, 
	7200, 7212, 577, 582, 609, 614, 7226, 7227, 
	105520, 105529, 105776, 105785, 106032, 106041, 7177, 7200, 
	7212, 7214, 13382, 13414, 13638, 13670, 13894, 13926, 
	112709, 112741, 112965, 112997, 113221, 113253, 113477, 113509, 
	113733, 113765, 113989, 114021, 114245, 114277, 7226, 7227, 
	13377, 13380, 13409, 13412, 13633, 13636, 13665, 13668, 
	13889, 13892, 13921, 13924, 112688, 112697, 112944, 112953, 
	113200, 113209, 113456, 113465, 113712, 113721, 113968, 113977, 
	114224, 114233, 1094, 1126, 7177, 7200, 7212, 7214, 
	106565, 106597, 106821, 106853, 107077, 107109, 1089, 1092, 
	1121, 1124, 7226, 7227, 106544, 106553, 106800, 106809, 
	107056, 107065, 7211, 7213, 1089, 1094, 1121, 1126, 
	106544, 106553, 106800, 106809, 107056, 107065, 7177, 7200, 
	7212, 1089, 1094, 1121, 1126, 7226, 7227, 106544, 
	106553, 106800, 106809, 107056, 107065, 7177, 7200, 7212, 
	7214, 10310, 10342, 10566, 10598, 10822, 10854, 11078, 
	11110, 11334, 11366, 11590, 11622, 11846, 11878, 114757, 
	114789, 115013, 115045, 115269, 115301, 115525, 115557, 115781, 
	115813, 116037, 116069, 116293, 116325, 116549, 116581, 116805, 
	116837, 117061, 117093, 117317, 117349, 117573, 117605, 117829, 
	117861, 118085, 118117, 118341, 118373, 7226, 7227, 10305, 
	10308, 10337, 10340, 10561, 10564, 10593, 10596, 10817, 
	10820, 10849, 10852, 11073, 11076, 11105, 11108, 11329, 
	11332, 11361, 11364, 11585, 11588, 11617, 11620, 11841, 
	11844, 11873, 11876, 114736, 114745, 114992, 115001, 115248, 
	115257, 115504, 115513, 115760, 115769, 116016, 116025, 116272, 
	116281, 116528, 116537, 116784, 116793, 117040, 117049, 117296, 
	117305, 117552, 117561, 117808, 117817, 118064, 118073, 118320, 
	118329, 7177, 7200, 7212, 7214, 9286, 9318, 9542, 
	9574, 9798, 9830, 107589, 107621, 107845, 107877, 108101, 
	108133, 108357, 108389, 108613, 108645, 108869, 108901, 109125, 
	109157, 7226, 7227, 9281, 9284, 9313, 9316, 9537, 
	9540, 9569, 9572, 9793, 9796, 9825, 9828, 107568, 
	107577, 107824, 107833, 108080, 108089, 108336, 108345, 108592, 
	108601, 108848, 108857, 109104, 109113, 7211, 7213, 9281, 
	9286, 9313, 9318, 9537, 9542, 9569, 9574, 9793, 
	9798, 9825, 9830, 107568, 107577, 107824, 107833, 108080, 
	108089, 108336, 108345, 108592, 108601, 108848, 108857, 109104, 
	109113, 7177, 7200, 7212, 7226, 7227, 9281, 9286, 
	9313, 9318, 9537, 9542, 9569, 9574, 9793, 9798, 
	9825, 9830, 107568, 107577, 107824, 107833, 108080, 108089, 
	108336, 108345, 108592, 108601, 108848, 108857, 109104, 109113, 
	2118, 2150, 7177, 7200, 7212, 7214, 118853, 118885, 
	119109, 119141, 119365, 119397, 2113, 2116, 2145, 2148, 
	7226, 7227, 118832, 118841, 119088, 119097, 119344, 119353, 
	7177, 7200, 7212, 7214, 18502, 18534, 18758, 18790, 
	19014, 19046, 119877, 119909, 120133, 120165, 120389, 120421, 
	120645, 120677, 120901, 120933, 121157, 121189, 121413, 121445, 
	7226, 7227, 18497, 18500, 18529, 18532, 18753, 18756, 
	18785, 18788, 19009, 19012, 19041, 19044, 119856, 119865, 
	120112, 120121, 120368, 120377, 120624, 120633, 120880, 120889, 
	121136, 121145, 121392, 121401, 582, 614, 7177, 7200, 
	7212, 7214, 105541, 105573, 105797, 105829, 106053, 106085, 
	577, 580, 609, 612, 7226, 7227, 105520, 105529, 
	105776, 105785, 106032, 106041, 7211, 7213, 577, 582, 
	609, 614, 105520, 105529, 105776, 105785, 106032, 106041, 
	7177, 7200, 7212, 577, 582, 609, 614, 7226, 
	7227, 105520, 105529, 105776, 105785, 106032, 106041, 7177, 
	7200, 7212, 7214, 19526, 19558, 19782, 19814, 20038, 
	20070, 121925, 121957, 122181, 122213, 122437, 122469, 122693, 
	122725, 122949, 122981, 123205, 123237, 123461, 123493, 7226, 
	7227, 19521, 19524, 19553, 19556, 19777, 19780, 19809, 
	19812, 20033, 20036, 20065, 20068, 121904, 121913, 122160, 
	122169, 122416, 122425, 122672, 122681, 122928, 122937, 123184, 
	123193, 123440, 123449, 1094, 1126, 7177, 7200, 7212, 
	7214, 106565, 106597, 106821, 106853, 107077, 107109, 1089, 
	1092, 1121, 1124, 7226, 7227, 106544, 106553, 106800, 
	106809, 107056, 107065, 7211, 7213, 1089, 1094, 1121, 
	1126, 106544, 106553, 106800, 106809, 107056, 107065, 7177, 
	7200, 7212, 1089, 1094, 1121, 1126, 7226, 7227, 
	106544, 106553, 106800, 106809, 107056, 107065, 7177, 7200, 
	7212, 7214, 20550, 20582, 20806, 20838, 21062, 21094, 
	21318, 21350, 21574, 21606, 21830, 21862, 22086, 22118, 
	123973, 124005, 124229, 124261, 124485, 124517, 124741, 124773, 
	124997, 125029, 125253, 125285, 125509, 125541, 125765, 125797, 
	126021, 126053, 126277, 126309, 126533, 126565, 126789, 126821, 
	127045, 127077, 127301, 127333, 127557, 127589, 7226, 7227, 
	20545, 20548, 20577, 20580, 20801, 20804, 20833, 20836, 
	21057, 21060, 21089, 21092, 21313, 21316, 21345, 21348, 
	21569, 21572, 21601, 21604, 21825, 21828, 21857, 21860, 
	22081, 22084, 22113, 22116, 123952, 123961, 124208, 124217, 
	124464, 124473, 124720, 124729, 124976, 124985, 125232, 125241, 
	125488, 125497, 125744, 125753, 126000, 126009, 126256, 126265, 
	126512, 126521, 126768, 126777, 127024, 127033, 127280, 127289, 
	127536, 127545, 7177, 7200, 7212, 7214, 9286, 9318, 
	9542, 9574, 9798, 9830, 107589, 107621, 107845, 107877, 
	108101, 108133, 108357, 108389, 108613, 108645, 108869, 108901, 
	109125, 109157, 7226, 7227, 9281, 9284, 9313, 9316, 
	9537, 9540, 9569, 9572, 9793, 9796, 9825, 9828, 
	107568, 107577, 107824, 107833, 108080, 108089, 108336, 108345, 
	108592, 108601, 108848, 108857, 109104, 109113, 7211, 7213, 
	9281, 9286, 9313, 9318, 9537, 9542, 9569, 9574, 
	9793, 9798, 9825, 9830, 107568, 107577, 107824, 107833, 
	108080, 108089, 108336, 108345, 108592, 108601, 108848, 108857, 
	109104, 109113, 7177, 7200, 7212, 7226, 7227, 9281, 
	9286, 9313, 9318, 9537, 9542, 9569, 9574, 9793, 
	9798, 9825, 9830, 107568, 107577, 107824, 107833, 108080, 
	108089, 108336, 108345, 108592, 108601, 108848, 108857, 109104, 
	109113, 7177, 7200, 7212, 7214, 22598, 22630, 22854, 
	22886, 23110, 23142, 128069, 128101, 128325, 128357, 128581, 
	128613, 128837, 128869, 129093, 129125, 129349, 129381, 129605, 
	129637, 7226, 7227, 22593, 22596, 22625, 22628, 22849, 
	22852, 22881, 22884, 23105, 23108, 23137, 23140, 128048, 
	128057, 128304, 128313, 128560, 128569, 128816, 128825, 129072, 
	129081, 129328, 129337, 129584, 129593, 1606, 1638, 7177, 
	7200, 7212, 7214, 109637, 109669, 109893, 109925, 110149, 
	110181, 1601, 1604, 1633, 1636, 7226, 7227, 109616, 
	109625, 109872, 109881, 110128, 110137, 7211, 7213, 1601, 
	1606, 1633, 1638, 109616, 109625, 109872, 109881, 110128, 
	110137, 7177, 7200, 7212, 1601, 1606, 1633, 1638, 
	7226, 7227, 109616, 109625, 109872, 109881, 110128, 110137, 
	7177, 7200, 7212, 7214, 23622, 23654, 23878, 23910, 
	24134, 24166, 24390, 24422, 24646, 24678, 24902, 24934, 
	25158, 25190, 130117, 130149, 130373, 130405, 130629, 130661, 
	130885, 130917, 131141, 131173, 131397, 131429, 131653, 131685, 
	131909, 131941, 132165, 132197, 132421, 132453, 132677, 132709, 
	132933, 132965, 133189, 133221, 133445, 133477, 133701, 133733, 
	7226, 7227, 23617, 23620, 23649, 23652, 23873, 23876, 
	23905, 23908, 24129, 24132, 24161, 24164, 24385, 24388, 
	24417, 24420, 24641, 24644, 24673, 24676, 24897, 24900, 
	24929, 24932, 25153, 25156, 25185, 25188, 130096, 130105, 
	130352, 130361, 130608, 130617, 130864, 130873, 131120, 131129, 
	131376, 131385, 131632, 131641, 131888, 131897, 132144, 132153, 
	132400, 132409, 132656, 132665, 132912, 132921, 133168, 133177, 
	133424, 133433, 133680, 133689, 7177, 7200, 7212, 7214, 
	12358, 12390, 12614, 12646, 12870, 12902, 110661, 110693, 
	110917, 110949, 111173, 111205, 111429, 111461, 111685, 111717, 
	111941, 111973, 112197, 112229, 7226, 7227, 12353, 12356, 
	12385, 12388, 12609, 12612, 12641, 12644, 12865, 12868, 
	12897, 12900, 110640, 110649, 110896, 110905, 111152, 111161, 
	111408, 111417, 111664, 111673, 111920, 111929, 112176, 112185, 
	7211, 7213, 12353, 12358, 12385, 12390, 12609, 12614, 
	12641, 12646, 12865, 12870, 12897, 12902, 110640, 110649, 
	110896, 110905, 111152, 111161, 111408, 111417, 111664, 111673, 
	111920, 111929, 112176, 112185, 7177, 7200, 7212, 7226, 
	7227, 12353, 12358, 12385, 12390, 12609, 12614, 12641, 
	12646, 12865, 12870, 12897, 12902, 110640, 110649, 110896, 
	110905, 111152, 111161, 111408, 111417, 111664, 111673, 111920, 
	111929, 112176, 112185, 7177, 7200, 7212, 7214, 25670, 
	25702, 25926, 25958, 26182, 26214, 26438, 26470, 26694, 
	26726, 26950, 26982, 27206, 27238, 134213, 134245, 134469, 
	134501, 134725, 134757, 134981, 135013, 135237, 135269, 135493, 
	135525, 135749, 135781, 136005, 136037, 136261, 136293, 136517, 
	136549, 136773, 136805, 137029, 137061, 137285, 137317, 137541, 
	137573, 137797, 137829, 7226, 7227, 25665, 25668, 25697, 
	25700, 25921, 25924, 25953, 25956, 26177, 26180, 26209, 
	26212, 26433, 26436, 26465, 26468, 26689, 26692, 26721, 
	26724, 26945, 26948, 26977, 26980, 27201, 27204, 27233, 
	27236, 134192, 134201, 134448, 134457, 134704, 134713, 134960, 
	134969, 135216, 135225, 135472, 135481, 135728, 135737, 135984, 
	135993, 136240, 136249, 136496, 136505, 136752, 136761, 137008, 
	137017, 137264, 137273, 137520, 137529, 137776, 137785, 7177, 
	7200, 7212, 7214, 13382, 13414, 13638, 13670, 13894, 
	13926, 112709, 112741, 112965, 112997, 113221, 113253, 113477, 
	113509, 113733, 113765, 113989, 114021, 114245, 114277, 7226, 
	7227, 13377, 13380, 13409, 13412, 13633, 13636, 13665, 
	13668, 13889, 13892, 13921, 13924, 112688, 112697, 112944, 
	112953, 113200, 113209, 113456, 113465, 113712, 113721, 113968, 
	113977, 114224, 114233, 7211, 7213, 13377, 13382, 13409, 
	13414, 13633, 13638, 13665, 13670, 13889, 13894, 13921, 
	13926, 112688, 112697, 112944, 112953, 113200, 113209, 113456, 
	113465, 113712, 113721, 113968, 113977, 114224, 114233, 7177, 
	7200, 7212, 7226, 7227, 13377, 13382, 13409, 13414, 
	13633, 13638, 13665, 13670, 13889, 13894, 13921, 13926, 
	112688, 112697, 112944, 112953, 113200, 113209, 113456, 113465, 
	113712, 113721, 113968, 113977, 114224, 114233, 7177, 7200, 
	7212, 7214, 14406, 14438, 14662, 14694, 14918, 14950, 
	15174, 15206, 15430, 15462, 15686, 15718, 15942, 15974, 
	16198, 16230, 16454, 16486, 16710, 16742, 16966, 16998, 
	17222, 17254, 17478, 17510, 17734, 17766, 17990, 18022, 
	138309, 138341, 138565, 138597, 138821, 138853, 139077, 139109, 
	139333, 139365, 139589, 139621, 139845, 139877, 140101, 140133, 
	140357, 140389, 140613, 140645, 140869, 140901, 141125, 141157, 
	141381, 141413, 141637, 141669, 141893, 141925, 142149, 142181, 
	142405, 142437, 142661, 142693, 142917, 142949, 143173, 143205, 
	143429, 143461, 143685, 143717, 143941, 143973, 144197, 144229, 
	144453, 144485, 144709, 144741, 144965, 144997, 145221, 145253, 
	145477, 145509, 145733, 145765, 145989, 146021, 7226, 7227, 
	14401, 14404, 14433, 14436, 14657, 14660, 14689, 14692, 
	14913, 14916, 14945, 14948, 15169, 15172, 15201, 15204, 
	15425, 15428, 15457, 15460, 15681, 15684, 15713, 15716, 
	15937, 15940, 15969, 15972, 16193, 16196, 16225, 16228, 
	16449, 16452, 16481, 16484, 16705, 16708, 16737, 16740, 
	16961, 16964, 16993, 16996, 17217, 17220, 17249, 17252, 
	17473, 17476, 17505, 17508, 17729, 17732, 17761, 17764, 
	17985, 17988, 18017, 18020, 138288, 138297, 138544, 138553, 
	138800, 138809, 139056, 139065, 139312, 139321, 139568, 139577, 
	139824, 139833, 140080, 140089, 140336, 140345, 140592, 140601, 
	140848, 140857, 141104, 141113, 141360, 141369, 141616, 141625, 
	141872, 141881, 142128, 142137, 142384, 142393, 142640, 142649, 
	142896, 142905, 143152, 143161, 143408, 143417, 143664, 143673, 
	143920, 143929, 144176, 144185, 144432, 144441, 144688, 144697, 
	144944, 144953, 145200, 145209, 145456, 145465, 145712, 145721, 
	145968, 145977, 7177, 7200, 7212, 7214, 10310, 10342, 
	10566, 10598, 10822, 10854, 11078, 11110, 11334, 11366, 
	11590, 11622, 11846, 11878, 114757, 114789, 115013, 115045, 
	115269, 115301, 115525, 115557, 115781, 115813, 116037, 116069, 
	116293, 116325, 116549, 116581, 116805, 116837, 117061, 117093, 
	117317, 117349, 117573, 117605, 117829, 117861, 118085, 118117, 
	118341, 118373, 7226, 7227, 10305, 10308, 10337, 10340, 
	10561, 10564, 10593, 10596, 10817, 10820, 10849, 10852, 
	11073, 11076, 11105, 11108, 11329, 11332, 11361, 11364, 
	11585, 11588, 11617, 11620, 11841, 11844, 11873, 11876, 
	114736, 114745, 114992, 115001, 115248, 115257, 115504, 115513, 
	115760, 115769, 116016, 116025, 116272, 116281, 116528, 116537, 
	116784, 116793, 117040, 117049, 117296, 117305, 117552, 117561, 
	117808, 117817, 118064, 118073, 118320, 118329, 7211, 7213, 
	10305, 10310, 10337, 10342, 10561, 10566, 10593, 10598, 
	10817, 10822, 10849, 10854, 11073, 11078, 11105, 11110, 
	11329, 11334, 11361, 11366, 11585, 11590, 11617, 11622, 
	11841, 11846, 11873, 11878, 114736, 114745, 114992, 115001, 
	115248, 115257, 115504, 115513, 115760, 115769, 116016, 116025, 
	116272, 116281, 116528, 116537, 116784, 116793, 117040, 117049, 
	117296, 117305, 117552, 117561, 117808, 117817, 118064, 118073, 
	118320, 118329, 7177, 7200, 7212, 7226, 7227, 10305, 
	10310, 10337, 10342, 10561, 10566, 10593, 10598, 10817, 
	10822, 10849, 10854, 11073, 11078, 11105, 11110, 11329, 
	11334, 11361, 11366, 11585, 11590, 11617, 11622, 11841, 
	11846, 11873, 11878, 114736, 114745, 114992, 115001, 115248, 
	115257, 115504, 115513, 115760, 115769, 116016, 116025, 116272, 
	116281, 116528, 116537, 116784, 116793, 117040, 117049, 117296, 
	117305, 117552, 117561, 117808, 117817, 118064, 118073, 118320, 
	118329, 7214, 7237, 7269, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, 146480, 146489, 146490, 
	146491, 146736, 146745, 146746, 146747, 146992, 147001, 147002, 
	147003, 45, 47, 6958, 6981, 7013, 7214, 7237, 
	7269, 146185, 146208, 146220, 146441, 146464, 146476, 146697, 
	146720, 146732, 146953, 146976, 146988, -128, 8, 10, 
	31, 33, 43, 60, 68, 70, 100, 102, 
	6408, 6410, 6431, 6433, 6443, 6445, 6447, 6460, 
	6664, 6666, 6687, 6689, 6699, 6701, 6703, 6716, 
	6920, 6922, 6943, 6945, 6955, 6957, 6959, 6972, 
	7176, 7178, 7199, 7201, 7211, 7213, 7215, 7228, 
	7268, 146224, 146235, 146480, 146489, 146490, 146491, 146736, 
	146745, 146746, 146747, 146992, 147001, 147002, 147003, 47, 
	6958, 6981, 7013, 7214, 7237, 7269, -128, 45, 
	58, 68, 70, 100, 102, 6447, 6458, 6703, 
	6714, 6959, 6970, 7215, 7226, 7268, 146224, 146233, 
	146480, 146489, 146736, 146745, 146992, 147001, 47, 6958, 
	6981, 7013, 7214, 7237, 7269, 146185, 146208, 146441, 
	146464, 146697, 146720, 146953, 146976, -128, 8, 10, 
	31, 33, 45, 58, 68, 70, 100, 102, 
	6408, 6410, 6431, 6433, 6447, 6458, 6664, 6666, 
	6687, 6689, 6703, 6714, 6920, 6922, 6943, 6945, 
	6959, 6970, 7176, 7178, 7199, 7201, 7215, 7226, 
	7268, 146224, 146233, 146480, 146489, 146736, 146745, 146992, 
	147001, 47, 6958, 6981, 7013, 7214, 7237, 7269, 
	-128, 45, 58, 68, 70, 100, 102, 6447, 
	6458, 6703, 6714, 6959, 6970, 7215, 7226, 7268, 
	146224, 146233, 146480, 146489, 146736, 146745, 146992, 147001, 
	582, 614, 7214, 105541, 105573, 105797, 105829, 106053, 
	106085, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 577, 580, 609, 612, 146490, 146491, 
	146746, 146747, 147002, 147003, 147504, 147513, 147760, 147769, 
	148016, 148025, 148272, 148281, 148528, 148537, 148784, 148793, 
	149040, 149049, 1094, 1126, 7214, 106565, 106597, 106821, 
	106853, 107077, 107109, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 1089, 1092, 1121, 1124, 
	146490, 146491, 146746, 146747, 147002, 147003, 149552, 149561, 
	149808, 149817, 150064, 150073, 150320, 150329, 150576, 150585, 
	150832, 150841, 151088, 151097, 7214, 9286, 9318, 9542, 
	9574, 9798, 9830, 107589, 107621, 107845, 107877, 108101, 
	108133, 108357, 108389, 108613, 108645, 108869, 108901, 109125, 
	109157, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 9281, 9284, 9313, 9316, 9537, 9540, 
	9569, 9572, 9793, 9796, 9825, 9828, 146490, 146491, 
	146746, 146747, 147002, 147003, 151600, 151609, 151856, 151865, 
	152112, 152121, 152368, 152377, 152624, 152633, 152880, 152889, 
	153136, 153145, 153392, 153401, 153648, 153657, 153904, 153913, 
	154160, 154169, 154416, 154425, 154672, 154681, 154928, 154937, 
	155184, 155193, 582, 614, 7214, 105541, 105573, 105797, 
	105829, 106053, 106085, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 577, 580, 609, 612, 
	146490, 146491, 146746, 146747, 147002, 147003, 147504, 147513, 
	147760, 147769, 148016, 148025, 148272, 148281, 148528, 148537, 
	148784, 148793, 149040, 149049, 1606, 1638, 7214, 109637, 
	109669, 109893, 109925, 110149, 110181, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 1601, 1604, 
	1633, 1636, 146490, 146491, 146746, 146747, 147002, 147003, 
	155696, 155705, 155952, 155961, 156208, 156217, 156464, 156473, 
	156720, 156729, 156976, 156985, 157232, 157241, 7214, 12358, 
	12390, 12614, 12646, 12870, 12902, 110661, 110693, 110917, 
	110949, 111173, 111205, 111429, 111461, 111685, 111717, 111941, 
	111973, 112197, 112229, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 12353, 12356, 12385, 12388, 
	12609, 12612, 12641, 12644, 12865, 12868, 12897, 12900, 
	146490, 146491, 146746, 146747, 147002, 147003, 157744, 157753, 
	158000, 158009, 158256, 158265, 158512, 158521, 158768, 158777, 
	159024, 159033, 159280, 159289, 159536, 159545, 159792, 159801, 
	160048, 160057, 160304, 160313, 160560, 160569, 160816, 160825, 
	161072, 161081, 161328, 161337, 582, 614, 7214, 105541, 
	105573, 105797, 105829, 106053, 106085, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 577, 580, 
	609, 612, 146490, 146491, 146746, 146747, 147002, 147003, 
	147504, 147513, 147760, 147769, 148016, 148025, 148272, 148281, 
	148528, 148537, 148784, 148793, 149040, 149049, 7214, 13382, 
	13414, 13638, 13670, 13894, 13926, 112709, 112741, 112965, 
	112997, 113221, 113253, 113477, 113509, 113733, 113765, 113989, 
	114021, 114245, 114277, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 13377, 13380, 13409, 13412, 
	13633, 13636, 13665, 13668, 13889, 13892, 13921, 13924, 
	146490, 146491, 146746, 146747, 147002, 147003, 161840, 161849, 
	162096, 162105, 162352, 162361, 162608, 162617, 162864, 162873, 
	163120, 163129, 163376, 163385, 163632, 163641, 163888, 163897, 
	164144, 164153, 164400, 164409, 164656, 164665, 164912, 164921, 
	165168, 165177, 165424, 165433, 1094, 1126, 7214, 106565, 
	106597, 106821, 106853, 107077, 107109, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 1089, 1092, 
	1121, 1124, 146490, 146491, 146746, 146747, 147002, 147003, 
	149552, 149561, 149808, 149817, 150064, 150073, 150320, 150329, 
	150576, 150585, 150832, 150841, 151088, 151097, 7214, 10310, 
	10342, 10566, 10598, 10822, 10854, 11078, 11110, 11334, 
	11366, 11590, 11622, 11846, 11878, 114757, 114789, 115013, 
	115045, 115269, 115301, 115525, 115557, 115781, 115813, 116037, 
	116069, 116293, 116325, 116549, 116581, 116805, 116837, 117061, 
	117093, 117317, 117349, 117573, 117605, 117829, 117861, 118085, 
	118117, 118341, 118373, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 10305, 10308, 10337, 10340, 
	10561, 10564, 10593, 10596, 10817, 10820, 10849, 10852, 
	11073, 11076, 11105, 11108, 11329, 11332, 11361, 11364, 
	11585, 11588, 11617, 11620, 11841, 11844, 11873, 11876, 
	146490, 146491, 146746, 146747, 147002, 147003, 165936, 165945, 
	166192, 166201, 166448, 166457, 166704, 166713, 166960, 166969, 
	167216, 167225, 167472, 167481, 167728, 167737, 167984, 167993, 
	168240, 168249, 168496, 168505, 168752, 168761, 169008, 169017, 
	169264, 169273, 169520, 169529, 169776, 169785, 170032, 170041, 
	170288, 170297, 170544, 170553, 170800, 170809, 171056, 171065, 
	171312, 171321, 171568, 171577, 171824, 171833, 172080, 172089, 
	172336, 172345, 172592, 172601, 172848, 172857, 173104, 173113, 
	173360, 173369, 173616, 173625, 7214, 9286, 9318, 9542, 
	9574, 9798, 9830, 107589, 107621, 107845, 107877, 108101, 
	108133, 108357, 108389, 108613, 108645, 108869, 108901, 109125, 
	109157, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 9281, 9284, 9313, 9316, 9537, 9540, 
	9569, 9572, 9793, 9796, 9825, 9828, 146490, 146491, 
	146746, 146747, 147002, 147003, 151600, 151609, 151856, 151865, 
	152112, 152121, 152368, 152377, 152624, 152633, 152880, 152889, 
	153136, 153145, 153392, 153401, 153648, 153657, 153904, 153913, 
	154160, 154169, 154416, 154425, 154672, 154681, 154928, 154937, 
	155184, 155193, 2118, 2150, 7214, 118853, 118885, 119109, 
	119141, 119365, 119397, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 2113, 2116, 2145, 2148, 
	146490, 146491, 146746, 146747, 147002, 147003, 174128, 174137, 
	174384, 174393, 174640, 174649, 174896, 174905, 175152, 175161, 
	175408, 175417, 175664, 175673, 7214, 18502, 18534, 18758, 
	18790, 19014, 19046, 119877, 119909, 120133, 120165, 120389, 
	120421, 120645, 120677, 120901, 120933, 121157, 121189, 121413, 
	121445, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 18497, 18500, 18529, 18532, 18753, 18756, 
	18785, 18788, 19009, 19012, 19041, 19044, 146490, 146491, 
	146746, 146747, 147002, 147003, 176176, 176185, 176432, 176441, 
	176688, 176697, 176944, 176953, 177200, 177209, 177456, 177465, 
	177712, 177721, 177968, 177977, 178224, 178233, 178480, 178489, 
	178736, 178745, 178992, 179001, 179248, 179257, 179504, 179513, 
	179760, 179769, 582, 614, 7214, 105541, 105573, 105797, 
	105829, 106053, 106085, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 577, 580, 609, 612, 
	146490, 146491, 146746, 146747, 147002, 147003, 147504, 147513, 
	147760, 147769, 148016, 148025, 148272, 148281, 148528, 148537, 
	148784, 148793, 149040, 149049, 7214, 19526, 19558, 19782, 
	19814, 20038, 20070, 121925, 121957, 122181, 122213, 122437, 
	122469, 122693, 122725, 122949, 122981, 123205, 123237, 123461, 
	123493, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 19521, 19524, 19553, 19556, 19777, 19780, 
	19809, 19812, 20033, 20036, 20065, 20068, 146490, 146491, 
	146746, 146747, 147002, 147003, 180272, 180281, 180528, 180537, 
	180784, 180793, 181040, 181049, 181296, 181305, 181552, 181561, 
	181808, 181817, 182064, 182073, 182320, 182329, 182576, 182585, 
	182832, 182841, 183088, 183097, 183344, 183353, 183600, 183609, 
	183856, 183865, 1094, 1126, 7214, 106565, 106597, 106821, 
	106853, 107077, 107109, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 1089, 1092, 1121, 1124, 
	146490, 146491, 146746, 146747, 147002, 147003, 149552, 149561, 
	149808, 149817, 150064, 150073, 150320, 150329, 150576, 150585, 
	150832, 150841, 151088, 151097, 7214, 20550, 20582, 20806, 
	20838, 21062, 21094, 21318, 21350, 21574, 21606, 21830, 
	21862, 22086, 22118, 123973, 124005, 124229, 124261, 124485, 
	124517, 124741, 124773, 124997, 125029, 125253, 125285, 125509, 
	125541, 125765, 125797, 126021, 126053, 126277, 126309, 126533, 
	126565, 126789, 126821, 127045, 127077, 127301, 127333, 127557, 
	127589, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 20545, 20548, 20577, 20580, 20801, 20804, 
	20833, 20836, 21057, 21060, 21089, 21092, 21313, 21316, 
	21345, 21348, 21569, 21572, 21601, 21604, 21825, 21828, 
	21857, 21860, 22081, 22084, 22113, 22116, 146490, 146491, 
	146746, 146747, 147002, 147003, 184368, 184377, 184624, 184633, 
	184880, 184889, 185136, 185145, 185392, 185401, 185648, 185657, 
	185904, 185913, 186160, 186169, 186416, 186425, 186672, 186681, 
	186928, 186937, 187184, 187193, 187440, 187449, 187696, 187705, 
	187952, 187961, 188208, 188217, 188464, 188473, 188720, 188729, 
	188976, 188985, 189232, 189241, 189488, 189497, 189744, 189753, 
	190000, 190009, 190256, 190265, 190512, 190521, 190768, 190777, 
	191024, 191033, 191280, 191289, 191536, 191545, 191792, 191801, 
	192048, 192057, 7214, 9286, 9318, 9542, 9574, 9798, 
	9830, 107589, 107621, 107845, 107877, 108101, 108133, 108357, 
	108389, 108613, 108645, 108869, 108901, 109125, 109157, 146441, 
	146464, 146476, 146697, 146720, 146732, 146953, 146976, 146988, 
	9281, 9284, 9313, 9316, 9537, 9540, 9569, 9572, 
	9793, 9796, 9825, 9828, 146490, 146491, 146746, 146747, 
	147002, 147003, 151600, 151609, 151856, 151865, 152112, 152121, 
	152368, 152377, 152624, 152633, 152880, 152889, 153136, 153145, 
	153392, 153401, 153648, 153657, 153904, 153913, 154160, 154169, 
	154416, 154425, 154672, 154681, 154928, 154937, 155184, 155193, 
	7214, 22598, 22630, 22854, 22886, 23110, 23142, 128069, 
	128101, 128325, 128357, 128581, 128613, 128837, 128869, 129093, 
	129125, 129349, 129381, 129605, 129637, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 22593, 22596, 
	22625, 22628, 22849, 22852, 22881, 22884, 23105, 23108, 
	23137, 23140, 146490, 146491, 146746, 146747, 147002, 147003, 
	192560, 192569, 192816, 192825, 193072, 193081, 193328, 193337, 
	193584, 193593, 193840, 193849, 194096, 194105, 194352, 194361, 
	194608, 194617, 194864, 194873, 195120, 195129, 195376, 195385, 
	195632, 195641, 195888, 195897, 196144, 196153, 1606, 1638, 
	7214, 109637, 109669, 109893, 109925, 110149, 110181, 146441, 
	146464, 146476, 146697, 146720, 146732, 146953, 146976, 146988, 
	1601, 1604, 1633, 1636, 146490, 146491, 146746, 146747, 
	147002, 147003, 155696, 155705, 155952, 155961, 156208, 156217, 
	156464, 156473, 156720, 156729, 156976, 156985, 157232, 157241, 
	7214, 23622, 23654, 23878, 23910, 24134, 24166, 24390, 
	24422, 24646, 24678, 24902, 24934, 25158, 25190, 130117, 
	130149, 130373, 130405, 130629, 130661, 130885, 130917, 131141, 
	131173, 131397, 131429, 131653, 131685, 131909, 131941, 132165, 
	132197, 132421, 132453, 132677, 132709, 132933, 132965, 133189, 
	133221, 133445, 133477, 133701, 133733, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 23617, 23620, 
	23649, 23652, 23873, 23876, 23905, 23908, 24129, 24132, 
	24161, 24164, 24385, 24388, 24417, 24420, 24641, 24644, 
	24673, 24676, 24897, 24900, 24929, 24932, 25153, 25156, 
	25185, 25188, 146490, 146491, 146746, 146747, 147002, 147003, 
	196656, 196665, 196912, 196921, 197168, 197177, 197424, 197433, 
	197680, 197689, 197936, 197945, 198192, 198201, 198448, 198457, 
	198704, 198713, 198960, 198969, 199216, 199225, 199472, 199481, 
	199728, 199737, 199984, 199993, 200240, 200249, 200496, 200505, 
	200752, 200761, 201008, 201017, 201264, 201273, 201520, 201529, 
	201776, 201785, 202032, 202041, 202288, 202297, 202544, 202553, 
	202800, 202809, 203056, 203065, 203312, 203321, 203568, 203577, 
	203824, 203833, 204080, 204089, 204336, 204345, 7214, 12358, 
	12390, 12614, 12646, 12870, 12902, 110661, 110693, 110917, 
	110949, 111173, 111205, 111429, 111461, 111685, 111717, 111941, 
	111973, 112197, 112229, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 12353, 12356, 12385, 12388, 
	12609, 12612, 12641, 12644, 12865, 12868, 12897, 12900, 
	146490, 146491, 146746, 146747, 147002, 147003, 157744, 157753, 
	158000, 158009, 158256, 158265, 158512, 158521, 158768, 158777, 
	159024, 159033, 159280, 159289, 159536, 159545, 159792, 159801, 
	160048, 160057, 160304, 160313, 160560, 160569, 160816, 160825, 
	161072, 161081, 161328, 161337, 7214, 25670, 25702, 25926, 
	25958, 26182, 26214, 26438, 26470, 26694, 26726, 26950, 
	26982, 27206, 27238, 134213, 134245, 134469, 134501, 134725, 
	134757, 134981, 135013, 135237, 135269, 135493, 135525, 135749, 
	135781, 136005, 136037, 136261, 136293, 136517, 136549, 136773, 
	136805, 137029, 137061, 137285, 137317, 137541, 137573, 137797, 
	137829, 146441, 146464, 146476, 146697, 146720, 146732, 146953, 
	146976, 146988, 25665, 25668, 25697, 25700, 25921, 25924, 
	25953, 25956, 26177, 26180, 26209, 26212, 26433, 26436, 
	26465, 26468, 26689, 26692, 26721, 26724, 26945, 26948, 
	26977, 26980, 27201, 27204, 27233, 27236, 146490, 146491, 
	146746, 146747, 147002, 147003, 204848, 204857, 205104, 205113, 
	205360, 205369, 205616, 205625, 205872, 205881, 206128, 206137, 
	206384, 206393, 206640, 206649, 206896, 206905, 207152, 207161, 
	207408, 207417, 207664, 207673, 207920, 207929, 208176, 208185, 
	208432, 208441, 208688, 208697, 208944, 208953, 209200, 209209, 
	209456, 209465, 209712, 209721, 209968, 209977, 210224, 210233, 
	210480, 210489, 210736, 210745, 210992, 211001, 211248, 211257, 
	211504, 211513, 211760, 211769, 212016, 212025, 212272, 212281, 
	212528, 212537, 7214, 13382, 13414, 13638, 13670, 13894, 
	13926, 112709, 112741, 112965, 112997, 113221, 113253, 113477, 
	113509, 113733, 113765, 113989, 114021, 114245, 114277, 146441, 
	146464, 146476, 146697, 146720, 146732, 146953, 146976, 146988, 
	13377, 13380, 13409, 13412, 13633, 13636, 13665, 13668, 
	13889, 13892, 13921, 13924, 146490, 146491, 146746, 146747, 
	147002, 147003, 161840, 161849, 162096, 162105, 162352, 162361, 
	162608, 162617, 162864, 162873, 163120, 163129, 163376, 163385, 
	163632, 163641, 163888, 163897, 164144, 164153, 164400, 164409, 
	164656, 164665, 164912, 164921, 165168, 165177, 165424, 165433, 
	7214, 14406, 14438, 14662, 14694, 14918, 14950, 15174, 
	15206, 15430, 15462, 15686, 15718, 15942, 15974, 16198, 
	16230, 16454, 16486, 16710, 16742, 16966, 16998, 17222, 
	17254, 17478, 17510, 17734, 17766, 17990, 18022, 138309, 
	138341, 138565, 138597, 138821, 138853, 139077, 139109, 139333, 
	139365, 139589, 139621, 139845, 139877, 140101, 140133, 140357, 
	140389, 140613, 140645, 140869, 140901, 141125, 141157, 141381, 
	141413, 141637, 141669, 141893, 141925, 142149, 142181, 142405, 
	142437, 142661, 142693, 142917, 142949, 143173, 143205, 143429, 
	143461, 143685, 143717, 143941, 143973, 144197, 144229, 144453, 
	144485, 144709, 144741, 144965, 144997, 145221, 145253, 145477, 
	145509, 145733, 145765, 145989, 146021, 146441, 146464, 146476, 
	146697, 146720, 146732, 146953, 146976, 146988, 14401, 14404, 
	14433, 14436, 14657, 14660, 14689, 14692, 14913, 14916, 
	14945, 14948, 15169, 15172, 15201, 15204, 15425, 15428, 
	15457, 15460, 15681, 15684, 15713, 15716, 15937, 15940, 
	15969, 15972, 16193, 16196, 16225, 16228, 16449, 16452, 
	16481, 16484, 16705, 16708, 16737, 16740, 16961, 16964, 
	16993, 16996, 17217, 17220, 17249, 17252, 17473, 17476, 
	17505, 17508, 17729, 17732, 17761, 17764, 17985, 17988, 
	18017, 18020, 89136, 89145, 89392, 89401, 89648, 89657, 
	89904, 89913, 90160, 90169, 90416, 90425, 90672, 90681, 
	90928, 90937, 91184, 91193, 91440, 91449, 91696, 91705, 
	91952, 91961, 92208, 92217, 92464, 92473, 92720, 92729, 
	92976, 92985, 93232, 93241, 93488, 93497, 93744, 93753, 
	94000, 94009, 94256, 94265, 94512, 94521, 94768, 94777, 
	95024, 95033, 95280, 95289, 95536, 95545, 95792, 95801, 
	96048, 96057, 96304, 96313, 96560, 96569, 96816, 96825, 
	97072, 97081, 97328, 97337, 97584, 97593, 97840, 97849, 
	98096, 98105, 98352, 98361, 98608, 98617, 98864, 98873, 
	99120, 99129, 99376, 99385, 99632, 99641, 99888, 99897, 
	100144, 100153, 100400, 100409, 100656, 100665, 100912, 100921, 
	101168, 101177, 101424, 101433, 101680, 101689, 101936, 101945, 
	102192, 102201, 102448, 102457, 102704, 102713, 102960, 102969, 
	103216, 103225, 103472, 103481, 103728, 103737, 103984, 103993, 
	104240, 104249, 104496, 104505, 104752, 104761, 105008, 105017, 
	146490, 146491, 146746, 146747, 147002, 147003, 7214, 10310, 
	10342, 10566, 10598, 10822, 10854, 11078, 11110, 11334, 
	11366, 11590, 11622, 11846, 11878, 114757, 114789, 115013, 
	115045, 115269, 115301, 115525, 115557, 115781, 115813, 116037, 
	116069, 116293, 116325, 116549, 116581, 116805, 116837, 117061, 
	117093, 117317, 117349, 117573, 117605, 117829, 117861, 118085, 
	118117, 118341, 118373, 146441, 146464, 146476, 146697, 146720, 
	146732, 146953, 146976, 146988, 10305, 10308, 10337, 10340, 
	10561, 10564, 10593, 10596, 10817, 10820, 10849, 10852, 
	11073, 11076, 11105, 11108, 11329, 11332, 11361, 11364, 
	11585, 11588, 11617, 11620, 11841, 11844, 11873, 11876, 
	146490, 146491, 146746, 146747, 147002, 147003, 165936, 165945, 
	166192, 166201, 166448, 166457, 166704, 166713, 166960, 166969, 
	167216, 167225, 167472, 167481, 167728, 167737, 167984, 167993, 
	168240, 168249, 168496, 168505, 168752, 168761, 169008, 169017, 
	169264, 169273, 169520, 169529, 169776, 169785, 170032, 170041, 
	170288, 170297, 170544, 170553, 170800, 170809, 171056, 171065, 
	171312, 171321, 171568, 171577, 171824, 171833, 172080, 172089, 
	172336, 172345, 172592, 172601, 172848, 172857, 173104, 173113, 
	173360, 173369, 173616, 173625, 7727, 8234, 7727, 8234, 
	560, 569, 577, 582, 609, 614, 1072, 1081, 
	1089, 1094, 1121, 1126, 9264, 9273, 9281, 9286, 
	9313, 9318, 9520, 9529, 9537, 9542, 9569, 9574, 
	9776, 9785, 9793, 9798, 9825, 9830, 560, 569, 
	577, 582, 609, 614, 1584, 1593, 1601, 1606, 
	1633, 1638, 12336, 12345, 12353, 12358, 12385, 12390, 
	12592, 12601, 12609, 12614, 12641, 12646, 12848, 12857, 
	12865, 12870, 12897, 12902, 560, 569, 577, 582, 
	609, 614, 13360, 13369, 13377, 13382, 13409, 13414, 
	13616, 13625, 13633, 13638, 13665, 13670, 13872, 13881, 
	13889, 13894, 13921, 13926, 1072, 1081, 1089, 1094, 
	1121, 1126, 10288, 10297, 10305, 10310, 10337, 10342, 
	10544, 10553, 10561, 10566, 10593, 10598, 10800, 10809, 
	10817, 10822, 10849, 10854, 11056, 11065, 11073, 11078, 
	11105, 11110, 11312, 11321, 11329, 11334, 11361, 11366, 
	11568, 11577, 11585, 11590, 11617, 11622, 11824, 11833, 
	11841, 11846, 11873, 11878, 9264, 9273, 9281, 9286, 
	9313, 9318, 9520, 9529, 9537, 9542, 9569, 9574, 
	9776, 9785, 9793, 9798, 9825, 9830, 2096, 2105, 
	2113, 2118, 2145, 2150, 18480, 18489, 18497, 18502, 
	18529, 18534, 18736, 18745, 18753, 18758, 18785, 18790, 
	18992, 19001, 19009, 19014, 19041, 19046, 560, 569, 
	577, 582, 609, 614, 19504, 19513, 19521, 19526, 
	19553, 19558, 19760, 19769, 19777, 19782, 19809, 19814, 
	20016, 20025, 20033, 20038, 20065, 20070, 1072, 1081, 
	1089, 1094, 1121, 1126, 20528, 20537, 20545, 20550, 
	20577, 20582, 20784, 20793, 20801, 20806, 20833, 20838, 
	21040, 21049, 21057, 21062, 21089, 21094, 21296, 21305, 
	21313, 21318, 21345, 21350, 21552, 21561, 21569, 21574, 
	21601, 21606, 21808, 21817, 21825, 21830, 21857, 21862, 
	22064, 22073, 22081, 22086, 22113, 22118, 9264, 9273, 
	9281, 9286, 9313, 9318, 9520, 9529, 9537, 9542, 
	9569, 9574, 9776, 9785, 9793, 9798, 9825, 9830, 
	22576, 22585, 22593, 22598, 22625, 22630, 22832, 22841, 
	22849, 22854, 22881, 22886, 23088, 23097, 23105, 23110, 
	23137, 23142, 1584, 1593, 1601, 1606, 1633, 1638, 
	23600, 23609, 23617, 23622, 23649, 23654, 23856, 23865, 
	23873, 23878, 23905, 23910, 24112, 24121, 24129, 24134, 
	24161, 24166, 24368, 24377, 24385, 24390, 24417, 24422, 
	24624, 24633, 24641, 24646, 24673, 24678, 24880, 24889, 
	24897, 24902, 24929, 24934, 25136, 25145, 25153, 25158, 
	25185, 25190, 12336, 12345, 12353, 12358, 12385, 12390, 
	12592, 12601, 12609, 12614, 12641, 12646, 12848, 12857, 
	12865, 12870, 12897, 12902, 25648, 25657, 25665, 25670, 
	25697, 25702, 25904, 25913, 25921, 25926, 25953, 25958, 
	26160, 26169, 26177, 26182, 26209, 26214, 26416, 26425, 
	26433, 26438, 26465, 26470, 26672, 26681, 26689, 26694, 
	26721, 26726, 26928, 26937, 26945, 26950, 26977, 26982, 
	27184, 27193, 27201, 27206, 27233, 27238, 13360, 13369, 
	13377, 13382, 13409, 13414, 13616, 13625, 13633, 13638, 
	13665, 13670, 13872, 13881, 13889, 13894, 13921, 13926, 
	14384, 14393, 14401, 14406, 14433, 14438, 14640, 14649, 
	14657, 14662, 14689, 14694, 14896, 14905, 14913, 14918, 
	14945, 14950, 15152, 15161, 15169, 15174, 15201, 15206, 
	15408, 15417, 15425, 15430, 15457, 15462, 15664, 15673, 
	15681, 15686, 15713, 15718, 15920, 15929, 15937, 15942, 
	15969, 15974, 16176, 16185, 16193, 16198, 16225, 16230, 
	16432, 16441, 16449, 16454, 16481, 16486, 16688, 16697, 
	16705, 16710, 16737, 16742, 16944, 16953, 16961, 16966, 
	16993, 16998, 17200, 17209, 17217, 17222, 17249, 17254, 
	17456, 17465, 17473, 17478, 17505, 17510, 17712, 17721, 
	17729, 17734, 17761, 17766, 17968, 17977, 17985, 17990, 
	18017, 18022, 10288, 10297, 10305, 10310, 10337, 10342, 
	10544, 10553, 10561, 10566, 10593, 10598, 10800, 10809, 
	10817, 10822, 10849, 10854, 11056, 11065, 11073, 11078, 
	11105, 11110, 11312, 11321, 11329, 11334, 11361, 11366, 
	11568, 11577, 11585, 11590, 11617, 11622, 11824, 11833, 
	11841, 11846, 11873, 11878, 10, 13, 42, 10, 
	47, 10, 13, 10, 0
};

static const char _text_file_single_lengths[] = {
	0, 0, 5, 4, 1, 0, 4, 5, 
	1, 0, 5, 1, 0, 2, 0, 2, 
	0, 2, 0, 2, 5, 5, 1, 0, 
	5, 3, 5, 1, 0, 3, 5, 1, 
	0, 2, 0, 2, 0, 2, 0, 3, 
	6, 2, 0, 3, 6, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 2, 1, 5, 1, 0, 
	6, 4, 1, 0, 5, 5, 1, 0, 
	7, 4, 5, 1, 0, 6, 3, 3, 
	2, 0, 4, 7, 4, 2, 0, 5, 
	8, 2, 2, 0, 3, 6, 5, 1, 
	0, 6, 3, 5, 1, 0, 6, 3, 
	3, 2, 0, 4, 7, 3, 2, 0, 
	4, 7, 3, 2, 0, 4, 7, 2, 
	2, 1, 5, 1, 0, 6, 4, 1, 
	0, 5, 5, 1, 0, 6, 3, 5, 
	1, 0, 6, 3, 3, 2, 0, 4, 
	7, 3, 2, 0, 4, 7, 2, 2, 
	0, 3, 6, 5, 1, 0, 6, 3, 
	5, 1, 0, 6, 3, 5, 1, 0, 
	6, 3, 3, 2, 0, 4, 7, 3, 
	2, 0, 4, 7, 3, 2, 0, 4, 
	7, 3, 2, 0, 4, 7, 6, 3, 
	2, 1, 5, 1, 0, 13, 4, 1, 
	0, 3, 2, 5, 1, 0, 3, 4, 
	5, 1, 0, 6, 3, 3, 2, 0, 
	4, 7, 2, 0, 1, 4, 2, 0, 
	1, 4, 5, 1, 0, 3, 3, 5, 
	1, 0, 3, 3, 2, 0, 1, 4, 
	2, 0, 1, 4, 2, 2, 3, 2, 
	0, 11, 2, 2, 2, 3, 2, 2, 
	3, 2, 2, 2, 3, 14, 2, 2, 
	1, 5, 1, 0, 13, 4, 1, 0, 
	3, 2, 5, 1, 0, 3, 3, 5, 
	1, 0, 6, 3, 3, 2, 0, 4, 
	7, 2, 0, 1, 4, 2, 0, 1, 
	4, 5, 1, 0, 3, 3, 5, 1, 
	0, 3, 3, 5, 1, 0, 6, 3, 
	3, 2, 0, 4, 7, 2, 0, 1, 
	4, 2, 0, 1, 4, 2, 2, 3, 
	2, 0, 11, 2, 2, 2, 3, 2, 
	2, 3, 2, 2, 2, 3, 14, 6, 
	3, 2, 2, 2, 1, 5, 1, 0, 
	5, 5, 1, 0, 5, 5, 1, 0, 
	13, 3, 2, 2, 3, 2, 0, 11, 
	2, 2, 2, 3, 2, 2, 3, 2, 
	2, 2, 3, 14, 2, 2, 0, 3, 
	6, 2, 2, 0, 3, 6, 2, 2, 
	2, 1, 5, 1, 0, 5, 5, 1, 
	0, 5, 5, 1, 0, 13, 3, 5, 
	1, 0, 6, 3, 3, 2, 0, 4, 
	7, 2, 2, 3, 2, 0, 11, 2, 
	2, 2, 3, 2, 2, 3, 2, 2, 
	2, 3, 14, 2, 2, 0, 3, 6, 
	2, 2, 0, 3, 6, 6, 6, 6, 
	3, 15, 3, 0, 15, 15, 3, 0, 
	15, 15, 3, 0, 38, 8, 6, 6, 
	8, 6, 0, 32, 6, 6, 6, 8, 
	6, 6, 8, 6, 6, 6, 8, 41, 
	6, 6, 0, 9, 18, 6, 6, 0, 
	9, 18, 2, 2, 2, 1, 5, 1, 
	0, 5, 5, 1, 0, 5, 5, 1, 
	0, 6, 3, 3, 2, 0, 4, 7, 
	2, 2, 0, 3, 6, 2, 2, 0, 
	3, 6, 6, 4, 6, 4, 14, 8, 
	2, 2, 2, 1, 5, 1, 0, 5, 
	5, 1, 0, 5, 5, 1, 0, 6, 
	3, 5, 1, 0, 6, 3, 3, 2, 
	0, 4, 7, 3, 2, 0, 4, 7, 
	2, 2, 0, 3, 6, 2, 2, 0, 
	3, 6, 6, 4, 6, 4, 14, 8, 
	6, 6, 6, 3, 15, 3, 0, 15, 
	15, 3, 0, 15, 15, 3, 0, 17, 
	8, 8, 6, 0, 11, 20, 6, 6, 
	0, 9, 18, 6, 6, 0, 9, 18, 
	14, 8, 14, 8, 30, 12, 3, 2, 
	3, 3, 2, 3, 2, 2, 2, 2, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 12, 
	12, 12, 12, 12, 12, 2, 2, 3, 
	2, 3, 2, 3, 2, 3, 2, 3, 
	12, 12, 12, 12, 2, 2, 3, 2, 
	3, 2, 3, 24, 24, 24, 24, 2, 
	2, 3, 2, 3, 2, 3, 12, 12, 
	2, 2, 3, 24, 24, 2, 2, 3, 
	24, 24, 2, 2, 3, 48, 48, 2, 
	2, 3, 12, 2, 24, 2, 24, 2, 
	48, 2, 24, 2, 48, 2, 48, 2, 
	96, 2, 12, 9, 12, 9, 9, 9, 
	9, 9, 9, 9, 12, 18, 18, 18, 
	18, 18, 18, 18, 18, 18, 18, 30, 
	30, 30, 30, 18, 18, 30, 30, 30, 
	30, 54, 54, 18, 30, 30, 54, 30, 
	54, 54, 102, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	93, 1, 1, 4, 2, 0, 3, 2, 
	5, 5, 2, 0, 3, 3, 6, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 2, 6, 2, 2, 6, 
	2, 2, 6, 2, 6, 6, 14, 2, 
	6, 6, 14, 6, 14, 14, 30, 3, 
	3, 0, 2, 0, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 6, 12, 
	2, 3, 12, 24, 12, 2, 3, 12, 
	24, 12, 2, 3, 24, 12, 2, 3, 
	48, 24, 2, 3, 12, 24, 12, 2, 
	3, 24, 12, 2, 3, 48, 24, 2, 
	3, 24, 12, 2, 3, 48, 24, 2, 
	3, 48, 24, 2, 3, 96, 48, 2, 
	3, 12, 20, 7, 15, 7, 18, 18, 
	30, 18, 18, 30, 18, 30, 18, 54, 
	30, 18, 30, 18, 30, 18, 54, 30, 
	30, 18, 54, 30, 54, 30, 102, 54, 
	1, 1, 2, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 3, 1, 1, 
	2, 1
};

static const char _text_file_range_lengths[] = {
	0, 1, 2, 3, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	2, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 2, 
	2, 1, 1, 2, 2, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 9, 
	9, 9, 9, 3, 3, 9, 9, 9, 
	9, 21, 21, 3, 9, 9, 21, 9, 
	21, 21, 45, 0, 0, 1, 1, 1, 
	1, 2, 1, 1, 1, 1, 1, 1, 
	1, 0, 1, 1, 1, 1, 0, 0, 
	1, 1, 1, 1, 0, 1, 1, 1, 
	1, 0, 1, 1, 1, 1, 1, 1, 
	1, 1, 0, 1, 1, 1, 1, 0, 
	0, 1, 1, 1, 1, 0, 1, 1, 
	1, 1, 0, 1, 1, 1, 1, 0, 
	0, 0, 1, 1, 1, 1, 2, 1, 
	1, 1, 1, 1, 1, 1, 0, 1, 
	1, 1, 1, 0, 0, 1, 1, 1, 
	1, 0, 1, 1, 1, 1, 0, 1, 
	1, 1, 1, 1, 1, 1, 1, 0, 
	1, 1, 1, 1, 0, 1, 1, 1, 
	1, 0, 0, 1, 1, 1, 1, 0, 
	1, 1, 1, 1, 0, 1, 1, 1, 
	1, 0, 1, 1, 1, 1, 0, 0, 
	0, 0, 1, 1, 1, 1, 2, 1, 
	1, 1, 0, 1, 1, 1, 1, 0, 
	1, 1, 1, 1, 0, 0, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 0, 1, 
	1, 1, 1, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 0, 0, 0, 1, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 1, 1, 1, 1, 2, 1, 1, 
	1, 0, 1, 1, 1, 1, 0, 1, 
	1, 1, 1, 0, 0, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 0, 1, 1, 
	1, 1, 0, 1, 1, 1, 1, 0, 
	0, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 0, 0, 0, 
	1, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 0, 0, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 0, 0, 0, 0, 1, 1, 1, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 1, 0, 1, 1, 1, 
	1, 0, 1, 1, 1, 1, 0, 0, 
	0, 0, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 0, 1, 
	1, 1, 1, 0, 0, 1, 1, 1, 
	1, 0, 0, 0, 1, 1, 1, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 1, 0, 1, 1, 1, 1, 
	0, 1, 1, 1, 1, 0, 0, 0, 
	0, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 0, 0, 0, 
	0, 3, 3, 3, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 3, 
	0, 3, 3, 3, 3, 0, 3, 3, 
	3, 3, 0, 0, 0, 0, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 0, 0, 1, 1, 1, 1, 
	0, 1, 1, 1, 1, 0, 1, 1, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	0, 1, 1, 1, 1, 0, 0, 1, 
	1, 1, 1, 0, 1, 1, 1, 1, 
	0, 1, 1, 1, 1, 0, 1, 1, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	0, 0, 3, 3, 3, 3, 0, 3, 
	3, 3, 3, 0, 3, 3, 3, 3, 
	0, 0, 0, 0, 0, 0, 2, 1, 
	2, 1, 1, 1, 1, 1, 1, 1, 
	2, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 14, 14, 14, 14, 6, 
	6, 14, 14, 14, 14, 30, 30, 6, 
	14, 14, 30, 14, 30, 30, 62, 6, 
	6, 6, 6, 6, 6, 5, 5, 6, 
	5, 6, 5, 6, 5, 6, 5, 6, 
	6, 6, 6, 6, 5, 5, 6, 5, 
	6, 5, 6, 14, 14, 14, 14, 13, 
	13, 14, 13, 14, 13, 14, 6, 6, 
	5, 5, 6, 14, 14, 13, 13, 14, 
	14, 14, 13, 13, 14, 30, 30, 29, 
	29, 30, 6, 5, 14, 13, 14, 13, 
	30, 29, 14, 13, 30, 29, 30, 29, 
	62, 61, 6, 3, 6, 3, 3, 3, 
	3, 3, 3, 3, 6, 12, 12, 12, 
	12, 12, 12, 12, 12, 12, 12, 24, 
	24, 24, 24, 12, 12, 24, 24, 24, 
	24, 48, 48, 12, 24, 24, 48, 24, 
	48, 48, 96, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 9, 
	9, 9, 9, 9, 3, 3, 3, 9, 
	9, 9, 9, 9, 9, 21, 21, 21, 
	3, 3, 9, 9, 9, 9, 21, 21, 
	9, 9, 21, 21, 21, 21, 45, 45, 
	106, 0, 1, 1, 1, 1, 1, 1, 
	1, 2, 1, 1, 1, 2, 2, 1, 
	3, 3, 9, 3, 3, 9, 3, 9, 
	3, 21, 9, 3, 9, 3, 9, 3, 
	21, 9, 9, 3, 21, 9, 21, 9, 
	45, 21, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 2, 
	2, 1, 1, 1, 6, 6, 14, 6, 
	6, 14, 6, 14, 6, 30, 14, 6, 
	14, 6, 14, 6, 30, 14, 14, 6, 
	30, 14, 30, 14, 62, 30, 2, 6, 
	1, 2, 6, 14, 6, 5, 6, 6, 
	14, 6, 5, 6, 14, 6, 5, 6, 
	30, 14, 13, 14, 6, 14, 6, 5, 
	6, 14, 6, 5, 6, 30, 14, 13, 
	14, 14, 6, 5, 6, 30, 14, 13, 
	14, 30, 14, 13, 14, 62, 30, 29, 
	30, 6, 29, 12, 22, 12, 12, 12, 
	24, 12, 12, 24, 12, 24, 12, 48, 
	24, 12, 24, 12, 24, 12, 48, 24, 
	24, 12, 48, 24, 48, 24, 96, 48, 
	0, 0, 0, 3, 3, 9, 3, 3, 
	9, 3, 9, 3, 21, 9, 3, 9, 
	3, 9, 3, 21, 9, 9, 3, 21, 
	9, 21, 9, 45, 21, 0, 0, 0, 
	0, 0
};

static const short _text_file_index_offsets[] = {
	0, 0, 2, 10, 18, 21, 23, 29, 
	36, 39, 41, 48, 51, 53, 57, 59, 
	63, 65, 69, 71, 75, 82, 89, 92, 
	94, 102, 107, 114, 117, 119, 124, 131, 
	134, 136, 140, 142, 146, 148, 152, 154, 
	160, 169, 173, 175, 181, 190, 194, 198, 
	202, 206, 210, 214, 218, 222, 226, 230, 
	240, 250, 260, 270, 274, 278, 288, 298, 
	308, 318, 340, 362, 366, 376, 386, 408, 
	418, 440, 462, 508, 511, 513, 520, 523, 
	525, 533, 540, 543, 545, 552, 559, 562, 
	564, 573, 578, 585, 588, 590, 598, 602, 
	606, 610, 612, 618, 627, 632, 636, 638, 
	645, 655, 658, 662, 664, 669, 677, 684, 
	687, 689, 697, 701, 708, 711, 713, 721, 
	725, 729, 733, 735, 741, 750, 754, 758, 
	760, 766, 775, 779, 783, 785, 791, 800, 
	803, 806, 808, 815, 818, 820, 828, 835, 
	838, 840, 847, 854, 857, 859, 867, 871, 
	878, 881, 883, 891, 895, 899, 903, 905, 
	911, 920, 924, 928, 930, 936, 945, 948, 
	952, 954, 959, 967, 974, 977, 979, 987, 
	991, 998, 1001, 1003, 1011, 1015, 1022, 1025, 
	1027, 1035, 1039, 1043, 1047, 1049, 1055, 1064, 
	1068, 1072, 1074, 1080, 1089, 1093, 1097, 1099, 
	1105, 1114, 1118, 1122, 1124, 1130, 1139, 1146, 
	1150, 1153, 1155, 1162, 1165, 1167, 1182, 1189, 
	1192, 1194, 1199, 1202, 1209, 1212, 1214, 1219, 
	1224, 1231, 1234, 1236, 1244, 1248, 1252, 1256, 
	1258, 1264, 1273, 1277, 1279, 1282, 1288, 1292, 
	1294, 1297, 1303, 1310, 1313, 1315, 1320, 1324, 
	1331, 1334, 1336, 1341, 1345, 1349, 1351, 1354, 
	1360, 1364, 1366, 1369, 1375, 1378, 1381, 1385, 
	1389, 1391, 1404, 1407, 1410, 1413, 1417, 1420, 
	1423, 1427, 1430, 1433, 1436, 1440, 1456, 1459, 
	1462, 1464, 1471, 1474, 1476, 1491, 1498, 1501, 
	1503, 1508, 1511, 1518, 1521, 1523, 1528, 1532, 
	1539, 1542, 1544, 1552, 1556, 1560, 1564, 1566, 
	1572, 1581, 1585, 1587, 1590, 1596, 1600, 1602, 
	1605, 1611, 1618, 1621, 1623, 1628, 1632, 1639, 
	1642, 1644, 1649, 1653, 1660, 1663, 1665, 1673, 
	1677, 1681, 1685, 1687, 1693, 1702, 1706, 1708, 
	1711, 1717, 1721, 1723, 1726, 1732, 1735, 1738, 
	1742, 1746, 1748, 1761, 1764, 1767, 1770, 1774, 
	1777, 1780, 1784, 1787, 1790, 1793, 1797, 1813, 
	1820, 1824, 1827, 1830, 1833, 1835, 1842, 1845, 
	1847, 1854, 1861, 1864, 1866, 1873, 1880, 1883, 
	1885, 1900, 1904, 1907, 1910, 1914, 1918, 1920, 
	1933, 1936, 1939, 1942, 1946, 1949, 1952, 1956, 
	1959, 1962, 1965, 1969, 1985, 1988, 1992, 1994, 
	1999, 2007, 2010, 2014, 2016, 2021, 2029, 2032, 
	2035, 2038, 2040, 2047, 2050, 2052, 2059, 2066, 
	2069, 2071, 2078, 2085, 2088, 2090, 2105, 2109, 
	2116, 2119, 2121, 2129, 2133, 2137, 2141, 2143, 
	2149, 2158, 2161, 2164, 2168, 2172, 2174, 2187, 
	2190, 2193, 2196, 2200, 2203, 2206, 2210, 2213, 
	2216, 2219, 2223, 2239, 2242, 2246, 2248, 2253, 
	2261, 2264, 2268, 2270, 2275, 2283, 2290, 2297, 
	2304, 2308, 2327, 2334, 2338, 2357, 2376, 2383, 
	2387, 2406, 2425, 2432, 2436, 2478, 2487, 2494, 
	2501, 2510, 2520, 2524, 2560, 2567, 2574, 2581, 
	2590, 2597, 2604, 2613, 2620, 2627, 2634, 2643, 
	2688, 2695, 2705, 2709, 2722, 2744, 2751, 2761, 
	2765, 2778, 2800, 2803, 2806, 2809, 2811, 2818, 
	2821, 2823, 2830, 2837, 2840, 2842, 2849, 2856, 
	2859, 2861, 2869, 2873, 2877, 2881, 2883, 2889, 
	2898, 2901, 2905, 2907, 2912, 2920, 2923, 2927, 
	2929, 2934, 2942, 2949, 2954, 2961, 2966, 2981, 
	2990, 2993, 2996, 2999, 3001, 3008, 3011, 3013, 
	3020, 3027, 3030, 3032, 3039, 3046, 3049, 3051, 
	3059, 3063, 3070, 3073, 3075, 3083, 3087, 3091, 
	3095, 3097, 3103, 3112, 3116, 3120, 3122, 3128, 
	3137, 3140, 3144, 3146, 3151, 3159, 3162, 3166, 
	3168, 3173, 3181, 3188, 3193, 3200, 3205, 3220, 
	3229, 3236, 3243, 3250, 3254, 3273, 3280, 3284, 
	3303, 3322, 3329, 3333, 3352, 3371, 3378, 3382, 
	3403, 3412, 3421, 3431, 3435, 3450, 3474, 3481, 
	3491, 3495, 3508, 3530, 3537, 3547, 3551, 3564, 
	3586, 3601, 3610, 3625, 3634, 3665, 3678, 3684, 
	3688, 3694, 3699, 3703, 3708, 3712, 3716, 3720, 
	3724, 3730, 3740, 3750, 3760, 3770, 3780, 3790, 
	3800, 3810, 3820, 3830, 3848, 3866, 3884, 3902, 
	3912, 3922, 3940, 3958, 3976, 3994, 4028, 4062, 
	4072, 4090, 4108, 4142, 4160, 4194, 4228, 4294, 
	4313, 4332, 4351, 4370, 4389, 4408, 4416, 4424, 
	4434, 4442, 4452, 4460, 4470, 4478, 4488, 4496, 
	4506, 4525, 4544, 4563, 4582, 4590, 4598, 4608, 
	4616, 4626, 4634, 4644, 4683, 4722, 4761, 4800, 
	4816, 4832, 4850, 4866, 4884, 4900, 4918, 4937, 
	4956, 4964, 4972, 4982, 5021, 5060, 5076, 5092, 
	5110, 5149, 5188, 5204, 5220, 5238, 5317, 5396, 
	5428, 5460, 5494, 5513, 5521, 5560, 5576, 5615, 
	5631, 5710, 5742, 5781, 5797, 5876, 5908, 5987, 
	6019, 6178, 6242, 6261, 6274, 6293, 6306, 6319, 
	6332, 6345, 6358, 6371, 6384, 6403, 6434, 6465, 
	6496, 6527, 6558, 6589, 6620, 6651, 6682, 6713, 
	6768, 6823, 6878, 6933, 6964, 6995, 7050, 7105, 
	7160, 7215, 7318, 7421, 7452, 7507, 7562, 7665, 
	7720, 7823, 7926, 8125, 8129, 8133, 8137, 8141, 
	8145, 8149, 8153, 8157, 8161, 8165, 8169, 8173, 
	8183, 8193, 8203, 8213, 8223, 8227, 8231, 8235, 
	8245, 8255, 8265, 8275, 8285, 8295, 8317, 8339, 
	8361, 8365, 8369, 8379, 8389, 8399, 8409, 8431, 
	8453, 8463, 8473, 8495, 8517, 8539, 8561, 8607, 
	8653, 8853, 8855, 8858, 8864, 8868, 8870, 8875, 
	8879, 8886, 8894, 8898, 8900, 8905, 8911, 8920, 
	8922, 8926, 8930, 8940, 8944, 8948, 8958, 8962, 
	8972, 8976, 8998, 9008, 9012, 9022, 9026, 9036, 
	9040, 9062, 9072, 9082, 9086, 9108, 9118, 9140, 
	9150, 9196, 9218, 9221, 9224, 9231, 9234, 9237, 
	9244, 9247, 9250, 9257, 9260, 9267, 9274, 9289, 
	9292, 9299, 9306, 9321, 9328, 9343, 9358, 9389, 
	9395, 9401, 9403, 9407, 9409, 9419, 9429, 9447, 
	9457, 9467, 9485, 9495, 9513, 9523, 9557, 9575, 
	9585, 9603, 9613, 9631, 9641, 9675, 9693, 9711, 
	9721, 9755, 9773, 9807, 9825, 9891, 9925, 9934, 
	9953, 9957, 9963, 9982, 10021, 10040, 10048, 10058, 
	10077, 10116, 10135, 10143, 10153, 10192, 10211, 10219, 
	10229, 10308, 10347, 10363, 10381, 10400, 10439, 10458, 
	10466, 10476, 10515, 10534, 10542, 10552, 10631, 10670, 
	10686, 10704, 10743, 10762, 10770, 10780, 10859, 10898, 
	10914, 10932, 11011, 11050, 11066, 11084, 11243, 11322, 
	11354, 11388, 11407, 11457, 11477, 11515, 11535, 11566, 
	11597, 11652, 11683, 11714, 11769, 11800, 11855, 11886, 
	11989, 12044, 12075, 12130, 12161, 12216, 12247, 12350, 
	12405, 12460, 12491, 12594, 12649, 12752, 12807, 13006, 
	13109, 13111, 13113, 13116, 13120, 13124, 13134, 13138, 
	13142, 13152, 13156, 13166, 13170, 13192, 13202, 13206, 
	13216, 13220, 13230, 13234, 13256, 13266, 13276, 13280, 
	13302, 13312, 13334, 13344, 13390, 13412, 13416, 13418, 
	13420, 13423
};

static const short _text_file_indicies[] = {
	1, 0, 2, 2, 3, 4, 4, 1, 
	3, 0, 5, 5, 7, 8, 6, 9, 
	7, 0, 10, 11, 0, 12, 0, 13, 
	13, 14, 14, 12, 0, 15, 15, 16, 
	16, 17, 18, 0, 19, 20, 0, 21, 
	0, 22, 22, 23, 23, 24, 25, 0, 
	26, 27, 0, 28, 0, 29, 29, 30, 
	0, 30, 0, 31, 31, 32, 0, 32, 
	0, 33, 33, 34, 0, 34, 0, 13, 
	13, 34, 0, 13, 13, 10, 14, 14, 
	11, 0, 7, 7, 35, 35, 36, 37, 
	0, 38, 39, 0, 40, 0, 41, 41, 
	42, 43, 43, 40, 42, 0, 44, 44, 
	45, 45, 0, 45, 45, 46, 46, 47, 
	48, 0, 49, 50, 0, 51, 0, 52, 
	52, 53, 53, 0, 53, 53, 54, 54, 
	55, 56, 0, 57, 58, 0, 59, 0, 
	60, 60, 61, 0, 61, 0, 62, 62, 
	63, 0, 63, 0, 64, 64, 65, 0, 
	65, 0, 41, 41, 42, 65, 42, 0, 
	41, 41, 42, 38, 43, 43, 39, 42, 
	0, 66, 66, 67, 0, 67, 0, 2, 
	2, 3, 67, 3, 0, 2, 2, 3, 
	68, 4, 4, 69, 3, 0, 71, 71, 
	71, 70, 72, 72, 72, 70, 73, 73, 
	73, 0, 74, 74, 74, 0, 75, 75, 
	75, 0, 76, 76, 76, 0, 77, 77, 
	77, 70, 78, 78, 78, 70, 79, 79, 
	79, 0, 80, 80, 80, 0, 71, 71, 
	71, 77, 77, 77, 81, 81, 81, 70, 
	72, 72, 72, 78, 78, 78, 82, 82, 
	82, 70, 73, 73, 73, 79, 79, 79, 
	83, 83, 83, 0, 74, 74, 74, 80, 
	80, 80, 84, 84, 84, 0, 85, 85, 
	85, 70, 86, 86, 86, 70, 71, 71, 
	71, 85, 85, 85, 87, 87, 87, 70, 
	72, 72, 72, 86, 86, 86, 88, 88, 
	88, 70, 77, 77, 77, 85, 85, 85, 
	89, 89, 89, 70, 78, 78, 78, 86, 
	86, 86, 90, 90, 90, 70, 71, 71, 
	71, 77, 77, 77, 81, 81, 81, 85, 
	85, 85, 87, 87, 87, 89, 89, 89, 
	91, 91, 91, 70, 72, 72, 72, 78, 
	78, 78, 82, 82, 82, 86, 86, 86, 
	88, 88, 88, 90, 90, 90, 92, 92, 
	92, 70, 93, 93, 93, 70, 71, 71, 
	71, 93, 93, 93, 94, 94, 94, 70, 
	77, 77, 77, 93, 93, 93, 95, 95, 
	95, 70, 71, 71, 71, 77, 77, 77, 
	81, 81, 81, 93, 93, 93, 94, 94, 
	94, 95, 95, 95, 96, 96, 96, 70, 
	85, 85, 85, 93, 93, 93, 97, 97, 
	97, 70, 71, 71, 71, 85, 85, 85, 
	87, 87, 87, 93, 93, 93, 94, 94, 
	94, 97, 97, 97, 98, 98, 98, 70, 
	77, 77, 77, 85, 85, 85, 89, 89, 
	89, 93, 93, 93, 95, 95, 95, 97, 
	97, 97, 99, 99, 99, 70, 71, 71, 
	71, 77, 77, 77, 81, 81, 81, 85, 
	85, 85, 87, 87, 87, 89, 89, 89, 
	91, 91, 91, 93, 93, 93, 94, 94, 
	94, 95, 95, 95, 96, 96, 96, 97, 
	97, 97, 98, 98, 98, 99, 99, 99, 
	100, 100, 100, 70, 101, 101, 70, 102, 
	70, 102, 102, 103, 103, 104, 105, 70, 
	106, 107, 70, 108, 70, 109, 109, 110, 
	111, 112, 112, 108, 70, 113, 113, 115, 
	116, 114, 117, 70, 118, 119, 70, 120, 
	70, 121, 121, 122, 123, 123, 120, 70, 
	124, 124, 125, 125, 126, 127, 70, 128, 
	129, 70, 130, 70, 131, 131, 132, 133, 
	134, 135, 135, 130, 70, 136, 136, 137, 
	138, 70, 138, 138, 139, 139, 140, 141, 
	70, 142, 143, 70, 144, 70, 145, 145, 
	146, 147, 148, 148, 144, 70, 149, 149, 
	150, 70, 151, 151, 152, 70, 153, 153, 
	154, 70, 154, 70, 145, 145, 146, 147, 
	154, 70, 145, 145, 146, 147, 142, 148, 
	148, 143, 70, 155, 155, 156, 157, 70, 
	158, 158, 159, 70, 159, 70, 131, 131, 
	132, 133, 134, 159, 70, 131, 131, 132, 
	133, 128, 134, 135, 135, 129, 70, 160, 
	160, 70, 161, 161, 162, 70, 162, 70, 
	121, 121, 122, 162, 70, 121, 121, 122, 
	118, 123, 123, 119, 70, 115, 115, 163, 
	163, 164, 165, 70, 166, 167, 70, 168, 
	70, 169, 169, 170, 171, 172, 172, 168, 
	70, 173, 173, 174, 70, 174, 174, 175, 
	175, 176, 177, 70, 178, 179, 70, 180, 
	70, 181, 181, 182, 183, 184, 184, 180, 
	70, 185, 185, 186, 70, 187, 187, 188, 
	70, 189, 189, 190, 70, 190, 70, 181, 
	181, 182, 183, 190, 70, 181, 181, 182, 
	183, 178, 184, 184, 179, 70, 191, 191, 
	192, 70, 193, 193, 194, 70, 194, 70, 
	169, 169, 170, 171, 194, 70, 169, 169, 
	170, 171, 166, 172, 172, 167, 70, 195, 
	195, 196, 70, 197, 197, 198, 70, 198, 
	70, 109, 109, 110, 111, 198, 70, 109, 
	109, 110, 111, 106, 112, 112, 107, 70, 
	199, 199, 70, 200, 200, 70, 201, 70, 
	201, 201, 202, 202, 203, 204, 70, 205, 
	206, 70, 207, 70, 208, 208, 209, 210, 
	211, 211, 207, 70, 212, 212, 214, 215, 
	213, 216, 70, 217, 218, 70, 219, 70, 
	220, 220, 221, 222, 222, 219, 70, 223, 
	223, 224, 224, 225, 226, 70, 227, 228, 
	70, 229, 70, 230, 230, 231, 232, 233, 
	233, 229, 70, 234, 234, 235, 70, 235, 
	235, 236, 236, 237, 238, 70, 239, 240, 
	70, 241, 70, 242, 242, 243, 244, 245, 
	245, 241, 70, 246, 246, 247, 70, 248, 
	248, 249, 70, 250, 250, 251, 70, 251, 
	70, 242, 242, 243, 244, 251, 70, 242, 
	242, 243, 244, 239, 245, 245, 240, 70, 
	252, 252, 253, 70, 254, 254, 255, 70, 
	255, 70, 230, 230, 231, 232, 255, 70, 
	230, 230, 231, 227, 232, 233, 233, 228, 
	70, 256, 256, 70, 257, 257, 258, 70, 
	258, 70, 220, 220, 221, 258, 70, 220, 
	220, 221, 217, 222, 222, 218, 70, 214, 
	214, 259, 259, 260, 261, 70, 262, 263, 
	70, 264, 70, 265, 265, 266, 267, 268, 
	268, 264, 70, 269, 269, 270, 70, 270, 
	270, 271, 271, 272, 273, 70, 274, 275, 
	70, 276, 70, 277, 277, 278, 279, 280, 
	280, 276, 70, 281, 281, 282, 70, 282, 
	282, 283, 283, 284, 285, 70, 286, 287, 
	70, 288, 70, 289, 289, 290, 291, 292, 
	292, 288, 70, 293, 293, 294, 70, 295, 
	295, 296, 70, 297, 297, 298, 70, 298, 
	70, 289, 289, 290, 291, 298, 70, 289, 
	289, 290, 291, 286, 292, 292, 287, 70, 
	299, 299, 300, 70, 301, 301, 302, 70, 
	302, 70, 277, 277, 278, 279, 302, 70, 
	277, 277, 278, 279, 274, 280, 280, 275, 
	70, 303, 303, 304, 70, 305, 305, 306, 
	70, 306, 70, 265, 265, 266, 267, 306, 
	70, 265, 265, 266, 267, 262, 268, 268, 
	263, 70, 307, 307, 308, 70, 309, 309, 
	310, 70, 310, 70, 208, 208, 209, 210, 
	310, 70, 208, 208, 209, 210, 205, 211, 
	211, 206, 70, 101, 101, 199, 199, 311, 
	311, 70, 102, 200, 200, 70, 312, 312, 
	70, 313, 70, 313, 313, 314, 314, 315, 
	316, 70, 317, 318, 70, 319, 70, 320, 
	320, 321, 322, 323, 324, 325, 326, 322, 
	323, 324, 325, 326, 319, 70, 327, 327, 
	329, 330, 328, 331, 70, 332, 333, 70, 
	334, 70, 335, 336, 336, 334, 70, 337, 
	337, 70, 337, 337, 338, 338, 339, 340, 
	70, 341, 342, 70, 343, 70, 344, 345, 
	345, 343, 70, 346, 346, 347, 348, 70, 
	348, 348, 349, 349, 350, 351, 70, 352, 
	353, 70, 354, 70, 355, 355, 356, 357, 
	358, 358, 354, 70, 359, 359, 360, 70, 
	361, 361, 362, 70, 363, 363, 364, 70, 
	364, 70, 355, 355, 356, 357, 364, 70, 
	355, 355, 356, 357, 352, 358, 358, 353, 
	70, 365, 365, 366, 70, 366, 70, 344, 
	366, 70, 344, 341, 345, 345, 342, 70, 
	367, 367, 368, 70, 368, 70, 335, 368, 
	70, 335, 332, 336, 336, 333, 70, 329, 
	329, 369, 369, 370, 371, 70, 372, 373, 
	70, 374, 70, 375, 376, 376, 374, 70, 
	377, 377, 378, 70, 378, 378, 379, 379, 
	380, 381, 70, 382, 383, 70, 384, 70, 
	385, 386, 386, 384, 70, 387, 387, 388, 
	70, 389, 389, 390, 70, 390, 70, 385, 
	390, 70, 385, 382, 386, 386, 383, 70, 
	391, 391, 392, 70, 392, 70, 375, 392, 
	70, 375, 372, 376, 376, 373, 70, 393, 
	393, 70, 394, 394, 70, 395, 395, 396, 
	70, 397, 397, 398, 70, 398, 70, 320, 
	320, 321, 322, 324, 325, 326, 322, 324, 
	325, 326, 398, 70, 399, 399, 70, 400, 
	400, 70, 401, 401, 70, 402, 402, 403, 
	70, 404, 404, 70, 405, 405, 70, 406, 
	406, 407, 70, 408, 408, 70, 409, 409, 
	70, 410, 410, 70, 411, 411, 412, 70, 
	320, 320, 321, 317, 322, 323, 324, 325, 
	326, 322, 323, 324, 325, 326, 318, 70, 
	413, 413, 70, 414, 414, 70, 415, 70, 
	415, 415, 416, 416, 417, 418, 70, 419, 
	420, 70, 421, 70, 422, 422, 423, 424, 
	425, 426, 427, 428, 424, 425, 426, 427, 
	428, 421, 70, 429, 429, 431, 432, 430, 
	433, 70, 434, 435, 70, 436, 70, 437, 
	438, 438, 436, 70, 439, 439, 70, 439, 
	439, 440, 440, 441, 442, 70, 443, 444, 
	70, 445, 70, 446, 447, 447, 445, 70, 
	448, 448, 449, 70, 449, 449, 450, 450, 
	451, 452, 70, 453, 454, 70, 455, 70, 
	456, 456, 457, 458, 459, 459, 455, 70, 
	460, 460, 461, 70, 462, 462, 463, 70, 
	464, 464, 465, 70, 465, 70, 456, 456, 
	457, 458, 465, 70, 456, 456, 457, 458, 
	453, 459, 459, 454, 70, 466, 466, 467, 
	70, 467, 70, 446, 467, 70, 446, 443, 
	447, 447, 444, 70, 468, 468, 469, 70, 
	469, 70, 437, 469, 70, 437, 434, 438, 
	438, 435, 70, 431, 431, 470, 470, 471, 
	472, 70, 473, 474, 70, 475, 70, 476, 
	477, 477, 475, 70, 478, 478, 479, 70, 
	479, 479, 480, 480, 481, 482, 70, 483, 
	484, 70, 485, 70, 486, 487, 487, 485, 
	70, 488, 488, 489, 70, 489, 489, 490, 
	490, 491, 492, 70, 493, 494, 70, 495, 
	70, 496, 496, 497, 498, 499, 499, 495, 
	70, 500, 500, 501, 70, 502, 502, 503, 
	70, 504, 504, 505, 70, 505, 70, 496, 
	496, 497, 498, 505, 70, 496, 496, 497, 
	498, 493, 499, 499, 494, 70, 506, 506, 
	507, 70, 507, 70, 486, 507, 70, 486, 
	483, 487, 487, 484, 70, 508, 508, 509, 
	70, 509, 70, 476, 509, 70, 476, 473, 
	477, 477, 474, 70, 510, 510, 70, 511, 
	511, 70, 512, 512, 513, 70, 514, 514, 
	515, 70, 515, 70, 422, 422, 423, 424, 
	426, 427, 428, 424, 426, 427, 428, 515, 
	70, 516, 516, 70, 517, 517, 70, 518, 
	518, 70, 519, 519, 520, 70, 521, 521, 
	70, 522, 522, 70, 523, 523, 524, 70, 
	525, 525, 70, 526, 526, 70, 527, 527, 
	70, 528, 528, 529, 70, 422, 422, 423, 
	419, 424, 425, 426, 427, 428, 424, 425, 
	426, 427, 428, 420, 70, 312, 312, 413, 
	413, 530, 530, 70, 313, 414, 414, 70, 
	531, 531, 70, 532, 532, 70, 533, 533, 
	70, 534, 70, 534, 534, 535, 535, 536, 
	537, 70, 538, 539, 70, 540, 70, 541, 
	541, 542, 543, 543, 540, 70, 544, 544, 
	545, 545, 546, 547, 70, 548, 549, 70, 
	550, 70, 551, 551, 552, 553, 553, 550, 
	70, 554, 554, 555, 555, 556, 557, 70, 
	558, 559, 70, 560, 70, 561, 561, 562, 
	563, 564, 565, 566, 567, 563, 564, 565, 
	566, 567, 560, 70, 568, 568, 569, 70, 
	570, 570, 70, 571, 571, 70, 572, 572, 
	573, 70, 574, 574, 575, 70, 575, 70, 
	561, 561, 562, 563, 565, 566, 567, 563, 
	565, 566, 567, 575, 70, 576, 576, 70, 
	577, 577, 70, 578, 578, 70, 579, 579, 
	580, 70, 581, 581, 70, 582, 582, 70, 
	583, 583, 584, 70, 585, 585, 70, 586, 
	586, 70, 587, 587, 70, 588, 588, 589, 
	70, 561, 561, 562, 558, 563, 564, 565, 
	566, 567, 563, 564, 565, 566, 567, 559, 
	70, 590, 590, 70, 591, 591, 592, 70, 
	592, 70, 551, 551, 552, 592, 70, 551, 
	551, 552, 548, 553, 553, 549, 70, 593, 
	593, 70, 594, 594, 595, 70, 595, 70, 
	541, 541, 542, 595, 70, 541, 541, 542, 
	538, 543, 543, 539, 70, 596, 596, 70, 
	597, 597, 70, 598, 598, 70, 599, 70, 
	599, 599, 600, 600, 601, 602, 70, 603, 
	604, 70, 605, 70, 606, 606, 607, 608, 
	608, 605, 70, 609, 609, 610, 610, 611, 
	612, 70, 613, 614, 70, 615, 70, 616, 
	616, 617, 618, 618, 615, 70, 619, 619, 
	620, 620, 621, 622, 70, 623, 624, 70, 
	625, 70, 626, 626, 627, 628, 629, 630, 
	631, 632, 628, 629, 630, 631, 632, 625, 
	70, 633, 633, 634, 70, 634, 634, 635, 
	635, 636, 637, 70, 638, 639, 70, 640, 
	70, 641, 641, 642, 643, 644, 644, 640, 
	70, 645, 645, 646, 70, 647, 647, 648, 
	70, 649, 649, 650, 70, 650, 70, 641, 
	641, 642, 643, 650, 70, 641, 641, 642, 
	643, 638, 644, 644, 639, 70, 651, 651, 
	70, 652, 652, 70, 653, 653, 654, 70, 
	655, 655, 656, 70, 656, 70, 626, 626, 
	627, 628, 630, 631, 632, 628, 630, 631, 
	632, 656, 70, 657, 657, 70, 658, 658, 
	70, 659, 659, 70, 660, 660, 661, 70, 
	662, 662, 70, 663, 663, 70, 664, 664, 
	665, 70, 666, 666, 70, 667, 667, 70, 
	668, 668, 70, 669, 669, 670, 70, 626, 
	626, 623, 627, 628, 629, 630, 631, 632, 
	628, 629, 630, 631, 632, 624, 70, 671, 
	671, 70, 672, 672, 673, 70, 673, 70, 
	616, 616, 617, 673, 70, 616, 616, 617, 
	613, 618, 618, 614, 70, 674, 674, 70, 
	675, 675, 676, 70, 676, 70, 606, 606, 
	607, 676, 70, 606, 606, 607, 603, 608, 
	608, 604, 70, 531, 531, 596, 596, 677, 
	677, 70, 532, 532, 597, 597, 678, 678, 
	70, 533, 533, 598, 598, 679, 679, 70, 
	534, 599, 680, 70, 534, 534, 535, 535, 
	536, 599, 599, 600, 600, 601, 680, 680, 
	681, 681, 682, 537, 602, 683, 70, 538, 
	603, 684, 539, 604, 685, 70, 540, 605, 
	686, 70, 541, 541, 542, 543, 543, 606, 
	606, 607, 608, 608, 687, 687, 688, 689, 
	689, 540, 605, 686, 70, 544, 544, 545, 
	545, 546, 609, 609, 610, 610, 611, 690, 
	690, 691, 691, 692, 547, 612, 693, 70, 
	548, 613, 694, 549, 614, 695, 70, 550, 
	615, 696, 70, 551, 551, 552, 553, 553, 
	616, 616, 617, 618, 618, 697, 697, 698, 
	699, 699, 550, 615, 696, 70, 554, 554, 
	555, 555, 556, 619, 619, 620, 620, 621, 
	700, 700, 701, 701, 702, 557, 622, 703, 
	70, 558, 623, 704, 559, 624, 705, 70, 
	560, 625, 706, 70, 562, 627, 561, 561, 
	563, 564, 565, 566, 567, 563, 564, 565, 
	566, 567, 626, 626, 628, 629, 630, 631, 
	632, 628, 629, 630, 631, 632, 707, 707, 
	708, 709, 710, 711, 712, 708, 709, 710, 
	711, 712, 560, 625, 706, 70, 569, 634, 
	568, 568, 633, 633, 713, 713, 70, 570, 
	570, 651, 651, 714, 714, 70, 571, 571, 
	652, 652, 715, 715, 70, 573, 654, 572, 
	572, 653, 653, 716, 716, 70, 574, 574, 
	655, 655, 717, 717, 575, 656, 718, 70, 
	575, 656, 718, 70, 562, 627, 561, 561, 
	563, 565, 566, 567, 563, 565, 566, 567, 
	626, 626, 628, 630, 631, 632, 628, 630, 
	631, 632, 707, 707, 708, 710, 711, 712, 
	708, 710, 711, 712, 575, 656, 718, 70, 
	576, 576, 657, 657, 719, 719, 70, 577, 
	577, 658, 658, 720, 720, 70, 578, 578, 
	659, 659, 721, 721, 70, 580, 661, 579, 
	579, 660, 660, 722, 722, 70, 581, 581, 
	662, 662, 723, 723, 70, 582, 582, 663, 
	663, 724, 724, 70, 584, 665, 583, 583, 
	664, 664, 725, 725, 70, 585, 585, 666, 
	666, 726, 726, 70, 586, 586, 667, 667, 
	727, 727, 70, 587, 587, 668, 668, 728, 
	728, 70, 589, 670, 588, 588, 669, 669, 
	729, 729, 70, 562, 627, 561, 561, 558, 
	563, 564, 565, 566, 567, 563, 564, 565, 
	566, 567, 626, 626, 623, 628, 629, 630, 
	631, 632, 628, 629, 630, 631, 632, 707, 
	707, 704, 708, 709, 710, 711, 712, 708, 
	709, 710, 711, 712, 559, 624, 705, 70, 
	590, 590, 671, 671, 730, 730, 70, 591, 
	591, 672, 672, 731, 731, 592, 673, 732, 
	70, 592, 673, 732, 70, 551, 551, 552, 
	616, 616, 617, 697, 697, 698, 592, 673, 
	732, 70, 551, 551, 552, 548, 553, 553, 
	616, 616, 617, 613, 618, 618, 697, 697, 
	698, 694, 699, 699, 549, 614, 695, 70, 
	593, 593, 674, 674, 733, 733, 70, 594, 
	594, 675, 675, 734, 734, 595, 676, 735, 
	70, 595, 676, 735, 70, 541, 541, 542, 
	606, 606, 607, 687, 687, 688, 595, 676, 
	735, 70, 541, 541, 542, 538, 543, 543, 
	606, 606, 607, 603, 608, 608, 687, 687, 
	688, 684, 689, 689, 539, 604, 685, 70, 
	736, 736, 70, 737, 737, 70, 738, 738, 
	70, 739, 70, 739, 739, 740, 740, 741, 
	742, 70, 743, 744, 70, 745, 70, 746, 
	746, 747, 748, 748, 745, 70, 749, 749, 
	750, 750, 751, 752, 70, 753, 754, 70, 
	755, 70, 756, 756, 757, 758, 758, 755, 
	70, 759, 759, 760, 760, 761, 762, 70, 
	763, 764, 70, 765, 70, 766, 766, 767, 
	768, 769, 769, 765, 70, 770, 770, 771, 
	70, 772, 772, 773, 70, 774, 774, 775, 
	70, 775, 70, 766, 766, 767, 768, 775, 
	70, 766, 766, 767, 768, 763, 769, 769, 
	764, 70, 776, 776, 70, 777, 777, 778, 
	70, 778, 70, 756, 756, 757, 778, 70, 
	756, 756, 757, 753, 758, 758, 754, 70, 
	779, 779, 70, 780, 780, 781, 70, 781, 
	70, 746, 746, 747, 781, 70, 746, 746, 
	747, 743, 748, 748, 744, 70, 531, 531, 
	736, 736, 782, 782, 70, 532, 532, 737, 
	737, 70, 596, 596, 736, 736, 783, 783, 
	70, 597, 597, 737, 737, 70, 531, 531, 
	596, 596, 677, 677, 736, 736, 782, 782, 
	783, 783, 784, 784, 70, 737, 737, 532, 
	532, 597, 597, 678, 678, 70, 785, 785, 
	70, 786, 786, 70, 787, 787, 70, 788, 
	70, 788, 788, 789, 789, 790, 791, 70, 
	792, 793, 70, 794, 70, 795, 795, 796, 
	797, 797, 794, 70, 798, 798, 799, 799, 
	800, 801, 70, 802, 803, 70, 804, 70, 
	805, 805, 806, 807, 807, 804, 70, 808, 
	808, 809, 809, 810, 811, 70, 812, 813, 
	70, 814, 70, 815, 815, 816, 817, 818, 
	818, 814, 70, 819, 819, 820, 70, 820, 
	820, 821, 821, 822, 823, 70, 824, 825, 
	70, 826, 70, 827, 827, 828, 829, 830, 
	830, 826, 70, 831, 831, 832, 70, 833, 
	833, 834, 70, 835, 835, 836, 70, 836, 
	70, 827, 827, 828, 829, 836, 70, 827, 
	827, 828, 829, 824, 830, 830, 825, 70, 
	837, 837, 838, 70, 839, 839, 840, 70, 
	840, 70, 815, 815, 816, 817, 840, 70, 
	815, 815, 816, 812, 817, 818, 818, 813, 
	70, 841, 841, 70, 842, 842, 843, 70, 
	843, 70, 805, 805, 806, 843, 70, 805, 
	805, 806, 802, 807, 807, 803, 70, 844, 
	844, 70, 845, 845, 846, 70, 846, 70, 
	795, 795, 796, 846, 70, 795, 795, 796, 
	792, 797, 797, 793, 70, 531, 531, 785, 
	785, 847, 847, 70, 532, 532, 786, 786, 
	70, 596, 596, 785, 785, 848, 848, 70, 
	597, 597, 786, 786, 70, 531, 531, 596, 
	596, 677, 677, 785, 785, 847, 847, 848, 
	848, 849, 849, 70, 786, 786, 532, 532, 
	597, 597, 678, 678, 70, 736, 736, 785, 
	785, 850, 850, 70, 737, 737, 786, 786, 
	851, 851, 70, 738, 738, 787, 787, 852, 
	852, 70, 739, 788, 853, 70, 739, 739, 
	740, 740, 741, 788, 788, 789, 789, 790, 
	853, 853, 854, 854, 855, 742, 791, 856, 
	70, 743, 792, 857, 744, 793, 858, 70, 
	745, 794, 859, 70, 746, 746, 747, 748, 
	748, 795, 795, 796, 797, 797, 860, 860, 
	861, 862, 862, 745, 794, 859, 70, 749, 
	749, 750, 750, 751, 798, 798, 799, 799, 
	800, 863, 863, 864, 864, 865, 752, 801, 
	866, 70, 753, 802, 867, 754, 803, 868, 
	70, 755, 804, 869, 70, 756, 756, 757, 
	758, 758, 805, 805, 806, 807, 807, 870, 
	870, 871, 872, 872, 755, 804, 869, 70, 
	759, 759, 760, 760, 761, 808, 808, 809, 
	809, 810, 873, 873, 874, 874, 875, 762, 
	811, 876, 70, 763, 812, 877, 764, 813, 
	878, 70, 765, 814, 879, 70, 768, 817, 
	766, 766, 767, 769, 769, 815, 815, 816, 
	818, 818, 880, 880, 881, 882, 882, 765, 
	814, 879, 70, 771, 820, 770, 770, 819, 
	819, 883, 883, 70, 773, 838, 772, 772, 
	837, 837, 884, 884, 70, 774, 774, 839, 
	839, 885, 885, 775, 840, 886, 70, 775, 
	840, 886, 70, 768, 817, 766, 766, 767, 
	815, 815, 816, 880, 880, 881, 775, 840, 
	886, 70, 768, 817, 766, 766, 767, 763, 
	769, 769, 815, 815, 816, 812, 818, 818, 
	880, 880, 881, 877, 882, 882, 764, 813, 
	878, 70, 776, 776, 841, 841, 887, 887, 
	70, 777, 777, 842, 842, 888, 888, 778, 
	843, 889, 70, 778, 843, 889, 70, 756, 
	756, 757, 805, 805, 806, 870, 870, 871, 
	778, 843, 889, 70, 756, 756, 757, 753, 
	758, 758, 805, 805, 806, 802, 807, 807, 
	870, 870, 871, 867, 872, 872, 754, 803, 
	868, 70, 779, 779, 844, 844, 890, 890, 
	70, 780, 780, 845, 845, 891, 891, 781, 
	846, 892, 70, 781, 846, 892, 70, 746, 
	746, 747, 795, 795, 796, 860, 860, 861, 
	781, 846, 892, 70, 746, 746, 747, 743, 
	748, 748, 795, 795, 796, 792, 797, 797, 
	860, 860, 861, 857, 862, 862, 744, 793, 
	858, 70, 531, 531, 736, 736, 782, 782, 
	785, 785, 847, 847, 850, 850, 893, 893, 
	70, 532, 532, 737, 737, 786, 786, 851, 
	851, 70, 596, 596, 736, 736, 783, 783, 
	785, 785, 848, 848, 850, 850, 894, 894, 
	70, 597, 597, 737, 737, 786, 786, 851, 
	851, 70, 531, 531, 596, 596, 677, 677, 
	736, 736, 782, 782, 783, 783, 784, 784, 
	785, 785, 847, 847, 848, 848, 849, 849, 
	850, 850, 893, 893, 894, 894, 895, 895, 
	70, 532, 532, 597, 597, 678, 678, 737, 
	737, 786, 786, 851, 851, 70, 896, 896, 
	897, 898, 897, 0, 897, 897, 899, 0, 
	900, 900, 901, 902, 901, 0, 903, 903, 
	904, 904, 0, 904, 904, 905, 0, 907, 
	907, 908, 908, 906, 908, 908, 909, 906, 
	910, 910, 911, 0, 912, 912, 913, 0, 
	915, 915, 916, 914, 917, 917, 918, 919, 
	918, 0, 917, 917, 918, 71, 71, 918, 
	71, 919, 920, 70, 917, 917, 918, 72, 
	72, 918, 72, 919, 921, 70, 917, 917, 
	918, 73, 73, 918, 73, 919, 922, 0, 
	917, 917, 918, 74, 74, 918, 74, 919, 
	923, 0, 917, 917, 918, 75, 75, 918, 
	75, 919, 924, 0, 917, 917, 918, 76, 
	76, 918, 76, 919, 925, 0, 917, 917, 
	918, 77, 77, 918, 77, 919, 926, 70, 
	917, 917, 918, 78, 78, 918, 78, 919, 
	927, 70, 917, 917, 918, 79, 79, 918, 
	79, 919, 928, 0, 917, 917, 918, 80, 
	80, 918, 80, 919, 929, 0, 917, 917, 
	918, 918, 71, 71, 77, 77, 81, 81, 
	71, 77, 81, 919, 920, 926, 930, 70, 
	917, 917, 918, 918, 72, 72, 78, 78, 
	82, 82, 72, 78, 82, 919, 921, 927, 
	931, 70, 917, 917, 918, 918, 73, 73, 
	79, 79, 83, 83, 73, 79, 83, 919, 
	922, 928, 932, 0, 917, 917, 918, 918, 
	74, 74, 80, 80, 84, 84, 74, 80, 
	84, 919, 923, 929, 933, 0, 917, 917, 
	918, 85, 85, 918, 85, 919, 934, 70, 
	917, 917, 918, 86, 86, 918, 86, 919, 
	935, 70, 917, 917, 918, 918, 71, 71, 
	85, 85, 87, 87, 71, 85, 87, 919, 
	920, 934, 936, 70, 917, 917, 918, 918, 
	72, 72, 86, 86, 88, 88, 72, 86, 
	88, 919, 921, 935, 937, 70, 917, 917, 
	918, 918, 77, 77, 85, 85, 89, 89, 
	77, 85, 89, 919, 926, 934, 938, 70, 
	917, 917, 918, 918, 78, 78, 86, 86, 
	90, 90, 78, 86, 90, 919, 927, 935, 
	939, 70, 917, 917, 918, 918, 71, 71, 
	77, 77, 81, 81, 85, 85, 87, 87, 
	89, 89, 91, 91, 71, 77, 81, 85, 
	87, 89, 91, 919, 920, 926, 930, 934, 
	936, 938, 940, 70, 917, 917, 918, 918, 
	72, 72, 78, 78, 82, 82, 86, 86, 
	88, 88, 90, 90, 92, 92, 72, 78, 
	82, 86, 88, 90, 92, 919, 921, 927, 
	931, 935, 937, 939, 941, 70, 917, 917, 
	918, 93, 93, 918, 93, 919, 942, 70, 
	917, 917, 918, 918, 71, 71, 93, 93, 
	94, 94, 71, 93, 94, 919, 920, 942, 
	943, 70, 917, 917, 918, 918, 77, 77, 
	93, 93, 95, 95, 77, 93, 95, 919, 
	926, 942, 944, 70, 917, 917, 918, 918, 
	71, 71, 77, 77, 81, 81, 93, 93, 
	94, 94, 95, 95, 96, 96, 71, 77, 
	81, 93, 94, 95, 96, 919, 920, 926, 
	930, 942, 943, 944, 945, 70, 917, 917, 
	918, 918, 85, 85, 93, 93, 97, 97, 
	85, 93, 97, 919, 934, 942, 946, 70, 
	917, 917, 918, 918, 71, 71, 85, 85, 
	87, 87, 93, 93, 94, 94, 97, 97, 
	98, 98, 71, 85, 87, 93, 94, 97, 
	98, 919, 920, 934, 936, 942, 943, 946, 
	947, 70, 917, 917, 918, 918, 77, 77, 
	85, 85, 89, 89, 93, 93, 95, 95, 
	97, 97, 99, 99, 77, 85, 89, 93, 
	95, 97, 99, 919, 926, 934, 938, 942, 
	944, 946, 948, 70, 917, 917, 918, 918, 
	71, 71, 77, 77, 81, 81, 85, 85, 
	87, 87, 89, 89, 91, 91, 93, 93, 
	94, 94, 95, 95, 96, 96, 97, 97, 
	98, 98, 99, 99, 100, 100, 71, 77, 
	81, 85, 87, 89, 91, 93, 94, 95, 
	96, 97, 98, 99, 100, 919, 920, 926, 
	930, 934, 936, 938, 940, 942, 943, 944, 
	945, 946, 947, 948, 949, 70, 71, 71, 
	2, 2, 3, 68, 71, 71, 4, 4, 
	951, 951, 71, 71, 3, 71, 69, 950, 
	70, 72, 72, 2, 2, 3, 68, 72, 
	72, 4, 4, 953, 953, 72, 72, 3, 
	72, 69, 952, 70, 73, 73, 2, 2, 
	3, 68, 73, 73, 4, 4, 955, 955, 
	73, 73, 3, 73, 69, 954, 0, 74, 
	74, 2, 2, 3, 68, 74, 74, 4, 
	4, 957, 957, 74, 74, 3, 74, 69, 
	956, 0, 75, 75, 2, 2, 3, 68, 
	75, 75, 4, 4, 959, 959, 75, 75, 
	3, 75, 69, 958, 0, 76, 76, 2, 
	2, 3, 68, 76, 76, 4, 4, 961, 
	961, 76, 76, 3, 76, 69, 960, 0, 
	66, 66, 76, 76, 76, 67, 962, 0, 
	66, 66, 75, 75, 75, 67, 963, 0, 
	2, 2, 3, 76, 76, 3, 76, 67, 
	962, 0, 66, 66, 74, 74, 74, 67, 
	964, 0, 2, 2, 3, 75, 75, 3, 
	75, 67, 963, 0, 66, 66, 73, 73, 
	73, 67, 965, 0, 2, 2, 3, 74, 
	74, 3, 74, 67, 964, 0, 66, 66, 
	72, 72, 72, 67, 966, 70, 2, 2, 
	3, 73, 73, 3, 73, 67, 965, 0, 
	66, 66, 71, 71, 71, 67, 967, 70, 
	2, 2, 3, 72, 72, 3, 72, 67, 
	966, 70, 77, 77, 2, 2, 3, 68, 
	77, 77, 4, 4, 969, 969, 77, 77, 
	3, 77, 69, 968, 70, 78, 78, 2, 
	2, 3, 68, 78, 78, 4, 4, 971, 
	971, 78, 78, 3, 78, 69, 970, 70, 
	79, 79, 2, 2, 3, 68, 79, 79, 
	4, 4, 973, 973, 79, 79, 3, 79, 
	69, 972, 0, 80, 80, 2, 2, 3, 
	68, 80, 80, 4, 4, 975, 975, 80, 
	80, 3, 80, 69, 974, 0, 66, 66, 
	80, 80, 80, 67, 976, 0, 66, 66, 
	79, 79, 79, 67, 977, 0, 2, 2, 
	3, 80, 80, 3, 80, 67, 976, 0, 
	66, 66, 78, 78, 78, 67, 978, 70, 
	2, 2, 3, 79, 79, 3, 79, 67, 
	977, 0, 66, 66, 77, 77, 77, 67, 
	979, 70, 2, 2, 3, 78, 78, 3, 
	78, 67, 978, 70, 2, 2, 3, 68, 
	71, 71, 77, 77, 81, 81, 71, 71, 
	77, 77, 81, 81, 4, 4, 951, 951, 
	969, 969, 981, 981, 3, 71, 71, 77, 
	77, 81, 81, 71, 77, 81, 69, 950, 
	968, 980, 70, 2, 2, 3, 68, 72, 
	72, 78, 78, 82, 82, 72, 72, 78, 
	78, 82, 82, 4, 4, 953, 953, 971, 
	971, 983, 983, 3, 72, 72, 78, 78, 
	82, 82, 72, 78, 82, 69, 952, 970, 
	982, 70, 2, 2, 3, 68, 73, 73, 
	79, 79, 83, 83, 73, 73, 79, 79, 
	83, 83, 4, 4, 955, 955, 973, 973, 
	985, 985, 3, 73, 73, 79, 79, 83, 
	83, 73, 79, 83, 69, 954, 972, 984, 
	0, 2, 2, 3, 68, 74, 74, 80, 
	80, 84, 84, 74, 74, 80, 80, 84, 
	84, 4, 4, 957, 957, 975, 975, 987, 
	987, 3, 74, 74, 80, 80, 84, 84, 
	74, 80, 84, 69, 956, 974, 986, 0, 
	66, 66, 74, 74, 80, 80, 84, 84, 
	74, 80, 84, 67, 964, 976, 988, 0, 
	66, 66, 73, 73, 79, 79, 83, 83, 
	73, 79, 83, 67, 965, 977, 989, 0, 
	2, 2, 3, 3, 74, 74, 80, 80, 
	84, 84, 74, 80, 84, 67, 964, 976, 
	988, 0, 66, 66, 72, 72, 78, 78, 
	82, 82, 72, 78, 82, 67, 966, 978, 
	990, 70, 2, 2, 3, 3, 73, 73, 
	79, 79, 83, 83, 73, 79, 83, 67, 
	965, 977, 989, 0, 66, 66, 71, 71, 
	77, 77, 81, 81, 71, 77, 81, 67, 
	967, 979, 991, 70, 2, 2, 3, 3, 
	72, 72, 78, 78, 82, 82, 72, 78, 
	82, 67, 966, 978, 990, 70, 85, 85, 
	2, 2, 3, 68, 85, 85, 4, 4, 
	993, 993, 85, 85, 3, 85, 69, 992, 
	70, 86, 86, 2, 2, 3, 68, 86, 
	86, 4, 4, 995, 995, 86, 86, 3, 
	86, 69, 994, 70, 66, 66, 86, 86, 
	86, 67, 996, 70, 66, 66, 85, 85, 
	85, 67, 997, 70, 2, 2, 3, 86, 
	86, 3, 86, 67, 996, 70, 2, 2, 
	3, 68, 71, 71, 85, 85, 87, 87, 
	71, 71, 85, 85, 87, 87, 4, 4, 
	951, 951, 993, 993, 999, 999, 3, 71, 
	71, 85, 85, 87, 87, 71, 85, 87, 
	69, 950, 992, 998, 70, 2, 2, 3, 
	68, 72, 72, 86, 86, 88, 88, 72, 
	72, 86, 86, 88, 88, 4, 4, 953, 
	953, 995, 995, 1001, 1001, 3, 72, 72, 
	86, 86, 88, 88, 72, 86, 88, 69, 
	952, 994, 1000, 70, 66, 66, 72, 72, 
	86, 86, 88, 88, 72, 86, 88, 67, 
	966, 996, 1002, 70, 66, 66, 71, 71, 
	85, 85, 87, 87, 71, 85, 87, 67, 
	967, 997, 1003, 70, 2, 2, 3, 3, 
	72, 72, 86, 86, 88, 88, 72, 86, 
	88, 67, 966, 996, 1002, 70, 2, 2, 
	3, 68, 77, 77, 85, 85, 89, 89, 
	77, 77, 85, 85, 89, 89, 4, 4, 
	969, 969, 993, 993, 1005, 1005, 3, 77, 
	77, 85, 85, 89, 89, 77, 85, 89, 
	69, 968, 992, 1004, 70, 2, 2, 3, 
	68, 78, 78, 86, 86, 90, 90, 78, 
	78, 86, 86, 90, 90, 4, 4, 971, 
	971, 995, 995, 1007, 1007, 3, 78, 78, 
	86, 86, 90, 90, 78, 86, 90, 69, 
	970, 994, 1006, 70, 66, 66, 78, 78, 
	86, 86, 90, 90, 78, 86, 90, 67, 
	978, 996, 1008, 70, 66, 66, 77, 77, 
	85, 85, 89, 89, 77, 85, 89, 67, 
	979, 997, 1009, 70, 2, 2, 3, 3, 
	78, 78, 86, 86, 90, 90, 78, 86, 
	90, 67, 978, 996, 1008, 70, 2, 2, 
	3, 68, 71, 71, 77, 77, 81, 81, 
	85, 85, 87, 87, 89, 89, 91, 91, 
	71, 71, 77, 77, 81, 81, 85, 85, 
	87, 87, 89, 89, 91, 91, 4, 4, 
	951, 951, 969, 969, 981, 981, 993, 993, 
	999, 999, 1005, 1005, 1011, 1011, 3, 71, 
	71, 77, 77, 81, 81, 85, 85, 87, 
	87, 89, 89, 91, 91, 71, 77, 81, 
	85, 87, 89, 91, 69, 950, 968, 980, 
	992, 998, 1004, 1010, 70, 2, 2, 3, 
	68, 72, 72, 78, 78, 82, 82, 86, 
	86, 88, 88, 90, 90, 92, 92, 72, 
	72, 78, 78, 82, 82, 86, 86, 88, 
	88, 90, 90, 92, 92, 4, 4, 953, 
	953, 971, 971, 983, 983, 995, 995, 1001, 
	1001, 1007, 1007, 1013, 1013, 3, 72, 72, 
	78, 78, 82, 82, 86, 86, 88, 88, 
	90, 90, 92, 92, 72, 78, 82, 86, 
	88, 90, 92, 69, 952, 970, 982, 994, 
	1000, 1006, 1012, 70, 66, 66, 72, 72, 
	78, 78, 82, 82, 86, 86, 88, 88, 
	90, 90, 92, 92, 72, 78, 82, 86, 
	88, 90, 92, 67, 966, 978, 990, 996, 
	1002, 1008, 1014, 70, 66, 66, 71, 71, 
	77, 77, 81, 81, 85, 85, 87, 87, 
	89, 89, 91, 91, 71, 77, 81, 85, 
	87, 89, 91, 67, 967, 979, 991, 997, 
	1003, 1009, 1015, 70, 2, 2, 3, 3, 
	72, 72, 78, 78, 82, 82, 86, 86, 
	88, 88, 90, 90, 92, 92, 72, 78, 
	82, 86, 88, 90, 92, 67, 966, 978, 
	990, 996, 1002, 1008, 1014, 70, 93, 93, 
	2, 2, 3, 68, 93, 93, 4, 4, 
	1017, 1017, 93, 93, 3, 93, 69, 1016, 
	70, 66, 66, 93, 93, 93, 67, 1018, 
	70, 2, 2, 3, 68, 71, 71, 93, 
	93, 94, 94, 71, 71, 93, 93, 94, 
	94, 4, 4, 951, 951, 1017, 1017, 1020, 
	1020, 3, 71, 71, 93, 93, 94, 94, 
	71, 93, 94, 69, 950, 1016, 1019, 70, 
	66, 66, 71, 71, 93, 93, 94, 94, 
	71, 93, 94, 67, 967, 1018, 1021, 70, 
	2, 2, 3, 68, 77, 77, 93, 93, 
	95, 95, 77, 77, 93, 93, 95, 95, 
	4, 4, 969, 969, 1017, 1017, 1023, 1023, 
	3, 77, 77, 93, 93, 95, 95, 77, 
	93, 95, 69, 968, 1016, 1022, 70, 66, 
	66, 77, 77, 93, 93, 95, 95, 77, 
	93, 95, 67, 979, 1018, 1024, 70, 2, 
	2, 3, 68, 71, 71, 77, 77, 81, 
	81, 93, 93, 94, 94, 95, 95, 96, 
	96, 71, 71, 77, 77, 81, 81, 93, 
	93, 94, 94, 95, 95, 96, 96, 4, 
	4, 951, 951, 969, 969, 981, 981, 1017, 
	1017, 1020, 1020, 1023, 1023, 1026, 1026, 3, 
	71, 71, 77, 77, 81, 81, 93, 93, 
	94, 94, 95, 95, 96, 96, 71, 77, 
	81, 93, 94, 95, 96, 69, 950, 968, 
	980, 1016, 1019, 1022, 1025, 70, 66, 66, 
	71, 71, 77, 77, 81, 81, 93, 93, 
	94, 94, 95, 95, 96, 96, 71, 77, 
	81, 93, 94, 95, 96, 67, 967, 979, 
	991, 1018, 1021, 1024, 1027, 70, 2, 2, 
	3, 68, 85, 85, 93, 93, 97, 97, 
	85, 85, 93, 93, 97, 97, 4, 4, 
	993, 993, 1017, 1017, 1029, 1029, 3, 85, 
	85, 93, 93, 97, 97, 85, 93, 97, 
	69, 992, 1016, 1028, 70, 66, 66, 85, 
	85, 93, 93, 97, 97, 85, 93, 97, 
	67, 997, 1018, 1030, 70, 2, 2, 3, 
	68, 71, 71, 85, 85, 87, 87, 93, 
	93, 94, 94, 97, 97, 98, 98, 71, 
	71, 85, 85, 87, 87, 93, 93, 94, 
	94, 97, 97, 98, 98, 4, 4, 951, 
	951, 993, 993, 999, 999, 1017, 1017, 1020, 
	1020, 1029, 1029, 1032, 1032, 3, 71, 71, 
	85, 85, 87, 87, 93, 93, 94, 94, 
	97, 97, 98, 98, 71, 85, 87, 93, 
	94, 97, 98, 69, 950, 992, 998, 1016, 
	1019, 1028, 1031, 70, 66, 66, 71, 71, 
	85, 85, 87, 87, 93, 93, 94, 94, 
	97, 97, 98, 98, 71, 85, 87, 93, 
	94, 97, 98, 67, 967, 997, 1003, 1018, 
	1021, 1030, 1033, 70, 2, 2, 3, 68, 
	77, 77, 85, 85, 89, 89, 93, 93, 
	95, 95, 97, 97, 99, 99, 77, 77, 
	85, 85, 89, 89, 93, 93, 95, 95, 
	97, 97, 99, 99, 4, 4, 969, 969, 
	993, 993, 1005, 1005, 1017, 1017, 1023, 1023, 
	1029, 1029, 1035, 1035, 3, 77, 77, 85, 
	85, 89, 89, 93, 93, 95, 95, 97, 
	97, 99, 99, 77, 85, 89, 93, 95, 
	97, 99, 69, 968, 992, 1004, 1016, 1022, 
	1028, 1034, 70, 66, 66, 77, 77, 85, 
	85, 89, 89, 93, 93, 95, 95, 97, 
	97, 99, 99, 77, 85, 89, 93, 95, 
	97, 99, 67, 979, 997, 1009, 1018, 1024, 
	1030, 1036, 70, 2, 2, 3, 68, 71, 
	71, 77, 77, 81, 81, 85, 85, 87, 
	87, 89, 89, 91, 91, 93, 93, 94, 
	94, 95, 95, 96, 96, 97, 97, 98, 
	98, 99, 99, 100, 100, 71, 71, 77, 
	77, 81, 81, 85, 85, 87, 87, 89, 
	89, 91, 91, 93, 93, 94, 94, 95, 
	95, 96, 96, 97, 97, 98, 98, 99, 
	99, 100, 100, 4, 4, 951, 951, 969, 
	969, 981, 981, 993, 993, 999, 999, 1005, 
	1005, 1011, 1011, 1017, 1017, 1020, 1020, 1023, 
	1023, 1026, 1026, 1029, 1029, 1032, 1032, 1035, 
	1035, 1038, 1038, 3, 71, 71, 77, 77, 
	81, 81, 85, 85, 87, 87, 89, 89, 
	91, 91, 93, 93, 94, 94, 95, 95, 
	96, 96, 97, 97, 98, 98, 99, 99, 
	100, 100, 71, 77, 81, 85, 87, 89, 
	91, 93, 94, 95, 96, 97, 98, 99, 
	100, 69, 950, 968, 980, 992, 998, 1004, 
	1010, 1016, 1019, 1022, 1025, 1028, 1031, 1034, 
	1037, 70, 66, 66, 71, 71, 77, 77, 
	81, 81, 85, 85, 87, 87, 89, 89, 
	91, 91, 93, 93, 94, 94, 95, 95, 
	96, 96, 97, 97, 98, 98, 99, 99, 
	100, 100, 71, 77, 81, 85, 87, 89, 
	91, 93, 94, 95, 96, 97, 98, 99, 
	100, 67, 967, 979, 991, 997, 1003, 1009, 
	1015, 1018, 1021, 1024, 1027, 1030, 1033, 1036, 
	1039, 70, 6, 6, 8, 896, 896, 897, 
	5, 5, 7, 1040, 1040, 1041, 898, 897, 
	9, 7, 1042, 1041, 0, 35, 35, 36, 
	897, 897, 7, 7, 1041, 1041, 899, 37, 
	1043, 0, 38, 43, 43, 900, 900, 901, 
	41, 41, 42, 1044, 1044, 1045, 902, 901, 
	39, 42, 1046, 1045, 0, 903, 903, 904, 
	44, 44, 45, 1047, 1047, 1048, 904, 45, 
	1048, 0, 46, 46, 47, 904, 904, 45, 
	45, 1048, 1048, 905, 48, 1049, 0, 907, 
	907, 908, 52, 52, 53, 1050, 1050, 1051, 
	908, 53, 1051, 906, 54, 54, 55, 908, 
	908, 53, 53, 1051, 1051, 909, 56, 1052, 
	906, 10, 14, 14, 910, 910, 13, 13, 
	1053, 1053, 911, 11, 1054, 0, 16, 16, 
	17, 912, 912, 15, 15, 1055, 1055, 913, 
	18, 1056, 0, 23, 23, 24, 915, 915, 
	22, 22, 1057, 1057, 916, 25, 1058, 914, 
	68, 4, 4, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 919, 918, 69, 3, 
	1061, 1060, 0, 71, 71, 68, 71, 71, 
	4, 4, 951, 951, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 71, 71, 918, 
	3, 1060, 71, 919, 920, 69, 950, 1061, 
	1062, 70, 72, 72, 68, 72, 72, 4, 
	4, 953, 953, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 72, 72, 918, 3, 
	1060, 72, 919, 921, 69, 952, 1061, 1063, 
	70, 73, 73, 68, 73, 73, 4, 4, 
	955, 955, 917, 917, 918, 2, 2, 3, 
	1059, 1059, 1060, 73, 73, 918, 3, 1060, 
	73, 919, 922, 69, 954, 1061, 1064, 0, 
	74, 74, 68, 74, 74, 4, 4, 957, 
	957, 917, 917, 918, 2, 2, 3, 1059, 
	1059, 1060, 74, 74, 918, 3, 1060, 74, 
	919, 923, 69, 956, 1061, 1065, 0, 75, 
	75, 68, 75, 75, 4, 4, 959, 959, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 75, 75, 918, 3, 1060, 75, 919, 
	924, 69, 958, 1061, 1066, 0, 76, 76, 
	68, 76, 76, 4, 4, 961, 961, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	76, 76, 918, 3, 1060, 76, 919, 925, 
	69, 960, 1061, 1067, 0, 77, 77, 68, 
	77, 77, 4, 4, 969, 969, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 77, 
	77, 918, 3, 1060, 77, 919, 926, 69, 
	968, 1061, 1068, 70, 78, 78, 68, 78, 
	78, 4, 4, 971, 971, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 78, 78, 
	918, 3, 1060, 78, 919, 927, 69, 970, 
	1061, 1069, 70, 79, 79, 68, 79, 79, 
	4, 4, 973, 973, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 79, 79, 918, 
	3, 1060, 79, 919, 928, 69, 972, 1061, 
	1070, 0, 80, 80, 68, 80, 80, 4, 
	4, 975, 975, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 80, 80, 918, 3, 
	1060, 80, 919, 929, 69, 974, 1061, 1071, 
	0, 68, 71, 71, 77, 77, 81, 81, 
	71, 71, 77, 77, 81, 81, 4, 4, 
	951, 951, 969, 969, 981, 981, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 71, 
	71, 77, 77, 81, 81, 918, 3, 1060, 
	71, 77, 81, 919, 920, 926, 930, 69, 
	950, 968, 980, 1061, 1062, 1068, 1072, 70, 
	68, 72, 72, 78, 78, 82, 82, 72, 
	72, 78, 78, 82, 82, 4, 4, 953, 
	953, 971, 971, 983, 983, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 72, 72, 
	78, 78, 82, 82, 918, 3, 1060, 72, 
	78, 82, 919, 921, 927, 931, 69, 952, 
	970, 982, 1061, 1063, 1069, 1073, 70, 68, 
	73, 73, 79, 79, 83, 83, 73, 73, 
	79, 79, 83, 83, 4, 4, 955, 955, 
	973, 973, 985, 985, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 73, 73, 79, 
	79, 83, 83, 918, 3, 1060, 73, 79, 
	83, 919, 922, 928, 932, 69, 954, 972, 
	984, 1061, 1064, 1070, 1074, 0, 68, 74, 
	74, 80, 80, 84, 84, 74, 74, 80, 
	80, 84, 84, 4, 4, 957, 957, 975, 
	975, 987, 987, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 74, 74, 80, 80, 
	84, 84, 918, 3, 1060, 74, 80, 84, 
	919, 923, 929, 933, 69, 956, 974, 986, 
	1061, 1065, 1071, 1075, 0, 85, 85, 68, 
	85, 85, 4, 4, 993, 993, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 85, 
	85, 918, 3, 1060, 85, 919, 934, 69, 
	992, 1061, 1076, 70, 86, 86, 68, 86, 
	86, 4, 4, 995, 995, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 86, 86, 
	918, 3, 1060, 86, 919, 935, 69, 994, 
	1061, 1077, 70, 68, 71, 71, 85, 85, 
	87, 87, 71, 71, 85, 85, 87, 87, 
	4, 4, 951, 951, 993, 993, 999, 999, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 71, 71, 85, 85, 87, 87, 918, 
	3, 1060, 71, 85, 87, 919, 920, 934, 
	936, 69, 950, 992, 998, 1061, 1062, 1076, 
	1078, 70, 68, 72, 72, 86, 86, 88, 
	88, 72, 72, 86, 86, 88, 88, 4, 
	4, 953, 953, 995, 995, 1001, 1001, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	72, 72, 86, 86, 88, 88, 918, 3, 
	1060, 72, 86, 88, 919, 921, 935, 937, 
	69, 952, 994, 1000, 1061, 1063, 1077, 1079, 
	70, 68, 77, 77, 85, 85, 89, 89, 
	77, 77, 85, 85, 89, 89, 4, 4, 
	969, 969, 993, 993, 1005, 1005, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 77, 
	77, 85, 85, 89, 89, 918, 3, 1060, 
	77, 85, 89, 919, 926, 934, 938, 69, 
	968, 992, 1004, 1061, 1068, 1076, 1080, 70, 
	68, 78, 78, 86, 86, 90, 90, 78, 
	78, 86, 86, 90, 90, 4, 4, 971, 
	971, 995, 995, 1007, 1007, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 78, 78, 
	86, 86, 90, 90, 918, 3, 1060, 78, 
	86, 90, 919, 927, 935, 939, 69, 970, 
	994, 1006, 1061, 1069, 1077, 1081, 70, 68, 
	71, 71, 77, 77, 81, 81, 85, 85, 
	87, 87, 89, 89, 91, 91, 71, 71, 
	77, 77, 81, 81, 85, 85, 87, 87, 
	89, 89, 91, 91, 4, 4, 951, 951, 
	969, 969, 981, 981, 993, 993, 999, 999, 
	1005, 1005, 1011, 1011, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 71, 71, 77, 
	77, 81, 81, 85, 85, 87, 87, 89, 
	89, 91, 91, 918, 3, 1060, 71, 77, 
	81, 85, 87, 89, 91, 919, 920, 926, 
	930, 934, 936, 938, 940, 69, 950, 968, 
	980, 992, 998, 1004, 1010, 1061, 1062, 1068, 
	1072, 1076, 1078, 1080, 1082, 70, 68, 72, 
	72, 78, 78, 82, 82, 86, 86, 88, 
	88, 90, 90, 92, 92, 72, 72, 78, 
	78, 82, 82, 86, 86, 88, 88, 90, 
	90, 92, 92, 4, 4, 953, 953, 971, 
	971, 983, 983, 995, 995, 1001, 1001, 1007, 
	1007, 1013, 1013, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 72, 72, 78, 78, 
	82, 82, 86, 86, 88, 88, 90, 90, 
	92, 92, 918, 3, 1060, 72, 78, 82, 
	86, 88, 90, 92, 919, 921, 927, 931, 
	935, 937, 939, 941, 69, 952, 970, 982, 
	994, 1000, 1006, 1012, 1061, 1063, 1069, 1073, 
	1077, 1079, 1081, 1083, 70, 93, 93, 68, 
	93, 93, 4, 4, 1017, 1017, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 93, 
	93, 918, 3, 1060, 93, 919, 942, 69, 
	1016, 1061, 1084, 70, 68, 71, 71, 93, 
	93, 94, 94, 71, 71, 93, 93, 94, 
	94, 4, 4, 951, 951, 1017, 1017, 1020, 
	1020, 917, 917, 918, 2, 2, 3, 1059, 
	1059, 1060, 71, 71, 93, 93, 94, 94, 
	918, 3, 1060, 71, 93, 94, 919, 920, 
	942, 943, 69, 950, 1016, 1019, 1061, 1062, 
	1084, 1085, 70, 68, 77, 77, 93, 93, 
	95, 95, 77, 77, 93, 93, 95, 95, 
	4, 4, 969, 969, 1017, 1017, 1023, 1023, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 77, 77, 93, 93, 95, 95, 918, 
	3, 1060, 77, 93, 95, 919, 926, 942, 
	944, 69, 968, 1016, 1022, 1061, 1068, 1084, 
	1086, 70, 68, 71, 71, 77, 77, 81, 
	81, 93, 93, 94, 94, 95, 95, 96, 
	96, 71, 71, 77, 77, 81, 81, 93, 
	93, 94, 94, 95, 95, 96, 96, 4, 
	4, 951, 951, 969, 969, 981, 981, 1017, 
	1017, 1020, 1020, 1023, 1023, 1026, 1026, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	71, 71, 77, 77, 81, 81, 93, 93, 
	94, 94, 95, 95, 96, 96, 918, 3, 
	1060, 71, 77, 81, 93, 94, 95, 96, 
	919, 920, 926, 930, 942, 943, 944, 945, 
	69, 950, 968, 980, 1016, 1019, 1022, 1025, 
	1061, 1062, 1068, 1072, 1084, 1085, 1086, 1087, 
	70, 68, 85, 85, 93, 93, 97, 97, 
	85, 85, 93, 93, 97, 97, 4, 4, 
	993, 993, 1017, 1017, 1029, 1029, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 85, 
	85, 93, 93, 97, 97, 918, 3, 1060, 
	85, 93, 97, 919, 934, 942, 946, 69, 
	992, 1016, 1028, 1061, 1076, 1084, 1088, 70, 
	68, 71, 71, 85, 85, 87, 87, 93, 
	93, 94, 94, 97, 97, 98, 98, 71, 
	71, 85, 85, 87, 87, 93, 93, 94, 
	94, 97, 97, 98, 98, 4, 4, 951, 
	951, 993, 993, 999, 999, 1017, 1017, 1020, 
	1020, 1029, 1029, 1032, 1032, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 71, 71, 
	85, 85, 87, 87, 93, 93, 94, 94, 
	97, 97, 98, 98, 918, 3, 1060, 71, 
	85, 87, 93, 94, 97, 98, 919, 920, 
	934, 936, 942, 943, 946, 947, 69, 950, 
	992, 998, 1016, 1019, 1028, 1031, 1061, 1062, 
	1076, 1078, 1084, 1085, 1088, 1089, 70, 68, 
	77, 77, 85, 85, 89, 89, 93, 93, 
	95, 95, 97, 97, 99, 99, 77, 77, 
	85, 85, 89, 89, 93, 93, 95, 95, 
	97, 97, 99, 99, 4, 4, 969, 969, 
	993, 993, 1005, 1005, 1017, 1017, 1023, 1023, 
	1029, 1029, 1035, 1035, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 77, 77, 85, 
	85, 89, 89, 93, 93, 95, 95, 97, 
	97, 99, 99, 918, 3, 1060, 77, 85, 
	89, 93, 95, 97, 99, 919, 926, 934, 
	938, 942, 944, 946, 948, 69, 968, 992, 
	1004, 1016, 1022, 1028, 1034, 1061, 1068, 1076, 
	1080, 1084, 1086, 1088, 1090, 70, 68, 71, 
	71, 77, 77, 81, 81, 85, 85, 87, 
	87, 89, 89, 91, 91, 93, 93, 94, 
	94, 95, 95, 96, 96, 97, 97, 98, 
	98, 99, 99, 100, 100, 71, 71, 77, 
	77, 81, 81, 85, 85, 87, 87, 89, 
	89, 91, 91, 93, 93, 94, 94, 95, 
	95, 96, 96, 97, 97, 98, 98, 99, 
	99, 100, 100, 4, 4, 951, 951, 969, 
	969, 981, 981, 993, 993, 999, 999, 1005, 
	1005, 1011, 1011, 1017, 1017, 1020, 1020, 1023, 
	1023, 1026, 1026, 1029, 1029, 1032, 1032, 1035, 
	1035, 1038, 1038, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 71, 71, 77, 77, 
	81, 81, 85, 85, 87, 87, 89, 89, 
	91, 91, 93, 93, 94, 94, 95, 95, 
	96, 96, 97, 97, 98, 98, 99, 99, 
	100, 100, 71, 77, 81, 85, 87, 89, 
	91, 93, 94, 95, 96, 97, 98, 99, 
	100, 919, 920, 926, 930, 934, 936, 938, 
	940, 942, 943, 944, 945, 946, 947, 948, 
	949, 69, 950, 968, 980, 992, 998, 1004, 
	1010, 1016, 1019, 1022, 1025, 1028, 1031, 1034, 
	1037, 1061, 1062, 1068, 1072, 1076, 1078, 1080, 
	1082, 1084, 1085, 1086, 1087, 1088, 1089, 1090, 
	1091, 918, 3, 1060, 70, 1092, 1092, 1092, 
	0, 1093, 1093, 1093, 0, 1094, 1094, 1094, 
	0, 1095, 1095, 1095, 0, 1096, 1096, 1096, 
	0, 1097, 1097, 1097, 0, 1098, 1098, 1098, 
	0, 1099, 1099, 1099, 0, 1100, 1100, 1100, 
	0, 1101, 1101, 1101, 0, 1102, 1102, 1102, 
	0, 1103, 1103, 1103, 0, 1092, 1092, 1092, 
	1099, 1099, 1099, 1104, 1104, 1104, 0, 1093, 
	1093, 1093, 1100, 1100, 1100, 1105, 1105, 1105, 
	0, 1094, 1094, 1094, 1101, 1101, 1101, 1106, 
	1106, 1106, 0, 1095, 1095, 1095, 1102, 1102, 
	1102, 1107, 1107, 1107, 0, 1096, 1096, 1096, 
	1103, 1103, 1103, 1108, 1108, 1108, 0, 1109, 
	1109, 1109, 0, 1110, 1110, 1110, 0, 1111, 
	1111, 1111, 0, 1092, 1092, 1092, 1109, 1109, 
	1109, 1112, 1112, 1112, 0, 1093, 1093, 1093, 
	1110, 1110, 1110, 1113, 1113, 1113, 0, 1094, 
	1094, 1094, 1111, 1111, 1111, 1114, 1114, 1114, 
	0, 1099, 1099, 1099, 1109, 1109, 1109, 1115, 
	1115, 1115, 0, 1100, 1100, 1100, 1110, 1110, 
	1110, 1116, 1116, 1116, 0, 1101, 1101, 1101, 
	1111, 1111, 1111, 1117, 1117, 1117, 0, 1092, 
	1092, 1092, 1099, 1099, 1099, 1104, 1104, 1104, 
	1109, 1109, 1109, 1112, 1112, 1112, 1115, 1115, 
	1115, 1118, 1118, 1118, 0, 1093, 1093, 1093, 
	1100, 1100, 1100, 1105, 1105, 1105, 1110, 1110, 
	1110, 1113, 1113, 1113, 1116, 1116, 1116, 1119, 
	1119, 1119, 0, 1094, 1094, 1094, 1101, 1101, 
	1101, 1106, 1106, 1106, 1111, 1111, 1111, 1114, 
	1114, 1114, 1117, 1117, 1117, 1120, 1120, 1120, 
	0, 1121, 1121, 1121, 0, 1122, 1122, 1122, 
	0, 1092, 1092, 1092, 1121, 1121, 1121, 1123, 
	1123, 1123, 0, 1093, 1093, 1093, 1122, 1122, 
	1122, 1124, 1124, 1124, 0, 1099, 1099, 1099, 
	1121, 1121, 1121, 1125, 1125, 1125, 0, 1100, 
	1100, 1100, 1122, 1122, 1122, 1126, 1126, 1126, 
	0, 1092, 1092, 1092, 1099, 1099, 1099, 1104, 
	1104, 1104, 1121, 1121, 1121, 1123, 1123, 1123, 
	1125, 1125, 1125, 1127, 1127, 1127, 0, 1093, 
	1093, 1093, 1100, 1100, 1100, 1105, 1105, 1105, 
	1122, 1122, 1122, 1124, 1124, 1124, 1126, 1126, 
	1126, 1128, 1128, 1128, 0, 1109, 1109, 1109, 
	1121, 1121, 1121, 1129, 1129, 1129, 0, 1110, 
	1110, 1110, 1122, 1122, 1122, 1130, 1130, 1130, 
	0, 1092, 1092, 1092, 1109, 1109, 1109, 1112, 
	1112, 1112, 1121, 1121, 1121, 1123, 1123, 1123, 
	1129, 1129, 1129, 1131, 1131, 1131, 0, 1093, 
	1093, 1093, 1110, 1110, 1110, 1113, 1113, 1113, 
	1122, 1122, 1122, 1124, 1124, 1124, 1130, 1130, 
	1130, 1132, 1132, 1132, 0, 1099, 1099, 1099, 
	1109, 1109, 1109, 1115, 1115, 1115, 1121, 1121, 
	1121, 1125, 1125, 1125, 1129, 1129, 1129, 1133, 
	1133, 1133, 0, 1100, 1100, 1100, 1110, 1110, 
	1110, 1116, 1116, 1116, 1122, 1122, 1122, 1126, 
	1126, 1126, 1130, 1130, 1130, 1134, 1134, 1134, 
	0, 1092, 1092, 1092, 1099, 1099, 1099, 1104, 
	1104, 1104, 1109, 1109, 1109, 1112, 1112, 1112, 
	1115, 1115, 1115, 1118, 1118, 1118, 1121, 1121, 
	1121, 1123, 1123, 1123, 1125, 1125, 1125, 1127, 
	1127, 1127, 1129, 1129, 1129, 1131, 1131, 1131, 
	1133, 1133, 1133, 1135, 1135, 1135, 0, 1093, 
	1093, 1093, 1100, 1100, 1100, 1105, 1105, 1105, 
	1110, 1110, 1110, 1113, 1113, 1113, 1116, 1116, 
	1116, 1119, 1119, 1119, 1122, 1122, 1122, 1124, 
	1124, 1124, 1126, 1126, 1126, 1128, 1128, 1128, 
	1130, 1130, 1130, 1132, 1132, 1132, 1134, 1134, 
	1134, 1136, 1136, 1136, 0, 1138, 1139, 1137, 
	1137, 1137, 1137, 1141, 1141, 1142, 1137, 1137, 
	1158, 1158, 1159, 1159, 1160, 1160, 1137, 1137, 
	1161, 1161, 1162, 1162, 1163, 1163, 1137, 1137, 
	1164, 1164, 1165, 1165, 1166, 1166, 1167, 1167, 
	1168, 1168, 1169, 1169, 1170, 1170, 1171, 1171, 
	1172, 1172, 1173, 1173, 1174, 1174, 1175, 1175, 
	1176, 1176, 1177, 1177, 1178, 1178, 1137, 1227, 
	1228, 1229, 1137, 1230, 1231, 1232, 1233, 1234, 
	1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242, 
	1243, 1244, 1245, 1246, 1247, 1248, 1249, 1250, 
	1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 
	1259, 1260, 1137, 1137, 1137, 1137, 1137, 1137, 
	1137, 1137, 1137, 1137, 1137, 1137, 1143, 1143, 
	1144, 1144, 1145, 1145, 1146, 1146, 1147, 1147, 
	1148, 1148, 1149, 1149, 1150, 1150, 1151, 1151, 
	1152, 1152, 1153, 1153, 1154, 1154, 1155, 1155, 
	1156, 1156, 1157, 1157, 1137, 1143, 1144, 1145, 
	1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153, 
	1154, 1155, 1156, 1157, 1179, 1180, 1181, 1182, 
	1183, 1184, 1185, 1186, 1187, 1188, 1189, 1190, 
	1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198, 
	1199, 1200, 1201, 1202, 1203, 1204, 1205, 1206, 
	1207, 1208, 1209, 1210, 1211, 1212, 1213, 1214, 
	1215, 1216, 1217, 1218, 1219, 1220, 1221, 1222, 
	1223, 1224, 1225, 1226, 1140, 1138, 1261, 68, 
	69, 1261, 1263, 1263, 1264, 1264, 21, 1262, 
	1265, 1265, 28, 1262, 30, 1262, 26, 1265, 
	1265, 27, 1262, 1263, 1263, 32, 1262, 1263, 
	1263, 19, 1264, 1264, 20, 1262, 1267, 1267, 
	1268, 1269, 1269, 51, 1268, 1266, 1270, 1270, 
	59, 1266, 61, 1266, 57, 1270, 1270, 58, 
	1266, 1267, 1267, 1268, 63, 1268, 1266, 1267, 
	1267, 1268, 49, 1269, 1269, 50, 1268, 1266, 
	1, 1261, 1271, 1271, 1271, 1261, 1272, 1272, 
	1272, 1261, 1271, 1271, 1271, 1272, 1272, 1272, 
	1273, 1273, 1273, 1261, 75, 75, 75, 1274, 
	1275, 1275, 1275, 1261, 1271, 1271, 1271, 1275, 
	1275, 1275, 1276, 1276, 1276, 1261, 73, 73, 
	73, 1277, 1272, 1272, 1272, 1275, 1275, 1275, 
	1278, 1278, 1278, 1261, 79, 79, 79, 1277, 
	1271, 1271, 1271, 1272, 1272, 1272, 1273, 1273, 
	1273, 1275, 1275, 1275, 1276, 1276, 1276, 1278, 
	1278, 1278, 1279, 1279, 1279, 1261, 73, 73, 
	73, 79, 79, 79, 83, 83, 83, 1277, 
	1280, 1280, 1280, 1261, 1271, 1271, 1271, 1280, 
	1280, 1280, 1281, 1281, 1281, 1261, 72, 72, 
	72, 1282, 1272, 1272, 1272, 1280, 1280, 1280, 
	1283, 1283, 1283, 1261, 78, 78, 78, 1282, 
	1271, 1271, 1271, 1272, 1272, 1272, 1273, 1273, 
	1273, 1280, 1280, 1280, 1281, 1281, 1281, 1283, 
	1283, 1283, 1284, 1284, 1284, 1261, 72, 72, 
	72, 78, 78, 78, 82, 82, 82, 1282, 
	1275, 1275, 1275, 1280, 1280, 1280, 1285, 1285, 
	1285, 1261, 86, 86, 86, 1282, 1271, 1271, 
	1271, 1275, 1275, 1275, 1276, 1276, 1276, 1280, 
	1280, 1280, 1281, 1281, 1281, 1285, 1285, 1285, 
	1286, 1286, 1286, 1261, 72, 72, 72, 86, 
	86, 86, 88, 88, 88, 1282, 1272, 1272, 
	1272, 1275, 1275, 1275, 1278, 1278, 1278, 1280, 
	1280, 1280, 1283, 1283, 1283, 1285, 1285, 1285, 
	1287, 1287, 1287, 1261, 78, 78, 78, 86, 
	86, 86, 90, 90, 90, 1282, 1271, 1271, 
	1271, 1272, 1272, 1272, 1273, 1273, 1273, 1275, 
	1275, 1275, 1276, 1276, 1276, 1278, 1278, 1278, 
	1279, 1279, 1279, 1280, 1280, 1280, 1281, 1281, 
	1281, 1283, 1283, 1283, 1284, 1284, 1284, 1285, 
	1285, 1285, 1286, 1286, 1286, 1287, 1287, 1287, 
	1288, 1288, 1288, 1261, 72, 72, 72, 78, 
	78, 78, 82, 82, 82, 86, 86, 86, 
	88, 88, 88, 90, 90, 90, 92, 92, 
	92, 1282, 1289, 1289, 1261, 1290, 1290, 1261, 
	1289, 1289, 1290, 1290, 1291, 1291, 1261, 1292, 
	1292, 1261, 1293, 1293, 1261, 1292, 1292, 1293, 
	1293, 1294, 1294, 1261, 1295, 1295, 1261, 1296, 
	1296, 1261, 1295, 1295, 1296, 1296, 1297, 1297, 
	1261, 1298, 1298, 1261, 1295, 1295, 1298, 1298, 
	1299, 1299, 1261, 1296, 1296, 1298, 1298, 1300, 
	1300, 1261, 1295, 1295, 1296, 1296, 1297, 1297, 
	1298, 1298, 1299, 1299, 1300, 1300, 1301, 1301, 
	1261, 1302, 1302, 1261, 1295, 1295, 1302, 1302, 
	1303, 1303, 1261, 1296, 1296, 1302, 1302, 1304, 
	1304, 1261, 1295, 1295, 1296, 1296, 1297, 1297, 
	1302, 1302, 1303, 1303, 1304, 1304, 1305, 1305, 
	1261, 1298, 1298, 1302, 1302, 1306, 1306, 1261, 
	1295, 1295, 1298, 1298, 1299, 1299, 1302, 1302, 
	1303, 1303, 1306, 1306, 1307, 1307, 1261, 1296, 
	1296, 1298, 1298, 1300, 1300, 1302, 1302, 1304, 
	1304, 1306, 1306, 1308, 1308, 1261, 1295, 1295, 
	1296, 1296, 1297, 1297, 1298, 1298, 1299, 1299, 
	1300, 1300, 1301, 1301, 1302, 1302, 1303, 1303, 
	1304, 1304, 1305, 1305, 1306, 1306, 1307, 1307, 
	1308, 1308, 1309, 1309, 1261, 917, 917, 918, 
	919, 918, 0, 1311, 1311, 1312, 1313, 1312, 
	1310, 1314, 1310, 1316, 1316, 1317, 1315, 1318, 
	1315, 917, 917, 918, 1271, 1271, 918, 1271, 
	919, 1319, 1261, 917, 917, 918, 1272, 1272, 
	918, 1272, 919, 1320, 1261, 917, 917, 918, 
	918, 1271, 1271, 1272, 1272, 1273, 1273, 1271, 
	1272, 1273, 919, 1319, 1320, 1321, 1261, 917, 
	917, 918, 75, 75, 918, 75, 919, 924, 
	1274, 917, 917, 918, 1275, 1275, 918, 1275, 
	919, 1322, 1261, 917, 917, 918, 918, 1271, 
	1271, 1275, 1275, 1276, 1276, 1271, 1275, 1276, 
	919, 1319, 1322, 1323, 1261, 917, 917, 918, 
	73, 73, 918, 73, 919, 922, 1277, 917, 
	917, 918, 918, 1272, 1272, 1275, 1275, 1278, 
	1278, 1272, 1275, 1278, 919, 1320, 1322, 1324, 
	1261, 917, 917, 918, 79, 79, 918, 79, 
	919, 928, 1277, 917, 917, 918, 918, 1271, 
	1271, 1272, 1272, 1273, 1273, 1275, 1275, 1276, 
	1276, 1278, 1278, 1279, 1279, 1271, 1272, 1273, 
	1275, 1276, 1278, 1279, 919, 1319, 1320, 1321, 
	1322, 1323, 1324, 1325, 1261, 917, 917, 918, 
	918, 73, 73, 79, 79, 83, 83, 73, 
	79, 83, 919, 922, 928, 932, 1277, 917, 
	917, 918, 1280, 1280, 918, 1280, 919, 1326, 
	1261, 917, 917, 918, 918, 1271, 1271, 1280, 
	1280, 1281, 1281, 1271, 1280, 1281, 919, 1319, 
	1326, 1327, 1261, 917, 917, 918, 72, 72, 
	918, 72, 919, 921, 1282, 917, 917, 918, 
	918, 1272, 1272, 1280, 1280, 1283, 1283, 1272, 
	1280, 1283, 919, 1320, 1326, 1328, 1261, 917, 
	917, 918, 78, 78, 918, 78, 919, 927, 
	1282, 917, 917, 918, 918, 1271, 1271, 1272, 
	1272, 1273, 1273, 1280, 1280, 1281, 1281, 1283, 
	1283, 1284, 1284, 1271, 1272, 1273, 1280, 1281, 
	1283, 1284, 919, 1319, 1320, 1321, 1326, 1327, 
	1328, 1329, 1261, 917, 917, 918, 918, 72, 
	72, 78, 78, 82, 82, 72, 78, 82, 
	919, 921, 927, 931, 1282, 917, 917, 918, 
	918, 1275, 1275, 1280, 1280, 1285, 1285, 1275, 
	1280, 1285, 919, 1322, 1326, 1330, 1261, 917, 
	917, 918, 86, 86, 918, 86, 919, 935, 
	1282, 917, 917, 918, 918, 1271, 1271, 1275, 
	1275, 1276, 1276, 1280, 1280, 1281, 1281, 1285, 
	1285, 1286, 1286, 1271, 1275, 1276, 1280, 1281, 
	1285, 1286, 919, 1319, 1322, 1323, 1326, 1327, 
	1330, 1331, 1261, 917, 917, 918, 918, 72, 
	72, 86, 86, 88, 88, 72, 86, 88, 
	919, 921, 935, 937, 1282, 917, 917, 918, 
	918, 1272, 1272, 1275, 1275, 1278, 1278, 1280, 
	1280, 1283, 1283, 1285, 1285, 1287, 1287, 1272, 
	1275, 1278, 1280, 1283, 1285, 1287, 919, 1320, 
	1322, 1324, 1326, 1328, 1330, 1332, 1261, 917, 
	917, 918, 918, 78, 78, 86, 86, 90, 
	90, 78, 86, 90, 919, 927, 935, 939, 
	1282, 917, 917, 918, 918, 1271, 1271, 1272, 
	1272, 1273, 1273, 1275, 1275, 1276, 1276, 1278, 
	1278, 1279, 1279, 1280, 1280, 1281, 1281, 1283, 
	1283, 1284, 1284, 1285, 1285, 1286, 1286, 1287, 
	1287, 1288, 1288, 1271, 1272, 1273, 1275, 1276, 
	1278, 1279, 1280, 1281, 1283, 1284, 1285, 1286, 
	1287, 1288, 919, 1319, 1320, 1321, 1322, 1323, 
	1324, 1325, 1326, 1327, 1328, 1329, 1330, 1331, 
	1332, 1333, 1261, 917, 917, 918, 918, 72, 
	72, 78, 78, 82, 82, 86, 86, 88, 
	88, 90, 90, 92, 92, 72, 78, 82, 
	86, 88, 90, 92, 919, 921, 927, 931, 
	935, 937, 939, 941, 1282, 2, 2, 3, 
	68, 4, 4, 69, 3, 0, 1271, 1271, 
	2, 2, 3, 68, 1271, 1271, 4, 4, 
	1335, 1335, 1271, 1271, 3, 1271, 69, 1334, 
	1261, 66, 66, 67, 0, 2, 2, 3, 
	67, 3, 0, 1272, 1272, 2, 2, 3, 
	68, 1272, 1272, 4, 4, 1337, 1337, 1272, 
	1272, 3, 1272, 69, 1336, 1261, 2, 2, 
	3, 68, 1271, 1271, 1272, 1272, 1273, 1273, 
	1271, 1271, 1272, 1272, 1273, 1273, 4, 4, 
	1335, 1335, 1337, 1337, 1339, 1339, 3, 1271, 
	1271, 1272, 1272, 1273, 1273, 1271, 1272, 1273, 
	69, 1334, 1336, 1338, 1261, 75, 75, 2, 
	2, 3, 68, 75, 75, 4, 4, 959, 
	959, 75, 75, 3, 75, 69, 958, 1274, 
	66, 66, 75, 75, 75, 67, 963, 1274, 
	2, 2, 3, 75, 75, 3, 75, 67, 
	963, 1274, 1275, 1275, 2, 2, 3, 68, 
	1275, 1275, 4, 4, 1341, 1341, 1275, 1275, 
	3, 1275, 69, 1340, 1261, 2, 2, 3, 
	68, 1271, 1271, 1275, 1275, 1276, 1276, 1271, 
	1271, 1275, 1275, 1276, 1276, 4, 4, 1335, 
	1335, 1341, 1341, 1343, 1343, 3, 1271, 1271, 
	1275, 1275, 1276, 1276, 1271, 1275, 1276, 69, 
	1334, 1340, 1342, 1261, 73, 73, 2, 2, 
	3, 68, 73, 73, 4, 4, 955, 955, 
	73, 73, 3, 73, 69, 954, 1277, 66, 
	66, 73, 73, 73, 67, 965, 1277, 2, 
	2, 3, 73, 73, 3, 73, 67, 965, 
	1277, 2, 2, 3, 68, 1272, 1272, 1275, 
	1275, 1278, 1278, 1272, 1272, 1275, 1275, 1278, 
	1278, 4, 4, 1337, 1337, 1341, 1341, 1345, 
	1345, 3, 1272, 1272, 1275, 1275, 1278, 1278, 
	1272, 1275, 1278, 69, 1336, 1340, 1344, 1261, 
	79, 79, 2, 2, 3, 68, 79, 79, 
	4, 4, 973, 973, 79, 79, 3, 79, 
	69, 972, 1277, 66, 66, 79, 79, 79, 
	67, 977, 1277, 2, 2, 3, 79, 79, 
	3, 79, 67, 977, 1277, 2, 2, 3, 
	68, 1271, 1271, 1272, 1272, 1273, 1273, 1275, 
	1275, 1276, 1276, 1278, 1278, 1279, 1279, 1271, 
	1271, 1272, 1272, 1273, 1273, 1275, 1275, 1276, 
	1276, 1278, 1278, 1279, 1279, 4, 4, 1335, 
	1335, 1337, 1337, 1339, 1339, 1341, 1341, 1343, 
	1343, 1345, 1345, 1347, 1347, 3, 1271, 1271, 
	1272, 1272, 1273, 1273, 1275, 1275, 1276, 1276, 
	1278, 1278, 1279, 1279, 1271, 1272, 1273, 1275, 
	1276, 1278, 1279, 69, 1334, 1336, 1338, 1340, 
	1342, 1344, 1346, 1261, 2, 2, 3, 68, 
	73, 73, 79, 79, 83, 83, 73, 73, 
	79, 79, 83, 83, 4, 4, 955, 955, 
	973, 973, 985, 985, 3, 73, 73, 79, 
	79, 83, 83, 73, 79, 83, 69, 954, 
	972, 984, 1277, 66, 66, 73, 73, 79, 
	79, 83, 83, 73, 79, 83, 67, 965, 
	977, 989, 1277, 2, 2, 3, 3, 73, 
	73, 79, 79, 83, 83, 73, 79, 83, 
	67, 965, 977, 989, 1277, 1280, 1280, 2, 
	2, 3, 68, 1280, 1280, 4, 4, 1349, 
	1349, 1280, 1280, 3, 1280, 69, 1348, 1261, 
	2, 2, 3, 68, 1271, 1271, 1280, 1280, 
	1281, 1281, 1271, 1271, 1280, 1280, 1281, 1281, 
	4, 4, 1335, 1335, 1349, 1349, 1351, 1351, 
	3, 1271, 1271, 1280, 1280, 1281, 1281, 1271, 
	1280, 1281, 69, 1334, 1348, 1350, 1261, 72, 
	72, 2, 2, 3, 68, 72, 72, 4, 
	4, 953, 953, 72, 72, 3, 72, 69, 
	952, 1282, 66, 66, 72, 72, 72, 67, 
	966, 1282, 2, 2, 3, 72, 72, 3, 
	72, 67, 966, 1282, 2, 2, 3, 68, 
	1272, 1272, 1280, 1280, 1283, 1283, 1272, 1272, 
	1280, 1280, 1283, 1283, 4, 4, 1337, 1337, 
	1349, 1349, 1353, 1353, 3, 1272, 1272, 1280, 
	1280, 1283, 1283, 1272, 1280, 1283, 69, 1336, 
	1348, 1352, 1261, 78, 78, 2, 2, 3, 
	68, 78, 78, 4, 4, 971, 971, 78, 
	78, 3, 78, 69, 970, 1282, 66, 66, 
	78, 78, 78, 67, 978, 1282, 2, 2, 
	3, 78, 78, 3, 78, 67, 978, 1282, 
	2, 2, 3, 68, 1271, 1271, 1272, 1272, 
	1273, 1273, 1280, 1280, 1281, 1281, 1283, 1283, 
	1284, 1284, 1271, 1271, 1272, 1272, 1273, 1273, 
	1280, 1280, 1281, 1281, 1283, 1283, 1284, 1284, 
	4, 4, 1335, 1335, 1337, 1337, 1339, 1339, 
	1349, 1349, 1351, 1351, 1353, 1353, 1355, 1355, 
	3, 1271, 1271, 1272, 1272, 1273, 1273, 1280, 
	1280, 1281, 1281, 1283, 1283, 1284, 1284, 1271, 
	1272, 1273, 1280, 1281, 1283, 1284, 69, 1334, 
	1336, 1338, 1348, 1350, 1352, 1354, 1261, 2, 
	2, 3, 68, 72, 72, 78, 78, 82, 
	82, 72, 72, 78, 78, 82, 82, 4, 
	4, 953, 953, 971, 971, 983, 983, 3, 
	72, 72, 78, 78, 82, 82, 72, 78, 
	82, 69, 952, 970, 982, 1282, 66, 66, 
	72, 72, 78, 78, 82, 82, 72, 78, 
	82, 67, 966, 978, 990, 1282, 2, 2, 
	3, 3, 72, 72, 78, 78, 82, 82, 
	72, 78, 82, 67, 966, 978, 990, 1282, 
	2, 2, 3, 68, 1275, 1275, 1280, 1280, 
	1285, 1285, 1275, 1275, 1280, 1280, 1285, 1285, 
	4, 4, 1341, 1341, 1349, 1349, 1357, 1357, 
	3, 1275, 1275, 1280, 1280, 1285, 1285, 1275, 
	1280, 1285, 69, 1340, 1348, 1356, 1261, 86, 
	86, 2, 2, 3, 68, 86, 86, 4, 
	4, 995, 995, 86, 86, 3, 86, 69, 
	994, 1282, 66, 66, 86, 86, 86, 67, 
	996, 1282, 2, 2, 3, 86, 86, 3, 
	86, 67, 996, 1282, 2, 2, 3, 68, 
	1271, 1271, 1275, 1275, 1276, 1276, 1280, 1280, 
	1281, 1281, 1285, 1285, 1286, 1286, 1271, 1271, 
	1275, 1275, 1276, 1276, 1280, 1280, 1281, 1281, 
	1285, 1285, 1286, 1286, 4, 4, 1335, 1335, 
	1341, 1341, 1343, 1343, 1349, 1349, 1351, 1351, 
	1357, 1357, 1359, 1359, 3, 1271, 1271, 1275, 
	1275, 1276, 1276, 1280, 1280, 1281, 1281, 1285, 
	1285, 1286, 1286, 1271, 1275, 1276, 1280, 1281, 
	1285, 1286, 69, 1334, 1340, 1342, 1348, 1350, 
	1356, 1358, 1261, 2, 2, 3, 68, 72, 
	72, 86, 86, 88, 88, 72, 72, 86, 
	86, 88, 88, 4, 4, 953, 953, 995, 
	995, 1001, 1001, 3, 72, 72, 86, 86, 
	88, 88, 72, 86, 88, 69, 952, 994, 
	1000, 1282, 66, 66, 72, 72, 86, 86, 
	88, 88, 72, 86, 88, 67, 966, 996, 
	1002, 1282, 2, 2, 3, 3, 72, 72, 
	86, 86, 88, 88, 72, 86, 88, 67, 
	966, 996, 1002, 1282, 2, 2, 3, 68, 
	1272, 1272, 1275, 1275, 1278, 1278, 1280, 1280, 
	1283, 1283, 1285, 1285, 1287, 1287, 1272, 1272, 
	1275, 1275, 1278, 1278, 1280, 1280, 1283, 1283, 
	1285, 1285, 1287, 1287, 4, 4, 1337, 1337, 
	1341, 1341, 1345, 1345, 1349, 1349, 1353, 1353, 
	1357, 1357, 1361, 1361, 3, 1272, 1272, 1275, 
	1275, 1278, 1278, 1280, 1280, 1283, 1283, 1285, 
	1285, 1287, 1287, 1272, 1275, 1278, 1280, 1283, 
	1285, 1287, 69, 1336, 1340, 1344, 1348, 1352, 
	1356, 1360, 1261, 2, 2, 3, 68, 78, 
	78, 86, 86, 90, 90, 78, 78, 86, 
	86, 90, 90, 4, 4, 971, 971, 995, 
	995, 1007, 1007, 3, 78, 78, 86, 86, 
	90, 90, 78, 86, 90, 69, 970, 994, 
	1006, 1282, 66, 66, 78, 78, 86, 86, 
	90, 90, 78, 86, 90, 67, 978, 996, 
	1008, 1282, 2, 2, 3, 3, 78, 78, 
	86, 86, 90, 90, 78, 86, 90, 67, 
	978, 996, 1008, 1282, 2, 2, 3, 68, 
	1271, 1271, 1272, 1272, 1273, 1273, 1275, 1275, 
	1276, 1276, 1278, 1278, 1279, 1279, 1280, 1280, 
	1281, 1281, 1283, 1283, 1284, 1284, 1285, 1285, 
	1286, 1286, 1287, 1287, 1288, 1288, 1271, 1271, 
	1272, 1272, 1273, 1273, 1275, 1275, 1276, 1276, 
	1278, 1278, 1279, 1279, 1280, 1280, 1281, 1281, 
	1283, 1283, 1284, 1284, 1285, 1285, 1286, 1286, 
	1287, 1287, 1288, 1288, 4, 4, 1335, 1335, 
	1337, 1337, 1339, 1339, 1341, 1341, 1343, 1343, 
	1345, 1345, 1347, 1347, 1349, 1349, 1351, 1351, 
	1353, 1353, 1355, 1355, 1357, 1357, 1359, 1359, 
	1361, 1361, 1363, 1363, 3, 1271, 1271, 1272, 
	1272, 1273, 1273, 1275, 1275, 1276, 1276, 1278, 
	1278, 1279, 1279, 1280, 1280, 1281, 1281, 1283, 
	1283, 1284, 1284, 1285, 1285, 1286, 1286, 1287, 
	1287, 1288, 1288, 1271, 1272, 1273, 1275, 1276, 
	1278, 1279, 1280, 1281, 1283, 1284, 1285, 1286, 
	1287, 1288, 69, 1334, 1336, 1338, 1340, 1342, 
	1344, 1346, 1348, 1350, 1352, 1354, 1356, 1358, 
	1360, 1362, 1261, 2, 2, 3, 68, 72, 
	72, 78, 78, 82, 82, 86, 86, 88, 
	88, 90, 90, 92, 92, 72, 72, 78, 
	78, 82, 82, 86, 86, 88, 88, 90, 
	90, 92, 92, 4, 4, 953, 953, 971, 
	971, 983, 983, 995, 995, 1001, 1001, 1007, 
	1007, 1013, 1013, 3, 72, 72, 78, 78, 
	82, 82, 86, 86, 88, 88, 90, 90, 
	92, 92, 72, 78, 82, 86, 88, 90, 
	92, 69, 952, 970, 982, 994, 1000, 1006, 
	1012, 1282, 66, 66, 72, 72, 78, 78, 
	82, 82, 86, 86, 88, 88, 90, 90, 
	92, 92, 72, 78, 82, 86, 88, 90, 
	92, 67, 966, 978, 990, 996, 1002, 1008, 
	1014, 1282, 2, 2, 3, 3, 72, 72, 
	78, 78, 82, 82, 86, 86, 88, 88, 
	90, 90, 92, 92, 72, 78, 82, 86, 
	88, 90, 92, 67, 966, 978, 990, 996, 
	1002, 1008, 1014, 1282, 68, 4, 4, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	919, 918, 69, 3, 1061, 1060, 0, 1364, 
	1364, 1364, 1364, 1364, 1367, 1368, 1368, 1364, 
	1364, 1364, 1369, 1369, 1370, 1372, 1372, 1373, 
	1375, 1375, 1376, 1364, 1364, 1364, 1364, 1364, 
	1364, 1364, 1364, 1364, 1364, 1364, 1364, 1364, 
	1366, 1366, 1366, 1366, 1366, 1366, 1366, 1366, 
	1366, 1364, 1371, 1370, 1374, 1373, 1377, 1376, 
	1365, 1364, 1364, 1364, 1364, 1378, 1379, 1379, 
	1364, 1364, 1364, 1364, 1364, 1366, 1366, 1366, 
	1364, 1380, 1381, 1382, 1365, 1383, 1383, 1383, 
	1383, 1386, 1387, 1387, 1383, 1383, 1388, 1388, 
	1390, 1390, 1392, 1392, 1383, 1383, 1383, 1383, 
	1383, 1383, 1383, 1383, 1383, 1383, 1383, 1385, 
	1385, 1385, 1385, 1385, 1385, 1385, 1383, 1389, 
	1391, 1393, 1384, 1383, 1383, 1383, 1383, 1394, 
	1395, 1395, 1383, 1383, 1383, 1383, 1383, 1385, 
	1385, 1385, 1383, 1396, 1397, 1398, 1384, 1271, 
	1271, 68, 1271, 1271, 4, 4, 1335, 1335, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 1271, 1271, 918, 3, 1060, 1271, 919, 
	1319, 69, 1334, 1061, 1399, 1261, 1272, 1272, 
	68, 1272, 1272, 4, 4, 1337, 1337, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	1272, 1272, 918, 3, 1060, 1272, 919, 1320, 
	69, 1336, 1061, 1400, 1261, 68, 1271, 1271, 
	1272, 1272, 1273, 1273, 1271, 1271, 1272, 1272, 
	1273, 1273, 4, 4, 1335, 1335, 1337, 1337, 
	1339, 1339, 917, 917, 918, 2, 2, 3, 
	1059, 1059, 1060, 1271, 1271, 1272, 1272, 1273, 
	1273, 918, 3, 1060, 1271, 1272, 1273, 919, 
	1319, 1320, 1321, 69, 1334, 1336, 1338, 1061, 
	1399, 1400, 1401, 1261, 75, 75, 68, 75, 
	75, 4, 4, 959, 959, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 75, 75, 
	918, 3, 1060, 75, 919, 924, 69, 958, 
	1061, 1066, 1274, 1275, 1275, 68, 1275, 1275, 
	4, 4, 1341, 1341, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 1275, 1275, 918, 
	3, 1060, 1275, 919, 1322, 69, 1340, 1061, 
	1402, 1261, 68, 1271, 1271, 1275, 1275, 1276, 
	1276, 1271, 1271, 1275, 1275, 1276, 1276, 4, 
	4, 1335, 1335, 1341, 1341, 1343, 1343, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	1271, 1271, 1275, 1275, 1276, 1276, 918, 3, 
	1060, 1271, 1275, 1276, 919, 1319, 1322, 1323, 
	69, 1334, 1340, 1342, 1061, 1399, 1402, 1403, 
	1261, 73, 73, 68, 73, 73, 4, 4, 
	955, 955, 917, 917, 918, 2, 2, 3, 
	1059, 1059, 1060, 73, 73, 918, 3, 1060, 
	73, 919, 922, 69, 954, 1061, 1064, 1277, 
	68, 1272, 1272, 1275, 1275, 1278, 1278, 1272, 
	1272, 1275, 1275, 1278, 1278, 4, 4, 1337, 
	1337, 1341, 1341, 1345, 1345, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 1272, 1272, 
	1275, 1275, 1278, 1278, 918, 3, 1060, 1272, 
	1275, 1278, 919, 1320, 1322, 1324, 69, 1336, 
	1340, 1344, 1061, 1400, 1402, 1404, 1261, 79, 
	79, 68, 79, 79, 4, 4, 973, 973, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 79, 79, 918, 3, 1060, 79, 919, 
	928, 69, 972, 1061, 1070, 1277, 68, 1271, 
	1271, 1272, 1272, 1273, 1273, 1275, 1275, 1276, 
	1276, 1278, 1278, 1279, 1279, 1271, 1271, 1272, 
	1272, 1273, 1273, 1275, 1275, 1276, 1276, 1278, 
	1278, 1279, 1279, 4, 4, 1335, 1335, 1337, 
	1337, 1339, 1339, 1341, 1341, 1343, 1343, 1345, 
	1345, 1347, 1347, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 1271, 1271, 1272, 1272, 
	1273, 1273, 1275, 1275, 1276, 1276, 1278, 1278, 
	1279, 1279, 918, 3, 1060, 1271, 1272, 1273, 
	1275, 1276, 1278, 1279, 919, 1319, 1320, 1321, 
	1322, 1323, 1324, 1325, 69, 1334, 1336, 1338, 
	1340, 1342, 1344, 1346, 1061, 1399, 1400, 1401, 
	1402, 1403, 1404, 1405, 1261, 68, 73, 73, 
	79, 79, 83, 83, 73, 73, 79, 79, 
	83, 83, 4, 4, 955, 955, 973, 973, 
	985, 985, 917, 917, 918, 2, 2, 3, 
	1059, 1059, 1060, 73, 73, 79, 79, 83, 
	83, 918, 3, 1060, 73, 79, 83, 919, 
	922, 928, 932, 69, 954, 972, 984, 1061, 
	1064, 1070, 1074, 1277, 1280, 1280, 68, 1280, 
	1280, 4, 4, 1349, 1349, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 1280, 1280, 
	918, 3, 1060, 1280, 919, 1326, 69, 1348, 
	1061, 1406, 1261, 68, 1271, 1271, 1280, 1280, 
	1281, 1281, 1271, 1271, 1280, 1280, 1281, 1281, 
	4, 4, 1335, 1335, 1349, 1349, 1351, 1351, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 1271, 1271, 1280, 1280, 1281, 1281, 918, 
	3, 1060, 1271, 1280, 1281, 919, 1319, 1326, 
	1327, 69, 1334, 1348, 1350, 1061, 1399, 1406, 
	1407, 1261, 72, 72, 68, 72, 72, 4, 
	4, 953, 953, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 72, 72, 918, 3, 
	1060, 72, 919, 921, 69, 952, 1061, 1063, 
	1282, 68, 1272, 1272, 1280, 1280, 1283, 1283, 
	1272, 1272, 1280, 1280, 1283, 1283, 4, 4, 
	1337, 1337, 1349, 1349, 1353, 1353, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 1272, 
	1272, 1280, 1280, 1283, 1283, 918, 3, 1060, 
	1272, 1280, 1283, 919, 1320, 1326, 1328, 69, 
	1336, 1348, 1352, 1061, 1400, 1406, 1408, 1261, 
	78, 78, 68, 78, 78, 4, 4, 971, 
	971, 917, 917, 918, 2, 2, 3, 1059, 
	1059, 1060, 78, 78, 918, 3, 1060, 78, 
	919, 927, 69, 970, 1061, 1069, 1282, 68, 
	1271, 1271, 1272, 1272, 1273, 1273, 1280, 1280, 
	1281, 1281, 1283, 1283, 1284, 1284, 1271, 1271, 
	1272, 1272, 1273, 1273, 1280, 1280, 1281, 1281, 
	1283, 1283, 1284, 1284, 4, 4, 1335, 1335, 
	1337, 1337, 1339, 1339, 1349, 1349, 1351, 1351, 
	1353, 1353, 1355, 1355, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 1271, 1271, 1272, 
	1272, 1273, 1273, 1280, 1280, 1281, 1281, 1283, 
	1283, 1284, 1284, 918, 3, 1060, 1271, 1272, 
	1273, 1280, 1281, 1283, 1284, 919, 1319, 1320, 
	1321, 1326, 1327, 1328, 1329, 69, 1334, 1336, 
	1338, 1348, 1350, 1352, 1354, 1061, 1399, 1400, 
	1401, 1406, 1407, 1408, 1409, 1261, 68, 72, 
	72, 78, 78, 82, 82, 72, 72, 78, 
	78, 82, 82, 4, 4, 953, 953, 971, 
	971, 983, 983, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 72, 72, 78, 78, 
	82, 82, 918, 3, 1060, 72, 78, 82, 
	919, 921, 927, 931, 69, 952, 970, 982, 
	1061, 1063, 1069, 1073, 1282, 68, 1275, 1275, 
	1280, 1280, 1285, 1285, 1275, 1275, 1280, 1280, 
	1285, 1285, 4, 4, 1341, 1341, 1349, 1349, 
	1357, 1357, 917, 917, 918, 2, 2, 3, 
	1059, 1059, 1060, 1275, 1275, 1280, 1280, 1285, 
	1285, 918, 3, 1060, 1275, 1280, 1285, 919, 
	1322, 1326, 1330, 69, 1340, 1348, 1356, 1061, 
	1402, 1406, 1410, 1261, 86, 86, 68, 86, 
	86, 4, 4, 995, 995, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 86, 86, 
	918, 3, 1060, 86, 919, 935, 69, 994, 
	1061, 1077, 1282, 68, 1271, 1271, 1275, 1275, 
	1276, 1276, 1280, 1280, 1281, 1281, 1285, 1285, 
	1286, 1286, 1271, 1271, 1275, 1275, 1276, 1276, 
	1280, 1280, 1281, 1281, 1285, 1285, 1286, 1286, 
	4, 4, 1335, 1335, 1341, 1341, 1343, 1343, 
	1349, 1349, 1351, 1351, 1357, 1357, 1359, 1359, 
	917, 917, 918, 2, 2, 3, 1059, 1059, 
	1060, 1271, 1271, 1275, 1275, 1276, 1276, 1280, 
	1280, 1281, 1281, 1285, 1285, 1286, 1286, 918, 
	3, 1060, 1271, 1275, 1276, 1280, 1281, 1285, 
	1286, 919, 1319, 1322, 1323, 1326, 1327, 1330, 
	1331, 69, 1334, 1340, 1342, 1348, 1350, 1356, 
	1358, 1061, 1399, 1402, 1403, 1406, 1407, 1410, 
	1411, 1261, 68, 72, 72, 86, 86, 88, 
	88, 72, 72, 86, 86, 88, 88, 4, 
	4, 953, 953, 995, 995, 1001, 1001, 917, 
	917, 918, 2, 2, 3, 1059, 1059, 1060, 
	72, 72, 86, 86, 88, 88, 918, 3, 
	1060, 72, 86, 88, 919, 921, 935, 937, 
	69, 952, 994, 1000, 1061, 1063, 1077, 1079, 
	1282, 68, 1272, 1272, 1275, 1275, 1278, 1278, 
	1280, 1280, 1283, 1283, 1285, 1285, 1287, 1287, 
	1272, 1272, 1275, 1275, 1278, 1278, 1280, 1280, 
	1283, 1283, 1285, 1285, 1287, 1287, 4, 4, 
	1337, 1337, 1341, 1341, 1345, 1345, 1349, 1349, 
	1353, 1353, 1357, 1357, 1361, 1361, 917, 917, 
	918, 2, 2, 3, 1059, 1059, 1060, 1272, 
	1272, 1275, 1275, 1278, 1278, 1280, 1280, 1283, 
	1283, 1285, 1285, 1287, 1287, 918, 3, 1060, 
	1272, 1275, 1278, 1280, 1283, 1285, 1287, 919, 
	1320, 1322, 1324, 1326, 1328, 1330, 1332, 69, 
	1336, 1340, 1344, 1348, 1352, 1356, 1360, 1061, 
	1400, 1402, 1404, 1406, 1408, 1410, 1412, 1261, 
	68, 78, 78, 86, 86, 90, 90, 78, 
	78, 86, 86, 90, 90, 4, 4, 971, 
	971, 995, 995, 1007, 1007, 917, 917, 918, 
	2, 2, 3, 1059, 1059, 1060, 78, 78, 
	86, 86, 90, 90, 918, 3, 1060, 78, 
	86, 90, 919, 927, 935, 939, 69, 970, 
	994, 1006, 1061, 1069, 1077, 1081, 1282, 68, 
	1271, 1271, 1272, 1272, 1273, 1273, 1275, 1275, 
	1276, 1276, 1278, 1278, 1279, 1279, 1280, 1280, 
	1281, 1281, 1283, 1283, 1284, 1284, 1285, 1285, 
	1286, 1286, 1287, 1287, 1288, 1288, 1271, 1271, 
	1272, 1272, 1273, 1273, 1275, 1275, 1276, 1276, 
	1278, 1278, 1279, 1279, 1280, 1280, 1281, 1281, 
	1283, 1283, 1284, 1284, 1285, 1285, 1286, 1286, 
	1287, 1287, 1288, 1288, 4, 4, 1335, 1335, 
	1337, 1337, 1339, 1339, 1341, 1341, 1343, 1343, 
	1345, 1345, 1347, 1347, 1349, 1349, 1351, 1351, 
	1353, 1353, 1355, 1355, 1357, 1357, 1359, 1359, 
	1361, 1361, 1363, 1363, 917, 917, 918, 2, 
	2, 3, 1059, 1059, 1060, 1271, 1271, 1272, 
	1272, 1273, 1273, 1275, 1275, 1276, 1276, 1278, 
	1278, 1279, 1279, 1280, 1280, 1281, 1281, 1283, 
	1283, 1284, 1284, 1285, 1285, 1286, 1286, 1287, 
	1287, 1288, 1288, 1271, 1272, 1273, 1275, 1276, 
	1278, 1279, 1280, 1281, 1283, 1284, 1285, 1286, 
	1287, 1288, 919, 1319, 1320, 1321, 1322, 1323, 
	1324, 1325, 1326, 1327, 1328, 1329, 1330, 1331, 
	1332, 1333, 69, 1334, 1336, 1338, 1340, 1342, 
	1344, 1346, 1348, 1350, 1352, 1354, 1356, 1358, 
	1360, 1362, 1061, 1399, 1400, 1401, 1402, 1403, 
	1404, 1405, 1406, 1407, 1408, 1409, 1410, 1411, 
	1412, 1413, 918, 3, 1060, 1261, 68, 72, 
	72, 78, 78, 82, 82, 86, 86, 88, 
	88, 90, 90, 92, 92, 72, 72, 78, 
	78, 82, 82, 86, 86, 88, 88, 90, 
	90, 92, 92, 4, 4, 953, 953, 971, 
	971, 983, 983, 995, 995, 1001, 1001, 1007, 
	1007, 1013, 1013, 917, 917, 918, 2, 2, 
	3, 1059, 1059, 1060, 72, 72, 78, 78, 
	82, 82, 86, 86, 88, 88, 90, 90, 
	92, 92, 918, 3, 1060, 72, 78, 82, 
	86, 88, 90, 92, 919, 921, 927, 931, 
	935, 937, 939, 941, 69, 952, 970, 982, 
	994, 1000, 1006, 1012, 1061, 1063, 1069, 1073, 
	1077, 1079, 1081, 1083, 1282, 1414, 1261, 1415, 
	1261, 1414, 1415, 1261, 1416, 1416, 1416, 0, 
	1417, 1417, 1417, 0, 1416, 1416, 1416, 1417, 
	1417, 1417, 1418, 1418, 1418, 0, 1097, 1097, 
	1097, 1419, 1420, 1420, 1420, 0, 1416, 1416, 
	1416, 1420, 1420, 1420, 1421, 1421, 1421, 0, 
	1095, 1095, 1095, 1422, 1417, 1417, 1417, 1420, 
	1420, 1420, 1423, 1423, 1423, 0, 1102, 1102, 
	1102, 1422, 1416, 1416, 1416, 1417, 1417, 1417, 
	1418, 1418, 1418, 1420, 1420, 1420, 1421, 1421, 
	1421, 1423, 1423, 1423, 1424, 1424, 1424, 0, 
	1095, 1095, 1095, 1102, 1102, 1102, 1107, 1107, 
	1107, 1422, 1425, 1425, 1425, 0, 1416, 1416, 
	1416, 1425, 1425, 1425, 1426, 1426, 1426, 0, 
	1094, 1094, 1094, 1427, 1417, 1417, 1417, 1425, 
	1425, 1425, 1428, 1428, 1428, 0, 1101, 1101, 
	1101, 1427, 1416, 1416, 1416, 1417, 1417, 1417, 
	1418, 1418, 1418, 1425, 1425, 1425, 1426, 1426, 
	1426, 1428, 1428, 1428, 1429, 1429, 1429, 0, 
	1094, 1094, 1094, 1101, 1101, 1101, 1106, 1106, 
	1106, 1427, 1420, 1420, 1420, 1425, 1425, 1425, 
	1430, 1430, 1430, 0, 1111, 1111, 1111, 1427, 
	1416, 1416, 1416, 1420, 1420, 1420, 1421, 1421, 
	1421, 1425, 1425, 1425, 1426, 1426, 1426, 1430, 
	1430, 1430, 1431, 1431, 1431, 0, 1094, 1094, 
	1094, 1111, 1111, 1111, 1114, 1114, 1114, 1427, 
	1417, 1417, 1417, 1420, 1420, 1420, 1423, 1423, 
	1423, 1425, 1425, 1425, 1428, 1428, 1428, 1430, 
	1430, 1430, 1432, 1432, 1432, 0, 1101, 1101, 
	1101, 1111, 1111, 1111, 1117, 1117, 1117, 1427, 
	1416, 1416, 1416, 1417, 1417, 1417, 1418, 1418, 
	1418, 1420, 1420, 1420, 1421, 1421, 1421, 1423, 
	1423, 1423, 1424, 1424, 1424, 1425, 1425, 1425, 
	1426, 1426, 1426, 1428, 1428, 1428, 1429, 1429, 
	1429, 1430, 1430, 1430, 1431, 1431, 1431, 1432, 
	1432, 1432, 1433, 1433, 1433, 0, 1094, 1094, 
	1094, 1101, 1101, 1101, 1106, 1106, 1106, 1111, 
	1111, 1111, 1114, 1114, 1114, 1117, 1117, 1117, 
	1120, 1120, 1120, 1427, 1435, 1436, 1437, 1434, 
	1435, 1438, 1439, 1438, 1441, 1442, 1440, 1441, 
	1443, 0
};

static const short _text_file_trans_targs[] = {
	832, 2, 3, 21, 41, 3, 4, 21, 
	5, 20, 5, 20, 6, 7, 17, 7, 
	8, 9, 840, 9, 840, 835, 10, 11, 
	12, 838, 12, 838, 836, 14, 837, 16, 
	839, 18, 19, 22, 23, 40, 23, 40, 
	24, 25, 26, 37, 25, 26, 27, 28, 
	846, 28, 846, 841, 29, 30, 31, 32, 
	844, 32, 844, 842, 34, 843, 36, 845, 
	38, 39, 42, 43, 1, 44, 832, 46, 
	47, 48, 49, 50, 832, 52, 53, 54, 
	832, 56, 57, 58, 851, 60, 832, 62, 
	854, 64, 856, 66, 858, 832, 861, 863, 
	865, 867, 869, 871, 873, 76, 77, 78, 
	79, 134, 79, 134, 80, 81, 130, 110, 
	131, 81, 82, 110, 83, 109, 83, 109, 
	84, 85, 105, 106, 85, 86, 87, 104, 
	87, 104, 88, 89, 100, 832, 90, 101, 
	89, 832, 90, 91, 92, 99, 92, 99, 
	93, 94, 95, 832, 96, 94, 832, 94, 
	832, 97, 98, 89, 832, 90, 102, 103, 
	85, 107, 108, 111, 112, 129, 112, 129, 
	113, 114, 125, 115, 126, 114, 115, 116, 
	117, 124, 117, 124, 118, 119, 120, 832, 
	121, 119, 832, 119, 832, 122, 123, 114, 
	115, 127, 128, 81, 110, 132, 133, 136, 
	137, 138, 139, 140, 205, 140, 205, 141, 
	142, 201, 171, 202, 142, 143, 171, 144, 
	170, 144, 170, 145, 146, 166, 167, 146, 
	147, 148, 165, 148, 165, 149, 150, 161, 
	151, 162, 150, 151, 152, 153, 160, 153, 
	160, 154, 155, 156, 832, 157, 155, 832, 
	155, 832, 158, 159, 150, 151, 163, 164, 
	146, 168, 169, 172, 173, 200, 173, 200, 
	174, 175, 196, 176, 197, 175, 176, 177, 
	178, 195, 178, 195, 179, 180, 191, 181, 
	192, 180, 181, 182, 183, 190, 183, 190, 
	184, 185, 186, 832, 187, 185, 832, 185, 
	832, 188, 189, 180, 181, 193, 194, 175, 
	176, 198, 199, 142, 171, 203, 204, 207, 
	209, 210, 211, 212, 277, 212, 277, 213, 
	214, 242, 260, 263, 266, 270, 273, 214, 
	215, 242, 216, 241, 216, 241, 217, 218, 
	238, 219, 220, 221, 237, 221, 237, 222, 
	223, 234, 223, 832, 224, 225, 226, 233, 
	226, 233, 227, 228, 229, 832, 230, 228, 
	832, 228, 832, 231, 232, 235, 236, 239, 
	240, 243, 244, 259, 244, 259, 245, 246, 
	256, 246, 247, 248, 249, 255, 249, 255, 
	250, 251, 252, 251, 832, 253, 254, 257, 
	258, 261, 262, 214, 242, 264, 265, 267, 
	268, 269, 214, 242, 271, 272, 214, 242, 
	274, 275, 276, 214, 242, 279, 280, 281, 
	282, 283, 358, 283, 358, 284, 285, 313, 
	341, 344, 347, 351, 354, 285, 286, 313, 
	287, 312, 287, 312, 288, 289, 309, 290, 
	291, 292, 308, 292, 308, 293, 294, 305, 
	294, 295, 296, 297, 304, 297, 304, 298, 
	299, 300, 832, 301, 299, 832, 299, 832, 
	302, 303, 306, 307, 310, 311, 314, 315, 
	340, 315, 340, 316, 317, 337, 317, 318, 
	319, 320, 336, 320, 336, 321, 322, 333, 
	322, 323, 324, 325, 332, 325, 332, 326, 
	327, 328, 832, 329, 327, 832, 327, 832, 
	330, 331, 334, 335, 338, 339, 342, 343, 
	285, 313, 345, 346, 348, 349, 350, 285, 
	313, 352, 353, 285, 313, 355, 356, 357, 
	285, 313, 360, 362, 363, 364, 365, 366, 
	367, 405, 367, 405, 368, 369, 401, 402, 
	369, 370, 371, 400, 371, 400, 372, 373, 
	396, 397, 373, 374, 375, 395, 375, 395, 
	376, 377, 832, 378, 381, 384, 388, 391, 
	377, 832, 379, 380, 377, 832, 382, 383, 
	385, 386, 387, 377, 832, 389, 390, 377, 
	832, 392, 393, 394, 377, 832, 373, 398, 
	399, 369, 403, 404, 407, 408, 409, 410, 
	411, 412, 460, 412, 460, 413, 414, 456, 
	457, 414, 415, 416, 455, 416, 455, 417, 
	418, 451, 452, 418, 419, 420, 450, 420, 
	450, 421, 422, 423, 433, 436, 439, 443, 
	446, 422, 423, 424, 425, 432, 425, 432, 
	426, 427, 428, 832, 429, 427, 832, 427, 
	832, 430, 431, 434, 435, 422, 423, 437, 
	438, 440, 441, 442, 422, 423, 444, 445, 
	422, 423, 447, 448, 449, 422, 423, 418, 
	453, 454, 414, 458, 459, 462, 463, 464, 
	465, 466, 467, 505, 467, 505, 468, 469, 
	501, 502, 469, 470, 471, 500, 471, 500, 
	472, 473, 496, 497, 473, 474, 475, 495, 
	475, 495, 476, 477, 478, 481, 484, 488, 
	491, 477, 479, 480, 477, 482, 483, 485, 
	486, 487, 477, 489, 490, 477, 492, 493, 
	494, 477, 473, 498, 499, 469, 503, 504, 
	507, 508, 509, 510, 511, 512, 537, 512, 
	537, 513, 514, 533, 534, 514, 515, 516, 
	532, 516, 532, 517, 518, 528, 529, 518, 
	519, 520, 527, 520, 527, 521, 522, 523, 
	832, 524, 522, 832, 522, 832, 525, 526, 
	518, 530, 531, 514, 535, 536, 539, 541, 
	543, 545, 546, 547, 548, 549, 550, 585, 
	550, 585, 551, 552, 581, 582, 552, 553, 
	554, 580, 554, 580, 555, 556, 576, 577, 
	556, 557, 558, 575, 558, 575, 559, 560, 
	571, 561, 572, 560, 561, 562, 563, 570, 
	563, 570, 564, 565, 566, 832, 567, 565, 
	832, 565, 832, 568, 569, 560, 561, 573, 
	574, 556, 578, 579, 552, 583, 584, 587, 
	589, 591, 593, 594, 595, 596, 597, 598, 
	623, 598, 623, 599, 600, 619, 620, 600, 
	601, 602, 618, 602, 618, 603, 604, 614, 
	615, 604, 605, 606, 613, 606, 613, 607, 
	608, 609, 610, 608, 608, 611, 612, 604, 
	616, 617, 600, 621, 622, 625, 627, 629, 
	630, 631, 637, 632, 633, 634, 632, 633, 
	634, 896, 832, 635, 636, 897, 638, 637, 
	638, 898, 832, 639, 899, 630, 631, 640, 
	642, 643, 644, 645, 646, 895, 648, 649, 
	650, 895, 652, 653, 654, 903, 656, 895, 
	658, 906, 660, 908, 662, 910, 895, 913, 
	915, 917, 919, 921, 923, 925, 672, 684, 
	673, 682, 674, 680, 675, 678, 676, 677, 
	926, 928, 929, 679, 681, 683, 685, 687, 
	689, 695, 690, 693, 691, 692, 926, 928, 
	929, 694, 696, 698, 700, 706, 701, 704, 
	702, 703, 932, 933, 934, 705, 707, 709, 
	711, 712, 926, 928, 929, 714, 716, 717, 
	937, 938, 939, 719, 721, 722, 941, 942, 
	943, 724, 726, 727, 945, 946, 947, 729, 
	926, 928, 929, 950, 951, 952, 954, 955, 
	956, 958, 959, 960, 962, 963, 964, 966, 
	967, 968, 970, 971, 972, 974, 975, 976, 
	746, 747, 753, 748, 749, 750, 748, 749, 
	750, 978, 751, 752, 979, 754, 753, 754, 
	980, 755, 981, 746, 747, 756, 758, 759, 
	760, 761, 762, 977, 764, 765, 766, 977, 
	768, 769, 770, 985, 772, 977, 774, 988, 
	776, 990, 778, 992, 977, 995, 997, 999, 
	1001, 1003, 1005, 1007, 788, 789, 790, 791, 
	792, 793, 832, 795, 796, 797, 798, 832, 
	800, 801, 802, 803, 1014, 805, 806, 832, 
	808, 809, 1017, 811, 812, 1019, 814, 815, 
	1021, 817, 832, 819, 1024, 821, 1026, 823, 
	1028, 825, 1030, 827, 1032, 829, 1034, 831, 
	1036, 832, 832, 833, 0, 834, 847, 848, 
	849, 850, 852, 853, 855, 857, 859, 860, 
	862, 864, 866, 868, 870, 872, 874, 875, 
	876, 877, 878, 879, 880, 881, 882, 883, 
	884, 885, 886, 887, 888, 889, 890, 891, 
	892, 893, 894, 895, 900, 901, 902, 904, 
	905, 907, 909, 911, 912, 914, 916, 918, 
	920, 922, 924, 926, 927, 930, 931, 935, 
	936, 940, 944, 948, 949, 953, 957, 961, 
	965, 969, 973, 977, 982, 983, 984, 986, 
	987, 989, 991, 993, 994, 996, 998, 1000, 
	1002, 1004, 1006, 1008, 1009, 1010, 1011, 1012, 
	1013, 1015, 1016, 1018, 1020, 1022, 1023, 1025, 
	1027, 1029, 1031, 1033, 1035, 832, 1011, 1012, 
	1013, 1015, 1016, 1018, 1020, 1022, 1023, 1025, 
	1027, 1029, 1031, 1033, 1035, 832, 832, 10, 
	15, 13, 832, 29, 30, 35, 33, 45, 
	51, 55, 832, 59, 61, 832, 63, 65, 
	67, 68, 832, 69, 70, 71, 72, 73, 
	74, 75, 135, 206, 208, 278, 359, 361, 
	406, 461, 506, 538, 540, 542, 544, 586, 
	588, 590, 592, 624, 626, 628, 832, 635, 
	636, 896, 897, 832, 639, 898, 899, 641, 
	647, 651, 655, 657, 659, 661, 663, 664, 
	665, 666, 667, 668, 669, 670, 671, 686, 
	688, 697, 699, 708, 710, 713, 715, 718, 
	720, 723, 725, 728, 730, 731, 732, 733, 
	734, 735, 736, 737, 738, 739, 740, 741, 
	742, 743, 744, 745, 832, 832, 832, 28, 
	35, 635, 636, 896, 29, 30, 846, 751, 
	752, 978, 32, 33, 897, 844, 979, 832, 
	832, 832, 9, 15, 639, 898, 10, 840, 
	755, 980, 12, 13, 899, 838, 981, 757, 
	763, 767, 771, 773, 775, 777, 779, 780, 
	781, 782, 783, 784, 785, 786, 832, 832, 
	787, 794, 799, 832, 804, 807, 832, 810, 
	813, 816, 818, 832, 820, 822, 824, 826, 
	828, 830, 1037, 1037, 1038, 1039, 1037, 1037, 
	1040, 1040, 1041, 1040
};

static const short _text_file_trans_actions[] = {
	123, 0, 7, 7, 0, 0, 5, 0, 
	5, 5, 0, 0, 0, 7, 0, 0, 
	5, 5, 318, 0, 269, 269, 0, 5, 
	5, 318, 0, 269, 269, 0, 0, 0, 
	269, 0, 0, 5, 5, 5, 0, 0, 
	0, 7, 7, 0, 0, 0, 5, 5, 
	314, 0, 266, 266, 0, 0, 5, 5, 
	314, 0, 266, 266, 0, 0, 0, 266, 
	0, 0, 0, 0, 0, 0, 121, 0, 
	0, 0, 0, 0, 55, 0, 0, 0, 
	57, 0, 0, 0, 257, 0, 59, 0, 
	260, 0, 260, 0, 260, 61, 263, 263, 
	263, 0, 263, 263, 263, 0, 0, 25, 
	25, 25, 0, 0, 0, 27, 27, 27, 
	0, 0, 25, 0, 25, 25, 0, 0, 
	0, 27, 27, 0, 0, 25, 25, 25, 
	0, 0, 0, 27, 27, 167, 27, 0, 
	0, 63, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 170, 0, 0, 65, 29, 
	203, 0, 0, 29, 200, 29, 0, 0, 
	29, 0, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 27, 0, 0, 0, 25, 
	25, 25, 0, 0, 0, 27, 27, 173, 
	0, 0, 67, 29, 206, 0, 0, 29, 
	29, 0, 0, 29, 29, 0, 0, 0, 
	0, 0, 25, 25, 25, 0, 0, 0, 
	27, 27, 27, 0, 0, 25, 0, 25, 
	25, 0, 0, 0, 27, 27, 0, 0, 
	25, 25, 25, 0, 0, 0, 27, 27, 
	27, 0, 0, 0, 25, 25, 25, 0, 
	0, 0, 27, 27, 176, 0, 0, 69, 
	29, 209, 0, 0, 29, 29, 0, 0, 
	29, 0, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 27, 0, 0, 0, 25, 
	25, 25, 0, 0, 0, 27, 27, 27, 
	0, 0, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 179, 0, 0, 71, 29, 
	212, 0, 0, 29, 29, 0, 0, 29, 
	29, 0, 0, 29, 29, 0, 0, 0, 
	0, 0, 9, 9, 9, 0, 0, 0, 
	11, 11, 11, 0, 11, 11, 11, 0, 
	21, 0, 21, 21, 0, 0, 0, 23, 
	0, 0, 21, 21, 21, 0, 0, 0, 
	23, 0, 0, 73, 0, 25, 25, 25, 
	0, 0, 0, 27, 27, 182, 0, 0, 
	75, 29, 215, 0, 0, 0, 0, 0, 
	0, 21, 21, 21, 0, 0, 0, 23, 
	0, 0, 0, 21, 21, 21, 0, 0, 
	0, 23, 0, 0, 77, 0, 0, 0, 
	0, 0, 0, 13, 13, 0, 0, 0, 
	0, 0, 15, 15, 0, 0, 19, 19, 
	0, 0, 0, 17, 17, 0, 0, 0, 
	9, 9, 9, 0, 0, 0, 11, 11, 
	11, 0, 11, 11, 11, 0, 21, 0, 
	21, 21, 0, 0, 0, 23, 0, 0, 
	21, 21, 21, 0, 0, 0, 23, 0, 
	0, 0, 25, 25, 25, 0, 0, 0, 
	27, 27, 185, 0, 0, 79, 29, 218, 
	0, 0, 0, 0, 0, 0, 21, 21, 
	21, 0, 0, 0, 23, 0, 0, 0, 
	21, 21, 21, 0, 0, 0, 23, 0, 
	0, 0, 25, 25, 25, 0, 0, 0, 
	27, 27, 188, 0, 0, 81, 29, 221, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	13, 13, 0, 0, 0, 0, 0, 15, 
	15, 0, 0, 19, 19, 0, 0, 0, 
	17, 17, 0, 0, 0, 0, 0, 25, 
	25, 25, 0, 0, 0, 27, 27, 0, 
	0, 25, 25, 25, 0, 0, 0, 27, 
	27, 0, 0, 9, 9, 9, 0, 0, 
	0, 11, 152, 11, 0, 11, 11, 11, 
	0, 83, 0, 0, 13, 155, 0, 0, 
	0, 0, 0, 15, 158, 0, 0, 19, 
	164, 0, 0, 0, 17, 161, 29, 0, 
	0, 29, 0, 0, 0, 0, 0, 0, 
	25, 25, 25, 0, 0, 0, 27, 27, 
	0, 0, 25, 25, 25, 0, 0, 0, 
	27, 27, 0, 0, 9, 9, 9, 0, 
	0, 0, 11, 11, 11, 0, 11, 11, 
	11, 0, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 191, 0, 0, 85, 29, 
	224, 0, 0, 0, 0, 13, 13, 0, 
	0, 0, 0, 0, 15, 15, 0, 0, 
	19, 19, 0, 0, 0, 17, 17, 29, 
	0, 0, 29, 0, 0, 0, 0, 0, 
	0, 25, 25, 25, 0, 0, 0, 27, 
	27, 0, 0, 25, 25, 25, 0, 0, 
	0, 27, 27, 0, 0, 9, 9, 9, 
	0, 0, 0, 11, 11, 0, 11, 11, 
	11, 0, 0, 0, 13, 0, 0, 0, 
	0, 0, 15, 0, 0, 19, 0, 0, 
	0, 17, 29, 0, 0, 29, 0, 0, 
	0, 0, 0, 0, 25, 25, 25, 0, 
	0, 0, 27, 27, 0, 0, 25, 25, 
	25, 0, 0, 0, 27, 27, 0, 0, 
	25, 25, 25, 0, 0, 0, 27, 27, 
	194, 0, 0, 87, 29, 227, 0, 0, 
	29, 0, 0, 29, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 25, 25, 25, 
	0, 0, 0, 27, 27, 0, 0, 25, 
	25, 25, 0, 0, 0, 27, 27, 0, 
	0, 25, 25, 25, 0, 0, 0, 27, 
	27, 27, 0, 0, 0, 25, 25, 25, 
	0, 0, 0, 27, 27, 197, 0, 0, 
	89, 29, 230, 0, 0, 29, 29, 0, 
	0, 29, 0, 0, 29, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 25, 25, 
	25, 0, 0, 0, 27, 27, 0, 0, 
	25, 25, 25, 0, 0, 0, 27, 27, 
	0, 0, 25, 25, 25, 0, 0, 0, 
	27, 27, 0, 0, 29, 0, 0, 29, 
	0, 0, 29, 0, 0, 0, 0, 0, 
	0, 0, 125, 125, 3, 3, 1, 0, 
	0, 290, 117, 0, 0, 125, 3, 1, 
	0, 290, 119, 0, 125, 3, 3, 1, 
	1, 1, 1, 1, 1, 326, 1, 1, 
	1, 330, 1, 1, 1, 330, 1, 334, 
	1, 334, 1, 334, 1, 334, 338, 338, 
	338, 338, 338, 338, 338, 338, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	254, 254, 254, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 257, 257, 
	257, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 257, 257, 257, 0, 0, 0, 
	0, 0, 260, 260, 260, 0, 0, 0, 
	260, 260, 260, 0, 0, 0, 260, 260, 
	260, 0, 0, 0, 260, 260, 260, 0, 
	263, 263, 263, 263, 263, 263, 263, 263, 
	263, 263, 263, 263, 263, 263, 263, 263, 
	263, 263, 263, 263, 263, 263, 263, 263, 
	0, 0, 278, 278, 131, 131, 1, 0, 
	0, 353, 0, 0, 353, 131, 1, 0, 
	359, 0, 359, 131, 131, 1, 1, 1, 
	1, 1, 1, 326, 1, 1, 1, 330, 
	1, 1, 1, 330, 1, 334, 1, 334, 
	1, 334, 1, 334, 338, 338, 338, 338, 
	338, 338, 338, 338, 0, 0, 0, 0, 
	0, 0, 47, 0, 0, 0, 0, 49, 
	0, 0, 0, 0, 245, 0, 0, 51, 
	0, 0, 248, 0, 0, 248, 0, 0, 
	248, 0, 53, 0, 251, 0, 251, 0, 
	251, 0, 0, 0, 251, 0, 251, 0, 
	251, 97, 239, 0, 0, 322, 322, 275, 
	275, 275, 35, 275, 275, 275, 35, 275, 
	275, 275, 35, 275, 275, 275, 35, 35, 
	35, 35, 35, 35, 35, 35, 35, 35, 
	35, 35, 35, 35, 35, 35, 35, 35, 
	35, 35, 35, 342, 342, 342, 342, 342, 
	342, 342, 342, 342, 342, 342, 342, 342, 
	342, 342, 342, 322, 322, 322, 322, 322, 
	322, 322, 322, 322, 322, 322, 322, 322, 
	322, 322, 322, 347, 347, 347, 347, 347, 
	347, 347, 347, 347, 347, 347, 347, 347, 
	347, 347, 347, 0, 0, 0, 275, 275, 
	275, 275, 275, 275, 275, 275, 275, 275, 
	275, 275, 275, 275, 275, 95, 272, 272, 
	272, 272, 272, 272, 272, 272, 272, 272, 
	272, 272, 272, 272, 272, 115, 149, 7, 
	0, 0, 146, 7, 7, 0, 0, 0, 
	0, 0, 105, 0, 0, 107, 0, 0, 
	0, 0, 109, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 134, 3, 
	3, 242, 1, 137, 3, 242, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 282, 111, 140, 3, 
	3, 131, 131, 294, 131, 131, 306, 131, 
	131, 298, 3, 3, 128, 306, 298, 286, 
	113, 143, 3, 3, 131, 294, 131, 310, 
	131, 302, 3, 3, 128, 310, 302, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 91, 93, 
	0, 0, 0, 99, 0, 0, 101, 0, 
	0, 0, 0, 103, 0, 0, 0, 0, 
	0, 0, 39, 233, 0, 0, 41, 37, 
	43, 236, 0, 45
};

static const short _text_file_to_state_actions[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	31, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 31, 0, 0, 
	31, 0
};

static const short _text_file_from_state_actions[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	33, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 33, 0, 0, 
	33, 0
};

static const short _text_file_eof_trans[] = {
	0, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 71, 71, 1, 
	1, 1, 1, 71, 71, 1, 1, 71, 
	71, 1, 1, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 1, 1, 
	1, 1, 1, 907, 907, 1, 1, 915, 
	1, 71, 71, 1, 1, 1, 1, 71, 
	71, 1, 1, 71, 71, 1, 1, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 71, 1, 71, 71, 
	71, 71, 1, 1, 1, 1, 1, 71, 
	1, 71, 71, 71, 71, 1, 1, 1, 
	1, 1, 71, 1, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 1, 1, 1, 1, 1, 907, 
	907, 1, 1, 915, 1, 71, 71, 1, 
	1, 1, 1, 71, 71, 1, 1, 71, 
	71, 1, 1, 71, 71, 71, 71, 71, 
	71, 71, 71, 71, 71, 71, 71, 71, 
	71, 71, 71, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	0, 1262, 1262, 1263, 1263, 1263, 1263, 1263, 
	1263, 1267, 1267, 1267, 1267, 1267, 1267, 1262, 
	1262, 1262, 1262, 1275, 1262, 1262, 1278, 1262, 
	1278, 1262, 1278, 1262, 1262, 1283, 1262, 1283, 
	1262, 1283, 1262, 1283, 1262, 1283, 1262, 1283, 
	1262, 1283, 1262, 1262, 1262, 1262, 1262, 1262, 
	1262, 1262, 1262, 1262, 1262, 1262, 1262, 1262, 
	1262, 1262, 1262, 1262, 1262, 1262, 1262, 1, 
	1311, 1311, 1316, 1316, 1262, 1262, 1262, 1275, 
	1262, 1262, 1278, 1262, 1278, 1262, 1278, 1262, 
	1262, 1283, 1262, 1283, 1262, 1283, 1262, 1283, 
	1262, 1283, 1262, 1283, 1262, 1283, 1, 1262, 
	1, 1, 1262, 1262, 1275, 1275, 1275, 1262, 
	1262, 1278, 1278, 1278, 1262, 1278, 1278, 1278, 
	1262, 1278, 1278, 1278, 1262, 1262, 1283, 1283, 
	1283, 1262, 1283, 1283, 1283, 1262, 1283, 1283, 
	1283, 1262, 1283, 1283, 1283, 1262, 1283, 1283, 
	1283, 1262, 1283, 1283, 1283, 1262, 1283, 1283, 
	1283, 1, 1365, 1365, 1384, 1384, 1262, 1262, 
	1262, 1275, 1262, 1262, 1278, 1262, 1278, 1262, 
	1278, 1262, 1262, 1283, 1262, 1283, 1262, 1283, 
	1262, 1283, 1262, 1283, 1262, 1283, 1262, 1283, 
	1262, 1262, 1262, 1, 1, 1, 1420, 1, 
	1, 1423, 1, 1423, 1, 1423, 1, 1, 
	1428, 1, 1428, 1, 1428, 1, 1428, 1, 
	1428, 1, 1428, 1, 1428, 0, 1439, 1439, 
	0, 1444
};

static const int text_file_start = 832;
static const int text_file_first_final = 832;
static const int text_file_error = 0;

static const int text_file_en_multiLineComment = 1037;
static const int text_file_en_singleLineComment = 1040;
static const int text_file_en_main = 832;


#line 291 "source/parser/TextFileParser.rl"

bool scanner(TextFile &textFile, const Configuration &configuration) {
	FSM fsm = {};
	bool parseError = false;
	fsm.addColor = [&textFile](const Color &color) {
		textFile.addColor(color.normalizeRgb());
	};
	
#line 7780 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
	{
	 fsm.cs = text_file_start;
	 fsm.ts = 0;
	 fsm.te = 0;
	 fsm.act = 0;
	}

#line 299 "source/parser/TextFileParser.rl"
	int have = 0;
	while (1) {
		char *p = fsm.buffer + have;
		int ws = sizeof(fsm.buffer) - have;
		if (ws == 0) {
			textFile.outOfMemory();
			break;
		}
		char *eof = 0;
		auto readSize = textFile.read(fsm.buffer + have, ws);
		char *pe = p + readSize;
		if (readSize > 0) {
			if (readSize < sizeof(fsm.buffer))
				eof = pe;
			
#line 7800 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
	{
	int _klen;
	unsigned int _trans;
	int _widec;
	const char *_acts;
	unsigned int _nacts;
	const int *_keys;

	if ( p == pe )
		goto _test_eof;
	if (  fsm.cs == 0 )
		goto _out;
_resume:
	_acts = _text_file_actions + _text_file_from_state_actions[ fsm.cs];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 ) {
		switch ( *_acts++ ) {
	case 18:
#line 1 "NONE"
	{ fsm.ts = p;}
	break;
#line 7820 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
		}
	}

	_widec = (*p);
	_klen = _text_file_cond_lengths[ fsm.cs];
	_keys = _text_file_cond_keys + (_text_file_cond_offsets[ fsm.cs]*2);
	if ( _klen > 0 ) {
		const int *_lower = _keys;
		const int *_mid;
		const int *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				switch ( _text_file_cond_spaces[_text_file_cond_offsets[ fsm.cs] + ((_mid - _keys)>>1)] ) {
	case 0: {
		_widec = (int)(128 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		break;
	}
	case 1: {
		_widec = (int)(640 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		break;
	}
	case 2: {
		_widec = (int)(1152 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		break;
	}
	case 3: {
		_widec = (int)(1664 + ((*p) - -128));
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 256;
		break;
	}
	case 4: {
		_widec = (int)(2176 + ((*p) - -128));
		if ( 
#line 232 "source/parser/TextFileParser.rl"
 configuration.cssRgb  ) _widec += 256;
		break;
	}
	case 5: {
		_widec = (int)(2688 + ((*p) - -128));
		if ( 
#line 233 "source/parser/TextFileParser.rl"
 configuration.cssRgba  ) _widec += 256;
		break;
	}
	case 6: {
		_widec = (int)(3200 + ((*p) - -128));
		if ( 
#line 234 "source/parser/TextFileParser.rl"
 configuration.cssHsl  ) _widec += 256;
		break;
	}
	case 7: {
		_widec = (int)(3712 + ((*p) - -128));
		if ( 
#line 235 "source/parser/TextFileParser.rl"
 configuration.cssHsla  ) _widec += 256;
		break;
	}
	case 8: {
		_widec = (int)(4224 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		break;
	}
	case 9: {
		_widec = (int)(4736 + ((*p) - -128));
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		break;
	}
	case 10: {
		_widec = (int)(5248 + ((*p) - -128));
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 256;
		break;
	}
	case 11: {
		_widec = (int)(5760 + ((*p) - -128));
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 256;
		break;
	}
	case 12: {
		_widec = (int)(6272 + ((*p) - -128));
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 256;
		break;
	}
	case 13: {
		_widec = (int)(6784 + ((*p) - -128));
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 256;
		break;
	}
	case 14: {
		_widec = (int)(7296 + ((*p) - -128));
		if ( 
#line 242 "source/parser/TextFileParser.rl"
 configuration.singleLineCComments  ) _widec += 256;
		break;
	}
	case 15: {
		_widec = (int)(7808 + ((*p) - -128));
		if ( 
#line 243 "source/parser/TextFileParser.rl"
 configuration.multiLineCComments  ) _widec += 256;
		break;
	}
	case 16: {
		_widec = (int)(8320 + ((*p) - -128));
		if ( 
#line 244 "source/parser/TextFileParser.rl"
 configuration.singleLineHashComments  ) _widec += 256;
		break;
	}
	case 17: {
		_widec = (int)(8832 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		break;
	}
	case 18: {
		_widec = (int)(11904 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		break;
	}
	case 19: {
		_widec = (int)(18048 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		break;
	}
	case 20: {
		_widec = (int)(55936 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		break;
	}
	case 21: {
		_widec = (int)(105088 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 512;
		break;
	}
	case 22: {
		_widec = (int)(12928 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		break;
	}
	case 23: {
		_widec = (int)(19072 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		break;
	}
	case 24: {
		_widec = (int)(56960 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		break;
	}
	case 25: {
		_widec = (int)(106112 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 512;
		break;
	}
	case 26: {
		_widec = (int)(22144 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		break;
	}
	case 27: {
		_widec = (int)(60032 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		break;
	}
	case 28: {
		_widec = (int)(109184 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 512;
		break;
	}
	case 29: {
		_widec = (int)(69248 + ((*p) - -128));
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		break;
	}
	case 30: {
		_widec = (int)(118400 + ((*p) - -128));
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 256;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 512;
		break;
	}
	case 31: {
		_widec = (int)(27264 + ((*p) - -128));
		if ( 
#line 232 "source/parser/TextFileParser.rl"
 configuration.cssRgb  ) _widec += 256;
		if ( 
#line 233 "source/parser/TextFileParser.rl"
 configuration.cssRgba  ) _widec += 512;
		break;
	}
	case 32: {
		_widec = (int)(28288 + ((*p) - -128));
		if ( 
#line 234 "source/parser/TextFileParser.rl"
 configuration.cssHsl  ) _widec += 256;
		if ( 
#line 235 "source/parser/TextFileParser.rl"
 configuration.cssHsla  ) _widec += 512;
		break;
	}
	case 33: {
		_widec = (int)(29312 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 512;
		break;
	}
	case 34: {
		_widec = (int)(32384 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		break;
	}
	case 35: {
		_widec = (int)(38528 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		break;
	}
	case 36: {
		_widec = (int)(33408 + ((*p) - -128));
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		break;
	}
	case 37: {
		_widec = (int)(39552 + ((*p) - -128));
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		break;
	}
	case 38: {
		_widec = (int)(42624 + ((*p) - -128));
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 256;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		break;
	}
	case 39: {
		_widec = (int)(146048 + ((*p) - -128));
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 256;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 512;
		break;
	}
	case 40: {
		_widec = (int)(212608 + ((*p) - -128));
		if ( 
#line 242 "source/parser/TextFileParser.rl"
 configuration.singleLineCComments  ) _widec += 256;
		if ( 
#line 243 "source/parser/TextFileParser.rl"
 configuration.multiLineCComments  ) _widec += 512;
		break;
	}
	case 41: {
		_widec = (int)(9856 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		break;
	}
	case 42: {
		_widec = (int)(20096 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		break;
	}
	case 43: {
		_widec = (int)(57984 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 44: {
		_widec = (int)(107136 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 45: {
		_widec = (int)(23168 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		break;
	}
	case 46: {
		_widec = (int)(61056 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 47: {
		_widec = (int)(110208 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 48: {
		_widec = (int)(70272 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 49: {
		_widec = (int)(119424 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 50: {
		_widec = (int)(147072 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 51: {
		_widec = (int)(25216 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		break;
	}
	case 52: {
		_widec = (int)(63104 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 53: {
		_widec = (int)(112256 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 54: {
		_widec = (int)(72320 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 55: {
		_widec = (int)(121472 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 56: {
		_widec = (int)(149120 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 57: {
		_widec = (int)(78464 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		break;
	}
	case 58: {
		_widec = (int)(127616 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 59: {
		_widec = (int)(155264 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 60: {
		_widec = (int)(173696 + ((*p) - -128));
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 256;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 512;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 1024;
		break;
	}
	case 61: {
		_widec = (int)(30336 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 512;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 1024;
		break;
	}
	case 62: {
		_widec = (int)(40576 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 512;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 1024;
		break;
	}
	case 63: {
		_widec = (int)(43648 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 1024;
		break;
	}
	case 64: {
		_widec = (int)(45696 + ((*p) - -128));
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 512;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 1024;
		break;
	}
	case 65: {
		_widec = (int)(13952 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 2048;
		break;
	}
	case 66: {
		_widec = (int)(65152 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		break;
	}
	case 67: {
		_widec = (int)(114304 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 68: {
		_widec = (int)(74368 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		break;
	}
	case 69: {
		_widec = (int)(123520 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 70: {
		_widec = (int)(151168 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 71: {
		_widec = (int)(80512 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		break;
	}
	case 72: {
		_widec = (int)(129664 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 73: {
		_widec = (int)(157312 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 74: {
		_widec = (int)(175744 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 75: {
		_widec = (int)(84608 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		break;
	}
	case 76: {
		_widec = (int)(133760 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 77: {
		_widec = (int)(161408 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 78: {
		_widec = (int)(179840 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 79: {
		_widec = (int)(192128 + ((*p) - -128));
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 256;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 512;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 1024;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 2048;
		break;
	}
	case 80: {
		_widec = (int)(34432 + ((*p) - -128));
		if ( 
#line 236 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 256;
		if ( 
#line 237 "source/parser/TextFileParser.rl"
 configuration.cssOklch  ) _widec += 512;
		if ( 
#line 238 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 1024;
		if ( 
#line 239 "source/parser/TextFileParser.rl"
 configuration.cssOklab  ) _widec += 2048;
		break;
	}
	case 81: {
		_widec = (int)(47744 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 2048;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 4096;
		break;
	}
	case 82: {
		_widec = (int)(137856 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 2048;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 4096;
		break;
	}
	case 83: {
		_widec = (int)(213632 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 2048;
		if ( 
#line 244 "source/parser/TextFileParser.rl"
 configuration.singleLineHashComments  ) _widec += 4096;
		break;
	}
	case 84: {
		_widec = (int)(165504 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 4096;
		break;
	}
	case 85: {
		_widec = (int)(183936 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 4096;
		break;
	}
	case 86: {
		_widec = (int)(196224 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 4096;
		break;
	}
	case 87: {
		_widec = (int)(204416 + ((*p) - -128));
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 256;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 512;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 1024;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 2048;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 4096;
		break;
	}
	case 88: {
		_widec = (int)(88704 + ((*p) - -128));
		if ( 
#line 228 "source/parser/TextFileParser.rl"
 configuration.fullHexWithAlpha  ) _widec += 256;
		if ( 
#line 229 "source/parser/TextFileParser.rl"
 configuration.fullHex  ) _widec += 512;
		if ( 
#line 230 "source/parser/TextFileParser.rl"
 configuration.shortHexWithAlpha  ) _widec += 1024;
		if ( 
#line 231 "source/parser/TextFileParser.rl"
 configuration.shortHex  ) _widec += 2048;
		if ( 
#line 240 "source/parser/TextFileParser.rl"
 configuration.intValues  ) _widec += 4096;
		if ( 
#line 241 "source/parser/TextFileParser.rl"
 configuration.floatValues  ) _widec += 8192;
		break;
	}
				}
				break;
			}
		}
	}

	_keys = _text_file_trans_keys + _text_file_key_offsets[ fsm.cs];
	_trans = _text_file_index_offsets[ fsm.cs];

	_klen = _text_file_single_lengths[ fsm.cs];
	if ( _klen > 0 ) {
		const int *_lower = _keys;
		const int *_mid;
		const int *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( _widec < *_mid )
				_upper = _mid - 1;
			else if ( _widec > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (unsigned int)(_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _text_file_range_lengths[ fsm.cs];
	if ( _klen > 0 ) {
		const int *_lower = _keys;
		const int *_mid;
		const int *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += (unsigned int)((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _text_file_indicies[_trans];
_eof_trans:
	 fsm.cs = _text_file_trans_targs[_trans];

	if ( _text_file_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _text_file_actions + _text_file_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 216 "source/parser/TextFileParser.rl"
	{ fsm.numberI64 = 0; }
	break;
	case 1:
#line 216 "source/parser/TextFileParser.rl"
	{ fsm.numberI64 = fsm.numberI64 * 10 + (*p - '0'); }
	break;
	case 2:
#line 219 "source/parser/TextFileParser.rl"
	{ fsm.numbersI64.emplace_back(fsm.numberI64, Unit::unitless); }
	break;
	case 3:
#line 220 "source/parser/TextFileParser.rl"
	{ fsm.numberStart = p; }
	break;
	case 4:
#line 220 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.emplace_back(fsm.parseDouble(fsm.numberStart, p), Unit::unitless); }
	break;
	case 5:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numberStart = p; }
	break;
	case 6:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.emplace_back(fsm.parseDouble(fsm.numberStart, p), Unit::unitless); }
	break;
	case 7:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.back().second = Unit::degree; }
	break;
	case 8:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.back().second = Unit::gradian; }
	break;
	case 9:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.back().second = Unit::turn; }
	break;
	case 10:
#line 221 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.back().second = Unit::radian; }
	break;
	case 11:
#line 222 "source/parser/TextFileParser.rl"
	{ fsm.numberStart = p; }
	break;
	case 12:
#line 222 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.emplace_back(fsm.parseDouble(fsm.numberStart, p), Unit::percentage); }
	break;
	case 13:
#line 223 "source/parser/TextFileParser.rl"
	{ fsm.numberStart = p; }
	break;
	case 14:
#line 223 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.emplace_back(fsm.parseDouble(fsm.numberStart, p), Unit::unitless); }
	break;
	case 15:
#line 223 "source/parser/TextFileParser.rl"
	{ fsm.numbersDouble.back().second = Unit::percentage; }
	break;
	case 16:
#line 224 "source/parser/TextFileParser.rl"
	{ fsm.handleNewline(); }
	break;
	case 19:
#line 1 "NONE"
	{ fsm.te = p+1;}
	break;
	case 20:
#line 247 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ { fsm.cs = 832;goto _again;} }}
	break;
	case 21:
#line 248 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ }}
	break;
	case 22:
#line 249 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ }}
	break;
	case 23:
#line 248 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ }}
	break;
	case 24:
#line 252 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ { fsm.cs = 832;goto _again;} }}
	break;
	case 25:
#line 253 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ }}
	break;
	case 26:
#line 253 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ }}
	break;
	case 27:
#line 257 "source/parser/TextFileParser.rl"
	{ fsm.act = 7;}
	break;
	case 28:
#line 258 "source/parser/TextFileParser.rl"
	{ fsm.act = 8;}
	break;
	case 29:
#line 259 "source/parser/TextFileParser.rl"
	{ fsm.act = 9;}
	break;
	case 30:
#line 260 "source/parser/TextFileParser.rl"
	{ fsm.act = 10;}
	break;
	case 31:
#line 261 "source/parser/TextFileParser.rl"
	{ fsm.act = 11;}
	break;
	case 32:
#line 262 "source/parser/TextFileParser.rl"
	{ fsm.act = 12;}
	break;
	case 33:
#line 263 "source/parser/TextFileParser.rl"
	{ fsm.act = 13;}
	break;
	case 34:
#line 278 "source/parser/TextFileParser.rl"
	{ fsm.act = 28;}
	break;
	case 35:
#line 279 "source/parser/TextFileParser.rl"
	{ fsm.act = 29;}
	break;
	case 36:
#line 280 "source/parser/TextFileParser.rl"
	{ fsm.act = 30;}
	break;
	case 37:
#line 281 "source/parser/TextFileParser.rl"
	{ fsm.act = 31;}
	break;
	case 38:
#line 284 "source/parser/TextFileParser.rl"
	{ fsm.act = 34;}
	break;
	case 39:
#line 285 "source/parser/TextFileParser.rl"
	{ fsm.act = 35;}
	break;
	case 40:
#line 256 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexWithAlphaFull(true); }}
	break;
	case 41:
#line 257 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexFull(true); }}
	break;
	case 42:
#line 258 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexWithAlphaShort(true); }}
	break;
	case 43:
#line 259 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexShort(true); }}
	break;
	case 44:
#line 260 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexWithAlphaFull(false); }}
	break;
	case 45:
#line 261 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexFull(false); }}
	break;
	case 46:
#line 262 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexWithAlphaShort(false); }}
	break;
	case 47:
#line 263 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHexShort(false); }}
	break;
	case 48:
#line 264 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorRgb(); }}
	break;
	case 49:
#line 265 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorRgb(); }}
	break;
	case 50:
#line 266 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorRgb(); }}
	break;
	case 51:
#line 267 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorRgb(); }}
	break;
	case 52:
#line 268 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorRgb(); }}
	break;
	case 53:
#line 269 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHsl(); }}
	break;
	case 54:
#line 270 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHsl(); }}
	break;
	case 55:
#line 271 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHsl(); }}
	break;
	case 56:
#line 272 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHsl(); }}
	break;
	case 57:
#line 273 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorHsl(); }}
	break;
	case 58:
#line 274 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorOklch(); }}
	break;
	case 59:
#line 275 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorOklch(); }}
	break;
	case 60:
#line 276 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorOklab(); }}
	break;
	case 61:
#line 277 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.colorOklab(); }}
	break;
	case 62:
#line 282 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ { fsm.cs = 1040;goto _again;} }}
	break;
	case 63:
#line 283 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ { fsm.cs = 1037;goto _again;} }}
	break;
	case 64:
#line 284 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ { fsm.cs = 1040;goto _again;} }}
	break;
	case 65:
#line 285 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.clearNumberStacks(); }}
	break;
	case 66:
#line 286 "source/parser/TextFileParser.rl"
	{ fsm.te = p+1;{ fsm.clearNumberStacks(); }}
	break;
	case 67:
#line 257 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexFull(true); }}
	break;
	case 68:
#line 258 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexWithAlphaShort(true); }}
	break;
	case 69:
#line 259 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexShort(true); }}
	break;
	case 70:
#line 261 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexFull(false); }}
	break;
	case 71:
#line 262 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexWithAlphaShort(false); }}
	break;
	case 72:
#line 263 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorHexShort(false); }}
	break;
	case 73:
#line 278 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorValueIntegers(); }}
	break;
	case 74:
#line 279 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorValueIntegers(); }}
	break;
	case 75:
#line 280 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorValues(); }}
	break;
	case 76:
#line 281 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.colorValues(); }}
	break;
	case 77:
#line 285 "source/parser/TextFileParser.rl"
	{ fsm.te = p;p--;{ fsm.clearNumberStacks(); }}
	break;
	case 78:
#line 278 "source/parser/TextFileParser.rl"
	{{p = (( fsm.te))-1;}{ fsm.colorValueIntegers(); }}
	break;
	case 79:
#line 279 "source/parser/TextFileParser.rl"
	{{p = (( fsm.te))-1;}{ fsm.colorValueIntegers(); }}
	break;
	case 80:
#line 285 "source/parser/TextFileParser.rl"
	{{p = (( fsm.te))-1;}{ fsm.clearNumberStacks(); }}
	break;
	case 81:
#line 1 "NONE"
	{	switch(  fsm.act ) {
	case 7:
	{{p = (( fsm.te))-1;} fsm.colorHexFull(true); }
	break;
	case 8:
	{{p = (( fsm.te))-1;} fsm.colorHexWithAlphaShort(true); }
	break;
	case 9:
	{{p = (( fsm.te))-1;} fsm.colorHexShort(true); }
	break;
	case 10:
	{{p = (( fsm.te))-1;} fsm.colorHexWithAlphaFull(false); }
	break;
	case 11:
	{{p = (( fsm.te))-1;} fsm.colorHexFull(false); }
	break;
	case 12:
	{{p = (( fsm.te))-1;} fsm.colorHexWithAlphaShort(false); }
	break;
	case 13:
	{{p = (( fsm.te))-1;} fsm.colorHexShort(false); }
	break;
	case 28:
	{{p = (( fsm.te))-1;} fsm.colorValueIntegers(); }
	break;
	case 29:
	{{p = (( fsm.te))-1;} fsm.colorValueIntegers(); }
	break;
	case 30:
	{{p = (( fsm.te))-1;} fsm.colorValues(); }
	break;
	case 31:
	{{p = (( fsm.te))-1;} fsm.colorValues(); }
	break;
	case 34:
	{{p = (( fsm.te))-1;} { fsm.cs = 1040;goto _again;} }
	break;
	case 35:
	{{p = (( fsm.te))-1;} fsm.clearNumberStacks(); }
	break;
	}
	}
	break;
#line 9030 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
		}
	}

_again:
	_acts = _text_file_actions + _text_file_to_state_actions[ fsm.cs];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 ) {
		switch ( *_acts++ ) {
	case 17:
#line 1 "NONE"
	{ fsm.ts = 0;}
	break;
#line 9041 "/build/gpick/src/gpick/build/CMakeFiles/TextFileParser.cpp"
		}
	}

	if (  fsm.cs == 0 )
		goto _out;
	if ( ++p != pe )
		goto _resume;
	_test_eof: {}
	if ( p == eof )
	{
	if ( _text_file_eof_trans[ fsm.cs] > 0 ) {
		_trans = _text_file_eof_trans[ fsm.cs] - 1;
		goto _eof_trans;
	}
	}

	_out: {}
	}

#line 314 "source/parser/TextFileParser.rl"
			if (fsm.cs == text_file_error) {
				parseError = true;
				textFile.syntaxError(fsm.line, fsm.ts - fsm.buffer - fsm.lineStart, fsm.line, fsm.te - fsm.buffer - fsm.lineStart);
				break;
			}
			if (fsm.ts == 0) {
				have = 0;
				fsm.lineStart -= sizeof(fsm.buffer);
			} else {
				have = pe - fsm.ts;
				std::memmove(fsm.buffer, fsm.ts, have);
				int bufferMovement = fsm.ts - fsm.buffer;
				fsm.te -= bufferMovement;
				fsm.lineStart -= bufferMovement;
				fsm.ts = fsm.buffer;
				fsm.bufferOffset += fsm.ts - fsm.buffer;
			}
		} else {
			break;
		}
	}
	return parseError == false;
}
}
