// Copyright 2018 Drone.IO Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package api

import (
	"net/http"

	"github.com/gin-gonic/gin"

	"go.woodpecker-ci.org/woodpecker/v3/server"
	"go.woodpecker-ci.org/woodpecker/v3/server/model"
	"go.woodpecker-ci.org/woodpecker/v3/server/router/middleware/session"
)

// GetSecret
//
//	@Summary	Get a repository secret by name
//	@Router		/repos/{repo_id}/secrets/{secretName} [get]
//	@Produce	json
//	@Success	200	{object}	Secret
//	@Tags		Repository secrets
//	@Param		Authorization	header	string	true	"Insert your personal access token"	default(Bearer <personal access token>)
//	@Param		repo_id			path	int		true	"the repository id"
//	@Param		secretName		path	string	true	"the secret name"
func GetSecret(c *gin.Context) {
	repo := session.Repo(c)
	name := c.Param("secret")

	secretService := server.Config.Services.Manager.SecretServiceFromRepo(repo)
	secret, err := secretService.SecretFind(repo, name)
	if err != nil {
		handleDBError(c, err)
		return
	}
	c.JSON(http.StatusOK, secret.Copy())
}

// PostSecret
//
//	@Summary	Create a repository secret
//	@Router		/repos/{repo_id}/secrets [post]
//	@Produce	json
//	@Success	200	{object}	Secret
//	@Tags		Repository secrets
//	@Param		Authorization	header	string	true	"Insert your personal access token"	default(Bearer <personal access token>)
//	@Param		repo_id			path	int		true	"the repository id"
//	@Param		secret			body	Secret	true	"the new secret"
func PostSecret(c *gin.Context) {
	repo := session.Repo(c)

	in := new(model.Secret)
	if err := c.Bind(in); err != nil {
		c.String(http.StatusBadRequest, "Error parsing secret. %s", err)
		return
	}
	secret := &model.Secret{
		RepoID: repo.ID,
		Name:   in.Name,
		Value:  in.Value,
		Events: in.Events,
		Images: in.Images,
		Note:   in.Note,
	}
	if err := secret.Validate(); err != nil {
		c.String(http.StatusUnprocessableEntity, "Error inserting secret. %s", err)
		return
	}

	secretService := server.Config.Services.Manager.SecretServiceFromRepo(repo)
	if err := secretService.SecretCreate(repo, secret); err != nil {
		c.String(http.StatusInternalServerError, "Error inserting secret %q. %s", in.Name, err)
		return
	}
	c.JSON(http.StatusOK, secret.Copy())
}

// PatchSecret
//
//	@Summary	Update a repository secret by name
//	@Router		/repos/{repo_id}/secrets/{secretName} [patch]
//	@Produce	json
//	@Success	200	{object}	Secret
//	@Tags		Repository secrets
//	@Param		Authorization	header	string		true	"Insert your personal access token"	default(Bearer <personal access token>)
//	@Param		repo_id			path	int			true	"the repository id"
//	@Param		secretName		path	string		true	"the secret name"
//	@Param		secret			body	SecretPatch	true	"the secret itself"
func PatchSecret(c *gin.Context) {
	var (
		repo = session.Repo(c)
		name = c.Param("secret")
	)

	in := new(model.SecretPatch)
	err := c.Bind(in)
	if err != nil {
		c.String(http.StatusBadRequest, "Error parsing secret. %s", err)
		return
	}

	secretService := server.Config.Services.Manager.SecretServiceFromRepo(repo)
	secret, err := secretService.SecretFind(repo, name)
	if err != nil {
		handleDBError(c, err)
		return
	}
	if in.Value != nil && *in.Value != "" {
		secret.Value = *in.Value
	}
	if in.Events != nil {
		secret.Events = in.Events
	}
	if in.Images != nil {
		secret.Images = in.Images
	}
	if in.Note != nil {
		secret.Note = *in.Note
	}

	if err := secret.Validate(); err != nil {
		c.String(http.StatusUnprocessableEntity, "Error updating secret. %s", err)
		return
	}
	if err := secretService.SecretUpdate(repo, secret); err != nil {
		c.String(http.StatusInternalServerError, "Error updating secret %q. %s", in.Name, err)
		return
	}
	c.JSON(http.StatusOK, secret.Copy())
}

// GetSecretList
//
//	@Summary	List repository secrets
//	@Router		/repos/{repo_id}/secrets [get]
//	@Produce	json
//	@Success	200	{array}	Secret
//	@Tags		Repository secrets
//	@Param		Authorization	header	string	true	"Insert your personal access token"	default(Bearer <personal access token>)
//	@Param		repo_id			path	int		true	"the repository id"
//	@Param		page			query	int		false	"for response pagination, page offset number"	default(1)
//	@Param		perPage			query	int		false	"for response pagination, max items per page"	default(50)
func GetSecretList(c *gin.Context) {
	repo := session.Repo(c)
	secretService := server.Config.Services.Manager.SecretServiceFromRepo(repo)
	list, err := secretService.SecretList(repo, session.Pagination(c))
	if err != nil {
		c.String(http.StatusInternalServerError, "Error getting secret list. %s", err)
		return
	}
	// copy the secret detail to remove the sensitive
	// password and token fields.
	for i, secret := range list {
		list[i] = secret.Copy()
	}
	c.JSON(http.StatusOK, list)
}

// DeleteSecret
//
//	@Summary	Delete a repository secret by name
//	@Router		/repos/{repo_id}/secrets/{secretName} [delete]
//	@Produce	plain
//	@Success	204
//	@Tags		Repository secrets
//	@Param		Authorization	header	string	true	"Insert your personal access token"	default(Bearer <personal access token>)
//	@Param		repo_id			path	int		true	"the repository id"
//	@Param		secretName		path	string	true	"the secret name"
func DeleteSecret(c *gin.Context) {
	repo := session.Repo(c)
	name := c.Param("secret")

	secretService := server.Config.Services.Manager.SecretServiceFromRepo(repo)
	if err := secretService.SecretDelete(repo, name); err != nil {
		handleDBError(c, err)
		return
	}
	c.Status(http.StatusNoContent)
}
