﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticmapreduce/EMRRequest.h>
#include <aws/elasticmapreduce/EMR_EXPORTS.h>
#include <aws/elasticmapreduce/model/AutoScalingPolicy.h>

#include <utility>

namespace Aws {
namespace EMR {
namespace Model {

/**
 */
class PutAutoScalingPolicyRequest : public EMRRequest {
 public:
  AWS_EMR_API PutAutoScalingPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutAutoScalingPolicy"; }

  AWS_EMR_API Aws::String SerializePayload() const override;

  AWS_EMR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the ID of a cluster. The instance group to which the automatic
   * scaling policy is applied is within this cluster.</p>
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  PutAutoScalingPolicyRequest& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the ID of the instance group to which the automatic scaling policy
   * is applied.</p>
   */
  inline const Aws::String& GetInstanceGroupId() const { return m_instanceGroupId; }
  inline bool InstanceGroupIdHasBeenSet() const { return m_instanceGroupIdHasBeenSet; }
  template <typename InstanceGroupIdT = Aws::String>
  void SetInstanceGroupId(InstanceGroupIdT&& value) {
    m_instanceGroupIdHasBeenSet = true;
    m_instanceGroupId = std::forward<InstanceGroupIdT>(value);
  }
  template <typename InstanceGroupIdT = Aws::String>
  PutAutoScalingPolicyRequest& WithInstanceGroupId(InstanceGroupIdT&& value) {
    SetInstanceGroupId(std::forward<InstanceGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the definition of the automatic scaling policy.</p>
   */
  inline const AutoScalingPolicy& GetAutoScalingPolicy() const { return m_autoScalingPolicy; }
  inline bool AutoScalingPolicyHasBeenSet() const { return m_autoScalingPolicyHasBeenSet; }
  template <typename AutoScalingPolicyT = AutoScalingPolicy>
  void SetAutoScalingPolicy(AutoScalingPolicyT&& value) {
    m_autoScalingPolicyHasBeenSet = true;
    m_autoScalingPolicy = std::forward<AutoScalingPolicyT>(value);
  }
  template <typename AutoScalingPolicyT = AutoScalingPolicy>
  PutAutoScalingPolicyRequest& WithAutoScalingPolicy(AutoScalingPolicyT&& value) {
    SetAutoScalingPolicy(std::forward<AutoScalingPolicyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterId;

  Aws::String m_instanceGroupId;

  AutoScalingPolicy m_autoScalingPolicy;
  bool m_clusterIdHasBeenSet = false;
  bool m_instanceGroupIdHasBeenSet = false;
  bool m_autoScalingPolicyHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMR
}  // namespace Aws
