#!/bin/bash
#
################################################################
#
# Used to build a MINI LiveCD
# - save diskspace by moving folders to /mini
#   disable this with "-nomove" option !
# - direct login to icewm as LOCALUSER
#   or as root (option -rootlogin)
# - configure icewm
# - run "mini_livecd -restore" to restore files from /mini again
#
# Urs Beyerle, PSI
#
################################################################

### -----------------------------------------------------------
### List of folders (or files) to move:
### -----------------------------------------------------------
FOLDERS="/usr/src \
         /usr/share/doc \
         /usr/share/info \
         /usr/share/anaconda \
         /usr/share/gdm \
         /var/cache/yum \
         /var/lib/rpm \
         /etc/cups/ppds.dat \
         /etc/gconf/gconf.xml.defaults \
         /usr/X11R6/lib/X11/doc \
         /usr/share/vim/vim63/doc \
         /usr/share/vim/vim63/lang \
         /usr/share/emacs/21.3/etc/DOC-21.3.1 \
         /usr/share/pixmaps/nautilus \
         /usr/share/cups \
	 /usr/share/icons"

### -----------------------------------------------------------
### locales we don't need, we keep: en, de, fr
### ----------------------------------------------------------- 

### /usr/share/locale
### ----------------------------------------------------------- 

# do we restore?
if [ -d /mini/usr/share/locale/zh_TW.Big5 ]; then
    MOVE_LOCALE=$( find /mini/usr/share/locale/* -type d -maxdepth 0 \
	      | while read f; do echo ${f##/mini}; done )
else
    MOVE_LOCALE=$( find /usr/share/locale/* -type d -maxdepth 0 \
              | grep -v en | grep -v de | grep -v fr )
fi

# add to FOLDERS
FOLDERS="$FOLDERS $MOVE_LOCALE"      

### /usr/lib/locale
### ----------------------------------------------------------- 

# do we restore?
if [ -d /mini/usr/lib/locale/zh_TW ]; then
    MOVE_LOCALE=$( find /mini/usr/lib/locale/* -type d -maxdepth 0 \
	      | while read f; do echo ${f##/mini}; done )
else
    MOVE_LOCALE=$( find /usr/lib/locale/* -type d -maxdepth 0 \
              | grep -v en | grep -v de | grep -v fr )
fi

# add to FOLDERS
FOLDERS="$FOLDERS $MOVE_LOCALE"      

### /usr/share/i18n/locales
### ----------------------------------------------------------- 

# do we restore?
if [ -d /mini/usr/share/i18n/locales/zh_TW ]; then
    MOVE_LOCALE=$( find /mini/usr/share/i18n/locales/* -type f -maxdepth 0 \
	      | while read f; do echo ${f##/mini}; done )
else
    MOVE_LOCALE=$( find /usr/share/i18n/locales/* -type f -maxdepth 0 \
              | grep -v en | grep -v de | grep -v fr | grep -v POSIX )
fi

# add to FOLDERS
FOLDERS="$FOLDERS $MOVE_LOCALE"      


### -----------------------------------------------------------
function usage() {

   ## Usage
   # ----------------------------------------------------------

   cat <<EOF

   Options:

    -h:         print this screen
    -restore:   restores the folders saved in /mini
    -nomove:    do not move to /mini
    -rootlogin: login directly as root and not as LOCALUSER
    -psi:       build for PSI

EOF

}


### -----------------------------------------------------------
### read options from command-line
RESTORE=""
while [ $# -gt 0 ]; do

    case "$1" in
       -h)
            usage; exit;;
       -restore)
            RESTORE="-restore"; shift; continue;;
       -psi)
            PSI="-psi"; shift; continue;;
       -nomove)
            NOMOVE="-nomove"; shift; continue;;
       -rootlogin)
            ROOTLOGIN="-rootlogin"; shift; continue;;
       *)
            usage; exit;;
    esac

done


### -----------------------------------------------------------
### LOCALUSER name?
if [ $PSI ]; then 
    LOCALUSER=l_psi
else
    LOCALUSER=centos_user
fi


### -----------------------------------------------------------
### Modifications for Mini LivdCD
if [ ! $RESTORE ]; then
    
    if [ ! $NOMOVE ]; then
	echo
	echo "--------------------------------------------"
	echo "Save diskspace by moving folders to /mini:"
	echo "--------------------------------------------"

	mkdir -p /mini
	for folder in $FOLDERS; do
	    echo "Move $folder"
	    name=$( basename $folder )
	    base=${folder%$name}
	    mkdir -p /mini/${base}
	    mv ${folder} /mini${base}
	done
    fi

    ### emacs-21.3 and emacs are hard links (and large 4.3MB) 
    if [ -f /usr/bin/emacs-21.3 ]; then
	rm /usr/bin/emacs-21.3
	ln -s emacs /usr/bin/emacs-21.3
    fi
 
    ### run updatedb
    echo "Run updatedb..."
    /usr/bin/updatedb -e /tmp,/boot,/livecd,/home,/mini
    echo "done."

    ### change init runlevel to 3
    sed -i "s/id\:.\:initdefault/id\:3\:initdefault/" /etc/inittab

    ### autologin at tty6
    if [ $ROOTLOGIN ]; then
	line="6:2345:respawn:\/sbin\/mingetty --autologin root tty6"
	# modify .bashrc for root
	grep -q "/dev/tty6" /root/.bashrc
	if [ "$?" = "1" ]; then
	    echo 'if [ "$( tty )" = "/dev/tty6" ]; then startx; fi' >> /root/.bashrc
	fi
    else
	line="6:2345:respawn:\/sbin\/mingetty --autologin $LOCALUSER tty6"
    fi
    sed -i "s/.*respawn.*tty6.*/$line/" /etc/inittab
    
    ### modify /etc/rc.d/init.d/runlast
    RUNLAST=/etc/rc.d/init.d/runlast
    echo >> $RUNLAST
    echo "### execute /etc/init.d/login for direct login" >> $RUNLAST
    echo "/etc/init.d/login" >> $RUNLAST
    echo >> $RUNLAST

    ### remove kde profile
    rm -f /etc/profile.d/kde*

    ### configure ICEWM
    cp -a customize/icewm/winoptions /usr/share/icewm/
    cp -a customize/icewm/toolbar /usr/share/icewm/
    cp -a customize/icewm/menu /usr/share/icewm/

    ### icons for icewm
    cp -a /usr/share/pixmaps/qtparted.xpm /usr/share/icewm/icons/ 2>/dev/null
    cp -a /usr/share/gftp/gftp.xpm /usr/share/icewm/icons/ 2>/dev/null

    ### no rdesktop menu entry on non PSI systems
    if [ ! $PSI ];then
	sed -i "/prog Rdesktop.*/d" /usr/share/icewm/menu
    fi

    ### create default html page
    mkdir -p /usr/share/doc/HTML/
    if [ $PSI ]; then
	cp -a customize/HTML/index_psi.html /usr/share/doc/HTML/index.html
    else
	cp -a customize/HTML/index.html /usr/share/doc/HTML/index.html
    fi

    ### create some pam.d files
    # cp -a /etc/pam.d/system-config-display /etc/pam.d/partimage


### -----------------------------------------------------------
### Restores some modifications
else

    if [ ! $NOMOVE ]; then
        ### remove default html page
	rm /usr/share/doc/HTML/index.html
	rmdir /usr/share/doc/HTML
	rmdir /usr/share/doc

	echo
	echo "--------------------------------------------"
	echo "Restore folders saved in /mini:"
	echo "--------------------------------------------"
	for folder in $FOLDERS; do
	    echo "Restore $folder"
	    name=$( basename $folder )
	    base=${folder%$name}
	    mv /mini${folder} ${base}
	done
	mv /mini/usr/share/i18n/locales/* /usr/share/i18n/locales/
	echo "done."
    fi
fi


echo "--------------------------------------------"
echo "End of mini_livecd.sh" 
echo "--------------------------------------------"
echo
