#ifndef CAPSTONE_AARCH64_H
#define CAPSTONE_AARCH64_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2013-2015 */

#ifdef __cplusplus
extern "C" {
#endif

#include "cs_operand.h"
#include "platform.h"

#include <assert.h>

#ifdef _MSC_VER
#pragma warning(disable : 4201)
#endif

/// AArch64 shift type
/// Those values do not correspond to the bit values encoded
/// in the instruction.
typedef enum aarch64_shifter {
  AARCH64_SFT_INVALID = 0,
  AARCH64_SFT_LSL = 1,
  AARCH64_SFT_MSL = 2,
  AARCH64_SFT_LSR = 3,
  AARCH64_SFT_ASR = 4,
  AARCH64_SFT_ROR = 5,
  AARCH64_SFT_LSL_REG = 6,
  AARCH64_SFT_MSL_REG = 7,
  AARCH64_SFT_LSR_REG = 8,
  AARCH64_SFT_ASR_REG = 9,
  AARCH64_SFT_ROR_REG = 10,
} aarch64_shifter;

/// AArch64 extender type
typedef enum aarch64_extender {
  AARCH64_EXT_INVALID = 0,
  AARCH64_EXT_UXTB = 1,
  AARCH64_EXT_UXTH = 2,
  AARCH64_EXT_UXTW = 3,
  AARCH64_EXT_UXTX = 4,
  AARCH64_EXT_SXTB = 5,
  AARCH64_EXT_SXTH = 6,
  AARCH64_EXT_SXTW = 7,
  AARCH64_EXT_SXTX = 8,
} aarch64_extender;

// Moved from AArch64BaseInfo.h and modified
// With extension of Q
typedef enum VectorLayout {
	AARCH64LAYOUT_INVALID = 0,
	// Bare layout for the 128-bit vector
	// (only show ".b", ".h", ".s", ".d" without vector number)
	AARCH64LAYOUT_VL_B = 8,
	AARCH64LAYOUT_VL_H = 16,
	AARCH64LAYOUT_VL_S = 32,
	AARCH64LAYOUT_VL_D = 64,
	AARCH64LAYOUT_VL_Q = 128,

	AARCH64LAYOUT_VL_4B = (4 << 8) | AARCH64LAYOUT_VL_B,
	AARCH64LAYOUT_VL_2H = (2 << 8) | AARCH64LAYOUT_VL_H,
	AARCH64LAYOUT_VL_1S = (1 << 8) | AARCH64LAYOUT_VL_S,

	AARCH64LAYOUT_VL_8B = (8 << 8) | AARCH64LAYOUT_VL_B,
	AARCH64LAYOUT_VL_4H = (4 << 8) | AARCH64LAYOUT_VL_H,
	AARCH64LAYOUT_VL_2S = (2 << 8) | AARCH64LAYOUT_VL_S,
	AARCH64LAYOUT_VL_1D = (1 << 8) | AARCH64LAYOUT_VL_D,

	AARCH64LAYOUT_VL_16B = (16 << 8) | AARCH64LAYOUT_VL_B,
	AARCH64LAYOUT_VL_8H = (8 << 8) | AARCH64LAYOUT_VL_H,
	AARCH64LAYOUT_VL_4S = (4 << 8) | AARCH64LAYOUT_VL_S,
	AARCH64LAYOUT_VL_2D = (2 << 8) | AARCH64LAYOUT_VL_D,
	AARCH64LAYOUT_VL_1Q = (1 << 8) | AARCH64LAYOUT_VL_Q,

	AARCH64LAYOUT_VL_64B = (64 << 8) | AARCH64LAYOUT_VL_B,
	AARCH64LAYOUT_VL_32H = (32 << 8) | AARCH64LAYOUT_VL_H,
	AARCH64LAYOUT_VL_16S = (16 << 8) | AARCH64LAYOUT_VL_S,
	AARCH64LAYOUT_VL_8D = (8 << 8) | AARCH64LAYOUT_VL_D,

	AARCH64LAYOUT_VL_COMPLETE, ///< Indicates that the complete matrix is used.
} AArch64Layout_VectorLayout;

// begin namespace AArch64CC

// Moved from AArch64BaseInfo.h
// The CondCodes constants map directly to the 4-bit encoding of the condition
// field for predicated instructions.
typedef enum AArch64CondCode { // Meaning (integer) Meaning (floating-point)
	AArch64CC_EQ = 0x0, // Equal                      Equal
	AArch64CC_NE = 0x1, // Not equal                  Not equal, or unordered
	AArch64CC_HS = 0x2, // Unsigned higher or same    >, ==, or unordered
	AArch64CC_LO = 0x3, // Unsigned lower             Less than
	AArch64CC_MI = 0x4, // Minus, negative            Less than
	AArch64CC_PL = 0x5, // Plus, positive or zero     >, ==, or unordered
	AArch64CC_VS = 0x6, // Overflow                   Unordered
	AArch64CC_VC = 0x7, // No overflow                Not unordered
	AArch64CC_HI = 0x8, // Unsigned higher            Greater than, or unordered
	AArch64CC_LS = 0x9, // Unsigned lower or same     Less than or equal
	AArch64CC_GE = 0xa, // Greater than or equal      Greater than or equal
	AArch64CC_LT = 0xb, // Less than                  Less than, or unordered
	AArch64CC_GT = 0xc, // Greater than               Greater than
	AArch64CC_LE = 0xd, // Less than or equal         <, ==, or unordered
	AArch64CC_AL = 0xe, // Always (unconditional)     Always (unconditional)
	AArch64CC_NV = 0xf, // Always (unconditional)     Always (unconditional)
	// Note the NV exists purely to disassemble 0b1111. Execution is "always".
	AArch64CC_Invalid,

	// Common aliases used for SVE.
	AArch64CC_ANY_ACTIVE = AArch64CC_NE,	 // (!Z)
	AArch64CC_FIRST_ACTIVE = AArch64CC_MI, // ( N)
	AArch64CC_LAST_ACTIVE = AArch64CC_LO,	 // (!C)
	AArch64CC_NONE_ACTIVE = AArch64CC_EQ,  // ( Z)
} AArch64CC_CondCode;

inline static const char *AArch64CC_getCondCodeName(AArch64CC_CondCode Code)
{
	switch (Code) {
	default:
		assert(0 && "Unknown condition code");
	case AArch64CC_EQ:
		return "eq";
	case AArch64CC_NE:
		return "ne";
	case AArch64CC_HS:
		return "hs";
	case AArch64CC_LO:
		return "lo";
	case AArch64CC_MI:
		return "mi";
	case AArch64CC_PL:
		return "pl";
	case AArch64CC_VS:
		return "vs";
	case AArch64CC_VC:
		return "vc";
	case AArch64CC_HI:
		return "hi";
	case AArch64CC_LS:
		return "ls";
	case AArch64CC_GE:
		return "ge";
	case AArch64CC_LT:
		return "lt";
	case AArch64CC_GT:
		return "gt";
	case AArch64CC_LE:
		return "le";
	case AArch64CC_AL:
		return "al";
	case AArch64CC_NV:
		return "nv";
	}
}

inline static AArch64CC_CondCode AArch64CC_getInvertedCondCode(AArch64CC_CondCode Code)
{
	// To reverse a condition it's necessary to only invert the low bit:

	return (AArch64CC_CondCode)((unsigned)(Code) ^ 0x1);
}

/// Given a condition code, return NZCV flags that would satisfy that condition.
/// The flag bits are in the format expected by the ccmp instructions.
/// Note that many different flag settings can satisfy a given condition code,
/// this function just returns one of them.
inline static unsigned AArch64CC_getNZCVToSatisfyCondCode(AArch64CC_CondCode Code)
{
	// NZCV flags encoded as expected by ccmp instructions, ARMv8 ISA 5.5.7.
	enum { N = 8, Z = 4, C = 2, V = 1 };
	switch (Code) {
	default:
		assert(0 && "Unknown condition code");
	case AArch64CC_EQ:
		return Z; // Z == 1
	case AArch64CC_NE:
		return 0; // Z == 0
	case AArch64CC_HS:
		return C; // C == 1
	case AArch64CC_LO:
		return 0; // C == 0
	case AArch64CC_MI:
		return N; // N == 1
	case AArch64CC_PL:
		return 0; // N == 0
	case AArch64CC_VS:
		return V; // V == 1
	case AArch64CC_VC:
		return 0; // V == 0
	case AArch64CC_HI:
		return C; // C == 1 && Z == 0
	case AArch64CC_LS:
		return 0; // C == 0 || Z == 1
	case AArch64CC_GE:
		return 0; // N == V
	case AArch64CC_LT:
		return N; // N != V
	case AArch64CC_GT:
		return 0; // Z == 0 && N == V
	case AArch64CC_LE:
		return Z; // Z == 1 || N != V
	}
}

/// Return true if Code is a reflexive relationship:
/// forall x. (CSET Code (CMP x x)) == 1
inline static bool AArch64CC_isReflexive(AArch64CC_CondCode Code)
{
	switch (Code) {
	case AArch64CC_EQ:
	case AArch64CC_HS:
	case AArch64CC_PL:
	case AArch64CC_LS:
	case AArch64CC_GE:
	case AArch64CC_LE:
	case AArch64CC_AL:
	case AArch64CC_NV:
		return true;
	default:
		return false;
	}
}

/// Return true if Code is an irreflexive relationship:
/// forall x. (CSET Code (CMP x x)) == 0
inline static bool AArch64CC_isIrreflexive(AArch64CC_CondCode Code)
{
	switch (Code) {
	case AArch64CC_NE:
	case AArch64CC_LO:
	case AArch64CC_MI:
	case AArch64CC_HI:
	case AArch64CC_LT:
	case AArch64CC_GT:
		return true;
	default:
		return false;
	}
}

// end namespace AArch64CC

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_TLBI> begin
	// clang-format off

	AARCH64_TLBI_ALLE1 = 0x243c,
	AARCH64_TLBI_ALLE1IS = 0x241c,
	AARCH64_TLBI_ALLE1ISNXS = 0x249c,
	AARCH64_TLBI_ALLE1NXS = 0x24bc,
	AARCH64_TLBI_ALLE1OS = 0x240c,
	AARCH64_TLBI_ALLE1OSNXS = 0x248c,
	AARCH64_TLBI_ALLE2 = 0x2438,
	AARCH64_TLBI_ALLE2IS = 0x2418,
	AARCH64_TLBI_ALLE2ISNXS = 0x2498,
	AARCH64_TLBI_ALLE2NXS = 0x24b8,
	AARCH64_TLBI_ALLE2OS = 0x2408,
	AARCH64_TLBI_ALLE2OSNXS = 0x2488,
	AARCH64_TLBI_ALLE3 = 0x3438,
	AARCH64_TLBI_ALLE3IS = 0x3418,
	AARCH64_TLBI_ALLE3ISNXS = 0x3498,
	AARCH64_TLBI_ALLE3NXS = 0x34b8,
	AARCH64_TLBI_ALLE3OS = 0x3408,
	AARCH64_TLBI_ALLE3OSNXS = 0x3488,
	AARCH64_TLBI_ASIDE1 = 0x43a,
	AARCH64_TLBI_ASIDE1IS = 0x41a,
	AARCH64_TLBI_ASIDE1ISNXS = 0x49a,
	AARCH64_TLBI_ASIDE1NXS = 0x4ba,
	AARCH64_TLBI_ASIDE1OS = 0x40a,
	AARCH64_TLBI_ASIDE1OSNXS = 0x48a,
	AARCH64_TLBI_IPAS2E1 = 0x2421,
	AARCH64_TLBI_IPAS2E1IS = 0x2401,
	AARCH64_TLBI_IPAS2E1ISNXS = 0x2481,
	AARCH64_TLBI_IPAS2E1NXS = 0x24a1,
	AARCH64_TLBI_IPAS2E1OS = 0x2420,
	AARCH64_TLBI_IPAS2E1OSNXS = 0x24a0,
	AARCH64_TLBI_IPAS2LE1 = 0x2425,
	AARCH64_TLBI_IPAS2LE1IS = 0x2405,
	AARCH64_TLBI_IPAS2LE1ISNXS = 0x2485,
	AARCH64_TLBI_IPAS2LE1NXS = 0x24a5,
	AARCH64_TLBI_IPAS2LE1OS = 0x2424,
	AARCH64_TLBI_IPAS2LE1OSNXS = 0x24a4,
	AARCH64_TLBI_PAALL = 0x343c,
	AARCH64_TLBI_PAALLNXS = 0x34bc,
	AARCH64_TLBI_PAALLOS = 0x340c,
	AARCH64_TLBI_PAALLOSNXS = 0x348c,
	AARCH64_TLBI_RIPAS2E1 = 0x2422,
	AARCH64_TLBI_RIPAS2E1IS = 0x2402,
	AARCH64_TLBI_RIPAS2E1ISNXS = 0x2482,
	AARCH64_TLBI_RIPAS2E1NXS = 0x24a2,
	AARCH64_TLBI_RIPAS2E1OS = 0x2423,
	AARCH64_TLBI_RIPAS2E1OSNXS = 0x24a3,
	AARCH64_TLBI_RIPAS2LE1 = 0x2426,
	AARCH64_TLBI_RIPAS2LE1IS = 0x2406,
	AARCH64_TLBI_RIPAS2LE1ISNXS = 0x2486,
	AARCH64_TLBI_RIPAS2LE1NXS = 0x24a6,
	AARCH64_TLBI_RIPAS2LE1OS = 0x2427,
	AARCH64_TLBI_RIPAS2LE1OSNXS = 0x24a7,
	AARCH64_TLBI_RPALOS = 0x3427,
	AARCH64_TLBI_RPALOSNXS = 0x34a7,
	AARCH64_TLBI_RPAOS = 0x3423,
	AARCH64_TLBI_RPAOSNXS = 0x34a3,
	AARCH64_TLBI_RVAAE1 = 0x433,
	AARCH64_TLBI_RVAAE1IS = 0x413,
	AARCH64_TLBI_RVAAE1ISNXS = 0x493,
	AARCH64_TLBI_RVAAE1NXS = 0x4b3,
	AARCH64_TLBI_RVAAE1OS = 0x42b,
	AARCH64_TLBI_RVAAE1OSNXS = 0x4ab,
	AARCH64_TLBI_RVAALE1 = 0x437,
	AARCH64_TLBI_RVAALE1IS = 0x417,
	AARCH64_TLBI_RVAALE1ISNXS = 0x497,
	AARCH64_TLBI_RVAALE1NXS = 0x4b7,
	AARCH64_TLBI_RVAALE1OS = 0x42f,
	AARCH64_TLBI_RVAALE1OSNXS = 0x4af,
	AARCH64_TLBI_RVAE1 = 0x431,
	AARCH64_TLBI_RVAE1IS = 0x411,
	AARCH64_TLBI_RVAE1ISNXS = 0x491,
	AARCH64_TLBI_RVAE1NXS = 0x4b1,
	AARCH64_TLBI_RVAE1OS = 0x429,
	AARCH64_TLBI_RVAE1OSNXS = 0x4a9,
	AARCH64_TLBI_RVAE2 = 0x2431,
	AARCH64_TLBI_RVAE2IS = 0x2411,
	AARCH64_TLBI_RVAE2ISNXS = 0x2491,
	AARCH64_TLBI_RVAE2NXS = 0x24b1,
	AARCH64_TLBI_RVAE2OS = 0x2429,
	AARCH64_TLBI_RVAE2OSNXS = 0x24a9,
	AARCH64_TLBI_RVAE3 = 0x3431,
	AARCH64_TLBI_RVAE3IS = 0x3411,
	AARCH64_TLBI_RVAE3ISNXS = 0x3491,
	AARCH64_TLBI_RVAE3NXS = 0x34b1,
	AARCH64_TLBI_RVAE3OS = 0x3429,
	AARCH64_TLBI_RVAE3OSNXS = 0x34a9,
	AARCH64_TLBI_RVALE1 = 0x435,
	AARCH64_TLBI_RVALE1IS = 0x415,
	AARCH64_TLBI_RVALE1ISNXS = 0x495,
	AARCH64_TLBI_RVALE1NXS = 0x4b5,
	AARCH64_TLBI_RVALE1OS = 0x42d,
	AARCH64_TLBI_RVALE1OSNXS = 0x4ad,
	AARCH64_TLBI_RVALE2 = 0x2435,
	AARCH64_TLBI_RVALE2IS = 0x2415,
	AARCH64_TLBI_RVALE2ISNXS = 0x2495,
	AARCH64_TLBI_RVALE2NXS = 0x24b5,
	AARCH64_TLBI_RVALE2OS = 0x242d,
	AARCH64_TLBI_RVALE2OSNXS = 0x24ad,
	AARCH64_TLBI_RVALE3 = 0x3435,
	AARCH64_TLBI_RVALE3IS = 0x3415,
	AARCH64_TLBI_RVALE3ISNXS = 0x3495,
	AARCH64_TLBI_RVALE3NXS = 0x34b5,
	AARCH64_TLBI_RVALE3OS = 0x342d,
	AARCH64_TLBI_RVALE3OSNXS = 0x34ad,
	AARCH64_TLBI_VAAE1 = 0x43b,
	AARCH64_TLBI_VAAE1IS = 0x41b,
	AARCH64_TLBI_VAAE1ISNXS = 0x49b,
	AARCH64_TLBI_VAAE1NXS = 0x4bb,
	AARCH64_TLBI_VAAE1OS = 0x40b,
	AARCH64_TLBI_VAAE1OSNXS = 0x48b,
	AARCH64_TLBI_VAALE1 = 0x43f,
	AARCH64_TLBI_VAALE1IS = 0x41f,
	AARCH64_TLBI_VAALE1ISNXS = 0x49f,
	AARCH64_TLBI_VAALE1NXS = 0x4bf,
	AARCH64_TLBI_VAALE1OS = 0x40f,
	AARCH64_TLBI_VAALE1OSNXS = 0x48f,
	AARCH64_TLBI_VAE1 = 0x439,
	AARCH64_TLBI_VAE1IS = 0x419,
	AARCH64_TLBI_VAE1ISNXS = 0x499,
	AARCH64_TLBI_VAE1NXS = 0x4b9,
	AARCH64_TLBI_VAE1OS = 0x409,
	AARCH64_TLBI_VAE1OSNXS = 0x489,
	AARCH64_TLBI_VAE2 = 0x2439,
	AARCH64_TLBI_VAE2IS = 0x2419,
	AARCH64_TLBI_VAE2ISNXS = 0x2499,
	AARCH64_TLBI_VAE2NXS = 0x24b9,
	AARCH64_TLBI_VAE2OS = 0x2409,
	AARCH64_TLBI_VAE2OSNXS = 0x2489,
	AARCH64_TLBI_VAE3 = 0x3439,
	AARCH64_TLBI_VAE3IS = 0x3419,
	AARCH64_TLBI_VAE3ISNXS = 0x3499,
	AARCH64_TLBI_VAE3NXS = 0x34b9,
	AARCH64_TLBI_VAE3OS = 0x3409,
	AARCH64_TLBI_VAE3OSNXS = 0x3489,
	AARCH64_TLBI_VALE1 = 0x43d,
	AARCH64_TLBI_VALE1IS = 0x41d,
	AARCH64_TLBI_VALE1ISNXS = 0x49d,
	AARCH64_TLBI_VALE1NXS = 0x4bd,
	AARCH64_TLBI_VALE1OS = 0x40d,
	AARCH64_TLBI_VALE1OSNXS = 0x48d,
	AARCH64_TLBI_VALE2 = 0x243d,
	AARCH64_TLBI_VALE2IS = 0x241d,
	AARCH64_TLBI_VALE2ISNXS = 0x249d,
	AARCH64_TLBI_VALE2NXS = 0x24bd,
	AARCH64_TLBI_VALE2OS = 0x240d,
	AARCH64_TLBI_VALE2OSNXS = 0x248d,
	AARCH64_TLBI_VALE3 = 0x343d,
	AARCH64_TLBI_VALE3IS = 0x341d,
	AARCH64_TLBI_VALE3ISNXS = 0x349d,
	AARCH64_TLBI_VALE3NXS = 0x34bd,
	AARCH64_TLBI_VALE3OS = 0x340d,
	AARCH64_TLBI_VALE3OSNXS = 0x348d,
	AARCH64_TLBI_VMALLE1 = 0x438,
	AARCH64_TLBI_VMALLE1IS = 0x418,
	AARCH64_TLBI_VMALLE1ISNXS = 0x498,
	AARCH64_TLBI_VMALLE1NXS = 0x4b8,
	AARCH64_TLBI_VMALLE1OS = 0x408,
	AARCH64_TLBI_VMALLE1OSNXS = 0x488,
	AARCH64_TLBI_VMALLS12E1 = 0x243e,
	AARCH64_TLBI_VMALLS12E1IS = 0x241e,
	AARCH64_TLBI_VMALLS12E1ISNXS = 0x249e,
	AARCH64_TLBI_VMALLS12E1NXS = 0x24be,
	AARCH64_TLBI_VMALLS12E1OS = 0x240e,
	AARCH64_TLBI_VMALLS12E1OSNXS = 0x248e,
	AARCH64_TLBI_VMALLWS2E1 = 0x2432,
	AARCH64_TLBI_VMALLWS2E1IS = 0x2412,
	AARCH64_TLBI_VMALLWS2E1ISNXS = 0x2492,
	AARCH64_TLBI_VMALLWS2E1NXS = 0x24b2,
	AARCH64_TLBI_VMALLWS2E1OS = 0x242a,
	AARCH64_TLBI_VMALLWS2E1OSNXS = 0x24aa,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_TLBI> end
	AARCH64_TLBI_ENDING,
} aarch64_tlbi;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_AT> begin
	// clang-format off

	AARCH64_AT_S12E0R = 0x23c6,
	AARCH64_AT_S12E0W = 0x23c7,
	AARCH64_AT_S12E1R = 0x23c4,
	AARCH64_AT_S12E1W = 0x23c5,
	AARCH64_AT_S1E0R = 0x3c2,
	AARCH64_AT_S1E0W = 0x3c3,
	AARCH64_AT_S1E1A = 0x3ca,
	AARCH64_AT_S1E1R = 0x3c0,
	AARCH64_AT_S1E1RP = 0x3c8,
	AARCH64_AT_S1E1W = 0x3c1,
	AARCH64_AT_S1E1WP = 0x3c9,
	AARCH64_AT_S1E2A = 0x23ca,
	AARCH64_AT_S1E2R = 0x23c0,
	AARCH64_AT_S1E2W = 0x23c1,
	AARCH64_AT_S1E3A = 0x33ca,
	AARCH64_AT_S1E3R = 0x33c0,
	AARCH64_AT_S1E3W = 0x33c1,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_AT> end
	AARCH64_AT_ENDING,
} aarch64_at;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_BTI> begin
	// clang-format off

	AARCH64_BTI_C = 0x2,
	AARCH64_BTI_J = 0x4,
	AARCH64_BTI_JC = 0x6,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_BTI> end
	AARCH64_BTI_ENDING,
} aarch64_bti;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DB> begin
	// clang-format off

	AARCH64_DB_ISH = 0xb,
	AARCH64_DB_ISHLD = 0x9,
	AARCH64_DB_ISHST = 0xa,
	AARCH64_DB_LD = 0xd,
	AARCH64_DB_NSH = 0x7,
	AARCH64_DB_NSHLD = 0x5,
	AARCH64_DB_NSHST = 0x6,
	AARCH64_DB_OSH = 0x3,
	AARCH64_DB_OSHLD = 0x1,
	AARCH64_DB_OSHST = 0x2,
	AARCH64_DB_ST = 0xe,
	AARCH64_DB_SY = 0xf,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DB> end
	AARCH64_DB_ENDING,
} aarch64_db;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DBnXS> begin
	// clang-format off

	AARCH64_DBNXS_ISHNXS = 0xb,
	AARCH64_DBNXS_NSHNXS = 0x7,
	AARCH64_DBNXS_OSHNXS = 0x3,
	AARCH64_DBNXS_SYNXS = 0xf,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DBnXS> end
	AARCH64_DBNXS_ENDING,
} aarch64_dbnxs;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DC> begin
	// clang-format off

	AARCH64_DC_CGDSW = 0x3d6,
	AARCH64_DC_CGDVAC = 0x1bd5,
	AARCH64_DC_CGDVADP = 0x1bed,
	AARCH64_DC_CGDVAP = 0x1be5,
	AARCH64_DC_CGSW = 0x3d4,
	AARCH64_DC_CGVAC = 0x1bd3,
	AARCH64_DC_CGVADP = 0x1beb,
	AARCH64_DC_CGVAP = 0x1be3,
	AARCH64_DC_CIGDPAE = 0x23f7,
	AARCH64_DC_CIGDSW = 0x3f6,
	AARCH64_DC_CIGDVAC = 0x1bf5,
	AARCH64_DC_CIGSW = 0x3f4,
	AARCH64_DC_CIGVAC = 0x1bf3,
	AARCH64_DC_CIPAE = 0x23f0,
	AARCH64_DC_CISW = 0x3f2,
	AARCH64_DC_CIVAC = 0x1bf1,
	AARCH64_DC_CSW = 0x3d2,
	AARCH64_DC_CVAC = 0x1bd1,
	AARCH64_DC_CVADP = 0x1be9,
	AARCH64_DC_CVAP = 0x1be1,
	AARCH64_DC_CVAU = 0x1bd9,
	AARCH64_DC_GVA = 0x1ba3,
	AARCH64_DC_GZVA = 0x1ba4,
	AARCH64_DC_IGDSW = 0x3b6,
	AARCH64_DC_IGDVAC = 0x3b5,
	AARCH64_DC_IGSW = 0x3b4,
	AARCH64_DC_IGVAC = 0x3b3,
	AARCH64_DC_ISW = 0x3b2,
	AARCH64_DC_IVAC = 0x3b1,
	AARCH64_DC_ZVA = 0x1ba1,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_DC> end
	AARCH64_DC_ENDING,
} aarch64_dc;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ExactFPImm> begin
	// clang-format off

	AARCH64_EXACTFPIMM_HALF = 0x1,
	AARCH64_EXACTFPIMM_ONE = 0x2,
	AARCH64_EXACTFPIMM_TWO = 0x3,
	AARCH64_EXACTFPIMM_ZERO = 0x0,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ExactFPImm> end
	AARCH64_EXACTFPIMM_INVALID = 0x4,

	AARCH64_EXACTFPIMM_ENDING,
} aarch64_exactfpimm;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_IC> begin
	// clang-format off

	AARCH64_IC_IALLU = 0x3a8,
	AARCH64_IC_IALLUIS = 0x388,
	AARCH64_IC_IVAU = 0x1ba9,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_IC> end
	AARCH64_IC_ENDING,
} aarch64_ic;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ISB> begin
	// clang-format off

	AARCH64_ISB_SY = 0xf,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ISB> end
	AARCH64_ISB_ENDING,
} aarch64_isb;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PRFM> begin
	// clang-format off

	AARCH64_PRFM_PLDL1KEEP = 0x0,
	AARCH64_PRFM_PLDL1STRM = 0x1,
	AARCH64_PRFM_PLDL2KEEP = 0x2,
	AARCH64_PRFM_PLDL2STRM = 0x3,
	AARCH64_PRFM_PLDL3KEEP = 0x4,
	AARCH64_PRFM_PLDL3STRM = 0x5,
	AARCH64_PRFM_PLDSLCKEEP = 0x6,
	AARCH64_PRFM_PLDSLCSTRM = 0x7,
	AARCH64_PRFM_PLIL1KEEP = 0x8,
	AARCH64_PRFM_PLIL1STRM = 0x9,
	AARCH64_PRFM_PLIL2KEEP = 0xa,
	AARCH64_PRFM_PLIL2STRM = 0xb,
	AARCH64_PRFM_PLIL3KEEP = 0xc,
	AARCH64_PRFM_PLIL3STRM = 0xd,
	AARCH64_PRFM_PLISLCKEEP = 0xe,
	AARCH64_PRFM_PLISLCSTRM = 0xf,
	AARCH64_PRFM_PSTL1KEEP = 0x10,
	AARCH64_PRFM_PSTL1STRM = 0x11,
	AARCH64_PRFM_PSTL2KEEP = 0x12,
	AARCH64_PRFM_PSTL2STRM = 0x13,
	AARCH64_PRFM_PSTL3KEEP = 0x14,
	AARCH64_PRFM_PSTL3STRM = 0x15,
	AARCH64_PRFM_PSTSLCKEEP = 0x16,
	AARCH64_PRFM_PSTSLCSTRM = 0x17,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PRFM> end
	AARCH64_PRFM_ENDING,
} aarch64_prfm;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PSB> begin
	// clang-format off

	AARCH64_PSB_CSYNC = 0x11,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PSB> end
	AARCH64_PSB_ENDING,
} aarch64_psb;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PStateImm0_1> begin
	// clang-format off

	AARCH64_PSTATEIMM0_1_ALLINT = 0x8,
	AARCH64_PSTATEIMM0_1_PM = 0x48,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PStateImm0_1> end
	AARCH64_PSTATEIMM0_1_ENDING,
} aarch64_pstateimm0_1;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PStateImm0_15> begin
	// clang-format off

	AARCH64_PSTATEIMM0_15_DAIFCLR = 0x1f,
	AARCH64_PSTATEIMM0_15_DAIFSET = 0x1e,
	AARCH64_PSTATEIMM0_15_DIT = 0x1a,
	AARCH64_PSTATEIMM0_15_PAN = 0x4,
	AARCH64_PSTATEIMM0_15_SPSEL = 0x5,
	AARCH64_PSTATEIMM0_15_SSBS = 0x19,
	AARCH64_PSTATEIMM0_15_TCO = 0x1c,
	AARCH64_PSTATEIMM0_15_UAO = 0x3,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_PStateImm0_15> end
	AARCH64_PSTATEIMM0_15_ENDING,
} aarch64_pstateimm0_15;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_RPRFM> begin
	// clang-format off

	AARCH64_RPRFM_PLDKEEP = 0x0,
	AARCH64_RPRFM_PLDSTRM = 0x4,
	AARCH64_RPRFM_PSTKEEP = 0x1,
	AARCH64_RPRFM_PSTSTRM = 0x5,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_RPRFM> end
	AARCH64_RPRFM_ENDING,
} aarch64_rprfm;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVCR> begin
	// clang-format off

	AARCH64_SVCR_SVCRSM = 0x1,
	AARCH64_SVCR_SVCRSMZA = 0x3,
	AARCH64_SVCR_SVCRZA = 0x2,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVCR> end
	AARCH64_SVCR_ENDING,
} aarch64_svcr;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEPREDPAT> begin
	// clang-format off

	AARCH64_SVEPREDPAT_ALL = 0x1f,
	AARCH64_SVEPREDPAT_MUL3 = 0x1e,
	AARCH64_SVEPREDPAT_MUL4 = 0x1d,
	AARCH64_SVEPREDPAT_POW2 = 0x0,
	AARCH64_SVEPREDPAT_VL1 = 0x1,
	AARCH64_SVEPREDPAT_VL128 = 0xc,
	AARCH64_SVEPREDPAT_VL16 = 0x9,
	AARCH64_SVEPREDPAT_VL2 = 0x2,
	AARCH64_SVEPREDPAT_VL256 = 0xd,
	AARCH64_SVEPREDPAT_VL3 = 0x3,
	AARCH64_SVEPREDPAT_VL32 = 0xa,
	AARCH64_SVEPREDPAT_VL4 = 0x4,
	AARCH64_SVEPREDPAT_VL5 = 0x5,
	AARCH64_SVEPREDPAT_VL6 = 0x6,
	AARCH64_SVEPREDPAT_VL64 = 0xb,
	AARCH64_SVEPREDPAT_VL7 = 0x7,
	AARCH64_SVEPREDPAT_VL8 = 0x8,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEPREDPAT> end
	AARCH64_SVEPREDPAT_ENDING,
} aarch64_svepredpat;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEPRFM> begin
	// clang-format off

	AARCH64_SVEPRFM_PLDL1KEEP = 0x0,
	AARCH64_SVEPRFM_PLDL1STRM = 0x1,
	AARCH64_SVEPRFM_PLDL2KEEP = 0x2,
	AARCH64_SVEPRFM_PLDL2STRM = 0x3,
	AARCH64_SVEPRFM_PLDL3KEEP = 0x4,
	AARCH64_SVEPRFM_PLDL3STRM = 0x5,
	AARCH64_SVEPRFM_PSTL1KEEP = 0x8,
	AARCH64_SVEPRFM_PSTL1STRM = 0x9,
	AARCH64_SVEPRFM_PSTL2KEEP = 0xa,
	AARCH64_SVEPRFM_PSTL2STRM = 0xb,
	AARCH64_SVEPRFM_PSTL3KEEP = 0xc,
	AARCH64_SVEPRFM_PSTL3STRM = 0xd,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEPRFM> end
	AARCH64_SVEPRFM_ENDING,
} aarch64_sveprfm;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEVECLENSPECIFIER> begin
	// clang-format off

	AARCH64_SVEVECLENSPECIFIER_VLX2 = 0x0,
	AARCH64_SVEVECLENSPECIFIER_VLX4 = 0x1,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SVEVECLENSPECIFIER> end
	AARCH64_SVEVECLENSPECIFIER_ENDING,
} aarch64_sveveclenspecifier;

typedef enum {
	AARCH64_SYSREG_INVALID = 0,
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SysReg> begin
	// clang-format off

	AARCH64_SYSREG_ACCDATA_EL1 = 0xc685,
	AARCH64_SYSREG_ACTLR_EL1 = 0xc081,
	AARCH64_SYSREG_ACTLR_EL2 = 0xe081,
	AARCH64_SYSREG_ACTLR_EL3 = 0xf081,
	AARCH64_SYSREG_AFSR0_EL1 = 0xc288,
	AARCH64_SYSREG_AFSR0_EL12 = 0xea88,
	AARCH64_SYSREG_AFSR0_EL2 = 0xe288,
	AARCH64_SYSREG_AFSR0_EL3 = 0xf288,
	AARCH64_SYSREG_AFSR1_EL1 = 0xc289,
	AARCH64_SYSREG_AFSR1_EL12 = 0xea89,
	AARCH64_SYSREG_AFSR1_EL2 = 0xe289,
	AARCH64_SYSREG_AFSR1_EL3 = 0xf289,
	AARCH64_SYSREG_AIDR_EL1 = 0xc807,
	AARCH64_SYSREG_ALLINT = 0xc218,
	AARCH64_SYSREG_AMAIR2_EL1 = 0xc519,
	AARCH64_SYSREG_AMAIR2_EL12 = 0xed19,
	AARCH64_SYSREG_AMAIR2_EL2 = 0xe519,
	AARCH64_SYSREG_AMAIR2_EL3 = 0xf519,
	AARCH64_SYSREG_AMAIR_EL1 = 0xc518,
	AARCH64_SYSREG_AMAIR_EL12 = 0xed18,
	AARCH64_SYSREG_AMAIR_EL2 = 0xe518,
	AARCH64_SYSREG_AMAIR_EL3 = 0xf518,
	AARCH64_SYSREG_AMCFGR_EL0 = 0xde91,
	AARCH64_SYSREG_AMCG1IDR_EL0 = 0xde96,
	AARCH64_SYSREG_AMCGCR_EL0 = 0xde92,
	AARCH64_SYSREG_AMCNTENCLR0_EL0 = 0xde94,
	AARCH64_SYSREG_AMCNTENCLR1_EL0 = 0xde98,
	AARCH64_SYSREG_AMCNTENSET0_EL0 = 0xde95,
	AARCH64_SYSREG_AMCNTENSET1_EL0 = 0xde99,
	AARCH64_SYSREG_AMCR_EL0 = 0xde90,
	AARCH64_SYSREG_AMEVCNTR00_EL0 = 0xdea0,
	AARCH64_SYSREG_AMEVCNTR01_EL0 = 0xdea1,
	AARCH64_SYSREG_AMEVCNTR02_EL0 = 0xdea2,
	AARCH64_SYSREG_AMEVCNTR03_EL0 = 0xdea3,
	AARCH64_SYSREG_AMEVCNTR10_EL0 = 0xdee0,
	AARCH64_SYSREG_AMEVCNTR110_EL0 = 0xdeea,
	AARCH64_SYSREG_AMEVCNTR111_EL0 = 0xdeeb,
	AARCH64_SYSREG_AMEVCNTR112_EL0 = 0xdeec,
	AARCH64_SYSREG_AMEVCNTR113_EL0 = 0xdeed,
	AARCH64_SYSREG_AMEVCNTR114_EL0 = 0xdeee,
	AARCH64_SYSREG_AMEVCNTR115_EL0 = 0xdeef,
	AARCH64_SYSREG_AMEVCNTR11_EL0 = 0xdee1,
	AARCH64_SYSREG_AMEVCNTR12_EL0 = 0xdee2,
	AARCH64_SYSREG_AMEVCNTR13_EL0 = 0xdee3,
	AARCH64_SYSREG_AMEVCNTR14_EL0 = 0xdee4,
	AARCH64_SYSREG_AMEVCNTR15_EL0 = 0xdee5,
	AARCH64_SYSREG_AMEVCNTR16_EL0 = 0xdee6,
	AARCH64_SYSREG_AMEVCNTR17_EL0 = 0xdee7,
	AARCH64_SYSREG_AMEVCNTR18_EL0 = 0xdee8,
	AARCH64_SYSREG_AMEVCNTR19_EL0 = 0xdee9,
	AARCH64_SYSREG_AMEVCNTVOFF00_EL2 = 0xe6c0,
	AARCH64_SYSREG_AMEVCNTVOFF010_EL2 = 0xe6ca,
	AARCH64_SYSREG_AMEVCNTVOFF011_EL2 = 0xe6cb,
	AARCH64_SYSREG_AMEVCNTVOFF012_EL2 = 0xe6cc,
	AARCH64_SYSREG_AMEVCNTVOFF013_EL2 = 0xe6cd,
	AARCH64_SYSREG_AMEVCNTVOFF014_EL2 = 0xe6ce,
	AARCH64_SYSREG_AMEVCNTVOFF015_EL2 = 0xe6cf,
	AARCH64_SYSREG_AMEVCNTVOFF01_EL2 = 0xe6c1,
	AARCH64_SYSREG_AMEVCNTVOFF02_EL2 = 0xe6c2,
	AARCH64_SYSREG_AMEVCNTVOFF03_EL2 = 0xe6c3,
	AARCH64_SYSREG_AMEVCNTVOFF04_EL2 = 0xe6c4,
	AARCH64_SYSREG_AMEVCNTVOFF05_EL2 = 0xe6c5,
	AARCH64_SYSREG_AMEVCNTVOFF06_EL2 = 0xe6c6,
	AARCH64_SYSREG_AMEVCNTVOFF07_EL2 = 0xe6c7,
	AARCH64_SYSREG_AMEVCNTVOFF08_EL2 = 0xe6c8,
	AARCH64_SYSREG_AMEVCNTVOFF09_EL2 = 0xe6c9,
	AARCH64_SYSREG_AMEVCNTVOFF10_EL2 = 0xe6d0,
	AARCH64_SYSREG_AMEVCNTVOFF110_EL2 = 0xe6da,
	AARCH64_SYSREG_AMEVCNTVOFF111_EL2 = 0xe6db,
	AARCH64_SYSREG_AMEVCNTVOFF112_EL2 = 0xe6dc,
	AARCH64_SYSREG_AMEVCNTVOFF113_EL2 = 0xe6dd,
	AARCH64_SYSREG_AMEVCNTVOFF114_EL2 = 0xe6de,
	AARCH64_SYSREG_AMEVCNTVOFF115_EL2 = 0xe6df,
	AARCH64_SYSREG_AMEVCNTVOFF11_EL2 = 0xe6d1,
	AARCH64_SYSREG_AMEVCNTVOFF12_EL2 = 0xe6d2,
	AARCH64_SYSREG_AMEVCNTVOFF13_EL2 = 0xe6d3,
	AARCH64_SYSREG_AMEVCNTVOFF14_EL2 = 0xe6d4,
	AARCH64_SYSREG_AMEVCNTVOFF15_EL2 = 0xe6d5,
	AARCH64_SYSREG_AMEVCNTVOFF16_EL2 = 0xe6d6,
	AARCH64_SYSREG_AMEVCNTVOFF17_EL2 = 0xe6d7,
	AARCH64_SYSREG_AMEVCNTVOFF18_EL2 = 0xe6d8,
	AARCH64_SYSREG_AMEVCNTVOFF19_EL2 = 0xe6d9,
	AARCH64_SYSREG_AMEVTYPER00_EL0 = 0xdeb0,
	AARCH64_SYSREG_AMEVTYPER01_EL0 = 0xdeb1,
	AARCH64_SYSREG_AMEVTYPER02_EL0 = 0xdeb2,
	AARCH64_SYSREG_AMEVTYPER03_EL0 = 0xdeb3,
	AARCH64_SYSREG_AMEVTYPER10_EL0 = 0xdef0,
	AARCH64_SYSREG_AMEVTYPER110_EL0 = 0xdefa,
	AARCH64_SYSREG_AMEVTYPER111_EL0 = 0xdefb,
	AARCH64_SYSREG_AMEVTYPER112_EL0 = 0xdefc,
	AARCH64_SYSREG_AMEVTYPER113_EL0 = 0xdefd,
	AARCH64_SYSREG_AMEVTYPER114_EL0 = 0xdefe,
	AARCH64_SYSREG_AMEVTYPER115_EL0 = 0xdeff,
	AARCH64_SYSREG_AMEVTYPER11_EL0 = 0xdef1,
	AARCH64_SYSREG_AMEVTYPER12_EL0 = 0xdef2,
	AARCH64_SYSREG_AMEVTYPER13_EL0 = 0xdef3,
	AARCH64_SYSREG_AMEVTYPER14_EL0 = 0xdef4,
	AARCH64_SYSREG_AMEVTYPER15_EL0 = 0xdef5,
	AARCH64_SYSREG_AMEVTYPER16_EL0 = 0xdef6,
	AARCH64_SYSREG_AMEVTYPER17_EL0 = 0xdef7,
	AARCH64_SYSREG_AMEVTYPER18_EL0 = 0xdef8,
	AARCH64_SYSREG_AMEVTYPER19_EL0 = 0xdef9,
	AARCH64_SYSREG_AMUSERENR_EL0 = 0xde93,
	AARCH64_SYSREG_APDAKEYHI_EL1 = 0xc111,
	AARCH64_SYSREG_APDAKEYLO_EL1 = 0xc110,
	AARCH64_SYSREG_APDBKEYHI_EL1 = 0xc113,
	AARCH64_SYSREG_APDBKEYLO_EL1 = 0xc112,
	AARCH64_SYSREG_APGAKEYHI_EL1 = 0xc119,
	AARCH64_SYSREG_APGAKEYLO_EL1 = 0xc118,
	AARCH64_SYSREG_APIAKEYHI_EL1 = 0xc109,
	AARCH64_SYSREG_APIAKEYLO_EL1 = 0xc108,
	AARCH64_SYSREG_APIBKEYHI_EL1 = 0xc10b,
	AARCH64_SYSREG_APIBKEYLO_EL1 = 0xc10a,
	AARCH64_SYSREG_BRBCR_EL1 = 0x8c80,
	AARCH64_SYSREG_BRBCR_EL12 = 0xac80,
	AARCH64_SYSREG_BRBCR_EL2 = 0xa480,
	AARCH64_SYSREG_BRBFCR_EL1 = 0x8c81,
	AARCH64_SYSREG_BRBIDR0_EL1 = 0x8c90,
	AARCH64_SYSREG_BRBINF0_EL1 = 0x8c00,
	AARCH64_SYSREG_BRBINF10_EL1 = 0x8c50,
	AARCH64_SYSREG_BRBINF11_EL1 = 0x8c58,
	AARCH64_SYSREG_BRBINF12_EL1 = 0x8c60,
	AARCH64_SYSREG_BRBINF13_EL1 = 0x8c68,
	AARCH64_SYSREG_BRBINF14_EL1 = 0x8c70,
	AARCH64_SYSREG_BRBINF15_EL1 = 0x8c78,
	AARCH64_SYSREG_BRBINF16_EL1 = 0x8c04,
	AARCH64_SYSREG_BRBINF17_EL1 = 0x8c0c,
	AARCH64_SYSREG_BRBINF18_EL1 = 0x8c14,
	AARCH64_SYSREG_BRBINF19_EL1 = 0x8c1c,
	AARCH64_SYSREG_BRBINF1_EL1 = 0x8c08,
	AARCH64_SYSREG_BRBINF20_EL1 = 0x8c24,
	AARCH64_SYSREG_BRBINF21_EL1 = 0x8c2c,
	AARCH64_SYSREG_BRBINF22_EL1 = 0x8c34,
	AARCH64_SYSREG_BRBINF23_EL1 = 0x8c3c,
	AARCH64_SYSREG_BRBINF24_EL1 = 0x8c44,
	AARCH64_SYSREG_BRBINF25_EL1 = 0x8c4c,
	AARCH64_SYSREG_BRBINF26_EL1 = 0x8c54,
	AARCH64_SYSREG_BRBINF27_EL1 = 0x8c5c,
	AARCH64_SYSREG_BRBINF28_EL1 = 0x8c64,
	AARCH64_SYSREG_BRBINF29_EL1 = 0x8c6c,
	AARCH64_SYSREG_BRBINF2_EL1 = 0x8c10,
	AARCH64_SYSREG_BRBINF30_EL1 = 0x8c74,
	AARCH64_SYSREG_BRBINF31_EL1 = 0x8c7c,
	AARCH64_SYSREG_BRBINF3_EL1 = 0x8c18,
	AARCH64_SYSREG_BRBINF4_EL1 = 0x8c20,
	AARCH64_SYSREG_BRBINF5_EL1 = 0x8c28,
	AARCH64_SYSREG_BRBINF6_EL1 = 0x8c30,
	AARCH64_SYSREG_BRBINF7_EL1 = 0x8c38,
	AARCH64_SYSREG_BRBINF8_EL1 = 0x8c40,
	AARCH64_SYSREG_BRBINF9_EL1 = 0x8c48,
	AARCH64_SYSREG_BRBINFINJ_EL1 = 0x8c88,
	AARCH64_SYSREG_BRBSRC0_EL1 = 0x8c01,
	AARCH64_SYSREG_BRBSRC10_EL1 = 0x8c51,
	AARCH64_SYSREG_BRBSRC11_EL1 = 0x8c59,
	AARCH64_SYSREG_BRBSRC12_EL1 = 0x8c61,
	AARCH64_SYSREG_BRBSRC13_EL1 = 0x8c69,
	AARCH64_SYSREG_BRBSRC14_EL1 = 0x8c71,
	AARCH64_SYSREG_BRBSRC15_EL1 = 0x8c79,
	AARCH64_SYSREG_BRBSRC16_EL1 = 0x8c05,
	AARCH64_SYSREG_BRBSRC17_EL1 = 0x8c0d,
	AARCH64_SYSREG_BRBSRC18_EL1 = 0x8c15,
	AARCH64_SYSREG_BRBSRC19_EL1 = 0x8c1d,
	AARCH64_SYSREG_BRBSRC1_EL1 = 0x8c09,
	AARCH64_SYSREG_BRBSRC20_EL1 = 0x8c25,
	AARCH64_SYSREG_BRBSRC21_EL1 = 0x8c2d,
	AARCH64_SYSREG_BRBSRC22_EL1 = 0x8c35,
	AARCH64_SYSREG_BRBSRC23_EL1 = 0x8c3d,
	AARCH64_SYSREG_BRBSRC24_EL1 = 0x8c45,
	AARCH64_SYSREG_BRBSRC25_EL1 = 0x8c4d,
	AARCH64_SYSREG_BRBSRC26_EL1 = 0x8c55,
	AARCH64_SYSREG_BRBSRC27_EL1 = 0x8c5d,
	AARCH64_SYSREG_BRBSRC28_EL1 = 0x8c65,
	AARCH64_SYSREG_BRBSRC29_EL1 = 0x8c6d,
	AARCH64_SYSREG_BRBSRC2_EL1 = 0x8c11,
	AARCH64_SYSREG_BRBSRC30_EL1 = 0x8c75,
	AARCH64_SYSREG_BRBSRC31_EL1 = 0x8c7d,
	AARCH64_SYSREG_BRBSRC3_EL1 = 0x8c19,
	AARCH64_SYSREG_BRBSRC4_EL1 = 0x8c21,
	AARCH64_SYSREG_BRBSRC5_EL1 = 0x8c29,
	AARCH64_SYSREG_BRBSRC6_EL1 = 0x8c31,
	AARCH64_SYSREG_BRBSRC7_EL1 = 0x8c39,
	AARCH64_SYSREG_BRBSRC8_EL1 = 0x8c41,
	AARCH64_SYSREG_BRBSRC9_EL1 = 0x8c49,
	AARCH64_SYSREG_BRBSRCINJ_EL1 = 0x8c89,
	AARCH64_SYSREG_BRBTGT0_EL1 = 0x8c02,
	AARCH64_SYSREG_BRBTGT10_EL1 = 0x8c52,
	AARCH64_SYSREG_BRBTGT11_EL1 = 0x8c5a,
	AARCH64_SYSREG_BRBTGT12_EL1 = 0x8c62,
	AARCH64_SYSREG_BRBTGT13_EL1 = 0x8c6a,
	AARCH64_SYSREG_BRBTGT14_EL1 = 0x8c72,
	AARCH64_SYSREG_BRBTGT15_EL1 = 0x8c7a,
	AARCH64_SYSREG_BRBTGT16_EL1 = 0x8c06,
	AARCH64_SYSREG_BRBTGT17_EL1 = 0x8c0e,
	AARCH64_SYSREG_BRBTGT18_EL1 = 0x8c16,
	AARCH64_SYSREG_BRBTGT19_EL1 = 0x8c1e,
	AARCH64_SYSREG_BRBTGT1_EL1 = 0x8c0a,
	AARCH64_SYSREG_BRBTGT20_EL1 = 0x8c26,
	AARCH64_SYSREG_BRBTGT21_EL1 = 0x8c2e,
	AARCH64_SYSREG_BRBTGT22_EL1 = 0x8c36,
	AARCH64_SYSREG_BRBTGT23_EL1 = 0x8c3e,
	AARCH64_SYSREG_BRBTGT24_EL1 = 0x8c46,
	AARCH64_SYSREG_BRBTGT25_EL1 = 0x8c4e,
	AARCH64_SYSREG_BRBTGT26_EL1 = 0x8c56,
	AARCH64_SYSREG_BRBTGT27_EL1 = 0x8c5e,
	AARCH64_SYSREG_BRBTGT28_EL1 = 0x8c66,
	AARCH64_SYSREG_BRBTGT29_EL1 = 0x8c6e,
	AARCH64_SYSREG_BRBTGT2_EL1 = 0x8c12,
	AARCH64_SYSREG_BRBTGT30_EL1 = 0x8c76,
	AARCH64_SYSREG_BRBTGT31_EL1 = 0x8c7e,
	AARCH64_SYSREG_BRBTGT3_EL1 = 0x8c1a,
	AARCH64_SYSREG_BRBTGT4_EL1 = 0x8c22,
	AARCH64_SYSREG_BRBTGT5_EL1 = 0x8c2a,
	AARCH64_SYSREG_BRBTGT6_EL1 = 0x8c32,
	AARCH64_SYSREG_BRBTGT7_EL1 = 0x8c3a,
	AARCH64_SYSREG_BRBTGT8_EL1 = 0x8c42,
	AARCH64_SYSREG_BRBTGT9_EL1 = 0x8c4a,
	AARCH64_SYSREG_BRBTGTINJ_EL1 = 0x8c8a,
	AARCH64_SYSREG_BRBTS_EL1 = 0x8c82,
	AARCH64_SYSREG_CCSIDR2_EL1 = 0xc802,
	AARCH64_SYSREG_CCSIDR_EL1 = 0xc800,
	AARCH64_SYSREG_CLIDR_EL1 = 0xc801,
	AARCH64_SYSREG_CNTFRQ_EL0 = 0xdf00,
	AARCH64_SYSREG_CNTHCTL_EL2 = 0xe708,
	AARCH64_SYSREG_CNTHPS_CTL_EL2 = 0xe729,
	AARCH64_SYSREG_CNTHPS_CVAL_EL2 = 0xe72a,
	AARCH64_SYSREG_CNTHPS_TVAL_EL2 = 0xe728,
	AARCH64_SYSREG_CNTHP_CTL_EL2 = 0xe711,
	AARCH64_SYSREG_CNTHP_CVAL_EL2 = 0xe712,
	AARCH64_SYSREG_CNTHP_TVAL_EL2 = 0xe710,
	AARCH64_SYSREG_CNTHVS_CTL_EL2 = 0xe721,
	AARCH64_SYSREG_CNTHVS_CVAL_EL2 = 0xe722,
	AARCH64_SYSREG_CNTHVS_TVAL_EL2 = 0xe720,
	AARCH64_SYSREG_CNTHV_CTL_EL2 = 0xe719,
	AARCH64_SYSREG_CNTHV_CVAL_EL2 = 0xe71a,
	AARCH64_SYSREG_CNTHV_TVAL_EL2 = 0xe718,
	AARCH64_SYSREG_CNTISCALE_EL2 = 0xe705,
	AARCH64_SYSREG_CNTKCTL_EL1 = 0xc708,
	AARCH64_SYSREG_CNTKCTL_EL12 = 0xef08,
	AARCH64_SYSREG_CNTPCTSS_EL0 = 0xdf05,
	AARCH64_SYSREG_CNTPCT_EL0 = 0xdf01,
	AARCH64_SYSREG_CNTPOFF_EL2 = 0xe706,
	AARCH64_SYSREG_CNTPS_CTL_EL1 = 0xff11,
	AARCH64_SYSREG_CNTPS_CVAL_EL1 = 0xff12,
	AARCH64_SYSREG_CNTPS_TVAL_EL1 = 0xff10,
	AARCH64_SYSREG_CNTP_CTL_EL0 = 0xdf11,
	AARCH64_SYSREG_CNTP_CTL_EL02 = 0xef11,
	AARCH64_SYSREG_CNTP_CVAL_EL0 = 0xdf12,
	AARCH64_SYSREG_CNTP_CVAL_EL02 = 0xef12,
	AARCH64_SYSREG_CNTP_TVAL_EL0 = 0xdf10,
	AARCH64_SYSREG_CNTP_TVAL_EL02 = 0xef10,
	AARCH64_SYSREG_CNTSCALE_EL2 = 0xe704,
	AARCH64_SYSREG_CNTVCTSS_EL0 = 0xdf06,
	AARCH64_SYSREG_CNTVCT_EL0 = 0xdf02,
	AARCH64_SYSREG_CNTVFRQ_EL2 = 0xe707,
	AARCH64_SYSREG_CNTVOFF_EL2 = 0xe703,
	AARCH64_SYSREG_CNTV_CTL_EL0 = 0xdf19,
	AARCH64_SYSREG_CNTV_CTL_EL02 = 0xef19,
	AARCH64_SYSREG_CNTV_CVAL_EL0 = 0xdf1a,
	AARCH64_SYSREG_CNTV_CVAL_EL02 = 0xef1a,
	AARCH64_SYSREG_CNTV_TVAL_EL0 = 0xdf18,
	AARCH64_SYSREG_CNTV_TVAL_EL02 = 0xef18,
	AARCH64_SYSREG_CONTEXTIDR_EL1 = 0xc681,
	AARCH64_SYSREG_CONTEXTIDR_EL12 = 0xee81,
	AARCH64_SYSREG_CONTEXTIDR_EL2 = 0xe681,
	AARCH64_SYSREG_CPACR_EL1 = 0xc082,
	AARCH64_SYSREG_CPACR_EL12 = 0xe882,
	AARCH64_SYSREG_CPM_IOACC_CTL_EL3 = 0xff90,
	AARCH64_SYSREG_CPTR_EL2 = 0xe08a,
	AARCH64_SYSREG_CPTR_EL3 = 0xf08a,
	AARCH64_SYSREG_CSSELR_EL1 = 0xd000,
	AARCH64_SYSREG_CTR_EL0 = 0xd801,
	AARCH64_SYSREG_CURRENTEL = 0xc212,
	AARCH64_SYSREG_DACR32_EL2 = 0xe180,
	AARCH64_SYSREG_DAIF = 0xda11,
	AARCH64_SYSREG_DBGAUTHSTATUS_EL1 = 0x83f6,
	AARCH64_SYSREG_DBGBCR0_EL1 = 0x8005,
	AARCH64_SYSREG_DBGBCR10_EL1 = 0x8055,
	AARCH64_SYSREG_DBGBCR11_EL1 = 0x805d,
	AARCH64_SYSREG_DBGBCR12_EL1 = 0x8065,
	AARCH64_SYSREG_DBGBCR13_EL1 = 0x806d,
	AARCH64_SYSREG_DBGBCR14_EL1 = 0x8075,
	AARCH64_SYSREG_DBGBCR15_EL1 = 0x807d,
	AARCH64_SYSREG_DBGBCR1_EL1 = 0x800d,
	AARCH64_SYSREG_DBGBCR2_EL1 = 0x8015,
	AARCH64_SYSREG_DBGBCR3_EL1 = 0x801d,
	AARCH64_SYSREG_DBGBCR4_EL1 = 0x8025,
	AARCH64_SYSREG_DBGBCR5_EL1 = 0x802d,
	AARCH64_SYSREG_DBGBCR6_EL1 = 0x8035,
	AARCH64_SYSREG_DBGBCR7_EL1 = 0x803d,
	AARCH64_SYSREG_DBGBCR8_EL1 = 0x8045,
	AARCH64_SYSREG_DBGBCR9_EL1 = 0x804d,
	AARCH64_SYSREG_DBGBVR0_EL1 = 0x8004,
	AARCH64_SYSREG_DBGBVR10_EL1 = 0x8054,
	AARCH64_SYSREG_DBGBVR11_EL1 = 0x805c,
	AARCH64_SYSREG_DBGBVR12_EL1 = 0x8064,
	AARCH64_SYSREG_DBGBVR13_EL1 = 0x806c,
	AARCH64_SYSREG_DBGBVR14_EL1 = 0x8074,
	AARCH64_SYSREG_DBGBVR15_EL1 = 0x807c,
	AARCH64_SYSREG_DBGBVR1_EL1 = 0x800c,
	AARCH64_SYSREG_DBGBVR2_EL1 = 0x8014,
	AARCH64_SYSREG_DBGBVR3_EL1 = 0x801c,
	AARCH64_SYSREG_DBGBVR4_EL1 = 0x8024,
	AARCH64_SYSREG_DBGBVR5_EL1 = 0x802c,
	AARCH64_SYSREG_DBGBVR6_EL1 = 0x8034,
	AARCH64_SYSREG_DBGBVR7_EL1 = 0x803c,
	AARCH64_SYSREG_DBGBVR8_EL1 = 0x8044,
	AARCH64_SYSREG_DBGBVR9_EL1 = 0x804c,
	AARCH64_SYSREG_DBGCLAIMCLR_EL1 = 0x83ce,
	AARCH64_SYSREG_DBGCLAIMSET_EL1 = 0x83c6,
	AARCH64_SYSREG_DBGDTRRX_EL0 = 0x9828,
	AARCH64_SYSREG_DBGDTRTX_EL0 = 0x9828,
	AARCH64_SYSREG_DBGDTR_EL0 = 0x9820,
	AARCH64_SYSREG_DBGPRCR_EL1 = 0x80a4,
	AARCH64_SYSREG_DBGVCR32_EL2 = 0xa038,
	AARCH64_SYSREG_DBGWCR0_EL1 = 0x8007,
	AARCH64_SYSREG_DBGWCR10_EL1 = 0x8057,
	AARCH64_SYSREG_DBGWCR11_EL1 = 0x805f,
	AARCH64_SYSREG_DBGWCR12_EL1 = 0x8067,
	AARCH64_SYSREG_DBGWCR13_EL1 = 0x806f,
	AARCH64_SYSREG_DBGWCR14_EL1 = 0x8077,
	AARCH64_SYSREG_DBGWCR15_EL1 = 0x807f,
	AARCH64_SYSREG_DBGWCR1_EL1 = 0x800f,
	AARCH64_SYSREG_DBGWCR2_EL1 = 0x8017,
	AARCH64_SYSREG_DBGWCR3_EL1 = 0x801f,
	AARCH64_SYSREG_DBGWCR4_EL1 = 0x8027,
	AARCH64_SYSREG_DBGWCR5_EL1 = 0x802f,
	AARCH64_SYSREG_DBGWCR6_EL1 = 0x8037,
	AARCH64_SYSREG_DBGWCR7_EL1 = 0x803f,
	AARCH64_SYSREG_DBGWCR8_EL1 = 0x8047,
	AARCH64_SYSREG_DBGWCR9_EL1 = 0x804f,
	AARCH64_SYSREG_DBGWVR0_EL1 = 0x8006,
	AARCH64_SYSREG_DBGWVR10_EL1 = 0x8056,
	AARCH64_SYSREG_DBGWVR11_EL1 = 0x805e,
	AARCH64_SYSREG_DBGWVR12_EL1 = 0x8066,
	AARCH64_SYSREG_DBGWVR13_EL1 = 0x806e,
	AARCH64_SYSREG_DBGWVR14_EL1 = 0x8076,
	AARCH64_SYSREG_DBGWVR15_EL1 = 0x807e,
	AARCH64_SYSREG_DBGWVR1_EL1 = 0x800e,
	AARCH64_SYSREG_DBGWVR2_EL1 = 0x8016,
	AARCH64_SYSREG_DBGWVR3_EL1 = 0x801e,
	AARCH64_SYSREG_DBGWVR4_EL1 = 0x8026,
	AARCH64_SYSREG_DBGWVR5_EL1 = 0x802e,
	AARCH64_SYSREG_DBGWVR6_EL1 = 0x8036,
	AARCH64_SYSREG_DBGWVR7_EL1 = 0x803e,
	AARCH64_SYSREG_DBGWVR8_EL1 = 0x8046,
	AARCH64_SYSREG_DBGWVR9_EL1 = 0x804e,
	AARCH64_SYSREG_DCZID_EL0 = 0xd807,
	AARCH64_SYSREG_DISR_EL1 = 0xc609,
	AARCH64_SYSREG_DIT = 0xda15,
	AARCH64_SYSREG_DLR_EL0 = 0xda29,
	AARCH64_SYSREG_DSPSR_EL0 = 0xda28,
	AARCH64_SYSREG_ELR_EL1 = 0xc201,
	AARCH64_SYSREG_ELR_EL12 = 0xea01,
	AARCH64_SYSREG_ELR_EL2 = 0xe201,
	AARCH64_SYSREG_ELR_EL3 = 0xf201,
	AARCH64_SYSREG_ERRIDR_EL1 = 0xc298,
	AARCH64_SYSREG_ERRSELR_EL1 = 0xc299,
	AARCH64_SYSREG_ERXADDR_EL1 = 0xc2a3,
	AARCH64_SYSREG_ERXCTLR_EL1 = 0xc2a1,
	AARCH64_SYSREG_ERXFR_EL1 = 0xc2a0,
	AARCH64_SYSREG_ERXGSR_EL1 = 0xc29a,
	AARCH64_SYSREG_ERXMISC0_EL1 = 0xc2a8,
	AARCH64_SYSREG_ERXMISC1_EL1 = 0xc2a9,
	AARCH64_SYSREG_ERXMISC2_EL1 = 0xc2aa,
	AARCH64_SYSREG_ERXMISC3_EL1 = 0xc2ab,
	AARCH64_SYSREG_ERXPFGCDN_EL1 = 0xc2a6,
	AARCH64_SYSREG_ERXPFGCTL_EL1 = 0xc2a5,
	AARCH64_SYSREG_ERXPFGF_EL1 = 0xc2a4,
	AARCH64_SYSREG_ERXSTATUS_EL1 = 0xc2a2,
	AARCH64_SYSREG_ESR_EL1 = 0xc290,
	AARCH64_SYSREG_ESR_EL12 = 0xea90,
	AARCH64_SYSREG_ESR_EL2 = 0xe290,
	AARCH64_SYSREG_ESR_EL3 = 0xf290,
	AARCH64_SYSREG_FAR_EL1 = 0xc300,
	AARCH64_SYSREG_FAR_EL12 = 0xeb00,
	AARCH64_SYSREG_FAR_EL2 = 0xe300,
	AARCH64_SYSREG_FAR_EL3 = 0xf300,
	AARCH64_SYSREG_FGWTE3_EL3 = 0xf08d,
	AARCH64_SYSREG_FPCR = 0xda20,
	AARCH64_SYSREG_FPEXC32_EL2 = 0xe298,
	AARCH64_SYSREG_FPMR = 0xda22,
	AARCH64_SYSREG_FPSR = 0xda21,
	AARCH64_SYSREG_GCR_EL1 = 0xc086,
	AARCH64_SYSREG_GCSCRE0_EL1 = 0xc12a,
	AARCH64_SYSREG_GCSCR_EL1 = 0xc128,
	AARCH64_SYSREG_GCSCR_EL12 = 0xe928,
	AARCH64_SYSREG_GCSCR_EL2 = 0xe128,
	AARCH64_SYSREG_GCSCR_EL3 = 0xf128,
	AARCH64_SYSREG_GCSPR_EL0 = 0xd929,
	AARCH64_SYSREG_GCSPR_EL1 = 0xc129,
	AARCH64_SYSREG_GCSPR_EL12 = 0xe929,
	AARCH64_SYSREG_GCSPR_EL2 = 0xe129,
	AARCH64_SYSREG_GCSPR_EL3 = 0xf129,
	AARCH64_SYSREG_GMID_EL1 = 0xc804,
	AARCH64_SYSREG_GPCCR_EL3 = 0xf10e,
	AARCH64_SYSREG_GPTBR_EL3 = 0xf10c,
	AARCH64_SYSREG_HACDBSBR_EL2 = 0xe11c,
	AARCH64_SYSREG_HACDBSCONS_EL2 = 0xe11d,
	AARCH64_SYSREG_HACR_EL2 = 0xe08f,
	AARCH64_SYSREG_HAFGRTR_EL2 = 0xe18e,
	AARCH64_SYSREG_HCRX_EL2 = 0xe092,
	AARCH64_SYSREG_HCR_EL2 = 0xe088,
	AARCH64_SYSREG_HDBSSBR_EL2 = 0xe11a,
	AARCH64_SYSREG_HDBSSPROD_EL2 = 0xe11b,
	AARCH64_SYSREG_HDFGRTR2_EL2 = 0xe188,
	AARCH64_SYSREG_HDFGRTR_EL2 = 0xe18c,
	AARCH64_SYSREG_HDFGWTR2_EL2 = 0xe189,
	AARCH64_SYSREG_HDFGWTR_EL2 = 0xe18d,
	AARCH64_SYSREG_HFGITR2_EL2 = 0xe18f,
	AARCH64_SYSREG_HFGITR_EL2 = 0xe08e,
	AARCH64_SYSREG_HFGRTR2_EL2 = 0xe18a,
	AARCH64_SYSREG_HFGRTR_EL2 = 0xe08c,
	AARCH64_SYSREG_HFGWTR2_EL2 = 0xe18b,
	AARCH64_SYSREG_HFGWTR_EL2 = 0xe08d,
	AARCH64_SYSREG_HPFAR_EL2 = 0xe304,
	AARCH64_SYSREG_HSTR_EL2 = 0xe08b,
	AARCH64_SYSREG_ICC_AP0R0_EL1 = 0xc644,
	AARCH64_SYSREG_ICC_AP0R1_EL1 = 0xc645,
	AARCH64_SYSREG_ICC_AP0R2_EL1 = 0xc646,
	AARCH64_SYSREG_ICC_AP0R3_EL1 = 0xc647,
	AARCH64_SYSREG_ICC_AP1R0_EL1 = 0xc648,
	AARCH64_SYSREG_ICC_AP1R1_EL1 = 0xc649,
	AARCH64_SYSREG_ICC_AP1R2_EL1 = 0xc64a,
	AARCH64_SYSREG_ICC_AP1R3_EL1 = 0xc64b,
	AARCH64_SYSREG_ICC_ASGI1R_EL1 = 0xc65e,
	AARCH64_SYSREG_ICC_BPR0_EL1 = 0xc643,
	AARCH64_SYSREG_ICC_BPR1_EL1 = 0xc663,
	AARCH64_SYSREG_ICC_CTLR_EL1 = 0xc664,
	AARCH64_SYSREG_ICC_CTLR_EL3 = 0xf664,
	AARCH64_SYSREG_ICC_DIR_EL1 = 0xc659,
	AARCH64_SYSREG_ICC_EOIR0_EL1 = 0xc641,
	AARCH64_SYSREG_ICC_EOIR1_EL1 = 0xc661,
	AARCH64_SYSREG_ICC_HPPIR0_EL1 = 0xc642,
	AARCH64_SYSREG_ICC_HPPIR1_EL1 = 0xc662,
	AARCH64_SYSREG_ICC_IAR0_EL1 = 0xc640,
	AARCH64_SYSREG_ICC_IAR1_EL1 = 0xc660,
	AARCH64_SYSREG_ICC_IGRPEN0_EL1 = 0xc666,
	AARCH64_SYSREG_ICC_IGRPEN1_EL1 = 0xc667,
	AARCH64_SYSREG_ICC_IGRPEN1_EL3 = 0xf667,
	AARCH64_SYSREG_ICC_NMIAR1_EL1 = 0xc64d,
	AARCH64_SYSREG_ICC_PMR_EL1 = 0xc230,
	AARCH64_SYSREG_ICC_RPR_EL1 = 0xc65b,
	AARCH64_SYSREG_ICC_SGI0R_EL1 = 0xc65f,
	AARCH64_SYSREG_ICC_SGI1R_EL1 = 0xc65d,
	AARCH64_SYSREG_ICC_SRE_EL1 = 0xc665,
	AARCH64_SYSREG_ICC_SRE_EL2 = 0xe64d,
	AARCH64_SYSREG_ICC_SRE_EL3 = 0xf665,
	AARCH64_SYSREG_ICH_AP0R0_EL2 = 0xe640,
	AARCH64_SYSREG_ICH_AP0R1_EL2 = 0xe641,
	AARCH64_SYSREG_ICH_AP0R2_EL2 = 0xe642,
	AARCH64_SYSREG_ICH_AP0R3_EL2 = 0xe643,
	AARCH64_SYSREG_ICH_AP1R0_EL2 = 0xe648,
	AARCH64_SYSREG_ICH_AP1R1_EL2 = 0xe649,
	AARCH64_SYSREG_ICH_AP1R2_EL2 = 0xe64a,
	AARCH64_SYSREG_ICH_AP1R3_EL2 = 0xe64b,
	AARCH64_SYSREG_ICH_EISR_EL2 = 0xe65b,
	AARCH64_SYSREG_ICH_ELRSR_EL2 = 0xe65d,
	AARCH64_SYSREG_ICH_HCR_EL2 = 0xe658,
	AARCH64_SYSREG_ICH_LR0_EL2 = 0xe660,
	AARCH64_SYSREG_ICH_LR10_EL2 = 0xe66a,
	AARCH64_SYSREG_ICH_LR11_EL2 = 0xe66b,
	AARCH64_SYSREG_ICH_LR12_EL2 = 0xe66c,
	AARCH64_SYSREG_ICH_LR13_EL2 = 0xe66d,
	AARCH64_SYSREG_ICH_LR14_EL2 = 0xe66e,
	AARCH64_SYSREG_ICH_LR15_EL2 = 0xe66f,
	AARCH64_SYSREG_ICH_LR1_EL2 = 0xe661,
	AARCH64_SYSREG_ICH_LR2_EL2 = 0xe662,
	AARCH64_SYSREG_ICH_LR3_EL2 = 0xe663,
	AARCH64_SYSREG_ICH_LR4_EL2 = 0xe664,
	AARCH64_SYSREG_ICH_LR5_EL2 = 0xe665,
	AARCH64_SYSREG_ICH_LR6_EL2 = 0xe666,
	AARCH64_SYSREG_ICH_LR7_EL2 = 0xe667,
	AARCH64_SYSREG_ICH_LR8_EL2 = 0xe668,
	AARCH64_SYSREG_ICH_LR9_EL2 = 0xe669,
	AARCH64_SYSREG_ICH_MISR_EL2 = 0xe65a,
	AARCH64_SYSREG_ICH_VMCR_EL2 = 0xe65f,
	AARCH64_SYSREG_ICH_VTR_EL2 = 0xe659,
	AARCH64_SYSREG_ID_AA64AFR0_EL1 = 0xc02c,
	AARCH64_SYSREG_ID_AA64AFR1_EL1 = 0xc02d,
	AARCH64_SYSREG_ID_AA64DFR0_EL1 = 0xc028,
	AARCH64_SYSREG_ID_AA64DFR1_EL1 = 0xc029,
	AARCH64_SYSREG_ID_AA64DFR2_EL1 = 0xc02a,
	AARCH64_SYSREG_ID_AA64FPFR0_EL1 = 0xc027,
	AARCH64_SYSREG_ID_AA64ISAR0_EL1 = 0xc030,
	AARCH64_SYSREG_ID_AA64ISAR1_EL1 = 0xc031,
	AARCH64_SYSREG_ID_AA64ISAR2_EL1 = 0xc032,
	AARCH64_SYSREG_ID_AA64ISAR3_EL1 = 0xc033,
	AARCH64_SYSREG_ID_AA64MMFR0_EL1 = 0xc038,
	AARCH64_SYSREG_ID_AA64MMFR1_EL1 = 0xc039,
	AARCH64_SYSREG_ID_AA64MMFR2_EL1 = 0xc03a,
	AARCH64_SYSREG_ID_AA64MMFR3_EL1 = 0xc03b,
	AARCH64_SYSREG_ID_AA64MMFR4_EL1 = 0xc03c,
	AARCH64_SYSREG_ID_AA64PFR0_EL1 = 0xc020,
	AARCH64_SYSREG_ID_AA64PFR1_EL1 = 0xc021,
	AARCH64_SYSREG_ID_AA64PFR2_EL1 = 0xc022,
	AARCH64_SYSREG_ID_AA64SMFR0_EL1 = 0xc025,
	AARCH64_SYSREG_ID_AA64ZFR0_EL1 = 0xc024,
	AARCH64_SYSREG_ID_AFR0_EL1 = 0xc00b,
	AARCH64_SYSREG_ID_DFR0_EL1 = 0xc00a,
	AARCH64_SYSREG_ID_DFR1_EL1 = 0xc01d,
	AARCH64_SYSREG_ID_ISAR0_EL1 = 0xc010,
	AARCH64_SYSREG_ID_ISAR1_EL1 = 0xc011,
	AARCH64_SYSREG_ID_ISAR2_EL1 = 0xc012,
	AARCH64_SYSREG_ID_ISAR3_EL1 = 0xc013,
	AARCH64_SYSREG_ID_ISAR4_EL1 = 0xc014,
	AARCH64_SYSREG_ID_ISAR5_EL1 = 0xc015,
	AARCH64_SYSREG_ID_ISAR6_EL1 = 0xc017,
	AARCH64_SYSREG_ID_MMFR0_EL1 = 0xc00c,
	AARCH64_SYSREG_ID_MMFR1_EL1 = 0xc00d,
	AARCH64_SYSREG_ID_MMFR2_EL1 = 0xc00e,
	AARCH64_SYSREG_ID_MMFR3_EL1 = 0xc00f,
	AARCH64_SYSREG_ID_MMFR4_EL1 = 0xc016,
	AARCH64_SYSREG_ID_MMFR5_EL1 = 0xc01e,
	AARCH64_SYSREG_ID_PFR0_EL1 = 0xc008,
	AARCH64_SYSREG_ID_PFR1_EL1 = 0xc009,
	AARCH64_SYSREG_ID_PFR2_EL1 = 0xc01c,
	AARCH64_SYSREG_IFSR32_EL2 = 0xe281,
	AARCH64_SYSREG_ISR_EL1 = 0xc608,
	AARCH64_SYSREG_LORC_EL1 = 0xc523,
	AARCH64_SYSREG_LOREA_EL1 = 0xc521,
	AARCH64_SYSREG_LORID_EL1 = 0xc527,
	AARCH64_SYSREG_LORN_EL1 = 0xc522,
	AARCH64_SYSREG_LORSA_EL1 = 0xc520,
	AARCH64_SYSREG_MAIR2_EL1 = 0xc511,
	AARCH64_SYSREG_MAIR2_EL12 = 0xed11,
	AARCH64_SYSREG_MAIR2_EL2 = 0xe509,
	AARCH64_SYSREG_MAIR2_EL3 = 0xf509,
	AARCH64_SYSREG_MAIR_EL1 = 0xc510,
	AARCH64_SYSREG_MAIR_EL12 = 0xed10,
	AARCH64_SYSREG_MAIR_EL2 = 0xe510,
	AARCH64_SYSREG_MAIR_EL3 = 0xf510,
	AARCH64_SYSREG_MDCCINT_EL1 = 0x8010,
	AARCH64_SYSREG_MDCCSR_EL0 = 0x9808,
	AARCH64_SYSREG_MDCR_EL2 = 0xe089,
	AARCH64_SYSREG_MDCR_EL3 = 0xf099,
	AARCH64_SYSREG_MDRAR_EL1 = 0x8080,
	AARCH64_SYSREG_MDSCR_EL1 = 0x8012,
	AARCH64_SYSREG_MDSELR_EL1 = 0x8022,
	AARCH64_SYSREG_MDSTEPOP_EL1 = 0x802a,
	AARCH64_SYSREG_MECIDR_EL2 = 0xe547,
	AARCH64_SYSREG_MECID_A0_EL2 = 0xe541,
	AARCH64_SYSREG_MECID_A1_EL2 = 0xe543,
	AARCH64_SYSREG_MECID_P0_EL2 = 0xe540,
	AARCH64_SYSREG_MECID_P1_EL2 = 0xe542,
	AARCH64_SYSREG_MECID_RL_A_EL3 = 0xf551,
	AARCH64_SYSREG_MFAR_EL3 = 0xf305,
	AARCH64_SYSREG_MIDR_EL1 = 0xc000,
	AARCH64_SYSREG_MPAM0_EL1 = 0xc529,
	AARCH64_SYSREG_MPAM1_EL1 = 0xc528,
	AARCH64_SYSREG_MPAM1_EL12 = 0xed28,
	AARCH64_SYSREG_MPAM2_EL2 = 0xe528,
	AARCH64_SYSREG_MPAM3_EL3 = 0xf528,
	AARCH64_SYSREG_MPAMHCR_EL2 = 0xe520,
	AARCH64_SYSREG_MPAMIDR_EL1 = 0xc524,
	AARCH64_SYSREG_MPAMSM_EL1 = 0xc52b,
	AARCH64_SYSREG_MPAMVPM0_EL2 = 0xe530,
	AARCH64_SYSREG_MPAMVPM1_EL2 = 0xe531,
	AARCH64_SYSREG_MPAMVPM2_EL2 = 0xe532,
	AARCH64_SYSREG_MPAMVPM3_EL2 = 0xe533,
	AARCH64_SYSREG_MPAMVPM4_EL2 = 0xe534,
	AARCH64_SYSREG_MPAMVPM5_EL2 = 0xe535,
	AARCH64_SYSREG_MPAMVPM6_EL2 = 0xe536,
	AARCH64_SYSREG_MPAMVPM7_EL2 = 0xe537,
	AARCH64_SYSREG_MPAMVPMV_EL2 = 0xe521,
	AARCH64_SYSREG_MPIDR_EL1 = 0xc005,
	AARCH64_SYSREG_MPUIR_EL1 = 0xc004,
	AARCH64_SYSREG_MPUIR_EL2 = 0xe004,
	AARCH64_SYSREG_MVFR0_EL1 = 0xc018,
	AARCH64_SYSREG_MVFR1_EL1 = 0xc019,
	AARCH64_SYSREG_MVFR2_EL1 = 0xc01a,
	AARCH64_SYSREG_NZCV = 0xda10,
	AARCH64_SYSREG_OSDLR_EL1 = 0x809c,
	AARCH64_SYSREG_OSDTRRX_EL1 = 0x8002,
	AARCH64_SYSREG_OSDTRTX_EL1 = 0x801a,
	AARCH64_SYSREG_OSECCR_EL1 = 0x8032,
	AARCH64_SYSREG_OSLAR_EL1 = 0x8084,
	AARCH64_SYSREG_OSLSR_EL1 = 0x808c,
	AARCH64_SYSREG_PAN = 0xc213,
	AARCH64_SYSREG_PAR_EL1 = 0xc3a0,
	AARCH64_SYSREG_PFAR_EL1 = 0xc305,
	AARCH64_SYSREG_PFAR_EL12 = 0xeb05,
	AARCH64_SYSREG_PFAR_EL2 = 0xe305,
	AARCH64_SYSREG_PIRE0_EL1 = 0xc512,
	AARCH64_SYSREG_PIRE0_EL12 = 0xed12,
	AARCH64_SYSREG_PIRE0_EL2 = 0xe512,
	AARCH64_SYSREG_PIR_EL1 = 0xc513,
	AARCH64_SYSREG_PIR_EL12 = 0xed13,
	AARCH64_SYSREG_PIR_EL2 = 0xe513,
	AARCH64_SYSREG_PIR_EL3 = 0xf513,
	AARCH64_SYSREG_PM = 0xc219,
	AARCH64_SYSREG_PMBIDR_EL1 = 0xc4d7,
	AARCH64_SYSREG_PMBLIMITR_EL1 = 0xc4d0,
	AARCH64_SYSREG_PMBPTR_EL1 = 0xc4d1,
	AARCH64_SYSREG_PMBSR_EL1 = 0xc4d3,
	AARCH64_SYSREG_PMCCFILTR_EL0 = 0xdf7f,
	AARCH64_SYSREG_PMCCNTR_EL0 = 0xdce8,
	AARCH64_SYSREG_PMCCNTSVR_EL1 = 0x875f,
	AARCH64_SYSREG_PMCEID0_EL0 = 0xdce6,
	AARCH64_SYSREG_PMCEID1_EL0 = 0xdce7,
	AARCH64_SYSREG_PMCNTENCLR_EL0 = 0xdce2,
	AARCH64_SYSREG_PMCNTENSET_EL0 = 0xdce1,
	AARCH64_SYSREG_PMCR_EL0 = 0xdce0,
	AARCH64_SYSREG_PMECR_EL1 = 0xc4f5,
	AARCH64_SYSREG_PMEVCNTR0_EL0 = 0xdf40,
	AARCH64_SYSREG_PMEVCNTR10_EL0 = 0xdf4a,
	AARCH64_SYSREG_PMEVCNTR11_EL0 = 0xdf4b,
	AARCH64_SYSREG_PMEVCNTR12_EL0 = 0xdf4c,
	AARCH64_SYSREG_PMEVCNTR13_EL0 = 0xdf4d,
	AARCH64_SYSREG_PMEVCNTR14_EL0 = 0xdf4e,
	AARCH64_SYSREG_PMEVCNTR15_EL0 = 0xdf4f,
	AARCH64_SYSREG_PMEVCNTR16_EL0 = 0xdf50,
	AARCH64_SYSREG_PMEVCNTR17_EL0 = 0xdf51,
	AARCH64_SYSREG_PMEVCNTR18_EL0 = 0xdf52,
	AARCH64_SYSREG_PMEVCNTR19_EL0 = 0xdf53,
	AARCH64_SYSREG_PMEVCNTR1_EL0 = 0xdf41,
	AARCH64_SYSREG_PMEVCNTR20_EL0 = 0xdf54,
	AARCH64_SYSREG_PMEVCNTR21_EL0 = 0xdf55,
	AARCH64_SYSREG_PMEVCNTR22_EL0 = 0xdf56,
	AARCH64_SYSREG_PMEVCNTR23_EL0 = 0xdf57,
	AARCH64_SYSREG_PMEVCNTR24_EL0 = 0xdf58,
	AARCH64_SYSREG_PMEVCNTR25_EL0 = 0xdf59,
	AARCH64_SYSREG_PMEVCNTR26_EL0 = 0xdf5a,
	AARCH64_SYSREG_PMEVCNTR27_EL0 = 0xdf5b,
	AARCH64_SYSREG_PMEVCNTR28_EL0 = 0xdf5c,
	AARCH64_SYSREG_PMEVCNTR29_EL0 = 0xdf5d,
	AARCH64_SYSREG_PMEVCNTR2_EL0 = 0xdf42,
	AARCH64_SYSREG_PMEVCNTR30_EL0 = 0xdf5e,
	AARCH64_SYSREG_PMEVCNTR3_EL0 = 0xdf43,
	AARCH64_SYSREG_PMEVCNTR4_EL0 = 0xdf44,
	AARCH64_SYSREG_PMEVCNTR5_EL0 = 0xdf45,
	AARCH64_SYSREG_PMEVCNTR6_EL0 = 0xdf46,
	AARCH64_SYSREG_PMEVCNTR7_EL0 = 0xdf47,
	AARCH64_SYSREG_PMEVCNTR8_EL0 = 0xdf48,
	AARCH64_SYSREG_PMEVCNTR9_EL0 = 0xdf49,
	AARCH64_SYSREG_PMEVCNTSVR0_EL1 = 0x8740,
	AARCH64_SYSREG_PMEVCNTSVR10_EL1 = 0x874a,
	AARCH64_SYSREG_PMEVCNTSVR11_EL1 = 0x874b,
	AARCH64_SYSREG_PMEVCNTSVR12_EL1 = 0x874c,
	AARCH64_SYSREG_PMEVCNTSVR13_EL1 = 0x874d,
	AARCH64_SYSREG_PMEVCNTSVR14_EL1 = 0x874e,
	AARCH64_SYSREG_PMEVCNTSVR15_EL1 = 0x874f,
	AARCH64_SYSREG_PMEVCNTSVR16_EL1 = 0x8750,
	AARCH64_SYSREG_PMEVCNTSVR17_EL1 = 0x8751,
	AARCH64_SYSREG_PMEVCNTSVR18_EL1 = 0x8752,
	AARCH64_SYSREG_PMEVCNTSVR19_EL1 = 0x8753,
	AARCH64_SYSREG_PMEVCNTSVR1_EL1 = 0x8741,
	AARCH64_SYSREG_PMEVCNTSVR20_EL1 = 0x8754,
	AARCH64_SYSREG_PMEVCNTSVR21_EL1 = 0x8755,
	AARCH64_SYSREG_PMEVCNTSVR22_EL1 = 0x8756,
	AARCH64_SYSREG_PMEVCNTSVR23_EL1 = 0x8757,
	AARCH64_SYSREG_PMEVCNTSVR24_EL1 = 0x8758,
	AARCH64_SYSREG_PMEVCNTSVR25_EL1 = 0x8759,
	AARCH64_SYSREG_PMEVCNTSVR26_EL1 = 0x875a,
	AARCH64_SYSREG_PMEVCNTSVR27_EL1 = 0x875b,
	AARCH64_SYSREG_PMEVCNTSVR28_EL1 = 0x875c,
	AARCH64_SYSREG_PMEVCNTSVR29_EL1 = 0x875d,
	AARCH64_SYSREG_PMEVCNTSVR2_EL1 = 0x8742,
	AARCH64_SYSREG_PMEVCNTSVR30_EL1 = 0x875e,
	AARCH64_SYSREG_PMEVCNTSVR3_EL1 = 0x8743,
	AARCH64_SYSREG_PMEVCNTSVR4_EL1 = 0x8744,
	AARCH64_SYSREG_PMEVCNTSVR5_EL1 = 0x8745,
	AARCH64_SYSREG_PMEVCNTSVR6_EL1 = 0x8746,
	AARCH64_SYSREG_PMEVCNTSVR7_EL1 = 0x8747,
	AARCH64_SYSREG_PMEVCNTSVR8_EL1 = 0x8748,
	AARCH64_SYSREG_PMEVCNTSVR9_EL1 = 0x8749,
	AARCH64_SYSREG_PMEVTYPER0_EL0 = 0xdf60,
	AARCH64_SYSREG_PMEVTYPER10_EL0 = 0xdf6a,
	AARCH64_SYSREG_PMEVTYPER11_EL0 = 0xdf6b,
	AARCH64_SYSREG_PMEVTYPER12_EL0 = 0xdf6c,
	AARCH64_SYSREG_PMEVTYPER13_EL0 = 0xdf6d,
	AARCH64_SYSREG_PMEVTYPER14_EL0 = 0xdf6e,
	AARCH64_SYSREG_PMEVTYPER15_EL0 = 0xdf6f,
	AARCH64_SYSREG_PMEVTYPER16_EL0 = 0xdf70,
	AARCH64_SYSREG_PMEVTYPER17_EL0 = 0xdf71,
	AARCH64_SYSREG_PMEVTYPER18_EL0 = 0xdf72,
	AARCH64_SYSREG_PMEVTYPER19_EL0 = 0xdf73,
	AARCH64_SYSREG_PMEVTYPER1_EL0 = 0xdf61,
	AARCH64_SYSREG_PMEVTYPER20_EL0 = 0xdf74,
	AARCH64_SYSREG_PMEVTYPER21_EL0 = 0xdf75,
	AARCH64_SYSREG_PMEVTYPER22_EL0 = 0xdf76,
	AARCH64_SYSREG_PMEVTYPER23_EL0 = 0xdf77,
	AARCH64_SYSREG_PMEVTYPER24_EL0 = 0xdf78,
	AARCH64_SYSREG_PMEVTYPER25_EL0 = 0xdf79,
	AARCH64_SYSREG_PMEVTYPER26_EL0 = 0xdf7a,
	AARCH64_SYSREG_PMEVTYPER27_EL0 = 0xdf7b,
	AARCH64_SYSREG_PMEVTYPER28_EL0 = 0xdf7c,
	AARCH64_SYSREG_PMEVTYPER29_EL0 = 0xdf7d,
	AARCH64_SYSREG_PMEVTYPER2_EL0 = 0xdf62,
	AARCH64_SYSREG_PMEVTYPER30_EL0 = 0xdf7e,
	AARCH64_SYSREG_PMEVTYPER3_EL0 = 0xdf63,
	AARCH64_SYSREG_PMEVTYPER4_EL0 = 0xdf64,
	AARCH64_SYSREG_PMEVTYPER5_EL0 = 0xdf65,
	AARCH64_SYSREG_PMEVTYPER6_EL0 = 0xdf66,
	AARCH64_SYSREG_PMEVTYPER7_EL0 = 0xdf67,
	AARCH64_SYSREG_PMEVTYPER8_EL0 = 0xdf68,
	AARCH64_SYSREG_PMEVTYPER9_EL0 = 0xdf69,
	AARCH64_SYSREG_PMIAR_EL1 = 0xc4f7,
	AARCH64_SYSREG_PMICFILTR_EL0 = 0xdcb0,
	AARCH64_SYSREG_PMICNTR_EL0 = 0xdca0,
	AARCH64_SYSREG_PMICNTSVR_EL1 = 0x8760,
	AARCH64_SYSREG_PMINTENCLR_EL1 = 0xc4f2,
	AARCH64_SYSREG_PMINTENSET_EL1 = 0xc4f1,
	AARCH64_SYSREG_PMMIR_EL1 = 0xc4f6,
	AARCH64_SYSREG_PMOVSCLR_EL0 = 0xdce3,
	AARCH64_SYSREG_PMOVSSET_EL0 = 0xdcf3,
	AARCH64_SYSREG_PMSCR_EL1 = 0xc4c8,
	AARCH64_SYSREG_PMSCR_EL12 = 0xecc8,
	AARCH64_SYSREG_PMSCR_EL2 = 0xe4c8,
	AARCH64_SYSREG_PMSDSFR_EL1 = 0xc4d4,
	AARCH64_SYSREG_PMSELR_EL0 = 0xdce5,
	AARCH64_SYSREG_PMSEVFR_EL1 = 0xc4cd,
	AARCH64_SYSREG_PMSFCR_EL1 = 0xc4cc,
	AARCH64_SYSREG_PMSICR_EL1 = 0xc4ca,
	AARCH64_SYSREG_PMSIDR_EL1 = 0xc4cf,
	AARCH64_SYSREG_PMSIRR_EL1 = 0xc4cb,
	AARCH64_SYSREG_PMSLATFR_EL1 = 0xc4ce,
	AARCH64_SYSREG_PMSNEVFR_EL1 = 0xc4c9,
	AARCH64_SYSREG_PMSSCR_EL1 = 0xc4eb,
	AARCH64_SYSREG_PMSWINC_EL0 = 0xdce4,
	AARCH64_SYSREG_PMUACR_EL1 = 0xc4f4,
	AARCH64_SYSREG_PMUSERENR_EL0 = 0xdcf0,
	AARCH64_SYSREG_PMXEVCNTR_EL0 = 0xdcea,
	AARCH64_SYSREG_PMXEVTYPER_EL0 = 0xdce9,
	AARCH64_SYSREG_PMZR_EL0 = 0xdcec,
	AARCH64_SYSREG_POR_EL0 = 0xdd14,
	AARCH64_SYSREG_POR_EL1 = 0xc514,
	AARCH64_SYSREG_POR_EL12 = 0xed14,
	AARCH64_SYSREG_POR_EL2 = 0xe514,
	AARCH64_SYSREG_POR_EL3 = 0xf514,
	AARCH64_SYSREG_PRBAR10_EL1 = 0xc368,
	AARCH64_SYSREG_PRBAR10_EL2 = 0xe368,
	AARCH64_SYSREG_PRBAR11_EL1 = 0xc36c,
	AARCH64_SYSREG_PRBAR11_EL2 = 0xe36c,
	AARCH64_SYSREG_PRBAR12_EL1 = 0xc370,
	AARCH64_SYSREG_PRBAR12_EL2 = 0xe370,
	AARCH64_SYSREG_PRBAR13_EL1 = 0xc374,
	AARCH64_SYSREG_PRBAR13_EL2 = 0xe374,
	AARCH64_SYSREG_PRBAR14_EL1 = 0xc378,
	AARCH64_SYSREG_PRBAR14_EL2 = 0xe378,
	AARCH64_SYSREG_PRBAR15_EL1 = 0xc37c,
	AARCH64_SYSREG_PRBAR15_EL2 = 0xe37c,
	AARCH64_SYSREG_PRBAR1_EL1 = 0xc344,
	AARCH64_SYSREG_PRBAR1_EL2 = 0xe344,
	AARCH64_SYSREG_PRBAR2_EL1 = 0xc348,
	AARCH64_SYSREG_PRBAR2_EL2 = 0xe348,
	AARCH64_SYSREG_PRBAR3_EL1 = 0xc34c,
	AARCH64_SYSREG_PRBAR3_EL2 = 0xe34c,
	AARCH64_SYSREG_PRBAR4_EL1 = 0xc350,
	AARCH64_SYSREG_PRBAR4_EL2 = 0xe350,
	AARCH64_SYSREG_PRBAR5_EL1 = 0xc354,
	AARCH64_SYSREG_PRBAR5_EL2 = 0xe354,
	AARCH64_SYSREG_PRBAR6_EL1 = 0xc358,
	AARCH64_SYSREG_PRBAR6_EL2 = 0xe358,
	AARCH64_SYSREG_PRBAR7_EL1 = 0xc35c,
	AARCH64_SYSREG_PRBAR7_EL2 = 0xe35c,
	AARCH64_SYSREG_PRBAR8_EL1 = 0xc360,
	AARCH64_SYSREG_PRBAR8_EL2 = 0xe360,
	AARCH64_SYSREG_PRBAR9_EL1 = 0xc364,
	AARCH64_SYSREG_PRBAR9_EL2 = 0xe364,
	AARCH64_SYSREG_PRBAR_EL1 = 0xc340,
	AARCH64_SYSREG_PRBAR_EL2 = 0xe340,
	AARCH64_SYSREG_PRENR_EL1 = 0xc309,
	AARCH64_SYSREG_PRENR_EL2 = 0xe309,
	AARCH64_SYSREG_PRLAR10_EL1 = 0xc369,
	AARCH64_SYSREG_PRLAR10_EL2 = 0xe369,
	AARCH64_SYSREG_PRLAR11_EL1 = 0xc36d,
	AARCH64_SYSREG_PRLAR11_EL2 = 0xe36d,
	AARCH64_SYSREG_PRLAR12_EL1 = 0xc371,
	AARCH64_SYSREG_PRLAR12_EL2 = 0xe371,
	AARCH64_SYSREG_PRLAR13_EL1 = 0xc375,
	AARCH64_SYSREG_PRLAR13_EL2 = 0xe375,
	AARCH64_SYSREG_PRLAR14_EL1 = 0xc379,
	AARCH64_SYSREG_PRLAR14_EL2 = 0xe379,
	AARCH64_SYSREG_PRLAR15_EL1 = 0xc37d,
	AARCH64_SYSREG_PRLAR15_EL2 = 0xe37d,
	AARCH64_SYSREG_PRLAR1_EL1 = 0xc345,
	AARCH64_SYSREG_PRLAR1_EL2 = 0xe345,
	AARCH64_SYSREG_PRLAR2_EL1 = 0xc349,
	AARCH64_SYSREG_PRLAR2_EL2 = 0xe349,
	AARCH64_SYSREG_PRLAR3_EL1 = 0xc34d,
	AARCH64_SYSREG_PRLAR3_EL2 = 0xe34d,
	AARCH64_SYSREG_PRLAR4_EL1 = 0xc351,
	AARCH64_SYSREG_PRLAR4_EL2 = 0xe351,
	AARCH64_SYSREG_PRLAR5_EL1 = 0xc355,
	AARCH64_SYSREG_PRLAR5_EL2 = 0xe355,
	AARCH64_SYSREG_PRLAR6_EL1 = 0xc359,
	AARCH64_SYSREG_PRLAR6_EL2 = 0xe359,
	AARCH64_SYSREG_PRLAR7_EL1 = 0xc35d,
	AARCH64_SYSREG_PRLAR7_EL2 = 0xe35d,
	AARCH64_SYSREG_PRLAR8_EL1 = 0xc361,
	AARCH64_SYSREG_PRLAR8_EL2 = 0xe361,
	AARCH64_SYSREG_PRLAR9_EL1 = 0xc365,
	AARCH64_SYSREG_PRLAR9_EL2 = 0xe365,
	AARCH64_SYSREG_PRLAR_EL1 = 0xc341,
	AARCH64_SYSREG_PRLAR_EL2 = 0xe341,
	AARCH64_SYSREG_PRSELR_EL1 = 0xc311,
	AARCH64_SYSREG_PRSELR_EL2 = 0xe311,
	AARCH64_SYSREG_RCWMASK_EL1 = 0xc686,
	AARCH64_SYSREG_RCWSMASK_EL1 = 0xc683,
	AARCH64_SYSREG_REVIDR_EL1 = 0xc006,
	AARCH64_SYSREG_RGSR_EL1 = 0xc085,
	AARCH64_SYSREG_RMR_EL1 = 0xc602,
	AARCH64_SYSREG_RMR_EL2 = 0xe602,
	AARCH64_SYSREG_RMR_EL3 = 0xf602,
	AARCH64_SYSREG_RNDR = 0xd920,
	AARCH64_SYSREG_RNDRRS = 0xd921,
	AARCH64_SYSREG_RVBAR_EL1 = 0xc601,
	AARCH64_SYSREG_RVBAR_EL2 = 0xe601,
	AARCH64_SYSREG_RVBAR_EL3 = 0xf601,
	AARCH64_SYSREG_S2PIR_EL2 = 0xe515,
	AARCH64_SYSREG_S2POR_EL1 = 0xc515,
	AARCH64_SYSREG_SCR_EL3 = 0xf088,
	AARCH64_SYSREG_SCTLR2_EL1 = 0xc083,
	AARCH64_SYSREG_SCTLR2_EL12 = 0xe883,
	AARCH64_SYSREG_SCTLR2_EL2 = 0xe083,
	AARCH64_SYSREG_SCTLR2_EL3 = 0xf083,
	AARCH64_SYSREG_SCTLR_EL1 = 0xc080,
	AARCH64_SYSREG_SCTLR_EL12 = 0xe880,
	AARCH64_SYSREG_SCTLR_EL2 = 0xe080,
	AARCH64_SYSREG_SCTLR_EL3 = 0xf080,
	AARCH64_SYSREG_SCXTNUM_EL0 = 0xde87,
	AARCH64_SYSREG_SCXTNUM_EL1 = 0xc687,
	AARCH64_SYSREG_SCXTNUM_EL12 = 0xee87,
	AARCH64_SYSREG_SCXTNUM_EL2 = 0xe687,
	AARCH64_SYSREG_SCXTNUM_EL3 = 0xf687,
	AARCH64_SYSREG_SDER32_EL2 = 0xe099,
	AARCH64_SYSREG_SDER32_EL3 = 0xf089,
	AARCH64_SYSREG_SMCR_EL1 = 0xc096,
	AARCH64_SYSREG_SMCR_EL12 = 0xe896,
	AARCH64_SYSREG_SMCR_EL2 = 0xe096,
	AARCH64_SYSREG_SMCR_EL3 = 0xf096,
	AARCH64_SYSREG_SMIDR_EL1 = 0xc806,
	AARCH64_SYSREG_SMPRIMAP_EL2 = 0xe095,
	AARCH64_SYSREG_SMPRI_EL1 = 0xc094,
	AARCH64_SYSREG_SPMACCESSR_EL1 = 0x84eb,
	AARCH64_SYSREG_SPMACCESSR_EL12 = 0xaceb,
	AARCH64_SYSREG_SPMACCESSR_EL2 = 0xa4eb,
	AARCH64_SYSREG_SPMACCESSR_EL3 = 0xb4eb,
	AARCH64_SYSREG_SPMCFGR_EL1 = 0x84ef,
	AARCH64_SYSREG_SPMCGCR0_EL1 = 0x84e8,
	AARCH64_SYSREG_SPMCGCR1_EL1 = 0x84e9,
	AARCH64_SYSREG_SPMCNTENCLR_EL0 = 0x9ce2,
	AARCH64_SYSREG_SPMCNTENSET_EL0 = 0x9ce1,
	AARCH64_SYSREG_SPMCR_EL0 = 0x9ce0,
	AARCH64_SYSREG_SPMDEVAFF_EL1 = 0x84ee,
	AARCH64_SYSREG_SPMDEVARCH_EL1 = 0x84ed,
	AARCH64_SYSREG_SPMEVCNTR0_EL0 = 0x9f00,
	AARCH64_SYSREG_SPMEVCNTR10_EL0 = 0x9f0a,
	AARCH64_SYSREG_SPMEVCNTR11_EL0 = 0x9f0b,
	AARCH64_SYSREG_SPMEVCNTR12_EL0 = 0x9f0c,
	AARCH64_SYSREG_SPMEVCNTR13_EL0 = 0x9f0d,
	AARCH64_SYSREG_SPMEVCNTR14_EL0 = 0x9f0e,
	AARCH64_SYSREG_SPMEVCNTR15_EL0 = 0x9f0f,
	AARCH64_SYSREG_SPMEVCNTR1_EL0 = 0x9f01,
	AARCH64_SYSREG_SPMEVCNTR2_EL0 = 0x9f02,
	AARCH64_SYSREG_SPMEVCNTR3_EL0 = 0x9f03,
	AARCH64_SYSREG_SPMEVCNTR4_EL0 = 0x9f04,
	AARCH64_SYSREG_SPMEVCNTR5_EL0 = 0x9f05,
	AARCH64_SYSREG_SPMEVCNTR6_EL0 = 0x9f06,
	AARCH64_SYSREG_SPMEVCNTR7_EL0 = 0x9f07,
	AARCH64_SYSREG_SPMEVCNTR8_EL0 = 0x9f08,
	AARCH64_SYSREG_SPMEVCNTR9_EL0 = 0x9f09,
	AARCH64_SYSREG_SPMEVFILT2R0_EL0 = 0x9f30,
	AARCH64_SYSREG_SPMEVFILT2R10_EL0 = 0x9f3a,
	AARCH64_SYSREG_SPMEVFILT2R11_EL0 = 0x9f3b,
	AARCH64_SYSREG_SPMEVFILT2R12_EL0 = 0x9f3c,
	AARCH64_SYSREG_SPMEVFILT2R13_EL0 = 0x9f3d,
	AARCH64_SYSREG_SPMEVFILT2R14_EL0 = 0x9f3e,
	AARCH64_SYSREG_SPMEVFILT2R15_EL0 = 0x9f3f,
	AARCH64_SYSREG_SPMEVFILT2R1_EL0 = 0x9f31,
	AARCH64_SYSREG_SPMEVFILT2R2_EL0 = 0x9f32,
	AARCH64_SYSREG_SPMEVFILT2R3_EL0 = 0x9f33,
	AARCH64_SYSREG_SPMEVFILT2R4_EL0 = 0x9f34,
	AARCH64_SYSREG_SPMEVFILT2R5_EL0 = 0x9f35,
	AARCH64_SYSREG_SPMEVFILT2R6_EL0 = 0x9f36,
	AARCH64_SYSREG_SPMEVFILT2R7_EL0 = 0x9f37,
	AARCH64_SYSREG_SPMEVFILT2R8_EL0 = 0x9f38,
	AARCH64_SYSREG_SPMEVFILT2R9_EL0 = 0x9f39,
	AARCH64_SYSREG_SPMEVFILTR0_EL0 = 0x9f20,
	AARCH64_SYSREG_SPMEVFILTR10_EL0 = 0x9f2a,
	AARCH64_SYSREG_SPMEVFILTR11_EL0 = 0x9f2b,
	AARCH64_SYSREG_SPMEVFILTR12_EL0 = 0x9f2c,
	AARCH64_SYSREG_SPMEVFILTR13_EL0 = 0x9f2d,
	AARCH64_SYSREG_SPMEVFILTR14_EL0 = 0x9f2e,
	AARCH64_SYSREG_SPMEVFILTR15_EL0 = 0x9f2f,
	AARCH64_SYSREG_SPMEVFILTR1_EL0 = 0x9f21,
	AARCH64_SYSREG_SPMEVFILTR2_EL0 = 0x9f22,
	AARCH64_SYSREG_SPMEVFILTR3_EL0 = 0x9f23,
	AARCH64_SYSREG_SPMEVFILTR4_EL0 = 0x9f24,
	AARCH64_SYSREG_SPMEVFILTR5_EL0 = 0x9f25,
	AARCH64_SYSREG_SPMEVFILTR6_EL0 = 0x9f26,
	AARCH64_SYSREG_SPMEVFILTR7_EL0 = 0x9f27,
	AARCH64_SYSREG_SPMEVFILTR8_EL0 = 0x9f28,
	AARCH64_SYSREG_SPMEVFILTR9_EL0 = 0x9f29,
	AARCH64_SYSREG_SPMEVTYPER0_EL0 = 0x9f10,
	AARCH64_SYSREG_SPMEVTYPER10_EL0 = 0x9f1a,
	AARCH64_SYSREG_SPMEVTYPER11_EL0 = 0x9f1b,
	AARCH64_SYSREG_SPMEVTYPER12_EL0 = 0x9f1c,
	AARCH64_SYSREG_SPMEVTYPER13_EL0 = 0x9f1d,
	AARCH64_SYSREG_SPMEVTYPER14_EL0 = 0x9f1e,
	AARCH64_SYSREG_SPMEVTYPER15_EL0 = 0x9f1f,
	AARCH64_SYSREG_SPMEVTYPER1_EL0 = 0x9f11,
	AARCH64_SYSREG_SPMEVTYPER2_EL0 = 0x9f12,
	AARCH64_SYSREG_SPMEVTYPER3_EL0 = 0x9f13,
	AARCH64_SYSREG_SPMEVTYPER4_EL0 = 0x9f14,
	AARCH64_SYSREG_SPMEVTYPER5_EL0 = 0x9f15,
	AARCH64_SYSREG_SPMEVTYPER6_EL0 = 0x9f16,
	AARCH64_SYSREG_SPMEVTYPER7_EL0 = 0x9f17,
	AARCH64_SYSREG_SPMEVTYPER8_EL0 = 0x9f18,
	AARCH64_SYSREG_SPMEVTYPER9_EL0 = 0x9f19,
	AARCH64_SYSREG_SPMIIDR_EL1 = 0x84ec,
	AARCH64_SYSREG_SPMINTENCLR_EL1 = 0x84f2,
	AARCH64_SYSREG_SPMINTENSET_EL1 = 0x84f1,
	AARCH64_SYSREG_SPMOVSCLR_EL0 = 0x9ce3,
	AARCH64_SYSREG_SPMOVSSET_EL0 = 0x9cf3,
	AARCH64_SYSREG_SPMROOTCR_EL3 = 0xb4f7,
	AARCH64_SYSREG_SPMSCR_EL1 = 0xbcf7,
	AARCH64_SYSREG_SPMSELR_EL0 = 0x9ce5,
	AARCH64_SYSREG_SPMZR_EL0 = 0x9ce4,
	AARCH64_SYSREG_SPSEL = 0xc210,
	AARCH64_SYSREG_SPSR_ABT = 0xe219,
	AARCH64_SYSREG_SPSR_EL1 = 0xc200,
	AARCH64_SYSREG_SPSR_EL12 = 0xea00,
	AARCH64_SYSREG_SPSR_EL2 = 0xe200,
	AARCH64_SYSREG_SPSR_EL3 = 0xf200,
	AARCH64_SYSREG_SPSR_FIQ = 0xe21b,
	AARCH64_SYSREG_SPSR_IRQ = 0xe218,
	AARCH64_SYSREG_SPSR_UND = 0xe21a,
	AARCH64_SYSREG_SP_EL0 = 0xc208,
	AARCH64_SYSREG_SP_EL1 = 0xe208,
	AARCH64_SYSREG_SP_EL2 = 0xf208,
	AARCH64_SYSREG_SSBS = 0xda16,
	AARCH64_SYSREG_SVCR = 0xda12,
	AARCH64_SYSREG_TCO = 0xda17,
	AARCH64_SYSREG_TCR2_EL1 = 0xc103,
	AARCH64_SYSREG_TCR2_EL12 = 0xe903,
	AARCH64_SYSREG_TCR2_EL2 = 0xe103,
	AARCH64_SYSREG_TCR_EL1 = 0xc102,
	AARCH64_SYSREG_TCR_EL12 = 0xe902,
	AARCH64_SYSREG_TCR_EL2 = 0xe102,
	AARCH64_SYSREG_TCR_EL3 = 0xf102,
	AARCH64_SYSREG_TEECR32_EL1 = 0x9000,
	AARCH64_SYSREG_TEEHBR32_EL1 = 0x9080,
	AARCH64_SYSREG_TFSRE0_EL1 = 0xc2b1,
	AARCH64_SYSREG_TFSR_EL1 = 0xc2b0,
	AARCH64_SYSREG_TFSR_EL12 = 0xeab0,
	AARCH64_SYSREG_TFSR_EL2 = 0xe2b0,
	AARCH64_SYSREG_TFSR_EL3 = 0xf2b0,
	AARCH64_SYSREG_TPIDR2_EL0 = 0xde85,
	AARCH64_SYSREG_TPIDRRO_EL0 = 0xde83,
	AARCH64_SYSREG_TPIDR_EL0 = 0xde82,
	AARCH64_SYSREG_TPIDR_EL1 = 0xc684,
	AARCH64_SYSREG_TPIDR_EL2 = 0xe682,
	AARCH64_SYSREG_TPIDR_EL3 = 0xf682,
	AARCH64_SYSREG_TRBBASER_EL1 = 0xc4da,
	AARCH64_SYSREG_TRBIDR_EL1 = 0xc4df,
	AARCH64_SYSREG_TRBLIMITR_EL1 = 0xc4d8,
	AARCH64_SYSREG_TRBMAR_EL1 = 0xc4dc,
	AARCH64_SYSREG_TRBPTR_EL1 = 0xc4d9,
	AARCH64_SYSREG_TRBSR_EL1 = 0xc4db,
	AARCH64_SYSREG_TRBTRG_EL1 = 0xc4de,
	AARCH64_SYSREG_TRCACATR0 = 0x8902,
	AARCH64_SYSREG_TRCACATR1 = 0x8912,
	AARCH64_SYSREG_TRCACATR10 = 0x8923,
	AARCH64_SYSREG_TRCACATR11 = 0x8933,
	AARCH64_SYSREG_TRCACATR12 = 0x8943,
	AARCH64_SYSREG_TRCACATR13 = 0x8953,
	AARCH64_SYSREG_TRCACATR14 = 0x8963,
	AARCH64_SYSREG_TRCACATR15 = 0x8973,
	AARCH64_SYSREG_TRCACATR2 = 0x8922,
	AARCH64_SYSREG_TRCACATR3 = 0x8932,
	AARCH64_SYSREG_TRCACATR4 = 0x8942,
	AARCH64_SYSREG_TRCACATR5 = 0x8952,
	AARCH64_SYSREG_TRCACATR6 = 0x8962,
	AARCH64_SYSREG_TRCACATR7 = 0x8972,
	AARCH64_SYSREG_TRCACATR8 = 0x8903,
	AARCH64_SYSREG_TRCACATR9 = 0x8913,
	AARCH64_SYSREG_TRCACVR0 = 0x8900,
	AARCH64_SYSREG_TRCACVR1 = 0x8910,
	AARCH64_SYSREG_TRCACVR10 = 0x8921,
	AARCH64_SYSREG_TRCACVR11 = 0x8931,
	AARCH64_SYSREG_TRCACVR12 = 0x8941,
	AARCH64_SYSREG_TRCACVR13 = 0x8951,
	AARCH64_SYSREG_TRCACVR14 = 0x8961,
	AARCH64_SYSREG_TRCACVR15 = 0x8971,
	AARCH64_SYSREG_TRCACVR2 = 0x8920,
	AARCH64_SYSREG_TRCACVR3 = 0x8930,
	AARCH64_SYSREG_TRCACVR4 = 0x8940,
	AARCH64_SYSREG_TRCACVR5 = 0x8950,
	AARCH64_SYSREG_TRCACVR6 = 0x8960,
	AARCH64_SYSREG_TRCACVR7 = 0x8970,
	AARCH64_SYSREG_TRCACVR8 = 0x8901,
	AARCH64_SYSREG_TRCACVR9 = 0x8911,
	AARCH64_SYSREG_TRCAUTHSTATUS = 0x8bf6,
	AARCH64_SYSREG_TRCAUXCTLR = 0x8830,
	AARCH64_SYSREG_TRCBBCTLR = 0x8878,
	AARCH64_SYSREG_TRCCCCTLR = 0x8870,
	AARCH64_SYSREG_TRCCIDCCTLR0 = 0x8982,
	AARCH64_SYSREG_TRCCIDCCTLR1 = 0x898a,
	AARCH64_SYSREG_TRCCIDCVR0 = 0x8980,
	AARCH64_SYSREG_TRCCIDCVR1 = 0x8990,
	AARCH64_SYSREG_TRCCIDCVR2 = 0x89a0,
	AARCH64_SYSREG_TRCCIDCVR3 = 0x89b0,
	AARCH64_SYSREG_TRCCIDCVR4 = 0x89c0,
	AARCH64_SYSREG_TRCCIDCVR5 = 0x89d0,
	AARCH64_SYSREG_TRCCIDCVR6 = 0x89e0,
	AARCH64_SYSREG_TRCCIDCVR7 = 0x89f0,
	AARCH64_SYSREG_TRCCIDR0 = 0x8be7,
	AARCH64_SYSREG_TRCCIDR1 = 0x8bef,
	AARCH64_SYSREG_TRCCIDR2 = 0x8bf7,
	AARCH64_SYSREG_TRCCIDR3 = 0x8bff,
	AARCH64_SYSREG_TRCCLAIMCLR = 0x8bce,
	AARCH64_SYSREG_TRCCLAIMSET = 0x8bc6,
	AARCH64_SYSREG_TRCCNTCTLR0 = 0x8825,
	AARCH64_SYSREG_TRCCNTCTLR1 = 0x882d,
	AARCH64_SYSREG_TRCCNTCTLR2 = 0x8835,
	AARCH64_SYSREG_TRCCNTCTLR3 = 0x883d,
	AARCH64_SYSREG_TRCCNTRLDVR0 = 0x8805,
	AARCH64_SYSREG_TRCCNTRLDVR1 = 0x880d,
	AARCH64_SYSREG_TRCCNTRLDVR2 = 0x8815,
	AARCH64_SYSREG_TRCCNTRLDVR3 = 0x881d,
	AARCH64_SYSREG_TRCCNTVR0 = 0x8845,
	AARCH64_SYSREG_TRCCNTVR1 = 0x884d,
	AARCH64_SYSREG_TRCCNTVR2 = 0x8855,
	AARCH64_SYSREG_TRCCNTVR3 = 0x885d,
	AARCH64_SYSREG_TRCCONFIGR = 0x8820,
	AARCH64_SYSREG_TRCDEVAFF0 = 0x8bd6,
	AARCH64_SYSREG_TRCDEVAFF1 = 0x8bde,
	AARCH64_SYSREG_TRCDEVARCH = 0x8bfe,
	AARCH64_SYSREG_TRCDEVID = 0x8b97,
	AARCH64_SYSREG_TRCDEVTYPE = 0x8b9f,
	AARCH64_SYSREG_TRCDVCMR0 = 0x8906,
	AARCH64_SYSREG_TRCDVCMR1 = 0x8926,
	AARCH64_SYSREG_TRCDVCMR2 = 0x8946,
	AARCH64_SYSREG_TRCDVCMR3 = 0x8966,
	AARCH64_SYSREG_TRCDVCMR4 = 0x8907,
	AARCH64_SYSREG_TRCDVCMR5 = 0x8927,
	AARCH64_SYSREG_TRCDVCMR6 = 0x8947,
	AARCH64_SYSREG_TRCDVCMR7 = 0x8967,
	AARCH64_SYSREG_TRCDVCVR0 = 0x8904,
	AARCH64_SYSREG_TRCDVCVR1 = 0x8924,
	AARCH64_SYSREG_TRCDVCVR2 = 0x8944,
	AARCH64_SYSREG_TRCDVCVR3 = 0x8964,
	AARCH64_SYSREG_TRCDVCVR4 = 0x8905,
	AARCH64_SYSREG_TRCDVCVR5 = 0x8925,
	AARCH64_SYSREG_TRCDVCVR6 = 0x8945,
	AARCH64_SYSREG_TRCDVCVR7 = 0x8965,
	AARCH64_SYSREG_TRCEVENTCTL0R = 0x8840,
	AARCH64_SYSREG_TRCEVENTCTL1R = 0x8848,
	AARCH64_SYSREG_TRCEXTINSELR = 0x8844,
	AARCH64_SYSREG_TRCEXTINSELR0 = 0x8844,
	AARCH64_SYSREG_TRCEXTINSELR1 = 0x884c,
	AARCH64_SYSREG_TRCEXTINSELR2 = 0x8854,
	AARCH64_SYSREG_TRCEXTINSELR3 = 0x885c,
	AARCH64_SYSREG_TRCIDR0 = 0x8847,
	AARCH64_SYSREG_TRCIDR1 = 0x884f,
	AARCH64_SYSREG_TRCIDR10 = 0x8816,
	AARCH64_SYSREG_TRCIDR11 = 0x881e,
	AARCH64_SYSREG_TRCIDR12 = 0x8826,
	AARCH64_SYSREG_TRCIDR13 = 0x882e,
	AARCH64_SYSREG_TRCIDR2 = 0x8857,
	AARCH64_SYSREG_TRCIDR3 = 0x885f,
	AARCH64_SYSREG_TRCIDR4 = 0x8867,
	AARCH64_SYSREG_TRCIDR5 = 0x886f,
	AARCH64_SYSREG_TRCIDR6 = 0x8877,
	AARCH64_SYSREG_TRCIDR7 = 0x887f,
	AARCH64_SYSREG_TRCIDR8 = 0x8806,
	AARCH64_SYSREG_TRCIDR9 = 0x880e,
	AARCH64_SYSREG_TRCIMSPEC0 = 0x8807,
	AARCH64_SYSREG_TRCIMSPEC1 = 0x880f,
	AARCH64_SYSREG_TRCIMSPEC2 = 0x8817,
	AARCH64_SYSREG_TRCIMSPEC3 = 0x881f,
	AARCH64_SYSREG_TRCIMSPEC4 = 0x8827,
	AARCH64_SYSREG_TRCIMSPEC5 = 0x882f,
	AARCH64_SYSREG_TRCIMSPEC6 = 0x8837,
	AARCH64_SYSREG_TRCIMSPEC7 = 0x883f,
	AARCH64_SYSREG_TRCITCTRL = 0x8b84,
	AARCH64_SYSREG_TRCITECR_EL1 = 0xc093,
	AARCH64_SYSREG_TRCITECR_EL12 = 0xe893,
	AARCH64_SYSREG_TRCITECR_EL2 = 0xe093,
	AARCH64_SYSREG_TRCITEEDCR = 0x8811,
	AARCH64_SYSREG_TRCLAR = 0x8be6,
	AARCH64_SYSREG_TRCLSR = 0x8bee,
	AARCH64_SYSREG_TRCOSLAR = 0x8884,
	AARCH64_SYSREG_TRCOSLSR = 0x888c,
	AARCH64_SYSREG_TRCPDCR = 0x88a4,
	AARCH64_SYSREG_TRCPDSR = 0x88ac,
	AARCH64_SYSREG_TRCPIDR0 = 0x8bc7,
	AARCH64_SYSREG_TRCPIDR1 = 0x8bcf,
	AARCH64_SYSREG_TRCPIDR2 = 0x8bd7,
	AARCH64_SYSREG_TRCPIDR3 = 0x8bdf,
	AARCH64_SYSREG_TRCPIDR4 = 0x8ba7,
	AARCH64_SYSREG_TRCPIDR5 = 0x8baf,
	AARCH64_SYSREG_TRCPIDR6 = 0x8bb7,
	AARCH64_SYSREG_TRCPIDR7 = 0x8bbf,
	AARCH64_SYSREG_TRCPRGCTLR = 0x8808,
	AARCH64_SYSREG_TRCPROCSELR = 0x8810,
	AARCH64_SYSREG_TRCQCTLR = 0x8809,
	AARCH64_SYSREG_TRCRSCTLR10 = 0x88d0,
	AARCH64_SYSREG_TRCRSCTLR11 = 0x88d8,
	AARCH64_SYSREG_TRCRSCTLR12 = 0x88e0,
	AARCH64_SYSREG_TRCRSCTLR13 = 0x88e8,
	AARCH64_SYSREG_TRCRSCTLR14 = 0x88f0,
	AARCH64_SYSREG_TRCRSCTLR15 = 0x88f8,
	AARCH64_SYSREG_TRCRSCTLR16 = 0x8881,
	AARCH64_SYSREG_TRCRSCTLR17 = 0x8889,
	AARCH64_SYSREG_TRCRSCTLR18 = 0x8891,
	AARCH64_SYSREG_TRCRSCTLR19 = 0x8899,
	AARCH64_SYSREG_TRCRSCTLR2 = 0x8890,
	AARCH64_SYSREG_TRCRSCTLR20 = 0x88a1,
	AARCH64_SYSREG_TRCRSCTLR21 = 0x88a9,
	AARCH64_SYSREG_TRCRSCTLR22 = 0x88b1,
	AARCH64_SYSREG_TRCRSCTLR23 = 0x88b9,
	AARCH64_SYSREG_TRCRSCTLR24 = 0x88c1,
	AARCH64_SYSREG_TRCRSCTLR25 = 0x88c9,
	AARCH64_SYSREG_TRCRSCTLR26 = 0x88d1,
	AARCH64_SYSREG_TRCRSCTLR27 = 0x88d9,
	AARCH64_SYSREG_TRCRSCTLR28 = 0x88e1,
	AARCH64_SYSREG_TRCRSCTLR29 = 0x88e9,
	AARCH64_SYSREG_TRCRSCTLR3 = 0x8898,
	AARCH64_SYSREG_TRCRSCTLR30 = 0x88f1,
	AARCH64_SYSREG_TRCRSCTLR31 = 0x88f9,
	AARCH64_SYSREG_TRCRSCTLR4 = 0x88a0,
	AARCH64_SYSREG_TRCRSCTLR5 = 0x88a8,
	AARCH64_SYSREG_TRCRSCTLR6 = 0x88b0,
	AARCH64_SYSREG_TRCRSCTLR7 = 0x88b8,
	AARCH64_SYSREG_TRCRSCTLR8 = 0x88c0,
	AARCH64_SYSREG_TRCRSCTLR9 = 0x88c8,
	AARCH64_SYSREG_TRCRSR = 0x8850,
	AARCH64_SYSREG_TRCSEQEVR0 = 0x8804,
	AARCH64_SYSREG_TRCSEQEVR1 = 0x880c,
	AARCH64_SYSREG_TRCSEQEVR2 = 0x8814,
	AARCH64_SYSREG_TRCSEQRSTEVR = 0x8834,
	AARCH64_SYSREG_TRCSEQSTR = 0x883c,
	AARCH64_SYSREG_TRCSSCCR0 = 0x8882,
	AARCH64_SYSREG_TRCSSCCR1 = 0x888a,
	AARCH64_SYSREG_TRCSSCCR2 = 0x8892,
	AARCH64_SYSREG_TRCSSCCR3 = 0x889a,
	AARCH64_SYSREG_TRCSSCCR4 = 0x88a2,
	AARCH64_SYSREG_TRCSSCCR5 = 0x88aa,
	AARCH64_SYSREG_TRCSSCCR6 = 0x88b2,
	AARCH64_SYSREG_TRCSSCCR7 = 0x88ba,
	AARCH64_SYSREG_TRCSSCSR0 = 0x88c2,
	AARCH64_SYSREG_TRCSSCSR1 = 0x88ca,
	AARCH64_SYSREG_TRCSSCSR2 = 0x88d2,
	AARCH64_SYSREG_TRCSSCSR3 = 0x88da,
	AARCH64_SYSREG_TRCSSCSR4 = 0x88e2,
	AARCH64_SYSREG_TRCSSCSR5 = 0x88ea,
	AARCH64_SYSREG_TRCSSCSR6 = 0x88f2,
	AARCH64_SYSREG_TRCSSCSR7 = 0x88fa,
	AARCH64_SYSREG_TRCSSPCICR0 = 0x8883,
	AARCH64_SYSREG_TRCSSPCICR1 = 0x888b,
	AARCH64_SYSREG_TRCSSPCICR2 = 0x8893,
	AARCH64_SYSREG_TRCSSPCICR3 = 0x889b,
	AARCH64_SYSREG_TRCSSPCICR4 = 0x88a3,
	AARCH64_SYSREG_TRCSSPCICR5 = 0x88ab,
	AARCH64_SYSREG_TRCSSPCICR6 = 0x88b3,
	AARCH64_SYSREG_TRCSSPCICR7 = 0x88bb,
	AARCH64_SYSREG_TRCSTALLCTLR = 0x8858,
	AARCH64_SYSREG_TRCSTATR = 0x8818,
	AARCH64_SYSREG_TRCSYNCPR = 0x8868,
	AARCH64_SYSREG_TRCTRACEIDR = 0x8801,
	AARCH64_SYSREG_TRCTSCTLR = 0x8860,
	AARCH64_SYSREG_TRCVDARCCTLR = 0x8852,
	AARCH64_SYSREG_TRCVDCTLR = 0x8842,
	AARCH64_SYSREG_TRCVDSACCTLR = 0x884a,
	AARCH64_SYSREG_TRCVICTLR = 0x8802,
	AARCH64_SYSREG_TRCVIIECTLR = 0x880a,
	AARCH64_SYSREG_TRCVIPCSSCTLR = 0x881a,
	AARCH64_SYSREG_TRCVISSCTLR = 0x8812,
	AARCH64_SYSREG_TRCVMIDCCTLR0 = 0x8992,
	AARCH64_SYSREG_TRCVMIDCCTLR1 = 0x899a,
	AARCH64_SYSREG_TRCVMIDCVR0 = 0x8981,
	AARCH64_SYSREG_TRCVMIDCVR1 = 0x8991,
	AARCH64_SYSREG_TRCVMIDCVR2 = 0x89a1,
	AARCH64_SYSREG_TRCVMIDCVR3 = 0x89b1,
	AARCH64_SYSREG_TRCVMIDCVR4 = 0x89c1,
	AARCH64_SYSREG_TRCVMIDCVR5 = 0x89d1,
	AARCH64_SYSREG_TRCVMIDCVR6 = 0x89e1,
	AARCH64_SYSREG_TRCVMIDCVR7 = 0x89f1,
	AARCH64_SYSREG_TRFCR_EL1 = 0xc091,
	AARCH64_SYSREG_TRFCR_EL12 = 0xe891,
	AARCH64_SYSREG_TRFCR_EL2 = 0xe091,
	AARCH64_SYSREG_TTBR0_EL1 = 0xc100,
	AARCH64_SYSREG_TTBR0_EL12 = 0xe900,
	AARCH64_SYSREG_TTBR0_EL2 = 0xe100,
	AARCH64_SYSREG_VSCTLR_EL2 = 0xe100,
	AARCH64_SYSREG_TTBR0_EL3 = 0xf100,
	AARCH64_SYSREG_TTBR1_EL1 = 0xc101,
	AARCH64_SYSREG_TTBR1_EL12 = 0xe901,
	AARCH64_SYSREG_TTBR1_EL2 = 0xe101,
	AARCH64_SYSREG_UAO = 0xc214,
	AARCH64_SYSREG_VBAR_EL1 = 0xc600,
	AARCH64_SYSREG_VBAR_EL12 = 0xee00,
	AARCH64_SYSREG_VBAR_EL2 = 0xe600,
	AARCH64_SYSREG_VBAR_EL3 = 0xf600,
	AARCH64_SYSREG_VDISR_EL2 = 0xe609,
	AARCH64_SYSREG_VDISR_EL3 = 0xf609,
	AARCH64_SYSREG_VMECID_A_EL2 = 0xe549,
	AARCH64_SYSREG_VMECID_P_EL2 = 0xe548,
	AARCH64_SYSREG_VMPIDR_EL2 = 0xe005,
	AARCH64_SYSREG_VNCR_EL2 = 0xe110,
	AARCH64_SYSREG_VPIDR_EL2 = 0xe000,
	AARCH64_SYSREG_VSESR_EL2 = 0xe293,
	AARCH64_SYSREG_VSESR_EL3 = 0xf293,
	AARCH64_SYSREG_VSTCR_EL2 = 0xe132,
	AARCH64_SYSREG_VSTTBR_EL2 = 0xe130,
	AARCH64_SYSREG_VTCR_EL2 = 0xe10a,
	AARCH64_SYSREG_VTTBR_EL2 = 0xe108,
	AARCH64_SYSREG_ZCR_EL1 = 0xc090,
	AARCH64_SYSREG_ZCR_EL12 = 0xe890,
	AARCH64_SYSREG_ZCR_EL2 = 0xe090,
	AARCH64_SYSREG_ZCR_EL3 = 0xf090,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_SysReg> end
	AARCH64_SYSREG_ENDING = UINT16_MAX,
} aarch64_sysreg;

typedef enum {
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_TSB> begin
	// clang-format off

	AARCH64_TSB_CSYNC = 0x0,

	// clang-format on
	// generated content <AArch64GenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_TSB> end
	AARCH64_TSB_ENDING,
} aarch64_tsb;

typedef union {
	aarch64_sysreg sysreg;
	aarch64_tlbi tlbi;
	aarch64_ic ic;
	int raw_val;
} aarch64_sysop_reg;

typedef union {
	aarch64_dbnxs dbnxs;
	aarch64_exactfpimm exactfpimm;
	int raw_val;
} aarch64_sysop_imm;

typedef union {
	aarch64_svcr svcr;
	aarch64_at at;
	aarch64_db db;
	aarch64_dc dc;
	aarch64_isb isb;
	aarch64_tsb tsb;
	aarch64_prfm prfm;
	aarch64_sveprfm sveprfm;
	aarch64_rprfm rprfm;
	aarch64_pstateimm0_15 pstateimm0_15;
	aarch64_pstateimm0_1 pstateimm0_1;
	aarch64_psb psb;
	aarch64_bti bti;
	aarch64_svepredpat svepredpat;
	aarch64_sveveclenspecifier sveveclenspecifier;
	int raw_val;
} aarch64_sysop_alias;

/// Operand type for instruction's operands
typedef enum aarch64_op_type {
  AARCH64_OP_INVALID = CS_OP_INVALID, ///< = CS_OP_INVALID (Uninitialized).
  AARCH64_OP_REG = CS_OP_REG,		///< = CS_OP_REG (Register operand).
  AARCH64_OP_IMM = CS_OP_IMM,		///< = CS_OP_IMM (Immediate operand).
  AARCH64_OP_MEM_REG = CS_OP_MEM_REG,		///< Register which references memory.
  AARCH64_OP_MEM_IMM = CS_OP_MEM_IMM,		///< = Immediate value which references memory.
  AARCH64_OP_MEM = CS_OP_MEM,		///< = CS_OP_MEM (Memory operand).
  AARCH64_OP_FP = CS_OP_FP,		///< = CS_OP_FP (Floating-Point operand).
  AARCH64_OP_CIMM = CS_OP_SPECIAL + 0,	///< C-Immediate
  AARCH64_OP_REG_MRS = CS_OP_SPECIAL + 1,	///< MRS register operand.
  AARCH64_OP_REG_MSR = CS_OP_SPECIAL + 2,	///< MSR register operand.
  AARCH64_OP_IMPLICIT_IMM_0 = CS_OP_SPECIAL + 3, ///< Implicit immediate operand 0
	// Different system operands.
	AARCH64_OP_SVCR = CS_OP_SPECIAL + 4,
	AARCH64_OP_AT = CS_OP_SPECIAL + 5,
	AARCH64_OP_DB = CS_OP_SPECIAL + 6,
	AARCH64_OP_DC = CS_OP_SPECIAL + 7,
	AARCH64_OP_ISB = CS_OP_SPECIAL + 8,
	AARCH64_OP_TSB = CS_OP_SPECIAL + 9,
	AARCH64_OP_PRFM = CS_OP_SPECIAL + 10,
	AARCH64_OP_SVEPRFM = CS_OP_SPECIAL + 11,
	AARCH64_OP_RPRFM = CS_OP_SPECIAL + 12,
	AARCH64_OP_PSTATEIMM0_15 = CS_OP_SPECIAL + 13,
	AARCH64_OP_PSTATEIMM0_1 = CS_OP_SPECIAL + 14,
	AARCH64_OP_PSB = CS_OP_SPECIAL + 15,
	AARCH64_OP_BTI = CS_OP_SPECIAL + 16,
	AARCH64_OP_SVEPREDPAT = CS_OP_SPECIAL + 17,
	AARCH64_OP_SVEVECLENSPECIFIER = CS_OP_SPECIAL + 18,
	AARCH64_OP_SME = CS_OP_SPECIAL + 19,
	AARCH64_OP_IMM_RANGE = CS_OP_SPECIAL + 20,
	AARCH64_OP_TLBI = CS_OP_SPECIAL + 21,
	AARCH64_OP_IC = CS_OP_SPECIAL + 22,
	AARCH64_OP_DBNXS = CS_OP_SPECIAL + 23,
	AARCH64_OP_EXACTFPIMM = CS_OP_SPECIAL + 24,
	AARCH64_OP_SYSREG = CS_OP_SPECIAL + 25,
	AARCH64_OP_SYSIMM = CS_OP_SPECIAL + 26,
	AARCH64_OP_SYSALIAS = CS_OP_SPECIAL + 27,
	AARCH64_OP_PRED = CS_OP_SPECIAL + 28,
} aarch64_op_type;

typedef struct {
	aarch64_sysop_reg reg;
	aarch64_sysop_imm imm;
	aarch64_sysop_alias alias;
	aarch64_op_type sub_type; ///< Specifies which field is set.
} aarch64_sysop;

/// AArch64 registers
typedef enum aarch64_reg {
	// generated content <AArch64GenCSRegEnum.inc> begin
	// clang-format off

	AARCH64_REG_INVALID = 0,
	AARCH64_REG_FFR = 1,
	AARCH64_REG_FP = 2,
	AARCH64_REG_FPCR = 3,
	AARCH64_REG_LR = 4,
	AARCH64_REG_NZCV = 5,
	AARCH64_REG_SP = 6,
	AARCH64_REG_VG = 7,
	AARCH64_REG_WSP = 8,
	AARCH64_REG_WZR = 9,
	AARCH64_REG_XZR = 10,
	AARCH64_REG_ZA = 11,
	AARCH64_REG_B0 = 12,
	AARCH64_REG_B1 = 13,
	AARCH64_REG_B2 = 14,
	AARCH64_REG_B3 = 15,
	AARCH64_REG_B4 = 16,
	AARCH64_REG_B5 = 17,
	AARCH64_REG_B6 = 18,
	AARCH64_REG_B7 = 19,
	AARCH64_REG_B8 = 20,
	AARCH64_REG_B9 = 21,
	AARCH64_REG_B10 = 22,
	AARCH64_REG_B11 = 23,
	AARCH64_REG_B12 = 24,
	AARCH64_REG_B13 = 25,
	AARCH64_REG_B14 = 26,
	AARCH64_REG_B15 = 27,
	AARCH64_REG_B16 = 28,
	AARCH64_REG_B17 = 29,
	AARCH64_REG_B18 = 30,
	AARCH64_REG_B19 = 31,
	AARCH64_REG_B20 = 32,
	AARCH64_REG_B21 = 33,
	AARCH64_REG_B22 = 34,
	AARCH64_REG_B23 = 35,
	AARCH64_REG_B24 = 36,
	AARCH64_REG_B25 = 37,
	AARCH64_REG_B26 = 38,
	AARCH64_REG_B27 = 39,
	AARCH64_REG_B28 = 40,
	AARCH64_REG_B29 = 41,
	AARCH64_REG_B30 = 42,
	AARCH64_REG_B31 = 43,
	AARCH64_REG_D0 = 44,
	AARCH64_REG_D1 = 45,
	AARCH64_REG_D2 = 46,
	AARCH64_REG_D3 = 47,
	AARCH64_REG_D4 = 48,
	AARCH64_REG_D5 = 49,
	AARCH64_REG_D6 = 50,
	AARCH64_REG_D7 = 51,
	AARCH64_REG_D8 = 52,
	AARCH64_REG_D9 = 53,
	AARCH64_REG_D10 = 54,
	AARCH64_REG_D11 = 55,
	AARCH64_REG_D12 = 56,
	AARCH64_REG_D13 = 57,
	AARCH64_REG_D14 = 58,
	AARCH64_REG_D15 = 59,
	AARCH64_REG_D16 = 60,
	AARCH64_REG_D17 = 61,
	AARCH64_REG_D18 = 62,
	AARCH64_REG_D19 = 63,
	AARCH64_REG_D20 = 64,
	AARCH64_REG_D21 = 65,
	AARCH64_REG_D22 = 66,
	AARCH64_REG_D23 = 67,
	AARCH64_REG_D24 = 68,
	AARCH64_REG_D25 = 69,
	AARCH64_REG_D26 = 70,
	AARCH64_REG_D27 = 71,
	AARCH64_REG_D28 = 72,
	AARCH64_REG_D29 = 73,
	AARCH64_REG_D30 = 74,
	AARCH64_REG_D31 = 75,
	AARCH64_REG_H0 = 76,
	AARCH64_REG_H1 = 77,
	AARCH64_REG_H2 = 78,
	AARCH64_REG_H3 = 79,
	AARCH64_REG_H4 = 80,
	AARCH64_REG_H5 = 81,
	AARCH64_REG_H6 = 82,
	AARCH64_REG_H7 = 83,
	AARCH64_REG_H8 = 84,
	AARCH64_REG_H9 = 85,
	AARCH64_REG_H10 = 86,
	AARCH64_REG_H11 = 87,
	AARCH64_REG_H12 = 88,
	AARCH64_REG_H13 = 89,
	AARCH64_REG_H14 = 90,
	AARCH64_REG_H15 = 91,
	AARCH64_REG_H16 = 92,
	AARCH64_REG_H17 = 93,
	AARCH64_REG_H18 = 94,
	AARCH64_REG_H19 = 95,
	AARCH64_REG_H20 = 96,
	AARCH64_REG_H21 = 97,
	AARCH64_REG_H22 = 98,
	AARCH64_REG_H23 = 99,
	AARCH64_REG_H24 = 100,
	AARCH64_REG_H25 = 101,
	AARCH64_REG_H26 = 102,
	AARCH64_REG_H27 = 103,
	AARCH64_REG_H28 = 104,
	AARCH64_REG_H29 = 105,
	AARCH64_REG_H30 = 106,
	AARCH64_REG_H31 = 107,
	AARCH64_REG_P0 = 108,
	AARCH64_REG_P1 = 109,
	AARCH64_REG_P2 = 110,
	AARCH64_REG_P3 = 111,
	AARCH64_REG_P4 = 112,
	AARCH64_REG_P5 = 113,
	AARCH64_REG_P6 = 114,
	AARCH64_REG_P7 = 115,
	AARCH64_REG_P8 = 116,
	AARCH64_REG_P9 = 117,
	AARCH64_REG_P10 = 118,
	AARCH64_REG_P11 = 119,
	AARCH64_REG_P12 = 120,
	AARCH64_REG_P13 = 121,
	AARCH64_REG_P14 = 122,
	AARCH64_REG_P15 = 123,
	AARCH64_REG_PN0 = 124,
	AARCH64_REG_PN1 = 125,
	AARCH64_REG_PN2 = 126,
	AARCH64_REG_PN3 = 127,
	AARCH64_REG_PN4 = 128,
	AARCH64_REG_PN5 = 129,
	AARCH64_REG_PN6 = 130,
	AARCH64_REG_PN7 = 131,
	AARCH64_REG_PN8 = 132,
	AARCH64_REG_PN9 = 133,
	AARCH64_REG_PN10 = 134,
	AARCH64_REG_PN11 = 135,
	AARCH64_REG_PN12 = 136,
	AARCH64_REG_PN13 = 137,
	AARCH64_REG_PN14 = 138,
	AARCH64_REG_PN15 = 139,
	AARCH64_REG_Q0 = 140,
	AARCH64_REG_Q1 = 141,
	AARCH64_REG_Q2 = 142,
	AARCH64_REG_Q3 = 143,
	AARCH64_REG_Q4 = 144,
	AARCH64_REG_Q5 = 145,
	AARCH64_REG_Q6 = 146,
	AARCH64_REG_Q7 = 147,
	AARCH64_REG_Q8 = 148,
	AARCH64_REG_Q9 = 149,
	AARCH64_REG_Q10 = 150,
	AARCH64_REG_Q11 = 151,
	AARCH64_REG_Q12 = 152,
	AARCH64_REG_Q13 = 153,
	AARCH64_REG_Q14 = 154,
	AARCH64_REG_Q15 = 155,
	AARCH64_REG_Q16 = 156,
	AARCH64_REG_Q17 = 157,
	AARCH64_REG_Q18 = 158,
	AARCH64_REG_Q19 = 159,
	AARCH64_REG_Q20 = 160,
	AARCH64_REG_Q21 = 161,
	AARCH64_REG_Q22 = 162,
	AARCH64_REG_Q23 = 163,
	AARCH64_REG_Q24 = 164,
	AARCH64_REG_Q25 = 165,
	AARCH64_REG_Q26 = 166,
	AARCH64_REG_Q27 = 167,
	AARCH64_REG_Q28 = 168,
	AARCH64_REG_Q29 = 169,
	AARCH64_REG_Q30 = 170,
	AARCH64_REG_Q31 = 171,
	AARCH64_REG_S0 = 172,
	AARCH64_REG_S1 = 173,
	AARCH64_REG_S2 = 174,
	AARCH64_REG_S3 = 175,
	AARCH64_REG_S4 = 176,
	AARCH64_REG_S5 = 177,
	AARCH64_REG_S6 = 178,
	AARCH64_REG_S7 = 179,
	AARCH64_REG_S8 = 180,
	AARCH64_REG_S9 = 181,
	AARCH64_REG_S10 = 182,
	AARCH64_REG_S11 = 183,
	AARCH64_REG_S12 = 184,
	AARCH64_REG_S13 = 185,
	AARCH64_REG_S14 = 186,
	AARCH64_REG_S15 = 187,
	AARCH64_REG_S16 = 188,
	AARCH64_REG_S17 = 189,
	AARCH64_REG_S18 = 190,
	AARCH64_REG_S19 = 191,
	AARCH64_REG_S20 = 192,
	AARCH64_REG_S21 = 193,
	AARCH64_REG_S22 = 194,
	AARCH64_REG_S23 = 195,
	AARCH64_REG_S24 = 196,
	AARCH64_REG_S25 = 197,
	AARCH64_REG_S26 = 198,
	AARCH64_REG_S27 = 199,
	AARCH64_REG_S28 = 200,
	AARCH64_REG_S29 = 201,
	AARCH64_REG_S30 = 202,
	AARCH64_REG_S31 = 203,
	AARCH64_REG_W0 = 204,
	AARCH64_REG_W1 = 205,
	AARCH64_REG_W2 = 206,
	AARCH64_REG_W3 = 207,
	AARCH64_REG_W4 = 208,
	AARCH64_REG_W5 = 209,
	AARCH64_REG_W6 = 210,
	AARCH64_REG_W7 = 211,
	AARCH64_REG_W8 = 212,
	AARCH64_REG_W9 = 213,
	AARCH64_REG_W10 = 214,
	AARCH64_REG_W11 = 215,
	AARCH64_REG_W12 = 216,
	AARCH64_REG_W13 = 217,
	AARCH64_REG_W14 = 218,
	AARCH64_REG_W15 = 219,
	AARCH64_REG_W16 = 220,
	AARCH64_REG_W17 = 221,
	AARCH64_REG_W18 = 222,
	AARCH64_REG_W19 = 223,
	AARCH64_REG_W20 = 224,
	AARCH64_REG_W21 = 225,
	AARCH64_REG_W22 = 226,
	AARCH64_REG_W23 = 227,
	AARCH64_REG_W24 = 228,
	AARCH64_REG_W25 = 229,
	AARCH64_REG_W26 = 230,
	AARCH64_REG_W27 = 231,
	AARCH64_REG_W28 = 232,
	AARCH64_REG_W29 = 233,
	AARCH64_REG_W30 = 234,
	AARCH64_REG_X0 = 235,
	AARCH64_REG_X1 = 236,
	AARCH64_REG_X2 = 237,
	AARCH64_REG_X3 = 238,
	AARCH64_REG_X4 = 239,
	AARCH64_REG_X5 = 240,
	AARCH64_REG_X6 = 241,
	AARCH64_REG_X7 = 242,
	AARCH64_REG_X8 = 243,
	AARCH64_REG_X9 = 244,
	AARCH64_REG_X10 = 245,
	AARCH64_REG_X11 = 246,
	AARCH64_REG_X12 = 247,
	AARCH64_REG_X13 = 248,
	AARCH64_REG_X14 = 249,
	AARCH64_REG_X15 = 250,
	AARCH64_REG_X16 = 251,
	AARCH64_REG_X17 = 252,
	AARCH64_REG_X18 = 253,
	AARCH64_REG_X19 = 254,
	AARCH64_REG_X20 = 255,
	AARCH64_REG_X21 = 256,
	AARCH64_REG_X22 = 257,
	AARCH64_REG_X23 = 258,
	AARCH64_REG_X24 = 259,
	AARCH64_REG_X25 = 260,
	AARCH64_REG_X26 = 261,
	AARCH64_REG_X27 = 262,
	AARCH64_REG_X28 = 263,
	AARCH64_REG_Z0 = 264,
	AARCH64_REG_Z1 = 265,
	AARCH64_REG_Z2 = 266,
	AARCH64_REG_Z3 = 267,
	AARCH64_REG_Z4 = 268,
	AARCH64_REG_Z5 = 269,
	AARCH64_REG_Z6 = 270,
	AARCH64_REG_Z7 = 271,
	AARCH64_REG_Z8 = 272,
	AARCH64_REG_Z9 = 273,
	AARCH64_REG_Z10 = 274,
	AARCH64_REG_Z11 = 275,
	AARCH64_REG_Z12 = 276,
	AARCH64_REG_Z13 = 277,
	AARCH64_REG_Z14 = 278,
	AARCH64_REG_Z15 = 279,
	AARCH64_REG_Z16 = 280,
	AARCH64_REG_Z17 = 281,
	AARCH64_REG_Z18 = 282,
	AARCH64_REG_Z19 = 283,
	AARCH64_REG_Z20 = 284,
	AARCH64_REG_Z21 = 285,
	AARCH64_REG_Z22 = 286,
	AARCH64_REG_Z23 = 287,
	AARCH64_REG_Z24 = 288,
	AARCH64_REG_Z25 = 289,
	AARCH64_REG_Z26 = 290,
	AARCH64_REG_Z27 = 291,
	AARCH64_REG_Z28 = 292,
	AARCH64_REG_Z29 = 293,
	AARCH64_REG_Z30 = 294,
	AARCH64_REG_Z31 = 295,
	AARCH64_REG_ZAB0 = 296,
	AARCH64_REG_ZAD0 = 297,
	AARCH64_REG_ZAD1 = 298,
	AARCH64_REG_ZAD2 = 299,
	AARCH64_REG_ZAD3 = 300,
	AARCH64_REG_ZAD4 = 301,
	AARCH64_REG_ZAD5 = 302,
	AARCH64_REG_ZAD6 = 303,
	AARCH64_REG_ZAD7 = 304,
	AARCH64_REG_ZAH0 = 305,
	AARCH64_REG_ZAH1 = 306,
	AARCH64_REG_ZAQ0 = 307,
	AARCH64_REG_ZAQ1 = 308,
	AARCH64_REG_ZAQ2 = 309,
	AARCH64_REG_ZAQ3 = 310,
	AARCH64_REG_ZAQ4 = 311,
	AARCH64_REG_ZAQ5 = 312,
	AARCH64_REG_ZAQ6 = 313,
	AARCH64_REG_ZAQ7 = 314,
	AARCH64_REG_ZAQ8 = 315,
	AARCH64_REG_ZAQ9 = 316,
	AARCH64_REG_ZAQ10 = 317,
	AARCH64_REG_ZAQ11 = 318,
	AARCH64_REG_ZAQ12 = 319,
	AARCH64_REG_ZAQ13 = 320,
	AARCH64_REG_ZAQ14 = 321,
	AARCH64_REG_ZAQ15 = 322,
	AARCH64_REG_ZAS0 = 323,
	AARCH64_REG_ZAS1 = 324,
	AARCH64_REG_ZAS2 = 325,
	AARCH64_REG_ZAS3 = 326,
	AARCH64_REG_ZT0 = 327,
	AARCH64_REG_D0_D1 = 328,
	AARCH64_REG_D1_D2 = 329,
	AARCH64_REG_D2_D3 = 330,
	AARCH64_REG_D3_D4 = 331,
	AARCH64_REG_D4_D5 = 332,
	AARCH64_REG_D5_D6 = 333,
	AARCH64_REG_D6_D7 = 334,
	AARCH64_REG_D7_D8 = 335,
	AARCH64_REG_D8_D9 = 336,
	AARCH64_REG_D9_D10 = 337,
	AARCH64_REG_D10_D11 = 338,
	AARCH64_REG_D11_D12 = 339,
	AARCH64_REG_D12_D13 = 340,
	AARCH64_REG_D13_D14 = 341,
	AARCH64_REG_D14_D15 = 342,
	AARCH64_REG_D15_D16 = 343,
	AARCH64_REG_D16_D17 = 344,
	AARCH64_REG_D17_D18 = 345,
	AARCH64_REG_D18_D19 = 346,
	AARCH64_REG_D19_D20 = 347,
	AARCH64_REG_D20_D21 = 348,
	AARCH64_REG_D21_D22 = 349,
	AARCH64_REG_D22_D23 = 350,
	AARCH64_REG_D23_D24 = 351,
	AARCH64_REG_D24_D25 = 352,
	AARCH64_REG_D25_D26 = 353,
	AARCH64_REG_D26_D27 = 354,
	AARCH64_REG_D27_D28 = 355,
	AARCH64_REG_D28_D29 = 356,
	AARCH64_REG_D29_D30 = 357,
	AARCH64_REG_D30_D31 = 358,
	AARCH64_REG_D31_D0 = 359,
	AARCH64_REG_D0_D1_D2_D3 = 360,
	AARCH64_REG_D1_D2_D3_D4 = 361,
	AARCH64_REG_D2_D3_D4_D5 = 362,
	AARCH64_REG_D3_D4_D5_D6 = 363,
	AARCH64_REG_D4_D5_D6_D7 = 364,
	AARCH64_REG_D5_D6_D7_D8 = 365,
	AARCH64_REG_D6_D7_D8_D9 = 366,
	AARCH64_REG_D7_D8_D9_D10 = 367,
	AARCH64_REG_D8_D9_D10_D11 = 368,
	AARCH64_REG_D9_D10_D11_D12 = 369,
	AARCH64_REG_D10_D11_D12_D13 = 370,
	AARCH64_REG_D11_D12_D13_D14 = 371,
	AARCH64_REG_D12_D13_D14_D15 = 372,
	AARCH64_REG_D13_D14_D15_D16 = 373,
	AARCH64_REG_D14_D15_D16_D17 = 374,
	AARCH64_REG_D15_D16_D17_D18 = 375,
	AARCH64_REG_D16_D17_D18_D19 = 376,
	AARCH64_REG_D17_D18_D19_D20 = 377,
	AARCH64_REG_D18_D19_D20_D21 = 378,
	AARCH64_REG_D19_D20_D21_D22 = 379,
	AARCH64_REG_D20_D21_D22_D23 = 380,
	AARCH64_REG_D21_D22_D23_D24 = 381,
	AARCH64_REG_D22_D23_D24_D25 = 382,
	AARCH64_REG_D23_D24_D25_D26 = 383,
	AARCH64_REG_D24_D25_D26_D27 = 384,
	AARCH64_REG_D25_D26_D27_D28 = 385,
	AARCH64_REG_D26_D27_D28_D29 = 386,
	AARCH64_REG_D27_D28_D29_D30 = 387,
	AARCH64_REG_D28_D29_D30_D31 = 388,
	AARCH64_REG_D29_D30_D31_D0 = 389,
	AARCH64_REG_D30_D31_D0_D1 = 390,
	AARCH64_REG_D31_D0_D1_D2 = 391,
	AARCH64_REG_D0_D1_D2 = 392,
	AARCH64_REG_D1_D2_D3 = 393,
	AARCH64_REG_D2_D3_D4 = 394,
	AARCH64_REG_D3_D4_D5 = 395,
	AARCH64_REG_D4_D5_D6 = 396,
	AARCH64_REG_D5_D6_D7 = 397,
	AARCH64_REG_D6_D7_D8 = 398,
	AARCH64_REG_D7_D8_D9 = 399,
	AARCH64_REG_D8_D9_D10 = 400,
	AARCH64_REG_D9_D10_D11 = 401,
	AARCH64_REG_D10_D11_D12 = 402,
	AARCH64_REG_D11_D12_D13 = 403,
	AARCH64_REG_D12_D13_D14 = 404,
	AARCH64_REG_D13_D14_D15 = 405,
	AARCH64_REG_D14_D15_D16 = 406,
	AARCH64_REG_D15_D16_D17 = 407,
	AARCH64_REG_D16_D17_D18 = 408,
	AARCH64_REG_D17_D18_D19 = 409,
	AARCH64_REG_D18_D19_D20 = 410,
	AARCH64_REG_D19_D20_D21 = 411,
	AARCH64_REG_D20_D21_D22 = 412,
	AARCH64_REG_D21_D22_D23 = 413,
	AARCH64_REG_D22_D23_D24 = 414,
	AARCH64_REG_D23_D24_D25 = 415,
	AARCH64_REG_D24_D25_D26 = 416,
	AARCH64_REG_D25_D26_D27 = 417,
	AARCH64_REG_D26_D27_D28 = 418,
	AARCH64_REG_D27_D28_D29 = 419,
	AARCH64_REG_D28_D29_D30 = 420,
	AARCH64_REG_D29_D30_D31 = 421,
	AARCH64_REG_D30_D31_D0 = 422,
	AARCH64_REG_D31_D0_D1 = 423,
	AARCH64_REG_P0_P1 = 424,
	AARCH64_REG_P1_P2 = 425,
	AARCH64_REG_P2_P3 = 426,
	AARCH64_REG_P3_P4 = 427,
	AARCH64_REG_P4_P5 = 428,
	AARCH64_REG_P5_P6 = 429,
	AARCH64_REG_P6_P7 = 430,
	AARCH64_REG_P7_P8 = 431,
	AARCH64_REG_P8_P9 = 432,
	AARCH64_REG_P9_P10 = 433,
	AARCH64_REG_P10_P11 = 434,
	AARCH64_REG_P11_P12 = 435,
	AARCH64_REG_P12_P13 = 436,
	AARCH64_REG_P13_P14 = 437,
	AARCH64_REG_P14_P15 = 438,
	AARCH64_REG_P15_P0 = 439,
	AARCH64_REG_Q0_Q1 = 440,
	AARCH64_REG_Q1_Q2 = 441,
	AARCH64_REG_Q2_Q3 = 442,
	AARCH64_REG_Q3_Q4 = 443,
	AARCH64_REG_Q4_Q5 = 444,
	AARCH64_REG_Q5_Q6 = 445,
	AARCH64_REG_Q6_Q7 = 446,
	AARCH64_REG_Q7_Q8 = 447,
	AARCH64_REG_Q8_Q9 = 448,
	AARCH64_REG_Q9_Q10 = 449,
	AARCH64_REG_Q10_Q11 = 450,
	AARCH64_REG_Q11_Q12 = 451,
	AARCH64_REG_Q12_Q13 = 452,
	AARCH64_REG_Q13_Q14 = 453,
	AARCH64_REG_Q14_Q15 = 454,
	AARCH64_REG_Q15_Q16 = 455,
	AARCH64_REG_Q16_Q17 = 456,
	AARCH64_REG_Q17_Q18 = 457,
	AARCH64_REG_Q18_Q19 = 458,
	AARCH64_REG_Q19_Q20 = 459,
	AARCH64_REG_Q20_Q21 = 460,
	AARCH64_REG_Q21_Q22 = 461,
	AARCH64_REG_Q22_Q23 = 462,
	AARCH64_REG_Q23_Q24 = 463,
	AARCH64_REG_Q24_Q25 = 464,
	AARCH64_REG_Q25_Q26 = 465,
	AARCH64_REG_Q26_Q27 = 466,
	AARCH64_REG_Q27_Q28 = 467,
	AARCH64_REG_Q28_Q29 = 468,
	AARCH64_REG_Q29_Q30 = 469,
	AARCH64_REG_Q30_Q31 = 470,
	AARCH64_REG_Q31_Q0 = 471,
	AARCH64_REG_Q0_Q1_Q2_Q3 = 472,
	AARCH64_REG_Q1_Q2_Q3_Q4 = 473,
	AARCH64_REG_Q2_Q3_Q4_Q5 = 474,
	AARCH64_REG_Q3_Q4_Q5_Q6 = 475,
	AARCH64_REG_Q4_Q5_Q6_Q7 = 476,
	AARCH64_REG_Q5_Q6_Q7_Q8 = 477,
	AARCH64_REG_Q6_Q7_Q8_Q9 = 478,
	AARCH64_REG_Q7_Q8_Q9_Q10 = 479,
	AARCH64_REG_Q8_Q9_Q10_Q11 = 480,
	AARCH64_REG_Q9_Q10_Q11_Q12 = 481,
	AARCH64_REG_Q10_Q11_Q12_Q13 = 482,
	AARCH64_REG_Q11_Q12_Q13_Q14 = 483,
	AARCH64_REG_Q12_Q13_Q14_Q15 = 484,
	AARCH64_REG_Q13_Q14_Q15_Q16 = 485,
	AARCH64_REG_Q14_Q15_Q16_Q17 = 486,
	AARCH64_REG_Q15_Q16_Q17_Q18 = 487,
	AARCH64_REG_Q16_Q17_Q18_Q19 = 488,
	AARCH64_REG_Q17_Q18_Q19_Q20 = 489,
	AARCH64_REG_Q18_Q19_Q20_Q21 = 490,
	AARCH64_REG_Q19_Q20_Q21_Q22 = 491,
	AARCH64_REG_Q20_Q21_Q22_Q23 = 492,
	AARCH64_REG_Q21_Q22_Q23_Q24 = 493,
	AARCH64_REG_Q22_Q23_Q24_Q25 = 494,
	AARCH64_REG_Q23_Q24_Q25_Q26 = 495,
	AARCH64_REG_Q24_Q25_Q26_Q27 = 496,
	AARCH64_REG_Q25_Q26_Q27_Q28 = 497,
	AARCH64_REG_Q26_Q27_Q28_Q29 = 498,
	AARCH64_REG_Q27_Q28_Q29_Q30 = 499,
	AARCH64_REG_Q28_Q29_Q30_Q31 = 500,
	AARCH64_REG_Q29_Q30_Q31_Q0 = 501,
	AARCH64_REG_Q30_Q31_Q0_Q1 = 502,
	AARCH64_REG_Q31_Q0_Q1_Q2 = 503,
	AARCH64_REG_Q0_Q1_Q2 = 504,
	AARCH64_REG_Q1_Q2_Q3 = 505,
	AARCH64_REG_Q2_Q3_Q4 = 506,
	AARCH64_REG_Q3_Q4_Q5 = 507,
	AARCH64_REG_Q4_Q5_Q6 = 508,
	AARCH64_REG_Q5_Q6_Q7 = 509,
	AARCH64_REG_Q6_Q7_Q8 = 510,
	AARCH64_REG_Q7_Q8_Q9 = 511,
	AARCH64_REG_Q8_Q9_Q10 = 512,
	AARCH64_REG_Q9_Q10_Q11 = 513,
	AARCH64_REG_Q10_Q11_Q12 = 514,
	AARCH64_REG_Q11_Q12_Q13 = 515,
	AARCH64_REG_Q12_Q13_Q14 = 516,
	AARCH64_REG_Q13_Q14_Q15 = 517,
	AARCH64_REG_Q14_Q15_Q16 = 518,
	AARCH64_REG_Q15_Q16_Q17 = 519,
	AARCH64_REG_Q16_Q17_Q18 = 520,
	AARCH64_REG_Q17_Q18_Q19 = 521,
	AARCH64_REG_Q18_Q19_Q20 = 522,
	AARCH64_REG_Q19_Q20_Q21 = 523,
	AARCH64_REG_Q20_Q21_Q22 = 524,
	AARCH64_REG_Q21_Q22_Q23 = 525,
	AARCH64_REG_Q22_Q23_Q24 = 526,
	AARCH64_REG_Q23_Q24_Q25 = 527,
	AARCH64_REG_Q24_Q25_Q26 = 528,
	AARCH64_REG_Q25_Q26_Q27 = 529,
	AARCH64_REG_Q26_Q27_Q28 = 530,
	AARCH64_REG_Q27_Q28_Q29 = 531,
	AARCH64_REG_Q28_Q29_Q30 = 532,
	AARCH64_REG_Q29_Q30_Q31 = 533,
	AARCH64_REG_Q30_Q31_Q0 = 534,
	AARCH64_REG_Q31_Q0_Q1 = 535,
	AARCH64_REG_X22_X23_X24_X25_X26_X27_X28_FP = 536,
	AARCH64_REG_X0_X1_X2_X3_X4_X5_X6_X7 = 537,
	AARCH64_REG_X2_X3_X4_X5_X6_X7_X8_X9 = 538,
	AARCH64_REG_X4_X5_X6_X7_X8_X9_X10_X11 = 539,
	AARCH64_REG_X6_X7_X8_X9_X10_X11_X12_X13 = 540,
	AARCH64_REG_X8_X9_X10_X11_X12_X13_X14_X15 = 541,
	AARCH64_REG_X10_X11_X12_X13_X14_X15_X16_X17 = 542,
	AARCH64_REG_X12_X13_X14_X15_X16_X17_X18_X19 = 543,
	AARCH64_REG_X14_X15_X16_X17_X18_X19_X20_X21 = 544,
	AARCH64_REG_X16_X17_X18_X19_X20_X21_X22_X23 = 545,
	AARCH64_REG_X18_X19_X20_X21_X22_X23_X24_X25 = 546,
	AARCH64_REG_X20_X21_X22_X23_X24_X25_X26_X27 = 547,
	AARCH64_REG_W30_WZR = 548,
	AARCH64_REG_W0_W1 = 549,
	AARCH64_REG_W2_W3 = 550,
	AARCH64_REG_W4_W5 = 551,
	AARCH64_REG_W6_W7 = 552,
	AARCH64_REG_W8_W9 = 553,
	AARCH64_REG_W10_W11 = 554,
	AARCH64_REG_W12_W13 = 555,
	AARCH64_REG_W14_W15 = 556,
	AARCH64_REG_W16_W17 = 557,
	AARCH64_REG_W18_W19 = 558,
	AARCH64_REG_W20_W21 = 559,
	AARCH64_REG_W22_W23 = 560,
	AARCH64_REG_W24_W25 = 561,
	AARCH64_REG_W26_W27 = 562,
	AARCH64_REG_W28_W29 = 563,
	AARCH64_REG_LR_XZR = 564,
	AARCH64_REG_X28_FP = 565,
	AARCH64_REG_X0_X1 = 566,
	AARCH64_REG_X2_X3 = 567,
	AARCH64_REG_X4_X5 = 568,
	AARCH64_REG_X6_X7 = 569,
	AARCH64_REG_X8_X9 = 570,
	AARCH64_REG_X10_X11 = 571,
	AARCH64_REG_X12_X13 = 572,
	AARCH64_REG_X14_X15 = 573,
	AARCH64_REG_X16_X17 = 574,
	AARCH64_REG_X18_X19 = 575,
	AARCH64_REG_X20_X21 = 576,
	AARCH64_REG_X22_X23 = 577,
	AARCH64_REG_X24_X25 = 578,
	AARCH64_REG_X26_X27 = 579,
	AARCH64_REG_Z0_Z1 = 580,
	AARCH64_REG_Z1_Z2 = 581,
	AARCH64_REG_Z2_Z3 = 582,
	AARCH64_REG_Z3_Z4 = 583,
	AARCH64_REG_Z4_Z5 = 584,
	AARCH64_REG_Z5_Z6 = 585,
	AARCH64_REG_Z6_Z7 = 586,
	AARCH64_REG_Z7_Z8 = 587,
	AARCH64_REG_Z8_Z9 = 588,
	AARCH64_REG_Z9_Z10 = 589,
	AARCH64_REG_Z10_Z11 = 590,
	AARCH64_REG_Z11_Z12 = 591,
	AARCH64_REG_Z12_Z13 = 592,
	AARCH64_REG_Z13_Z14 = 593,
	AARCH64_REG_Z14_Z15 = 594,
	AARCH64_REG_Z15_Z16 = 595,
	AARCH64_REG_Z16_Z17 = 596,
	AARCH64_REG_Z17_Z18 = 597,
	AARCH64_REG_Z18_Z19 = 598,
	AARCH64_REG_Z19_Z20 = 599,
	AARCH64_REG_Z20_Z21 = 600,
	AARCH64_REG_Z21_Z22 = 601,
	AARCH64_REG_Z22_Z23 = 602,
	AARCH64_REG_Z23_Z24 = 603,
	AARCH64_REG_Z24_Z25 = 604,
	AARCH64_REG_Z25_Z26 = 605,
	AARCH64_REG_Z26_Z27 = 606,
	AARCH64_REG_Z27_Z28 = 607,
	AARCH64_REG_Z28_Z29 = 608,
	AARCH64_REG_Z29_Z30 = 609,
	AARCH64_REG_Z30_Z31 = 610,
	AARCH64_REG_Z31_Z0 = 611,
	AARCH64_REG_Z0_Z1_Z2_Z3 = 612,
	AARCH64_REG_Z1_Z2_Z3_Z4 = 613,
	AARCH64_REG_Z2_Z3_Z4_Z5 = 614,
	AARCH64_REG_Z3_Z4_Z5_Z6 = 615,
	AARCH64_REG_Z4_Z5_Z6_Z7 = 616,
	AARCH64_REG_Z5_Z6_Z7_Z8 = 617,
	AARCH64_REG_Z6_Z7_Z8_Z9 = 618,
	AARCH64_REG_Z7_Z8_Z9_Z10 = 619,
	AARCH64_REG_Z8_Z9_Z10_Z11 = 620,
	AARCH64_REG_Z9_Z10_Z11_Z12 = 621,
	AARCH64_REG_Z10_Z11_Z12_Z13 = 622,
	AARCH64_REG_Z11_Z12_Z13_Z14 = 623,
	AARCH64_REG_Z12_Z13_Z14_Z15 = 624,
	AARCH64_REG_Z13_Z14_Z15_Z16 = 625,
	AARCH64_REG_Z14_Z15_Z16_Z17 = 626,
	AARCH64_REG_Z15_Z16_Z17_Z18 = 627,
	AARCH64_REG_Z16_Z17_Z18_Z19 = 628,
	AARCH64_REG_Z17_Z18_Z19_Z20 = 629,
	AARCH64_REG_Z18_Z19_Z20_Z21 = 630,
	AARCH64_REG_Z19_Z20_Z21_Z22 = 631,
	AARCH64_REG_Z20_Z21_Z22_Z23 = 632,
	AARCH64_REG_Z21_Z22_Z23_Z24 = 633,
	AARCH64_REG_Z22_Z23_Z24_Z25 = 634,
	AARCH64_REG_Z23_Z24_Z25_Z26 = 635,
	AARCH64_REG_Z24_Z25_Z26_Z27 = 636,
	AARCH64_REG_Z25_Z26_Z27_Z28 = 637,
	AARCH64_REG_Z26_Z27_Z28_Z29 = 638,
	AARCH64_REG_Z27_Z28_Z29_Z30 = 639,
	AARCH64_REG_Z28_Z29_Z30_Z31 = 640,
	AARCH64_REG_Z29_Z30_Z31_Z0 = 641,
	AARCH64_REG_Z30_Z31_Z0_Z1 = 642,
	AARCH64_REG_Z31_Z0_Z1_Z2 = 643,
	AARCH64_REG_Z0_Z1_Z2 = 644,
	AARCH64_REG_Z1_Z2_Z3 = 645,
	AARCH64_REG_Z2_Z3_Z4 = 646,
	AARCH64_REG_Z3_Z4_Z5 = 647,
	AARCH64_REG_Z4_Z5_Z6 = 648,
	AARCH64_REG_Z5_Z6_Z7 = 649,
	AARCH64_REG_Z6_Z7_Z8 = 650,
	AARCH64_REG_Z7_Z8_Z9 = 651,
	AARCH64_REG_Z8_Z9_Z10 = 652,
	AARCH64_REG_Z9_Z10_Z11 = 653,
	AARCH64_REG_Z10_Z11_Z12 = 654,
	AARCH64_REG_Z11_Z12_Z13 = 655,
	AARCH64_REG_Z12_Z13_Z14 = 656,
	AARCH64_REG_Z13_Z14_Z15 = 657,
	AARCH64_REG_Z14_Z15_Z16 = 658,
	AARCH64_REG_Z15_Z16_Z17 = 659,
	AARCH64_REG_Z16_Z17_Z18 = 660,
	AARCH64_REG_Z17_Z18_Z19 = 661,
	AARCH64_REG_Z18_Z19_Z20 = 662,
	AARCH64_REG_Z19_Z20_Z21 = 663,
	AARCH64_REG_Z20_Z21_Z22 = 664,
	AARCH64_REG_Z21_Z22_Z23 = 665,
	AARCH64_REG_Z22_Z23_Z24 = 666,
	AARCH64_REG_Z23_Z24_Z25 = 667,
	AARCH64_REG_Z24_Z25_Z26 = 668,
	AARCH64_REG_Z25_Z26_Z27 = 669,
	AARCH64_REG_Z26_Z27_Z28 = 670,
	AARCH64_REG_Z27_Z28_Z29 = 671,
	AARCH64_REG_Z28_Z29_Z30 = 672,
	AARCH64_REG_Z29_Z30_Z31 = 673,
	AARCH64_REG_Z30_Z31_Z0 = 674,
	AARCH64_REG_Z31_Z0_Z1 = 675,
	AARCH64_REG_Z16_Z24 = 676,
	AARCH64_REG_Z17_Z25 = 677,
	AARCH64_REG_Z18_Z26 = 678,
	AARCH64_REG_Z19_Z27 = 679,
	AARCH64_REG_Z20_Z28 = 680,
	AARCH64_REG_Z21_Z29 = 681,
	AARCH64_REG_Z22_Z30 = 682,
	AARCH64_REG_Z23_Z31 = 683,
	AARCH64_REG_Z0_Z8 = 684,
	AARCH64_REG_Z1_Z9 = 685,
	AARCH64_REG_Z2_Z10 = 686,
	AARCH64_REG_Z3_Z11 = 687,
	AARCH64_REG_Z4_Z12 = 688,
	AARCH64_REG_Z5_Z13 = 689,
	AARCH64_REG_Z6_Z14 = 690,
	AARCH64_REG_Z7_Z15 = 691,
	AARCH64_REG_Z16_Z20_Z24_Z28 = 692,
	AARCH64_REG_Z17_Z21_Z25_Z29 = 693,
	AARCH64_REG_Z18_Z22_Z26_Z30 = 694,
	AARCH64_REG_Z19_Z23_Z27_Z31 = 695,
	AARCH64_REG_Z0_Z4_Z8_Z12 = 696,
	AARCH64_REG_Z1_Z5_Z9_Z13 = 697,
	AARCH64_REG_Z2_Z6_Z10_Z14 = 698,
	AARCH64_REG_Z3_Z7_Z11_Z15 = 699,
	AARCH64_REG_ENDING, // 700

	// clang-format on
	// generated content <AArch64GenCSRegEnum.inc> end

  // alias registers
  AARCH64_REG_IP0 = AARCH64_REG_X16,
  AARCH64_REG_IP1 = AARCH64_REG_X17,
  AARCH64_REG_X29 = AARCH64_REG_FP,
  AARCH64_REG_X30 = AARCH64_REG_LR,
} aarch64_reg;

/// Instruction's operand referring to memory
typedef struct aarch64_op_mem {
  aarch64_reg base;  ///< base register
  aarch64_reg index; ///< index register
  int32_t disp;	   ///< displacement/offset value
} aarch64_op_mem;

typedef enum {
	AARCH64_SME_OP_INVALID,
	AARCH64_SME_OP_TILE, ///< SME operand is a single tile.
	AARCH64_SME_OP_TILE_VEC, ///< SME operand is a tile indexed by a register and/or immediate
} aarch64_sme_op_type;

#define AARCH64_SLICE_IMM_INVALID UINT16_MAX
#define AARCH64_SLICE_IMM_RANGE_INVALID UINT8_MAX

typedef struct {
	uint8_t first;
	uint8_t offset;
} aarch64_imm_range;

/// SME Instruction's matrix operand
typedef struct {
  aarch64_sme_op_type type; ///< AArch64_SME_OP_TILE, AArch64_SME_OP_TILE_VEC
  aarch64_reg tile; ///< Matrix tile register
  aarch64_reg slice_reg; ///< slice index reg
	union {
		uint16_t imm; ///< Invalid if equal to AARCH64_SLICE_IMM_INVALID
		aarch64_imm_range imm_range; ///< Members are set to AARCH64_SLICE_IMM_RANGE_INVALID if invalid.
	} slice_offset; ///< slice index offset.
	bool has_range_offset; ///< If true, the offset is a range.
  bool is_vertical;	///< Flag if slice is vertical or horizontal
} aarch64_op_sme;

/// SME Instruction's operand has index
typedef struct {
  aarch64_reg reg; ///< Vector predicate register
  aarch64_reg vec_select; ///< Vector select register.
  int32_t imm_index; ///< Index in range 0 to one less of vector elements in a 128bit reg.
} aarch64_op_pred;

/// Instruction operand
typedef struct cs_aarch64_op {
  int vector_index; ///< Vector Index for some vector operands (or -1 if
		    ///< irrelevant)
  AArch64Layout_VectorLayout vas;    ///< Vector Arrangement Specifier
  struct {
    aarch64_shifter type; ///< shifter type of this operand
    /// Shift value of this operand.
    /// If the type indicates a shift with a register this value should be
    /// interpreted as aarch64_reg.
    unsigned int value;
  } shift;
  aarch64_extender ext; ///< extender type of this operand
  aarch64_op_type type; ///< operand type
	/// Q and V regs share the same identifiers (because they are the same registers).
	/// If this flag is set, the register operand is interpreted as a V-register.
  bool is_vreg;
  union {
    aarch64_reg reg;	 ///< register value for REG operand
    int64_t imm;	 ///< immediate value, or index for C-IMM or IMM operand
		aarch64_imm_range imm_range; ///< An immediate range
    double fp;		 ///< floating point value for FP operand
    aarch64_op_mem mem;	 ///< base/index/scale/disp value for MEM operand
    aarch64_op_sme sme; ///< SME matrix operand
    aarch64_op_pred pred; ///< Predicate register
  };
  aarch64_sysop sysop; ///< System operand

  /// How is this operand accessed? (READ, WRITE or READ|WRITE)
  /// This field is combined of cs_ac_type.
  /// NOTE: this field is irrelevant if engine is compiled in DIET mode.
  uint8_t access;
  bool is_list_member; ///< True if this operand is part of a register or vector list.
} cs_aarch64_op;

typedef struct {
	cs_ac_type mem_acc; ///< CGI memory access according to mayLoad and mayStore
} aarch64_suppl_info;

#define NUM_AARCH64_OPS 16

/// Instruction structure
typedef struct cs_aarch64 {
  AArch64CC_CondCode cc;	     ///< conditional code for this insn
  bool update_flags; ///< does this insn update flags?
  bool post_index;   ///< only set if writeback is 'True', if 'False' pre-index, otherwise post.
  bool is_doing_sme; ///< True if a SME or SVE operand is currently edited.

  /// Number of operands of this instruction,
  /// or 0 when instruction has no operand.
  uint8_t op_count;

  cs_aarch64_op operands[NUM_AARCH64_OPS]; ///< operands for this instruction.
} cs_aarch64;

/// AArch64 instruction
typedef enum aarch64_insn {
	// generated content <AArch64GenCSInsnEnum.inc> begin
	// clang-format off

	AARCH64_INS_INVALID,
	AARCH64_INS_ABS,
	AARCH64_INS_ADCLB,
	AARCH64_INS_ADCLT,
	AARCH64_INS_ADCS,
	AARCH64_INS_ADC,
	AARCH64_INS_ADDG,
	AARCH64_INS_ADDHA,
	AARCH64_INS_ADDHNB,
	AARCH64_INS_ADDHNT,
	AARCH64_INS_ADDHN,
	AARCH64_INS_ADDHN2,
	AARCH64_INS_ADDPL,
	AARCH64_INS_ADDPT,
	AARCH64_INS_ADDP,
	AARCH64_INS_ADDQV,
	AARCH64_INS_ADDSPL,
	AARCH64_INS_ADDSVL,
	AARCH64_INS_ADDS,
	AARCH64_INS_ADDVA,
	AARCH64_INS_ADDVL,
	AARCH64_INS_ADDV,
	AARCH64_INS_ADD,
	AARCH64_INS_ADR,
	AARCH64_INS_ADRP,
	AARCH64_INS_AESD,
	AARCH64_INS_AESE,
	AARCH64_INS_AESIMC,
	AARCH64_INS_AESMC,
	AARCH64_INS_ANDQV,
	AARCH64_INS_ANDS,
	AARCH64_INS_ANDV,
	AARCH64_INS_AND,
	AARCH64_INS_ASRD,
	AARCH64_INS_ASRR,
	AARCH64_INS_ASR,
	AARCH64_INS_AUTDA,
	AARCH64_INS_AUTDB,
	AARCH64_INS_AUTDZA,
	AARCH64_INS_AUTDZB,
	AARCH64_INS_AUTIA,
	AARCH64_INS_HINT,
	AARCH64_INS_AUTIA171615,
	AARCH64_INS_AUTIASPPC,
	AARCH64_INS_AUTIB,
	AARCH64_INS_AUTIB171615,
	AARCH64_INS_AUTIBSPPC,
	AARCH64_INS_AUTIZA,
	AARCH64_INS_AUTIZB,
	AARCH64_INS_AXFLAG,
	AARCH64_INS_B,
	AARCH64_INS_BCAX,
	AARCH64_INS_BC,
	AARCH64_INS_BDEP,
	AARCH64_INS_BEXT,
	AARCH64_INS_BFDOT,
	AARCH64_INS_BF1CVTL2,
	AARCH64_INS_BF1CVTLT,
	AARCH64_INS_BF1CVTL,
	AARCH64_INS_BF1CVT,
	AARCH64_INS_BF2CVTL2,
	AARCH64_INS_BF2CVTLT,
	AARCH64_INS_BF2CVTL,
	AARCH64_INS_BF2CVT,
	AARCH64_INS_BFADD,
	AARCH64_INS_BFCLAMP,
	AARCH64_INS_BFCVT,
	AARCH64_INS_BFCVTN,
	AARCH64_INS_BFCVTN2,
	AARCH64_INS_BFCVTNT,
	AARCH64_INS_BFMAXNM,
	AARCH64_INS_BFMAX,
	AARCH64_INS_BFMINNM,
	AARCH64_INS_BFMIN,
	AARCH64_INS_BFMLALB,
	AARCH64_INS_BFMLALT,
	AARCH64_INS_BFMLAL,
	AARCH64_INS_BFMLA,
	AARCH64_INS_BFMLSLB,
	AARCH64_INS_BFMLSLT,
	AARCH64_INS_BFMLSL,
	AARCH64_INS_BFMLS,
	AARCH64_INS_BFMMLA,
	AARCH64_INS_BFMOPA,
	AARCH64_INS_BFMOPS,
	AARCH64_INS_BFMUL,
	AARCH64_INS_BFM,
	AARCH64_INS_BFSUB,
	AARCH64_INS_BFVDOT,
	AARCH64_INS_BGRP,
	AARCH64_INS_BICS,
	AARCH64_INS_BIC,
	AARCH64_INS_BIF,
	AARCH64_INS_BIT,
	AARCH64_INS_BL,
	AARCH64_INS_BLR,
	AARCH64_INS_BLRAA,
	AARCH64_INS_BLRAAZ,
	AARCH64_INS_BLRAB,
	AARCH64_INS_BLRABZ,
	AARCH64_INS_BMOPA,
	AARCH64_INS_BMOPS,
	AARCH64_INS_BR,
	AARCH64_INS_BRAA,
	AARCH64_INS_BRAAZ,
	AARCH64_INS_BRAB,
	AARCH64_INS_BRABZ,
	AARCH64_INS_BRB,
	AARCH64_INS_BRK,
	AARCH64_INS_BRKAS,
	AARCH64_INS_BRKA,
	AARCH64_INS_BRKBS,
	AARCH64_INS_BRKB,
	AARCH64_INS_BRKNS,
	AARCH64_INS_BRKN,
	AARCH64_INS_BRKPAS,
	AARCH64_INS_BRKPA,
	AARCH64_INS_BRKPBS,
	AARCH64_INS_BRKPB,
	AARCH64_INS_BSL1N,
	AARCH64_INS_BSL2N,
	AARCH64_INS_BSL,
	AARCH64_INS_CADD,
	AARCH64_INS_CASAB,
	AARCH64_INS_CASAH,
	AARCH64_INS_CASALB,
	AARCH64_INS_CASALH,
	AARCH64_INS_CASAL,
	AARCH64_INS_CASA,
	AARCH64_INS_CASB,
	AARCH64_INS_CASH,
	AARCH64_INS_CASLB,
	AARCH64_INS_CASLH,
	AARCH64_INS_CASL,
	AARCH64_INS_CASPAL,
	AARCH64_INS_CASPA,
	AARCH64_INS_CASPL,
	AARCH64_INS_CASP,
	AARCH64_INS_CAS,
	AARCH64_INS_CBNZ,
	AARCH64_INS_CBZ,
	AARCH64_INS_CCMN,
	AARCH64_INS_CCMP,
	AARCH64_INS_CDOT,
	AARCH64_INS_CFINV,
	AARCH64_INS_CLASTA,
	AARCH64_INS_CLASTB,
	AARCH64_INS_CLREX,
	AARCH64_INS_CLS,
	AARCH64_INS_CLZ,
	AARCH64_INS_CMEQ,
	AARCH64_INS_CMGE,
	AARCH64_INS_CMGT,
	AARCH64_INS_CMHI,
	AARCH64_INS_CMHS,
	AARCH64_INS_CMLA,
	AARCH64_INS_CMLE,
	AARCH64_INS_CMLT,
	AARCH64_INS_CMPEQ,
	AARCH64_INS_CMPGE,
	AARCH64_INS_CMPGT,
	AARCH64_INS_CMPHI,
	AARCH64_INS_CMPHS,
	AARCH64_INS_CMPLE,
	AARCH64_INS_CMPLO,
	AARCH64_INS_CMPLS,
	AARCH64_INS_CMPLT,
	AARCH64_INS_CMPNE,
	AARCH64_INS_CMTST,
	AARCH64_INS_CNOT,
	AARCH64_INS_CNTB,
	AARCH64_INS_CNTD,
	AARCH64_INS_CNTH,
	AARCH64_INS_CNTP,
	AARCH64_INS_CNTW,
	AARCH64_INS_CNT,
	AARCH64_INS_COMPACT,
	AARCH64_INS_CPYE,
	AARCH64_INS_CPYEN,
	AARCH64_INS_CPYERN,
	AARCH64_INS_CPYERT,
	AARCH64_INS_CPYERTN,
	AARCH64_INS_CPYERTRN,
	AARCH64_INS_CPYERTWN,
	AARCH64_INS_CPYET,
	AARCH64_INS_CPYETN,
	AARCH64_INS_CPYETRN,
	AARCH64_INS_CPYETWN,
	AARCH64_INS_CPYEWN,
	AARCH64_INS_CPYEWT,
	AARCH64_INS_CPYEWTN,
	AARCH64_INS_CPYEWTRN,
	AARCH64_INS_CPYEWTWN,
	AARCH64_INS_CPYFE,
	AARCH64_INS_CPYFEN,
	AARCH64_INS_CPYFERN,
	AARCH64_INS_CPYFERT,
	AARCH64_INS_CPYFERTN,
	AARCH64_INS_CPYFERTRN,
	AARCH64_INS_CPYFERTWN,
	AARCH64_INS_CPYFET,
	AARCH64_INS_CPYFETN,
	AARCH64_INS_CPYFETRN,
	AARCH64_INS_CPYFETWN,
	AARCH64_INS_CPYFEWN,
	AARCH64_INS_CPYFEWT,
	AARCH64_INS_CPYFEWTN,
	AARCH64_INS_CPYFEWTRN,
	AARCH64_INS_CPYFEWTWN,
	AARCH64_INS_CPYFM,
	AARCH64_INS_CPYFMN,
	AARCH64_INS_CPYFMRN,
	AARCH64_INS_CPYFMRT,
	AARCH64_INS_CPYFMRTN,
	AARCH64_INS_CPYFMRTRN,
	AARCH64_INS_CPYFMRTWN,
	AARCH64_INS_CPYFMT,
	AARCH64_INS_CPYFMTN,
	AARCH64_INS_CPYFMTRN,
	AARCH64_INS_CPYFMTWN,
	AARCH64_INS_CPYFMWN,
	AARCH64_INS_CPYFMWT,
	AARCH64_INS_CPYFMWTN,
	AARCH64_INS_CPYFMWTRN,
	AARCH64_INS_CPYFMWTWN,
	AARCH64_INS_CPYFP,
	AARCH64_INS_CPYFPN,
	AARCH64_INS_CPYFPRN,
	AARCH64_INS_CPYFPRT,
	AARCH64_INS_CPYFPRTN,
	AARCH64_INS_CPYFPRTRN,
	AARCH64_INS_CPYFPRTWN,
	AARCH64_INS_CPYFPT,
	AARCH64_INS_CPYFPTN,
	AARCH64_INS_CPYFPTRN,
	AARCH64_INS_CPYFPTWN,
	AARCH64_INS_CPYFPWN,
	AARCH64_INS_CPYFPWT,
	AARCH64_INS_CPYFPWTN,
	AARCH64_INS_CPYFPWTRN,
	AARCH64_INS_CPYFPWTWN,
	AARCH64_INS_CPYM,
	AARCH64_INS_CPYMN,
	AARCH64_INS_CPYMRN,
	AARCH64_INS_CPYMRT,
	AARCH64_INS_CPYMRTN,
	AARCH64_INS_CPYMRTRN,
	AARCH64_INS_CPYMRTWN,
	AARCH64_INS_CPYMT,
	AARCH64_INS_CPYMTN,
	AARCH64_INS_CPYMTRN,
	AARCH64_INS_CPYMTWN,
	AARCH64_INS_CPYMWN,
	AARCH64_INS_CPYMWT,
	AARCH64_INS_CPYMWTN,
	AARCH64_INS_CPYMWTRN,
	AARCH64_INS_CPYMWTWN,
	AARCH64_INS_CPYP,
	AARCH64_INS_CPYPN,
	AARCH64_INS_CPYPRN,
	AARCH64_INS_CPYPRT,
	AARCH64_INS_CPYPRTN,
	AARCH64_INS_CPYPRTRN,
	AARCH64_INS_CPYPRTWN,
	AARCH64_INS_CPYPT,
	AARCH64_INS_CPYPTN,
	AARCH64_INS_CPYPTRN,
	AARCH64_INS_CPYPTWN,
	AARCH64_INS_CPYPWN,
	AARCH64_INS_CPYPWT,
	AARCH64_INS_CPYPWTN,
	AARCH64_INS_CPYPWTRN,
	AARCH64_INS_CPYPWTWN,
	AARCH64_INS_CPY,
	AARCH64_INS_CRC32B,
	AARCH64_INS_CRC32CB,
	AARCH64_INS_CRC32CH,
	AARCH64_INS_CRC32CW,
	AARCH64_INS_CRC32CX,
	AARCH64_INS_CRC32H,
	AARCH64_INS_CRC32W,
	AARCH64_INS_CRC32X,
	AARCH64_INS_CSEL,
	AARCH64_INS_CSINC,
	AARCH64_INS_CSINV,
	AARCH64_INS_CSNEG,
	AARCH64_INS_CTERMEQ,
	AARCH64_INS_CTERMNE,
	AARCH64_INS_CTZ,
	AARCH64_INS_DCPS1,
	AARCH64_INS_DCPS2,
	AARCH64_INS_DCPS3,
	AARCH64_INS_DECB,
	AARCH64_INS_DECD,
	AARCH64_INS_DECH,
	AARCH64_INS_DECP,
	AARCH64_INS_DECW,
	AARCH64_INS_DMB,
	AARCH64_INS_DRPS,
	AARCH64_INS_DSB,
	AARCH64_INS_DUPM,
	AARCH64_INS_DUPQ,
	AARCH64_INS_DUP,
	AARCH64_INS_MOV,
	AARCH64_INS_EON,
	AARCH64_INS_EOR3,
	AARCH64_INS_EORBT,
	AARCH64_INS_EORQV,
	AARCH64_INS_EORS,
	AARCH64_INS_EORTB,
	AARCH64_INS_EORV,
	AARCH64_INS_EOR,
	AARCH64_INS_ERET,
	AARCH64_INS_ERETAA,
	AARCH64_INS_ERETAB,
	AARCH64_INS_EXTQ,
	AARCH64_INS_MOVA,
	AARCH64_INS_EXTR,
	AARCH64_INS_EXT,
	AARCH64_INS_F1CVTL2,
	AARCH64_INS_F1CVTLT,
	AARCH64_INS_F1CVTL,
	AARCH64_INS_F1CVT,
	AARCH64_INS_F2CVTL2,
	AARCH64_INS_F2CVTLT,
	AARCH64_INS_F2CVTL,
	AARCH64_INS_F2CVT,
	AARCH64_INS_FABD,
	AARCH64_INS_FABS,
	AARCH64_INS_FACGE,
	AARCH64_INS_FACGT,
	AARCH64_INS_FADDA,
	AARCH64_INS_FADD,
	AARCH64_INS_FADDP,
	AARCH64_INS_FADDQV,
	AARCH64_INS_FADDV,
	AARCH64_INS_FAMAX,
	AARCH64_INS_FAMIN,
	AARCH64_INS_FCADD,
	AARCH64_INS_FCCMP,
	AARCH64_INS_FCCMPE,
	AARCH64_INS_FCLAMP,
	AARCH64_INS_FCMEQ,
	AARCH64_INS_FCMGE,
	AARCH64_INS_FCMGT,
	AARCH64_INS_FCMLA,
	AARCH64_INS_FCMLE,
	AARCH64_INS_FCMLT,
	AARCH64_INS_FCMNE,
	AARCH64_INS_FCMP,
	AARCH64_INS_FCMPE,
	AARCH64_INS_FCMUO,
	AARCH64_INS_FCPY,
	AARCH64_INS_FCSEL,
	AARCH64_INS_FCVTAS,
	AARCH64_INS_FCVTAU,
	AARCH64_INS_FCVT,
	AARCH64_INS_FCVTLT,
	AARCH64_INS_FCVTL,
	AARCH64_INS_FCVTL2,
	AARCH64_INS_FCVTMS,
	AARCH64_INS_FCVTMU,
	AARCH64_INS_FCVTNB,
	AARCH64_INS_FCVTNS,
	AARCH64_INS_FCVTNT,
	AARCH64_INS_FCVTNU,
	AARCH64_INS_FCVTN,
	AARCH64_INS_FCVTN2,
	AARCH64_INS_FCVTPS,
	AARCH64_INS_FCVTPU,
	AARCH64_INS_FCVTXNT,
	AARCH64_INS_FCVTXN,
	AARCH64_INS_FCVTXN2,
	AARCH64_INS_FCVTX,
	AARCH64_INS_FCVTZS,
	AARCH64_INS_FCVTZU,
	AARCH64_INS_FDIV,
	AARCH64_INS_FDIVR,
	AARCH64_INS_FDOT,
	AARCH64_INS_FDUP,
	AARCH64_INS_FEXPA,
	AARCH64_INS_FJCVTZS,
	AARCH64_INS_FLOGB,
	AARCH64_INS_FMADD,
	AARCH64_INS_FMAD,
	AARCH64_INS_FMAX,
	AARCH64_INS_FMAXNM,
	AARCH64_INS_FMAXNMP,
	AARCH64_INS_FMAXNMQV,
	AARCH64_INS_FMAXNMV,
	AARCH64_INS_FMAXP,
	AARCH64_INS_FMAXQV,
	AARCH64_INS_FMAXV,
	AARCH64_INS_FMIN,
	AARCH64_INS_FMINNM,
	AARCH64_INS_FMINNMP,
	AARCH64_INS_FMINNMQV,
	AARCH64_INS_FMINNMV,
	AARCH64_INS_FMINP,
	AARCH64_INS_FMINQV,
	AARCH64_INS_FMINV,
	AARCH64_INS_FMLAL2,
	AARCH64_INS_FMLALB,
	AARCH64_INS_FMLALLBB,
	AARCH64_INS_FMLALLBT,
	AARCH64_INS_FMLALLTB,
	AARCH64_INS_FMLALLTT,
	AARCH64_INS_FMLALL,
	AARCH64_INS_FMLALT,
	AARCH64_INS_FMLAL,
	AARCH64_INS_FMLA,
	AARCH64_INS_FMLSL2,
	AARCH64_INS_FMLSLB,
	AARCH64_INS_FMLSLT,
	AARCH64_INS_FMLSL,
	AARCH64_INS_FMLS,
	AARCH64_INS_FMMLA,
	AARCH64_INS_FMOPA,
	AARCH64_INS_FMOPS,
	AARCH64_INS_FMOV,
	AARCH64_INS_FMSB,
	AARCH64_INS_FMSUB,
	AARCH64_INS_FMUL,
	AARCH64_INS_FMULX,
	AARCH64_INS_FNEG,
	AARCH64_INS_FNMADD,
	AARCH64_INS_FNMAD,
	AARCH64_INS_FNMLA,
	AARCH64_INS_FNMLS,
	AARCH64_INS_FNMSB,
	AARCH64_INS_FNMSUB,
	AARCH64_INS_FNMUL,
	AARCH64_INS_FRECPE,
	AARCH64_INS_FRECPS,
	AARCH64_INS_FRECPX,
	AARCH64_INS_FRINT32X,
	AARCH64_INS_FRINT32Z,
	AARCH64_INS_FRINT64X,
	AARCH64_INS_FRINT64Z,
	AARCH64_INS_FRINTA,
	AARCH64_INS_FRINTI,
	AARCH64_INS_FRINTM,
	AARCH64_INS_FRINTN,
	AARCH64_INS_FRINTP,
	AARCH64_INS_FRINTX,
	AARCH64_INS_FRINTZ,
	AARCH64_INS_FRSQRTE,
	AARCH64_INS_FRSQRTS,
	AARCH64_INS_FSCALE,
	AARCH64_INS_FSQRT,
	AARCH64_INS_FSUB,
	AARCH64_INS_FSUBR,
	AARCH64_INS_FTMAD,
	AARCH64_INS_FTSMUL,
	AARCH64_INS_FTSSEL,
	AARCH64_INS_FVDOTB,
	AARCH64_INS_FVDOTT,
	AARCH64_INS_FVDOT,
	AARCH64_INS_GCSPOPCX,
	AARCH64_INS_GCSPOPM,
	AARCH64_INS_GCSPOPX,
	AARCH64_INS_GCSPUSHM,
	AARCH64_INS_GCSPUSHX,
	AARCH64_INS_GCSSS1,
	AARCH64_INS_GCSSS2,
	AARCH64_INS_GCSSTR,
	AARCH64_INS_GCSSTTR,
	AARCH64_INS_LD1B,
	AARCH64_INS_LD1D,
	AARCH64_INS_LD1H,
	AARCH64_INS_LD1Q,
	AARCH64_INS_LD1SB,
	AARCH64_INS_LD1SH,
	AARCH64_INS_LD1SW,
	AARCH64_INS_LD1W,
	AARCH64_INS_LDFF1B,
	AARCH64_INS_LDFF1D,
	AARCH64_INS_LDFF1H,
	AARCH64_INS_LDFF1SB,
	AARCH64_INS_LDFF1SH,
	AARCH64_INS_LDFF1SW,
	AARCH64_INS_LDFF1W,
	AARCH64_INS_GMI,
	AARCH64_INS_HISTCNT,
	AARCH64_INS_HISTSEG,
	AARCH64_INS_HLT,
	AARCH64_INS_HVC,
	AARCH64_INS_INCB,
	AARCH64_INS_INCD,
	AARCH64_INS_INCH,
	AARCH64_INS_INCP,
	AARCH64_INS_INCW,
	AARCH64_INS_INDEX,
	AARCH64_INS_INSR,
	AARCH64_INS_INS,
	AARCH64_INS_IRG,
	AARCH64_INS_ISB,
	AARCH64_INS_LASTA,
	AARCH64_INS_LASTB,
	AARCH64_INS_LD1,
	AARCH64_INS_LD1RB,
	AARCH64_INS_LD1RD,
	AARCH64_INS_LD1RH,
	AARCH64_INS_LD1ROB,
	AARCH64_INS_LD1ROD,
	AARCH64_INS_LD1ROH,
	AARCH64_INS_LD1ROW,
	AARCH64_INS_LD1RQB,
	AARCH64_INS_LD1RQD,
	AARCH64_INS_LD1RQH,
	AARCH64_INS_LD1RQW,
	AARCH64_INS_LD1RSB,
	AARCH64_INS_LD1RSH,
	AARCH64_INS_LD1RSW,
	AARCH64_INS_LD1RW,
	AARCH64_INS_LD1R,
	AARCH64_INS_LD2B,
	AARCH64_INS_LD2D,
	AARCH64_INS_LD2H,
	AARCH64_INS_LD2Q,
	AARCH64_INS_LD2R,
	AARCH64_INS_LD2,
	AARCH64_INS_LD2W,
	AARCH64_INS_LD3B,
	AARCH64_INS_LD3D,
	AARCH64_INS_LD3H,
	AARCH64_INS_LD3Q,
	AARCH64_INS_LD3R,
	AARCH64_INS_LD3,
	AARCH64_INS_LD3W,
	AARCH64_INS_LD4B,
	AARCH64_INS_LD4D,
	AARCH64_INS_LD4,
	AARCH64_INS_LD4H,
	AARCH64_INS_LD4Q,
	AARCH64_INS_LD4R,
	AARCH64_INS_LD4W,
	AARCH64_INS_LD64B,
	AARCH64_INS_LDADDAB,
	AARCH64_INS_LDADDAH,
	AARCH64_INS_LDADDALB,
	AARCH64_INS_LDADDALH,
	AARCH64_INS_LDADDAL,
	AARCH64_INS_LDADDA,
	AARCH64_INS_LDADDB,
	AARCH64_INS_LDADDH,
	AARCH64_INS_LDADDLB,
	AARCH64_INS_LDADDLH,
	AARCH64_INS_LDADDL,
	AARCH64_INS_LDADD,
	AARCH64_INS_LDAP1,
	AARCH64_INS_LDAPRB,
	AARCH64_INS_LDAPRH,
	AARCH64_INS_LDAPR,
	AARCH64_INS_LDAPURB,
	AARCH64_INS_LDAPURH,
	AARCH64_INS_LDAPURSB,
	AARCH64_INS_LDAPURSH,
	AARCH64_INS_LDAPURSW,
	AARCH64_INS_LDAPUR,
	AARCH64_INS_LDARB,
	AARCH64_INS_LDARH,
	AARCH64_INS_LDAR,
	AARCH64_INS_LDAXP,
	AARCH64_INS_LDAXRB,
	AARCH64_INS_LDAXRH,
	AARCH64_INS_LDAXR,
	AARCH64_INS_LDCLRAB,
	AARCH64_INS_LDCLRAH,
	AARCH64_INS_LDCLRALB,
	AARCH64_INS_LDCLRALH,
	AARCH64_INS_LDCLRAL,
	AARCH64_INS_LDCLRA,
	AARCH64_INS_LDCLRB,
	AARCH64_INS_LDCLRH,
	AARCH64_INS_LDCLRLB,
	AARCH64_INS_LDCLRLH,
	AARCH64_INS_LDCLRL,
	AARCH64_INS_LDCLRP,
	AARCH64_INS_LDCLRPA,
	AARCH64_INS_LDCLRPAL,
	AARCH64_INS_LDCLRPL,
	AARCH64_INS_LDCLR,
	AARCH64_INS_LDEORAB,
	AARCH64_INS_LDEORAH,
	AARCH64_INS_LDEORALB,
	AARCH64_INS_LDEORALH,
	AARCH64_INS_LDEORAL,
	AARCH64_INS_LDEORA,
	AARCH64_INS_LDEORB,
	AARCH64_INS_LDEORH,
	AARCH64_INS_LDEORLB,
	AARCH64_INS_LDEORLH,
	AARCH64_INS_LDEORL,
	AARCH64_INS_LDEOR,
	AARCH64_INS_LDG,
	AARCH64_INS_LDGM,
	AARCH64_INS_LDIAPP,
	AARCH64_INS_LDLARB,
	AARCH64_INS_LDLARH,
	AARCH64_INS_LDLAR,
	AARCH64_INS_LDNF1B,
	AARCH64_INS_LDNF1D,
	AARCH64_INS_LDNF1H,
	AARCH64_INS_LDNF1SB,
	AARCH64_INS_LDNF1SH,
	AARCH64_INS_LDNF1SW,
	AARCH64_INS_LDNF1W,
	AARCH64_INS_LDNP,
	AARCH64_INS_LDNT1B,
	AARCH64_INS_LDNT1D,
	AARCH64_INS_LDNT1H,
	AARCH64_INS_LDNT1SB,
	AARCH64_INS_LDNT1SH,
	AARCH64_INS_LDNT1SW,
	AARCH64_INS_LDNT1W,
	AARCH64_INS_LDP,
	AARCH64_INS_LDPSW,
	AARCH64_INS_LDRAA,
	AARCH64_INS_LDRAB,
	AARCH64_INS_LDRB,
	AARCH64_INS_LDR,
	AARCH64_INS_LDRH,
	AARCH64_INS_LDRSB,
	AARCH64_INS_LDRSH,
	AARCH64_INS_LDRSW,
	AARCH64_INS_LDSETAB,
	AARCH64_INS_LDSETAH,
	AARCH64_INS_LDSETALB,
	AARCH64_INS_LDSETALH,
	AARCH64_INS_LDSETAL,
	AARCH64_INS_LDSETA,
	AARCH64_INS_LDSETB,
	AARCH64_INS_LDSETH,
	AARCH64_INS_LDSETLB,
	AARCH64_INS_LDSETLH,
	AARCH64_INS_LDSETL,
	AARCH64_INS_LDSETP,
	AARCH64_INS_LDSETPA,
	AARCH64_INS_LDSETPAL,
	AARCH64_INS_LDSETPL,
	AARCH64_INS_LDSET,
	AARCH64_INS_LDSMAXAB,
	AARCH64_INS_LDSMAXAH,
	AARCH64_INS_LDSMAXALB,
	AARCH64_INS_LDSMAXALH,
	AARCH64_INS_LDSMAXAL,
	AARCH64_INS_LDSMAXA,
	AARCH64_INS_LDSMAXB,
	AARCH64_INS_LDSMAXH,
	AARCH64_INS_LDSMAXLB,
	AARCH64_INS_LDSMAXLH,
	AARCH64_INS_LDSMAXL,
	AARCH64_INS_LDSMAX,
	AARCH64_INS_LDSMINAB,
	AARCH64_INS_LDSMINAH,
	AARCH64_INS_LDSMINALB,
	AARCH64_INS_LDSMINALH,
	AARCH64_INS_LDSMINAL,
	AARCH64_INS_LDSMINA,
	AARCH64_INS_LDSMINB,
	AARCH64_INS_LDSMINH,
	AARCH64_INS_LDSMINLB,
	AARCH64_INS_LDSMINLH,
	AARCH64_INS_LDSMINL,
	AARCH64_INS_LDSMIN,
	AARCH64_INS_LDTRB,
	AARCH64_INS_LDTRH,
	AARCH64_INS_LDTRSB,
	AARCH64_INS_LDTRSH,
	AARCH64_INS_LDTRSW,
	AARCH64_INS_LDTR,
	AARCH64_INS_LDUMAXAB,
	AARCH64_INS_LDUMAXAH,
	AARCH64_INS_LDUMAXALB,
	AARCH64_INS_LDUMAXALH,
	AARCH64_INS_LDUMAXAL,
	AARCH64_INS_LDUMAXA,
	AARCH64_INS_LDUMAXB,
	AARCH64_INS_LDUMAXH,
	AARCH64_INS_LDUMAXLB,
	AARCH64_INS_LDUMAXLH,
	AARCH64_INS_LDUMAXL,
	AARCH64_INS_LDUMAX,
	AARCH64_INS_LDUMINAB,
	AARCH64_INS_LDUMINAH,
	AARCH64_INS_LDUMINALB,
	AARCH64_INS_LDUMINALH,
	AARCH64_INS_LDUMINAL,
	AARCH64_INS_LDUMINA,
	AARCH64_INS_LDUMINB,
	AARCH64_INS_LDUMINH,
	AARCH64_INS_LDUMINLB,
	AARCH64_INS_LDUMINLH,
	AARCH64_INS_LDUMINL,
	AARCH64_INS_LDUMIN,
	AARCH64_INS_LDURB,
	AARCH64_INS_LDUR,
	AARCH64_INS_LDURH,
	AARCH64_INS_LDURSB,
	AARCH64_INS_LDURSH,
	AARCH64_INS_LDURSW,
	AARCH64_INS_LDXP,
	AARCH64_INS_LDXRB,
	AARCH64_INS_LDXRH,
	AARCH64_INS_LDXR,
	AARCH64_INS_LSLR,
	AARCH64_INS_LSL,
	AARCH64_INS_LSRR,
	AARCH64_INS_LSR,
	AARCH64_INS_LUTI2,
	AARCH64_INS_LUTI4,
	AARCH64_INS_MADDPT,
	AARCH64_INS_MADD,
	AARCH64_INS_MADPT,
	AARCH64_INS_MAD,
	AARCH64_INS_MATCH,
	AARCH64_INS_MLAPT,
	AARCH64_INS_MLA,
	AARCH64_INS_MLS,
	AARCH64_INS_SETGE,
	AARCH64_INS_SETGEN,
	AARCH64_INS_SETGET,
	AARCH64_INS_SETGETN,
	AARCH64_INS_MOVAZ,
	AARCH64_INS_MOVI,
	AARCH64_INS_MOVK,
	AARCH64_INS_MOVN,
	AARCH64_INS_MOVPRFX,
	AARCH64_INS_MOVT,
	AARCH64_INS_MOVZ,
	AARCH64_INS_MRRS,
	AARCH64_INS_MRS,
	AARCH64_INS_MSB,
	AARCH64_INS_MSR,
	AARCH64_INS_MSRR,
	AARCH64_INS_MSUBPT,
	AARCH64_INS_MSUB,
	AARCH64_INS_MUL,
	AARCH64_INS_MVNI,
	AARCH64_INS_NANDS,
	AARCH64_INS_NAND,
	AARCH64_INS_NBSL,
	AARCH64_INS_NEG,
	AARCH64_INS_NMATCH,
	AARCH64_INS_NORS,
	AARCH64_INS_NOR,
	AARCH64_INS_NOT,
	AARCH64_INS_ORNS,
	AARCH64_INS_ORN,
	AARCH64_INS_ORQV,
	AARCH64_INS_ORRS,
	AARCH64_INS_ORR,
	AARCH64_INS_ORV,
	AARCH64_INS_PACDA,
	AARCH64_INS_PACDB,
	AARCH64_INS_PACDZA,
	AARCH64_INS_PACDZB,
	AARCH64_INS_PACGA,
	AARCH64_INS_PACIA,
	AARCH64_INS_PACIA171615,
	AARCH64_INS_PACIASPPC,
	AARCH64_INS_PACIB,
	AARCH64_INS_PACIB171615,
	AARCH64_INS_PACIBSPPC,
	AARCH64_INS_PACIZA,
	AARCH64_INS_PACIZB,
	AARCH64_INS_PACNBIASPPC,
	AARCH64_INS_PACNBIBSPPC,
	AARCH64_INS_PEXT,
	AARCH64_INS_PFALSE,
	AARCH64_INS_PFIRST,
	AARCH64_INS_PMOV,
	AARCH64_INS_PMULLB,
	AARCH64_INS_PMULLT,
	AARCH64_INS_PMULL2,
	AARCH64_INS_PMULL,
	AARCH64_INS_PMUL,
	AARCH64_INS_PNEXT,
	AARCH64_INS_PRFB,
	AARCH64_INS_PRFD,
	AARCH64_INS_PRFH,
	AARCH64_INS_PRFM,
	AARCH64_INS_PRFUM,
	AARCH64_INS_PRFW,
	AARCH64_INS_PSEL,
	AARCH64_INS_PTEST,
	AARCH64_INS_PTRUES,
	AARCH64_INS_PTRUE,
	AARCH64_INS_PUNPKHI,
	AARCH64_INS_PUNPKLO,
	AARCH64_INS_RADDHNB,
	AARCH64_INS_RADDHNT,
	AARCH64_INS_RADDHN,
	AARCH64_INS_RADDHN2,
	AARCH64_INS_RAX1,
	AARCH64_INS_RBIT,
	AARCH64_INS_RCWCAS,
	AARCH64_INS_RCWCASA,
	AARCH64_INS_RCWCASAL,
	AARCH64_INS_RCWCASL,
	AARCH64_INS_RCWCASP,
	AARCH64_INS_RCWCASPA,
	AARCH64_INS_RCWCASPAL,
	AARCH64_INS_RCWCASPL,
	AARCH64_INS_RCWCLR,
	AARCH64_INS_RCWCLRA,
	AARCH64_INS_RCWCLRAL,
	AARCH64_INS_RCWCLRL,
	AARCH64_INS_RCWCLRP,
	AARCH64_INS_RCWCLRPA,
	AARCH64_INS_RCWCLRPAL,
	AARCH64_INS_RCWCLRPL,
	AARCH64_INS_RCWSCLR,
	AARCH64_INS_RCWSCLRA,
	AARCH64_INS_RCWSCLRAL,
	AARCH64_INS_RCWSCLRL,
	AARCH64_INS_RCWSCLRP,
	AARCH64_INS_RCWSCLRPA,
	AARCH64_INS_RCWSCLRPAL,
	AARCH64_INS_RCWSCLRPL,
	AARCH64_INS_RCWSCAS,
	AARCH64_INS_RCWSCASA,
	AARCH64_INS_RCWSCASAL,
	AARCH64_INS_RCWSCASL,
	AARCH64_INS_RCWSCASP,
	AARCH64_INS_RCWSCASPA,
	AARCH64_INS_RCWSCASPAL,
	AARCH64_INS_RCWSCASPL,
	AARCH64_INS_RCWSET,
	AARCH64_INS_RCWSETA,
	AARCH64_INS_RCWSETAL,
	AARCH64_INS_RCWSETL,
	AARCH64_INS_RCWSETP,
	AARCH64_INS_RCWSETPA,
	AARCH64_INS_RCWSETPAL,
	AARCH64_INS_RCWSETPL,
	AARCH64_INS_RCWSSET,
	AARCH64_INS_RCWSSETA,
	AARCH64_INS_RCWSSETAL,
	AARCH64_INS_RCWSSETL,
	AARCH64_INS_RCWSSETP,
	AARCH64_INS_RCWSSETPA,
	AARCH64_INS_RCWSSETPAL,
	AARCH64_INS_RCWSSETPL,
	AARCH64_INS_RCWSWP,
	AARCH64_INS_RCWSWPA,
	AARCH64_INS_RCWSWPAL,
	AARCH64_INS_RCWSWPL,
	AARCH64_INS_RCWSWPP,
	AARCH64_INS_RCWSWPPA,
	AARCH64_INS_RCWSWPPAL,
	AARCH64_INS_RCWSWPPL,
	AARCH64_INS_RCWSSWP,
	AARCH64_INS_RCWSSWPA,
	AARCH64_INS_RCWSSWPAL,
	AARCH64_INS_RCWSSWPL,
	AARCH64_INS_RCWSSWPP,
	AARCH64_INS_RCWSSWPPA,
	AARCH64_INS_RCWSSWPPAL,
	AARCH64_INS_RCWSSWPPL,
	AARCH64_INS_RDFFRS,
	AARCH64_INS_RDFFR,
	AARCH64_INS_RDSVL,
	AARCH64_INS_RDVL,
	AARCH64_INS_RET,
	AARCH64_INS_RETAA,
	AARCH64_INS_RETAASPPC,
	AARCH64_INS_RETAB,
	AARCH64_INS_RETABSPPC,
	AARCH64_INS_REV16,
	AARCH64_INS_REV32,
	AARCH64_INS_REV64,
	AARCH64_INS_REVB,
	AARCH64_INS_REVD,
	AARCH64_INS_REVH,
	AARCH64_INS_REVW,
	AARCH64_INS_REV,
	AARCH64_INS_RMIF,
	AARCH64_INS_ROR,
	AARCH64_INS_RPRFM,
	AARCH64_INS_RSHRNB,
	AARCH64_INS_RSHRNT,
	AARCH64_INS_RSHRN2,
	AARCH64_INS_RSHRN,
	AARCH64_INS_RSUBHNB,
	AARCH64_INS_RSUBHNT,
	AARCH64_INS_RSUBHN,
	AARCH64_INS_RSUBHN2,
	AARCH64_INS_SABALB,
	AARCH64_INS_SABALT,
	AARCH64_INS_SABAL2,
	AARCH64_INS_SABAL,
	AARCH64_INS_SABA,
	AARCH64_INS_SABDLB,
	AARCH64_INS_SABDLT,
	AARCH64_INS_SABDL2,
	AARCH64_INS_SABDL,
	AARCH64_INS_SABD,
	AARCH64_INS_SADALP,
	AARCH64_INS_SADDLBT,
	AARCH64_INS_SADDLB,
	AARCH64_INS_SADDLP,
	AARCH64_INS_SADDLT,
	AARCH64_INS_SADDLV,
	AARCH64_INS_SADDL2,
	AARCH64_INS_SADDL,
	AARCH64_INS_SADDV,
	AARCH64_INS_SADDWB,
	AARCH64_INS_SADDWT,
	AARCH64_INS_SADDW2,
	AARCH64_INS_SADDW,
	AARCH64_INS_SB,
	AARCH64_INS_SBCLB,
	AARCH64_INS_SBCLT,
	AARCH64_INS_SBCS,
	AARCH64_INS_SBC,
	AARCH64_INS_SBFM,
	AARCH64_INS_SCLAMP,
	AARCH64_INS_SCVTF,
	AARCH64_INS_SDIVR,
	AARCH64_INS_SDIV,
	AARCH64_INS_SDOT,
	AARCH64_INS_SEL,
	AARCH64_INS_SETE,
	AARCH64_INS_SETEN,
	AARCH64_INS_SETET,
	AARCH64_INS_SETETN,
	AARCH64_INS_SETF16,
	AARCH64_INS_SETF8,
	AARCH64_INS_SETFFR,
	AARCH64_INS_SETGM,
	AARCH64_INS_SETGMN,
	AARCH64_INS_SETGMT,
	AARCH64_INS_SETGMTN,
	AARCH64_INS_SETGP,
	AARCH64_INS_SETGPN,
	AARCH64_INS_SETGPT,
	AARCH64_INS_SETGPTN,
	AARCH64_INS_SETM,
	AARCH64_INS_SETMN,
	AARCH64_INS_SETMT,
	AARCH64_INS_SETMTN,
	AARCH64_INS_SETP,
	AARCH64_INS_SETPN,
	AARCH64_INS_SETPT,
	AARCH64_INS_SETPTN,
	AARCH64_INS_SHA1C,
	AARCH64_INS_SHA1H,
	AARCH64_INS_SHA1M,
	AARCH64_INS_SHA1P,
	AARCH64_INS_SHA1SU0,
	AARCH64_INS_SHA1SU1,
	AARCH64_INS_SHA256H2,
	AARCH64_INS_SHA256H,
	AARCH64_INS_SHA256SU0,
	AARCH64_INS_SHA256SU1,
	AARCH64_INS_SHA512H,
	AARCH64_INS_SHA512H2,
	AARCH64_INS_SHA512SU0,
	AARCH64_INS_SHA512SU1,
	AARCH64_INS_SHADD,
	AARCH64_INS_SHLL2,
	AARCH64_INS_SHLL,
	AARCH64_INS_SHL,
	AARCH64_INS_SHRNB,
	AARCH64_INS_SHRNT,
	AARCH64_INS_SHRN2,
	AARCH64_INS_SHRN,
	AARCH64_INS_SHSUBR,
	AARCH64_INS_SHSUB,
	AARCH64_INS_SLI,
	AARCH64_INS_SM3PARTW1,
	AARCH64_INS_SM3PARTW2,
	AARCH64_INS_SM3SS1,
	AARCH64_INS_SM3TT1A,
	AARCH64_INS_SM3TT1B,
	AARCH64_INS_SM3TT2A,
	AARCH64_INS_SM3TT2B,
	AARCH64_INS_SM4E,
	AARCH64_INS_SM4EKEY,
	AARCH64_INS_SMADDL,
	AARCH64_INS_SMAXP,
	AARCH64_INS_SMAXQV,
	AARCH64_INS_SMAXV,
	AARCH64_INS_SMAX,
	AARCH64_INS_SMC,
	AARCH64_INS_SMINP,
	AARCH64_INS_SMINQV,
	AARCH64_INS_SMINV,
	AARCH64_INS_SMIN,
	AARCH64_INS_SMLALB,
	AARCH64_INS_SMLALL,
	AARCH64_INS_SMLALT,
	AARCH64_INS_SMLAL,
	AARCH64_INS_SMLAL2,
	AARCH64_INS_SMLSLB,
	AARCH64_INS_SMLSLL,
	AARCH64_INS_SMLSLT,
	AARCH64_INS_SMLSL,
	AARCH64_INS_SMLSL2,
	AARCH64_INS_SMMLA,
	AARCH64_INS_SMOPA,
	AARCH64_INS_SMOPS,
	AARCH64_INS_SMOV,
	AARCH64_INS_SMSUBL,
	AARCH64_INS_SMULH,
	AARCH64_INS_SMULLB,
	AARCH64_INS_SMULLT,
	AARCH64_INS_SMULL2,
	AARCH64_INS_SMULL,
	AARCH64_INS_SPLICE,
	AARCH64_INS_SQABS,
	AARCH64_INS_SQADD,
	AARCH64_INS_SQCADD,
	AARCH64_INS_SQCVTN,
	AARCH64_INS_SQCVTUN,
	AARCH64_INS_SQCVTU,
	AARCH64_INS_SQCVT,
	AARCH64_INS_SQDECB,
	AARCH64_INS_SQDECD,
	AARCH64_INS_SQDECH,
	AARCH64_INS_SQDECP,
	AARCH64_INS_SQDECW,
	AARCH64_INS_SQDMLALBT,
	AARCH64_INS_SQDMLALB,
	AARCH64_INS_SQDMLALT,
	AARCH64_INS_SQDMLAL,
	AARCH64_INS_SQDMLAL2,
	AARCH64_INS_SQDMLSLBT,
	AARCH64_INS_SQDMLSLB,
	AARCH64_INS_SQDMLSLT,
	AARCH64_INS_SQDMLSL,
	AARCH64_INS_SQDMLSL2,
	AARCH64_INS_SQDMULH,
	AARCH64_INS_SQDMULLB,
	AARCH64_INS_SQDMULLT,
	AARCH64_INS_SQDMULL,
	AARCH64_INS_SQDMULL2,
	AARCH64_INS_SQINCB,
	AARCH64_INS_SQINCD,
	AARCH64_INS_SQINCH,
	AARCH64_INS_SQINCP,
	AARCH64_INS_SQINCW,
	AARCH64_INS_SQNEG,
	AARCH64_INS_SQRDCMLAH,
	AARCH64_INS_SQRDMLAH,
	AARCH64_INS_SQRDMLSH,
	AARCH64_INS_SQRDMULH,
	AARCH64_INS_SQRSHLR,
	AARCH64_INS_SQRSHL,
	AARCH64_INS_SQRSHRNB,
	AARCH64_INS_SQRSHRNT,
	AARCH64_INS_SQRSHRN,
	AARCH64_INS_SQRSHRN2,
	AARCH64_INS_SQRSHRUNB,
	AARCH64_INS_SQRSHRUNT,
	AARCH64_INS_SQRSHRUN,
	AARCH64_INS_SQRSHRUN2,
	AARCH64_INS_SQRSHRU,
	AARCH64_INS_SQRSHR,
	AARCH64_INS_SQSHLR,
	AARCH64_INS_SQSHLU,
	AARCH64_INS_SQSHL,
	AARCH64_INS_SQSHRNB,
	AARCH64_INS_SQSHRNT,
	AARCH64_INS_SQSHRN,
	AARCH64_INS_SQSHRN2,
	AARCH64_INS_SQSHRUNB,
	AARCH64_INS_SQSHRUNT,
	AARCH64_INS_SQSHRUN,
	AARCH64_INS_SQSHRUN2,
	AARCH64_INS_SQSUBR,
	AARCH64_INS_SQSUB,
	AARCH64_INS_SQXTNB,
	AARCH64_INS_SQXTNT,
	AARCH64_INS_SQXTN2,
	AARCH64_INS_SQXTN,
	AARCH64_INS_SQXTUNB,
	AARCH64_INS_SQXTUNT,
	AARCH64_INS_SQXTUN2,
	AARCH64_INS_SQXTUN,
	AARCH64_INS_SRHADD,
	AARCH64_INS_SRI,
	AARCH64_INS_SRSHLR,
	AARCH64_INS_SRSHL,
	AARCH64_INS_SRSHR,
	AARCH64_INS_SRSRA,
	AARCH64_INS_SSHLLB,
	AARCH64_INS_SSHLLT,
	AARCH64_INS_SSHLL2,
	AARCH64_INS_SSHLL,
	AARCH64_INS_SSHL,
	AARCH64_INS_SSHR,
	AARCH64_INS_SSRA,
	AARCH64_INS_ST1B,
	AARCH64_INS_ST1D,
	AARCH64_INS_ST1H,
	AARCH64_INS_ST1Q,
	AARCH64_INS_ST1W,
	AARCH64_INS_SSUBLBT,
	AARCH64_INS_SSUBLB,
	AARCH64_INS_SSUBLTB,
	AARCH64_INS_SSUBLT,
	AARCH64_INS_SSUBL2,
	AARCH64_INS_SSUBL,
	AARCH64_INS_SSUBWB,
	AARCH64_INS_SSUBWT,
	AARCH64_INS_SSUBW2,
	AARCH64_INS_SSUBW,
	AARCH64_INS_ST1,
	AARCH64_INS_ST2B,
	AARCH64_INS_ST2D,
	AARCH64_INS_ST2G,
	AARCH64_INS_ST2H,
	AARCH64_INS_ST2Q,
	AARCH64_INS_ST2,
	AARCH64_INS_ST2W,
	AARCH64_INS_ST3B,
	AARCH64_INS_ST3D,
	AARCH64_INS_ST3H,
	AARCH64_INS_ST3Q,
	AARCH64_INS_ST3,
	AARCH64_INS_ST3W,
	AARCH64_INS_ST4B,
	AARCH64_INS_ST4D,
	AARCH64_INS_ST4,
	AARCH64_INS_ST4H,
	AARCH64_INS_ST4Q,
	AARCH64_INS_ST4W,
	AARCH64_INS_ST64B,
	AARCH64_INS_ST64BV,
	AARCH64_INS_ST64BV0,
	AARCH64_INS_STGM,
	AARCH64_INS_STGP,
	AARCH64_INS_STG,
	AARCH64_INS_STILP,
	AARCH64_INS_STL1,
	AARCH64_INS_STLLRB,
	AARCH64_INS_STLLRH,
	AARCH64_INS_STLLR,
	AARCH64_INS_STLRB,
	AARCH64_INS_STLRH,
	AARCH64_INS_STLR,
	AARCH64_INS_STLURB,
	AARCH64_INS_STLURH,
	AARCH64_INS_STLUR,
	AARCH64_INS_STLXP,
	AARCH64_INS_STLXRB,
	AARCH64_INS_STLXRH,
	AARCH64_INS_STLXR,
	AARCH64_INS_STNP,
	AARCH64_INS_STNT1B,
	AARCH64_INS_STNT1D,
	AARCH64_INS_STNT1H,
	AARCH64_INS_STNT1W,
	AARCH64_INS_STP,
	AARCH64_INS_STRB,
	AARCH64_INS_STR,
	AARCH64_INS_STRH,
	AARCH64_INS_STTRB,
	AARCH64_INS_STTRH,
	AARCH64_INS_STTR,
	AARCH64_INS_STURB,
	AARCH64_INS_STUR,
	AARCH64_INS_STURH,
	AARCH64_INS_STXP,
	AARCH64_INS_STXRB,
	AARCH64_INS_STXRH,
	AARCH64_INS_STXR,
	AARCH64_INS_STZ2G,
	AARCH64_INS_STZGM,
	AARCH64_INS_STZG,
	AARCH64_INS_SUBG,
	AARCH64_INS_SUBHNB,
	AARCH64_INS_SUBHNT,
	AARCH64_INS_SUBHN,
	AARCH64_INS_SUBHN2,
	AARCH64_INS_SUBP,
	AARCH64_INS_SUBPS,
	AARCH64_INS_SUBPT,
	AARCH64_INS_SUBR,
	AARCH64_INS_SUBS,
	AARCH64_INS_SUB,
	AARCH64_INS_SUDOT,
	AARCH64_INS_SUMLALL,
	AARCH64_INS_SUMOPA,
	AARCH64_INS_SUMOPS,
	AARCH64_INS_SUNPKHI,
	AARCH64_INS_SUNPKLO,
	AARCH64_INS_SUNPK,
	AARCH64_INS_SUQADD,
	AARCH64_INS_SUVDOT,
	AARCH64_INS_SVC,
	AARCH64_INS_SVDOT,
	AARCH64_INS_SWPAB,
	AARCH64_INS_SWPAH,
	AARCH64_INS_SWPALB,
	AARCH64_INS_SWPALH,
	AARCH64_INS_SWPAL,
	AARCH64_INS_SWPA,
	AARCH64_INS_SWPB,
	AARCH64_INS_SWPH,
	AARCH64_INS_SWPLB,
	AARCH64_INS_SWPLH,
	AARCH64_INS_SWPL,
	AARCH64_INS_SWPP,
	AARCH64_INS_SWPPA,
	AARCH64_INS_SWPPAL,
	AARCH64_INS_SWPPL,
	AARCH64_INS_SWP,
	AARCH64_INS_SXTB,
	AARCH64_INS_SXTH,
	AARCH64_INS_SXTW,
	AARCH64_INS_SYSL,
	AARCH64_INS_SYSP,
	AARCH64_INS_SYS,
	AARCH64_INS_TBLQ,
	AARCH64_INS_TBL,
	AARCH64_INS_TBNZ,
	AARCH64_INS_TBXQ,
	AARCH64_INS_TBX,
	AARCH64_INS_TBZ,
	AARCH64_INS_TCANCEL,
	AARCH64_INS_TCOMMIT,
	AARCH64_INS_TRCIT,
	AARCH64_INS_TRN1,
	AARCH64_INS_TRN2,
	AARCH64_INS_TSB,
	AARCH64_INS_TSTART,
	AARCH64_INS_TTEST,
	AARCH64_INS_UABALB,
	AARCH64_INS_UABALT,
	AARCH64_INS_UABAL2,
	AARCH64_INS_UABAL,
	AARCH64_INS_UABA,
	AARCH64_INS_UABDLB,
	AARCH64_INS_UABDLT,
	AARCH64_INS_UABDL2,
	AARCH64_INS_UABDL,
	AARCH64_INS_UABD,
	AARCH64_INS_UADALP,
	AARCH64_INS_UADDLB,
	AARCH64_INS_UADDLP,
	AARCH64_INS_UADDLT,
	AARCH64_INS_UADDLV,
	AARCH64_INS_UADDL2,
	AARCH64_INS_UADDL,
	AARCH64_INS_UADDV,
	AARCH64_INS_UADDWB,
	AARCH64_INS_UADDWT,
	AARCH64_INS_UADDW2,
	AARCH64_INS_UADDW,
	AARCH64_INS_UBFM,
	AARCH64_INS_UCLAMP,
	AARCH64_INS_UCVTF,
	AARCH64_INS_UDF,
	AARCH64_INS_UDIVR,
	AARCH64_INS_UDIV,
	AARCH64_INS_UDOT,
	AARCH64_INS_UHADD,
	AARCH64_INS_UHSUBR,
	AARCH64_INS_UHSUB,
	AARCH64_INS_UMADDL,
	AARCH64_INS_UMAXP,
	AARCH64_INS_UMAXQV,
	AARCH64_INS_UMAXV,
	AARCH64_INS_UMAX,
	AARCH64_INS_UMINP,
	AARCH64_INS_UMINQV,
	AARCH64_INS_UMINV,
	AARCH64_INS_UMIN,
	AARCH64_INS_UMLALB,
	AARCH64_INS_UMLALL,
	AARCH64_INS_UMLALT,
	AARCH64_INS_UMLAL,
	AARCH64_INS_UMLAL2,
	AARCH64_INS_UMLSLB,
	AARCH64_INS_UMLSLL,
	AARCH64_INS_UMLSLT,
	AARCH64_INS_UMLSL,
	AARCH64_INS_UMLSL2,
	AARCH64_INS_UMMLA,
	AARCH64_INS_UMOPA,
	AARCH64_INS_UMOPS,
	AARCH64_INS_UMOV,
	AARCH64_INS_UMSUBL,
	AARCH64_INS_UMULH,
	AARCH64_INS_UMULLB,
	AARCH64_INS_UMULLT,
	AARCH64_INS_UMULL2,
	AARCH64_INS_UMULL,
	AARCH64_INS_UQADD,
	AARCH64_INS_UQCVTN,
	AARCH64_INS_UQCVT,
	AARCH64_INS_UQDECB,
	AARCH64_INS_UQDECD,
	AARCH64_INS_UQDECH,
	AARCH64_INS_UQDECP,
	AARCH64_INS_UQDECW,
	AARCH64_INS_UQINCB,
	AARCH64_INS_UQINCD,
	AARCH64_INS_UQINCH,
	AARCH64_INS_UQINCP,
	AARCH64_INS_UQINCW,
	AARCH64_INS_UQRSHLR,
	AARCH64_INS_UQRSHL,
	AARCH64_INS_UQRSHRNB,
	AARCH64_INS_UQRSHRNT,
	AARCH64_INS_UQRSHRN,
	AARCH64_INS_UQRSHRN2,
	AARCH64_INS_UQRSHR,
	AARCH64_INS_UQSHLR,
	AARCH64_INS_UQSHL,
	AARCH64_INS_UQSHRNB,
	AARCH64_INS_UQSHRNT,
	AARCH64_INS_UQSHRN,
	AARCH64_INS_UQSHRN2,
	AARCH64_INS_UQSUBR,
	AARCH64_INS_UQSUB,
	AARCH64_INS_UQXTNB,
	AARCH64_INS_UQXTNT,
	AARCH64_INS_UQXTN2,
	AARCH64_INS_UQXTN,
	AARCH64_INS_URECPE,
	AARCH64_INS_URHADD,
	AARCH64_INS_URSHLR,
	AARCH64_INS_URSHL,
	AARCH64_INS_URSHR,
	AARCH64_INS_URSQRTE,
	AARCH64_INS_URSRA,
	AARCH64_INS_USDOT,
	AARCH64_INS_USHLLB,
	AARCH64_INS_USHLLT,
	AARCH64_INS_USHLL2,
	AARCH64_INS_USHLL,
	AARCH64_INS_USHL,
	AARCH64_INS_USHR,
	AARCH64_INS_USMLALL,
	AARCH64_INS_USMMLA,
	AARCH64_INS_USMOPA,
	AARCH64_INS_USMOPS,
	AARCH64_INS_USQADD,
	AARCH64_INS_USRA,
	AARCH64_INS_USUBLB,
	AARCH64_INS_USUBLT,
	AARCH64_INS_USUBL2,
	AARCH64_INS_USUBL,
	AARCH64_INS_USUBWB,
	AARCH64_INS_USUBWT,
	AARCH64_INS_USUBW2,
	AARCH64_INS_USUBW,
	AARCH64_INS_USVDOT,
	AARCH64_INS_UUNPKHI,
	AARCH64_INS_UUNPKLO,
	AARCH64_INS_UUNPK,
	AARCH64_INS_UVDOT,
	AARCH64_INS_UXTB,
	AARCH64_INS_UXTH,
	AARCH64_INS_UXTW,
	AARCH64_INS_UZP1,
	AARCH64_INS_UZP2,
	AARCH64_INS_UZPQ1,
	AARCH64_INS_UZPQ2,
	AARCH64_INS_UZP,
	AARCH64_INS_WFET,
	AARCH64_INS_WFIT,
	AARCH64_INS_WHILEGE,
	AARCH64_INS_WHILEGT,
	AARCH64_INS_WHILEHI,
	AARCH64_INS_WHILEHS,
	AARCH64_INS_WHILELE,
	AARCH64_INS_WHILELO,
	AARCH64_INS_WHILELS,
	AARCH64_INS_WHILELT,
	AARCH64_INS_WHILERW,
	AARCH64_INS_WHILEWR,
	AARCH64_INS_WRFFR,
	AARCH64_INS_XAFLAG,
	AARCH64_INS_XAR,
	AARCH64_INS_XPACD,
	AARCH64_INS_XPACI,
	AARCH64_INS_XTN2,
	AARCH64_INS_XTN,
	AARCH64_INS_ZERO,
	AARCH64_INS_ZIP1,
	AARCH64_INS_ZIP2,
	AARCH64_INS_ZIPQ1,
	AARCH64_INS_ZIPQ2,
	AARCH64_INS_ZIP,

	// clang-format on
	// generated content <AArch64GenCSInsnEnum.inc> end

  AARCH64_INS_ENDING, // <-- mark the end of the list of insn

	AARCH64_INS_ALIAS_BEGIN,
	// generated content <AArch64GenCSAliasEnum.inc> begin
	// clang-format off

	AARCH64_INS_ALIAS_ADDPT, // Real instr.: AARCH64_ADDPT_shift
	AARCH64_INS_ALIAS_GCSB, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_GCSPOPM, // Real instr.: AARCH64_GCSPOPM
	AARCH64_INS_ALIAS_LDAPUR, // Real instr.: AARCH64_LDAPURbi
	AARCH64_INS_ALIAS_STLLRB, // Real instr.: AARCH64_STLLRB
	AARCH64_INS_ALIAS_STLLRH, // Real instr.: AARCH64_STLLRH
	AARCH64_INS_ALIAS_STLLR, // Real instr.: AARCH64_STLLRW
	AARCH64_INS_ALIAS_STLRB, // Real instr.: AARCH64_STLRB
	AARCH64_INS_ALIAS_STLRH, // Real instr.: AARCH64_STLRH
	AARCH64_INS_ALIAS_STLR, // Real instr.: AARCH64_STLRW
	AARCH64_INS_ALIAS_STLUR, // Real instr.: AARCH64_STLURbi
	AARCH64_INS_ALIAS_SUBPT, // Real instr.: AARCH64_SUBPT_shift
	AARCH64_INS_ALIAS_LDRAA, // Real instr.: AARCH64_LDRAAindexed
	AARCH64_INS_ALIAS_ADD, // Real instr.: AARCH64_ADDWrs
	AARCH64_INS_ALIAS_CMN, // Real instr.: AARCH64_ADDSWri
	AARCH64_INS_ALIAS_ADDS, // Real instr.: AARCH64_ADDSWrs
	AARCH64_INS_ALIAS_AND, // Real instr.: AARCH64_ANDWrs
	AARCH64_INS_ALIAS_ANDS, // Real instr.: AARCH64_ANDSWrs
	AARCH64_INS_ALIAS_LDR, // Real instr.: AARCH64_LDRXui
	AARCH64_INS_ALIAS_STR, // Real instr.: AARCH64_STRBui
	AARCH64_INS_ALIAS_LDRB, // Real instr.: AARCH64_LDRBBroX
	AARCH64_INS_ALIAS_STRB, // Real instr.: AARCH64_STRBBroX
	AARCH64_INS_ALIAS_LDRH, // Real instr.: AARCH64_LDRHHroX
	AARCH64_INS_ALIAS_STRH, // Real instr.: AARCH64_STRHHroX
	AARCH64_INS_ALIAS_PRFM, // Real instr.: AARCH64_PRFMroX
	AARCH64_INS_ALIAS_LDAPURB, // Real instr.: AARCH64_LDAPURBi
	AARCH64_INS_ALIAS_STLURB, // Real instr.: AARCH64_STLURBi
	AARCH64_INS_ALIAS_LDUR, // Real instr.: AARCH64_LDURXi
	AARCH64_INS_ALIAS_STUR, // Real instr.: AARCH64_STURXi
	AARCH64_INS_ALIAS_PRFUM, // Real instr.: AARCH64_PRFUMi
	AARCH64_INS_ALIAS_LDTR, // Real instr.: AARCH64_LDTRXi
	AARCH64_INS_ALIAS_STTR, // Real instr.: AARCH64_STTRWi
	AARCH64_INS_ALIAS_LDP, // Real instr.: AARCH64_LDPWi
	AARCH64_INS_ALIAS_STGP, // Real instr.: AARCH64_STGPi
	AARCH64_INS_ALIAS_LDNP, // Real instr.: AARCH64_LDNPWi
	AARCH64_INS_ALIAS_STNP, // Real instr.: AARCH64_STNPWi
	AARCH64_INS_ALIAS_STG, // Real instr.: AARCH64_STGi
	AARCH64_INS_ALIAS_MOV, // Real instr.: AARCH64_UMOVvi32_idx0
	AARCH64_INS_ALIAS_LD1, // Real instr.: AARCH64_LD1Onev16b_POST
	AARCH64_INS_ALIAS_LD1R, // Real instr.: AARCH64_LD1Rv8b_POST
	AARCH64_INS_ALIAS_STADDLB, // Real instr.: AARCH64_LDADDLB
	AARCH64_INS_ALIAS_STADDLH, // Real instr.: AARCH64_LDADDLH
	AARCH64_INS_ALIAS_STADDL, // Real instr.: AARCH64_LDADDLW
	AARCH64_INS_ALIAS_STADDB, // Real instr.: AARCH64_LDADDB
	AARCH64_INS_ALIAS_STADDH, // Real instr.: AARCH64_LDADDH
	AARCH64_INS_ALIAS_STADD, // Real instr.: AARCH64_LDADDW
	AARCH64_INS_ALIAS_PTRUE, // Real instr.: AARCH64_PTRUE_B
	AARCH64_INS_ALIAS_PTRUES, // Real instr.: AARCH64_PTRUES_B
	AARCH64_INS_ALIAS_CNTB, // Real instr.: AARCH64_CNTB_XPiI
	AARCH64_INS_ALIAS_SQINCH, // Real instr.: AARCH64_SQINCH_ZPiI
	AARCH64_INS_ALIAS_INCB, // Real instr.: AARCH64_INCB_XPiI
	AARCH64_INS_ALIAS_SQINCB, // Real instr.: AARCH64_SQINCB_XPiWdI
	AARCH64_INS_ALIAS_UQINCB, // Real instr.: AARCH64_UQINCB_WPiI
	AARCH64_INS_ALIAS_ORR, // Real instr.: AARCH64_ORR_ZI
	AARCH64_INS_ALIAS_DUPM, // Real instr.: AARCH64_DUPM_ZI
	AARCH64_INS_ALIAS_FMOV, // Real instr.: AARCH64_DUP_ZI_H
	AARCH64_INS_ALIAS_EOR3, // Real instr.: AARCH64_EOR3_ZZZZ
	AARCH64_INS_ALIAS_ST1B, // Real instr.: AARCH64_ST1B_IMM
	AARCH64_INS_ALIAS_ST2B, // Real instr.: AARCH64_ST2B_IMM
	AARCH64_INS_ALIAS_ST2Q, // Real instr.: AARCH64_ST2Q_IMM
	AARCH64_INS_ALIAS_STNT1B, // Real instr.: AARCH64_STNT1B_ZRI
	AARCH64_INS_ALIAS_LD1B, // Real instr.: AARCH64_LD1B_IMM
	AARCH64_INS_ALIAS_LDNT1B, // Real instr.: AARCH64_LDNT1B_ZRI
	AARCH64_INS_ALIAS_LD1RQB, // Real instr.: AARCH64_LD1RQ_B_IMM
	AARCH64_INS_ALIAS_LD1RB, // Real instr.: AARCH64_LD1RB_IMM
	AARCH64_INS_ALIAS_LDFF1B, // Real instr.: AARCH64_LDFF1B_REAL
	AARCH64_INS_ALIAS_LDNF1B, // Real instr.: AARCH64_LDNF1B_IMM_REAL
	AARCH64_INS_ALIAS_LD2B, // Real instr.: AARCH64_LD2B_IMM
	AARCH64_INS_ALIAS_LD1SB, // Real instr.: AARCH64_GLD1SB_S_IMM_REAL
	AARCH64_INS_ALIAS_PRFB, // Real instr.: AARCH64_PRFB_PRI
	AARCH64_INS_ALIAS_LDNT1SB, // Real instr.: AARCH64_LDNT1SB_ZZR_S_REAL
	AARCH64_INS_ALIAS_LD1ROB, // Real instr.: AARCH64_LD1RO_B_IMM
	AARCH64_INS_ALIAS_LD1Q, // Real instr.: AARCH64_GLD1Q
	AARCH64_INS_ALIAS_ST1Q, // Real instr.: AARCH64_SST1Q
	AARCH64_INS_ALIAS_LD1W, // Real instr.: AARCH64_LD1W_Q_IMM
	AARCH64_INS_ALIAS_PMOV, // Real instr.: AARCH64_PMOV_PZI_B
	AARCH64_INS_ALIAS_SMSTART, // Real instr.: AARCH64_MSRpstatesvcrImm1
	AARCH64_INS_ALIAS_SMSTOP, // Real instr.: AARCH64_MSRpstatesvcrImm1
	AARCH64_INS_ALIAS_ZERO, // Real instr.: AARCH64_ZERO_M
	AARCH64_INS_ALIAS_MOVT, // Real instr.: AARCH64_MOVT
	AARCH64_INS_ALIAS_NOP, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_YIELD, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_WFE, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_WFI, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_SEV, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_SEVL, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_DGH, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_ESB, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_CSDB, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_BTI, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_PSB, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_CHKFEAT, // Real instr.: AARCH64_CHKFEAT
	AARCH64_INS_ALIAS_PACIAZ, // Real instr.: AARCH64_PACIAZ
	AARCH64_INS_ALIAS_PACIBZ, // Real instr.: AARCH64_PACIBZ
	AARCH64_INS_ALIAS_AUTIAZ, // Real instr.: AARCH64_AUTIAZ
	AARCH64_INS_ALIAS_AUTIBZ, // Real instr.: AARCH64_AUTIBZ
	AARCH64_INS_ALIAS_PACIASP, // Real instr.: AARCH64_PACIASP
	AARCH64_INS_ALIAS_PACIBSP, // Real instr.: AARCH64_PACIBSP
	AARCH64_INS_ALIAS_AUTIASP, // Real instr.: AARCH64_AUTIASP
	AARCH64_INS_ALIAS_AUTIBSP, // Real instr.: AARCH64_AUTIBSP
	AARCH64_INS_ALIAS_PACIA1716, // Real instr.: AARCH64_PACIA1716
	AARCH64_INS_ALIAS_PACIB1716, // Real instr.: AARCH64_PACIB1716
	AARCH64_INS_ALIAS_AUTIA1716, // Real instr.: AARCH64_AUTIA1716
	AARCH64_INS_ALIAS_AUTIB1716, // Real instr.: AARCH64_AUTIB1716
	AARCH64_INS_ALIAS_XPACLRI, // Real instr.: AARCH64_XPACLRI
	AARCH64_INS_ALIAS_LDRAB, // Real instr.: AARCH64_LDRABindexed
	AARCH64_INS_ALIAS_PACM, // Real instr.: AARCH64_PACM
	AARCH64_INS_ALIAS_CLREX, // Real instr.: AARCH64_CLREX
	AARCH64_INS_ALIAS_ISB, // Real instr.: AARCH64_ISB
	AARCH64_INS_ALIAS_SSBB, // Real instr.: AARCH64_DSB
	AARCH64_INS_ALIAS_PSSBB, // Real instr.: AARCH64_DSB
	AARCH64_INS_ALIAS_DFB, // Real instr.: AARCH64_DSB
	AARCH64_INS_ALIAS_SYS, // Real instr.: AARCH64_SYSxt
	AARCH64_INS_ALIAS_MOVN, // Real instr.: AARCH64_MOVNWi
	AARCH64_INS_ALIAS_MOVZ, // Real instr.: AARCH64_MOVZWi
	AARCH64_INS_ALIAS_NGC, // Real instr.: AARCH64_SBCWr
	AARCH64_INS_ALIAS_NGCS, // Real instr.: AARCH64_SBCSWr
	AARCH64_INS_ALIAS_SUB, // Real instr.: AARCH64_SUBWrs
	AARCH64_INS_ALIAS_CMP, // Real instr.: AARCH64_SUBSWri
	AARCH64_INS_ALIAS_SUBS, // Real instr.: AARCH64_SUBSWrs
	AARCH64_INS_ALIAS_NEG, // Real instr.: AARCH64_SUBWrs
	AARCH64_INS_ALIAS_NEGS, // Real instr.: AARCH64_SUBSWrs
	AARCH64_INS_ALIAS_MUL, // Real instr.: AARCH64_MADDWrrr
	AARCH64_INS_ALIAS_MNEG, // Real instr.: AARCH64_MSUBWrrr
	AARCH64_INS_ALIAS_SMULL, // Real instr.: AARCH64_SMADDLrrr
	AARCH64_INS_ALIAS_SMNEGL, // Real instr.: AARCH64_SMSUBLrrr
	AARCH64_INS_ALIAS_UMULL, // Real instr.: AARCH64_UMADDLrrr
	AARCH64_INS_ALIAS_UMNEGL, // Real instr.: AARCH64_UMSUBLrrr
	AARCH64_INS_ALIAS_STCLRLB, // Real instr.: AARCH64_LDCLRLB
	AARCH64_INS_ALIAS_STCLRLH, // Real instr.: AARCH64_LDCLRLH
	AARCH64_INS_ALIAS_STCLRL, // Real instr.: AARCH64_LDCLRLW
	AARCH64_INS_ALIAS_STCLRB, // Real instr.: AARCH64_LDCLRB
	AARCH64_INS_ALIAS_STCLRH, // Real instr.: AARCH64_LDCLRH
	AARCH64_INS_ALIAS_STCLR, // Real instr.: AARCH64_LDCLRW
	AARCH64_INS_ALIAS_STEORLB, // Real instr.: AARCH64_LDEORLB
	AARCH64_INS_ALIAS_STEORLH, // Real instr.: AARCH64_LDEORLH
	AARCH64_INS_ALIAS_STEORL, // Real instr.: AARCH64_LDEORLW
	AARCH64_INS_ALIAS_STEORB, // Real instr.: AARCH64_LDEORB
	AARCH64_INS_ALIAS_STEORH, // Real instr.: AARCH64_LDEORH
	AARCH64_INS_ALIAS_STEOR, // Real instr.: AARCH64_LDEORW
	AARCH64_INS_ALIAS_STSETLB, // Real instr.: AARCH64_LDSETLB
	AARCH64_INS_ALIAS_STSETLH, // Real instr.: AARCH64_LDSETLH
	AARCH64_INS_ALIAS_STSETL, // Real instr.: AARCH64_LDSETLW
	AARCH64_INS_ALIAS_STSETB, // Real instr.: AARCH64_LDSETB
	AARCH64_INS_ALIAS_STSETH, // Real instr.: AARCH64_LDSETH
	AARCH64_INS_ALIAS_STSET, // Real instr.: AARCH64_LDSETW
	AARCH64_INS_ALIAS_STSMAXLB, // Real instr.: AARCH64_LDSMAXLB
	AARCH64_INS_ALIAS_STSMAXLH, // Real instr.: AARCH64_LDSMAXLH
	AARCH64_INS_ALIAS_STSMAXL, // Real instr.: AARCH64_LDSMAXLW
	AARCH64_INS_ALIAS_STSMAXB, // Real instr.: AARCH64_LDSMAXB
	AARCH64_INS_ALIAS_STSMAXH, // Real instr.: AARCH64_LDSMAXH
	AARCH64_INS_ALIAS_STSMAX, // Real instr.: AARCH64_LDSMAXW
	AARCH64_INS_ALIAS_STSMINLB, // Real instr.: AARCH64_LDSMINLB
	AARCH64_INS_ALIAS_STSMINLH, // Real instr.: AARCH64_LDSMINLH
	AARCH64_INS_ALIAS_STSMINL, // Real instr.: AARCH64_LDSMINLW
	AARCH64_INS_ALIAS_STSMINB, // Real instr.: AARCH64_LDSMINB
	AARCH64_INS_ALIAS_STSMINH, // Real instr.: AARCH64_LDSMINH
	AARCH64_INS_ALIAS_STSMIN, // Real instr.: AARCH64_LDSMINW
	AARCH64_INS_ALIAS_STUMAXLB, // Real instr.: AARCH64_LDUMAXLB
	AARCH64_INS_ALIAS_STUMAXLH, // Real instr.: AARCH64_LDUMAXLH
	AARCH64_INS_ALIAS_STUMAXL, // Real instr.: AARCH64_LDUMAXLW
	AARCH64_INS_ALIAS_STUMAXB, // Real instr.: AARCH64_LDUMAXB
	AARCH64_INS_ALIAS_STUMAXH, // Real instr.: AARCH64_LDUMAXH
	AARCH64_INS_ALIAS_STUMAX, // Real instr.: AARCH64_LDUMAXW
	AARCH64_INS_ALIAS_STUMINLB, // Real instr.: AARCH64_LDUMINLB
	AARCH64_INS_ALIAS_STUMINLH, // Real instr.: AARCH64_LDUMINLH
	AARCH64_INS_ALIAS_STUMINL, // Real instr.: AARCH64_LDUMINLW
	AARCH64_INS_ALIAS_STUMINB, // Real instr.: AARCH64_LDUMINB
	AARCH64_INS_ALIAS_STUMINH, // Real instr.: AARCH64_LDUMINH
	AARCH64_INS_ALIAS_STUMIN, // Real instr.: AARCH64_LDUMINW
	AARCH64_INS_ALIAS_IRG, // Real instr.: AARCH64_IRG
	AARCH64_INS_ALIAS_LDG, // Real instr.: AARCH64_LDG
	AARCH64_INS_ALIAS_STZG, // Real instr.: AARCH64_STZGi
	AARCH64_INS_ALIAS_ST2G, // Real instr.: AARCH64_ST2Gi
	AARCH64_INS_ALIAS_STZ2G, // Real instr.: AARCH64_STZ2Gi
	AARCH64_INS_ALIAS_BICS, // Real instr.: AARCH64_BICSWrs
	AARCH64_INS_ALIAS_BIC, // Real instr.: AARCH64_BICWrs
	AARCH64_INS_ALIAS_EON, // Real instr.: AARCH64_EONWrs
	AARCH64_INS_ALIAS_EOR, // Real instr.: AARCH64_EORWrs
	AARCH64_INS_ALIAS_ORN, // Real instr.: AARCH64_ORNWrs
	AARCH64_INS_ALIAS_MVN, // Real instr.: AARCH64_ORNWrs
	AARCH64_INS_ALIAS_TST, // Real instr.: AARCH64_ANDSWri
	AARCH64_INS_ALIAS_ROR, // Real instr.: AARCH64_EXTRWrri
	AARCH64_INS_ALIAS_ASR, // Real instr.: AARCH64_SBFMWri
	AARCH64_INS_ALIAS_SXTB, // Real instr.: AARCH64_SBFMWri
	AARCH64_INS_ALIAS_SXTH, // Real instr.: AARCH64_SBFMWri
	AARCH64_INS_ALIAS_SXTW, // Real instr.: AARCH64_SBFMXri
	AARCH64_INS_ALIAS_LSR, // Real instr.: AARCH64_UBFMWri
	AARCH64_INS_ALIAS_UXTB, // Real instr.: AARCH64_UBFMWri
	AARCH64_INS_ALIAS_UXTH, // Real instr.: AARCH64_UBFMWri
	AARCH64_INS_ALIAS_UXTW, // Real instr.: AARCH64_UBFMXri
	AARCH64_INS_ALIAS_CSET, // Real instr.: AARCH64_CSINCWr
	AARCH64_INS_ALIAS_CSETM, // Real instr.: AARCH64_CSINVWr
	AARCH64_INS_ALIAS_CINC, // Real instr.: AARCH64_CSINCWr
	AARCH64_INS_ALIAS_CINV, // Real instr.: AARCH64_CSINVWr
	AARCH64_INS_ALIAS_CNEG, // Real instr.: AARCH64_CSNEGWr
	AARCH64_INS_ALIAS_RET, // Real instr.: AARCH64_RET
	AARCH64_INS_ALIAS_DCPS1, // Real instr.: AARCH64_DCPS1
	AARCH64_INS_ALIAS_DCPS2, // Real instr.: AARCH64_DCPS2
	AARCH64_INS_ALIAS_DCPS3, // Real instr.: AARCH64_DCPS3
	AARCH64_INS_ALIAS_LDPSW, // Real instr.: AARCH64_LDPSWi
	AARCH64_INS_ALIAS_LDRSH, // Real instr.: AARCH64_LDRSHWroX
	AARCH64_INS_ALIAS_LDRSB, // Real instr.: AARCH64_LDRSBWroX
	AARCH64_INS_ALIAS_LDRSW, // Real instr.: AARCH64_LDRSWroX
	AARCH64_INS_ALIAS_LDURH, // Real instr.: AARCH64_LDURHHi
	AARCH64_INS_ALIAS_LDURB, // Real instr.: AARCH64_LDURBBi
	AARCH64_INS_ALIAS_LDURSH, // Real instr.: AARCH64_LDURSHWi
	AARCH64_INS_ALIAS_LDURSB, // Real instr.: AARCH64_LDURSBWi
	AARCH64_INS_ALIAS_LDURSW, // Real instr.: AARCH64_LDURSWi
	AARCH64_INS_ALIAS_LDTRH, // Real instr.: AARCH64_LDTRHi
	AARCH64_INS_ALIAS_LDTRB, // Real instr.: AARCH64_LDTRBi
	AARCH64_INS_ALIAS_LDTRSH, // Real instr.: AARCH64_LDTRSHWi
	AARCH64_INS_ALIAS_LDTRSB, // Real instr.: AARCH64_LDTRSBWi
	AARCH64_INS_ALIAS_LDTRSW, // Real instr.: AARCH64_LDTRSWi
	AARCH64_INS_ALIAS_STP, // Real instr.: AARCH64_STPWi
	AARCH64_INS_ALIAS_STURH, // Real instr.: AARCH64_STURHHi
	AARCH64_INS_ALIAS_STURB, // Real instr.: AARCH64_STURBBi
	AARCH64_INS_ALIAS_STLURH, // Real instr.: AARCH64_STLURHi
	AARCH64_INS_ALIAS_LDAPURSB, // Real instr.: AARCH64_LDAPURSBWi
	AARCH64_INS_ALIAS_LDAPURH, // Real instr.: AARCH64_LDAPURHi
	AARCH64_INS_ALIAS_LDAPURSH, // Real instr.: AARCH64_LDAPURSHWi
	AARCH64_INS_ALIAS_LDAPURSW, // Real instr.: AARCH64_LDAPURSWi
	AARCH64_INS_ALIAS_STTRH, // Real instr.: AARCH64_STTRHi
	AARCH64_INS_ALIAS_STTRB, // Real instr.: AARCH64_STTRBi
	AARCH64_INS_ALIAS_BIC_4H, // Real instr.: AARCH64_BICv4i16
	AARCH64_INS_ALIAS_BIC_8H, // Real instr.: AARCH64_BICv8i16
	AARCH64_INS_ALIAS_BIC_2S, // Real instr.: AARCH64_BICv2i32
	AARCH64_INS_ALIAS_BIC_4S, // Real instr.: AARCH64_BICv4i32
	AARCH64_INS_ALIAS_ORR_4H, // Real instr.: AARCH64_ORRv4i16
	AARCH64_INS_ALIAS_ORR_8H, // Real instr.: AARCH64_ORRv8i16
	AARCH64_INS_ALIAS_ORR_2S, // Real instr.: AARCH64_ORRv2i32
	AARCH64_INS_ALIAS_ORR_4S, // Real instr.: AARCH64_ORRv4i32
	AARCH64_INS_ALIAS_SXTL_8H, // Real instr.: AARCH64_SSHLLv8i8_shift
	AARCH64_INS_ALIAS_SXTL, // Real instr.: AARCH64_SSHLLv8i8_shift
	AARCH64_INS_ALIAS_SXTL_4S, // Real instr.: AARCH64_SSHLLv4i16_shift
	AARCH64_INS_ALIAS_SXTL_2D, // Real instr.: AARCH64_SSHLLv2i32_shift
	AARCH64_INS_ALIAS_SXTL2_8H, // Real instr.: AARCH64_SSHLLv16i8_shift
	AARCH64_INS_ALIAS_SXTL2, // Real instr.: AARCH64_SSHLLv16i8_shift
	AARCH64_INS_ALIAS_SXTL2_4S, // Real instr.: AARCH64_SSHLLv8i16_shift
	AARCH64_INS_ALIAS_SXTL2_2D, // Real instr.: AARCH64_SSHLLv4i32_shift
	AARCH64_INS_ALIAS_UXTL_8H, // Real instr.: AARCH64_USHLLv8i8_shift
	AARCH64_INS_ALIAS_UXTL, // Real instr.: AARCH64_USHLLv8i8_shift
	AARCH64_INS_ALIAS_UXTL_4S, // Real instr.: AARCH64_USHLLv4i16_shift
	AARCH64_INS_ALIAS_UXTL_2D, // Real instr.: AARCH64_USHLLv2i32_shift
	AARCH64_INS_ALIAS_UXTL2_8H, // Real instr.: AARCH64_USHLLv16i8_shift
	AARCH64_INS_ALIAS_UXTL2, // Real instr.: AARCH64_USHLLv16i8_shift
	AARCH64_INS_ALIAS_UXTL2_4S, // Real instr.: AARCH64_USHLLv8i16_shift
	AARCH64_INS_ALIAS_UXTL2_2D, // Real instr.: AARCH64_USHLLv4i32_shift
	AARCH64_INS_ALIAS_LD2, // Real instr.: AARCH64_LD2Twov16b_POST
	AARCH64_INS_ALIAS_LD3, // Real instr.: AARCH64_LD3Threev16b_POST
	AARCH64_INS_ALIAS_LD4, // Real instr.: AARCH64_LD4Fourv16b_POST
	AARCH64_INS_ALIAS_ST1, // Real instr.: AARCH64_ST1Onev16b_POST
	AARCH64_INS_ALIAS_ST2, // Real instr.: AARCH64_ST2Twov16b_POST
	AARCH64_INS_ALIAS_ST3, // Real instr.: AARCH64_ST3Threev16b_POST
	AARCH64_INS_ALIAS_ST4, // Real instr.: AARCH64_ST4Fourv16b_POST
	AARCH64_INS_ALIAS_LD2R, // Real instr.: AARCH64_LD2Rv8b_POST
	AARCH64_INS_ALIAS_LD3R, // Real instr.: AARCH64_LD3Rv8b_POST
	AARCH64_INS_ALIAS_LD4R, // Real instr.: AARCH64_LD4Rv8b_POST
	AARCH64_INS_ALIAS_CLRBHB, // Real instr.: AARCH64_HINT
	AARCH64_INS_ALIAS_STILP, // Real instr.: AARCH64_STILPW
	AARCH64_INS_ALIAS_STL1, // Real instr.: AARCH64_STL1
	AARCH64_INS_ALIAS_SYSP, // Real instr.: AARCH64_SYSPxt_XZR
	AARCH64_INS_ALIAS_LD1SW, // Real instr.: AARCH64_LD1SW_D_IMM
	AARCH64_INS_ALIAS_LD1H, // Real instr.: AARCH64_LD1H_IMM
	AARCH64_INS_ALIAS_LD1SH, // Real instr.: AARCH64_LD1SH_D_IMM
	AARCH64_INS_ALIAS_LD1D, // Real instr.: AARCH64_LD1D_IMM
	AARCH64_INS_ALIAS_LD1RSW, // Real instr.: AARCH64_LD1RSW_IMM
	AARCH64_INS_ALIAS_LD1RH, // Real instr.: AARCH64_LD1RH_IMM
	AARCH64_INS_ALIAS_LD1RSH, // Real instr.: AARCH64_LD1RSH_D_IMM
	AARCH64_INS_ALIAS_LD1RW, // Real instr.: AARCH64_LD1RW_IMM
	AARCH64_INS_ALIAS_LD1RSB, // Real instr.: AARCH64_LD1RSB_D_IMM
	AARCH64_INS_ALIAS_LD1RD, // Real instr.: AARCH64_LD1RD_IMM
	AARCH64_INS_ALIAS_LD1RQH, // Real instr.: AARCH64_LD1RQ_H_IMM
	AARCH64_INS_ALIAS_LD1RQW, // Real instr.: AARCH64_LD1RQ_W_IMM
	AARCH64_INS_ALIAS_LD1RQD, // Real instr.: AARCH64_LD1RQ_D_IMM
	AARCH64_INS_ALIAS_LDNF1SW, // Real instr.: AARCH64_LDNF1SW_D_IMM_REAL
	AARCH64_INS_ALIAS_LDNF1H, // Real instr.: AARCH64_LDNF1H_IMM_REAL
	AARCH64_INS_ALIAS_LDNF1SH, // Real instr.: AARCH64_LDNF1SH_D_IMM_REAL
	AARCH64_INS_ALIAS_LDNF1W, // Real instr.: AARCH64_LDNF1W_IMM_REAL
	AARCH64_INS_ALIAS_LDNF1SB, // Real instr.: AARCH64_LDNF1SB_D_IMM_REAL
	AARCH64_INS_ALIAS_LDNF1D, // Real instr.: AARCH64_LDNF1D_IMM_REAL
	AARCH64_INS_ALIAS_LDFF1SW, // Real instr.: AARCH64_LDFF1SW_D_REAL
	AARCH64_INS_ALIAS_LDFF1H, // Real instr.: AARCH64_LDFF1H_REAL
	AARCH64_INS_ALIAS_LDFF1SH, // Real instr.: AARCH64_LDFF1SH_D_REAL
	AARCH64_INS_ALIAS_LDFF1W, // Real instr.: AARCH64_LDFF1W_REAL
	AARCH64_INS_ALIAS_LDFF1SB, // Real instr.: AARCH64_LDFF1SB_D_REAL
	AARCH64_INS_ALIAS_LDFF1D, // Real instr.: AARCH64_LDFF1D_REAL
	AARCH64_INS_ALIAS_LD3B, // Real instr.: AARCH64_LD3B_IMM
	AARCH64_INS_ALIAS_LD4B, // Real instr.: AARCH64_LD4B_IMM
	AARCH64_INS_ALIAS_LD2H, // Real instr.: AARCH64_LD2H_IMM
	AARCH64_INS_ALIAS_LD3H, // Real instr.: AARCH64_LD3H_IMM
	AARCH64_INS_ALIAS_LD4H, // Real instr.: AARCH64_LD4H_IMM
	AARCH64_INS_ALIAS_LD2W, // Real instr.: AARCH64_LD2W_IMM
	AARCH64_INS_ALIAS_LD3W, // Real instr.: AARCH64_LD3W_IMM
	AARCH64_INS_ALIAS_LD4W, // Real instr.: AARCH64_LD4W_IMM
	AARCH64_INS_ALIAS_LD2D, // Real instr.: AARCH64_LD2D_IMM
	AARCH64_INS_ALIAS_LD3D, // Real instr.: AARCH64_LD3D_IMM
	AARCH64_INS_ALIAS_LD4D, // Real instr.: AARCH64_LD4D_IMM
	AARCH64_INS_ALIAS_LD2Q, // Real instr.: AARCH64_LD2Q_IMM
	AARCH64_INS_ALIAS_LD3Q, // Real instr.: AARCH64_LD3Q_IMM
	AARCH64_INS_ALIAS_LD4Q, // Real instr.: AARCH64_LD4Q_IMM
	AARCH64_INS_ALIAS_LDNT1H, // Real instr.: AARCH64_LDNT1H_ZRI
	AARCH64_INS_ALIAS_LDNT1W, // Real instr.: AARCH64_LDNT1W_ZRI
	AARCH64_INS_ALIAS_LDNT1D, // Real instr.: AARCH64_LDNT1D_ZRI
	AARCH64_INS_ALIAS_ST1H, // Real instr.: AARCH64_ST1H_IMM
	AARCH64_INS_ALIAS_ST1W, // Real instr.: AARCH64_ST1W_IMM
	AARCH64_INS_ALIAS_ST1D, // Real instr.: AARCH64_ST1D_IMM
	AARCH64_INS_ALIAS_ST3B, // Real instr.: AARCH64_ST3B_IMM
	AARCH64_INS_ALIAS_ST4B, // Real instr.: AARCH64_ST4B_IMM
	AARCH64_INS_ALIAS_ST2H, // Real instr.: AARCH64_ST2H_IMM
	AARCH64_INS_ALIAS_ST3H, // Real instr.: AARCH64_ST3H_IMM
	AARCH64_INS_ALIAS_ST4H, // Real instr.: AARCH64_ST4H_IMM
	AARCH64_INS_ALIAS_ST2W, // Real instr.: AARCH64_ST2W_IMM
	AARCH64_INS_ALIAS_ST3W, // Real instr.: AARCH64_ST3W_IMM
	AARCH64_INS_ALIAS_ST4W, // Real instr.: AARCH64_ST4W_IMM
	AARCH64_INS_ALIAS_ST2D, // Real instr.: AARCH64_ST2D_IMM
	AARCH64_INS_ALIAS_ST3D, // Real instr.: AARCH64_ST3D_IMM
	AARCH64_INS_ALIAS_ST4D, // Real instr.: AARCH64_ST4D_IMM
	AARCH64_INS_ALIAS_ST3Q, // Real instr.: AARCH64_ST3Q_IMM
	AARCH64_INS_ALIAS_ST4Q, // Real instr.: AARCH64_ST4Q_IMM
	AARCH64_INS_ALIAS_STNT1H, // Real instr.: AARCH64_STNT1H_ZRI
	AARCH64_INS_ALIAS_STNT1W, // Real instr.: AARCH64_STNT1W_ZRI
	AARCH64_INS_ALIAS_STNT1D, // Real instr.: AARCH64_STNT1D_ZRI
	AARCH64_INS_ALIAS_PRFH, // Real instr.: AARCH64_PRFH_PRI
	AARCH64_INS_ALIAS_PRFW, // Real instr.: AARCH64_PRFW_PRI
	AARCH64_INS_ALIAS_PRFD, // Real instr.: AARCH64_PRFD_PRI
	AARCH64_INS_ALIAS_CNTH, // Real instr.: AARCH64_CNTH_XPiI
	AARCH64_INS_ALIAS_CNTW, // Real instr.: AARCH64_CNTW_XPiI
	AARCH64_INS_ALIAS_CNTD, // Real instr.: AARCH64_CNTD_XPiI
	AARCH64_INS_ALIAS_DECB, // Real instr.: AARCH64_DECB_XPiI
	AARCH64_INS_ALIAS_INCH, // Real instr.: AARCH64_INCH_XPiI
	AARCH64_INS_ALIAS_DECH, // Real instr.: AARCH64_DECH_XPiI
	AARCH64_INS_ALIAS_INCW, // Real instr.: AARCH64_INCW_XPiI
	AARCH64_INS_ALIAS_DECW, // Real instr.: AARCH64_DECW_XPiI
	AARCH64_INS_ALIAS_INCD, // Real instr.: AARCH64_INCD_XPiI
	AARCH64_INS_ALIAS_DECD, // Real instr.: AARCH64_DECD_XPiI
	AARCH64_INS_ALIAS_SQDECB, // Real instr.: AARCH64_SQDECB_XPiWdI
	AARCH64_INS_ALIAS_UQDECB, // Real instr.: AARCH64_UQDECB_WPiI
	AARCH64_INS_ALIAS_UQINCH, // Real instr.: AARCH64_UQINCH_WPiI
	AARCH64_INS_ALIAS_SQDECH, // Real instr.: AARCH64_SQDECH_XPiWdI
	AARCH64_INS_ALIAS_UQDECH, // Real instr.: AARCH64_UQDECH_WPiI
	AARCH64_INS_ALIAS_SQINCW, // Real instr.: AARCH64_SQINCW_XPiWdI
	AARCH64_INS_ALIAS_UQINCW, // Real instr.: AARCH64_UQINCW_WPiI
	AARCH64_INS_ALIAS_SQDECW, // Real instr.: AARCH64_SQDECW_XPiWdI
	AARCH64_INS_ALIAS_UQDECW, // Real instr.: AARCH64_UQDECW_WPiI
	AARCH64_INS_ALIAS_SQINCD, // Real instr.: AARCH64_SQINCD_XPiWdI
	AARCH64_INS_ALIAS_UQINCD, // Real instr.: AARCH64_UQINCD_WPiI
	AARCH64_INS_ALIAS_SQDECD, // Real instr.: AARCH64_SQDECD_XPiWdI
	AARCH64_INS_ALIAS_UQDECD, // Real instr.: AARCH64_UQDECD_WPiI
	AARCH64_INS_ALIAS_MOVS, // Real instr.: AARCH64_ORRS_PPzPP
	AARCH64_INS_ALIAS_NOT, // Real instr.: AARCH64_EOR_PPzPP
	AARCH64_INS_ALIAS_NOTS, // Real instr.: AARCH64_EORS_PPzPP
	AARCH64_INS_ALIAS_LD1ROH, // Real instr.: AARCH64_LD1RO_H_IMM
	AARCH64_INS_ALIAS_LD1ROW, // Real instr.: AARCH64_LD1RO_W_IMM
	AARCH64_INS_ALIAS_LD1ROD, // Real instr.: AARCH64_LD1RO_D_IMM
	AARCH64_INS_ALIAS_BCAX, // Real instr.: AARCH64_BCAX_ZZZZ
	AARCH64_INS_ALIAS_BSL, // Real instr.: AARCH64_BSL_ZZZZ
	AARCH64_INS_ALIAS_BSL1N, // Real instr.: AARCH64_BSL1N_ZZZZ
	AARCH64_INS_ALIAS_BSL2N, // Real instr.: AARCH64_BSL2N_ZZZZ
	AARCH64_INS_ALIAS_NBSL, // Real instr.: AARCH64_NBSL_ZZZZ
	AARCH64_INS_ALIAS_LDNT1SH, // Real instr.: AARCH64_LDNT1SH_ZZR_S_REAL
	AARCH64_INS_ALIAS_LDNT1SW, // Real instr.: AARCH64_LDNT1SW_ZZR_D_REAL

	// clang-format on
	// generated content <AArch64GenCSAliasEnum.inc> end

	// Hardcoded in LLVM printer
	AARCH64_INS_ALIAS_CFP,
	AARCH64_INS_ALIAS_DVP,
	AARCH64_INS_ALIAS_COSP,
	AARCH64_INS_ALIAS_CPP,
	AARCH64_INS_ALIAS_IC,
	AARCH64_INS_ALIAS_DC,
	AARCH64_INS_ALIAS_AT,
	AARCH64_INS_ALIAS_TLBI,
	AARCH64_INS_ALIAS_TLBIP,
	AARCH64_INS_ALIAS_RPRFM,
	AARCH64_INS_ALIAS_LSL,
	AARCH64_INS_ALIAS_SBFX,
	AARCH64_INS_ALIAS_UBFX,
	AARCH64_INS_ALIAS_SBFIZ,
	AARCH64_INS_ALIAS_UBFIZ,
	AARCH64_INS_ALIAS_BFC,
	AARCH64_INS_ALIAS_BFI,
	AARCH64_INS_ALIAS_BFXIL,

	AARCH64_INS_ALIAS_END,
} aarch64_insn;

/// Group of AArch64 instructions
typedef enum aarch64_insn_group {
  AARCH64_GRP_INVALID = 0, ///< = CS_GRP_INVALID

  // Generic groups
  // all jump instructions (conditional+direct+indirect jumps)
  AARCH64_GRP_JUMP, ///< = CS_GRP_JUMP
  AARCH64_GRP_CALL,
  AARCH64_GRP_RET,
  AARCH64_GRP_INT,
  AARCH64_GRP_PRIVILEGE = 6,   ///< = CS_GRP_PRIVILEGE
  AARCH64_GRP_BRANCH_RELATIVE, ///< = CS_GRP_BRANCH_RELATIVE
	// generated content <AArch64GenCSFeatureEnum.inc> begin
	// clang-format off

	AARCH64_FEATURE_HASV8_0A = 128,
	AARCH64_FEATURE_HASV8_1A,
	AARCH64_FEATURE_HASV8_2A,
	AARCH64_FEATURE_HASV8_3A,
	AARCH64_FEATURE_HASV8_4A,
	AARCH64_FEATURE_HASV8_5A,
	AARCH64_FEATURE_HASV8_6A,
	AARCH64_FEATURE_HASV8_7A,
	AARCH64_FEATURE_HASV8_8A,
	AARCH64_FEATURE_HASV8_9A,
	AARCH64_FEATURE_HASV9_0A,
	AARCH64_FEATURE_HASV9_1A,
	AARCH64_FEATURE_HASV9_2A,
	AARCH64_FEATURE_HASV9_3A,
	AARCH64_FEATURE_HASV9_4A,
	AARCH64_FEATURE_HASV8_0R,
	AARCH64_FEATURE_HASEL2VMSA,
	AARCH64_FEATURE_HASEL3,
	AARCH64_FEATURE_HASVH,
	AARCH64_FEATURE_HASLOR,
	AARCH64_FEATURE_HASPAUTH,
	AARCH64_FEATURE_HASPAUTHLR,
	AARCH64_FEATURE_HASJS,
	AARCH64_FEATURE_HASCCIDX,
	AARCH64_FEATURE_HASCOMPLXNUM,
	AARCH64_FEATURE_HASNV,
	AARCH64_FEATURE_HASMPAM,
	AARCH64_FEATURE_HASDIT,
	AARCH64_FEATURE_HASTRACEV8_4,
	AARCH64_FEATURE_HASAM,
	AARCH64_FEATURE_HASSEL2,
	AARCH64_FEATURE_HASTLB_RMI,
	AARCH64_FEATURE_HASFLAGM,
	AARCH64_FEATURE_HASRCPC_IMMO,
	AARCH64_FEATURE_HASFPARMV8,
	AARCH64_FEATURE_HASNEON,
	AARCH64_FEATURE_HASSM4,
	AARCH64_FEATURE_HASSHA3,
	AARCH64_FEATURE_HASSHA2,
	AARCH64_FEATURE_HASAES,
	AARCH64_FEATURE_HASDOTPROD,
	AARCH64_FEATURE_HASCRC,
	AARCH64_FEATURE_HASCSSC,
	AARCH64_FEATURE_HASLSE,
	AARCH64_FEATURE_HASRAS,
	AARCH64_FEATURE_HASRDM,
	AARCH64_FEATURE_HASFULLFP16,
	AARCH64_FEATURE_HASFP16FML,
	AARCH64_FEATURE_HASSPE,
	AARCH64_FEATURE_HASFUSEAES,
	AARCH64_FEATURE_HASSVE,
	AARCH64_FEATURE_HASSVE2,
	AARCH64_FEATURE_HASSVE2P1,
	AARCH64_FEATURE_HASSVE2AES,
	AARCH64_FEATURE_HASSVE2SM4,
	AARCH64_FEATURE_HASSVE2SHA3,
	AARCH64_FEATURE_HASSVE2BITPERM,
	AARCH64_FEATURE_HASB16B16,
	AARCH64_FEATURE_HASSME,
	AARCH64_FEATURE_HASSMEF64F64,
	AARCH64_FEATURE_HASSMEF16F16,
	AARCH64_FEATURE_HASSMEFA64,
	AARCH64_FEATURE_HASSMEI16I64,
	AARCH64_FEATURE_HASSME2,
	AARCH64_FEATURE_HASSME2P1,
	AARCH64_FEATURE_HASFPMR,
	AARCH64_FEATURE_HASFP8,
	AARCH64_FEATURE_HASFAMINMAX,
	AARCH64_FEATURE_HASFP8FMA,
	AARCH64_FEATURE_HASSSVE_FP8FMA,
	AARCH64_FEATURE_HASFP8DOT2,
	AARCH64_FEATURE_HASSSVE_FP8DOT2,
	AARCH64_FEATURE_HASFP8DOT4,
	AARCH64_FEATURE_HASSSVE_FP8DOT4,
	AARCH64_FEATURE_HASLUT,
	AARCH64_FEATURE_HASSME_LUTV2,
	AARCH64_FEATURE_HASSMEF8F16,
	AARCH64_FEATURE_HASSMEF8F32,
	AARCH64_FEATURE_HASSVEORSME,
	AARCH64_FEATURE_HASSVE2ORSME,
	AARCH64_FEATURE_HASSVE2ORSME2,
	AARCH64_FEATURE_HASSVE2P1_OR_HASSME,
	AARCH64_FEATURE_HASSVE2P1_OR_HASSME2,
	AARCH64_FEATURE_HASSVE2P1_OR_HASSME2P1,
	AARCH64_FEATURE_HASNEONORSME,
	AARCH64_FEATURE_HASRCPC,
	AARCH64_FEATURE_HASALTNZCV,
	AARCH64_FEATURE_HASFRINT3264,
	AARCH64_FEATURE_HASSB,
	AARCH64_FEATURE_HASPREDRES,
	AARCH64_FEATURE_HASCCDP,
	AARCH64_FEATURE_HASBTI,
	AARCH64_FEATURE_HASMTE,
	AARCH64_FEATURE_HASTME,
	AARCH64_FEATURE_HASETE,
	AARCH64_FEATURE_HASTRBE,
	AARCH64_FEATURE_HASBF16,
	AARCH64_FEATURE_HASMATMULINT8,
	AARCH64_FEATURE_HASMATMULFP32,
	AARCH64_FEATURE_HASMATMULFP64,
	AARCH64_FEATURE_HASXS,
	AARCH64_FEATURE_HASWFXT,
	AARCH64_FEATURE_HASLS64,
	AARCH64_FEATURE_HASBRBE,
	AARCH64_FEATURE_HASSPE_EEF,
	AARCH64_FEATURE_HASHBC,
	AARCH64_FEATURE_HASMOPS,
	AARCH64_FEATURE_HASCLRBHB,
	AARCH64_FEATURE_HASSPECRES2,
	AARCH64_FEATURE_HASITE,
	AARCH64_FEATURE_HASTHE,
	AARCH64_FEATURE_HASRCPC3,
	AARCH64_FEATURE_HASLSE128,
	AARCH64_FEATURE_HASD128,
	AARCH64_FEATURE_HASCHK,
	AARCH64_FEATURE_HASGCS,
	AARCH64_FEATURE_HASCPA,
	AARCH64_FEATURE_USENEGATIVEIMMEDIATES,
	AARCH64_FEATURE_HASCCPP,
	AARCH64_FEATURE_HASPAN,
	AARCH64_FEATURE_HASPSUAO,
	AARCH64_FEATURE_HASPAN_RWV,
	AARCH64_FEATURE_HASCONTEXTIDREL2,

	// clang-format on
	// generated content <AArch64GenCSFeatureEnum.inc> end

  AARCH64_GRP_ENDING, // <-- mark the end of the list of groups
} aarch64_insn_group;

#ifdef __cplusplus
}
#endif

#endif
