; SV - Symbolic Vector Hardware Analysis Framework
; Copyright (C) 2014-2015 Centaur Technology
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; License: (An MIT/X11-style license)
;
;   Permission is hereby granted, free of charge, to any person obtaining a
;   copy of this software and associated documentation files (the "Software"),
;   to deal in the Software without restriction, including without limitation
;   the rights to use, copy, modify, merge, publish, distribute, sublicense,
;   and/or sell copies of the Software, and to permit persons to whom the
;   Software is furnished to do so, subject to the following conditions:
;
;   The above copyright notice and this permission notice shall be included in
;   all copies or substantial portions of the Software.
;
;   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
;   DEALINGS IN THE SOFTWARE.
;
; Original authors: Sol Swords <sswords@centtech.com>

(in-package "SV")

;; (include-book "debug")
(include-book "eval-phases")
(include-book "probe")
(include-book "fsm-obj")
(include-book "../svex/rewrite")
(include-book "../mods/moddb")
(include-book "../mods/path-string")
(include-book "std/strings/hexify" :dir :system)
(include-book "expand")
(include-book "../svex/env-ops")
(include-book "../svex/override-types")
(include-book "centaur/misc/hons-extra" :Dir :system)
(local (in-theory (disable (tau-system))))

;; (include-book "std/stobjs/updater-independence" :dir :system)
;; This tool operates from an initialized DEBUGDATA stobj, as generated by
;; svtv-debug-core, or (without dumping a vcd) by svtv-debug-init followed by
;; svtv-debug-set-svtv.

;; It takes an input alist and expands it to an environment for each phase and
;; initial state so that we can evaluate a signal at a given phase.

;; Then we start from some given signal (an output/internal name or path and
;; phase).  We can evaluate that signal using svex-eval-svtv-phases.


(defprod chase-position
  ((path path-p)
   ;; Integerp largely because it's occasionally convenient to have a phase of -1 to denote initial states.
   (phase integerp :rule-classes :type-prescription)
   (rsh natp :rule-classes :type-prescription)
   (mask 4vmask-p)))


(deflist chase-stack :elt-type chase-position :true-listp t)


(defprod svtv-chase-evaldata
  ((evaldata svtv-evaldata :default (make-svtv-evaldata))
   (updates svex-alist-p))
  :layout :list
  :extra-binder-names (nextstate inputs initst)
  ///
  (define svtv-chase-evaldata->nextstate ((x svtv-chase-evaldata-p))
    :enabled t
    (svtv-evaldata->nextstate (svtv-chase-evaldata->evaldata x)))

  (define svtv-chase-evaldata->inputs ((x svtv-chase-evaldata-p))
    :enabled t
    (svtv-evaldata->inputs (svtv-chase-evaldata->evaldata x)))

  (define svtv-chase-evaldata->initst ((x svtv-chase-evaldata-p))
    :enabled t
    (svtv-evaldata->initst (svtv-chase-evaldata->evaldata x))))


(local (defun svtv-chase-data-renaming (field-names)
         (b* (((when (atom field-names)) nil)
              (field (car field-names))
              (new-field (intern$ (cat "SVTV-CHASE-DATA->" (symbol-name field)) "SV"))
              (update (intern$ (cat "UPDATE-" (symbol-name field)) "SV"))
              (new-update (intern$ (cat "SET-SVTV-CHASE-DATA->" (symbol-name field)) "SV"))
              (pred (intern$ (cat (symbol-name field) "P") "SV"))
              (new-pred (intern$ (cat "SVTV-CHASE-DATA->" (symbol-name field) "P") "SV")))
           (cons (list field new-field)
                 (cons (list update new-update)
                       (cons (list pred new-pred)
                             (svtv-chase-data-renaming (cdr field-names))))))))

(fty::defoption maybe-svtv-chase-evaldata svtv-chase-evaldata-p)



(make-event
 (b* ((fields
       `((stack :type (satisfies chase-stack-p) :initially nil)
         (sigtype :type symbol)
         (vars :type (satisfies 4vmask-alist-p))
         (expr :type (satisfies svex-p) :initially ,(svex-x))
         ;; (new-phase :type (integer 0 *) :initially 0)
         (evaldata :type (satisfies svtv-chase-evaldata-p) :initially ,(make-svtv-chase-evaldata))
         (evaldata2 :type (satisfies maybe-svtv-chase-evaldata-p) :initially nil)
         (data2-offset :type integer :initially 0)
         (smartp :initially t)
         (phaselabels :type (satisfies symbol-listp))
         ;; (updates :type (satisfies svex-alist-p))
         (delays :type (satisfies svex-alist-p))
         (assigns :type (satisfies svex-alist-p))
         (override-alist :type (satisfies svex-alist-p))
         (modidx :type (integer 0 *) :initially 0)
         (probes :type (satisfies svtv-probealist-p))
         (namemap :type (satisfies svtv-name-lhs-map-p))
         (print-with-mask-mode :type symbol :initially :default)
         (print-overrides-mode :type symbol :initially nil)))
      (field-names (strip-cars fields))
      (renaming (svtv-chase-data-renaming field-names))
      ;; (fns (append '(debugdatap create-debugdata)
      ;;              (acl2::strip-cadrs renaming)))
      (make-binder (std::da-make-binder 'svtv-chase-data field-names)))
   
   `(progn
      (defstobj svtv-chase-data
        ,@fields
        :renaming ,renaming)
      (in-theory (disable create-svtv-chase-data svtv-chase-datap))
      ,make-binder)))





;; There are basically three types of signals/svex variables used here:
;;  - If V is a key of nextstates/delays, then it is a previous state signal.
;;  - If V is a key of updates/assigns, then it is an internal signal.
;;  - If V appears in an expression (value) of updates or nextstates but is not
;;  a key of either one, then it is a primary input.

;;    A previous state signal cannot also be an internal signal (a key of
;;    nextstates cannot also be a key of updates).  But each V that is a
;;    previous state/key of updates is also a key of delays, and the
;;    corresponding value in delays is an internal signal or primary input that
;;    is also the nextstate of V.

;; The keys of nextstates should be the same as those of delays.  The keys of
;; updates should be the same as those of assigns.


;; We want to support exploring the design by walking through following
;; dependencies back to drivers. Generally we'll have a stack starting from
;; some signal and going back though its drivers (across phases).

;; The current position will be tracked as a variable, phase, right-shift, and
;; mask.  The mask is relative to the right-shift, so the caremask for the
;; whole variable is mask << right-shift.

;; We'll find dependencies of a variable modulo the caremask, and the resulting
;; dependency variables will be reported along with their caremasks.
;; Internally we'll use right-shifts and masks, but when we print things out to
;; the user we'll want to translate back to relative indices.  E.g. if a
;; variable is declared as x[10:2] and we have a rightshift of 3 and mask
;; #b101, we'll show the range as x[7:5].


(define svtv-chase-normalize-var/phase ((var svar-p)
                                        (phase integerp))
  :returns (mv (new-var svar-p)
               (new-phase natp :rule-classes :type-prescription))
  (b* (((svar var))
       (eff-delay (min var.delay (lifix phase)))
       (new-delay (- var.delay eff-delay)))
    (mv (change-svar var :delay new-delay) (- (lifix phase) eff-delay)))
  ///
  ;; (defret svar-addr-p-of-<fn>
  ;;   (implies (svar-addr-p var)
  ;;            (svar-addr-p new-var))
  ;;   :hints(("Goal" :in-theory (enable svar-addr-p))))
  )

(define svtv-chase-eval-override ((var svar-p)
                                  (phase integerp)
                                  (evaldata svtv-chase-evaldata-p)
                                  &key
                                  (svtv-chase-data 'svtv-chase-data))
  :returns (mv (override-mask integerp)
               (override-val 4vec-p)
               (computed-val 4vec-p))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  (b* (((mv var phase) (svtv-chase-normalize-var/phase var phase))
       ((svtv-chase-data svtv-chase-data))
       ((svtv-chase-evaldata evaldata))
       ((when (< 0 (svar->delay var)))
        (b* ((val (svex-env-lookup var evaldata.initst)))
          (mv 0 0 val)))
       (override-mux (svex-fastlookup var svtv-chase-data.override-alist))
       (computed-expr (svex-compose (svex-var var) evaldata.updates))
       (computed-val (svex-eval-svtv-phases computed-expr phase evaldata.evaldata))
       ((unless override-mux)
        (mv 0 0 computed-val))
       (override-test-expr (svex-var (svar-change-override var :test)))
       (override-val-expr (svex-var (svar-change-override var :val))))
    (mv (4vec-1mask (svex-eval-svtv-phases override-test-expr phase evaldata.evaldata))
        (svex-eval-svtv-phases override-val-expr phase evaldata.evaldata)
        computed-val)))
  
(define svtv-chase-eval ((var svar-p)
                         (phase integerp)
                         (evaldata svtv-chase-evaldata-p)
                         &key
                         (svtv-chase-data 'svtv-chase-data))
  :returns (val 4vec-p)
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  (b* (((mv var phase) (svtv-chase-normalize-var/phase var phase))
       ((svtv-chase-data svtv-chase-data))
       ((svtv-chase-evaldata evaldata))
       ((when (< 0 (svar->delay var)))
        (svex-env-lookup var evaldata.initst))
       (override-mux (svex-fastlookup var svtv-chase-data.override-alist))
       (overridden-val (svex-compose (or override-mux (svex-var var)) evaldata.updates)))
    (svex-eval-svtv-phases overridden-val phase evaldata.evaldata)))


(define svtv-chase-evallist ((vars svarlist-p)
                             (phase integerp)
                             (evaldata svtv-chase-evaldata-p)
                             &key
                             (svtv-chase-data 'svtv-chase-data))
  :returns (vals 4veclist-p)
  (if (atom vars)
      nil
    (cons (svtv-chase-eval (car vars) phase evaldata)
          (svtv-chase-evallist (cdr vars) phase evaldata)))
  ///
  (defret len-of-<fn>
    (equal (len vals) (len vars))))
       


(define svex-mask-alist-to-4vmask-alist ((x svex-mask-alist-p))
  :returns (new-x 4vmask-alist-p)
  (if (atom x)
      nil
    (if (and (mbt (and (consp (car x)) (svex-p (caar x))))
             (svex-case (caar x) :var)
             (not (sparseint-equal (cdar x) 0)))
        (cons (cons (svex-var->name (caar x))
                    (sparseint-fix (cdar x)))
              (svex-mask-alist-to-4vmask-alist (cdr x)))
      (svex-mask-alist-to-4vmask-alist (cdr x)))))


(local (defthm svarlist-p-alist-keys-when-4vmask-alist-p
         (implies (4vmask-alist-p x)
                  (svarlist-p (alist-keys x)))
         :hints(("Goal" :in-theory (enable alist-keys)))))


;; (local
;;  (defsection svarlist-addr-p-of-svexlist-compute-masks
;;    (defret member-vars-of-svex-args-apply-masks
;;      (implies (and (not (member v (svexlist-vars args)))
;;                    (not (member v (svexlist-vars (svex-mask-alist-keys mask-al)))))
;;               (not (member v (svexlist-vars (svex-mask-alist-keys mask-al1)))))
;;      :hints(("Goal" :in-theory (enable svex-args-apply-masks)))
;;      :fn svex-args-apply-masks)

;;    (defret member-vars-of-svexlist-compute-masks
;;      (implies (and (not (member v (svexlist-vars x)))
;;                    (not (member v (svexlist-vars (svex-mask-alist-keys mask-al)))))
;;               (not (member v (svexlist-vars (svex-mask-alist-keys mask-al1)))))
;;      :hints(("Goal" :in-theory (enable svexlist-compute-masks)))
;;      :fn svexlist-compute-masks)

;;    (defret member-alist-keys-of-svex-mask-alist-to-4vmask-alist
;;      (implies (not (member v (svexlist-vars (svex-mask-alist-keys x))))
;;               (not (member v (alist-keys new-x))))
;;      :hints(("Goal" :in-theory (enable svex-mask-alist-to-4vmask-alist
;;                                        svex-mask-alist-keys
;;                                        alist-keys)))
;;      :fn svex-mask-alist-to-4vmask-alist)))


(local (in-theory (disable fast-alist-clean)))

(local
 (encapsulate nil
   (defthm svex-mask-alist-p-of-fast-alist-fork
     (implies (and (svex-mask-alist-p x)
                   (svex-mask-alist-p y))
              (svex-mask-alist-p (fast-alist-fork x y))))

   (local (defthm cdr-last-when-svex-mask-alist-p
            (implies (svex-mask-alist-p x)
                     (equal (cdr (last x)) nil))))



   (defthm svex-mask-alist-p-of-fast-alist-clean
     (implies (svex-mask-alist-p x)
              (svex-mask-alist-p (fast-alist-clean x)))
     :hints(("Goal" :in-theory (enable fast-alist-clean svex-mask-alist-p))))))



(define svexlist-compute-masks-with-eval ((x svexlist-p)
                                          (mask-al svex-mask-alist-p)
                                          (env svex-env-p))
  :returns (mask-al1 svex-mask-alist-p)
  (b* (((when (atom x))
        (mbe :logic (svex-mask-alist-fix mask-al)
             :exec mask-al))
       (first (car x))
       ((when (not (eq (svex-kind first) :call)))
        (svexlist-compute-masks-with-eval (cdr x) mask-al env))
       (mask (svex-mask-lookup first mask-al))
       ((when (sparseint-equal mask 0))
        (svexlist-compute-masks-with-eval (cdr x) mask-al env))
       (args (svex-call->args first))
       (argvals (4veclist-quote (svexlist-eval args env)))
       (argmasks (svex-argmasks mask
                                (svex-call->fn first)
                                argvals))
       (mask-al (svex-args-apply-masks args argmasks mask-al)))
    (svexlist-compute-masks-with-eval (cdr x) mask-al env)))

(local (defthm true-listp-when-svarlist-p
         (implies (svarlist-p x)
                  (true-listp x))))

(define svtv-chase-expr-deps ((expr svex-p)
                              (phase natp)
                              (rsh natp)
                              (mask 4vmask-p)
                              (evaldata svtv-chase-evaldata-p)
                              &key
                              (svtv-chase-data 'svtv-chase-data))
  :returns (deps svex-mask-alist-p)
  (b* (((mv toposort al) (svex-toposort expr nil nil))
       (- (fast-alist-free al))
       ((svtv-chase-data svtv-chase-data))
       (start-mask-al (svex-mask-acons expr (sparseint-ash mask rsh) nil))
       ((unless svtv-chase-data.smartp)
        (fast-alist-free
         (fast-alist-clean
          (svexlist-compute-masks toposort start-mask-al))))
       (vars (svex-collect-vars expr))
       (env (make-fast-alist (pairlis$ vars (svtv-chase-evallist vars phase evaldata))))
       (mask-al (fast-alist-free
                 (fast-alist-clean
                  (svexlist-compute-masks-with-eval
                   toposort (svex-mask-acons expr (sparseint-ash mask rsh) nil)
                   env)))))
    (fast-alist-free env)
    mask-al))
       


;; Non-X-monotonic
;; Results in a 1 at bits where the inputs differ, 0 where they don't
(define 4vec-bitdiff ((x 4vec-p)
                      (y 4vec-p))
  :returns (bitdiff 4vec-p)
  (b* (((4vec x)) ((4vec y)))
    (2vec (logior (logxor x.upper y.upper)
                  (logxor x.lower y.lower)))))

(define 4veclist-bitdiff ((x 4veclist-p)
                          (y 4veclist-p))
  :guard (eql (len x) (len y))
  :returns (diffs 4veclist-p)
  (if (or (atom x) (atom y))
      nil
    (cons (4vec-bitdiff (car x) (car y))
          (4veclist-bitdiff (cdr x) (cdr y))))
  ///
  (defret len-of-<fn>
    (equal (len diffs)
           (min (len x) (len y)))))


(define svtv-chase-deps ((var svar-p)
                         (phase natp)
                         (rsh natp)
                         (mask 4vmask-p)
                         &key
                         (svtv-chase-data 'svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (mv (type symbolp :rule-classes :type-prescription)
               (vars 4vmask-alist-p)
               (vars2 4vmask-alist-p)
               (diffs svex-env-p)
               (expr svex-p))
  (b* (((svtv-chase-data svtv-chase-data))
       (phase (lnfix phase))
       (var (svar-fix var))
       (type
        (b* ((svex (svex-fastlookup var (svtv-chase-evaldata->updates svtv-chase-data.evaldata)))
             ((when svex)
              :update)
             (prev-var-look (hons-get var (svex-alist-fix svtv-chase-data.delays)))
             ((when prev-var-look)
              (if (eql phase 0)
                  :initst
                :prevst)))
          :input))

       ((when (or (eq type :input)
                  (eq type :initst)))
        (mv type nil nil nil (svex-var var)))

       ((when (eq type :prevst))
        (mv type (list (cons var (sparseint-ash mask rsh))) (list (cons var (sparseint-ash mask rsh)))
            nil ;; diffs?
            (svex-var var)))

       (expr (svex-fastlookup var svtv-chase-data.assigns))

       ((unless expr)
        (mv :error nil nil nil (svex-x)))

       
       
       (mask-al (svtv-chase-expr-deps expr phase rsh mask svtv-chase-data.evaldata))
       ((mv mask-al2 diffs)
        (if svtv-chase-data.evaldata2
            (let ((phase2  (+ phase svtv-chase-data.data2-offset)))
              (if (<= 0 phase2)
                  (mv (svtv-chase-expr-deps expr phase2 rsh mask svtv-chase-data.evaldata2)
                      (b* ((expr-vars (svex-collect-vars expr))
                           (expr-var-vals1 (svtv-chase-evallist expr-vars phase svtv-chase-data.evaldata))
                           (expr-var-vals2 (svtv-chase-evallist expr-vars phase2 svtv-chase-data.evaldata2))
                           (diffs (4veclist-bitdiff expr-var-vals1 expr-var-vals2)))
                        (make-fast-alist (pairlis$ expr-vars diffs))))
                (mv nil nil)))
          (mv nil nil)))
       
       (vars (svex-mask-alist-to-4vmask-alist mask-al))
       (vars2 (svex-mask-alist-to-4vmask-alist mask-al2)))
    (mv type vars vars2 diffs expr))
  ///
  ;; (local (defthm svar-addr-p-lookup-in-svar-map
  ;;          (implies (And (svarlist-addr-p (svar-map-vars x))
  ;;                        (hons-assoc-equal k (svar-map-fix x)))
  ;;                   (svar-addr-p (cdr (hons-assoc-equal k (svar-map-fix x)))))
  ;;          :hints(("Goal" :in-theory (e/d (svar-map-vars svar-map-fix)
  ;;                                         (hons-assoc-equal-of-svar-map-fix))
  ;;                  :induct (svar-map-vars x)))
  ;;          :rule-classes
  ;;          ((:rewrite :corollary
  ;;            (implies (And (svarlist-addr-p (svar-map-vars x))
  ;;                          (svar-p k)
  ;;                          (hons-assoc-equal k x))
  ;;                   (svar-addr-p (cdr (hons-assoc-equal k x))))))))

  ;; (local (defthm member-svex-mask-alist-keys-of-fast-alist-fork
  ;;          (implies (and (not (member v (svexlist-vars (svex-mask-alist-keys x))))
  ;;                        (not (member v (svexlist-vars (svex-mask-alist-keys y)))))
  ;;                   (not (member v (svexlist-vars (svex-mask-alist-keys (fast-alist-fork x y))))))
  ;;          :hints(("Goal" :in-theory (enable svex-mask-alist-keys)))))

  ;; (local (defthm svex-mask-alist-keys-of-atom
  ;;          (implies (atom x)
  ;;                   (equal (svex-mask-alist-keys x) nil))
  ;;          :hints(("Goal" :in-theory (enable svex-mask-alist-keys)))))

  ;; (defret svarlist-addr-p-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (alist-keys vars)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))

  

  ;; (defret svarlist-addr-p-expr-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (svex-vars expr)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))
  )


(define svtv-chase-normalize-masks ((rsh natp) (mask 4vmask-p) (mask2 4vmask-p))
  :returns (mv (new-rsh natp :rule-classes :type-prescription)
               (new-mask 4vmask-p)
               (new-mask2 4vmask-p)
               (union 4vmask-p))
  (b* ((union (sparseint-bitor (4vmask-fix mask) (4vmask-fix mask2)))
       (mask-rsh (sparseint-trailing-0-count union))
       (norm-mask (sparseint-ash (4vmask-fix mask) (- mask-rsh)))
       (norm-mask2 (sparseint-ash (4vmask-fix mask2) (- mask-rsh))))
    (mv (+ mask-rsh (lnfix rsh))
        norm-mask norm-mask2
        (sparseint-ash union (- mask-rsh))))
  ///
  (local (include-book "centaur/bitops/ihsext-basics" :dir :system))
  (local (include-book "arithmetic/top" :dir :system))


  (local (defthm trailing-0-count-of-logior-lte
           (and (implies (not (zip x))
                         (<= (bitops::trailing-0-count (logior x y)) (bitops::trailing-0-count x)))
                (implies (not (zip y))
                         (<= (bitops::trailing-0-count (logior x y)) (bitops::trailing-0-count y))))
           :hints(("Goal" :in-theory (enable* bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs
                                              bitops::ihsext-inductions)
                   :induct (logior x y)))))
  
  (local (defthm ash-of-trailing-0-count
           (equal (ash (logtail (bitops::trailing-0-count x) x)
                       (bitops::trailing-0-count x))
                  (ifix x))
           :hints(("Goal" :in-theory (enable* bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs)))))

  (local (defthm trailing-0-count-of-logtail
           (implies (<= (nfix n) (bitops::trailing-0-count x))
                    (equal (bitops::trailing-0-count (logtail n x))
                           (- (bitops::trailing-0-count x) (nfix n))))
           :hints(("Goal" :in-theory (e/d* (bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs
                                              bitops::ihsext-inductions)
                                           (bitops::logcdr-of-logtail))
                   :induct (logtail n x)))))
  
  (local (defthm trailing-0-count-of-ash
           (implies (<= (- (ifix n)) (bitops::trailing-0-count x))
                    (equal (bitops::trailing-0-count (ash x n))
                           (if (zip x)
                               0
                             (+ (bitops::trailing-0-count x) (ifix n)))))
           :hints(("Goal" :in-theory (enable* bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs
                                              bitops::ihsext-inductions)
                   :induct (ash x n)))))
           

  (local (defthm trailing-0-count-when-logtail-equal-0
           (implies (and (not (zip x))
                         (equal (logtail n x) 0))
                    (< (bitops::trailing-0-count x) (nfix n)))
           :hints(("Goal" :in-theory (enable* bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs
                                              bitops::ihsext-inductions)))
           :rule-classes :linear))
  
  (local (defthm ash-of-trailing-0-count-gen
           (implies (and (natp n)
                         (<= n (bitops::trailing-0-count x)))
                    (equal (ash (logtail n x)
                                n)
                           (ifix x)))
           :hints (("goal" :use ((:instance ash-of-trailing-0-count
                                  (x (ash (logtail n x)
                                          n)))
                                 (:instance ash-of-trailing-0-count
                                  (x x)))
                    :in-theory (disable ash-of-trailing-0-count)))))

  (defret <fn>-correct
    (equal (ash (sparseint-val new-mask) new-rsh)
           (ash (sparseint-val (4vmask-fix mask)) (nfix rsh)))
    :hints (("goal" :in-theory (disable bitops::ash-of-ash)
             :use ((:instance bitops::ash-of-ash
                    (x (logtail (BITOPS::TRAILING-0-COUNT (logior (SPARSEINT-VAL (4vmask-fix MASK))
                                                                  (SPARSEINT-VAL (4vmask-fix MASK2))))
                                (sparseint-val (4vmask-fix mask))))
                    (sh1 (BITOPS::TRAILING-0-COUNT (logior (SPARSEINT-VAL (4vmask-fix MASK))
                                                                  (SPARSEINT-VAL (4vmask-fix MASK2)))))
                    (sh2 (nfix rsh))))
             :cases ((zip (sparseint-val (4vmask-fix mask)))))))

  (defret <fn>-correct2
    (equal (ash (sparseint-val new-mask2) new-rsh)
           (ash (sparseint-val (4vmask-fix mask2)) (nfix rsh)))
    :hints (("goal" :in-theory (disable bitops::ash-of-ash)
             :use ((:instance bitops::ash-of-ash
                    (x (logtail (BITOPS::TRAILING-0-COUNT (logior (SPARSEINT-VAL (4vmask-fix MASK))
                                                                  (SPARSEINT-VAL (4vmask-fix MASK2))))
                                (sparseint-val (4vmask-fix mask2))))
                    (sh1 (BITOPS::TRAILING-0-COUNT (logior (SPARSEINT-VAL (4vmask-fix MASK))
                                                                  (SPARSEINT-VAL (4vmask-fix MASK2)))))
                    (sh2 (nfix rsh))))
             :cases ((zip (sparseint-val (4vmask-fix mask2))))))))

(define svtv-chase-var-declared-width ((var svar-p)
                                       (modidx natp)
                                       &key
                                       ((moddb moddb-ok) 'moddb))
  :returns (width maybe-posp :rule-classes :type-prescription)
  :guard (< modidx (moddb->nmods moddb))
  (b* (((svar var))
       ((unless (address-p var.name)) nil)
       (path (address->path var.name))
       ((mv err wire & &) (moddb-path->wireidx/decl path modidx moddb))
       ((when err) nil)
       ((wire wire)))
    wire.width))


(define svtv-chase-var-name/range ((var svar-p)
                                   (rsh natp)
                                   (mask 4vmask-p)
                                   (modidx natp)
                                   &key
                                   ((moddb moddb-ok) 'moddb))
  :returns (mv (name-range-msg)
               (width maybe-posp :rule-classes :type-prescription))
  :guard (< modidx (moddb->nmods moddb))
  (b* (((svar var))
       (maskwidth (and (not (sparseint-< mask 0))
                       (sparseint-length mask)))
       ((unless (address-p var.name))
        (b* (((when maskwidth)
              (mv (msg "~x0[~x1:~x2]" var.name (1- maskwidth) rsh) (max 1 maskwidth))))
          (cw! "Warning: For non-address variable ~x0, caremask was ~
                unbounded!~%" var)
          (mv (msg "~x0[??:~x1]" var.name rsh) nil)))
       (path (address->path var.name))
       (name (path->string-top path))
       ((mv err wire & &) (moddb-path->wireidx/decl path modidx moddb))
       ((when err)
        (cw! "[ERROR finding wire ~s0]: ~@1~%" (path->string-top path) err)
        (b* (((when maskwidth)
              (mv (msg "~s0[~x1:~x2]" name (1- maskwidth) rsh) (max 1 maskwidth))))
          (cw! "Warning: For unrecognized wire ~s0, caremask was unbounded!~%" name)
          (mv (msg "~s0[??:~x1]" (path->string-top path) rsh) nil)))
       ((wire wire))
       (width (if maskwidth
                  (min (max 1 maskwidth) wire.width)
                wire.width))
       (lsb (if wire.revp
                (- (+ wire.low-idx (- wire.width 1)) rsh)
              (+ rsh wire.low-idx)))
       (msb (if wire.revp
                (- lsb (- width 1))
              (+ lsb (- width 1)))))
    (mv (msg "~s0[~x1:~x2]" name msb lsb) width)))

;;  signame[4:0]:                #ux1E
;;  caremask:                    #ux1F
;; Value results from an override--
;;  override mask:                #uxc
;;  override value:               #uxf
;;  computed value:              #ux16
;; (Phase NEXT-PHASE = #x1.)


;; Printing for Chase --

;; Most things we print have a label on the left and a 4vec or integer value on
;; the right, to be printed in hex. If we are doing a chase comparison, we
;; might have another column of 4vec/integer values.  We want to right-justify
;; all such integer and 4vec values to the same point in the line.
;; Unfortunately, though, we need to support arbitrary ACL2 msg objects as the
;; labels in the left column, which means we can't determine how wide they will
;; print.  Instead, we'll just arbitrarily fix a left column width and use the
;; ~t directive to tab out to it, which means if the message doesn't fit in
;; that width we'll just go to the next line.

;; Non-integer 4vec values will be printed as:     ( #uxabcdabcd        ;; <-- "upper"
;;                                                   . #uxcadbca )      ;; <-- "lower"
;; Non-integer 4vec values will be printed as:    ( #uxABCD_ABCD
;;                                                  . #uxCA_DBCA )       (actual output)
;; If the "lower" is longer then it might instead look like:
;;                                                 (     #uxabcd        ;; <-- "upper"
;;                                                  . #uxdcbabcd )      ;; <-- "lower"
;; If the "lower" is longer then it might instead look like:
;;                                                (      #uxABCD
;;                                                 . #uxDCB_ABCD )       (actual output)
;;  -- the left paren is always at least 1 character to the left of the dot
;;  -- the dot is always one space before from the start of the "lower"
;;  -- the right paren is one space after the end of the "lower"
;;  -- the rightmost characters of "upper" and "lower" are in the same column.
;; Integer values are printed lined up with the 4vec uppers and lowers
;;  as in:                                                #ux10a
;;  as in:                                                #ux10A         (actual output)
;; We also allow lines with strings only and no 4vec.


;; BOZO we should put this in a more general book
(define msg-fix ((x msgp))
  :returns (new-x msgp)
  :inline t
  (mbe :logic (if (msgp x) x "")
       :exec x)
  ///
  (defthm msg-fix-when-msgp
    (implies (msgp x)
             (equal (msg-fix x) x)))

  (defthm msgp-of-msg
    (implies (and (stringp str)
                  (character-alistp args))
             (msgp (cons str args))))

  (defthm msgp-of-str
    (implies (stringp x)
             (msgp x)))
  
  (in-theory (disable msgp))
  
  (fty::deffixtype msg :pred msgp :fix msg-fix :equiv msg-equiv :define t))

(defprod 3col4vecline
  ((label msgp)
   (val maybe-4vec-p)
   (val2 maybe-4vec-p))
  :layout :tree)

(deflist 3col4vecs :elt-type 3col4vecline :true-listp t)

(define 4vec-print-width ((x 4vec-p))
  :returns (width posp :rule-classes :type-prescription)
  (if (2vec-p x)
      (length (str::hexify (2vec->val x)))
    (max (+ 3 (length (str::hexify (4vec->lower x))))
         (+ 2 (length (str::hexify (4vec->upper x)))))))

(define 3col4vec-min-space-between-columns ()
  :inline t
  2)

(define 3col4vec-left-column-width ()
  :inline t
  40)

(define 3col4vecline-valcolumn-width ((x 3col4vecline-p))
  :returns (width posp :rule-classes :type-prescription)
  (b* (((3col4vecline x)))
    (+ (3col4vec-min-space-between-columns)
       (if x.val (4vec-print-width x.val) 0))))

(define 3col4vecs-valcolumn-max-width ((x 3col4vecs-p))
  :returns (width posp :rule-classes :type-prescription)
  (if (atom x)
      (3col4vec-min-space-between-columns)
    (max (3col4vecline-valcolumn-width (car x))
         (3col4vecs-valcolumn-max-width (cdr x)))))

(define 3col4vecline-val2column-width ((x 3col4vecline-p))
  :returns (width posp :rule-classes :type-prescription)
  (b* (((3col4vecline x)))
    (+ (3col4vec-min-space-between-columns)
       (if x.val2 (4vec-print-width x.val2) 0))))

(define 3col4vecs-val2column-max-width ((x 3col4vecs-p))
  :returns (width posp :rule-classes :type-prescription)
  (if (atom x)
      (3col4vec-min-space-between-columns)
    (max (3col4vecline-val2column-width (car x))
         (3col4vecs-val2column-max-width (cdr x)))))

(local (in-theory (disable hexify explode-atom max)))


(define print-3col4vec-maybe-4vec ((column-width posp)
                                   (val maybe-4vec-p))
  :returns (mv (upper-msg msgp)
               (lower-msg msgp))
  ;; This prints out to column-width+2 to accommodate the right paren after the lower value.
  (b* (((unless val)
        (b* ((msg (msg "~_0" (+ 2 column-width))))
          (mv msg msg)))
       ((when (2vec-p val))
        (b* ((str (str::hexify (2vec->val val)))
             (len (length str))
             (spaces (- (lposfix column-width) len)))
          (mv (msg "~_0~s1  " spaces str)
              (msg "~_0" (+ 2 column-width)))))
       ((4vec val))
       (upper-str (str::hexify val.upper))
       (lower-str (str::hexify val.lower))
       (upper-len (length upper-str))
       (lower-len (length lower-str))
       (full-len (max (+ 3 lower-len) (+ 2 upper-len)))
       (spaces-before-lparen (- (lposfix column-width) full-len))
       (spaces-after-lparen (+ 1 (max 0 (+ 1 (- lower-len upper-len)))))
       (spaces-before-dot (- (lposfix column-width) (+ 2 lower-len))))
    (mv (msg "~_0(~_1~s2  " spaces-before-lparen spaces-after-lparen upper-str)
        (msg "~_0. ~s1 )" spaces-before-dot lower-str))))


(define print-3col4vecline ((valcolumn-width posp)
                            (val2column-width posp)
                            (x 3col4vecline-p))
  :returns (msg msgp)
  (b* (((3col4vecline x))
       ((unless (or x.val x.val2))
        (msg "~@0~%" x.label))
       ((mv val-upper val-lower) (print-3col4vec-maybe-4vec valcolumn-width x.val))
       ((mv val2-upper val2-lower) (print-3col4vec-maybe-4vec val2column-width x.val2))
       ((when (not x.val2))
        (if (2vec-p x.val)
            (msg "~@0~t1~@2~%" x.label (3col4vec-left-column-width) val-upper)
          (msg "~@0~t1~@2~%~t3~@4~%" x.label (3col4vec-left-column-width) val-upper (3col4vec-left-column-width) val-lower)))
       ((when (and (or (not x.val) (2vec-p x.val))
                   (2vec-p x.val2)))
        (msg "~@0~t1~@2~@3~%" x.label (3col4vec-left-column-width) val-upper val2-upper)))
    (msg "~@0~t1~@2~@3~%~t4~@5~@6~%" x.label (3col4vec-left-column-width) val-upper val2-upper
         (3col4vec-left-column-width) val-lower val2-lower)))

(define print-3col4vecs-aux ((valcolumn-width posp)
                             (val2column-width posp)
                             (x 3col4vecs-p))
  :returns (msg msgp)
  (if (atom x)
      ""
    (msg "~@0~@1"
         (print-3col4vecline valcolumn-width val2column-width (car x))
         (print-3col4vecs-aux valcolumn-width val2column-width (cdr x)))))

(define print-3col4vecs ((x 3col4vecs-p))
  :returns (msg msgp)
  (b* ((val-column (3col4vecs-valcolumn-max-width x))
       (val2-column (3col4vecs-val2column-max-width x)))
    (print-3col4vecs-aux val-column val2-column x)))


#||
(set-fmt-hard-right-margin 200 state)
(set-fmt-soft-right-margin 100 state)
(cw "~@0"
(print-3col4vecs
 (list
  (3col4vecline "Most things we print have a label on the left and a 4vec or integer value on" nil nil)
  (3col4vecline "the right, to be printed in hex. We want to right-justify all such integer" nil nil)
  (3col4vecline "and 4vec values to the same point in the line." nil nil)
  (3col4vecline "Non-integer 4vec values will be printed as:"
                (4vec #uxabcdabcd #uxcadbca ) nil)
  (3col4vecline "If the \"lower\" is longer then it" nil nil)
  (3col4vecline "might instead look like:" (4vec #uxabcd #uxdcbabcd ) nil)
  (3col4vecline " -- the left paren is always at least 1 character to the left of the dot" nil nil)
  (3col4vecline " -- the dot is always one space before from the start of the \"lower\"" nil nil)
  (3col4vecline " -- the right paren is one space after the end of the \"lower\"" nil nil)
  (3col4vecline " -- the rightmost characters of \"upper\" and \"lower\" are in the same column." nil nil)
  (3col4vecline "Integer values are printed lined up with the 4vec uppers and lowers" nil nil)
  (3col4vecline " as in:" #ux10a nil)
  (3col4vecline "We also allow lines with strings only and no 4vec." nil nil))))


(cw "~@0"
(print-3col4vecs
 (list
  (3col4vecline "Now testing printing two 4vecs" nil nil)
  (3col4vecline "Both 4vecs" (4vec #uxabcdabcd #uxcadbca) (4vec #uxcad #uxabcdabcda))
  (3col4vecline "First empty" nil (4vec #uxabcdabcd #uxcadbca))
  (3col4vecline "Second empty" (4vec #uxcad #uxabcdabcda) nil)
  (3col4vecline "Both 2vecs" 100 40)
  (3col4vecline "first 2vec" 1000 (4vec #uxabcdabcd #uxcadbca))
  (3col4vecline "second 2vec" (4vec #uxabcdabcd #uxcadbca) 1000000)
  (3col4vecline "empty/2vec" nil 30)
  (3col4vecline "2vec/empty" 800 nil))))

;; should output (minus the ";; " left column:)

;; Most things we print have a label on the left and a 4vec or integer value on
;; the right, to be printed in hex. We want to right-justify all such integer
;; and 4vec values to the same point in the line.
;; Non-integer 4vec values will be printed as:                  ( #uxABCD_ABCD
;;                                                                . #uxCA_DBCA )
;; If the "lower" is longer then it might instead look like:    (      #uxABCD
;;                                                               . #uxDCB_ABCD )
;;  -- the left paren is always at least 1 character to the left of the dot
;;  -- the dot is always one space before from the start of the "lower"
;;  -- the right paren is one space after the end of the "lower"
;;  -- the rightmost characters of "upper" and "lower" are in the same column.
;; Integer values are printed lined up with the 4vec uppers and lowers
;;  as in:                                                              #ux10A
;; We also allow lines with strings only and no 4vec.
||#

(define 3col4vec-maybe-line (cond
                             (label msgp)
                             (val maybe-4vec-p)
                             (val2 maybe-4vec-p))
  :returns (line 3col4vecs-p)
  (and cond
       (list (3col4vecline label val val2))))

(define maybe-loghead ((width maybe-posp)
                       (x integerp))
  :returns (new-x integerp :rule-classes :type-prescription)
  (if width (loghead (lposfix width) x) (lifix x)))

(define maybe-4vec-zero-ext ((width maybe-posp)
                             (x 4vec-p))
  :returns (new-x 4vec-p)
  (if width (4vec-zero-ext (2vec (lposfix width)) x) (4vec-fix x)))


(define svtv-chase-basic-signal-data ((val 4vec-p)
                                      (rsh natp)
                                      (width maybe-posp)
                                      (mask integerp))
  :returns (mv (shifted-val 4vec-p)
               (masked-val 4vec-p)
               (nonbool-mask integerp))
  (b* ((shifted-val  (maybe-4vec-zero-ext width (4vec-shift-core (- (lnfix rsh)) val)))
       (masked-val (4vec-bitand shifted-val (2vec mask)))
       (nonbool-mask (logxor (4vec->upper shifted-val) (4vec->lower shifted-val))))
    (mv shifted-val masked-val nonbool-mask)))



(define svtv-chase-override-signal-data ((var svar-p)
                                         (phase natp)
                                         (rsh natp)
                                         (width maybe-posp)
                                         (mask integerp)
                                         (evaldata svtv-chase-evaldata-p)
                                         (print-with-masked-val)
                                         &key
                                         (svtv-chase-data 'svtv-chase-data))
  :returns (mv (override-mask integerp)
               (masked-override-mask integerp)
               (override-val 4vec-p)
               (computed-val 4vec-p))

  (b* (((mv override-mask override-val computed-val)
        (svtv-chase-eval-override var phase evaldata))
       (override-mask (maybe-loghead width (ash override-mask (- (lnfix rsh)))))
       (masked-override-mask (logand override-mask mask))

       (override-val (maybe-4vec-zero-ext width (4vec-shift-core (- (lnfix rsh)) override-val)))
       (override-val (if print-with-masked-val
                         (4vec-bitand override-val (2vec masked-override-mask ))
                       override-val))
       (computed-val (maybe-4vec-zero-ext width (4vec-shift-core (- (lnfix rsh)) computed-val)))
       (computed-val (if print-with-masked-val
                         (4vec-bitand computed-val
                                      (2vec (logand mask (lognot override-mask))))
                       computed-val)))
    (mv override-mask
        masked-override-mask
        override-val
        computed-val)))

(define svtv-chase-print-with-masked-val ((print-with-mask-mode symbolp)
                                          (shifted-val 4vec-p)
                                          (masked-val 4vec-p)
                                          (shifted-val2 maybe-4vec-p)
                                          (masked-val2 maybe-4vec-p))
  (case print-with-mask-mode
    ((t) t)
    ((nil) nil)
    (t (and (or (not (2vec-p shifted-val))
                (and shifted-val2
                     (not (2vec-p shifted-val2))))
            (2vec-p masked-val)
            (or (not masked-val2)
                (2vec-p masked-val2))))))


(define svtv-chase-print-signal ((index acl2::maybe-natp "index to print at the beginning, if applicable")
                                 (var svar-p)
                                 (phase natp)
                                 (rsh natp)
                                 (mask 4vmask-p)
                                 (mask2 4vmask-p)
                                 (print-overrides)
                                 &key
                                 ((override-name/range msgp) '"")
                                 (skip-caremask)
                                 ((moddb moddb-ok) 'moddb)
                                 (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (svar-addr-p var)
          (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :prepwork ((local (in-theory (e/d (svar-addr-p)
                                    (str::hexify max member-equal ash logior logxor lognot
                                                 append acl2::append-of-cons
                                                 acl2::append-when-not-consp not
                                                 moddb->nmods moddb->nmods1p
                                                 nth
                                                 moddbp
                                                 if*)))))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (lines 3col4vecs-p)

  (b* (((mv rsh mask mask2 union-mask) (svtv-chase-normalize-masks rsh mask mask2))
       ((mv name/range width)
        (svtv-chase-var-name/range var rsh union-mask (svtv-chase-data->modidx svtv-chase-data)))
       ;; Mask and width now are both relative to rsh.  I.e., shift first, then mask/truncate.
       (delay (svar->delay var))
       (delay-msg (if (eql delay 0)
                      ""
                    (msg " (delay ~x0)" delay)))
       ((svtv-chase-data svtv-chase-data))

       (mask (sparseint-val mask))
       (mask2 (sparseint-val mask2))
       
       (val (svtv-chase-eval var phase svtv-chase-data.evaldata))
       ((mv shifted-val masked-val nonbool-mask)
        (svtv-chase-basic-signal-data val rsh width mask))

       (val2 (and svtv-chase-data.evaldata2
                  (let ((phase2  (+ phase svtv-chase-data.data2-offset)))
                    (and (<= 0 phase2)
                         (svtv-chase-eval var phase2 svtv-chase-data.evaldata2)))))
       ((mv shifted-val2 masked-val2 nonbool-mask2)
        (if val2
            (svtv-chase-basic-signal-data val2 rsh width mask2)
          (mv nil nil nil)))
       
       (print-with-mask-mode (svtv-chase-data->print-with-mask-mode svtv-chase-data))
       (print-with-masked-val (svtv-chase-print-with-masked-val print-with-mask-mode
                                                                shifted-val masked-val shifted-val2 masked-val2))
       (printed-val (if print-with-masked-val
                        masked-val
                      shifted-val))
       (printed-val2 (if print-with-masked-val
                         masked-val2
                       shifted-val2))
       (name/range-display (if* (equal (msg-fix override-name/range) "")
                                name/range
                                (msg-fix override-name/range)))
       (first-lines (append (list (3col4vecline
                                   (msg "~@0 ~@1~@2"
                                        (if index (msg "~x0." index) "")
                                        name/range-display
                                        delay-msg)
                                   printed-val
                                   printed-val2))
                            (3col4vec-maybe-line
                             print-with-masked-val
                             (if (eq print-with-mask-mode t)
                                 "(value masked due to MASK-ALWAYS setting -- ? for help)"
                               "(value masked to make it 2valued -- ? for help)")
                              nil nil)
                            (3col4vec-maybe-line (or (not (2vec-p shifted-val))
                                                     (and val2 (not (2vec-p shifted-val2))))
                                                 "  non-Boolean portion:"
                                                 (2vec nonbool-mask)
                                                 (and val2 (2vec nonbool-mask2)))))
       (caremask-lines (and (not skip-caremask)
                            (list (3col4vecline "  caremask:" (2vec mask) (and val2 (2vec mask2))))))
       ((unless print-overrides)
        (append first-lines caremask-lines))


       ((mv override-mask masked-override-mask override-val computed-val)
        (svtv-chase-override-signal-data var phase rsh width mask svtv-chase-data.evaldata print-with-masked-val))

       ((mv override-mask2 masked-override-mask2 override-val2 computed-val2)
        (if val2
            (svtv-chase-override-signal-data var (+ phase svtv-chase-data.data2-offset) rsh width mask2 svtv-chase-data.evaldata2 print-with-masked-val)
          (mv nil nil nil nil)))
       
       ((when (and (eql masked-override-mask 0)
                   (or (not val2)
                       (eql masked-override-mask2 0))))
        (append first-lines caremask-lines))

       (override-lines (list
                        (3col4vecline "Value results from an override --" nil nil)
                        (3col4vecline "  Override mask:" (2vec override-mask) (and val2 (2vec override-mask2)))
                        (3col4vecline "  Override val:"  override-val override-val2)
                        (3col4vecline "  Computed val:"  computed-val computed-val2))))
    (append first-lines caremask-lines override-lines)))


(local (defthm len-equal-0
         (equal (equal (len x) 0)
                (not (consp x)))))

(define svtv-chase-print-signals ((index natp)
                                  (vars svarlist-p)
                                  (masks 4vmask-alist-p)
                                  (masks2 4vmask-alist-p)
                                  (phase natp)
                                  (print-overrides)
                                  &key
                                  ((moddb moddb-ok) 'moddb)
                                  (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints (("goal" :in-theory (enable alist-keys))
                (and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (lines 3col4vecs-p)
  (if (atom vars)
      nil
    (append (list (3col4vecline
                   (if (svtv-chase-data->evaldata2 svtv-chase-data)
                       "----              -----              -----              -----              -----              -----              -----              -----"
                     "----              -----              -----              -----              -----")
                   nil nil))
            (svtv-chase-print-signal (lnfix index)
                                     (car vars)
                                     phase
                                     0
                                     (or (cdr (hons-get (car vars) (4vmask-alist-fix masks))) 0)
                                     (or (cdr (hons-get (car vars) (4vmask-alist-fix masks2))) 0)
                                     print-overrides)
            (svtv-chase-print-signals (1+ (lnfix index)) (cdr vars) masks masks2 phase print-overrides))))

      
                     
         
(local (defthm len-alist-keys-when-4vmask-alist-p
         (implies (4vmask-alist-p x)
                  (equal (len (alist-keys x))
                         (len x)))
         :hints(("Goal" :in-theory (enable alist-keys)))))



(define svtv-labelpair-p (x)
  (or (integerp x) ;; no label, just offset
      (symbolp x) ;; no offset, just label
      (and (consp x)
           (symbolp (car x))
           (consp (cdr x))
           (integerp (cadr x))
           (not (cddr x)))))

(define svtv-chase-phase-labelpair-aux ((phase natp)
                                        (phaselabels symbol-listp))
  ;; Scans through the phase labels and returns either the phase label at
  ;; phase, the last phase label before phase and the offset of phase from that
  ;; label, or NIL if no label was found at or before phase.
  :returns (labelpair svtv-labelpair-p
                      :hints(("Goal" :in-theory (enable svtv-labelpair-p))))
  (if (zp phase)
      (mbe :logic (acl2::symbol-fix (car phaselabels))
           :exec (car phaselabels))
    (if (car phaselabels)
        (or (svtv-chase-phase-labelpair-aux (1- phase) (cdr phaselabels))
            (list (mbe :logic (acl2::symbol-fix (car phaselabels))
                       :exec (car phaselabels))
                  phase))
      (svtv-chase-phase-labelpair-aux (1- phase) (cdr phaselabels)))))

(define svtv-chase-phase-labelpair ((phase integerp)
                                    (phaselabels symbol-listp))
  :returns (labelpair svtv-labelpair-p
                      :hints ((and stable-under-simplificationp
                                   '(:in-theory (enable svtv-labelpair-p)))))
  ;; Returns either:
  ;;  - the phase label exactly at phase,
  ;;  - the last phase label before phase and the offset of phase at that label,
  ;;  - phase itself if no labels before phase.
  (if (< (lifix phase) 0)
      (lifix phase)
    (or (svtv-chase-phase-labelpair-aux phase phaselabels)
        (lnfix phase))))

(define svtv-chase-labelpair-phase ((labelpair svtv-labelpair-p)
                                    (phaselabels symbol-listp))
  :prepwork ((local (in-theory (enable svtv-labelpair-p))))
  :returns (phase acl2::maybe-integerp :rule-classes :type-prescription)
  (b* (((when (integerp labelpair))
        labelpair)
       (name (if (consp labelpair)
                 (car labelpair)
               labelpair))
       (offset (if (consp labelpair)
                   (lifix (cadr labelpair))
                 0))
       (label-index (acl2::index-of name phaselabels)))
    (and label-index (+ label-index offset))))


    
(local (in-theory (disable nth update-nth)))


(define svtv-chase-mask-alists-sort-vars ((vars svarlist-p)
                                          (masks 4vmask-alist-p)
                                          (masks2 4vmask-alist-p)
                                          (diffs svex-env-p))
  :returns (mv (diff-overlap svarlist-p)
               (overlap svarlist-p)
               (both svarlist-p)
               (first svarlist-p)
               (second svarlist-p))
  :prepwork ((local (in-theory (disable svarlist-p-when-subsetp-equal logand member-equal hons-assoc-equal))))
  (b* (((when (atom vars)) (mv nil nil nil nil nil))
       (var1 (svar-fix (car vars)))
       (mask1 (or (cdr (hons-get var1 (4vmask-alist-fix masks))) 0))
       (mask2 (or (cdr (hons-get var1 (4vmask-alist-fix masks2))) 0))
       (diff (or (cdr (hons-get var1 (svex-env-fix diffs))) 0))
       ((mv diff-overlap overlap both first second)
        (svtv-chase-mask-alists-sort-vars (cdr vars) masks masks2 diffs))
       (mask-overlap (sparseint-bitand mask1 mask2))
       ((when (not (sparseint-equal 0 mask-overlap)))
        (b* ((masked-diff (4vec-bitand (2vec (sparseint-val mask-overlap)) diff)))
          (if (eql masked-diff 0)
              (mv diff-overlap (cons var1 overlap) both first second)
            (mv (cons var1 diff-overlap) overlap both first second))))
       ((when (not (sparseint-equal 0 mask1)))
        (if (sparseint-equal 0 mask2)
            (mv diff-overlap overlap both (cons var1 first) second)
          (mv diff-overlap overlap (cons var1 both) first second)))
       ((when (not (sparseint-equal 0 mask2)))
        (mv diff-overlap overlap both first (cons var1 second))))
    (mv diff-overlap overlap both first second)))


(local
 (include-book "std/osets/element-list" :dir :system))

(local
 (fty::deflist svarlist :elt-type svar :true-listp t :elementp-of-nil nil))

(define svtv-chase-mask-alists-sorted-vars ((masks 4vmask-alist-p)
                                            (masks2 4vmask-alist-p)
                                            (diffs svex-env-p))
  :prepwork ((local (defthm true-listp-when-svarlist-p-rw
                      (implies (svarlist-p x) (true-listp x)))))
  :returns (sorted-vars svarlist-p)
  (b* ((masks (4vmask-alist-fix masks))
       (masks2 (4vmask-alist-fix masks2))
       ((when (atom masks2)) (mergesort (alist-keys masks)))
       (vars (union (mergesort (alist-keys masks)) (mergesort (alist-keys masks2))))
       ((mv diff-overlap overlap both first second)
        (svtv-chase-mask-alists-sort-vars vars masks masks2 diffs)))
    (append diff-overlap overlap both first second)))

(define svtv-chase-mask-intersect-alist ((vars svarlist-p)
                                         (masks 4vmask-alist-p)
                                         (masks2 4vmask-alist-p)
                                         (include-masks2))
  :returns (intersect-masks 4vmask-alist-p)
  (b* (((when (atom vars)) nil)
       (var1 (svar-fix (car vars)))
       (masks (4vmask-alist-fix masks))
       (masks2 (4vmask-alist-fix masks2))
       (mask1 (or (cdr (hons-get var1 masks)) 0)))
    (cons (cons (svar-fix (car vars))
                (if include-masks2
                    (sparseint-bitand mask1
                                      (or (cdr (hons-get var1 masks2)) 0))
                  mask1))
          (svtv-chase-mask-intersect-alist (cdr vars) masks masks2 include-masks2))))

(define svtv-chase-signal ((var svar-p)
                           (phase integerp)
                           (rsh natp)
                           (mask 4vmask-p)
                           &key
                           (svtv-chase-data 'svtv-chase-data)
                           ((moddb moddb-ok) 'moddb))
  :guard (and ;; (svar-addr-p var)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (mv (type symbolp :rule-classes :type-prescription)
               (vars 4vmask-alist-p)
               (expr svex-p))
  (b* (((svtv-chase-data svtv-chase-data))
       ((mv var phase) (svtv-chase-normalize-var/phase var phase))
       (signal-lines (svtv-chase-print-signal nil var phase rsh mask mask t))
       (phase-line (list (3col4vecline
                          (msg "(Phase ~@0.)"
                               (b* ((pair (svtv-chase-phase-labelpair phase svtv-chase-data.phaselabels)))
                                 (if (equal pair phase)
                                     (msg "~x0" phase)
                                   (msg "~x0 = ~x1" pair phase))))
                          nil nil)))
       ((mv type mask-alist mask-alist2 diffs expr)
        (svtv-chase-deps var phase rsh mask))
       ((acl2::with-fast mask-alist mask-alist2))
       (vars (svtv-chase-mask-alists-sorted-vars mask-alist mask-alist2 diffs))
       (deps-lines
        (b* (((when (eq type :error))
              (list (3col4vecline "Error! Somehow this signal wasn't what we expected." nil nil)))
             ((when (eq type :input))
              (list (3col4vecline "Primary input." nil nil)))
             ((when (eq type :initst))
              (list (3col4vecline "Initial state." nil nil)))
             ((when (eq type :prevst))
              (cons (3col4vecline "Previous state var." nil nil)
                    (svtv-chase-print-signals 0 vars mask-alist mask-alist2 phase svtv-chase-data.print-overrides-mode))))
          (cons (3col4vecline "Internal signal; dependencies:" nil nil)
                (svtv-chase-print-signals 0 vars mask-alist mask-alist2 phase
                                          svtv-chase-data.print-overrides-mode))))
       (last-line (list (3col4vecline "================================================================================" nil nil)))
       (msg (print-3col4vecs
             (append signal-lines phase-line deps-lines last-line))))
    (cw! "~@0" msg)
    (mv type (svtv-chase-mask-intersect-alist
              vars mask-alist mask-alist2
              svtv-chase-data.evaldata2)
        expr))
  ///
  ;; (defret svarlist-addr-p-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (alist-keys vars)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))

  

  ;; (defret svarlist-addr-p-expr-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (svex-vars expr)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))
  )

;; (define svar-addr-p! (x)
;;   :enabled t
;;   (and (svar-p x)
;;        (svar-addr-p x)))

;; (define 4vmask-alist-addr-p! (x)
;;   :enabled t
;;   (and (4vmask-alist-p x)
;;        (svarlist-addr-p (alist-keys x))))

;; (define svex-addr-p! (x)
;;   :enabled t
;;   (and (svex-p x)
;;        (svarlist-addr-p (svex-vars x))))


(define svtv-chase-signal-data ((pos chase-position-p)
                                &key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (chase-position-addr-p pos)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints (("goal" :in-theory (enable svtv-chase-datap
                                           ;; chase-position-addr-p
                                           ;; chase-stack-addr-p
                                           )))
  :returns (new-svtv-chase-data)
  (b* (((chase-position pos))
       ((mv type vars expr)
        (svtv-chase-signal (make-svar :name (make-address :path pos.path))
                           pos.phase pos.rsh pos.mask))
       ((when (eq type :error))
        (cw! "[Error -- discrepancy between stored updates and assignments!]~%")
        svtv-chase-data)
       (svtv-chase-data (set-svtv-chase-data->stack (cons pos (svtv-chase-data->stack svtv-chase-data))
                                            svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->sigtype type svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->vars vars svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->expr expr svtv-chase-data))
       ;; (svtv-chase-data (set-svtv-chase-data->new-phase new-phase svtv-chase-data))
       )
    svtv-chase-data)
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))



(define svtv-chase-range ((pos chase-position-p)
                          (msb integerp)
                          (lsb integerp)
                          &key
                          ((moddb moddb-ok) 'moddb)
                          (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (chase-position-addr-p pos)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable ;; chase-position-addr-p
                                   ;; svar-addr-p
                                   svtv-chase-datap))))
  :returns (new-svtv-chase-data)
  :prepwork ((local (in-theory (disable logmask not))))
  (b* ((modidx (svtv-chase-data->modidx svtv-chase-data))
       ((chase-position pos))
       ;; (path (address->path (svar->name pos.var)))
       ((mv err wire & &) (moddb-path->wireidx/decl pos.path modidx moddb))
       ((when err)
        (cw! "[ERROR finding wire ~s0]: ~@1~%" (path->string-top pos.path) err)
        svtv-chase-data)
       ((wire wire))
       (msb (lifix msb))
       (lsb (lifix lsb))
       (wire-lsb (if wire.revp
                     (+ wire.low-idx (- wire.width 1))
                   wire.low-idx))
       (wire-msb (if wire.revp
                     wire.low-idx
                   (+ wire.low-idx (- wire.width 1))))
       ((unless (if wire.revp
                    (and (<= wire-msb msb) (<= msb lsb) (<= lsb wire-lsb))
                  (and (<= wire-lsb lsb) (<= lsb msb) (<= msb wire-msb))))
        (cw! "Bad range for ~s0: declared range is [~x1:~x2]~%"
             (path->string-top pos.path) wire-msb wire-lsb)
        svtv-chase-data)
       (width (if wire.revp
                  (+ 1 (- lsb msb))
                (+ 1 (- msb lsb))))
       (rsh (if wire.revp (- wire-lsb lsb) (- lsb wire-lsb)))
       (mask (logmask width))
       (new-pos (change-chase-position pos
                                       :rsh rsh
                                       :mask (int-to-sparseint mask))))
    (svtv-chase-signal-data new-pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

;; (local
;;  (defsection lhs-addr-p-of-svtv-wire->lhs

;;    (defret lhs-addr-p-of-svtv-1wire->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-1wire->lhs
;;                                        )))
;;      :fn svtv-1wire->lhs)

;;    (local (defthm member-lhs-vars-of-append
;;             (implies (and (not (member v (lhs-vars a)))
;;                           (not (member v (lhs-vars b))))
;;                      (not (member v (lhs-vars (append a b )))))
;;             :hints(("Goal" :in-theory (enable lhs-vars)))))

;;    (defret lhs-addr-p-of-svtv-concat->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-concat->lhs
;;                                        )))
;;      :fn svtv-concat->lhs)

;;    (defret lhs-addr-p-of-svtv-wire->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-wire->lhs)))
;;      :fn svtv-wire->lhs)))


;; (local (include-book "centaur/bitops/ihsext-basics" :dir :system
(local (in-theory (disable logmask)))

(define svtv-chase-goto-lhs ((lhs lhs-p)
                             (phase natp)
                             (debug-source-obj)
                             &key
                             ((moddb moddb-ok) 'moddb)
                             (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  ;; :prepwork ((local (defthm svar-addr-p-of-lhatom-var->name
  ;;                     (implies (and (lhatom-case x :var)
  ;;                                   (svarlist-addr-p (lhatom-vars x)))
  ;;                              (svar-addr-p (lhatom-var->name x)))
  ;;                     :hints(("Goal" :in-theory (enable lhatom-vars)))))
  ;;            (local (defthm member-vars-of-lhrange->atom
  ;;                     (implies (and (not (member v (lhs-vars x)))
  ;;                                   (consp x))
  ;;                              (not (member v (lhatom-vars (lhrange->atom (car x))))))
  ;;                     :hints(("Goal" :in-theory (enable lhs-vars)))))
  ;;            (local (in-theory (disable lhs-vars-when-consp))))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((when (atom lhs))
        (cw! "Error interpreting name: ~x0~%" debug-source-obj)
        svtv-chase-data)
       ((when (consp (cdr lhs)))
        (cw! "Error interpreting name: ~x0 was a concatenation~%" debug-source-obj)
        svtv-chase-data)
       ((lhrange lhrange) (car lhs))
       ((unless (lhatom-case lhrange.atom :var))
        (cw! "Error interpreting name: ~x0 had no variable component~%" debug-source-obj)
        svtv-chase-data)
       ((lhatom-var lhrange.atom))
       ((svar lhrange.atom.name))
       ((unless (address-p lhrange.atom.name.name))
        (cw! "Error interpreting name: ~x0 produced a variable that was not an address~%" debug-source-obj)
        svtv-chase-data)
       (pos (make-chase-position :path (address->path lhrange.atom.name.name)
                                 :phase phase
                                 :rsh lhrange.atom.rsh
                                 :mask (int-to-sparseint (logmask lhrange.w)))))
    (svtv-chase-signal-data pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))
       

(define svtv-chase-goto ((str stringp)
                         (labelpair svtv-labelpair-p)
                         &key
                         ((moddb moddb-ok) 'moddb)
                         (aliases 'aliases)
                         (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases)))
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((mv err lhs) (svtv-wire->lhs str (svtv-chase-data->modidx svtv-chase-data) moddb aliases))
       ((when err)
        (cw! "Error interpreting name: ~s0~%" str)
        svtv-chase-data)
       (phase (svtv-chase-labelpair-phase labelpair (svtv-chase-data->phaselabels svtv-chase-data)))
       ((unless phase)
        (cw! "Error interpreting phase: ~x0 -- label not found~%" labelpair)
        svtv-chase-data)
       ((unless (<= 0 phase))
        (cw! "Error interpreting phase: ~x0 -- normalized to negative value: ~x1~%" labelpair phase)
        svtv-chase-data))
    (svtv-chase-goto-lhs lhs phase str))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

(define svtv-chase-goto-output ((name)
                                &key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((svtv-chase-data svtv-chase-data))
       (probe? (hons-assoc-equal name svtv-chase-data.probes))
       ((unless probe?)
        (cw! "Error: no output named ~x0~%" name)
        svtv-chase-data)
       ((svtv-probe probe) (cdr probe?))
       (lhs? (hons-assoc-equal probe.signal svtv-chase-data.namemap))
       ((unless lhs?)
        (cw! "Error: found output named ~x0 pointing to signal ~x1 but no such entry in namemap~%"
             name probe.signal)
        svtv-chase-data)
       (lhs (cdr lhs?)))
    (svtv-chase-goto-lhs lhs probe.time name))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))
       

(define svtv-chase-print (&key
                          ((moddb moddb-ok) 'moddb)
                          (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :returns new-svtv-chase-data
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t))
                ;; (and stable-under-simplificationp
                ;;      '(:in-theory (enable chase-position-addr-p)))
                )
  (b* ((stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        svtv-chase-data)
       (pos (car stack))
       (svtv-chase-data (set-svtv-chase-data->stack (cdr stack) svtv-chase-data))
       (svtv-chase-data (svtv-chase-signal-data pos)))
    svtv-chase-data)
  ///
  (defmacro svtv-chase-print! (&rest args)
    `(b* ((svtv-chase-data (svtv-chase-print . ,args)))
       (mv nil svtv-chase-data state)))

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))


(define svtv-chase-print-stack-rec ((stack chase-stack-p)
                                    (index natp)
                                    &key
                                    ((moddb moddb-ok) 'moddb)
                                    (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (lines 3col4vecs-p)
  (b* (((when (atom stack)) nil)
       ((chase-position pos) (car stack))
       (var (make-svar :name (make-address :path pos.path)))
       ((mv var phase) (svtv-chase-normalize-var/phase var pos.phase))
       (signal-lines (svtv-chase-print-signal index var phase pos.rsh pos.mask pos.mask t))
       (phase-line (list (3col4vecline
                          (msg "(Phase ~@0.)"
                               (b* ((pair (svtv-chase-phase-labelpair
                                           phase (svtv-chase-data->phaselabels svtv-chase-data))))
                                 (if (equal pair phase)
                                     (msg "~x0" phase)
                                   (msg "~x0 = ~x1" pair phase))))
                          nil nil)))
       (rest-lines (svtv-chase-print-stack-rec (cdr stack) (1+ (lnfix index)))))
    (append (list (3col4vecline
                   (if (svtv-chase-data->evaldata2 svtv-chase-data)
                       "----              -----              -----              -----              -----              -----              -----              -----"
                     "----              -----              -----              -----              -----")
                   nil nil))
            signal-lines phase-line rest-lines)))

(define svtv-chase-print-stack (&key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))

  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  (b* ((lines (svtv-chase-print-stack-rec (svtv-chase-data->stack svtv-chase-data) 0)))
    (cw! "~@0" (print-3col4vecs lines))))


(define svtv-chase-print-history-range ((pos chase-position-p)
                                        (last integerp)
                                        (incr integerp)
                                        &key
                                        ((moddb moddb-ok) 'moddb)
                                        (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (lines 3col4vecs-p)
  :measure (if (< 0 (ifix incr))
               (nfix (- (ifix last)
                        (chase-position->phase pos)))
             (nfix (- (chase-position->phase pos)
                      (ifix last))))
  (b* (((chase-position pos))
       (var (make-svar :name (make-address :path pos.path)))
       ((mv var phase) (svtv-chase-normalize-var/phase var pos.phase))
       (phase-label (msg "(Phase ~@0.)"
                         (b* ((pair (svtv-chase-phase-labelpair
                                     phase (svtv-chase-data->phaselabels svtv-chase-data))))
                           (if (equal pair phase)
                               (msg "~x0" phase)
                             (msg "~x0 = ~x1" pair phase)))))
       (signal-lines (svtv-chase-print-signal
                      nil var phase pos.rsh pos.mask pos.mask t
                      :override-name/range phase-label
                      :skip-caremask t))
       (next-phase (+ pos.phase (lifix incr)))
       ((when (or (< next-phase 0)
                  (zip incr)
                  (and (< 0 incr) (< (lifix last) next-phase))
                  (and (> 0 incr) (> (lifix last) next-phase))))
        (cons (3col4vecline
               (if (svtv-chase-data->evaldata2 svtv-chase-data)
                   "----              -----              -----              -----              -----              -----              -----              -----"
                 "----              -----              -----              -----              -----")
               nil nil)
              signal-lines))
       (next-pos (change-chase-position pos :phase next-phase))
       (rest-lines (svtv-chase-print-history-range next-pos last incr)))
    (cons (3col4vecline
           (if (svtv-chase-data->evaldata2 svtv-chase-data)
               "----              -----              -----              -----              -----              -----              -----              -----"
             "----              -----              -----              -----              -----")
           nil nil)
          (append signal-lines rest-lines))))


(define svtv-chase-print-history ((by posp)
                                  &key
                                  ((moddb moddb-ok) 'moddb)
                                  (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :prepwork ((local (in-theory (disable mod)))
             (local (defthm integerp-mod
                      (implies (and (integerp x) (integerp y))
                               (integerp (mod x y)))
                      :hints(("Goal" :in-theory (enable mod)))
                      :rule-classes :type-prescription)))
  (b* ((stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack"))
       ((chase-position pos) (car stack))
       (nphases (len (svtv-chase-evaldata->inputs (svtv-chase-data->evaldata svtv-chase-data))))
       (first-phase (mod pos.phase (lposfix by)))
       (first-pos (change-chase-position pos :phase first-phase))
       (lines (svtv-chase-print-history-range first-pos (1- nphases) by)))
    (cw! "~@0" (print-3col4vecs lines))))



(defmacro setup-ev-for-chase ()
  '(progn
     (defttag trans)
     (defattach (simple-translate-and-eval-logic acl2::simple-translate-and-eval-cmp)
       :skip-checks t)))

(defmacro unsetup-ev-for-chase ()
  '(progn
     (defttag nil)
     (defattach (simple-translate-and-eval-logic nil))))

;; (local (defthm chase-position-addr-p-car-when-chase-stack-addr-p
;;          (implies (and (svtv-chase-data->stack-addr-p x)
;;                        (consp x))
;;                   (svtv-chase-data->position-addr-p (car x)))
;;          :hints(("Goal" :in-theory (enable chase-stack-addr-p)))))

;; (local (defthm chase-stack-addr-p-cdr-when-chase-stack-addr-p
;;          (implies (svtv-chase-data->stack-addr-p x)
;;                   (svtv-chase-data->stack-addr-p (cdr x)))
;;          :hints(("Goal" :in-theory (enable chase-stack-addr-p)))))


(local (defthm nth-when-4vmask-alist-p
         (implies (and (4vmask-alist-p x)
                       (< (nfix n) (len x)))
                  (and (consp (nth n x))
                       (svar-p (car (nth n x)))
                       (sparseint-p (cdr (nth n x)))))
         :hints(("Goal" :in-theory (enable nth)))))

;; (local (defthm nth-svar-addr-p-when-4vmask-alist-p
;;          (implies (and (4vmask-alist-p x)
;;                        (svarlist-addr-p (alist-keys x))
;;                        (< (nfix n) (len x)))
;;                   (svar-addr-p (car (nth n x))))
;;          :hints(("Goal" :in-theory (enable nth alist-keys)))))

(local (in-theory (disable read-object
                           open-input-channel-p1
                           member)))

(include-book "std/io/file-measure" :dir :system)
(local (include-book "std/io/open-channels" :dir :system))
(local (in-theory (disable file-measure)))

(verify-termination evisc-tuple)
(verify-guards evisc-tuple)


(encapsulate
  (((simple-translate-and-eval-logic
     * * * * * * state * * *) => (mv * *)
    :formals (x alist ok-stobj-names msg ctx wrld state aok safe-mode gc-off)
    :guard t))
  (set-ignore-ok t)
  (set-irrelevant-formals-ok t)
  (local (defun simple-translate-and-eval-logic (x alist ok-stobj-names msg ctx wrld state aok safe-mode gc-off)
           (declare (xargs :stobjs state))
           (mv nil nil))))


(local (in-theory (disable w)))

(local (defthm w-of-read-object
         (equal (w (mv-nth 2 (read-object channel state)))
                (w state))
         :hints(("Goal" :in-theory (enable w read-object
                                           ;; Matt K. addition for 5/8/2023 change to
                                           ;; read-object to call iprint-oracle-updates:
                                           read-acl2-oracle update-acl2-oracle
                                           ;; Matt K. mod for conversion of
                                           ;; eviscerate-top to logic mode:
                                           acl2::iprint-oracle-updates)))))


(define svtv-chase-follow-x-select-var ((vars 4vmask-alist-p)
                                        (vals 4veclist-p))
  :returns (mv multiple ;; true if we gave up because there was more than one
               (chosen-var (iff (svar-p chosen-var) chosen-var)))
  :guard (eql (len vals) (len vars))
  (b* (((when (atom vars)) (mv nil nil))
       ((unless (mbt (consp (car vars))))
        (svtv-chase-follow-x-select-var (cdr vars) vals))
       ((cons var mask) (car vars))
       (val (car vals))
       ((4vec masked-val) (4vec-mask-to-zero mask val))
       (has-xes (not (eql 0 (logandc2 masked-val.upper masked-val.lower))))
       ((mv multi rest-var) (svtv-chase-follow-x-select-var (cdr vars) (cdr vals)))
       (multi (or multi (and has-xes rest-var))))
    (mv multi
        (and (not multi)
             (if has-xes (svar-fix var) rest-var))))
  ///
  (defret <fn>-lookup-in-vars
    (implies (and chosen-var
                  (4vmask-alist-p vars))
             (hons-assoc-equal chosen-var vars))))


(define svtv-chase-follow-x-step (&key
                                  ((moddb moddb-ok) 'moddb)
                                  (svtv-chase-data 'svtv-chase-data))
  :returns (mv traversedp new-svtv-chase-data)
  :verify-guards nil
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  (b* ((vars (svtv-chase-data->vars svtv-chase-data))
       (stack (svtv-chase-data->stack svtv-chase-data))
       (evaldata (svtv-chase-data->evaldata svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        (mv nil svtv-chase-data))
       ((chase-position pos) (car stack))
       (pos.var  (make-svar :name (make-address :path pos.path)))
       ((mv pos.var pos.phase) (svtv-chase-normalize-var/phase pos.var pos.phase))
       ((mv ?override-mask masked-override-mask ?override-val ?computed-val)
        (svtv-chase-override-signal-data
         pos.var pos.phase pos.rsh nil (sparseint-val pos.mask) evaldata nil))
       ((unless (eql masked-override-mask 0))
        ;; Stop because our signal was overridden.
        (mv nil svtv-chase-data))
       (varnames (alist-keys vars))
       (vals (svtv-chase-evallist varnames pos.phase evaldata))
       ((mv & chosen-var) (svtv-chase-follow-x-select-var vars vals))
       ((unless chosen-var)
        ;; Couldn't follow it back, so just stay here
        (mv nil svtv-chase-data))
       (new-mask (cdr (hons-assoc-equal chosen-var vars)))
       ((mv chosen-var phase) (svtv-chase-normalize-var/phase chosen-var pos.phase))
       (name (svar->name chosen-var))
       ((unless (address-p name))
        (cw! "The chosen signal isn't an address, so it must be an ~
                    auxiliary variable supporting an override.~%Enter P to ~
                    print current state, ? for more options.~%")
        (mv nil svtv-chase-data))
       (svtv-chase-data (svtv-chase-signal-data
                         (make-chase-position
                          :path (address->path name)
                          :phase (- phase (svar->delay chosen-var))
                          :rsh 0 :mask new-mask))))
    (mv t svtv-chase-data))
  ///
  (verify-guards svtv-chase-follow-x-step-fn
    :hints ((and stable-under-simplificationp
                 '(:in-theory (enable svtv-chase-datap)
                   :do-not-induct t))))

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-follow-xes-rec ((count natp)
                                   &key
                                   ((moddb moddb-ok) 'moddb)
                                   (svtv-chase-data 'svtv-chase-data))
  :returns (mv (remainder natp :rule-classes :type-prescription)
               new-svtv-chase-data)
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((when (zp count))
        (mv 0 svtv-chase-data))
       ((mv keep-going svtv-chase-data)
        (svtv-chase-follow-x-step))
       ((when keep-going)
        (svtv-chase-follow-xes-rec (1- count))))
    (mv count svtv-chase-data))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-follow-xes (args
                               &key
                               ((moddb moddb-ok) 'moddb)
                               (svtv-chase-data 'svtv-chase-data))
  :returns new-svtv-chase-data
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((unless (and (consp args)
                     (natp (car args))
                     (not (cdr args))))
        (cw! "FOLLOW-XES directive must be of the form (FOLLOW-XES COUNT) where COUNT is natp.  ? for more options.~%")
        svtv-chase-data)
       (count (car args))
       ((mv remaining-count svtv-chase-data)
        (svtv-chase-follow-xes-rec count)))
    (cw! "FOLLOW-XES complete: ~x0 steps.~%" (- count remaining-count))
    (svtv-chase-print))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))


(define svtv-chase-follow-compare-select-var ((vars 4vmask-alist-p)
                                              (vals 4veclist-p)
                                              (vals2 4veclist-p))
  :returns (mv multiple (chosen-var (iff (svar-p chosen-var) chosen-var)))
  :guard (and (eql (len vals) (len vars))
              (eql (len vals2) (len vars)))
  (b* (((when (atom vars)) (mv nil nil))
       ((unless (mbt (consp (car vars))))
        (svtv-chase-follow-compare-select-var (cdr vars) vals vals2))
       ((cons var mask) (car vars))
       (val1 (car vals))
       (val2 (car vals2))
       (masked-val1 (4vec-mask-to-zero mask val1))
       (masked-val2 (4vec-mask-to-zero mask val2))
       (diff (not (equal masked-val1 masked-val2)))
       ((mv multi rest-var) (svtv-chase-follow-compare-select-var (cdr vars) (cdr vals) (cdr vals2)))
       (multi (or multi (and diff rest-var))))
    (mv multi
        (and (not multi)
             (if diff
                 (svar-fix var)
               rest-var))))
  ///
  (defret <fn>-lookup-in-vars
    (implies (and chosen-var
                  (4vmask-alist-p vars))
             (hons-assoc-equal chosen-var vars))))


(define svtv-chase-follow-compare-step (&key
                                  ((moddb moddb-ok) 'moddb)
                                  (svtv-chase-data 'svtv-chase-data))
  :returns (mv traversedp new-svtv-chase-data)
  :verify-guards nil
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (svtv-chase-data->evaldata2 svtv-chase-data))
  (b* ((vars (svtv-chase-data->vars svtv-chase-data))
       (stack (svtv-chase-data->stack svtv-chase-data))
       (evaldata (svtv-chase-data->evaldata svtv-chase-data))
       (evaldata2 (svtv-chase-data->evaldata2 svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        (mv nil svtv-chase-data))
       ((chase-position pos) (car stack))
       (pos.var  (make-svar :name (make-address :path pos.path)))
       ((mv pos.var pos.phase) (svtv-chase-normalize-var/phase pos.var pos.phase))
       ((mv ?override-mask masked-override-mask ?override-val ?computed-val)
        (svtv-chase-override-signal-data
         pos.var pos.phase pos.rsh nil (sparseint-val pos.mask) evaldata nil))
       ((unless (eql masked-override-mask 0))
        ;; Stop because our signal was overridden.
        (mv nil svtv-chase-data))
       (varnames (alist-keys vars))
       (vals (svtv-chase-evallist varnames pos.phase evaldata))
       (vals2 (svtv-chase-evallist varnames pos.phase evaldata2))
       ((mv & chosen-var) (svtv-chase-follow-compare-select-var vars vals vals2))
       ((unless chosen-var)
        ;; Couldn't follow it back, so just stay here
        (mv nil svtv-chase-data))
       (new-mask (cdr (hons-assoc-equal chosen-var vars)))
       ((mv chosen-var phase) (svtv-chase-normalize-var/phase chosen-var pos.phase))
       (name (svar->name chosen-var))
       ((unless (address-p name))
        (cw! "The chosen signal isn't an address, so it must be an ~
                    auxiliary variable supporting an override.~%Enter P to ~
                    print current state, ? for more options.~%")
        (mv nil svtv-chase-data))
       (svtv-chase-data (svtv-chase-signal-data
                         (make-chase-position
                          :path (address->path name)
                          :phase (- phase (svar->delay chosen-var))
                          :rsh 0 :mask new-mask))))
    (mv t svtv-chase-data))
  ///
  (verify-guards svtv-chase-follow-compare-step-fn
    :hints ((and stable-under-simplificationp
                 '(:in-theory (enable svtv-chase-datap)
                   :do-not-induct t))))

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-follow-compare-rec ((count natp)
                                   &key
                                   ((moddb moddb-ok) 'moddb)
                                   (svtv-chase-data 'svtv-chase-data))
  :returns (mv (remainder natp :rule-classes :type-prescription)
               new-svtv-chase-data)
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (svtv-chase-data->evaldata2 svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((when (zp count))
        (mv 0 svtv-chase-data))
       ((mv keep-going svtv-chase-data)
        (svtv-chase-follow-compare-step))
       ((when keep-going)
        (svtv-chase-follow-compare-rec (1- count))))
    (mv count svtv-chase-data))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))


(define svtv-chase-follow-compare (args
                                   &key
                                   ((moddb moddb-ok) 'moddb)
                                   (svtv-chase-data 'svtv-chase-data))
  :returns new-svtv-chase-data
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((unless (and (consp args)
                     (natp (car args))
                     (not (cdr args))))
        (cw! "FOLLOW-COMPARE directive must be of the form (FOLLOW-COMPARE COUNT) where COUNT is natp.  ? for more options.~%")
        svtv-chase-data)
       ((unless (svtv-chase-data->evaldata2 svtv-chase-data))
        (cw! "FOLLOW-COMPARE directive can only be run in compare mode. ? for more options.~%")
        svtv-chase-data)
       (count (car args))
       ((mv remaining-count svtv-chase-data)
        (svtv-chase-follow-compare-rec count)))
    (cw! "FOLLOW-COMPARE complete: ~x0 steps.~%" (- count remaining-count))
    (svtv-chase-print))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))





(define svtv-chase-detect-clock-deps ((vars 4vmask-alist-p)
                                      (all-vars 4vmask-alist-p))
  :returns (clockvars svarlist-p)
  (b* (((when (atom vars)) nil)
       ((unless (mbt (consp (car vars))))
        (svtv-chase-detect-clock-deps (cdr vars) all-vars))
       ((cons var mask) (car vars))
       ((svar var) (svar-fix var))
       (mask (4vmask-fix mask))
       ((unless (and (eql var.delay 1)
                     (or (sparseint-equal mask 1)
                         (sparseint-equal mask 0))
                     (let ((look (hons-assoc-equal (change-svar var :delay 0) all-vars)))
                       (and look
                            (or (sparseint-equal (4vmask-fix (cdr look)) 1)
                                (sparseint-equal (4vmask-fix (cdr look)) 0))))))
        (svtv-chase-detect-clock-deps (cdr vars) all-vars)))
    (cons var
          (cons (change-svar var :delay 0)
                (svtv-chase-detect-clock-deps (cdr vars) all-vars)))))
       
    

(define svtv-chase-follow-data-select-var ((in-val 4vec-p)
                                           (in-mask 4vmask-p)
                                           (clocks svarlist-p)
                                           (vars 4vmask-alist-p)
                                           (vals 4veclist-p))
  :returns (mv multiple (chosen-var (iff (svar-p chosen-var) chosen-var)))
  :guard (eql (len vals) (len vars))
  (b* (((when (atom vars)) (mv nil nil))
       ((unless (mbt (consp (car vars))))
        (svtv-chase-follow-x-select-var (cdr vars) vals))
       ((cons var mask) (car vars))
       ((when (member-equal (svar-fix var) (svarlist-fix clocks)))
        (svtv-chase-follow-data-select-var in-val in-mask clocks (cdr vars) (cdr vals)))
       (mask (4vmask-fix mask))
       (val (car vals))
       (mask-rsh (sparseint-trailing-0-count mask))
       (norm-mask (sparseint-ash mask (- mask-rsh)))
       (norm-val (4vec-mask-to-zero norm-mask (4vec-rsh (2vec mask-rsh) val)))
       (match (and (sparseint-equal norm-mask (4vmask-fix in-mask))
                   (equal norm-val (4vec-fix in-val))))
       ((mv multi rest-var) (svtv-chase-follow-data-select-var in-val in-mask clocks (cdr vars) (cdr vals)))
       (multi (or multi (and match rest-var))))
    (mv multi
        (and (not multi)
             (if match
                 (svar-fix var)
               rest-var))))
  ///
  (defret <fn>-lookup-in-vars
    (implies (and chosen-var
                  (4vmask-alist-p vars))
             (hons-assoc-equal chosen-var vars))))


(define svtv-chase-follow-data-step (&key
                                     ((moddb moddb-ok) 'moddb)
                                     (svtv-chase-data 'svtv-chase-data))
  :returns (mv traversedp new-svtv-chase-data)
  :verify-guards nil
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  (b* ((vars (svtv-chase-data->vars svtv-chase-data))
       (stack (svtv-chase-data->stack svtv-chase-data))
       (evaldata (svtv-chase-data->evaldata svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        (mv nil svtv-chase-data))
       ((chase-position pos) (car stack))
       (pos.var  (make-svar :name (make-address :path pos.path)))
       ((mv pos.var pos.phase) (svtv-chase-normalize-var/phase pos.var pos.phase))
       (mask-rsh (sparseint-trailing-0-count pos.mask))
       (norm-mask (sparseint-ash pos.mask (- mask-rsh)))
       (val (4vec-mask-to-zero norm-mask
                               (4vec-rsh (2vec mask-rsh)
                                         (svtv-chase-eval pos.var pos.phase evaldata))))
       ((mv ?override-mask masked-override-mask ?override-val ?computed-val)
        (svtv-chase-override-signal-data pos.var pos.phase mask-rsh nil (sparseint-val norm-mask)
                                         evaldata nil))
       ((unless (eql masked-override-mask 0))
        ;; Stop because our signal was overridden
        (mv nil svtv-chase-data))
       (varnames (alist-keys vars))
       (vals (svtv-chase-evallist varnames pos.phase evaldata))
       (clocks (svtv-chase-detect-clock-deps vars vars))
       ((mv & chosen-var) (svtv-chase-follow-data-select-var val norm-mask clocks vars vals))
       ((unless chosen-var)
        ;; Couldn't follow it back, so just stay here
        (mv nil svtv-chase-data))
       (new-mask (cdr (hons-assoc-equal chosen-var vars)))
       ((mv chosen-var phase) (svtv-chase-normalize-var/phase chosen-var pos.phase))
       (name (svar->name chosen-var))
       ((unless (address-p name))
        (cw! "The chosen signal isn't an address, so it must be an ~
                    auxiliary variable supporting an override.~%Enter P to ~
                    print current state, ? for more options.~%")
        (mv nil svtv-chase-data))
       (svtv-chase-data (svtv-chase-signal-data
                         (make-chase-position
                          :path (address->path name)
                          :phase (- phase (svar->delay chosen-var))
                          :rsh 0 :mask new-mask))))
    (mv t svtv-chase-data))
  ///
  (verify-guards svtv-chase-follow-data-step-fn
    :hints ((and stable-under-simplificationp
                 '(:in-theory (enable svtv-chase-datap)
                   :do-not-induct t))))

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-follow-data-rec ((count natp)
                                   &key
                                   ((moddb moddb-ok) 'moddb)
                                   (svtv-chase-data 'svtv-chase-data))
  :returns (mv (remainder natp :rule-classes :type-prescription)
               new-svtv-chase-data)
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((when (zp count))
        (mv 0 svtv-chase-data))
       ((mv keep-going svtv-chase-data)
        (svtv-chase-follow-data-step))
       ((when keep-going)
        (svtv-chase-follow-data-rec (1- count))))
    (mv count svtv-chase-data))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))


(define svtv-chase-follow-data (args
                                   &key
                                   ((moddb moddb-ok) 'moddb)
                                   (svtv-chase-data 'svtv-chase-data))
  :returns new-svtv-chase-data
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((unless (and (consp args)
                     (natp (car args))
                     (not (cdr args))))
        (cw! "FOLLOW-DATA directive must be of the form (FOLLOW-DATA COUNT) where COUNT is natp.  ? for more options.~%")
        svtv-chase-data)
       (count (car args))
       ((mv remaining-count svtv-chase-data)
        (svtv-chase-follow-data-rec count)))
    (cw! "FOLLOW-DATA complete: ~x0 steps.~%" (- count remaining-count))
    (svtv-chase-print))
  ///

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))






(local (defthm chase-stack-p-of-nthcdr
         (implies (chase-stack-p x)
                  (chase-stack-p (nthcdr n x)))))
    

(define svtv-chase-goto-driver ((n natp)
                                &key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* ((n (lnfix n))
       (vars (svtv-chase-data->vars svtv-chase-data))
       (stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        svtv-chase-data)
       ((chase-position pos) (car stack))
       ((unless (< n (len vars)))
        (cw "Out of range! Enter P to print current state, ? for more options.~%")
        svtv-chase-data)
       ((cons new-var new-mask) (nth n vars))
       ;; ((mv rsh mask) (svtv-chase-normalize-mask 0 new-mask))
       ((mv new-var new-phase) (svtv-chase-normalize-var/phase new-var pos.phase))
       (name (svar->name new-var))
       ((unless (address-p name))
        (cw! "The chosen signal isn't an address, so it must be an ~
                    auxiliary variable supporting an override.~%Enter P to ~
                    print current state, ? for more options.~%")
        svtv-chase-data))
    (svtv-chase-signal-data
     (make-chase-position
      :path (address->path name)
      :phase (- new-phase (svar->delay new-var))
      :rsh 0 :mask new-mask)))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix k) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth k new-svtv-chase-data)
                    (nth k svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))


(define svtv-chase-back (args
                         &key
                         ((moddb moddb-ok) 'moddb)
                         (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (or (not args)
                    (and (consp args)
                         (natp (car args))
                         (not (cdr args)))))
        (cw! "B directive with an argument must be of the form (B COUNT) where COUNT is natp.  ? for more options.~%")
        svtv-chase-data)
       (count (if args (car args) 1))
       (stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (< count (len stack)))
        (cw! "Insufficient stack -- length ~x0~%" (len stack))
        svtv-chase-data)
       (svtv-chase-data (set-svtv-chase-data->stack (nthcdr count stack) svtv-chase-data)))
    (svtv-chase-print))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-range-cmd (args
                              &key
                              ((moddb moddb-ok) 'moddb)
                              (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (and (consp args)
                     (integerp (car args))
                     (consp (cdr args))
                     (integerp (cadr args))
                     (not (cddr args))))
        (cw! "R directive must be of the form (R MSB LSB) where MSB and LSB are integers.  ? for more options.~%")
        svtv-chase-data)
       (stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\") to choose a signal, ? for more options.~%")
        svtv-chase-data)
       (pos (car stack)))
    (svtv-chase-range pos (car args) (cadr args)))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-goto-cmd (args
                             &key
                             ((moddb moddb-ok) 'moddb)
                             (aliases 'aliases)
                             (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases)))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (and (consp args)
                     (stringp (car args))
                     (consp (cdr args))
                     (svtv-labelpair-p (cadr args))
                     (not (cddr args))))
        (cw! "G directive must be of the form (G \"path\" phase) ~
                          where the first argument is a string and the second ~
                          is either a natural number offset, a phase label ~
                          (symbol), or a list (label offset).~%")
        svtv-chase-data))
    (svtv-chase-goto (car args) (cadr args)))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-ph-cmd (args
                           &key
                           ((moddb moddb-ok) 'moddb)
                           (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (and (consp args)
                     (svtv-labelpair-p (car args))
                     (not (cdr args))))
        (cw! "PH directive must be of the form (PH phase) ~
                          where the first argument is a string and the second ~
                          is either a natural number offset, a phase label ~
                          (symbol), or a list (label offset).~%")
        svtv-chase-data)
       (stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        svtv-chase-data)
       (labelpair (car args))
       (phase (svtv-chase-labelpair-phase labelpair (svtv-chase-data->phaselabels svtv-chase-data)))
       ((unless phase)
        (cw! "Error interpreting phase: ~x0 -- label not found~%" labelpair)
        svtv-chase-data)
       (pos (change-chase-position (car stack) :phase phase)))
    (svtv-chase-signal-data pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-phplusminus-cmd (plusminus
                                    args
                                    &key
                                    ((moddb moddb-ok) 'moddb)
                                    (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (and (consp args)
                     (natp (car args))
                     (not (cdr args))))
        (cw! "The PH+/PH- directive must be of the form (PH+/- n) ~
              where the argument is a natural number.~%")
        svtv-chase-data)
       (stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        svtv-chase-data)
       ((chase-position pos) (car stack))
       (incr (if (eq plusminus '+)
                 (car args)
               (- (car args))))
       (new-phase (+ pos.phase incr))
       ((unless (<= 0 new-phase))
        (cw! "The phase offset in a PH- command must be less than or equal to ~
              the current phase, which is ~x0." pos.phase)
        svtv-chase-data)
       (new-pos (change-chase-position pos :phase new-phase)))
    (svtv-chase-signal-data new-pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-toggle-smartp (&key
                                    ((moddb moddb-ok) 'moddb)
                                    (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* ((smartp (svtv-chase-data->smartp svtv-chase-data))
       (new-smartp (not smartp))
       (svtv-chase-data (set-svtv-chase-data->smartp new-smartp svtv-chase-data)))
    (cw! "Turned data-aware dependency reduction ~s0.~%"
         (if new-smartp "on" "off"))
    (svtv-chase-print))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-data-set-mask-mode ((val symbolp)
                                  &key
                                    (svtv-chase-data 'svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* ((svtv-chase-data (set-svtv-chase-data->print-with-mask-mode val svtv-chase-data)))
    (case val
      ((t)   (cw! "Changed mask mode: always apply caremask to signal values~%"))
      ((nil) (cw! "Changed mask mode: never  apply caremask to signal values~%"))
      (t (cw! "Changed mask mode: apply caremask to signal values if it results in a 2vec~%")))
    svtv-chase-data)
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->print-with-mask-mode*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-data-set-print-overrides-mode ((val symbolp)
                                                  &key
                                                  (svtv-chase-data 'svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* ((svtv-chase-data (set-svtv-chase-data->print-overrides-mode val svtv-chase-data)))
    (if val
        (cw! "Changed override mode: print overrides for dependencies~%")
      (cw! "Changed override mode: print overrides only for current signal~%"))
    svtv-chase-data)
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->print-overrides-mode*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))



(define svtv-chase-goto-output-cmd (args
                                      &key
                                    ((moddb moddb-ok) 'moddb)
                                    (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns new-svtv-chase-data
  (b* (((unless (and (consp args)
                     (not (cdr args))))
        (cw! "O directive must be of the form (O name).~%")
        svtv-chase-data)
       (svtv-chase-data (svtv-chase-goto-output (car args))))
    svtv-chase-data)
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

(define svtv-chase-print-expr-cmd (args
                                   &key
                                   (svtv-chase-data 'svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  (b* (((unless (and (consp args)
                     (acl2::maybe-natp (car args))
                     (not (cdr args))))
        (cw! "EXPR directive must be of the form (EXPR depth).~%")))
    (acl2::fmt-to-comment-window! "~x0~%"
                                  `((#\0 . ,(svtv-chase-data->expr svtv-chase-data)))
                                  0 (evisc-tuple (car args) nil nil nil) nil)))


(define svtv-chase-ev-cmd (args
                           &key (state 'state))
  (b* (((unless (and (consp args)
                     (not (cdr args))))
        (cw! "EV directive must be of the form (EV term).~%"))
       (attachment (fgetprop 'simple-translate-and-eval-logic 'acl2::attachment nil (w state)))
       ((unless (and attachment
                     (alistp attachment)
                     (eq (cdr (assoc-eq 'simple-translate-and-eval-logic attachment))
                         'acl2::simple-translate-and-eval-cmp)))
        (cw! "In order to use EV you must set ~x0 as the ~
                          attachment for ~x1, as in the following ~
                          form:~%~x2~%Note that to (mostly) undo this you may ~
                          do:~%~x3~%"
             'acl2::simple-translate-and-eval-cmp
             'simple-translate-and-eval-logic
             '(setup-ev-for-chase)
             '(unsetup-ev-for-chase)))
       ((mv err term-dot-val)
        (simple-translate-and-eval-logic (car args) nil nil "The argument to EV"
                                         'svtv-chase-rep (w state) state t nil nil))
       ((when (or err (not (consp term-dot-val))))
        (cw! "Failed to evaluate: ~@0~%" term-dot-val)))
    (cw! "~x0~%" (cdr term-dot-val))))


(define svtv-chase-hist-cmd (args
                             &key
                             ((moddb moddb-ok) 'moddb)
                             (svtv-chase-data 'svtv-chase-data))
  :guard (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :guard-debug t
  (b* (((unless (or (atom args)
                    (and (acl2::maybe-posp (car args))
                         (not (cdr args)))))
        (cw! "HIST directive takes at most 1 argument which must be a natural number.~%")))
    (svtv-chase-print-history (or (and (consp args) (car args)) 1))))

(defconst *chase-usage*
  "
What you can enter at the SVTV-CHASE prompt:

 ?                  prints this help message

 X                  Exit the chase read-eval-print loop.

 P                  prints the current state, including the next signal choices

 (G \"path\" phase) Go to the signal named by the given path at the given phase.
                    The phase may be specified as a natural number (offset), 
                    a phase label from the defsvtv form, or a combination
                    (label num), meaning the numth phase after label.

 (PH phase)         Go to the given phase, keeping the same current signal.
 (PH+ n)            Go forward in time N phases.
 (PH- n)            Go backward in time N phases.

 HIST               Show the history of the current signal from time 0 to 
 (HIST n)           the last phase. For the form with an argument, skip by
                    increments of N phases, picking the first phase
                    such that the current phase is included.

 (O name)           Go to the signal/phase corresponding to the named pipeline output.

 (R MSB LSB)        Select the given MSB:LSB range of the current signal

 Natural number     Select the given choice of next signal
 B                  Go back to the previous signal on the stack.
 (B N)              Go back N stack frames. (B 1) is the same as just B.
 STACK              Print the stack, showing the signals you can go back to.

 (FOLLOW-XES N)     For up to N times, if there is a unique driver that has an X
                    in its care-bits, follow that driver.

 (FOLLOW-DATA N)    For up to N times, if there is a unique driver that matches the
                    caremask and data value of the current signal (and doesn't seem
                    to be a clock), follow that driver.

 (FOLLOW-COMPARE N) In compare mode only, for up to N times, if there is a unique
                    driver which has a different value in the intersection of the
                    care bits of its two evaluations, follow that driver.

 EXPR               Print the assignment for the current signal.
 (EXPR N)           Print the assignment expression, limiting nesting depth to N.

 SMARTP             Toggle data-aware dependency reduction feature
                    (reduces the number of irrelevant signals listed).
                    On by default.

 (EV form)          Evaluates form using simple-translate-and-eval 
                    and prints the result.  You need to set up an attachment
                    to do this, which you can do by running
                    (sv::setup-ev-for-chase) in the ACL2 loop.  You can undo this
                    with (sv::unsetup-ev-for-chase).

 MASK-ALWAYS
 MASK-NEVER
 MASK-DEFAULT       Affects whether signal values are ANDed with the caremask before
                    printing.  The default is to only AND the value with the mask if
                    the unmasked value is not 2valued and the masked value is 2valued.
                    The -always and -never settings change this to always/never 
                    (respectively) AND the value with the mask.

 OVERRIDES-VERBOSE
 OVERRIDES-DEFAULT  Affects whether the derivation of signal values from overrides is
                    printed for only the current signal (default) or for all
                    dependencies as well (verbose).
")

(define svtv-chase-rep (&key
                        ((moddb moddb-ok) 'moddb)
                        (svtv-chase-data 'svtv-chase-data)
                        (aliases 'aliases)
                        (state 'state))
  :guard (and (open-input-channel-p *standard-oi* :object state)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)
                ;; (and stable-under-simplificationp
                ;;      '(:in-theory (enable chase-position-addr-p)))
                ))
  :returns (mv exitp new-svtv-chase-data new-state)
  (b* ((- (cw! "SVTV-CHASE > "))
       ((mv err obj state) (read-object *standard-oi* state))
       ((when err)
        (mv t svtv-chase-data state))
       ((unless (or (natp obj)
                    (symbolp obj)
                    (and (true-listp obj) (symbolp (car obj)))))
        (cw! "Bad command: must be a natural number, symbol, or list beginning with a symbol~%")
        (mv nil svtv-chase-data state))
       ((when (natp obj))
        (b* ((svtv-chase-data (svtv-chase-goto-driver obj)))
          (mv nil svtv-chase-data state)))
       (cmd (symbol-name (if (symbolp obj) obj (car obj))))
       (args (if (symbolp obj) nil (cdr obj)))
       ((when (and (member-equal cmd '("X" "?" "P" "EXPR" "STACK" "SMARTP" "MASK-ALWAYS"
                                       "MASK-NEVER" "MASK-DEFAULT"
                                       "OVERRIDES-VERBOSE" "OVERRIDES-DEFAULT"))
                   args))
        (cw! "Command ~s0 should not have arguments~%" cmd)
        (mv nil svtv-chase-data state))
       ((when (equal cmd "X"))
        (mv t svtv-chase-data state))
       (svtv-chase-data
        (b* (((when (equal cmd "?"))
              (cw! *chase-usage*)
              svtv-chase-data)
             ((when (equal cmd "P"))
              (svtv-chase-print))
             ((when (equal cmd "EXPR"))
              (cw! "~x0~%" (svtv-chase-data->expr svtv-chase-data))
              svtv-chase-data)
             ((when (equal cmd "B")) (svtv-chase-back args))
             ((when (equal cmd "STACK"))
              (prog2$ (svtv-chase-print-stack) svtv-chase-data))
             ((when (equal cmd "SMARTP")) (svtv-chase-toggle-smartp))
             ((when (equal cmd "MASK-ALWAYS"))
              (svtv-chase-data-set-mask-mode t))
             ((when (equal cmd "MASK-NEVER"))
              (svtv-chase-data-set-mask-mode nil))
             ((when (equal cmd "MASK-DEFAULT"))
              (svtv-chase-data-set-mask-mode :default))
             ((when (equal cmd "OVERRIDES-VERBOSE"))
              (svtv-chase-data-set-print-overrides-mode t))
             ((when (equal cmd "OVERRIDES-DEFAULT"))
              (svtv-chase-data-set-print-overrides-mode nil))
             ((when (equal cmd "FOLLOW-XES")) (svtv-chase-follow-xes args))
             ((when (equal cmd "FOLLOW-COMPARE")) (svtv-chase-follow-compare args))
             ((when (equal cmd "FOLLOW-DATA")) (svtv-chase-follow-data args))
             ((when (equal cmd "R")) (svtv-chase-range-cmd args))
             ((when (equal cmd "G")) (svtv-chase-goto-cmd args))
             ((when (equal cmd "PH")) (svtv-chase-ph-cmd args))
             ((when (equal cmd "PH+")) (svtv-chase-phplusminus-cmd '+ args))
             ((when (equal cmd "PH-")) (svtv-chase-phplusminus-cmd '- args))
             ((when (equal cmd "HIST")) (svtv-chase-hist-cmd args) svtv-chase-data)
             ((when (equal cmd "O"))
              (svtv-chase-goto-output-cmd args))
             ((when (equal cmd "EXPR"))
              (svtv-chase-print-expr-cmd args)
              svtv-chase-data)
             ((when (equal cmd "EV"))
              (svtv-chase-ev-cmd args)
              svtv-chase-data))
          (cw! "Error -- unrecognized directive: ~x0~%Type ? for allowed commands.~%" obj)
          svtv-chase-data)))
    (mv nil svtv-chase-data state))
  ///
  (defret file-measure-of-svtv-chase-rep-weak
    (<= (file-measure *standard-oi* new-state)
        (file-measure *standard-oi* state))
    :rule-classes :linear)

  (defret file-measure-of-svtv-chase-rep-strong
    (implies (not exitp)
             (< (file-measure *standard-oi* new-state)
                (file-measure *standard-oi* state)))
    :rule-classes :linear)

  (defret open-input-channel-p1-of-<fn>
    (implies (open-input-channel-p1 *standard-oi* :object state)
             (open-input-channel-p1 *standard-oi* :object new-state)))
  
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*
                                               *svtv-chase-data->print-with-mask-mode*
                                               *svtv-chase-data->print-overrides-mode*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

  

(define svtv-chase-repl1 (&key
                         ((moddb moddb-ok) 'moddb)
                         (aliases 'aliases)
                         (svtv-chase-data 'svtv-chase-data)
                         (state 'state))
  :guard (and (open-input-channel-p *standard-oi* :object state)
              ;; (svarlist-addr-p (svexlist-collect-vars (svex-alist-vals (debugdata->override-assigns debugdata))))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns (mv new-svtv-chase-data new-state)
  :measure (file-measure *standard-oi* state)
  :parents (svtv-chase)
  :short "Re-enter the @(see svtv-chase) read-eval-print loop, with no change to the environment or SVTV."
  (b* (((mv exitp svtv-chase-data state) (svtv-chase-rep))
       ((when exitp)
        (cw! "~%Exiting SVTV-CHASE.  You may execute ~x0 to re-enter or ~x1 ~
              to change the simulation inputs.~%"
             '(svtv-chase-repl) '(svtv-chase-update env))
        (mv svtv-chase-data state)))
    (svtv-chase-repl1))

  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*
                                               *svtv-chase-data->print-with-mask-mode*
                                               *svtv-chase-data->print-overrides-mode*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))



; Matt K. mod to accommodate change to guards for fmt-hard-right-margin and
; fmt-soft-right-margin around August 2023.
(local (in-theory (enable acl2::all-boundp-initial-global-table)))

(verify-termination fmt-hard-right-margin)
(verify-termination fmt-soft-right-margin)


(local (defthm open-input-channel-p1-of-put-global
         (equal (open-input-channel-p1 channel type (put-global key val state))
                (open-input-channel-p1 channel type state))
         :hints(("Goal" :in-theory (enable open-input-channel-p1
                                           open-input-channels
                                           put-global
                                           update-global-table)))))

(local (in-theory (disable put-global)))

(define svtv-chase-repl (&key
                         ((moddb moddb-ok) 'moddb)
                         (aliases 'aliases)
                         (svtv-chase-data 'svtv-chase-data)
                         (state 'state))
  :guard (and (open-input-channel-p *standard-oi* :object state)
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases)))
  :prepwork ((local (in-theory (disable nth))))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns (mv new-svtv-chase-data new-state)

  ;; Just wraps svtv-chase-repl1 with a setting of the fmt margins.
  (b* ((hard (pos-fix (and (boundp-global 'fmt-hard-right-margin state)
                           (f-get-global 'fmt-hard-right-margin state))))
       (soft (pos-fix (and (boundp-global 'fmt-soft-right-margin state)
                           (f-get-global 'fmt-soft-right-margin state))))
       (state (set-fmt-hard-right-margin 200 state))
       (state (set-fmt-soft-right-margin 150 state))
       ((mv svtv-chase-data state) (svtv-chase-repl1))
       (state (set-fmt-soft-right-margin soft state))
       (state (set-fmt-hard-right-margin hard state)))
    (mv svtv-chase-data state))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*
                                               *svtv-chase-data->print-with-mask-mode*
                                               *svtv-chase-data->print-overrides-mode*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))


