/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { RuntimeError } from '../errors';
import { stringify } from '../util/stringify';
import { stringifyForError } from './util/stringify_utils';
/** Called when directives inject each other (creating a circular dependency) */
export function throwCyclicDependencyError(token, path) {
    const depPath = path ? `. Dependency path: ${path.join(' > ')} > ${token}` : '';
    throw new RuntimeError(-200 /* RuntimeErrorCode.CYCLIC_DI_DEPENDENCY */, `Circular dependency in DI detected for ${token}${depPath}`);
}
export function throwMixedMultiProviderError() {
    throw new Error(`Cannot mix multi providers and regular providers`);
}
export function throwInvalidProviderError(ngModuleType, providers, provider) {
    if (ngModuleType && providers) {
        const providerDetail = providers.map(v => v == provider ? '?' + provider + '?' : '...');
        throw new Error(`Invalid provider for the NgModule '${stringify(ngModuleType)}' - only instances of Provider and Type are allowed, got: [${providerDetail.join(', ')}]`);
    }
    else if (provider.ɵproviders) {
        throw new RuntimeError(207 /* RuntimeErrorCode.PROVIDER_IN_WRONG_CONTEXT */, `Invalid providers from 'importProvidersFrom' present in a non-environment injector. 'importProvidersFrom' can't be used for component providers.`);
    }
    else {
        throw new Error('Invalid provider');
    }
}
/** Throws an error when a token is not found in DI. */
export function throwProviderNotFoundError(token, injectorName) {
    const injectorDetails = injectorName ? ` in ${injectorName}` : '';
    throw new RuntimeError(-201 /* RuntimeErrorCode.PROVIDER_NOT_FOUND */, ngDevMode && `No provider for ${stringifyForError(token)} found${injectorDetails}`);
}
//# sourceMappingURL=data:application/json;base64,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