/***************************************************************************
 *                                                                         *
 *   copyright : (C) 2007 The University of Toronto                        *
 *                   netterfield@astro.utoronto.ca                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "fftoptions.h"
#include "dialogdefaults.h"

namespace Kst {

FFTOptions::FFTOptions(QWidget *parent)
    : QWidget(parent) {
  setupUi(this);

  connect(_interleavedAverage, SIGNAL(clicked()), this, SLOT(clickedInterleaved()));
  connect(_apodize, SIGNAL(clicked()), this, SLOT(clickedApodize()));
  connect(_apodizeFunction, SIGNAL(currentIndexChanged(int)), this, SLOT(changedApodizeFxn()));
  connect(_apodize, SIGNAL(clicked()), this, SLOT(changedApodizeFxn()));

  connect(_apodizeFunction, SIGNAL(currentIndexChanged(int)), this, SIGNAL(modified()));
  connect(_output, SIGNAL(currentIndexChanged(int)), this, SIGNAL(modified()));
  connect(_apodize, SIGNAL(clicked()), this, SIGNAL(modified()));
  connect(_removeMean, SIGNAL(clicked()), this, SIGNAL(modified()));
  connect(_interleavedAverage, SIGNAL(clicked()), this, SIGNAL(modified()));
  connect(_sampleRate, SIGNAL(textChanged(QString)), this, SIGNAL(modified()));
  connect(_vectorUnits, SIGNAL(textChanged(QString)), this, SIGNAL(modified()));
  connect(_rateUnits, SIGNAL(textChanged(QString)), this, SIGNAL(modified()));
  connect(_sigma, SIGNAL(valueChanged(double)), this, SIGNAL(modified()));
  connect(_FFTLength, SIGNAL(valueChanged(int)), this, SIGNAL(modified()));


  _FFTLength->setEnabled(false);

}


FFTOptions::~FFTOptions() {}


void FFTOptions::init() {
  update();
}

double FFTOptions::sampleRate() const {
  return _sampleRate->text().toDouble();
}


bool FFTOptions::sampleRateDirty() const {
  return !_sampleRate->text().isEmpty();
}


void FFTOptions::setSampleRate(const double sampleRate) {
  _sampleRate->setText(QString::number(sampleRate));
}


double FFTOptions::sigma() const {
  return _sigma->value();
}


bool FFTOptions::sigmaDirty() const {
  return !_sigma->text().isEmpty();
}


void FFTOptions::setSigma(const double sigma) {
  _sigma->setValue(sigma);
}


bool FFTOptions::interleavedAverage() const {
  return _interleavedAverage->isChecked();
}


bool FFTOptions::interleavedAverageDirty() const {
  return _interleavedAverage->checkState() == Qt::PartiallyChecked;
}


void FFTOptions::setInterleavedAverage(const bool interleavedAverage) {
  _interleavedAverage->setChecked(interleavedAverage);
  _FFTLength->setEnabled(interleavedAverage);
}


bool FFTOptions::apodize() const {
  return _apodize->isChecked();
}


bool FFTOptions::apodizeDirty() const {
  return _apodize->checkState() == Qt::PartiallyChecked;
}


void FFTOptions::setApodize(const bool apodize) {
  _apodize->setChecked(apodize);
  _apodizeFunction->setEnabled(apodize);
}


bool FFTOptions::removeMean() const {
  return _removeMean->isChecked();
}


bool FFTOptions::removeMeanDirty() const {
  return _removeMean->checkState() == Qt::PartiallyChecked;
}


void FFTOptions::setRemoveMean(const bool removeMean) {
  _removeMean->setChecked(removeMean);
}


int FFTOptions::FFTLength() const {
  return _FFTLength->value();
}


bool FFTOptions::FFTLengthDirty() const {
  return !_FFTLength->text().isEmpty();
}


void FFTOptions::setFFTLength(const int FFTLength) {
  _FFTLength->setValue(FFTLength);
}


QString FFTOptions::vectorUnits() const {
  return _vectorUnits->text();
}


bool FFTOptions::vectorUnitsDirty() const {
  return !_vectorUnits->text().isEmpty();
}


void FFTOptions::setVectorUnits(const QString vectorUnits) {
  _vectorUnits->setText(vectorUnits);
}


QString FFTOptions::rateUnits() const {
  return _rateUnits->text();
}


bool FFTOptions::rateUnitsDirty() const {
  return !_rateUnits->text().isEmpty();
}


void FFTOptions::setRateUnits(const QString rateUnits) {
  _rateUnits->setText(rateUnits);
}


ApodizeFunction FFTOptions::apodizeFunction() const {
  return (ApodizeFunction)_apodizeFunction->currentIndex();
}


bool FFTOptions::apodizeFunctionDirty() const {
  return _apodizeFunction->currentIndex() != -1;
}


void FFTOptions::setApodizeFunction(const ApodizeFunction apodizeFunction) {
  _apodizeFunction->setCurrentIndex((ApodizeFunction)apodizeFunction);
}


PSDType FFTOptions::output() const {
  return (PSDType)_output->currentIndex();
}


bool FFTOptions::outputDirty() const {
  return _output->currentIndex() != -1;
}


void FFTOptions::setOutput(const PSDType output) {
  _output->setCurrentIndex((PSDType)output);
}


void FFTOptions::clearValues() {
  _sigma->clear();
  _FFTLength->clear();
  _sampleRate->clear();
  _vectorUnits->clear();
  _rateUnits->clear();
  _apodize->setCheckState(Qt::PartiallyChecked);
  _interleavedAverage->setCheckState(Qt::PartiallyChecked);
  _removeMean->setCheckState(Qt::PartiallyChecked);
  _apodizeFunction->setCurrentIndex(-1);
  _output->setCurrentIndex(-1);
}


void FFTOptions::changedApodizeFxn() {
  int gaussianIndex = 5;
  if (_apodizeFunction->itemText(0).isEmpty()) {
    ++gaussianIndex;
  }
  _sigma->setEnabled(_apodizeFunction->currentIndex() == gaussianIndex && _apodize->isChecked());
  _sigmaLabel->setEnabled(_apodizeFunction->currentIndex() == gaussianIndex && _apodize->isChecked());
}


void FFTOptions::clickedInterleaved() {
  _FFTLength->setEnabled(_interleavedAverage->isChecked());
  _FFTLengthLabel->setEnabled(_interleavedAverage->isChecked());
}


void FFTOptions::clickedApodize() {
  _apodizeFunction->setEnabled(_apodize->isChecked());
}


void FFTOptions::synch() {
  clickedInterleaved();
  clickedApodize();
}


bool FFTOptions::checkValues() {
  double new_freq = _sampleRate->text().toDouble();
  int new_len = _FFTLength->text().toInt();
  return checkGivenValues(new_freq, new_len);
}


bool FFTOptions::checkGivenValues(double sampleRate, int FFTLength) {
  if (sampleRate <= 0) {
    return false;
  }
  if (FFTLength < 2) {
    return false;
  }
  return true;
}

// store the current state of the widget as the default
void FFTOptions::setWidgetDefaults() {
  dialogDefaults().setValue("spectrum/freq", sampleRate());
  dialogDefaults().setValue("spectrum/average", interleavedAverage());
  dialogDefaults().setValue("spectrum/len", FFTLength());
  dialogDefaults().setValue("spectrum/apodize", apodize());
  dialogDefaults().setValue("spectrum/removeMean", removeMean());
  dialogDefaults().setValue("spectrum/vUnits", vectorUnits());
  dialogDefaults().setValue("spectrum/rUnits", rateUnits());
  dialogDefaults().setValue("spectrum/apodizeFxn", apodizeFunction());
  dialogDefaults().setValue("spectrum/gaussianSigma", sigma());
  dialogDefaults().setValue("spectrum/output", output());
}

// set the widget to the stored default values
void FFTOptions::loadWidgetDefaults() {
  setSampleRate(dialogDefaults().value("spectrum/freq",100.0).toDouble());
  setInterleavedAverage(dialogDefaults().value("spectrum/average",true).toBool());
  setFFTLength(dialogDefaults().value("spectrum/len",12).toInt());
  setApodize(dialogDefaults().value("spectrum/apodize",true).toBool());
  setRemoveMean(dialogDefaults().value("spectrum/removeMean",true).toBool());
  setVectorUnits(dialogDefaults().value("spectrum/vUnits","V").toString());
  setRateUnits(dialogDefaults().value("spectrum/rUnits","Hz").toString());
  setApodizeFunction(ApodizeFunction(dialogDefaults().value("spectrum/apodizeFxn",WindowOriginal).toInt()));
  setSigma(dialogDefaults().value("spectrum/gaussianSigma",1.0).toDouble());
  setOutput(PSDType(dialogDefaults().value("spectrum/output",PSDPowerSpectralDensity).toInt()));
}

}
// vim: ts=2 sw=2 et
