// Copyright 2020 Consensys Software Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package bw6756

import (
	"github.com/consensys/gnark-crypto/ecc/bw6-756/fp"

	"math/big"
)

//Note: This only works for simple extensions

func g2IsogenyXNumerator(dst *fp.Element, x *fp.Element) {
	g2EvalPolynomial(dst,
		false,
		[]fp.Element{
			{18063257888214992592, 1393496779859630263, 8564933104033120466, 1364340956348512280, 2179459776642261738, 7246713717490156421, 11399242774977305051, 4643984571529960155, 5648608782916908274, 3095467176315578154, 3467607453741162011, 2961445036873290},
			{11658835549372914133, 12286399758052775709, 12748454159994037005, 8627322543101307557, 8488782471683565130, 3311860832147265783, 16666746629317462942, 3993283414811786302, 9484238281806419324, 9079179108706802563, 4853121176603729598, 4115220294490375},
			{9804516088925396249, 1689484976811888131, 6709027706281311158, 15148698413255486787, 4456196932928240544, 13089600824592779987, 4915748753406253244, 8824016979977565722, 5801663995596799579, 18101997615639527323, 2024314854882138752, 3982056613208256},
			{2442176974808264130, 1036024567776342463, 3553154388034084009, 5407139347352086633, 18271230407247584526, 15484355645761637943, 751188679793694145, 4984314664481512509, 610694156402706023, 4400103409344523273, 10219908803514347843, 147859773786044},
			{4103257481960472996, 3407771621173017780, 10892974841255474934, 16253724974777807377, 6376172419506182724, 15853776788744583196, 1223963103902005581, 13511207157402706386, 11263217018011554306, 4332839954383721397, 5081830234251762489, 2677604750100570},
			{16143817283423817884, 16868200401550225159, 17247373971233108250, 4471527367127116953, 3288264420010388700, 13080898939531555317, 309418061729162402, 4570902553714880875, 12501140036399581948, 16460091225458830175, 3432750371563496540, 1856425305308313},
			{6616034961392165287, 13262605314780833274, 11095673701643109564, 15659197846522771275, 6221107675939471385, 10040356795873210634, 7142688090102529240, 5217229900361698120, 107142442904534986, 4880883056892442152, 17848994686119140771, 2209292322719356},
			{8209889887433583591, 11125531176963853968, 10429304465365845536, 9698924055228808053, 11572483637041790867, 8471024052732620731, 16171107090788573972, 17389518512515618487, 2446717027925821414, 13845569984714024499, 10358707895980715431, 1981955952337795},
			{17440571817547589032, 14821760434103399038, 3467158545118505628, 1052069839390006688, 16597307299261243349, 1514696147834692288, 1502298074058409421, 9724159673512954906, 17146431833140791833, 12568150970058967052, 4051586399753035900, 3034883876673207},
			{6885431716405590573, 11285374911458481328, 8536774051778827233, 537308629644561775, 4978081861472261373, 6895244255083366290, 12771428336377158707, 18366942941316985453, 3885842168611052213, 15733883558726650552, 1898691920676539973, 1053947946004101},
			{6438046397076616827, 8748193276523668278, 12924065944264387518, 10396411676633151501, 8884810883874904938, 11581924250560173624, 7958603540887013994, 11929384227933921591, 12058158172733608199, 13011125749180986554, 8011584745771515935, 2769359570595999},
			{8485125453864892728, 9496478782187944243, 9835406788412012827, 13148749213470338156, 14346720249036467428, 11042761504554563134, 9885278452952011517, 2699400820004207727, 6861670924206724915, 10779917832009994523, 6892501690509198304, 1120589998747771},
			{17370092359602020084, 5939626380863061634, 15363278959077316883, 17700203104572168868, 6623108653336641881, 12109660681271032827, 6997648647599827246, 7989523002331856142, 2551599959583933382, 13522924635998791246, 10964309953846145036, 2038629017267546},
			{8905023910868005352, 2644351245783489261, 12712934134422018629, 2320762562258764217, 18070651629628507397, 9497759956968634620, 1196234375306649517, 11762569576125533470, 12760978227532514829, 2100402702948591945, 14717565707450064032, 3190123599240964},
			{2321122152288065791, 8559934930487477171, 2816091705874298985, 12816347129408254318, 13759444057793103650, 9245275761719025150, 4583905162646958584, 193512411034013027, 5373879090500334116, 14965705556666027214, 15938856173769808949, 1679542184021728},
			{3064863255627260738, 1322412353639848785, 18016733159894272674, 10335342398228990822, 1019043548854784599, 12367531741647476417, 16806389067180361238, 11090164111872500153, 8574474748101498214, 10816633183836176713, 6244604899848898765, 3568151292359233},
			{13882469591933839029, 3780726533369959129, 356529851879299843, 4882617127020139027, 7330450063362196924, 16839599917433694663, 15301540460601913522, 9954563827398335290, 1349170473954715280, 9235680880529941043, 1342314491728118367, 3440448802314722},
			{9829752897409242023, 15704098846998971558, 3953782753040335701, 13733604128466395046, 15940385395599933276, 11943859108918239708, 14026342461240088190, 5157027042114130164, 17730504431960750988, 5346615480493660819, 1155792137343476704, 317557884360030},
		},
		x)
}

func g2IsogenyXDenominator(dst *fp.Element, x *fp.Element) {
	g2EvalPolynomial(dst,
		true,
		[]fp.Element{
			{7676806743992704411, 13899270453609261655, 12451810503333105973, 7586377905044480253, 4400073814052305663, 8772770626841823456, 13952615522127684289, 14011211693111766274, 8877781094782098197, 15891608636494308341, 15564328328945326253, 2909487247725599},
			{9453865609050903521, 16187066946506957518, 11903964155097034943, 8713536912762661644, 14536674352530403432, 9468882702901471210, 7150526403476766966, 5620694243891780279, 8606838083802511379, 15607157215838277797, 15500728334841342059, 2231399860031313},
			{16980338680259670981, 11521927360174526230, 3296865033313109705, 17336540695467956468, 16363282394107312970, 8395304954809411637, 4445751931845641868, 15986692066173048677, 10373032380811327027, 7515107668367104040, 1285158972258773404, 3381474465871678},
			{12590065689114661726, 5946446231937317556, 6252935342450242612, 3946482967954549718, 5477460783051802954, 9554038960651143508, 14284409639302464644, 8381919432766671976, 15064388857143239334, 11198709445699977935, 1505776422996134004, 228166822329897},
			{8718455273244450707, 18257164560137771694, 63167664726716276, 10939327297441407123, 6906792779173733504, 6910059409960120006, 15613529393540514307, 9642032594321541892, 11604953731100391742, 12576353621847180703, 16967498256620051397, 3178252700562399},
			{868685242467462437, 6117772598772167454, 180017912061117271, 6682693383856096681, 14087880244443394467, 10065045297523127645, 12651808868021479222, 253494832475484277, 13284798185615450963, 9622862694522664923, 12723192387819738881, 3881837484675755},
			{16652886415926729528, 15222425659919396075, 7883409001976488877, 108334016999777799, 2139096560884051070, 9235143524812320273, 362394594825970392, 1985075437469393064, 15285298004551295011, 4694377468798011199, 14002413277245730823, 1031059376762955},
			{11607763233042652646, 15107090951981795493, 750429153195033507, 15447410638764619883, 17517491898415918026, 6872460098882921300, 15826126419240939753, 13286003994964494101, 18044211163024483308, 996549296944201754, 658907883085288864, 3183797990537945},
			{4230116250557350613, 3657094985405752339, 7217616869566592695, 1233781281998910073, 17857880953005547064, 6893178357644011918, 14339433834014725723, 15278301383782487737, 14319056202540821048, 9161061358875487159, 1721449113640690872, 4232195864691114},
			{9776057899978497849, 18295317405102606195, 499915739732625100, 9734977242896698887, 8865331344776907994, 11499915146337835230, 2528236165698619045, 12194855496531305052, 9067458767906246251, 18260802818211288667, 2059444010111999228, 1594971276166605},
			{16948142999539174921, 7749442011298877799, 8782365616033113710, 10062902276000792382, 8578445819659278006, 18441606647876159181, 12027817196043418722, 10967829751748336848, 13807613859264788489, 14146619962209840535, 237985113988347841, 3780579450083530},
			{2994714778176611012, 15395498331776333933, 11906372358806379274, 3870528935878244793, 4010010522904287132, 9137750020304910250, 15901170280015152991, 10139251366754058601, 11005484872982035363, 8422618689683690561, 14842869397711928390, 3247505200278147},
			{101073571214872246, 18383559315189060379, 6907519649047275056, 13062498176165399437, 16983943441557859234, 18299740594093156677, 5880497128789512121, 3932944625598144568, 5690929158366038599, 7602942967413953653, 13778217666122444261, 1907343141630194},
			{2065675606533188185, 835926001041832944, 2586561621060881533, 8657377230123209903, 10089711764069753420, 3314659207458483825, 17753035059206418973, 14321321176141429822, 14378558285243871759, 12481010464609732173, 1367439684858703947, 117846213348560},
			{1753375113789138573, 5582726390451056543, 18111137772277889143, 8599937063653169974, 9539333829070716249, 10105075059414936548, 13701354987781444925, 15726824122155311967, 8194284482795973630, 17314739116151717281, 9538018392868702110, 1367705449205283},
			{9090248073906778481, 2107788472131405778, 12924810070024706928, 15005609447201957126, 961945466227986008, 8749282965734757112, 7718473509162793608, 10637582677835211604, 13790836079088289949, 4207625851985012931, 2762901280724194593, 1893895310850181},
		},
		x)
}

func g2IsogenyYNumerator(dst *fp.Element, x *fp.Element, y *fp.Element) {
	var _dst fp.Element
	g2EvalPolynomial(&_dst,
		false,
		[]fp.Element{
			{5051598811638517557, 737162936682204453, 18266277940781649426, 8297934493309658251, 14587852221828932567, 17520883813942983557, 10498215533406532450, 5832211223751606295, 3600880350437357340, 2825910535639299749, 12647163569588255163, 275061126760636},
			{12559678814741632895, 7859048468261858149, 4236488078359479360, 16841239042007152658, 5312075531742262702, 3178741308357192977, 14664365762224592012, 14895611710779307411, 17003497102161051726, 12561805960382829864, 1689612490434121511, 4339709385293787},
			{4476631763572289814, 16082167790703866320, 7394388887204004896, 3955363014434416656, 13817310402283032879, 6647511166122297131, 14720286221254837855, 4472005002784771460, 4703528211196572387, 15590281238800563442, 15593255919960024691, 2166688893510683},
			{2503229986694465765, 8594864623388797577, 11771007318796339592, 12178353071709987741, 9395567834543143117, 1623478240275898563, 10422013481936289821, 5751815158810662048, 6129775232805856798, 16015977650288458746, 11063526906033396515, 359428695871498},
			{3172204394861266886, 7178324459329989193, 12055192196751786661, 9556747627156995614, 7108346223162691745, 12196486755584093700, 13296299274967806249, 743835019303585217, 12362670244290053814, 6976826164013006385, 6645425301774832941, 3420842202901129},
			{7103471633677248769, 10501892449765124953, 7979329109825360150, 16750428365225432440, 15586582932171624974, 14076198087256240428, 2628477998626483788, 7821198768462694495, 7466756118781704838, 13948888709477971062, 7927369204535991101, 2058102271014739},
			{5288823391484243258, 13731831771514171493, 12881614458960554030, 12148729456237475052, 8848547805942475948, 12243669284994705772, 16485291633832482603, 16058960579070538785, 17547062151649259172, 6905305693038127501, 18220013145790156749, 2972569751643637},
			{13361596835977281893, 17117667659587365983, 6339637372565868085, 15717095851481368394, 16547894374671072774, 5617618166823667001, 14514184715672603774, 7631006970098262425, 6062683165562369640, 14437434371647350980, 18414922128376530006, 2265037525519879},
			{9106851493396134797, 3871350947879893854, 11419315593021252893, 5677821281985076357, 11422846285152591355, 9656976370798298175, 11940691516517696790, 8167972214220646424, 12377408730237178633, 18329224238304815706, 5868665003522831600, 4308812217155104},
			{7457550484179549606, 17785821431645724103, 8610295788728590146, 4940001953956769495, 3943521744044194208, 4168544217828160098, 5813372651329180488, 9347943620560663705, 15479905250276510060, 14862637707067876829, 17259565500085355864, 1633018135761863},
			{10835542273568905129, 8092036257982094713, 14709920468851814741, 16853250056678220902, 6812464373424184914, 5889056178203244050, 17630558818321130987, 11291602263438171858, 5363215832179755730, 1406328387408005099, 17557253559133413174, 4179578472005619},
			{14855153380663865487, 13851563109375821599, 13068709803362046654, 11885935936878191666, 5065974774005451260, 2466622550050362922, 18101639232595559187, 7571004564246793426, 11590196373886920649, 4421221247140818609, 1615676370004474465, 1368767916401153},
			{3229602409663261297, 18370132806978542041, 8156118142877923158, 10568467618403222100, 4717245830849486971, 1321623212661780313, 2402246841054709322, 1497696416423214573, 478732305369471040, 4937841867770020731, 15317670803771231546, 1916834553357246},
			{5311741649814063446, 7140916177349472547, 4629497167543615722, 3247838709755899354, 9349955712890521943, 10806931643455376261, 9797786755886426509, 8936361737713300483, 4680758664059980220, 12406007169225434383, 3533709596282820669, 2455674357456026},
			{12488981058804589629, 12457922297183532626, 451262455699675331, 2589192099576276852, 15580980055328892440, 18403052471820082137, 12189972068214284937, 13256129327810144998, 3772655204155038395, 16041014234959313151, 72181953470569885, 4281388365540118},
			{14355041116718584904, 6518908914215076528, 103136633313959269, 11248008213672993918, 6970489461061342522, 11790861983726804408, 13550185278440680395, 13812178300427159463, 16743587002127429862, 4048335839682807858, 14698087464497806541, 2419921007991860},
			{5234990458074203996, 13017287354230506146, 14528142910692005959, 3772795329349177430, 5743699393195056243, 7634427128601001554, 13604641011547791987, 135487290822320499, 3790885956722941871, 11074651945233114222, 18082432112342758574, 574821069767001},
			{52578721445132518, 4807297948059579044, 13300282182261617411, 1966048695126804313, 9698133404944599207, 8820248505019030680, 3493492344229743544, 2207289649260572501, 10632810226164047214, 4971453237616087970, 913429712705966969, 1297201987563834},
			{17825584147787228187, 9643369156770337109, 4780283242127859977, 17498229400171506032, 16108828831928294847, 2084860329621159147, 18414157538260388952, 800287692256631753, 5695704543389678503, 4041482112338895649, 11390130574267716400, 2960465040246311},
			{6300502775430373558, 8369258902650415287, 3125753630970140323, 18204543611164718383, 526661945080208710, 15161036982699244642, 17578969572176151165, 12378105092679056730, 18183133059714169258, 12446959345931710099, 15917824696898340511, 2833318602758342},
			{13591805356567102552, 7114002070737981430, 15792963314576163910, 2439908364129503908, 287177262616586422, 15991174737764895668, 6785081176756496415, 4252801190810180185, 15974406619965461794, 5898178132521952779, 16735071778182207400, 3747243900171232},
			{15813641761354694888, 5775215055800921627, 6840760130944862528, 6877362349591737745, 15249413449075939135, 4616164489636862746, 10144566619146579982, 13413064070702175791, 1351336168301780689, 13223512313899745023, 2505242198188713520, 4008932646590333},
			{12443271823023628293, 10023560508435423279, 223996864810707239, 9381771480330342249, 6071616917663934748, 8368415673838482178, 9943888360412422135, 16466687579138830466, 13092709466108236949, 9672861101133407978, 10061835049751520054, 1548449159771126},
			{682372491237998985, 11642050129783406054, 11148560572438111964, 14522736662327934823, 11397999623021472703, 1908450234014681122, 7015571940646005434, 12379468932656542568, 16774893396095877790, 6713564321785078684, 16841673995724381835, 2735911680003515},
			{13599451869524933025, 12884519722266096890, 17760075748912444940, 7988627628311688106, 14881807801342021418, 13083553921104407380, 14041750000390895271, 2472546647851334075, 3041343147406232878, 12920007657510457765, 995078962479205134, 2579040578744874},
		},
		x)

	dst.Mul(&_dst, y)
}

func g2IsogenyYDenominator(dst *fp.Element, x *fp.Element) {
	g2EvalPolynomial(dst,
		true,
		[]fp.Element{
			{1334792841914619146, 679589714283882072, 17083068605088219648, 4813351246215290848, 18172042438030301686, 14277203889184557608, 15837808323580910883, 10306580648728757893, 11003641200468979478, 940386280777395627, 3499226008293819953, 2670356276894702},
			{9755707125415732584, 11547150959279383035, 10562244305064472457, 10066300461986435643, 7148237540886167626, 14216704987748624229, 16406871087277952232, 12806845016466455100, 2351888649504415376, 4581239853380640174, 16076162830476753346, 568274864174746},
			{893765329958948899, 6025675675869057371, 17671761618169526031, 290857361517030098, 10896841519848110098, 17285132374146627243, 6578246538169581645, 14698668347892092920, 11847506233132912482, 13196346148126584369, 8267604772654803804, 1874938974390476},
			{6735883464220501378, 12278479419325863497, 14385176350002363003, 327505777929575292, 12879051046191573898, 5244035810968697501, 3180567461135135676, 17276308001343046319, 3006667409138758213, 15967849879716326079, 16187453789268986652, 605590141583629},
			{15477429905846444566, 874980550924067157, 812459596667087749, 15729930408665072903, 14188396940358881836, 8936483967067860411, 11560783477954519230, 3137560215625292057, 9575790530359643406, 17927687113810822901, 2632829296588147564, 435712685367393},
			{17106890255742260517, 15652864607222471998, 10170811537593627639, 1453924004107446790, 11329170583151456244, 5141789719850306806, 16125025042683025570, 15076092565634391873, 8561206140914607069, 4463443131371537634, 11670432227453805701, 1689349098401896},
			{9129230268637416312, 16131424300862894054, 11599455708249086086, 13712061106571700271, 3222547574018203825, 5028201422192587329, 12665893808043057006, 7910268944325741984, 13694452156601909132, 7594944072090734412, 6879517203248637902, 1138400281829526},
			{535993526973048097, 3775829409162107138, 13717328028075297397, 931342578972386679, 9677314225136962692, 1800755985773403875, 10333191064113944512, 493249530510038265, 3476098351926766495, 1181720279204825908, 12777038111429138841, 3384692890585801},
			{6772381476956843695, 1547184579308225297, 9184478537476982368, 18151871132032417756, 11438493594152051480, 13120718087287853722, 906650696512912608, 13671423743799524436, 5153011690436366245, 3698430439766668188, 15092448703784862617, 3030866256020577},
			{8895000324450291201, 15634640715766263880, 7490240763823358372, 3817243160942000768, 6890890435018787185, 9346708229944942004, 18064456205166879593, 12575272864892718868, 15621790773858233563, 14813107721424927109, 8430498945985935887, 867090342578861},
			{12739036589867187355, 9795881745418909683, 5361817503043430525, 7798704069278097907, 18096088569918482706, 2937445070828306418, 9991593439280666445, 15110215917620419457, 921826231237854254, 5913792261041157634, 18090796787901390905, 2793222464357693},
			{3438032613460386180, 6204279332812840808, 14376358175686985834, 18381747228820135059, 14447820798834719386, 16106544860316335529, 14025545057246822816, 2539847764773437436, 4552716052497346485, 5833901919148093744, 13521223476405376994, 2797828028642414},
			{4675764524655489549, 7417596943369946677, 9595278207029085025, 5342154649589543352, 18404482716153763072, 2184348815728369995, 14066858184312939843, 16474267930709347218, 2362065103222405087, 3078344645281881403, 13469353821803478908, 304023594704714},
			{13336540334000055874, 12363933549556690779, 9868198113486195987, 17200559107951779810, 10375132447814441523, 16052049732850218957, 8448578710116104683, 271552525374008518, 8581766105752285893, 6711806820578914344, 14469754450868659033, 1966815926308711},
			{5101880310190341644, 15159071985462456973, 7149774646728826197, 12576407457185946487, 17860866655468525238, 4507443374425691963, 17618967833350108716, 7421484418298085264, 6158022389899675973, 108715159554595854, 9388611835741856695, 3474499278497195},
			{15767622754071481373, 6666535779016477106, 4339596343074584784, 4593977422804012446, 8260949087674697005, 2604766566145864187, 10520452650074955497, 9902394558193624589, 12775751068121307426, 10756048839832512283, 7768310516012640156, 1089103621897159},
			{5654788162836550163, 13749357347467323269, 13974653046468144914, 15362323059394904107, 10078076543768098232, 11650909370168368583, 12022622616989434446, 16634402826677993967, 6252490137121470998, 10291036775792765672, 13911313412251103552, 1186595065825944},
			{9121325598321594489, 4133870066251362254, 8684200896151736292, 11861689787908128953, 15931378051893636612, 12451213672520635731, 12914385543399447420, 8572569051941496555, 2726874492944369081, 10801779996113951741, 5864104807527920738, 1982990352964359},
			{11090133365261209413, 15814955091366535667, 18339652705270534313, 12470411874283941774, 7241404949791996603, 8825390212707063972, 9965333245992945637, 8672796411653931019, 1540315761125104703, 12195021924674475366, 18070568062581496335, 3519904720183927},
			{1301757369757148520, 2291752182431491104, 18031597538102297383, 1002578022922400175, 17456614351673438697, 13042781829346793120, 15050871415347786395, 14032535732147643747, 1819269103672142070, 5052963464011981037, 3935730851297570856, 3098099795194852},
			{1598327534129828645, 1100792835020160421, 14836203351660379192, 14908939120570101326, 16000114216335714301, 3466300856246268575, 3868982654067330071, 9459325155447144247, 16298126741239179846, 4600431113223808736, 16925178278642030656, 2162687411385134},
			{17788148994062165922, 7922294369928924429, 14353729795536090711, 11780816328988316738, 4791479909649638698, 4114510181199450025, 1665112860230268902, 9464476004182348196, 10513922006780843432, 7647020907782691752, 907039416571889407, 2600070448243251},
			{18279740539772034858, 13488091108525447666, 7632730442377627346, 14964872363027481484, 10761640429414937302, 12381919936976899683, 11478319527418241156, 9634723465607073757, 6359127698243469512, 2747481342146307926, 7704083516220795600, 1774160264476624},
			{13635372110860167721, 1296080719790173115, 11867537485597319624, 10870737671048455066, 11726493726432719410, 12745805811649554768, 12333923534385142391, 17801909532926800273, 11608940035451353880, 2975460672602133984, 5200998817217096522, 664536368560130},
		},
		x)
}

func g2Isogeny(p *G2Affine) {

	den := make([]fp.Element, 2)

	g2IsogenyYDenominator(&den[1], &p.X)
	g2IsogenyXDenominator(&den[0], &p.X)

	g2IsogenyYNumerator(&p.Y, &p.X, &p.Y)
	g2IsogenyXNumerator(&p.X, &p.X)

	den = fp.BatchInvert(den)

	p.X.Mul(&p.X, &den[0])
	p.Y.Mul(&p.Y, &den[1])
}

// g2SqrtRatio computes the square root of u/v and returns 0 iff u/v was indeed a quadratic residue
// if not, we get sqrt(Z * u / v). Recall that Z is non-residue
// If v = 0, u/v is meaningless and the output is unspecified, without raising an error.
// The main idea is that since the computation of the square root involves taking large powers of u/v, the inversion of v can be avoided
func g2SqrtRatio(z *fp.Element, u *fp.Element, v *fp.Element) uint64 {

	// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-sqrt_ratio-for-any-field

	tv1 := fp.Element{17302715199413996045, 15077845457253267709, 8842885729139027579, 12189878420705505575, 12380986790262239346, 585111498723936856, 4947215576903759546, 1186632482028566920, 14543050817583235372, 5644943604719368358, 9440830989708189862, 1039766423535362} //tv1 = c6

	var tv2, tv3, tv4, tv5 fp.Element
	var exp big.Int
	// c4 = 4835703278458516698824703 = 2⁸² - 1
	// q is odd so c1 is at least 1.
	exp.SetBytes([]byte{3, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255})

	tv2.Exp(*v, &exp) // 2. tv2 = vᶜ⁴
	tv3.Square(&tv2)  // 3. tv3 = tv2²
	tv3.Mul(&tv3, v)  // 4. tv3 = tv3 * v
	tv5.Mul(u, &tv3)  // 5. tv5 = u * tv3

	// c3 = 37877157660731232732990269576663233239936484746509109593426423261538632780449313352717366389444912082695314931794809746268936574949192324351273838279701014606648452884726586254167471840902479876056412368
	exp.SetBytes([]byte{1, 238, 213, 183, 107, 119, 49, 92, 85, 130, 79, 195, 198, 173, 25, 235, 146, 241, 154, 95, 88, 89, 209, 63, 126, 70, 68, 40, 170, 44, 116, 217, 152, 213, 206, 120, 133, 72, 219, 61, 96, 89, 2, 93, 64, 159, 85, 65, 79, 214, 57, 103, 160, 220, 200, 220, 82, 89, 162, 189, 182, 200, 212, 168, 96, 85, 71, 132, 177, 188, 251, 218, 22, 208, 189, 13, 10, 73, 216, 6, 120, 252, 199, 240, 208})

	tv5.Exp(tv5, &exp)  // 6. tv5 = tv5ᶜ³
	tv5.Mul(&tv5, &tv2) // 7. tv5 = tv5 * tv2
	tv2.Mul(&tv5, v)    // 8. tv2 = tv5 * v
	tv3.Mul(&tv5, u)    // 9. tv3 = tv5 * u
	tv4.Mul(&tv3, &tv2) // 10. tv4 = tv3 * tv2

	// c5 = 2417851639229258349412352
	exp.SetBytes([]byte{2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0})
	tv5.Exp(tv4, &exp)      // 11. tv5 = tv4ᶜ⁵
	isQNr := g2NotOne(&tv5) // 12. isQR = tv5 == 1
	c7 := fp.Element{13990906742184113945, 15879050380504523621, 13768460034940508157, 12337541071329853620, 6296858130192020747, 9289986178217863086, 18403114759403589657, 4546259071787184045, 5504643400205978814, 13830311104669138548, 96107744534255859, 1024735223965534}
	tv2.Mul(&tv3, &c7)                 // 13. tv2 = tv3 * c7
	tv5.Mul(&tv4, &tv1)                // 14. tv5 = tv4 * tv1
	tv3.Select(int(isQNr), &tv3, &tv2) // 15. tv3 = CMOV(tv2, tv3, isQR)
	tv4.Select(int(isQNr), &tv4, &tv5) // 16. tv4 = CMOV(tv5, tv4, isQR)
	exp.Lsh(big.NewInt(1), 82-2)       // 18, 19: tv5 = 2ⁱ⁻² for i = c1

	for i := 82; i >= 2; i-- { // 17. for i in (c1, c1 - 1, ..., 2):

		tv5.Exp(tv4, &exp)               // 20.    tv5 = tv4ᵗᵛ⁵
		nE1 := g2NotOne(&tv5)            // 21.    e1 = tv5 == 1
		tv2.Mul(&tv3, &tv1)              // 22.    tv2 = tv3 * tv1
		tv1.Mul(&tv1, &tv1)              // 23.    tv1 = tv1 * tv1    Why not write square?
		tv5.Mul(&tv4, &tv1)              // 24.    tv5 = tv4 * tv1
		tv3.Select(int(nE1), &tv3, &tv2) // 25.    tv3 = CMOV(tv2, tv3, e1)
		tv4.Select(int(nE1), &tv4, &tv5) // 26.    tv4 = CMOV(tv5, tv4, e1)

		if i > 2 {
			exp.Rsh(&exp, 1) // 18, 19. tv5 = 2ⁱ⁻²
		}
	}

	*z = tv3
	return isQNr
}

func g2NotOne(x *fp.Element) uint64 {

	var one fp.Element
	return one.SetOne().NotEqual(x)

}

// g2MulByZ multiplies x by [11] and stores the result in z
func g2MulByZ(z *fp.Element, x *fp.Element) {

	res := *x

	res.Double(&res)
	res.Double(&res)
	res.Add(&res, x)
	res.Double(&res)
	res.Add(&res, x)

	*z = res
}

// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-simplified-swu-method
// MapToCurve2 implements the SSWU map
// No cofactor clearing or isogeny
func MapToCurve2(u *fp.Element) G2Affine {

	var sswuIsoCurveCoeffA = fp.Element{11188695195863236139, 18339800635248689929, 13644954250665578253, 16122525194076552550, 1985822167495960177, 11021218035968661748, 12951199075167016614, 18080500199774882647, 3065668365127963650, 1810223365641727596, 18249180996905802984, 4351293214471385}
	var sswuIsoCurveCoeffB = fp.Element{3597427888115195847, 8485485194496420669, 9451115945982544412, 10217463679676360079, 3023875305953960937, 5866766270380139867, 15059909646037855295, 1065687373540957157, 12978541562777068958, 18112033168403904062, 11632286302244735111, 1469792042332206}

	var tv1 fp.Element
	tv1.Square(u) // 1.  tv1 = u²

	//mul tv1 by Z
	g2MulByZ(&tv1, &tv1) // 2.  tv1 = Z * tv1

	var tv2 fp.Element
	tv2.Square(&tv1)    // 3.  tv2 = tv1²
	tv2.Add(&tv2, &tv1) // 4.  tv2 = tv2 + tv1

	var tv3 fp.Element
	var tv4 fp.Element
	tv4.SetOne()
	tv3.Add(&tv2, &tv4)                // 5.  tv3 = tv2 + 1
	tv3.Mul(&tv3, &sswuIsoCurveCoeffB) // 6.  tv3 = B * tv3

	tv2NZero := g2NotZero(&tv2)

	// tv4 = Z
	tv4 = fp.Element{18446744073709504998, 11529623972028612607, 739483395258014634, 5527028560780200701, 11477868704616895891, 15905434021829949368, 2844651761892435780, 17567410508478669002, 4162242322955979641, 15743938111024983262, 11916654042695069468, 4062866236140222}

	tv2.Neg(&tv2)
	tv4.Select(int(tv2NZero), &tv4, &tv2) // 7.  tv4 = CMOV(Z, -tv2, tv2 != 0)
	tv4.Mul(&tv4, &sswuIsoCurveCoeffA)    // 8.  tv4 = A * tv4

	tv2.Square(&tv3) // 9.  tv2 = tv3²

	var tv6 fp.Element
	tv6.Square(&tv4) // 10. tv6 = tv4²

	var tv5 fp.Element
	tv5.Mul(&tv6, &sswuIsoCurveCoeffA) // 11. tv5 = A * tv6

	tv2.Add(&tv2, &tv5) // 12. tv2 = tv2 + tv5
	tv2.Mul(&tv2, &tv3) // 13. tv2 = tv2 * tv3
	tv6.Mul(&tv6, &tv4) // 14. tv6 = tv6 * tv4

	tv5.Mul(&tv6, &sswuIsoCurveCoeffB) // 15. tv5 = B * tv6
	tv2.Add(&tv2, &tv5)                // 16. tv2 = tv2 + tv5

	var x fp.Element
	x.Mul(&tv1, &tv3) // 17.   x = tv1 * tv3

	var y1 fp.Element
	gx1NSquare := g2SqrtRatio(&y1, &tv2, &tv6) // 18. (is_gx1_square, y1) = sqrt_ratio(tv2, tv6)

	var y fp.Element
	y.Mul(&tv1, u) // 19.   y = tv1 * u

	y.Mul(&y, &y1) // 20.   y = y * y1

	x.Select(int(gx1NSquare), &tv3, &x) // 21.   x = CMOV(x, tv3, is_gx1_square)
	y.Select(int(gx1NSquare), &y1, &y)  // 22.   y = CMOV(y, y1, is_gx1_square)

	y1.Neg(&y)
	y.Select(int(g2Sgn0(u)^g2Sgn0(&y)), &y, &y1)

	// 23.  e1 = sgn0(u) == sgn0(y)
	// 24.   y = CMOV(-y, y, e1)

	x.Div(&x, &tv4) // 25.   x = x / tv4

	return G2Affine{x, y}
}

func g2EvalPolynomial(z *fp.Element, monic bool, coefficients []fp.Element, x *fp.Element) {
	dst := coefficients[len(coefficients)-1]

	if monic {
		dst.Add(&dst, x)
	}

	for i := len(coefficients) - 2; i >= 0; i-- {
		dst.Mul(&dst, x)
		dst.Add(&dst, &coefficients[i])
	}

	z.Set(&dst)
}

// g2Sgn0 is an algebraic substitute for the notion of sign in ordered fields
// Namely, every non-zero quadratic residue in a finite field of characteristic =/= 2 has exactly two square roots, one of each sign
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-the-sgn0-function
// The sign of an element is not obviously related to that of its Montgomery form
func g2Sgn0(z *fp.Element) uint64 {

	nonMont := z.Bits()

	// m == 1
	return nonMont[0] % 2

}

// MapToG2 invokes the SSWU map, and guarantees that the result is in g2
func MapToG2(u fp.Element) G2Affine {
	res := MapToCurve2(&u)
	//this is in an isogenous curve
	g2Isogeny(&res)
	res.ClearCofactor(&res)
	return res
}

// EncodeToG2 hashes a message to a point on the G2 curve using the SSWU map.
// It is faster than HashToG2, but the result is not uniformly distributed. Unsuitable as a random oracle.
// dst stands for "domain separation tag", a string unique to the construction using the hash function
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#roadmap
func EncodeToG2(msg, dst []byte) (G2Affine, error) {

	var res G2Affine
	u, err := fp.Hash(msg, dst, 1)
	if err != nil {
		return res, err
	}

	res = MapToCurve2(&u[0])

	//this is in an isogenous curve
	g2Isogeny(&res)
	res.ClearCofactor(&res)
	return res, nil
}

// HashToG2 hashes a message to a point on the G2 curve using the SSWU map.
// Slower than EncodeToG2, but usable as a random oracle.
// dst stands for "domain separation tag", a string unique to the construction using the hash function
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#roadmap
func HashToG2(msg, dst []byte) (G2Affine, error) {
	u, err := fp.Hash(msg, dst, 2*1)
	if err != nil {
		return G2Affine{}, err
	}

	Q0 := MapToCurve2(&u[0])
	Q1 := MapToCurve2(&u[1])

	//TODO (perf): Add in E' first, then apply isogeny
	g2Isogeny(&Q0)
	g2Isogeny(&Q1)

	var _Q0, _Q1 G2Jac
	_Q0.FromAffine(&Q0)
	_Q1.FromAffine(&Q1).AddAssign(&_Q0)

	_Q1.ClearCofactor(&_Q1)

	Q1.FromJacobian(&_Q1)
	return Q1, nil
}

func g2NotZero(x *fp.Element) uint64 {

	return x[0] | x[1] | x[2] | x[3] | x[4] | x[5] | x[6] | x[7] | x[8] | x[9] | x[10] | x[11]

}
