#!/bin/bash

#
# Exit codes
#

EXIT_SUCCESS=0

# Exit with this when a test fails.
EXIT_FAILURE=1

# We use a few system users in the tests. If these users aren't
# present, we exit with a different (non-EXIT_FAILURE).
EXIT_MISSING_USERS=2

# Define the users and groups that we'll use in the tests below. We
# store the names as variables to avoid repeating them everywhere.
# Since GROUPS is already part of everyone's environment, we need
# a different name.
#
# WARNING: These must be in alphabetical order; otherwise the getfacl
# output will not match.
#
USERS=( bin daemon )
TESTGROUPS=( bin daemon )

# Check to see if the above users exist. If not, bail.
for idx in $( seq 0 $((${#USERS[@]} - 1)) ); do
    id "${USERS[idx]}" >/dev/null 2>&1

    if [ $? -ne $EXIT_SUCCESS ]; then
	echo "Error: missing test user ${USERS[idx]}." 1>&2
	exit $EXIT_MISSING_USERS
    fi
done

# The program name.
BIN=$(realpath src/apply-default-acl)

# The directory where we'll do all the ACL manipulation.
TESTDIR=test

# Will auto-increment.
TESTNUM=0

acl_reset() {
    # Remove any ACLs on our test directory and remove its contents.
    setfacl --remove-all --recursive "${TESTDIR}"
    chmod 755 "${TESTDIR}"
    rm -rf "${TESTDIR}"/*
}

compare() {
    if [[ "${ACTUAL}" == "${EXPECTED}" ]]; then
	echo "Success (#${TESTNUM})"
	acl_reset
    else
	echo "Failure (#${TESTNUM})"
	echo 'Expected result:'
	echo '================'
	echo "${EXPECTED}"
	echo '================'
	echo 'Actual result:'
	echo '================'
	echo "${ACTUAL}"
	echo '================'
	exit $EXIT_FAILURE
    fi
}

# Start by removing and recreating the 'acl' directory.
rm -rf "${TESTDIR}"
mkdir "${TESTDIR}"


# When using a minimal ACL, the default user, group, and other
# permissions should all be propagated to the mode bits.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 777 "${TARGET}"
setfacl -d -m user::r--  "${TESTDIR}"
setfacl -d -m group::r-- "${TESTDIR}"
setfacl -d -m other::r-- "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::r--
group::r--
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare

# Do the same thing as the last test, except with an extended ACL.
((TESTNUM++))
setfacl -d -m user::r--     "${TESTDIR}"
setfacl -d -m group::r--    "${TESTDIR}"
setfacl -d -m other::r--    "${TESTDIR}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
touch "${TARGET}"
chmod 777 "${TARGET}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::r--
user:${USERS[0]}:rwx
group::r--
mask::rwx
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# A file shared by a group, should still be group-writable
# afterwards.
((TESTNUM++))
touch "${TARGET}"
chmod 644 "${TARGET}"
setfacl -d -m group:${USERS[0]}:rwx "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rw-
group::r--
group:${USERS[0]}:rwx	#effective:rw-
mask::rw-
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Same test as before except with a directory.
((TESTNUM++))
setfacl -d -m group:${USERS[0]}:rwx "${TESTDIR}"
mkdir "${TARGET}"
chmod 755 "${TARGET}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
group::r-x
group:${USERS[0]}:rwx
mask::rwx
other::r-x
default:user::rwx
default:group::r-x
default:group:${USERS[0]}:rwx
default:mask::rwx
default:other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# With no default, things are left alone.
((TESTNUM++))
touch "${TARGET}"
chmod 744 "${TARGET}"
$BIN "${TARGET}"


EXPECTED=$(cat <<EOF
user::rwx
group::r--
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare



# Since the default ACL will grant r-x to group/other, they will wind
# up with it.
((TESTNUM++))
touch "${TARGET}"
chmod 744 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
$BIN "${TARGET}"


EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Some named entries can be granted execute permissions as the result
# of reapplication.
((TESTNUM++))
touch "${TARGET}"
chmod 744 "${TARGET}"
setfacl -m user:${USERS[1]}:rw "${TARGET}"
# If we don't add 'x' to the mask here, nobody can execute the file.
# setfacl will update the mask for us under most circumstances, but
# note that we didn't create an entry with an 'x' bit using setfacl --
# therefore, setfacl won't unmask 'x' for us.
setfacl -m mask::rwx "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
setfacl -d -m user:${USERS[1]}:rwx "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
user:${USERS[1]}:rwx
group::r-x
mask::rwx
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# We should not retain any entries that aren't in the default.
((TESTNUM++))
touch "${TARGET}"
chmod 644 "${TARGET}"
setfacl -m user:${USERS[1]}:rw "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rw-
user:${USERS[0]}:rwx	#effective:rw-
group::r--
mask::rw-
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# A slightly modified version of the first test, to make sure it works.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 777 "${TARGET}"
setfacl -d -m user::r--  "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::r--
group::r-x
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# If the default ACL mask denies execute, we should respect that
# regardless of the existing execute permissions.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 777 "${TARGET}"
setfacl -m user:${USERS[0]}:rwx "${TESTDIR}"
setfacl -d -m user:${USERS[0]}:rwx  "${TESTDIR}"
setfacl -d -m mask::rw-  "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx	#effective:rw-
group::r-x	#effective:r--
mask::rw-
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare



# The --recursive mode should work normally if the argument is a
# normal file. See the first test.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
setfacl -d -m user::r--  "${TESTDIR}"
setfacl -d -m group::r-- "${TESTDIR}"
setfacl -d -m other::r-- "${TESTDIR}"
touch "${TARGET}"
chmod 777 "${TARGET}"
$BIN --recursive "${TARGET}"

EXPECTED=$(cat <<EOF
user::r--
group::r--
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# The --recursive mode should work recursively.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
mkdir -p "${TARGET}"
touch "${TARGET}"/baz
mkdir -p "${TARGET}"/bar
touch "${TARGET}"/bar/quux
setfacl -d -m user::rwx  "${TESTDIR}"
setfacl -d -m group::r-- "${TESTDIR}"
setfacl -d -m other::r-- "${TESTDIR}"
chmod -R 777 "${TARGET}"
$BIN --recursive "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
group::r--
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}"/bar/quux)
compare


# The --recursive mode should work recursively. This time
# check a directory, and pass the short command-line flag.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
mkdir -p "${TARGET}"
touch "${TARGET}"/baz
mkdir -p "${TARGET}"/bar
touch "${TARGET}"/bar/quux
setfacl -d -m user::rwx  "${TESTDIR}"
setfacl -d -m group::r-- "${TESTDIR}"
setfacl -d -m other::r-- "${TESTDIR}"
chmod -R 777 "${TARGET}"
$BIN -r "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
group::r--
other::r--
default:user::rwx
default:group::r--
default:other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}"/bar)
compare


# Test double application on a directory.
#
((TESTNUM++))
TARGET="${TESTDIR}"/baz
mkdir "${TARGET}"
chmod 644 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"

$BIN "${TARGET}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x
default:user::rwx
default:user:${USERS[0]}:rwx
default:group::r-x
default:mask::rwx
default:other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Same as previous test, with 755 initial perms.
#
((TESTNUM++))
TARGET="${TESTDIR}"/baz
mkdir "${TARGET}"
chmod 755 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"

$BIN "${TARGET}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x
default:user::rwx
default:user:${USERS[0]}:rwx
default:group::r-x
default:mask::rwx
default:other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Same as previous two tests, only with a file.
#
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 644 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"

$BIN "${TARGET}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rw-
user:${USERS[0]}:rwx	#effective:rw-
group::r--
mask::rw-
other::r--
EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# User-executable files should not wind up exec-masked.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 700 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx  "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Group-executable files should not wind up exec-masked.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 670 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx  "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Other-executable files should not wind up exec-masked.
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 607 "${TARGET}"
setfacl -d -m user:${USERS[0]}:rwx  "${TESTDIR}"
$BIN "${TARGET}"

EXPECTED=$(cat <<EOF
user::rwx
user:${USERS[0]}:rwx
group::r-x
mask::rwx
other::r-x

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare




# Make sure a mask with an execute bit doesn't count as being
# executable.
#
((TESTNUM++))
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
chmod 644 "${TARGET}"
setfacl -m user::rw "${TARGET}"
setfacl -m group::rw "${TARGET}"
# Even though the mask has an 'x' bit, nobody can execute it.
setfacl -m mask::rwx "${TARGET}"
setfacl -d -m user::rwx "${TESTDIR}"
setfacl -d -m group::rwx "${TESTDIR}"
$BIN "${TARGET}"


EXPECTED=$(cat <<EOF
user::rw-
group::rw-
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# Same as the second test, except we pass multiple files on the
# command line and check the result of the first one.
((TESTNUM++))
setfacl -d -m user::r--     "${TESTDIR}"
setfacl -d -m group::r--    "${TESTDIR}"
setfacl -d -m other::r--    "${TESTDIR}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
DUMMY="${TESTDIR}/dummy"
touch "${DUMMY}"
chmod 777 "${DUMMY}"
touch "${TARGET}"
chmod 777 "${TARGET}"
$BIN "${TARGET}" "${DUMMY}"

EXPECTED=$(cat <<EOF
user::r--
user:${USERS[0]}:rwx
group::r--
mask::rwx
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare



# Same as the previous test with the argument order switched.
((TESTNUM++))
setfacl -d -m user::r--     "${TESTDIR}"
setfacl -d -m group::r--    "${TESTDIR}"
setfacl -d -m other::r--    "${TESTDIR}"
setfacl -d -m user:${USERS[0]}:rwx "${TESTDIR}"
DUMMY="${TESTDIR}/dummy"
touch "${DUMMY}"
chmod 777 "${DUMMY}"
touch "${TARGET}"
chmod 777 "${TARGET}"
$BIN "${DUMMY}" "${TARGET}"

EXPECTED=$(cat <<EOF
user::r--
user:${USERS[0]}:rwx
group::r--
mask::rwx
other::r--

EOF
)

ACTUAL=$(getfacl --omit-header "${TARGET}")
compare


# If we call apply-default-acl on a single file that does not exist,
# we get the expected error.
((TESTNUM++))
ACTUAL=$( "${BIN}" test/nonexistent 2>&1 )
EXPECTED="test/nonexistent: No such file or directory"
compare


# Same as the previous test, but with --recursive.
((TESTNUM++))
ACTUAL=$( "${BIN}" --recursive test/nonexistent 2>&1 )
EXPECTED="test/nonexistent: No such file or directory"
compare


# If we call apply-default-acl on more than one file, it should report any
# that don't exist (but proceed to operate on the others).
((TESTNUM++))
DUMMY1="${TESTDIR}/dummy1"
DUMMY2="${TESTDIR}/dummy2"
touch "${DUMMY1}" "${DUMMY2}"
ACTUAL=$( "${BIN}" "${DUMMY1}" test/nonexistent "${DUMMY2}" 2>&1 )
EXPECTED="test/nonexistent: No such file or directory"
compare


# Ensure that symlinks are not followed.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
LINK2TARGET="${TESTDIR}/foo-sym"
touch "${TARGET}"
ln -s "${TARGET#${TESTDIR}/}" "${LINK2TARGET}"
setfacl --default --modify user:${USERS[0]}:rwx "${TESTDIR}"
"${BIN}" "${LINK2TARGET}"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
EXPECTED=$(cat <<EOF
user::rw-
group::r--
other::r--

EOF
)
compare


# Ensure that symlinks are not followed in subdirectories (recursively).
((TESTNUM++))
TARGET="${TESTDIR}/bar"
touch "${TARGET}"
mkdir "${TESTDIR}/foo"
LINK2TARGET="${TESTDIR}/foo/bar-sym"
ln -s "../bar" "${LINK2TARGET}"
setfacl --default --modify user:${USERS[0]}:rwx "${TESTDIR}/foo"
EXPECTED=$(getfacl --omit-header "${TARGET}")
"${BIN}" --recursive "${TESTDIR}/foo"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare


# Ensure that hard links are ignored.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
LINK2TARGET="${TESTDIR}/bar"
touch "${TARGET}"
ln "${TARGET}" "${LINK2TARGET}"
setfacl --default --modify user:${USERS[0]}:rwx "${TESTDIR}"
"${BIN}" "${LINK2TARGET}"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
EXPECTED=$(cat <<EOF
user::rw-
group::r--
other::r--

EOF
)
compare


# We should be able to run the tool with a relative path from within a
# directory that contains a symlink, so long as the relative path
# doesn't contain one.
((TESTNUM++))
TARGET="${TESTDIR}/foo/bar"
LINK2TARGET="${TESTDIR}/baz"
mkdir -p $(dirname "${TARGET}")
touch "${TARGET}"
ln -s foo "${TESTDIR}/baz"
setfacl --default --modify user:${USERS[0]}:rw $(dirname "${TARGET}")
pushd "${TESTDIR}/baz" > /dev/null
"${BIN}" bar
popd > /dev/null
ACTUAL=$( getfacl --omit-header "${TARGET}" )
EXPECTED=$(cat <<EOF
user::rw-
user:${USERS[0]}:rw-
group::r--
mask::rw-
other::r--

EOF
)
compare


# Ensure that symlinks in non-terminal path components are not followed.
((TESTNUM++))
TARGET="${TESTDIR}/foo/bar/baz"
LINK2FOO="${TESTDIR}/quux"
mkdir -p $(dirname "${TARGET}")
touch "${TARGET}"
ln -s foo "${LINK2FOO}"
setfacl --default --modify user:${USERS[0]}:rw $(dirname "${TARGET}")
EXPECTED=$(getfacl --omit-header "${TARGET}")
"${BIN}" "${LINK2FOO}/bar/baz"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare


# Test that our exit code succeeds on a single, normal path.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
touch "${TARGET}"
setfacl --default --modify user:${USERS[0]}:rw "${TESTDIR}"
"${BIN}" "${TARGET}"
ACTUAL="$?"
EXPECTED="0"
compare


# Test that our exit code fails on a symlink.
((TESTNUM++))
TARGET="${TESTDIR}/bar"
touch "${TESTDIR}/foo"
ln -s foo "${TARGET}"
setfacl --default --modify user:${USERS[0]}:rw "${TESTDIR}"
"${BIN}" "${TARGET}"
ACTUAL="$?"
EXPECTED="1"
compare


# The previous test should fail, even if we use --recursive.
((TESTNUM++))
TARGET="${TESTDIR}/bar"
touch "${TESTDIR}/foo"
ln -s foo "${TARGET}"
setfacl --default --modify user:${USERS[0]}:rw "${TESTDIR}"
"${BIN}" --recursive "${TARGET}"
ACTUAL="$?"
EXPECTED="1"
compare


# Test the return value for nonexistent paths.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
"${BIN}" "${TARGET}" &>/dev/null
ACTUAL="$?"
EXPECTED="1"
compare


# Test that one "failure" exit code overrides two "successes"
# We need a default ACL on ${TESTDIR} because otherwise we do
# nothing, successfully, on the symlink path.
((TESTNUM++))
mkdir "${TESTDIR}/foo"
ln -s foo "${TESTDIR}/bar"
mkdir "${TESTDIR}/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TESTDIR}"
"${BIN}" "${TESTDIR}/foo" "${TESTDIR}/bar" "${TESTDIR}/baz"
ACTUAL="$?"
EXPECTED="1"
compare


# The failure should prevail when using --recursive, too.
((TESTNUM++))
mkdir "${TESTDIR}/foo"
ln -s foo "${TESTDIR}/bar"
mkdir "${TESTDIR}/baz"
"${BIN}" --recursive "${TESTDIR}"
ACTUAL="$?"
EXPECTED="1"
compare


# We should get "Not a directory" if we stick a trailing slash on the
# end of the path to a file.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
touch "${TARGET}"
ACTUAL=$( "${BIN}" "${TARGET}/" 2>&1 )
EXPECTED="${TARGET}/: Not a directory"
compare


# We should be a no-op on files contained in directories that have no
# default ACL.
((TESTNUM++))
TARGET="${TESTDIR}/foo"
touch "${TARGET}"
setfacl --modify user:${USERS[0]}:rw "${TARGET}"
EXPECTED=$( getfacl --omit-header "${TARGET}" )
"${BIN}" "${TARGET}"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare


# We should be a no-op on directories contained in directories that
# have no default ACL (same as the previous test, but with a directory).
((TESTNUM++))
TARGET="${TESTDIR}/foo"
mkdir "${TARGET}"
setfacl --modify user:${USERS[0]}:rw "${TARGET}"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
EXPECTED=$( getfacl --omit-header "${TARGET}" )
"${BIN}" --recursive "${TARGET}"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare


# Make sure we descend into subdirectories that don't have default ACLs.
((TESTNUM++))
TARGET="${TESTDIR}/foo/bar/baz"
mkdir -p $(dirname "${TARGET}")
touch "${TARGET}"
touch "${TARGET}-direct"
setfacl --default --modify user:${USERS[0]}:rw $(dirname "${TARGET}")
"${BIN}" "${TARGET}-direct"
EXPECTED=$( getfacl --omit-header "${TARGET}-direct" )
"${BIN}" --recursive "${TESTDIR}"
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare


# Ensure that we don't get "error" results for symlinks encountered
# during a recursive traversal.
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir "${TARGET}/foo"
mkdir "${TARGET}/bar"
ln -s "../foo" "${TARGET}/bar/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
EXPECTED="1"
"${BIN}" --recursive "${TARGET}"
ACTUAL=$?
compare


# Ensure that "." works as an argument.
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir "${TARGET}/foo"
mkdir "${TARGET}/bar"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo" )
pushd "${TARGET}/bar" > /dev/null
"${BIN}" "."
ACTUAL=$( getfacl --omit-header "." )
popd > /dev/null
compare

# Ensure that "." works as an argument (recursive).
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir -p "${TARGET}/foo/baz"
mkdir -p "${TARGET}/bar/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" --recursive "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo/baz" )
pushd "${TARGET}/bar" > /dev/null
"${BIN}" --recursive "."
ACTUAL=$( getfacl --omit-header "./baz" )
popd > /dev/null
compare

# Ensure that "./" works as an argument.
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir "${TARGET}/foo"
mkdir "${TARGET}/bar"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo" )
pushd "${TARGET}/bar" > /dev/null
"${BIN}" "./"
ACTUAL=$( getfacl --omit-header "./" )
popd > /dev/null
compare

# Ensure that ".." works as an argument.
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir "${TARGET}/foo"
mkdir -p "${TARGET}/bar/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo" )
pushd "${TARGET}/bar/baz" > /dev/null
"${BIN}" ".."
ACTUAL=$( getfacl --omit-header ".." )
popd > /dev/null
compare

# Ensure that ".." works as an argument (recursive).
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir -p "${TARGET}/foo/baz"
mkdir -p "${TARGET}/bar/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" --recursive "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo/baz" )
pushd "${TARGET}/bar/baz" > /dev/null
"${BIN}" --recursive ".."
ACTUAL=$( getfacl --omit-header "." )
popd > /dev/null
compare

# Ensure that "../" works as an argument.
((TESTNUM++))
TARGET="${TESTDIR}"
mkdir "${TARGET}/foo"
mkdir -p "${TARGET}/bar/baz"
setfacl --default --modify user:${USERS[0]}:rw "${TARGET}"
"${BIN}" "${TARGET}/foo"
EXPECTED=$( getfacl --omit-header "${TARGET}/foo" )
pushd "${TARGET}/bar/baz" > /dev/null
"${BIN}" "../"
ACTUAL=$( getfacl --omit-header "../" )
popd > /dev/null
compare


# Ensure that multiple named-user and named-group entries all get
# applied individually rather than the last one taking precedence.
# This is a regression test against a bug that made it into a release
# and was reported by Michał Bartoszkiewicz.
((TESTNUM++))
TARGET="${TESTDIR}"
TARGET="${TESTDIR}"/foo
touch "${TARGET}"
setfacl -d -m user:${USERS[0]}:rw- "${TESTDIR}"
setfacl -d -m group:${TESTGROUPS[0]}:rw- "${TESTDIR}"
setfacl -d -m user:${USERS[1]}:--- "${TESTDIR}"
setfacl -d -m group:${TESTGROUPS[1]}:--- "${TESTDIR}"
"${BIN}" "${TARGET}"
EXPECTED=$(cat <<EOF
user::rw-
user:${USERS[0]}:rw-
user:${USERS[1]}:---
group::r--
group:${TESTGROUPS[0]}:rw-
group:${TESTGROUPS[1]}:---
mask::rw-
other::r--

EOF
)
ACTUAL=$( getfacl --omit-header "${TARGET}" )
compare
