import type { VariableDeclarator } from '@swc/core';
import type { ScopeInfo, I18nextToolkitConfig } from '../../types';
export declare class ScopeManager {
    private scopeStack;
    private config;
    private scope;
    private simpleConstants;
    constructor(config: Omit<I18nextToolkitConfig, 'plugins'>);
    /**
     * Reset per-file scope state.
     *
     * This clears both the scope stack and the legacy scope map. It should be
     * called at the start of processing each file so that scope info does not
     * leak between files.
     */
    reset(): void;
    /**
     * Enters a new variable scope by pushing a new scope map onto the stack.
     * Used when entering functions to isolate variable declarations.
     */
    enterScope(): void;
    /**
     * Exits the current variable scope by popping the top scope map.
     * Used when leaving functions to clean up variable tracking.
     */
    exitScope(): void;
    /**
     * Stores variable information in the current scope.
     * Used to track translation functions and their configuration.
     *
     * @param name - Variable name to store
     * @param info - Scope information about the variable
     */
    setVarInScope(name: string, info: ScopeInfo): void;
    /**
     * Retrieves variable information from the scope chain.
     * Searches from innermost to outermost scope.
     *
     * @param name - Variable name to look up
     * @returns Scope information if found, undefined otherwise
     */
    getVarFromScope(name: string): ScopeInfo | undefined;
    private getUseTranslationConfig;
    /**
     * Resolve simple identifier declared in-file to its string literal value, if known.
     */
    private resolveSimpleStringIdentifier;
    /**
     * Handles variable declarations that might define translation functions.
     *
     * Processes two patterns:
     * 1. `const { t } = useTranslation(...)` - React i18next pattern
     * 2. `const t = i18next.getFixedT(...)` - Core i18next pattern
     *
     * Extracts namespace and key prefix information for later use.
     *
     * @param node - Variable declarator node to process
     */
    handleVariableDeclarator(node: VariableDeclarator): void;
    /**
     * Handles useTranslation calls for comment scope resolution.
     * This is a separate method to store scope info in the legacy scope map
     * that the comment parser can access.
     *
     * @param node - Variable declarator with useTranslation call
     * @param callExpr - The CallExpression node representing the useTranslation invocation
     * @param hookConfig - Configuration describing argument positions for namespace and keyPrefix
     */
    private handleUseTranslationForComments;
    /**
     * Processes useTranslation hook declarations to extract scope information.
     *
     * Handles various destructuring patterns:
     * - `const [t] = useTranslation('ns')` - Array destructuring
     * - `const { t } = useTranslation('ns')` - Object destructuring
     * - `const { t: myT } = useTranslation('ns')` - Aliased destructuring
     *
     * Extracts namespace from the first argument and keyPrefix from options.
     *
     * @param node - Variable declarator with useTranslation call
     * @param callExpr - The CallExpression node representing the useTranslation invocation
     * @param hookConfig - Configuration describing argument positions for namespace and keyPrefix
     */
    private handleUseTranslationDeclarator;
    /**
     * Processes getFixedT function declarations to extract scope information.
     *
     * Handles the pattern: `const t = i18next.getFixedT(lng, ns, keyPrefix)`
     * - Ignores the first argument (language)
     * - Extracts namespace from the second argument
     * - Extracts key prefix from the third argument
     *
     * @param node - Variable declarator with getFixedT call
     * @param callExpr - The CallExpression node representing the getFixedT invocation
     */
    private handleGetFixedTDeclarator;
    /**
     * Handles cases where a getFixedT-like function is a variable (from a custom hook)
     * and is invoked to produce a bound `t` function, e.g.:
     *   const { getFixedT } = useTranslate('prefix')
     *   const t = getFixedT('en', 'ns')
     *
     * We combine the original source variable's scope (keyPrefix/defaultNs) with
     * any namespace/keyPrefix arguments provided to this call and attach the
     * resulting scope to the newly declared variable.
     */
    private handleGetFixedTFromVariableDeclarator;
}
//# sourceMappingURL=scope-manager.d.ts.map