//***********************************************************************************************************
// Concatenate_BinToStd_GammaAtCreation.C
// Root command file
// Type: root Concatenate_BinToStd_GammaAtCreation.C
//
// It is used in case of interruption
// Read 2 output files GammaAtCreation_1.dat and GammaAtCreation_2.dat that are generated by Geant4
// tomography simulation. It reads all the gamma at creation information, and rewrite the events in
// a binary file PixeEvent_std_AtCreation.DAT
//
// More information is available in UserGuide
// Created by Z.LI LP2i Bordeaux 2022
//***********************************************************************************************************

#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include <vector>
// using namespace std;

// Define a structure to read and write each event in the required binary format
struct PixeEvent
{
  uint16_t energy_10eV;
  uint16_t pixelIndex;
  uint16_t sliceIndex;
  uint8_t projectionIndex;
};
struct ParticleInfo
{
  float energy_keV;
  float mx;
  float my;
  float mz;
};
struct RunInfo
{
  // uint_16t
  uint8_t projectionIndex;  // 1 byte
  uint16_t sliceIndex;  //
  uint16_t pixelIndex;
  uint32_t nbParticle;  // 4 bytes int
};

struct Point
{
  double m_x;
  double m_y;
  double m_z;
};
bool IsDetected(Point poi1, Point poi2, double theta)
{
  double a = (poi1.m_x * poi2.m_x + poi1.m_y * poi2.m_y + poi1.m_z * poi2.m_z)
             / sqrt(poi1.m_x * poi1.m_x + poi1.m_y * poi1.m_y + poi1.m_z * poi1.m_z)
             / sqrt(poi2.m_x * poi2.m_x + poi2.m_y * poi2.m_y + poi2.m_z * poi2.m_z);
  if (a > 1.0) a = 1;
  if (a < -1.0) a = -1;
  double r = acos(a);
  if (r > theta)
    return false;
  else
    return true;
}
void Concatenate_BinToStd_GammaAtCreation()
{
  //***********************************************************************
  //**************************Detection parameters (begin)*****************
  //***********************************************************************

  const int nbProjection = 10;
  const int nbSlice = 1;
  const int nbPixel = 20;
  double totalAngleSpan = 180.;  // in degree

  double angleOfDetector =
    135.;  // angle of detector relative to the incident direction of the primary protons //
  double distanceObjectDetector = 22.;  // 22 mm
  double radiusOfDetector = 5.;  // 5 mm
  // double theta = atan(radiusOfDetector/distanceObjectDetector); //half apex angle of the right
  // circular cone in radian
  double theta = 70 * TMath::DegToRad();  // in radian

  int P_interrupt = 6;  // Projection of interruption

  //***********************************************************************
  //**************************Detection parameters (end)*******************
  //***********************************************************************

  // assuming there is one interruption
  FILE* input1 = fopen("../build/GammaAtCreation_1.dat", "rb");
  FILE* input2 = fopen("../build/GammaAtCreation_2.dat", "rb");
  FILE* out = fopen("../build/PixeEvent_std_AtCreation.DAT", "wb");

  if (input1 == NULL) {
    printf("error for opening the input GammaAtCreation_1.dat file\n");
    return;
  }
  if (input2 == NULL) {
    printf("error for opening the input GammaAtCreation_2.dat file\n");
    return;
  }

  RunInfo runInfo;
  PixeEvent pixeEvent;
  Point centerOfDetector;
  Point gammaMomentum;
  long long count1 = 0;
  long long count2 = 0;
  int runID = -1;  // index of simulations, namely runID, starting from 0

  // ************************************************************(begin)
  // **********************READ FIRST FILE***********************
  // ************************************************************
  while (fread(&runInfo, sizeof(RunInfo), 1, input1)) {
    runID++;

    //(begin)***************************************************************
    // the following codes are used only when in the simulation
    // the index of projection, slice and pixel is not
    // correctly configured
    runInfo.projectionIndex = runID / (nbSlice * nbPixel);
    int remain = runID % (nbSlice * nbPixel);
    runInfo.sliceIndex = remain / nbPixel;
    runInfo.pixelIndex = remain % nbPixel;
    //(end)******************************************************************

    if (runInfo.projectionIndex == P_interrupt) {
      runID--;
      break;
    }

    int nbParticle = runInfo.nbParticle;

    std::vector<ParticleInfo> gammaAtCreation(nbParticle);
    fread(&gammaAtCreation[0], sizeof(ParticleInfo), nbParticle, input1);

    // if(runInfo.sliceIndex!=1) continue;
    // if(runInfo.sliceIndex!=31&&runInfo.sliceIndex!=32) continue;
    // if(runInfo.sliceIndex!=31) continue;

    //***********************************************************************
    //**************************Print information (begin)********************
    //***********************************************************************

    printf("-1--runId %d, ProjectionIndex=%d, SliceIndex=%d, PixelIndex=%d, nbParticle = %d\n",
           runID, runInfo.projectionIndex, runInfo.sliceIndex, runInfo.pixelIndex, nbParticle);

    //***********************************************************************
    //**************************Print information (end)**********************
    //***********************************************************************

    // angleOfDetector+totalAngleSpan/nbProjection*runInfo.projectionIndex means the angle between
    // source direction and detector, which should be constant when source is rotating
    double ra = TMath::DegToRad()
                * (angleOfDetector + totalAngleSpan / nbProjection * runInfo.projectionIndex);
    centerOfDetector.m_x = distanceObjectDetector * cos(ra);
    centerOfDetector.m_y = distanceObjectDetector * sin(ra);
    centerOfDetector.m_z = 0;

    for (int i = 0; i < nbParticle; ++i) {
      // gamma selection: energy should be lower than 4095*10eV = 49.45 keV
      if (gammaAtCreation[i].energy_keV >= 40.95 || gammaAtCreation[i].energy_keV <= 0.9)
        continue;  // gamma selection

      gammaMomentum.m_x = gammaAtCreation[i].mx;
      gammaMomentum.m_y = gammaAtCreation[i].my;
      gammaMomentum.m_z = gammaAtCreation[i].mz;

      if (!IsDetected(centerOfDetector, gammaMomentum, theta))
        continue;
      else {
        pixeEvent.energy_10eV = floor(100 * gammaAtCreation[i].energy_keV + 0.5);
        pixeEvent.projectionIndex = runInfo.projectionIndex;
        pixeEvent.sliceIndex = runInfo.sliceIndex;
        pixeEvent.pixelIndex = runInfo.pixelIndex;
        fwrite(&pixeEvent, 7, 1, out);
        count1++;
      }
    }
  }
  printf("---------------Number of PixeEvent in the first file: %lld------------------------\n",
         count1);
  fclose(input1);

  // ************************************************************
  // **********************READ FIRST FILE (end)*****************
  // ************************************************************

  // ************************************************************
  // **********************READ SECOND FILE (begin)**************
  // ************************************************************
  while (fread(&runInfo, sizeof(RunInfo), 1, input2)) {
    runID++;

    //(begin)***************************************************************
    // the following codes are used only when in the simulation
    // the index of projection, slice and pixel is not
    // correctly configured
    runInfo.projectionIndex = runID / (nbSlice * nbPixel);
    int remain = runID % (nbSlice * nbPixel);
    runInfo.sliceIndex = remain / nbPixel;
    runInfo.pixelIndex = remain % nbPixel;
    //(end)******************************************************************

    int nbParticle = runInfo.nbParticle;

    //***********************************************************************
    //**************************Print information (begin)********************
    //***********************************************************************

    printf("-2--runId %d, ProjectionIndex=%d, SliceIndex=%d, PixelIndex=%d, nbParticle = %d\n",
           runID, runInfo.projectionIndex, runInfo.sliceIndex, runInfo.pixelIndex, nbParticle);

    //***********************************************************************
    //**************************Print information (end)**********************
    //***********************************************************************

    if (!nbParticle) continue;
    std::vector<ParticleInfo> gammaAtCreation(nbParticle);
    fread(&gammaAtCreation[0], sizeof(ParticleInfo), nbParticle, input2);

    // if(runInfo.sliceIndex!=1) continue;
    // if(runInfo.sliceIndex!=31) continue;
    // if(runInfo.sliceIndex!=31&&runInfo.sliceIndex!=32) continue;

    // angleOfDetector+totalAngleSpan/nbProjection*runInfo.projectionIndex means the angle between
    // source direction and detector, which should be constant when source is rotating
    double ra = TMath::DegToRad()
                * (angleOfDetector + totalAngleSpan / nbProjection * runInfo.projectionIndex);
    centerOfDetector.m_x = distanceObjectDetector * cos(ra);
    centerOfDetector.m_y = distanceObjectDetector * sin(ra);
    centerOfDetector.m_z = 0;

    for (int i = 0; i < nbParticle; ++i) {
      // gamma selection: energy should be lower than 4095*10eV = 49.45 keV
      if (gammaAtCreation[i].energy_keV >= 40.95 || gammaAtCreation[i].energy_keV <= 0.9)
        continue;  // gamma selection

      gammaMomentum.m_x = gammaAtCreation[i].mx;
      gammaMomentum.m_y = gammaAtCreation[i].my;
      gammaMomentum.m_z = gammaAtCreation[i].mz;

      if (!IsDetected(centerOfDetector, gammaMomentum, theta))
        continue;
      else {
        pixeEvent.energy_10eV = floor(100 * gammaAtCreation[i].energy_keV + 0.5);
        pixeEvent.projectionIndex = runInfo.projectionIndex;
        pixeEvent.sliceIndex = runInfo.sliceIndex;
        pixeEvent.pixelIndex = runInfo.pixelIndex;
        fwrite(&pixeEvent, 7, 1, out);
        count2++;
      }
    }
  }
  printf("---------------Number of PixeEvent in in the second file: %lld------------------------\n",
         count2);

  // ************************************************************
  // **********************READ SECOND FILE (end)****************
  // ************************************************************

  printf("---------------Number of PixeEvent in total: %lld------------------------\n",
         count1 + count2);
  fclose(input2);
  fclose(out);

  // Recheck the output file in case
  // FILE* input2 = fopen("PixeEvent_std_AtCreation.DAT","rb");
  // PixeEvent p;
  // while(fread(&p, 7, 1, input2))
  // {
  // printf("__ProjectionIndex=%d, SliceIndex=%d, PixelIndex=%d, Energy_10eV=%d\n",
  // p.projectionIndex, p.sliceIndex, p.pixelIndex, p.energy_10eV);

  // }
  // fclose(input2);
}
