//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// INCL++ intra-nuclear cascade model
// Alain Boudard, CEA-Saclay, France
// Joseph Cugnon, University of Liege, Belgium
// Jean-Christophe David, CEA-Saclay, France
// Pekka Kaitaniemi, CEA-Saclay, France, and Helsinki Institute of Physics, Finland
// Sylvie Leray, CEA-Saclay, France
// Davide Mancusi, CEA-Saclay, France
//
#define INCLXX_IN_GEANT4_MODE 1

#include "globals.hh"

#include "G4INCLNKbToLpiChannel.hh"
#include "G4INCLKinematicsUtils.hh"
#include "G4INCLBinaryCollisionAvatar.hh"
#include "G4INCLRandom.hh"
#include "G4INCLGlobals.hh"
#include "G4INCLLogger.hh"
#include <algorithm>
#include "G4INCLPhaseSpaceGenerator.hh"

namespace G4INCL {
	
	NKbToLpiChannel::NKbToLpiChannel(Particle *p1, Particle *p2)
		: particle1(p1), particle2(p2)
		{}
	
	NKbToLpiChannel::~NKbToLpiChannel(){}
	
	void NKbToLpiChannel::fillFinalState(FinalState *fs) {
		
		Particle *nucleon;
		Particle *kaon;
		
		if(particle1->isNucleon()){
			nucleon = particle1;
			kaon = particle2;
		}
		else{
			nucleon = particle2;
			kaon = particle1;
		}
		const G4int iso = ParticleTable::getIsospin(nucleon->getType()) + ParticleTable::getIsospin(kaon->getType());
		
		
		ThreeVector mom_pion = KaonMomentum(kaon,nucleon); // based on K- p -> Lambda pi0
		
		kaon->setType(ParticleTable::getPionType(iso));
		nucleon->setType(Lambda);
		
		G4double norm = KinematicsUtils::momentumInCM(kaon,nucleon);

		kaon->setMomentum(mom_pion*norm);
		nucleon->setMomentum(-mom_pion*norm);
		
		kaon->adjustEnergyFromMomentum();
		nucleon->adjustEnergyFromMomentum();
		
		fs->addModifiedParticle(nucleon);
		fs->addModifiedParticle(kaon);
				
	}
	
	ThreeVector NKbToLpiChannel::KaonMomentum(Particle const * const kaon, Particle const * const nucleon){
		
		const G4double pLab = KinematicsUtils::momentumInLab(kaon,nucleon);
		
		if(pLab < 435.) return Random::normVector(); // isotropic
		
		G4double cos_theta = 1.;
		G4double sin_theta = 0.;
		const G4double cos_phi = std::cos(Random::shoot()*Math::twoPi);
		const G4double sin_phi = std::sqrt(1-cos_phi*cos_phi);
		
		const G4double x = kaon->getMomentum().getX();
		const G4double y = kaon->getMomentum().getY();
		const G4double z = kaon->getMomentum().getZ();
		
		const G4double r = std::sqrt(x*x+y*y+z*z);
		const G4double rho = std::sqrt(x*x+y*y);
		
		if(pLab >= 1845.){
			const G4double b = 12. * pLab/2375.; // correspond to the forward slope description at 2375 MeV/c in K- p elastic
			cos_theta = std::log(Random::shoot()*(std::exp(b)-std::exp(-b))+std::exp(-b))/b;
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
			
		}
		else{
			const G4double Legendre_coef[283][8] = {
				{435,1.29381,0.87668,0.24531,0.00844,-0.0067,4e-05,0},
				{440,1.28561,0.87491,0.24024,0.00529,-0.00609,3e-05,0},
				{445,1.27743,0.87315,0.23522,0.00212,-0.00547,3e-05,0},
				{450,1.26928,0.87139,0.2303,-0.00105,-0.00485,2e-05,0},
				{455,1.2612,0.86965,0.22553,-0.00424,-0.0042,1e-05,0},
				{460,1.25317,0.86792,0.22099,-0.00745,-0.00354,0,0},
				{465,1.24512,0.86623,0.21678,-0.01069,-0.00284,0,0},
				{470,1.23692,0.86461,0.21304,-0.01397,-0.00211,-1e-05,0},
				{475,1.22848,0.86311,0.20991,-0.01729,-0.00132,-2e-05,0},
				{480,1.21967,0.86172,0.20749,-0.02066,-0.00048,-3e-05,0},
				{485,1.21035,0.86044,0.20577,-0.02407,0.00044,-3e-05,0},
				{490,1.2004,0.85923,0.20473,-0.02748,0.00143,-4e-05,0},
				{495,1.18967,0.85804,0.20434,-0.03087,0.00251,-5e-05,0},
				{500,1.17802,0.85684,0.20455,-0.03423,0.00369,-6e-05,0},
				{505,1.16533,0.8556,0.20535,-0.03753,0.00497,-7e-05,0},
				{510,1.15145,0.85427,0.20669,-0.04075,0.00637,-8e-05,0},
				{515,1.13624,0.85282,0.20855,-0.04386,0.0079,-9e-05,0},
				{520,1.11972,0.85121,0.21086,-0.0468,0.00955,-1e-04,0},
				{525,1.10208,0.84953,0.21379,-0.04947,0.01133,-0.00011,0},
				{530,1.0835,0.84789,0.21759,-0.05178,0.01323,-0.00012,0},
				{535,1.06417,0.84639,0.2225,-0.05361,0.01525,-0.00013,0},
				{540,1.04423,0.8451,0.22862,-0.05493,0.01738,-0.00014,0},
				{545,1.02374,0.844,0.23576,-0.05583,0.0196,-0.00015,0},
				{550,1.00274,0.84308,0.2437,-0.05642,0.02189,-0.00016,0},
				{555,0.98125,0.8423,0.25219,-0.05681,0.02423,-0.00017,0},
				{560,0.95932,0.84165,0.26101,-0.0571,0.02659,-0.00017,0},
				{565,0.93696,0.84108,0.26988,-0.05743,0.02895,-0.00018,0},
				{570,0.91408,0.84053,0.27837,-0.05801,0.03126,-0.00018,0},
				{575,0.89056,0.83994,0.28602,-0.05907,0.03345,-0.00018,0},
				{580,0.86629,0.8393,0.29249,-0.06069,0.03543,-0.00017,0},
				{585,0.84122,0.83869,0.29765,-0.06277,0.0371,-0.00016,0},
				{590,0.81531,0.83818,0.30154,-0.06516,0.03842,-0.00015,0},
				{595,0.78853,0.83781,0.30425,-0.06771,0.03943,-0.00012,0},
				{600,0.76086,0.83762,0.30587,-0.07025,0.04015,-9e-05,1e-05},
				{605,0.73242,0.83777,0.30661,-0.07259,0.0406,-6e-05,1e-05},
				{610,0.7034,0.8385,0.30673,-0.07453,0.04081,-1e-05,1e-05},
				{615,0.67402,0.84003,0.3065,-0.07588,0.0408,5e-05,1e-05},
				{620,0.6445,0.8426,0.30618,-0.07642,0.0406,0.00012,1e-05},
				{625,0.61503,0.84637,0.30581,-0.07599,0.04021,2e-04,2e-05},
				{630,0.5858,0.85147,0.30525,-0.07444,0.03962,3e-04,2e-05},
				{635,0.557,0.85801,0.30443,-0.07169,0.03881,0.00041,2e-05},
				{640,0.5289,0.86611,0.30346,-0.06786,0.03772,0.00053,3e-05},
				{645,0.50176,0.87586,0.30245,-0.06312,0.03626,0.00068,3e-05},
				{650,0.47582,0.8874,0.30153,-0.05762,0.03438,0.00084,3e-05},
				{655,0.45135,0.90083,0.30082,-0.05148,0.03201,0.00101,4e-05},
				{660,0.42865,0.91629,0.30054,-0.0447,0.0292,0.0012,4e-05},
				{665,0.40794,0.93393,0.30079,-0.03707,0.02606,0.00141,4e-05},
				{670,0.38927,0.95387,0.30149,-0.0283,0.02276,0.00163,5e-05},
				{675,0.37268,0.97623,0.30251,-0.01806,0.01946,0.00187,5e-05},
				{680,0.35818,1.0011,0.30373,-0.00607,0.01635,0.0021,5e-05},
				{685,0.34574,1.02858,0.30527,0.00796,0.0136,0.00234,5e-05},
				{690,0.33527,1.05871,0.30741,0.02426,0.01138,0.00257,5e-05},
				{695,0.32658,1.09138,0.31026,0.04292,0.00982,0.0028,5e-05},
				{700,0.31939,1.12634,0.31382,0.06382,0.00895,0.003,5e-05},
				{705,0.31344,1.16334,0.31815,0.08676,0.00879,0.00317,4e-05},
				{710,0.3086,1.20215,0.32345,0.11138,0.00935,0.00328,4e-05},
				{715,0.30471,1.24255,0.32994,0.13731,0.01061,0.00334,3e-05},
				{720,0.30161,1.28419,0.33773,0.16421,0.01265,0.00332,2e-05},
				{725,0.29909,1.32659,0.34682,0.19187,0.01564,0.0032,0},
				{730,0.29695,1.36945,0.35725,0.22005,0.01956,0.00296,-2e-05},
				{735,0.29502,1.41255,0.36915,0.24854,0.0243,0.00258,-4e-05},
				{740,0.29311,1.45571,0.38262,0.27709,0.02974,0.00202,-7e-05},
				{745,0.29109,1.49871,0.39781,0.3055,0.03575,0.00127,-0.00011},
				{750,0.28898,1.54138,0.41503,0.33366,0.0422,0.00029,-0.00015},
				{755,0.28683,1.58345,0.43457,0.36147,0.04894,-0.00094,-0.00019},
				{760,0.28451,1.6243,0.45625,0.38871,0.05591,-0.00244,-0.00025},
				{765,0.28175,1.66337,0.47979,0.41518,0.06308,-0.00425,-0.00031},
				{770,0.27835,1.70025,0.50482,0.44067,0.07042,-0.00639,-0.00037},
				{775,0.27417,1.73456,0.53095,0.46502,0.07785,-0.00889,-0.00044},
				{780,0.26902,1.76599,0.55773,0.48808,0.08531,-0.01176,-0.00052},
				{785,0.26276,1.79435,0.58464,0.50969,0.09264,-0.01503,-6e-04},
				{790,0.2552,1.81947,0.61114,0.52972,0.09971,-0.0187,-0.00068},
				{795,0.24617,1.84117,0.6367,0.54802,0.10638,-0.02279,-0.00077},
				{800,0.23554,1.85933,0.66076,0.56451,0.11244,-0.0273,-0.00086},
				{805,0.22322,1.87397,0.68275,0.57917,0.11765,-0.03222,-0.00095},
				{810,0.20914,1.8851,0.70209,0.592,0.12173,-0.03752,-0.00104},
				{815,0.1932,1.89272,0.7182,0.60299,0.12443,-0.04319,-0.00113},
				{820,0.17533,1.89686,0.73052,0.61214,0.1255,-0.04922,-0.00121},
				{825,0.15565,1.89756,0.73887,0.61947,0.12491,-0.05557,-0.00128},
				{830,0.13439,1.89493,0.74336,0.62504,0.12283,-0.06219,-0.00134},
				{835,0.11178,1.88905,0.74408,0.62889,0.11942,-0.06905,-0.00139},
				{840,0.08807,1.88001,0.74115,0.63106,0.11484,-0.07609,-0.00143},
				{845,0.0635,1.86791,0.73467,0.63161,0.10925,-0.08327,-0.00144},
				{850,0.03831,1.85283,0.72475,0.63058,0.10282,-0.09055,-0.00144},
				{855,0.01278,1.83488,0.7116,0.62802,0.09574,-0.09784,-0.00142},
				{860,-0.01274,1.81419,0.69565,0.624,0.0883,-0.105,-0.00136},
				{865,-0.03789,1.79087,0.67736,0.61858,0.08079,-0.11186,-0.00127},
				{870,-0.06232,1.76507,0.65719,0.61182,0.07351,-0.11824,-0.00115},
				{875,-0.08566,1.73691,0.63561,0.60379,0.06676,-0.12399,-0.00099},
				{880,-0.10755,1.70652,0.61308,0.59454,0.06082,-0.12894,-0.00077},
				{885,-0.12763,1.67402,0.59005,0.58414,0.05598,-0.13293,-0.00051},
				{890,-0.1458,1.63947,0.56679,0.57245,0.05242,-0.13584,-0.00019},
				{895,-0.16195,1.60305,0.54339,0.5594,0.05012,-0.13773,0.00019},
				{900,-0.17602,1.56493,0.51984,0.54502,0.04902,-0.13871,0.00064},
				{905,-0.18798,1.52525,0.4961,0.52929,0.049,-0.13888,0.00117},
				{910,-0.19778,1.48414,0.47216,0.51223,0.04998,-0.13838,0.00177},
				{915,-0.20538,1.44176,0.44798,0.49384,0.05185,-0.13732,0.00247},
				{920,-0.21074,1.39824,0.42355,0.47412,0.05451,-0.13581,0.00325},
				{925,-0.21388,1.35383,0.39886,0.45319,0.05787,-0.13396,0.00414},
				{930,-0.21484,1.30892,0.37401,0.4313,0.06178,-0.13183,0.00512},
				{935,-0.21369,1.2639,0.34905,0.40872,0.06613,-0.1295,0.0062},
				{940,-0.2105,1.21913,0.32406,0.38569,0.07078,-0.12704,0.00739},
				{945,-0.20536,1.17491,0.29905,0.36232,0.07556,-0.12458,0.00869},
				{950,-0.19836,1.13145,0.27401,0.33866,0.08027,-0.12224,0.01009},
				{955,-0.18958,1.08898,0.24895,0.31474,0.08473,-0.12018,0.01158},
				{960,-0.17912,1.04775,0.22386,0.2906,0.08874,-0.11852,0.01318},
				{965,-0.16707,1.00794,0.19872,0.26627,0.09217,-0.11741,0.01487},
				{970,-0.15345,0.96965,0.17358,0.24171,0.09504,-0.11702,0.01662},
				{975,-0.1383,0.93292,0.14845,0.21687,0.09744,-0.1175,0.01841},
				{980,-0.12165,0.89782,0.12337,0.19178,0.09944,-0.11883,0.0202},
				{985,-0.10352,0.86439,0.09835,0.16649,0.10108,-0.12091,0.02197},
				{990,-0.08393,0.83266,0.07345,0.14106,0.1024,-0.12362,0.02367},
				{995,-0.06289,0.80271,0.04868,0.11554,0.10346,-0.12687,0.02527},
				{1000,-0.04044,0.77456,0.02407,0.09,0.10431,-0.13055,0.02675},
				{1005,-0.0166,0.74824,-3e-04,0.06448,0.10496,-0.13453,0.02801},
				{1010,0.00852,0.72368,-0.02409,0.03902,0.1053,-0.13858,0.02872},
				{1015,0.03479,0.70081,-0.0472,0.01366,0.10528,-0.14262,0.02878},
				{1020,0.06212,0.67958,-0.06962,-0.01158,0.10492,-0.14664,0.02821},
				{1025,0.0904,0.65992,-0.09133,-0.03664,0.10421,-0.15064,0.02703},
				{1030,0.11951,0.64179,-0.11234,-0.0615,0.10315,-0.15461,0.02526},
				{1035,0.14935,0.62513,-0.13261,-0.08611,0.10174,-0.15854,0.02291},
				{1040,0.1798,0.60989,-0.15215,-0.11044,0.09999,-0.16241,0.02},
				{1045,0.21078,0.59602,-0.17094,-0.13445,0.0979,-0.16622,0.01656},
				{1050,0.24215,0.58346,-0.18897,-0.15811,0.09547,-0.16996,0.01259},
				{1055,0.27383,0.57216,-0.20623,-0.18137,0.0927,-0.17363,0.00812},
				{1060,0.30569,0.56207,-0.2227,-0.20421,0.08959,-0.1772,0.00316},
				{1065,0.33764,0.55313,-0.23838,-0.22657,0.08615,-0.18068,-0.00227},
				{1070,0.36956,0.54529,-0.25326,-0.24843,0.08237,-0.18405,-0.00815},
				{1075,0.40134,0.53849,-0.26732,-0.26974,0.07826,-0.1873,-0.01447},
				{1080,0.43288,0.53269,-0.28055,-0.29047,0.07382,-0.19043,-0.02119},
				{1085,0.46407,0.52783,-0.29295,-0.31059,0.06905,-0.19342,-0.02832},
				{1090,0.49481,0.52386,-0.30452,-0.33004,0.06396,-0.19627,-0.03582},
				{1095,0.52503,0.52073,-0.31538,-0.34881,0.05856,-0.1989,-0.04359},
				{1100,0.55465,0.51841,-0.3257,-0.36683,0.05286,-0.20125,-0.05154},
				{1105,0.58361,0.51686,-0.3356,-0.38409,0.0469,-0.20327,-0.05957},
				{1110,0.61184,0.51604,-0.34525,-0.40052,0.04068,-0.20488,-0.06757},
				{1115,0.63927,0.5159,-0.35478,-0.4161,0.03422,-0.20602,-0.07545},
				{1120,0.66583,0.51641,-0.36435,-0.43079,0.02755,-0.20662,-0.08311},
				{1125,0.69146,0.51753,-0.37411,-0.44454,0.02068,-0.20662,-0.09043},
				{1130,0.71609,0.51922,-0.38418,-0.45732,0.01364,-0.20594,-0.09736},
				{1135,0.73967,0.5214,-0.3946,-0.46913,0.00649,-0.20451,-0.10389},
				{1140,0.76215,0.52403,-0.40542,-0.47994,-0.00069,-0.20222,-0.11009},
				{1145,0.78348,0.52703,-0.41665,-0.48977,-0.00782,-0.19898,-0.11598},
				{1150,0.80362,0.53033,-0.42833,-0.49859,-0.01484,-0.19469,-0.1216},
				{1155,0.82251,0.53388,-0.44048,-0.50642,-0.02166,-0.18927,-0.12701},
				{1160,0.8401,0.53761,-0.45313,-0.51323,-0.02821,-0.18261,-0.13223},
				{1165,0.85635,0.54144,-0.46631,-0.51903,-0.03442,-0.17462,-0.13732},
				{1170,0.87122,0.54533,-0.48004,-0.52382,-0.04023,-0.16525,-0.14231},
				{1175,0.88472,0.54924,-0.49427,-0.52765,-0.0457,-0.15454,-0.1472},
				{1180,0.89689,0.55314,-0.50896,-0.5306,-0.0509,-0.1426,-0.15202},
				{1185,0.90774,0.55701,-0.52406,-0.53273,-0.05591,-0.12951,-0.15675},
				{1190,0.91731,0.56082,-0.53952,-0.53411,-0.06079,-0.11537,-0.16142},
				{1195,0.92561,0.56455,-0.55528,-0.53481,-0.06563,-0.10027,-0.16601},
				{1200,0.93269,0.56816,-0.57129,-0.53488,-0.07049,-0.08429,-0.17056},
				{1205,0.93856,0.57163,-0.58751,-0.53441,-0.07546,-0.06754,-0.17505},
				{1210,0.94325,0.57494,-0.60389,-0.53345,-0.0806,-0.0501,-0.17949},
				{1215,0.94679,0.57806,-0.62037,-0.53208,-0.086,-0.03207,-0.1839},
				{1220,0.94921,0.58096,-0.6369,-0.53036,-0.09172,-0.01354,-0.18827},
				{1225,0.95052,0.58361,-0.65343,-0.52836,-0.09784,0.00541,-0.19262},
				{1230,0.95077,0.58599,-0.66992,-0.52614,-0.10444,0.02468,-0.19694},
				{1235,0.94997,0.58808,-0.68631,-0.52378,-0.11159,0.04417,-0.20126},
				{1240,0.94816,0.58984,-0.70255,-0.52133,-0.11936,0.06379,-0.20557},
				{1245,0.94535,0.59125,-0.71859,-0.51888,-0.12783,0.08346,-0.20987},
				{1250,0.94157,0.59229,-0.73437,-0.51648,-0.13706,0.10305,-0.21416},
				{1255,0.93683,0.59293,-0.74976,-0.51419,-0.1471,0.12231,-0.21828},
				{1260,0.93113,0.59315,-0.76463,-0.51207,-0.15797,0.14096,-0.22206},
				{1265,0.92447,0.59294,-0.77883,-0.51019,-0.1697,0.15872,-0.22534},
				{1270,0.91689,0.59232,-0.79218,-0.50845,-0.18223,0.17547,-0.22804},
				{1275,0.90848,0.5914,-0.80445,-0.5066,-0.19545,0.19131,-0.23018},
				{1280,0.89933,0.59026,-0.81539,-0.50438,-0.20919,0.20634,-0.2318},
				{1285,0.88952,0.58902,-0.82477,-0.50151,-0.22331,0.22066,-0.23293},
				{1290,0.87914,0.58776,-0.83234,-0.49778,-0.23768,0.23437,-0.23361},
				{1295,0.86827,0.58655,-0.83797,-0.49307,-0.25217,0.24756,-0.23388},
				{1300,0.85696,0.58546,-0.84148,-0.48733,-0.26668,0.26028,-0.23375},
				{1305,0.8453,0.58454,-0.84274,-0.4805,-0.2811,0.27262,-0.23327},
				{1310,0.83334,0.58385,-0.84159,-0.47251,-0.29532,0.28466,-0.23245},
				{1315,0.82116,0.58345,-0.8379,-0.46331,-0.30922,0.29645,-0.23134},
				{1320,0.80881,0.58341,-0.8315,-0.45284,-0.32269,0.30807,-0.22995},
				{1325,0.79636,0.58376,-0.82231,-0.44106,-0.33567,0.31957,-0.22833},
				{1330,0.78383,0.58452,-0.81043,-0.42806,-0.34819,0.33087,-0.22653},
				{1335,0.77121,0.58565,-0.79603,-0.41395,-0.36036,0.34185,-0.22459},
				{1340,0.75852,0.58715,-0.77928,-0.39886,-0.37227,0.3524,-0.22258},
				{1345,0.74577,0.589,-0.76033,-0.38289,-0.38399,0.36242,-0.22055},
				{1350,0.73295,0.59118,-0.73935,-0.36617,-0.39563,0.37178,-0.21857},
				{1355,0.72008,0.59368,-0.71651,-0.3488,-0.40726,0.38039,-0.21667},
				{1360,0.70715,0.59646,-0.69194,-0.3309,-0.41893,0.38818,-0.2149},
				{1365,0.6941,0.59943,-0.66566,-0.31253,-0.43043,0.39531,-0.21314},
				{1370,0.68084,0.60248,-0.63764,-0.29373,-0.44151,0.40198,-0.21126},
				{1375,0.66737,0.60553,-0.60795,-0.27456,-0.45201,0.40833,-0.20918},
				{1380,0.65371,0.60856,-0.57672,-0.25501,-0.46189,0.41438,-0.2069},
				{1385,0.63991,0.61158,-0.5441,-0.2351,-0.47112,0.42015,-0.20442},
				{1390,0.62603,0.61456,-0.51023,-0.21484,-0.47966,0.42567,-0.20173},
				{1395,0.6121,0.6175,-0.47525,-0.19424,-0.48749,0.43095,-0.19884},
				{1400,0.59816,0.62039,-0.43932,-0.17332,-0.49457,0.43601,-0.19575},
				{1405,0.58428,0.62321,-0.40257,-0.15208,-0.50088,0.44087,-0.19246},
				{1410,0.57048,0.62596,-0.36515,-0.13053,-0.50637,0.44556,-0.18897},
				{1415,0.55682,0.62862,-0.32721,-0.1087,-0.51102,0.4501,-0.18528},
				{1420,0.54335,0.63118,-0.28889,-0.08658,-0.5148,0.4545,-0.18139},
				{1425,0.5301,0.63363,-0.25033,-0.06419,-0.51767,0.45878,-0.1773},
				{1430,0.51712,0.63597,-0.21167,-0.04154,-0.5196,0.46298,-0.173},
				{1435,0.50446,0.63817,-0.17307,-0.01864,-0.52057,0.4671,-0.16851},
				{1440,0.49217,0.64024,-0.13467,0.0045,-0.52053,0.47116,-0.16382},
				{1445,0.48028,0.64215,-0.09661,0.02786,-0.51946,0.4752,-0.15892},
				{1450,0.46885,0.64391,-0.05904,0.05144,-0.51733,0.47922,-0.15383},
				{1455,0.45792,0.64548,-0.02209,0.07522,-0.51411,0.48325,-0.14854},
				{1460,0.44754,0.64688,0.01408,0.09919,-0.50975,0.48732,-0.14305},
				{1465,0.43774,0.64808,0.04933,0.12335,-0.50424,0.49143,-0.13736},
				{1470,0.42855,0.64906,0.08361,0.14764,-0.49758,0.49559,-0.13154},
				{1475,0.41997,0.6498,0.11691,0.17201,-0.4898,0.49979,-0.12566},
				{1480,0.41199,0.65025,0.14924,0.19638,-0.48094,0.50402,-0.11983},
				{1485,0.4046,0.65039,0.1806,0.2207,-0.47103,0.50827,-0.11413},
				{1490,0.39781,0.65018,0.211,0.2449,-0.4601,0.51253,-0.10866},
				{1495,0.39161,0.6496,0.24043,0.26892,-0.44819,0.51679,-0.10352},
				{1500,0.38599,0.64862,0.26889,0.29269,-0.43534,0.52104,-0.09879},
				{1505,0.38096,0.64719,0.2964,0.31615,-0.42158,0.52526,-0.09457},
				{1510,0.37651,0.6453,0.32295,0.33923,-0.40693,0.52944,-0.09095},
				{1515,0.37264,0.6429,0.34854,0.36186,-0.39144,0.53359,-0.08803},
				{1520,0.36933,0.63998,0.37315,0.38401,-0.37512,0.53773,-0.08583},
				{1525,0.36653,0.63654,0.39667,0.40564,-0.35794,0.54203,-0.08422},
				{1530,0.3642,0.63257,0.41901,0.42676,-0.33984,0.54668,-0.08304},
				{1535,0.36228,0.62809,0.44005,0.44733,-0.3208,0.55185,-0.08216},
				{1540,0.36072,0.62309,0.45968,0.46736,-0.30076,0.55773,-0.08141},
				{1545,0.35948,0.61757,0.47781,0.48683,-0.27969,0.5645,-0.08065},
				{1550,0.35851,0.61154,0.49434,0.50572,-0.25756,0.57231,-0.07974},
				{1555,0.35781,0.60502,0.50929,0.52405,-0.23439,0.58116,-0.07861},
				{1560,0.35738,0.59804,0.52273,0.54186,-0.21023,0.59098,-0.07724},
				{1565,0.35725,0.59063,0.53473,0.55915,-0.18514,0.60169,-0.07558},
				{1570,0.35742,0.58282,0.54538,0.57597,-0.15918,0.61322,-0.07362},
				{1575,0.35792,0.57463,0.55473,0.59234,-0.13239,0.62549,-0.07132},
				{1580,0.35874,0.5661,0.56287,0.60828,-0.10484,0.63843,-0.06864},
				{1585,0.35991,0.55725,0.56986,0.62383,-0.07657,0.65196,-0.06557},
				{1590,0.36144,0.54811,0.57578,0.63899,-0.04764,0.66601,-0.06206},
				{1595,0.36335,0.53871,0.5807,0.65382,-0.0181,0.6805,-0.05809},
				{1600,0.36564,0.52909,0.5847,0.66832,0.01199,0.69536,-0.05363},
				{1605,0.36833,0.51926,0.58783,0.68253,0.04258,0.7105,-0.04864},
				{1610,0.37143,0.50926,0.59019,0.69647,0.07361,0.72586,-0.0431},
				{1615,0.37496,0.49911,0.59183,0.71017,0.10503,0.74137,-0.03697},
				{1620,0.37894,0.48885,0.59284,0.72365,0.13679,0.75693,-0.03023},
				{1625,0.38336,0.47851,0.59329,0.73695,0.16883,0.77249,-0.02284},
				{1630,0.38826,0.46811,0.59324,0.75008,0.2011,0.78797,-0.01477},
				{1635,0.39363,0.45767,0.59277,0.76308,0.23354,0.80328,-0.00599},
				{1640,0.39951,0.44724,0.59195,0.77597,0.26611,0.81836,0.00352},
				{1645,0.40588,0.43684,0.59086,0.78878,0.29874,0.83312,0.01381},
				{1650,0.41278,0.4265,0.58956,0.80153,0.33138,0.84751,0.0249},
				{1655,0.42022,0.41624,0.58813,0.81425,0.36397,0.86143,0.03681},
				{1660,0.4282,0.40614,0.58665,0.82695,0.39646,0.87483,0.04954},
				{1665,0.43672,0.39632,0.58517,0.83966,0.42876,0.88768,0.06298},
				{1670,0.44579,0.38689,0.58378,0.85238,0.46079,0.89994,0.07703},
				{1675,0.45542,0.37799,0.58253,0.86511,0.49245,0.91158,0.09161},
				{1680,0.46559,0.36972,0.5815,0.87787,0.52367,0.92256,0.1066},
				{1685,0.47631,0.36223,0.58075,0.89067,0.55437,0.93285,0.12193},
				{1690,0.48759,0.35562,0.58036,0.90351,0.58446,0.94242,0.13748},
				{1695,0.49943,0.35002,0.58039,0.91642,0.61386,0.95122,0.15316},
				{1700,0.51182,0.34555,0.58092,0.92938,0.64248,0.95923,0.16887},
				{1705,0.52478,0.34234,0.58201,0.94243,0.67025,0.96642,0.18453},
				{1710,0.5383,0.34049,0.58374,0.95556,0.69713,0.97277,0.20007},
				{1715,0.55239,0.34002,0.58616,0.96883,0.72336,0.97841,0.21569},
				{1720,0.56707,0.34092,0.58936,0.98229,0.74924,0.98349,0.23159},
				{1725,0.58235,0.34319,0.59342,0.99599,0.77504,0.98814,0.24802},
				{1730,0.59825,0.34683,0.59839,1.00998,0.80106,0.99252,0.26518},
				{1735,0.61478,0.35185,0.60435,1.02432,0.82759,0.99678,0.28332},
				{1740,0.63195,0.35823,0.61138,1.03905,0.85493,1.00106,0.30266},
				{1745,0.64977,0.36598,0.61954,1.05422,0.88333,1.00549,0.32339},
				{1750,0.66822,0.37504,0.6288,1.06985,0.91282,1.01013,0.34556},
				{1755,0.68725,0.38533,0.6391,1.08592,0.94333,1.01496,0.36913},
				{1760,0.70685,0.39679,0.65037,1.10245,0.9748,1.01998,0.39406},
				{1765,0.72695,0.40934,0.66257,1.11943,1.00714,1.02519,0.42031},
				{1770,0.74754,0.42291,0.67563,1.13684,1.0403,1.03058,0.44784},
				{1775,0.76857,0.43743,0.68947,1.15471,1.07419,1.03615,0.47661},
				{1780,0.79,0.45282,0.70406,1.17301,1.10875,1.04191,0.50658},
				{1785,0.81179,0.46902,0.71931,1.19174,1.14391,1.04784,0.53771},
				{1790,0.83392,0.48594,0.73518,1.21092,1.17959,1.05394,0.56997},
				{1795,0.85634,0.50352,0.75159,1.23052,1.21573,1.06022,0.60332},
				{1800,0.87901,0.52169,0.76849,1.25055,1.25225,1.06666,0.63771},
				{1805,0.9019,0.54038,0.78582,1.27101,1.28908,1.07326,0.6731},
				{1810,0.92496,0.5595,0.80351,1.2919,1.32615,1.08003,0.70946},
				{1815,0.94817,0.57899,0.82151,1.31321,1.36339,1.08695,0.74675},
				{1820,0.97149,0.59878,0.83975,1.33493,1.40073,1.09403,0.78493},
				{1825,0.99487,0.61879,0.85816,1.35707,1.4381,1.10125,0.82396},
				{1830,1.01828,0.63895,0.8767,1.37963,1.47543,1.10863,0.86381},
				{1835,1.04168,0.65919,0.89529,1.4026,1.51264,1.11616,0.90442},
				{1840,1.06504,0.67943,0.91388,1.42598,1.54966,1.12382,0.94576},
				{1845,1.08832,0.69962,0.93241,1.44973,1.58645,1.13162,0.98775}};
			
			const G4int coef_ener = G4int((pLab-Legendre_coef[0][0])/5);
			const G4double sup_ener = pLab/5. - coef_ener -Legendre_coef[0][0]/5;
			
// assert(pLab >= Legendre_coef[coef_ener][0] && pLab < Legendre_coef[coef_ener+1][0]);
			
			// Legendre coefficient normalized
			const G4double A0 = 1.;
			const G4double A1 = (1-sup_ener)*Legendre_coef[coef_ener][1] + sup_ener*Legendre_coef[coef_ener+1][1];
			const G4double A2 = (1-sup_ener)*Legendre_coef[coef_ener][2] + sup_ener*Legendre_coef[coef_ener+1][2];
			const G4double A3 = (1-sup_ener)*Legendre_coef[coef_ener][3] + sup_ener*Legendre_coef[coef_ener+1][3];
			const G4double A4 = (1-sup_ener)*Legendre_coef[coef_ener][4] + sup_ener*Legendre_coef[coef_ener+1][4];
			const G4double A5 = (1-sup_ener)*Legendre_coef[coef_ener][5] + sup_ener*Legendre_coef[coef_ener+1][5];
			const G4double A6 = (1-sup_ener)*Legendre_coef[coef_ener][6] + sup_ener*Legendre_coef[coef_ener+1][6];
			const G4double A7 = (1-sup_ener)*Legendre_coef[coef_ener][7] + sup_ener*Legendre_coef[coef_ener+1][7];
			
			// Theoritical max if all Ai > 0 (often the case)
			const G4double A = std::fabs(A0) + std::fabs(A1) + std::fabs(A2) + std::fabs(A3) + std::fabs(A4) + std::fabs(A5) + std::fabs(A6) + std::fabs(A7);
			
			G4bool success = false;
			G4int maxloop = 0;
			
			while(!success && maxloop < 1000){
				
				cos_theta = Random::shoot()*2-1.; // not optimized
				
				// Legendre Polynomial
				G4double P0 = A0;
				G4double P1 = A1*cos_theta;
				G4double P2 = A2/2.*(3*std::pow(cos_theta,2)-1);
				G4double P3 = A3/2.*(5*std::pow(cos_theta,3)-3*cos_theta);
				G4double P4 = A4/8.*(35*std::pow(cos_theta,4)-30*std::pow(cos_theta,2)+3);
				G4double P5 = A5/8.*(63*std::pow(cos_theta,5)-70*std::pow(cos_theta,3)+15*cos_theta);
				G4double P6 = A6/16.*(231*std::pow(cos_theta,6)-315*std::pow(cos_theta,4)+105*std::pow(cos_theta,2)-5);
				G4double P7 = A7/16.*(429*std::pow(cos_theta,7)-693*std::pow(cos_theta,5)+315*std::pow(cos_theta,3)-35*cos_theta);
				
				G4double P = (P0 + P1 + P2 + P3 + P4 + P5 + P6 + P7)/2; // /2 for the normalisation
				
				if(Random::shoot()*A < P) success = true;
				maxloop +=1 ;
				if(maxloop==1000) cos_theta = std::log(Random::shoot()*(std::exp(10.)-std::exp(-10.))+std::exp(-10.))/10.; // if no success in 1E4 shoot, probably angulard distribution piked very forward 
			}
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
		}
		
		if(rho == 0) return ThreeVector(sin_theta*cos_phi,sin_theta*sin_phi,cos_theta);
		// Rotation in the direction of the incident kaon
		const G4double px = x/r*cos_theta - y/rho*sin_theta*cos_phi + z/r*x/rho*sin_theta*sin_phi;
		const G4double py = y/r*cos_theta + x/rho*sin_theta*cos_phi + z/r*y/rho*sin_theta*sin_phi;
		const G4double pz = z/r*cos_theta - rho/r*sin_theta*sin_phi;
		
		return ThreeVector(px,py,pz);
	}

}
