/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Patch by Alan Carriou <cariou_alan@yahoo.fr> (C) 2004-2005
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>

 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.

 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.

 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include "lib_acpi.h"
#include "wmacpiload.h"
#include "battery.h"

/************************************************************************
 * Macro definitions
 ************************************************************************/

/* used to parse the battery state file */
#define CAPACITY_UNIT " mAh"

/************************************************************************
 * Global constants definitions
 ************************************************************************/

const char * const battery_dir = "/battery/";
const char * const battery_status_file = "/state";
const char * const battery_info_file = "/info";

/************************************************************************
 * Function definitions
 ************************************************************************/

void *
battery_init(struct dirent *fd)
{
    Battery *bat;

    if (fd == NULL)
        return NULL;

    bat = malloc(sizeof(*bat));
    if (!bat) {
        fprintf(stderr, "%s : could not allocate memory "
                "for battery information concerning battery '%s'\n",
                PACKAGE, fd->d_name);
        return NULL;
    }

    bat->name = NULL;
    bat->statefile = NULL;
    bat->infofile = NULL;
    bat->next = NULL;
    bat->percentage = -1;

    bat->name = malloc(strlen(fd->d_name) + 1);
    if (!bat->name) {
        fprintf(stderr, "%s : could not allocate memory "
                "for battery name concerning battery '%s'\n",
                PACKAGE, fd->d_name);
        free(bat);
        return NULL;
    }
    strcpy(bat->name, fd->d_name);

    bat->statefile =
        strcat4("/proc/acpi", battery_dir, bat->name, battery_status_file);
    if (!bat->statefile) {
        fprintf(stderr, "%s : could not allocate memory "
                "for battery statefile concerning battery '%s'\n",
                PACKAGE, bat->name);
        free(bat->name);
        free(bat);
        return NULL;
    }

    bat->infofile =
        strcat4("/proc/acpi", battery_dir, bat->name, battery_info_file);
    if (!bat->infofile) {
        fprintf(stderr, "%s : could not allocate memory "
                "for battery infofile concerning battery '%s'\n",
                PACKAGE, bat->name);
        free(bat->statefile);
        free(bat->name);
        free(bat);
        return NULL;
    }

    battery_check_plugged(bat);
    if (bat->plugged)
        battery_get_capacity(bat);

    if (verbose)
        printf("Found battery slot %s ; battery is plugged %s\n",
               bat->name, bat->plugged ? "in" : "out");

    if (verbose > 1)
        printf("Battery slot %s: infofile %s, statefile %s\n", bat->name,
               bat->infofile, bat->statefile);

    return bat;
}

void
battery_check_plugged(Battery *bat)
{
    char buf[5];

    if (bat != NULL) {
        if (parse_file(bat->statefile, "present:", buf, sizeof buf) != 0) {
            if (strcmp(buf, "yes") == 0)
                bat->plugged = 1;
            else {
                bat->plugged = 0;
#if 0
                if (strcmp(buf, "no") != 0)
                    /* "present" is neither "yes" nor "no" */
                    fprintf(stderr,
                            "%s : Unconsistent data about battery %s\n",
                            PACKAGE, bat->name);
#endif
            }
        } else {
            bat->plugged = 0;
        }
    }
}

void
battery_get_capacity(Battery *bat)
{
    unsigned long int capacity;
    char buf[10];

    if (bat != NULL) {
        capacity = 0;
        if (parse_file(bat->infofile, "last full capacity:", buf,
                       sizeof buf) != NULL) {
            char *endptr;

            capacity = strtoul(buf, &endptr, 10);
            if (strcmp(endptr, CAPACITY_UNIT) != 0) {
                fprintf(stderr,
                        "%s : Battery %s unplugged or unconsistent data\n",
                        PACKAGE, bat->name);
                capacity = 0;
            }
        }
        bat->capacity = capacity;

        if (verbose > 1)
            printf("Battery %s: new capacity %lu\n", bat->name, capacity);
    }
}

void
battery_next(AcpiInfos *infos)
{
    if (infos && infos->bat) {
        if (infos->bat->next == NULL)
            infos->bat = infos->bat_first;
        else
            infos->bat = infos->bat->next;

        if (verbose)
            printf("Now monitoring battery %s.\n", infos->bat->name);
    }
}

void
battery_update_plugged(Battery *bat)
{
    if (bat) {
        if (bat->plugged) {
            /* checks whether the battery has been plugged out since last update */
            battery_check_plugged(bat);
            if (verbose && !bat->plugged)
                printf("Battery %s has been plugged out\n", bat->name);
        } else {
            /* checks whether the battery has been plugged in since last update */
            battery_check_plugged(bat);

            if (bat->plugged) {
                if (verbose)
                    printf("Battery %s has been plugged in\n", bat->name);
                battery_get_capacity(bat);
            }
        }
    }
}

void
battery_update_status(Battery *bat)
{
    char buf[13];
    long remain;

    if (bat && bat->plugged) {
        if (parse_file(bat->statefile, "charging state:", buf,
                       sizeof buf) != NULL) {
            if (strcmp(buf, "charging") == 0)
                bat->status = Charging;
            else if (strcmp(buf, "discharging") == 0)
                bat->status = Discharging;
            else
                bat->status = Unknown;
        } else {
            bat->status = Unknown;
        }

        if (parse_file(bat->statefile, "remaining capacity:", buf,
                       sizeof buf) != NULL) {
            char *endptr;

            remain = strtol(buf, &endptr, 10);
            if (strcmp(endptr, CAPACITY_UNIT) == 0)
                bat->percentage = (int) ((remain * 100.0) / bat->capacity);
            else
                bat->percentage = -1;
        }
    }
    if (verbose > 1)
        printf("Battery %s: new percentage %d\n", bat->name,
               bat->percentage);
}

void
battery_free(Battery *bat)
{
    if (!bat) {
        MYFREE(bat->name);
        MYFREE(bat->statefile);
        MYFREE(bat->infofile);
        free(bat);
    }
}
