/*
Copyright 2017 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package cpumanager

import (
	"k8s.io/kubernetes/pkg/kubelet/cm/cpumanager/topology"
)

var (
	topoSingleSocketHT = &topology.CPUTopology{
		NumCPUs:    8,
		NumSockets: 1,
		NumCores:   4,
		CPUDetails: map[int]topology.CPUInfo{
			0: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			2: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			4: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			5: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			6: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			7: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
		},
	}

	topoDualSocketHT = &topology.CPUTopology{
		NumCPUs:        12,
		NumSockets:     2,
		NumCores:       6,
		NumUncoreCache: 1,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1:  {CoreID: 1, SocketID: 1, NUMANodeID: 1},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3:  {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			5:  {CoreID: 5, SocketID: 1, NUMANodeID: 1},
			6:  {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			7:  {CoreID: 1, SocketID: 1, NUMANodeID: 1},
			8:  {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			9:  {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			10: {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			11: {CoreID: 5, SocketID: 1, NUMANodeID: 1},
		},
	}

	topoUncoreDualSocketNoSMT = &topology.CPUTopology{
		NumCPUs:        16,
		NumSockets:     2,
		NumCores:       16,
		NumUncoreCache: 4,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			8:  {CoreID: 8, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 2},
			9:  {CoreID: 9, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 2},
			10: {CoreID: 10, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 2},
			11: {CoreID: 11, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 2},
			12: {CoreID: 12, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 3},
			13: {CoreID: 13, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 3},
			14: {CoreID: 14, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 3},
			15: {CoreID: 15, SocketID: 1, NUMANodeID: 0, UncoreCacheID: 3},
		},
	}

	topoUncoreSingleSocketMultiNuma = &topology.CPUTopology{
		NumCPUs:        16,
		NumSockets:     1,
		NumCores:       16,
		NumUncoreCache: 4,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			8:  {CoreID: 8, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 2},
			9:  {CoreID: 9, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 2},
			10: {CoreID: 10, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 2},
			11: {CoreID: 11, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 2},
			12: {CoreID: 12, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 3},
			13: {CoreID: 13, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 3},
			14: {CoreID: 14, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 3},
			15: {CoreID: 15, SocketID: 0, NUMANodeID: 1, UncoreCacheID: 3},
		},
	}

	topoUncoreSingleSocketSMT = &topology.CPUTopology{
		NumCPUs:        16,
		NumSockets:     1,
		NumCores:       8,
		NumUncoreCache: 2,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			8:  {CoreID: 0, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			9:  {CoreID: 1, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			10: {CoreID: 2, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			11: {CoreID: 3, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			12: {CoreID: 4, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			13: {CoreID: 5, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			14: {CoreID: 6, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			15: {CoreID: 7, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
		},
	}

	topoUncoreSingleSocketNoSMT = &topology.CPUTopology{
		NumCPUs:        16,
		NumSockets:     1,
		NumCores:       16,
		NumUncoreCache: 4,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 1},
			8:  {CoreID: 8, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 2},
			9:  {CoreID: 9, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 2},
			10: {CoreID: 10, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 2},
			11: {CoreID: 11, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 2},
			12: {CoreID: 12, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 3},
			13: {CoreID: 13, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 3},
			14: {CoreID: 14, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 3},
			15: {CoreID: 15, SocketID: 0, NUMANodeID: 0, UncoreCacheID: 3},
		},
	}

	topoDualSocketNoHT = &topology.CPUTopology{
		NumCPUs:    8,
		NumSockets: 2,
		NumCores:   8,
		CPUDetails: map[int]topology.CPUInfo{
			0: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			2: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			4: {CoreID: 4, SocketID: 1, NUMANodeID: 1},
			5: {CoreID: 5, SocketID: 1, NUMANodeID: 1},
			6: {CoreID: 6, SocketID: 1, NUMANodeID: 1},
			7: {CoreID: 7, SocketID: 1, NUMANodeID: 1},
		},
	}

	// fake topology for testing purposes only
	topoTripleSocketHT = &topology.CPUTopology{
		NumCPUs:    18,
		NumSockets: 3,
		NumCores:   9,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 1, NUMANodeID: 1},
			1:  {CoreID: 0, SocketID: 1, NUMANodeID: 1},
			2:  {CoreID: 1, SocketID: 1, NUMANodeID: 1},
			3:  {CoreID: 1, SocketID: 1, NUMANodeID: 1},
			4:  {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			5:  {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			6:  {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			7:  {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			8:  {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			9:  {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			10: {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			11: {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			12: {CoreID: 6, SocketID: 2, NUMANodeID: 2},
			13: {CoreID: 6, SocketID: 2, NUMANodeID: 2},
			14: {CoreID: 7, SocketID: 2, NUMANodeID: 2},
			15: {CoreID: 7, SocketID: 2, NUMANodeID: 2},
			16: {CoreID: 8, SocketID: 2, NUMANodeID: 2},
			17: {CoreID: 8, SocketID: 2, NUMANodeID: 2},
		},
	}

	/*
		Topology from https://www.open-mpi.org/projects/hwloc/lstopo/images/KNL.SNC4.H50.v1.11.png.
		Socket0:
		0-2,9-10,13-14,21-22,25-26,33-34,38-39,46-47,50,57-58,71-72,79-80,87-88,95-96,103-104,109-110,117-118,
		131-132,139-140,147-148,155-156,163-164,169-170,177-178,191-192,199-200,207-208,215-216,223-224,229-230,
		237-238,251-252,259-260,267-268,275-276,283-284
		Socket1:
		3-4,11-12,15-16,23-24,27-28,35-36,40-41,48-49,51-52,59-60,65-66,73-74,81-82,89-90,97-98,111-112,119-120,125-126,
		133-134,141-142,149-150,157-158,171-172,179-180,185-186,193-194,201-202,209-210,217-218,231-232,239-240,245-246,
		253-254,261-262,269-270,277-278
		Socket2:
		5-6,17-18,29-30,42-43,53-54,61-62,67-68,75-76,83-84,91-92,99-100,105-106,113-114,121-122,127-128,135-136,
		143-144,151-152,159-160,165-166,173-174,181-182,187-188,195-196,203-204,211-212,219-220,225-226,233-234,241-242,
		247-248,255-256,263-264,271-272,279-280,285-286
		Socket3:
		7-8,19-20,31-32,37,44-45,55-56,63-64,69-70,77-78,85-86,93-94,101-102,107-108,115-116,123-124,129-130,137-138,
		145-146,153-154,161-162,167-168,175-176,183-184,189-190,197-198,205-206,213-214,221-222,227-228,235-236,243-244,
		249-250,257-258,265-266,273-274,281-282,287
	*/
	topoQuadSocketFourWayHT = &topology.CPUTopology{
		NumCPUs:    288,
		NumSockets: 4,
		NumCores:   72,
		CPUDetails: map[int]topology.CPUInfo{
			0:   {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			169: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			109: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			229: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			50:  {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			170: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			110: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			230: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			1:   {CoreID: 64, SocketID: 0, NUMANodeID: 0},
			25:  {CoreID: 64, SocketID: 0, NUMANodeID: 0},
			13:  {CoreID: 64, SocketID: 0, NUMANodeID: 0},
			38:  {CoreID: 64, SocketID: 0, NUMANodeID: 0},
			2:   {CoreID: 65, SocketID: 0, NUMANodeID: 0},
			26:  {CoreID: 65, SocketID: 0, NUMANodeID: 0},
			14:  {CoreID: 65, SocketID: 0, NUMANodeID: 0},
			39:  {CoreID: 65, SocketID: 0, NUMANodeID: 0},
			9:   {CoreID: 72, SocketID: 0, NUMANodeID: 0},
			33:  {CoreID: 72, SocketID: 0, NUMANodeID: 0},
			21:  {CoreID: 72, SocketID: 0, NUMANodeID: 0},
			46:  {CoreID: 72, SocketID: 0, NUMANodeID: 0},
			10:  {CoreID: 73, SocketID: 0, NUMANodeID: 0},
			34:  {CoreID: 73, SocketID: 0, NUMANodeID: 0},
			22:  {CoreID: 73, SocketID: 0, NUMANodeID: 0},
			47:  {CoreID: 73, SocketID: 0, NUMANodeID: 0},
			57:  {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			177: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			117: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			237: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			58:  {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			178: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			118: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			238: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			71:  {CoreID: 24, SocketID: 0, NUMANodeID: 0},
			191: {CoreID: 24, SocketID: 0, NUMANodeID: 0},
			131: {CoreID: 24, SocketID: 0, NUMANodeID: 0},
			251: {CoreID: 24, SocketID: 0, NUMANodeID: 0},
			72:  {CoreID: 25, SocketID: 0, NUMANodeID: 0},
			192: {CoreID: 25, SocketID: 0, NUMANodeID: 0},
			132: {CoreID: 25, SocketID: 0, NUMANodeID: 0},
			252: {CoreID: 25, SocketID: 0, NUMANodeID: 0},
			79:  {CoreID: 32, SocketID: 0, NUMANodeID: 0},
			199: {CoreID: 32, SocketID: 0, NUMANodeID: 0},
			139: {CoreID: 32, SocketID: 0, NUMANodeID: 0},
			259: {CoreID: 32, SocketID: 0, NUMANodeID: 0},
			80:  {CoreID: 33, SocketID: 0, NUMANodeID: 0},
			200: {CoreID: 33, SocketID: 0, NUMANodeID: 0},
			140: {CoreID: 33, SocketID: 0, NUMANodeID: 0},
			260: {CoreID: 33, SocketID: 0, NUMANodeID: 0},
			87:  {CoreID: 40, SocketID: 0, NUMANodeID: 0},
			207: {CoreID: 40, SocketID: 0, NUMANodeID: 0},
			147: {CoreID: 40, SocketID: 0, NUMANodeID: 0},
			267: {CoreID: 40, SocketID: 0, NUMANodeID: 0},
			88:  {CoreID: 41, SocketID: 0, NUMANodeID: 0},
			208: {CoreID: 41, SocketID: 0, NUMANodeID: 0},
			148: {CoreID: 41, SocketID: 0, NUMANodeID: 0},
			268: {CoreID: 41, SocketID: 0, NUMANodeID: 0},
			95:  {CoreID: 48, SocketID: 0, NUMANodeID: 0},
			215: {CoreID: 48, SocketID: 0, NUMANodeID: 0},
			155: {CoreID: 48, SocketID: 0, NUMANodeID: 0},
			275: {CoreID: 48, SocketID: 0, NUMANodeID: 0},
			96:  {CoreID: 49, SocketID: 0, NUMANodeID: 0},
			216: {CoreID: 49, SocketID: 0, NUMANodeID: 0},
			156: {CoreID: 49, SocketID: 0, NUMANodeID: 0},
			276: {CoreID: 49, SocketID: 0, NUMANodeID: 0},
			103: {CoreID: 56, SocketID: 0, NUMANodeID: 0},
			223: {CoreID: 56, SocketID: 0, NUMANodeID: 0},
			163: {CoreID: 56, SocketID: 0, NUMANodeID: 0},
			283: {CoreID: 56, SocketID: 0, NUMANodeID: 0},
			104: {CoreID: 57, SocketID: 0, NUMANodeID: 0},
			224: {CoreID: 57, SocketID: 0, NUMANodeID: 0},
			164: {CoreID: 57, SocketID: 0, NUMANodeID: 0},
			284: {CoreID: 57, SocketID: 0, NUMANodeID: 0},
			3:   {CoreID: 66, SocketID: 1, NUMANodeID: 1},
			27:  {CoreID: 66, SocketID: 1, NUMANodeID: 1},
			15:  {CoreID: 66, SocketID: 1, NUMANodeID: 1},
			40:  {CoreID: 66, SocketID: 1, NUMANodeID: 1},
			4:   {CoreID: 67, SocketID: 1, NUMANodeID: 1},
			28:  {CoreID: 67, SocketID: 1, NUMANodeID: 1},
			16:  {CoreID: 67, SocketID: 1, NUMANodeID: 1},
			41:  {CoreID: 67, SocketID: 1, NUMANodeID: 1},
			11:  {CoreID: 74, SocketID: 1, NUMANodeID: 1},
			35:  {CoreID: 74, SocketID: 1, NUMANodeID: 1},
			23:  {CoreID: 74, SocketID: 1, NUMANodeID: 1},
			48:  {CoreID: 74, SocketID: 1, NUMANodeID: 1},
			12:  {CoreID: 75, SocketID: 1, NUMANodeID: 1},
			36:  {CoreID: 75, SocketID: 1, NUMANodeID: 1},
			24:  {CoreID: 75, SocketID: 1, NUMANodeID: 1},
			49:  {CoreID: 75, SocketID: 1, NUMANodeID: 1},
			51:  {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			171: {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			111: {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			231: {CoreID: 2, SocketID: 1, NUMANodeID: 1},
			52:  {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			172: {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			112: {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			232: {CoreID: 3, SocketID: 1, NUMANodeID: 1},
			59:  {CoreID: 10, SocketID: 1, NUMANodeID: 1},
			179: {CoreID: 10, SocketID: 1, NUMANodeID: 1},
			119: {CoreID: 10, SocketID: 1, NUMANodeID: 1},
			239: {CoreID: 10, SocketID: 1, NUMANodeID: 1},
			60:  {CoreID: 11, SocketID: 1, NUMANodeID: 1},
			180: {CoreID: 11, SocketID: 1, NUMANodeID: 1},
			120: {CoreID: 11, SocketID: 1, NUMANodeID: 1},
			240: {CoreID: 11, SocketID: 1, NUMANodeID: 1},
			65:  {CoreID: 18, SocketID: 1, NUMANodeID: 1},
			185: {CoreID: 18, SocketID: 1, NUMANodeID: 1},
			125: {CoreID: 18, SocketID: 1, NUMANodeID: 1},
			245: {CoreID: 18, SocketID: 1, NUMANodeID: 1},
			66:  {CoreID: 19, SocketID: 1, NUMANodeID: 1},
			186: {CoreID: 19, SocketID: 1, NUMANodeID: 1},
			126: {CoreID: 19, SocketID: 1, NUMANodeID: 1},
			246: {CoreID: 19, SocketID: 1, NUMANodeID: 1},
			73:  {CoreID: 26, SocketID: 1, NUMANodeID: 1},
			193: {CoreID: 26, SocketID: 1, NUMANodeID: 1},
			133: {CoreID: 26, SocketID: 1, NUMANodeID: 1},
			253: {CoreID: 26, SocketID: 1, NUMANodeID: 1},
			74:  {CoreID: 27, SocketID: 1, NUMANodeID: 1},
			194: {CoreID: 27, SocketID: 1, NUMANodeID: 1},
			134: {CoreID: 27, SocketID: 1, NUMANodeID: 1},
			254: {CoreID: 27, SocketID: 1, NUMANodeID: 1},
			81:  {CoreID: 34, SocketID: 1, NUMANodeID: 1},
			201: {CoreID: 34, SocketID: 1, NUMANodeID: 1},
			141: {CoreID: 34, SocketID: 1, NUMANodeID: 1},
			261: {CoreID: 34, SocketID: 1, NUMANodeID: 1},
			82:  {CoreID: 35, SocketID: 1, NUMANodeID: 1},
			202: {CoreID: 35, SocketID: 1, NUMANodeID: 1},
			142: {CoreID: 35, SocketID: 1, NUMANodeID: 1},
			262: {CoreID: 35, SocketID: 1, NUMANodeID: 1},
			89:  {CoreID: 42, SocketID: 1, NUMANodeID: 1},
			209: {CoreID: 42, SocketID: 1, NUMANodeID: 1},
			149: {CoreID: 42, SocketID: 1, NUMANodeID: 1},
			269: {CoreID: 42, SocketID: 1, NUMANodeID: 1},
			90:  {CoreID: 43, SocketID: 1, NUMANodeID: 1},
			210: {CoreID: 43, SocketID: 1, NUMANodeID: 1},
			150: {CoreID: 43, SocketID: 1, NUMANodeID: 1},
			270: {CoreID: 43, SocketID: 1, NUMANodeID: 1},
			97:  {CoreID: 50, SocketID: 1, NUMANodeID: 1},
			217: {CoreID: 50, SocketID: 1, NUMANodeID: 1},
			157: {CoreID: 50, SocketID: 1, NUMANodeID: 1},
			277: {CoreID: 50, SocketID: 1, NUMANodeID: 1},
			98:  {CoreID: 51, SocketID: 1, NUMANodeID: 1},
			218: {CoreID: 51, SocketID: 1, NUMANodeID: 1},
			158: {CoreID: 51, SocketID: 1, NUMANodeID: 1},
			278: {CoreID: 51, SocketID: 1, NUMANodeID: 1},
			5:   {CoreID: 68, SocketID: 2, NUMANodeID: 2},
			29:  {CoreID: 68, SocketID: 2, NUMANodeID: 2},
			17:  {CoreID: 68, SocketID: 2, NUMANodeID: 2},
			42:  {CoreID: 68, SocketID: 2, NUMANodeID: 2},
			6:   {CoreID: 69, SocketID: 2, NUMANodeID: 2},
			30:  {CoreID: 69, SocketID: 2, NUMANodeID: 2},
			18:  {CoreID: 69, SocketID: 2, NUMANodeID: 2},
			43:  {CoreID: 69, SocketID: 2, NUMANodeID: 2},
			53:  {CoreID: 4, SocketID: 2, NUMANodeID: 2},
			173: {CoreID: 4, SocketID: 2, NUMANodeID: 2},
			113: {CoreID: 4, SocketID: 2, NUMANodeID: 2},
			233: {CoreID: 4, SocketID: 2, NUMANodeID: 2},
			54:  {CoreID: 5, SocketID: 2, NUMANodeID: 2},
			174: {CoreID: 5, SocketID: 2, NUMANodeID: 2},
			114: {CoreID: 5, SocketID: 2, NUMANodeID: 2},
			234: {CoreID: 5, SocketID: 2, NUMANodeID: 2},
			61:  {CoreID: 12, SocketID: 2, NUMANodeID: 2},
			181: {CoreID: 12, SocketID: 2, NUMANodeID: 2},
			121: {CoreID: 12, SocketID: 2, NUMANodeID: 2},
			241: {CoreID: 12, SocketID: 2, NUMANodeID: 2},
			62:  {CoreID: 13, SocketID: 2, NUMANodeID: 2},
			182: {CoreID: 13, SocketID: 2, NUMANodeID: 2},
			122: {CoreID: 13, SocketID: 2, NUMANodeID: 2},
			242: {CoreID: 13, SocketID: 2, NUMANodeID: 2},
			67:  {CoreID: 20, SocketID: 2, NUMANodeID: 2},
			187: {CoreID: 20, SocketID: 2, NUMANodeID: 2},
			127: {CoreID: 20, SocketID: 2, NUMANodeID: 2},
			247: {CoreID: 20, SocketID: 2, NUMANodeID: 2},
			68:  {CoreID: 21, SocketID: 2, NUMANodeID: 2},
			188: {CoreID: 21, SocketID: 2, NUMANodeID: 2},
			128: {CoreID: 21, SocketID: 2, NUMANodeID: 2},
			248: {CoreID: 21, SocketID: 2, NUMANodeID: 2},
			75:  {CoreID: 28, SocketID: 2, NUMANodeID: 2},
			195: {CoreID: 28, SocketID: 2, NUMANodeID: 2},
			135: {CoreID: 28, SocketID: 2, NUMANodeID: 2},
			255: {CoreID: 28, SocketID: 2, NUMANodeID: 2},
			76:  {CoreID: 29, SocketID: 2, NUMANodeID: 2},
			196: {CoreID: 29, SocketID: 2, NUMANodeID: 2},
			136: {CoreID: 29, SocketID: 2, NUMANodeID: 2},
			256: {CoreID: 29, SocketID: 2, NUMANodeID: 2},
			83:  {CoreID: 36, SocketID: 2, NUMANodeID: 2},
			203: {CoreID: 36, SocketID: 2, NUMANodeID: 2},
			143: {CoreID: 36, SocketID: 2, NUMANodeID: 2},
			263: {CoreID: 36, SocketID: 2, NUMANodeID: 2},
			84:  {CoreID: 37, SocketID: 2, NUMANodeID: 2},
			204: {CoreID: 37, SocketID: 2, NUMANodeID: 2},
			144: {CoreID: 37, SocketID: 2, NUMANodeID: 2},
			264: {CoreID: 37, SocketID: 2, NUMANodeID: 2},
			91:  {CoreID: 44, SocketID: 2, NUMANodeID: 2},
			211: {CoreID: 44, SocketID: 2, NUMANodeID: 2},
			151: {CoreID: 44, SocketID: 2, NUMANodeID: 2},
			271: {CoreID: 44, SocketID: 2, NUMANodeID: 2},
			92:  {CoreID: 45, SocketID: 2, NUMANodeID: 2},
			212: {CoreID: 45, SocketID: 2, NUMANodeID: 2},
			152: {CoreID: 45, SocketID: 2, NUMANodeID: 2},
			272: {CoreID: 45, SocketID: 2, NUMANodeID: 2},
			99:  {CoreID: 52, SocketID: 2, NUMANodeID: 2},
			219: {CoreID: 52, SocketID: 2, NUMANodeID: 2},
			159: {CoreID: 52, SocketID: 2, NUMANodeID: 2},
			279: {CoreID: 52, SocketID: 2, NUMANodeID: 2},
			100: {CoreID: 53, SocketID: 2, NUMANodeID: 2},
			220: {CoreID: 53, SocketID: 2, NUMANodeID: 2},
			160: {CoreID: 53, SocketID: 2, NUMANodeID: 2},
			280: {CoreID: 53, SocketID: 2, NUMANodeID: 2},
			105: {CoreID: 60, SocketID: 2, NUMANodeID: 2},
			225: {CoreID: 60, SocketID: 2, NUMANodeID: 2},
			165: {CoreID: 60, SocketID: 2, NUMANodeID: 2},
			285: {CoreID: 60, SocketID: 2, NUMANodeID: 2},
			106: {CoreID: 61, SocketID: 2, NUMANodeID: 2},
			226: {CoreID: 61, SocketID: 2, NUMANodeID: 2},
			166: {CoreID: 61, SocketID: 2, NUMANodeID: 2},
			286: {CoreID: 61, SocketID: 2, NUMANodeID: 2},
			7:   {CoreID: 70, SocketID: 3, NUMANodeID: 3},
			31:  {CoreID: 70, SocketID: 3, NUMANodeID: 3},
			19:  {CoreID: 70, SocketID: 3, NUMANodeID: 3},
			44:  {CoreID: 70, SocketID: 3, NUMANodeID: 3},
			8:   {CoreID: 71, SocketID: 3, NUMANodeID: 3},
			32:  {CoreID: 71, SocketID: 3, NUMANodeID: 3},
			20:  {CoreID: 71, SocketID: 3, NUMANodeID: 3},
			45:  {CoreID: 71, SocketID: 3, NUMANodeID: 3},
			37:  {CoreID: 63, SocketID: 3, NUMANodeID: 3},
			168: {CoreID: 63, SocketID: 3, NUMANodeID: 3},
			108: {CoreID: 63, SocketID: 3, NUMANodeID: 3},
			228: {CoreID: 63, SocketID: 3, NUMANodeID: 3},
			107: {CoreID: 62, SocketID: 3, NUMANodeID: 3},
			227: {CoreID: 62, SocketID: 3, NUMANodeID: 3},
			167: {CoreID: 62, SocketID: 3, NUMANodeID: 3},
			287: {CoreID: 62, SocketID: 3, NUMANodeID: 3},
			55:  {CoreID: 6, SocketID: 3, NUMANodeID: 3},
			175: {CoreID: 6, SocketID: 3, NUMANodeID: 3},
			115: {CoreID: 6, SocketID: 3, NUMANodeID: 3},
			235: {CoreID: 6, SocketID: 3, NUMANodeID: 3},
			56:  {CoreID: 7, SocketID: 3, NUMANodeID: 3},
			176: {CoreID: 7, SocketID: 3, NUMANodeID: 3},
			116: {CoreID: 7, SocketID: 3, NUMANodeID: 3},
			236: {CoreID: 7, SocketID: 3, NUMANodeID: 3},
			63:  {CoreID: 14, SocketID: 3, NUMANodeID: 3},
			183: {CoreID: 14, SocketID: 3, NUMANodeID: 3},
			123: {CoreID: 14, SocketID: 3, NUMANodeID: 3},
			243: {CoreID: 14, SocketID: 3, NUMANodeID: 3},
			64:  {CoreID: 15, SocketID: 3, NUMANodeID: 3},
			184: {CoreID: 15, SocketID: 3, NUMANodeID: 3},
			124: {CoreID: 15, SocketID: 3, NUMANodeID: 3},
			244: {CoreID: 15, SocketID: 3, NUMANodeID: 3},
			69:  {CoreID: 22, SocketID: 3, NUMANodeID: 3},
			189: {CoreID: 22, SocketID: 3, NUMANodeID: 3},
			129: {CoreID: 22, SocketID: 3, NUMANodeID: 3},
			249: {CoreID: 22, SocketID: 3, NUMANodeID: 3},
			70:  {CoreID: 23, SocketID: 3, NUMANodeID: 3},
			190: {CoreID: 23, SocketID: 3, NUMANodeID: 3},
			130: {CoreID: 23, SocketID: 3, NUMANodeID: 3},
			250: {CoreID: 23, SocketID: 3, NUMANodeID: 3},
			77:  {CoreID: 30, SocketID: 3, NUMANodeID: 3},
			197: {CoreID: 30, SocketID: 3, NUMANodeID: 3},
			137: {CoreID: 30, SocketID: 3, NUMANodeID: 3},
			257: {CoreID: 30, SocketID: 3, NUMANodeID: 3},
			78:  {CoreID: 31, SocketID: 3, NUMANodeID: 3},
			198: {CoreID: 31, SocketID: 3, NUMANodeID: 3},
			138: {CoreID: 31, SocketID: 3, NUMANodeID: 3},
			258: {CoreID: 31, SocketID: 3, NUMANodeID: 3},
			85:  {CoreID: 38, SocketID: 3, NUMANodeID: 3},
			205: {CoreID: 38, SocketID: 3, NUMANodeID: 3},
			145: {CoreID: 38, SocketID: 3, NUMANodeID: 3},
			265: {CoreID: 38, SocketID: 3, NUMANodeID: 3},
			86:  {CoreID: 39, SocketID: 3, NUMANodeID: 3},
			206: {CoreID: 39, SocketID: 3, NUMANodeID: 3},
			146: {CoreID: 39, SocketID: 3, NUMANodeID: 3},
			266: {CoreID: 39, SocketID: 3, NUMANodeID: 3},
			93:  {CoreID: 46, SocketID: 3, NUMANodeID: 3},
			213: {CoreID: 46, SocketID: 3, NUMANodeID: 3},
			153: {CoreID: 46, SocketID: 3, NUMANodeID: 3},
			273: {CoreID: 46, SocketID: 3, NUMANodeID: 3},
			94:  {CoreID: 47, SocketID: 3, NUMANodeID: 3},
			214: {CoreID: 47, SocketID: 3, NUMANodeID: 3},
			154: {CoreID: 47, SocketID: 3, NUMANodeID: 3},
			274: {CoreID: 47, SocketID: 3, NUMANodeID: 3},
			101: {CoreID: 54, SocketID: 3, NUMANodeID: 3},
			221: {CoreID: 54, SocketID: 3, NUMANodeID: 3},
			161: {CoreID: 54, SocketID: 3, NUMANodeID: 3},
			281: {CoreID: 54, SocketID: 3, NUMANodeID: 3},
			102: {CoreID: 55, SocketID: 3, NUMANodeID: 3},
			222: {CoreID: 55, SocketID: 3, NUMANodeID: 3},
			162: {CoreID: 55, SocketID: 3, NUMANodeID: 3},
			282: {CoreID: 55, SocketID: 3, NUMANodeID: 3},
		},
	}
	/*
		Topology from dual xeon gold 6230; lscpu excerpt
		CPU(s):              80
		On-line CPU(s) list: 0-79
		Thread(s) per core:  2
		Core(s) per socket:  20
		Socket(s):           2
		NUMA node(s):        4
		NUMA node0 CPU(s):   0-9,40-49
		NUMA node1 CPU(s):   10-19,50-59
		NUMA node2 CPU(s):   20-29,60-69
		NUMA node3 CPU(s):   30-39,70-79
	*/
	topoDualSocketMultiNumaPerSocketHT = &topology.CPUTopology{
		NumCPUs:      80,
		NumSockets:   2,
		NumCores:     40,
		NumNUMANodes: 4,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			8:  {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			9:  {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			10: {CoreID: 10, SocketID: 0, NUMANodeID: 1},
			11: {CoreID: 11, SocketID: 0, NUMANodeID: 1},
			12: {CoreID: 12, SocketID: 0, NUMANodeID: 1},
			13: {CoreID: 13, SocketID: 0, NUMANodeID: 1},
			14: {CoreID: 14, SocketID: 0, NUMANodeID: 1},
			15: {CoreID: 15, SocketID: 0, NUMANodeID: 1},
			16: {CoreID: 16, SocketID: 0, NUMANodeID: 1},
			17: {CoreID: 17, SocketID: 0, NUMANodeID: 1},
			18: {CoreID: 18, SocketID: 0, NUMANodeID: 1},
			19: {CoreID: 19, SocketID: 0, NUMANodeID: 1},
			20: {CoreID: 20, SocketID: 1, NUMANodeID: 2},
			21: {CoreID: 21, SocketID: 1, NUMANodeID: 2},
			22: {CoreID: 22, SocketID: 1, NUMANodeID: 2},
			23: {CoreID: 23, SocketID: 1, NUMANodeID: 2},
			24: {CoreID: 24, SocketID: 1, NUMANodeID: 2},
			25: {CoreID: 25, SocketID: 1, NUMANodeID: 2},
			26: {CoreID: 26, SocketID: 1, NUMANodeID: 2},
			27: {CoreID: 27, SocketID: 1, NUMANodeID: 2},
			28: {CoreID: 28, SocketID: 1, NUMANodeID: 2},
			29: {CoreID: 29, SocketID: 1, NUMANodeID: 2},
			30: {CoreID: 30, SocketID: 1, NUMANodeID: 3},
			31: {CoreID: 31, SocketID: 1, NUMANodeID: 3},
			32: {CoreID: 32, SocketID: 1, NUMANodeID: 3},
			33: {CoreID: 33, SocketID: 1, NUMANodeID: 3},
			34: {CoreID: 34, SocketID: 1, NUMANodeID: 3},
			35: {CoreID: 35, SocketID: 1, NUMANodeID: 3},
			36: {CoreID: 36, SocketID: 1, NUMANodeID: 3},
			37: {CoreID: 37, SocketID: 1, NUMANodeID: 3},
			38: {CoreID: 38, SocketID: 1, NUMANodeID: 3},
			39: {CoreID: 39, SocketID: 1, NUMANodeID: 3},
			40: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			41: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			42: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			43: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			44: {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			45: {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			46: {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			47: {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			48: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			49: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			50: {CoreID: 10, SocketID: 0, NUMANodeID: 1},
			51: {CoreID: 11, SocketID: 0, NUMANodeID: 1},
			52: {CoreID: 12, SocketID: 0, NUMANodeID: 1},
			53: {CoreID: 13, SocketID: 0, NUMANodeID: 1},
			54: {CoreID: 14, SocketID: 0, NUMANodeID: 1},
			55: {CoreID: 15, SocketID: 0, NUMANodeID: 1},
			56: {CoreID: 16, SocketID: 0, NUMANodeID: 1},
			57: {CoreID: 17, SocketID: 0, NUMANodeID: 1},
			58: {CoreID: 18, SocketID: 0, NUMANodeID: 1},
			59: {CoreID: 19, SocketID: 0, NUMANodeID: 1},
			60: {CoreID: 20, SocketID: 1, NUMANodeID: 2},
			61: {CoreID: 21, SocketID: 1, NUMANodeID: 2},
			62: {CoreID: 22, SocketID: 1, NUMANodeID: 2},
			63: {CoreID: 23, SocketID: 1, NUMANodeID: 2},
			64: {CoreID: 24, SocketID: 1, NUMANodeID: 2},
			65: {CoreID: 25, SocketID: 1, NUMANodeID: 2},
			66: {CoreID: 26, SocketID: 1, NUMANodeID: 2},
			67: {CoreID: 27, SocketID: 1, NUMANodeID: 2},
			68: {CoreID: 28, SocketID: 1, NUMANodeID: 2},
			69: {CoreID: 29, SocketID: 1, NUMANodeID: 2},
			70: {CoreID: 30, SocketID: 1, NUMANodeID: 3},
			71: {CoreID: 31, SocketID: 1, NUMANodeID: 3},
			72: {CoreID: 32, SocketID: 1, NUMANodeID: 3},
			73: {CoreID: 33, SocketID: 1, NUMANodeID: 3},
			74: {CoreID: 34, SocketID: 1, NUMANodeID: 3},
			75: {CoreID: 35, SocketID: 1, NUMANodeID: 3},
			76: {CoreID: 36, SocketID: 1, NUMANodeID: 3},
			77: {CoreID: 37, SocketID: 1, NUMANodeID: 3},
			78: {CoreID: 38, SocketID: 1, NUMANodeID: 3},
			79: {CoreID: 39, SocketID: 1, NUMANodeID: 3},
		},
	}
	/*
		FAKE Topology from dual xeon gold 6230
		(see: topoDualSocketMultiNumaPerSocketHT).
		We flip NUMA cells and Sockets to exercise the code.
		TODO(fromanirh): replace with a real-world topology
		once we find a suitable one.
	*/
	fakeTopoMultiSocketDualSocketPerNumaHT = &topology.CPUTopology{
		NumCPUs:      80,
		NumSockets:   4,
		NumCores:     40,
		NumNUMANodes: 2,
		CPUDetails: map[int]topology.CPUInfo{
			0:  {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1:  {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			2:  {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3:  {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			4:  {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			5:  {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			6:  {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			7:  {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			8:  {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			9:  {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			10: {CoreID: 10, SocketID: 1, NUMANodeID: 0},
			11: {CoreID: 11, SocketID: 1, NUMANodeID: 0},
			12: {CoreID: 12, SocketID: 1, NUMANodeID: 0},
			13: {CoreID: 13, SocketID: 1, NUMANodeID: 0},
			14: {CoreID: 14, SocketID: 1, NUMANodeID: 0},
			15: {CoreID: 15, SocketID: 1, NUMANodeID: 0},
			16: {CoreID: 16, SocketID: 1, NUMANodeID: 0},
			17: {CoreID: 17, SocketID: 1, NUMANodeID: 0},
			18: {CoreID: 18, SocketID: 1, NUMANodeID: 0},
			19: {CoreID: 19, SocketID: 1, NUMANodeID: 0},
			20: {CoreID: 20, SocketID: 2, NUMANodeID: 1},
			21: {CoreID: 21, SocketID: 2, NUMANodeID: 1},
			22: {CoreID: 22, SocketID: 2, NUMANodeID: 1},
			23: {CoreID: 23, SocketID: 2, NUMANodeID: 1},
			24: {CoreID: 24, SocketID: 2, NUMANodeID: 1},
			25: {CoreID: 25, SocketID: 2, NUMANodeID: 1},
			26: {CoreID: 26, SocketID: 2, NUMANodeID: 1},
			27: {CoreID: 27, SocketID: 2, NUMANodeID: 1},
			28: {CoreID: 28, SocketID: 2, NUMANodeID: 1},
			29: {CoreID: 29, SocketID: 2, NUMANodeID: 1},
			30: {CoreID: 30, SocketID: 3, NUMANodeID: 1},
			31: {CoreID: 31, SocketID: 3, NUMANodeID: 1},
			32: {CoreID: 32, SocketID: 3, NUMANodeID: 1},
			33: {CoreID: 33, SocketID: 3, NUMANodeID: 1},
			34: {CoreID: 34, SocketID: 3, NUMANodeID: 1},
			35: {CoreID: 35, SocketID: 3, NUMANodeID: 1},
			36: {CoreID: 36, SocketID: 3, NUMANodeID: 1},
			37: {CoreID: 37, SocketID: 3, NUMANodeID: 1},
			38: {CoreID: 38, SocketID: 3, NUMANodeID: 1},
			39: {CoreID: 39, SocketID: 3, NUMANodeID: 1},
			40: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			41: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			42: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			43: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			44: {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			45: {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			46: {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			47: {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			48: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			49: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			50: {CoreID: 10, SocketID: 1, NUMANodeID: 0},
			51: {CoreID: 11, SocketID: 1, NUMANodeID: 0},
			52: {CoreID: 12, SocketID: 1, NUMANodeID: 0},
			53: {CoreID: 13, SocketID: 1, NUMANodeID: 0},
			54: {CoreID: 14, SocketID: 1, NUMANodeID: 0},
			55: {CoreID: 15, SocketID: 1, NUMANodeID: 0},
			56: {CoreID: 16, SocketID: 1, NUMANodeID: 0},
			57: {CoreID: 17, SocketID: 1, NUMANodeID: 0},
			58: {CoreID: 18, SocketID: 1, NUMANodeID: 0},
			59: {CoreID: 19, SocketID: 1, NUMANodeID: 0},
			60: {CoreID: 20, SocketID: 2, NUMANodeID: 1},
			61: {CoreID: 21, SocketID: 2, NUMANodeID: 1},
			62: {CoreID: 22, SocketID: 2, NUMANodeID: 1},
			63: {CoreID: 23, SocketID: 2, NUMANodeID: 1},
			64: {CoreID: 24, SocketID: 2, NUMANodeID: 1},
			65: {CoreID: 25, SocketID: 2, NUMANodeID: 1},
			66: {CoreID: 26, SocketID: 2, NUMANodeID: 1},
			67: {CoreID: 27, SocketID: 2, NUMANodeID: 1},
			68: {CoreID: 28, SocketID: 2, NUMANodeID: 1},
			69: {CoreID: 29, SocketID: 2, NUMANodeID: 1},
			70: {CoreID: 30, SocketID: 3, NUMANodeID: 1},
			71: {CoreID: 31, SocketID: 3, NUMANodeID: 1},
			72: {CoreID: 32, SocketID: 3, NUMANodeID: 1},
			73: {CoreID: 33, SocketID: 3, NUMANodeID: 1},
			74: {CoreID: 34, SocketID: 3, NUMANodeID: 1},
			75: {CoreID: 35, SocketID: 3, NUMANodeID: 1},
			76: {CoreID: 36, SocketID: 3, NUMANodeID: 1},
			77: {CoreID: 37, SocketID: 3, NUMANodeID: 1},
			78: {CoreID: 38, SocketID: 3, NUMANodeID: 1},
			79: {CoreID: 39, SocketID: 3, NUMANodeID: 1},
		},
	}

	/*
		Topology from dual AMD EPYC 7742 64-Core Processor; lscpu excerpt
		CPU(s):              256
		On-line CPU(s) list: 0-255
		Thread(s) per core:  2
		Core(s) per socket:  64
		Socket(s):           2
		NUMA node(s):        8 (NPS=4)
		NUMA node0 CPU(s):   0-15,128-143
		NUMA node1 CPU(s):   16-31,144-159
		NUMA node2 CPU(s):   32-47,160-175
		NUMA node3 CPU(s):   48-63,176-191
		NUMA node4 CPU(s):   64-79,192-207
		NUMA node5 CPU(s):   80-95,208-223
		NUMA node6 CPU(s):   96-111,224-239
		NUMA node7 CPU(s):   112-127,240-255
	*/
	topoDualSocketMultiNumaPerSocketHTLarge = &topology.CPUTopology{
		NumCPUs:      256,
		NumSockets:   2,
		NumCores:     128,
		NumNUMANodes: 8,
		CPUDetails: map[int]topology.CPUInfo{
			0:   {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			1:   {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			2:   {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			3:   {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			4:   {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			5:   {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			6:   {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			7:   {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			8:   {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			9:   {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			10:  {CoreID: 10, SocketID: 0, NUMANodeID: 0},
			11:  {CoreID: 11, SocketID: 0, NUMANodeID: 0},
			12:  {CoreID: 12, SocketID: 0, NUMANodeID: 0},
			13:  {CoreID: 13, SocketID: 0, NUMANodeID: 0},
			14:  {CoreID: 14, SocketID: 0, NUMANodeID: 0},
			15:  {CoreID: 15, SocketID: 0, NUMANodeID: 0},
			16:  {CoreID: 16, SocketID: 0, NUMANodeID: 1},
			17:  {CoreID: 17, SocketID: 0, NUMANodeID: 1},
			18:  {CoreID: 18, SocketID: 0, NUMANodeID: 1},
			19:  {CoreID: 19, SocketID: 0, NUMANodeID: 1},
			20:  {CoreID: 20, SocketID: 0, NUMANodeID: 1},
			21:  {CoreID: 21, SocketID: 0, NUMANodeID: 1},
			22:  {CoreID: 22, SocketID: 0, NUMANodeID: 1},
			23:  {CoreID: 23, SocketID: 0, NUMANodeID: 1},
			24:  {CoreID: 24, SocketID: 0, NUMANodeID: 1},
			25:  {CoreID: 25, SocketID: 0, NUMANodeID: 1},
			26:  {CoreID: 26, SocketID: 0, NUMANodeID: 1},
			27:  {CoreID: 27, SocketID: 0, NUMANodeID: 1},
			28:  {CoreID: 28, SocketID: 0, NUMANodeID: 1},
			29:  {CoreID: 29, SocketID: 0, NUMANodeID: 1},
			30:  {CoreID: 30, SocketID: 0, NUMANodeID: 1},
			31:  {CoreID: 31, SocketID: 0, NUMANodeID: 1},
			32:  {CoreID: 32, SocketID: 0, NUMANodeID: 2},
			33:  {CoreID: 33, SocketID: 0, NUMANodeID: 2},
			34:  {CoreID: 34, SocketID: 0, NUMANodeID: 2},
			35:  {CoreID: 35, SocketID: 0, NUMANodeID: 2},
			36:  {CoreID: 36, SocketID: 0, NUMANodeID: 2},
			37:  {CoreID: 37, SocketID: 0, NUMANodeID: 2},
			38:  {CoreID: 38, SocketID: 0, NUMANodeID: 2},
			39:  {CoreID: 39, SocketID: 0, NUMANodeID: 2},
			40:  {CoreID: 40, SocketID: 0, NUMANodeID: 2},
			41:  {CoreID: 41, SocketID: 0, NUMANodeID: 2},
			42:  {CoreID: 42, SocketID: 0, NUMANodeID: 2},
			43:  {CoreID: 43, SocketID: 0, NUMANodeID: 2},
			44:  {CoreID: 44, SocketID: 0, NUMANodeID: 2},
			45:  {CoreID: 45, SocketID: 0, NUMANodeID: 2},
			46:  {CoreID: 46, SocketID: 0, NUMANodeID: 2},
			47:  {CoreID: 47, SocketID: 0, NUMANodeID: 2},
			48:  {CoreID: 48, SocketID: 0, NUMANodeID: 3},
			49:  {CoreID: 49, SocketID: 0, NUMANodeID: 3},
			50:  {CoreID: 50, SocketID: 0, NUMANodeID: 3},
			51:  {CoreID: 51, SocketID: 0, NUMANodeID: 3},
			52:  {CoreID: 52, SocketID: 0, NUMANodeID: 3},
			53:  {CoreID: 53, SocketID: 0, NUMANodeID: 3},
			54:  {CoreID: 54, SocketID: 0, NUMANodeID: 3},
			55:  {CoreID: 55, SocketID: 0, NUMANodeID: 3},
			56:  {CoreID: 56, SocketID: 0, NUMANodeID: 3},
			57:  {CoreID: 57, SocketID: 0, NUMANodeID: 3},
			58:  {CoreID: 58, SocketID: 0, NUMANodeID: 3},
			59:  {CoreID: 59, SocketID: 0, NUMANodeID: 3},
			60:  {CoreID: 60, SocketID: 0, NUMANodeID: 3},
			61:  {CoreID: 61, SocketID: 0, NUMANodeID: 3},
			62:  {CoreID: 62, SocketID: 0, NUMANodeID: 3},
			63:  {CoreID: 63, SocketID: 0, NUMANodeID: 3},
			64:  {CoreID: 64, SocketID: 1, NUMANodeID: 4},
			65:  {CoreID: 65, SocketID: 1, NUMANodeID: 4},
			66:  {CoreID: 66, SocketID: 1, NUMANodeID: 4},
			67:  {CoreID: 67, SocketID: 1, NUMANodeID: 4},
			68:  {CoreID: 68, SocketID: 1, NUMANodeID: 4},
			69:  {CoreID: 69, SocketID: 1, NUMANodeID: 4},
			70:  {CoreID: 70, SocketID: 1, NUMANodeID: 4},
			71:  {CoreID: 71, SocketID: 1, NUMANodeID: 4},
			72:  {CoreID: 72, SocketID: 1, NUMANodeID: 4},
			73:  {CoreID: 73, SocketID: 1, NUMANodeID: 4},
			74:  {CoreID: 74, SocketID: 1, NUMANodeID: 4},
			75:  {CoreID: 75, SocketID: 1, NUMANodeID: 4},
			76:  {CoreID: 76, SocketID: 1, NUMANodeID: 4},
			77:  {CoreID: 77, SocketID: 1, NUMANodeID: 4},
			78:  {CoreID: 78, SocketID: 1, NUMANodeID: 4},
			79:  {CoreID: 79, SocketID: 1, NUMANodeID: 4},
			80:  {CoreID: 80, SocketID: 1, NUMANodeID: 5},
			81:  {CoreID: 81, SocketID: 1, NUMANodeID: 5},
			82:  {CoreID: 82, SocketID: 1, NUMANodeID: 5},
			83:  {CoreID: 83, SocketID: 1, NUMANodeID: 5},
			84:  {CoreID: 84, SocketID: 1, NUMANodeID: 5},
			85:  {CoreID: 85, SocketID: 1, NUMANodeID: 5},
			86:  {CoreID: 86, SocketID: 1, NUMANodeID: 5},
			87:  {CoreID: 87, SocketID: 1, NUMANodeID: 5},
			88:  {CoreID: 88, SocketID: 1, NUMANodeID: 5},
			89:  {CoreID: 89, SocketID: 1, NUMANodeID: 5},
			90:  {CoreID: 90, SocketID: 1, NUMANodeID: 5},
			91:  {CoreID: 91, SocketID: 1, NUMANodeID: 5},
			92:  {CoreID: 92, SocketID: 1, NUMANodeID: 5},
			93:  {CoreID: 93, SocketID: 1, NUMANodeID: 5},
			94:  {CoreID: 94, SocketID: 1, NUMANodeID: 5},
			95:  {CoreID: 95, SocketID: 1, NUMANodeID: 5},
			96:  {CoreID: 96, SocketID: 1, NUMANodeID: 6},
			97:  {CoreID: 97, SocketID: 1, NUMANodeID: 6},
			98:  {CoreID: 98, SocketID: 1, NUMANodeID: 6},
			99:  {CoreID: 99, SocketID: 1, NUMANodeID: 6},
			100: {CoreID: 100, SocketID: 1, NUMANodeID: 6},
			101: {CoreID: 101, SocketID: 1, NUMANodeID: 6},
			102: {CoreID: 102, SocketID: 1, NUMANodeID: 6},
			103: {CoreID: 103, SocketID: 1, NUMANodeID: 6},
			104: {CoreID: 104, SocketID: 1, NUMANodeID: 6},
			105: {CoreID: 105, SocketID: 1, NUMANodeID: 6},
			106: {CoreID: 106, SocketID: 1, NUMANodeID: 6},
			107: {CoreID: 107, SocketID: 1, NUMANodeID: 6},
			108: {CoreID: 108, SocketID: 1, NUMANodeID: 6},
			109: {CoreID: 109, SocketID: 1, NUMANodeID: 6},
			110: {CoreID: 110, SocketID: 1, NUMANodeID: 6},
			111: {CoreID: 111, SocketID: 1, NUMANodeID: 6},
			112: {CoreID: 112, SocketID: 1, NUMANodeID: 7},
			113: {CoreID: 113, SocketID: 1, NUMANodeID: 7},
			114: {CoreID: 114, SocketID: 1, NUMANodeID: 7},
			115: {CoreID: 115, SocketID: 1, NUMANodeID: 7},
			116: {CoreID: 116, SocketID: 1, NUMANodeID: 7},
			117: {CoreID: 117, SocketID: 1, NUMANodeID: 7},
			118: {CoreID: 118, SocketID: 1, NUMANodeID: 7},
			119: {CoreID: 119, SocketID: 1, NUMANodeID: 7},
			120: {CoreID: 120, SocketID: 1, NUMANodeID: 7},
			121: {CoreID: 121, SocketID: 1, NUMANodeID: 7},
			122: {CoreID: 122, SocketID: 1, NUMANodeID: 7},
			123: {CoreID: 123, SocketID: 1, NUMANodeID: 7},
			124: {CoreID: 124, SocketID: 1, NUMANodeID: 7},
			125: {CoreID: 125, SocketID: 1, NUMANodeID: 7},
			126: {CoreID: 126, SocketID: 1, NUMANodeID: 7},
			127: {CoreID: 127, SocketID: 1, NUMANodeID: 7},
			128: {CoreID: 0, SocketID: 0, NUMANodeID: 0},
			129: {CoreID: 1, SocketID: 0, NUMANodeID: 0},
			130: {CoreID: 2, SocketID: 0, NUMANodeID: 0},
			131: {CoreID: 3, SocketID: 0, NUMANodeID: 0},
			132: {CoreID: 4, SocketID: 0, NUMANodeID: 0},
			133: {CoreID: 5, SocketID: 0, NUMANodeID: 0},
			134: {CoreID: 6, SocketID: 0, NUMANodeID: 0},
			135: {CoreID: 7, SocketID: 0, NUMANodeID: 0},
			136: {CoreID: 8, SocketID: 0, NUMANodeID: 0},
			137: {CoreID: 9, SocketID: 0, NUMANodeID: 0},
			138: {CoreID: 10, SocketID: 0, NUMANodeID: 0},
			139: {CoreID: 11, SocketID: 0, NUMANodeID: 0},
			140: {CoreID: 12, SocketID: 0, NUMANodeID: 0},
			141: {CoreID: 13, SocketID: 0, NUMANodeID: 0},
			142: {CoreID: 14, SocketID: 0, NUMANodeID: 0},
			143: {CoreID: 15, SocketID: 0, NUMANodeID: 0},
			144: {CoreID: 16, SocketID: 0, NUMANodeID: 1},
			145: {CoreID: 17, SocketID: 0, NUMANodeID: 1},
			146: {CoreID: 18, SocketID: 0, NUMANodeID: 1},
			147: {CoreID: 19, SocketID: 0, NUMANodeID: 1},
			148: {CoreID: 20, SocketID: 0, NUMANodeID: 1},
			149: {CoreID: 21, SocketID: 0, NUMANodeID: 1},
			150: {CoreID: 22, SocketID: 0, NUMANodeID: 1},
			151: {CoreID: 23, SocketID: 0, NUMANodeID: 1},
			152: {CoreID: 24, SocketID: 0, NUMANodeID: 1},
			153: {CoreID: 25, SocketID: 0, NUMANodeID: 1},
			154: {CoreID: 26, SocketID: 0, NUMANodeID: 1},
			155: {CoreID: 27, SocketID: 0, NUMANodeID: 1},
			156: {CoreID: 28, SocketID: 0, NUMANodeID: 1},
			157: {CoreID: 29, SocketID: 0, NUMANodeID: 1},
			158: {CoreID: 30, SocketID: 0, NUMANodeID: 1},
			159: {CoreID: 31, SocketID: 0, NUMANodeID: 1},
			160: {CoreID: 32, SocketID: 0, NUMANodeID: 2},
			161: {CoreID: 33, SocketID: 0, NUMANodeID: 2},
			162: {CoreID: 34, SocketID: 0, NUMANodeID: 2},
			163: {CoreID: 35, SocketID: 0, NUMANodeID: 2},
			164: {CoreID: 36, SocketID: 0, NUMANodeID: 2},
			165: {CoreID: 37, SocketID: 0, NUMANodeID: 2},
			166: {CoreID: 38, SocketID: 0, NUMANodeID: 2},
			167: {CoreID: 39, SocketID: 0, NUMANodeID: 2},
			168: {CoreID: 40, SocketID: 0, NUMANodeID: 2},
			169: {CoreID: 41, SocketID: 0, NUMANodeID: 2},
			170: {CoreID: 42, SocketID: 0, NUMANodeID: 2},
			171: {CoreID: 43, SocketID: 0, NUMANodeID: 2},
			172: {CoreID: 44, SocketID: 0, NUMANodeID: 2},
			173: {CoreID: 45, SocketID: 0, NUMANodeID: 2},
			174: {CoreID: 46, SocketID: 0, NUMANodeID: 2},
			175: {CoreID: 47, SocketID: 0, NUMANodeID: 2},
			176: {CoreID: 48, SocketID: 0, NUMANodeID: 3},
			177: {CoreID: 49, SocketID: 0, NUMANodeID: 3},
			178: {CoreID: 50, SocketID: 0, NUMANodeID: 3},
			179: {CoreID: 51, SocketID: 0, NUMANodeID: 3},
			180: {CoreID: 52, SocketID: 0, NUMANodeID: 3},
			181: {CoreID: 53, SocketID: 0, NUMANodeID: 3},
			182: {CoreID: 54, SocketID: 0, NUMANodeID: 3},
			183: {CoreID: 55, SocketID: 0, NUMANodeID: 3},
			184: {CoreID: 56, SocketID: 0, NUMANodeID: 3},
			185: {CoreID: 57, SocketID: 0, NUMANodeID: 3},
			186: {CoreID: 58, SocketID: 0, NUMANodeID: 3},
			187: {CoreID: 59, SocketID: 0, NUMANodeID: 3},
			188: {CoreID: 60, SocketID: 0, NUMANodeID: 3},
			189: {CoreID: 61, SocketID: 0, NUMANodeID: 3},
			190: {CoreID: 62, SocketID: 0, NUMANodeID: 3},
			191: {CoreID: 63, SocketID: 0, NUMANodeID: 3},
			192: {CoreID: 64, SocketID: 1, NUMANodeID: 4},
			193: {CoreID: 65, SocketID: 1, NUMANodeID: 4},
			194: {CoreID: 66, SocketID: 1, NUMANodeID: 4},
			195: {CoreID: 67, SocketID: 1, NUMANodeID: 4},
			196: {CoreID: 68, SocketID: 1, NUMANodeID: 4},
			197: {CoreID: 69, SocketID: 1, NUMANodeID: 4},
			198: {CoreID: 70, SocketID: 1, NUMANodeID: 4},
			199: {CoreID: 71, SocketID: 1, NUMANodeID: 4},
			200: {CoreID: 72, SocketID: 1, NUMANodeID: 4},
			201: {CoreID: 73, SocketID: 1, NUMANodeID: 4},
			202: {CoreID: 74, SocketID: 1, NUMANodeID: 4},
			203: {CoreID: 75, SocketID: 1, NUMANodeID: 4},
			204: {CoreID: 76, SocketID: 1, NUMANodeID: 4},
			205: {CoreID: 77, SocketID: 1, NUMANodeID: 4},
			206: {CoreID: 78, SocketID: 1, NUMANodeID: 4},
			207: {CoreID: 79, SocketID: 1, NUMANodeID: 4},
			208: {CoreID: 80, SocketID: 1, NUMANodeID: 5},
			209: {CoreID: 81, SocketID: 1, NUMANodeID: 5},
			210: {CoreID: 82, SocketID: 1, NUMANodeID: 5},
			211: {CoreID: 83, SocketID: 1, NUMANodeID: 5},
			212: {CoreID: 84, SocketID: 1, NUMANodeID: 5},
			213: {CoreID: 85, SocketID: 1, NUMANodeID: 5},
			214: {CoreID: 86, SocketID: 1, NUMANodeID: 5},
			215: {CoreID: 87, SocketID: 1, NUMANodeID: 5},
			216: {CoreID: 88, SocketID: 1, NUMANodeID: 5},
			217: {CoreID: 89, SocketID: 1, NUMANodeID: 5},
			218: {CoreID: 90, SocketID: 1, NUMANodeID: 5},
			219: {CoreID: 91, SocketID: 1, NUMANodeID: 5},
			220: {CoreID: 92, SocketID: 1, NUMANodeID: 5},
			221: {CoreID: 93, SocketID: 1, NUMANodeID: 5},
			222: {CoreID: 94, SocketID: 1, NUMANodeID: 5},
			223: {CoreID: 95, SocketID: 1, NUMANodeID: 5},
			224: {CoreID: 96, SocketID: 1, NUMANodeID: 6},
			225: {CoreID: 97, SocketID: 1, NUMANodeID: 6},
			226: {CoreID: 98, SocketID: 1, NUMANodeID: 6},
			227: {CoreID: 99, SocketID: 1, NUMANodeID: 6},
			228: {CoreID: 100, SocketID: 1, NUMANodeID: 6},
			229: {CoreID: 101, SocketID: 1, NUMANodeID: 6},
			230: {CoreID: 102, SocketID: 1, NUMANodeID: 6},
			231: {CoreID: 103, SocketID: 1, NUMANodeID: 6},
			232: {CoreID: 104, SocketID: 1, NUMANodeID: 6},
			233: {CoreID: 105, SocketID: 1, NUMANodeID: 6},
			234: {CoreID: 106, SocketID: 1, NUMANodeID: 6},
			235: {CoreID: 107, SocketID: 1, NUMANodeID: 6},
			236: {CoreID: 108, SocketID: 1, NUMANodeID: 6},
			237: {CoreID: 109, SocketID: 1, NUMANodeID: 6},
			238: {CoreID: 110, SocketID: 1, NUMANodeID: 6},
			239: {CoreID: 111, SocketID: 1, NUMANodeID: 6},
			240: {CoreID: 112, SocketID: 1, NUMANodeID: 7},
			241: {CoreID: 113, SocketID: 1, NUMANodeID: 7},
			242: {CoreID: 114, SocketID: 1, NUMANodeID: 7},
			243: {CoreID: 115, SocketID: 1, NUMANodeID: 7},
			244: {CoreID: 116, SocketID: 1, NUMANodeID: 7},
			245: {CoreID: 117, SocketID: 1, NUMANodeID: 7},
			246: {CoreID: 118, SocketID: 1, NUMANodeID: 7},
			247: {CoreID: 119, SocketID: 1, NUMANodeID: 7},
			248: {CoreID: 120, SocketID: 1, NUMANodeID: 7},
			249: {CoreID: 121, SocketID: 1, NUMANodeID: 7},
			250: {CoreID: 122, SocketID: 1, NUMANodeID: 7},
			251: {CoreID: 123, SocketID: 1, NUMANodeID: 7},
			252: {CoreID: 124, SocketID: 1, NUMANodeID: 7},
			253: {CoreID: 125, SocketID: 1, NUMANodeID: 7},
			254: {CoreID: 126, SocketID: 1, NUMANodeID: 7},
			255: {CoreID: 127, SocketID: 1, NUMANodeID: 7},
		},
	}
	/*
		Topology from dual AMD EPYC 9654 96-Core Processor; lscpu excerpt
		CPU(s):                384
		On-line CPU(s) list:   0-383
		Thread(s) per core:    2
		Core(s) per socket:    96
		Socket(s):             2
		NUMA node(s):          2
		NUMA node0 CPU(s):     0-95,192-287
		NUMA node1 CPU(s):     96-191,288-383
	*/
	topoDualSocketSingleNumaPerSocketSMTUncore = &topology.CPUTopology{
		NumCPUs:        384,
		NumCores:       192,
		NumUncoreCache: 24,
		NumSockets:     2,
		NumNUMANodes:   2,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			32:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			33:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			34:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			35:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			36:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			37:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			38:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			39:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			40:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			41:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			42:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			43:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			44:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			45:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			46:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			47:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			48:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			49:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			50:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			51:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			52:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			53:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			54:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			55:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			56:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			57:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			58:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			59:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			60:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			61:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			62:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			63:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			64:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			65:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			66:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			67:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			68:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			69:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			70:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			71:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			72:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			73:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			74:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			75:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			76:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			77:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			78:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			79:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			80:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			81:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			82:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			83:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			84:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			85:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			86:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			87:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			88:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			89:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			90:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			91:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			92:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			93:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			94:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			95:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			96:  topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 96, UncoreCacheID: 12},
			97:  topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 97, UncoreCacheID: 12},
			98:  topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 98, UncoreCacheID: 12},
			99:  topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 99, UncoreCacheID: 12},
			100: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 100, UncoreCacheID: 12},
			101: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 101, UncoreCacheID: 12},
			102: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 102, UncoreCacheID: 12},
			103: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 103, UncoreCacheID: 12},
			104: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 104, UncoreCacheID: 13},
			105: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 105, UncoreCacheID: 13},
			106: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 106, UncoreCacheID: 13},
			107: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 107, UncoreCacheID: 13},
			108: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 108, UncoreCacheID: 13},
			109: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 109, UncoreCacheID: 13},
			110: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 110, UncoreCacheID: 13},
			111: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 111, UncoreCacheID: 13},
			112: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 112, UncoreCacheID: 14},
			113: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 113, UncoreCacheID: 14},
			114: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 114, UncoreCacheID: 14},
			115: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 115, UncoreCacheID: 14},
			116: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 116, UncoreCacheID: 14},
			117: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 117, UncoreCacheID: 14},
			118: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 118, UncoreCacheID: 14},
			119: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 119, UncoreCacheID: 14},
			120: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 120, UncoreCacheID: 15},
			121: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 121, UncoreCacheID: 15},
			122: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 122, UncoreCacheID: 15},
			123: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 123, UncoreCacheID: 15},
			124: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 124, UncoreCacheID: 15},
			125: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 125, UncoreCacheID: 15},
			126: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 126, UncoreCacheID: 15},
			127: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 127, UncoreCacheID: 15},
			128: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 128, UncoreCacheID: 16},
			129: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 129, UncoreCacheID: 16},
			130: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 130, UncoreCacheID: 16},
			131: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 131, UncoreCacheID: 16},
			132: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 132, UncoreCacheID: 16},
			133: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 133, UncoreCacheID: 16},
			134: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 134, UncoreCacheID: 16},
			135: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 135, UncoreCacheID: 16},
			136: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 136, UncoreCacheID: 17},
			137: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 137, UncoreCacheID: 17},
			138: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 138, UncoreCacheID: 17},
			139: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 139, UncoreCacheID: 17},
			140: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 140, UncoreCacheID: 17},
			141: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 141, UncoreCacheID: 17},
			142: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 142, UncoreCacheID: 17},
			143: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 143, UncoreCacheID: 17},
			144: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 144, UncoreCacheID: 18},
			145: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 145, UncoreCacheID: 18},
			146: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 146, UncoreCacheID: 18},
			147: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 147, UncoreCacheID: 18},
			148: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 148, UncoreCacheID: 18},
			149: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 149, UncoreCacheID: 18},
			150: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 150, UncoreCacheID: 18},
			151: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 151, UncoreCacheID: 18},
			152: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 152, UncoreCacheID: 19},
			153: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 153, UncoreCacheID: 19},
			154: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 154, UncoreCacheID: 19},
			155: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 155, UncoreCacheID: 19},
			156: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 156, UncoreCacheID: 19},
			157: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 157, UncoreCacheID: 19},
			158: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 158, UncoreCacheID: 19},
			159: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 159, UncoreCacheID: 19},
			160: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 160, UncoreCacheID: 20},
			161: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 161, UncoreCacheID: 20},
			162: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 162, UncoreCacheID: 20},
			163: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 163, UncoreCacheID: 20},
			164: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 164, UncoreCacheID: 20},
			165: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 165, UncoreCacheID: 20},
			166: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 166, UncoreCacheID: 20},
			167: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 167, UncoreCacheID: 20},
			168: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 168, UncoreCacheID: 21},
			169: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 169, UncoreCacheID: 21},
			170: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 170, UncoreCacheID: 21},
			171: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 171, UncoreCacheID: 21},
			172: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 172, UncoreCacheID: 21},
			173: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 173, UncoreCacheID: 21},
			174: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 174, UncoreCacheID: 21},
			175: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 175, UncoreCacheID: 21},
			176: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 176, UncoreCacheID: 22},
			177: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 177, UncoreCacheID: 22},
			178: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 178, UncoreCacheID: 22},
			179: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 179, UncoreCacheID: 22},
			180: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 180, UncoreCacheID: 22},
			181: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 181, UncoreCacheID: 22},
			182: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 182, UncoreCacheID: 22},
			183: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 183, UncoreCacheID: 22},
			184: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 184, UncoreCacheID: 23},
			185: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 185, UncoreCacheID: 23},
			186: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 186, UncoreCacheID: 23},
			187: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 187, UncoreCacheID: 23},
			188: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 188, UncoreCacheID: 23},
			189: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 189, UncoreCacheID: 23},
			190: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 190, UncoreCacheID: 23},
			191: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 191, UncoreCacheID: 23},
			192: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			193: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			194: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			195: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			196: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			197: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			198: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			199: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			200: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			201: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			202: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			203: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			204: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			205: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			206: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			207: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			208: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			209: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			210: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			211: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			212: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			213: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			214: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			215: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			216: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			217: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			218: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			219: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			220: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			221: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			222: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			223: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			224: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			225: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			226: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			227: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			228: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			229: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			230: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			231: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			232: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			233: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			234: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			235: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			236: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			237: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			238: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			239: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			240: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			241: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			242: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			243: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			244: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			245: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			246: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			247: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			248: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			249: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			250: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			251: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			252: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			253: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			254: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			255: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			256: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			257: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			258: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			259: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			260: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			261: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			262: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			263: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			264: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			265: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			266: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			267: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			268: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			269: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			270: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			271: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			272: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			273: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			274: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			275: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			276: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			277: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			278: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			279: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			280: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			281: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			282: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			283: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			284: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			285: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			286: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			287: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			288: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 96, UncoreCacheID: 12},
			289: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 97, UncoreCacheID: 12},
			290: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 98, UncoreCacheID: 12},
			291: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 99, UncoreCacheID: 12},
			292: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 100, UncoreCacheID: 12},
			293: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 101, UncoreCacheID: 12},
			294: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 102, UncoreCacheID: 12},
			295: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 103, UncoreCacheID: 12},
			296: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 104, UncoreCacheID: 13},
			297: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 105, UncoreCacheID: 13},
			298: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 106, UncoreCacheID: 13},
			299: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 107, UncoreCacheID: 13},
			300: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 108, UncoreCacheID: 13},
			301: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 109, UncoreCacheID: 13},
			302: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 110, UncoreCacheID: 13},
			303: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 111, UncoreCacheID: 13},
			304: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 112, UncoreCacheID: 14},
			305: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 113, UncoreCacheID: 14},
			306: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 114, UncoreCacheID: 14},
			307: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 115, UncoreCacheID: 14},
			308: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 116, UncoreCacheID: 14},
			309: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 117, UncoreCacheID: 14},
			310: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 118, UncoreCacheID: 14},
			311: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 119, UncoreCacheID: 14},
			312: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 120, UncoreCacheID: 15},
			313: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 121, UncoreCacheID: 15},
			314: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 122, UncoreCacheID: 15},
			315: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 123, UncoreCacheID: 15},
			316: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 124, UncoreCacheID: 15},
			317: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 125, UncoreCacheID: 15},
			318: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 126, UncoreCacheID: 15},
			319: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 127, UncoreCacheID: 15},
			320: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 128, UncoreCacheID: 16},
			321: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 129, UncoreCacheID: 16},
			322: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 130, UncoreCacheID: 16},
			323: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 131, UncoreCacheID: 16},
			324: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 132, UncoreCacheID: 16},
			325: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 133, UncoreCacheID: 16},
			326: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 134, UncoreCacheID: 16},
			327: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 135, UncoreCacheID: 16},
			328: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 136, UncoreCacheID: 17},
			329: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 137, UncoreCacheID: 17},
			330: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 138, UncoreCacheID: 17},
			331: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 139, UncoreCacheID: 17},
			332: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 140, UncoreCacheID: 17},
			333: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 141, UncoreCacheID: 17},
			334: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 142, UncoreCacheID: 17},
			335: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 143, UncoreCacheID: 17},
			336: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 144, UncoreCacheID: 18},
			337: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 145, UncoreCacheID: 18},
			338: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 146, UncoreCacheID: 18},
			339: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 147, UncoreCacheID: 18},
			340: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 148, UncoreCacheID: 18},
			341: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 149, UncoreCacheID: 18},
			342: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 150, UncoreCacheID: 18},
			343: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 151, UncoreCacheID: 18},
			344: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 152, UncoreCacheID: 19},
			345: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 153, UncoreCacheID: 19},
			346: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 154, UncoreCacheID: 19},
			347: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 155, UncoreCacheID: 19},
			348: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 156, UncoreCacheID: 19},
			349: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 157, UncoreCacheID: 19},
			350: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 158, UncoreCacheID: 19},
			351: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 159, UncoreCacheID: 19},
			352: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 160, UncoreCacheID: 20},
			353: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 161, UncoreCacheID: 20},
			354: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 162, UncoreCacheID: 20},
			355: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 163, UncoreCacheID: 20},
			356: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 164, UncoreCacheID: 20},
			357: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 165, UncoreCacheID: 20},
			358: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 166, UncoreCacheID: 20},
			359: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 167, UncoreCacheID: 20},
			360: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 168, UncoreCacheID: 21},
			361: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 169, UncoreCacheID: 21},
			362: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 170, UncoreCacheID: 21},
			363: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 171, UncoreCacheID: 21},
			364: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 172, UncoreCacheID: 21},
			365: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 173, UncoreCacheID: 21},
			366: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 174, UncoreCacheID: 21},
			367: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 175, UncoreCacheID: 21},
			368: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 176, UncoreCacheID: 22},
			369: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 177, UncoreCacheID: 22},
			370: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 178, UncoreCacheID: 22},
			371: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 179, UncoreCacheID: 22},
			372: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 180, UncoreCacheID: 22},
			373: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 181, UncoreCacheID: 22},
			374: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 182, UncoreCacheID: 22},
			375: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 183, UncoreCacheID: 22},
			376: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 184, UncoreCacheID: 23},
			377: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 185, UncoreCacheID: 23},
			378: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 186, UncoreCacheID: 23},
			379: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 187, UncoreCacheID: 23},
			380: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 188, UncoreCacheID: 23},
			381: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 189, UncoreCacheID: 23},
			382: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 190, UncoreCacheID: 23},
			383: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 191, UncoreCacheID: 23},
		},
	}
	/*
		Topology from single AMD EPYC 7702P 64C; lscpu excerpt
		CPU(s):                128
		On-line CPU(s) list:   0-127
		Thread(s) per core:    2
		Core(s) per socket:    64
		Socket(s):             1
		NUMA node(s):          1
		NUMA node0 CPU(s):     0-127
	*/
	topoSingleSocketSingleNumaPerSocketSMTSmallUncore = &topology.CPUTopology{
		NumCPUs:        128,
		NumCores:       64,
		NumUncoreCache: 16,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 1},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 1},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 1},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 1},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 2},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 2},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 2},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 2},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 3},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 3},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 3},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 3},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 4},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 4},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 4},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 4},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 5},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 5},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 5},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 5},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 6},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 6},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 6},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 6},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 7},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 7},
			30:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 7},
			31:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 7},
			32:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 8},
			33:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 8},
			34:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 8},
			35:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 8},
			36:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 9},
			37:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 9},
			38:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 9},
			39:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 9},
			40:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 10},
			41:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 10},
			42:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 10},
			43:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 10},
			44:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 11},
			45:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 11},
			46:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 11},
			47:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 11},
			48:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 12},
			49:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 12},
			50:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 12},
			51:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 12},
			52:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 13},
			53:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 13},
			54:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 13},
			55:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 13},
			56:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 14},
			57:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 14},
			58:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 14},
			59:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 14},
			60:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 15},
			61:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 15},
			62:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 15},
			63:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 15},
			64:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			65:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			66:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			67:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			68:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 1},
			69:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 1},
			70:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 1},
			71:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 1},
			72:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 2},
			73:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 2},
			74:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 2},
			75:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 2},
			76:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 3},
			77:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 3},
			78:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 3},
			79:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 3},
			80:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 4},
			81:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 4},
			82:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 4},
			83:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 4},
			84:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 5},
			85:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 5},
			86:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 5},
			87:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 5},
			88:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 6},
			89:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 6},
			90:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 6},
			91:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 6},
			92:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 7},
			93:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 7},
			94:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 7},
			95:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 7},
			96:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 8},
			97:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 8},
			98:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 8},
			99:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 8},
			100: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 9},
			101: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 9},
			102: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 9},
			103: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 9},
			104: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 10},
			105: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 10},
			106: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 10},
			107: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 10},
			108: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 11},
			109: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 11},
			110: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 11},
			111: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 11},
			112: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 12},
			113: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 12},
			114: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 12},
			115: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 12},
			116: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 13},
			117: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 13},
			118: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 13},
			119: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 13},
			120: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 14},
			121: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 14},
			122: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 14},
			123: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 14},
			124: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 15},
			125: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 15},
			126: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 15},
			127: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 15},
		},
	}
	/*
		Topology from single AMD EPYC 7702P 64C; lscpu excerpt
		CPU(s):                64
		On-line CPU(s) list:   0-63
		Thread(s) per core:    1
		Core(s) per socket:    64
		Socket(s):             1
		NUMA node(s):          1
		NUMA node0 CPU(s):     0-63
	*/
	topoSingleSocketSingleNumaPerSocketNoSMTSmallUncore = &topology.CPUTopology{
		NumCPUs:        128,
		NumCores:       64,
		NumUncoreCache: 16,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 1},
			5:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 1},
			6:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 1},
			7:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 1},
			8:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 2},
			9:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 2},
			10: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 2},
			11: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 2},
			12: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 3},
			13: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 3},
			14: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 3},
			15: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 3},
			16: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 4},
			17: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 4},
			18: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 4},
			19: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 4},
			20: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 5},
			21: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 5},
			22: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 5},
			23: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 5},
			24: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 6},
			25: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 6},
			26: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 6},
			27: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 6},
			28: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 7},
			29: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 7},
			30: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 7},
			31: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 7},
			32: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 8},
			33: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 8},
			34: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 8},
			35: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 8},
			36: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 9},
			37: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 9},
			38: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 9},
			39: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 9},
			40: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 10},
			41: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 10},
			42: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 10},
			43: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 10},
			44: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 11},
			45: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 11},
			46: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 11},
			47: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 11},
			48: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 12},
			49: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 12},
			50: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 12},
			51: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 12},
			52: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 13},
			53: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 13},
			54: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 13},
			55: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 13},
			56: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 14},
			57: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 14},
			58: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 14},
			59: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 14},
			60: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 15},
			61: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 15},
			62: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 15},
			63: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 15},
		},
	}
	/*
		Topology from dual AMD EPYC 7303 32C; lscpu excerpt
		CPU(s):                64
		On-line CPU(s) list:   0-127
		Thread(s) per core:    1
		Core(s) per socket:    32
		Socket(s):             2
		NUMA node(s):          2
		NUMA node0 CPU(s):     0-31,64-95
		NUMA node1 CPU(s):     32-63,96-127
	*/
	topoSmallDualSocketSingleNumaPerSocketNoSMTUncore = &topology.CPUTopology{
		NumCPUs:        64,
		NumCores:       64,
		NumUncoreCache: 8,
		NumSockets:     2,
		NumNUMANodes:   2,
		CPUDetails: topology.CPUDetails{
			0:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			32: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 32, UncoreCacheID: 4},
			33: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 33, UncoreCacheID: 4},
			34: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 34, UncoreCacheID: 4},
			35: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 35, UncoreCacheID: 4},
			36: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 36, UncoreCacheID: 4},
			37: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 37, UncoreCacheID: 4},
			38: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 38, UncoreCacheID: 4},
			39: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 39, UncoreCacheID: 4},
			40: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 40, UncoreCacheID: 5},
			41: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 41, UncoreCacheID: 5},
			42: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 42, UncoreCacheID: 5},
			43: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 43, UncoreCacheID: 5},
			44: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 44, UncoreCacheID: 5},
			45: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 45, UncoreCacheID: 5},
			46: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 46, UncoreCacheID: 5},
			47: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 47, UncoreCacheID: 5},
			48: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 48, UncoreCacheID: 6},
			49: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 49, UncoreCacheID: 6},
			50: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 50, UncoreCacheID: 6},
			51: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 51, UncoreCacheID: 6},
			52: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 52, UncoreCacheID: 6},
			53: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 53, UncoreCacheID: 6},
			54: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 54, UncoreCacheID: 6},
			55: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 55, UncoreCacheID: 6},
			56: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 56, UncoreCacheID: 7},
			57: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 57, UncoreCacheID: 7},
			58: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 58, UncoreCacheID: 7},
			59: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 59, UncoreCacheID: 7},
			60: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 60, UncoreCacheID: 7},
			61: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 61, UncoreCacheID: 7},
			62: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 62, UncoreCacheID: 7},
			63: topology.CPUInfo{NUMANodeID: 1, SocketID: 1, CoreID: 63, UncoreCacheID: 7},
		},
	}
	/*
		Topology from single AMD EPYC 7303 32C; lscpu excerpt
		CPU(s):                32
		On-line CPU(s) list:   0-31
		Thread(s) per core:    1
		Core(s) per socket:    32
		Socket(s):             1
		NUMA node(s):          1
		NUMA node0 CPU(s):     0-31
	*/
	topoSmallSingleSocketSingleNumaPerSocketNoSMTUncore = &topology.CPUTopology{
		NumCPUs:        32,
		NumCores:       32,
		NumUncoreCache: 4,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
		},
	}
	/*
		Topology from dual AMD EPYC 9754 128C; lscpu excerpt
		CPU(s):                128
		On-line CPU(s) list:   0-255
		Thread(s) per core:    2
		Core(s) per socket:    128
		Socket(s):             1
		NUMA node(s):          1
		NUMA node0 CPU(s):     0-255
	*/
	topoLargeSingleSocketSingleNumaPerSocketSMTUncore = &topology.CPUTopology{
		NumCPUs:        256,
		NumCores:       128,
		NumUncoreCache: 16,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			32:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			33:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			34:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			35:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			36:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			37:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			38:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			39:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			40:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			41:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			42:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			43:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			44:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			45:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			46:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			47:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			48:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			49:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			50:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			51:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			52:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			53:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			54:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			55:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			56:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			57:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			58:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			59:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			60:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			61:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			62:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			63:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			64:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			65:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			66:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			67:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			68:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			69:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			70:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			71:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			72:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			73:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			74:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			75:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			76:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			77:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			78:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			79:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			80:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			81:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			82:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			83:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			84:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			85:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			86:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			87:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			88:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			89:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			90:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			91:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			92:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			93:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			94:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			95:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			96:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 96, UncoreCacheID: 12},
			97:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 97, UncoreCacheID: 12},
			98:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 98, UncoreCacheID: 12},
			99:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 99, UncoreCacheID: 12},
			100: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 100, UncoreCacheID: 12},
			101: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 101, UncoreCacheID: 12},
			102: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 102, UncoreCacheID: 12},
			103: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 103, UncoreCacheID: 12},
			104: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 104, UncoreCacheID: 13},
			105: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 105, UncoreCacheID: 13},
			106: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 106, UncoreCacheID: 13},
			107: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 107, UncoreCacheID: 13},
			108: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 108, UncoreCacheID: 13},
			109: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 109, UncoreCacheID: 13},
			110: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 110, UncoreCacheID: 13},
			111: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 111, UncoreCacheID: 13},
			112: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 112, UncoreCacheID: 14},
			113: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 113, UncoreCacheID: 14},
			114: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 114, UncoreCacheID: 14},
			115: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 115, UncoreCacheID: 14},
			116: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 116, UncoreCacheID: 14},
			117: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 117, UncoreCacheID: 14},
			118: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 118, UncoreCacheID: 14},
			119: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 119, UncoreCacheID: 14},
			120: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 120, UncoreCacheID: 15},
			121: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 121, UncoreCacheID: 15},
			122: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 122, UncoreCacheID: 15},
			123: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 123, UncoreCacheID: 15},
			124: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 124, UncoreCacheID: 15},
			125: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 125, UncoreCacheID: 15},
			126: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 126, UncoreCacheID: 15},
			127: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 127, UncoreCacheID: 15},
			128: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			129: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			130: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			131: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			132: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			133: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			134: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			135: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			136: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			137: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			138: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			139: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			140: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			141: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			142: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			143: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			144: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			145: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			146: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			147: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			148: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			149: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			150: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			151: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			152: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			153: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			154: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			155: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			156: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			157: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			158: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			159: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			160: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			161: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			162: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			163: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			164: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			165: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			166: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			167: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			168: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			169: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			170: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			171: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			172: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			173: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			174: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			175: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			176: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			177: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			178: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			179: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			180: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			181: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			182: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			183: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			184: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			185: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			186: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			187: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			188: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			189: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			190: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			191: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			192: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			193: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			194: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			195: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			196: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			197: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			198: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			199: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			200: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			201: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			202: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			203: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			204: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			205: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			206: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			207: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			208: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			209: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			210: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			211: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			212: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			213: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			214: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			215: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			216: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			217: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			218: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			219: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			220: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			221: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			222: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			223: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			224: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 96, UncoreCacheID: 12},
			225: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 97, UncoreCacheID: 12},
			226: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 98, UncoreCacheID: 12},
			227: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 99, UncoreCacheID: 12},
			228: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 100, UncoreCacheID: 12},
			229: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 101, UncoreCacheID: 12},
			230: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 102, UncoreCacheID: 12},
			231: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 103, UncoreCacheID: 12},
			232: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 104, UncoreCacheID: 13},
			233: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 105, UncoreCacheID: 13},
			234: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 106, UncoreCacheID: 13},
			235: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 107, UncoreCacheID: 13},
			236: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 108, UncoreCacheID: 13},
			237: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 109, UncoreCacheID: 13},
			238: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 110, UncoreCacheID: 13},
			239: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 111, UncoreCacheID: 13},
			240: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 112, UncoreCacheID: 14},
			241: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 113, UncoreCacheID: 14},
			242: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 114, UncoreCacheID: 14},
			243: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 115, UncoreCacheID: 14},
			244: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 116, UncoreCacheID: 14},
			245: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 117, UncoreCacheID: 14},
			246: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 118, UncoreCacheID: 14},
			247: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 119, UncoreCacheID: 14},
			248: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 120, UncoreCacheID: 15},
			249: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 121, UncoreCacheID: 15},
			250: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 122, UncoreCacheID: 15},
			251: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 123, UncoreCacheID: 15},
			252: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 124, UncoreCacheID: 15},
			253: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 125, UncoreCacheID: 15},
			254: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 126, UncoreCacheID: 15},
			255: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 127, UncoreCacheID: 15},
		},
	}
	/*
		Topology from dual AMD EPYC 9654 96-Core Processor; lscpu excerpt
		CPU(s):                192
		On-line CPU(s) list:   0-191
		Thread(s) per core:    1
		Core(s) per socket:    96
		Socket(s):             2
		NUMA node(s):          4
		NUMA node0 CPU(s):     0-47
		NUMA node1 CPU(s):     48-95
		NUMA node2 CPU(s):     96-143
		NUMA node3 CPU(s):     144-191
	*/
	topoDualSocketMultiNumaPerSocketUncore = &topology.CPUTopology{
		NumCPUs:        192,
		NumCores:       192,
		NumUncoreCache: 24,
		NumSockets:     2,
		NumNUMANodes:   4,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			32:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			33:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			34:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			35:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			36:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			37:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			38:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			39:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			40:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			41:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			42:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			43:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			44:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			45:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			46:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			47:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			48:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			49:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			50:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			51:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			52:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			53:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			54:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			55:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			56:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			57:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			58:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			59:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			60:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			61:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			62:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			63:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			64:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			65:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			66:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			67:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			68:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			69:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			70:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			71:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			72:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			73:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			74:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			75:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			76:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			77:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			78:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			79:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			80:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			81:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			82:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			83:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			84:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			85:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			86:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			87:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			88:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			89:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			90:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			91:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			92:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			93:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			94:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			95:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			96:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 0, UncoreCacheID: 12},
			97:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 1, UncoreCacheID: 12},
			98:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 2, UncoreCacheID: 12},
			99:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 3, UncoreCacheID: 12},
			100: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 4, UncoreCacheID: 12},
			101: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 5, UncoreCacheID: 12},
			102: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 6, UncoreCacheID: 12},
			103: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 7, UncoreCacheID: 12},
			104: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 8, UncoreCacheID: 13},
			105: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 9, UncoreCacheID: 13},
			106: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 10, UncoreCacheID: 13},
			107: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 11, UncoreCacheID: 13},
			108: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 12, UncoreCacheID: 13},
			109: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 13, UncoreCacheID: 13},
			110: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 14, UncoreCacheID: 13},
			111: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 15, UncoreCacheID: 13},
			112: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 16, UncoreCacheID: 14},
			113: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 17, UncoreCacheID: 14},
			114: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 18, UncoreCacheID: 14},
			115: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 19, UncoreCacheID: 14},
			116: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 20, UncoreCacheID: 14},
			117: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 21, UncoreCacheID: 14},
			118: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 22, UncoreCacheID: 14},
			119: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 23, UncoreCacheID: 14},
			120: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 24, UncoreCacheID: 15},
			121: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 25, UncoreCacheID: 15},
			122: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 26, UncoreCacheID: 15},
			123: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 27, UncoreCacheID: 15},
			124: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 28, UncoreCacheID: 15},
			125: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 29, UncoreCacheID: 15},
			126: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 30, UncoreCacheID: 15},
			127: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 31, UncoreCacheID: 15},
			128: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 32, UncoreCacheID: 16},
			129: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 33, UncoreCacheID: 16},
			130: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 34, UncoreCacheID: 16},
			131: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 35, UncoreCacheID: 16},
			132: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 36, UncoreCacheID: 16},
			133: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 37, UncoreCacheID: 16},
			134: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 38, UncoreCacheID: 16},
			135: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 39, UncoreCacheID: 16},
			136: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 40, UncoreCacheID: 17},
			137: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 41, UncoreCacheID: 17},
			138: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 42, UncoreCacheID: 17},
			139: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 43, UncoreCacheID: 17},
			140: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 44, UncoreCacheID: 17},
			141: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 45, UncoreCacheID: 17},
			142: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 46, UncoreCacheID: 17},
			143: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 47, UncoreCacheID: 17},
			144: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 48, UncoreCacheID: 18},
			145: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 49, UncoreCacheID: 18},
			146: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 50, UncoreCacheID: 18},
			147: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 51, UncoreCacheID: 18},
			148: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 52, UncoreCacheID: 18},
			149: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 53, UncoreCacheID: 18},
			150: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 54, UncoreCacheID: 18},
			151: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 55, UncoreCacheID: 18},
			152: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 56, UncoreCacheID: 19},
			153: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 57, UncoreCacheID: 19},
			154: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 58, UncoreCacheID: 19},
			155: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 59, UncoreCacheID: 19},
			156: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 60, UncoreCacheID: 19},
			157: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 61, UncoreCacheID: 19},
			158: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 62, UncoreCacheID: 19},
			159: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 63, UncoreCacheID: 19},
			160: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 64, UncoreCacheID: 20},
			161: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 65, UncoreCacheID: 20},
			162: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 66, UncoreCacheID: 20},
			163: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 67, UncoreCacheID: 20},
			164: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 68, UncoreCacheID: 20},
			165: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 69, UncoreCacheID: 20},
			166: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 70, UncoreCacheID: 20},
			167: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 71, UncoreCacheID: 20},
			168: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 72, UncoreCacheID: 21},
			169: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 73, UncoreCacheID: 21},
			170: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 74, UncoreCacheID: 21},
			171: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 75, UncoreCacheID: 21},
			172: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 76, UncoreCacheID: 21},
			173: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 77, UncoreCacheID: 21},
			174: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 78, UncoreCacheID: 21},
			175: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 79, UncoreCacheID: 21},
			176: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 80, UncoreCacheID: 22},
			177: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 81, UncoreCacheID: 22},
			178: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 82, UncoreCacheID: 22},
			179: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 83, UncoreCacheID: 22},
			180: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 84, UncoreCacheID: 22},
			181: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 85, UncoreCacheID: 22},
			182: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 86, UncoreCacheID: 22},
			183: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 87, UncoreCacheID: 22},
			184: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 88, UncoreCacheID: 23},
			185: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 89, UncoreCacheID: 23},
			186: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 90, UncoreCacheID: 23},
			187: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 91, UncoreCacheID: 23},
			188: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 92, UncoreCacheID: 23},
			189: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 93, UncoreCacheID: 23},
			190: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 94, UncoreCacheID: 23},
			191: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 95, UncoreCacheID: 23},
		},
	}
	/*
		Topology from dual Intel Xeon Platinum 8490H 60-Core Processor; lscpu excerpt
		CPU(s):                240
		On-line CPU(s) list:   0-239
		Thread(s) per core:    2
		Core(s) per socket:    60
		Socket(s):             2
		NUMA node(s):          4
		NUMA node0 CPU(s):     0-29,120-149
		NUMA node1 CPU(s):     30-59,150-179
		NUMA node2 CPU(s):     60-89,180-209
		NUMA node3 CPU(s):     90-119,210-239
	*/
	topoDualSocketSubNumaPerSocketHTMonolithicUncore = &topology.CPUTopology{
		NumCPUs:        240,
		NumCores:       120,
		NumUncoreCache: 2,
		NumSockets:     2,
		NumNUMANodes:   4,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 0},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 0},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 0},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 0},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 0},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 0},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 0},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 0},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 0},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 0},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 0},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 0},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 0},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 0},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 0},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 0},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 0},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 0},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 0},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 0},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 0},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 0},
			30:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 30, UncoreCacheID: 0},
			31:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 31, UncoreCacheID: 0},
			32:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 32, UncoreCacheID: 0},
			33:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 33, UncoreCacheID: 0},
			34:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 34, UncoreCacheID: 0},
			35:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 35, UncoreCacheID: 0},
			36:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 36, UncoreCacheID: 0},
			37:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 37, UncoreCacheID: 0},
			38:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 38, UncoreCacheID: 0},
			39:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 39, UncoreCacheID: 0},
			40:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 40, UncoreCacheID: 0},
			41:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 41, UncoreCacheID: 0},
			42:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 42, UncoreCacheID: 0},
			43:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 43, UncoreCacheID: 0},
			44:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 44, UncoreCacheID: 0},
			45:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 45, UncoreCacheID: 0},
			46:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 46, UncoreCacheID: 0},
			47:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 47, UncoreCacheID: 0},
			48:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 48, UncoreCacheID: 0},
			49:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 49, UncoreCacheID: 0},
			50:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 50, UncoreCacheID: 0},
			51:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 51, UncoreCacheID: 0},
			52:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 52, UncoreCacheID: 0},
			53:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 53, UncoreCacheID: 0},
			54:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 54, UncoreCacheID: 0},
			55:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 55, UncoreCacheID: 0},
			56:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 56, UncoreCacheID: 0},
			57:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 57, UncoreCacheID: 0},
			58:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 58, UncoreCacheID: 0},
			59:  topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 59, UncoreCacheID: 0},
			60:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 60, UncoreCacheID: 1},
			61:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 61, UncoreCacheID: 1},
			62:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 62, UncoreCacheID: 1},
			63:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 63, UncoreCacheID: 1},
			64:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 64, UncoreCacheID: 1},
			65:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 65, UncoreCacheID: 1},
			66:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 66, UncoreCacheID: 1},
			67:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 67, UncoreCacheID: 1},
			68:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 68, UncoreCacheID: 1},
			69:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 69, UncoreCacheID: 1},
			70:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 70, UncoreCacheID: 1},
			71:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 71, UncoreCacheID: 1},
			72:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 72, UncoreCacheID: 1},
			73:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 73, UncoreCacheID: 1},
			74:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 74, UncoreCacheID: 1},
			75:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 75, UncoreCacheID: 1},
			76:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 76, UncoreCacheID: 1},
			77:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 77, UncoreCacheID: 1},
			78:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 78, UncoreCacheID: 1},
			79:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 79, UncoreCacheID: 1},
			80:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 80, UncoreCacheID: 1},
			81:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 81, UncoreCacheID: 1},
			82:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 82, UncoreCacheID: 1},
			83:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 83, UncoreCacheID: 1},
			84:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 84, UncoreCacheID: 1},
			85:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 85, UncoreCacheID: 1},
			86:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 86, UncoreCacheID: 1},
			87:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 87, UncoreCacheID: 1},
			88:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 88, UncoreCacheID: 1},
			89:  topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 89, UncoreCacheID: 1},
			90:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 90, UncoreCacheID: 1},
			91:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 91, UncoreCacheID: 1},
			92:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 92, UncoreCacheID: 1},
			93:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 93, UncoreCacheID: 1},
			94:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 94, UncoreCacheID: 1},
			95:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 95, UncoreCacheID: 1},
			96:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 96, UncoreCacheID: 1},
			97:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 97, UncoreCacheID: 1},
			98:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 98, UncoreCacheID: 1},
			99:  topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 99, UncoreCacheID: 1},
			100: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 100, UncoreCacheID: 1},
			101: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 101, UncoreCacheID: 1},
			102: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 102, UncoreCacheID: 1},
			103: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 103, UncoreCacheID: 1},
			104: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 104, UncoreCacheID: 1},
			105: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 105, UncoreCacheID: 1},
			106: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 106, UncoreCacheID: 1},
			107: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 107, UncoreCacheID: 1},
			108: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 108, UncoreCacheID: 1},
			109: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 109, UncoreCacheID: 1},
			110: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 110, UncoreCacheID: 1},
			111: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 111, UncoreCacheID: 1},
			112: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 112, UncoreCacheID: 1},
			113: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 113, UncoreCacheID: 1},
			114: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 114, UncoreCacheID: 1},
			115: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 115, UncoreCacheID: 1},
			116: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 116, UncoreCacheID: 1},
			117: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 117, UncoreCacheID: 1},
			118: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 118, UncoreCacheID: 1},
			119: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 119, UncoreCacheID: 1},
			120: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			121: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			122: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			123: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			124: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			125: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			126: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			127: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			128: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 0},
			129: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 0},
			130: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 0},
			131: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 0},
			132: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 0},
			133: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 0},
			134: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 0},
			135: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 0},
			136: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 0},
			137: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 0},
			138: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 0},
			139: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 0},
			140: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 0},
			141: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 0},
			142: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 0},
			143: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 0},
			144: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 0},
			145: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 0},
			146: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 0},
			147: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 0},
			148: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 0},
			149: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 0},
			150: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 30, UncoreCacheID: 0},
			151: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 31, UncoreCacheID: 0},
			152: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 32, UncoreCacheID: 0},
			153: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 33, UncoreCacheID: 0},
			154: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 34, UncoreCacheID: 0},
			155: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 35, UncoreCacheID: 0},
			156: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 36, UncoreCacheID: 0},
			157: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 37, UncoreCacheID: 0},
			158: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 38, UncoreCacheID: 0},
			159: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 39, UncoreCacheID: 0},
			160: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 40, UncoreCacheID: 0},
			161: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 41, UncoreCacheID: 0},
			162: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 42, UncoreCacheID: 0},
			163: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 43, UncoreCacheID: 0},
			164: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 44, UncoreCacheID: 0},
			165: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 45, UncoreCacheID: 0},
			166: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 46, UncoreCacheID: 0},
			167: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 47, UncoreCacheID: 0},
			168: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 48, UncoreCacheID: 0},
			169: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 49, UncoreCacheID: 0},
			170: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 50, UncoreCacheID: 0},
			171: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 51, UncoreCacheID: 0},
			172: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 52, UncoreCacheID: 0},
			173: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 53, UncoreCacheID: 0},
			174: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 54, UncoreCacheID: 0},
			175: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 55, UncoreCacheID: 0},
			176: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 56, UncoreCacheID: 0},
			177: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 57, UncoreCacheID: 0},
			178: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 58, UncoreCacheID: 0},
			179: topology.CPUInfo{NUMANodeID: 1, SocketID: 0, CoreID: 59, UncoreCacheID: 0},
			180: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 60, UncoreCacheID: 1},
			181: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 61, UncoreCacheID: 1},
			182: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 62, UncoreCacheID: 1},
			183: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 63, UncoreCacheID: 1},
			184: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 64, UncoreCacheID: 1},
			185: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 65, UncoreCacheID: 1},
			186: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 66, UncoreCacheID: 1},
			187: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 67, UncoreCacheID: 1},
			188: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 68, UncoreCacheID: 1},
			189: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 69, UncoreCacheID: 1},
			190: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 70, UncoreCacheID: 1},
			191: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 71, UncoreCacheID: 1},
			192: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 72, UncoreCacheID: 1},
			193: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 73, UncoreCacheID: 1},
			194: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 74, UncoreCacheID: 1},
			195: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 75, UncoreCacheID: 1},
			196: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 76, UncoreCacheID: 1},
			197: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 77, UncoreCacheID: 1},
			198: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 78, UncoreCacheID: 1},
			199: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 79, UncoreCacheID: 1},
			200: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 80, UncoreCacheID: 1},
			201: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 81, UncoreCacheID: 1},
			202: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 82, UncoreCacheID: 1},
			203: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 83, UncoreCacheID: 1},
			204: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 84, UncoreCacheID: 1},
			205: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 85, UncoreCacheID: 1},
			206: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 86, UncoreCacheID: 1},
			207: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 87, UncoreCacheID: 1},
			208: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 88, UncoreCacheID: 1},
			209: topology.CPUInfo{NUMANodeID: 2, SocketID: 1, CoreID: 89, UncoreCacheID: 1},
			210: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 90, UncoreCacheID: 1},
			211: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 91, UncoreCacheID: 1},
			212: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 92, UncoreCacheID: 1},
			213: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 93, UncoreCacheID: 1},
			214: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 94, UncoreCacheID: 1},
			215: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 95, UncoreCacheID: 1},
			216: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 96, UncoreCacheID: 1},
			217: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 97, UncoreCacheID: 1},
			218: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 98, UncoreCacheID: 1},
			219: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 99, UncoreCacheID: 1},
			220: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 100, UncoreCacheID: 1},
			221: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 101, UncoreCacheID: 1},
			222: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 102, UncoreCacheID: 1},
			223: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 103, UncoreCacheID: 1},
			224: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 104, UncoreCacheID: 1},
			225: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 105, UncoreCacheID: 1},
			226: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 106, UncoreCacheID: 1},
			227: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 107, UncoreCacheID: 1},
			228: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 108, UncoreCacheID: 1},
			229: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 109, UncoreCacheID: 1},
			230: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 110, UncoreCacheID: 1},
			231: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 111, UncoreCacheID: 1},
			232: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 112, UncoreCacheID: 1},
			233: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 113, UncoreCacheID: 1},
			234: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 114, UncoreCacheID: 1},
			235: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 115, UncoreCacheID: 1},
			236: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 116, UncoreCacheID: 1},
			237: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 117, UncoreCacheID: 1},
			238: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 118, UncoreCacheID: 1},
			239: topology.CPUInfo{NUMANodeID: 3, SocketID: 1, CoreID: 119, UncoreCacheID: 1},
		},
	}
	/*
		Topology from dual Intel Core i7-12850HX 10-Core Processor; lscpu excerpt
		CPU(s):                24
		On-line CPU(s) list:   0-23
		Thread(s) per core:    2
		Core(s) per socket:    16
		Socket(s):             1
		NUMA node(s):          1
	*/
	topoSingleSocketSingleNumaPerSocketPCoreHTMonolithicUncore = &topology.CPUTopology{
		NumCPUs:        24,
		NumCores:       16, // 8 E-Cores and 8 P-Cores
		NumUncoreCache: 1,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 0},
			9:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 0},
			10: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 0},
			11: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 0},
			12: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 0},
			13: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 0},
			14: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 0},
			15: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 0},
			16: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 0},
			17: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 0},
			18: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 0},
			19: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 0},
			20: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 0},
			21: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 0},
			22: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 0},
			23: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 0},
		},
	}
	/*
		Topology from 1P ARM Ampere Altra 128C; lscpu excerpt
		CPU(s):                128
		On-line CPU(s) list:   0-127
		Thread(s) per core:    1
		Core(s) per socket:    128
		Socket(s):             1
		NUMA node(s):          1
	*/
	topoLargeSingleSocketSingleNumaPerSocketUncore = &topology.CPUTopology{
		NumCPUs:        128,
		NumCores:       128,
		NumUncoreCache: 16,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 1},
			9:   topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 1},
			10:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 1},
			11:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 1},
			12:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 1},
			13:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 1},
			14:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 1},
			15:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 1},
			16:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 2},
			17:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 2},
			18:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 2},
			19:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 2},
			20:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 2},
			21:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 2},
			22:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 2},
			23:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 2},
			24:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 3},
			25:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 3},
			26:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 3},
			27:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 3},
			28:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 3},
			29:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 3},
			30:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 3},
			31:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 3},
			32:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 4},
			33:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 4},
			34:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 4},
			35:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 4},
			36:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 4},
			37:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 4},
			38:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 4},
			39:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 4},
			40:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 5},
			41:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 5},
			42:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 5},
			43:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 5},
			44:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 5},
			45:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 5},
			46:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 5},
			47:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 5},
			48:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 6},
			49:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 6},
			50:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 6},
			51:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 6},
			52:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 6},
			53:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 6},
			54:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 6},
			55:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 6},
			56:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 7},
			57:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 7},
			58:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 7},
			59:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 7},
			60:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 7},
			61:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 7},
			62:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 7},
			63:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 7},
			64:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 64, UncoreCacheID: 8},
			65:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 65, UncoreCacheID: 8},
			66:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 66, UncoreCacheID: 8},
			67:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 67, UncoreCacheID: 8},
			68:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 68, UncoreCacheID: 8},
			69:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 69, UncoreCacheID: 8},
			70:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 70, UncoreCacheID: 8},
			71:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 71, UncoreCacheID: 8},
			72:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 72, UncoreCacheID: 9},
			73:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 73, UncoreCacheID: 9},
			74:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 74, UncoreCacheID: 9},
			75:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 75, UncoreCacheID: 9},
			76:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 76, UncoreCacheID: 9},
			77:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 77, UncoreCacheID: 9},
			78:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 78, UncoreCacheID: 9},
			79:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 79, UncoreCacheID: 9},
			80:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 80, UncoreCacheID: 10},
			81:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 81, UncoreCacheID: 10},
			82:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 82, UncoreCacheID: 10},
			83:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 83, UncoreCacheID: 10},
			84:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 84, UncoreCacheID: 10},
			85:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 85, UncoreCacheID: 10},
			86:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 86, UncoreCacheID: 10},
			87:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 87, UncoreCacheID: 10},
			88:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 88, UncoreCacheID: 11},
			89:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 89, UncoreCacheID: 11},
			90:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 90, UncoreCacheID: 11},
			91:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 91, UncoreCacheID: 11},
			92:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 92, UncoreCacheID: 11},
			93:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 93, UncoreCacheID: 11},
			94:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 94, UncoreCacheID: 11},
			95:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 95, UncoreCacheID: 11},
			96:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 96, UncoreCacheID: 12},
			97:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 97, UncoreCacheID: 12},
			98:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 98, UncoreCacheID: 12},
			99:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 99, UncoreCacheID: 12},
			100: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 100, UncoreCacheID: 12},
			101: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 101, UncoreCacheID: 12},
			102: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 102, UncoreCacheID: 12},
			103: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 103, UncoreCacheID: 12},
			104: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 104, UncoreCacheID: 13},
			105: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 105, UncoreCacheID: 13},
			106: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 106, UncoreCacheID: 13},
			107: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 107, UncoreCacheID: 13},
			108: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 108, UncoreCacheID: 13},
			109: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 109, UncoreCacheID: 13},
			110: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 110, UncoreCacheID: 13},
			111: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 111, UncoreCacheID: 13},
			112: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 112, UncoreCacheID: 14},
			113: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 113, UncoreCacheID: 14},
			114: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 114, UncoreCacheID: 14},
			115: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 115, UncoreCacheID: 14},
			116: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 116, UncoreCacheID: 14},
			117: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 117, UncoreCacheID: 14},
			118: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 118, UncoreCacheID: 14},
			119: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 119, UncoreCacheID: 14},
			120: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 120, UncoreCacheID: 15},
			121: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 121, UncoreCacheID: 15},
			122: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 122, UncoreCacheID: 15},
			123: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 123, UncoreCacheID: 15},
			124: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 124, UncoreCacheID: 15},
			125: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 125, UncoreCacheID: 15},
			126: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 126, UncoreCacheID: 15},
			127: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 127, UncoreCacheID: 15},
		},
	}
	/*
		Topology from 1P AWS Graviton3 64C; lscpu excerpt
		CPU(s):                64
		On-line CPU(s) list:   0-63
		Thread(s) per core:    1
		Core(s) per socket:    64
		Socket(s):             1
		NUMA node(s):          1
	*/
	topoSingleSocketSingleNumaPerSocketUncore = &topology.CPUTopology{
		NumCPUs:        64,
		NumCores:       64,
		NumUncoreCache: 1,
		NumSockets:     1,
		NumNUMANodes:   1,
		CPUDetails: topology.CPUDetails{
			0:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 0, UncoreCacheID: 0},
			1:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 1, UncoreCacheID: 0},
			2:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 2, UncoreCacheID: 0},
			3:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 3, UncoreCacheID: 0},
			4:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 4, UncoreCacheID: 0},
			5:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 5, UncoreCacheID: 0},
			6:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 6, UncoreCacheID: 0},
			7:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 7, UncoreCacheID: 0},
			8:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 8, UncoreCacheID: 0},
			9:  topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 9, UncoreCacheID: 0},
			10: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 10, UncoreCacheID: 0},
			11: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 11, UncoreCacheID: 0},
			12: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 12, UncoreCacheID: 0},
			13: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 13, UncoreCacheID: 0},
			14: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 14, UncoreCacheID: 0},
			15: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 15, UncoreCacheID: 0},
			16: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 16, UncoreCacheID: 0},
			17: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 17, UncoreCacheID: 0},
			18: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 18, UncoreCacheID: 0},
			19: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 19, UncoreCacheID: 0},
			20: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 20, UncoreCacheID: 0},
			21: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 21, UncoreCacheID: 0},
			22: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 22, UncoreCacheID: 0},
			23: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 23, UncoreCacheID: 0},
			24: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 24, UncoreCacheID: 0},
			25: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 25, UncoreCacheID: 0},
			26: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 26, UncoreCacheID: 0},
			27: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 27, UncoreCacheID: 0},
			28: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 28, UncoreCacheID: 0},
			29: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 29, UncoreCacheID: 0},
			30: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 30, UncoreCacheID: 0},
			31: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 31, UncoreCacheID: 0},
			32: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 32, UncoreCacheID: 0},
			33: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 33, UncoreCacheID: 0},
			34: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 34, UncoreCacheID: 0},
			35: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 35, UncoreCacheID: 0},
			36: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 36, UncoreCacheID: 0},
			37: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 37, UncoreCacheID: 0},
			38: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 38, UncoreCacheID: 0},
			39: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 39, UncoreCacheID: 0},
			40: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 40, UncoreCacheID: 0},
			41: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 41, UncoreCacheID: 0},
			42: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 42, UncoreCacheID: 0},
			43: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 43, UncoreCacheID: 0},
			44: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 44, UncoreCacheID: 0},
			45: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 45, UncoreCacheID: 0},
			46: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 46, UncoreCacheID: 0},
			47: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 47, UncoreCacheID: 0},
			48: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 48, UncoreCacheID: 0},
			49: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 49, UncoreCacheID: 0},
			50: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 50, UncoreCacheID: 0},
			51: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 51, UncoreCacheID: 0},
			52: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 52, UncoreCacheID: 0},
			53: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 53, UncoreCacheID: 0},
			54: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 54, UncoreCacheID: 0},
			55: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 55, UncoreCacheID: 0},
			56: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 56, UncoreCacheID: 0},
			57: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 57, UncoreCacheID: 0},
			58: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 58, UncoreCacheID: 0},
			59: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 59, UncoreCacheID: 0},
			60: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 60, UncoreCacheID: 0},
			61: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 61, UncoreCacheID: 0},
			62: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 62, UncoreCacheID: 0},
			63: topology.CPUInfo{NUMANodeID: 0, SocketID: 0, CoreID: 63, UncoreCacheID: 0},
		},
	}
)
