/*============================================================================

This C source file is part of the SoftFloat IEEE Floating-Point Arithmetic
Package, Release 3e, by John R. Hauser.

Copyright 2011, 2012, 2013, 2014 The Regents of the University of California.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice,
    this list of conditions, and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions, and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

 3. Neither the name of the University nor the names of its contributors may
    be used to endorse or promote products derived from this software without
    specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS "AS IS", AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE
DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=============================================================================*/

#include <stdbool.h>
#include <stdint.h>
#include "internals.h"
#include "primitives.h"
#include "specialize.h"
#include "softfloat.h"

float64 f64_div(float64 a, float64 b, struct softfloat_status_t *status)
{
    bool signA;
    int16_t expA;
    uint64_t sigA;
    bool signB;
    int16_t expB;
    uint64_t sigB;
    bool signZ;
    struct exp16_sig64 normExpSig;
    int16_t expZ;
    uint32_t recip32, sig32Z, doubleTerm;
    uint64_t rem;
    uint32_t q;
    uint64_t sigZ;

    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    signA = signF64UI(a);
    expA  = expF64UI(a);
    sigA  = fracF64UI(a);
    signB = signF64UI(b);
    expB  = expF64UI(b);
    sigB  = fracF64UI(b);
    signZ = signA ^ signB;
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    if (softfloat_denormalsAreZeros(status)) {
        if (!expA) sigA = 0;
        if (!expB) sigB = 0;
    }
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    if (expA == 0x7FF) {
        if (sigA) goto propagateNaN;
        if (expB == 0x7FF) {
            if (sigB) goto propagateNaN;
            goto invalid;
        }
        if (sigB && !expB)
            softfloat_raiseFlags(status, softfloat_flag_denormal);
        goto infinity;
    }
    if (expB == 0x7FF) {
        if (sigB) goto propagateNaN;
        if (sigA && !expA)
            softfloat_raiseFlags(status, softfloat_flag_denormal);
        goto zero;
    }
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    if (! expB) {
        if (! sigB) {
            if (! (expA | sigA)) goto invalid;
            softfloat_raiseFlags(status, softfloat_flag_infinite);
            goto infinity;
        }
        softfloat_raiseFlags(status, softfloat_flag_denormal);
        normExpSig = softfloat_normSubnormalF64Sig(sigB);
        expB = normExpSig.exp;
        sigB = normExpSig.sig;
    }
    if (! expA) {
        if (! sigA) goto zero;
        softfloat_raiseFlags(status, softfloat_flag_denormal);
        normExpSig = softfloat_normSubnormalF64Sig(sigA);
        expA = normExpSig.exp;
        sigA = normExpSig.sig;
    }
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    expZ = expA - expB + 0x3FE;
    sigA |= UINT64_C(0x0010000000000000);
    sigB |= UINT64_C(0x0010000000000000);
    if (sigA < sigB) {
        --expZ;
        sigA <<= 11;
    } else {
        sigA <<= 10;
    }
    sigB <<= 11;
    recip32 = softfloat_approxRecip32_1(sigB>>32) - 2;
    sig32Z = ((uint32_t) (sigA>>32) * (uint64_t) recip32)>>32;
    doubleTerm = sig32Z<<1;
    rem =
        ((sigA - (uint64_t) doubleTerm * (uint32_t) (sigB>>32))<<28)
            - (uint64_t) doubleTerm * ((uint32_t) sigB>>4);
    q = (((uint32_t) (rem>>32) * (uint64_t) recip32)>>32) + 4;
    sigZ = ((uint64_t) sig32Z<<32) + ((uint64_t) q<<4);
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
    if ((sigZ & 0x1FF) < 4<<4) {
        q &= ~7;
        sigZ &= ~(uint64_t) 0x7F;
        doubleTerm = q<<1;
        rem =
            ((rem - (uint64_t) doubleTerm * (uint32_t) (sigB>>32))<<28)
                - (uint64_t) doubleTerm * ((uint32_t) sigB>>4);
        if (rem & UINT64_C(0x8000000000000000)) {
            sigZ -= 1<<7;
        } else {
            if (rem) sigZ |= 1;
        }
    }
    return softfloat_roundPackToF64(signZ, expZ, sigZ, status);
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
 propagateNaN:
    return softfloat_propagateNaNF64UI(a, b, status);
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
 invalid:
    softfloat_raiseFlags(status, softfloat_flag_invalid);
    return defaultNaNF64UI;
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
 infinity:
    return packToF64UI(signZ, 0x7FF, 0);
    /*------------------------------------------------------------------------
    *------------------------------------------------------------------------*/
 zero:
    return packToF64UI(signZ, 0, 0);
}
