/**************************************************************************
 *
 * Adapted from MIT-Licensed: https://github.com/alitrack/mman-win32
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
#include <windows.h>
#include <errno.h>
#include <io.h>

#include "mman_win32.h"

#ifndef FILE_MAP_EXECUTE
#define FILE_MAP_EXECUTE    0x0020
#endif /* FILE_MAP_EXECUTE */

static int __map_mman_error(const DWORD err, const int deferr)
{
   (void)deferr;
   if (err == 0)
      return 0;
   //TODO: implement
   return err;
}

static DWORD __map_mmap_prot_page(const int prot)
{
   DWORD protect = 0;

   if (prot == PROT_NONE)
      return protect;

   if ((prot & PROT_EXEC) != 0) {
      protect = ((prot & PROT_WRITE) != 0) ?
         PAGE_EXECUTE_READWRITE : PAGE_EXECUTE_READ;
   } else {
      protect = ((prot & PROT_WRITE) != 0) ?
         PAGE_READWRITE : PAGE_READONLY;
   }

   return protect;
}

static DWORD __map_mmap_prot_file(const int prot)
{
   DWORD desiredAccess = 0;

   if (prot == PROT_NONE)
      return desiredAccess;

   if ((prot & PROT_READ) != 0)
      desiredAccess |= FILE_MAP_READ;
   if ((prot & PROT_WRITE) != 0)
      desiredAccess |= FILE_MAP_WRITE;
   if ((prot & PROT_EXEC) != 0)
      desiredAccess |= FILE_MAP_EXECUTE;

   return desiredAccess;
}

void *mmap(void *addr, size_t len, int prot, int flags, int fildes, intptr_t off)
{
   HANDLE fm, h;

   void *map = MAP_FAILED;

#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable: 4293)
#endif

   const DWORD dwFileOffsetLow = (sizeof(intptr_t) <= sizeof(DWORD)) ?
      (DWORD)off : (DWORD)(off & 0xFFFFFFFFL);
   const DWORD dwFileOffsetHigh = (sizeof(intptr_t) <= sizeof(DWORD)) ?
      (DWORD)0 : (DWORD)((off >> 32) & 0xFFFFFFFFL);
   const DWORD protect = __map_mmap_prot_page(prot);
   const DWORD desiredAccess = __map_mmap_prot_file(prot);

   const intptr_t maxSize = off + (intptr_t)len;

   const DWORD dwMaxSizeLow = (sizeof(intptr_t) <= sizeof(DWORD)) ?
      (DWORD)maxSize : (DWORD)(maxSize & 0xFFFFFFFFL);
   const DWORD dwMaxSizeHigh = (sizeof(intptr_t) <= sizeof(DWORD)) ?
      (DWORD)0 : (DWORD)((maxSize >> 32) & 0xFFFFFFFFL);

#ifdef _MSC_VER
#pragma warning(pop)
#endif

   errno = 0;

   if (len == 0
      /* Usupported protection combinations */
      || prot == PROT_EXEC) {
      errno = EINVAL;
      return MAP_FAILED;
   }

   h = ((flags & MAP_ANONYMOUS) == 0) ?
      (HANDLE)_get_osfhandle(fildes) : INVALID_HANDLE_VALUE;

   if ((flags & MAP_ANONYMOUS) == 0 && h == INVALID_HANDLE_VALUE) {
      errno = EBADF;
      return MAP_FAILED;
   }

   fm = CreateFileMappingW(h, NULL, protect, dwMaxSizeHigh, dwMaxSizeLow, NULL);

   if (fm == NULL) {
      errno = __map_mman_error(GetLastError(), EPERM);
      return MAP_FAILED;
   }

   if ((flags & MAP_FIXED) == 0) {
      map = MapViewOfFile(fm, desiredAccess, dwFileOffsetHigh, dwFileOffsetLow, len);
   } else {
      map = MapViewOfFileEx(fm, desiredAccess, dwFileOffsetHigh, dwFileOffsetLow, len, addr);
   }

   CloseHandle(fm);

   if (map == NULL) {
      errno = __map_mman_error(GetLastError(), EPERM);
      return MAP_FAILED;
   }

   return map;
}

int munmap(void *addr, size_t len)
{
   (void)len;
   if (UnmapViewOfFile(addr))
      return 0;

   errno = __map_mman_error(GetLastError(), EPERM);

   return -1;
}

int _mprotect(void *addr, size_t len, int prot)
{
   DWORD newProtect = __map_mmap_prot_page(prot);
   DWORD oldProtect = 0;

   if (VirtualProtect(addr, len, newProtect, &oldProtect))
      return 0;

   errno = __map_mman_error(GetLastError(), EPERM);

   return -1;
}

int msync(void *addr, size_t len, int flags)
{
   (void)flags;
   if (FlushViewOfFile(addr, len))
      return 0;

   errno = __map_mman_error(GetLastError(), EPERM);

   return -1;
}

int mlock(const void *addr, size_t len)
{
   if (VirtualLock((LPVOID)addr, len))
      return 0;

   errno = __map_mman_error(GetLastError(), EPERM);

   return -1;
}

int munlock(const void *addr, size_t len)
{
   if (VirtualUnlock((LPVOID)addr, len))
      return 0;

   errno = __map_mman_error(GetLastError(), EPERM);

   return -1;
}
