/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

#pragma once

#include <inttypes.h>
#include <cstddef>

/* Record types */
#define TS_TAPE 	1	/* dump tape header */
#define TS_INODE	2	/* beginning of file record */
#define TS_BITS 	3	/* map of inodes on tape */
#define TS_ADDR 	4	/* continuation of file record */
#define TS_END  	5	/* end of volume marker */
#define TS_CLRI 	6	/* map of inodes deleted since last dump */

/* flags */
#define DR_NEWHEADER	0x0001	/* new format tape header */
#define DR_NEWINODEFMT	0x0002	/* new format inodes on tape */
#define DR_COMPRESSED	0x0080	/* dump tape is compressed */
#define DR_METAONLY	0x0100	/* only the metadata of the inode has
				 * been dumped */
#define DR_EXTATTRIBUTES	0x8000

typedef uint32_t	dump_ino_t;

struct bsdtimeval {
	uint32_t   tv_sec;
	uint32_t   tv_usec;
};

#define NDADDR		12
#define NIADDR		 3

struct new_bsd_inode {
	uint16_t		di_mode;
	int16_t		di_nlink;
	union {
		uint16_t		oldids[2];
		uint32_t		inumber;
	}		di_u;
	uint64_t	di_size;
	bsdtimeval	di_atime;
	bsdtimeval	di_mtime;
	bsdtimeval	di_ctime;
	uint32_t		di_db[NDADDR];
	uint32_t		di_ib[NIADDR];
	uint32_t		di_flags;
	int32_t		di_blocks;
	int32_t		di_gen;
	uint32_t		di_uid;
	uint32_t		di_gid;
	int32_t		di_spare[2];
};
static_assert(sizeof(new_bsd_inode) == 128);

#define TP_BSIZE	1024
#define TP_NINDIR	(TP_BSIZE/2)
#define TP_NINOS	(TP_NINDIR / sizeof (dump_ino_t))

extern union u_data {
	char	s_addrs[TP_NINDIR];	/* 1 => data; 0 => hole in inode */
	dump_ino_t	s_inos[TP_NINOS];	/* table of first inode on each volume */
} u_data;
static_assert(sizeof(u_data) == 512);

#define LBLSIZE		16
#define NAMELEN		64

struct	header {
	int32_t	c_type;		    /* record type (see below) */
	int32_t	c_date;		    /* date of this dump */
	int32_t	c_ddate;	    /* date of previous dump */
	int32_t	c_volume;	    /* dump volume number */
	uint32_t c_tapea_lo;	    /* logical block of this record */
	dump_ino_t c_inumber;	    /* number of inode */
	int32_t	c_magic;	    /* magic number (see above) */
	int32_t	c_checksum;	    /* record checksum */
	new_bsd_inode c_dinode;
	int32_t	c_count;	    /* number of valid c_addr entries */
	union u_data c_data;	    /* see above */
	char	c_label[LBLSIZE];   /* dump label */
	int32_t	c_level;	    /* level of this dump */
	char	c_filesys[NAMELEN]; /* name of dumpped file system */
	char	c_dev[NAMELEN];	    /* name of dumpped device */
	char	c_host[NAMELEN];    /* name of dumpped host */
	int32_t	c_flags;	    /* additional information */
	uint32_t	c_firstrec_lo; /* first record on volume */
	int32_t	c_ntrec;	    /* blocksize on volume */
	int32_t	c_extattributes;    /* additional inode info */
	uint32_t	c_tapea_hi; /* top 32 bits of c_tapea */
	uint32_t	c_firstrec_hi; /* top 32 bits of c_firstrec */
	int32_t	c_spare[28];	    /* reserved for future uses */
};
static_assert(sizeof(header) == 1024);

struct direct {
	uint32_t d_ino;
	uint16_t d_reclen;
	uint8_t d_type;
	uint8_t d_namlen;
	char d_name[0];
};
static_assert(sizeof(direct) == 8);

#define NFS_MAGIC 60012

struct bswap_context {
	bool to_cpu = false;
	bool from_cpu = false;
	bool to_host = false;
	uint8_t s_addr[512] = {0};
	uint8_t* s_addr_ptr = 0;
	uint32_t subcount = 0;
	int state = 0;
	size_t count = 0;
	bool inode_is_directory = false;
	uint64_t inode_data_size = 0;
	bool seen_end = true;
	bool new_vol = false;
	bool oldinofmt = false;
	dump_ino_t inode_num;
};

extern uint32_t get_s_addr_length(const uint8_t* ptr);
extern bool checksum(header* hdr);
extern void setchecksum(header* hdr);
extern void bswap_header(uint8_t* data, size_t bytes, bswap_context* context);
extern void swap1(int count, uint8_t*& data);
extern void swap2(int count, uint8_t*& data);
extern void swap4(int count, uint8_t*& data);
extern void swap8(int count, uint8_t*& data);

extern void do_bswap_hdr(header* hdr);

template<typename T> T tohost(const T& v) {
	T n(v);
#if BYTE_ORDER == BIG_ENDIAN
	auto np = reinterpret_cast<uint8_t*>(&n);
	if constexpr (sizeof(T) == 8) {
		swap8(1, np);
	} else if constexpr (sizeof(T) == 4) {
		swap4(1, np);
	} else if constexpr (sizeof(T) == 2) {
		swap2(1, np);
	} else if constexpr (sizeof(T) == 1) {
		swap1(1, np);
	}
#endif
	return n;
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
