/*
 * Copyright Supranational LLC
 * Licensed under the Apache License, Version 2.0, see LICENSE for details.
 * SPDX-License-Identifier: Apache-2.0
 */
/*
 * The "magic" number is (BLS12_381_P-3)/4. Exponentiation to which
 * yields reciprocal of sqrt(x), which is used in simplified Shallue-
 * van de Woestijne-Ulas map-to-curve method, but it's trivial to adapt
 * it for more "traditional" sqrt(x) as 'x*ret' (or for is_square(x)
 * as 'x*ret^2==1').
 *
 * Generated with 'addchain 1000602388805416848354447456433976039139220704984751971333014534031007912622709466110671907282253916009473568139946'
 * https://github.com/kwantam/addchain
 *
 * # Bos-Coster (win=4)           :  458 (16) <<<
 * # Bos-Coster (win=5)           :  460 (28)
 * # Bos-Coster (win=6)           :  461 (33)
 * # Bos-Coster (win=7)           :  460 (28)
 * # Bos-Coster (win=3)           :  462 ( 9)
 * # Bos-Coster (win=8)           :  466 (34)
 * # Bos-Coster (win=9)           :  464 (31)
 * # Yacobi                       :  478 (31)
 * # Bos-Coster (win=10)          :  473 (30)
 * # Bos-Coster (win=2)           :  486 ( 5)
 * # Bergeron-Berstel-Brlek-Duboc :  489 ( 5)
 */

#define RECIP_SQRT_MOD_BLS12_381_P(out, inp, ptype) do { \
ptype t[16]; \
vec_copy(t[13], inp, sizeof(ptype));/*    0: 1 */\
sqr(t[0], t[13]);                   /*    1: 2 */\
mul(t[8], t[0], t[13]);             /*    2: 3 */\
sqr(t[4], t[0]);                    /*    3: 4 */\
mul(t[1], t[8], t[0]);              /*    4: 5 */\
mul(t[6], t[4], t[8]);              /*    5: 7 */\
mul(t[9], t[1], t[4]);              /*    6: 9 */\
mul(t[12], t[6], t[4]);             /*    7: b */\
mul(t[3], t[9], t[4]);              /*    8: d */\
mul(t[7], t[12], t[4]);             /*    9: f */\
mul(t[15], t[3], t[4]);             /*   10: 11 */\
mul(t[10], t[7], t[4]);             /*   11: 13 */\
mul(t[2], t[15], t[4]);             /*   12: 15 */\
mul(t[11], t[10], t[4]);            /*   13: 17 */\
sqr(t[0], t[3]);                    /*   14: 1a */\
mul(t[14], t[11], t[4]);            /*   15: 1b */\
mul(t[5], t[0], t[8]);              /*   16: 1d */\
mul(t[4], t[0], t[1]);              /*   17: 1f */\
/* sqr(t[0], t[0]); */              /*   18: 34 */\
/* sqr(t[0], t[0]); */              /*   19: 68 */\
/* sqr(t[0], t[0]); */              /*   20: d0 */\
/* sqr(t[0], t[0]); */              /*   21: 1a0 */\
/* sqr(t[0], t[0]); */              /*   22: 340 */\
/* sqr(t[0], t[0]); */              /*   23: 680 */\
/* sqr(t[0], t[0]); */              /*   24: d00 */\
/* sqr(t[0], t[0]); */              /*   25: 1a00 */\
/* sqr(t[0], t[0]); */              /*   26: 3400 */\
/* sqr(t[0], t[0]); */              /*   27: 6800 */\
/* sqr(t[0], t[0]); */              /*   28: d000 */\
/* sqr(t[0], t[0]); */              /*   29: 1a000 */\
sqr_n_mul(t[0], t[0], 12, t[15]);   /*   30: 1a011 */\
/* sqr(t[0], t[0]); */              /*   31: 34022 */\
/* sqr(t[0], t[0]); */              /*   32: 68044 */\
/* sqr(t[0], t[0]); */              /*   33: d0088 */\
/* sqr(t[0], t[0]); */              /*   34: 1a0110 */\
/* sqr(t[0], t[0]); */              /*   35: 340220 */\
/* sqr(t[0], t[0]); */              /*   36: 680440 */\
/* sqr(t[0], t[0]); */              /*   37: d00880 */\
sqr_n_mul(t[0], t[0], 7, t[7]);     /*   38: d0088f */\
/* sqr(t[0], t[0]); */              /*   39: 1a0111e */\
/* sqr(t[0], t[0]); */              /*   40: 340223c */\
/* sqr(t[0], t[0]); */              /*   41: 6804478 */\
/* sqr(t[0], t[0]); */              /*   42: d0088f0 */\
sqr_n_mul(t[0], t[0], 4, t[1]);     /*   43: d0088f5 */\
/* sqr(t[0], t[0]); */              /*   44: 1a0111ea */\
/* sqr(t[0], t[0]); */              /*   45: 340223d4 */\
/* sqr(t[0], t[0]); */              /*   46: 680447a8 */\
/* sqr(t[0], t[0]); */              /*   47: d0088f50 */\
/* sqr(t[0], t[0]); */              /*   48: 1a0111ea0 */\
/* sqr(t[0], t[0]); */              /*   49: 340223d40 */\
sqr_n_mul(t[0], t[0], 6, t[6]);     /*   50: 340223d47 */\
/* sqr(t[0], t[0]); */              /*   51: 680447a8e */\
/* sqr(t[0], t[0]); */              /*   52: d0088f51c */\
/* sqr(t[0], t[0]); */              /*   53: 1a0111ea38 */\
/* sqr(t[0], t[0]); */              /*   54: 340223d470 */\
/* sqr(t[0], t[0]); */              /*   55: 680447a8e0 */\
/* sqr(t[0], t[0]); */              /*   56: d0088f51c0 */\
/* sqr(t[0], t[0]); */              /*   57: 1a0111ea380 */\
sqr_n_mul(t[0], t[0], 7, t[11]);    /*   58: 1a0111ea397 */\
/* sqr(t[0], t[0]); */              /*   59: 340223d472e */\
/* sqr(t[0], t[0]); */              /*   60: 680447a8e5c */\
/* sqr(t[0], t[0]); */              /*   61: d0088f51cb8 */\
/* sqr(t[0], t[0]); */              /*   62: 1a0111ea3970 */\
/* sqr(t[0], t[0]); */              /*   63: 340223d472e0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*   64: 340223d472ff */\
/* sqr(t[0], t[0]); */              /*   65: 680447a8e5fe */\
/* sqr(t[0], t[0]); */              /*   66: d0088f51cbfc */\
sqr_n_mul(t[0], t[0], 2, t[8]);     /*   67: d0088f51cbff */\
/* sqr(t[0], t[0]); */              /*   68: 1a0111ea397fe */\
/* sqr(t[0], t[0]); */              /*   69: 340223d472ffc */\
/* sqr(t[0], t[0]); */              /*   70: 680447a8e5ff8 */\
/* sqr(t[0], t[0]); */              /*   71: d0088f51cbff0 */\
/* sqr(t[0], t[0]); */              /*   72: 1a0111ea397fe0 */\
/* sqr(t[0], t[0]); */              /*   73: 340223d472ffc0 */\
sqr_n_mul(t[0], t[0], 6, t[3]);     /*   74: 340223d472ffcd */\
/* sqr(t[0], t[0]); */              /*   75: 680447a8e5ff9a */\
/* sqr(t[0], t[0]); */              /*   76: d0088f51cbff34 */\
/* sqr(t[0], t[0]); */              /*   77: 1a0111ea397fe68 */\
/* sqr(t[0], t[0]); */              /*   78: 340223d472ffcd0 */\
/* sqr(t[0], t[0]); */              /*   79: 680447a8e5ff9a0 */\
/* sqr(t[0], t[0]); */              /*   80: d0088f51cbff340 */\
sqr_n_mul(t[0], t[0], 6, t[3]);     /*   81: d0088f51cbff34d */\
/* sqr(t[0], t[0]); */              /*   82: 1a0111ea397fe69a */\
/* sqr(t[0], t[0]); */              /*   83: 340223d472ffcd34 */\
/* sqr(t[0], t[0]); */              /*   84: 680447a8e5ff9a68 */\
/* sqr(t[0], t[0]); */              /*   85: d0088f51cbff34d0 */\
/* sqr(t[0], t[0]); */              /*   86: 1a0111ea397fe69a0 */\
/* sqr(t[0], t[0]); */              /*   87: 340223d472ffcd340 */\
sqr_n_mul(t[0], t[0], 6, t[9]);     /*   88: 340223d472ffcd349 */\
/* sqr(t[0], t[0]); */              /*   89: 680447a8e5ff9a692 */\
/* sqr(t[0], t[0]); */              /*   90: d0088f51cbff34d24 */\
/* sqr(t[0], t[0]); */              /*   91: 1a0111ea397fe69a48 */\
sqr_n_mul(t[0], t[0], 3, t[8]);     /*   92: 1a0111ea397fe69a4b */\
/* sqr(t[0], t[0]); */              /*   93: 340223d472ffcd3496 */\
/* sqr(t[0], t[0]); */              /*   94: 680447a8e5ff9a692c */\
/* sqr(t[0], t[0]); */              /*   95: d0088f51cbff34d258 */\
/* sqr(t[0], t[0]); */              /*   96: 1a0111ea397fe69a4b0 */\
/* sqr(t[0], t[0]); */              /*   97: 340223d472ffcd34960 */\
/* sqr(t[0], t[0]); */              /*   98: 680447a8e5ff9a692c0 */\
/* sqr(t[0], t[0]); */              /*   99: d0088f51cbff34d2580 */\
sqr_n_mul(t[0], t[0], 7, t[3]);     /*  100: d0088f51cbff34d258d */\
/* sqr(t[0], t[0]); */              /*  101: 1a0111ea397fe69a4b1a */\
/* sqr(t[0], t[0]); */              /*  102: 340223d472ffcd349634 */\
/* sqr(t[0], t[0]); */              /*  103: 680447a8e5ff9a692c68 */\
/* sqr(t[0], t[0]); */              /*  104: d0088f51cbff34d258d0 */\
sqr_n_mul(t[0], t[0], 4, t[3]);     /*  105: d0088f51cbff34d258dd */\
/* sqr(t[0], t[0]); */              /*  106: 1a0111ea397fe69a4b1ba */\
/* sqr(t[0], t[0]); */              /*  107: 340223d472ffcd3496374 */\
/* sqr(t[0], t[0]); */              /*  108: 680447a8e5ff9a692c6e8 */\
/* sqr(t[0], t[0]); */              /*  109: d0088f51cbff34d258dd0 */\
/* sqr(t[0], t[0]); */              /*  110: 1a0111ea397fe69a4b1ba0 */\
/* sqr(t[0], t[0]); */              /*  111: 340223d472ffcd34963740 */\
sqr_n_mul(t[0], t[0], 6, t[7]);     /*  112: 340223d472ffcd3496374f */\
/* sqr(t[0], t[0]); */              /*  113: 680447a8e5ff9a692c6e9e */\
/* sqr(t[0], t[0]); */              /*  114: d0088f51cbff34d258dd3c */\
/* sqr(t[0], t[0]); */              /*  115: 1a0111ea397fe69a4b1ba78 */\
/* sqr(t[0], t[0]); */              /*  116: 340223d472ffcd3496374f0 */\
/* sqr(t[0], t[0]); */              /*  117: 680447a8e5ff9a692c6e9e0 */\
/* sqr(t[0], t[0]); */              /*  118: d0088f51cbff34d258dd3c0 */\
sqr_n_mul(t[0], t[0], 6, t[14]);    /*  119: d0088f51cbff34d258dd3db */\
/* sqr(t[0], t[0]); */              /*  120: 1a0111ea397fe69a4b1ba7b6 */\
/* sqr(t[0], t[0]); */              /*  121: 340223d472ffcd3496374f6c */\
/* sqr(t[0], t[0]); */              /*  122: 680447a8e5ff9a692c6e9ed8 */\
sqr_n_mul(t[0], t[0], 3, t[13]);    /*  123: 680447a8e5ff9a692c6e9ed9 */\
/* sqr(t[0], t[0]); */              /*  124: d0088f51cbff34d258dd3db2 */\
/* sqr(t[0], t[0]); */              /*  125: 1a0111ea397fe69a4b1ba7b64 */\
/* sqr(t[0], t[0]); */              /*  126: 340223d472ffcd3496374f6c8 */\
/* sqr(t[0], t[0]); */              /*  127: 680447a8e5ff9a692c6e9ed90 */\
/* sqr(t[0], t[0]); */              /*  128: d0088f51cbff34d258dd3db20 */\
/* sqr(t[0], t[0]); */              /*  129: 1a0111ea397fe69a4b1ba7b640 */\
/* sqr(t[0], t[0]); */              /*  130: 340223d472ffcd3496374f6c80 */\
/* sqr(t[0], t[0]); */              /*  131: 680447a8e5ff9a692c6e9ed900 */\
sqr_n_mul(t[0], t[0], 8, t[3]);     /*  132: 680447a8e5ff9a692c6e9ed90d */\
/* sqr(t[0], t[0]); */              /*  133: d0088f51cbff34d258dd3db21a */\
/* sqr(t[0], t[0]); */              /*  134: 1a0111ea397fe69a4b1ba7b6434 */\
/* sqr(t[0], t[0]); */              /*  135: 340223d472ffcd3496374f6c868 */\
/* sqr(t[0], t[0]); */              /*  136: 680447a8e5ff9a692c6e9ed90d0 */\
/* sqr(t[0], t[0]); */              /*  137: d0088f51cbff34d258dd3db21a0 */\
/* sqr(t[0], t[0]); */              /*  138: 1a0111ea397fe69a4b1ba7b64340 */\
/* sqr(t[0], t[0]); */              /*  139: 340223d472ffcd3496374f6c8680 */\
sqr_n_mul(t[0], t[0], 7, t[11]);    /*  140: 340223d472ffcd3496374f6c8697 */\
/* sqr(t[0], t[0]); */              /*  141: 680447a8e5ff9a692c6e9ed90d2e */\
/* sqr(t[0], t[0]); */              /*  142: d0088f51cbff34d258dd3db21a5c */\
/* sqr(t[0], t[0]); */              /*  143: 1a0111ea397fe69a4b1ba7b6434b8 */\
/* sqr(t[0], t[0]); */              /*  144: 340223d472ffcd3496374f6c86970 */\
/* sqr(t[0], t[0]); */              /*  145: 680447a8e5ff9a692c6e9ed90d2e0 */\
sqr_n_mul(t[0], t[0], 5, t[12]);    /*  146: 680447a8e5ff9a692c6e9ed90d2eb */\
/* sqr(t[0], t[0]); */              /*  147: d0088f51cbff34d258dd3db21a5d6 */\
/* sqr(t[0], t[0]); */              /*  148: 1a0111ea397fe69a4b1ba7b6434bac */\
/* sqr(t[0], t[0]); */              /*  149: 340223d472ffcd3496374f6c869758 */\
/* sqr(t[0], t[0]); */              /*  150: 680447a8e5ff9a692c6e9ed90d2eb0 */\
/* sqr(t[0], t[0]); */              /*  151: d0088f51cbff34d258dd3db21a5d60 */\
/* sqr(t[0], t[0]); */              /*  152: 1a0111ea397fe69a4b1ba7b6434bac0 */\
sqr_n_mul(t[0], t[0], 6, t[3]);     /*  153: 1a0111ea397fe69a4b1ba7b6434bacd */\
/* sqr(t[0], t[0]); */              /*  154: 340223d472ffcd3496374f6c869759a */\
/* sqr(t[0], t[0]); */              /*  155: 680447a8e5ff9a692c6e9ed90d2eb34 */\
/* sqr(t[0], t[0]); */              /*  156: d0088f51cbff34d258dd3db21a5d668 */\
/* sqr(t[0], t[0]); */              /*  157: 1a0111ea397fe69a4b1ba7b6434bacd0 */\
/* sqr(t[0], t[0]); */              /*  158: 340223d472ffcd3496374f6c869759a0 */\
/* sqr(t[0], t[0]); */              /*  159: 680447a8e5ff9a692c6e9ed90d2eb340 */\
sqr_n_mul(t[0], t[0], 6, t[5]);     /*  160: 680447a8e5ff9a692c6e9ed90d2eb35d */\
/* sqr(t[0], t[0]); */              /*  161: d0088f51cbff34d258dd3db21a5d66ba */\
/* sqr(t[0], t[0]); */              /*  162: 1a0111ea397fe69a4b1ba7b6434bacd74 */\
/* sqr(t[0], t[0]); */              /*  163: 340223d472ffcd3496374f6c869759ae8 */\
/* sqr(t[0], t[0]); */              /*  164: 680447a8e5ff9a692c6e9ed90d2eb35d0 */\
sqr_n_mul(t[0], t[0], 4, t[9]);     /*  165: 680447a8e5ff9a692c6e9ed90d2eb35d9 */\
/* sqr(t[0], t[0]); */              /*  166: d0088f51cbff34d258dd3db21a5d66bb2 */\
/* sqr(t[0], t[0]); */              /*  167: 1a0111ea397fe69a4b1ba7b6434bacd764 */\
/* sqr(t[0], t[0]); */              /*  168: 340223d472ffcd3496374f6c869759aec8 */\
/* sqr(t[0], t[0]); */              /*  169: 680447a8e5ff9a692c6e9ed90d2eb35d90 */\
/* sqr(t[0], t[0]); */              /*  170: d0088f51cbff34d258dd3db21a5d66bb20 */\
/* sqr(t[0], t[0]); */              /*  171: 1a0111ea397fe69a4b1ba7b6434bacd7640 */\
/* sqr(t[0], t[0]); */              /*  172: 340223d472ffcd3496374f6c869759aec80 */\
/* sqr(t[0], t[0]); */              /*  173: 680447a8e5ff9a692c6e9ed90d2eb35d900 */\
sqr_n_mul(t[0], t[0], 8, t[5]);     /*  174: 680447a8e5ff9a692c6e9ed90d2eb35d91d */\
/* sqr(t[0], t[0]); */              /*  175: d0088f51cbff34d258dd3db21a5d66bb23a */\
/* sqr(t[0], t[0]); */              /*  176: 1a0111ea397fe69a4b1ba7b6434bacd76474 */\
/* sqr(t[0], t[0]); */              /*  177: 340223d472ffcd3496374f6c869759aec8e8 */\
/* sqr(t[0], t[0]); */              /*  178: 680447a8e5ff9a692c6e9ed90d2eb35d91d0 */\
sqr_n_mul(t[0], t[0], 4, t[3]);     /*  179: 680447a8e5ff9a692c6e9ed90d2eb35d91dd */\
/* sqr(t[0], t[0]); */              /*  180: d0088f51cbff34d258dd3db21a5d66bb23ba */\
/* sqr(t[0], t[0]); */              /*  181: 1a0111ea397fe69a4b1ba7b6434bacd764774 */\
/* sqr(t[0], t[0]); */              /*  182: 340223d472ffcd3496374f6c869759aec8ee8 */\
/* sqr(t[0], t[0]); */              /*  183: 680447a8e5ff9a692c6e9ed90d2eb35d91dd0 */\
/* sqr(t[0], t[0]); */              /*  184: d0088f51cbff34d258dd3db21a5d66bb23ba0 */\
/* sqr(t[0], t[0]); */              /*  185: 1a0111ea397fe69a4b1ba7b6434bacd7647740 */\
/* sqr(t[0], t[0]); */              /*  186: 340223d472ffcd3496374f6c869759aec8ee80 */\
sqr_n_mul(t[0], t[0], 7, t[11]);    /*  187: 340223d472ffcd3496374f6c869759aec8ee97 */\
/* sqr(t[0], t[0]); */              /*  188: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e */\
/* sqr(t[0], t[0]); */              /*  189: d0088f51cbff34d258dd3db21a5d66bb23ba5c */\
/* sqr(t[0], t[0]); */              /*  190: 1a0111ea397fe69a4b1ba7b6434bacd764774b8 */\
/* sqr(t[0], t[0]); */              /*  191: 340223d472ffcd3496374f6c869759aec8ee970 */\
/* sqr(t[0], t[0]); */              /*  192: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e0 */\
/* sqr(t[0], t[0]); */              /*  193: d0088f51cbff34d258dd3db21a5d66bb23ba5c0 */\
/* sqr(t[0], t[0]); */              /*  194: 1a0111ea397fe69a4b1ba7b6434bacd764774b80 */\
/* sqr(t[0], t[0]); */              /*  195: 340223d472ffcd3496374f6c869759aec8ee9700 */\
/* sqr(t[0], t[0]); */              /*  196: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e00 */\
sqr_n_mul(t[0], t[0], 9, t[10]);    /*  197: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13 */\
/* sqr(t[0], t[0]); */              /*  198: d0088f51cbff34d258dd3db21a5d66bb23ba5c26 */\
/* sqr(t[0], t[0]); */              /*  199: 1a0111ea397fe69a4b1ba7b6434bacd764774b84c */\
sqr_n_mul(t[0], t[0], 2, t[8]);     /*  200: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f */\
/* sqr(t[0], t[0]); */              /*  201: 340223d472ffcd3496374f6c869759aec8ee9709e */\
/* sqr(t[0], t[0]); */              /*  202: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13c */\
/* sqr(t[0], t[0]); */              /*  203: d0088f51cbff34d258dd3db21a5d66bb23ba5c278 */\
/* sqr(t[0], t[0]); */              /*  204: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f0 */\
/* sqr(t[0], t[0]); */              /*  205: 340223d472ffcd3496374f6c869759aec8ee9709e0 */\
sqr_n_mul(t[0], t[0], 5, t[6]);     /*  206: 340223d472ffcd3496374f6c869759aec8ee9709e7 */\
/* sqr(t[0], t[0]); */              /*  207: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce */\
/* sqr(t[0], t[0]); */              /*  208: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c */\
/* sqr(t[0], t[0]); */              /*  209: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38 */\
/* sqr(t[0], t[0]); */              /*  210: 340223d472ffcd3496374f6c869759aec8ee9709e70 */\
/* sqr(t[0], t[0]); */              /*  211: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce0 */\
/* sqr(t[0], t[0]); */              /*  212: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c0 */\
/* sqr(t[0], t[0]); */              /*  213: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f380 */\
sqr_n_mul(t[0], t[0], 7, t[1]);     /*  214: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f385 */\
/* sqr(t[0], t[0]); */              /*  215: 340223d472ffcd3496374f6c869759aec8ee9709e70a */\
/* sqr(t[0], t[0]); */              /*  216: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce14 */\
/* sqr(t[0], t[0]); */              /*  217: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c28 */\
/* sqr(t[0], t[0]); */              /*  218: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f3850 */\
/* sqr(t[0], t[0]); */              /*  219: 340223d472ffcd3496374f6c869759aec8ee9709e70a0 */\
/* sqr(t[0], t[0]); */              /*  220: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce140 */\
/* sqr(t[0], t[0]); */              /*  221: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c280 */\
sqr_n_mul(t[0], t[0], 7, t[9]);     /*  222: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c289 */\
/* sqr(t[0], t[0]); */              /*  223: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512 */\
/* sqr(t[0], t[0]); */              /*  224: 340223d472ffcd3496374f6c869759aec8ee9709e70a24 */\
/* sqr(t[0], t[0]); */              /*  225: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce1448 */\
/* sqr(t[0], t[0]); */              /*  226: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2890 */\
/* sqr(t[0], t[0]); */              /*  227: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f385120 */\
/* sqr(t[0], t[0]); */              /*  228: 340223d472ffcd3496374f6c869759aec8ee9709e70a240 */\
sqr_n_mul(t[0], t[0], 6, t[11]);    /*  229: 340223d472ffcd3496374f6c869759aec8ee9709e70a257 */\
/* sqr(t[0], t[0]); */              /*  230: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144ae */\
/* sqr(t[0], t[0]); */              /*  231: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895c */\
/* sqr(t[0], t[0]); */              /*  232: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512b8 */\
/* sqr(t[0], t[0]); */              /*  233: 340223d472ffcd3496374f6c869759aec8ee9709e70a2570 */\
/* sqr(t[0], t[0]); */              /*  234: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144ae0 */\
sqr_n_mul(t[0], t[0], 5, t[5]);     /*  235: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd */\
/* sqr(t[0], t[0]); */              /*  236: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fa */\
/* sqr(t[0], t[0]); */              /*  237: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf4 */\
/* sqr(t[0], t[0]); */              /*  238: 340223d472ffcd3496374f6c869759aec8ee9709e70a257e8 */\
/* sqr(t[0], t[0]); */              /*  239: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd0 */\
/* sqr(t[0], t[0]); */              /*  240: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fa0 */\
sqr_n_mul(t[0], t[0], 5, t[10]);    /*  241: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb3 */\
/* sqr(t[0], t[0]); */              /*  242: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf66 */\
/* sqr(t[0], t[0]); */              /*  243: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ecc */\
/* sqr(t[0], t[0]); */              /*  244: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd98 */\
/* sqr(t[0], t[0]); */              /*  245: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb30 */\
/* sqr(t[0], t[0]); */              /*  246: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf660 */\
sqr_n_mul(t[0], t[0], 5, t[10]);    /*  247: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf673 */\
/* sqr(t[0], t[0]); */              /*  248: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece6 */\
/* sqr(t[0], t[0]); */              /*  249: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc */\
/* sqr(t[0], t[0]); */              /*  250: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb398 */\
/* sqr(t[0], t[0]); */              /*  251: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730 */\
/* sqr(t[0], t[0]); */              /*  252: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece60 */\
/* sqr(t[0], t[0]); */              /*  253: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc0 */\
/* sqr(t[0], t[0]); */              /*  254: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb3980 */\
/* sqr(t[0], t[0]); */              /*  255: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf67300 */\
sqr_n_mul(t[0], t[0], 8, t[3]);     /*  256: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d */\
/* sqr(t[0], t[0]); */              /*  257: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a */\
/* sqr(t[0], t[0]); */              /*  258: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34 */\
/* sqr(t[0], t[0]); */              /*  259: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39868 */\
/* sqr(t[0], t[0]); */              /*  260: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d0 */\
/* sqr(t[0], t[0]); */              /*  261: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a0 */\
/* sqr(t[0], t[0]); */              /*  262: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc340 */\
/* sqr(t[0], t[0]); */              /*  263: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb398680 */\
sqr_n_mul(t[0], t[0], 7, t[2]);     /*  264: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb398695 */\
/* sqr(t[0], t[0]); */              /*  265: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a */\
/* sqr(t[0], t[0]); */              /*  266: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a54 */\
/* sqr(t[0], t[0]); */              /*  267: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a8 */\
/* sqr(t[0], t[0]); */              /*  268: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb3986950 */\
/* sqr(t[0], t[0]); */              /*  269: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0 */\
/* sqr(t[0], t[0]); */              /*  270: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a540 */\
/* sqr(t[0], t[0]); */              /*  271: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a80 */\
/* sqr(t[0], t[0]); */              /*  272: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869500 */\
/* sqr(t[0], t[0]); */              /*  273: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a00 */\
sqr_n_mul(t[0], t[0], 9, t[7]);     /*  274: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f */\
/* sqr(t[0], t[0]); */              /*  275: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541e */\
/* sqr(t[0], t[0]); */              /*  276: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83c */\
/* sqr(t[0], t[0]); */              /*  277: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb398695078 */\
/* sqr(t[0], t[0]); */              /*  278: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f0 */\
/* sqr(t[0], t[0]); */              /*  279: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541e0 */\
sqr_n_mul(t[0], t[0], 5, t[3]);     /*  280: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed */\
/* sqr(t[0], t[0]); */              /*  281: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83da */\
/* sqr(t[0], t[0]); */              /*  282: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b4 */\
/* sqr(t[0], t[0]); */              /*  283: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f68 */\
sqr_n_mul(t[0], t[0], 3, t[8]);     /*  284: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b */\
/* sqr(t[0], t[0]); */              /*  285: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed6 */\
/* sqr(t[0], t[0]); */              /*  286: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac */\
/* sqr(t[0], t[0]); */              /*  287: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b58 */\
/* sqr(t[0], t[0]); */              /*  288: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0 */\
/* sqr(t[0], t[0]); */              /*  289: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed60 */\
/* sqr(t[0], t[0]); */              /*  290: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac0 */\
/* sqr(t[0], t[0]); */              /*  291: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b580 */\
/* sqr(t[0], t[0]); */              /*  292: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b00 */\
sqr_n_mul(t[0], t[0], 8, t[7]);     /*  293: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f */\
/* sqr(t[0], t[0]); */              /*  294: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61e */\
/* sqr(t[0], t[0]); */              /*  295: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3c */\
/* sqr(t[0], t[0]); */              /*  296: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b5878 */\
sqr_n_mul(t[0], t[0], 3, t[8]);     /*  297: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b */\
/* sqr(t[0], t[0]); */              /*  298: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6 */\
/* sqr(t[0], t[0]); */              /*  299: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec */\
/* sqr(t[0], t[0]); */              /*  300: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8 */\
/* sqr(t[0], t[0]); */              /*  301: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b0 */\
/* sqr(t[0], t[0]); */              /*  302: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f60 */\
/* sqr(t[0], t[0]); */              /*  303: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec0 */\
/* sqr(t[0], t[0]); */              /*  304: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d80 */\
sqr_n_mul(t[0], t[0], 7, t[9]);     /*  305: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d89 */\
/* sqr(t[0], t[0]); */              /*  306: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b12 */\
/* sqr(t[0], t[0]); */              /*  307: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f624 */\
/* sqr(t[0], t[0]); */              /*  308: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec48 */\
/* sqr(t[0], t[0]); */              /*  309: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d890 */\
/* sqr(t[0], t[0]); */              /*  310: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120 */\
/* sqr(t[0], t[0]); */              /*  311: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6240 */\
/* sqr(t[0], t[0]); */              /*  312: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec480 */\
/* sqr(t[0], t[0]); */              /*  313: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8900 */\
/* sqr(t[0], t[0]); */              /*  314: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b1200 */\
sqr_n_mul(t[0], t[0], 9, t[7]);     /*  315: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f */\
/* sqr(t[0], t[0]); */              /*  316: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241e */\
/* sqr(t[0], t[0]); */              /*  317: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483c */\
/* sqr(t[0], t[0]); */              /*  318: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d89078 */\
/* sqr(t[0], t[0]); */              /*  319: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f0 */\
/* sqr(t[0], t[0]); */              /*  320: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241e0 */\
/* sqr(t[0], t[0]); */              /*  321: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483c0 */\
sqr_n_mul(t[0], t[0], 6, t[2]);     /*  322: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d5 */\
/* sqr(t[0], t[0]); */              /*  323: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aa */\
/* sqr(t[0], t[0]); */              /*  324: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f54 */\
/* sqr(t[0], t[0]); */              /*  325: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241ea8 */\
/* sqr(t[0], t[0]); */              /*  326: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d50 */\
/* sqr(t[0], t[0]); */              /*  327: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aa0 */\
/* sqr(t[0], t[0]); */              /*  328: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f540 */\
sqr_n_mul(t[0], t[0], 6, t[4]);     /*  329: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55f */\
/* sqr(t[0], t[0]); */              /*  330: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabe */\
/* sqr(t[0], t[0]); */              /*  331: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57c */\
/* sqr(t[0], t[0]); */              /*  332: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaf8 */\
/* sqr(t[0], t[0]); */              /*  333: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55f0 */\
/* sqr(t[0], t[0]); */              /*  334: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  335: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabff */\
/* sqr(t[0], t[0]); */              /*  336: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fe */\
/* sqr(t[0], t[0]); */              /*  337: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffc */\
/* sqr(t[0], t[0]); */              /*  338: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ff8 */\
/* sqr(t[0], t[0]); */              /*  339: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabff0 */\
/* sqr(t[0], t[0]); */              /*  340: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  341: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fff */\
/* sqr(t[0], t[0]); */              /*  342: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aafffe */\
/* sqr(t[0], t[0]); */              /*  343: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55fffc */\
/* sqr(t[0], t[0]); */              /*  344: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfff8 */\
/* sqr(t[0], t[0]); */              /*  345: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fff0 */\
sqr_n_mul(t[0], t[0], 4, t[3]);     /*  346: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd */\
/* sqr(t[0], t[0]); */              /*  347: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffa */\
/* sqr(t[0], t[0]); */              /*  348: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff4 */\
/* sqr(t[0], t[0]); */              /*  349: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffe8 */\
sqr_n_mul(t[0], t[0], 3, t[8]);     /*  350: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb */\
/* sqr(t[0], t[0]); */              /*  351: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd6 */\
/* sqr(t[0], t[0]); */              /*  352: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac */\
/* sqr(t[0], t[0]); */              /*  353: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58 */\
/* sqr(t[0], t[0]); */              /*  354: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb0 */\
/* sqr(t[0], t[0]); */              /*  355: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd60 */\
/* sqr(t[0], t[0]); */              /*  356: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac0 */\
/* sqr(t[0], t[0]); */              /*  357: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff580 */\
/* sqr(t[0], t[0]); */              /*  358: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb00 */\
sqr_n_mul(t[0], t[0], 8, t[2]);     /*  359: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb15 */\
/* sqr(t[0], t[0]); */              /*  360: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a */\
/* sqr(t[0], t[0]); */              /*  361: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54 */\
/* sqr(t[0], t[0]); */              /*  362: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a8 */\
/* sqr(t[0], t[0]); */              /*  363: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb150 */\
/* sqr(t[0], t[0]); */              /*  364: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a0 */\
/* sqr(t[0], t[0]); */              /*  365: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac540 */\
/* sqr(t[0], t[0]); */              /*  366: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a80 */\
sqr_n_mul(t[0], t[0], 7, t[4]);     /*  367: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9f */\
/* sqr(t[0], t[0]); */              /*  368: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153e */\
/* sqr(t[0], t[0]); */              /*  369: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7c */\
/* sqr(t[0], t[0]); */              /*  370: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54f8 */\
/* sqr(t[0], t[0]); */              /*  371: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9f0 */\
/* sqr(t[0], t[0]); */              /*  372: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153e0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  373: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ff */\
/* sqr(t[0], t[0]); */              /*  374: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7fe */\
/* sqr(t[0], t[0]); */              /*  375: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffc */\
/* sqr(t[0], t[0]); */              /*  376: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ff8 */\
/* sqr(t[0], t[0]); */              /*  377: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ff0 */\
/* sqr(t[0], t[0]); */              /*  378: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7fe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  379: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7fff */\
/* sqr(t[0], t[0]); */              /*  380: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54fffe */\
/* sqr(t[0], t[0]); */              /*  381: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9fffc */\
/* sqr(t[0], t[0]); */              /*  382: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153fff8 */\
/* sqr(t[0], t[0]); */              /*  383: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7fff0 */\
sqr_n_mul(t[0], t[0], 4, t[7]);     /*  384: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff */\
/* sqr(t[0], t[0]); */              /*  385: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffe */\
/* sqr(t[0], t[0]); */              /*  386: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffc */\
/* sqr(t[0], t[0]); */              /*  387: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffff8 */\
/* sqr(t[0], t[0]); */              /*  388: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff0 */\
sqr_n_mul(t[0], t[0], 4, t[6]);     /*  389: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff7 */\
/* sqr(t[0], t[0]); */              /*  390: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee */\
/* sqr(t[0], t[0]); */              /*  391: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdc */\
/* sqr(t[0], t[0]); */              /*  392: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb8 */\
/* sqr(t[0], t[0]); */              /*  393: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff70 */\
/* sqr(t[0], t[0]); */              /*  394: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee0 */\
/* sqr(t[0], t[0]); */              /*  395: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdc0 */\
/* sqr(t[0], t[0]); */              /*  396: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb80 */\
sqr_n_mul(t[0], t[0], 7, t[4]);     /*  397: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9f */\
/* sqr(t[0], t[0]); */              /*  398: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73e */\
/* sqr(t[0], t[0]); */              /*  399: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7c */\
/* sqr(t[0], t[0]); */              /*  400: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcf8 */\
/* sqr(t[0], t[0]); */              /*  401: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9f0 */\
/* sqr(t[0], t[0]); */              /*  402: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73e0 */\
sqr_n_mul(t[0], t[0], 5, t[5]);     /*  403: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fd */\
/* sqr(t[0], t[0]); */              /*  404: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fa */\
/* sqr(t[0], t[0]); */              /*  405: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff4 */\
/* sqr(t[0], t[0]); */              /*  406: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9fe8 */\
/* sqr(t[0], t[0]); */              /*  407: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fd0 */\
/* sqr(t[0], t[0]); */              /*  408: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fa0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  409: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbf */\
/* sqr(t[0], t[0]); */              /*  410: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7e */\
/* sqr(t[0], t[0]); */              /*  411: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9fefc */\
/* sqr(t[0], t[0]); */              /*  412: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdf8 */\
/* sqr(t[0], t[0]); */              /*  413: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbf0 */\
/* sqr(t[0], t[0]); */              /*  414: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7e0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  415: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7ff */\
/* sqr(t[0], t[0]); */              /*  416: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffe */\
/* sqr(t[0], t[0]); */              /*  417: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffc */\
/* sqr(t[0], t[0]); */              /*  418: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbff8 */\
/* sqr(t[0], t[0]); */              /*  419: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7ff0 */\
/* sqr(t[0], t[0]); */              /*  420: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  421: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffff */\
/* sqr(t[0], t[0]); */              /*  422: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdfffe */\
/* sqr(t[0], t[0]); */              /*  423: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbfffc */\
/* sqr(t[0], t[0]); */              /*  424: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fff8 */\
/* sqr(t[0], t[0]); */              /*  425: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffff0 */\
/* sqr(t[0], t[0]); */              /*  426: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdfffe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  427: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdfffff */\
/* sqr(t[0], t[0]); */              /*  428: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbffffe */\
/* sqr(t[0], t[0]); */              /*  429: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7ffffc */\
/* sqr(t[0], t[0]); */              /*  430: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9fefffff8 */\
/* sqr(t[0], t[0]); */              /*  431: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdfffff0 */\
/* sqr(t[0], t[0]); */              /*  432: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbffffe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  433: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbffffff */\
/* sqr(t[0], t[0]); */              /*  434: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffe */\
/* sqr(t[0], t[0]); */              /*  435: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffffffc */\
/* sqr(t[0], t[0]); */              /*  436: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffff8 */\
/* sqr(t[0], t[0]); */              /*  437: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbffffff0 */\
/* sqr(t[0], t[0]); */              /*  438: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffe0 */\
sqr_n_mul(t[0], t[0], 5, t[4]);     /*  439: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffff */\
/* sqr(t[0], t[0]); */              /*  440: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9fefffffffe */\
/* sqr(t[0], t[0]); */              /*  441: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdfffffffc */\
/* sqr(t[0], t[0]); */              /*  442: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbfffffff8 */\
/* sqr(t[0], t[0]); */              /*  443: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffff0 */\
sqr_n_mul(t[0], t[0], 4, t[3]);     /*  444: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffffd */\
/* sqr(t[0], t[0]); */              /*  445: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffffffffa */\
/* sqr(t[0], t[0]); */              /*  446: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffffff4 */\
/* sqr(t[0], t[0]); */              /*  447: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbfffffffe8 */\
/* sqr(t[0], t[0]); */              /*  448: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffffd0 */\
/* sqr(t[0], t[0]); */              /*  449: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffffffffa0 */\
/* sqr(t[0], t[0]); */              /*  450: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffffff40 */\
sqr_n_mul(t[0], t[0], 6, t[2]);     /*  451: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffffff55 */\
/* sqr(t[0], t[0]); */              /*  452: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbfffffffeaa */\
/* sqr(t[0], t[0]); */              /*  453: d0088f51cbff34d258dd3db21a5d66bb23ba5c279c2895fb39869507b587b120f55ffff58a9ffffdcff7fffffffd54 */\
/* sqr(t[0], t[0]); */              /*  454: 1a0111ea397fe69a4b1ba7b6434bacd764774b84f38512bf6730d2a0f6b0f6241eabfffeb153ffffb9feffffffffaa8 */\
/* sqr(t[0], t[0]); */              /*  455: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffffff550 */\
sqr_n_mul(t[0], t[0], 4, t[1]);     /*  456: 340223d472ffcd3496374f6c869759aec8ee9709e70a257ece61a541ed61ec483d57fffd62a7ffff73fdffffffff555 */\
sqr(out, t[0]);                     /*  457: 680447a8e5ff9a692c6e9ed90d2eb35d91dd2e13ce144afd9cc34a83dac3d8907aaffffac54ffffee7fbfffffffeaaa */\
} while(0)
