/*
Copyright 2015 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package args

import (
	"fmt"

	"github.com/spf13/pflag"

	"k8s.io/code-generator/cmd/client-gen/types"
)

type Args struct {
	// The directory for the generated results.
	OutputDir string

	// The Go import-path of the generated results.
	OutputPkg string

	// The boilerplate header for Go files.
	GoHeaderFile string

	// A sorted list of group versions to generate. For each of them the package path is found
	// in GroupVersionToInputPath.
	Groups []types.GroupVersions

	// Overrides for which types should be included in the client.
	IncludedTypesOverrides map[types.GroupVersion][]string

	// ClientsetName is the name of the clientset to be generated. It's
	// populated from command-line arguments.
	ClientsetName string
	// ClientsetAPIPath is the default API HTTP path for generated clients.
	ClientsetAPIPath string
	// ClientsetOnly determines if we should generate the clients for groups and
	// types along with the clientset. It's populated from command-line
	// arguments.
	ClientsetOnly bool
	// FakeClient determines if client-gen generates the fake clients.
	FakeClient bool
	// PluralExceptions specify list of exceptions used when pluralizing certain types.
	// For example 'Endpoints:Endpoints', otherwise the pluralizer will generate 'Endpointes'.
	PluralExceptions []string

	// ApplyConfigurationPackage is the package of apply builders generated by
	// applyconfiguration-gen.
	// If non-empty, Apply functions are generated for each type and reference the apply builders.
	// If empty (""), Apply functions are not generated.
	ApplyConfigurationPackage string

	// PrefersProtobuf determines if the generated clientset uses protobuf for API requests.
	PrefersProtobuf bool
}

func New() *Args {
	return &Args{
		ClientsetName:             "internalclientset",
		ClientsetAPIPath:          "/apis",
		ClientsetOnly:             false,
		FakeClient:                true,
		ApplyConfigurationPackage: "",
	}
}

func (args *Args) AddFlags(fs *pflag.FlagSet, inputBase string) {
	gvsBuilder := NewGroupVersionsBuilder(&args.Groups)
	fs.StringVar(&args.OutputDir, "output-dir", "",
		"the base directory under which to generate results")
	fs.StringVar(&args.OutputPkg, "output-pkg", args.OutputPkg,
		"the Go import-path of the generated results")
	fs.StringVar(&args.GoHeaderFile, "go-header-file", "",
		"the path to a file containing boilerplate header text; the string \"YEAR\" will be replaced with the current 4-digit year")
	fs.Var(NewGVPackagesValue(gvsBuilder, nil), "input",
		"group/versions that client-gen will generate clients for. At most one version per group is allowed. Specified in the format \"group1/version1,group2/version2...\".")
	fs.Var(NewGVTypesValue(&args.IncludedTypesOverrides, []string{}), "included-types-overrides",
		"list of group/version/type for which client should be generated. By default, client is generated for all types which have genclient in types.go. This overrides that. For each groupVersion in this list, only the types mentioned here will be included. The default check of genclient will be used for other group versions.")
	fs.Var(NewInputBasePathValue(gvsBuilder, inputBase), "input-base",
		"base path to look for the api group.")
	fs.StringVarP(&args.ClientsetName, "clientset-name", "n", args.ClientsetName,
		"the name of the generated clientset package.")
	fs.StringVarP(&args.ClientsetAPIPath, "clientset-api-path", "", args.ClientsetAPIPath,
		"the value of default API HTTP path, starting with / and without trailing /.")
	fs.BoolVar(&args.ClientsetOnly, "clientset-only", args.ClientsetOnly,
		"when set, client-gen only generates the clientset shell, without generating the individual typed clients")
	fs.BoolVar(&args.FakeClient, "fake-clientset", args.FakeClient,
		"when set, client-gen will generate the fake clientset that can be used in tests")
	fs.StringSliceVar(&args.PluralExceptions, "plural-exceptions", args.PluralExceptions,
		"list of comma separated plural exception definitions in Type:PluralizedType form")
	fs.StringVar(&args.ApplyConfigurationPackage, "apply-configuration-package", args.ApplyConfigurationPackage,
		"optional package of apply configurations, generated by applyconfiguration-gen, that are required to generate Apply functions for each type in the clientset. By default Apply functions are not generated.")
	fs.BoolVar(&args.PrefersProtobuf, "prefers-protobuf", args.PrefersProtobuf,
		"when set, client-gen will generate a clientset that uses protobuf for API requests")

	// support old flags
	fs.SetNormalizeFunc(mapFlagName("clientset-path", "output-pkg", fs.GetNormalizeFunc()))
}

func (args *Args) Validate() error {
	if len(args.OutputDir) == 0 {
		return fmt.Errorf("--output-dir must be specified")
	}
	if len(args.OutputPkg) == 0 {
		return fmt.Errorf("--output-pkg must be specified")
	}
	if len(args.ClientsetName) == 0 {
		return fmt.Errorf("--clientset-name must be specified")
	}
	if len(args.ClientsetAPIPath) == 0 {
		return fmt.Errorf("--clientset-api-path cannot be empty")
	}

	return nil
}

// GroupVersionPackages returns a map from GroupVersion to the package with the types.go.
func (args *Args) GroupVersionPackages() map[types.GroupVersion]string {
	res := map[types.GroupVersion]string{}
	for _, pkg := range args.Groups {
		for _, v := range pkg.Versions {
			res[types.GroupVersion{Group: pkg.Group, Version: v.Version}] = v.Package
		}
	}
	return res
}

func mapFlagName(from, to string, old func(fs *pflag.FlagSet, name string) pflag.NormalizedName) func(fs *pflag.FlagSet, name string) pflag.NormalizedName {
	return func(fs *pflag.FlagSet, name string) pflag.NormalizedName {
		if name == from {
			name = to
		}
		return old(fs, name)
	}
}
