#!/bin/sh
# Copyright (C) 2026 Greenbone AG
#
# SPDX-License-Identifier: GPL-3.0-or-later

set -eu

: "${POSTGRES_USER:=gvmd}"
: "${POSTGRES_DB:=gvmd}"
: "${POSTGRES_DATA:=/var/lib/postgresql}"
: "${POSTGRES_PASSWORD:=}"
: "${POSTGRES_HOST_AUTH_METHOD:=md5}"
: "${POSTGRES_CLUSTER:=main}"
: "${POSTGRES_SOCKET_DIR:=/var/run/postgresql}"
: "${POSTGRES_LISTEN_ADDRESSES:=*}"

rm -f "${POSTGRES_DATA}/started"

detect_existing_major() {
  for f in "${POSTGRES_DATA}"/*/"${POSTGRES_CLUSTER}"/PG_VERSION; do
    [ -f "$f" ] || continue
    cat "$f"
    return 0
  done
  return 1
}

detect_installed_major() {
  for v in 17 16 15 14 13; do
    [ -x "/usr/lib/postgresql/$v/bin/initdb" ] && { echo "$v"; return 0; }
  done
  return 1
}

POSTGRES_VERSION="${POSTGRES_VERSION:-}"
if [ -n "${POSTGRES_VERSION}" ]; then
  if [ ! -x "/usr/lib/postgresql/${POSTGRES_VERSION}/bin/initdb" ]; then
    echo "ERROR: POSTGRES_VERSION=${POSTGRES_VERSION} requested but initdb not found in /usr/lib/postgresql/${POSTGRES_VERSION}/bin" >&2
    exit 1
  fi
else
  EXISTING="$(detect_existing_major)"
  INSTALLED="$(detect_installed_major)"

  if [ -n "${EXISTING}" ]; then
    POSTGRES_VERSION="$EXISTING"
  else
    if [ -z "${INSTALLED}" ]; then
      echo "ERROR: no PostgreSQL initdb found (checked 13-17 under /usr/lib/postgresql/<major>/bin)" >&2
      exit 1
    fi
    POSTGRES_VERSION="$INSTALLED"
  fi

  if [ "${EXISTING}" -ne "${INSTALLED}" ]; then
    echo "ERROR: existing data is for PostgreSQL ${EXISTING} but installed version is ${INSTALLED}" >&2
    echo "ERROR: please update your compose file for a postgres database migration or" \
      "download the newest version from https://greenbone.github.io/docs/latest/_static/docker-compose.yml" >&2
    exit 1
  fi
fi

mkdir -p "${POSTGRES_SOCKET_DIR}"
chown postgres:postgres "${POSTGRES_SOCKET_DIR}"
chmod 2775 "${POSTGRES_SOCKET_DIR}"

CONF_DIR="/etc/postgresql/${POSTGRES_VERSION}/${POSTGRES_CLUSTER}"
if [ ! -d "${CONF_DIR}" ]; then
  pg_createcluster "${POSTGRES_VERSION}" "${POSTGRES_CLUSTER}" --start-conf=manual
fi

HBA="${CONF_DIR}/pg_hba.conf"
[ -f "${HBA}" ] || echo "# generated by container startup" > "${HBA}"

{
  echo "local   all             all                                     trust"
  echo "host    all             all             0.0.0.0/0               ${POSTGRES_HOST_AUTH_METHOD}"
  echo "host    all             all             ::/0                    ${POSTGRES_HOST_AUTH_METHOD}"
} > "${HBA}"

pg_ctlcluster \
  -o "-k ${POSTGRES_SOCKET_DIR}" \
  -o "-c listen_addresses='${POSTGRES_LISTEN_ADDRESSES}'" \
  "${POSTGRES_VERSION}" "${POSTGRES_CLUSTER}" start

psql_sock() {
  psql --host="${POSTGRES_SOCKET_DIR}" --no-psqlrc -v ON_ERROR_STOP=1 "$@"
}

i=0
while ! pg_isready -h "${POSTGRES_SOCKET_DIR}" -p 5432 -U postgres >/dev/null 2>&1; do
  i=$((i + 1))
  [ "$i" -lt 60 ] || { echo "Postgres did not become ready"; exit 1; }
  sleep 1
done

USER_EXISTS="$(psql_sock -d postgres -tAc "SELECT 1 FROM pg_roles WHERE rolname='${POSTGRES_USER}'" | tr -d '[:space:]')"
if [ -z "${USER_EXISTS}" ]; then
  createuser --host="${POSTGRES_SOCKET_DIR}" -DRS "${POSTGRES_USER}"
fi

if [ -n "${POSTGRES_PASSWORD}" ]; then
  ESCAPED_PASSWORD="$(printf "%s" "${POSTGRES_PASSWORD}" | sed "s/'/''/g")"
  psql_sock -d postgres -c "ALTER ROLE \"${POSTGRES_USER}\" PASSWORD '${ESCAPED_PASSWORD}';"
fi

DB_EXISTS="$(psql_sock -d postgres -tAc "SELECT 1 FROM pg_database WHERE datname='${POSTGRES_DB}'" | tr -d '[:space:]')"
if [ -z "${DB_EXISTS}" ]; then
  createdb --host="${POSTGRES_SOCKET_DIR}" -O "${POSTGRES_USER}" "${POSTGRES_DB}"
fi

psql_sock -d "${POSTGRES_DB}" -v ON_ERROR_STOP=1 <<'SQL'
DO $$
BEGIN
  IF NOT EXISTS (SELECT 1 FROM pg_roles WHERE rolname='dba') THEN
    CREATE ROLE dba WITH superuser noinherit;
  END IF;
END $$;
SQL
psql_sock -d "${POSTGRES_DB}" -c "GRANT dba TO \"${POSTGRES_USER}\";"

ensure_extension() {
  ext="$1"
  EXISTS="$(psql_sock -d "${POSTGRES_DB}" -tAc "SELECT 1 FROM pg_extension WHERE extname='${ext}'" | tr -d '[:space:]')"
  [ -n "${EXISTS}" ] || psql_sock -d "${POSTGRES_DB}" -c "CREATE EXTENSION \"${ext}\";"
}

cleanup() {
  rm -f "${POSTGRES_DATA}/started"
  # in case of unclean shutdowns remove the lock file
  rm -f "${POSTGRES_SOCKET_DIR}/.s.PGSQL.5432.lock"
}

ensure_extension "uuid-ossp"
ensure_extension "pgcrypto"
ensure_extension "pg-gvm"

pg_ctlcluster "${POSTGRES_VERSION}" "${POSTGRES_CLUSTER}" stop

touch "${POSTGRES_DATA}/started"
trap cleanup EXIT

exec pg_ctlcluster \
  -o "-k ${POSTGRES_SOCKET_DIR}" \
  -o "-c listen_addresses='${POSTGRES_LISTEN_ADDRESSES}'" \
  --foreground \
  "${POSTGRES_VERSION}" "${POSTGRES_CLUSTER}" start
