/* optimizer
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "treerep.h"
#include "treetools.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "parser.h"
#include "regalloc.h"
#include "eval.h"

/* incremented on each optimization,
  used to detect whether further optimization should be attempted */
int optimizations;

/* forward */
tree optimize_statement(tree p);

/* report whether the value of p is known */
boolean value_known(tree p)
{
    tree q;
    stack_guard;
    assert_pointer(NULL, p);
    q = follow(p);
    while (q->kind == node_ref)
        q = q->first;
    if (q->kind == node_const) {
        assert_pointer(NULL, q->value);
        return true;
    }
    return false;
}

/* optimize an op node */
tree optimize_op(tree p)
{
    stack_guard;
    assert_kind(p->loc, p, node_op);
    if (verbose_optimizer)
        log((m, "optimize op %d", p->nr));

    if (optimize_constant_folding) {
        /* replace operation on two constants by a constant */
        if (value_known(p->first)
            && (is_monop(p->op) || value_known(p->next))
            ) {
            optimizations++;
            if (verbose_optimizer) {
                log((m, "optimization : fold %d", p->nr));
            }
            evaluate(p);
            p = new_ref(p->loc, ref_const, new_const(p->value), NULL);
            return p;
        }
    }

    if (optimize_trivial_expressions) {

        /* replace monadic minus constant by (- constant) */
        if ((p->kind == node_op)
            && (p->op == op_mminus)
            && node_is_some_constant(p->first)
            ) {
            tree q = new_node(NULL, node_const);
#ifdef __DEBUG__
            trace;
#endif
            optimizations++;
            if (verbose_optimizer) {
                log((m, "optimization : replace minus constant %d", p->nr));
            }
            q->value->x = -p->first->value->x;
            p->op = op_plus;
            p->next = q;
#ifdef __DEBUG__
            trace;
#endif
            return p;
        }

        /* optimize trivial constant operators like +0, *1, ==true etc. */
        if ((p->next != NULL)
            && (p->next->kind == node_ref)
            && node_is_some_constant(p->next->first)
            ) {

            if ((p->op == op_plus)
                || (p->op == op_minus)
                || (p->op == op_shift_left)
                || (p->op == op_shift_right)
                ) {
                if (node_is_constant(p->next, 0)) {
#ifdef __DEBUG__
                    trace;
#endif
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : removed <op> 0 %d", p->nr));
                    }
                    return p->first;
                }
            } else if ((p->op == op_times || p->op == op_divide)) {
                if (node_is_constant(p->next, 1)) {
#ifdef __DEBUG__
                    trace;
#endif
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : removed *1 or /1 %d", p->nr));
                    }
                    return p->first;
                }
            } else if ((p->op == op_times)
                       && (node_is_constant(p->next, 0))
                ) {
#ifdef __DEBUG__
                trace;
#endif
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : folded times 0 %d", p->nr));
                }
                return p->next;
            } else if ((p->type == type_bit)
                       && (p->op == op_and)) {
#ifdef __DEBUG__
                trace;
#endif
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : folded and %d %d", p->nr, p->next->value));
                }
                if (node_is_constant(p->next, 1)) {
                    return p->first;
                } else {
                    return p->next;
                }
            } else if ((p->type == type_bit)
                       && (p->op == op_or)) {
#ifdef __DEBUG__
                trace;
#endif
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : folded or %d %d", p->nr, p->next->value));
                }
                if (node_is_constant(p->next, 1)) {
                    return p->next;
                } else {
                    return p->first;
                }
            } else if ((p->first->type == type_bit)
                       && ((p->op == op_equal)
                           || (p->op == op_not_equal))
                ) {
#ifdef __DEBUG__
                trace;
#endif
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : folded == %d %s %d", p->nr, op_name[p->op],
                         p->next->value));
                }
                if ((p->op == op_equal) == node_is_constant(p->next, 1)) {
                    return p->first;
                } else {
                    return new_op(p->loc, type_bit, op_mnot, p->first, NULL);
                }
            }
        }
    }

    if (optimize_strength_reduction && ((p->op == op_divide)
                                        || (p->op == op_times)
                                        || (p->op == op_modulo)
        )
        ) {
        int i;
        int factors[] = { 0, 2, 4, 8, 16, 32, 64, 128 };
        for (i = 1; i < 8; i++) {
            if (node_is_constant(p->next, factors[i])) {
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : reduced strength %d", p->nr));
                }
                if (p->op == op_modulo) {
                    p->next =
                        new_ref(p->loc, ref_const, new_const(new_value(type_byte, factors[i] - 1)),
                                NULL);
                    p->op = op_and;
                } else {
                    p->next = new_ref(p->loc, ref_const, new_const(new_value(type_byte, i)), NULL);
                    if (p->op == op_divide) {
                        p->op = op_shift_right;
                    } else {
                        p->op = op_shift_left;
                    }
                }
                p->impl = find_operator(p->op, p->first, p->next);
                return p;
            }
        }
    }

    if (optimize_tree_shape) {

        /* move a constant to the right side */
        if (node_is_some_constant(p->first)) {
            int newop = op_inverted[p->op];
            if (newop != 0) {
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : constant moved to right side %d", p->nr));
                }
                swapp(p->first, p->next);
                return p;
            }
        }

        /* for commuting operators: migrate a constant down
           ((a op N) op b) becomes (a op (b op N)) */
#ifdef __DEBUG__
        if (                    /* could cause infinite looping! */
               op_commutates[p->op]
               && (p->first->kind == node_op)
               && (p->first->op == p->op)
               && (node_is_some_constant(p->first->next))
            ) {
            optimizations++;
            if (verbose_optimizer) {
                log((m, "optimization : migrate constant down %d", p->nr));
            }
            swapp(p->first, p->next);
            swapp(p->first, p->next->first);
            return p;
        }
#endif

        /* for commuting operators: migrate deeper expression to the left */
        if (op_commutates[p->op]
            && (expression_depth(p->first) < expression_depth(p->next))
            ) {
            optimizations++;
            if (verbose_optimizer) {
                log((m, "optimization : deeper expression to left side %d", p->nr));
            }
            swapp(p->first, p->next);
        }
    }

    return p;
}

/* return a possibly optimized expression node p */
tree optimize_expression(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    if (verbose_optimizer)
        log((m, "optimize expression %d", p->nr));

    switch (p->kind) {

        /* nodes which can not be optimized yet */
    case node_ref:
    case node_var:
    case node_w:
    case node_const:{
            break;
        }

        /* operator : first optimize subexpresions */
    case node_op:{
            p->first = optimize_expression(p->first);
            if (p->next != NULL)
                p->next = optimize_expression(p->next);
            if (verbose_optimizer) {
                log((m, "optimize expression continue %d", p->nr));
            }
            p = optimize_op(p);
            break;
        }

        /* a chain can insert preparation statements */
    case node_chain:{
            p->first = optimize_statement(p->first);
            if (p->next != NULL) {
                if (p->next->kind == node_chain) {
                    p->next = optimize_statement(p->next);
                } else {
                    p->next = optimize_expression(p->next);
                }
            }
            break;
        }

        /* no other nodes should appear in an expression */
    default:{
            snark_node(p->loc, p);
            break;
        }
    }
    return p;
}

/* general statement optimization */
tree optimize_statement(tree p)
{
    stack_guard;
    if (p == NULL)
        return NULL;
    if (verbose_optimizer) {
        log((m, "optimize statement %d %s", p->nr, node_name[p->kind]));
    }

    switch (p->kind) {

        /* nodes which can not be optimized */
    case node_precall:
    case node_call:
    case node_test:
    case node_type:
    case node_const:
    case node_label:
    case node_return:
    case node_error:
    case node_var:
    case node_ref:
    case node_w:
    case node_org:
    case node_asm:{
            break;
        }

    case node_procedure:{
            p->first = optimize_statement(p->first);

            /* tail call? */
            if ((optimize_tail_calls)
                && (!p->is_interrupt)
                && (!p->is_chained)
                ) {

                /* find last statement, just before the end label */
                tree q = p->first;
#ifdef __DEBUG__
                trace_subtree(q);
#endif
                while ((q != NULL)
                       && (q->kind == node_chain)
                       && (q->next != NULL)
                       && (q->next->next != NULL)
                    ) {
                    q = q->next;
                }
#ifdef __DEBUG__
                trace_subtree(q)
#endif
                    /* follow a chain's first */
                    while ((q != NULL)
                           && (q->kind == node_chain)
                    ) {
                    q = q->first;
                    while ((q != NULL)
                           && (q->kind == node_chain)
                           && (q->next != NULL)
                        ) {
                        q = q->next;
                    }
                }
#ifdef __DEBUG__
                trace_subtree(q)
#endif
                    /* is it a not-yet-chained call? */
                    if ((q != NULL)
                        && (q->kind == node_call)
                        && (!q->is_chained)
                    ) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : tail call %d %d", p->nr, q->nr));
                    }
                    p->is_chained = true;
                    q->is_chained = true;
                }
                if ((q != NULL)
                    && (q->kind == node_return)
                    && (!q->is_chained)
                    ) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : tail return %d %d", p->nr, q->nr));
                    }
                    q->is_chained = true;
                }
            }
            break;
        }

        /* just a first to optimize */
    case node_decl:{
            p->first = optimize_statement(p->first);
            break;
        }

        /* optimize the statement and its continuation */
    case node_chain:{
            p->first = optimize_statement(p->first);
            p->next = optimize_statement(p->next);
            if (verbose_optimizer) {
                log((m, "optimize statement continue %d", p->nr));
            }

            /* remove an empty statement */
            if (optimize_dead_code) {

                if (p->first == NULL) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : chain reduced to next %d", p->nr));
                    }
                    return p->next;
                }
#ifdef __DEBUG__
                else {
/* this causes problems with the register allocation! */
                    tree q = follow(p->first);
                    if ((q->kind == node_chain)
                        && (q->next == NULL)) {
                        optimizations++;
                        if (verbose_optimizer) {
                            log((m, "optimization : chain sunk to lower level %d", p->nr));
                        }
                        q->next = p->next;
                        p->next = p->first;
                        p->first = NULL;
                    }
                }
#endif

                /* be carefull not to violate the chain-based structure */
                if ((p->next == NULL)
                    && (p->first != NULL)
                    && (p->first->kind == node_chain)
                    ) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : chain reduced to first %d", p->nr));
                    }
                    return p->first;
                }
            }
            break;
        }

        /* optimize the expression in an assignment */
    case node_assign:{
            if (verbose_optimizer) {
                log((m, "optimize statement continue %d", p->nr));
            }
            p->next = optimize_expression(p->next);

            if ((p->next->kind == node_ref)
                && (p->next->first == p->first->first)
                && (true)
                ) {
                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimize trivial assignment %d", p->nr));
                }
                return NULL;
            }

            break;
        }

        /* optimize the condition and the blocks in an if */
    case node_if:{
            p->condition = optimize_expression(p->condition);
            p->first = optimize_statement(p->first);
            p->next = optimize_statement(p->next);
            if (verbose_optimizer) {
                log((m, "optimize statement continue %d", p->nr));
            }

            if (optimize_dead_code) {

                /* inline when the condition is a constant */
                if (node_is_constant(p->condition, 1)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization: constant if folded %d", p->nr));
                    }
                    return p->first;
                }
                if (node_is_constant(p->condition, 0)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : constant if folded %d", p->nr));
                    }
                    return p->next;
                }

                /* remove the if when there is neither then nor else body */
                if ((p->first == NULL)
                    && (p->next == NULL)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : empty if removed %d", p->nr));
                    }
                    return NULL;
                }

            }
            break;
        }

        /* optimize the condition and the block in a while */
    case node_while:{
            cassert(p->next == NULL);
            p->condition = optimize_expression(p->condition);
            p->first = optimize_statement(p->first);
            if (verbose_optimizer) {
                log((m, "optimize statement continue %d", p->nr));
            }

            if (optimize_dead_code) {

                /* remove when the condition is constant false */
                if (node_is_constant(p->condition, 0)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : dead while removed %d", p->nr));
                    }
                    p = NULL;
                }
            }

            break;
        }

        /* optimize all parts of a for */
    case node_for:{

            if (p->start != NULL)
                p->start = optimize_expression(p->start);
            if (p->step != NULL)
                p->step = optimize_expression(p->step);
            p->end = optimize_expression(p->end);
            p->first = optimize_statement(p->first);

            if (optimize_dead_code && (p->first == NULL)) {

                optimizations++;
                if (verbose_optimizer) {
                    log((m, "optimization : empty for removed %d", p->nr));
                }
                p = NULL;

            } else if (optimize_dead_code && (p->start == NULL)
                       && (p->step == NULL)
                ) {

                if (node_is_constant(p->end, 0)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : dead for removed %d", p->nr));
                    }
                    p = NULL;

                } else if (node_is_constant(p->end, 1)) {
                    optimizations++;
                    if (verbose_optimizer) {
                        log((m, "optimization : singleton for inlined %d", p->nr));
                    }
                    p = p->first;
                }
            }

            break;
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }
    return p;
}

/* remove unused declarations */
void remove_unused(tree p)
{
    stack_guard;
    while (p != NULL) {
        assert_pointer(NULL, p);
        if (verbose_optimizer) {
            log((m, "attempt remove %d %s", p->nr, node_name[p->kind]));
        }
        switch (p->kind) {

            /* nodes which are not interesting */
        case node_test:
        case node_type:
        case node_const:
        case node_value:
        case node_return:
        case node_precall:
        case node_error:
        case node_w:
        case node_org:
        case node_label:{
                break;
            }

            /* uses its subject */
        case node_asm:{
                if ((p->opcode == opcode_goto)
                    || (p->opcode == opcode_call)) {
                    assert_pointer(p->loc, p->first);
#ifdef __DEBUG__
                    trace_subtree(p);
                    assert_kind(p->loc, p->first, node_ref);
                    log((m, "loca=%s", location_string(p->loc)));
#endif
                    if (p->first->kind == node_ref) {
                        assert_pointer(p->loc, p->first->first);
                        if (p->first->first->kind == node_procedure) {
                            p->first->first->used++;
                        }
                    }
                }
                /* fallthrough */
            }

            /* nodes which must be explored further */
        case node_procedure:
        case node_assign:
        case node_decl:
        case node_if:
        case node_while:
        case node_for:{
                remove_unused(p->first);
                remove_unused(p->start);
                remove_unused(p->step);
                remove_unused(p->end);
                remove_unused(p->condition);
                remove_unused(p->type);
                break;
            }

            /* nodes which use their subject */
        case node_var:{
                if (p->master1 != NULL) {
                    assert_kind(p->loc, p->master1, node_var);
                    p->master1->used++;
                }
                break;
            }
        case node_call:{
                assert_pointer(p->loc, p->first);
                p->first->used++;
                break;
            }
        case node_ref:{
                assert_pointer(p->loc, p->first);
                p->first->used++;
                if (p->indirect) {
                    if (p->first->put != NULL) {
                        p->first->put->used++;
                    }
                    if (p->first->get != NULL) {
                        p->first->get->used++;
                    }
                }
                if (p->is_target) {
                    if (p->first->put != NULL) {
                        p->first->put->used++;
                    }
                } else {
                    if (p->first->get != NULL) {
                        p->first->get->used++;
                    }
                }
                break;
            }
        case node_op:{
                if (p->impl != NULL) {
                    p->impl->used++;
                }
                remove_unused(p->first);
                break;
            }

            /* declarations */
        case node_chain:{

                /* declaration */
                if ((p->first != NULL)
                    && (p->first->kind == node_decl)
                    && (p->first->first != NULL)
                    && (!p->first->first->is_argument)
                    ) {
                    tree q = p->first->first;
                    assert_pointer(p->loc, q);
                    q->used = false;
                    remove_unused(p->next);
                    if ((q->is_interrupt)
                        || (q->used > 0)
                        || (q->sacred)
                        ) {
                        remove_unused(p->first);
                    } else {
                        if (verbose_optimizer) {
                            log((m, "remove unused %d", p->first->nr));
                        }
                        p->first = NULL;
                        optimizations++;
                    }
                    p = NULL;
                } else {
                    remove_unused(p->first);
                }
                break;
            }

        default:{
                snark_node(p->loc, p);
                break;
            }
        }
        if (p != NULL)
            p = p->next;
    }
}

/* copy var properties */
void propagation(tree p)
{
    stack_guard;
    if (p == NULL)
        return;
    assert_pointer(NULL, p);
    if (verbose_optimizer) {
        log((m, "propagation %d %s", p->nr, node_name[p->kind]));
    }

    switch (p->kind) {

    case node_var:{
            if (p->uncle != NULL) {
                if (p->get == NULL) {
                    p->get = p->uncle->get;
                }
                if (p->put == NULL) {
                    p->put = p->uncle->put;
                }
                if (p->uncle->is_volatile) {
                    p->is_volatile = true;
                }
                p->fixed = true;
                p->master1 = p->uncle;
                p->master2 = p->uncle;
            }
            break;
        }

    case node_ref:
    case node_precall:
    case node_call:
    case node_test:
    case node_type:
    case node_const:
    case node_label:
    case node_return:
    case node_error:
    case node_w:
    case node_org:
    case node_asm:{
            break;
        }

    case node_decl:
    case node_procedure:{
            propagation(p->first);
            break;
        }

    case node_op:
    case node_assign:
    case node_chain:{
            propagation(p->first);
            propagation(p->next);
            break;
        }

    case node_for:
    case node_while:
    case node_if:{
            propagation(p->condition);
            propagation(p->first);
            propagation(p->next);
            propagation(p->start);
            propagation(p->step);
            propagation(p->end);
            break;
        }

    default:{
            snark(NULL);
            snark_node(p->loc, p);
            break;
        }
    }
}

void show_optimization_settings(void)
{
    log((m, "optimize:"));
    log((m, "   constant_folding    = %d", optimize_constant_folding));
    log((m, "   strength_reduction  = %d", optimize_strength_reduction));
    log((m, "   tree_shape          = %d", optimize_tree_shape));
    log((m, "   tail_calls          = %d", optimize_tail_calls));
    log((m, "   trivial_expressions = %d", optimize_trivial_expressions));
    log((m, "   dead_code           = %d", optimize_dead_code));
}

void optimize1(tree * p)
{
    stack_guard;

    propagation(*p);

    if (verbose_optimizer) {
        show_optimization_settings();
    }

    optimizations = 1;
    while (optimizations) {
        optimizations = 0;
        *p = optimize_statement(*p);
        remove_unused(*p);
        if (verbose_optimizer) {
            log((m, "optimization 1 found %d case(s)", optimizations));
        }
    }
}

void optimize2(tree * p)
{
    stack_guard;
#ifdef __DEBUG__
    optimize_constant_folding = false;
    optimize_strength_reduction = false;
    optimize_tree_shape = false;
    optimize_tail_calls = false;
    optimize_trivial_expressions = false;
    optimize_dead_code = false;
#endif

    if (verbose_optimizer) {
        show_optimization_settings();
    }

    optimizations = 1;
    while (optimizations) {
        optimizations = 0;
        *p = optimize_statement(*p);
        remove_unused(*p);
        if (verbose_optimizer) {
            log((m, "optimization 2 found %d case(s)", optimizations));
        }
    }
}
