import { IIterator } from '@lumino/algorithm';
import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '..';
import * as Kernel from './kernel';
import { BaseManager } from '../basemanager';
import { IKernelOptions } from './restapi';
/**
 * An implementation of a kernel manager.
 */
export declare class KernelManager extends BaseManager implements Kernel.IManager {
    /**
     * Construct a new kernel manager.
     *
     * @param options - The default options for kernel.
     */
    constructor(options?: KernelManager.IOptions);
    /**
     * The server settings for the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * A signal emitted when the running kernels change.
     */
    get runningChanged(): ISignal<this, Kernel.IModel[]>;
    /**
     * A signal emitted when there is a connection failure.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * Connect to an existing kernel.
     *
     * @returns The new kernel connection.
     *
     * #### Notes
     * This will use the manager's server settings and ignore any server
     * settings passed in the options.
     */
    connectTo(options: Omit<Kernel.IKernelConnection.IOptions, 'serverSettings'>): Kernel.IKernelConnection;
    /**
     * Create an iterator over the most recent running kernels.
     *
     * @returns A new iterator over the running kernels.
     */
    running(): IIterator<Kernel.IModel>;
    /**
     * Force a refresh of the running kernels.
     *
     * @returns A promise that resolves when the running list has been refreshed.
     *
     * #### Notes
     * This is not typically meant to be called by the user, since the
     * manager maintains its own internal state.
     */
    refreshRunning(): Promise<void>;
    /**
     * Start a new kernel.
     *
     * @param createOptions - The kernel creation options
     *
     * @param connectOptions - The kernel connection options
     *
     * @returns A promise that resolves with the kernel connection.
     *
     * #### Notes
     * The manager `serverSettings` will be always be used.
     */
    startNew(createOptions?: IKernelOptions, connectOptions?: Omit<Kernel.IKernelConnection.IOptions, 'model' | 'serverSettings'>): Promise<Kernel.IKernelConnection>;
    /**
     * Shut down a kernel by id.
     *
     * @param id - The id of the target kernel.
     *
     * @returns A promise that resolves when the operation is complete.
     */
    shutdown(id: string): Promise<void>;
    /**
     * Shut down all kernels.
     *
     * @returns A promise that resolves when all of the kernels are shut down.
     */
    shutdownAll(): Promise<void>;
    /**
     * Find a kernel by id.
     *
     * @param id - The id of the target kernel.
     *
     * @returns A promise that resolves with the kernel's model.
     */
    findById(id: string): Promise<Kernel.IModel | undefined>;
    /**
     * Execute a request to the server to poll running kernels and update state.
     */
    protected requestRunning(): Promise<void>;
    /**
     * Handle a kernel starting.
     */
    private _onStarted;
    private _onDisposed;
    private _onStatusChanged;
    private _isReady;
    private _ready;
    private _kernelConnections;
    private _models;
    private _pollModels;
    private _runningChanged;
    private _connectionFailure;
}
/**
 * The namespace for `KernelManager` class statics.
 */
export declare namespace KernelManager {
    /**
     * The options used to initialize a KernelManager.
     */
    interface IOptions extends BaseManager.IOptions {
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby;
    }
}
