import { ServerConnection } from '../serverconnection';
/**
 * The kernel model provided by the server.
 *
 * #### Notes
 * See the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels).
 */
export interface IModel {
    /**
     * Unique identifier of the kernel on the server.
     */
    readonly id: string;
    /**
     * The name of the kernel.
     */
    readonly name: string;
    /**
     * The kernel execution state.
     */
    readonly execution_state?: string;
    /**
     * The timestamp of the last activity on the kernel.
     */
    readonly last_activity?: string;
    /**
     * The number of active connections to the kernel.
     */
    readonly connections?: number;
    /**
     * The reason the kernel died, if applicable.
     */
    readonly reason?: string;
    /**
     * The traceback for a dead kernel, if applicable.
     */
    readonly traceback?: string;
}
/**
 * The url for the kernel service.
 */
export declare const KERNEL_SERVICE_URL = "api/kernels";
/**
 * Fetch the running kernels.
 *
 * @param settings - The optional server settings.
 *
 * @returns A promise that resolves with the list of running kernels.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
export declare function listRunning(settings?: ServerConnection.ISettings): Promise<IModel[]>;
/**
 * Start a new kernel.
 *
 * @param options - The options used to create the kernel.
 *
 * @returns A promise that resolves with a kernel connection object.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
export declare function startNew(options?: IKernelOptions, settings?: ServerConnection.ISettings): Promise<IModel>;
/**
 * The options object used to initialize a kernel.
 */
export declare type IKernelOptions = Partial<Pick<IModel, 'name'>>;
/**
 * Restart a kernel.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response (and thus after a restart) and rejected otherwise.
 */
export declare function restartKernel(id: string, settings?: ServerConnection.ISettings): Promise<void>;
/**
 * Interrupt a kernel.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
export declare function interruptKernel(id: string, settings?: ServerConnection.ISettings): Promise<void>;
/**
 * Shut down a kernel.
 *
 * @param id - The id of the running kernel.
 *
 * @param settings - The server settings for the request.
 *
 * @returns A promise that resolves when the kernel is shut down.
 *
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
export declare function shutdownKernel(id: string, settings?: ServerConnection.ISettings): Promise<void>;
/**
 * Get a full kernel model from the server by kernel id string.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
export declare function getKernelModel(id: string, settings?: ServerConnection.ISettings): Promise<IModel | undefined>;
