/* Copyright (C) 2023 Alif Semiconductor - All Rights Reserved.
 * Use, distribution and modification of this code is permitted under the
 * terms stated in the Alif Semiconductor Software License Agreement
 *
 * You should have received a copy of the Alif Semiconductor Software
 * License Agreement with this file. If not, please write to:
 * contact@alifsemi.com, or visit: https://alifsemi.com/license
 *
 */
/**************************************************************************//**
 * @file     arx3A0_camera_sensor.c
 * @author   Prasanna Ravi
 * @email    prasanna.ravi@alifsemi.com
 * @version  V1.0.0
 * @date     19-April-2023
 * @brief    ONsemi ARX3A0 Camera Sensor driver.
 * @bug      None.
 * @Note     None.
 ******************************************************************************/
/* System Includes */
#include "RTE_Device.h"
#include "RTE_Components.h"
#include CMSIS_device_header
#include "Camera_Sensor.h"
#include "Camera_Sensor_i2c.h"
#include "Driver_GPIO.h"
#include "Driver_CPI.h"
#include "sys_ctrl_cpi.h"

/* Proceed only if ARX3A0 Camera Sensor is enabled. */
#if RTE_ARX3A0_CAMERA_SENSOR_CSI_ENABLE

/* I2C Instance */
#if(RTE_ARX3A0_CAMERA_SENSOR_I2C_INSTANCE == 4)
#define CAMERA_SENSOR_I2C_INSTANCE                           I3C
#else
#define CAMERA_SENSOR_I2C_INSTANCE                           RTE_ARX3A0_CAMERA_SENSOR_I2C_INSTANCE
#endif

/* ARX3A0 Camera Sensor Slave Address. */
#define ARX3A0_CAMERA_SENSOR_SLAVE_ADDR                      0x36

/* ARX3A0 Camera Sensor CHIP-ID registers */
#define ARX3A0_CHIP_ID_REGISTER                              0x3000
#define ARX3A0_CHIP_ID_REGISTER_VALUE                        0x0353

/* ARX3A0 Camera Sensor registers index */
#define ARX3A0_SOFTWARE_RESET_REGISTER                       0x0103
#define ARX3A0_MODE_SELECT_REGISTER                          0x0100
#define ARX3A0_MIPI_CONFIG_REGISTER                          0x31BE
#define ARX3A0_RESET_REGISTER                                0x301A
#define ARX3A0_COARSE_INTEGRATION_TIME_REGISTER              0x3012
#define ARX3A0_GLOBAL_GAIN_REGISTER                          0x305E
#define ARX3A0_DATA_PEDESTAL_REGISTER                        0x301E

/* Wrapper function for i2c read
 *  read register value from ARX3A0 Camera Sensor registers
 *   using i2c read API \ref camera_sensor_i2c_read
 *
 *  for ARX3A0 Camera Sensor specific i2c configurations
 *   see \ref ARX3A0_camera_sensor_i2c_cnfg
 */
#define ARX3A0_READ_REG(reg_addr, reg_value, reg_size) \
    camera_sensor_i2c_read(&arx3A0_camera_sensor_i2c_cnfg, \
            reg_addr,  \
            reg_value, \
            (CAMERA_SENSOR_I2C_REG_SIZE)reg_size)

/* Wrapper function for i2c write
 *  write register value to ARX3A0 Camera Sensor registers
 *   using i2c write API \ref camera_sensor_i2c_write.
 *
 *  for ARX3A0 Camera Sensor specific i2c configurations
 *   see \ref ARX3A0_camera_sensor_i2c_cnfg
 */
#define ARX3A0_WRITE_REG(reg_addr, reg_value, reg_size) \
    camera_sensor_i2c_write(&arx3A0_camera_sensor_i2c_cnfg, \
            reg_addr,  \
            reg_value, \
            (CAMERA_SENSOR_I2C_REG_SIZE)reg_size)

/**
\brief ARX3A0 Camera Sensor Register Array Structure
       used for Camera Configuration.
*/
typedef struct _ARX3A0_REG {
    uint16_t reg_addr;             /* ARX3A0 Camera Sensor Register Address*/
    uint16_t reg_value;            /* ARX3A0 Camera Sensor Register Value*/
} ARX3A0_REG;

/* Wrapper function for Delay
 * Delay for microsecond:
 * Provide busy loop delay
 */
#define ARX3A0_DELAY_uSEC(usec)       sys_busy_loop_us(usec)

/* ARX3A) integration time */
static uint32_t current_integration_time;
static uint32_t max_integration_time;

/* ARX3A0 Camera reset GPIO port */
extern ARM_DRIVER_GPIO ARM_Driver_GPIO_(RTE_ARX3A0_CAMERA_SENSOR_RESET_GPIO_PORT);
static ARM_DRIVER_GPIO *GPIO_Driver_CAM_RST = &ARM_Driver_GPIO_(RTE_ARX3A0_CAMERA_SENSOR_RESET_GPIO_PORT);

/* ARX3A0 Camera power GPIO port */
extern ARM_DRIVER_GPIO ARM_Driver_GPIO_(RTE_ARX3A0_CAMERA_SENSOR_POWER_GPIO_PORT);
static ARM_DRIVER_GPIO *GPIO_Driver_CAM_PWR = &ARM_Driver_GPIO_(RTE_ARX3A0_CAMERA_SENSOR_POWER_GPIO_PORT);

/* I2C Driver Instance */
extern ARM_DRIVER_I2C ARM_Driver_I2C_(CAMERA_SENSOR_I2C_INSTANCE);

/**
\brief ARX3A0 Camera Sensor slave i2c Configuration
\ref CAMERA_SENSOR_SLAVE_I2C_CONFIG
*/
static CAMERA_SENSOR_SLAVE_I2C_CONFIG arx3A0_camera_sensor_i2c_cnfg =
{
    .drv_i2c                        = &ARM_Driver_I2C_(CAMERA_SENSOR_I2C_INSTANCE),
    .bus_speed                      = ARM_I2C_BUS_SPEED_STANDARD,
    .cam_sensor_slave_addr          = ARX3A0_CAMERA_SENSOR_SLAVE_ADDR,
    .cam_sensor_slave_reg_addr_type = CAMERA_SENSOR_I2C_REG_ADDR_TYPE_16BIT,
};

/**
\brief ARX3A0 Camera Sensor Resolution 560x560
*/
static const ARX3A0_REG arx3a0_560_regs[] = {

#if (RTE_ARX3A0_CAMERA_SENSOR_CSI_CFG_FPS == 90)
    {0x304C, 0x3000 }, // OTPM_RECORD
    {0x304A, 0x0070 }, // OTPM_CONTROL
    {0x0103, 0x01 }, // SOFTWARE_RESET

    {0xFFFF, 500 }, //DELAY= 100

    {0x3ED0, 0x0748 }, // RESERVED_MFR_3ED0
    {0x3ED6, 0x3136 }, // RESERVED_MFR_3ED6
    {0x3EDC, 0x1020 }, // RESERVED_MFR_3EDC
    {0x3EDE, 0x1D2A }, // RESERVED_MFR_3EDE
    {0x3EE0, 0x282A }, // RESERVED_MFR_3EE0
    {0x3EE2, 0x2821 }, // RESERVED_MFR_3EE2
    {0x3EC8, 0x0401 }, // RESERVED_MFR_3EC8
    {0x3ED2, 0x3903 }, // RESERVED_MFR_3ED2
    {0x3EC0, 0x0011 }, // RESERVED_MFR_3EC0
    {0x3ECA, 0x826F }, // RESERVED_MFR_3ECA
    {0x3EBC, 0xA8AA }, // RESERVED_MFR_3EBC
    {0x3EC4, 0x1000 }, // RESERVED_MFR_3EC4
    {0x3EBA, 0x0044 }, // RESERVED_MFR_3EBA
    {0x3ED0, 0x0745 }, // RESERVED_MFR_3ED0
    {0x3ED4, 0x0016 }, // RESERVED_MFR_3ED4
    {0x3EC6, 0x80F2 }, // RESERVED_MFR_3EC6
    {0x3ED8, 0x55FF }, // RESERVED_MFR_3ED8
    {0x3EE6, 0x8000 }, // RESERVED_MFR_3EE6
    {0x30D2, 0x0000 }, // RESERVED_MFR_30D2
    {0x31E0, 0x00F1 }, // PIX_DEF_ID
    {0x31E6, 0xA35F }, // PIX_DEF_ID_2
    {0x3180, 0x9096 }, // RESERVED_MFR_3180
    {0x3120, 0x0001 }, // GAIN_DITHER_CONTROL
    {0x301E, 0x002A }, // DATA_PEDESTAL_
    {0x3D00, 0x0436 }, // RESERVED_MFR_3D00
    {0x3D02, 0x435A }, // RESERVED_MFR_3D02
    {0x3D04, 0xFFFF }, // RESERVED_MFR_3D04
    {0x3D06, 0xFFFF }, // RESERVED_MFR_3D06
    {0x3D08, 0x2180 }, // RESERVED_MFR_3D08
    {0x3D0A, 0x0005 }, // RESERVED_MFR_3D0A
    {0x3D0C, 0x108F }, // RESERVED_MFR_3D0C
    {0x3D0E, 0x0802 }, // RESERVED_MFR_3D0E
    {0x3D10, 0x5248 }, // RESERVED_MFR_3D10
    {0x3D12, 0x801B }, // RESERVED_MFR_3D12
    {0x3D14, 0x006F }, // RESERVED_MFR_3D14
    {0x3D16, 0x8269 }, // RESERVED_MFR_3D16
    {0x3D18, 0x6A82 }, // RESERVED_MFR_3D18
    {0x3D1A, 0x5148 }, // RESERVED_MFR_3D1A
    {0x3D1C, 0x5A80 }, // RESERVED_MFR_3D1C
    {0x3D1E, 0x5902 }, // RESERVED_MFR_3D1E
    {0x3D20, 0x8082 }, // RESERVED_MFR_3D20
    {0x3D22, 0x3060 }, // RESERVED_MFR_3D22
    {0x3D24, 0x8567 }, // RESERVED_MFR_3D24
    {0x3D26, 0x5C20 }, // RESERVED_MFR_3D26
    {0x3D28, 0x4880 }, // RESERVED_MFR_3D28
    {0x3D2A, 0x0284 }, // RESERVED_MFR_3D2A
    {0x3D2C, 0x6084 }, // RESERVED_MFR_3D2C
    {0x3D2E, 0x5C91 }, // RESERVED_MFR_3D2E
    {0x3D30, 0x5980 }, // RESERVED_MFR_3D30
    {0x3D32, 0x5883 }, // RESERVED_MFR_3D32
    {0x3D34, 0x6462 }, // RESERVED_MFR_3D34
    {0x3D36, 0x8056 }, // RESERVED_MFR_3D36
    {0x3D38, 0x8162 }, // RESERVED_MFR_3D38
    {0x3D3A, 0x8422 }, // RESERVED_MFR_3D3A
    {0x3D3C, 0x20A2 }, // RESERVED_MFR_3D3C
    {0x3D3E, 0x2220 }, // RESERVED_MFR_3D3E
    {0x3D40, 0x804B }, // RESERVED_MFR_3D40
    {0x3D42, 0x8110 }, // RESERVED_MFR_3D42
    {0x3D44, 0x0943 }, // RESERVED_MFR_3D44
    {0x3D46, 0x9243 }, // RESERVED_MFR_3D46
    {0x3D48, 0x8050 }, // RESERVED_MFR_3D48
    {0x3D4A, 0x9A4B }, // RESERVED_MFR_3D4A
    {0x3D4C, 0x8563 }, // RESERVED_MFR_3D4C
    {0x3D4E, 0x8363 }, // RESERVED_MFR_3D4E
    {0x3D50, 0x8422 }, // RESERVED_MFR_3D50
    {0x3D52, 0x20A2 }, // RESERVED_MFR_3D52
    {0x3D54, 0x61C6 }, // RESERVED_MFR_3D54
    {0x3D56, 0x6F99 }, // RESERVED_MFR_3D56
    {0x3D58, 0x3009 }, // RESERVED_MFR_3D58
    {0x3D5A, 0x1FF6 }, // RESERVED_MFR_3D5A
    {0x3D5C, 0x20ED }, // RESERVED_MFR_3D5C
    {0x3D5E, 0x0874 }, // RESERVED_MFR_3D5E
    {0x3D60, 0x8230 }, // RESERVED_MFR_3D60
    {0x3D62, 0x609B }, // RESERVED_MFR_3D62
    {0x3D64, 0x3060 }, // RESERVED_MFR_3D64
    {0x3D66, 0x4600 }, // RESERVED_MFR_3D66
    {0x3D68, 0x3783 }, // RESERVED_MFR_3D68
    {0x3D6A, 0x7070 }, // RESERVED_MFR_3D6A
    {0x3D6C, 0x8040 }, // RESERVED_MFR_3D6C
    {0x3D6E, 0x4A44 }, // RESERVED_MFR_3D6E
    {0x3D70, 0x8003 }, // RESERVED_MFR_3D70
    {0x3D72, 0x0086 }, // RESERVED_MFR_3D72
    {0x3D74, 0x4588 }, // RESERVED_MFR_3D74
    {0x3D76, 0x46BA }, // RESERVED_MFR_3D76
    {0x3D78, 0x0300 }, // RESERVED_MFR_3D78
    {0x3D7A, 0xFFD7 }, // RESERVED_MFR_3D7A
    {0x3D7C, 0x4688 }, // RESERVED_MFR_3D7C
    {0x3D7E, 0x4588 }, // RESERVED_MFR_3D7E
    {0x3D80, 0x4492 }, // RESERVED_MFR_3D80
    {0x3D82, 0x4A9B }, // RESERVED_MFR_3D82
    {0x3D84, 0x4070 }, // RESERVED_MFR_3D84
    {0x3D86, 0x8040 }, // RESERVED_MFR_3D86
    {0x3D88, 0x4AAD }, // RESERVED_MFR_3D88
    {0x3D8A, 0x0070 }, // RESERVED_MFR_3D8A
    {0x3D8C, 0xAE47 }, // RESERVED_MFR_3D8C
    {0x3D8E, 0x8547 }, // RESERVED_MFR_3D8E
    {0x3D90, 0xAD78 }, // RESERVED_MFR_3D90
    {0x3D92, 0x6B85 }, // RESERVED_MFR_3D92
    {0x3D94, 0x6A80 }, // RESERVED_MFR_3D94
    {0x3D96, 0x6984 }, // RESERVED_MFR_3D96
    {0x3D98, 0x6B8A }, // RESERVED_MFR_3D98
    {0x3D9A, 0x6B80 }, // RESERVED_MFR_3D9A
    {0x3D9C, 0x6980 }, // RESERVED_MFR_3D9C
    {0x3D9E, 0x6A85 }, // RESERVED_MFR_3D9E
    {0x3DA0, 0x7C93 }, // RESERVED_MFR_3DA0
    {0x3DA2, 0x846B }, // RESERVED_MFR_3DA2
    {0x3DA4, 0x8465 }, // RESERVED_MFR_3DA4
    {0x3DA6, 0x46FF }, // RESERVED_MFR_3DA6
    {0x3DA8, 0xAA65 }, // RESERVED_MFR_3DA8
    {0x3DAA, 0x9C79 }, // RESERVED_MFR_3DAA
    {0x3DAC, 0x4A00 }, // RESERVED_MFR_3DAC
    {0x3DAE, 0x2180 }, // RESERVED_MFR_3DAE
    {0x3DB0, 0x44AC }, // RESERVED_MFR_3DB0
    {0x3DB2, 0x7070 }, // RESERVED_MFR_3DB2
    {0x3DB4, 0x2180 }, // RESERVED_MFR_3DB4
    {0x3DB6, 0x0005 }, // RESERVED_MFR_3DB6
    {0x3DB8, 0x108F }, // RESERVED_MFR_3DB8
    {0x3DBA, 0x0802 }, // RESERVED_MFR_3DBA
    {0x3DBC, 0x5248 }, // RESERVED_MFR_3DBC
    {0x3DBE, 0x801B }, // RESERVED_MFR_3DBE
    {0x3DC0, 0x006F }, // RESERVED_MFR_3DC0
    {0x3DC2, 0x8269 }, // RESERVED_MFR_3DC2
    {0x3DC4, 0x6A82 }, // RESERVED_MFR_3DC4
    {0x3DC6, 0x5148 }, // RESERVED_MFR_3DC6
    {0x3DC8, 0x5A80 }, // RESERVED_MFR_3DC8
    {0x3DCA, 0x5902 }, // RESERVED_MFR_3DCA
    {0x3DCC, 0x8082 }, // RESERVED_MFR_3DCC
    {0x3DCE, 0x3060 }, // RESERVED_MFR_3DCE
    {0x3DD0, 0x8567 }, // RESERVED_MFR_3DD0
    {0x3DD2, 0x5C20 }, // RESERVED_MFR_3DD2
    {0x3DD4, 0x4880 }, // RESERVED_MFR_3DD4
    {0x3DD6, 0x0284 }, // RESERVED_MFR_3DD6
    {0x3DD8, 0x6084 }, // RESERVED_MFR_3DD8
    {0x3DDA, 0x5C91 }, // RESERVED_MFR_3DDA
    {0x3DDC, 0x5980 }, // RESERVED_MFR_3DDC
    {0x3DDE, 0x5883 }, // RESERVED_MFR_3DDE
    {0x3DE0, 0x6462 }, // RESERVED_MFR_3DE0
    {0x3DE2, 0x8056 }, // RESERVED_MFR_3DE2
    {0x3DE4, 0x8162 }, // RESERVED_MFR_3DE4
    {0x3DE6, 0x8422 }, // RESERVED_MFR_3DE6
    {0x3DE8, 0x209C }, // RESERVED_MFR_3DE8
    {0x3DEA, 0x2220 }, // RESERVED_MFR_3DEA
    {0x3DEC, 0x514B }, // RESERVED_MFR_3DEC
    {0x3DEE, 0x8110 }, // RESERVED_MFR_3DEE
    {0x3DF0, 0x0943 }, // RESERVED_MFR_3DF0
    {0x3DF2, 0x9843 }, // RESERVED_MFR_3DF2
    {0x3DF4, 0x8050 }, // RESERVED_MFR_3DF4
    {0x3DF6, 0x8B51 }, // RESERVED_MFR_3DF6
    {0x3DF8, 0x8D4B }, // RESERVED_MFR_3DF8
    {0x3DFA, 0x9063 }, // RESERVED_MFR_3DFA
    {0x3DFC, 0x8363 }, // RESERVED_MFR_3DFC
    {0x3DFE, 0x8422 }, // RESERVED_MFR_3DFE
    {0x3E00, 0x209C }, // RESERVED_MFR_3E00
    {0x3E02, 0x61D3 }, // RESERVED_MFR_3E02
    {0x3E04, 0x1FB6 }, // RESERVED_MFR_3E04
    {0x3E06, 0x20ED }, // RESERVED_MFR_3E06
    {0x3E08, 0x3809 }, // RESERVED_MFR_3E08
    {0x3E0A, 0x524B }, // RESERVED_MFR_3E0A
    {0x3E0C, 0x0014 }, // RESERVED_MFR_3E0C
    {0x3E0E, 0x4580 }, // RESERVED_MFR_3E0E
    {0x3E10, 0x4681 }, // RESERVED_MFR_3E10
    {0x3E12, 0x3060 }, // RESERVED_MFR_3E12
    {0x3E14, 0x9D30 }, // RESERVED_MFR_3E14
    {0x3E16, 0x6083 }, // RESERVED_MFR_3E16
    {0x3E18, 0x4645 }, // RESERVED_MFR_3E18
    {0x3E1A, 0x0017 }, // RESERVED_MFR_3E1A
    {0x3E1C, 0x8170 }, // RESERVED_MFR_3E1C
    {0x3E1E, 0x7070 }, // RESERVED_MFR_3E1E
    {0x3E20, 0x7070 }, // RESERVED_MFR_3E20
    {0x3E22, 0x7070 }, // RESERVED_MFR_3E22
    {0x3E24, 0x7070 }, // RESERVED_MFR_3E24
    {0x3E26, 0x7070 }, // RESERVED_MFR_3E26
    {0x3E28, 0x7070 }, // RESERVED_MFR_3E28
    {0x3E2A, 0x7070 }, // RESERVED_MFR_3E2A
    {0x3E2C, 0x7070 }, // RESERVED_MFR_3E2C
    {0x3E2E, 0x7070 }, // RESERVED_MFR_3E2E
    {0x3E30, 0x7070 }, // RESERVED_MFR_3E30
    {0x3E32, 0x7070 }, // RESERVED_MFR_3E32
    {0x3E34, 0x7070 }, // RESERVED_MFR_3E34
    {0x3E36, 0x7070 }, // RESERVED_MFR_3E36
    {0x3E38, 0x7070 }, // RESERVED_MFR_3E38
    {0x3E3A, 0x7070 }, // RESERVED_MFR_3E3A
    {0x3E3C, 0x7070 }, // RESERVED_MFR_3E3C
    {0x3E3E, 0x7070 }, // RESERVED_MFR_3E3E
    {0x3E40, 0x7070 }, // RESERVED_MFR_3E40
    {0x3E42, 0x7070 }, // RESERVED_MFR_3E42
    {0x3E44, 0x7070 }, // RESERVED_MFR_3E44
    {0x3E46, 0x7070 }, // RESERVED_MFR_3E46
    {0x3E48, 0x7070 }, // RESERVED_MFR_3E48
    {0x3E4A, 0x7070 }, // RESERVED_MFR_3E4A
    {0x3E4C, 0x7070 }, // RESERVED_MFR_3E4C
    {0x3E4E, 0x7070 }, // RESERVED_MFR_3E4E
    {0x3E50, 0x7070 }, // RESERVED_MFR_3E50
    {0x3E52, 0x7070 }, // RESERVED_MFR_3E52
    {0x3E54, 0x7070 }, // RESERVED_MFR_3E54
    {0x3E56, 0x7070 }, // RESERVED_MFR_3E56
    {0x3E58, 0x7070 }, // RESERVED_MFR_3E58
    {0x3E5A, 0x7070 }, // RESERVED_MFR_3E5A
    {0x3E5C, 0x7070 }, // RESERVED_MFR_3E5C
    {0x3E5E, 0x7070 }, // RESERVED_MFR_3E5E
    {0x3E60, 0x7070 }, // RESERVED_MFR_3E60
    {0x3E62, 0x7070 }, // RESERVED_MFR_3E62
    {0x3E64, 0x7070 }, // RESERVED_MFR_3E64
    {0x3E66, 0x7070 }, // RESERVED_MFR_3E66
    {0x3E68, 0x7070 }, // RESERVED_MFR_3E68
    {0x3E6A, 0x7070 }, // RESERVED_MFR_3E6A
    {0x3E6C, 0x7070 }, // RESERVED_MFR_3E6C
    {0x3E6E, 0x7070 }, // RESERVED_MFR_3E6E
    {0x3E70, 0x7070 }, // RESERVED_MFR_3E70
    {0x3E72, 0x7070 }, // RESERVED_MFR_3E72
    {0x3E74, 0x7070 }, // RESERVED_MFR_3E74
    {0x3E76, 0x7070 }, // RESERVED_MFR_3E76
    {0x3E78, 0x7070 }, // RESERVED_MFR_3E78
    {0x3E7A, 0x7070 }, // RESERVED_MFR_3E7A
    {0x3E7C, 0x7070 }, // RESERVED_MFR_3E7C
    {0x3E7E, 0x7070 }, // RESERVED_MFR_3E7E
    {0x3E80, 0x7070 }, // RESERVED_MFR_3E80
    {0x3E82, 0x7070 }, // RESERVED_MFR_3E82
    {0x3E84, 0x7070 }, // RESERVED_MFR_3E84
    {0x3E86, 0x7070 }, // RESERVED_MFR_3E86
    {0x3E88, 0x7070 }, // RESERVED_MFR_3E88
    {0x3E8A, 0x7070 }, // RESERVED_MFR_3E8A
    {0x3E8C, 0x7070 }, // RESERVED_MFR_3E8C
    {0x3E8E, 0x7070 }, // RESERVED_MFR_3E8E
    {0x3E90, 0x7070 }, // RESERVED_MFR_3E90
    {0x3E92, 0x7070 }, // RESERVED_MFR_3E92
    {0x3E94, 0x7070 }, // RESERVED_MFR_3E94
    {0x3E96, 0x7070 }, // RESERVED_MFR_3E96
    {0x3E98, 0x7070 }, // RESERVED_MFR_3E98
    {0x3E9A, 0x7070 }, // RESERVED_MFR_3E9A
    {0x3E9C, 0x7070 }, // RESERVED_MFR_3E9C
    {0x3E9E, 0x7070 }, // RESERVED_MFR_3E9E
    {0x3EA0, 0x7070 }, // RESERVED_MFR_3EA0
    {0x3EA2, 0x7070 }, // RESERVED_MFR_3EA2
    {0x3EA4, 0x7070 }, // RESERVED_MFR_3EA4
    {0x3EA6, 0x7070 }, // RESERVED_MFR_3EA6
    {0x3EA8, 0x7070 }, // RESERVED_MFR_3EA8
    {0x3EAA, 0x7070 }, // RESERVED_MFR_3EAA
    {0x3EAC, 0x7070 }, // RESERVED_MFR_3EAC
    {0x3EAE, 0x7070 }, // RESERVED_MFR_3EAE
    {0x3EB0, 0x7070 }, // RESERVED_MFR_3EB0
    {0x3EB2, 0x7070 }, // RESERVED_MFR_3EB2

    /*PLL Setting*/
    {0x0300, 0x000C}, //VT_PIX_CLK_DIV
    {0x0302, 0x0001}, //VT_SYS_CLK_DIV
    {0x0304, 0x0101}, //PRE_PLL_CLK_DIV2 //PRE_PLL_CLK_DIV1
    {0x0306, 0x2830}, //PLL_MULTIPLIER2 //PLL_MULTIPLIER1
    {0x0308, 0x000A}, //OP_PIX_CLK_DIV
    {0x030A, 0x0001}, //OP_SYS_CLK_DIV

    {0x31B0, 0x0047}, //FRAME_PREAMBLE
    {0x31B2, 0x0026}, //LINE_PREAMBLE
    {0x31B4, 0x328C}, //MIPI_TIMING_0
    {0x31B6, 0x3308}, //MIPI_TIMING_1
    {0x31B8, 0x1C12}, //MIPI_TIMING_2
    {0x31BA, 0x1452}, //MIPI_TIMING_3
    {0x31BC, 0x8488}, //MIPI_TIMING_4

    /*Group_Parameter Hold*/
    {0x0104, 0x0001},

    /*Mode Configuration*/
    {0x0122, 0x0A0A}, //CCP_DATA_FORMAT
    {0x31AE, 0x0202}, //SERIAL_FROMAT

    {0x3004, 0x0004}, //X_ADDR_START
    {0x3008, 0x0234}, //X_ADDR_END
    {0x3002, 0x0004}, //Y_ADDR_START
    {0x3006, 0x0233}, //Y_ADDR_END
    {0x034C, 0x0230}, //X_OUTPUT_SIZE
    {0x034E, 0x0230}, //Y_OUTPUT_SIZE

    {0x300C, 0x02EA}, //LINE_LENGTH_PCK
    {0x300A, 0x04A6}, //FRAME_LENGTH_LINES
    {0x3040, 0x0001}, //Y_ODD_INC = 1
    {0x30A4, 0x0001}, //Y_EVEN_INC_
    {0x30AA, 0x0000}, //DIGITAL_HSKIP
    {0x0400, 0x0000}, //SCALE_ENABLE
    {0x0404, 0x0010}, //SCALE_M
    {0x317A, 0x416E}, //SF_BIN_ENABLE0(FIELD_WR= ANALOG_CONTROL6, PARALLEL_SHUTTER)
    {0x3F3C, 0x0003}, //SF_BIN_ENABLE1(FIELD_WR= ANALOG_CONTROL9, ANA_BLOCKS_ENABLE)
    {0x3012, 0x0441}, //COARSE_INTEGRATION_TIME
    {0x3730, 0x0000}, //FB_CTRL
    {0x305E, 0x2035}, // GLOBAL_GAIN

    /*Group_Parameter Hold*/
    {0x0104, 0x0000},
#elif (RTE_ARX3A0_CAMERA_SENSOR_CSI_CFG_FPS == 60)
    {0x304C, 0x3000 }, // OTPM_RECORD
    {0x304A, 0x0070 }, // OTPM_CONTROL
    {0x0103, 0x01 }, // SOFTWARE_RESET

    {0xFFFF, 500 }, //DELAY= 100

    {0x3ED0, 0x0748 }, // RESERVED_MFR_3ED0
    {0x3ED6, 0x3136 }, // RESERVED_MFR_3ED6
    {0x3EDC, 0x1020 }, // RESERVED_MFR_3EDC
    {0x3EDE, 0x1D2A }, // RESERVED_MFR_3EDE
    {0x3EE0, 0x282A }, // RESERVED_MFR_3EE0
    {0x3EE2, 0x2821 }, // RESERVED_MFR_3EE2
    {0x3EC8, 0x0401 }, // RESERVED_MFR_3EC8
    {0x3ED2, 0x3903 }, // RESERVED_MFR_3ED2
    {0x3EC0, 0x0011 }, // RESERVED_MFR_3EC0
    {0x3ECA, 0x826F }, // RESERVED_MFR_3ECA
    {0x3EBC, 0xA8AA }, // RESERVED_MFR_3EBC
    {0x3EC4, 0x1000 }, // RESERVED_MFR_3EC4
    {0x3EBA, 0x0044 }, // RESERVED_MFR_3EBA
    {0x3ED0, 0x0745 }, // RESERVED_MFR_3ED0
    {0x3ED4, 0x0016 }, // RESERVED_MFR_3ED4
    {0x3EC6, 0x80F2 }, // RESERVED_MFR_3EC6
    {0x3ED8, 0x55FF }, // RESERVED_MFR_3ED8
    {0x3EE6, 0x8000 }, // RESERVED_MFR_3EE6
    {0x30D2, 0x0000 }, // RESERVED_MFR_30D2
    {0x31E0, 0x00F1 }, // PIX_DEF_ID
    {0x31E6, 0xA35F }, // PIX_DEF_ID_2
    {0x3180, 0x9096 }, // RESERVED_MFR_3180
    {0x3120, 0x0001 }, // GAIN_DITHER_CONTROL
    {0x301E, 0x002A }, // DATA_PEDESTAL_
    {0x3D00, 0x0436 }, // RESERVED_MFR_3D00
    {0x3D02, 0x435A }, // RESERVED_MFR_3D02
    {0x3D04, 0xFFFF }, // RESERVED_MFR_3D04
    {0x3D06, 0xFFFF }, // RESERVED_MFR_3D06
    {0x3D08, 0x2180 }, // RESERVED_MFR_3D08
    {0x3D0A, 0x0005 }, // RESERVED_MFR_3D0A
    {0x3D0C, 0x108F }, // RESERVED_MFR_3D0C
    {0x3D0E, 0x0802 }, // RESERVED_MFR_3D0E
    {0x3D10, 0x5248 }, // RESERVED_MFR_3D10
    {0x3D12, 0x801B }, // RESERVED_MFR_3D12
    {0x3D14, 0x006F }, // RESERVED_MFR_3D14
    {0x3D16, 0x8269 }, // RESERVED_MFR_3D16
    {0x3D18, 0x6A82 }, // RESERVED_MFR_3D18
    {0x3D1A, 0x5148 }, // RESERVED_MFR_3D1A
    {0x3D1C, 0x5A80 }, // RESERVED_MFR_3D1C
    {0x3D1E, 0x5902 }, // RESERVED_MFR_3D1E
    {0x3D20, 0x8082 }, // RESERVED_MFR_3D20
    {0x3D22, 0x3060 }, // RESERVED_MFR_3D22
    {0x3D24, 0x8567 }, // RESERVED_MFR_3D24
    {0x3D26, 0x5C20 }, // RESERVED_MFR_3D26
    {0x3D28, 0x4880 }, // RESERVED_MFR_3D28
    {0x3D2A, 0x0284 }, // RESERVED_MFR_3D2A
    {0x3D2C, 0x6084 }, // RESERVED_MFR_3D2C
    {0x3D2E, 0x5C91 }, // RESERVED_MFR_3D2E
    {0x3D30, 0x5980 }, // RESERVED_MFR_3D30
    {0x3D32, 0x5883 }, // RESERVED_MFR_3D32
    {0x3D34, 0x6462 }, // RESERVED_MFR_3D34
    {0x3D36, 0x8056 }, // RESERVED_MFR_3D36
    {0x3D38, 0x8162 }, // RESERVED_MFR_3D38
    {0x3D3A, 0x8422 }, // RESERVED_MFR_3D3A
    {0x3D3C, 0x20A2 }, // RESERVED_MFR_3D3C
    {0x3D3E, 0x2220 }, // RESERVED_MFR_3D3E
    {0x3D40, 0x804B }, // RESERVED_MFR_3D40
    {0x3D42, 0x8110 }, // RESERVED_MFR_3D42
    {0x3D44, 0x0943 }, // RESERVED_MFR_3D44
    {0x3D46, 0x9243 }, // RESERVED_MFR_3D46
    {0x3D48, 0x8050 }, // RESERVED_MFR_3D48
    {0x3D4A, 0x9A4B }, // RESERVED_MFR_3D4A
    {0x3D4C, 0x8563 }, // RESERVED_MFR_3D4C
    {0x3D4E, 0x8363 }, // RESERVED_MFR_3D4E
    {0x3D50, 0x8422 }, // RESERVED_MFR_3D50
    {0x3D52, 0x20A2 }, // RESERVED_MFR_3D52
    {0x3D54, 0x61C6 }, // RESERVED_MFR_3D54
    {0x3D56, 0x6F99 }, // RESERVED_MFR_3D56
    {0x3D58, 0x3009 }, // RESERVED_MFR_3D58
    {0x3D5A, 0x1FF6 }, // RESERVED_MFR_3D5A
    {0x3D5C, 0x20ED }, // RESERVED_MFR_3D5C
    {0x3D5E, 0x0874 }, // RESERVED_MFR_3D5E
    {0x3D60, 0x8230 }, // RESERVED_MFR_3D60
    {0x3D62, 0x609B }, // RESERVED_MFR_3D62
    {0x3D64, 0x3060 }, // RESERVED_MFR_3D64
    {0x3D66, 0x4600 }, // RESERVED_MFR_3D66
    {0x3D68, 0x3783 }, // RESERVED_MFR_3D68
    {0x3D6A, 0x7070 }, // RESERVED_MFR_3D6A
    {0x3D6C, 0x8040 }, // RESERVED_MFR_3D6C
    {0x3D6E, 0x4A44 }, // RESERVED_MFR_3D6E
    {0x3D70, 0x8003 }, // RESERVED_MFR_3D70
    {0x3D72, 0x0086 }, // RESERVED_MFR_3D72
    {0x3D74, 0x4588 }, // RESERVED_MFR_3D74
    {0x3D76, 0x46BA }, // RESERVED_MFR_3D76
    {0x3D78, 0x0300 }, // RESERVED_MFR_3D78
    {0x3D7A, 0xFFD7 }, // RESERVED_MFR_3D7A
    {0x3D7C, 0x4688 }, // RESERVED_MFR_3D7C
    {0x3D7E, 0x4588 }, // RESERVED_MFR_3D7E
    {0x3D80, 0x4492 }, // RESERVED_MFR_3D80
    {0x3D82, 0x4A9B }, // RESERVED_MFR_3D82
    {0x3D84, 0x4070 }, // RESERVED_MFR_3D84
    {0x3D86, 0x8040 }, // RESERVED_MFR_3D86
    {0x3D88, 0x4AAD }, // RESERVED_MFR_3D88
    {0x3D8A, 0x0070 }, // RESERVED_MFR_3D8A
    {0x3D8C, 0xAE47 }, // RESERVED_MFR_3D8C
    {0x3D8E, 0x8547 }, // RESERVED_MFR_3D8E
    {0x3D90, 0xAD78 }, // RESERVED_MFR_3D90
    {0x3D92, 0x6B85 }, // RESERVED_MFR_3D92
    {0x3D94, 0x6A80 }, // RESERVED_MFR_3D94
    {0x3D96, 0x6984 }, // RESERVED_MFR_3D96
    {0x3D98, 0x6B8A }, // RESERVED_MFR_3D98
    {0x3D9A, 0x6B80 }, // RESERVED_MFR_3D9A
    {0x3D9C, 0x6980 }, // RESERVED_MFR_3D9C
    {0x3D9E, 0x6A85 }, // RESERVED_MFR_3D9E
    {0x3DA0, 0x7C93 }, // RESERVED_MFR_3DA0
    {0x3DA2, 0x846B }, // RESERVED_MFR_3DA2
    {0x3DA4, 0x8465 }, // RESERVED_MFR_3DA4
    {0x3DA6, 0x46FF }, // RESERVED_MFR_3DA6
    {0x3DA8, 0xAA65 }, // RESERVED_MFR_3DA8
    {0x3DAA, 0x9C79 }, // RESERVED_MFR_3DAA
    {0x3DAC, 0x4A00 }, // RESERVED_MFR_3DAC
    {0x3DAE, 0x2180 }, // RESERVED_MFR_3DAE
    {0x3DB0, 0x44AC }, // RESERVED_MFR_3DB0
    {0x3DB2, 0x7070 }, // RESERVED_MFR_3DB2
    {0x3DB4, 0x2180 }, // RESERVED_MFR_3DB4
    {0x3DB6, 0x0005 }, // RESERVED_MFR_3DB6
    {0x3DB8, 0x108F }, // RESERVED_MFR_3DB8
    {0x3DBA, 0x0802 }, // RESERVED_MFR_3DBA
    {0x3DBC, 0x5248 }, // RESERVED_MFR_3DBC
    {0x3DBE, 0x801B }, // RESERVED_MFR_3DBE
    {0x3DC0, 0x006F }, // RESERVED_MFR_3DC0
    {0x3DC2, 0x8269 }, // RESERVED_MFR_3DC2
    {0x3DC4, 0x6A82 }, // RESERVED_MFR_3DC4
    {0x3DC6, 0x5148 }, // RESERVED_MFR_3DC6
    {0x3DC8, 0x5A80 }, // RESERVED_MFR_3DC8
    {0x3DCA, 0x5902 }, // RESERVED_MFR_3DCA
    {0x3DCC, 0x8082 }, // RESERVED_MFR_3DCC
    {0x3DCE, 0x3060 }, // RESERVED_MFR_3DCE
    {0x3DD0, 0x8567 }, // RESERVED_MFR_3DD0
    {0x3DD2, 0x5C20 }, // RESERVED_MFR_3DD2
    {0x3DD4, 0x4880 }, // RESERVED_MFR_3DD4
    {0x3DD6, 0x0284 }, // RESERVED_MFR_3DD6
    {0x3DD8, 0x6084 }, // RESERVED_MFR_3DD8
    {0x3DDA, 0x5C91 }, // RESERVED_MFR_3DDA
    {0x3DDC, 0x5980 }, // RESERVED_MFR_3DDC
    {0x3DDE, 0x5883 }, // RESERVED_MFR_3DDE
    {0x3DE0, 0x6462 }, // RESERVED_MFR_3DE0
    {0x3DE2, 0x8056 }, // RESERVED_MFR_3DE2
    {0x3DE4, 0x8162 }, // RESERVED_MFR_3DE4
    {0x3DE6, 0x8422 }, // RESERVED_MFR_3DE6
    {0x3DE8, 0x209C }, // RESERVED_MFR_3DE8
    {0x3DEA, 0x2220 }, // RESERVED_MFR_3DEA
    {0x3DEC, 0x514B }, // RESERVED_MFR_3DEC
    {0x3DEE, 0x8110 }, // RESERVED_MFR_3DEE
    {0x3DF0, 0x0943 }, // RESERVED_MFR_3DF0
    {0x3DF2, 0x9843 }, // RESERVED_MFR_3DF2
    {0x3DF4, 0x8050 }, // RESERVED_MFR_3DF4
    {0x3DF6, 0x8B51 }, // RESERVED_MFR_3DF6
    {0x3DF8, 0x8D4B }, // RESERVED_MFR_3DF8
    {0x3DFA, 0x9063 }, // RESERVED_MFR_3DFA
    {0x3DFC, 0x8363 }, // RESERVED_MFR_3DFC
    {0x3DFE, 0x8422 }, // RESERVED_MFR_3DFE
    {0x3E00, 0x209C }, // RESERVED_MFR_3E00
    {0x3E02, 0x61D3 }, // RESERVED_MFR_3E02
    {0x3E04, 0x1FB6 }, // RESERVED_MFR_3E04
    {0x3E06, 0x20ED }, // RESERVED_MFR_3E06
    {0x3E08, 0x3809 }, // RESERVED_MFR_3E08
    {0x3E0A, 0x524B }, // RESERVED_MFR_3E0A
    {0x3E0C, 0x0014 }, // RESERVED_MFR_3E0C
    {0x3E0E, 0x4580 }, // RESERVED_MFR_3E0E
    {0x3E10, 0x4681 }, // RESERVED_MFR_3E10
    {0x3E12, 0x3060 }, // RESERVED_MFR_3E12
    {0x3E14, 0x9D30 }, // RESERVED_MFR_3E14
    {0x3E16, 0x6083 }, // RESERVED_MFR_3E16
    {0x3E18, 0x4645 }, // RESERVED_MFR_3E18
    {0x3E1A, 0x0017 }, // RESERVED_MFR_3E1A
    {0x3E1C, 0x8170 }, // RESERVED_MFR_3E1C
    {0x3E1E, 0x7070 }, // RESERVED_MFR_3E1E
    {0x3E20, 0x7070 }, // RESERVED_MFR_3E20
    {0x3E22, 0x7070 }, // RESERVED_MFR_3E22
    {0x3E24, 0x7070 }, // RESERVED_MFR_3E24
    {0x3E26, 0x7070 }, // RESERVED_MFR_3E26
    {0x3E28, 0x7070 }, // RESERVED_MFR_3E28
    {0x3E2A, 0x7070 }, // RESERVED_MFR_3E2A
    {0x3E2C, 0x7070 }, // RESERVED_MFR_3E2C
    {0x3E2E, 0x7070 }, // RESERVED_MFR_3E2E
    {0x3E30, 0x7070 }, // RESERVED_MFR_3E30
    {0x3E32, 0x7070 }, // RESERVED_MFR_3E32
    {0x3E34, 0x7070 }, // RESERVED_MFR_3E34
    {0x3E36, 0x7070 }, // RESERVED_MFR_3E36
    {0x3E38, 0x7070 }, // RESERVED_MFR_3E38
    {0x3E3A, 0x7070 }, // RESERVED_MFR_3E3A
    {0x3E3C, 0x7070 }, // RESERVED_MFR_3E3C
    {0x3E3E, 0x7070 }, // RESERVED_MFR_3E3E
    {0x3E40, 0x7070 }, // RESERVED_MFR_3E40
    {0x3E42, 0x7070 }, // RESERVED_MFR_3E42
    {0x3E44, 0x7070 }, // RESERVED_MFR_3E44
    {0x3E46, 0x7070 }, // RESERVED_MFR_3E46
    {0x3E48, 0x7070 }, // RESERVED_MFR_3E48
    {0x3E4A, 0x7070 }, // RESERVED_MFR_3E4A
    {0x3E4C, 0x7070 }, // RESERVED_MFR_3E4C
    {0x3E4E, 0x7070 }, // RESERVED_MFR_3E4E
    {0x3E50, 0x7070 }, // RESERVED_MFR_3E50
    {0x3E52, 0x7070 }, // RESERVED_MFR_3E52
    {0x3E54, 0x7070 }, // RESERVED_MFR_3E54
    {0x3E56, 0x7070 }, // RESERVED_MFR_3E56
    {0x3E58, 0x7070 }, // RESERVED_MFR_3E58
    {0x3E5A, 0x7070 }, // RESERVED_MFR_3E5A
    {0x3E5C, 0x7070 }, // RESERVED_MFR_3E5C
    {0x3E5E, 0x7070 }, // RESERVED_MFR_3E5E
    {0x3E60, 0x7070 }, // RESERVED_MFR_3E60
    {0x3E62, 0x7070 }, // RESERVED_MFR_3E62
    {0x3E64, 0x7070 }, // RESERVED_MFR_3E64
    {0x3E66, 0x7070 }, // RESERVED_MFR_3E66
    {0x3E68, 0x7070 }, // RESERVED_MFR_3E68
    {0x3E6A, 0x7070 }, // RESERVED_MFR_3E6A
    {0x3E6C, 0x7070 }, // RESERVED_MFR_3E6C
    {0x3E6E, 0x7070 }, // RESERVED_MFR_3E6E
    {0x3E70, 0x7070 }, // RESERVED_MFR_3E70
    {0x3E72, 0x7070 }, // RESERVED_MFR_3E72
    {0x3E74, 0x7070 }, // RESERVED_MFR_3E74
    {0x3E76, 0x7070 }, // RESERVED_MFR_3E76
    {0x3E78, 0x7070 }, // RESERVED_MFR_3E78
    {0x3E7A, 0x7070 }, // RESERVED_MFR_3E7A
    {0x3E7C, 0x7070 }, // RESERVED_MFR_3E7C
    {0x3E7E, 0x7070 }, // RESERVED_MFR_3E7E
    {0x3E80, 0x7070 }, // RESERVED_MFR_3E80
    {0x3E82, 0x7070 }, // RESERVED_MFR_3E82
    {0x3E84, 0x7070 }, // RESERVED_MFR_3E84
    {0x3E86, 0x7070 }, // RESERVED_MFR_3E86
    {0x3E88, 0x7070 }, // RESERVED_MFR_3E88
    {0x3E8A, 0x7070 }, // RESERVED_MFR_3E8A
    {0x3E8C, 0x7070 }, // RESERVED_MFR_3E8C
    {0x3E8E, 0x7070 }, // RESERVED_MFR_3E8E
    {0x3E90, 0x7070 }, // RESERVED_MFR_3E90
    {0x3E92, 0x7070 }, // RESERVED_MFR_3E92
    {0x3E94, 0x7070 }, // RESERVED_MFR_3E94
    {0x3E96, 0x7070 }, // RESERVED_MFR_3E96
    {0x3E98, 0x7070 }, // RESERVED_MFR_3E98
    {0x3E9A, 0x7070 }, // RESERVED_MFR_3E9A
    {0x3E9C, 0x7070 }, // RESERVED_MFR_3E9C
    {0x3E9E, 0x7070 }, // RESERVED_MFR_3E9E
    {0x3EA0, 0x7070 }, // RESERVED_MFR_3EA0
    {0x3EA2, 0x7070 }, // RESERVED_MFR_3EA2
    {0x3EA4, 0x7070 }, // RESERVED_MFR_3EA4
    {0x3EA6, 0x7070 }, // RESERVED_MFR_3EA6
    {0x3EA8, 0x7070 }, // RESERVED_MFR_3EA8
    {0x3EAA, 0x7070 }, // RESERVED_MFR_3EAA
    {0x3EAC, 0x7070 }, // RESERVED_MFR_3EAC
    {0x3EAE, 0x7070 }, // RESERVED_MFR_3EAE
    {0x3EB0, 0x7070 }, // RESERVED_MFR_3EB0
    {0x3EB2, 0x7070 }, // RESERVED_MFR_3EB2

    /*PLL Setting*/
    {0x0300, 0x000C}, //VT_PIX_CLK_DIV
    {0x0302, 0x0001}, //VT_SYS_CLK_DIV
    {0x0304, 0x0101}, //PRE_PLL_CLK_DIV2 //PRE_PLL_CLK_DIV1
    {0x0306, 0x2830}, //PLL_MULTIPLIER2 //PLL_MULTIPLIER1
    {0x0308, 0x000A}, //OP_PIX_CLK_DIV
    {0x030A, 0x0001}, //OP_SYS_CLK_DIV

    {0x31B0, 0x0047}, //FRAME_PREAMBLE
    {0x31B2, 0x0026}, //LINE_PREAMBLE
    {0x31B4, 0x328C}, //MIPI_TIMING_0
    {0x31B6, 0x3308}, //MIPI_TIMING_1
    {0x31B8, 0x1C12}, //MIPI_TIMING_2
    {0x31BA, 0x1452}, //MIPI_TIMING_3
    {0x31BC, 0x8488}, //MIPI_TIMING_4

    /*Group_Parameter Hold*/
    {0x0104, 0x0001},

    /*Mode Configuration*/
    {0x0122, 0x0A0A}, //CCP_DATA_FORMAT
    {0x31AE, 0x0202}, //SERIAL_FROMAT

    {0x3004, 0x0004}, //X_ADDR_START
    {0x3008, 0x0234}, //X_ADDR_END
    {0x3002, 0x0004}, //Y_ADDR_START
    {0x3006, 0x0233}, //Y_ADDR_END
    {0x034C, 0x0230}, //X_OUTPUT_SIZE
    {0x034E, 0x0230}, //Y_OUTPUT_SIZE

    {0x300C, 0x02EA}, //LINE_LENGTH_PCK
    {0x300A, 0x06FA}, //FRAME_LENGTH_LINES
    {0x3040, 0x0001}, //Y_ODD_INC = 1
    {0x30A4, 0x0001}, //Y_EVEN_INC_
    {0x30AA, 0x0000}, //DIGITAL_HSKIP
    {0x0400, 0x0000}, //SCALE_ENABLE
    {0x0404, 0x0010}, //SCALE_M
    {0x317A, 0x416E}, //SF_BIN_ENABLE0(FIELD_WR= ANALOG_CONTROL6, PARALLEL_SHUTTER)
    {0x3F3C, 0x0003}, //SF_BIN_ENABLE1(FIELD_WR= ANALOG_CONTROL9, ANA_BLOCKS_ENABLE)
    {0x3012, 0x0441}, //COARSE_INTEGRATION_TIME
    {0x3730, 0x0000}, //FB_CTRL
    {0x305E, 0x2035}, // GLOBAL_GAIN

    /*Group_Parameter Hold*/
    {0x0104, 0x0000},
#elif  (RTE_ARX3A0_CAMERA_SENSOR_CSI_CFG_FPS == 40)
    {0x304C, 0x3000 }, // OTPM_RECORD
    {0x304A, 0x0070 }, // OTPM_CONTROL
    {0x0103, 0x01 }, // SOFTWARE_RESET

    {0xFFFF, 500 }, //DELAY= 100

    {0x0300, 0x000A }, // VT_PIX_CLK_DIV
    {0x0302, 0x0001 }, // VT_SYS_CLK_DIV
    {0x0304, 0x0101 }, // PRE_PLL_CLK_DIV
    {0x0306, 0x2828 }, // PLL_MULTIPLIER
    {0x0308, 0x000A }, // OP_PIX_CLK_DIV
    {0x030A, 0x0001 }, // OP_SYS_CLK_DIV

    {0x31B0, 0x0047 }, // FRAME_PREAMBLE
    {0x31B2, 0x0026 }, // LINE_PREAMBLE
    {0x31B4, 0x328C }, // MIPI_TIMING_0
    {0x31B6, 0x32E8 }, // MIPI_TIMING_1
    {0x31B8, 0x1C12 }, // MIPI_TIMING_2
    {0x31BA, 0x1452 }, // MIPI_TIMING_3
    {0x31BC, 0x8488 }, // MIPI_TIMING_4

    {0x3ED0, 0x0748 }, // RESERVED_MFR_3ED0
    {0x3ED6, 0x3136 }, // RESERVED_MFR_3ED6
    {0x3EDC, 0x1020 }, // RESERVED_MFR_3EDC
    {0x3EDE, 0x1D2A }, // RESERVED_MFR_3EDE
    {0x3EE0, 0x282A }, // RESERVED_MFR_3EE0
    {0x3EE2, 0x2821 }, // RESERVED_MFR_3EE2
    {0x3EC8, 0x0401 }, // RESERVED_MFR_3EC8
    {0x3ED2, 0x3903 }, // RESERVED_MFR_3ED2
    {0x3EC0, 0x0011 }, // RESERVED_MFR_3EC0
    {0x3ECA, 0x826F }, // RESERVED_MFR_3ECA
    {0x3EBC, 0xA8AA }, // RESERVED_MFR_3EBC
    {0x3EC4, 0x1000 }, // RESERVED_MFR_3EC4
    {0x3EBA, 0x0044 }, // RESERVED_MFR_3EBA
    {0x3ED0, 0x0745 }, // RESERVED_MFR_3ED0
    {0x3ED4, 0x0016 }, // RESERVED_MFR_3ED4
    {0x3EC6, 0x80F2 }, // RESERVED_MFR_3EC6
    {0x3ED8, 0x55FF }, // RESERVED_MFR_3ED8
    {0x3EE6, 0x8000 }, // RESERVED_MFR_3EE6
    {0x30D2, 0x0000 }, // RESERVED_MFR_30D2
    {0x31E0, 0x00F1 }, // PIX_DEF_ID
    {0x31E6, 0xA35F }, // PIX_DEF_ID_2
    {0x3180, 0x9096 }, // RESERVED_MFR_3180
    {0x3120, 0x0001 }, // GAIN_DITHER_CONTROL
    {0x301E, 0x002A }, // DATA_PEDESTAL_
    {0x3D00, 0x0436 }, // RESERVED_MFR_3D00
    {0x3D02, 0x435A }, // RESERVED_MFR_3D02
    {0x3D04, 0xFFFF }, // RESERVED_MFR_3D04
    {0x3D06, 0xFFFF }, // RESERVED_MFR_3D06
    {0x3D08, 0x2180 }, // RESERVED_MFR_3D08
    {0x3D0A, 0x0005 }, // RESERVED_MFR_3D0A
    {0x3D0C, 0x108F }, // RESERVED_MFR_3D0C
    {0x3D0E, 0x0802 }, // RESERVED_MFR_3D0E
    {0x3D10, 0x5248 }, // RESERVED_MFR_3D10
    {0x3D12, 0x801B }, // RESERVED_MFR_3D12
    {0x3D14, 0x006F }, // RESERVED_MFR_3D14
    {0x3D16, 0x8269 }, // RESERVED_MFR_3D16
    {0x3D18, 0x6A82 }, // RESERVED_MFR_3D18
    {0x3D1A, 0x5148 }, // RESERVED_MFR_3D1A
    {0x3D1C, 0x5A80 }, // RESERVED_MFR_3D1C
    {0x3D1E, 0x5902 }, // RESERVED_MFR_3D1E
    {0x3D20, 0x8082 }, // RESERVED_MFR_3D20
    {0x3D22, 0x3060 }, // RESERVED_MFR_3D22
    {0x3D24, 0x8567 }, // RESERVED_MFR_3D24
    {0x3D26, 0x5C20 }, // RESERVED_MFR_3D26
    {0x3D28, 0x4880 }, // RESERVED_MFR_3D28
    {0x3D2A, 0x0284 }, // RESERVED_MFR_3D2A
    {0x3D2C, 0x6084 }, // RESERVED_MFR_3D2C
    {0x3D2E, 0x5C91 }, // RESERVED_MFR_3D2E
    {0x3D30, 0x5980 }, // RESERVED_MFR_3D30
    {0x3D32, 0x5883 }, // RESERVED_MFR_3D32
    {0x3D34, 0x6462 }, // RESERVED_MFR_3D34
    {0x3D36, 0x8056 }, // RESERVED_MFR_3D36
    {0x3D38, 0x8162 }, // RESERVED_MFR_3D38
    {0x3D3A, 0x8422 }, // RESERVED_MFR_3D3A
    {0x3D3C, 0x20A2 }, // RESERVED_MFR_3D3C
    {0x3D3E, 0x2220 }, // RESERVED_MFR_3D3E
    {0x3D40, 0x804B }, // RESERVED_MFR_3D40
    {0x3D42, 0x8110 }, // RESERVED_MFR_3D42
    {0x3D44, 0x0943 }, // RESERVED_MFR_3D44
    {0x3D46, 0x9243 }, // RESERVED_MFR_3D46
    {0x3D48, 0x8050 }, // RESERVED_MFR_3D48
    {0x3D4A, 0x9A4B }, // RESERVED_MFR_3D4A
    {0x3D4C, 0x8563 }, // RESERVED_MFR_3D4C
    {0x3D4E, 0x8363 }, // RESERVED_MFR_3D4E
    {0x3D50, 0x8422 }, // RESERVED_MFR_3D50
    {0x3D52, 0x20A2 }, // RESERVED_MFR_3D52
    {0x3D54, 0x61C6 }, // RESERVED_MFR_3D54
    {0x3D56, 0x6F99 }, // RESERVED_MFR_3D56
    {0x3D58, 0x3009 }, // RESERVED_MFR_3D58
    {0x3D5A, 0x1FF6 }, // RESERVED_MFR_3D5A
    {0x3D5C, 0x20ED }, // RESERVED_MFR_3D5C
    {0x3D5E, 0x0874 }, // RESERVED_MFR_3D5E
    {0x3D60, 0x8230 }, // RESERVED_MFR_3D60
    {0x3D62, 0x609B }, // RESERVED_MFR_3D62
    {0x3D64, 0x3060 }, // RESERVED_MFR_3D64
    {0x3D66, 0x4600 }, // RESERVED_MFR_3D66
    {0x3D68, 0x3783 }, // RESERVED_MFR_3D68
    {0x3D6A, 0x7070 }, // RESERVED_MFR_3D6A
    {0x3D6C, 0x8040 }, // RESERVED_MFR_3D6C
    {0x3D6E, 0x4A44 }, // RESERVED_MFR_3D6E
    {0x3D70, 0x8003 }, // RESERVED_MFR_3D70
    {0x3D72, 0x0086 }, // RESERVED_MFR_3D72
    {0x3D74, 0x4588 }, // RESERVED_MFR_3D74
    {0x3D76, 0x46BA }, // RESERVED_MFR_3D76
    {0x3D78, 0x0300 }, // RESERVED_MFR_3D78
    {0x3D7A, 0xFFD7 }, // RESERVED_MFR_3D7A
    {0x3D7C, 0x4688 }, // RESERVED_MFR_3D7C
    {0x3D7E, 0x4588 }, // RESERVED_MFR_3D7E
    {0x3D80, 0x4492 }, // RESERVED_MFR_3D80
    {0x3D82, 0x4A9B }, // RESERVED_MFR_3D82
    {0x3D84, 0x4070 }, // RESERVED_MFR_3D84
    {0x3D86, 0x8040 }, // RESERVED_MFR_3D86
    {0x3D88, 0x4AAD }, // RESERVED_MFR_3D88
    {0x3D8A, 0x0070 }, // RESERVED_MFR_3D8A
    {0x3D8C, 0xAE47 }, // RESERVED_MFR_3D8C
    {0x3D8E, 0x8547 }, // RESERVED_MFR_3D8E
    {0x3D90, 0xAD78 }, // RESERVED_MFR_3D90
    {0x3D92, 0x6B85 }, // RESERVED_MFR_3D92
    {0x3D94, 0x6A80 }, // RESERVED_MFR_3D94
    {0x3D96, 0x6984 }, // RESERVED_MFR_3D96
    {0x3D98, 0x6B8A }, // RESERVED_MFR_3D98
    {0x3D9A, 0x6B80 }, // RESERVED_MFR_3D9A
    {0x3D9C, 0x6980 }, // RESERVED_MFR_3D9C
    {0x3D9E, 0x6A85 }, // RESERVED_MFR_3D9E
    {0x3DA0, 0x7C93 }, // RESERVED_MFR_3DA0
    {0x3DA2, 0x846B }, // RESERVED_MFR_3DA2
    {0x3DA4, 0x8465 }, // RESERVED_MFR_3DA4
    {0x3DA6, 0x46FF }, // RESERVED_MFR_3DA6
    {0x3DA8, 0xAA65 }, // RESERVED_MFR_3DA8
    {0x3DAA, 0x9C79 }, // RESERVED_MFR_3DAA
    {0x3DAC, 0x4A00 }, // RESERVED_MFR_3DAC
    {0x3DAE, 0x2180 }, // RESERVED_MFR_3DAE
    {0x3DB0, 0x44AC }, // RESERVED_MFR_3DB0
    {0x3DB2, 0x7070 }, // RESERVED_MFR_3DB2
    {0x3DB4, 0x2180 }, // RESERVED_MFR_3DB4
    {0x3DB6, 0x0005 }, // RESERVED_MFR_3DB6
    {0x3DB8, 0x108F }, // RESERVED_MFR_3DB8
    {0x3DBA, 0x0802 }, // RESERVED_MFR_3DBA
    {0x3DBC, 0x5248 }, // RESERVED_MFR_3DBC
    {0x3DBE, 0x801B }, // RESERVED_MFR_3DBE
    {0x3DC0, 0x006F }, // RESERVED_MFR_3DC0
    {0x3DC2, 0x8269 }, // RESERVED_MFR_3DC2
    {0x3DC4, 0x6A82 }, // RESERVED_MFR_3DC4
    {0x3DC6, 0x5148 }, // RESERVED_MFR_3DC6
    {0x3DC8, 0x5A80 }, // RESERVED_MFR_3DC8
    {0x3DCA, 0x5902 }, // RESERVED_MFR_3DCA
    {0x3DCC, 0x8082 }, // RESERVED_MFR_3DCC
    {0x3DCE, 0x3060 }, // RESERVED_MFR_3DCE
    {0x3DD0, 0x8567 }, // RESERVED_MFR_3DD0
    {0x3DD2, 0x5C20 }, // RESERVED_MFR_3DD2
    {0x3DD4, 0x4880 }, // RESERVED_MFR_3DD4
    {0x3DD6, 0x0284 }, // RESERVED_MFR_3DD6
    {0x3DD8, 0x6084 }, // RESERVED_MFR_3DD8
    {0x3DDA, 0x5C91 }, // RESERVED_MFR_3DDA
    {0x3DDC, 0x5980 }, // RESERVED_MFR_3DDC
    {0x3DDE, 0x5883 }, // RESERVED_MFR_3DDE
    {0x3DE0, 0x6462 }, // RESERVED_MFR_3DE0
    {0x3DE2, 0x8056 }, // RESERVED_MFR_3DE2
    {0x3DE4, 0x8162 }, // RESERVED_MFR_3DE4
    {0x3DE6, 0x8422 }, // RESERVED_MFR_3DE6
    {0x3DE8, 0x209C }, // RESERVED_MFR_3DE8
    {0x3DEA, 0x2220 }, // RESERVED_MFR_3DEA
    {0x3DEC, 0x514B }, // RESERVED_MFR_3DEC
    {0x3DEE, 0x8110 }, // RESERVED_MFR_3DEE
    {0x3DF0, 0x0943 }, // RESERVED_MFR_3DF0
    {0x3DF2, 0x9843 }, // RESERVED_MFR_3DF2
    {0x3DF4, 0x8050 }, // RESERVED_MFR_3DF4
    {0x3DF6, 0x8B51 }, // RESERVED_MFR_3DF6
    {0x3DF8, 0x8D4B }, // RESERVED_MFR_3DF8
    {0x3DFA, 0x9063 }, // RESERVED_MFR_3DFA
    {0x3DFC, 0x8363 }, // RESERVED_MFR_3DFC
    {0x3DFE, 0x8422 }, // RESERVED_MFR_3DFE
    {0x3E00, 0x209C }, // RESERVED_MFR_3E00
    {0x3E02, 0x61D3 }, // RESERVED_MFR_3E02
    {0x3E04, 0x1FB6 }, // RESERVED_MFR_3E04
    {0x3E06, 0x20ED }, // RESERVED_MFR_3E06
    {0x3E08, 0x3809 }, // RESERVED_MFR_3E08
    {0x3E0A, 0x524B }, // RESERVED_MFR_3E0A
    {0x3E0C, 0x0014 }, // RESERVED_MFR_3E0C
    {0x3E0E, 0x4580 }, // RESERVED_MFR_3E0E
    {0x3E10, 0x4681 }, // RESERVED_MFR_3E10
    {0x3E12, 0x3060 }, // RESERVED_MFR_3E12
    {0x3E14, 0x9D30 }, // RESERVED_MFR_3E14
    {0x3E16, 0x6083 }, // RESERVED_MFR_3E16
    {0x3E18, 0x4645 }, // RESERVED_MFR_3E18
    {0x3E1A, 0x0017 }, // RESERVED_MFR_3E1A
    {0x3E1C, 0x8170 }, // RESERVED_MFR_3E1C
    {0x3E1E, 0x7070 }, // RESERVED_MFR_3E1E
    {0x3E20, 0x7070 }, // RESERVED_MFR_3E20
    {0x3E22, 0x7070 }, // RESERVED_MFR_3E22
    {0x3E24, 0x7070 }, // RESERVED_MFR_3E24
    {0x3E26, 0x7070 }, // RESERVED_MFR_3E26
    {0x3E28, 0x7070 }, // RESERVED_MFR_3E28
    {0x3E2A, 0x7070 }, // RESERVED_MFR_3E2A
    {0x3E2C, 0x7070 }, // RESERVED_MFR_3E2C
    {0x3E2E, 0x7070 }, // RESERVED_MFR_3E2E
    {0x3E30, 0x7070 }, // RESERVED_MFR_3E30
    {0x3E32, 0x7070 }, // RESERVED_MFR_3E32
    {0x3E34, 0x7070 }, // RESERVED_MFR_3E34
    {0x3E36, 0x7070 }, // RESERVED_MFR_3E36
    {0x3E38, 0x7070 }, // RESERVED_MFR_3E38
    {0x3E3A, 0x7070 }, // RESERVED_MFR_3E3A
    {0x3E3C, 0x7070 }, // RESERVED_MFR_3E3C
    {0x3E3E, 0x7070 }, // RESERVED_MFR_3E3E
    {0x3E40, 0x7070 }, // RESERVED_MFR_3E40
    {0x3E42, 0x7070 }, // RESERVED_MFR_3E42
    {0x3E44, 0x7070 }, // RESERVED_MFR_3E44
    {0x3E46, 0x7070 }, // RESERVED_MFR_3E46
    {0x3E48, 0x7070 }, // RESERVED_MFR_3E48
    {0x3E4A, 0x7070 }, // RESERVED_MFR_3E4A
    {0x3E4C, 0x7070 }, // RESERVED_MFR_3E4C
    {0x3E4E, 0x7070 }, // RESERVED_MFR_3E4E
    {0x3E50, 0x7070 }, // RESERVED_MFR_3E50
    {0x3E52, 0x7070 }, // RESERVED_MFR_3E52
    {0x3E54, 0x7070 }, // RESERVED_MFR_3E54
    {0x3E56, 0x7070 }, // RESERVED_MFR_3E56
    {0x3E58, 0x7070 }, // RESERVED_MFR_3E58
    {0x3E5A, 0x7070 }, // RESERVED_MFR_3E5A
    {0x3E5C, 0x7070 }, // RESERVED_MFR_3E5C
    {0x3E5E, 0x7070 }, // RESERVED_MFR_3E5E
    {0x3E60, 0x7070 }, // RESERVED_MFR_3E60
    {0x3E62, 0x7070 }, // RESERVED_MFR_3E62
    {0x3E64, 0x7070 }, // RESERVED_MFR_3E64
    {0x3E66, 0x7070 }, // RESERVED_MFR_3E66
    {0x3E68, 0x7070 }, // RESERVED_MFR_3E68
    {0x3E6A, 0x7070 }, // RESERVED_MFR_3E6A
    {0x3E6C, 0x7070 }, // RESERVED_MFR_3E6C
    {0x3E6E, 0x7070 }, // RESERVED_MFR_3E6E
    {0x3E70, 0x7070 }, // RESERVED_MFR_3E70
    {0x3E72, 0x7070 }, // RESERVED_MFR_3E72
    {0x3E74, 0x7070 }, // RESERVED_MFR_3E74
    {0x3E76, 0x7070 }, // RESERVED_MFR_3E76
    {0x3E78, 0x7070 }, // RESERVED_MFR_3E78
    {0x3E7A, 0x7070 }, // RESERVED_MFR_3E7A
    {0x3E7C, 0x7070 }, // RESERVED_MFR_3E7C
    {0x3E7E, 0x7070 }, // RESERVED_MFR_3E7E
    {0x3E80, 0x7070 }, // RESERVED_MFR_3E80
    {0x3E82, 0x7070 }, // RESERVED_MFR_3E82
    {0x3E84, 0x7070 }, // RESERVED_MFR_3E84
    {0x3E86, 0x7070 }, // RESERVED_MFR_3E86
    {0x3E88, 0x7070 }, // RESERVED_MFR_3E88
    {0x3E8A, 0x7070 }, // RESERVED_MFR_3E8A
    {0x3E8C, 0x7070 }, // RESERVED_MFR_3E8C
    {0x3E8E, 0x7070 }, // RESERVED_MFR_3E8E
    {0x3E90, 0x7070 }, // RESERVED_MFR_3E90
    {0x3E92, 0x7070 }, // RESERVED_MFR_3E92
    {0x3E94, 0x7070 }, // RESERVED_MFR_3E94
    {0x3E96, 0x7070 }, // RESERVED_MFR_3E96
    {0x3E98, 0x7070 }, // RESERVED_MFR_3E98
    {0x3E9A, 0x7070 }, // RESERVED_MFR_3E9A
    {0x3E9C, 0x7070 }, // RESERVED_MFR_3E9C
    {0x3E9E, 0x7070 }, // RESERVED_MFR_3E9E
    {0x3EA0, 0x7070 }, // RESERVED_MFR_3EA0
    {0x3EA2, 0x7070 }, // RESERVED_MFR_3EA2
    {0x3EA4, 0x7070 }, // RESERVED_MFR_3EA4
    {0x3EA6, 0x7070 }, // RESERVED_MFR_3EA6
    {0x3EA8, 0x7070 }, // RESERVED_MFR_3EA8
    {0x3EAA, 0x7070 }, // RESERVED_MFR_3EAA
    {0x3EAC, 0x7070 }, // RESERVED_MFR_3EAC
    {0x3EAE, 0x7070 }, // RESERVED_MFR_3EAE
    {0x3EB0, 0x7070 }, // RESERVED_MFR_3EB0
    {0x3EB2, 0x7070 }, // RESERVED_MFR_3EB2
    {0x3EB4, 0x7070 }, // RESERVED_MFR_3EB4

    {0x0344, 0x0004 }, // X_ADDR_START
    {0x0346, 0x0004 }, // Y_ADDR_START
    {0x0348, 0x0233 }, // X_ADDR_END
    {0x034A, 0x0233 }, // Y_ADDR_END
    {0x034C, 0x0230 }, // X_OUTPUT_SIZE
    {0x034E, 0x0230 }, // Y_OUTPUT_SIZE
    {0x3040, 0x0041 }, // READ_MODE
    {0x30A4, 0x0001 }, // Y_EVEN_INC_
    {0x0342, 0x02F8 }, // LINE_LENGTH_PCK
    {0x0340, 0x0248 }, // FRAME_LENGTH_LINES
    {0x3012, 0x0107 }, // COARSE_INTEGRATION_TIME_

    {0xFFFF, 50 }, //DELAY= 50

    {0x301A, 0x001C }, // RESET_REGISTER
    {0x31E0, 0x0000 }, // PIX_DEF_ID
    {0x3022, 0x01 }, // GROUPED_PARAMETER_HOLD_
    {0x300A, 0x1491 }, // FRAME_LENGTH_LINES_
    {0x0202, 0x148F }, // COARSE_INTEGRATION_TIME

    {0x3022, 0x00 }, // GROUPED_PARAMETER_HOLD_
    {0x305E, 0x2010 }, // GLOBAL_GAIN
#elif (RTE_ARX3A0_CAMERA_SENSOR_CSI_CFG_FPS == 5)
    /*LOAD= PLL_360Fps_80MHz_80MHz_20MHz*/
    {0x300, 0xA    },
    {0x302, 0x1    },
    {0x304, 0x101  },
    {0x306, 0x2828 },
    {0x308, 0xA    },
    {0x30A, 0x1    },

    /*LOAD= MIPI_TIMING_10bit*/
    {0x31B0, 0x47   },
    {0x31B2, 0x26   },
    {0x31B4, 0x328C },
    {0x31B6, 0x32E8 },
    {0x31B8, 0x1C12 },
    {0x31BA, 0x1452 },
    {0x31BC, 0x8488 },

    /*LOAD= Analog_Setup_Recommended*/
    {0x3ED0, 0x748  },
    {0x3ED6, 0x3136 },
    {0x3EDC, 0x1020 },
    {0x3EDE, 0x1D2A },
    {0x3EE0, 0x282A },
    {0x3EE2, 0x2821 },
    {0x3EC8, 0x401  },
    {0x3ED2, 0x3903 },
    {0x3EC0, 0x0011 },
    {0x3ECA, 0x826F },
    {0x3EBC, 0xA8AA },
    {0x3EC4, 0x1000 },
    {0x3EBA, 0x44   },

    /*LOAD= Corrections_Recommended*/
    {0x3ED0, 0x745  },
    {0x3ED4, 0x16   },
    {0x3EC6, 0x80F2 },
    {0x3ED8, 0x55FF },
    {0x3EE6, 0x8000 },
    {0x30D2, 0x0    },
    {0x31E0, 0x00F1 },
    {0x31E6, 0xA35F },
    {0x3180, 0x9096 },
    {0x3120, 0x1    },
    {0x301E, 0x2A   },

    /*LOAD= Pixel_Timing_Recommended_10bit*/
    {0x3D00, 0x0436 },
    {0x3D02, 0x435A },
    {0x3D04, 0xFFFF },
    {0x3D06, 0xFFFF },
    {0x3D08, 0x2180 },
    {0x3D0A, 0x0005 },
    {0x3D0C, 0x108F },
    {0x3D0E, 0x0802 },
    {0x3D10, 0x5248 },
    {0x3D12, 0x801B },
    {0x3D14, 0x006F },
    {0x3D16, 0x8269 },
    {0x3D18, 0x6A82 },
    {0x3D1A, 0x5148 },
    {0x3D1C, 0x5A80 },
    {0x3D1E, 0x5902 },
    {0x3D20, 0x8082 },
    {0x3D22, 0x3060 },
    {0x3D24, 0x8567 },
    {0x3D26, 0x5C20 },
    {0x3D28, 0x4880 },
    {0x3D2A, 0x0284 },
    {0x3D2C, 0x6084 },
    {0x3D2E, 0x5C91 },
    {0x3D30, 0x5980 },
    {0x3D32, 0x5883 },
    {0x3D34, 0x6462 },
    {0x3D36, 0x8056 },
    {0x3D38, 0x8162 },
    {0x3D3A, 0x8422 },
    {0x3D3C, 0x20A2 },
    {0x3D3E, 0x2220 },
    {0x3D40, 0x804B },
    {0x3D42, 0x8110 },
    {0x3D44, 0x0943 },
    {0x3D46, 0x9243 },
    {0x3D48, 0x8050 },
    {0x3D4A, 0x9A4B },
    {0x3D4C, 0x8563 },
    {0x3D4E, 0x8363 },
    {0x3D50, 0x8422 },
    {0x3D52, 0x20A2 },
    {0x3D54, 0x61C6 },
    {0x3D56, 0x6F99 },
    {0x3D58, 0x3009 },
    {0x3D5A, 0x1FF6 },
    {0x3D5C, 0x20ED },
    {0x3D5E, 0x0874 },
    {0x3D60, 0x8230 },
    {0x3D62, 0x609B },
    {0x3D64, 0x3060 },
    {0x3D66, 0x4600 },
    {0x3D68, 0x3783 },
    {0x3D6A, 0x7070 },
    {0x3D6C, 0x8040 },
    {0x3D6E, 0x4A44 },
    {0x3D70, 0x8003 },
    {0x3D72, 0x0086 },
    {0x3D74, 0x4588 },
    {0x3D76, 0x46BA },
    {0x3D78, 0x0300 },
    {0x3D7A, 0xFFD7 },
    {0x3D7C, 0x4688 },
    {0x3D7E, 0x4588 },
    {0x3D80, 0x4492 },
    {0x3D82, 0x4A9B },
    {0x3D84, 0x4070 },
    {0x3D86, 0x8040 },
    {0x3D88, 0x4AAD },
    {0x3D8A, 0x0070 },
    {0x3D8C, 0xAE47 },
    {0x3D8E, 0x8547 },
    {0x3D90, 0xAD78 },
    {0x3D92, 0x6B85 },
    {0x3D94, 0x6A80 },
    {0x3D96, 0x6984 },
    {0x3D98, 0x6B8A },
    {0x3D9A, 0x6B80 },
    {0x3D9C, 0x6980 },
    {0x3D9E, 0x6A85 },
    {0x3DA0, 0x7C93 },
    {0x3DA2, 0x846B },
    {0x3DA4, 0x8465 },
    {0x3DA6, 0x46FF },
    {0x3DA8, 0xAA65 },
    {0x3DAA, 0x9C79 },
    {0x3DAC, 0x4A00 },
    {0x3DAE, 0x2180 },
    {0x3DB0, 0x44AC },
    {0x3DB2, 0x7070 },
    {0x3DB4, 0x2180 },
    {0x3DB6, 0x0005 },
    {0x3DB8, 0x108F },
    {0x3DBA, 0x0802 },
    {0x3DBC, 0x5248 },
    {0x3DBE, 0x801B },
    {0x3DC0, 0x006F },
    {0x3DC2, 0x8269 },
    {0x3DC4, 0x6A82 },
    {0x3DC6, 0x5148 },
    {0x3DC8, 0x5A80 },
    {0x3DCA, 0x5902 },
    {0x3DCC, 0x8082 },
    {0x3DCE, 0x3060 },
    {0x3DD0, 0x8567 },
    {0x3DD2, 0x5C20 },
    {0x3DD4, 0x4880 },
    {0x3DD6, 0x0284 },
    {0x3DD8, 0x6084 },
    {0x3DDA, 0x5C91 },
    {0x3DDC, 0x5980 },
    {0x3DDE, 0x5883 },
    {0x3DE0, 0x6462 },
    {0x3DE2, 0x8056 },
    {0x3DE4, 0x8162 },
    {0x3DE6, 0x8422 },
    {0x3DE8, 0x209C },
    {0x3DEA, 0x2220 },
    {0x3DEC, 0x514B },
    {0x3DEE, 0x8110 },
    {0x3DF0, 0x0943 },
    {0x3DF2, 0x9843 },
    {0x3DF4, 0x8050 },
    {0x3DF6, 0x8B51 },
    {0x3DF8, 0x8D4B },
    {0x3DFA, 0x9063 },
    {0x3DFC, 0x8363 },
    {0x3DFE, 0x8422 },
    {0x3E00, 0x209C },
    {0x3E02, 0x61D3 },
    {0x3E04, 0x1FB6 },
    {0x3E06, 0x20ED },
    {0x3E08, 0x3809 },
    {0x3E0A, 0x524B },
    {0x3E0C, 0x0014 },
    {0x3E0E, 0x4580 },
    {0x3E10, 0x4681 },
    {0x3E12, 0x3060 },
    {0x3E14, 0x9D30 },
    {0x3E16, 0x6083 },
    {0x3E18, 0x4645 },
    {0x3E1A, 0x0017 },
    {0x3E1C, 0x8170 },
    {0x3E1E, 0x7070 },
    {0x3E20, 0x7070 },
    {0x3E22, 0x7070 },
    {0x3E24, 0x7070 },
    {0x3E26, 0x7070 },
    {0x3E28, 0x7070 },
    {0x3E2A, 0x7070 },
    {0x3E2C, 0x7070 },
    {0x3E2E, 0x7070 },
    {0x3E30, 0x7070 },
    {0x3E32, 0x7070 },
    {0x3E34, 0x7070 },
    {0x3E36, 0x7070 },
    {0x3E38, 0x7070 },
    {0x3E3A, 0x7070 },
    {0x3E3C, 0x7070 },
    {0x3E3E, 0x7070 },
    {0x3E40, 0x7070 },
    {0x3E42, 0x7070 },
    {0x3E44, 0x7070 },
    {0x3E46, 0x7070 },
    {0x3E48, 0x7070 },
    {0x3E4A, 0x7070 },
    {0x3E4C, 0x7070 },
    {0x3E4E, 0x7070 },
    {0x3E50, 0x7070 },
    {0x3E52, 0x7070 },
    {0x3E54, 0x7070 },
    {0x3E56, 0x7070 },
    {0x3E58, 0x7070 },
    {0x3E5A, 0x7070 },
    {0x3E5C, 0x7070 },
    {0x3E5E, 0x7070 },
    {0x3E60, 0x7070 },
    {0x3E62, 0x7070 },
    {0x3E64, 0x7070 },
    {0x3E66, 0x7070 },
    {0x3E68, 0x7070 },
    {0x3E6A, 0x7070 },
    {0x3E6C, 0x7070 },
    {0x3E6E, 0x7070 },
    {0x3E70, 0x7070 },
    {0x3E72, 0x7070 },
    {0x3E74, 0x7070 },
    {0x3E76, 0x7070 },
    {0x3E78, 0x7070 },
    {0x3E7A, 0x7070 },
    {0x3E7C, 0x7070 },
    {0x3E7E, 0x7070 },
    {0x3E80, 0x7070 },
    {0x3E82, 0x7070 },
    {0x3E84, 0x7070 },
    {0x3E86, 0x7070 },
    {0x3E88, 0x7070 },
    {0x3E8A, 0x7070 },
    {0x3E8C, 0x7070 },
    {0x3E8E, 0x7070 },
    {0x3E90, 0x7070 },
    {0x3E92, 0x7070 },
    {0x3E94, 0x7070 },
    {0x3E96, 0x7070 },
    {0x3E98, 0x7070 },
    {0x3E9A, 0x7070 },
    {0x3E9C, 0x7070 },
    {0x3E9E, 0x7070 },
    {0x3EA0, 0x7070 },
    {0x3EA2, 0x7070 },
    {0x3EA4, 0x7070 },
    {0x3EA6, 0x7070 },
    {0x3EA8, 0x7070 },
    {0x3EAA, 0x7070 },
    {0x3EAC, 0x7070 },
    {0x3EAE, 0x7070 },
    {0x3EB0, 0x7070 },
    {0x3EB2, 0x7070 },
    {0x3EB4, 0x7070 },
    {0x0104, 0x0001 }, //Group_Parameter Hold
    {0x0344, 0x0004 }, //X_ADDR_START
    {0x0346, 0x0004 }, //Y_ADDR_START
    {0x0348, 0x0233 }, //X_ADDR_END
    {0x034A, 0x0233 }, //Y_ADDR_END
    {0x034C, 0x0230 }, //X_OUTPUT_SIZE
    {0x034E, 0x0230 }, //Y_OUTPUT_SIZE
    {0x3040, 0x41   }, //Y_ODD_INC
    {0x30A4, 0x1    }, //Y_EVEN_INC
    {0x342,  0x2F8  }, //line_length_pck
    {0x340,  0x248  }, //frame_length_lines ??
    {0x3012, 0x0107 }, //coarse_integration_time ??
    {0x112 , 0xA0A  }, //RAW10
    {0x202 , 0x366A },
    {0x300 , 0xA    },
    {0x302 , 0x1    },
    {0x304 , 0x101  },
    {0x306 , 0x2828 },
    {0x308 , 0xA    },
    {0x30A , 0x1    },
    {0x340 , 0xA47B },
    {0x342 , 0x2F8  },
    {0x344 , 0x4    },
    {0x346 , 0x4    },
    {0x348 , 0x233  },
    {0x34A , 0x233  },
    {0x34C , 0x230  },
    {0x34E , 0x230  },
    {0x382 , 0x1    },
    {0x386 , 0x1    },
    {0x400 , 0x0    },
    {0x402 , 0x0    },
    {0x404 , 0x10   },
    {0x3000, 0x353  },
    {0x3002, 0x4    },
    {0x3004, 0x4    },
    {0x3006, 0x233  },
    {0x3008, 0x233  },
    {0x300A, 0xA47B },
    {0x300C, 0x2F8  },
    {0x3012, 0x366A },
    {0x3018, 0x0    },
    {0x301A, 0x1C   },
    {0x301C, 0x1    },
    {0x301D, 0x0    },
    {0x301E, 0x2A   },
    {0x3021, 0x0    },
    {0x3022, 0x0    },
    {0x3023, 0x0    },
    {0x3026, 0xFFFF },
    {0x3028, 0x4    },
    {0x3032, 0x100  },
    {0x303A, 0xA    },
    {0x303B, 0xF7   },
    {0x303C, 0x0    },
    {0x303E, 0x0    },
    {0x3040, 0x41   },
    {0x3044, 0x10C0 },
    {0x3046, 0x608  },
    {0x3048, 0x8    },
    {0x304A, 0x60   },
    {0x304C, 0x200  },
    {0x305E, 0x2000 },
    {0x3064, 0x5840 },
    {0x3068, 0x0    },
    {0x306A, 0x0    },
    {0x306E, 0x9000 },
    {0x3070, 0x0    },
    {0x3072, 0x0    },
    {0x3074, 0x0    },
    {0x3076, 0x0    },
    {0x3078, 0x0    },
    {0x307A, 0x0    },
    {0x307E, 0x20   },
    {0x3088, 0x1    },
    {0x30A0, 0x1    },
    {0x30A2, 0x1    },
    {0x30A4, 0x1    },
    {0x30A6, 0x1    },
    {0x30AA, 0x0    },
    {0x30B0, 0x400  },
    {0x30BC, 0x0    },
    {0x30BE, 0x0    },
    {0x30C0, 0x2000 },
    {0x30C2, 0x0    },
    {0x30E8, 0x0    },
    {0x30EA, 0x0    },
    {0x30EC, 0x5AE7 },
    {0x30F8, 0x33   },
    {0x30FA, 0xFC4C },
    {0x3120, 0x1    },
    {0x3122, 0x7    },
    {0x3124, 0x1A7  },
    {0x3126, 0x0    },
    {0x3128, 0x1CF  },
    {0x312A, 0x4567 },
    {0x312C, 0x89AB },
    {0x312E, 0xCDEF },
    {0x3152, 0x10   },
    {0x3154, 0x3200 },
    {0x3156, 0xC8F7 },
    {0x3158, 0x0    },
    {0x315A, 0x0    },
    {0x315C, 0x0    },
    {0x315E, 0x0    },
    {0x3160, 0xEC   },
    {0x3162, 0x317  },
    {0x3164, 0x0    },
    {0x0104, 0x0000 }, //Group_Parameter Hold
#else
#error "Selected FPS configuration not found"
#endif
};

/**
  \fn           int32_t ARX3A0_Bulk_Write_Reg(const ARX3A0_REG arx3A0_reg[],
                                              uint32_t total_num, uint32_t reg_size))
  \brief        write array of registers value to ARX3A0 Camera Sensor registers.
  \param[in]    arx3A0_reg : ARX3A0 Camera Sensor Register Array Structure
  \ref ARX3A0_REG
  \param[in]    total_num   : total number of registers(size of array)
  \return       \ref execution_status
  */
static int32_t ARX3A0_Bulk_Write_Reg(const ARX3A0_REG arx3A0_reg[],
                                     uint32_t total_num, uint32_t reg_size)
{
    uint32_t i  = 0;
    int32_t ret = 0;

    for(i = 0; i < total_num; i++)
    {
        ret = ARX3A0_WRITE_REG(arx3A0_reg[i].reg_addr, arx3A0_reg[i].reg_value, \
                reg_size);
        if(ret != ARM_DRIVER_OK)
            return ret;
    }

    return ARM_DRIVER_OK;
}

/**
  \fn           void ARX3A0_Sensor_Enable_Clk_Src(void)
  \brief        Enable ARX3A0 Camera Sensor external clock source configuration.
  \param[in]    none
  \return       none
  */
static void ARX3A0_Sensor_Enable_Clk_Src(void)
{
    set_cpi_pixel_clk(CPI_PIX_CLKSEL_400MZ, RTE_ARX3A0_CAMERA_SENSOR_CSI_CLK_SCR_DIV);
}

/**
  \fn           void ARX3A0_Sensor_Disable_Clk_Src(void)
  \brief        Disable ARX3A0 Camera Sensor external clock source configuration.
  \param[in]    none
  \return       none
  */
static void ARX3A0_Sensor_Disable_Clk_Src(void)
{
    clear_cpi_pixel_clk();
}

/**
  \fn           int32_t ARX3A0_Camera_Hard_Reseten(void)
  \brief        Hard Reset ARX3A0 Camera Sensor
  \param[in]    none
  \return       \ref execution_status
  */
static int32_t ARX3A0_Camera_Hard_Reseten(void)
{
    int32_t ret = 0;

    ret = GPIO_Driver_CAM_RST->Initialize(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, NULL);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_RST->PowerControl(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, ARM_POWER_FULL);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_RST->SetDirection(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, GPIO_PIN_DIRECTION_OUTPUT);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->Initialize(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, NULL);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->PowerControl(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, ARM_POWER_FULL);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->SetDirection(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, GPIO_PIN_DIRECTION_OUTPUT);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_RST->SetValue(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, GPIO_PIN_OUTPUT_STATE_LOW);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ARX3A0_DELAY_uSEC(2000);

    ret = GPIO_Driver_CAM_PWR->SetValue(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, GPIO_PIN_OUTPUT_STATE_HIGH);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ARX3A0_DELAY_uSEC(1000);

    ret = GPIO_Driver_CAM_RST->SetValue(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, GPIO_PIN_OUTPUT_STATE_HIGH);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ARX3A0_DELAY_uSEC(100000);

    return ARM_DRIVER_OK;
}

/**
  \fn           int32_t ARX3A0_Camera_Soft_Reseten(void)
  \brief        Software Reset ARX3A0 Camera Sensor
  \param[in]    none
  \return       \ref execution_status
  */
static int32_t ARX3A0_Camera_Soft_Reseten(void)
{
    int32_t ret = 0;

    ret = ARX3A0_WRITE_REG(ARX3A0_SOFTWARE_RESET_REGISTER, 0x01, 1);
    if(ret != ARM_DRIVER_OK)
        return ret;

    /* @Observation: more delay is required for Camera Sensor
     *               to setup after Soft Reset.
     */
    ARX3A0_DELAY_uSEC(1000);

    return ARM_DRIVER_OK;
}

/**
  \fn           int32_t ARX3A0_Camera_Cfg(void)
  \brief        Initialize ARX3A0 Camera Sensor.
  this function will
  - configure Camera Sensor resolution registers as per input parameter.
  (currently supports only 560x560(WxH) Camera resolution)
  - configure Camera Sensor slew rate.
  \ref ARX3A0_Camera_Resolution
  \return       \ref execution_status
  */
static int32_t ARX3A0_Camera_Cfg(void)
{
    uint32_t total_num     = 0;

    /* Camera Sensor Resolution: 560x560(WxH) */
    total_num = (sizeof(arx3a0_560_regs) / sizeof(ARX3A0_REG));
    return ARX3A0_Bulk_Write_Reg(arx3a0_560_regs, total_num, 2);
}

/**
  \fn           int32_t ARX3A0_Camera_Gain(uint32_t gain)
  \brief        Set camera gain
  this function will
  - configure Camera Sensor gain and integration time as per input parameter.

  Gain 1 is the default integration time with camera gain set to 1.

  For requested gain > 1, we use the default integration time and adjust the analogue+digital gain.
  For requested gain < 1, we set gain to 1 and use a reduced integration time.

  \param[in]    gain    : gain value * 65536 (so 1.0 gain = 65536); 0 to read
  \return       \ref actual gain
  */
static int32_t ARX3A0_Camera_Gain(const uint32_t gain)
{
    uint32_t digital_gain;
    uint32_t fine_gain = gain;
    uint32_t coarse_gain;

    if (max_integration_time == 0)
    {
        /* Record the integration time set by the configuration tables. We won't adjust it upwards, as it may
         * interfere with frame timing, but we can freely adjust it downwards.
         */
        int32_t ret = ARX3A0_READ_REG(ARX3A0_COARSE_INTEGRATION_TIME_REGISTER, &current_integration_time, 2);
        if (ret != 0)
            return ret;

        max_integration_time = current_integration_time;

        /* Slight hackery - if this API is in use, assume the caller does not want the data pedestal - disable it */
        uint32_t reset_register;
        ret = ARX3A0_READ_REG(ARX3A0_RESET_REGISTER, &reset_register, 2);
        if (ret != 0)
            return ret;

        /* Unlock the data pedestal setting */
        ret = ARX3A0_WRITE_REG(ARX3A0_RESET_REGISTER, reset_register &~ 8, 2);
        if (ret != 0)
            return ret;

        /* Set it to zero */
        ret = ARX3A0_WRITE_REG(ARX3A0_DATA_PEDESTAL_REGISTER, 0, 2);
        if (ret != 0)
            return ret;

        /* Restore the lock */
        ret = ARX3A0_WRITE_REG(ARX3A0_RESET_REGISTER, reset_register, 2);
        if (ret != 0)
            return ret;
    }

    if (gain != 0)
    {
        /* Request to set gain */

        /* From the Design Guide:
         *
         * Total Gain = (R0x305E[0:3]/16+1)*2^R0x0305E[4:6]*(R0x305E[7:15]/64)
         *
         * (Register guide is misleading, no doubt in part due to the backwards-
         * compatibility analogue gain API in register 0x3028).
         *
         * First clamp analogue gain, using digital gain to get more if
         * necessary. Otherwise digital gain is used to fine adjust.
         */
        uint32_t new_integration_time = max_integration_time;
        if (gain < 0x10000)
        {
            /* Minimum gain is 1.0 */
            fine_gain = 0x10000;

            /* But we can lower integration time */
            new_integration_time = (uint32_t) (((float) max_integration_time * gain) * 0x1p-16f + 0.5f);
        }
        else if (gain > 0x80000)
        {
            /* Maximum analogue gain is 8.0 */
            fine_gain = 0x80000;
        }

        /* Set integration time */
        if (new_integration_time != current_integration_time)
        {
            int32_t ret = ARX3A0_WRITE_REG(ARX3A0_COARSE_INTEGRATION_TIME_REGISTER, new_integration_time, 2);
            if (ret != 0)
            {
                return ret;
            }
            current_integration_time = new_integration_time;
        }

        /*
         * First get coarse analogue power of two, leaving fine gain in range [0x10000,0x1FFFF]
         */
        coarse_gain = 0;
        while (fine_gain >= 0x20000)
        {
            coarse_gain++;
            fine_gain /= 2;
        }

        /* And we then have 16 steps of fine gain. Round down here, and
         * we tune with digital gain >= 1.0
         */
        fine_gain = (fine_gain - 0x10000) / 0x1000;

        /* Use digital gain to extend gain beyond the analogue limits of
         * x1 to x8, or to fine-tune within that range.
         *
         * We don't let digital gain go below 1.0 - it just loses information,
         * and clamping it lets an auto-gain controller see that we are
         * unable to improve exposure by such lowering. Another camera might
         * be able to usefully set gain to <1.0, so a controller could try it.
         *
         * (When we're fine tuning, digital gain is always >=1.0, because we
         * round down analogue gain, so it can only go below 1.0 by the user
         * requesting total gain < 1.0).
         */
        uint32_t resulting_analogue_gain = ((fine_gain + 16) << coarse_gain) * 0x1000;
        digital_gain = (64 * gain + (resulting_analogue_gain / 2)) / resulting_analogue_gain;
        if (digital_gain > 0x1FF)
        {
            /* Maximum digital gain is just under 8.0 (limited by register size) */
            digital_gain = 0x1FF;
        }
        else if (digital_gain < 64)
        {
            /* Digital gain >= 1.0, as per discussion above */
            digital_gain = 64;
        }
        int32_t ret = ARX3A0_WRITE_REG(ARX3A0_GLOBAL_GAIN_REGISTER, (digital_gain << 7) | (coarse_gain << 4) | fine_gain, 2);
        if (ret != 0)
            return ret;
    }
    else
    {
        uint32_t reg_value;

        int32_t ret = ARX3A0_READ_REG(ARX3A0_GLOBAL_GAIN_REGISTER, &reg_value, 2);
        if (ret != 0)
            return ret;

        digital_gain = reg_value >> 7;
        coarse_gain = (reg_value >> 4) & 7;
        fine_gain = reg_value & 0xF;
    }

    // Fixed point factors of 16 and 64 in registers, so multiply by 64 to get our *0x10000 scale
    uint32_t resulting_gain = ((fine_gain + 16) << coarse_gain) * digital_gain * 64;

    // And adjust for any reduction of integration time (in which case gain should be 0x10000 initially...)
    if (current_integration_time != max_integration_time)
    {
        resulting_gain = (uint32_t) (((float) resulting_gain * current_integration_time) / max_integration_time + 0.5f);
    }
    return resulting_gain;
}

/**
  \fn           int32_t ARX3A0_Init(ARM_CAMERA_RESOLUTION cam_resolution)
  \brief        Initialize ARX3A0 Camera Sensor
  this function will
  - initialize i2c using i3c instance
  - software reset ARX3A0 Camera Sensor
  - read ARX3A0 chip-id, proceed only it is correct.
  \param[in]    cam_resolution  : Camera Resolution \ref ARM_CAMERA_RESOLUTION
  \return       \ref execution_status
  */
static int32_t ARX3A0_Init(void)
{
    int32_t  ret = 0;
    uint32_t rcv_data = 0;

    /*Enable camera sensor clock source config*/
    ARX3A0_Sensor_Enable_Clk_Src();

    /*camera sensor resten*/
    ret = ARX3A0_Camera_Hard_Reseten();
    if(ret != ARM_DRIVER_OK)
        return ret;

    /* Initialize i2c using i3c driver instance depending on
     *  ARX3A0 Camera Sensor specific i2c configurations
     *   \ref arx3A0_camera_sensor_i2c_cnfg
     */
    ret = camera_sensor_i2c_init(&arx3A0_camera_sensor_i2c_cnfg);
    if(ret != ARM_DRIVER_OK)
        return ret;

    /* Soft Reset ARX3A0 Camera Sensor */
    ret = ARX3A0_Camera_Soft_Reseten();
    if(ret != ARM_DRIVER_OK)
        return ret;

    /* Read ARX3A0 Camera Sensor CHIP ID */
    ret = ARX3A0_READ_REG(ARX3A0_CHIP_ID_REGISTER, &rcv_data, 2);
    if(ret != ARM_DRIVER_OK)
        return ret;

    /* Proceed only if CHIP ID is correct. */
    if(rcv_data != ARX3A0_CHIP_ID_REGISTER_VALUE)
        return ARM_DRIVER_ERROR_UNSUPPORTED;

    /*Putting sensor in standby mode*/
    ret = ARX3A0_WRITE_REG(ARX3A0_MODE_SELECT_REGISTER, 0x00, 1);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = ARX3A0_READ_REG(ARX3A0_MIPI_CONFIG_REGISTER, &rcv_data, 2);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = ARX3A0_WRITE_REG(ARX3A0_MIPI_CONFIG_REGISTER, rcv_data | (1U << 7), 2);
    if(ret != ARM_DRIVER_OK)
        return ret;

    /*start streaming*/
    ret = ARX3A0_WRITE_REG(ARX3A0_MODE_SELECT_REGISTER, 0x01, 1);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ARX3A0_DELAY_uSEC(50000);

    /*stop streaming*/
    ret = ARX3A0_WRITE_REG(ARX3A0_MODE_SELECT_REGISTER, 0x00, 1);
    if(ret != ARM_DRIVER_OK)
        return ret;

    /*Adding delay to finish streaming*/
    for(uint32_t count = 0; count  < 5; count++)
        ARX3A0_DELAY_uSEC(100000);

    /* Force re-reading of the registers */
    max_integration_time = current_integration_time = 0;

    return ARM_DRIVER_OK;
}

/**
  \fn           int32_t ARX3A0_Start(void)
  \brief        Start ARX3A0 Camera Sensor Streaming.
  \param[in]    none
  \return       \ref execution_status
  */
static int32_t ARX3A0_Start(void)
{
    /* Start streaming */
    return ARX3A0_WRITE_REG(ARX3A0_MODE_SELECT_REGISTER, 0x01, 1);;
}

/**
  \fn           int32_t ARX3A0_Stop(void)
  \brief        Stop ARX3A0 Camera Sensor Streaming.
  \param[in]    none
  \return       \ref execution_status
  */
static int32_t ARX3A0_Stop(void)
{
    /* Suspend any stream */
    return ARX3A0_WRITE_REG(ARX3A0_MODE_SELECT_REGISTER, 0x00, 1);;
}

/**
  \fn           int32_t ARX3A0_Control(uint32_t control, uint32_t arg)
  \brief        Control ARX3A0 Camera Sensor.
  \param[in]    control  : Operation
  \param[in]    arg      : Argument of operation
  \return       \ref execution_status
  */
static int32_t ARX3A0_Control(uint32_t control, uint32_t arg)
{

    switch (control)
    {
        case CPI_CAMERA_SENSOR_CONFIGURE:
            return ARX3A0_Camera_Cfg();
            break;
        case CPI_CAMERA_SENSOR_GAIN:
            return ARX3A0_Camera_Gain(arg);
            break;
        default:
            return ARM_DRIVER_ERROR_PARAMETER;
    }
}

/**
  \fn           int32_t ARX3A0_Uninit(void)
  \brief        Un-initialize ARX3A0 Camera Sensor.
  \param[in]    none
  \return       \ref execution_status
  */
static int32_t ARX3A0_Uninit(void)
{
    int32_t ret;

    /*Disable camera sensor clock source config*/
    ARX3A0_Sensor_Disable_Clk_Src();

    ret = GPIO_Driver_CAM_RST->SetValue(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, GPIO_PIN_OUTPUT_STATE_LOW);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_RST->PowerControl(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO, ARM_POWER_OFF);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_RST->Uninitialize(RTE_ARX3A0_CAMERA_SENSOR_RESET_PIN_NO);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->SetValue(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, GPIO_PIN_OUTPUT_STATE_LOW);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->PowerControl(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO, ARM_POWER_OFF);
    if(ret != ARM_DRIVER_OK)
        return ret;

    ret = GPIO_Driver_CAM_PWR->Uninitialize(RTE_ARX3A0_CAMERA_SENSOR_POWER_PIN_NO);
    if(ret != ARM_DRIVER_OK)
        return ret;

    return ARM_DRIVER_OK;
}

/**
\brief ARX3A0 Camera Sensor CSi informations
\ref CSI_INFO
*/
static CSI_INFO arx3a0_csi_info =
{
    .frequency                = RTE_ARX3A0_CAMERA_SENSOR_CSI_FREQ,
    .dt                       = RTE_ARX3A0_CAMERA_SENSOR_CSI_DATA_TYPE,
    .n_lanes                  = RTE_ARX3A0_CAMERA_SENSOR_CSI_N_LANES,
    .vc_id                    = RTE_ARX3A0_CAMERA_SENSOR_CSI_VC_ID,
    .cpi_cfg.override         = RTE_ARX3A0_CAMERA_SENSOR_OVERRIDE_CPI_COLOR_MODE,
    .cpi_cfg.cpi_color_mode   = RTE_ARX3A0_CAMERA_SENSOR_CPI_COLOR_MODE
};

/**
\brief ARX3A0 Camera Sensor Operations
\ref CAMERA_SENSOR_OPERATIONS
*/
static CAMERA_SENSOR_OPERATIONS arx3a0_ops =
{
    .Init    = ARX3A0_Init,
    .Uninit  = ARX3A0_Uninit,
    .Start   = ARX3A0_Start,
    .Stop    = ARX3A0_Stop,
    .Control = ARX3A0_Control,
};

/**
\brief ARX3A0 Camera Sensor Device Structure
\ref CAMERA_SENSOR_DEVICE
*/
static CAMERA_SENSOR_DEVICE arx3a0_camera_sensor =
{
    .interface  = CAMERA_SENSOR_INTERFACE_MIPI,
    .width      = RTE_ARX3A0_CAMERA_SENSOR_FRAME_WIDTH,
    .height     = RTE_ARX3A0_CAMERA_SENSOR_FRAME_HEIGHT,
    .csi_info   = &arx3a0_csi_info,
    .ops        = &arx3a0_ops,
};

/* Registering CPI sensor */
CAMERA_SENSOR(arx3a0_camera_sensor)

#endif /* RTE_ARX3A0_CAMERA_SENSOR_CSI_ENABLE */
