/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  dirzip.c

  Get games sources from zipped files in a directory
*/

#include "common.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>

#include <mamory/readinfo.h>
#include <sortbox/sortbox.h>
#include <mamory/zip.h>

unsigned char recursive_level;
s_SortBox *RomContentsSBox;

static const unsigned long crctable[256] = {
  0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
  0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
  0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
  0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
  0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
  0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
  0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
  0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
  0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
  0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
  0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
  0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
  0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
  0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
  0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
  0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
  0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
  0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
  0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
  0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
  0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
  0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
  0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
  0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
  0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
  0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
  0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
  0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
  0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
  0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
  0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
  0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
  0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
  0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
  0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
  0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
  0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
  0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
  0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
  0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
  0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
  0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
  0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
  0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
  0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
  0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
  0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
  0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
  0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
  0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
  0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
  0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
  0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
  0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
  0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
  0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
  0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
  0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
  0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
  0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
  0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
  0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
  0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
  0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d,
};

unsigned int GetFileCRC32(unsigned char *File)
{
  unsigned char Buffer[BUFSIZ];
  unsigned char *p;
  FILE *fp;
  unsigned int crc=~0;
  int nbread;

  fp=fopen(File,"r");
  while((nbread=fread(Buffer,sizeof(unsigned char),BUFSIZ,fp))>0)
    for(p=Buffer;nbread--;++p)
      crc=(crc>>8)^crctable[(crc ^ *p)&0xff];

  fclose(fp);

  if(nbread<0)
    return 0;
  
  return ~crc;
}

void AddGameToGamesList(s_GamesList *GamesList,
			s_GameInfo *Game,
			unsigned char *Source)
{
  unsigned int i,j;

  for(i=0;i<GamesList->NbrSources;i++)
    if(strcmp(GamesList->Sources[i]->Source,Source)==0)
      break;
  if(i==GamesList->NbrSources)
    {
      GamesList->Sources=XREALLOC(GamesList->Sources,
				  s_GamesListSource*,
				  GamesList->NbrSources+1);
      GamesList->Sources[i]=XCALLOC(s_GamesListSource,1);
      GamesList->Sources[i]->Type=TYPE_DIR;
      GamesList->Sources[i]->Source=XSTRDUP(Source);
      GamesList->NbrSources++;
    }
  
  for(j=0;j<Game->NbrRoms;j++)
    Game->Roms[j]->Source=GamesList->Sources[i];
  
  GamesList->Games=XREALLOC(GamesList->Games,
			    s_GameInfo*,GamesList->NbrGames+1);
  
  GamesList->Games[GamesList->NbrGames]=Game;
  GamesList->NbrGames++;  
}

void SetRomContent(s_GamesList *GamesList,
		   s_GameInfo *Game,
		   s_RomInfo *Rom, 
		   s_RomContent *RomContent)
{
  unsigned int i,nbr;
  s_RomContent **RomContents=NULL;
  
  // Test if this size/CRC combination is already known
  RomContents=(s_RomContent**)SortBox_FindTheUint(RomContentsSBox,
						  RomContent->CRC,
						  (int*)&nbr);
  if(RomContents!=NULL)
    for(i=0;i<nbr;i++)
      if(RomContents[i]->Size==RomContent->Size)
	break;
  
  if(RomContents==NULL || (RomContents!=NULL && i==nbr))
    // size/CRC unknown, therefore it is added
    {
      SortBox_AddOneUint(RomContentsSBox,
			 RomContent->CRC,RomContent);
      Rom->Content=RomContent;
      GamesList->RomContents=XREALLOC(GamesList->RomContents,
				      s_RomContent*,
				      GamesList->NbrRomContents+1);
      GamesList->RomContents[GamesList->NbrRomContents]=Rom->Content;
      GamesList->NbrRomContents++;
    }
  else
    // size/CRC known
    {
      Rom->Content=RomContents[i];
      XFREE(RomContent);
    };
  
  for(i=0;i<nbr;i++)
    RomContents[i]=NULL;
  XFREE(RomContents);
  
  if (Rom->Content->CRC==0)
    Rom->Status=NO_ROM;
  else
    Rom->Status=UNKNOWN_DUMP;
  
  // Update the size/CRC game association
  for(i=0;i<Rom->Content->NbrAssociatedGames;i++)
    if(Rom->Content->AssociatedGames[i]->Game==Game)
      break;
  
  if(i==Rom->Content->NbrAssociatedGames)
    {	
      Rom->Content->AssociatedGames=
	XREALLOC(Rom->Content->AssociatedGames,
		 s_RomAssociatedGame*,
		 Rom->Content->NbrAssociatedGames+1);
      
      Rom->Content->AssociatedGames[Rom->Content->NbrAssociatedGames]=
	XCALLOC(s_RomAssociatedGame,1);
      
      Rom->Content->AssociatedGames[Rom->Content->NbrAssociatedGames]->NbFiles=1;
      Rom->Content->AssociatedGames[Rom->Content->NbrAssociatedGames]->Game=Game;
      Rom->Content->NbrAssociatedGames++;
    }
  else
    Rom->Content->AssociatedGames[i]->NbFiles++;
  
};

s_GameInfo *GetGameInfoFromZip(s_GamesList *GamesList,
			       unsigned char *File,
			       s_Zip *Zip)
{
  s_GameInfo *Game=NULL;
  s_RomInfo *Rom;
  s_RomContent *RomContent;
  unsigned char *GameName=NULL;
  unsigned int i,GameNameLen;
  
  if(!Zip)
    return NULL;

  if(strstr(File,".zip")!=NULL || strstr(File,".ZIP")!=NULL)
    GameNameLen=strlen(File)-strlen(".zip");
  else
    GameNameLen=strlen(File);

  GameName=XCALLOC(unsigned char,GameNameLen+1);
  strncpy(GameName,File,GameNameLen);

  for(i=0;i<GamesList->NbrGames;i++)
    if(strcmp(GamesList->Games[i]->Name,GameName)==0)
      {
	XFREE(GameName);
	Game=GamesList->Games[i];
	break;
      };

  if(i==GamesList->NbrGames)
    {
      Game=XCALLOC(s_GameInfo,1);
      Game->Name=GameName;
    };
       
  for(i=0;i<Zip->NbCentralDirEntries;i++)
    {
      // Create the potential rom
      Rom=XCALLOC(s_RomInfo,1);
      Rom->Name=XCALLOC(unsigned char,
			Zip->CentralDirEntries[i]->Const->FileNameLen+1);
      strncpy(Rom->Name,
	      Zip->CentralDirEntries[i]->FileName,
	      Zip->CentralDirEntries[i]->Const->FileNameLen);      
      Rom->GameName=XSTRDUP(Game->Name);
      RomContent=XCALLOC(s_RomContent,1);
      RomContent->Size=Zip->CentralDirEntries[i]->Const->UncompSize;
      RomContent->CompSize=Zip->CentralDirEntries[i]->Const->CompSize;
      RomContent->CRC=Zip->CentralDirEntries[i]->Const->CRC;
      
      SetRomContent(GamesList,Game,Rom,RomContent);

      // Add the potential rom to the game
      Game->Roms=XREALLOC(Game->Roms,
			      s_RomInfo*,
			      Game->NbrRoms+1);
      Game->Roms[Game->NbrRoms]=Rom;
      
      Game->NbrRoms++;      
    };
  
  if(GameName==NULL)
    return NULL;
  
  return Game;
}

  

s_GameInfo *GetGameInfoFromDir(s_GamesList *GamesList,
			       unsigned char *Directory,
			       unsigned int Recursive)
{
  
  
  DIR *dirp=NULL;
  struct dirent *DirEntry=NULL;
  unsigned char *path;
  struct stat stat_buf;
  s_Zip *Zip;

  s_GameInfo *Game=NULL, *SelfGame=NULL;
//  s_RomInfo *Rom;
//  s_RomContent *RomContent;
  unsigned char *GameName=NULL;
//  unsigned int i,GameNameLen;

  dirp=opendir(Directory);
  if(!dirp)
    return NULL;

  while((DirEntry=readdir(dirp))!=NULL)
    // Parse the whole directory
    {
      if(strncmp(DirEntry->d_name,".",1)==0)
	// Skip the hidden files
	continue;
      
      path=XCALLOC(unsigned char,
		   strlen(Directory)+1+strlen(DirEntry->d_name)+1);
      strcat(path,Directory);
      strcat(path,"/");
      strcat(path,DirEntry->d_name);
      
      if(stat(path,&stat_buf)!=0)
	{
	  LPRINTF(WARNING,"%s",strerror(errno));
	  continue;
	};
      
      if(S_ISDIR(stat_buf.st_mode))
	// path is a directory
	{
	  if(Recursive ||
	     (!Recursive && recursive_level==0))
	    {
	      recursive_level++;
	      Game=GetGameInfoFromDir(GamesList,path,Recursive);
	      recursive_level--;
	      if(Game!=NULL)
		AddGameToGamesList(GamesList,Game,Directory);
	    }
	  XFREE(path);
	  continue;
	}

      // Test if path is one of the known archive formats
      // Zip files
      Zip=Zipfopen(path,"r");
      if(Zip!=NULL)
	{
	  if(Recursive ||
	     (!Recursive && recursive_level==0))
	    {
	      recursive_level++;
	      Game=GetGameInfoFromZip(GamesList,DirEntry->d_name,Zip);
	      recursive_level--;
	      if(Game!=NULL)
		AddGameToGamesList(GamesList,Game,Directory);
	    };
	  Zipfclose(Zip);
	  Zip=NULL;
	  XFREE(path);
	  continue;
	}
      
      // TODO: Test CHD files
/*
      // path is a simple file, therefore a potential rom
      if(SelfGame==NULL)
	// Directory is considered as the game owning the rom
	// If the game does not exist, we initialize it
	{
	  i=strlen(Directory)-2;
	  while(Directory[i]!='/')
	    i--;
	  if(Directory[strlen(Directory)-1]=='/')
	    GameNameLen=strlen(Directory)-2-i;
	  else
	    GameNameLen=strlen(Directory)-1-i;
	  GameName=XCALLOC(unsigned char,GameNameLen+1);
	  strncpy(GameName,&Directory[i+1],GameNameLen);
	  
	  for(i=0;i<GamesList->NbrGames;i++)
	    if(strcmp(GamesList->Games[i]->Name,GameName)==0)
	      {
		XFREE(GameName);
		SelfGame=GamesList->Games[i];
		break;
	      };
	  
	  if(i==GamesList->NbrGames)
	    {
	      SelfGame=XCALLOC(s_GameInfo,1);
	      SelfGame->Name=GameName;
	    };
	};
      
      // Create the potential rom
      Rom=XCALLOC(s_RomInfo,1);
      Rom->Name=XSTRDUP(DirEntry->d_name);
      Rom->GameName=XSTRDUP(SelfGame->Name);
      RomContent=XCALLOC(s_RomContent,1);
      RomContent->Size=stat_buf.st_size;
      RomContent->CRC=GetFileCRC32(path);
      
      SetRomContent(GamesList,SelfGame,Rom,RomContent);
      
      // Add the potential rom to the game
      SelfGame->Roms=XREALLOC(SelfGame->Roms,
			      s_RomInfo*,
			      SelfGame->NbrRoms+1);
      SelfGame->Roms[SelfGame->NbrRoms]=Rom;
      
      SelfGame->NbrRoms++;
*/
      XFREE(path);
    };
  
  closedir(dirp);

  if(GameName==NULL)
    return NULL;

  return SelfGame;
};

s_GameInfo *GetGameInfoFromPath(s_GamesList *GamesList,
				unsigned char *File,
				unsigned int Recursive)
{
  struct stat stat_buf;
  s_GameInfo *Game;
  s_Zip *Zip;

  if(!File || stat(File,&stat_buf)!=0)
    {
      LPRINTF(WARNING,"'%s' is not a valid path or file",File);
      return(NULL);
    };

  if(!S_ISDIR(stat_buf.st_mode))
    {
      Zip=Zipfopen(File,"r");
      if(Zip!=NULL)
	{
	  Game=GetGameInfoFromZip(GamesList,File,Zip);
	  Zipfclose(Zip);
	  Zip=NULL;
	  return Game;
	}
      LPRINTF(WARNING,"'%s' is not an archive type",File);
      return NULL;
    }

  Game=GetGameInfoFromDir(GamesList,File,Recursive);

  return Game;
}

s_GamesList *GetGamesListFromPath(unsigned char *File,
				  unsigned int Recursive)
{
  s_GamesList *GamesList;
  s_GameInfo *Game;
  unsigned int i;
  unsigned char *path=NULL;
  recursive_level=0;

  GamesList=XCALLOC(s_GamesList,1);

  GamesList->MergedGamesList=FALSE;  

  RomContentsSBox=InitSortBox(SB_NUM,NULL);  

  Game=GetGameInfoFromPath(GamesList,File,Recursive);

  if(Game!=NULL)
    {
      // Ajouter une source en prenant path1/path2 pour un File de type
      // path1/path2/File/
      i=strlen(File)-2;
      while(File[i]!='/')
	i--;
      path=XCALLOC(unsigned char,i+1);
      strncpy(path,File,i);

      AddGameToGamesList(GamesList,Game,path);
    };
  
  if(GamesList->NbrSources==0)
    {
      GamesList->Sources=XREALLOC(GamesList->Sources,
				  s_GamesListSource*,
				  GamesList->NbrSources+1);
      GamesList->Sources[GamesList->NbrSources]=XCALLOC(s_GamesListSource,1);
      GamesList->Sources[GamesList->NbrSources]->Type=TYPE_DIR;
      GamesList->Sources[GamesList->NbrSources]->Source=XSTRDUP(File);
      GamesList->NbrSources++;
    };

  XFREE(path);

  FreeSortBox(RomContentsSBox);
  RomContentsSBox=NULL;
  return GamesList;
};

