// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Define the non-opaque part to the LDAP-C-API
//
// - Automatic Version Information via RCS:
//   $Id: LDAP_SFilter.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/LDAP_SFilter.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h>
#include <stdexcept>
#include "LDAP_SFilter.h"

#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: LDAP_SFilter.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $";
static const char vcHid[] = __LDAP_SFILTER_H;
#endif

static char * fill_in_sequence_of_filters(LDAP_ArrayOf_Filter &aof, char *filter);

#define e_not 2

LDAP_SFilter::LDAP_SFilter(char *filter) {
  char	*next;
  int	parens, balance, escape;
  
  parens = 0;
  while ( *filter ) {
    switch ( *filter ) {
      case '(':
	filter++;
	parens++;
      
	/* skip spaces */
	while( isspace( *filter ) ) filter++;
      
	switch ( *filter ) {
	  case '&': {
	    tag=e_and;
	    this->CreateObject();
	    LDAP_ArrayOf_Filter &aof=(*this);
	    filter=fill_in_sequence_of_filters(aof, filter);
	    choice->SetTag(e_and);
	    parens--;
	    break;
	  }
	  case '|': {
	    tag=e_or;
	    this->CreateObject();
	    LDAP_ArrayOf_Filter &aof=(*this);
	    filter=fill_in_sequence_of_filters(aof, filter);
	    choice->SetTag(e_or);
	    parens--;
	    break;
	  }
	  case '!': {
	    tag=e_not;
	    this->CreateObject();
	    LDAP_Filter &f=(*this);
	    char *s=filter;
	    s++;
	    int p=1;
	    while(*s && p) {
	      if(*s=='(')
		p++;
	      if(*s==')')
		p--;
	      s++;
	    }
	    if(*s) {
	      s='\0';
	      f=LDAP_SFilter(filter);
	      filter=++s;
	    } else {
	      filter=s;	  
	    }
	    parens--;
	  }
	    break;
	  default:
	    balance = 1;
	    escape = 0;
	    next = filter;
	    while ( *next && balance ) {
	      if ( escape == 0 ) {
		if ( *next == '(' )
		  balance++;
		else if ( *next == ')' )
		  balance--;
	      }
	      if ( *next == '\\' && ! escape )
		escape = 1;
	      else
		escape = 0;
	      if ( balance )
		next++;
	    }
	    if ( balance != 0 )
	      return;

	    *next = '\0';
	    /* Simple filter */
	    set_simple_filter(filter);
	
	    *next++ = ')';
	    filter = next;
	    parens--;
	    break;
	}
	break;
      
      case ')':
	filter++;
	parens--;
	break;
      
      case ' ':
	filter++;
	break;
      
      default:	/* assume it's a simple type=value filter */
	next = strchr( filter, '\0' );
	/* Simple filter */
	set_simple_filter(filter);
	filter = next;
	break;
    }
  }
}


static char * fill_in_sequence_of_filters(LDAP_ArrayOf_Filter &aof, char *filter){
  char *filterbegin=filter+1;
  char *s=filter+1;
  int number_of_filter=-1;
  int parens=0;
  while(*s!='\0') {
    while((*s!='\0') && isspace(*s))
      s++;
    if(*s=='\0')
      return s;
    switch (*s) {
    case '(':
      parens++;
      break;
    case ')':
      parens--;
    }
    if(parens==0){
      char *filter;
      number_of_filter++;
      filter=strndup(filterbegin+1,s-filterbegin-1);
      aof.SetSize(number_of_filter+1);      
      aof[number_of_filter]=LDAP_SFilter(filter);
      filterbegin=s+1;
    }
    if(parens<0)
      return --s;
    s++;
  }
  return s;
}


static int ldap_is_attr_desc ( const char *attr )
{
	/* cheap attribute description check */
	int i, c;

	for( i = 0; (c = attr[i]) != 0; i++ ) {
		if (( c >= '0' && c <= '9' )
			|| ( c >= 'A' && c <= 'Z' )
			|| ( c >= 'a' && c <= 'z' )
			|| ( c == '.' || c == '-' )
			|| ( c == ';' )) continue;

		return 0;
	}

	return i > 0;
}

char * find_wildcard( const char *s )
{
  for( ; *s != '\0' ; s++ ) {
    switch( *s ) {
    case '*':	/* found wildcard */
      return (char *) s;
      
    case '\\':
      s++; /* skip over escape */
      if ( *s == '\0' )
	return NULL;	/* escape at end of string */
    }
  }
  
  return NULL;
}

void LDAP_SFilter::set_simple_filter(char *filter) {
  char *str;
  char *s;
  char *value;
  char op;
  
  
  str = strndup(filter,strlen(filter));
  if( str == NULL ) 
    throw length_error("Out of memory");

  if ( (s = strchr( str, '=' )) == NULL ) {
    ldap_memfree(str);
    return;    
  }

  value = s + 1;
  op=*s;
  *s = '\0';

  switch ( op ) {
    case '<': {
      tag=e_lessOrEqual;
      this->CreateObject();
      LDAP_AttributeValueAssertion & ava=(*this);
      ava.m_attributeType=str;
      ava.m_attributeValue=value;
      *s = '\0';
      if(!ldap_is_attr_desc(str)){
	ldap_memfree(str);
	throw invalid_argument("Bad Attribute");
      }
      break;
    }
    case '>': {
      tag=e_greaterOrEqual;
      this->CreateObject();
      LDAP_AttributeValueAssertion &ava=(*this);
      ava.m_attributeType=str;
      ava.m_attributeValue=value;
      *s = '\0';
      if(!ldap_is_attr_desc(str)){
	ldap_memfree(str);
	throw invalid_argument("Bad Attribute");
      }
      break;
    }
    case '~': {
      tag=e_approxMatch;
      this->CreateObject();
      LDAP_AttributeValueAssertion & ava=(*this);
      ava.m_attributeType=str;
      ava.m_attributeValue=value;
      *s = '\0';
      if(!ldap_is_attr_desc(str)) {
	ldap_memfree(str);
	throw invalid_argument("Bad Attribute");
      }
      break;
    }
    default:
      if ( find_wildcard( value ) == NULL ) {
	tag=e_equalityMatch;
	this->CreateObject();
	LDAP_AttributeValueAssertion &ava=(*this);
	ava.m_attributeType=str;
	ava.m_attributeValue=value;
      } else if ( strcmp( value, "*" ) == 0 ) {
	tag=e_present;
	this->CreateObject();
	LDAP_AttributeType &attrib=(*this); // Looks funny, does it?
	attrib.SetValue(str);
	ldap_memfree(str);
	return;
      } else {
	tag=e_substrings;
	this->CreateObject();
	LDAP_SubstringFilter &substr=(*this);
	substr.m_type=str;
	char *begin=value;
	int number_in_sequence=0;
	int initial_value=1;
	while(*value!='\0') {
	  substr.m_value.SetSize(number_in_sequence+1);
	  switch (*value){
	    case '\\':
	      *value++;
	      if (*value=='\0') {
		substr.m_value[number_in_sequence]=LDAP_SubstringFilter_value_subtype(LDAP_SubstringFilter_value_subtype::e_final);
		substr.m_value[number_in_sequence].CreateObject();
		LDAP_LDAPString &lstr=substr.m_value[number_in_sequence];
		lstr.SetValue(begin);
		ldap_memfree(str);
		return;
	      }
	      break;
	    case '*':
	      *value='\0';
	      if(begin==value){ /* first char a * */
		initial_value=0;
		value++;
		begin++;
		continue;
	      }
	      if(initial_value){
		initial_value=0;
		substr.m_value[number_in_sequence]=LDAP_SubstringFilter_value_subtype(LDAP_SubstringFilter_value_subtype::e_initial);
		substr.m_value[number_in_sequence].CreateObject();
		LDAP_LDAPString &lstr=substr.m_value[number_in_sequence];
		lstr.SetValue(begin);
	      } else {
		substr.m_value[number_in_sequence]=LDAP_SubstringFilter_value_subtype(LDAP_SubstringFilter_value_subtype::e_any);
		substr.m_value[number_in_sequence].CreateObject();
		LDAP_LDAPString &lstr=substr.m_value[number_in_sequence];
		lstr.SetValue(begin);
	      }
	      begin=++value;
	      break;
	    case '\0':
	      substr.m_value[number_in_sequence]=LDAP_SubstringFilter_value_subtype(LDAP_SubstringFilter_value_subtype::e_final);
	      substr.m_value[number_in_sequence].CreateObject();
	      LDAP_LDAPString &lstr=substr.m_value[number_in_sequence];
	      lstr.SetValue(begin);
	      break;
	  }
	  value++;
	}
	ldap_memfree(str);
	return;
      }
  }

  ldap_memfree(str);
  return;
}
//
// End of LDAP_SFilter.cxx
//
