/*
 *	$Id: x_term_manager.c,v 1.71 2003/09/15 13:49:14 arakiken Exp $
 */

#include  "x_term_manager.h"

#include  <stdio.h>		/* sprintf */
#include  <string.h>		/* memset/memcpy */
#include  <pwd.h>		/* getpwuid */
#include  <sys/time.h>		/* timeval */
#include  <unistd.h>		/* getpid/select/unlink */
#include  <sys/wait.h>		/* wait */
#include  <signal.h>		/* kill */
#include  <stdlib.h>		/* getenv */
#include  <errno.h>
#include  <fcntl.h>
#include  <kiklib/kik_debug.h>
#include  <kiklib/kik_str.h>	/* kik_str_sep/kik_str_to_int/kik_str_alloca_dup/strdup */
#include  <kiklib/kik_path.h>	/* kik_basename */
#include  <kiklib/kik_util.h>	/* DIGIT_STR_LEN */
#include  <kiklib/kik_mem.h>	/* alloca/kik_alloca_garbage_collect/malloc/free */
#include  <kiklib/kik_conf.h>
#include  <kiklib/kik_conf_io.h>
#include  <kiklib/kik_net.h>	/* socket/bind/listen/sockaddr_un */
#include  <kiklib/kik_types.h>	/* u_int */
#include  <kiklib/kik_sig_child.h>

#include  "version.h"
#include  "x_xim.h"
#include  "x_sb_screen.h"
#include  "x_display.h"
#include  "x_termcap.h"
#include  "x_imagelib.h"
#include  "ml_term_manager.h"


#define  MAX_SCREENS  (8*sizeof(dead_mask))


typedef struct main_config
{
	int  x ;
	int  y ;
	int  geom_hint ;
	u_int  cols ;
	u_int  rows ;
	u_int  screen_width_ratio ;
	u_int  screen_height_ratio ;
	u_int  font_size ;
	u_int  num_of_log_lines ;
	u_int  line_space ;
	u_int  tab_size ;
	ml_iscii_lang_type_t  iscii_lang_type ;
	x_mod_meta_mode_t  mod_meta_mode ;
	x_bel_mode_t  bel_mode ;
	x_sb_mode_t  sb_mode ;
	u_int  col_size_a ;
	ml_char_encoding_t  encoding ;
	int  is_auto_encoding ;
	x_font_present_t  font_present ;
	ml_vertical_mode_t  vertical_mode ;
	ml_bs_mode_t  bs_mode ;
	ml_unicode_font_policy_t  unicode_font_policy ;

	char *  disp_name ;
	char *  app_name ;
	char *  title ;
	char *  icon_name ;
	char *  term_type ;
	char *  scrollbar_view_name ;
	char *  pic_file_path ;
	char *  conf_menu_path_1 ;
	char *  conf_menu_path_2 ;
	char *  conf_menu_path_3 ;
	char *  fg_color ;
	char *  bg_color ;
	char *  cursor_fg_color ;
	char *  cursor_bg_color ;
	char *  sb_fg_color ;
	char *  sb_bg_color ;
	char *  mod_meta_key ;
	char *  icon_path ;
	char *  init_str ;
	char *  cmd_path ;
	char **  cmd_argv ;
	
	u_int8_t  step_in_changing_font_size ;
	u_int16_t  brightness ;
	u_int16_t  contrast ;
	u_int16_t  gamma ;
	u_int8_t  fade_ratio ;
	int8_t  use_scrollbar ;
	int8_t  use_login_shell ;
	int8_t  xim_open_in_startup ;
	int8_t  use_bidi ;
	int8_t  big5_buggy ;
	int8_t  iso88591_font_for_usascii ;
	int8_t  receive_string_via_ucs ;
	int8_t  use_transbg ;
	int8_t  use_char_combining ;
	int8_t  use_multi_col_char ;
	int8_t  use_vertical_cursor ;
	int8_t  use_extended_scroll_shortcut ;
	int8_t  use_dynamic_comb ;
	int8_t  logging_vt_seq ;

	/* cache */
	x_termcap_entry_t *  tent ;
	
} main_config_t ;


/* --- static variables --- */

static x_screen_t **  screens ;
static u_int  num_of_screens ;
static u_long  dead_mask ;

static u_int  num_of_startup_screens ;
static u_int  num_of_startup_ptys ;

static x_system_event_listener_t  system_listener ;

static char *  version ;

static main_config_t  main_config ;

static x_color_custom_t  color_custom ;
static x_shortcut_t  shortcut ;
static x_termcap_t  termcap ;

static int  sock_fd ;
static char *  un_file ;
static int8_t  is_genuine_daemon ;


/* --- static functions --- */

static int
get_font_size_range(
	u_int *  min ,
	u_int *  max ,
	char *  str
	)
{
	char *  str_p ;
	char *  p ;

	if( ( str_p = kik_str_alloca_dup(str)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " alloca() failed.\n") ;
	#endif

		return  0 ;
	}

	if( ( p = kik_str_sep( &str_p , "-")) == NULL)
	{
		return  0 ;
	}

	if( ! kik_str_to_uint( min , p))
	{
		kik_msg_printf( "min font size %s is not valid.\n" , p) ;

		return  0 ;
	}

	if( ! kik_str_to_uint( max , str_p))
	{
		kik_msg_printf( "max font size %s is not valid.\n" , str_p) ;

		return  0 ;
	}

	return  1 ;
}

static ml_term_t *
create_term_intern(void)
{
	ml_term_t *  term ;
	
	if( ( term = ml_create_term( main_config.cols , main_config.rows ,
			main_config.tab_size , main_config.num_of_log_lines ,
			main_config.encoding , main_config.is_auto_encoding , 
			main_config.unicode_font_policy ,
			main_config.col_size_a , main_config.use_char_combining ,
			main_config.use_multi_col_char , main_config.use_bidi ,
			x_termcap_get_bool_field( main_config.tent , ML_BCE) ,
			main_config.use_dynamic_comb , main_config.bs_mode ,
			main_config.vertical_mode , main_config.iscii_lang_type)) == NULL)
	{
		return  NULL ;
	}

	if( main_config.title)
	{
		ml_term_set_window_name( term , main_config.title) ;
	}
	
	if( main_config.icon_name)
	{
		ml_term_set_icon_name( term , main_config.icon_name) ;
	}

	if( main_config.logging_vt_seq)
	{
		ml_term_enable_logging_vt_seq( term) ;
	}

	return  term ;
}

static int
open_pty_intern(
	ml_term_t *  term ,
	char *  cmd_path ,
	char **  cmd_argv ,
	char *  display ,
	Window  window ,
	char *  term_type ,
	int  use_login_shell
	)
{
	char *  env[5] ;	/* MLTERM,TERM,WINDOWID,DISPLAY,NULL */
	char **  env_p ;
	char  wid_env[9 + DIGIT_STR_LEN(Window) + 1] ;	/* "WINDOWID="(9) + [32bit digit] + NULL(1) */
	char *  ver_env ;
	char *  disp_env ;
	char *  term_env ;
	
	env_p = env ;

	if( version && ( ver_env = alloca( 7 + strlen( version) + 1)))
	{
		sprintf( ver_env , "MLTERM=%s" , version) ;
		*(env_p ++) = ver_env ;
	}
	
	sprintf( wid_env , "WINDOWID=%ld" , window) ;
	*(env_p ++) = wid_env ;
	
	/* "DISPLAY="(8) + NULL(1) */
	if( ( disp_env = alloca( 8 + strlen( display) + 1)))
	{
		sprintf( disp_env , "DISPLAY=%s" , display) ;
		*(env_p ++) = disp_env ;
	}

	/* "TERM="(5) + NULL(1) */
	if( ( term_env = alloca( 5 + strlen( term_type) + 1)))
	{
		sprintf( term_env , "TERM=%s" , term_type) ;
		*(env_p ++) = term_env ;
	}

	/* NULL terminator */
	*env_p = NULL ;

	if( ! cmd_path)
	{
		struct passwd *  pw ;

		/*
		 * SHELL env var -> /etc/passwd -> /bin/sh
		 */
		if( ( cmd_path = getenv( "SHELL")) == NULL || *cmd_path == '\0')
		{
			if( ( pw = getpwuid(getuid())) == NULL ||
				*( cmd_path = pw->pw_shell) == '\0')
			{
				cmd_path = "/bin/sh" ;
			}
		}
	}

	if( ! cmd_argv)
	{
		char *  cmd_file ;
		
		if( ( cmd_argv = alloca( sizeof( char*) * 2)) == NULL)
		{
		#ifdef  DEBUG
			kik_warn_printf( KIK_DEBUG_TAG " alloca() failed.\n") ;
		#endif

			return  0 ;
		}

		cmd_file = kik_basename( cmd_path) ;

		/* 2 = `-' and NULL */
		if( ( cmd_argv[0] = alloca( strlen( cmd_file) + 2)) == NULL)
		{
			return  0 ;
		}

		if( use_login_shell)
		{
			sprintf( cmd_argv[0] , "-%s" , cmd_file) ;
		}
		else
		{
			strcpy( cmd_argv[0] , cmd_file) ;
		}

		cmd_argv[1] = NULL ;
	}

	return  ml_term_open_pty( term , cmd_path , cmd_argv , env , display) ;
}

static int
open_screen_intern(
	char *  pty
	)
{
	ml_term_t *  term ;
	x_display_t *  disp ;
	x_screen_t *  screen ;
	x_sb_screen_t *  sb_screen ;
	x_font_manager_t *  font_man ;
	x_color_manager_t *  color_man ;
	x_window_t *  root ;
	mkf_charset_t  usascii_font_cs ;
	int  usascii_font_cs_changable ;
	void *  p ;
	
	/*
	 * these are dynamically allocated.
	 */
	term = NULL ;
	disp = NULL ;
	font_man = NULL ;
	color_man = NULL ;
	screen = NULL ;
	sb_screen = NULL ;
	root = NULL ;

	if( MAX_SCREENS <= num_of_screens)
	{
		return  0 ;
	}

	if( pty)
	{
		if( ( term = ml_get_detached_term( pty)) == NULL)
		{
			return  0 ;
		}
	}
	else
	{
	#if  0
		if( ( term = ml_get_detached_term( NULL)) == NULL &&
			( term = create_term_intern()) == NULL)
	#else
		if( ( term = create_term_intern()) == NULL)
	#endif
		{
			return  0 ;
		}
	}

	if( ( disp = x_display_open( main_config.disp_name)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_display_open failed.\n") ;
	#endif
	
		goto  error ;
	}

	if( main_config.unicode_font_policy == NOT_USE_UNICODE_FONT ||
		main_config.iso88591_font_for_usascii)
	{
		usascii_font_cs = x_get_usascii_font_cs( ML_ISO8859_1) ;
		usascii_font_cs_changable = 0 ;
	}
	else if( main_config.unicode_font_policy == ONLY_USE_UNICODE_FONT)
	{
		usascii_font_cs = x_get_usascii_font_cs( ML_UTF8) ;
		usascii_font_cs_changable = 0 ;
	}
	else
	{
		usascii_font_cs = x_get_usascii_font_cs( ml_term_get_encoding( term)) ;
		usascii_font_cs_changable = 1 ;
	}
	
	if( ( font_man = x_font_manager_new( disp->display ,
		main_config.font_present , main_config.font_size ,
		usascii_font_cs , usascii_font_cs_changable ,
		main_config.use_multi_col_char ,
		main_config.step_in_changing_font_size)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_font_manager_new() failed.\n") ;
	#endif
	
		goto  error ;
	}

	if( ( color_man = x_color_manager_new( disp->display ,
				DefaultScreen( disp->display) , &color_custom ,
				main_config.fg_color , main_config.bg_color ,
				main_config.cursor_fg_color , main_config.cursor_bg_color)) == NULL)
	{
		goto  error ;
	}

	if( ( screen = x_screen_new( term , font_man , color_man , main_config.tent ,
			main_config.brightness , main_config.contrast , main_config.gamma ,
			main_config.fade_ratio , &shortcut ,
			main_config.screen_width_ratio , main_config.screen_height_ratio ,
			main_config.xim_open_in_startup , main_config.mod_meta_key ,
			main_config.mod_meta_mode , main_config.bel_mode ,
			main_config.receive_string_via_ucs , main_config.pic_file_path ,
			main_config.use_transbg , main_config.use_vertical_cursor ,
			main_config.big5_buggy , main_config.conf_menu_path_1 ,
			main_config.conf_menu_path_2 , main_config.conf_menu_path_3 ,
			main_config.use_extended_scroll_shortcut , main_config.line_space)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_screen_new() failed.\n") ;
	#endif

		goto  error ;
	}

	if( ! x_set_system_listener( screen , &system_listener))
	{
		goto  error ;
	}

	if( main_config.use_scrollbar)
	{
		if( ( sb_screen = x_sb_screen_new( screen ,
					main_config.scrollbar_view_name ,
					main_config.sb_fg_color , main_config.sb_bg_color ,
					main_config.sb_mode)) == NULL)
		{
		#ifdef  DEBUG
			kik_warn_printf( KIK_DEBUG_TAG " x_sb_screen_new() failed.\n") ;
		#endif

			goto  error ;
		}

		root = &sb_screen->window ;
	}
	else
	{
		root = &screen->window ;
	}

	if( ! x_window_manager_show_root( &disp->win_man , root ,
		main_config.x , main_config.y , main_config.geom_hint , main_config.app_name))
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_window_manager_show_root() failed.\n") ;
	#endif
	
		goto  error ;
	}

	if( main_config.icon_path)
	{
		if( !(disp->win_man.icon) && !(disp->win_man.mask) && !(disp->win_man.cardinal))
		{
			x_window_t  dummy ;
			int  icon_size = 48 ;
			x_imagelib_load_file( disp->display , main_config.icon_path,
					      &(disp->win_man.cardinal),
					      &(disp->win_man.icon),
					      &(disp->win_man.mask),
					      &icon_size ,&icon_size) ;

			dummy.my_window = disp->win_man.group_leader ;
			dummy.display = disp->win_man.display ;
			x_window_set_icon( &dummy,
					   disp->win_man.icon,
					   disp->win_man.mask,
					   disp->win_man.cardinal) ;
		}
		x_window_set_icon( root,
				   disp->win_man.icon,
				   disp->win_man.mask,
				   disp->win_man.cardinal) ;

	}

	if( pty && main_config.cmd_argv)
	{
		int  count ;
		
		for( count = 0 ; main_config.cmd_argv[count] ; count ++)
		{
			ml_term_write( term , main_config.cmd_argv[count] ,
				strlen( main_config.cmd_argv[count]) , 0) ;
			ml_term_write( term , " " , 1 , 0) ;
		}

		ml_term_write( term , "\n" , 1 , 0) ;
	}
	else
	{
		if( ! open_pty_intern( term , main_config.cmd_path , main_config.cmd_argv ,
			XDisplayString( disp->display) , root->my_window ,
			main_config.term_type , main_config.use_login_shell))
		{
			goto  error ;
		}
	}

	if( main_config.init_str)
	{
		ml_term_write( term , main_config.init_str , strlen( main_config.init_str) , 0) ;
	}
	
	if( ( p = realloc( screens , sizeof( x_screen_t*) * (num_of_screens + 1))) == NULL)
	{
		goto  error ;
	}

	screens = p ;
	screens[num_of_screens++] = screen ;
	
	return  1 ;
	
error:
	if( font_man)
	{
		x_font_manager_delete( font_man) ;
	}

	if( color_man)
	{
		x_color_manager_delete( color_man) ;
	}

	if( ! root || ! x_window_manager_remove_root( &disp->win_man , root))
	{
		/*
		 * If root is still NULL or is not registered to win_man yet.
		 */
		 
		if( screen)
		{
			x_screen_delete( screen) ;
		}

		if( sb_screen)
		{
			x_sb_screen_delete( sb_screen) ;
		}
	}

	if( disp)
	{
		x_display_close( disp) ;
	}

	return  0 ;
}

static int
close_screen_intern(
	x_screen_t *  screen
	)
{
	x_window_t *  root ;
	x_window_manager_t *  win_man ;

	x_screen_detach( screen) ;
	x_font_manager_delete( screen->font_man) ;
	x_color_manager_delete( screen->color_man) ;

	root = x_get_root_window( &screen->window) ;
	win_man = root->win_man ;
	
	if( win_man->num_of_roots == 1)
	{
		Display *  display_to_close ;

		display_to_close = root->display ;
		x_window_manager_remove_root( win_man , root) ;
		x_display_close_2( display_to_close) ;
	}
	else
	{
		x_window_manager_remove_root( win_man , root) ;
	}

	return  1 ;
}


/*
 * callbacks of x_system_event_listener_t
 */
 
/*
 * EXIT_PROGRAM shortcut calls this at last.
 * this is for debugging.
 */
#ifdef  KIK_DEBUG
#include  <kiklib/kik_locale.h>		/* kik_locale_final */
#endif
static void
__exit(
	void *  p ,
	int  status
	)
{
#ifdef  KIK_DEBUG
	x_term_manager_final() ;
	
	kik_locale_final() ;

	kik_alloca_garbage_collect() ;

	kik_msg_printf( "reporting unfreed memories --->\n") ;
	kik_mem_free_all() ;
#endif
	
	if( un_file)
	{
		unlink( un_file) ;
	}
	
	exit(status) ;
}

static void
open_pty(
	void *  p ,
	x_screen_t *  screen ,
	char *  dev
	)
{
	int  count ;

	for( count = 0 ; count < num_of_screens ; count ++)
	{
		if( screen == screens[count])
		{
			ml_term_t *  new ;

			if( dev)
			{
				if( ( new = ml_get_detached_term( dev)) == NULL)
				{
					return ;
				}
			}
			else
			{
				if( ( new = create_term_intern()) == NULL)
				{
					return ;
				}

				if( ! open_pty_intern( new , main_config.cmd_path , main_config.cmd_argv ,
					XDisplayString( screen->window.display) ,
					x_get_root_window( &screen->window)->my_window ,
					main_config.term_type , main_config.use_login_shell))
				{
					return ;
				}
			}
			
			x_screen_detach( screen) ;
			x_screen_attach( screen , new) ;
			
			return ;
		}
	}
}

static void
next_pty(
	void *  p ,
	x_screen_t *  screen
	)
{
	int  count ;
	
	for( count = 0 ; count < num_of_screens ; count ++)
	{
		if( screen == screens[count])
		{
			ml_term_t *  old ;
			ml_term_t *  new ;

			if( ( old = x_screen_detach( screen)) == NULL)
			{
				return ;
			}

			if( ( new = ml_next_term( old)) == NULL)
			{
				x_screen_attach( screen , old) ;
			}
			else
			{
				x_screen_attach( screen , new) ;
			}
			
			return ;
		}
	}
}

static void
prev_pty(
	void *  p ,
	x_screen_t *  screen
	)
{
	int  count ;
	
	for( count = 0 ; count < num_of_screens ; count ++)
	{
		if( screen == screens[count])
		{
			ml_term_t *  old ;
			ml_term_t *  new ;

			if( ( old = x_screen_detach( screen)) == NULL)
			{
				return ;
			}
			
			if( ( new = ml_prev_term( old)) == NULL)
			{
				x_screen_attach( screen , old) ;
			}
			else
			{
				x_screen_attach( screen , new) ;
			}
			
			return ;
		}
	}
}
	
static void
pty_closed(
	void *  p ,
	x_screen_t *  screen	/* screen->term was already deleted. */
	)
{
	int  count ;

	for( count = num_of_screens - 1 ; count >= 0 ; count --)
	{
		if( screen == screens[count])
		{
			ml_term_t *  term ;
			
			if( ( term = ml_get_detached_term( NULL)) == NULL)
			{
				close_screen_intern( screen) ;
				screens[count] = screens[--num_of_screens] ;
			}
			else
			{
				x_screen_attach( screen , term) ;
			}

			return ;
		}
	}
}

static void
open_screen(
	void *  p
	)
{
	if( ! open_screen_intern(NULL))
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " open_screen_intern failed.\n") ;
	#endif
	}
}
	
static void
close_screen(
	void *  p ,
	x_screen_t *  screen
	)
{
	int  count ;
	
	for( count = 0 ; count < num_of_screens ; count ++)
	{
		if( screen == screens[count])
		{
			dead_mask |= (1 << count) ;
			
			return ;
		}
	}
}

static ml_term_t *
get_pty(
	void *  p ,
	char *  dev
	)
{
	return  ml_get_term( dev) ;
}

static char *
pty_list(
	void *  p
	)
{
	return  ml_get_pty_list() ;
}


/*
 * signal handlers.
 */
 
static void
sig_fatal( int  sig)
{
#ifdef  DEBUG
	kik_warn_printf( KIK_DEBUG_TAG "signal %d is received\n" , sig) ;
#endif

	if( un_file)
	{
		unlink( un_file) ;
	}

	/* reset */
	signal( sig , SIG_DFL) ;
	
	kill( getpid() , sig) ;
}


static int
start_daemon(void)
{
	pid_t  pid ;
	int  sock_fd ;
	struct sockaddr_un  servaddr ;
	char * const path = servaddr.sun_path ;

	memset( &servaddr , 0 , sizeof( servaddr)) ;
	servaddr.sun_family = AF_LOCAL ;
	kik_snprintf( path , sizeof( servaddr.sun_path) - 1 , "/tmp/.mlterm-%d.unix" , getuid()) ;

	if( ( sock_fd = socket( AF_LOCAL , SOCK_STREAM , 0)) < 0)
	{
		return  -1 ;
	}

	while( bind( sock_fd , (struct sockaddr *) &servaddr , sizeof( servaddr)) < 0)
	{
		if( errno == EADDRINUSE)
		{
			if( connect( sock_fd , (struct sockaddr*) &servaddr , sizeof( servaddr)) == 0)
			{
				close( sock_fd) ;
				
				kik_msg_printf( "daemon is already running.\n") ;
				
				return  -1 ;
			}

			kik_msg_printf( "removing stale lock file %s.\n" , path) ;
			
			if( unlink( path) == 0)
			{
				continue ;
			}
		}
		else
		{
			close( sock_fd) ;

			kik_msg_printf( "failed to lock file %s: %s\n" , path , strerror(errno)) ;

			return  -1 ;
		}
	}

	pid = fork() ;

	if( pid == -1)
	{
		return  -1 ;
	}

	if( pid != 0)
	{
		exit(0) ;
	}
	
	/*
	 * child
	 */

	/*
	 * This process becomes a session leader and purged from control terminal.
	 */
	setsid() ;

	/*
	 * SIGHUP signal when the child process exits must not be sent to
	 * the grandchild process.
	 */
	signal( SIGHUP , SIG_IGN) ;

	pid = fork() ;

	if( pid == -1)
	{
		exit(1) ;
	}

	if( pid != 0)
	{
		exit(0) ;
	}

	/*
	 * grandchild
	 */

	if( listen( sock_fd , 1024) < 0)
	{
		close( sock_fd) ;
		unlink( path) ;
		
		return  -1 ;
	}

	un_file = strdup( path) ;

	return  sock_fd ;
}

static kik_conf_t *
get_min_conf(
	int  argc ,
	char **  argv
	)
{
	kik_conf_t *  conf ;
	char *  rcpath ;
	
	if( ( conf = kik_conf_new( "mlterm" ,
		MAJOR_VERSION , MINOR_VERSION , REVISION , PATCH_LEVEL , CVS_REVISION)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " kik_conf_new() failed.\n") ;
	#endif
	
		return  NULL ;
	}

	/*
	 * XXX
	 * "mlterm/core" is for backward compatibility with 1.9.44
	 */
	 
	if( ( rcpath = kik_get_sys_rc_path( "mlterm/core")))
	{
		kik_conf_read( conf , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_user_rc_path( "mlterm/core")))
	{
		kik_conf_read( conf , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_sys_rc_path( "mlterm/main")))
	{
		kik_conf_read( conf , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_user_rc_path( "mlterm/main")))
	{
		kik_conf_read( conf , rcpath) ;
		free( rcpath) ;
	}

	kik_conf_add_opt( conf , '#' , "initstr" , 0 , "init_str" ,
		"initial string sent to pty") ;
	kik_conf_add_opt( conf , '$' , "mc" , 0 , "click_interval" ,
		"click interval(milisecond)[250]") ;
	kik_conf_add_opt( conf , '%' , "logseq" , 1 , "logging_vt_seq" ,
		"enable logging vt100 sequence") ;
	kik_conf_add_opt( conf , '1' , "wscr" , 0 , "screen_width_ratio" ,
		"screen width in percent against font width [default = 100]") ;
	kik_conf_add_opt( conf , '2' , "hscr" , 0 , "screen_height_ratio" ,
		"screen height in percent against font height [100]") ;
#if defined(USE_IMLIB) || defined(USE_GDK_PIXBUF)
	kik_conf_add_opt( conf , '3' , "contrast" , 0 , "contrast" ,
		"contrast of background image in percent [100]") ;
	kik_conf_add_opt( conf , '4' , "gamma" , 0 , "gamma" ,
		"gamma of background image in percent [100]") ;
#endif
	kik_conf_add_opt( conf , '5' , "big5bug" , 1 , "big5_buggy" ,
		"manage buggy Big5 CTEXT in XFree86 4.1 or earlier [false]") ;
	kik_conf_add_opt( conf , '6' , "stbs" , 1 , "static_backscroll_mode" ,
		"screen is static under backscroll mode [false]") ;
	kik_conf_add_opt( conf , '7' , "bel" , 0 , "bel_mode" , 
		"bel (0x07) mode [none/sound/visual, default = sound]") ;
	kik_conf_add_opt( conf , '8' , "88591" , 1 , "iso88591_font_for_usascii" ,
		"use ISO-8859-1 font for ASCII part of any encoding [false]") ;
	kik_conf_add_opt( conf , '9' , "crfg" , 0 , "cursor_fg_color" ,
		"cursor foreground color") ;
	kik_conf_add_opt( conf , '0' , "crbg" , 0 , "cursor_bg_color" ,
		"cursor background color") ;
#ifdef  ANTI_ALIAS
	kik_conf_add_opt( conf , 'A' , "aa" , 1 , "use_anti_alias" , 
		"use anti-alias font by using Xft [false]") ;
#endif
	kik_conf_add_opt( conf , 'B' , "sbbg" , 0 , "sb_bg_color" , 
		"scrollbar background color") ;
#ifdef  USE_IND
	kik_conf_add_opt( conf , 'C' , "iscii" , 0 , "iscii_lang" , 
		"language to be used in ISCII encoding") ;
#endif
#ifdef  USE_FRIBIDI
	kik_conf_add_opt( conf , 'D' , "bi" , 1 , "use_bidi" , 
		"use bidi (bi-directional text) [false]") ;
#endif
	kik_conf_add_opt( conf , 'E' , "km" , 0 , "ENCODING" , 
		"character encoding [AUTO/ISO-8859-*/EUC-*/UTF-8/...]") ;
	kik_conf_add_opt( conf , 'F' , "sbfg" , 0 , "sb_fg_color" , 
		"scrollbar foreground color") ;
	kik_conf_add_opt( conf , 'G' , "vertical" , 0 , "vertical_mode" ,
		"vertical mode [none/cjk/mongol]") ;
#if defined(USE_IMLIB) || defined(USE_GDK_PIXBUF)
	kik_conf_add_opt( conf , 'H' , "bright" , 0 , "brightness" ,
		"brightness of background image in percent [100]") ;
#endif
	kik_conf_add_opt( conf , 'I' , "icon" , 0 , "icon_name" , 
		"icon name") ;
	kik_conf_add_opt( conf , 'J' , "dyncomb" , 1 , "use_dynamic_comb" ,
		"use dynamic combining [false]") ;
	kik_conf_add_opt( conf , 'K' , "metakey" , 0 , "mod_meta_key" ,
		"specify meta key [none]") ;
	kik_conf_add_opt( conf , 'L' , "ls" , 1 , "use_login_shell" , 
		"turn on login shell [false]") ;
	kik_conf_add_opt( conf , 'M' , "menu" , 0 , "conf_menu_path_3" ,
		"command path of mlconfig (GUI configurator)") ;
	kik_conf_add_opt( conf , 'N' , "name" , 0 , "app_name" , 
		"application name") ;
	kik_conf_add_opt( conf , 'O' , "sbmod" , 0 , "scrollbar_mode" ,
		"scrollbar mode [none/left/right]") ;
	kik_conf_add_opt( conf , 'Q' , "vcur" , 1 , "use_vertical_cursor" ,
		"rearrange cursor key for vertical mode [false]") ;
	kik_conf_add_opt( conf , 'S' , "sbview" , 0 , "scrollbar_view_name" , 
		"scrollbar view name [simple/sample/athena/motif/...]") ;
	kik_conf_add_opt( conf , 'T' , "title" , 0 , "title" , 
		"title name") ;
	kik_conf_add_opt( conf , 'U' , "viaucs" , 1 , "receive_string_via_ucs" ,
		"process received (pasted) strings via Unicode [false]") ;
	kik_conf_add_opt( conf , 'V' , "varwidth" , 1 , "use_variable_column_width" ,
		"variable column width (for proportional/ISCII) [false]") ;
	kik_conf_add_opt( conf , 'X' , "openim" , 1 , "xim_open_in_startup" , 
		"open XIM (X Input Method) in starting up [true]") ;
	kik_conf_add_opt( conf , 'Z' , "multicol" , 1 , "use_multi_column_char" ,
		"fullwidth character occupies two logical columns [true]") ;
	kik_conf_add_opt( conf , 'a' , "ac" , 0 , "col_size_of_width_a" ,
		"columns for Unicode \"EastAsianAmbiguous\" character [1]") ;
	kik_conf_add_opt( conf , 'b' , "bg" , 0 , "bg_color" , 
		"background color") ;
	kik_conf_add_opt( conf , 'd' , "display" , 0 , "display" , 
		"X server to connect") ;
	kik_conf_add_opt( conf , 'f' , "fg" , 0 , "fg_color" , 
		"foreground color") ;
	kik_conf_add_opt( conf , 'g' , "geometry" , 0 , "geometry" , 
		"size (in characters) and position [80x24]") ;
	kik_conf_add_opt( conf , 'k' , "meta" , 0 , "mod_meta_mode" , 
		"mode in pressing meta key [none/esc/8bit]") ;
	kik_conf_add_opt( conf , 'l' , "sl" , 0 , "logsize" , 
		"number of backlog (scrolled lines to save) [128]") ;
	kik_conf_add_opt( conf , 'm' , "comb" , 1 , "use_combining" , 
		"use combining characters [true]") ;
	kik_conf_add_opt( conf , 'n' , "noucsfont" , 1 , "not_use_unicode_font" ,
		"use non-Unicode fonts even in UTF-8 mode [false]") ;
	kik_conf_add_opt( conf , 'o' , "lsp" , 0 , "line_space" ,
		"extra space between lines in pixels [0]") ;
#if defined(USE_IMLIB) || defined(USE_GDK_PIXBUF)
	kik_conf_add_opt( conf , 'p' , "pic" , 0 , "wall_picture" , 
		"path for wallpaper (background) image") ;
#endif
	kik_conf_add_opt( conf , 'q' , "extkey" , 1 , "use_extended_scroll_shortcut" ,
		"use extended scroll shortcut keys [false]") ;
	kik_conf_add_opt( conf , 'r' , "fade" , 0 , "fade_ratio" , 
		"fade ratio in percent when window unfocued [100]") ;
	kik_conf_add_opt( conf , 's' , "sb" , 1 , "use_scrollbar" , 
		"use scrollbar [false]") ;
	kik_conf_add_opt( conf , 't' , "transbg" , 1 , "use_transbg" , 
		"use transparent background [false]") ;
	kik_conf_add_opt( conf , 'u' , "onlyucsfont" , 1 , "only_use_unicode_font" ,
		"use a Unicode font even in non-UTF-8 modes [false]") ;
	kik_conf_add_opt( conf , 'w' , "fontsize" , 0 , "fontsize" , 
		"font size in pixels [16]") ;
	kik_conf_add_opt( conf , 'x' , "tw" , 0 , "tabsize" , 
		"tab width in columns [8]") ;
	kik_conf_add_opt( conf , 'y' , "term" , 0 , "termtype" , 
		"terminal type for TERM variable [xterm]") ;
	kik_conf_add_opt( conf , 'z' ,  "largesmall" , 0 , "step_in_changing_font_size" ,
		"step in changing font size in GUI configurator [1]") ;

	kik_conf_set_end_opt( conf , 'e' , NULL , "exec_cmd" , 
		"execute external command") ;

	return  conf ;	
}

static int
config_init(
	kik_conf_t *  conf ,
	int  argc ,
	char **  argv
	)
{
	char *  value ;
	
	if( ( value = kik_conf_get_value( conf , "display")) == NULL)
	{
		value = "" ;
	}

	if( ( main_config.disp_name = strdup( value)) == NULL)
	{
		return  0 ;
	}
	
	if( ( value = kik_conf_get_value( conf , "fontsize")) == NULL)
	{
		main_config.font_size = 16 ;
	}
	else if( ! kik_str_to_uint( &main_config.font_size , value))
	{
		kik_msg_printf( "font size %s is not valid.\n" , value) ;

		/* default value is used. */
		main_config.font_size = 16 ;
	}

	if( main_config.font_size > x_get_max_font_size())
	{
		kik_msg_printf( "font size %d is too large. %d is used.\n" ,
			main_config.font_size , x_get_max_font_size()) ;
		
		main_config.font_size = x_get_max_font_size() ;
	}
	else if( main_config.font_size < x_get_min_font_size())
	{
		kik_msg_printf( "font size %d is too small. %d is used.\n" ,
			main_config.font_size , x_get_min_font_size()) ;
			
		main_config.font_size = x_get_min_font_size() ;
	}

	main_config.app_name = NULL ;

	if( ( value = kik_conf_get_value( conf , "app_name")))
	{
		main_config.app_name = strdup( value) ;
	}

	main_config.title = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "title")))
	{
		main_config.title = strdup( value) ;
	}

	main_config.icon_name = NULL ;

	if( ( value = kik_conf_get_value( conf , "icon_name")))
	{
		main_config.icon_name = strdup( value) ;
	}

	/*
	 * conf_menu_path_[12] are set x_term_manager_init() once.
	 * conf_menu_path_3 alone is changable.
	 */	
	main_config.conf_menu_path_3 = NULL ;

	if( ( value = kik_conf_get_value( conf , "conf_menu_path_3")))
	{
		main_config.conf_menu_path_3 = strdup( value) ;
	}

	/* use default value */
	main_config.scrollbar_view_name = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "scrollbar_view_name")))
	{
		main_config.scrollbar_view_name = strdup( value) ;
	}

	main_config.use_char_combining = 1 ;
	
	if( ( value = kik_conf_get_value( conf , "use_combining")))
	{
		if( strcmp( value , "false") == 0)
		{
			main_config.use_char_combining = 0 ;
		}
	}

	main_config.use_dynamic_comb = 0 ;
	if( ( value = kik_conf_get_value( conf , "use_dynamic_comb")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.use_dynamic_comb = 1 ;
		}
	}

	main_config.logging_vt_seq = 0 ;
	if( ( value = kik_conf_get_value( conf , "logging_vt_seq")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.logging_vt_seq = 1 ;
		}
	}

	main_config.font_present = 0 ;

	if( ( value = kik_conf_get_value( conf , "use_variable_column_width")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.font_present |= FONT_VAR_WIDTH ;
		}
	}

	main_config.step_in_changing_font_size = 1 ;
	
	if( ( value = kik_conf_get_value( conf , "step_in_changing_font_size")))
	{
		u_int  size ;
		
		if( kik_str_to_uint( &size , value))
		{
			main_config.step_in_changing_font_size = size ;
		}
		else
		{
			kik_msg_printf( "step in changing font size %s is not valid.\n" , value) ;
		}
	}

#ifdef  ANTI_ALIAS	
	if( ( value = kik_conf_get_value( conf , "use_anti_alias")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.font_present |= FONT_AA ;
		}
	}
#endif

	main_config.vertical_mode = 0 ;
	
	if( ( value = kik_conf_get_value( conf , "vertical_mode")))
	{
		if( ( main_config.vertical_mode = ml_get_vertical_mode( value)))
		{
			/*
			 * vertical font is automatically used under vertical mode.
			 * similler processing is done in x_screen.c:change_vertical_mode.
			 */
			main_config.font_present |= FONT_VERTICAL ;
			main_config.font_present &= ~FONT_VAR_WIDTH ;
		}
	}

	main_config.fg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "fg_color")))
	{
		main_config.fg_color = strdup( value) ;
	}

	main_config.bg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "bg_color")))
	{
		main_config.bg_color = strdup( value) ;
	}

	main_config.cursor_fg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "cursor_fg_color")))
	{
		main_config.cursor_fg_color = strdup( value) ;
	}

	main_config.cursor_bg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "cursor_bg_color")))
	{
		main_config.cursor_bg_color = strdup( value) ;
	}
	
	main_config.sb_fg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "sb_fg_color")))
	{
		main_config.sb_fg_color = strdup( value) ;
	}

	main_config.sb_bg_color = NULL ;
	
	if( ( value = kik_conf_get_value( conf , "sb_bg_color")))
	{
		main_config.sb_bg_color = strdup( value) ;
	}
	
	if( ( value = kik_conf_get_value( conf , "termtype")))
	{
		main_config.term_type = strdup( value) ;
	}
	else
	{
		main_config.term_type = strdup( "xterm") ;
	}
	
	main_config.tent = x_termcap_get_entry( &termcap , main_config.term_type) ;
	
	main_config.x = 0 ;
	main_config.y = 0 ;
	main_config.cols = 80 ;
	main_config.rows = 24 ;
	if( ( value = kik_conf_get_value( conf , "geometry")))
	{
		/* For each value not found, the argument is left unchanged.(see man XParseGeometry(3)) */
		main_config.geom_hint = XParseGeometry( value , &main_config.x , &main_config.y ,
						&main_config.cols , &main_config.rows) ;

		if( main_config.cols == 0 || main_config.rows == 0)
		{
			kik_msg_printf( "geometry option %s is illegal.\n" , value) ;
			
			main_config.cols = 80 ;
			main_config.rows = 24 ;
		}
	}
	else
	{
		main_config.geom_hint = 0 ;
	}

	main_config.screen_width_ratio = 100 ;
	
	if( ( value = kik_conf_get_value( conf , "screen_width_ratio")))
	{
		u_int  ratio ;

		if( kik_str_to_uint( &ratio , value))
		{
			main_config.screen_width_ratio = ratio ;
		}
	}

	main_config.screen_height_ratio = 100 ;
	
	if( ( value = kik_conf_get_value( conf , "screen_height_ratio")))
	{
		u_int  ratio ;

		if( kik_str_to_uint( &ratio , value))
		{
			main_config.screen_height_ratio = ratio ;
		}
	}
	
	main_config.use_multi_col_char = 1 ;

	if( ( value = kik_conf_get_value( conf , "use_multi_column_char")))
	{
		if( strcmp( value , "false") == 0)
		{
			main_config.use_multi_col_char = 0 ;
		}
	}

	main_config.line_space = 0 ;

	if( ( value = kik_conf_get_value( conf , "line_space")))
	{
		u_int  size ;

		if( kik_str_to_uint( &size , value))
		{
			main_config.line_space = size ;
		}
		else
		{
			kik_msg_printf( "line space %s is not valid.\n" , value) ;
		}
	}

	if( ( value = kik_conf_get_value( conf , "logsize")) == NULL)
	{
		main_config.num_of_log_lines = 128 ;
	}
	else if( ! kik_str_to_uint( &main_config.num_of_log_lines , value))
	{
		kik_msg_printf( "log size %s is not valid.\n" , value) ;

		/* default value is used. */
		main_config.num_of_log_lines = 128 ;
	}

	if( ( value = kik_conf_get_value( conf , "tabsize")) == NULL)
	{
		/* default value is used. */
		main_config.tab_size = 8 ;
	}
	else if( ! kik_str_to_uint( &main_config.tab_size , value))
	{
		kik_msg_printf( "tab size %s is not valid.\n" , value) ;

		/* default value is used. */
		main_config.tab_size = 8 ;
	}
	
	main_config.use_login_shell = 0 ;
	
	if( ( value = kik_conf_get_value( conf , "use_login_shell")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.use_login_shell = 1 ;
		}
	}

	main_config.big5_buggy = 0 ;

	if( ( value = kik_conf_get_value( conf , "big5_buggy")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.big5_buggy = 1 ;
		}
	}

	main_config.use_scrollbar = 1 ;

	if( ( value = kik_conf_get_value( conf , "use_scrollbar")))
	{
		if( strcmp( value , "false") == 0)
		{
			main_config.use_scrollbar = 0 ;
		}
	}

	if( ( value = kik_conf_get_value( conf , "scrollbar_mode")))
	{
		main_config.sb_mode = x_get_sb_mode( value) ;
	}
	else
	{
		main_config.sb_mode = SB_LEFT ;
	}

	main_config.iso88591_font_for_usascii = 0 ;

	if( ( value = kik_conf_get_value( conf , "iso88591_font_for_usascii")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.iso88591_font_for_usascii = 1 ;
		}
	}

	main_config.unicode_font_policy = 0 ;

	if( ( value = kik_conf_get_value( conf , "not_use_unicode_font")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.unicode_font_policy = NOT_USE_UNICODE_FONT ;
		}
	}

	if( ( value = kik_conf_get_value( conf , "only_use_unicode_font")))
	{
		if( strcmp( value , "true") == 0)
		{
			if( main_config.unicode_font_policy == NOT_USE_UNICODE_FONT)
			{
				kik_msg_printf(
					"only_use_unicode_font and not_use_unicode_font options "
					"cannot be used at the same time.\n") ;

				/* default values are used */
				main_config.unicode_font_policy = 0 ;
			}
			else
			{
				main_config.unicode_font_policy = ONLY_USE_UNICODE_FONT ;
			}
		}
	}

	main_config.receive_string_via_ucs = 0 ;

	if( ( value = kik_conf_get_value( conf , "receive_string_via_ucs")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.receive_string_via_ucs = 1 ;
		}
	}
	
	/* default value is used */
	main_config.col_size_a = 1 ;
	
	if( ( value = kik_conf_get_value( conf , "col_size_of_width_a")))
	{
		u_int  col_size_a ;
		
		if( kik_str_to_uint( &col_size_a , value))
		{
			main_config.col_size_a = col_size_a ;
		}
		else
		{
			kik_msg_printf( "col size of width a %s is not valid.\n" , value) ;
		}
	}

	main_config.pic_file_path = NULL ;

	if( ( value = kik_conf_get_value( conf , "wall_picture")))
	{
		if( *value != '\0')
		{
			main_config.pic_file_path = strdup( value) ;
		}
	}

	main_config.use_transbg = 0 ;

	if( ( value = kik_conf_get_value( conf , "use_transbg")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.use_transbg = 1 ;
		}
	}

	if( main_config.pic_file_path && main_config.use_transbg)
	{
		kik_msg_printf(
			"wall picture and transparent background cannot be used at the same time.\n") ;

		/* using wall picture */
		main_config.use_transbg = 0 ;
	}

	main_config.brightness = 100 ;

	if( ( value = kik_conf_get_value( conf , "brightness")))
	{
		u_int  brightness ;
		
		if( kik_str_to_uint( &brightness , value))
		{
			main_config.brightness = brightness ;
		}
		else
		{
			kik_msg_printf( "shade ratio %s is not valid.\n" , value) ;
		}
	}
	
	main_config.contrast = 100 ;

	if( ( value = kik_conf_get_value( conf , "contrast")))
	{
		u_int  contrast ;
		
		if( kik_str_to_uint( &contrast , value))
		{
			main_config.contrast = contrast ;
		}
		else
		{
			kik_msg_printf( "contrast ratio %s is not valid.\n" , value) ;
		}
	}
	
	main_config.gamma = 100 ;

	if( ( value = kik_conf_get_value( conf , "gamma")))
	{
		u_int  gamma ;
		
		if( kik_str_to_uint( &gamma , value))
		{
			main_config.gamma = gamma ;
		}
		else
		{
			kik_msg_printf( "gamma ratio %s is not valid.\n" , value) ;
		}
	}
	
	main_config.fade_ratio = 100 ;
	
	if( ( value = kik_conf_get_value( conf , "fade_ratio")))
	{
		u_int  fade_ratio ;
		
		if( kik_str_to_uint( &fade_ratio , value) && fade_ratio <= 100)
		{
			main_config.fade_ratio = fade_ratio ;
		}
		else
		{
			kik_msg_printf( "fade ratio %s is not valid.\n" , value) ;
		}
	}

	main_config.is_auto_encoding = 0 ;
	if( ( value = kik_conf_get_value( conf , "ENCODING")))
	{
		if( ( main_config.encoding = ml_get_char_encoding( value)) == ML_UNKNOWN_ENCODING)
		{
			kik_msg_printf(
				"%s encoding is not supported. Auto detected encoding is used.\n" ,
				value) ;
				
			main_config.encoding = ml_get_char_encoding( "auto") ;
			main_config.is_auto_encoding = 1 ;
		}
	}
	else
	{
		main_config.encoding = ml_get_char_encoding( "auto") ;
		main_config.is_auto_encoding = 1 ;
	}

	if( main_config.encoding == ML_UNKNOWN_ENCODING)
	{
		main_config.encoding = ML_ISO8859_1 ;
	}

	main_config.xim_open_in_startup = 1 ;
	
	if( ( value = kik_conf_get_value( conf , "xim_open_in_startup")))
	{
		if( strcmp( value , "false") == 0)
		{
			main_config.xim_open_in_startup = 0 ;
		}
	}

	main_config.use_bidi = 1 ;

	if( ( value = kik_conf_get_value( conf , "use_bidi")))
	{
		if( strcmp( value , "false") == 0)
		{
			main_config.use_bidi = 0 ;
		}
	}

	/* If value is "none" or not is also checked in x_screen.c */
	if( ( value = kik_conf_get_value( conf , "mod_meta_key")) &&
		strcmp( value , "none") != 0)
	{
		main_config.mod_meta_key = strdup( value) ;
	}
	else
	{
		main_config.mod_meta_key = NULL ;
	}
	
	if( ( value = kik_conf_get_value( conf , "mod_meta_mode")))
	{
		main_config.mod_meta_mode = x_get_mod_meta_mode( value) ;
	}
	else
	{
		main_config.mod_meta_mode = MOD_META_SET_MSB ;
	}

	if( ( value = kik_conf_get_value( conf , "bel_mode")))
	{
		main_config.bel_mode = x_get_bel_mode( value) ;
	}
	else
	{
		main_config.bel_mode = BEL_SOUND ;
	}

	main_config.use_vertical_cursor = 0 ;

	if( ( value = kik_conf_get_value( conf , "use_vertical_cursor")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.use_vertical_cursor = 1 ;
		}
	}
	
	main_config.iscii_lang_type = ISCIILANG_MALAYALAM ;
	
	if( ( value = kik_conf_get_value( conf , "iscii_lang")))
	{
		ml_iscii_lang_type_t  type ;
		
		if( ( type = ml_iscii_get_lang( value)) != ISCIILANG_UNKNOWN)
		{
			main_config.iscii_lang_type = type ;
		}
	}

	main_config.use_extended_scroll_shortcut = 0 ;
	
	if( ( value = kik_conf_get_value( conf , "use_extended_scroll_shortcut")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.use_extended_scroll_shortcut = 1 ;
		}
	}

	main_config.bs_mode = BSM_VOLATILE ;

	if( ( value = kik_conf_get_value( conf , "static_backscroll_mode")))
	{
		if( strcmp( value , "true") == 0)
		{
			main_config.bs_mode = BSM_STATIC ;
		}
	}

	if( ( value = kik_conf_get_value( conf , "icon_path")))
	{
		main_config.icon_path = strdup( value) ;
	}
	else
	{
		main_config.icon_path = NULL ;
	}

	if( ( value = kik_conf_get_value( conf , "init_str")))
	{
		if( ( main_config.init_str = malloc( strlen( value) + 1)))
		{
			char *  p1 ;
			char *  p2 ;

			p1 = value ;
			p2 = main_config.init_str ;

			while( *p1)
			{
				if( *p1 == '\\')
				{
					p1 ++ ;
					
					if( *p1 == '\0')
					{
						break ;
					}
					else if( *p1 == 'n')
					{
						*(p2 ++) = '\n' ;
					}
					else if( *p1 == 'r')
					{
						*(p2 ++) = '\r' ;
					}
					else if( *p1 == 't')
					{
						*(p2 ++) = '\t' ;
					}
					else if( *p1 == 'e')
					{
						*(p2 ++) = '\e' ;
					}
					else
					{
						*(p2 ++) = *p1 ;
					}
				}
				else
				{
					*(p2 ++) = *p1 ;
				}

				p1 ++ ;
			}

			*p2 = '\0' ;
		}
	}
	else
	{
		main_config.init_str = NULL ;
	}

	if( ( value = kik_conf_get_value( conf , "exec_cmd")) && strcmp( value , "true") == 0)
	{
		if( ( main_config.cmd_argv = malloc( sizeof( char*) * (argc + 1))) == NULL)
		{
		#ifdef  DEBUG
			kik_warn_printf( KIK_DEBUG_TAG " malloc() failed.\n") ;
		#endif
		
			main_config.cmd_path = NULL ;
			main_config.cmd_argv = NULL ;
		}
		else
		{
			/*
			 * !! Notice !!
			 * cmd_path and strings in cmd_argv vector should be allocated
			 * by the caller.
			 */
			 
			main_config.cmd_path = argv[0] ;
			
			memcpy( &main_config.cmd_argv[0] , argv , sizeof( char*) * argc) ;
			main_config.cmd_argv[argc] = NULL ;
		}
	}
	else
	{
		main_config.cmd_path = NULL ;
		main_config.cmd_argv = NULL ;
	}

	return  1 ;
}

static int
config_final(void)
{
	free( main_config.disp_name) ;
	free( main_config.app_name) ;
	free( main_config.title) ;
	free( main_config.icon_name) ;
	free( main_config.term_type) ;
	free( main_config.conf_menu_path_1) ;
	free( main_config.conf_menu_path_2) ;
	free( main_config.conf_menu_path_3) ;
	free( main_config.pic_file_path) ;
	free( main_config.scrollbar_view_name) ;
	free( main_config.fg_color) ;
	free( main_config.bg_color) ;
	free( main_config.cursor_fg_color) ;
	free( main_config.cursor_bg_color) ;
	free( main_config.sb_fg_color) ;
	free( main_config.sb_bg_color) ;
	free( main_config.icon_path) ;
	free( main_config.mod_meta_key) ;
	free( main_config.init_str) ;
	free( main_config.cmd_argv) ;

	return  1 ;
}

static void
client_connected(void)
{
	struct sockaddr_un  addr ;
	socklen_t  sock_len ;
	int  fd ;
	FILE *  fp ;
	kik_file_t *  from ;
	char *  line ;
	size_t  line_len ;
	char *  args ;
	char **  argv ;
	int  argc ;
	char *  args_dup ;
	char *  p ;

	fp = NULL ;

	sock_len = sizeof( addr) ;

	if( ( fd = accept( sock_fd , (struct sockaddr *) &addr , &sock_len)) < 0)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " accept failed.\n") ;
	#endif
	
		return ;
	}

	/*
	 * Set the close-on-exec flag.
	 * If this flag off, this fd remained open until the child process forked in
	 * open_screen_intern()(ml_term_open_pty()) close it.
	 */
	fcntl( fd , F_SETFD , 1) ;

	if( ( fp = fdopen( fd , "r+")) == NULL)
	{
		goto  crit_error ;
	}

	if( ( from = kik_file_new( fp)) == NULL)
	{
		goto  crit_error ;
	}

	if( ( line = kik_file_get_line( from , &line_len)) == NULL)
	{
		kik_file_delete( from) ;
		
		goto  crit_error ;
	}

	if( ( args = alloca( line_len)) == NULL)
	{
		kik_file_delete( from) ;

		goto  error ;
	}

	strncpy( args , line , line_len - 1) ;
	args[line_len - 1] = '\0' ;

	kik_file_delete( from) ;

#ifdef  __DEBUG
	kik_debug_printf( KIK_DEBUG_TAG " %s\n" , args) ;
#endif
	
	/*
	 * parsing options.
	 */

	argc = 0 ;

	if( ( argv = alloca( sizeof( char*) * line_len)) == NULL)
	{
		goto  error ;
	}

	if( ( args_dup = alloca( line_len)) == NULL)
	{
		goto  error ;
	}
	
	p = args_dup ;

	while( *args)
	{
		int  quoted ;

		while( *args == ' ' || *args == '\t')
		{
			if( *args == '\0')
			{
				goto  parse_end ;
			}

			args ++ ;
		}

		if( *args == '\"')
		{
			quoted = 1 ;
			args ++ ;
		}
		else
		{
			quoted = 0 ;
		}
		
		while( *args)
		{
			if( quoted)
			{
				if( *args == '\"')
				{
					args ++ ;
					
					break ;
				}
			}
			else
			{
				if( *args == ' ' || *args == '\t')
				{
					args ++ ;
					
					break ;
				}
			}
			
			if( *args == '\\' && *(args + 1) == '\"')
			{
				*(p ++) = *(++ args) ;
			}
			else
			{
				*(p ++) = *args ;
			}

			args ++ ;
		}

		*(p ++) = '\0' ;
		argv[argc ++] = args_dup ;
		args_dup = p ;
	}
parse_end:

#ifdef  __DEBUG
	{
		int  i ;

		for( i = 0 ; i < argc ; i ++)
		{
			kik_msg_printf( "%s\n" , argv[i]) ;
		}
	}
#endif

	if( argc == 0)
	{
		goto  error ;
	}

	if( argc == 2 &&
		( strncmp( argv[1] , "-P" , 2) == 0 || strncmp( argv[1] , "--ptylist" , 9) == 0))
	{
		/*
		 * mlclient -P or mlclient --ptylist
		 */
		
		ml_term_t **  terms ;
		u_int  num ;
		int  count ;
		
		num = ml_get_all_terms( &terms) ;
		for( count = 0 ; count < num ; count ++)
		{
			fprintf( fp , "%s" , ml_term_get_slave_name( terms[count])) ;
			if( ml_term_window_name( terms[count]))
			{
				fprintf( fp , "(whose title is %s)" , ml_term_window_name( terms[count])) ;
			}
			if( ml_term_is_attached( terms[count]))
			{
				fprintf( fp , " is active:)\n") ;
			}
			else
			{
				fprintf( fp , " is sleeping.zZ\n") ;
			}
		}
	}
	else
	{
		kik_conf_t *  conf ;
		main_config_t  orig_conf ;
		char *  pty ;
		
		if( argc >= 2 && *argv[1] != '-')
		{
			/*
			 * mlclient [dev] [options...]
			 */
			 
			pty = argv[1] ;
			argv[1] = argv[0] ;
			argv = &argv[1] ;
			argc -- ;
		}
		else
		{
			pty = NULL ;
		}

		if( ( conf = get_min_conf( argc , argv)) == NULL)
		{
			goto  error ;
		}

		if( ! kik_conf_parse_args( conf , &argc , &argv))
		{
			kik_conf_delete( conf) ;

			goto  error ;
		}

		orig_conf = main_config ;

		config_init( conf , argc , argv) ;

		kik_conf_delete( conf) ;

		if( ! open_screen_intern( pty))
		{
		#ifdef  DEBUG
			kik_warn_printf( KIK_DEBUG_TAG " open_screen_intern() failed.\n") ;
		#endif
		}
		
		config_final() ;

		main_config = orig_conf ;
	}

	fclose( fp) ;
	
	return ;

error:
	{
		char  msg[] = "Error happened.\n" ;

		write( fd , msg , sizeof( msg)) ;
	}

	/* If fd may be invalid, jump to here directly. */
crit_error:
	if( fp)
	{
		fclose( fp) ;
	}
	else
	{
		close( fd) ;
	}
}

static void
receive_next_event(void)
{
	int  count ;
	int  xfd ;
	int  ptyfd ;
	int  maxfd ;
	int  ret ;
	fd_set  read_fds ;
	struct timeval  tval ;
	x_display_t **  displays ;
	u_int  num_of_displays ;
	ml_term_t **  terms ;
	u_int  num_of_terms ;

	num_of_terms = ml_get_all_terms( &terms) ;
	
	/*
	 * flush buffer
	 */

	for( count = 0 ; count < num_of_terms ; count ++)
	{
		ml_term_flush( terms[count]) ;
	}
	
	while( 1)
	{
		/* on Linux tv_usec,tv_sec members are zero cleared after select() */
		tval.tv_usec = 50000 ;	/* 0.05 sec */
		tval.tv_sec = 0 ;

		maxfd = 0 ;
		FD_ZERO( &read_fds) ;

		displays = x_get_opened_displays( &num_of_displays) ;
		
		for( count = 0 ; count < num_of_displays ; count ++)
		{
			/*
			 * it is necessary to flush events here since some events
			 * may have happened in idling
			 */
			x_window_manager_receive_next_event( &displays[count]->win_man) ;

			xfd = x_display_fd( displays[count]) ;
			
			FD_SET( xfd , &read_fds) ;
		
			if( xfd > maxfd)
			{
				maxfd = xfd ;
			}
		}

		for( count = 0 ; count < num_of_terms ; count ++)
		{
			ptyfd = ml_term_get_pty_fd( terms[count]) ;
			FD_SET( ptyfd , &read_fds) ;

			if( ptyfd > maxfd)
			{
				maxfd = ptyfd ;
			}
		}
		
		if( sock_fd >= 0)
		{
			FD_SET( sock_fd , &read_fds) ;
			
			if( sock_fd > maxfd)
			{
				maxfd = sock_fd ;
			}
		}

		if( ( ret = select( maxfd + 1 , &read_fds , NULL , NULL , &tval)) != 0)
		{
			break ;
		}

		for( count = 0 ; count < num_of_displays ; count ++)
		{
			x_window_manager_idling( &displays[count]->win_man) ;
		}
	}
	
	if( ret < 0)
	{
		/* error happened */
		
		return ;
	}

	/*
	 * Processing order should be as follows.
	 *
	 * PTY -> X WINDOW -> Socket
	 */
	 
	for( count = 0 ; count < num_of_terms ; count ++)
	{
		if( FD_ISSET( ml_term_get_pty_fd( terms[count]) , &read_fds))
		{
			ml_term_parse_vt100_sequence( terms[count]) ;
		}
	}
	
	for( count = 0 ; count < num_of_displays ; count ++)
	{
		if( FD_ISSET( x_display_fd( displays[count]) , &read_fds))
		{
			x_window_manager_receive_next_event( &displays[count]->win_man) ;
		}
	}

	if( sock_fd >= 0)
	{
		if( FD_ISSET( sock_fd , &read_fds))
		{
			client_connected() ;
		}
	}
}


/* --- global functions --- */

int
x_term_manager_init(
	int  argc ,
	char **  argv
	)
{
	kik_conf_t *  conf ;
	int  use_xim ;
	u_int  min_font_size ;
	u_int  max_font_size ;
	char *  rcpath ;
	char *  value ;

	if( ( conf = get_min_conf( argc , argv)) == NULL)
	{
		return  0 ;
	}

	kik_conf_add_opt( conf , '@' , "screens" , 0 , "startup_screens" ,
		"number of screens to open in start up [1]") ;
	kik_conf_add_opt( conf , 'h' , "help" , 1 , "help" ,
		"show this help message") ;
	kik_conf_add_opt( conf , 'v' , "version" , 1 , "version" ,
		"show version message") ;
	kik_conf_add_opt( conf , 'P' , "ptys" , 0 , "startup_ptys" ,
		"number of ptys to open in start up [1]") ;
	kik_conf_add_opt( conf , 'R' , "fsrange" , 0 , "font_size_range" , 
		"font size range for GUI configurator [6-30]") ;
	kik_conf_add_opt( conf , 'W' , "sep" , 0 , "word_separators" , 
		"word-separating characters for double-click [,.:;/@]") ;
	kik_conf_add_opt( conf , 'Y' , "decsp" , 1 , "compose_dec_special_font" ,
		"compose dec special font [false]") ;
#ifdef  ANTI_ALIAS
	kik_conf_add_opt( conf , 'c' , "cp932" , 1 , "use_cp932_ucs_for_xft" , 
		"use CP932-Unicode mapping table for JISX0208 [false]") ;
#endif
	kik_conf_add_opt( conf , 'i' , "xim" , 1 , "use_xim" , 
		"use XIM (X Input Method) [true]") ;
	kik_conf_add_opt( conf , 'j' , "daemon" , 0 , "daemon_mode" ,
		"start as a daemon [none/blend/genuine]") ;
	
	if( ! kik_conf_parse_args( conf , &argc , &argv))
	{
		kik_conf_delete( conf) ;

		return  0 ;
	}

	/*
	 * daemon
	 */

	is_genuine_daemon = 0 ;
	sock_fd = -1 ;
	
	if( ( value = kik_conf_get_value( conf , "daemon_mode")))
	{
		if( strcmp( value , "genuine") == 0)
		{
			if( ( sock_fd = start_daemon()) < 0)
			{
				kik_msg_printf( "mlterm failed to become daemon.\n") ;
			}
			else
			{
				is_genuine_daemon = 1 ;
			}
		}
		else if( strcmp( value , "blend") == 0)
		{
			if( ( sock_fd = start_daemon()) < 0)
			{
				kik_msg_printf( "mlterm failed to become daemon.\n") ;
			}
		}
	#if  0
		else if( strcmp( value , "none") == 0)
		{
		}
	#endif
	}

	use_xim = 1 ;
	
	if( ( value = kik_conf_get_value( conf , "use_xim")))
	{
		if( strcmp( value , "false") == 0)
		{
			use_xim = 0 ;
		}
	}

	x_xim_init( use_xim) ;
	
	if( ( value = kik_conf_get_value( conf , "font_size_range")))
	{
		if( ! get_font_size_range( &min_font_size , &max_font_size , value))
		{
			kik_msg_printf( "font_size_range = %s is illegal.\n" , value) ;

			/* default values are used */
			min_font_size = 6 ;
			max_font_size = 30 ;
		}
	}
	else
	{
		/* default values are used */
		min_font_size = 6 ;
		max_font_size = 30 ;
	}

	x_set_font_size_range( min_font_size , max_font_size) ;

	if( ( value = kik_conf_get_value( conf , "click_interval")))
	{
		int  interval ;

		if( kik_str_to_int( &interval , value))
		{
			x_set_click_interval( interval) ;
		}
	}

	/*
	 * conf_menu_path_3 is set config_init().
	 * conf_menu_path_[12] aren't changable.
	 */	
	main_config.conf_menu_path_1 = NULL ;

	if( ( value = kik_conf_get_value( conf , "conf_menu_path_1")))
	{
		main_config.conf_menu_path_1 = strdup( value) ;
	}

	main_config.conf_menu_path_2 = NULL ;

	if( ( value = kik_conf_get_value( conf , "conf_menu_path_2")))
	{
		main_config.conf_menu_path_2 = strdup( value) ;
	}

	if( ( value = kik_conf_get_value( conf , "compose_dec_special_font")))
	{
		if( strcmp( value , "true") == 0)
		{
			x_compose_dec_special_font() ;
		}
	}

#ifdef  ANTI_ALIAS
	if( ( value = kik_conf_get_value( conf , "use_cp932_ucs_for_xft")) == NULL ||
		strcmp( value , "true") == 0)
	{
		ml_use_cp932_ucs_for_xft() ;
	}
#endif
	
	if( ! x_color_custom_init( &color_custom))
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_color_custom_init failed.\n") ;
	#endif
	
		return  0 ;
	}
	
	if( ( rcpath = kik_get_sys_rc_path( "mlterm/color")))
	{
		x_color_custom_read_conf( &color_custom , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_user_rc_path( "mlterm/color")))
	{
		x_color_custom_read_conf( &color_custom , rcpath) ;
		free( rcpath) ;
	}

	if( ! x_shortcut_init( &shortcut))
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_shortcut_init failed.\n") ;
	#endif
	
		return  0 ;
	}

	if( ( rcpath = kik_get_sys_rc_path( "mlterm/key")))
	{
		x_shortcut_read_conf( &shortcut , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_user_rc_path( "mlterm/key")))
	{
		x_shortcut_read_conf( &shortcut , rcpath) ;
		free( rcpath) ;
	}

	if( ! x_termcap_init( &termcap))
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " x_termcap_init failed.\n") ;
	#endif
	
		return  0 ;
	}

	if( ( rcpath = kik_get_sys_rc_path( "mlterm/termcap")))
	{
		x_termcap_read_conf( &termcap , rcpath) ;
		free( rcpath) ;
	}
	
	if( ( rcpath = kik_get_user_rc_path( "mlterm/termcap")))
	{
		x_termcap_read_conf( &termcap , rcpath) ;
		free( rcpath) ;
	}
	
	/*
	 * others
	 */

	num_of_startup_screens = 1 ;
	
#if  0
	if( ( value = kik_conf_get_value( conf , "ptys")))
#else
	if( ( value = kik_conf_get_value( conf , "startup_screens")))
#endif
	{
		u_int  n ;
		
		if( ! kik_str_to_uint( &n , value) || ( ! is_genuine_daemon && n == 0))
		{
			kik_msg_printf( "startup_screens %s is not valid.\n" , value) ;
		}
		else
		{
			if( n > MAX_SCREENS)
			{
				n = MAX_SCREENS ;
			}
			
			num_of_startup_screens = n ;
		}
	}
	
	num_of_startup_ptys = 0 ;

	if( ( value = kik_conf_get_value( conf , "startup_ptys")))
	{
		u_int  n ;
		
		if( ! kik_str_to_uint( &n , value) || ( ! is_genuine_daemon && n == 0))
		{
			kik_msg_printf( "startup_ptys %s is not valid.\n" , value) ;
		}
		else
		{
			if( n <= num_of_startup_screens)
			{
				num_of_startup_ptys = 0 ;
			}
			else
			{
				if( n > MAX_SCREENS)
				{
					n = MAX_SCREENS ;
				}

				num_of_startup_ptys = n - num_of_startup_screens ;
			}
		}
	}

	if( ( value = kik_conf_get_value( conf , "word_separators")))
	{
		ml_set_word_separators( value) ;
	}

	if( ( version = kik_conf_get_version( conf)) == NULL)
	{
	#ifdef  DEBUG
		kik_warn_printf( KIK_DEBUG_TAG " version string is NULL\n") ;
	#endif
	}

	config_init( conf , argc , argv) ;

	kik_conf_delete( conf) ;

	if( *main_config.disp_name)
	{
		/*
		 * setting DISPLAY environment variable to match "--display" option.
		 */
		
		char *  env ;

		if( ( env = malloc( strlen( main_config.disp_name) + 9)))
		{
			sprintf( env , "DISPLAY=%s" , main_config.disp_name) ;
			putenv( env) ;
		}
	}

	system_listener.self = NULL ;
	system_listener.exit = __exit ;
	system_listener.open_screen = open_screen ;
	system_listener.close_screen = close_screen ;
	system_listener.open_pty = open_pty ;
	system_listener.next_pty = next_pty ;
	system_listener.prev_pty = prev_pty ;
	system_listener.close_pty = NULL ;
	system_listener.pty_closed = pty_closed ;
	system_listener.get_pty = get_pty ;
	system_listener.pty_list = pty_list ;

	signal( SIGHUP , sig_fatal) ;
	signal( SIGINT , sig_fatal) ;
	signal( SIGQUIT , sig_fatal) ;
	signal( SIGTERM , sig_fatal) ;

	ml_term_manager_init() ;
	
	kik_alloca_garbage_collect() ;
	
	return  1 ;
}

int
x_term_manager_final(void)
{
	int  count ;

	free( version) ;
	
	config_final() ;
	
	ml_free_word_separators() ;
	
	for( count = 0 ; count < num_of_screens ; count ++)
	{
		close_screen_intern( screens[count]) ;
	}

	free( screens) ;

	ml_term_manager_final() ;

	x_display_close_all() ;

	x_xim_final() ;
	
	x_color_custom_final( &color_custom) ;
	
	x_shortcut_final( &shortcut) ;
	x_termcap_final( &termcap) ;

	kik_sig_child_final() ;
	
	return  1 ;
}

void
x_term_manager_event_loop(void)
{
	int  count ;

	for( count = 0 ; count < num_of_startup_screens ; count ++)
	{
		if( ! open_screen_intern(NULL))
		{
		#ifdef  DEBUG
			kik_warn_printf( KIK_DEBUG_TAG " open_screen_intern() failed.\n") ;
		#endif

			if( count == 0 && ! is_genuine_daemon)
			{
				kik_msg_printf( "Unable to start - open_screen_intern() failed.\n") ;

				exit(1) ;
			}
			else
			{
				break ;
			}
		}
	}

	for( count = 0 ; count < num_of_startup_ptys ; count ++)
	{
		ml_term_t *  term ;

		if( ( term = create_term_intern()) == NULL)
		{
			break ;
		}

		if( ! open_pty_intern( term , main_config.cmd_path , main_config.cmd_argv ,
			":0.0" , 0 , main_config.term_type , main_config.use_login_shell))
		{
			return ;
		}
	}
	
	while( 1)
	{
		int  count ;
		
		kik_alloca_begin_stack_frame() ;

		receive_next_event() ;

		ml_close_dead_terms() ;

		if( dead_mask)
		{
			for( count = 0 ; count < num_of_screens ; count ++)
			{
				if( dead_mask & (0x1 << count))
				{
					close_screen_intern( screens[count]) ;
					
					screens[count] = screens[--num_of_screens] ;
				}
			}

			dead_mask = 0 ;
		}
		
		if( num_of_screens == 0 && ! is_genuine_daemon)
		{
			if( un_file)
			{
				unlink( un_file) ;
			}
			
			exit( 0) ;
		}

		kik_alloca_end_stack_frame() ;
	}
}
