/*
 * X Window System communication module
 *
 * 1998/04/25
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>

#include "xyaku.h"


/* Constant numbers */
#define MAX_SELECTION_DATA	8192

/* Private function prototype */
static Atom scan_atomlist(Atom atomlist[], int num_alist);
static uchar *dup_buffer(uchar *src, int src_len);


/**
 * req_selection:
 * Request selection
 * This is just a wrapper of XConvertSelection().
 **/
void
req_selection(Atom selection, Atom target, Atom prop, const Window w, const Time time)
{
	XConvertSelection(dpy, selection, target, prop, w, time);
}

/**
 * get_selected_string:
 * Using X inter-client communication such as selection, cut-buffer,
 * retrieve the selected string.
 * Input:
 * const XEvent *pxev;
 * const Window w;
 * Output:
 * Return value;the selected string, terminated by a null byte.
 * 				If no selected string is found, return NULL.
 **/
uchar*
get_selected_string(const XEvent *pxev, const Window w)
{
	Atom	ret_type;
	Atom	target_atom;
	int		ret_format;
	unsigned long ret_len = 0, ret_after;
	uchar	*str = NULL;
	XTextProperty	textprop;
	char	**list;
	int		count;
	uchar	*ret_str;/* Returned string allocated here, and freed by a caller */
	Atom	target;
	

	if (pxev->xselection.property != None) {
		if (pxev->xselection.target == XA_TARGETS) {
			target = XA_ATOM;
		} else {
			target = pxev->xselection.target;
		}
		XGetWindowProperty(dpy, pxev->xselection.requestor,
						   pxev->xselection.property,
						   0, MAX_SELECTION_DATA, True, target,
						   &ret_type, &ret_format, &ret_len,
						   &ret_after, &str);
#ifdef DEBUG
		fprintf(stderr, "prop = %ld:%s, %ld:%s %s\n", 
				ret_type, XGetAtomName(dpy, ret_type),
				pxev->xselection.property, XGetAtomName(dpy, ret_type),
				str);
#endif

		if (ret_type == XA_ATOM) {
			/* response to XA_TARGETS */
			if (ret_len > 0) {
				/* Scan list of available selection data type */
				target_atom = scan_atomlist((Atom*)str, ret_len);
				if (target_atom > 0) {
					req_selection(XA_PRIMARY, target_atom, target_atom,
								  w, pxev->xselection.time);
					XFree(str);
					return NULL;
				}
			} else {
				goto DEFAULT_TYPE_REQUEST;/* XA_STRING */
			}
		} else if (ret_type == XA_TARGETS) {
			XFree(str);
			/* response to XA_TARGETS, especially from cmdtool on Solaris */
			XGetWindowProperty(dpy, pxev->xselection.requestor,
							   pxev->xselection.property,
							   0, MAX_SELECTION_DATA, True, XA_TARGETS,
							   &ret_type, &ret_format, &ret_len,
							   &ret_after, &str);
			if (ret_after > 0)
				puts("after");
			if (ret_type == XA_TARGETS) {
				/* Scan list of available selection data type */
				target_atom = scan_atomlist((Atom*)str, ret_len);
				if (target_atom > 0) {
					req_selection(XA_PRIMARY, target_atom, target_atom,
								  w, pxev->xselection.time);
					XFree(str);
					return NULL;
				}
			} else {
				goto DEFAULT_TYPE_REQUEST;/* XA_STRING */
			}
		} else if (ret_type == XA_STRING) {
			/* ASCII string, "ret_len" doesn't include a null byte. */
			if (ret_len > 0) {
				ret_str = dup_buffer(str, ret_len);
				XFree(str);
				return ret_str;
			}
		} else if (ret_type == XA_TEXT || ret_type == XA_COMPOUND_TEXT) {
			/* Compound-text, which could be ASCII or not. */
			if (ret_len > 0) {
				int ret;
				/* Convert from compound-text to a native string */
				textprop.value = str;
				textprop.encoding = ret_type;
				textprop.format = ret_format;
				textprop.nitems = ret_len;
				ret = XmbTextPropertyToTextList(dpy, &textprop, &list, &count);
				if (ret == XNoMemory || ret == XLocaleNotSupported || ret == XConverterNotFound) {
					perror("XmbTextPropertyToTextList");
					XFree(str);
					return NULL;
				} else {
					ret_str = dup_buffer((uchar*)list[0], ret_len);
					XFreeStringList(list);
					XFree(str);
					return ret_str;
				}
			} else {
				goto DEFAULT_TYPE_REQUEST;/* XA_STRING */
			}
		} else {
			/* unknown type */
			/* try cut-buffer */
		}
	}

	/* Try cut-buffer */
	XFree(str);
	str = (uchar*)XFetchBytes(dpy, (int*)&ret_len);
	if (ret_len > 0) {
		ret_str = dup_buffer(str, ret_len);
		XFree(str);
		return ret_str;
	}
	
	return NULL;

DEFAULT_TYPE_REQUEST:/* XA_STRING */
	req_selection(XA_PRIMARY, XA_STRING,
				  XA_STRING, w, pxev->xselection.time);
	XFree(str);
	return NULL;
}


/**
 * scan_atomlist:
 * Scan the available type list, which is a response to XA_TARGETS request,
 * and choose the best type. 
 * Input:
 * Atom atomlist[];	X selection type list available
 * int num_alist;	number of list
 * Output:
 * Return value;	the best type atom to request data. 
 **/
static Atom
scan_atomlist(Atom atomlist[], int num_alist)
{
	Atom	atoms[3];
	/*	Atom	atoms[] = {XA_COMPOUND_TEXT, XA_TEXT, XA_STRING,};*/
	int		n_atoms = sizeof(atoms)/sizeof(Atom);
	int i, j;

	atoms[0] = XA_COMPOUND_TEXT;
	atoms[1] = XA_TEXT;
	atoms[2] = XA_STRING;
	for (i = 0; i < n_atoms; i++) {
		for (j = 0; j < num_alist; j++) {
			if (atoms[i] == atomlist[j]) {
				return atoms[i];
			}
		}
	}
	return (Atom)0;
}

/**
 * dup_buffer:
 * Return duplicated buffer.
 * "src" could not be terminated by a null byte. (This is why I don't use a standard strdup().)
 * However, "return buffer" must be terminated by a null byte.
 **/
static uchar*
dup_buffer(uchar *src, int src_len)
{
	uchar	*dest;

	if ((dest = (uchar*)malloc(src_len + 1)) == NULL) {
		perror("malloc");
		exit(1);
	}
	memcpy(dest, src, src_len);
	dest[src_len] = '\0';

	return dest;
}
