/* build-config.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/configs.h"
#include "commands/cmd.h"
#include "commands/build-config.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] config");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("cd to DIR first")) \
  OP (opt_no_pristines, 0, "no-pristines", 0, \
      N_("don't create pristine copies")) \
  OP (opt_hardlinks, 0, "link", 0, \
      N_("hardlink files to revision library instead of copying")) \
  OP (opt_library, 0, "library", 0, \
      N_("ensure revisions are in the revision library")) \
  OP (opt_sparse, 0, "sparse", 0, \
      N_("add library entries sparsely (--link, --library)")) \
  OP (opt_no_greedy_add, 0, "no-greedy-add", 0, \
      "do not allow greedy libraries to add revisions")\
  OP (opt_release_id, "r", "release-id", 0, \
      N_("overwrite ./=RELEASE-ID for this config"))

t_uchar arch_cmd_build_config_help[] = N_(
        "instantiate a multi-project config\n"
        "Build the named configuration.  See also baz cat-config -H\".\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_build_config (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * dir = 0;
  struct arch_build_config_params params = {0, };

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_build_config_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_no_pristines:
          {
            params.no_pristines = 1;
            break;
          }

        case opt_hardlinks:
          {
            params.hardlinks = 1;
            break;
          }

        case opt_library:
          {
            params.library = 1;
            break;
          }

        case opt_sparse:
          {
            params.sparse = 1;
            break;
          }

        case opt_no_greedy_add:
          {
            params.no_greedy_add = 1;
            break;
          }

        case opt_release_id:
          {
            params.release_id = 1;
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  {
    t_uchar * config = NULL;
    t_uchar * tree_root = NULL;
    arch_project_tree_t * tree = NULL;


    tree = arch_project_tree_new(talloc_context, dir);

    config = str_save (0, argv[1]);

    if (!arch_valid_config_name (config))
      {
        safe_printfmt (2, "%s: invalid config name (%s)\n",
                       argv[0], config);
        exit (2);
      }

    arch_build_config (tree, config, &params, tree->archive);

    arch_project_tree_delete(tree);
    lim_free (0, tree_root);
    lim_free (0, config);

  }

  lim_free (0, dir);
  return 0;
}



/* tag: Tom Lord Mon Jun  2 21:41:48 2003 (buildcfg.c)
 */
