/* cmd-register-archive.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include <neon/ne_alloc.h>
#include <neon/ne_uri.h>

#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/mem/alloc-limits.h"
#include "hackerlab/mem/mem.h"
#include "hackerlab/os/malloc.h"
#include "libfsutils/ensure-dir.h"
#include "libarch/my.h"
#include "libarch/namespace.h"
#include "libarch/archive.h"
#include "libarch/archives.h"
#include "libarch/pfs.h"
#include "commands/cmdutils.h"
#include "commands/register-archive.h"
#include "libawk/trim.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [-d name] location");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_force, "f", "force", 0, \
      N_("overwrite existing location")) \
  OP (opt_present_ok, 0, "present-ok", 0, \
      N_("return 0 even if archive exists")) \
  OP (opt_delete, "d", "delete", 0, \
      N_("delete archive registration"))

t_uchar arch_cmd_register_archive_help[] = N_("change an archive location registration\n"
                                            "Record the location of ARCHIVE.\n"
                                            "\n"
                                            "With -d, remove the registration of a previously registered\n"
                                            "archive.  When accompanied by -f, override permissions on\n"
                                            "the registration file and don't complain if the archive is\n"
                                            "not registered.\n"
                                            "If you are deleting a location that is inaccesible, you may\n"
                                            "provide the name of the archive that the location belongs to\n"
                                            "This may not be provided when registering an archive:\n"
                                            "inaccessible archives may not be registered.\n"
                                            "\n"
                                            "A LOCATION should be either a directory name or a distant URL.\n"
                                            "\n"
                                            "When registering a new archive, then the archive's name will be\n"
                                            "read automatically from the archive's meta data.\n"
                                            "\n"
                                            "Archive locations are stored in ~/.arch-params/archives/ARCHIVENAME.\n"
                                            "\n"
                                            "Registering archives is optional. If you use a url to a command, the\n"
                                            "archive will be automatically registered.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};




int
arch_cmd_register_archive (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  t_uchar * target_location = 0;
  struct opt_parsed * option;
  struct arch_archive * archive = 0;
  int delete;
  int force;
  int quietly_fail;

  delete = 0;
  force = 0;
  quietly_fail = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_register_archive_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_delete:
          {
            delete = 1;
            break;
          }
        case opt_force:
          {
            force = 1;
            break;
          }
        case opt_present_ok:
          {
            quietly_fail = ARCH_REG_FAIL_NOFAIL;
          }
        }
    }

  if (argc < 2 || argc > 3)
    goto usage_error;

  if (delete)
    {
      if (argc == 2)
        {
          t_uchar * archive_name = argv[1];

          if (arch_valid_archive_name (archive_name))
            {
              if (arch_archives_has_registry_entry (archive_name))
                  {
                    ar_archive_location locations = arch_archive_locations (archive_name);
                    int index;
                    ar_for_each (locations, index)
                      {
                        force = ARCH_REG_FAIL_NOFAIL;
                        arch_archives_delete_archive_location (archive_name, locations[index]->url, force);
                      }
                    ar_free_archive_location (&locations);
                  }
              arch_delete_archive_location (archive_name, force);
            }
          else
            {
              /* must be URL */
              t_uchar * temp_location = escape_location (archive_name);
              struct arch_archive *archive = arch_archive_connect_location_ext (NULL, temp_location, NULL, 1, 1);
              t_uchar * realname;
              lim_free (0, temp_location);
              if (!archive)
                {
                  safe_printfmt (2, _("%s: could not determine archive name from location (%s)\n"),
                                 argv[0], archive_name);
                  exit (1);
                }
              if (archive->in_registry)
                {
                  t_uchar * old_location;
                  arch_archives_delete_archive_location (archive->official_name, archive->location, force);
                  old_location = arch_archive_location (archive->official_name, 1);
                  if (old_location)
                    arch_delete_archive_location (archive->official_name, ARCH_REG_FAIL_NOFAIL);
                  lim_free (0, old_location);
                }
              else
                {
                  /* FIXME-REMOVENAME this can go with baz 1.4 when we remove =locations except for upgrades */
                  realname = str_save (0, archive->registered_name);
                  arch_delete_archive_location (realname, force);
                  lim_free (0, realname);
                }
              arch_archive_close (archive);
            }
        }
      else
        {
          t_uchar * temp_location = escape_location (argv[1]);
          arch_archives_delete_archive_location (argv[2], temp_location, force);
          lim_free (0, temp_location);
        }
    }
  else
    {
      t_uchar * archive_location;

      if (argc == 3)
        goto usage_error;
      
      target_location = argv[1];
      
      archive_location = escape_location(target_location);
      archive_location = str_replace (archive_location, arch_pfs_abs_path(archive_location));
      arch_check_uri (archive_location);
      
      archive = arch_archive_connect_location (archive_location, 0);

      if (!arch_valid_archive_name (archive->official_name))
        {
          safe_printfmt (2, "%s: invalid archive name (%s)\n",
                         argv[0], archive->official_name);
          exit (1);
        }

      safe_printfmt (1, _("Registering Archive: %s\n"), archive->official_name);
      arch_archive_register (archive);
      arch_archive_close (archive);
      lim_free (0, archive_location);
    }

  return 0;
}




/* tag: Tom Lord Sun May 18 20:34:28 2003 (register-archive.c)
 */
