// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999-2001 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

/* 
 * $B!VEPO?=q@RL>4IM}%U%!%$%k!W(B $B=hM}It(B
 */

#include <cstdio>
#include <cstdlib>
#include <cctype>

#include "def.h"
#include "catalog.h"
#include "codeconv.h"
#include "bfile.h"
#include "ndtp.h"
#include "filename.h"

static bool StripCharacter(std::string& str,char ch){
  while(str.length()!=0 && str[str.length()-1]==ch){
    str.resize(str.length()-1);
  }
  return true;
}

static std::string FixFileName(const std::string& fname){
  std::string tmp=fname;
  if (fname.substr(0,5)=="file:") tmp=fname.substr(5);
  StripCharacter(tmp,'\0');
  StripCharacter(tmp,' ');
  StripCharacter(tmp,'/');
  /*
  std::string::iterator i;
  for(i=tmp.begin();i!=tmp.end();i++){
    if (isalpha(*i)){
      *i=tolower(*i);
    }
  }
  */
  return tmp;
}

const char* DICT::BaseDir() const {
  return baseDir.c_str(); 
};

const char* DICT::DicPath() const {
  return (baseDir+dicPath).c_str();
};

const char* DICT::StreamPath() const {
  if (streamPath=="") return NULL;
  return (baseDir+streamPath).c_str();
};

const char* DICT::GraphicsPath() const {
  if (graphicsPath=="") return NULL;
  return (baseDir+graphicsPath).c_str();
};

const char* DICT::SoundPath() const {
  if (soundPath=="") return NULL;
  return (baseDir+soundPath).c_str();
};

const char* DICT::MoviePath() const {
  if (moviePath=="") return NULL;
  return (baseDir+moviePath).c_str();
};

const char* DICT::GaijiPath(int isHankaku,int size) const {
  if (isHankaku<0 || isHankaku>1) return NULL;
  if (size<0 || size>3) return NULL;
  if (gaijiPath[isHankaku][size]=="") return NULL;
  return (gaijiPath[isHankaku][size]).c_str();
};

void DICT::DebugOut(int level) const {
#ifdef DEBUG
  Debug::DebugOut(level,"  DicName %s\n",DicName().c_str());
  Debug::DebugOut(level,"    Version = %d\n",Version());
  Debug::DebugOut(level,"    isEB    = %s\n",isEB()?"True":"False");
  Debug::DebugOut(level,"    BaseDir = %s\n",BaseDir());
  if (DicPath()!=NULL){
    Debug::DebugOut(level,"       Dictionary = %s %s\n",dicPath.c_str(),DicPath());
  }
  if (StreamPath()!=NULL){
    Debug::DebugOut(level,"       Stream     = %s\n",streamPath.c_str());
  }
  if (GraphicsPath()!=NULL){
    Debug::DebugOut(level,"       Graphics   = %s\n",graphicsPath.c_str());
  }
  if (SoundPath()!=NULL){
    Debug::DebugOut(level,"       Sound      = %s\n",soundPath.c_str());
  }
  for(int i=0;i<4;i++){
    if (GaijiPath(0,i)!=NULL || GaijiPath(1,i)!=NULL){
      Debug::DebugOut(level,"       Gaiji  ");
      if (GaijiPath(0,i)!=NULL){
	Debug::DebugOut(level," Z = %-20s",GaijiPath(0,i));
      } else {
	Debug::DebugOut(level,"                         ");
      }
      if (GaijiPath(1,i)!=NULL){
	Debug::DebugOut(level," H = %s\n",GaijiPath(1,i));
      } else {
	Debug::DebugOut(level,"\n");
      }
    }
  }
  Debug::DebugOut(level,
		   "    CharOffset = %d, GaijiOffset = %d, CharExtend = %d\n",
		   charOffset,gaijiOffset,charExtend);
  Debug::DebugOut(level,"    AutoImageDisplay = %s\n",
		  autoImageDisplay?"True":"False");
  Debug::DebugOut(level,"    StartBlock = %d\n",StartBlock());
#endif
}

CATALOG::CATALOG(){
  //  Debug::DebugOut(Debug::CATALOG_DECODE,"new CATALOG\n");
}
CATALOG::~CATALOG(){
  //  Debug::DebugOut(Debug::CATALOG_DECODE,"delete CATALOG\n");
}

bool DICT::GetDictViaNDTP(std::string baseDirName){
  bool retcode=false;
  NdtpPath* np=new NdtpPath(baseDirName.c_str());
  if (np->host!=NULL){
    NdtpIO* pf=new NdtpIO(np->host,np->port);
    if (pf->SelectDict(np->path)){
      version=0;
      bookNumber=0;
      dicName=pf->GetTitle(np->path);
      iseb=np->isEB;
      baseDir=np->base;
      dicPath=np->path;
      startBlock=1;

      soundPath="";
      graphicsPath="";
      moviePath="";
      retcode=true;
    } else {
      Debug::DebugOut(Debug::CATALOG_DECODE,
		      "Can't open ndtp dictionary '%s'\n",
		      baseDirName.c_str());
    }
    delete pf;
  }
  delete np;
  return retcode;
}

bool CATALOG::Append(std::string baseDirName,std::string title,
		     int charOffset,int gaijiOffset,
		     int charExtend,bool autoImage,
		     bool disable, std::string disableSubbook) {
  byte buffer[2048];
  int i,j,k;
  bool isEB;
  BOOK bk;
  DICT dct;

  if (title.length()==0){ 
    bk.title=_("Dictionary");
  } else {
    bk.title=title;
  }
  bk.path=baseDirName;
  bk.charOffset=charOffset;
  bk.gaijiOffset=gaijiOffset;
  bk.charExtend=charExtend;
  bk.autoImage=autoImage;
  bk.disable=disable;

  if (baseDirName.substr(0,5)=="ndtp:"){
    bool ret=dct.GetDictViaNDTP(baseDirName);
    dct.charOffset=charOffset;
    dct.gaijiOffset=gaijiOffset;
    dct.charExtend=charExtend;
    dct.autoImageDisplay=autoImage;
    if (ret) {
      if (!disable) {
	bk.dict.push_back(dct);
      }
    } 
    books.push_back(bk);
    return ret;
  } 
  FILE *fp=fopen(FILENAME::FuzzyFind(FixFileName(baseDirName)+"/catalogs").c_str(),"r");
  isEB=false;
  if (fp==NULL) {
    fp=fopen((FILENAME::FuzzyFind(FixFileName(baseDirName)+"/catalog")).c_str(),"r");
    isEB=true;
  }
  if (fp==NULL) {
    fp=fopen((FILENAME::FuzzyFind(FixFileName(baseDirName)+"/catalog.")).c_str(),"r");
    isEB=true;
  }
  if (fp==NULL) {
    Debug::DebugOut(Debug::WARNINGS,"Can't open catalog file at %s.\n",
		    baseDirName.c_str());
    return false;
  }
  fread(buffer,1,2048,fp);
  fclose(fp);

  int nDir=(buffer[0]<<8)+buffer[1];
  int version=(buffer[2]<<8)+buffer[3];
  if (version==0) isEB=true;
  byte* bp=buffer+16;
  //  Debug::DebugOut(Debug::CATALOG_DECODE,"nDir=%d\n",nDir);
  for(i=0;i<nDir;i++){
    dct.bookNumber=i;
    dct.version=(256*bp[0])+bp[1];
    dct.iseb=isEB;
    dct.charOffset=charOffset;
    dct.gaijiOffset=gaijiOffset;
    dct.charExtend=charExtend;
    dct.autoImageDisplay=autoImage;
    dct.soundPath="";
    dct.graphicsPath="";
    dct.moviePath="";
    bp+=2;
    if (isEB) {
      dct.dicName=_EC(_JE(std::string((char *)bp,30)));
      bp+=30;
    } else {
      dct.dicName=_EC(_JE(std::string((char *)bp,80)));
      bp+=80;
    }
    //    Debug::DebugOut(Debug::CATALOG_DECODE,"Dict %d/%d\n",i+1,nDir);

    std::string dirName=FixFileName(std::string((char*)bp,8));
    bp+=8;
    dct.baseDir=FixFileName(baseDirName)+"/"+dirName;
    if (isEB){
      dct.dicPath="/start";
      dct.soundPath="/sound";
      FILE* fp=fopen(dct.SoundPath(),"r");
      if (fp==NULL) {
	dct.soundPath="";
      } else {
	fclose(fp);
      }
      dct.startBlock=1;
    } else {      
      dct.dicPath="/data/honmon";
      bp+=4;
      dct.startBlock=GetWord(bp);
      bp+=6;
      for(j=0;j<2;j++){
	for(k=0;k<4;k++){
	  if (*bp!=0){
	    dct.gaijiPath[j][k]=dct.baseDir+
	      "/gaiji/"+FixFileName(std::string((char*)bp,8));
	  } else {
	    dct.gaijiPath[j][k]=std::string("");
	  }
	  bp+=8;
	}
      }
      dct.moviePath="/movie";
    }

    if (!isEB && version>3){
      byte* bp2=buffer+0x10+164*nDir+i*164;
      if (GetDWord(bp2)!=0){
	std::string honmonFileName=
	  FixFileName(std::string(reinterpret_cast<char*>(bp2+4),8));
	if (honmonFileName.c_str()[0]!=0){
	  dct.dicPath="/data/"+honmonFileName;
	}
      }
      bp2+=12;
      if (GetDWord(bp2)!=0){
	std::string xxFileName=
	  FixFileName(std::string(reinterpret_cast<char*>(bp2+4),8));
	Debug::DebugOut(Debug::CATALOG_DECODE,
			"XX=%s\n",xxFileName.c_str());
      }
      bp2+=12;
      if (bp2[4]!=0){
	std::string streamFileName=
	  FixFileName(std::string(reinterpret_cast<char*>(bp2+4),8));
	if (streamFileName.c_str()[0]!=0){
	  dct.streamPath="/stream/"+streamFileName;
	}
      }
      bp2+=12;
      if (GetDWord(bp2)==0) bp2+=4;
      if (GetDWord(bp2)!=0){
	std::string graphicsFileName=
	  FixFileName(std::string(reinterpret_cast<char*>(bp2+4),8));
	if (graphicsFileName.c_str()[0]!=0){
	  dct.graphicsPath="/data/"+graphicsFileName;
	}
      }
      bp2+=12;
      if (GetDWord(bp2)!=0){
	std::string soundFileName=
	  FixFileName(std::string(reinterpret_cast<char*>(bp2+4),8));
	if (soundFileName.c_str()[0]!=0){
	  dct.soundPath="/data/"+soundFileName;
	}
      }
      bp2+=12;
    }

    {
      // BlockIO *pf=OpenDict(dct.DicPath());
      //            if (pf->Initialized()){
      bk.dict.push_back(dct);
      //      }
      //      if (pf!=NULL) delete pf;
    }
    dct.DebugOut();
  }
  books.push_back(bk);
  return true;
}

bool CATALOG::Clear(void){
  if (books.size()<=0) return false;
  while(books.size()>0){
    books.pop_back();
  }
  return true;
}

int CATALOG::nDic() {
  int count=0;
  for(std::vector<BOOK>::iterator i=books.begin();i!=books.end();++i){
    count+=(*i).dict.size();
  }
  return count;
}

DICT* CATALOG::Dic(int n){
  int count=0;
  for(std::vector<BOOK>::iterator i=books.begin();i!=books.end();++i){
    int size=(*i).dict.size();
    if (count<=n && n<count+size) {
      return &((*i).dict[n-count]);
    }
    count+=size;
  }
  return NULL;
}

DICT::DICT(const DICT& dict) {
  bookNumber=dict.bookNumber;
  version=dict.version;
  iseb=dict.iseb;
  dicName=dict.dicName;
  baseDir=dict.baseDir;
  dicPath=dict.dicPath;
  streamPath=dict.streamPath;
  graphicsPath=dict.graphicsPath;
  soundPath=dict.soundPath;
  moviePath=dict.moviePath;

  for(int i=0;i<2;i++){
    for(int j=0;j<4;j++){
      gaijiPath[i][j]=dict.gaijiPath[i][j];
    }
  }
  charOffset=dict.charOffset;
  gaijiOffset=dict.gaijiOffset;
  charExtend=dict.charExtend;
  autoImageDisplay=dict.autoImageDisplay;
  startBlock=dict.startBlock;
}
