// -*- C++ -*-     EUC-JP
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "dict.h"
#include <algorithm>
#include <cstdio>
#include <cstdlib>

void DICTIONARY::SetupKinsoku(byte* buf){
  if (*buf!=1) {
    SetupDummyKinsoku();
    return;
  }
  for(int i=0;i<3;i++){
    int start=GetBCD(buf+i*4+26,4);
    int len=GetBCD(buf+i*4+28,4);
    for(int j=0;j<len;j+=2){
      kinsoku[i][GetWord(buf+start+j)]=true;
    }
  }
}

void DICTIONARY::SetupCombinedIndexName(byte* buf){
  int menuNameStart=70*GetWord(buf);
  int combinedMenuNameStart=70*GetWord(buf+2);
  int indexSearchNameStart=70*3;

  for(int i=0;GetWord(buf+indexSearchNameStart+i*70)==0x0002;i++){
    if (IndexExist(0x81+i)){
      indexHead[0x81+i].typeName=_EC(_JE(reinterpret_cast<char*>
					 (buf+indexSearchNameStart+i*70+2)));
    } else {
      break;
    }
  }
  for(uint i=0;GetWord(buf+combinedMenuNameStart+i*70)==0x0002;i++){
    if (mIndexHead.size()<=i) break;
    mIndexHead[i].typeName=_EC(_JE(reinterpret_cast<char*>
					  (buf+combinedMenuNameStart+i*70+2)));
  }
  for(int i=0;GetWord(buf+menuNameStart+i*70)==0x0001;i++){
    if (IndexExist(0x10+i)){
      indexHead[0x10+i].typeName=_EC(_JE(reinterpret_cast<char*>
					 (buf+menuNameStart+i*70+2)));
    } else {
      break;
    }
  }
}

void DICTIONARY::SetupDummyKinsoku(){  // ȴ
  char* kin[]={
    "\xa1\xa1\xa1\xa4\xa1\xa5\xa1\xa3" 			// ""
      "\xa1\xa5\xa1\xa6\xa1\xa7\xa1\xa8"       		// ""
      "\xa1\xa9\xa1\xaa\xa1\xd7\xa1\xcf\xa1\xcb",    	// "סϡ"
      "\a1\a1\a1\a4\a1\a5\a1\a3\a1\a5\a1\a6\a1\a7\a1\a8\a1\a9\a1\aa",	
      // "" 
      ""};
  for(int j=0;j<3;j++){
    int len=strlen(kin[j]);
    for(int i=0;i<len;i+=2){
      int code=(((0xff & kin[j][i]) <<8)+(0xff & kin[j][i+1]))&0x7f7f;
      kinsoku[j][code]=true;
    }
  }
}

bool DICTIONARY::IsKinsoku(int type,int code){
  if (type<0 || type>2) return false;
  if (kinsoku[type].find(code)==kinsoku[type].end()){
    return false;
  } else {
    return kinsoku[type][code];
  }
}

TAG DICTIONARY::RandomJump(const TAG& start){
  pf->Seek(start);
  pf->GetWord();
  int nItems=pf->GetByte();
  pf->GetByte();
  pf->GetDWord();
  int ran=random();
  ran>>=16;
  int pos=ran % nItems;
  for(int i=0;i<pos;i++){
    pf->GetBCDTag();
  }
  return pf->GetBCDTag();
}

/*
TAG DICTIONARY::NextJump(const TAG& start){
  pf->Seek(start);
  pf->GetWord();
  int nItems=pf->GetByte();
  pf->GetByte();
  pf->GetDWord();
  TAG top=pf->Tell();
  for(int i=0;i<pos;i++){
    pf->GetBCDTag();
  }
  return pf->GetBCDTag();
}
*/

bool DICTIONARY::InitializeDictionary(DICT* dic) {
  srandom(time(NULL));
  sel=new SELECTION_MENU();
  currentIndex=NULL;
  dictFileStruct=dic;
  charOffset=dic->CharOffset();
  gaijiOffset=dic->GaijiOffset();
  charExtend=dic->CharExtend();
  isEB=dic->isEB();
  autoImageDisplay=dic->AutoImageDisplay();
  Debug::DebugOut(Debug::TEMP,"Initialize %s\n",dictFileStruct->DicPath());
  pf=OpenDict(dictFileStruct->DicPath());
  Debug::DebugOut(Debug::TEMP,"Initialize %s Ok?\n",dictFileStruct->DicPath());
  if (pf==NULL) {
    Debug::DebugOut(Debug::TEMP,"%s\n",dictFileStruct->DicPath());
    return false;
  }
  byte buf0[cBlockSize],buf1[cBlockSize];
  byte* b0,* b1;
  int i;
  pf->ReadBlock(dic->StartBlock(),buf0);
  int nIndex=GetWord(buf0);
  indexDecodeMethod=buf0[4];
  b0=buf0+16;
  for(i=0;i<nIndex;i++){
    INDEX_HEAD ih(b0,indexDecodeMethod);
    ih.DebugOut(Debug::INDEX_DECODE);
    if (ih.type!=255) {
      indexHead[ih.type]=ih;
    } else {
      pf->ReadBlock(ih.start,buf1);
      b1=buf1;
      MINDEX_HEAD mih(b1,indexDecodeMethod);
      mIndexHead.push_back(mih);
      mih.DebugOut(Debug::INDEX_DECODE);
    }
  }
  if (*b0 ==1){
    defaultIndexDisplayMethod=b0[4];
    defaultHonmonDisplayMethod=b0[5];
  } else {
    defaultIndexDisplayMethod=0;
    defaultHonmonDisplayMethod=0;
  }
  int s=charHeight>20?0xf3:0xf1; // 24/16dot Gaiji 
  if (IndexExist(s) || IndexExist(s+1)){
    int s0=0,s1=0;
    if (IndexExist(s)) s0=indexHead[s].start;
    if (IndexExist(s+1)) s1=indexHead[s+1].start;
    if (s0==0) { s0=s1; s1=0; };
    gaiji=new GAIJI(pf,s0,s1);
  } else if (pf->IsNDTPMode()){
    gaiji=new GAIJI(pf,charHeight);
  } else {
    int scode=0;
    if (charHeight>20) scode++;
    if (charHeight>28) scode++;
    if (charHeight>36) scode++;
    gaiji=new GAIJI(dictFileStruct->GaijiPath(0,scode),
		    dictFileStruct->GaijiPath(1,scode));
  }
  if (IndexExist(0x12)){ // §ʸν
    pf->ReadBlock(indexHead[0x12].start,buf1);
    SetupKinsoku(buf1);
  } else {
    SetupDummyKinsoku();
  }

  if (IndexExist(0x16)){
    pf->ReadBlock(indexHead[0x16].start,buf1);
    SetupCombinedIndexName(buf1);
  }
  return true;
}

bool DICTIONARY::FindWordHead(const std::string& key0,
			      const INDEX_HEAD& iHead) {
  std::string key=CODECONV::Euc2Jis(key0);
  int t=iHead.type;
  if (key0[0]=='*' && (t & 0xf0)==0x90) { // 
    return FindWordHead(CODECONV::Jis2Euc(CODECONV::RevJis(key)),t-0x20);
  }

  // keyword for upper index
  std::string key1=CODECONV::ConvertForCompare(key,iHead.codeConvMode);

  int start=iHead.start;
  if (currentIndex!=NULL) delete currentIndex;
  currentIndex=new INDEX(pf,start,t,isEB);

  for(;;){
    int nItems=currentIndex->indexItem.size();
    currentIndex->currentPoint=-1;
    
    // ǽ
    if (!currentIndex->isUpperIndex()){
      for(int i=0;i<nItems;i++){
	std::string keyword=
	  currentIndex->indexItem[i].keyword.substr(0,key1.length());
	if (iHead.type==0x70 || iHead.type==0x90){
	  keyword=CODECONV::ConvertForCompare(keyword,iHead.codeConvMode);
	} 
	if (key1==keyword){
	  currentIndex->currentPoint=i;
	  return true;
	}
	if (key1<keyword) {
	  currentIndex->currentPoint=i;
	  return false;
	}
      }
      return false;  // Not Found.
    }

    for(int i=0;i<nItems;i++){
      if (currentIndex->indexItem[i].keyword >= key1) {
	currentIndex->currentPoint=i;
	break;
      }
    }
    if (currentIndex->currentPoint<0){ 
      currentIndex->currentPoint=nItems-1;
      return false;
    }
    INDEX *idx=new INDEX(pf,currentIndex->indexItem[currentIndex->currentPoint].adrs.Block(),t,isEB);
    // ========
    if (currentIndex!=NULL) delete currentIndex;
    currentIndex=idx;
    currentIndex->DebugOut(Debug::INDEX_DECODE);
  } 
}

bool DICTIONARY::FindWordHead(const std::string& key,int indexType){
  if (indexType<0){
    bool retCode=false;
    indexType=-indexType & 0xf0;
    sel->Clear();

    if (CODECONV::isKana(CODECONV::Euc2Jis(key)) && 
	IndexExist(indexType)){ // Kana Search
      bool r=FindWordHead(key,indexType);



      retCode|=r;
      sel->Append(currentIndex,r);
    } 
    if (IndexExist(indexType+1)){
      bool r=FindWordHead(key,indexType+1);
      retCode|=r;
      if (r) sel->Append(currentIndex,r);
    }
    if (IndexExist(0x61)){
      bool r=FindWordHead(key,0x61);
      retCode|=r;
      if (r) sel->Append(currentIndex,r);
    }
    return retCode;
  }
  if (indexHead.find(indexType)==indexHead.end()) {
    return false;
  }
  return FindWordHead(key,indexHead[indexType]);
}

INDEXITEMLIST_T DICTIONARY::FindWord(const std::string& key,
					  int indexType) {
  Debug::DebugOut(Debug::DICTIONARY,"%s %d\n",
		  CODECONV::Jis2Euc(key).c_str(),key.length());

  if (indexHead.find(indexType)==indexHead.end()) {
    return INDEXITEMLIST_T ();
  }
  return FindWord(key,indexHead[indexType]);
}

INDEXITEMLIST_T 
DICTIONARY::FindWord(const std::string& key0,const INDEX_HEAD& iHead) {

  std::string key1=CODECONV::Euc2Jis(key0);
  std::string key=CODECONV::ConvertForCompare(key1,iHead.codeConvMode);

  Debug::DebugOut(Debug::DICTIONARY,"%s %d\n",CODECONV::Jis2Euc(key).c_str(),
		  key.length());

  int start=iHead.start;
  INDEXITEMLIST_T selectionList;
  if (currentIndex!=NULL) delete currentIndex;
  currentIndex=new INDEX(pf,start,iHead.type,isEB);
  currentIndex->DebugOut(Debug::INDEX_DECODE);
  do {
    std::vector<INDEXITEM>::iterator i;
    for(i=currentIndex->indexItem.begin();
	i!=currentIndex->indexItem.end();++i){
      Debug::DebugOut(Debug::DICTIONARY|Debug::VERBOSE,
	       "Compare (%d) %s ",i->keyword.length(),
		      CODECONV::Jis2Euc(i->keyword).c_str());
      Debug::DebugOut(Debug::DICTIONARY|Debug::VERBOSE,">= (%d) %s : %c\n",
		      key.length(),
		      CODECONV::Jis2Euc(key).c_str(),i->keyword>=key?'t':'f');
      if (i->keyword >= key) {
	break;
      }
    }
    if (i==currentIndex->indexItem.end()) {
      Debug::ErrorOut("Illigal Index!\n");
      //exit(1);
      return selectionList;
    }
    INDEX *idx=new INDEX(pf,i->adrs.Block(),iHead.type,isEB);
    if (idx==NULL) break;
    if (currentIndex!=NULL) delete currentIndex;
    currentIndex=idx;
    currentIndex->DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE);
  } while(currentIndex->isUpperIndex());

  while(currentIndex->CurrentIndexItem().keyword<=key){
    const INDEXITEM& item=currentIndex->CurrentIndexItem();

      Debug::DebugOut(Debug::DICTIONARY|Debug::VERBOSE,
	       "X Compare (%d) %s ",item.keyword.length(),
		      CODECONV::Jis2Euc(item.keyword).c_str());
      Debug::DebugOut(Debug::DICTIONARY|Debug::VERBOSE,">= (%d) %s : %c\n",
		      key.length(),
		      CODECONV::Jis2Euc(key).c_str(),item.keyword>=key?'t':'f');

    if (item.keyword==key){
      //  if (item.keyword==key){
      if (item.adrs != selectionList.back().adrs){
	selectionList.push_back(item);
      }
    }
    
    if (!currentIndex->Next()) break;
  }

#ifdef DEBUG
  if ((Debug::Level()&Debug::VERBOSE)!=0 && selectionList.size()!=0){
    INDEXITEMLIST_T::iterator i;
    for(i=selectionList.begin();i!=selectionList.end();++i){
      (*i).DebugOut(Debug::INDEX_DECODE | Debug::VERBOSE);
      std::string str=GetLine((*i).keywordTag);
      Debug::DebugOut(Debug::TEMP | Debug::VERBOSE,"%s\n",
		      CODECONV::Jis2Euc(str).c_str());
    }
  }
#endif

  if (selectionList.size()==0) {
    Debug::DebugOut(Debug::DICTIONARY,"Can't find word\n");
  }
  return selectionList;
}

INDEXITEMLIST_T DICTIONARY::FindWord(const std::string& key,
					  int extnum,int extnum2) {
  Debug::DebugOut(Debug::DICTIONARY,"%s %d\n",CODECONV::Jis2Euc(key).c_str(),
		  key.length());
  //  if (currentIndex!=NULL) delete currentIndex;
  //  currentIndex=NULL;
  if (extnum<0 || extnum>=(int)mIndexHead.size() || extnum2<0) {
    return INDEXITEMLIST_T();
  }
  MINDEX_HEAD mIndex=mIndexHead[extnum];
  
  mIndex.DebugOut();

  if (extnum2>=(int)mIndex.subIndex.size()) return INDEXITEMLIST_T();
  SUBINDEX sIndex=mIndex.subIndex[extnum2];
  
  sIndex.DebugOut();

  if (sIndex.subSubIndex.find(0xa1)!=sIndex.subSubIndex.end()) {
    return FindWord(key,sIndex.subSubIndex[0xa1]);
  } else if (sIndex.subSubIndex.find(0x91)!=sIndex.subSubIndex.end()) {
    return FindWord(key,sIndex.subSubIndex[0x91]);
  }
  return INDEXITEMLIST_T();
}

std::string DICTIONARY::GetLine(const TAG& t) {
  std::string str;
  pf->Seek(t);
  byte ch;
  for(;;) { 
    ch=pf->GetByte();
    if (ch==0x1f) {
      int ch2=pf->GetByte();
      if (ch2==0x0a) break;
      str+="\x21\x2a";
    } else {
      str+=static_cast<char>(ch);
    }
  }
  return str;
}

DICTIONARY::~DICTIONARY(){
  if (gaiji!=NULL) delete gaiji;
  if (pf!=NULL) delete pf;
  if (sel!=NULL) delete sel;
  //  if (dictFileStruct!=NULL) delete dictFileStruct;
}


SELECTION_MENU::~SELECTION_MENU(){
  Clear();
}

SELECTION_MENU::SELECTION_MENU(){
  Clear();
}

bool SELECTION_MENU::Clear(){
  items.clear();
  idx.clear();
  currentPoint.adrs=TAG(0,0);
  nReverse=nForward=0;
  return true;
}

bool SELECTION_MENU::Append(INDEX* index,bool validFlag){
  if (index==NULL) return false;
  if (validFlag && 
      (currentPoint.adrs==TAG(0,0) || 
      index->CurrentIndexItem()<currentPoint)){
    currentPoint=index->CurrentIndexItem();
  }
  idx.push_back(INDEX(index));
  return true;
};

/*
const INDEXITEM& SELECTION_MENU::CurrentPoint() {
  std::vector<INDEX>::iterator imin=idx.begin();
  for(std::vector<INDEX>::iterator ii=idx.begin()+1;ii!=idx.end();++ii){
    if ((*ii).CurrentIndexItem()<(*imin).CurrentIndexItem()) imin=ii;
  }
  return (*imin).CurrentIndexItem();
}
*/

/*
static int CompareSelection(INDEX& a,INDEX& b){
  return a.CurrentIndexItem()<b.CurrentIndexItem();
}
*/

bool SELECTION_MENU::Prev(std::vector<INDEX>& idx){
  std::vector<bool> validFlag;
  int i,j=-1;
  int nItems=idx.size();
  if (idx.begin()==idx.end()) return false;
  for(i=0;i<nItems;i++){
    validFlag.push_back(true);
    if (currentPoint < idx[i].CurrentIndexItem()) {
      validFlag[i]=idx[i].Prev();
    }
  }
  for(i=0;i<nItems;i++){
    if (!validFlag[i]) continue;
    while (idx[i].CurrentIndexItem().FuzzyCompare(currentPoint)==0){
      if (!idx[i].Prev()){
	validFlag[i]=false;
	break;
      }
    }
    j=i;
  }

  if (j<0) return false;

  for(i=0;i<nItems;i++){
    if (idx[j].CurrentIndexItem()<idx[i].CurrentIndexItem()){
      j=i;
    }
  }
  if (idx[j].CurrentIndexItem()<currentPoint) {
    currentPoint=idx[j].CurrentIndexItem();
    return true;
  } else {
    return false;
  }
}

bool SELECTION_MENU::Prev(){
  return Prev(idx);
}

bool SELECTION_MENU::Next(std::vector<INDEX>& idx){
  if (idx.begin()==idx.end()) return false;
  std::vector<bool> validFlag;
  int i,j=-1;

  int nIndex=idx.size();

  for(i=0;i<nIndex;i++){
    validFlag.push_back(true);
    if (idx[i].CurrentIndexItem()<currentPoint){
      validFlag[i]=idx[i].Next();
    }
  }
  for(i=0;i<nIndex;i++){
    if (!validFlag[i]) continue;
    while (idx[i].CurrentIndexItem().FuzzyCompare(currentPoint)==0){
      if (!idx[i].Next()) {
	validFlag[i]=false;
	break;
      }
    }
    if (validFlag[i]) j=i;
  }
  if (j<0) return false;

  for(i=0;i<nIndex;i++){
    if (idx[i].CurrentIndexItem().FuzzyCompare(idx[j].CurrentIndexItem())<0){
      j=i;
    }
  }
  currentPoint=idx[j].CurrentIndexItem();
  return true;
}

bool SELECTION_MENU::Next(){
  return Next(idx);
}

/*
void SELECTION_MENU::Setup(int nPrev,int nForw){
  items.clear();

  std::vector<INDEX> tmpidx;
  CODE_CONV_MODE convMode;
  convMode.Setup(NULL);

  if (currentPoint.adrs==TAG(0,0)) return;

  INDEXITEM cp=currentPoint;
  INDEXITEM lp=currentPoint;
  itemCheckListF.clear();
  itemCheckListR.clear();

  std::vector<INDEX>::iterator ii;

  // Ȥꤢ ǥå򥳥ԡ
  for(std::vector<INDEX*>::iterator i=idx.begin();i!=idx.end();++i){
    tmpidx.push_back(**i);
  }
  // ֥ɤ˶ᤤΤ
  while(itemCheckListR.size()<nPrev){
    ii=max_element(tmpidx.begin(),tmpidx.end(),CompareSelection);
    if (ii==tmpidx.end()) break;
    if (ii->CurrentIndexItem()>=lp){
      if (!ii->isIndexHead() && ii->Prev()) continue;
      break;
    }
    itemCheckListR.push_front((*ii).CurrentIndexItem());
    lp=(*ii).CurrentIndexItem();
  }
  copy(itemCheckListR.begin(),itemCheckListR.end(),back_inserter(items));
  nReverse=itemCheckListR.size();

  items.push_back(cp);
  
  tmpidx.clear();

  for(std::vector<INDEX*>::iterator ii=idx.begin();ii!=idx.end();++ii){
    tmpidx.push_back(**ii);
  }

  lp=cp;

  while(itemCheckListF.size()<nForw){ 
    ii=min_element(tmpidx.begin(),tmpidx.end(),CompareSelection);
    if (ii==tmpidx.end()) break;
    if (ii->CurrentIndexItem()<=lp){
      if (ii->Next()) {
	continue;
      } else {
	break;
      }
    }
    itemCheckListF.push_back(ii->CurrentIndexItem());
    lp=(*ii).CurrentIndexItem();
  }

  nForward=itemCheckListF.size();
  copy(itemCheckListF.begin(),itemCheckListF.end(),back_inserter(items));
}
*/

void SELECTION_MENU::Setup(size_t nPrev,size_t nForw){
  items.clear();

  std::vector<INDEX> tmpidx;
  CODE_CONV_MODE convMode;
  convMode.Setup(NULL);

  if (currentPoint.adrs==TAG(0,0)) return;

  INDEXITEM cp=currentPoint;
  std::deque<INDEXITEM> itemCheckList;

  // Ȥꤢ ǥå򥳥ԡ
  copy(idx.begin(),idx.end(),back_inserter(tmpidx));

  // ֥ɤ˶ᤤΤ
  while(itemCheckList.size()<nPrev){
    if (!Prev(tmpidx)) break;
    itemCheckList.push_front(currentPoint);
  }
  copy(itemCheckList.begin(),itemCheckList.end(),back_inserter(items));
  nReverse=itemCheckList.size();

  items.push_back(cp);

  tmpidx.clear();
  copy(idx.begin(),idx.end(),back_inserter(tmpidx));
  itemCheckList.clear();
  currentPoint=cp;

  while(itemCheckList.size()<nForw){ 
    if (!Next(tmpidx)) break;
    itemCheckList.push_back(currentPoint);
  }
  nForward=itemCheckList.size();
  copy(itemCheckList.begin(),itemCheckList.end(),back_inserter(items));
  currentPoint=cp;
  fprintf(stderr,"Forw=%d, Back=%d, nItem=%d\n",nForward, nReverse, items.size());
}

INDEXITEMLIST_T And(const INDEXITEMLIST_T& a,
			 const INDEXITEMLIST_T& b) {
  if (a.size()==0) return b;
  if (b.size()==0) return a;
  INDEXITEMLIST_T out;
  INDEXITEMLIST_T::const_iterator aa,bb;
  aa=a.begin();
  bb=b.begin();
  while(aa!=a.end() && bb!=b.end()){
    int x=aa->FuzzyCompare(*bb);
    if (x==0){
      out.push_back(*aa);
      ++aa;
      ++bb;
    } else if (x<0) {
      ++aa;
    } else {
      ++bb;
    }
  }
  return out;
}

const char* DICTIONARY::MIndexName(int num) {
  static char buf[20];
  if (num<0 || num>=nmIndex()) return NULL;
  if (mIndexHead[num].typeName==std::string("")) {
    snprintf(buf,20,"%s %d",_("Multi Param. Search"),num+1);
    return buf;
  }
  return mIndexHead[num].typeName.c_str();
};
