// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"
#include "../src/regression.h"

template< typename T,
          int Size,
          typename VectorType,
          typename BigVecType >
void makeNoisyCohyperplanarPoints( int numPoints,
                                   VectorType * points,
                                   BigVecType * coeffs,
                                   T noiseAmplitude )
{
    int i, j;

    // pick a random hyperplane, store the coefficients of its equation
    coeffs->resize( Size + 1 );
    for( j = 0; j < Size + 1; j++ )
    {
        (*coeffs)(j) = static_cast<T>(0);
        while( std::abs( (*coeffs)(j) ) < 0.5 ) pickRandom((*coeffs)(j));
    }

    // now pick numPoints random points on this hyperplane
    for( i = 0; i < numPoints; i++ )
    {
        VectorType cur_point(Size);
        do
        {
            // make cur_point a random vector
            for( j = 0; j < Size; j++ )
                pickRandom( cur_point(j) );
            cur_point.normalize();

            // project cur_point onto the hyperplane
            T x = static_cast<T>(0);
            for( j = 0; j < Size; j++ )
                x -= (*coeffs)(j) * cur_point(j);
            cur_point *= (*coeffs)(Size) / x;
        } while( std::abs(cur_point.norm()) < 0.5
              || std::abs(cur_point.norm()) > 2.0 );

        points[i] = cur_point;
    }

    // add some noise to these points
    for( i = 0; i < numPoints; i++ )
    {
        for( j = 0; j < Size; j++ )
        {
            T someNoise;
            pickRandom( someNoise );
            points[i](j) += someNoise * noiseAmplitude;
        }
    }
}

template< typename T,
          typename VectorType,
          typename BigVecType,
          typename MatrixType >
void helper_checkComputeFittingHyperplane( int numPoints,
                                           const VectorType * points,
                                           const BigVecType * coeffs,
                                           T allowedError )
{
    int size = points[0].size();
    BigVecType result( size + 1 );
    computeFittingHyperplane_internal< T, VectorType,
                                       BigVecType, MatrixType >
        ( numPoints, points, & result );
    result /= result( size );
    result *= (*coeffs)( size );
    T error = (result - *coeffs).norm() / coeffs->norm();
    QVERIFY( std::abs(error) < std::abs(allowedError) );
}

void MainTest::checkRegression()
{
    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        Vector2f points2f [1000];
        Vector3f coeffs3f;
        makeNoisyCohyperplanarPoints< float, 2, Vector2f, Vector3f >
            ( 1000, points2f, &coeffs3f, 0.01 );
        helper_checkComputeFittingHyperplane
                    < float, Vector2f, Vector3f, Matrix2f >
                    ( 10, points2f, &coeffs3f, 0.05 );
        helper_checkComputeFittingHyperplane
                    < float, Vector2f, Vector3f, Matrix2f >
                    ( 100, points2f, &coeffs3f, 0.01 );
        helper_checkComputeFittingHyperplane
                    < float, Vector2f, Vector3f, Matrix2f >
                    ( 1000, points2f, &coeffs3f, 0.002 );

        Vector4d points4d [1000];
        Vector<double, 5> coeffs5d;
        makeNoisyCohyperplanarPoints< double, 4, Vector4d, Vector<double,5> >
            ( 1000, points4d, &coeffs5d, 0.01 );
        helper_checkComputeFittingHyperplane
                    < double, Vector4d, Vector<double, 5>, Matrix4d >
                    ( 10, points4d, &coeffs5d, 0.05 );
        helper_checkComputeFittingHyperplane
                    < double, Vector4d, Vector<double, 5>, Matrix4d >
                    ( 100, points4d, &coeffs5d, 0.01 );
        helper_checkComputeFittingHyperplane
                    < double, Vector4d, Vector<double, 5>, Matrix4d >
                    ( 1000, points4d, &coeffs5d, 0.002 );

        VectorXcd points8cd [1000];
        VectorXcd coeffs9cd;
        makeNoisyCohyperplanarPoints< complex<double>, 8, VectorXcd, VectorXcd >
            ( 1000, points8cd, &coeffs9cd, 0.01 );
        helper_checkComputeFittingHyperplane
                    < complex<double>, VectorXcd, VectorXcd, MatrixXcd >
                    ( 10, points8cd, &coeffs9cd, 0.35 );
        helper_checkComputeFittingHyperplane
                    < complex<double>, VectorXcd, VectorXcd, MatrixXcd >
                    ( 100, points8cd, &coeffs9cd, 0.025 );
        helper_checkComputeFittingHyperplane
                    < complex<double>, VectorXcd, VectorXcd, MatrixXcd >
                    ( 1000, points8cd, &coeffs9cd, 0.006 );

        VectorXd points20d [1000];
        VectorXd coeffs21d;
        makeNoisyCohyperplanarPoints< double, 20, VectorXd, VectorXd >
            ( 1000, points20d, &coeffs21d, 0.01 );
        helper_checkComputeFittingHyperplane
                    < double, VectorXd, VectorXd, MatrixXd >
                    ( 100, points20d, &coeffs21d, 0.1 );
        helper_checkComputeFittingHyperplane
                    < double, VectorXd, VectorXd, MatrixXd >
                    ( 1000, points20d, &coeffs21d, 0.015 );
    }
}
