/*
 * Gnome Attacks
 *
 * Copyright (c) 2003 by Andy Balaam
 *
 * Released under the GNU General Public License
 * with ABSOLUTELY NO WARRANTY.
 *
 * See the file COPYING for more information.
 *
 */

#include "graphics.h"

#include <math.h>

#include "main.h"
#include "game.h"
#include "hiscore.h"

/**
 * Called when the screen size is changed (and at start).
 */
gboolean graphics_canvas_configure( GtkWidget *widget,
  GdkEventConfigure *event, gpointer user_data)
{ TRACE(graphics_canvas_configure)
  
  /* Get the new screen size */
  drawing_area = widget;
  drawing_area_width = drawing_area->allocation.width;
  drawing_area_height = drawing_area->allocation.height;
  
  /* Scale the images */
  _scale_nonlevel_images();
  _scale_level_images();
  
  /* Prepare the pixmap */
  if( pixmap != NULL ) {
    g_object_unref( pixmap );
    pixmap == NULL;
  }

  pixmap = gdk_pixmap_new( drawing_area->window,
	  drawing_area_width, drawing_area_height, -1 );

  /* Draw the current image onto the pixmap */
  graphics_redraw_whole_screen();

  return TRUE;
  
}

/**
 * Called when the widget is uncovered or needs refreshing.
 */
gboolean graphics_canvas_expose( GtkWidget *widget, GdkEventExpose  *event,
  gpointer user_data )
{ TRACE(graphics_canvas_expose)

  /* Just re-draw our pixmap onto the screen. */
  gdk_draw_drawable( widget->window,
    widget->style->white_gc,
		pixmap,
		event->area.x, event->area.y,
		event->area.x, event->area.y,
		event->area.width, event->area.height );
    
    return FALSE;
  
}

/**
 * Load all the non-level graphics images
 */
void graphics_setup()
{ TRACE(graphics_setup)
  
  /* Null everything */
  title_screen_original = NULL;
  title_screen_scaled = NULL;
  hiscore_screen_original = NULL;
  hiscore_screen_scaled = NULL;
  instructions_screen_original = NULL;
  instructions_screen_scaled = NULL;
  ingame_background_original = NULL;
  ingame_background_scaled = NULL;
  ingame_foreground_original = NULL;
  ingame_foreground_scaled = NULL;
  ship_original = NULL;
  ship_scaled = NULL;
  dead_ship_original = NULL;
  dead_ship_scaled = NULL;
  bullet_original = NULL;
  bullet_scaled = NULL;
  
  /* Load the images */
  _load_nonlevel_original_images();
  
  /* but they will get scaled and drawn in the configure event */
  
}

/**
 * Tidy up our memory usage.
 */
void graphics_quit()
{ TRACE(graphics_quit)
  
  _unref_level_original_images();
  _unref_level_scaled_images();
  _unref_nonlevel_original_images();
  _unref_nonlevel_scaled_images();
  
}

/**
 * Load all the non-level graphics images
 */
void _load_nonlevel_original_images()
{ TRACE(_load_nonlevel_original_images)
  
  title_screen_original = GDK_PIXBUF( levelset_get_pixbuf( "title",
    TRUE ) );
  
  hiscore_screen_original = GDK_PIXBUF( levelset_get_pixbuf( "hiscore",
    TRUE ) );
  
  instructions_screen_original = GDK_PIXBUF(
    levelset_get_pixbuf( "instructions", TRUE ) );
  
  ship_original = GDK_PIXBUF( levelset_get_pixbuf( "ship", TRUE ) );
  
  dead_ship_original = GDK_PIXBUF( levelset_get_pixbuf( "dead_ship", TRUE ) );
  
  bullet_original = GDK_PIXBUF( levelset_get_pixbuf( "bullet", TRUE ) );
  
}

void _unref_nonlevel_original_images()
{ TRACE(_unref_nonlevel_original_images)
  
  if( title_screen_original != NULL) {
    g_object_unref( title_screen_original );
    title_screen_original = NULL;
  }
  
  if( hiscore_screen_original != NULL) {
    g_object_unref( hiscore_screen_original );
    hiscore_screen_original = NULL;
  }
  
  if( instructions_screen_original != NULL) {
    g_object_unref( instructions_screen_original );
    instructions_screen_original = NULL;
  }
  
  if( ship_original != NULL) {
    g_object_unref( ship_original );
    ship_original = NULL;
  }
  
  if( dead_ship_original != NULL) {
    g_object_unref( dead_ship_original );
    dead_ship_original = NULL;
  }
  
  if( bullet_original != NULL) {
    g_object_unref( bullet_original );
    bullet_original = NULL;
  }
  
}

void _unref_nonlevel_scaled_images()
{ TRACE(_unref_nonlevel_scaled_images)
  
  if( title_screen_scaled != NULL) {
    g_object_unref( title_screen_scaled );
    title_screen_scaled = NULL;
  }
  
  if( hiscore_screen_scaled != NULL) {
    g_object_unref( hiscore_screen_scaled );
    hiscore_screen_scaled = NULL;
  }
  
  if( instructions_screen_scaled != NULL) {
    g_object_unref( instructions_screen_scaled );
    instructions_screen_scaled = NULL;
  }
  
  if( ship_scaled != NULL) {
    g_object_unref( ship_scaled );
    ship_scaled = NULL;
  }
  
  if( dead_ship_scaled != NULL) {
    g_object_unref( dead_ship_scaled );
    dead_ship_scaled = NULL;
  }
  
  if( bullet_scaled != NULL) {
    g_object_unref( bullet_scaled );
    bullet_scaled = NULL;
  }
  
}

void _load_level_original_images()
{ TRACE(_load_level_original_images)
  
  gint drawing_area_width;
  gint drawing_area_height;
  
  GError* err;
  
  GString* file_path;
  
  drawing_area_width = drawing_area->allocation.width;
  drawing_area_height = drawing_area->allocation.height;
  
  err = NULL;
  
  file_path = g_string_new( NULL );
  
  g_string_printf( file_path, "level_%d_bg", graphics_level );
  
  ingame_background_original = levelset_get_pixbuf( file_path->str, FALSE );
  
  TRACEPTR(ingame_background_original)
  
  /* If the file doesn't exist, we assume we've got to the end of the levels
     and start again at level 1 */  
  if( ingame_background_original == NULL ) {
    
    graphics_level = 1;
    
    g_string_printf( file_path, "level_%d_bg", graphics_level );
    
    ingame_background_original = levelset_get_pixbuf( file_path->str, FALSE );
    
    if( ingame_background_original == NULL ) {
      
      fprintf( stderr, "gnomeattacks: No background available for level %d!\n",
        graphics_level );
      exit( 2 );
      
    }
    
  }
  
  g_string_printf( file_path, "level_%d_fg", graphics_level );
  
  ingame_foreground_original = levelset_get_pixbuf( file_path->str, FALSE );
  
  /* If the file doesn't exist, we assume we've got to the end of the levels
     and start again at level 1 */  
  if( ingame_foreground_original == NULL ) {
    
    fprintf( stderr, "gnomeattacks: No foreground available for level %d!\n",
        graphics_level );
    exit( 3 );
    
  }
  
  TRACEPTR(ingame_background_original)
  TRACEPTR(ingame_background_scaled)
  TRACEINT(drawing_area_width)
  TRACEINT(drawing_area_height)
  
  ingame_background_scaled = gdk_pixbuf_scale_simple(
    ingame_background_original, drawing_area_width, drawing_area_height,
    GDK_INTERP_BILINEAR );
    
  ingame_foreground_scaled = gdk_pixbuf_scale_simple(
    ingame_foreground_original, drawing_area_width, drawing_area_height,
    GDK_INTERP_BILINEAR );
  
  g_string_free( file_path, TRUE );
  
}

void _unref_level_original_images()
{ TRACE(_unref_level_original_images)
  
  if( ingame_background_original != NULL ) {
    g_object_unref( ingame_background_original );
    ingame_background_original = NULL;
  }
  
  if( ingame_foreground_original != NULL ) {
    g_object_unref( ingame_foreground_original );
    ingame_foreground_original == NULL;
  }
  
}

void _unref_level_scaled_images()
{ TRACE(_unref_level_scaled_images)
  
  if( ingame_background_scaled != NULL ) {
    g_object_unref( ingame_background_scaled );
    ingame_background_scaled == NULL;
  }
  
  if( ingame_foreground_scaled != NULL ) {
    g_object_unref( ingame_foreground_scaled );
    ingame_foreground_scaled == NULL;
  }
}

/**
 * Unreference and reload all the images (e.g. because a new levelset has been
 * chosen).
 */
void graphics_reload_nonlevel_images()
{ TRACE(graphics_reload_nonlevel_images)
  
  /* First unreference all of the images (if loaded) */
  _unref_nonlevel_original_images();
  _unref_nonlevel_scaled_images();
  
  /* Now load the images */
  _load_nonlevel_original_images();
  
  /* and scale them */
  _scale_nonlevel_images();
  
  /* and refresh the screen */
  graphics_redraw_whole_screen();
  
}

/**
 * Do everything needed to set up graphics for the game to start.
 */
void graphics_setup_level()
{ TRACE(graphics_setup_level)
  

  /* Unref old images (if any) */
  _unref_level_original_images();
  _unref_level_scaled_images();
    
  /* load them */
  _load_level_original_images();
    
  /* and scale them to the right size */
  _scale_level_images(); 
  
  
}

void _scale_level_images()
{ TRACE(_scale_level_images)
  
  _unref_level_scaled_images();
  
  /* Only scale ingame images if they are loaded */
  if( ingame_background_original != NULL ) {
    ingame_background_scaled = gdk_pixbuf_scale_simple(
      ingame_background_original, drawing_area_width, drawing_area_height,
      GDK_INTERP_BILINEAR );
  } else {
    ingame_background_scaled = NULL;
  }
  
  if( ingame_foreground_original != NULL ) {
    ingame_foreground_scaled = gdk_pixbuf_scale_simple(
      ingame_foreground_original, drawing_area_width, drawing_area_height,
      GDK_INTERP_BILINEAR );
  } else {
    ingame_foreground_scaled = NULL;
  }
  
}

/** (re)scale all the level images */ 
void _scale_nonlevel_images()
{ TRACE(_scale_nonlevel_images)
  
  GdkRectangle sh_shape;
  GdkRectangle bl_shape;
    
  sh_shape = _ship_shape( 0, 0 );
  bl_shape = _bullet_shape( 0, 0 );
  
  /* Delete the old scaled images */
  _unref_nonlevel_scaled_images();

  TRACEPTR(title_screen_scaled)
  
  /* Create new scaled ones */
  title_screen_scaled = gdk_pixbuf_scale_simple( title_screen_original,
    drawing_area_width, drawing_area_height, GDK_INTERP_BILINEAR );
  
  TRACEPTR(title_screen_scaled)
  
  hiscore_screen_scaled = gdk_pixbuf_scale_simple( hiscore_screen_original,
    drawing_area_width, drawing_area_height, GDK_INTERP_BILINEAR );
  
  instructions_screen_scaled = gdk_pixbuf_scale_simple(
    instructions_screen_original, drawing_area_width, drawing_area_height,
    GDK_INTERP_BILINEAR );
  
  ship_scaled = gdk_pixbuf_scale_simple(
    ship_original, sh_shape.width, sh_shape.height, GDK_INTERP_BILINEAR );
  
  dead_ship_scaled = gdk_pixbuf_scale_simple(
    dead_ship_original, sh_shape.width, sh_shape.height,
    GDK_INTERP_BILINEAR );
  
  bullet_scaled = gdk_pixbuf_scale_simple(
    bullet_original, bl_shape.width, bl_shape.height,
    GDK_INTERP_BILINEAR );
  
}

void graphics_redraw_whole_screen()
{ TRACE(graphics_redraw_whole_screen)
  
  /* Different behaviour depending on what stae we are in. */
  switch(status)
  {
    case STATUS_IN_GAME:
      _ingame_redraw_whole_screen();
      break;
      
    case STATUS_DIED:
      _died_redraw_whole_screen();
      break;
      
    case STATUS_LEVEL_COMPLETE:
      _level_complete_redraw_whole_screen();
      break;
      
    case STATUS_TITLE_SCREEN:
      _title_screen_redraw_whole_screen();
      break;
    
    case STATUS_INSTRUCTIONS_SCREEN:
      _instructions_screen_redraw_whole_screen();
      break;
    
    case STATUS_HISCORE_SCREEN:
      _hiscore_screen_redraw_whole_screen();
      break;
    
  }
  
}

void _ingame_redraw_whole_screen()
{ TRACE(_ingame_redraw_whole_screen)
    
  int i;
    
  /* Clear the screen */
    
  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
  
  _draw_ship();
    
  if( bullet_exists )
  {
      _draw_bullet();
  }
    
  for( i=0; i<BLOCKS_WIDTH; i++ )
  {
        
      _draw_building( i );
        
  }
    
  gtk_widget_queue_draw( drawing_area ); 
    
}

void _died_redraw_whole_screen()
{ TRACE(_died_redraw_whole_screen)

  int i;
  
  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
    
  if( bullet_exists )
  {
    _draw_bullet();
  }
    
  for( i=0; i<BLOCKS_WIDTH; i++ )
  {
        
    _draw_building( i );
        
  }
  
  _draw_dead_ship();
  
  _draw_string( 0.5, 0.5, 0.2, "Game Over!" );
  
  gtk_widget_queue_draw( drawing_area );    
    
}

void _level_complete_redraw_whole_screen()
{ TRACE(_level_complete_redraw_whole_screen)

  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
  
  _draw_string( 0.5, 0.5, 0.15, "Level Complete!" );
  
  gtk_widget_queue_draw( drawing_area );  

}

void _title_screen_redraw_whole_screen()
{ TRACE(_title_screen_redraw_whole_screen)

  /* NOTE TO SELF: you can modify the VERSION in the file configure.in */
  
  TRACEPTR(title_screen_scaled)
  TRACEPTR(pixmap)
  
  /* Clear the screen by drawing the background picture onto the pixmap */
  gdk_pixbuf_render_to_drawable( title_screen_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
  
  _draw_string( 0.5, 0.90, 0.035, "by Andy Balaam" );
  _draw_string( 0.5, 0.95, 0.035, "v"VERSION"" );
  
  gtk_widget_queue_draw( drawing_area );  
}

void _hiscore_screen_redraw_whole_screen()
{ TRACE(_hiscore_screen_redraw_whole_screen)
    
  int i;
  double y;
  gchar tmp[30];
  
  gdk_pixbuf_render_to_drawable( hiscore_screen_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
  
  y = 0.32;
  for( i=0; i<10; i++ ) {

    _draw_string( 0.35, y, 0.05, high_score_names[i]->str );
    
    snprintf( tmp, 30, "%d", high_scores[i] );
    _draw_string( 0.8, y, 0.05, tmp );
    
    y += 0.07;
      
  }
  
  gtk_widget_queue_draw( drawing_area );  
    
}

void _instructions_screen_redraw_whole_screen()
{ TRACE(_instructions_screen_redraw_whole_screen)
    
  gdk_pixbuf_render_to_drawable( instructions_screen_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0,
    0 );
    
  gtk_widget_queue_draw( drawing_area );  
    
}

// -------------------------------------------------------------------------

void graphics_redraw_ship()
{ TRACE(graphics_redraw_ship)
    
    gint old_ship_x;
    gint old_ship_y;
    
    /* Draw the new ship */
    _draw_ship();
    
    /* Remove the old one */
    old_ship_x = ship_x - 1;
    old_ship_y = ship_y;
    
    if( old_ship_x < 0 )
    {
        old_ship_y--;
        old_ship_x = BLOCKS_WIDTH - 1;
    }
    
    if( old_ship_y >= 0 )
    {
        
        _remove_ship_at( old_ship_x, old_ship_y );
        
    }
    
    _queue_draw_ship();
    _queue_draw_ship_at(old_ship_x, old_ship_y );
    
}

void _draw_ship()
{ TRACE(_draw_ship)

  GdkRectangle shape;
    
  shape = _ship_shape( ship_x, ship_y );
  
  gdk_pixbuf_render_to_drawable( ship_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
  
}

void _draw_dead_ship()
{ TRACE(_draw_dead_ship)
  
  GdkRectangle shape;
    
  shape = _ship_shape( ship_x, ship_y );
  
  gdk_pixbuf_render_to_drawable( dead_ship_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
  
}

void _remove_ship_at( const gint pos_x, const gint pos_y )
{ TRACE(_remove_ship_at)
  
  GdkRectangle shape;
    
  shape = _ship_shape( pos_x, pos_y );
  
  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, shape.x, shape.y, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
  
}

void _queue_draw_ship()
{ TRACE(_queue_draw_ship)
    _queue_draw_ship_at( ship_x, ship_y );
}

void _queue_draw_ship_at( const gint pos_x, const gint pos_y )
{ TRACE(_queue_draw_ship_at)
  
    GdkRectangle shape;
    
    shape = _ship_shape( pos_x, pos_y );
        
    gtk_widget_queue_draw_area( drawing_area,
        shape.x,
        shape.y,
        shape.width,
        shape.height );
    
}

GdkRectangle _ship_shape( const gint pos_x, const gint pos_y )
{ TRACE(_ship_shape)
    
    GdkRectangle answer;
    
    answer.x = _scaleup_pos_x( pos_x );
    answer.y = _scaleup_pos_y( pos_y );
    answer.width = _scaleup_size_x( 1 );
    answer.height = _scaleup_size_y( 1 );
    
    return answer;
    
}


void graphics_redraw_bullet()
{ TRACE(graphics_redraw_bullet)
    
    gint old_bullet_y;
    
    /* Draw the new bullet */
    if( bullet_y > ship_y )
    {
        _draw_bullet();
    }
    
    /* Remove the old one */
    old_bullet_y = bullet_y - 1;
      
    if( (old_bullet_y > ship_y) || (bullet_x != ship_x) )
    {
        
        _remove_bullet_at( bullet_x, old_bullet_y );
        
    }
    
    _queue_draw_bullet();
    _queue_draw_bullet_at( bullet_x, old_bullet_y );
    
}

void _draw_bullet()
{ TRACE(_draw_bullet)
  
  GdkRectangle shape;
    
  shape = _bullet_shape( bullet_x, bullet_y );
      
  gdk_pixbuf_render_to_drawable( bullet_scaled, pixmap,
    drawing_area->style->white_gc, 0, 0, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
        
        
}

void _remove_bullet_at( const gint pos_x, const gint pos_y )
{ TRACE(_remove_bullet_at)
    
  GdkRectangle shape;
    
  shape = _bullet_shape( pos_x, pos_y );
    
  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, shape.x, shape.y, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
    
}

void _queue_draw_bullet()
{ TRACE(_queue_draw_bullet)
    _queue_draw_bullet_at( bullet_x, bullet_y );
}

void _queue_draw_bullet_at( const gint pos_x, const gint pos_y )
{ TRACE(_queue_draw_bullet_at)
    GdkRectangle shape;
    
    shape = _bullet_shape( pos_x, pos_y );
        
    gtk_widget_queue_draw_area( drawing_area,
        shape.x,
        shape.y,
        shape.width,
        shape.height );
    
}

GdkRectangle _bullet_shape( const gint pos_x, const gint pos_y )
{ TRACE(_bullet_shape)
    
    GdkRectangle answer;
    
    answer.x = _scaleup_pos_x( pos_x + 0.3 );
    answer.y = _scaleup_pos_y( pos_y + 0.3 );
    answer.width = _scaleup_size_x( 1 - 0.6 );
    answer.height = _scaleup_size_y( 1 - 0.6 );
    
    return answer;
    
}



void graphics_redraw_building( const gint building_no, const gint old_height )
{ TRACE(graphics_redraw_building)
    
  GdkRectangle shape;
  gint max_y;
    
  /* Just clear the destroyed area (and the bullet above it) */
   
  shape.x = _scaleup_pos_x( building_no + 0.01 );
  shape.y = _scaleup_pos_y( BLOCKS_HEIGHT - old_height - 1 - 0.05 );
  shape.width = _scaleup_size_x( 1 - 0.02 );
  shape.height = _scaleup_size_y( ( 1 + old_height
    - building_heights[building_no] ) + 0.1 );
  
  // Adjust to screen height if needed
  max_y = shape.y + shape.height;
  if( max_y > drawing_area->allocation.height ) {
    shape.height -= ( max_y - drawing_area->allocation.height );
  }
        
  gdk_pixbuf_render_to_drawable( ingame_background_scaled, pixmap,
    drawing_area->style->white_gc, shape.x, shape.y, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
    
  gtk_widget_queue_draw_area( drawing_area,
    shape.x,
    shape.y,
    shape.width,
    shape.height );
    
}

void _draw_building( const gint building_no )
{ TRACE(_draw_building)
    
  /* Draw the building */
    
  GdkRectangle shape;
  gint drawing_area_height, max_y;
    
  shape.x = _scaleup_pos_x( building_no + 0.01 );
  shape.y = _scaleup_pos_y( BLOCKS_HEIGHT
    - building_heights[building_no] + 0.01 );
  shape.width = _scaleup_size_x( 1 - 0.02 );
  shape.height = _scaleup_size_y( building_heights[building_no]
    - 0.02 );
  
  drawing_area_height  = drawing_area->allocation.height;
  
  // Adjust to screen size if needed
  max_y = shape.y + shape.height;
  if( max_y > drawing_area_height ) {
    shape.height -= ( max_y - drawing_area_height );
  }
  
  if( shape.height < 0 ) {
    shape.height = 0;
  }
  
  gdk_pixbuf_render_to_drawable( ingame_foreground_scaled, pixmap,
    drawing_area->style->white_gc, shape.x, shape.y, shape.x, shape.y,
    shape.width, shape.height, GDK_RGB_DITHER_NONE, 0, 0 );
    
    gtk_widget_queue_draw_area( drawing_area,
        shape.x,
        shape.y,
        shape.width,
        shape.height );
    
}


/*--------------------------------------------------------------------------
    utilities
  --------------------------------------------------------------------------*/

gint _scaleup_pos_x( const gdouble block_val )
{ TRACE(_scaleup_pos_x)
  
  return floor( drawing_area->allocation.width * block_val / BLOCKS_WIDTH );
  
}

gint _scaleup_pos_y( const gdouble block_val )
{ TRACE(_scaleup_pos_y)
  return floor( drawing_area->allocation.height * block_val / BLOCKS_HEIGHT );
}

gint _scaleup_size_x( const gdouble block_val )
{ TRACE(_scaleup_size_x)
  return floor( drawing_area->allocation.width * block_val / BLOCKS_WIDTH );
}

gint _scaleup_size_y( const gdouble block_val )
{ TRACE(_scaleup_size_y)
  return floor( drawing_area->allocation.height * block_val / BLOCKS_HEIGHT );
}

/*--------------------------------------------------------------------------
    text
  --------------------------------------------------------------------------*/

void _draw_string( double x, double y, double height, gchar* text )
{ TRACE(_draw_string)
  
  GdkFont* fnt;
  gchar tmp[100];
  
  snprintf( tmp, 100, "-*-helvetica-bold-r-*-*-%d-*-*-*-*-*-*-*",
    (int)(drawing_area->allocation.height*height) );
  
  fnt = gdk_font_load( tmp );
  
  gdk_draw_string( pixmap, fnt, drawing_area->style->black_gc,
    (x*drawing_area->allocation.width)
      -  gdk_string_measure( fnt, text ) / 2,
    y*drawing_area->allocation.height,
    text );

  gdk_font_unref( fnt );
  
}





